/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.pinot.segment.local.io.compression;

import java.io.IOException;
import java.nio.ByteBuffer;
import org.apache.pinot.segment.spi.compression.ChunkCompressionType;
import org.apache.pinot.segment.spi.compression.ChunkCompressor;
import org.xerial.snappy.Snappy;


/**
 * Implementation of {@link ChunkCompressor} using Snappy.
 */
class SnappyCompressor implements ChunkCompressor {

  static final SnappyCompressor INSTANCE = new SnappyCompressor();

  private SnappyCompressor() {
  }

  @Override
  public int compress(ByteBuffer inDecompressed, ByteBuffer outCompressed)
      throws IOException {
    return Snappy.compress(inDecompressed, outCompressed);
  }

  @Override
  public int maxCompressedSize(int uncompressedSize) {
    return Snappy.maxCompressedLength(uncompressedSize);
  }

  @Override
  public ChunkCompressionType compressionType() {
    return ChunkCompressionType.SNAPPY;
  }
}
