/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.geometry;

import com.google.common.annotations.GwtCompatible;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.geometry.ParametrizedS2Point;
import com.google.common.geometry.S2Cell;
import com.google.common.geometry.S2CellId;
import com.google.common.geometry.S2Edge;
import com.google.common.geometry.S2EdgeUtil;
import com.google.common.geometry.S2Point;
import com.google.common.geometry.S2Projections;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

@GwtCompatible
public strictfp abstract class S2EdgeIndex {
    private static final double THICKENING = 0.01;
    private long[] cells;
    private int[] edges;
    private int minimumS2LevelUsed;
    private boolean indexComputed;
    private int queryCount;

    public void reset() {
        this.minimumS2LevelUsed = 30;
        this.indexComputed = false;
        this.queryCount = 0;
        this.cells = null;
        this.edges = null;
    }

    private static final int compare(long cell1, int edge1, long cell2, int edge2) {
        if (cell1 < cell2) {
            return -1;
        }
        if (cell1 > cell2) {
            return 1;
        }
        if (edge1 < edge2) {
            return -1;
        }
        if (edge1 > edge2) {
            return 1;
        }
        return 0;
    }

    public final void computeIndex() {
        int i;
        if (this.indexComputed) {
            return;
        }
        ArrayList<Long> cellList = Lists.newArrayList();
        ArrayList<Integer> edgeList = Lists.newArrayList();
        for (i = 0; i < this.getNumEdges(); ++i) {
            S2Point from = this.edgeFrom(i);
            S2Point to = this.edgeTo(i);
            ArrayList<S2CellId> cover = Lists.newArrayList();
            int level = this.getCovering(from, to, true, cover);
            this.minimumS2LevelUsed = Math.min(this.minimumS2LevelUsed, level);
            for (S2CellId cellId : cover) {
                cellList.add(cellId.id());
                edgeList.add(i);
            }
        }
        this.cells = new long[cellList.size()];
        this.edges = new int[edgeList.size()];
        for (i = 0; i < this.cells.length; ++i) {
            this.cells[i] = (Long)cellList.get(i);
            this.edges[i] = (Integer)edgeList.get(i);
        }
        this.sortIndex();
        this.indexComputed = true;
    }

    private void sortIndex() {
        Integer[] indices = new Integer[this.cells.length];
        for (int i = 0; i < indices.length; ++i) {
            indices[i] = i;
        }
        Arrays.sort(indices, new Comparator<Integer>(){

            @Override
            public int compare(Integer index1, Integer index2) {
                return S2EdgeIndex.compare(S2EdgeIndex.this.cells[index1], S2EdgeIndex.this.edges[index1], S2EdgeIndex.this.cells[index2], S2EdgeIndex.this.edges[index2]);
            }
        });
        long[] newCells = new long[this.cells.length];
        int[] newEdges = new int[this.edges.length];
        for (int i = 0; i < indices.length; ++i) {
            newCells[i] = this.cells[indices[i]];
            newEdges[i] = this.edges[indices[i]];
        }
        this.cells = newCells;
        this.edges = newEdges;
    }

    public final boolean isIndexComputed() {
        return this.indexComputed;
    }

    protected final void incrementQueryCount() {
        ++this.queryCount;
    }

    public final void predictAdditionalCalls(int n) {
        if (this.indexComputed) {
            return;
        }
        if (this.getNumEdges() > 100 && this.queryCount + n > 30) {
            this.computeIndex();
        }
    }

    public abstract int getNumEdges();

    public abstract S2Point edgeFrom(int var1);

    public abstract S2Point edgeTo(int var1);

    public S2Edge edgeFromTo(int index) {
        return new S2Edge(this.edgeFrom(index), this.edgeTo(index));
    }

    protected void findCandidateCrossings(S2Point a, S2Point b, List<Integer> candidateCrossings) {
        Preconditions.checkState(this.indexComputed);
        ArrayList<S2CellId> cover = Lists.newArrayList();
        this.getCovering(a, b, false, cover);
        HashSet<Integer> uniqueSet = new HashSet<Integer>();
        this.getEdgesInParentCells(cover, uniqueSet);
        this.getEdgesInChildrenCells(a, b, cover, uniqueSet);
        candidateCrossings.clear();
        candidateCrossings.addAll(uniqueSet);
    }

    private static S2CellId containingCell(S2Point pa, S2Point pb, S2Point pc, S2Point pd) {
        S2CellId a = S2CellId.fromPoint(pa);
        S2CellId b = S2CellId.fromPoint(pb);
        S2CellId c = S2CellId.fromPoint(pc);
        S2CellId d = S2CellId.fromPoint(pd);
        if (a.face() != b.face() || a.face() != c.face() || a.face() != d.face()) {
            return S2CellId.sentinel();
        }
        while (!(a.equals(b) && a.equals(c) && a.equals(d))) {
            a = a.parent();
            b = b.parent();
            c = c.parent();
            d = d.parent();
        }
        return a;
    }

    private static S2CellId containingCell(S2Point pa, S2Point pb) {
        S2CellId a = S2CellId.fromPoint(pa);
        S2CellId b = S2CellId.fromPoint(pb);
        if (a.face() != b.face()) {
            return S2CellId.sentinel();
        }
        while (!a.equals(b)) {
            a = a.parent();
            b = b.parent();
        }
        return a;
    }

    private int getCovering(S2Point a, S2Point b, boolean thickenEdge, ArrayList<S2CellId> edgeCovering) {
        S2CellId containingCellId;
        edgeCovering.clear();
        double edgeLength = a.angle(b);
        int idealLevel = S2Projections.PROJ.minWidth.getMaxLevel(edgeLength * 1.02);
        if (!thickenEdge) {
            containingCellId = S2EdgeIndex.containingCell(a, b);
        } else if (idealLevel == 30) {
            containingCellId = new S2CellId(65520L).parent(3);
        } else {
            S2Point pq = S2Point.mul(S2Point.minus(b, a), 0.01);
            S2Point ortho = S2Point.mul(S2Point.normalize(S2Point.crossProd(pq, a)), edgeLength * 0.01);
            S2Point p = S2Point.minus(a, pq);
            S2Point q = S2Point.add(b, pq);
            containingCellId = S2EdgeIndex.containingCell(S2Point.minus(p, ortho), S2Point.add(p, ortho), S2Point.minus(q, ortho), S2Point.add(q, ortho));
        }
        if (!containingCellId.equals(S2CellId.sentinel()) && containingCellId.level() >= idealLevel - 2) {
            edgeCovering.add(containingCellId);
            return containingCellId.level();
        }
        if (idealLevel == 0) {
            S2CellId cellid = S2CellId.begin(0);
            while (!cellid.equals(S2CellId.end(0))) {
                edgeCovering.add(cellid);
                cellid = cellid.next();
            }
            return 0;
        }
        S2Point middle = S2Point.normalize(S2Point.div(S2Point.add(a, b), 2.0));
        int actualLevel = Math.min(idealLevel, 29);
        S2CellId.fromPoint(middle).getVertexNeighbors(actualLevel, edgeCovering);
        return actualLevel;
    }

    private int[] getEdges(long cell1, long cell2) {
        if (cell1 > cell2) {
            long temp = cell1;
            cell1 = cell2;
            cell2 = temp;
        }
        return new int[]{-1 - this.binarySearch(cell1, Integer.MIN_VALUE), -1 - this.binarySearch(cell2, Integer.MAX_VALUE)};
    }

    private int binarySearch(long cell, int edge) {
        int low = 0;
        int high = this.cells.length - 1;
        while (low <= high) {
            int mid = low + high >>> 1;
            int cmp = S2EdgeIndex.compare(this.cells[mid], this.edges[mid], cell, edge);
            if (cmp < 0) {
                low = mid + 1;
                continue;
            }
            if (cmp > 0) {
                high = mid - 1;
                continue;
            }
            return mid;
        }
        return -(low + 1);
    }

    private void getEdgesInParentCells(List<S2CellId> cover, Set<Integer> candidateCrossings) {
        HashSet<S2CellId> parentCells = Sets.newHashSet();
        for (S2CellId coverCell : cover) {
            for (int parentLevel = coverCell.level() - 1; parentLevel >= this.minimumS2LevelUsed && parentCells.add(coverCell.parent(parentLevel)); --parentLevel) {
            }
        }
        for (S2CellId parentCell : parentCells) {
            int[] bounds = this.getEdges(parentCell.id(), parentCell.id());
            for (int i = bounds[0]; i < bounds[1]; ++i) {
                candidateCrossings.add(this.edges[i]);
            }
        }
    }

    private static boolean edgeIntersectsCellBoundary(S2Point a, S2Point b, S2Cell cell) {
        int i;
        S2Point[] vertices = new S2Point[4];
        for (i = 0; i < 4; ++i) {
            vertices[i] = cell.getVertex(i);
        }
        for (i = 0; i < 4; ++i) {
            S2Point fromPoint = vertices[i];
            S2Point toPoint = vertices[(i + 1) % 4];
            if (!S2EdgeUtil.lenientCrossing(a, b, fromPoint, toPoint)) continue;
            return true;
        }
        return false;
    }

    private void getEdgesInChildrenCells(S2Point a, S2Point b, List<S2CellId> cover, Set<Integer> candidateCrossings) {
        S2Cell[] children = null;
        while (!cover.isEmpty()) {
            int i;
            S2CellId cell = cover.remove(cover.size() - 1);
            int[] bounds = this.getEdges(cell.rangeMin().id(), cell.rangeMax().id());
            if (bounds[1] - bounds[0] <= 16) {
                for (i = bounds[0]; i < bounds[1]; ++i) {
                    candidateCrossings.add(this.edges[i]);
                }
                continue;
            }
            bounds = this.getEdges(cell.id(), cell.id());
            for (i = bounds[0]; i < bounds[1]; ++i) {
                candidateCrossings.add(this.edges[i]);
            }
            if (children == null) {
                children = new S2Cell[4];
                for (i = 0; i < 4; ++i) {
                    children[i] = new S2Cell();
                }
            }
            new S2Cell(cell).subdivide(children);
            for (S2Cell child : children) {
                if (!S2EdgeIndex.edgeIntersectsCellBoundary(a, b, child)) continue;
                cover.add(child.id());
            }
        }
    }

    public void clipEdge(S2Point a0, S2Point a1, boolean addSharedEdges, Collection<ParametrizedS2Point> intersections) {
        DataEdgeIterator it = new DataEdgeIterator(this);
        S2EdgeUtil.EdgeCrosser crosser = new S2EdgeUtil.EdgeCrosser(a0, a1, a0);
        S2Point b0 = null;
        S2Point b1 = null;
        it.getCandidates(a0, a1);
        while (it.hasNext()) {
            int crossing;
            S2Point previous = b1;
            S2Edge bEdge = this.edgeFromTo(it.index());
            b0 = bEdge.getStart();
            b1 = bEdge.getEnd();
            if (previous == null || !previous.equals(b0)) {
                crosser.restartAt(b0);
            }
            if ((crossing = crosser.robustCrossing(b1)) >= 0) {
                if (crossing > 0) {
                    S2Point x = S2EdgeUtil.getIntersection(a0, a1, b0, b1);
                    double t = S2EdgeUtil.getDistanceFraction(x, a0, a1);
                    intersections.add(new ParametrizedS2Point(t, x));
                } else if (S2EdgeUtil.vertexCrossing(a0, a1, b0, b1)) {
                    double t;
                    double d = t = a0.equals(b0) || a0.equals(b1) ? 0.0 : 1.0;
                    if (!addSharedEdges && a1.equals(b1)) {
                        t = 1.0;
                    }
                    intersections.add(new ParametrizedS2Point(t, t == 0.0 ? a0 : a1));
                }
            }
            it.next();
        }
    }

    public strictfp static class DataEdgeIterator {
        private final S2EdgeIndex edgeIndex;
        private boolean isBruteForce;
        private int currentIndex;
        private int numEdges;
        ArrayList<Integer> candidates;
        private int currentIndexInCandidates;

        public DataEdgeIterator(S2EdgeIndex edgeIndex) {
            this.edgeIndex = edgeIndex;
            this.candidates = Lists.newArrayList();
        }

        public void getCandidates(S2Point a, S2Point b) {
            this.edgeIndex.predictAdditionalCalls(1);
            boolean bl = this.isBruteForce = !this.edgeIndex.isIndexComputed();
            if (this.isBruteForce) {
                this.edgeIndex.incrementQueryCount();
                this.currentIndex = 0;
                this.numEdges = this.edgeIndex.getNumEdges();
            } else {
                this.candidates.clear();
                this.edgeIndex.findCandidateCrossings(a, b, this.candidates);
                this.currentIndexInCandidates = 0;
                if (!this.candidates.isEmpty()) {
                    this.currentIndex = this.candidates.get(0);
                }
            }
        }

        public int index() {
            Preconditions.checkState(this.hasNext());
            return this.currentIndex;
        }

        public boolean hasNext() {
            if (this.isBruteForce) {
                return this.currentIndex < this.numEdges;
            }
            return this.currentIndexInCandidates < this.candidates.size();
        }

        public void next() {
            Preconditions.checkState(this.hasNext());
            if (this.isBruteForce) {
                ++this.currentIndex;
            } else {
                ++this.currentIndexInCandidates;
                if (this.currentIndexInCandidates < this.candidates.size()) {
                    this.currentIndex = this.candidates.get(this.currentIndexInCandidates);
                }
            }
        }
    }
}

