-- This test suite checks the WITH SCHEMA TYPE EVOLUTION clause

-- In EVOLUTION mode Spark will inherit everything from the query, unless
--  a column list is given. In that case it behaves like TYPE EVOLUTION
DROP TABLE IF EXISTS t;
CREATE TABLE t(c1 INT NOT NULL, c2 INT) USING PARQUET;
INSERT INTO t VALUES (1, 2);
CREATE OR REPLACE VIEW v WITH SCHEMA EVOLUTION AS SELECT * FROM t;
SELECT * FROM v;
DESCRIBE EXTENDED v;

-- In EVOLUTION views inherit column type changes and name changes
DROP TABLE IF EXISTS t;
CREATE TABLE t(c4 STRING NOT NULL, c5 DOUBLE) USING PARQUET;
INSERT INTO t VALUES ('1', 2.0);
SELECT * FROM v;
-- The view now describes as v(c3 STRING, c4 DOUBLE)
DESCRIBE EXTENDED v;

-- In EVOLUTION new columns are inherited
DROP TABLE IF EXISTS t;
CREATE TABLE t(c4 STRING, c5 DOUBLE, c6 DATE) USING PARQUET;
INSERT INTO t VALUES ('1', 2.0, DATE'2022-01-01');
SELECT * FROM v;
-- The view describes as v(c4 STRING, c5 DOUBLE, c6 DATE)
DESCRIBE EXTENDED v;

-- We can even drop columns
DROP TABLE IF EXISTS t;
CREATE TABLE t(c1 INT, c2 INT) USING PARQUET;
INSERT INTO t VALUES (1, 2);
CREATE OR REPLACE VIEW v WITH SCHEMA EVOLUTION AS SELECT * FROM t;
SELECT * FROM v;
-- Describes as v(c1 INT, c2 INT)
DESCRIBE EXTENDED v;

DROP TABLE IF EXISTS t;
CREATE TABLE t(c1 INT) USING PARQUET;
SELECT * FROM v;
-- The view describes as v(c1 INT)
DESCRIBE EXTENDED v;

-- If a column list is given it behaves like TYPE EVOLUTION
DROP TABLE IF EXISTS t;
CREATE TABLE t(c1 INT NOT NULL, c2 INT) USING PARQUET;
INSERT INTO t VALUES (1, 2);
CREATE OR REPLACE VIEW v(a1, a2) WITH SCHEMA EVOLUTION AS SELECT * FROM t;
SELECT * FROM v;
DESCRIBE EXTENDED v;

-- In EVOLUTION views with explicit column lists still inherit column type changes
DROP TABLE IF EXISTS t;
CREATE TABLE t(c1 STRING NOT NULL, c2 DOUBLE) USING PARQUET;
INSERT INTO t VALUES ('1', 2.0);
SELECT * FROM v;
-- The view now describes as v(a1 STRING, a2 DOUBLE)
DESCRIBE EXTENDED v;

-- In EVOLUTION views with explicit column lists no new columns are inherited
DROP TABLE IF EXISTS t;
CREATE TABLE t(c1 STRING, c2 DOUBLE, c3 DATE) USING PARQUET;
INSERT INTO t VALUES ('1', 2.0, DATE'2022-01-01');
SELECT * FROM v;
-- The view still describes as v(a1 STRING, a2 DOUBLE)
DESCRIBE EXTENDED v;

-- In EVOLUTION views with explicit column lists can't drop a column
DROP TABLE IF EXISTS t;
CREATE TABLE t(c1 INT, c2 INT) USING PARQUET;
INSERT INTO t VALUES (1, 2);
CREATE OR REPLACE VIEW v(a1, a2) WITH SCHEMA EVOLUTION AS SELECT * FROM t;
SELECT * FROM v;

-- Describes as v(a1 INT, a2 INT)
DESCRIBE EXTENDED v;

DROP TABLE IF EXISTS t;
CREATE TABLE t(c1 INT) USING PARQUET;

-- The view should be invalidated, it lost a column
SELECT * FROM v;
DESCRIBE EXTENDED v;

-- Attempt to rename a column
DROP TABLE IF EXISTS t;
CREATE TABLE t(c3 INT, c2 INT) USING PARQUET;
SELECT * FROM v;
DESCRIBE EXTENDED v;

-- Test preservation of comments
DROP TABLE IF EXISTS t;
CREATE TABLE t(c1 INT COMMENT 'c1', c2 INT COMMENT 'c2') USING PARQUET;

-- EVOLUTION, column list, but no comments
CREATE OR REPLACE VIEW v(a1, a2) WITH SCHEMA TYPE EVOLUTION AS SELECT * FROM t;
DESCRIBE EXTENDED v;

-- EVOLUTION, column list, but no comments
CREATE OR REPLACE VIEW v(a1, a2) WITH SCHEMA EVOLUTION AS SELECT * FROM t;
DESCRIBE EXTENDED v;

DROP TABLE IF EXISTS t;
CREATE TABLE t(c1 BIGINT COMMENT 'c1 6c', c2 STRING COMMENT 'c2 6c') USING PARQUET;
SELECT * FROM v;
DESCRIBE EXTENDED v;

-- EVOLUTION, column list with comments
CREATE OR REPLACE VIEW v(a1 COMMENT 'a1', a2 COMMENT 'a2') WITH SCHEMA EVOLUTION AS SELECT * FROM t;
DESCRIBE EXTENDED v;

DROP TABLE IF EXISTS t;
CREATE TABLE t(c1 BIGINT COMMENT 'c1 6d', c2 STRING COMMENT 'c2 6d') USING PARQUET;
SELECT * FROM v;
DESCRIBE EXTENDED v;

-- EVOLUTION, no column list
CREATE OR REPLACE VIEW v WITH SCHEMA EVOLUTION AS SELECT * FROM t;
DESCRIBE EXTENDED v;

DROP TABLE IF EXISTS t;
CREATE TABLE t(c1 BIGINT COMMENT 'c1 6e', c2 STRING COMMENT 'c2 6e') USING PARQUET;
SELECT * FROM v;
DESCRIBE EXTENDED v;

-- Test error condition where a duplicate column name is produced
DROP TABLE IF EXISTS t1;
CREATE TABLE t1(c1 INT) USING PARQUET;
DROP TABLE IF EXISTS t2;
CREATE TABLE t2(c2 INT) USING PARQUET;
CREATE OR REPLACE VIEW v WITH SCHEMA EVOLUTION AS SELECT * FROM t1, t2;
SELECT * FROM v;
DROP TABLE IF EXISTS t2;
CREATE TABLE t2(c1 INT) USING PARQUET;
-- This should fail with a duplicate column error
SELECT * FROM v;
DROP TABLE IF EXISTS t1;
DROP TABLE IF EXISTS t2;

-- Test ALTER VIEW ... WITH SCHEMA EVOLUTION

DROP TABLE IF EXISTS t;
CREATE TABLE t(c1 INT) USING PARQUET;
INSERT INTO t VALUES(1);
CREATE OR REPLACE VIEW v AS SELECT * FROM t;

ALTER VIEW v WITH SCHEMA EVOLUTION;
DROP TABLE IF EXISTS t;
CREATE TABLE t(c1 STRING, c2 INT) USING PARQUET;
-- No error, extra column
SELECT * FROM v;
DESCRIBE EXTENDED v;

-- clean up
DROP VIEW IF EXISTS v;
DROP TABLE IF EXISTS t;
