package Module::CoreList;
use strict;

our ( %released, %version, %families, %upstream, %bug_tracker, %deprecated, %delta );

use version;
our $VERSION = '5.20260220';

sub PKG_PATTERN () { q#\A[a-zA-Z_][0-9a-zA-Z_]*(?:(::|')[0-9a-zA-Z_]+)*\z# }
sub _looks_like_invocant ($) { local $@; !!eval { $_[0]->isa(__PACKAGE__) } }

sub _undelta {
    my ($delta) = @_;
    my (%expanded, $delta_from, $base, $changed, $removed);
    for my $v (sort keys %$delta) {
        ($delta_from, $changed, $removed) = @{$delta->{$v}}{qw( delta_from changed removed )};
        $base = $delta_from ? $expanded{$delta_from} : {};
        my %full = ( %$base, %{$changed || {}} );
        delete @full{ keys %$removed };
        $expanded{$v} = \%full;
    }
    return %expanded;
}

sub _released_order {   # Sort helper, to make '?' sort after everything else
    (substr($released{$a}, 0, 1) eq "?")
    ? ((substr($released{$b}, 0, 1) eq "?")
        ? 0
        : 1)
    : ((substr($released{$b}, 0, 1) eq "?")
        ? -1
        : $released{$a} cmp $released{$b} )
}

my $dumpinc = 0;
sub import {
    my $self = shift;
    my $what = shift || '';
    if ($what eq 'dumpinc') {
        $dumpinc = 1;
    }
}

END {
    print "---INC---\n", join "\n" => keys %INC
      if $dumpinc;
}


sub first_release_raw {
    shift if defined $_[1] and $_[1] =~ PKG_PATTERN and _looks_like_invocant $_[0];
    my $module = shift;
    my $version = shift;

    my @perls = $version
        ? grep { defined $version{$_}{ $module } &&
                        $version{$_}{ $module } ge $version } keys %version
        : grep { exists $version{$_}{ $module }             } keys %version;

    return @perls;
}

sub first_release_by_date {
    my @perls = &first_release_raw;
    return unless @perls;
    return (sort _released_order @perls)[0];
}

sub first_release {
    my @perls = &first_release_raw;
    return unless @perls;
    return (sort { $a cmp $b } @perls)[0];
}

sub find_modules {
    shift if _looks_like_invocant $_[0];
    my $regex = shift;
    my @perls = @_ ? @_ : keys %version;

    my %mods;
    foreach (@perls) {
        while (my ($k, $v) = each %{$version{$_}}) {
            $mods{$k}++ if $k =~ $regex;
        }
    }
    return sort keys %mods
}

sub find_version {
    shift if _looks_like_invocant $_[0];
    my $v = shift;
    return $version{$v} if defined $v and defined $version{$v};
    return;
}

sub is_deprecated {
    shift if defined $_[1] and $_[1] =~ PKG_PATTERN and _looks_like_invocant $_[0];
    my $module = shift;
    my $perl_version = shift || $];
    return unless $module && exists $deprecated{$perl_version}{$module};
    return $deprecated{$perl_version}{$module};
}

sub deprecated_in {
    shift if defined $_[1] and $_[1] =~ PKG_PATTERN and _looks_like_invocant $_[0];
    my $module = shift or return;
    my @perls = grep { exists $deprecated{$_}{$module} } keys %deprecated;
    return unless @perls;
    require List::Util;
    return List::Util::minstr(@perls);
}

sub removed_from {
  my @perls = &removed_raw;
  return shift @perls;
}

sub removed_from_by_date {
  my @perls = sort _released_order &removed_raw;
  return shift @perls;
}

sub removed_raw {
  shift if defined $_[1] and $_[1] =~ PKG_PATTERN and _looks_like_invocant $_[0];
  my $mod = shift;
  return unless my @perls = sort { $a cmp $b } first_release_raw($mod);
  my $last = pop @perls;
  my @removed = grep { $_ > $last } sort { $a cmp $b } keys %version;
  return @removed;
}

sub changes_between {
  shift if _looks_like_invocant $_[0];
  my $left_ver = shift;
  my $right_ver = shift;

  my $left  = $version{ $left_ver } || {};
  my $right = $version{ $right_ver } || {};

  my %uniq = (%$left, %$right);

  my %changes;
  for my $lib (keys %uniq) {
      my $lhs = exists $left->{ $lib }
              ? (defined $left->{ $lib } ? $left->{ $lib } : '(undef)')
              : '(absent)';
      my $rhs = exists $right->{ $lib }
              ? (defined $right->{ $lib } ? $right->{ $lib } : '(undef)')
              : '(absent)';

      next if $lhs eq $rhs;

      my $change = {
        (exists $left->{$lib}  ? (left  => $left->{$lib})  : ()),
        (exists $right->{$lib} ? (right => $right->{$lib}) : ()),
      };

      $changes{$lib} = $change;
  }

  return %changes;
}

# When things escaped.
# NB. If you put version numbers with trailing zeroes here, you
# should also add an alias for the numerical ($]) version; see
# just before the __END__ of this module.
%released = (
    5.000    => '1994-10-17',
    5.001    => '1995-03-14',
    5.002    => '1996-02-29',
    5.00307  => '1996-10-10',
    5.004    => '1997-05-15',
    5.005    => '1998-07-22',
    5.00503  => '1999-03-28',
    5.00405  => '1999-04-29',
    5.006    => '2000-03-22',
    5.006001 => '2001-04-08',
    5.007003 => '2002-03-05',
    5.008    => '2002-07-19',
    5.008001 => '2003-09-25',
    5.009    => '2003-10-27',
    5.008002 => '2003-11-05',
    5.006002 => '2003-11-15',
    5.008003 => '2004-01-14',
    5.00504  => '2004-02-23',
    5.009001 => '2004-03-16',
    5.008004 => '2004-04-21',
    5.008005 => '2004-07-19',
    5.008006 => '2004-11-27',
    5.009002 => '2005-04-01',
    5.008007 => '2005-05-30',
    5.009003 => '2006-01-28',
    5.008008 => '2006-01-31',
    5.009004 => '2006-08-15',
    5.009005 => '2007-07-07',
    5.010000 => '2007-12-18',
    5.008009 => '2008-12-14',
    5.010001 => '2009-08-22',
    5.011000 => '2009-10-02',
    5.011001 => '2009-10-20',
    5.011002 => '2009-11-20',
    5.011003 => '2009-12-20',
    5.011004 => '2010-01-20',
    5.011005 => '2010-02-20',
    5.012000 => '2010-04-12',
    5.013000 => '2010-04-20',
    5.012001 => '2010-05-16',
    5.013001 => '2010-05-20',
    5.013002 => '2010-06-22',
    5.013003 => '2010-07-20',
    5.013004 => '2010-08-20',
    5.012002 => '2010-09-06',
    5.013005 => '2010-09-19',
    5.013006 => '2010-10-20',
    5.013007 => '2010-11-20',
    5.013008 => '2010-12-20',
    5.012003 => '2011-01-21',
    5.013009 => '2011-01-20',
    5.013010 => '2011-02-20',
    5.013011 => '2011-03-20',
    5.014000 => '2011-05-14',
    5.012004 => '2011-06-20',
    5.012005 => '2012-11-10',
    5.014001 => '2011-06-16',
    5.015000 => '2011-06-20',
    5.015001 => '2011-07-20',
    5.015002 => '2011-08-20',
    5.014002 => '2011-09-26',
    5.015003 => '2011-09-20',
    5.015004 => '2011-10-20',
    5.015005 => '2011-11-20',
    5.015006 => '2011-12-20',
    5.015007 => '2012-01-20',
    5.015008 => '2012-02-20',
    5.015009 => '2012-03-20',
    5.016000 => '2012-05-20',
    5.016001 => '2012-08-08',
    5.016002 => '2012-11-01',
    5.017000 => '2012-05-26',
    5.017001 => '2012-06-20',
    5.017002 => '2012-07-20',
    5.017003 => '2012-08-20',
    5.017004 => '2012-09-20',
    5.014003 => '2012-10-12',
    5.017005 => '2012-10-20',
    5.017006 => '2012-11-20',
    5.017007 => '2012-12-18',
    5.017008 => '2013-01-20',
    5.017009 => '2013-02-20',
    5.014004 => '2013-03-10',
    5.016003 => '2013-03-11',
    5.017010 => '2013-03-21',
    5.017011 => '2013-04-20',
    5.018000 => '2013-05-18',
    5.019000 => '2013-05-20',
    5.019001 => '2013-06-21',
    5.019002 => '2013-07-22',
    5.018001 => '2013-08-12',
    5.019003 => '2013-08-20',
    5.019004 => '2013-09-20',
    5.019005 => '2013-10-20',
    5.019006 => '2013-11-20',
    5.019007 => '2013-12-20',
    5.018002 => '2014-01-06',
    5.018003 => '2014-10-01',
    5.018004 => '2014-10-01',
    5.019008 => '2014-01-20',
    5.019009 => '2014-02-20',
    5.01901  => '2014-03-20',
    5.019011 => '2014-04-20',
    5.020000 => '2014-05-27',
    5.021000 => '2014-05-27',
    5.021001 => '2014-06-20',
    5.021002 => '2014-07-20',
    5.021003 => '2014-08-20',
    5.020001 => '2014-09-14',
    5.021004 => '2014-09-20',
    5.021005 => '2014-10-20',
    5.021006 => '2014-11-20',
    5.021007 => '2014-12-20',
    5.021008 => '2015-01-20',
    5.020002 => '2015-02-14',
    5.021009 => '2015-02-21',
    5.021010 => '2015-03-20',
    5.021011 => '2015-04-20',
    5.022000 => '2015-06-01',
    5.023000 => '2015-06-20',
    5.023001 => '2015-07-20',
    5.023002 => '2015-08-20',
    5.020003 => '2015-09-12',
    5.023003 => '2015-09-20',
    5.023004 => '2015-10-20',
    5.023005 => '2015-11-20',
    5.022001 => '2015-12-13',
    5.023006 => '2015-12-21',
    5.023007 => '2016-01-20',
    5.023008 => '2016-02-20',
    5.023009 => '2016-03-20',
    5.022002 => '2016-04-29',
    5.024000 => '2016-05-09',
    5.025000 => '2016-05-09',
    5.025001 => '2016-05-20',
    5.025002 => '2016-06-20',
    5.025003 => '2016-07-20',
    5.025004 => '2016-08-20',
    5.025005 => '2016-09-20',
    5.025006 => '2016-10-20',
    5.025007 => '2016-11-20',
    5.025008 => '2016-12-20',
    5.022003 => '2017-01-14',
    5.024001 => '2017-01-14',
    5.025009 => '2017-01-20',
    5.025010 => '2017-02-20',
    5.025011 => '2017-03-20',
    5.025012 => '2017-04-20',
    5.026000 => '2017-05-30',
    5.027000 => '2017-05-31',
    5.027001 => '2017-06-20',
    5.022004 => '2017-07-15',
    5.024002 => '2017-07-15',
    5.027002 => '2017-07-20',
    5.027003 => '2017-08-21',
    5.027004 => '2017-09-20',
    5.024003 => '2017-09-22',
    5.026001 => '2017-09-22',
    5.027005 => '2017-10-20',
    5.027006 => '2017-11-20',
    5.027007 => '2017-12-20',
    5.027008 => '2018-01-20',
    5.027009 => '2018-02-20',
    5.027010 => '2018-03-20',
    5.024004 => '2018-04-14',
    5.026002 => '2018-04-14',
    5.027011 => '2018-04-20',
    5.028000 => '2018-06-22',
    5.029000 => '2018-06-26',
    5.029001 => '2018-07-20',
    5.029002 => '2018-08-20',
    5.029003 => '2018-09-20',
    5.029004 => '2018-10-20',
    5.029005 => '2018-11-20',
    5.026003 => '2018-11-29',
    5.028001 => '2018-11-29',
    5.029006 => '2018-12-18',
    5.029007 => '2019-01-20',
    5.029008 => '2019-02-20',
    5.029009 => '2019-03-20',
    5.028002 => '2019-04-19',
    5.029010 => '2019-04-20',
    5.030000 => '2019-05-22',
    5.031000 => '2019-05-24',
    5.031001 => '2019-06-20',
    5.031002 => '2019-07-20',
    5.031003 => '2019-08-20',
    5.031004 => '2019-09-20',
    5.031005 => '2019-10-20',
    5.030001 => '2019-11-10',
    5.031006 => '2019-11-20',
    5.031007 => '2019-12-20',
    5.031008 => '2020-01-20',
    5.031009 => '2020-02-20',
    5.030002 => '2020-03-14',
    5.031010 => '2020-03-20',
    5.031011 => '2020-04-28',
    5.028003 => '2020-06-01',
    5.030003 => '2020-06-01',
    5.032000 => '2020-06-20',
    5.033000 => '2020-07-17',
    5.033001 => '2020-08-20',
    5.033002 => '2020-09-20',
    5.033003 => '2020-10-20',
    5.033004 => '2020-11-20',
    5.033005 => '2020-12-20',
    5.033006 => '2021-01-20',
    5.032001 => '2021-01-23',
    5.033007 => '2021-02-20',
    5.033008 => '2021-03-20',
    5.033009 => '2021-04-20',
    5.034000 => '2021-05-20',
    5.035000 => '2021-05-21',
    5.035001 => '2021-06-20',
    5.035002 => '2021-07-23',
    5.035003 => '2021-08-20',
    5.035004 => '2021-09-20',
    5.035005 => '2021-10-21',
    5.035006 => '2021-11-20',
    5.035007 => '2021-12-20',
    5.035008 => '2022-01-20',
    5.035009 => '2022-02-20',
    5.034001 => '2022-03-13',
    5.035010 => '2022-03-20',
    5.035011 => '2022-04-20',
    5.036000 => '2022-05-27',
    5.037000 => '2022-05-27',
    5.037001 => '2022-06-20',
    5.037002 => '2022-07-20',
    5.037003 => '2022-08-20',
    5.037004 => '2022-09-20',
    5.037005 => '2022-10-20',
    5.037006 => '2022-11-20',
    5.037007 => '2022-12-20',
    5.037008 => '2023-01-20',
    5.037009 => '2023-02-20',
    5.037010 => '2023-03-20',
    5.037011 => '2023-04-20',
    5.036001 => '2023-04-23',
    5.038000 => '2023-07-02',
    5.039001 => '2023-07-20',
    5.039002 => '2023-08-20',
    5.039003 => '2023-09-20',
    5.039004 => '2023-10-25',
    5.039005 => '2023-11-20',
    5.034002 => '2023-11-25',
    5.036002 => '2023-11-25',
    5.038001 => '2023-11-25',
    5.034003 => '2023-11-29',
    5.036003 => '2023-11-29',
    5.038002 => '2023-11-29',
    5.039006 => '2023-12-30',
    5.039007 => '2024-01-20',
    5.039008 => '2024-02-23',
    5.039009 => '2024-03-20',
    5.039010 => '2024-04-27',
    5.040000 => '2024-06-09',
    5.041000 => '2024-06-10',
    5.041001 => '2024-07-02',
    5.041002 => '2024-07-20',
    5.041003 => '2024-08-29',
    5.041004 => '2024-09-20',
    5.041005 => '2024-10-20',
    5.041006 => '2024-11-20',
    5.041007 => '2024-12-20',
    5.038003 => '2025-01-18',
    5.040001 => '2025-01-18',
    5.041008 => '2025-01-20',
    5.041009 => '2025-02-24',
    5.041010 => '2025-03-21',
    5.038004 => '2025-04-13',
    5.040002 => '2025-04-13',
    5.041011 => '2025-04-20',
    5.041012 => '2025-04-21',
    5.041013 => '2025-05-28',
    5.042000 => '2025-07-02',
    5.043000 => '2025-07-03',
    5.043001 => '2025-07-20',
    5.038005 => '2025-08-03',
    5.040003 => '2025-08-03',
    5.043002 => '2025-08-20',
    5.043003 => '2025-09-23',
    5.043004 => '2025-10-22',
    5.043005 => '2025-11-20',
    5.043006 => '2025-12-20',
    5.043007 => '2026-01-19',
    5.043008 => '2026-02-20',
  );

for my $version ( sort { $a <=> $b } keys %released ) {
    my $family = int ($version * 1000) / 1000;
    push @{ $families{ $family }} , $version;
}

%delta = (
    5 => {
        changed => {
            'AnyDBM_File'           => undef,
            'AutoLoader'            => undef,
            'AutoSplit'             => undef,
            'Benchmark'             => undef,
            'Carp'                  => undef,
            'Cwd'                   => undef,
            'DB_File'               => undef,
            'DynaLoader'            => undef,
            'English'               => undef,
            'Env'                   => undef,
            'Exporter'              => undef,
            'ExtUtils::MakeMaker'   => undef,
            'Fcntl'                 => undef,
            'File::Basename'        => undef,
            'File::CheckTree'       => undef,
            'File::Find'            => undef,
            'FileHandle'            => undef,
            'GDBM_File'             => undef,
            'Getopt::Long'          => undef,
            'Getopt::Std'           => undef,
            'I18N::Collate'         => undef,
            'IPC::Open2'            => undef,
            'IPC::Open3'            => undef,
            'Math::BigFloat'        => undef,
            'Math::BigInt'          => undef,
            'Math::Complex'         => undef,
            'NDBM_File'             => undef,
            'Net::Ping'             => undef,
            'ODBM_File'             => undef,
            'POSIX'                 => undef,
            'SDBM_File'             => undef,
            'Search::Dict'          => undef,
            'Shell'                 => undef,
            'Socket'                => undef,
            'Sys::Hostname'         => undef,
            'Sys::Syslog'           => undef,
            'Term::Cap'             => undef,
            'Term::Complete'        => undef,
            'Test::Harness'         => undef,
            'Text::Abbrev'          => undef,
            'Text::ParseWords'      => undef,
            'Text::Soundex'         => undef,
            'Text::Tabs'            => undef,
            'TieHash'               => undef,
            'Time::Local'           => undef,
            'integer'               => undef,
            'less'                  => undef,
            'sigtrap'               => undef,
            'strict'                => undef,
            'subs'                  => undef,
        },
        removed => {
        }
    },
    5.001 => {
        delta_from => 5,
        changed => {
            'ExtUtils::Liblist'     => undef,
            'ExtUtils::Manifest'    => undef,
            'ExtUtils::Mkbootstrap' => undef,
            'File::Path'            => undef,
            'SubstrHash'            => undef,
            'lib'                   => undef,
        },
        removed => {
        }
    },
    5.002 => {
        delta_from => 5.001,
        changed => {
            'DB_File'               => '1.01',
            'Devel::SelfStubber'    => '1.01',
            'DirHandle'             => undef,
            'DynaLoader'            => '1.00',
            'ExtUtils::Install'     => undef,
            'ExtUtils::MM_OS2'      => undef,
            'ExtUtils::MM_Unix'     => undef,
            'ExtUtils::MM_VMS'      => undef,
            'ExtUtils::MakeMaker'   => '5.21',
            'ExtUtils::Manifest'    => '1.22',
            'ExtUtils::Mksymlists'  => '1.00',
            'Fcntl'                 => '1.00',
            'File::Copy'            => '1.5',
            'File::Path'            => '1.01',
            'FileCache'             => undef,
            'FileHandle'            => '1.00',
            'GDBM_File'             => '1.00',
            'Getopt::Long'          => '2.01',
            'NDBM_File'             => '1.00',
            'Net::Ping'             => '1',
            'ODBM_File'             => '1.00',
            'POSIX'                 => '1.00',
            'Pod::Functions'        => undef,
            'Pod::Text'             => undef,
            'SDBM_File'             => '1.00',
            'Safe'                  => '1.00',
            'SelectSaver'           => undef,
            'SelfLoader'            => '1.06',
            'Socket'                => '1.5',
            'Symbol'                => undef,
            'Term::ReadLine'        => undef,
            'Test::Harness'         => '1.07',
            'Text::Wrap'            => undef,
            'Tie::Hash'             => undef,
            'Tie::Scalar'           => undef,
            'Tie::SubstrHash'       => undef,
            'diagnostics'           => undef,
            'overload'              => undef,
            'vars'                  => undef,
        },
        removed => {
            'SubstrHash'            => 1,
            'TieHash'               => 1,
        }
    },
    5.00307 => {
        delta_from => 5.002,
        changed => {
            'Config'                => undef,
            'DB_File'               => '1.03',
            'ExtUtils::Embed'       => '1.18',
            'ExtUtils::Install'     => '1.15',
            'ExtUtils::Liblist'     => '1.20',
            'ExtUtils::MM_Unix'     => '1.107',
            'ExtUtils::MakeMaker'   => '5.38',
            'ExtUtils::Manifest'    => '1.27',
            'ExtUtils::Mkbootstrap' => '1.13',
            'ExtUtils::Mksymlists'  => '1.12',
            'ExtUtils::testlib'     => '1.11',
            'Fatal'                 => undef,
            'File::Basename'        => '2.4',
            'FindBin'               => '1.04',
            'Getopt::Long'          => '2.04',
            'IO'                    => undef,
            'IO::File'              => '1.05',
            'IO::Handle'            => '1.12',
            'IO::Pipe'              => '1.07',
            'IO::Seekable'          => '1.05',
            'IO::Select'            => '1.09',
            'IO::Socket'            => '1.13',
            'Net::Ping'             => '1.01',
            'OS2::ExtAttr'          => '0.01',
            'OS2::PrfDB'            => '0.02',
            'OS2::Process'          => undef,
            'OS2::REXX'             => undef,
            'Opcode'                => '1.01',
            'Safe'                  => '2.06',
            'Test::Harness'         => '1.13',
            'Text::Tabs'            => '96.051501',
            'Text::Wrap'            => '96.041801',
            'UNIVERSAL'             => undef,
            'VMS::Filespec'         => undef,
            'VMS::Stdio'            => '2.0',
            'ops'                   => undef,
            'sigtrap'               => '1.01',
        },
        removed => {
        }
    },
    5.004 => {
        delta_from => 5.00307,
        changed => {
            'Bundle::CPAN'          => '0.02',
            'CGI'                   => '2.36',
            'CGI::Apache'           => '1.01',
            'CGI::Carp'             => '1.06',
            'CGI::Fast'             => '1.00a',
            'CGI::Push'             => '1.00',
            'CGI::Switch'           => '0.05',
            'CPAN'                  => '1.2401',
            'CPAN::FirstTime'       => '1.18',
            'CPAN::Nox'             => undef,
            'Class::Struct'         => undef,
            'Cwd'                   => '2.00',
            'DB_File'               => '1.14',
            'DynaLoader'            => '1.02',
            'ExtUtils::Command'     => '1.00',
            'ExtUtils::Embed'       => '1.2501',
            'ExtUtils::Install'     => '1.16',
            'ExtUtils::Liblist'     => '1.2201',
            'ExtUtils::MM_Unix'     => '1.114',
            'ExtUtils::MM_Win32'    => undef,
            'ExtUtils::MakeMaker'   => '5.4002',
            'ExtUtils::Manifest'    => '1.33',
            'ExtUtils::Mksymlists'  => '1.13',
            'ExtUtils::XSSymSet'    => '1.0',
            'Fcntl'                 => '1.03',
            'File::Basename'        => '2.5',
            'File::Compare'         => '1.1001',
            'File::Copy'            => '2.02',
            'File::Path'            => '1.04',
            'File::stat'            => undef,
            'FileHandle'            => '2.00',
            'Getopt::Long'          => '2.10',
            'IO::File'              => '1.0602',
            'IO::Handle'            => '1.1504',
            'IO::Pipe'              => '1.0901',
            'IO::Seekable'          => '1.06',
            'IO::Select'            => '1.10',
            'IO::Socket'            => '1.1602',
            'IPC::Open2'            => '1.01',
            'IPC::Open3'            => '1.0101',
            'Math::Complex'         => '1.01',
            'Math::Trig'            => '1',
            'Net::Ping'             => '2.02',
            'Net::hostent'          => undef,
            'Net::netent'           => undef,
            'Net::protoent'         => undef,
            'Net::servent'          => undef,
            'Opcode'                => '1.04',
            'POSIX'                 => '1.02',
            'Pod::Html'             => undef,
            'Pod::Text'             => '1.0203',
            'SelfLoader'            => '1.07',
            'Socket'                => '1.6',
            'Symbol'                => '1.02',
            'Test::Harness'         => '1.1502',
            'Text::Tabs'            => '96.121201',
            'Text::Wrap'            => '97.011701',
            'Tie::RefHash'          => undef,
            'Time::gmtime'          => '1.01',
            'Time::localtime'       => '1.01',
            'Time::tm'              => undef,
            'User::grent'           => undef,
            'User::pwent'           => undef,
            'VMS::DCLsym'           => '1.01',
            'VMS::Stdio'            => '2.02',
            'autouse'               => '1.01',
            'blib'                  => undef,
            'constant'              => '1.00',
            'locale'                => undef,
            'sigtrap'               => '1.02',
            'vmsish'                => undef,
        },
        removed => {
            'Fatal'                 => 1,
        }
    },
    5.00405 => {
        delta_from => 5.004,
        changed => {
            'AutoLoader'            => '5.56',
            'AutoSplit'             => '1.0303',
            'Bundle::CPAN'          => '0.03',
            'CGI'                   => '2.42',
            'CGI::Apache'           => '1.1',
            'CGI::Carp'             => '1.10',
            'CGI::Cookie'           => '1.06',
            'CGI::Push'             => '1.01',
            'CGI::Switch'           => '0.06',
            'CPAN'                  => '1.40',
            'CPAN::FirstTime'       => '1.30',
            'Cwd'                   => '2.01',
            'DB_File'               => '1.15',
            'DynaLoader'            => '1.03',
            'ExtUtils::Command'     => '1.01',
            'ExtUtils::Embed'       => '1.2505',
            'ExtUtils::Install'     => '1.28',
            'ExtUtils::Liblist'     => '1.25',
            'ExtUtils::MM_Unix'     => '1.118',
            'ExtUtils::MakeMaker'   => '5.42',
            'ExtUtils::Mkbootstrap' => '1.14',
            'ExtUtils::Mksymlists'  => '1.16',
            'File::Basename'        => '2.6',
            'File::DosGlob'         => undef,
            'File::Path'            => '1.0402',
            'File::Spec'            => '0.6',
            'File::Spec::Mac'       => '1.0',
            'File::Spec::OS2'       => undef,
            'File::Spec::Unix'      => undef,
            'File::Spec::VMS'       => undef,
            'File::Spec::Win32'     => undef,
            'FindBin'               => '1.41',
            'Getopt::Long'          => '2.19',
            'IO::File'              => '1.06021',
            'IO::Socket'            => '1.1603',
            'IPC::Open3'            => '1.0103',
            'Math::Complex'         => '1.25',
            'NDBM_File'             => '1.01',
            'Pod::Html'             => '1.0101',
            'Pod::Text'             => '1.0204',
            'SelfLoader'            => '1.08',
            'Socket'                => '1.7',
            'Test'                  => '1.04',
            'Test::Harness'         => '1.1602',
            'Text::ParseWords'      => '3.1001',
            'Text::Wrap'            => '98.112902',
            'Tie::Handle'           => undef,
            'attrs'                 => '0.1',
            'base'                  => undef,
            'blib'                  => '1.00',
            're'                    => undef,
            'strict'                => '1.01',
        },
        removed => {
        }
    },
    5.005 => {
        delta_from => 5.00405,
        changed => {
            'AutoLoader'            => undef,
            'AutoSplit'             => '1.0302',
            'B'                     => undef,
            'B::Asmdata'            => undef,
            'B::Assembler'          => undef,
            'B::Bblock'             => undef,
            'B::Bytecode'           => undef,
            'B::C'                  => undef,
            'B::CC'                 => undef,
            'B::Debug'              => undef,
            'B::Deparse'            => '0.56',
            'B::Disassembler'       => undef,
            'B::Lint'               => undef,
            'B::Showlex'            => undef,
            'B::Stackobj'           => undef,
            'B::Terse'              => undef,
            'B::Xref'               => undef,
            'CGI::Carp'             => '1.101',
            'CPAN'                  => '1.3901',
            'CPAN::FirstTime'       => '1.29',
            'DB_File'               => '1.60',
            'Data::Dumper'          => '2.09',
            'Errno'                 => '1.09',
            'ExtUtils::Installed'   => '0.02',
            'ExtUtils::MM_Unix'     => '1.12601',
            'ExtUtils::MakeMaker'   => '5.4301',
            'ExtUtils::Mkbootstrap' => '1.13',
            'ExtUtils::Mksymlists'  => '1.17',
            'ExtUtils::Packlist'    => '0.03',
            'Fatal'                 => '1.02',
            'File::Path'            => '1.0401',
            'Getopt::Long'          => '2.17',
            'IO::Handle'            => '1.1505',
            'IPC::Msg'              => '1.00',
            'IPC::Open3'            => '1.0102',
            'IPC::Semaphore'        => '1.00',
            'IPC::SysV'             => '1.03',
            'O'                     => undef,
            'OS2::Process'          => '0.2',
            'Pod::Html'             => '1.01',
            'Pod::Text'             => '1.0203',
            'Text::ParseWords'      => '3.1',
            'Text::Wrap'            => '97.02',
            'Thread'                => '1.0',
            'Thread::Queue'         => undef,
            'Thread::Semaphore'     => undef,
            'Thread::Signal'        => undef,
            'Thread::Specific'      => undef,
            'Tie::Array'            => '1.00',
            'VMS::Stdio'            => '2.1',
            'attrs'                 => '1.0',
            'fields'                => '0.02',
            're'                    => '0.02',
        },
        removed => {
            'Bundle::CPAN'          => 1,
        }
    },
    5.00503 => {
        delta_from => 5.005,
        changed => {
            'AutoSplit'             => '1.0303',
            'CGI'                   => '2.46',
            'CGI::Carp'             => '1.13',
            'CGI::Fast'             => '1.01',
            'CPAN'                  => '1.48',
            'CPAN::FirstTime'       => '1.36',
            'CPAN::Nox'             => '1.00',
            'DB_File'               => '1.65',
            'Data::Dumper'          => '2.101',
            'Dumpvalue'             => undef,
            'Errno'                 => '1.111',
            'ExtUtils::Install'     => '1.28',
            'ExtUtils::Liblist'     => '1.25',
            'ExtUtils::MM_Unix'     => '1.12602',
            'ExtUtils::MakeMaker'   => '5.4302',
            'ExtUtils::Manifest'    => '1.33',
            'ExtUtils::Mkbootstrap' => '1.14',
            'ExtUtils::Mksymlists'  => '1.17',
            'ExtUtils::testlib'     => '1.11',
            'FindBin'               => '1.42',
            'Getopt::Long'          => '2.19',
            'Getopt::Std'           => '1.01',
            'IO::Pipe'              => '1.0902',
            'IPC::Open3'            => '1.0103',
            'Math::Complex'         => '1.26',
            'Test'                  => '1.122',
            'Text::Wrap'            => '98.112902',
        },
        removed => {
        }
    },
    5.00504 => {
        delta_from => 5.00503,
        changed => {
            'CPAN::FirstTime'       => '1.36',
            'DB_File'               => '1.807',
            'ExtUtils::Install'     => '1.28',
            'ExtUtils::Liblist'     => '1.25',
            'ExtUtils::MM_Unix'     => '1.12602',
            'ExtUtils::Manifest'    => '1.33',
            'ExtUtils::Miniperl'    => undef,
            'ExtUtils::Mkbootstrap' => '1.14',
            'ExtUtils::Mksymlists'  => '1.17',
            'ExtUtils::testlib'     => '1.11',
            'File::Compare'         => '1.1002',
            'File::Spec'            => '0.8',
            'File::Spec::Functions' => undef,
            'File::Spec::Mac'       => undef,
            'Getopt::Long'          => '2.20',
            'Pod::Html'             => '1.02',
        },
        removed => {
        }
    },
    5.006 => {
        delta_from => 5.00504,
        changed => {
            'AutoLoader'            => '5.57',
            'AutoSplit'             => '1.0305',
            'B::Deparse'            => '0.59',
            'B::Stash'              => undef,
            'Benchmark'             => '1',
            'ByteLoader'            => '0.03',
            'CGI'                   => '2.56',
            'CGI::Apache'           => undef,
            'CGI::Carp'             => '1.14',
            'CGI::Cookie'           => '1.12',
            'CGI::Fast'             => '1.02',
            'CGI::Pretty'           => '1.03',
            'CGI::Switch'           => undef,
            'CPAN'                  => '1.52',
            'CPAN::FirstTime'       => '1.38',
            'Carp::Heavy'           => undef,
            'Class::Struct'         => '0.58',
            'Cwd'                   => '2.02',
            'DB'                    => '1.0',
            'DB_File'               => '1.72',
            'Devel::DProf'          => '20000000.00_00',
            'Devel::Peek'           => '1.00_01',
            'DynaLoader'            => '1.04',
            'Exporter'              => '5.562',
            'Exporter::Heavy'       => undef,
            'ExtUtils::MM_Cygwin'   => undef,
            'ExtUtils::MM_Unix'     => '1.12603',
            'ExtUtils::MakeMaker'   => '5.45',
            'File::Copy'            => '2.03',
            'File::Glob'            => '0.991',
            'File::Path'            => '1.0403',
            'GDBM_File'             => '1.03',
            'Getopt::Long'          => '2.23',
            'Getopt::Std'           => '1.02',
            'IO'                    => '1.20',
            'IO::Dir'               => '1.03',
            'IO::File'              => '1.08',
            'IO::Handle'            => '1.21',
            'IO::Pipe'              => '1.121',
            'IO::Poll'              => '0.01',
            'IO::Seekable'          => '1.08',
            'IO::Select'            => '1.14',
            'IO::Socket'            => '1.26',
            'IO::Socket::INET'      => '1.25',
            'IO::Socket::UNIX'      => '1.20',
            'JNI'                   => '0.01',
            'JPL::AutoLoader'       => undef,
            'JPL::Class'            => undef,
            'JPL::Compile'          => undef,
            'NDBM_File'             => '1.03',
            'ODBM_File'             => '1.02',
            'OS2::DLL'              => undef,
            'POSIX'                 => '1.03',
            'Pod::Checker'          => '1.098',
            'Pod::Find'             => '0.12',
            'Pod::Html'             => '1.03',
            'Pod::InputObjects'     => '1.12',
            'Pod::Man'              => '1.02',
            'Pod::ParseUtils'       => '0.2',
            'Pod::Parser'           => '1.12',
            'Pod::Plainer'          => '0.01',
            'Pod::Select'           => '1.12',
            'Pod::Text'             => '2.03',
            'Pod::Text::Color'      => '0.05',
            'Pod::Text::Termcap'    => '0.04',
            'Pod::Usage'            => '1.12',
            'SDBM_File'             => '1.02',
            'SelfLoader'            => '1.0901',
            'Shell'                 => '0.2',
            'Socket'                => '1.72',
            'Sys::Hostname'         => '1.1',
            'Sys::Syslog'           => '0.01',
            'Term::ANSIColor'       => '1.01',
            'Test'                  => '1.13',
            'Test::Harness'         => '1.1604',
            'Text::ParseWords'      => '3.2',
            'Text::Soundex'         => '1.0',
            'Text::Tabs'            => '98.112801',
            'Tie::Array'            => '1.01',
            'Tie::Handle'           => '1.0',
            'VMS::Stdio'            => '2.2',
            'XSLoader'              => '0.01',
            'attributes'            => '0.03',
            'autouse'               => '1.02',
            'base'                  => '1.01',
            'bytes'                 => undef,
            'charnames'             => undef,
            'constant'              => '1.02',
            'diagnostics'           => '1.0',
            'fields'                => '1.01',
            'filetest'              => undef,
            'lib'                   => '0.5564',
            'open'                  => undef,
            'utf8'                  => undef,
            'warnings'              => undef,
            'warnings::register'    => undef,
        },
        removed => {
        }
    },
    5.006001 => {
        delta_from => 5.006,
        changed => {
            'AutoLoader'            => '5.58',
            'B::Assembler'          => '0.02',
            'B::Concise'            => '0.51',
            'B::Deparse'            => '0.6',
            'ByteLoader'            => '0.04',
            'CGI'                   => '2.752',
            'CGI::Carp'             => '1.20',
            'CGI::Cookie'           => '1.18',
            'CGI::Pretty'           => '1.05',
            'CGI::Push'             => '1.04',
            'CGI::Util'             => '1.1',
            'CPAN'                  => '1.59_54',
            'CPAN::FirstTime'       => '1.53',
            'Class::Struct'         => '0.59',
            'Cwd'                   => '2.04',
            'DB_File'               => '1.75',
            'Data::Dumper'          => '2.102',
            'ExtUtils::Install'     => '1.28',
            'ExtUtils::Liblist'     => '1.26',
            'ExtUtils::MM_Unix'     => '1.12603',
            'ExtUtils::Manifest'    => '1.33',
            'ExtUtils::Mkbootstrap' => '1.14',
            'ExtUtils::Mksymlists'  => '1.17',
            'ExtUtils::testlib'     => '1.11',
            'File::Path'            => '1.0404',
            'File::Spec'            => '0.82',
            'File::Spec::Epoc'      => undef,
            'File::Spec::Functions' => '1.1',
            'File::Spec::Mac'       => '1.2',
            'File::Spec::OS2'       => '1.1',
            'File::Spec::Unix'      => '1.2',
            'File::Spec::VMS'       => '1.1',
            'File::Spec::Win32'     => '1.2',
            'File::Temp'            => '0.12',
            'GDBM_File'             => '1.05',
            'Getopt::Long'          => '2.25',
            'IO::Poll'              => '0.05',
            'JNI'                   => '0.1',
            'Math::BigFloat'        => '0.02',
            'Math::BigInt'          => '0.01',
            'Math::Complex'         => '1.31',
            'NDBM_File'             => '1.04',
            'ODBM_File'             => '1.03',
            'OS2::REXX'             => '1.00',
            'Pod::Checker'          => '1.2',
            'Pod::Find'             => '0.21',
            'Pod::InputObjects'     => '1.13',
            'Pod::LaTeX'            => '0.53',
            'Pod::Man'              => '1.15',
            'Pod::ParseUtils'       => '0.22',
            'Pod::Parser'           => '1.13',
            'Pod::Select'           => '1.13',
            'Pod::Text'             => '2.08',
            'Pod::Text::Color'      => '0.06',
            'Pod::Text::Overstrike' => '1.01',
            'Pod::Text::Termcap'    => '1',
            'Pod::Usage'            => '1.14',
            'SDBM_File'             => '1.03',
            'SelfLoader'            => '1.0902',
            'Shell'                 => '0.3',
            'Term::ANSIColor'       => '1.03',
            'Test'                  => '1.15',
            'Text::Wrap'            => '2001.0131',
            'Tie::Handle'           => '4.0',
            'Tie::RefHash'          => '1.3',
        },
        removed => {
        }
    },
    5.006002 => {
        delta_from => 5.006001,
        changed => {
            'CPAN::FirstTime'       => '1.53',
            'DB_File'               => '1.806',
            'Data::Dumper'          => '2.121',
            'ExtUtils::Command'     => '1.05',
            'ExtUtils::Command::MM' => '0.03',
            'ExtUtils::Install'     => '1.32',
            'ExtUtils::Installed'   => '0.08',
            'ExtUtils::Liblist'     => '1.01',
            'ExtUtils::Liblist::Kid'=> '1.3',
            'ExtUtils::MM'          => '0.04',
            'ExtUtils::MM_Any'      => '0.07',
            'ExtUtils::MM_BeOS'     => '1.04',
            'ExtUtils::MM_Cygwin'   => '1.06',
            'ExtUtils::MM_DOS'      => '0.02',
            'ExtUtils::MM_MacOS'    => '1.07',
            'ExtUtils::MM_NW5'      => '2.06',
            'ExtUtils::MM_OS2'      => '1.04',
            'ExtUtils::MM_UWIN'     => '0.02',
            'ExtUtils::MM_Unix'     => '1.42',
            'ExtUtils::MM_VMS'      => '5.70',
            'ExtUtils::MM_Win32'    => '1.09',
            'ExtUtils::MM_Win95'    => '0.03',
            'ExtUtils::MY'          => '0.01',
            'ExtUtils::MakeMaker'   => '6.17',
            'ExtUtils::MakeMaker::bytes'=> '0.01',
            'ExtUtils::MakeMaker::vmsish'=> '0.01',
            'ExtUtils::Manifest'    => '1.42',
            'ExtUtils::Mkbootstrap' => '1.15',
            'ExtUtils::Mksymlists'  => '1.19',
            'ExtUtils::Packlist'    => '0.04',
            'ExtUtils::testlib'     => '1.15',
            'File::Spec'            => '0.86',
            'File::Spec::Cygwin'    => '1.1',
            'File::Spec::Epoc'      => '1.1',
            'File::Spec::Functions' => '1.3',
            'File::Spec::Mac'       => '1.4',
            'File::Spec::OS2'       => '1.2',
            'File::Spec::Unix'      => '1.5',
            'File::Spec::VMS'       => '1.4',
            'File::Spec::Win32'     => '1.4',
            'File::Temp'            => '0.14',
            'Safe'                  => '2.10',
            'Test'                  => '1.24',
            'Test::Builder'         => '0.17',
            'Test::Harness'         => '2.30',
            'Test::Harness::Assert' => '0.01',
            'Test::Harness::Iterator'=> '0.01',
            'Test::Harness::Straps' => '0.15',
            'Test::More'            => '0.47',
            'Test::Simple'          => '0.47',
            'Unicode'               => '3.0.1',
            'if'                    => '0.03',
            'ops'                   => '1.00',
        },
        removed => {
        }
    },
    5.007003 => {
        delta_from => 5.006001,
        changed => {
            'AnyDBM_File'           => '1.00',
            'Attribute::Handlers'   => '0.76',
            'AutoLoader'            => '5.59',
            'AutoSplit'             => '1.0307',
            'B'                     => '1.00',
            'B::Asmdata'            => '1.00',
            'B::Assembler'          => '0.04',
            'B::Bblock'             => '1.00',
            'B::Bytecode'           => '1.00',
            'B::C'                  => '1.01',
            'B::CC'                 => '1.00',
            'B::Concise'            => '0.52',
            'B::Debug'              => '1.00',
            'B::Deparse'            => '0.63',
            'B::Disassembler'       => '1.01',
            'B::Lint'               => '1.00',
            'B::Showlex'            => '1.00',
            'B::Stackobj'           => '1.00',
            'B::Stash'              => '1.00',
            'B::Terse'              => '1.00',
            'B::Xref'               => '1.00',
            'Benchmark'             => '1.04',
            'CGI'                   => '2.80',
            'CGI::Apache'           => '1.00',
            'CGI::Carp'             => '1.22',
            'CGI::Cookie'           => '1.20',
            'CGI::Fast'             => '1.04',
            'CGI::Pretty'           => '1.05_00',
            'CGI::Switch'           => '1.00',
            'CGI::Util'             => '1.3',
            'CPAN'                  => '1.59_56',
            'CPAN::FirstTime'       => '1.54',
            'CPAN::Nox'             => '1.00_01',
            'Carp'                  => '1.01',
            'Carp::Heavy'           => '1.01',
            'Class::ISA'            => '0.32',
            'Class::Struct'         => '0.61',
            'Cwd'                   => '2.06',
            'DB_File'               => '1.804',
            'Data::Dumper'          => '2.12',
            'Devel::DProf'          => '20000000.00_01',
            'Devel::PPPort'         => '2.0002',
            'Devel::Peek'           => '1.00_03',
            'Devel::SelfStubber'    => '1.03',
            'Digest'                => '1.00',
            'Digest::MD5'           => '2.16',
            'DirHandle'             => '1.00',
            'Dumpvalue'             => '1.10',
            'Encode'                => '0.40',
            'Encode::CN'            => '0.02',
            'Encode::CN::HZ'        => undef,
            'Encode::Encoding'      => '0.02',
            'Encode::Internal'      => '0.30',
            'Encode::JP'            => '0.02',
            'Encode::JP::Constants' => '1.02',
            'Encode::JP::H2Z'       => '0.77',
            'Encode::JP::ISO_2022_JP'=> undef,
            'Encode::JP::JIS'       => undef,
            'Encode::JP::Tr'        => '0.77',
            'Encode::KR'            => '0.02',
            'Encode::TW'            => '0.02',
            'Encode::Tcl'           => '1.01',
            'Encode::Tcl::Escape'   => '1.01',
            'Encode::Tcl::Extended' => '1.01',
            'Encode::Tcl::HanZi'    => '1.01',
            'Encode::Tcl::Table'    => '1.01',
            'Encode::Unicode'       => '0.30',
            'Encode::XS'            => '0.40',
            'Encode::iso10646_1'    => '0.30',
            'Encode::usc2_le'       => '0.30',
            'Encode::utf8'          => '0.30',
            'English'               => '1.00',
            'Env'                   => '1.00',
            'Exporter'              => '5.566',
            'Exporter::Heavy'       => '5.562',
            'ExtUtils::Command'     => '1.02',
            'ExtUtils::Constant'    => '0.11',
            'ExtUtils::Embed'       => '1.250601',
            'ExtUtils::Install'     => '1.29',
            'ExtUtils::Installed'   => '0.04',
            'ExtUtils::Liblist'     => '1.2701',
            'ExtUtils::MM_BeOS'     => '1.00',
            'ExtUtils::MM_Cygwin'   => '1.00',
            'ExtUtils::MM_OS2'      => '1.00',
            'ExtUtils::MM_Unix'     => '1.12607',
            'ExtUtils::MM_VMS'      => '5.56',
            'ExtUtils::MM_Win32'    => '1.00_02',
            'ExtUtils::MakeMaker'   => '5.48_03',
            'ExtUtils::Manifest'    => '1.35',
            'ExtUtils::Mkbootstrap' => '1.1401',
            'ExtUtils::Mksymlists'  => '1.18',
            'ExtUtils::Packlist'    => '0.04',
            'ExtUtils::testlib'     => '1.1201',
            'Fatal'                 => '1.03',
            'Fcntl'                 => '1.04',
            'File::Basename'        => '2.71',
            'File::CheckTree'       => '4.1',
            'File::Compare'         => '1.1003',
            'File::Copy'            => '2.05',
            'File::DosGlob'         => '1.00',
            'File::Find'            => '1.04',
            'File::Glob'            => '1.01',
            'File::Path'            => '1.05',
            'File::Spec'            => '0.83',
            'File::Spec::Cygwin'    => '1.0',
            'File::Spec::Epoc'      => '1.00',
            'File::Spec::Functions' => '1.2',
            'File::Spec::Mac'       => '1.3',
            'File::Spec::Unix'      => '1.4',
            'File::Spec::VMS'       => '1.2',
            'File::Spec::Win32'     => '1.3',
            'File::Temp'            => '0.13',
            'File::stat'            => '1.00',
            'FileCache'             => '1.00',
            'FileHandle'            => '2.01',
            'Filter::Simple'        => '0.77',
            'Filter::Util::Call'    => '1.06',
            'FindBin'               => '1.43',
            'GDBM_File'             => '1.06',
            'Getopt::Long'          => '2.28',
            'Getopt::Std'           => '1.03',
            'I18N::Collate'         => '1.00',
            'I18N::LangTags'        => '0.27',
            'I18N::LangTags::List'  => '0.25',
            'I18N::Langinfo'        => '0.01',
            'IO::Dir'               => '1.03_00',
            'IO::File'              => '1.09',
            'IO::Handle'            => '1.21_00',
            'IO::Pipe'              => '1.122',
            'IO::Poll'              => '0.06',
            'IO::Seekable'          => '1.08_00',
            'IO::Select'            => '1.15',
            'IO::Socket'            => '1.27',
            'IO::Socket::INET'      => '1.26',
            'IO::Socket::UNIX'      => '1.20_00',
            'IPC::Msg'              => '1.00_00',
            'IPC::Open3'            => '1.0104',
            'IPC::Semaphore'        => '1.00_00',
            'IPC::SysV'             => '1.03_00',
            'List::Util'            => '1.06_00',
            'Locale::Constants'     => '2.01',
            'Locale::Country'       => '2.01',
            'Locale::Currency'      => '2.01',
            'Locale::Language'      => '2.01',
            'Locale::Maketext'      => '1.03',
            'Locale::Script'        => '2.01',
            'MIME::Base64'          => '2.12',
            'MIME::QuotedPrint'     => '2.03',
            'Math::BigFloat'        => '1.30',
            'Math::BigInt'          => '1.54',
            'Math::BigInt::Calc'    => '0.25',
            'Math::Complex'         => '1.34',
            'Math::Trig'            => '1.01',
            'Memoize'               => '0.66',
            'Memoize::AnyDBM_File'  => '0.65',
            'Memoize::Expire'       => '0.66',
            'Memoize::ExpireFile'   => '0.65',
            'Memoize::ExpireTest'   => '0.65',
            'Memoize::NDBM_File'    => '0.65',
            'Memoize::SDBM_File'    => '0.65',
            'Memoize::Storable'     => '0.65',
            'NEXT'                  => '0.50',
            'Net::Cmd'              => '2.21',
            'Net::Config'           => '1.10',
            'Net::Domain'           => '2.17',
            'Net::FTP'              => '2.64',
            'Net::FTP::A'           => '1.15',
            'Net::FTP::E'           => '0.01',
            'Net::FTP::I'           => '1.12',
            'Net::FTP::L'           => '0.01',
            'Net::FTP::dataconn'    => '0.10',
            'Net::NNTP'             => '2.21',
            'Net::Netrc'            => '2.12',
            'Net::POP3'             => '2.23',
            'Net::Ping'             => '2.12',
            'Net::SMTP'             => '2.21',
            'Net::Time'             => '2.09',
            'Net::hostent'          => '1.00',
            'Net::netent'           => '1.00',
            'Net::protoent'         => '1.00',
            'Net::servent'          => '1.00',
            'O'                     => '1.00',
            'OS2::DLL'              => '1.00',
            'OS2::Process'          => '1.0',
            'OS2::REXX'             => '1.01',
            'Opcode'                => '1.05',
            'POSIX'                 => '1.05',
            'PerlIO'                => '1.00',
            'PerlIO::Scalar'        => '0.01',
            'PerlIO::Via'           => '0.01',
            'Pod::Checker'          => '1.3',
            'Pod::Find'             => '0.22',
            'Pod::Functions'        => '1.01',
            'Pod::Html'             => '1.04',
            'Pod::LaTeX'            => '0.54',
            'Pod::Man'              => '1.32',
            'Pod::ParseLink'        => '1.05',
            'Pod::Text'             => '2.18',
            'Pod::Text::Color'      => '1.03',
            'Pod::Text::Overstrike' => '1.08',
            'Pod::Text::Termcap'    => '1.09',
            'Safe'                  => '2.07',
            'Scalar::Util'          => '1.06_00',
            'Search::Dict'          => '1.02',
            'SelectSaver'           => '1.00',
            'SelfLoader'            => '1.0903',
            'Shell'                 => '0.4',
            'Socket'                => '1.75',
            'Storable'              => '1.015',
            'Switch'                => '2.06',
            'Symbol'                => '1.04',
            'Sys::Syslog'           => '0.02',
            'Term::ANSIColor'       => '1.04',
            'Term::Cap'             => '1.07',
            'Term::Complete'        => '1.4',
            'Term::ReadLine'        => '1.00',
            'Test'                  => '1.18',
            'Test::Builder'         => '0.11',
            'Test::Harness'         => '2.01',
            'Test::Harness::Assert' => '0.01',
            'Test::Harness::Iterator'=> '0.01',
            'Test::Harness::Straps' => '0.08',
            'Test::More'            => '0.41',
            'Test::Simple'          => '0.41',
            'Text::Abbrev'          => '1.00',
            'Text::Balanced'        => '1.89',
            'Text::ParseWords'      => '3.21',
            'Text::Soundex'         => '1.01',
            'Text::Wrap'            => '2001.0929',
            'Thread'                => '2.00',
            'Thread::Queue'         => '1.00',
            'Thread::Semaphore'     => '1.00',
            'Thread::Signal'        => '1.00',
            'Thread::Specific'      => '1.00',
            'Tie::Array'            => '1.02',
            'Tie::File'             => '0.17',
            'Tie::Handle'           => '4.1',
            'Tie::Hash'             => '1.00',
            'Tie::Memoize'          => '1.0',
            'Tie::RefHash'          => '1.3_00',
            'Tie::Scalar'           => '1.00',
            'Tie::SubstrHash'       => '1.00',
            'Time::HiRes'           => '1.20_00',
            'Time::Local'           => '1.04',
            'Time::gmtime'          => '1.02',
            'Time::localtime'       => '1.02',
            'Time::tm'              => '1.00',
            'UNIVERSAL'             => '1.00',
            'Unicode::Collate'      => '0.10',
            'Unicode::Normalize'    => '0.14',
            'Unicode::UCD'          => '0.2',
            'User::grent'           => '1.00',
            'User::pwent'           => '1.00',
            'VMS::DCLsym'           => '1.02',
            'VMS::Filespec'         => '1.1',
            'VMS::Stdio'            => '2.3',
            'XS::Typemap'           => '0.01',
            'attributes'            => '0.04_01',
            'attrs'                 => '1.01',
            'autouse'               => '1.03',
            'base'                  => '1.02',
            'blib'                  => '1.01',
            'bytes'                 => '1.00',
            'charnames'             => '1.01',
            'constant'              => '1.04',
            'diagnostics'           => '1.1',
            'encoding'              => '1.00',
            'fields'                => '1.02',
            'filetest'              => '1.00',
            'if'                    => '0.01',
            'integer'               => '1.00',
            'less'                  => '0.01',
            'locale'                => '1.00',
            'open'                  => '1.01',
            'ops'                   => '1.00',
            'overload'              => '1.00',
            're'                    => '0.03',
            'sort'                  => '1.00',
            'strict'                => '1.02',
            'subs'                  => '1.00',
            'threads'               => '0.05',
            'threads::shared'       => '0.90',
            'utf8'                  => '1.00',
            'vars'                  => '1.01',
            'vmsish'                => '1.00',
            'warnings'              => '1.00',
            'warnings::register'    => '1.00',
        },
        removed => {
        }
    },
    5.008 => {
        delta_from => 5.007003,
        changed => {
            'Attribute::Handlers'   => '0.77',
            'B'                     => '1.01',
            'B::Lint'               => '1.01',
            'B::Xref'               => '1.01',
            'CGI'                   => '2.81',
            'CGI::Carp'             => '1.23',
            'CPAN'                  => '1.61',
            'CPAN::FirstTime'       => '1.56',
            'CPAN::Nox'             => '1.02',
            'Digest::MD5'           => '2.20',
            'Dumpvalue'             => '1.11',
            'Encode'                => '1.75',
            'Encode::Alias'         => '1.32',
            'Encode::Byte'          => '1.22',
            'Encode::CJKConstants'  => '1.00',
            'Encode::CN'            => '1.24',
            'Encode::CN::HZ'        => '1.04',
            'Encode::Config'        => '1.06',
            'Encode::EBCDIC'        => '1.21',
            'Encode::Encoder'       => '0.05',
            'Encode::Encoding'      => '1.30',
            'Encode::Guess'         => '1.06',
            'Encode::JP'            => '1.25',
            'Encode::JP::H2Z'       => '1.02',
            'Encode::JP::JIS7'      => '1.08',
            'Encode::KR'            => '1.22',
            'Encode::KR::2022_KR'   => '1.05',
            'Encode::MIME::Header'  => '1.05',
            'Encode::Symbol'        => '1.22',
            'Encode::TW'            => '1.26',
            'Encode::Unicode'       => '1.37',
            'Exporter::Heavy'       => '5.566',
            'ExtUtils::Command'     => '1.04',
            'ExtUtils::Command::MM' => '0.01',
            'ExtUtils::Constant'    => '0.12',
            'ExtUtils::Installed'   => '0.06',
            'ExtUtils::Liblist'     => '1.00',
            'ExtUtils::Liblist::Kid'=> '1.29',
            'ExtUtils::MM'          => '0.04',
            'ExtUtils::MM_Any'      => '0.04',
            'ExtUtils::MM_BeOS'     => '1.03',
            'ExtUtils::MM_Cygwin'   => '1.04',
            'ExtUtils::MM_DOS'      => '0.01',
            'ExtUtils::MM_MacOS'    => '1.03',
            'ExtUtils::MM_NW5'      => '2.05',
            'ExtUtils::MM_OS2'      => '1.03',
            'ExtUtils::MM_UWIN'     => '0.01',
            'ExtUtils::MM_Unix'     => '1.33',
            'ExtUtils::MM_VMS'      => '5.65',
            'ExtUtils::MM_Win32'    => '1.05',
            'ExtUtils::MM_Win95'    => '0.02',
            'ExtUtils::MY'          => '0.01',
            'ExtUtils::MakeMaker'   => '6.03',
            'ExtUtils::Manifest'    => '1.38',
            'ExtUtils::Mkbootstrap' => '1.15',
            'ExtUtils::Mksymlists'  => '1.19',
            'ExtUtils::testlib'     => '1.15',
            'File::CheckTree'       => '4.2',
            'FileCache'             => '1.021',
            'Filter::Simple'        => '0.78',
            'Getopt::Long'          => '2.32',
            'Hash::Util'            => '0.04',
            'List::Util'            => '1.07_00',
            'Locale::Country'       => '2.04',
            'Math::BigFloat'        => '1.35',
            'Math::BigFloat::Trace' => '0.01',
            'Math::BigInt'          => '1.60',
            'Math::BigInt::Calc'    => '0.30',
            'Math::BigInt::Trace'   => '0.01',
            'Math::BigRat'          => '0.07',
            'Memoize'               => '1.01',
            'Memoize::Expire'       => '1.00',
            'Memoize::ExpireFile'   => '1.01',
            'Net::FTP'              => '2.65',
            'Net::FTP::dataconn'    => '0.11',
            'Net::Ping'             => '2.19',
            'Net::SMTP'             => '2.24',
            'PerlIO'                => '1.01',
            'PerlIO::encoding'      => '0.06',
            'PerlIO::scalar'        => '0.01',
            'PerlIO::via'           => '0.01',
            'PerlIO::via::QuotedPrint'=> '0.04',
            'Pod::Man'              => '1.33',
            'Pod::Text'             => '2.19',
            'Scalar::Util'          => '1.07_00',
            'Storable'              => '2.04',
            'Switch'                => '2.09',
            'Sys::Syslog'           => '0.03',
            'Test'                  => '1.20',
            'Test::Builder'         => '0.15',
            'Test::Harness'         => '2.26',
            'Test::Harness::Straps' => '0.14',
            'Test::More'            => '0.45',
            'Test::Simple'          => '0.45',
            'Thread::Queue'         => '2.00',
            'Thread::Semaphore'     => '2.00',
            'Tie::File'             => '0.93',
            'Tie::RefHash'          => '1.30',
            'Unicode'               => '3.2.0',
            'Unicode::Collate'      => '0.12',
            'Unicode::Normalize'    => '0.17',
            'XS::APItest'           => '0.01',
            'attributes'            => '0.05',
            'base'                  => '1.03',
            'bigint'                => '0.02',
            'bignum'                => '0.11',
            'bigrat'                => '0.04',
            'blib'                  => '1.02',
            'encoding'              => '1.35',
            'sort'                  => '1.01',
            'threads'               => '0.99',
        },
        removed => {
            'Encode::Internal'      => 1,
            'Encode::JP::Constants' => 1,
            'Encode::JP::ISO_2022_JP'=> 1,
            'Encode::JP::JIS'       => 1,
            'Encode::JP::Tr'        => 1,
            'Encode::Tcl'           => 1,
            'Encode::Tcl::Escape'   => 1,
            'Encode::Tcl::Extended' => 1,
            'Encode::Tcl::HanZi'    => 1,
            'Encode::Tcl::Table'    => 1,
            'Encode::XS'            => 1,
            'Encode::iso10646_1'    => 1,
            'Encode::usc2_le'       => 1,
            'Encode::utf8'          => 1,
            'PerlIO::Scalar'        => 1,
            'PerlIO::Via'           => 1,
        }
    },
    5.008001 => {
        delta_from => 5.008,
        changed => {
            'Attribute::Handlers'   => '0.78',
            'AutoLoader'            => '5.60',
            'AutoSplit'             => '1.04',
            'B'                     => '1.02',
            'B::Asmdata'            => '1.01',
            'B::Assembler'          => '0.06',
            'B::Bblock'             => '1.02',
            'B::Bytecode'           => '1.01',
            'B::C'                  => '1.02',
            'B::Concise'            => '0.56',
            'B::Debug'              => '1.01',
            'B::Deparse'            => '0.64',
            'B::Disassembler'       => '1.03',
            'B::Lint'               => '1.02',
            'B::Terse'              => '1.02',
            'Benchmark'             => '1.051',
            'ByteLoader'            => '0.05',
            'CGI'                   => '3.00',
            'CGI::Carp'             => '1.26',
            'CGI::Cookie'           => '1.24',
            'CGI::Fast'             => '1.041',
            'CGI::Pretty'           => '1.07_00',
            'CGI::Util'             => '1.31',
            'CPAN'                  => '1.76_01',
            'CPAN::FirstTime'       => '1.60',
            'CPAN::Nox'             => '1.03',
            'Class::Struct'         => '0.63',
            'Cwd'                   => '2.08',
            'DB_File'               => '1.806',
            'Data::Dumper'          => '2.121',
            'Devel::DProf'          => '20030813.00',
            'Devel::PPPort'         => '2.007',
            'Devel::Peek'           => '1.01',
            'Digest'                => '1.02',
            'Digest::MD5'           => '2.27',
            'Encode'                => '1.9801',
            'Encode::Alias'         => '1.38',
            'Encode::Byte'          => '1.23',
            'Encode::CJKConstants'  => '1.02',
            'Encode::CN::HZ'        => '1.05',
            'Encode::Config'        => '1.07',
            'Encode::Encoder'       => '0.07',
            'Encode::Encoding'      => '1.33',
            'Encode::Guess'         => '1.09',
            'Encode::JP::JIS7'      => '1.12',
            'Encode::KR'            => '1.23',
            'Encode::KR::2022_KR'   => '1.06',
            'Encode::MIME::Header'  => '1.09',
            'Encode::Unicode'       => '1.40',
            'Encode::Unicode::UTF7' => '0.02',
            'English'               => '1.01',
            'Errno'                 => '1.09_00',
            'Exporter'              => '5.567',
            'Exporter::Heavy'       => '5.567',
            'ExtUtils::Command'     => '1.05',
            'ExtUtils::Command::MM' => '0.03',
            'ExtUtils::Constant'    => '0.14',
            'ExtUtils::Install'     => '1.32',
            'ExtUtils::Installed'   => '0.08',
            'ExtUtils::Liblist'     => '1.01',
            'ExtUtils::Liblist::Kid'=> '1.3',
            'ExtUtils::MM_Any'      => '0.07',
            'ExtUtils::MM_BeOS'     => '1.04',
            'ExtUtils::MM_Cygwin'   => '1.06',
            'ExtUtils::MM_DOS'      => '0.02',
            'ExtUtils::MM_MacOS'    => '1.07',
            'ExtUtils::MM_NW5'      => '2.06',
            'ExtUtils::MM_OS2'      => '1.04',
            'ExtUtils::MM_UWIN'     => '0.02',
            'ExtUtils::MM_Unix'     => '1.42',
            'ExtUtils::MM_VMS'      => '5.70',
            'ExtUtils::MM_Win32'    => '1.09',
            'ExtUtils::MM_Win95'    => '0.03',
            'ExtUtils::MakeMaker'   => '6.17',
            'ExtUtils::MakeMaker::bytes'=> '0.01',
            'ExtUtils::MakeMaker::vmsish'=> '0.01',
            'ExtUtils::Manifest'    => '1.42',
            'Fcntl'                 => '1.05',
            'File::Basename'        => '2.72',
            'File::Copy'            => '2.06',
            'File::Find'            => '1.05',
            'File::Glob'            => '1.02',
            'File::Path'            => '1.06',
            'File::Spec'            => '0.86',
            'File::Spec::Cygwin'    => '1.1',
            'File::Spec::Epoc'      => '1.1',
            'File::Spec::Functions' => '1.3',
            'File::Spec::Mac'       => '1.4',
            'File::Spec::OS2'       => '1.2',
            'File::Spec::Unix'      => '1.5',
            'File::Spec::VMS'       => '1.4',
            'File::Spec::Win32'     => '1.4',
            'File::Temp'            => '0.14',
            'FileCache'             => '1.03',
            'Filter::Util::Call'    => '1.0601',
            'GDBM_File'             => '1.07',
            'Getopt::Long'          => '2.34',
            'Getopt::Std'           => '1.04',
            'Hash::Util'            => '0.05',
            'I18N::LangTags'        => '0.28',
            'I18N::LangTags::List'  => '0.26',
            'I18N::Langinfo'        => '0.02',
            'IO'                    => '1.21',
            'IO::Dir'               => '1.04',
            'IO::File'              => '1.10',
            'IO::Handle'            => '1.23',
            'IO::Seekable'          => '1.09',
            'IO::Select'            => '1.16',
            'IO::Socket'            => '1.28',
            'IO::Socket::INET'      => '1.27',
            'IO::Socket::UNIX'      => '1.21',
            'IPC::Msg'              => '1.02',
            'IPC::Open3'            => '1.0105',
            'IPC::Semaphore'        => '1.02',
            'IPC::SysV'             => '1.04',
            'JNI'                   => '0.2',
            'List::Util'            => '1.13',
            'Locale::Country'       => '2.61',
            'Locale::Currency'      => '2.21',
            'Locale::Language'      => '2.21',
            'Locale::Maketext'      => '1.06',
            'Locale::Maketext::Guts'=> undef,
            'Locale::Maketext::GutsLoader'=> undef,
            'Locale::Script'        => '2.21',
            'MIME::Base64'          => '2.20',
            'MIME::QuotedPrint'     => '2.20',
            'Math::BigFloat'        => '1.40',
            'Math::BigInt'          => '1.66',
            'Math::BigInt::Calc'    => '0.36',
            'Math::BigInt::Scalar'  => '0.11',
            'Math::BigRat'          => '0.10',
            'Math::Trig'            => '1.02',
            'NDBM_File'             => '1.05',
            'NEXT'                  => '0.60',
            'Net::Cmd'              => '2.24',
            'Net::Domain'           => '2.18',
            'Net::FTP'              => '2.71',
            'Net::FTP::A'           => '1.16',
            'Net::NNTP'             => '2.22',
            'Net::POP3'             => '2.24',
            'Net::Ping'             => '2.31',
            'Net::SMTP'             => '2.26',
            'Net::hostent'          => '1.01',
            'Net::servent'          => '1.01',
            'ODBM_File'             => '1.04',
            'OS2::DLL'              => '1.01',
            'OS2::ExtAttr'          => '0.02',
            'OS2::PrfDB'            => '0.03',
            'OS2::Process'          => '1.01',
            'OS2::REXX'             => '1.02',
            'POSIX'                 => '1.06',
            'PerlIO'                => '1.02',
            'PerlIO::encoding'      => '0.07',
            'PerlIO::scalar'        => '0.02',
            'PerlIO::via'           => '0.02',
            'PerlIO::via::QuotedPrint'=> '0.05',
            'Pod::Checker'          => '1.41',
            'Pod::Find'             => '0.24',
            'Pod::Functions'        => '1.02',
            'Pod::Html'             => '1.0501',
            'Pod::InputObjects'     => '1.14',
            'Pod::LaTeX'            => '0.55',
            'Pod::Man'              => '1.37',
            'Pod::ParseLink'        => '1.06',
            'Pod::ParseUtils'       => '0.3',
            'Pod::Perldoc'          => '3.10',
            'Pod::Perldoc::BaseTo'  => undef,
            'Pod::Perldoc::GetOptsOO'=> undef,
            'Pod::Perldoc::ToChecker'=> undef,
            'Pod::Perldoc::ToMan'   => undef,
            'Pod::Perldoc::ToNroff' => undef,
            'Pod::Perldoc::ToPod'   => undef,
            'Pod::Perldoc::ToRtf'   => undef,
            'Pod::Perldoc::ToText'  => undef,
            'Pod::Perldoc::ToTk'    => undef,
            'Pod::Perldoc::ToXml'   => undef,
            'Pod::PlainText'        => '2.01',
            'Pod::Text'             => '2.21',
            'Pod::Text::Color'      => '1.04',
            'Pod::Text::Overstrike' => '1.1',
            'Pod::Text::Termcap'    => '1.11',
            'Pod::Usage'            => '1.16',
            'SDBM_File'             => '1.04',
            'Safe'                  => '2.10',
            'Scalar::Util'          => '1.13',
            'SelfLoader'            => '1.0904',
            'Shell'                 => '0.5',
            'Socket'                => '1.76',
            'Storable'              => '2.08',
            'Switch'                => '2.10',
            'Symbol'                => '1.05',
            'Sys::Hostname'         => '1.11',
            'Sys::Syslog'           => '0.04',
            'Term::ANSIColor'       => '1.07',
            'Term::Cap'             => '1.08',
            'Term::Complete'        => '1.401',
            'Term::ReadLine'        => '1.01',
            'Test'                  => '1.24',
            'Test::Builder'         => '0.17',
            'Test::Harness'         => '2.30',
            'Test::Harness::Straps' => '0.15',
            'Test::More'            => '0.47',
            'Test::Simple'          => '0.47',
            'Text::Abbrev'          => '1.01',
            'Text::Balanced'        => '1.95',
            'Text::Wrap'            => '2001.09291',
            'Thread::Semaphore'     => '2.01',
            'Tie::Array'            => '1.03',
            'Tie::File'             => '0.97',
            'Tie::RefHash'          => '1.31',
            'Time::HiRes'           => '1.51',
            'Time::Local'           => '1.07',
            'UNIVERSAL'             => '1.01',
            'Unicode'               => '4.0.0',
            'Unicode::Collate'      => '0.28',
            'Unicode::Normalize'    => '0.23',
            'Unicode::UCD'          => '0.21',
            'VMS::Filespec'         => '1.11',
            'XS::APItest'           => '0.02',
            'XSLoader'              => '0.02',
            'attributes'            => '0.06',
            'base'                  => '2.03',
            'bigint'                => '0.04',
            'bignum'                => '0.14',
            'bigrat'                => '0.06',
            'bytes'                 => '1.01',
            'charnames'             => '1.02',
            'diagnostics'           => '1.11',
            'encoding'              => '1.47',
            'fields'                => '2.03',
            'filetest'              => '1.01',
            'if'                    => '0.03',
            'lib'                   => '0.5565',
            'open'                  => '1.02',
            'overload'              => '1.01',
            're'                    => '0.04',
            'sort'                  => '1.02',
            'strict'                => '1.03',
            'threads'               => '1.00',
            'threads::shared'       => '0.91',
            'utf8'                  => '1.02',
            'vmsish'                => '1.01',
            'warnings'              => '1.03',
        },
        removed => {
        }
    },
    5.008002 => {
        delta_from => 5.008001,
        changed => {
            'DB_File'               => '1.807',
            'Devel::PPPort'         => '2.009',
            'Digest::MD5'           => '2.30',
            'I18N::LangTags'        => '0.29',
            'I18N::LangTags::List'  => '0.29',
            'MIME::Base64'          => '2.21',
            'MIME::QuotedPrint'     => '2.21',
            'Net::Domain'           => '2.19',
            'Net::FTP'              => '2.72',
            'Pod::Perldoc'          => '3.11',
            'Time::HiRes'           => '1.52',
            'Unicode::Collate'      => '0.30',
            'Unicode::Normalize'    => '0.25',
        },
        removed => {
        }
    },
    5.008003 => {
        delta_from => 5.008002,
        changed => {
            'Benchmark'             => '1.052',
            'CGI'                   => '3.01',
            'CGI::Carp'             => '1.27',
            'CGI::Fast'             => '1.05',
            'CGI::Pretty'           => '1.08',
            'CGI::Util'             => '1.4',
            'Cwd'                   => '2.12',
            'DB_File'               => '1.808',
            'Devel::PPPort'         => '2.011',
            'Digest'                => '1.05',
            'Digest::MD5'           => '2.33',
            'Digest::base'          => '1.00',
            'Encode'                => '1.99',
            'Exporter'              => '5.57',
            'File::CheckTree'       => '4.3',
            'File::Copy'            => '2.07',
            'File::Find'            => '1.06',
            'File::Spec'            => '0.87',
            'FindBin'               => '1.44',
            'Getopt::Std'           => '1.05',
            'Math::BigFloat'        => '1.42',
            'Math::BigInt'          => '1.68',
            'Math::BigInt::Calc'    => '0.38',
            'Math::BigInt::CalcEmu' => '0.02',
            'OS2::DLL'              => '1.02',
            'POSIX'                 => '1.07',
            'PerlIO'                => '1.03',
            'PerlIO::via::QuotedPrint'=> '0.06',
            'Pod::Html'             => '1.0502',
            'Pod::Parser'           => '1.14',
            'Pod::Perldoc'          => '3.12',
            'Pod::PlainText'        => '2.02',
            'Storable'              => '2.09',
            'Test::Harness'         => '2.40',
            'Test::Harness::Assert' => '0.02',
            'Test::Harness::Iterator'=> '0.02',
            'Test::Harness::Straps' => '0.19',
            'Tie::Hash'             => '1.01',
            'Unicode::Collate'      => '0.33',
            'Unicode::Normalize'    => '0.28',
            'XS::APItest'           => '0.03',
            'base'                  => '2.04',
            'diagnostics'           => '1.12',
            'encoding'              => '1.48',
            'threads'               => '1.01',
            'threads::shared'       => '0.92',
        },
        removed => {
            'Math::BigInt::Scalar'  => 1,
        }
    },
    5.008004 => {
        delta_from => 5.008003,
        changed => {
            'Attribute::Handlers'   => '0.78_01',
            'B::Assembler'          => '0.07',
            'B::Concise'            => '0.60',
            'B::Deparse'            => '0.66',
            'Benchmark'             => '1.06',
            'CGI'                   => '3.04',
            'Carp'                  => '1.02',
            'Cwd'                   => '2.17',
            'DBM_Filter'            => '0.01',
            'DBM_Filter::compress'  => '0.01',
            'DBM_Filter::encode'    => '0.01',
            'DBM_Filter::int32'     => '0.01',
            'DBM_Filter::null'      => '0.01',
            'DBM_Filter::utf8'      => '0.01',
            'Digest'                => '1.06',
            'DynaLoader'            => '1.05',
            'Encode'                => '1.99_01',
            'Encode::CN::HZ'        => '1.0501',
            'Exporter'              => '5.58',
            'Exporter::Heavy'       => '5.57',
            'ExtUtils::Liblist::Kid'=> '1.3001',
            'ExtUtils::MM_NW5'      => '2.07_02',
            'ExtUtils::MM_Win95'    => '0.0301',
            'File::Find'            => '1.07',
            'IO::Handle'            => '1.24',
            'IO::Pipe'              => '1.123',
            'IPC::Open3'            => '1.0106',
            'Locale::Maketext'      => '1.08',
            'MIME::Base64'          => '3.01',
            'MIME::QuotedPrint'     => '3.01',
            'Math::BigFloat'        => '1.44',
            'Math::BigInt'          => '1.70',
            'Math::BigInt::Calc'    => '0.40',
            'Math::BigInt::CalcEmu' => '0.04',
            'Math::BigRat'          => '0.12',
            'ODBM_File'             => '1.05',
            'POSIX'                 => '1.08',
            'Shell'                 => '0.5.2',
            'Socket'                => '1.77',
            'Storable'              => '2.12',
            'Sys::Syslog'           => '0.05',
            'Term::ANSIColor'       => '1.08',
            'Time::HiRes'           => '1.59',
            'Unicode'               => '4.0.1',
            'Unicode::UCD'          => '0.22',
            'Win32'                 => '0.23',
            'base'                  => '2.05',
            'bigint'                => '0.05',
            'bignum'                => '0.15',
            'charnames'             => '1.03',
            'open'                  => '1.03',
            'threads'               => '1.03',
            'utf8'                  => '1.03',
        },
        removed => {
        }
    },
    5.008005 => {
        delta_from => 5.008004,
        changed => {
            'B::Concise'            => '0.61',
            'B::Deparse'            => '0.67',
            'CGI'                   => '3.05',
            'CGI::Carp'             => '1.28',
            'CGI::Util'             => '1.5',
            'Carp'                  => '1.03',
            'Carp::Heavy'           => '1.03',
            'Cwd'                   => '2.19',
            'DB_File'               => '1.809',
            'Digest'                => '1.08',
            'Encode'                => '2.01',
            'Encode::Alias'         => '2.00',
            'Encode::Byte'          => '2.00',
            'Encode::CJKConstants'  => '2.00',
            'Encode::CN'            => '2.00',
            'Encode::CN::HZ'        => '2.01',
            'Encode::Config'        => '2.00',
            'Encode::EBCDIC'        => '2.00',
            'Encode::Encoder'       => '2.00',
            'Encode::Encoding'      => '2.00',
            'Encode::Guess'         => '2.00',
            'Encode::JP'            => '2.00',
            'Encode::JP::H2Z'       => '2.00',
            'Encode::JP::JIS7'      => '2.00',
            'Encode::KR'            => '2.00',
            'Encode::KR::2022_KR'   => '2.00',
            'Encode::MIME::Header'  => '2.00',
            'Encode::Symbol'        => '2.00',
            'Encode::TW'            => '2.00',
            'Encode::Unicode'       => '2.00',
            'Encode::Unicode::UTF7' => '2.01',
            'File::Basename'        => '2.73',
            'File::Copy'            => '2.08',
            'File::Glob'            => '1.03',
            'FileCache'             => '1.04_01',
            'I18N::LangTags'        => '0.33',
            'I18N::LangTags::Detect'=> '1.03',
            'List::Util'            => '1.14',
            'Locale::Constants'     => '2.07',
            'Locale::Country'       => '2.07',
            'Locale::Currency'      => '2.07',
            'Locale::Language'      => '2.07',
            'Locale::Maketext'      => '1.09',
            'Locale::Script'        => '2.07',
            'Net::Cmd'              => '2.26',
            'Net::FTP'              => '2.75',
            'Net::NNTP'             => '2.23',
            'Net::POP3'             => '2.28',
            'Net::SMTP'             => '2.29',
            'Net::Time'             => '2.10',
            'Pod::Checker'          => '1.42',
            'Pod::Find'             => '0.2401',
            'Pod::LaTeX'            => '0.56',
            'Pod::ParseUtils'       => '1.2',
            'Pod::Perldoc'          => '3.13',
            'Safe'                  => '2.11',
            'Scalar::Util'          => '1.14',
            'Shell'                 => '0.6',
            'Storable'              => '2.13',
            'Term::Cap'             => '1.09',
            'Test'                  => '1.25',
            'Test::Harness'         => '2.42',
            'Text::ParseWords'      => '3.22',
            'Text::Wrap'            => '2001.09292',
            'Time::Local'           => '1.10',
            'Unicode::Collate'      => '0.40',
            'Unicode::Normalize'    => '0.30',
            'XS::APItest'           => '0.04',
            'autouse'               => '1.04',
            'base'                  => '2.06',
            'charnames'             => '1.04',
            'diagnostics'           => '1.13',
            'encoding'              => '2.00',
            'threads'               => '1.05',
            'utf8'                  => '1.04',
        },
        removed => {
        }
    },
    5.008006 => {
        delta_from => 5.008005,
        changed => {
            'B'                     => '1.07',
            'B::C'                  => '1.04',
            'B::Concise'            => '0.64',
            'B::Debug'              => '1.02',
            'B::Deparse'            => '0.69',
            'B::Lint'               => '1.03',
            'B::Showlex'            => '1.02',
            'Cwd'                   => '3.01',
            'DB_File'               => '1.810',
            'Data::Dumper'          => '2.121_02',
            'Devel::PPPort'         => '3.03',
            'Devel::Peek'           => '1.02',
            'Encode'                => '2.08',
            'Encode::Alias'         => '2.02',
            'Encode::Encoding'      => '2.02',
            'Encode::JP'            => '2.01',
            'Encode::Unicode'       => '2.02',
            'Exporter::Heavy'       => '5.58',
            'ExtUtils::Constant'    => '0.1401',
            'File::Spec'            => '3.01',
            'File::Spec::Win32'     => '1.5',
            'I18N::LangTags'        => '0.35',
            'I18N::LangTags::List'  => '0.35',
            'MIME::Base64'          => '3.05',
            'MIME::QuotedPrint'     => '3.03',
            'Math::BigFloat'        => '1.47',
            'Math::BigInt'          => '1.73',
            'Math::BigInt::Calc'    => '0.43',
            'Math::BigRat'          => '0.13',
            'Text::ParseWords'      => '3.23',
            'Time::HiRes'           => '1.65',
            'XS::APItest'           => '0.05',
            'diagnostics'           => '1.14',
            'encoding'              => '2.01',
            'open'                  => '1.04',
            'overload'              => '1.02',
        },
        removed => {
        }
    },
    5.008007 => {
        delta_from => 5.008006,
        changed => {
            'B'                     => '1.09',
            'B::Concise'            => '0.65',
            'B::Deparse'            => '0.7',
            'B::Disassembler'       => '1.04',
            'B::Terse'              => '1.03',
            'Benchmark'             => '1.07',
            'CGI'                   => '3.10',
            'CGI::Carp'             => '1.29',
            'CGI::Cookie'           => '1.25',
            'Carp'                  => '1.04',
            'Carp::Heavy'           => '1.04',
            'Class::ISA'            => '0.33',
            'Cwd'                   => '3.05',
            'DB_File'               => '1.811',
            'Data::Dumper'          => '2.121_04',
            'Devel::DProf'          => '20050310.00',
            'Devel::PPPort'         => '3.06',
            'Digest'                => '1.10',
            'Digest::file'          => '0.01',
            'Encode'                => '2.10',
            'Encode::Alias'         => '2.03',
            'Errno'                 => '1.09_01',
            'ExtUtils::Constant'    => '0.16',
            'ExtUtils::Constant::Base'=> '0.01',
            'ExtUtils::Constant::Utils'=> '0.01',
            'ExtUtils::Constant::XS'=> '0.01',
            'File::Find'            => '1.09',
            'File::Glob'            => '1.04',
            'File::Path'            => '1.07',
            'File::Spec'            => '3.05',
            'File::Temp'            => '0.16',
            'FileCache'             => '1.05',
            'IO::File'              => '1.11',
            'IO::Socket::INET'      => '1.28',
            'Math::BigFloat'        => '1.51',
            'Math::BigInt'          => '1.77',
            'Math::BigInt::Calc'    => '0.47',
            'Math::BigInt::CalcEmu' => '0.05',
            'Math::BigRat'          => '0.15',
            'Pod::Find'             => '1.3',
            'Pod::Html'             => '1.0503',
            'Pod::InputObjects'     => '1.3',
            'Pod::LaTeX'            => '0.58',
            'Pod::ParseUtils'       => '1.3',
            'Pod::Parser'           => '1.3',
            'Pod::Perldoc'          => '3.14',
            'Pod::Select'           => '1.3',
            'Pod::Usage'            => '1.3',
            'SelectSaver'           => '1.01',
            'Symbol'                => '1.06',
            'Sys::Syslog'           => '0.06',
            'Term::ANSIColor'       => '1.09',
            'Term::Complete'        => '1.402',
            'Test::Builder'         => '0.22',
            'Test::Harness'         => '2.48',
            'Test::Harness::Point'  => '0.01',
            'Test::Harness::Straps' => '0.23',
            'Test::More'            => '0.54',
            'Test::Simple'          => '0.54',
            'Text::ParseWords'      => '3.24',
            'Text::Wrap'            => '2001.09293',
            'Tie::RefHash'          => '1.32',
            'Time::HiRes'           => '1.66',
            'Time::Local'           => '1.11',
            'Unicode'               => '4.1.0',
            'Unicode::Normalize'    => '0.32',
            'Unicode::UCD'          => '0.23',
            'Win32'                 => '0.24',
            'XS::APItest'           => '0.06',
            'base'                  => '2.07',
            'bigint'                => '0.07',
            'bignum'                => '0.17',
            'bigrat'                => '0.08',
            'bytes'                 => '1.02',
            'constant'              => '1.05',
            'overload'              => '1.03',
            'threads::shared'       => '0.93',
            'utf8'                  => '1.05',
        },
        removed => {
            'JNI'                   => 1,
            'JPL::AutoLoader'       => 1,
            'JPL::Class'            => 1,
            'JPL::Compile'          => 1,
        }
    },
    5.008008 => {
        delta_from => 5.008007,
        changed => {
            'Attribute::Handlers'   => '0.78_02',
            'B'                     => '1.09_01',
            'B::Bblock'             => '1.02_01',
            'B::Bytecode'           => '1.01_01',
            'B::C'                  => '1.04_01',
            'B::CC'                 => '1.00_01',
            'B::Concise'            => '0.66',
            'B::Debug'              => '1.02_01',
            'B::Deparse'            => '0.71',
            'B::Disassembler'       => '1.05',
            'B::Terse'              => '1.03_01',
            'ByteLoader'            => '0.06',
            'CGI'                   => '3.15',
            'CGI::Cookie'           => '1.26',
            'CPAN'                  => '1.76_02',
            'Cwd'                   => '3.12',
            'DB'                    => '1.01',
            'DB_File'               => '1.814',
            'Data::Dumper'          => '2.121_08',
            'Devel::DProf'          => '20050603.00',
            'Devel::PPPort'         => '3.06_01',
            'Devel::Peek'           => '1.03',
            'Digest'                => '1.14',
            'Digest::MD5'           => '2.36',
            'Digest::file'          => '1.00',
            'Dumpvalue'             => '1.12',
            'Encode'                => '2.12',
            'Encode::Alias'         => '2.04',
            'Encode::Config'        => '2.01',
            'Encode::MIME::Header'  => '2.01',
            'Encode::MIME::Header::ISO_2022_JP'=> '1.01',
            'English'               => '1.02',
            'ExtUtils::Command'     => '1.09',
            'ExtUtils::Command::MM' => '0.05',
            'ExtUtils::Constant'    => '0.17',
            'ExtUtils::Embed'       => '1.26',
            'ExtUtils::Install'     => '1.33',
            'ExtUtils::Liblist::Kid'=> '1.3',
            'ExtUtils::MM'          => '0.05',
            'ExtUtils::MM_AIX'      => '0.03',
            'ExtUtils::MM_Any'      => '0.13',
            'ExtUtils::MM_BeOS'     => '1.05',
            'ExtUtils::MM_Cygwin'   => '1.08',
            'ExtUtils::MM_MacOS'    => '1.08',
            'ExtUtils::MM_NW5'      => '2.08',
            'ExtUtils::MM_OS2'      => '1.05',
            'ExtUtils::MM_QNX'      => '0.02',
            'ExtUtils::MM_Unix'     => '1.50',
            'ExtUtils::MM_VMS'      => '5.73',
            'ExtUtils::MM_VOS'      => '0.02',
            'ExtUtils::MM_Win32'    => '1.12',
            'ExtUtils::MM_Win95'    => '0.04',
            'ExtUtils::MakeMaker'   => '6.30',
            'ExtUtils::MakeMaker::Config'=> '0.02',
            'ExtUtils::Manifest'    => '1.46',
            'File::Basename'        => '2.74',
            'File::Copy'            => '2.09',
            'File::Find'            => '1.10',
            'File::Glob'            => '1.05',
            'File::Path'            => '1.08',
            'File::Spec'            => '3.12',
            'File::Spec::Win32'     => '1.6',
            'FileCache'             => '1.06',
            'Filter::Simple'        => '0.82',
            'FindBin'               => '1.47',
            'GDBM_File'             => '1.08',
            'Getopt::Long'          => '2.35',
            'IO'                    => '1.22',
            'IO::Dir'               => '1.05',
            'IO::File'              => '1.13',
            'IO::Handle'            => '1.25',
            'IO::Pipe'              => '1.13',
            'IO::Poll'              => '0.07',
            'IO::Seekable'          => '1.10',
            'IO::Select'            => '1.17',
            'IO::Socket'            => '1.29',
            'IO::Socket::INET'      => '1.29',
            'IO::Socket::UNIX'      => '1.22',
            'IPC::Open2'            => '1.02',
            'IPC::Open3'            => '1.02',
            'List::Util'            => '1.18',
            'MIME::Base64'          => '3.07',
            'MIME::QuotedPrint'     => '3.07',
            'Math::Complex'         => '1.35',
            'Math::Trig'            => '1.03',
            'NDBM_File'             => '1.06',
            'ODBM_File'             => '1.06',
            'OS2::PrfDB'            => '0.04',
            'OS2::Process'          => '1.02',
            'OS2::REXX'             => '1.03',
            'Opcode'                => '1.06',
            'POSIX'                 => '1.09',
            'PerlIO'                => '1.04',
            'PerlIO::encoding'      => '0.09',
            'PerlIO::scalar'        => '0.04',
            'PerlIO::via'           => '0.03',
            'Pod::Checker'          => '1.43',
            'Pod::Find'             => '1.34',
            'Pod::Functions'        => '1.03',
            'Pod::Html'             => '1.0504',
            'Pod::ParseUtils'       => '1.33',
            'Pod::Parser'           => '1.32',
            'Pod::Usage'            => '1.33',
            'SDBM_File'             => '1.05',
            'Safe'                  => '2.12',
            'Scalar::Util'          => '1.18',
            'Socket'                => '1.78',
            'Storable'              => '2.15',
            'Switch'                => '2.10_01',
            'Sys::Syslog'           => '0.13',
            'Term::ANSIColor'       => '1.10',
            'Term::ReadLine'        => '1.02',
            'Test::Builder'         => '0.32',
            'Test::Builder::Module' => '0.02',
            'Test::Builder::Tester' => '1.02',
            'Test::Builder::Tester::Color'=> undef,
            'Test::Harness'         => '2.56',
            'Test::Harness::Straps' => '0.26',
            'Test::More'            => '0.62',
            'Test::Simple'          => '0.62',
            'Text::Tabs'            => '2005.0824',
            'Text::Wrap'            => '2005.082401',
            'Tie::Hash'             => '1.02',
            'Time::HiRes'           => '1.86',
            'Unicode::Collate'      => '0.52',
            'Unicode::UCD'          => '0.24',
            'User::grent'           => '1.01',
            'Win32'                 => '0.2601',
            'XS::APItest'           => '0.08',
            'XS::Typemap'           => '0.02',
            'XSLoader'              => '0.06',
            'attrs'                 => '1.02',
            'autouse'               => '1.05',
            'blib'                  => '1.03',
            'charnames'             => '1.05',
            'diagnostics'           => '1.15',
            'encoding'              => '2.02',
            'if'                    => '0.05',
            'open'                  => '1.05',
            'ops'                   => '1.01',
            'overload'              => '1.04',
            're'                    => '0.05',
            'threads'               => '1.07',
            'threads::shared'       => '0.94',
            'utf8'                  => '1.06',
            'vmsish'                => '1.02',
            'warnings'              => '1.05',
            'warnings::register'    => '1.01',
        },
        removed => {
        }
    },
    5.008009 => {
        delta_from => 5.008008,
        changed => {
            'Attribute::Handlers'   => '0.78_03',
            'AutoLoader'            => '5.67',
            'AutoSplit'             => '1.06',
            'B'                     => '1.19',
            'B::Asmdata'            => '1.02',
            'B::Assembler'          => '0.08',
            'B::C'                  => '1.05',
            'B::Concise'            => '0.76',
            'B::Debug'              => '1.05',
            'B::Deparse'            => '0.87',
            'B::Lint'               => '1.11',
            'B::Lint::Debug'        => undef,
            'B::Terse'              => '1.05',
            'Benchmark'             => '1.1',
            'CGI'                   => '3.42',
            'CGI::Carp'             => '1.30_01',
            'CGI::Cookie'           => '1.29',
            'CGI::Fast'             => '1.07',
            'CGI::Util'             => '1.5_01',
            'CPAN'                  => '1.9301',
            'CPAN::Debug'           => '5.5',
            'CPAN::DeferedCode'     => '5.50',
            'CPAN::Distroprefs'     => '6',
            'CPAN::FirstTime'       => '5.5_01',
            'CPAN::HandleConfig'    => '5.5',
            'CPAN::Kwalify'         => '5.50',
            'CPAN::Nox'             => '5.50',
            'CPAN::Queue'           => '5.5',
            'CPAN::Tarzip'          => '5.5',
            'CPAN::Version'         => '5.5',
            'Carp'                  => '1.10',
            'Carp::Heavy'           => '1.10',
            'Cwd'                   => '3.29',
            'DBM_Filter'            => '0.02',
            'DBM_Filter::compress'  => '0.02',
            'DBM_Filter::encode'    => '0.02',
            'DBM_Filter::int32'     => '0.02',
            'DBM_Filter::null'      => '0.02',
            'DBM_Filter::utf8'      => '0.02',
            'DB_File'               => '1.817',
            'Data::Dumper'          => '2.121_17',
            'Devel::DProf'          => '20080331.00',
            'Devel::InnerPackage'   => '0.3',
            'Devel::PPPort'         => '3.14',
            'Devel::Peek'           => '1.04',
            'Digest'                => '1.15',
            'Digest::MD5'           => '2.37',
            'DirHandle'             => '1.02',
            'DynaLoader'            => '1.09',
            'Encode'                => '2.26',
            'Encode::Alias'         => '2.10',
            'Encode::Byte'          => '2.03',
            'Encode::CJKConstants'  => '2.02',
            'Encode::CN'            => '2.02',
            'Encode::CN::HZ'        => '2.05',
            'Encode::Config'        => '2.05',
            'Encode::EBCDIC'        => '2.02',
            'Encode::Encoder'       => '2.01',
            'Encode::Encoding'      => '2.05',
            'Encode::GSM0338'       => '2.01',
            'Encode::Guess'         => '2.02',
            'Encode::JP'            => '2.03',
            'Encode::JP::H2Z'       => '2.02',
            'Encode::JP::JIS7'      => '2.04',
            'Encode::KR'            => '2.02',
            'Encode::KR::2022_KR'   => '2.02',
            'Encode::MIME::Header'  => '2.05',
            'Encode::MIME::Header::ISO_2022_JP'=> '1.03',
            'Encode::MIME::Name'    => '1.01',
            'Encode::Symbol'        => '2.02',
            'Encode::TW'            => '2.02',
            'Encode::Unicode'       => '2.05',
            'Encode::Unicode::UTF7' => '2.04',
            'English'               => '1.03',
            'Errno'                 => '1.10',
            'Exporter'              => '5.63',
            'Exporter::Heavy'       => '5.63',
            'ExtUtils::Command'     => '1.15',
            'ExtUtils::Command::MM' => '6.48',
            'ExtUtils::Constant'    => '0.21',
            'ExtUtils::Constant::Base'=> '0.04',
            'ExtUtils::Constant::ProxySubs'=> '0.06',
            'ExtUtils::Constant::Utils'=> '0.02',
            'ExtUtils::Constant::XS'=> '0.02',
            'ExtUtils::Embed'       => '1.28',
            'ExtUtils::Install'     => '1.50_01',
            'ExtUtils::Installed'   => '1.43',
            'ExtUtils::Liblist'     => '6.48',
            'ExtUtils::Liblist::Kid'=> '6.48',
            'ExtUtils::MM'          => '6.48',
            'ExtUtils::MM_AIX'      => '6.48',
            'ExtUtils::MM_Any'      => '6.48',
            'ExtUtils::MM_BeOS'     => '6.48',
            'ExtUtils::MM_Cygwin'   => '6.48',
            'ExtUtils::MM_DOS'      => '6.48',
            'ExtUtils::MM_Darwin'   => '6.48',
            'ExtUtils::MM_MacOS'    => '6.48',
            'ExtUtils::MM_NW5'      => '6.48',
            'ExtUtils::MM_OS2'      => '6.48',
            'ExtUtils::MM_QNX'      => '6.48',
            'ExtUtils::MM_UWIN'     => '6.48',
            'ExtUtils::MM_Unix'     => '6.48',
            'ExtUtils::MM_VMS'      => '6.48',
            'ExtUtils::MM_VOS'      => '6.48',
            'ExtUtils::MM_Win32'    => '6.48',
            'ExtUtils::MM_Win95'    => '6.48',
            'ExtUtils::MY'          => '6.48',
            'ExtUtils::MakeMaker'   => '6.48',
            'ExtUtils::MakeMaker::Config'=> '6.48',
            'ExtUtils::MakeMaker::bytes'=> '6.48',
            'ExtUtils::MakeMaker::vmsish'=> '6.48',
            'ExtUtils::Manifest'    => '1.55',
            'ExtUtils::Mkbootstrap' => '6.48',
            'ExtUtils::Mksymlists'  => '6.48',
            'ExtUtils::Packlist'    => '1.43',
            'ExtUtils::ParseXS'     => '2.19',
            'ExtUtils::XSSymSet'    => '1.1',
            'ExtUtils::testlib'     => '6.48',
            'Fatal'                 => '1.06',
            'Fcntl'                 => '1.06',
            'File::Basename'        => '2.77',
            'File::CheckTree'       => '4.4',
            'File::Compare'         => '1.1005',
            'File::Copy'            => '2.13',
            'File::DosGlob'         => '1.01',
            'File::Find'            => '1.13',
            'File::Glob'            => '1.06',
            'File::Path'            => '2.07_02',
            'File::Spec'            => '3.29',
            'File::Spec::Cygwin'    => '3.29',
            'File::Spec::Epoc'      => '3.29',
            'File::Spec::Functions' => '3.29',
            'File::Spec::Mac'       => '3.29',
            'File::Spec::OS2'       => '3.29',
            'File::Spec::Unix'      => '3.29',
            'File::Spec::VMS'       => '3.29',
            'File::Spec::Win32'     => '3.29',
            'File::Temp'            => '0.20',
            'File::stat'            => '1.01',
            'FileCache'             => '1.07',
            'Filter::Simple'        => '0.83',
            'Filter::Util::Call'    => '1.07',
            'FindBin'               => '1.49',
            'GDBM_File'             => '1.09',
            'Getopt::Long'          => '2.37',
            'Getopt::Std'           => '1.06',
            'Hash::Util'            => '0.06',
            'IO'                    => '1.23',
            'IO::Dir'               => '1.06',
            'IO::File'              => '1.14',
            'IO::Handle'            => '1.27',
            'IO::Socket'            => '1.30',
            'IO::Socket::INET'      => '1.31',
            'IO::Socket::UNIX'      => '1.23',
            'IPC::Msg'              => '2.00',
            'IPC::Open2'            => '1.03',
            'IPC::Open3'            => '1.03',
            'IPC::Semaphore'        => '2.00',
            'IPC::SharedMem'        => '2.00',
            'IPC::SysV'             => '2.00',
            'List::Util'            => '1.19',
            'Locale::Maketext'      => '1.13',
            'Locale::Maketext::Guts'=> '1.13',
            'Locale::Maketext::GutsLoader'=> '1.13',
            'Math::BigFloat'        => '1.60',
            'Math::BigInt'          => '1.89',
            'Math::BigInt::Calc'    => '0.52',
            'Math::BigRat'          => '0.22',
            'Math::Complex'         => '1.54',
            'Math::Trig'            => '1.18',
            'Module::CoreList'      => '2.17',
            'Module::Pluggable'     => '3.8',
            'Module::Pluggable::Object'=> '3.6',
            'NDBM_File'             => '1.07',
            'NEXT'                  => '0.61',
            'Net::Cmd'              => '2.29',
            'Net::Config'           => '1.11',
            'Net::Domain'           => '2.20',
            'Net::FTP'              => '2.77',
            'Net::FTP::A'           => '1.18',
            'Net::NNTP'             => '2.24',
            'Net::POP3'             => '2.29',
            'Net::Ping'             => '2.35',
            'Net::SMTP'             => '2.31',
            'O'                     => '1.01',
            'ODBM_File'             => '1.07',
            'OS2::DLL'              => '1.03',
            'OS2::Process'          => '1.03',
            'Opcode'                => '1.0601',
            'POSIX'                 => '1.15',
            'PerlIO'                => '1.05',
            'PerlIO::encoding'      => '0.11',
            'PerlIO::scalar'        => '0.06',
            'PerlIO::via'           => '0.05',
            'Pod::Html'             => '1.09',
            'Pod::ParseUtils'       => '1.35',
            'Pod::Parser'           => '1.35',
            'Pod::Select'           => '1.35',
            'Pod::Usage'            => '1.35',
            'SDBM_File'             => '1.06',
            'Safe'                  => '2.16',
            'Scalar::Util'          => '1.19',
            'SelfLoader'            => '1.17',
            'Shell'                 => '0.72',
            'Socket'                => '1.81',
            'Storable'              => '2.19',
            'Switch'                => '2.13',
            'Sys::Syslog'           => '0.27',
            'Sys::Syslog::win32::Win32'=> undef,
            'Term::ANSIColor'       => '1.12',
            'Term::Cap'             => '1.12',
            'Term::ReadLine'        => '1.03',
            'Test::Builder'         => '0.80',
            'Test::Builder::Module' => '0.80',
            'Test::Builder::Tester' => '1.13',
            'Test::Harness'         => '2.64',
            'Test::Harness::Results'=> '0.01_01',
            'Test::Harness::Straps' => '0.26_01',
            'Test::Harness::Util'   => '0.01',
            'Test::More'            => '0.80',
            'Test::Simple'          => '0.80',
            'Text::Balanced'        => '1.98',
            'Text::ParseWords'      => '3.27',
            'Text::Soundex'         => '3.03',
            'Text::Tabs'            => '2007.1117',
            'Text::Wrap'            => '2006.1117',
            'Thread'                => '2.01',
            'Thread::Queue'         => '2.11',
            'Thread::Semaphore'     => '2.09',
            'Tie::Handle'           => '4.2',
            'Tie::Hash'             => '1.03',
            'Tie::Memoize'          => '1.1',
            'Tie::RefHash'          => '1.38',
            'Tie::Scalar'           => '1.01',
            'Tie::StdHandle'        => '4.2',
            'Time::HiRes'           => '1.9715',
            'Time::Local'           => '1.1901',
            'Time::gmtime'          => '1.03',
            'Unicode'               => '5.1.0',
            'Unicode::Normalize'    => '1.02',
            'Unicode::UCD'          => '0.25',
            'VMS::DCLsym'           => '1.03',
            'VMS::Stdio'            => '2.4',
            'Win32'                 => '0.38',
            'Win32API::File'        => '0.1001_01',
            'Win32API::File::ExtUtils::Myconst2perl'=> '1',
            'Win32CORE'             => '0.02',
            'XS::APItest'           => '0.15',
            'XS::Typemap'           => '0.03',
            'XSLoader'              => '0.10',
            'attributes'            => '0.09',
            'autouse'               => '1.06',
            'base'                  => '2.13',
            'bigint'                => '0.23',
            'bignum'                => '0.23',
            'bigrat'                => '0.23',
            'blib'                  => '1.04',
            'charnames'             => '1.06',
            'constant'              => '1.17',
            'diagnostics'           => '1.16',
            'encoding'              => '2.6_01',
            'fields'                => '2.12',
            'filetest'              => '1.02',
            'lib'                   => '0.61',
            'open'                  => '1.06',
            'ops'                   => '1.02',
            'overload'              => '1.06',
            're'                    => '0.0601',
            'sigtrap'               => '1.04',
            'threads'               => '1.71',
            'threads::shared'       => '1.27',
            'utf8'                  => '1.07',
            'warnings'              => '1.05_01',
        },
        removed => {
        }
    },
    5.009 => {
        delta_from => 5.008002,
        changed => {
            'B'                     => '1.03',
            'B::C'                  => '1.03',
            'B::Concise'            => '0.57',
            'B::Deparse'            => '0.65',
            'DB_File'               => '1.806',
            'Devel::PPPort'         => '2.008',
            'English'               => '1.02',
            'Fatal'                 => '1.04',
            'OS2::DLL'              => '1.02',
            'Opcode'                => '1.06',
            'Time::HiRes'           => '1.51',
            'Unicode::Collate'      => '0.28',
            'Unicode::Normalize'    => '0.23',
            'XSLoader'              => '0.03',
            'assertions'            => '0.01',
            'assertions::activate'  => '0.01',
            'overload'              => '1.02',
            'version'               => '0.29',
        },
        removed => {
        }
    },
    5.009001 => {
        delta_from => 5.008004,
        changed => {
            'B'                     => '1.05',
            'B::Assembler'          => '0.06',
            'B::C'                  => '1.04',
            'B::Concise'            => '0.59',
            'B::Debug'              => '1.02',
            'B::Deparse'            => '0.65',
            'DB_File'               => '1.808_01',
            'Devel::PPPort'         => '2.011_01',
            'Digest'                => '1.05',
            'DynaLoader'            => '1.04',
            'English'               => '1.02',
            'Exporter::Heavy'       => '5.567',
            'ExtUtils::Command'     => '1.07',
            'ExtUtils::Liblist::Kid'=> '1.3',
            'ExtUtils::MM_Any'      => '0.0901',
            'ExtUtils::MM_Cygwin'   => '1.07',
            'ExtUtils::MM_NW5'      => '2.07_01',
            'ExtUtils::MM_Unix'     => '1.45_01',
            'ExtUtils::MM_VMS'      => '5.71_01',
            'ExtUtils::MM_Win32'    => '1.10_01',
            'ExtUtils::MM_Win95'    => '0.03',
            'ExtUtils::MakeMaker'   => '6.21_02',
            'ExtUtils::Manifest'    => '1.43',
            'Fatal'                 => '1.04',
            'Getopt::Long'          => '2.3401',
            'IO::Handle'            => '1.23',
            'IO::Pipe'              => '1.122',
            'IPC::Open3'            => '1.0105',
            'MIME::Base64'          => '3.00_01',
            'MIME::QuotedPrint'     => '3.00',
            'Memoize'               => '1.01_01',
            'ODBM_File'             => '1.04',
            'Opcode'                => '1.06',
            'POSIX'                 => '1.07',
            'Storable'              => '2.11',
            'Time::HiRes'           => '1.56',
            'Time::Local'           => '1.07_94',
            'UNIVERSAL'             => '1.02',
            'Unicode'               => '4.0.0',
            'Unicode::UCD'          => '0.21',
            'XSLoader'              => '0.03',
            'assertions'            => '0.01',
            'assertions::activate'  => '0.01',
            'base'                  => '2.04',
            'if'                    => '0.0401',
            'open'                  => '1.02',
            'overload'              => '1.02',
            'threads'               => '1.02',
            'utf8'                  => '1.02',
            'version'               => '0.36',
        },
        removed => {
        }
    },
    5.009002 => {
        delta_from => 5.008007,
        changed => {
            'B'                     => '1.07',
            'B::Concise'            => '0.64',
            'B::Deparse'            => '0.69',
            'B::Disassembler'       => '1.03',
            'B::Terse'              => '1.02',
            'CGI'                   => '3.07',
            'Config::Extensions'    => '0.01',
            'Devel::DProf'          => '20030813.00',
            'DynaLoader'            => '1.07',
            'Encode'                => '2.09',
            'Encode::Alias'         => '2.02',
            'English'               => '1.03',
            'Exporter'              => '5.59',
            'Exporter::Heavy'       => '5.59',
            'ExtUtils::Command'     => '1.07',
            'ExtUtils::Command::MM' => '0.03_01',
            'ExtUtils::Embed'       => '1.26',
            'ExtUtils::Liblist::Kid'=> '1.3',
            'ExtUtils::MM_Any'      => '0.10',
            'ExtUtils::MM_Cygwin'   => '1.07',
            'ExtUtils::MM_MacOS'    => '1.08',
            'ExtUtils::MM_NW5'      => '2.07',
            'ExtUtils::MM_Unix'     => '1.46_01',
            'ExtUtils::MM_VMS'      => '5.71',
            'ExtUtils::MM_Win32'    => '1.10',
            'ExtUtils::MM_Win95'    => '0.03',
            'ExtUtils::MakeMaker'   => '6.25',
            'ExtUtils::Manifest'    => '1.44',
            'Fatal'                 => '1.04',
            'File::Path'            => '1.06',
            'FileCache'             => '1.04_01',
            'Getopt::Long'          => '2.3401',
            'IO::File'              => '1.10',
            'IO::Socket::INET'      => '1.27',
            'Math::BigFloat'        => '1.49',
            'Math::BigInt'          => '1.75',
            'Math::BigInt::Calc'    => '0.45',
            'Math::BigRat'          => '0.14',
            'Memoize'               => '1.01_01',
            'Module::CoreList'      => '1.99',
            'NEXT'                  => '0.60_01',
            'Opcode'                => '1.06',
            'Pod::Html'             => '1.0502',
            'Scalar::Util'          => '1.14_1',
            'Storable'              => '2.14',
            'Symbol'                => '1.05',
            'Test::Harness'         => '2.46',
            'Test::Harness::Straps' => '0.20_01',
            'Text::Balanced'        => '1.95_01',
            'Text::Wrap'            => '2001.09292',
            'UNIVERSAL'             => '1.02',
            'Unicode'               => '4.0.1',
            'Unicode::Normalize'    => '0.30',
            'Unicode::UCD'          => '0.22',
            'Win32'                 => '0.23',
            'XS::APItest'           => '0.05',
            'XSLoader'              => '0.03',
            'assertions'            => '0.01',
            'assertions::activate'  => '0.01',
            'base'                  => '2.06',
            'bigint'                => '0.06',
            'bignum'                => '0.16',
            'bigrat'                => '0.07',
            'bytes'                 => '1.01',
            'encoding::warnings'    => '0.05',
            'if'                    => '0.0401',
            're'                    => '0.05',
            'threads::shared'       => '0.92',
            'utf8'                  => '1.04',
            'version'               => '0.42',
            'warnings'              => '1.04',
        },
        removed => {
            'Test::Harness::Point'  => 1,
        }
    },
    5.009003 => {
        delta_from => 5.008008,
        changed => {
            'Archive::Tar'          => '1.26_01',
            'Archive::Tar::Constant'=> '0.02',
            'Archive::Tar::File'    => '0.02',
            'AutoSplit'             => '1.04_01',
            'B'                     => '1.10',
            'B::Bblock'             => '1.02',
            'B::Bytecode'           => '1.01',
            'B::C'                  => '1.04',
            'B::CC'                 => '1.00',
            'B::Concise'            => '0.67',
            'B::Debug'              => '1.02',
            'B::Deparse'            => '0.73',
            'B::Lint'               => '1.04',
            'B::Terse'              => '1.03',
            'CGI'                   => '3.15_01',
            'CPAN'                  => '1.83_58',
            'CPAN::Debug'           => '4.44',
            'CPAN::FirstTime'       => '4.50',
            'CPAN::HandleConfig'    => '4.31',
            'CPAN::Nox'             => '2.31',
            'CPAN::Tarzip'          => '3.36',
            'CPAN::Version'         => '2.55',
            'Carp'                  => '1.05',
            'Carp::Heavy'           => '1.05',
            'Compress::Zlib'        => '2.000_07',
            'Compress::Zlib::Common'=> '2.000_07',
            'Compress::Zlib::Compress::Gzip::Constants'=> '2.000_07',
            'Compress::Zlib::Compress::Zip::Constants'=> '1.00',
            'Compress::Zlib::CompressPlugin::Deflate'=> '2.000_05',
            'Compress::Zlib::CompressPlugin::Identity'=> '2.000_05',
            'Compress::Zlib::File::GlobMapper'=> '0.000_02',
            'Compress::Zlib::FileConstants'=> '2.000_07',
            'Compress::Zlib::IO::Compress::Base'=> '2.000_05',
            'Compress::Zlib::IO::Compress::Deflate'=> '2.000_07',
            'Compress::Zlib::IO::Compress::Gzip'=> '2.000_07',
            'Compress::Zlib::IO::Compress::RawDeflate'=> '2.000_07',
            'Compress::Zlib::IO::Compress::Zip'=> '2.000_04',
            'Compress::Zlib::IO::Uncompress::AnyInflate'=> '2.000_07',
            'Compress::Zlib::IO::Uncompress::AnyUncompress'=> '2.000_05',
            'Compress::Zlib::IO::Uncompress::Base'=> '2.000_05',
            'Compress::Zlib::IO::Uncompress::Gunzip'=> '2.000_07',
            'Compress::Zlib::IO::Uncompress::Inflate'=> '2.000_07',
            'Compress::Zlib::IO::Uncompress::RawInflate'=> '2.000_07',
            'Compress::Zlib::IO::Uncompress::Unzip'=> '2.000_05',
            'Compress::Zlib::ParseParameters'=> '2.000_07',
            'Compress::Zlib::UncompressPlugin::Identity'=> '2.000_05',
            'Compress::Zlib::UncompressPlugin::Inflate'=> '2.000_05',
            'Config::Extensions'    => '0.01',
            'Cwd'                   => '3.15',
            'Devel::PPPort'         => '3.08',
            'Digest::SHA'           => '5.32',
            'DirHandle'             => '1.01',
            'DynaLoader'            => '1.07',
            'Encode'                => '2.14',
            'Encode::CN::HZ'        => '2.02',
            'Encode::MIME::Header'  => '2.02',
            'English'               => '1.04',
            'Exporter'              => '5.59',
            'Exporter::Heavy'       => '5.59',
            'ExtUtils::CBuilder'    => '0.15',
            'ExtUtils::CBuilder::Base'=> '0.12',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.12',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.12',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.12',
            'ExtUtils::CBuilder::Platform::aix'=> '0.12',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.12',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.12',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.01',
            'ExtUtils::CBuilder::Platform::os2'=> '0.13',
            'ExtUtils::Command::MM' => '0.05_01',
            'ExtUtils::Constant'    => '0.2',
            'ExtUtils::Constant::Base'=> '0.02',
            'ExtUtils::Constant::ProxySubs'=> '0.01',
            'ExtUtils::Constant::XS'=> '0.02',
            'ExtUtils::MM_Any'      => '0.13_01',
            'ExtUtils::MM_Unix'     => '1.50_01',
            'ExtUtils::MakeMaker'   => '6.30_01',
            'ExtUtils::ParseXS'     => '2.15_02',
            'Fatal'                 => '1.04',
            'File::Compare'         => '1.1005',
            'File::Spec'            => '3.15',
            'File::Temp'            => '0.16_01',
            'IO::File'              => '1.13_01',
            'IO::Handle'            => '1.26',
            'IO::Socket'            => '1.29_01',
            'IO::Socket::INET'      => '1.29_02',
            'IO::Socket::UNIX'      => '1.22_01',
            'IO::Zlib'              => '1.04_02',
            'Locale::Maketext'      => '1.10_01',
            'Math::BigInt::FastCalc'=> '0.10',
            'Memoize'               => '1.01_01',
            'Module::CoreList'      => '2.02',
            'Moped::Msg'            => '0.01',
            'NEXT'                  => '0.60_01',
            'Net::Cmd'              => '2.26_01',
            'Net::Domain'           => '2.19_01',
            'Net::Ping'             => '2.31_04',
            'Opcode'                => '1.08',
            'POSIX'                 => '1.10',
            'Pod::Escapes'          => '1.04',
            'Pod::Man'              => '2.04',
            'Pod::Perldoc'          => '3.14_01',
            'Pod::Simple'           => '3.04',
            'Pod::Simple::BlackBox' => undef,
            'Pod::Simple::Checker'  => '2.02',
            'Pod::Simple::Debug'    => undef,
            'Pod::Simple::DumpAsText'=> '2.02',
            'Pod::Simple::DumpAsXML'=> '2.02',
            'Pod::Simple::HTML'     => '3.03',
            'Pod::Simple::HTMLBatch'=> '3.02',
            'Pod::Simple::HTMLLegacy'=> '5.01',
            'Pod::Simple::LinkSection'=> undef,
            'Pod::Simple::Methody'  => '2.02',
            'Pod::Simple::Progress' => '1.01',
            'Pod::Simple::PullParser'=> '2.02',
            'Pod::Simple::PullParserEndToken'=> undef,
            'Pod::Simple::PullParserStartToken'=> undef,
            'Pod::Simple::PullParserTextToken'=> undef,
            'Pod::Simple::PullParserToken'=> '2.02',
            'Pod::Simple::RTF'      => '2.02',
            'Pod::Simple::Search'   => '3.04',
            'Pod::Simple::SimpleTree'=> '2.02',
            'Pod::Simple::Text'     => '2.02',
            'Pod::Simple::TextContent'=> '2.02',
            'Pod::Simple::TiedOutFH'=> undef,
            'Pod::Simple::Transcode'=> undef,
            'Pod::Simple::TranscodeDumb'=> '2.02',
            'Pod::Simple::TranscodeSmart'=> undef,
            'Pod::Simple::XMLOutStream'=> '2.02',
            'Pod::Text'             => '3.01',
            'Pod::Text::Color'      => '2.01',
            'Pod::Text::Overstrike' => '2',
            'Pod::Text::Termcap'    => '2.01',
            'Pod::Usage'            => '1.33_01',
            'SelfLoader'            => '1.0905',
            'Storable'              => '2.15_02',
            'Test::Builder::Module' => '0.03',
            'Text::Balanced'        => '1.95_01',
            'Tie::File'             => '0.97_01',
            'UNIVERSAL'             => '1.03',
            'XS::APItest'           => '0.09',
            'assertions'            => '0.02',
            'assertions::activate'  => '0.02',
            'assertions::compat'    => undef,
            'constant'              => '1.07',
            'encoding::warnings'    => '0.05',
            'feature'               => '1.00',
            're'                    => '0.06',
            'sort'                  => '2.00',
            'version'               => '0.53',
        },
        removed => {
        }
    },
    5.009004 => {
        delta_from => 5.009003,
        changed => {
            'Archive::Tar'          => '1.30_01',
            'AutoLoader'            => '5.61',
            'B'                     => '1.11',
            'B::Bytecode'           => '1.02',
            'B::C'                  => '1.05',
            'B::Concise'            => '0.69',
            'B::Deparse'            => '0.76',
            'B::Lint'               => '1.08',
            'Benchmark'             => '1.08',
            'CGI'                   => '3.20',
            'CGI::Cookie'           => '1.27',
            'CGI::Fast'             => '1.07',
            'CPAN'                  => '1.87_55',
            'CPAN::Debug'           => '5.400561',
            'CPAN::FirstTime'       => '5.400742',
            'CPAN::HandleConfig'    => '5.400740',
            'CPAN::Nox'             => '5.400561',
            'CPAN::Tarzip'          => '5.400714',
            'CPAN::Version'         => '5.400561',
            'Compress::Raw::Zlib'   => '2.000_13',
            'Compress::Zlib'        => '2.000_13',
            'Cwd'                   => '3.19',
            'Devel::PPPort'         => '3.10',
            'Digest'                => '1.15',
            'Digest::SHA'           => '5.43',
            'Encode'                => '2.18_01',
            'Encode::Alias'         => '2.06',
            'Encode::Byte'          => '2.02',
            'Encode::CJKConstants'  => '2.02',
            'Encode::CN'            => '2.02',
            'Encode::CN::HZ'        => '2.04',
            'Encode::Config'        => '2.03',
            'Encode::EBCDIC'        => '2.02',
            'Encode::Encoder'       => '2.01',
            'Encode::Encoding'      => '2.04',
            'Encode::Guess'         => '2.02',
            'Encode::JP'            => '2.03',
            'Encode::JP::H2Z'       => '2.02',
            'Encode::JP::JIS7'      => '2.02',
            'Encode::KR'            => '2.02',
            'Encode::KR::2022_KR'   => '2.02',
            'Encode::MIME::Header'  => '2.04',
            'Encode::MIME::Header::ISO_2022_JP'=> '1.03',
            'Encode::Symbol'        => '2.02',
            'Encode::TW'            => '2.02',
            'Encode::Unicode'       => '2.03',
            'Encode::Unicode::UTF7' => '2.04',
            'ExtUtils::CBuilder'    => '0.18',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.12_01',
            'ExtUtils::Constant::Base'=> '0.03',
            'ExtUtils::Constant::ProxySubs'=> '0.03',
            'ExtUtils::Install'     => '1.41',
            'ExtUtils::Installed'   => '1.41',
            'ExtUtils::MM_Any'      => '0.13_02',
            'ExtUtils::MM_NW5'      => '2.08_01',
            'ExtUtils::MM_Unix'     => '1.5003',
            'ExtUtils::MM_VMS'      => '5.73_03',
            'ExtUtils::MM_Win32'    => '1.12_02',
            'ExtUtils::MM_Win95'    => '0.04_01',
            'ExtUtils::MakeMaker'   => '6.30_02',
            'ExtUtils::Manifest'    => '1.46_01',
            'ExtUtils::Mkbootstrap' => '1.15_01',
            'ExtUtils::Mksymlists'  => '1.19_01',
            'ExtUtils::Packlist'    => '1.41',
            'File::Basename'        => '2.75',
            'File::Find'            => '1.11',
            'File::GlobMapper'      => '0.000_02',
            'File::Spec'            => '3.19',
            'FileCache'             => '1.07',
            'Getopt::Long'          => '2.3501',
            'Hash::Util'            => '0.07',
            'Hash::Util::FieldHash' => '0.01',
            'IO'                    => '1.23_01',
            'IO::Compress::Adapter::Deflate'=> '2.000_13',
            'IO::Compress::Adapter::Identity'=> '2.000_13',
            'IO::Compress::Base'    => '2.000_13',
            'IO::Compress::Base::Common'=> '2.000_13',
            'IO::Compress::Deflate' => '2.000_13',
            'IO::Compress::Gzip'    => '2.000_13',
            'IO::Compress::Gzip::Constants'=> '2.000_13',
            'IO::Compress::RawDeflate'=> '2.000_13',
            'IO::Compress::Zip'     => '2.000_13',
            'IO::Compress::Zip::Constants'=> '2.000_13',
            'IO::Compress::Zlib::Constants'=> '2.000_13',
            'IO::Compress::Zlib::Extra'=> '2.000_13',
            'IO::Dir'               => '1.06',
            'IO::File'              => '1.14',
            'IO::Handle'            => '1.27',
            'IO::Socket'            => '1.30_01',
            'IO::Socket::INET'      => '1.31',
            'IO::Socket::UNIX'      => '1.23',
            'IO::Uncompress::Adapter::Identity'=> '2.000_13',
            'IO::Uncompress::Adapter::Inflate'=> '2.000_13',
            'IO::Uncompress::AnyInflate'=> '2.000_13',
            'IO::Uncompress::AnyUncompress'=> '2.000_13',
            'IO::Uncompress::Base'  => '2.000_13',
            'IO::Uncompress::Gunzip'=> '2.000_13',
            'IO::Uncompress::Inflate'=> '2.000_13',
            'IO::Uncompress::RawInflate'=> '2.000_13',
            'IO::Uncompress::Unzip' => '2.000_13',
            'MIME::Base64'          => '3.07_01',
            'Math::Complex'         => '1.36',
            'Math::Trig'            => '1.04',
            'Module::Build'         => '0.2805',
            'Module::Build::Base'   => undef,
            'Module::Build::Compat' => '0.03',
            'Module::Build::ConfigData'=> undef,
            'Module::Build::Cookbook'=> undef,
            'Module::Build::ModuleInfo'=> undef,
            'Module::Build::Notes'  => undef,
            'Module::Build::PPMMaker'=> undef,
            'Module::Build::Platform::Amiga'=> undef,
            'Module::Build::Platform::Default'=> undef,
            'Module::Build::Platform::EBCDIC'=> undef,
            'Module::Build::Platform::MPEiX'=> undef,
            'Module::Build::Platform::MacOS'=> undef,
            'Module::Build::Platform::RiscOS'=> undef,
            'Module::Build::Platform::Unix'=> undef,
            'Module::Build::Platform::VMS'=> undef,
            'Module::Build::Platform::VOS'=> undef,
            'Module::Build::Platform::Windows'=> undef,
            'Module::Build::Platform::aix'=> undef,
            'Module::Build::Platform::cygwin'=> undef,
            'Module::Build::Platform::darwin'=> undef,
            'Module::Build::Platform::os2'=> undef,
            'Module::Build::PodParser'=> undef,
            'Module::Build::Version'=> '0',
            'Module::Build::YAML'   => '0.50',
            'Module::CoreList'      => '2.08',
            'Module::Load'          => '0.10',
            'Module::Loaded'        => '0.01',
            'Package::Constants'    => '0.01',
            'Pod::Html'             => '1.07',
            'Pod::Man'              => '2.09',
            'Pod::Text'             => '3.07',
            'Pod::Text::Color'      => '2.03',
            'Pod::Text::Termcap'    => '2.03',
            'SDBM_File'             => '1.06',
            'Shell'                 => '0.7',
            'Sys::Syslog'           => '0.17',
            'Term::ANSIColor'       => '1.11',
            'Test::Builder'         => '0.33',
            'Test::Builder::Tester' => '1.04',
            'Test::Harness'         => '2.62',
            'Test::Harness::Util'   => '0.01',
            'Test::More'            => '0.64',
            'Test::Simple'          => '0.64',
            'Text::Balanced'        => '1.98_01',
            'Text::ParseWords'      => '3.25',
            'Text::Tabs'            => '2007.071101',
            'Text::Wrap'            => '2006.0711',
            'Tie::RefHash'          => '1.34_01',
            'Time::HiRes'           => '1.87',
            'Time::Local'           => '1.13',
            'Time::gmtime'          => '1.03',
            'UNIVERSAL'             => '1.04',
            'Unicode::Normalize'    => '1.01',
            'Win32API::File'        => '0.1001',
            'Win32API::File::ExtUtils::Myconst2perl'=> '1',
            'assertions'            => '0.03',
            'assertions::compat'    => '0.02',
            'autouse'               => '1.06',
            'diagnostics'           => '1.16',
            'encoding'              => '2.04',
            'encoding::warnings'    => '0.10',
            'feature'               => '1.01',
            're'                    => '0.0601',
            'threads'               => '1.38',
            'threads::shared'       => '0.94_01',
            'version'               => '0.67',
        },
        removed => {
            'Compress::Zlib::Common'=> 1,
            'Compress::Zlib::Compress::Gzip::Constants'=> 1,
            'Compress::Zlib::Compress::Zip::Constants'=> 1,
            'Compress::Zlib::CompressPlugin::Deflate'=> 1,
            'Compress::Zlib::CompressPlugin::Identity'=> 1,
            'Compress::Zlib::File::GlobMapper'=> 1,
            'Compress::Zlib::FileConstants'=> 1,
            'Compress::Zlib::IO::Compress::Base'=> 1,
            'Compress::Zlib::IO::Compress::Deflate'=> 1,
            'Compress::Zlib::IO::Compress::Gzip'=> 1,
            'Compress::Zlib::IO::Compress::RawDeflate'=> 1,
            'Compress::Zlib::IO::Compress::Zip'=> 1,
            'Compress::Zlib::IO::Uncompress::AnyInflate'=> 1,
            'Compress::Zlib::IO::Uncompress::AnyUncompress'=> 1,
            'Compress::Zlib::IO::Uncompress::Base'=> 1,
            'Compress::Zlib::IO::Uncompress::Gunzip'=> 1,
            'Compress::Zlib::IO::Uncompress::Inflate'=> 1,
            'Compress::Zlib::IO::Uncompress::RawInflate'=> 1,
            'Compress::Zlib::IO::Uncompress::Unzip'=> 1,
            'Compress::Zlib::ParseParameters'=> 1,
            'Compress::Zlib::UncompressPlugin::Identity'=> 1,
            'Compress::Zlib::UncompressPlugin::Inflate'=> 1,
        }
    },
    5.009005 => {
        delta_from => 5.009004,
        changed => {
            'Archive::Extract'      => '0.22_01',
            'Archive::Tar'          => '1.32',
            'Attribute::Handlers'   => '0.78_06',
            'AutoLoader'            => '5.63',
            'AutoSplit'             => '1.05',
            'B'                     => '1.16',
            'B::Concise'            => '0.72',
            'B::Debug'              => '1.05',
            'B::Deparse'            => '0.82',
            'B::Lint'               => '1.09',
            'B::Terse'              => '1.05',
            'Benchmark'             => '1.1',
            'CGI'                   => '3.29',
            'CGI::Cookie'           => '1.28',
            'CGI::Util'             => '1.5_01',
            'CPAN'                  => '1.9102',
            'CPAN::Debug'           => '5.400955',
            'CPAN::FirstTime'       => '5.401669',
            'CPAN::HandleConfig'    => '5.401744',
            'CPAN::Kwalify'         => '5.401418',
            'CPAN::Nox'             => '5.400844',
            'CPAN::Queue'           => '5.401704',
            'CPAN::Tarzip'          => '5.401717',
            'CPAN::Version'         => '5.401387',
            'CPANPLUS'              => '0.81_01',
            'CPANPLUS::Backend'     => undef,
            'CPANPLUS::Backend::RV' => undef,
            'CPANPLUS::Config'      => undef,
            'CPANPLUS::Configure'   => undef,
            'CPANPLUS::Configure::Setup'=> undef,
            'CPANPLUS::Dist'        => undef,
            'CPANPLUS::Dist::Base'  => '0.01',
            'CPANPLUS::Dist::Build' => '0.06_01',
            'CPANPLUS::Dist::Build::Constants'=> '0.01',
            'CPANPLUS::Dist::MM'    => undef,
            'CPANPLUS::Dist::Sample'=> undef,
            'CPANPLUS::Error'       => undef,
            'CPANPLUS::Internals'   => '0.81_01',
            'CPANPLUS::Internals::Constants'=> '0.01',
            'CPANPLUS::Internals::Constants::Report'=> '0.01',
            'CPANPLUS::Internals::Extract'=> undef,
            'CPANPLUS::Internals::Fetch'=> undef,
            'CPANPLUS::Internals::Report'=> undef,
            'CPANPLUS::Internals::Search'=> undef,
            'CPANPLUS::Internals::Source'=> undef,
            'CPANPLUS::Internals::Utils'=> undef,
            'CPANPLUS::Internals::Utils::Autoflush'=> undef,
            'CPANPLUS::Module'      => undef,
            'CPANPLUS::Module::Author'=> undef,
            'CPANPLUS::Module::Author::Fake'=> undef,
            'CPANPLUS::Module::Checksums'=> undef,
            'CPANPLUS::Module::Fake'=> undef,
            'CPANPLUS::Module::Signature'=> undef,
            'CPANPLUS::Selfupdate'  => undef,
            'CPANPLUS::Shell'       => undef,
            'CPANPLUS::Shell::Classic'=> '0.0562',
            'CPANPLUS::Shell::Default'=> '0.81_01',
            'CPANPLUS::Shell::Default::Plugins::Remote'=> undef,
            'CPANPLUS::Shell::Default::Plugins::Source'=> undef,
            'CPANPLUS::inc'         => undef,
            'Carp'                  => '1.07',
            'Carp::Heavy'           => '1.07',
            'Compress::Raw::Zlib'   => '2.005',
            'Compress::Zlib'        => '2.005',
            'Cwd'                   => '3.25',
            'DBM_Filter'            => '0.02',
            'DB_File'               => '1.815',
            'Data::Dumper'          => '2.121_13',
            'Devel::InnerPackage'   => '0.3',
            'Devel::PPPort'         => '3.11_01',
            'Digest::MD5'           => '2.36_01',
            'Digest::SHA'           => '5.44',
            'DynaLoader'            => '1.08',
            'Encode'                => '2.23',
            'Encode::Alias'         => '2.07',
            'Encode::Byte'          => '2.03',
            'Encode::Config'        => '2.04',
            'Encode::Encoding'      => '2.05',
            'Encode::GSM0338'       => '2.00',
            'Encode::JP::JIS7'      => '2.03',
            'Encode::MIME::Header'  => '2.05',
            'Encode::MIME::Name'    => '1.01',
            'Encode::Unicode'       => '2.05',
            'Errno'                 => '1.10',
            'Exporter'              => '5.60',
            'Exporter::Heavy'       => '5.60',
            'ExtUtils::CBuilder'    => '0.19',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.13',
            'ExtUtils::Command'     => '1.13',
            'ExtUtils::Command::MM' => '0.07',
            'ExtUtils::Constant::Base'=> '0.04',
            'ExtUtils::Install'     => '1.41_01',
            'ExtUtils::Liblist'     => '1.03',
            'ExtUtils::Liblist::Kid'=> '1.33',
            'ExtUtils::MM'          => '0.07',
            'ExtUtils::MM_AIX'      => '0.05',
            'ExtUtils::MM_Any'      => '0.15',
            'ExtUtils::MM_BeOS'     => '1.07',
            'ExtUtils::MM_Cygwin'   => '1.1',
            'ExtUtils::MM_DOS'      => '0.04',
            'ExtUtils::MM_MacOS'    => '1.1',
            'ExtUtils::MM_NW5'      => '2.1',
            'ExtUtils::MM_OS2'      => '1.07',
            'ExtUtils::MM_QNX'      => '0.04',
            'ExtUtils::MM_UWIN'     => '0.04',
            'ExtUtils::MM_Unix'     => '1.54_01',
            'ExtUtils::MM_VMS'      => '5.76',
            'ExtUtils::MM_VOS'      => '0.04',
            'ExtUtils::MM_Win32'    => '1.15',
            'ExtUtils::MM_Win95'    => '0.06',
            'ExtUtils::MY'          => '0.03',
            'ExtUtils::MakeMaker'   => '6.36',
            'ExtUtils::MakeMaker::Config'=> '0.04',
            'ExtUtils::MakeMaker::bytes'=> '0.03',
            'ExtUtils::MakeMaker::vmsish'=> '0.03',
            'ExtUtils::Manifest'    => '1.51_01',
            'ExtUtils::Mkbootstrap' => '1.17',
            'ExtUtils::Mksymlists'  => '1.21',
            'ExtUtils::ParseXS'     => '2.18',
            'ExtUtils::XSSymSet'    => '1.1',
            'ExtUtils::testlib'     => '1.17',
            'Fatal'                 => '1.05',
            'Fcntl'                 => '1.06',
            'File::Basename'        => '2.76',
            'File::Copy'            => '2.10',
            'File::Fetch'           => '0.10',
            'File::Glob'            => '1.06',
            'File::Path'            => '2.01',
            'File::Spec'            => '3.25',
            'File::Spec::Cygwin'    => '1.1_01',
            'File::Spec::VMS'       => '1.4_01',
            'File::Temp'            => '0.18',
            'Filter::Util::Call'    => '1.0602',
            'FindBin'               => '1.49',
            'Getopt::Long'          => '2.36',
            'Hash::Util::FieldHash' => '1.01',
            'IO::Compress::Adapter::Deflate'=> '2.005',
            'IO::Compress::Adapter::Identity'=> '2.005',
            'IO::Compress::Base'    => '2.005',
            'IO::Compress::Base::Common'=> '2.005',
            'IO::Compress::Deflate' => '2.005',
            'IO::Compress::Gzip'    => '2.005',
            'IO::Compress::Gzip::Constants'=> '2.005',
            'IO::Compress::RawDeflate'=> '2.005',
            'IO::Compress::Zip'     => '2.005',
            'IO::Compress::Zip::Constants'=> '2.005',
            'IO::Compress::Zlib::Constants'=> '2.005',
            'IO::Compress::Zlib::Extra'=> '2.005',
            'IO::Uncompress::Adapter::Identity'=> '2.005',
            'IO::Uncompress::Adapter::Inflate'=> '2.005',
            'IO::Uncompress::AnyInflate'=> '2.005',
            'IO::Uncompress::AnyUncompress'=> '2.005',
            'IO::Uncompress::Base'  => '2.005',
            'IO::Uncompress::Gunzip'=> '2.005',
            'IO::Uncompress::Inflate'=> '2.005',
            'IO::Uncompress::RawInflate'=> '2.005',
            'IO::Uncompress::Unzip' => '2.005',
            'IO::Zlib'              => '1.05_01',
            'IPC::Cmd'              => '0.36_01',
            'List::Util'            => '1.19',
            'Locale::Maketext::Simple'=> '0.18',
            'Log::Message'          => '0.01',
            'Log::Message::Config'  => '0.01',
            'Log::Message::Handlers'=> undef,
            'Log::Message::Item'    => undef,
            'Log::Message::Simple'  => '0.0201',
            'Math::BigFloat'        => '1.58',
            'Math::BigInt'          => '1.87',
            'Math::BigInt::Calc'    => '0.51',
            'Math::BigInt::FastCalc'=> '0.15_01',
            'Math::BigRat'          => '0.19',
            'Math::Complex'         => '1.37',
            'Memoize'               => '1.01_02',
            'Module::Build'         => '0.2808',
            'Module::Build::Config' => undef,
            'Module::Build::Version'=> '0.7203',
            'Module::CoreList'      => '2.12',
            'Module::Load::Conditional'=> '0.16',
            'Module::Pluggable'     => '3.6',
            'Module::Pluggable::Object'=> '3.6',
            'NDBM_File'             => '1.07',
            'Net::Cmd'              => '2.28',
            'Net::Config'           => '1.11',
            'Net::Domain'           => '2.20',
            'Net::FTP'              => '2.77',
            'Net::FTP::A'           => '1.18',
            'Net::NNTP'             => '2.24',
            'Net::POP3'             => '2.29',
            'Net::SMTP'             => '2.31',
            'ODBM_File'             => '1.07',
            'OS2::DLL'              => '1.03',
            'Object::Accessor'      => '0.32',
            'Opcode'                => '1.09',
            'POSIX'                 => '1.13',
            'Params::Check'         => '0.26',
            'PerlIO::encoding'      => '0.10',
            'PerlIO::scalar'        => '0.05',
            'PerlIO::via'           => '0.04',
            'Pod::Html'             => '1.08',
            'Pod::Man'              => '2.12',
            'Pod::ParseUtils'       => '1.35',
            'Pod::Parser'           => '1.35',
            'Pod::Select'           => '1.35',
            'Pod::Simple'           => '3.05',
            'Pod::Text'             => '3.08',
            'Pod::Usage'            => '1.35',
            'Scalar::Util'          => '1.19',
            'SelfLoader'            => '1.11',
            'Shell'                 => '0.72_01',
            'Socket'                => '1.79',
            'Storable'              => '2.16',
            'Switch'                => '2.13',
            'Sys::Syslog'           => '0.18_01',
            'Term::ANSIColor'       => '1.12',
            'Term::UI'              => '0.14_01',
            'Term::UI::History'     => undef,
            'Test::Builder'         => '0.70',
            'Test::Builder::Module' => '0.68',
            'Test::Builder::Tester' => '1.07',
            'Test::Harness'         => '2.64',
            'Test::Harness::Results'=> '0.01',
            'Test::More'            => '0.70',
            'Test::Simple'          => '0.70',
            'Text::Balanced'        => '2.0.0',
            'Text::Soundex'         => '3.02',
            'Text::Tabs'            => '2007.1117',
            'Text::Wrap'            => '2006.1117',
            'Thread'                => '3.02',
            'Tie::File'             => '0.97_02',
            'Tie::Hash::NamedCapture'=> '0.06',
            'Tie::Memoize'          => '1.1',
            'Tie::RefHash'          => '1.37',
            'Time::HiRes'           => '1.9707',
            'Time::Local'           => '1.17',
            'Time::Piece'           => '1.11_02',
            'Time::Seconds'         => undef,
            'Unicode'               => '5.0.0',
            'Unicode::Normalize'    => '1.02',
            'Unicode::UCD'          => '0.25',
            'VMS::DCLsym'           => '1.03',
            'Win32'                 => '0.30',
            'Win32API::File'        => '0.1001_01',
            'Win32CORE'             => '0.02',
            'XS::APItest'           => '0.12',
            'XSLoader'              => '0.08',
            'attributes'            => '0.08',
            'base'                  => '2.12',
            'bigint'                => '0.22',
            'bignum'                => '0.22',
            'bigrat'                => '0.22',
            'bytes'                 => '1.03',
            'charnames'             => '1.06',
            'constant'              => '1.10',
            'diagnostics'           => '1.17',
            'encoding'              => '2.06',
            'encoding::warnings'    => '0.11',
            'feature'               => '1.10',
            'fields'                => '2.12',
            'less'                  => '0.02',
            'mro'                   => '1.00',
            'overload'              => '1.06',
            're'                    => '0.08',
            'sigtrap'               => '1.04',
            'sort'                  => '2.01',
            'strict'                => '1.04',
            'threads'               => '1.63',
            'threads::shared'       => '1.12',
            'utf8'                  => '1.07',
            'version'               => '0.7203',
            'warnings'              => '1.06',
        },
        removed => {
            'B::Asmdata'            => 1,
            'B::Assembler'          => 1,
            'B::Bblock'             => 1,
            'B::Bytecode'           => 1,
            'B::C'                  => 1,
            'B::CC'                 => 1,
            'B::Disassembler'       => 1,
            'B::Stackobj'           => 1,
            'B::Stash'              => 1,
            'ByteLoader'            => 1,
            'Thread::Signal'        => 1,
            'Thread::Specific'      => 1,
            'assertions'            => 1,
            'assertions::activate'  => 1,
            'assertions::compat'    => 1,
        }
    },
    5.01 => {
        delta_from => 5.009005,
        changed => {
            'Archive::Extract'      => '0.24',
            'Archive::Tar'          => '1.38',
            'Attribute::Handlers'   => '0.79',
            'B'                     => '1.17',
            'B::Concise'            => '0.74',
            'B::Deparse'            => '0.83',
            'CPAN'                  => '1.9205',
            'CPAN::API::HOWTO'      => undef,
            'CPAN::Debug'           => '5.402212',
            'CPAN::DeferedCode'     => '5.50',
            'CPAN::FirstTime'       => '5.402229',
            'CPAN::HandleConfig'    => '5.402212',
            'CPAN::Nox'             => '5.402411',
            'CPAN::Queue'           => '5.402212',
            'CPAN::Tarzip'          => '5.402213',
            'CPAN::Version'         => '5.5',
            'CPANPLUS'              => '0.84',
            'CPANPLUS::Dist::Build' => '0.06_02',
            'CPANPLUS::Internals'   => '0.84',
            'CPANPLUS::Shell::Default'=> '0.84',
            'CPANPLUS::Shell::Default::Plugins::CustomSource'=> undef,
            'Carp'                  => '1.08',
            'Carp::Heavy'           => '1.08',
            'Compress::Raw::Zlib'   => '2.008',
            'Compress::Zlib'        => '2.008',
            'Cwd'                   => '3.2501',
            'DB_File'               => '1.816_1',
            'Data::Dumper'          => '2.121_14',
            'Devel::PPPort'         => '3.13',
            'Digest::SHA'           => '5.45',
            'Exporter'              => '5.62',
            'Exporter::Heavy'       => '5.62',
            'ExtUtils::CBuilder'    => '0.21',
            'ExtUtils::CBuilder::Base'=> '0.21',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.21',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.22',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.21',
            'ExtUtils::CBuilder::Platform::aix'=> '0.21',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.21',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.21',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.21',
            'ExtUtils::CBuilder::Platform::os2'=> '0.21',
            'ExtUtils::Command::MM' => '6.42',
            'ExtUtils::Constant::ProxySubs'=> '0.05',
            'ExtUtils::Embed'       => '1.27',
            'ExtUtils::Install'     => '1.44',
            'ExtUtils::Installed'   => '1.43',
            'ExtUtils::Liblist'     => '6.42',
            'ExtUtils::Liblist::Kid'=> '6.42',
            'ExtUtils::MM'          => '6.42',
            'ExtUtils::MM_AIX'      => '6.42',
            'ExtUtils::MM_Any'      => '6.42',
            'ExtUtils::MM_BeOS'     => '6.42',
            'ExtUtils::MM_Cygwin'   => '6.42',
            'ExtUtils::MM_DOS'      => '6.42',
            'ExtUtils::MM_MacOS'    => '6.42',
            'ExtUtils::MM_NW5'      => '6.42',
            'ExtUtils::MM_OS2'      => '6.42',
            'ExtUtils::MM_QNX'      => '6.42',
            'ExtUtils::MM_UWIN'     => '6.42',
            'ExtUtils::MM_Unix'     => '6.42',
            'ExtUtils::MM_VMS'      => '6.42',
            'ExtUtils::MM_VOS'      => '6.42',
            'ExtUtils::MM_Win32'    => '6.42',
            'ExtUtils::MM_Win95'    => '6.42',
            'ExtUtils::MY'          => '6.42',
            'ExtUtils::MakeMaker'   => '6.42',
            'ExtUtils::MakeMaker::Config'=> '6.42',
            'ExtUtils::MakeMaker::bytes'=> '6.42',
            'ExtUtils::MakeMaker::vmsish'=> '6.42',
            'ExtUtils::Mkbootstrap' => '6.42',
            'ExtUtils::Mksymlists'  => '6.42',
            'ExtUtils::Packlist'    => '1.43',
            'ExtUtils::ParseXS'     => '2.18_02',
            'ExtUtils::testlib'     => '6.42',
            'File::Copy'            => '2.11',
            'File::Fetch'           => '0.14',
            'File::Find'            => '1.12',
            'File::Path'            => '2.04',
            'File::Spec'            => '3.2501',
            'File::Spec::Cygwin'    => '3.2501',
            'File::Spec::Epoc'      => '3.2501',
            'File::Spec::Functions' => '3.2501',
            'File::Spec::Mac'       => '3.2501',
            'File::Spec::OS2'       => '3.2501',
            'File::Spec::Unix'      => '3.2501',
            'File::Spec::VMS'       => '3.2501',
            'File::Spec::Win32'     => '3.2501',
            'Filter::Util::Call'    => '1.07',
            'Getopt::Long'          => '2.37',
            'Hash::Util::FieldHash' => '1.03',
            'IO::Compress::Adapter::Deflate'=> '2.008',
            'IO::Compress::Adapter::Identity'=> '2.008',
            'IO::Compress::Base'    => '2.008',
            'IO::Compress::Base::Common'=> '2.008',
            'IO::Compress::Deflate' => '2.008',
            'IO::Compress::Gzip'    => '2.008',
            'IO::Compress::Gzip::Constants'=> '2.008',
            'IO::Compress::RawDeflate'=> '2.008',
            'IO::Compress::Zip'     => '2.008',
            'IO::Compress::Zip::Constants'=> '2.008',
            'IO::Compress::Zlib::Constants'=> '2.008',
            'IO::Compress::Zlib::Extra'=> '2.008',
            'IO::Uncompress::Adapter::Identity'=> '2.008',
            'IO::Uncompress::Adapter::Inflate'=> '2.008',
            'IO::Uncompress::AnyInflate'=> '2.008',
            'IO::Uncompress::AnyUncompress'=> '2.008',
            'IO::Uncompress::Base'  => '2.008',
            'IO::Uncompress::Gunzip'=> '2.008',
            'IO::Uncompress::Inflate'=> '2.008',
            'IO::Uncompress::RawInflate'=> '2.008',
            'IO::Uncompress::Unzip' => '2.008',
            'IO::Zlib'              => '1.07',
            'IPC::Cmd'              => '0.40_1',
            'IPC::SysV'             => '1.05',
            'Locale::Maketext'      => '1.12',
            'Log::Message::Simple'  => '0.04',
            'Math::BigFloat'        => '1.59',
            'Math::BigInt'          => '1.88',
            'Math::BigInt::Calc'    => '0.52',
            'Math::BigInt::FastCalc'=> '0.16',
            'Math::BigRat'          => '0.21',
            'Module::Build'         => '0.2808_01',
            'Module::Build::Base'   => '0.2808_01',
            'Module::Build::Compat' => '0.2808_01',
            'Module::Build::Config' => '0.2808_01',
            'Module::Build::Dumper' => undef,
            'Module::Build::ModuleInfo'=> '0.2808_01',
            'Module::Build::Notes'  => '0.2808_01',
            'Module::Build::PPMMaker'=> '0.2808_01',
            'Module::Build::Platform::Amiga'=> '0.2808_01',
            'Module::Build::Platform::Default'=> '0.2808_01',
            'Module::Build::Platform::EBCDIC'=> '0.2808_01',
            'Module::Build::Platform::MPEiX'=> '0.2808_01',
            'Module::Build::Platform::MacOS'=> '0.2808_01',
            'Module::Build::Platform::RiscOS'=> '0.2808_01',
            'Module::Build::Platform::Unix'=> '0.2808_01',
            'Module::Build::Platform::VMS'=> '0.2808_01',
            'Module::Build::Platform::VOS'=> '0.2808_01',
            'Module::Build::Platform::Windows'=> '0.2808_01',
            'Module::Build::Platform::aix'=> '0.2808_01',
            'Module::Build::Platform::cygwin'=> '0.2808_01',
            'Module::Build::Platform::darwin'=> '0.2808_01',
            'Module::Build::Platform::os2'=> '0.2808_01',
            'Module::Build::PodParser'=> '0.2808_01',
            'Module::CoreList'      => '2.13',
            'Module::Load'          => '0.12',
            'Module::Load::Conditional'=> '0.22',
            'Net::Cmd'              => '2.29',
            'Net::Ping'             => '2.33',
            'Opcode'                => '1.11',
            'Pod::Checker'          => '1.43_01',
            'Pod::Man'              => '2.16',
            'Pod::Perldoc'          => '3.14_02',
            'Socket'                => '1.80',
            'Storable'              => '2.18',
            'Sys::Syslog'           => '0.22',
            'Sys::Syslog::win32::Win32'=> undef,
            'Term::Cap'             => '1.12',
            'Term::ReadLine'        => '1.03',
            'Term::UI'              => '0.18',
            'Test::Builder'         => '0.72',
            'Test::Builder::Module' => '0.72',
            'Test::Builder::Tester' => '1.09',
            'Test::Harness::Straps' => '0.26_01',
            'Test::More'            => '0.72',
            'Test::Simple'          => '0.72',
            'Text::ParseWords'      => '3.26',
            'Text::Soundex'         => '3.03',
            'Tie::StdHandle'        => undef,
            'Time::HiRes'           => '1.9711',
            'Time::Local'           => '1.18',
            'Time::Piece'           => '1.12',
            'VMS::Filespec'         => '1.12',
            'Win32'                 => '0.34',
            'base'                  => '2.13',
            'constant'              => '1.13',
            'feature'               => '1.11',
            'fields'                => '2.13',
            'filetest'              => '1.02',
            'open'                  => '1.06',
            'threads'               => '1.67',
            'threads::shared'       => '1.14',
            'version'               => '0.74',
        },
        removed => {
        }
    },
    5.010001 => {
        delta_from => 5.01,
        changed => {
            'App::Prove'            => '3.17',
            'App::Prove::State'     => '3.17',
            'App::Prove::State::Result'=> '3.17',
            'App::Prove::State::Result::Test'=> '3.17',
            'Archive::Extract'      => '0.34',
            'Archive::Tar'          => '1.52',
            'Attribute::Handlers'   => '0.85',
            'AutoLoader'            => '5.68',
            'AutoSplit'             => '1.06',
            'B'                     => '1.22',
            'B::Concise'            => '0.76',
            'B::Debug'              => '1.11',
            'B::Deparse'            => '0.89',
            'B::Lint'               => '1.11',
            'B::Lint::Debug'        => undef,
            'B::Xref'               => '1.02',
            'Benchmark'             => '1.11',
            'CGI'                   => '3.43',
            'CGI::Carp'             => '1.30_01',
            'CGI::Cookie'           => '1.29',
            'CPAN'                  => '1.9402',
            'CPAN::Author'          => '5.5',
            'CPAN::Bundle'          => '5.5',
            'CPAN::CacheMgr'        => '5.5',
            'CPAN::Complete'        => '5.5',
            'CPAN::Debug'           => '5.5',
            'CPAN::DeferredCode'    => '5.50',
            'CPAN::Distribution'    => '1.93',
            'CPAN::Distroprefs'     => '6',
            'CPAN::Distrostatus'    => '5.5',
            'CPAN::Exception::RecursiveDependency'=> '5.5',
            'CPAN::Exception::blocked_urllist'=> '1.0',
            'CPAN::Exception::yaml_not_installed'=> '5.5',
            'CPAN::FTP'             => '5.5001',
            'CPAN::FTP::netrc'      => '1.00',
            'CPAN::FirstTime'       => '5.53',
            'CPAN::HandleConfig'    => '5.5',
            'CPAN::Index'           => '1.93',
            'CPAN::InfoObj'         => '5.5',
            'CPAN::Kwalify'         => '5.50',
            'CPAN::LWP::UserAgent'  => '1.00',
            'CPAN::Module'          => '5.5',
            'CPAN::Nox'             => '5.50',
            'CPAN::Prompt'          => '5.5',
            'CPAN::Queue'           => '5.5',
            'CPAN::Shell'           => '5.5',
            'CPAN::Tarzip'          => '5.501',
            'CPAN::URL'             => '5.5',
            'CPANPLUS'              => '0.88',
            'CPANPLUS::Dist::Autobundle'=> undef,
            'CPANPLUS::Dist::Base'  => undef,
            'CPANPLUS::Dist::Build' => '0.36',
            'CPANPLUS::Dist::Build::Constants'=> '0.36',
            'CPANPLUS::Internals'   => '0.88',
            'CPANPLUS::Internals::Constants'=> undef,
            'CPANPLUS::Internals::Constants::Report'=> undef,
            'CPANPLUS::Internals::Source::Memory'=> undef,
            'CPANPLUS::Internals::Source::SQLite'=> undef,
            'CPANPLUS::Internals::Source::SQLite::Tie'=> undef,
            'CPANPLUS::Shell::Default'=> '0.88',
            'Carp'                  => '1.11',
            'Carp::Heavy'           => '1.11',
            'Compress::Raw::Bzip2'  => '2.020',
            'Compress::Raw::Zlib'   => '2.020',
            'Compress::Zlib'        => '2.020',
            'Cwd'                   => '3.30',
            'DB'                    => '1.02',
            'DBM_Filter::compress'  => '0.02',
            'DBM_Filter::encode'    => '0.02',
            'DBM_Filter::int32'     => '0.02',
            'DBM_Filter::null'      => '0.02',
            'DBM_Filter::utf8'      => '0.02',
            'DB_File'               => '1.820',
            'Data::Dumper'          => '2.124',
            'Devel::DProf'          => '20080331.00',
            'Devel::PPPort'         => '3.19',
            'Devel::Peek'           => '1.04',
            'Digest'                => '1.16',
            'Digest::MD5'           => '2.39',
            'Digest::SHA'           => '5.47',
            'Digest::base'          => '1.16',
            'Digest::file'          => '1.16',
            'DirHandle'             => '1.03',
            'Dumpvalue'             => '1.13',
            'DynaLoader'            => '1.10',
            'Encode'                => '2.35',
            'Encode::Alias'         => '2.12',
            'Encode::CN::HZ'        => '2.05',
            'Encode::Config'        => '2.05',
            'Encode::GSM0338'       => '2.01',
            'Encode::Guess'         => '2.03',
            'Encode::JP::JIS7'      => '2.04',
            'Encode::MIME::Header'  => '2.11',
            'Encode::Unicode'       => '2.06',
            'Errno'                 => '1.11',
            'Exporter'              => '5.63',
            'Exporter::Heavy'       => '5.63',
            'ExtUtils::CBuilder'    => '0.2602',
            'ExtUtils::CBuilder::Base'=> '0.2602',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.2602',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.2602',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.2602',
            'ExtUtils::CBuilder::Platform::aix'=> '0.2602',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.2602',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.2602',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.2602',
            'ExtUtils::CBuilder::Platform::os2'=> '0.2602',
            'ExtUtils::Command'     => '1.16',
            'ExtUtils::Command::MM' => '6.55_02',
            'ExtUtils::Constant'    => '0.22',
            'ExtUtils::Constant::ProxySubs'=> '0.06',
            'ExtUtils::Constant::Utils'=> '0.02',
            'ExtUtils::Constant::XS'=> '0.03',
            'ExtUtils::Embed'       => '1.28',
            'ExtUtils::Install'     => '1.54',
            'ExtUtils::Installed'   => '1.999_001',
            'ExtUtils::Liblist'     => '6.55_02',
            'ExtUtils::Liblist::Kid'=> '6.5502',
            'ExtUtils::MM'          => '6.55_02',
            'ExtUtils::MM_AIX'      => '6.55_02',
            'ExtUtils::MM_Any'      => '6.55_02',
            'ExtUtils::MM_BeOS'     => '6.55_02',
            'ExtUtils::MM_Cygwin'   => '6.55_02',
            'ExtUtils::MM_DOS'      => '6.5502',
            'ExtUtils::MM_Darwin'   => '6.55_02',
            'ExtUtils::MM_MacOS'    => '6.5502',
            'ExtUtils::MM_NW5'      => '6.55_02',
            'ExtUtils::MM_OS2'      => '6.55_02',
            'ExtUtils::MM_QNX'      => '6.55_02',
            'ExtUtils::MM_UWIN'     => '6.5502',
            'ExtUtils::MM_Unix'     => '6.55_02',
            'ExtUtils::MM_VMS'      => '6.55_02',
            'ExtUtils::MM_VOS'      => '6.55_02',
            'ExtUtils::MM_Win32'    => '6.55_02',
            'ExtUtils::MM_Win95'    => '6.55_02',
            'ExtUtils::MY'          => '6.5502',
            'ExtUtils::MakeMaker'   => '6.55_02',
            'ExtUtils::MakeMaker::Config'=> '6.55_02',
            'ExtUtils::Manifest'    => '1.56',
            'ExtUtils::Mkbootstrap' => '6.55_02',
            'ExtUtils::Mksymlists'  => '6.55_02',
            'ExtUtils::ParseXS'     => '2.2002',
            'ExtUtils::testlib'     => '6.5502',
            'Fatal'                 => '2.06_01',
            'File::Basename'        => '2.77',
            'File::CheckTree'       => '4.4',
            'File::Compare'         => '1.1006',
            'File::Copy'            => '2.14',
            'File::DosGlob'         => '1.01',
            'File::Fetch'           => '0.20',
            'File::Find'            => '1.14',
            'File::GlobMapper'      => '1.000',
            'File::Path'            => '2.07_03',
            'File::Spec'            => '3.30',
            'File::Spec::Cygwin'    => '3.30',
            'File::Spec::Epoc'      => '3.30',
            'File::Spec::Functions' => '3.30',
            'File::Spec::Mac'       => '3.30',
            'File::Spec::OS2'       => '3.30',
            'File::Spec::Unix'      => '3.30',
            'File::Spec::VMS'       => '3.30',
            'File::Spec::Win32'     => '3.30',
            'File::Temp'            => '0.22',
            'File::stat'            => '1.01',
            'FileCache'             => '1.08',
            'FileHandle'            => '2.02',
            'Filter::Simple'        => '0.84',
            'Filter::Util::Call'    => '1.08',
            'FindBin'               => '1.50',
            'GDBM_File'             => '1.09',
            'Getopt::Long'          => '2.38',
            'Getopt::Std'           => '1.06',
            'Hash::Util::FieldHash' => '1.04',
            'I18N::Collate'         => '1.01',
            'IO'                    => '1.25',
            'IO::Compress::Adapter::Bzip2'=> '2.020',
            'IO::Compress::Adapter::Deflate'=> '2.020',
            'IO::Compress::Adapter::Identity'=> '2.020',
            'IO::Compress::Base'    => '2.020',
            'IO::Compress::Base::Common'=> '2.020',
            'IO::Compress::Bzip2'   => '2.020',
            'IO::Compress::Deflate' => '2.020',
            'IO::Compress::Gzip'    => '2.020',
            'IO::Compress::Gzip::Constants'=> '2.020',
            'IO::Compress::RawDeflate'=> '2.020',
            'IO::Compress::Zip'     => '2.020',
            'IO::Compress::Zip::Constants'=> '2.020',
            'IO::Compress::Zlib::Constants'=> '2.020',
            'IO::Compress::Zlib::Extra'=> '2.020',
            'IO::Dir'               => '1.07',
            'IO::Handle'            => '1.28',
            'IO::Socket'            => '1.31',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.020',
            'IO::Uncompress::Adapter::Identity'=> '2.020',
            'IO::Uncompress::Adapter::Inflate'=> '2.020',
            'IO::Uncompress::AnyInflate'=> '2.020',
            'IO::Uncompress::AnyUncompress'=> '2.020',
            'IO::Uncompress::Base'  => '2.020',
            'IO::Uncompress::Bunzip2'=> '2.020',
            'IO::Uncompress::Gunzip'=> '2.020',
            'IO::Uncompress::Inflate'=> '2.020',
            'IO::Uncompress::RawInflate'=> '2.020',
            'IO::Uncompress::Unzip' => '2.020',
            'IO::Zlib'              => '1.09',
            'IPC::Cmd'              => '0.46',
            'IPC::Msg'              => '2.01',
            'IPC::Open2'            => '1.03',
            'IPC::Open3'            => '1.04',
            'IPC::Semaphore'        => '2.01',
            'IPC::SharedMem'        => '2.01',
            'IPC::SysV'             => '2.01',
            'List::Util'            => '1.21',
            'List::Util::PP'        => '1.21',
            'List::Util::XS'        => '1.21',
            'Locale::Maketext'      => '1.13',
            'Locale::Maketext::Guts'=> '1.13',
            'Locale::Maketext::GutsLoader'=> '1.13',
            'Log::Message'          => '0.02',
            'MIME::Base64'          => '3.08',
            'MIME::QuotedPrint'     => '3.08',
            'Math::BigFloat'        => '1.60',
            'Math::BigInt'          => '1.89',
            'Math::BigInt::FastCalc'=> '0.19',
            'Math::BigRat'          => '0.22',
            'Math::Complex'         => '1.56',
            'Math::Trig'            => '1.2',
            'Memoize'               => '1.01_03',
            'Module::Build'         => '0.340201',
            'Module::Build::Base'   => '0.340201',
            'Module::Build::Compat' => '0.340201',
            'Module::Build::Config' => '0.340201',
            'Module::Build::Cookbook'=> '0.340201',
            'Module::Build::Dumper' => '0.340201',
            'Module::Build::ModuleInfo'=> '0.340201',
            'Module::Build::Notes'  => '0.340201',
            'Module::Build::PPMMaker'=> '0.340201',
            'Module::Build::Platform::Amiga'=> '0.340201',
            'Module::Build::Platform::Default'=> '0.340201',
            'Module::Build::Platform::EBCDIC'=> '0.340201',
            'Module::Build::Platform::MPEiX'=> '0.340201',
            'Module::Build::Platform::MacOS'=> '0.340201',
            'Module::Build::Platform::RiscOS'=> '0.340201',
            'Module::Build::Platform::Unix'=> '0.340201',
            'Module::Build::Platform::VMS'=> '0.340201',
            'Module::Build::Platform::VOS'=> '0.340201',
            'Module::Build::Platform::Windows'=> '0.340201',
            'Module::Build::Platform::aix'=> '0.340201',
            'Module::Build::Platform::cygwin'=> '0.340201',
            'Module::Build::Platform::darwin'=> '0.340201',
            'Module::Build::Platform::os2'=> '0.340201',
            'Module::Build::PodParser'=> '0.340201',
            'Module::Build::Version'=> '0.77',
            'Module::CoreList'      => '2.18',
            'Module::Load'          => '0.16',
            'Module::Load::Conditional'=> '0.30',
            'Module::Loaded'        => '0.02',
            'Module::Pluggable'     => '3.9',
            'Module::Pluggable::Object'=> '3.9',
            'NDBM_File'             => '1.08',
            'NEXT'                  => '0.64',
            'Net::Ping'             => '2.36',
            'O'                     => '1.01',
            'OS2::Process'          => '1.03',
            'OS2::REXX'             => '1.04',
            'Object::Accessor'      => '0.34',
            'POSIX'                 => '1.17',
            'Package::Constants'    => '0.02',
            'Parse::CPAN::Meta'     => '1.39',
            'PerlIO'                => '1.06',
            'PerlIO::encoding'      => '0.11',
            'PerlIO::scalar'        => '0.07',
            'PerlIO::via'           => '0.07',
            'Pod::Checker'          => '1.45',
            'Pod::Find'             => '1.35',
            'Pod::Html'             => '1.09',
            'Pod::InputObjects'     => '1.31',
            'Pod::Man'              => '2.22',
            'Pod::ParseLink'        => '1.09',
            'Pod::ParseUtils'       => '1.36',
            'Pod::Parser'           => '1.37',
            'Pod::Perldoc'          => '3.14_04',
            'Pod::PlainText'        => '2.04',
            'Pod::Select'           => '1.36',
            'Pod::Simple'           => '3.07',
            'Pod::Simple::XHTML'    => '3.04',
            'Pod::Text'             => '3.13',
            'Pod::Text::Color'      => '2.05',
            'Pod::Text::Overstrike' => '2.03',
            'Pod::Text::Termcap'    => '2.05',
            'Pod::Usage'            => '1.36',
            'Safe'                  => '2.18',
            'Scalar::Util'          => '1.21',
            'Scalar::Util::PP'      => '1.21',
            'SelectSaver'           => '1.02',
            'SelfLoader'            => '1.17',
            'Socket'                => '1.82',
            'Storable'              => '2.20',
            'Switch'                => '2.14',
            'Symbol'                => '1.07',
            'Sys::Syslog'           => '0.27',
            'TAP::Base'             => '3.17',
            'TAP::Formatter::Base'  => '3.17',
            'TAP::Formatter::Color' => '3.17',
            'TAP::Formatter::Console'=> '3.17',
            'TAP::Formatter::Console::ParallelSession'=> '3.17',
            'TAP::Formatter::Console::Session'=> '3.17',
            'TAP::Formatter::File'  => '3.17',
            'TAP::Formatter::File::Session'=> '3.17',
            'TAP::Formatter::Session'=> '3.17',
            'TAP::Harness'          => '3.17',
            'TAP::Object'           => '3.17',
            'TAP::Parser'           => '3.17',
            'TAP::Parser::Aggregator'=> '3.17',
            'TAP::Parser::Grammar'  => '3.17',
            'TAP::Parser::Iterator' => '3.17',
            'TAP::Parser::Iterator::Array'=> '3.17',
            'TAP::Parser::Iterator::Process'=> '3.17',
            'TAP::Parser::Iterator::Stream'=> '3.17',
            'TAP::Parser::IteratorFactory'=> '3.17',
            'TAP::Parser::Multiplexer'=> '3.17',
            'TAP::Parser::Result'   => '3.17',
            'TAP::Parser::Result::Bailout'=> '3.17',
            'TAP::Parser::Result::Comment'=> '3.17',
            'TAP::Parser::Result::Plan'=> '3.17',
            'TAP::Parser::Result::Pragma'=> '3.17',
            'TAP::Parser::Result::Test'=> '3.17',
            'TAP::Parser::Result::Unknown'=> '3.17',
            'TAP::Parser::Result::Version'=> '3.17',
            'TAP::Parser::Result::YAML'=> '3.17',
            'TAP::Parser::ResultFactory'=> '3.17',
            'TAP::Parser::Scheduler'=> '3.17',
            'TAP::Parser::Scheduler::Job'=> '3.17',
            'TAP::Parser::Scheduler::Spinner'=> '3.17',
            'TAP::Parser::Source'   => '3.17',
            'TAP::Parser::Source::Perl'=> '3.17',
            'TAP::Parser::Utils'    => '3.17',
            'TAP::Parser::YAMLish::Reader'=> '3.17',
            'TAP::Parser::YAMLish::Writer'=> '3.17',
            'Term::ANSIColor'       => '2.00',
            'Term::ReadLine'        => '1.04',
            'Term::UI'              => '0.20',
            'Test'                  => '1.25_02',
            'Test::Builder'         => '0.92',
            'Test::Builder::Module' => '0.92',
            'Test::Builder::Tester' => '1.18',
            'Test::Builder::Tester::Color'=> '1.18',
            'Test::Harness'         => '3.17',
            'Test::More'            => '0.92',
            'Test::Simple'          => '0.92',
            'Text::ParseWords'      => '3.27',
            'Text::Tabs'            => '2009.0305',
            'Text::Wrap'            => '2009.0305',
            'Thread::Queue'         => '2.11',
            'Thread::Semaphore'     => '2.09',
            'Tie::Handle'           => '4.2',
            'Tie::Hash'             => '1.03',
            'Tie::RefHash'          => '1.38',
            'Tie::Scalar'           => '1.01',
            'Tie::StdHandle'        => '4.2',
            'Time::HiRes'           => '1.9719',
            'Time::Local'           => '1.1901',
            'Time::Piece'           => '1.15',
            'UNIVERSAL'             => '1.05',
            'Unicode'               => '5.1.0',
            'Unicode::Normalize'    => '1.03',
            'Unicode::UCD'          => '0.27',
            'VMS::Stdio'            => '2.4',
            'Win32'                 => '0.39',
            'Win32API::File'        => '0.1101',
            'XS::APItest'           => '0.15',
            'XS::Typemap'           => '0.03',
            'XSLoader'              => '0.10',
            'attributes'            => '0.09',
            'attrs'                 => '1.03',
            'autodie'               => '2.06_01',
            'autodie::exception'    => '2.06_01',
            'autodie::exception::system'=> '2.06_01',
            'autodie::hints'        => '2.06_01',
            'base'                  => '2.14',
            'bigint'                => '0.23',
            'bignum'                => '0.23',
            'bigrat'                => '0.23',
            'blib'                  => '1.04',
            'charnames'             => '1.07',
            'constant'              => '1.17',
            'encoding'              => '2.6_01',
            'feature'               => '1.13',
            'fields'                => '2.14',
            'lib'                   => '0.62',
            'mro'                   => '1.01',
            'open'                  => '1.07',
            'ops'                   => '1.02',
            'overload'              => '1.07',
            'overload::numbers'     => undef,
            'overloading'           => '0.01',
            'parent'                => '0.221',
            're'                    => '0.09',
            'threads'               => '1.72',
            'threads::shared'       => '1.29',
            'version'               => '0.77',
        },
        removed => {
            'CPAN::API::HOWTO'      => 1,
            'CPAN::DeferedCode'     => 1,
            'CPANPLUS::inc'         => 1,
            'ExtUtils::MakeMaker::bytes'=> 1,
            'ExtUtils::MakeMaker::vmsish'=> 1,
            'Test::Harness::Assert' => 1,
            'Test::Harness::Iterator'=> 1,
            'Test::Harness::Point'  => 1,
            'Test::Harness::Results'=> 1,
            'Test::Harness::Straps' => 1,
            'Test::Harness::Util'   => 1,
        }
    },
    5.011 => {
        delta_from => 5.010001,
        changed => {
            'Archive::Tar'          => '1.54',
            'Attribute::Handlers'   => '0.87',
            'AutoLoader'            => '5.70',
            'B::Deparse'            => '0.91',
            'B::Lint'               => '1.11_01',
            'B::Lint::Debug'        => '0.01',
            'CGI'                   => '3.45',
            'CGI::Apache'           => '1.01',
            'CGI::Carp'             => '3.45',
            'CGI::Pretty'           => '3.44',
            'CGI::Switch'           => '1.01',
            'CGI::Util'             => '3.45',
            'CPAN'                  => '1.94_51',
            'CPAN::Distribution'    => '1.94',
            'CPAN::FTP'             => '5.5002',
            'CPAN::Index'           => '1.94',
            'CPAN::LWP::UserAgent'  => '1.94',
            'CPANPLUS::Dist::Build' => '0.40',
            'CPANPLUS::Dist::Build::Constants'=> '0.40',
            'Carp'                  => '1.12',
            'Carp::Heavy'           => '1.12',
            'Class::ISA'            => '0.36',
            'Compress::Raw::Bzip2'  => '2.021',
            'Compress::Raw::Zlib'   => '2.021',
            'Compress::Zlib'        => '2.021',
            'Cwd'                   => '3.3002',
            'Data::Dumper'          => '2.125',
            'Encode'                => '2.37',
            'Exporter'              => '5.64',
            'Exporter::Heavy'       => '5.64',
            'ExtUtils::ParseXS'     => '2.200403',
            'File::Basename'        => '2.78',
            'File::Copy'            => '2.16',
            'File::stat'            => '1.02',
            'IO'                    => '1.25_01',
            'IO::Compress::Adapter::Bzip2'=> '2.021',
            'IO::Compress::Adapter::Deflate'=> '2.021',
            'IO::Compress::Adapter::Identity'=> '2.021',
            'IO::Compress::Base'    => '2.021',
            'IO::Compress::Base::Common'=> '2.021',
            'IO::Compress::Bzip2'   => '2.021',
            'IO::Compress::Deflate' => '2.021',
            'IO::Compress::Gzip'    => '2.021',
            'IO::Compress::Gzip::Constants'=> '2.021',
            'IO::Compress::RawDeflate'=> '2.021',
            'IO::Compress::Zip'     => '2.021',
            'IO::Compress::Zip::Constants'=> '2.021',
            'IO::Compress::Zlib::Constants'=> '2.021',
            'IO::Compress::Zlib::Extra'=> '2.021',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.021',
            'IO::Uncompress::Adapter::Identity'=> '2.021',
            'IO::Uncompress::Adapter::Inflate'=> '2.021',
            'IO::Uncompress::AnyInflate'=> '2.021',
            'IO::Uncompress::AnyUncompress'=> '2.021',
            'IO::Uncompress::Base'  => '2.021',
            'IO::Uncompress::Bunzip2'=> '2.021',
            'IO::Uncompress::Gunzip'=> '2.021',
            'IO::Uncompress::Inflate'=> '2.021',
            'IO::Uncompress::RawInflate'=> '2.021',
            'IO::Uncompress::Unzip' => '2.021',
            'IO::Zlib'              => '1.10',
            'IPC::Cmd'              => '0.50',
            'IPC::Open3'            => '1.05',
            'Locale::Maketext::Simple'=> '0.21',
            'Log::Message::Simple'  => '0.06',
            'Math::BigInt'          => '1.89_01',
            'Math::BigRat'          => '0.24',
            'Module::Build'         => '0.35',
            'Module::Build::Base'   => '0.35',
            'Module::Build::Compat' => '0.35',
            'Module::Build::Config' => '0.35',
            'Module::Build::Cookbook'=> '0.35',
            'Module::Build::Dumper' => '0.35',
            'Module::Build::ModuleInfo'=> '0.35',
            'Module::Build::Notes'  => '0.35',
            'Module::Build::PPMMaker'=> '0.35',
            'Module::Build::Platform::Amiga'=> '0.35',
            'Module::Build::Platform::Default'=> '0.35',
            'Module::Build::Platform::EBCDIC'=> '0.35',
            'Module::Build::Platform::MPEiX'=> '0.35',
            'Module::Build::Platform::MacOS'=> '0.35',
            'Module::Build::Platform::RiscOS'=> '0.35',
            'Module::Build::Platform::Unix'=> '0.35',
            'Module::Build::Platform::VMS'=> '0.35',
            'Module::Build::Platform::VOS'=> '0.35',
            'Module::Build::Platform::Windows'=> '0.35',
            'Module::Build::Platform::aix'=> '0.35',
            'Module::Build::Platform::cygwin'=> '0.35',
            'Module::Build::Platform::darwin'=> '0.35',
            'Module::Build::Platform::os2'=> '0.35',
            'Module::Build::PodParser'=> '0.35',
            'Module::CoreList'      => '2.19',
            'Module::Loaded'        => '0.06',
            'Opcode'                => '1.13',
            'PerlIO::via'           => '0.08',
            'Pod::Perldoc'          => '3.15_01',
            'Pod::Plainer'          => '1.01',
            'Safe'                  => '2.19',
            'Socket'                => '1.84',
            'Switch'                => '2.14_01',
            'Term::ANSIColor'       => '2.02',
            'Term::ReadLine'        => '1.05',
            'Text::Balanced'        => '2.02',
            'Text::Soundex'         => '3.03_01',
            'Time::Local'           => '1.1901_01',
            'Unicode::Collate'      => '0.52_01',
            'attributes'            => '0.12',
            'constant'              => '1.19',
            'deprecate'             => '0.01',
            'overload'              => '1.08',
            'parent'                => '0.223',
            're'                    => '0.10',
            'threads'               => '1.74',
            'threads::shared'       => '1.31',
            'warnings'              => '1.07',
        },
        removed => {
            'attrs'                 => 1,
        }
    },
    5.011001 => {
        delta_from => 5.011,
        changed => {
            'B'                     => '1.23',
            'B::Concise'            => '0.77',
            'B::Deparse'            => '0.92',
            'CGI'                   => '3.48',
            'CGI::Pretty'           => '3.46',
            'CGI::Util'             => '3.48',
            'CPANPLUS'              => '0.89_03',
            'CPANPLUS::Internals'   => '0.89_03',
            'CPANPLUS::Shell::Default'=> '0.89_03',
            'Carp'                  => '1.13',
            'Carp::Heavy'           => '1.13',
            'ExtUtils::CBuilder'    => '0.260301',
            'ExtUtils::CBuilder::Base'=> '0.260301',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.260301',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.260301',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.260301',
            'ExtUtils::CBuilder::Platform::aix'=> '0.260301',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.260301',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.260301',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.260301',
            'ExtUtils::CBuilder::Platform::os2'=> '0.260301',
            'ExtUtils::Install'     => '1.55',
            'ExtUtils::Manifest'    => '1.57',
            'ExtUtils::Packlist'    => '1.44',
            'ExtUtils::ParseXS'     => '2.21',
            'File::Glob'            => '1.07',
            'File::Path'            => '2.08',
            'IO'                    => '1.25_02',
            'Module::CoreList'      => '2.21',
            'OS2::DLL'              => '1.04',
            'OS2::Process'          => '1.04',
            'Object::Accessor'      => '0.36',
            'Opcode'                => '1.15',
            'POSIX'                 => '1.18',
            'Parse::CPAN::Meta'     => '1.40',
            'PerlIO::via'           => '0.09',
            'Pod::Simple'           => '3.08',
            'Socket'                => '1.85',
            'Storable'              => '2.22',
            'Switch'                => '2.15',
            'Test::Builder'         => '0.94',
            'Test::Builder::Module' => '0.94',
            'Test::More'            => '0.94',
            'Test::Simple'          => '0.94',
            'XS::APItest'           => '0.16',
            'mro'                   => '1.02',
            'overload'              => '1.09',
            'threads::shared'       => '1.32',
        },
        removed => {
        }
    },
    5.011002 => {
        delta_from => 5.011001,
        changed => {
            'B::Concise'            => '0.78',
            'B::Deparse'            => '0.93',
            'CPANPLUS'              => '0.89_09',
            'CPANPLUS::Dist::Build' => '0.44',
            'CPANPLUS::Dist::Build::Constants'=> '0.44',
            'CPANPLUS::Internals'   => '0.89_09',
            'CPANPLUS::Shell::Default'=> '0.89_09',
            'Carp'                  => '1.14',
            'Carp::Heavy'           => '1.14',
            'Compress::Zlib'        => '2.022',
            'DBM_Filter'            => '0.03',
            'Encode'                => '2.38',
            'Encode::Byte'          => '2.04',
            'Encode::CN'            => '2.03',
            'Encode::JP'            => '2.04',
            'Encode::KR'            => '2.03',
            'Encode::TW'            => '2.03',
            'Encode::Unicode'       => '2.07',
            'Env'                   => '1.01',
            'Exporter'              => '5.64_01',
            'Exporter::Heavy'       => '5.64_01',
            'ExtUtils::CBuilder'    => '0.27',
            'ExtUtils::CBuilder::Base'=> '0.27',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.27',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.27',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.27',
            'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.27',
            'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.27',
            'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.27',
            'ExtUtils::CBuilder::Platform::aix'=> '0.27',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.27',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.27',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.27',
            'ExtUtils::CBuilder::Platform::os2'=> '0.27',
            'File::Fetch'           => '0.22',
            'I18N::LangTags::Detect'=> '1.04',
            'I18N::Langinfo'        => '0.03',
            'IO::Compress::Adapter::Bzip2'=> '2.022',
            'IO::Compress::Adapter::Deflate'=> '2.022',
            'IO::Compress::Adapter::Identity'=> '2.022',
            'IO::Compress::Base'    => '2.022',
            'IO::Compress::Base::Common'=> '2.022',
            'IO::Compress::Bzip2'   => '2.022',
            'IO::Compress::Deflate' => '2.022',
            'IO::Compress::Gzip'    => '2.022',
            'IO::Compress::Gzip::Constants'=> '2.022',
            'IO::Compress::RawDeflate'=> '2.022',
            'IO::Compress::Zip'     => '2.022',
            'IO::Compress::Zip::Constants'=> '2.022',
            'IO::Compress::Zlib::Constants'=> '2.022',
            'IO::Compress::Zlib::Extra'=> '2.022',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.022',
            'IO::Uncompress::Adapter::Identity'=> '2.022',
            'IO::Uncompress::Adapter::Inflate'=> '2.022',
            'IO::Uncompress::AnyInflate'=> '2.022',
            'IO::Uncompress::AnyUncompress'=> '2.022',
            'IO::Uncompress::Base'  => '2.022',
            'IO::Uncompress::Bunzip2'=> '2.022',
            'IO::Uncompress::Gunzip'=> '2.022',
            'IO::Uncompress::Inflate'=> '2.022',
            'IO::Uncompress::RawInflate'=> '2.022',
            'IO::Uncompress::Unzip' => '2.022',
            'IPC::Cmd'              => '0.54',
            'List::Util'            => '1.22',
            'List::Util::PP'        => '1.22',
            'List::Util::XS'        => '1.22',
            'Locale::Maketext'      => '1.14',
            'Module::Build'         => '0.35_09',
            'Module::Build::Base'   => '0.35_09',
            'Module::Build::Compat' => '0.35_09',
            'Module::Build::Config' => '0.35_09',
            'Module::Build::Cookbook'=> '0.35_09',
            'Module::Build::Dumper' => '0.35_09',
            'Module::Build::ModuleInfo'=> '0.35_09',
            'Module::Build::Notes'  => '0.35_09',
            'Module::Build::PPMMaker'=> '0.35_09',
            'Module::Build::Platform::Amiga'=> '0.35_09',
            'Module::Build::Platform::Default'=> '0.35_09',
            'Module::Build::Platform::EBCDIC'=> '0.35_09',
            'Module::Build::Platform::MPEiX'=> '0.35_09',
            'Module::Build::Platform::MacOS'=> '0.35_09',
            'Module::Build::Platform::RiscOS'=> '0.35_09',
            'Module::Build::Platform::Unix'=> '0.35_09',
            'Module::Build::Platform::VMS'=> '0.35_09',
            'Module::Build::Platform::VOS'=> '0.35_09',
            'Module::Build::Platform::Windows'=> '0.35_09',
            'Module::Build::Platform::aix'=> '0.35_09',
            'Module::Build::Platform::cygwin'=> '0.35_09',
            'Module::Build::Platform::darwin'=> '0.35_09',
            'Module::Build::Platform::os2'=> '0.35_09',
            'Module::Build::PodParser'=> '0.35_09',
            'Module::Build::YAML'   => '1.40',
            'Module::CoreList'      => '2.23',
            'Module::Load::Conditional'=> '0.34',
            'Pod::Simple'           => '3.10',
            'Pod::Simple::XHTML'    => '3.10',
            'Scalar::Util'          => '1.22',
            'Scalar::Util::PP'      => '1.22',
            'Switch'                => '2.16',
            'XS::APItest'           => '0.17',
            'XS::APItest::KeywordRPN'=> '0.003',
            'base'                  => '2.15',
            'diagnostics'           => '1.18',
            'fields'                => '2.15',
            'inc::latest'           => '0.35_09',
            'legacy'                => '1.00',
            'overload'              => '1.10',
        },
        removed => {
        }
    },
    5.011003 => {
        delta_from => 5.011002,
        changed => {
            'App::Cpan'             => '1.570001',
            'Archive::Extract'      => '0.36',
            'CPAN'                  => '1.94_5301',
            'CPAN::FTP'             => '5.5004',
            'CPAN::FirstTime'       => '5.530001',
            'CPAN::Mirrors'         => '1.770001',
            'CPANPLUS'              => '0.90',
            'CPANPLUS::Internals'   => '0.90',
            'CPANPLUS::Shell::Default'=> '0.90',
            'Cwd'                   => '3.31',
            'Encode'                => '2.39',
            'ExtUtils::Command::MM' => '6.56',
            'ExtUtils::Liblist'     => '6.56',
            'ExtUtils::Liblist::Kid'=> '6.56',
            'ExtUtils::MM'          => '6.56',
            'ExtUtils::MM_AIX'      => '6.56',
            'ExtUtils::MM_Any'      => '6.56',
            'ExtUtils::MM_BeOS'     => '6.56',
            'ExtUtils::MM_Cygwin'   => '6.56',
            'ExtUtils::MM_DOS'      => '6.56',
            'ExtUtils::MM_Darwin'   => '6.56',
            'ExtUtils::MM_MacOS'    => '6.56',
            'ExtUtils::MM_NW5'      => '6.56',
            'ExtUtils::MM_OS2'      => '6.56',
            'ExtUtils::MM_QNX'      => '6.56',
            'ExtUtils::MM_UWIN'     => '6.56',
            'ExtUtils::MM_Unix'     => '6.56',
            'ExtUtils::MM_VMS'      => '6.56',
            'ExtUtils::MM_VOS'      => '6.56',
            'ExtUtils::MM_Win32'    => '6.56',
            'ExtUtils::MM_Win95'    => '6.56',
            'ExtUtils::MY'          => '6.56',
            'ExtUtils::MakeMaker'   => '6.56',
            'ExtUtils::MakeMaker::Config'=> '6.56',
            'ExtUtils::Mkbootstrap' => '6.56',
            'ExtUtils::Mksymlists'  => '6.56',
            'ExtUtils::testlib'     => '6.56',
            'File::Find'            => '1.15',
            'File::Path'            => '2.08_01',
            'File::Spec'            => '3.31',
            'Module::Build'         => '0.36',
            'Module::Build::Base'   => '0.36',
            'Module::Build::Compat' => '0.36',
            'Module::Build::Config' => '0.36',
            'Module::Build::Cookbook'=> '0.36',
            'Module::Build::Dumper' => '0.36',
            'Module::Build::ModuleInfo'=> '0.36',
            'Module::Build::Notes'  => '0.36',
            'Module::Build::PPMMaker'=> '0.36',
            'Module::Build::Platform::Amiga'=> '0.36',
            'Module::Build::Platform::Default'=> '0.36',
            'Module::Build::Platform::EBCDIC'=> '0.36',
            'Module::Build::Platform::MPEiX'=> '0.36',
            'Module::Build::Platform::MacOS'=> '0.36',
            'Module::Build::Platform::RiscOS'=> '0.36',
            'Module::Build::Platform::Unix'=> '0.36',
            'Module::Build::Platform::VMS'=> '0.36',
            'Module::Build::Platform::VOS'=> '0.36',
            'Module::Build::Platform::Windows'=> '0.36',
            'Module::Build::Platform::aix'=> '0.36',
            'Module::Build::Platform::cygwin'=> '0.36',
            'Module::Build::Platform::darwin'=> '0.36',
            'Module::Build::Platform::os2'=> '0.36',
            'Module::Build::PodParser'=> '0.36',
            'Module::CoreList'      => '2.24',
            'POSIX'                 => '1.19',
            'Pod::Simple'           => '3.13',
            'Pod::Simple::BlackBox' => '3.13',
            'Pod::Simple::Checker'  => '3.13',
            'Pod::Simple::Debug'    => '3.13',
            'Pod::Simple::DumpAsText'=> '3.13',
            'Pod::Simple::DumpAsXML'=> '3.13',
            'Pod::Simple::HTML'     => '3.13',
            'Pod::Simple::HTMLBatch'=> '3.13',
            'Pod::Simple::LinkSection'=> '3.13',
            'Pod::Simple::Methody'  => '3.13',
            'Pod::Simple::Progress' => '3.13',
            'Pod::Simple::PullParser'=> '3.13',
            'Pod::Simple::PullParserEndToken'=> '3.13',
            'Pod::Simple::PullParserStartToken'=> '3.13',
            'Pod::Simple::PullParserTextToken'=> '3.13',
            'Pod::Simple::PullParserToken'=> '3.13',
            'Pod::Simple::RTF'      => '3.13',
            'Pod::Simple::Search'   => '3.13',
            'Pod::Simple::SimpleTree'=> '3.13',
            'Pod::Simple::Text'     => '3.13',
            'Pod::Simple::TextContent'=> '3.13',
            'Pod::Simple::TiedOutFH'=> '3.13',
            'Pod::Simple::Transcode'=> '3.13',
            'Pod::Simple::TranscodeDumb'=> '3.13',
            'Pod::Simple::TranscodeSmart'=> '3.13',
            'Pod::Simple::XHTML'    => '3.13',
            'Pod::Simple::XMLOutStream'=> '3.13',
            'Safe'                  => '2.20',
            'Unicode'               => '5.2.0',
            'constant'              => '1.20',
            'diagnostics'           => '1.19',
            'feature'               => '1.14',
            'inc::latest'           => '0.36',
            'threads'               => '1.75',
            'warnings'              => '1.08',
        },
        removed => {
            'legacy'                => 1,
        }
    },
    5.011004 => {
        delta_from => 5.011003,
        changed => {
            'App::Cpan'             => '1.5701',
            'Archive::Extract'      => '0.38',
            'B::Deparse'            => '0.94',
            'CPAN'                  => '1.94_54',
            'CPAN::FirstTime'       => '5.53',
            'CPAN::Mirrors'         => '1.77',
            'Carp'                  => '1.15',
            'Carp::Heavy'           => '1.15',
            'Compress::Raw::Bzip2'  => '2.024',
            'Compress::Raw::Zlib'   => '2.024',
            'Compress::Zlib'        => '2.024',
            'File::Copy'            => '2.17',
            'File::Fetch'           => '0.24',
            'GDBM_File'             => '1.10',
            'IO::Compress::Adapter::Bzip2'=> '2.024',
            'IO::Compress::Adapter::Deflate'=> '2.024',
            'IO::Compress::Adapter::Identity'=> '2.024',
            'IO::Compress::Base'    => '2.024',
            'IO::Compress::Base::Common'=> '2.024',
            'IO::Compress::Bzip2'   => '2.024',
            'IO::Compress::Deflate' => '2.024',
            'IO::Compress::Gzip'    => '2.024',
            'IO::Compress::Gzip::Constants'=> '2.024',
            'IO::Compress::RawDeflate'=> '2.024',
            'IO::Compress::Zip'     => '2.024',
            'IO::Compress::Zip::Constants'=> '2.024',
            'IO::Compress::Zlib::Constants'=> '2.024',
            'IO::Compress::Zlib::Extra'=> '2.024',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.024',
            'IO::Uncompress::Adapter::Identity'=> '2.024',
            'IO::Uncompress::Adapter::Inflate'=> '2.024',
            'IO::Uncompress::AnyInflate'=> '2.024',
            'IO::Uncompress::AnyUncompress'=> '2.024',
            'IO::Uncompress::Base'  => '2.024',
            'IO::Uncompress::Bunzip2'=> '2.024',
            'IO::Uncompress::Gunzip'=> '2.024',
            'IO::Uncompress::Inflate'=> '2.024',
            'IO::Uncompress::RawInflate'=> '2.024',
            'IO::Uncompress::Unzip' => '2.024',
            'Module::Build'         => '0.3603',
            'Module::Build::Base'   => '0.3603',
            'Module::Build::Compat' => '0.3603',
            'Module::Build::Config' => '0.3603',
            'Module::Build::Cookbook'=> '0.3603',
            'Module::Build::Dumper' => '0.3603',
            'Module::Build::ModuleInfo'=> '0.3603',
            'Module::Build::Notes'  => '0.3603',
            'Module::Build::PPMMaker'=> '0.3603',
            'Module::Build::Platform::Amiga'=> '0.3603',
            'Module::Build::Platform::Default'=> '0.3603',
            'Module::Build::Platform::EBCDIC'=> '0.3603',
            'Module::Build::Platform::MPEiX'=> '0.3603',
            'Module::Build::Platform::MacOS'=> '0.3603',
            'Module::Build::Platform::RiscOS'=> '0.3603',
            'Module::Build::Platform::Unix'=> '0.3603',
            'Module::Build::Platform::VMS'=> '0.3603',
            'Module::Build::Platform::VOS'=> '0.3603',
            'Module::Build::Platform::Windows'=> '0.3603',
            'Module::Build::Platform::aix'=> '0.3603',
            'Module::Build::Platform::cygwin'=> '0.3603',
            'Module::Build::Platform::darwin'=> '0.3603',
            'Module::Build::Platform::os2'=> '0.3603',
            'Module::Build::PodParser'=> '0.3603',
            'Module::CoreList'      => '2.25',
            'PerlIO::encoding'      => '0.12',
            'Safe'                  => '2.21',
            'UNIVERSAL'             => '1.06',
            'feature'               => '1.15',
            'inc::latest'           => '0.3603',
            'less'                  => '0.03',
            're'                    => '0.11',
            'version'               => '0.81',
            'warnings'              => '1.09',
        },
        removed => {
        }
    },
    5.011005 => {
        delta_from => 5.011004,
        changed => {
            'B::Debug'              => '1.12',
            'CPAN'                  => '1.94_56',
            'CPAN::Debug'           => '5.5001',
            'CPAN::Distribution'    => '1.9456',
            'CPAN::FirstTime'       => '5.5301',
            'CPAN::HandleConfig'    => '5.5001',
            'CPAN::Shell'           => '5.5001',
            'CPAN::Tarzip'          => '5.5011',
            'CPANPLUS::Dist::Build' => '0.46',
            'CPANPLUS::Dist::Build::Constants'=> '0.46',
            'Module::CoreList'      => '2.26',
            'Pod::Man'              => '2.23',
            'Pod::ParseLink'        => '1.10',
            'Pod::Perldoc'          => '3.15_02',
            'Pod::Plainer'          => '1.02',
            'Pod::Text'             => '3.14',
            'Pod::Text::Color'      => '2.06',
            'Pod::Text::Overstrike' => '2.04',
            'Pod::Text::Termcap'    => '2.06',
            'Safe'                  => '2.22',
            'Socket'                => '1.86',
            'version'               => '0.82',
        },
        removed => {
        }
    },
    5.012 => {
        delta_from => 5.011005,
        changed => {
            'B::Deparse'            => '0.96',
            'CPAN::Distribution'    => '1.9456_01',
            'Module::CoreList'      => '2.29',
            'Safe'                  => '2.25',
            'Socket'                => '1.87',
            'Tie::Scalar'           => '1.02',
            'Time::Piece'           => '1.15_01',
            'bytes'                 => '1.04',
            'feature'               => '1.16',
            'utf8'                  => '1.08',
        },
        removed => {
        }
    },
    5.012001 => {
        delta_from => 5.012,
        changed => {
            'B::Deparse'            => '0.97',
            'CGI'                   => '3.49',
            'CGI::Fast'             => '1.08',
            'Carp'                  => '1.16',
            'Carp::Heavy'           => '1.16',
            'File::Copy'            => '2.18',
            'Module::CoreList'      => '2.32',
            'Pod::Functions'        => '1.04',
            'Pod::Simple'           => '3.14',
            'Pod::Simple::BlackBox' => '3.14',
            'Pod::Simple::Checker'  => '3.14',
            'Pod::Simple::Debug'    => '3.14',
            'Pod::Simple::DumpAsText'=> '3.14',
            'Pod::Simple::DumpAsXML'=> '3.14',
            'Pod::Simple::HTML'     => '3.14',
            'Pod::Simple::HTMLBatch'=> '3.14',
            'Pod::Simple::LinkSection'=> '3.14',
            'Pod::Simple::Methody'  => '3.14',
            'Pod::Simple::Progress' => '3.14',
            'Pod::Simple::PullParser'=> '3.14',
            'Pod::Simple::PullParserEndToken'=> '3.14',
            'Pod::Simple::PullParserStartToken'=> '3.14',
            'Pod::Simple::PullParserTextToken'=> '3.14',
            'Pod::Simple::PullParserToken'=> '3.14',
            'Pod::Simple::RTF'      => '3.14',
            'Pod::Simple::Search'   => '3.14',
            'Pod::Simple::SimpleTree'=> '3.14',
            'Pod::Simple::Text'     => '3.14',
            'Pod::Simple::TextContent'=> '3.14',
            'Pod::Simple::TiedOutFH'=> '3.14',
            'Pod::Simple::Transcode'=> '3.14',
            'Pod::Simple::TranscodeDumb'=> '3.14',
            'Pod::Simple::TranscodeSmart'=> '3.14',
            'Pod::Simple::XHTML'    => '3.14',
            'Pod::Simple::XMLOutStream'=> '3.14',
            'Safe'                  => '2.27',
        },
        removed => {
        }
    },
    5.012002 => {
        delta_from => 5.012001,
        changed => {
            'Carp'                  => '1.17',
            'Carp::Heavy'           => '1.17',
            'File::Spec'            => '3.31_01',
            'Module::CoreList'      => '2.38',
            'Module::Load::Conditional'=> '0.38',
            'PerlIO::scalar'        => '0.08',
        },
        removed => {
        }
    },
    5.012003 => {
        delta_from => 5.012002,
        changed => {
            'B::Deparse'            => '0.9701',
            'Module::Build::Platform::cygwin'=> '0.360301',
            'Module::CoreList'      => '2.43',
            'Socket'                => '1.87_01',
        },
        removed => {
        }
    },
    5.012004 => {
        delta_from => 5.012003,
        changed => {
            'Module::CoreList'      => '2.50',
        },
        removed => {
        }
    },
    5.012005 => {
        delta_from => 5.012004,
        changed => {
            'B::Concise'            => '0.78_01',
            'Encode'                => '2.39_01',
            'File::Glob'            => '1.07_01',
            'Module::CoreList'      => '2.50_02',
            'Unicode::UCD'          => '0.29',
            'charnames'             => '1.07_01',
        },
        removed => {
        }
    },
    5.013 => {
        delta_from => 5.012,
        changed => {
            'CGI'                   => '3.49',
            'CGI::Fast'             => '1.08',
            'Data::Dumper'          => '2.126',
            'ExtUtils::MM_Unix'     => '6.5601',
            'ExtUtils::MakeMaker'   => '6.5601',
            'File::Copy'            => '2.18',
            'IPC::Open3'            => '1.06',
            'MIME::Base64'          => '3.09',
            'MIME::QuotedPrint'     => '3.09',
            'Module::CoreList'      => '2.31',
            'Pod::Functions'        => '1.04',
            'XS::APItest'           => '0.18',
            'XS::APItest::KeywordRPN'=> '0.004',
            'feature'               => '1.17',
            'threads'               => '1.77_01',
            'threads::shared'       => '1.33',
        },
        removed => {
        }
    },
    5.013001 => {
        delta_from => 5.012001,
        changed => {
            'Data::Dumper'          => '2.126',
            'Dumpvalue'             => '1.14',
            'Errno'                 => '1.12',
            'ExtUtils::MM_Unix'     => '6.5601',
            'ExtUtils::MakeMaker'   => '6.5601',
            'ExtUtils::ParseXS'     => '2.2205',
            'File::Find'            => '1.16',
            'IPC::Cmd'              => '0.58',
            'IPC::Open3'            => '1.06',
            'List::Util'            => '1.23',
            'List::Util::PP'        => '1.23',
            'List::Util::XS'        => '1.23',
            'Locale::Codes'         => '3.12',
            'Locale::Codes::Country'=> '3.12',
            'Locale::Codes::Currency'=> '3.12',
            'Locale::Codes::Language'=> '3.12',
            'Locale::Codes::Script' => '3.12',
            'Locale::Constants'     => '3.12',
            'Locale::Country'       => '3.12',
            'Locale::Currency'      => '3.12',
            'Locale::Language'      => '3.12',
            'Locale::Script'        => '3.12',
            'MIME::Base64'          => '3.09',
            'MIME::QuotedPrint'     => '3.09',
            'Module::Build::Platform::cygwin'=> '0.360301',
            'Module::CoreList'      => '2.34',
            'Module::Load::Conditional'=> '0.38',
            'PerlIO::scalar'        => '0.08',
            'Scalar::Util'          => '1.23',
            'Scalar::Util::PP'      => '1.23',
            'Socket'                => '1.88',
            'Term::ReadLine'        => '1.06',
            'Unicode::UCD'          => '0.28',
            'XS::APItest'           => '0.19',
            'XS::APItest::KeywordRPN'=> '0.004',
            'charnames'             => '1.08',
            'feature'               => '1.17',
            'threads'               => '1.77_01',
            'threads::shared'       => '1.33',
        },
        removed => {
            'Class::ISA'            => 1,
            'Pod::Plainer'          => 1,
            'Switch'                => 1,
        }
    },
    5.013002 => {
        delta_from => 5.013001,
        changed => {
            'B::Concise'            => '0.79',
            'B::Deparse'            => '0.98',
            'CPAN'                  => '1.94_57',
            'CPAN::Distribution'    => '1.9600',
            'Exporter'              => '5.64_02',
            'Exporter::Heavy'       => '5.64_02',
            'File::Copy'            => '2.19',
            'Hash::Util'            => '0.08',
            'IO::Socket'            => '1.32',
            'Locale::Codes'         => '3.13',
            'Locale::Codes::Country'=> '3.13',
            'Locale::Codes::Currency'=> '3.13',
            'Locale::Codes::Language'=> '3.13',
            'Locale::Codes::Script' => '3.13',
            'Locale::Constants'     => '3.13',
            'Locale::Country'       => '3.13',
            'Locale::Currency'      => '3.13',
            'Locale::Language'      => '3.13',
            'Locale::Script'        => '3.13',
            'Search::Dict'          => '1.03',
            'Socket'                => '1.89',
            'Thread::Semaphore'     => '2.11',
            'UNIVERSAL'             => '1.07',
            'VMS::DCLsym'           => '1.04',
            'mro'                   => '1.03',
            'threads'               => '1.77_02',
            'threads::shared'       => '1.33_01',
        },
        removed => {
        }
    },
    5.013003 => {
        delta_from => 5.013002,
        changed => {
            'App::Prove'            => '3.21',
            'App::Prove::State'     => '3.21',
            'App::Prove::State::Result'=> '3.21',
            'App::Prove::State::Result::Test'=> '3.21',
            'Archive::Extract'      => '0.42',
            'Archive::Tar'          => '1.64',
            'Archive::Tar::Constant'=> '1.64',
            'Archive::Tar::File'    => '1.64',
            'Attribute::Handlers'   => '0.88',
            'CPANPLUS'              => '0.9007',
            'CPANPLUS::Internals'   => '0.9007',
            'CPANPLUS::Shell::Default'=> '0.9007',
            'Compress::Raw::Bzip2'  => '2.027',
            'Compress::Raw::Zlib'   => '2.027_01',
            'Compress::Zlib'        => '2.027',
            'DB'                    => '1.03',
            'Digest::MD5'           => '2.40',
            'Digest::SHA'           => '5.48',
            'Exporter'              => '5.64_03',
            'Exporter::Heavy'       => '5.64_03',
            'ExtUtils::CBuilder'    => '0.2703',
            'ExtUtils::CBuilder::Base'=> '0.2703_01',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.2703',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.2703',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.2703',
            'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.2703',
            'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.2703',
            'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.2703',
            'ExtUtils::CBuilder::Platform::aix'=> '0.2703',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.2703',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.2703',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.2703',
            'ExtUtils::CBuilder::Platform::os2'=> '0.2703',
            'ExtUtils::Manifest'    => '1.58',
            'ExtUtils::ParseXS'     => '2.2206',
            'Fatal'                 => '2.10',
            'File::Basename'        => '2.79',
            'File::Copy'            => '2.20',
            'File::DosGlob'         => '1.02',
            'File::Find'            => '1.17',
            'File::Glob'            => '1.08',
            'File::stat'            => '1.03',
            'I18N::LangTags'        => '0.35_01',
            'I18N::LangTags::List'  => '0.35_01',
            'IO::Compress::Adapter::Bzip2'=> '2.027',
            'IO::Compress::Adapter::Deflate'=> '2.027',
            'IO::Compress::Adapter::Identity'=> '2.027',
            'IO::Compress::Base'    => '2.027',
            'IO::Compress::Base::Common'=> '2.027',
            'IO::Compress::Bzip2'   => '2.027',
            'IO::Compress::Deflate' => '2.027',
            'IO::Compress::Gzip'    => '2.027',
            'IO::Compress::Gzip::Constants'=> '2.027',
            'IO::Compress::RawDeflate'=> '2.027',
            'IO::Compress::Zip'     => '2.027',
            'IO::Compress::Zip::Constants'=> '2.027',
            'IO::Compress::Zlib::Constants'=> '2.027',
            'IO::Compress::Zlib::Extra'=> '2.027',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.027',
            'IO::Uncompress::Adapter::Identity'=> '2.027',
            'IO::Uncompress::Adapter::Inflate'=> '2.027',
            'IO::Uncompress::AnyInflate'=> '2.027',
            'IO::Uncompress::AnyUncompress'=> '2.027',
            'IO::Uncompress::Base'  => '2.027',
            'IO::Uncompress::Bunzip2'=> '2.027',
            'IO::Uncompress::Gunzip'=> '2.027',
            'IO::Uncompress::Inflate'=> '2.027',
            'IO::Uncompress::RawInflate'=> '2.027',
            'IO::Uncompress::Unzip' => '2.027',
            'IPC::Cmd'              => '0.60',
            'IPC::Msg'              => '2.03',
            'IPC::Semaphore'        => '2.03',
            'IPC::SharedMem'        => '2.03',
            'IPC::SysV'             => '2.03',
            'Locale::Maketext'      => '1.15',
            'Locale::Maketext::Guts'=> undef,
            'Locale::Maketext::GutsLoader'=> undef,
            'Module::Build'         => '0.3607',
            'Module::Build::Base'   => '0.3607',
            'Module::Build::Compat' => '0.3607',
            'Module::Build::Config' => '0.3607',
            'Module::Build::Cookbook'=> '0.3607',
            'Module::Build::Dumper' => '0.3607',
            'Module::Build::ModuleInfo'=> '0.3607',
            'Module::Build::Notes'  => '0.3607',
            'Module::Build::PPMMaker'=> '0.3607',
            'Module::Build::Platform::Amiga'=> '0.3607',
            'Module::Build::Platform::Default'=> '0.3607',
            'Module::Build::Platform::EBCDIC'=> '0.3607',
            'Module::Build::Platform::MPEiX'=> '0.3607',
            'Module::Build::Platform::MacOS'=> '0.3607',
            'Module::Build::Platform::RiscOS'=> '0.3607',
            'Module::Build::Platform::Unix'=> '0.3607',
            'Module::Build::Platform::VMS'=> '0.3607',
            'Module::Build::Platform::VOS'=> '0.3607',
            'Module::Build::Platform::Windows'=> '0.3607',
            'Module::Build::Platform::aix'=> '0.3607',
            'Module::Build::Platform::cygwin'=> '0.3607',
            'Module::Build::Platform::darwin'=> '0.3607',
            'Module::Build::Platform::os2'=> '0.3607',
            'Module::Build::PodParser'=> '0.3607',
            'Module::CoreList'      => '2.36',
            'Module::Load'          => '0.18',
            'TAP::Base'             => '3.21',
            'TAP::Formatter::Base'  => '3.21',
            'TAP::Formatter::Color' => '3.21',
            'TAP::Formatter::Console'=> '3.21',
            'TAP::Formatter::Console::ParallelSession'=> '3.21',
            'TAP::Formatter::Console::Session'=> '3.21',
            'TAP::Formatter::File'  => '3.21',
            'TAP::Formatter::File::Session'=> '3.21',
            'TAP::Formatter::Session'=> '3.21',
            'TAP::Harness'          => '3.21',
            'TAP::Object'           => '3.21',
            'TAP::Parser'           => '3.21',
            'TAP::Parser::Aggregator'=> '3.21',
            'TAP::Parser::Grammar'  => '3.21',
            'TAP::Parser::Iterator' => '3.21',
            'TAP::Parser::Iterator::Array'=> '3.21',
            'TAP::Parser::Iterator::Process'=> '3.21',
            'TAP::Parser::Iterator::Stream'=> '3.21',
            'TAP::Parser::IteratorFactory'=> '3.21',
            'TAP::Parser::Multiplexer'=> '3.21',
            'TAP::Parser::Result'   => '3.21',
            'TAP::Parser::Result::Bailout'=> '3.21',
            'TAP::Parser::Result::Comment'=> '3.21',
            'TAP::Parser::Result::Plan'=> '3.21',
            'TAP::Parser::Result::Pragma'=> '3.21',
            'TAP::Parser::Result::Test'=> '3.21',
            'TAP::Parser::Result::Unknown'=> '3.21',
            'TAP::Parser::Result::Version'=> '3.21',
            'TAP::Parser::Result::YAML'=> '3.21',
            'TAP::Parser::ResultFactory'=> '3.21',
            'TAP::Parser::Scheduler'=> '3.21',
            'TAP::Parser::Scheduler::Job'=> '3.21',
            'TAP::Parser::Scheduler::Spinner'=> '3.21',
            'TAP::Parser::Source'   => '3.21',
            'TAP::Parser::SourceHandler'=> '3.21',
            'TAP::Parser::SourceHandler::Executable'=> '3.21',
            'TAP::Parser::SourceHandler::File'=> '3.21',
            'TAP::Parser::SourceHandler::Handle'=> '3.21',
            'TAP::Parser::SourceHandler::Perl'=> '3.21',
            'TAP::Parser::SourceHandler::RawTAP'=> '3.21',
            'TAP::Parser::SourceHandler::pgTAP'=> '3.21',
            'TAP::Parser::Utils'    => '3.21',
            'TAP::Parser::YAMLish::Reader'=> '3.21',
            'TAP::Parser::YAMLish::Writer'=> '3.21',
            'Term::ANSIColor'       => '3.00',
            'Term::ReadLine'        => '1.07',
            'Test::Harness'         => '3.21',
            'Tie::Array'            => '1.04',
            'Time::HiRes'           => '1.9721',
            'Time::Piece'           => '1.20_01',
            'Unicode::Collate'      => '0.53',
            'Unicode::Normalize'    => '1.06',
            'Unicode::UCD'          => '0.29',
            'autodie'               => '2.10',
            'autodie::exception'    => '2.10',
            'autodie::exception::system'=> '2.10',
            'autodie::hints'        => '2.10',
            'blib'                  => '1.05',
            'charnames'             => '1.11',
            'diagnostics'           => '1.20',
            'inc::latest'           => '0.3607',
            'lib'                   => '0.63',
            're'                    => '0.12',
            'threads'               => '1.77_03',
            'threads::shared'       => '1.33_02',
            'vars'                  => '1.02',
            'warnings'              => '1.10',
        },
        removed => {
            'TAP::Parser::Source::Perl'=> 1,
        }
    },
    5.013004 => {
        delta_from => 5.013003,
        changed => {
            'App::Prove'            => '3.22',
            'App::Prove::State'     => '3.22',
            'App::Prove::State::Result'=> '3.22',
            'App::Prove::State::Result::Test'=> '3.22',
            'Archive::Tar'          => '1.68',
            'Archive::Tar::Constant'=> '1.68',
            'Archive::Tar::File'    => '1.68',
            'B::Lint'               => '1.12',
            'B::Lint::Debug'        => '1.12',
            'Carp'                  => '1.18',
            'Carp::Heavy'           => '1.18',
            'Compress::Raw::Bzip2'  => '2.030',
            'Compress::Raw::Zlib'   => '2.030',
            'Compress::Zlib'        => '2.030',
            'ExtUtils::ParseXS'     => '2.2207',
            'File::Spec'            => '3.31_01',
            'I18N::Langinfo'        => '0.04',
            'IO::Compress::Adapter::Bzip2'=> '2.030',
            'IO::Compress::Adapter::Deflate'=> '2.030',
            'IO::Compress::Adapter::Identity'=> '2.030',
            'IO::Compress::Base'    => '2.030',
            'IO::Compress::Base::Common'=> '2.030',
            'IO::Compress::Bzip2'   => '2.030',
            'IO::Compress::Deflate' => '2.030',
            'IO::Compress::Gzip'    => '2.030',
            'IO::Compress::Gzip::Constants'=> '2.030',
            'IO::Compress::RawDeflate'=> '2.030',
            'IO::Compress::Zip'     => '2.030',
            'IO::Compress::Zip::Constants'=> '2.030',
            'IO::Compress::Zlib::Constants'=> '2.030',
            'IO::Compress::Zlib::Extra'=> '2.030',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.030',
            'IO::Uncompress::Adapter::Identity'=> '2.030',
            'IO::Uncompress::Adapter::Inflate'=> '2.030',
            'IO::Uncompress::AnyInflate'=> '2.030',
            'IO::Uncompress::AnyUncompress'=> '2.030',
            'IO::Uncompress::Base'  => '2.030',
            'IO::Uncompress::Bunzip2'=> '2.030',
            'IO::Uncompress::Gunzip'=> '2.030',
            'IO::Uncompress::Inflate'=> '2.030',
            'IO::Uncompress::RawInflate'=> '2.030',
            'IO::Uncompress::Unzip' => '2.030',
            'Module::CoreList'      => '2.37',
            'TAP::Base'             => '3.22',
            'TAP::Formatter::Base'  => '3.22',
            'TAP::Formatter::Color' => '3.22',
            'TAP::Formatter::Console'=> '3.22',
            'TAP::Formatter::Console::ParallelSession'=> '3.22',
            'TAP::Formatter::Console::Session'=> '3.22',
            'TAP::Formatter::File'  => '3.22',
            'TAP::Formatter::File::Session'=> '3.22',
            'TAP::Formatter::Session'=> '3.22',
            'TAP::Harness'          => '3.22',
            'TAP::Object'           => '3.22',
            'TAP::Parser'           => '3.22',
            'TAP::Parser::Aggregator'=> '3.22',
            'TAP::Parser::Grammar'  => '3.22',
            'TAP::Parser::Iterator' => '3.22',
            'TAP::Parser::Iterator::Array'=> '3.22',
            'TAP::Parser::Iterator::Process'=> '3.22',
            'TAP::Parser::Iterator::Stream'=> '3.22',
            'TAP::Parser::IteratorFactory'=> '3.22',
            'TAP::Parser::Multiplexer'=> '3.22',
            'TAP::Parser::Result'   => '3.22',
            'TAP::Parser::Result::Bailout'=> '3.22',
            'TAP::Parser::Result::Comment'=> '3.22',
            'TAP::Parser::Result::Plan'=> '3.22',
            'TAP::Parser::Result::Pragma'=> '3.22',
            'TAP::Parser::Result::Test'=> '3.22',
            'TAP::Parser::Result::Unknown'=> '3.22',
            'TAP::Parser::Result::Version'=> '3.22',
            'TAP::Parser::Result::YAML'=> '3.22',
            'TAP::Parser::ResultFactory'=> '3.22',
            'TAP::Parser::Scheduler'=> '3.22',
            'TAP::Parser::Scheduler::Job'=> '3.22',
            'TAP::Parser::Scheduler::Spinner'=> '3.22',
            'TAP::Parser::Source'   => '3.22',
            'TAP::Parser::SourceHandler'=> '3.22',
            'TAP::Parser::SourceHandler::Executable'=> '3.22',
            'TAP::Parser::SourceHandler::File'=> '3.22',
            'TAP::Parser::SourceHandler::Handle'=> '3.22',
            'TAP::Parser::SourceHandler::Perl'=> '3.22',
            'TAP::Parser::SourceHandler::RawTAP'=> '3.22',
            'TAP::Parser::Utils'    => '3.22',
            'TAP::Parser::YAMLish::Reader'=> '3.22',
            'TAP::Parser::YAMLish::Writer'=> '3.22',
            'Test::Builder'         => '0.96',
            'Test::Builder::Module' => '0.96',
            'Test::Builder::Tester' => '1.20',
            'Test::Builder::Tester::Color'=> '1.20',
            'Test::Harness'         => '3.22',
            'Test::More'            => '0.96',
            'Test::Simple'          => '0.96',
            'Unicode::Collate'      => '0.56',
            'Unicode::Collate::Locale'=> '0.56',
            'XS::APItest'           => '0.20',
            'charnames'             => '1.15',
            'feature'               => '1.18',
        },
        removed => {
            'TAP::Parser::SourceHandler::pgTAP'=> 1,
        }
    },
    5.013005 => {
        delta_from => 5.013004,
        changed => {
            'B::Debug'              => '1.16',
            'CPANPLUS::Dist::Build' => '0.48',
            'CPANPLUS::Dist::Build::Constants'=> '0.48',
            'Data::Dumper'          => '2.128',
            'Encode'                => '2.40',
            'Encode::Guess'         => '2.04',
            'Encode::MIME::Header'  => '2.12',
            'Encode::Unicode::UTF7' => '2.05',
            'Errno'                 => '1.13',
            'ExtUtils::Command::MM' => '6.57_05',
            'ExtUtils::Liblist'     => '6.57_05',
            'ExtUtils::Liblist::Kid'=> '6.5705',
            'ExtUtils::MM'          => '6.57_05',
            'ExtUtils::MM_AIX'      => '6.57_05',
            'ExtUtils::MM_Any'      => '6.57_05',
            'ExtUtils::MM_BeOS'     => '6.57_05',
            'ExtUtils::MM_Cygwin'   => '6.57_05',
            'ExtUtils::MM_DOS'      => '6.5705',
            'ExtUtils::MM_Darwin'   => '6.57_05',
            'ExtUtils::MM_MacOS'    => '6.5705',
            'ExtUtils::MM_NW5'      => '6.57_05',
            'ExtUtils::MM_OS2'      => '6.57_05',
            'ExtUtils::MM_QNX'      => '6.57_05',
            'ExtUtils::MM_UWIN'     => '6.5705',
            'ExtUtils::MM_Unix'     => '6.57_05',
            'ExtUtils::MM_VMS'      => '6.57_05',
            'ExtUtils::MM_VOS'      => '6.57_05',
            'ExtUtils::MM_Win32'    => '6.57_05',
            'ExtUtils::MM_Win95'    => '6.57_05',
            'ExtUtils::MY'          => '6.5705',
            'ExtUtils::MakeMaker'   => '6.57_05',
            'ExtUtils::MakeMaker::Config'=> '6.57_05',
            'ExtUtils::MakeMaker::YAML'=> '1.44',
            'ExtUtils::Mkbootstrap' => '6.57_05',
            'ExtUtils::Mksymlists'  => '6.57_05',
            'ExtUtils::testlib'     => '6.5705',
            'Filter::Simple'        => '0.85',
            'Hash::Util'            => '0.09',
            'Math::BigFloat'        => '1.62',
            'Math::BigInt'          => '1.95',
            'Math::BigInt::Calc'    => '0.54',
            'Math::BigInt::CalcEmu' => '0.06',
            'Math::BigInt::FastCalc'=> '0.22',
            'Math::BigRat'          => '0.26',
            'Module::CoreList'      => '2.39',
            'POSIX'                 => '1.20',
            'PerlIO::scalar'        => '0.09',
            'Safe'                  => '2.28',
            'Test::Builder'         => '0.97_01',
            'Test::Builder::Module' => '0.97_01',
            'Test::Builder::Tester' => '1.21_01',
            'Test::Builder::Tester::Color'=> '1.21_01',
            'Test::More'            => '0.97_01',
            'Test::Simple'          => '0.97_01',
            'Tie::Hash'             => '1.04',
            'Unicode::Collate'      => '0.59',
            'Unicode::Collate::Locale'=> '0.59',
            'XS::APItest'           => '0.21',
            'XS::APItest::KeywordRPN'=> '0.005',
            'XSLoader'              => '0.11',
            'bigint'                => '0.25',
            'bignum'                => '0.25',
            'bigrat'                => '0.25',
            'blib'                  => '1.06',
            'open'                  => '1.08',
            'threads::shared'       => '1.33_03',
            'warnings'              => '1.11',
            'warnings::register'    => '1.02',
        },
        removed => {
        }
    },
    5.013006 => {
        delta_from => 5.013005,
        changed => {
            'Archive::Extract'      => '0.44',
            'B'                     => '1.24',
            'B::Deparse'            => '0.99',
            'CPAN'                  => '1.94_61',
            'CPAN::FTP'             => '5.5005',
            'CPAN::Queue'           => '5.5001',
            'CPAN::Version'         => '5.5001',
            'Carp'                  => '1.19',
            'Carp::Heavy'           => '1.19',
            'Compress::Raw::Bzip2'  => '2.031',
            'Cwd'                   => '3.34',
            'Data::Dumper'          => '2.129',
            'Devel::Peek'           => '1.05',
            'Digest::MD5'           => '2.51',
            'ExtUtils::Constant::Base'=> '0.05',
            'ExtUtils::Constant::ProxySubs'=> '0.07',
            'ExtUtils::Embed'       => '1.29',
            'ExtUtils::XSSymSet'    => '1.2',
            'Fcntl'                 => '1.09',
            'File::DosGlob'         => '1.03',
            'File::Find'            => '1.18',
            'File::Glob'            => '1.09',
            'File::Spec'            => '3.33',
            'File::Spec::Cygwin'    => '3.33',
            'File::Spec::Epoc'      => '3.33',
            'File::Spec::Functions' => '3.33',
            'File::Spec::Mac'       => '3.33',
            'File::Spec::OS2'       => '3.33',
            'File::Spec::Unix'      => '3.33',
            'File::Spec::VMS'       => '3.33',
            'File::Spec::Win32'     => '3.33',
            'GDBM_File'             => '1.11',
            'Hash::Util::FieldHash' => '1.05',
            'I18N::Langinfo'        => '0.06',
            'IPC::Cmd'              => '0.64',
            'IPC::Open3'            => '1.07',
            'Locale::Codes'         => '3.14',
            'Locale::Codes::Country'=> '3.14',
            'Locale::Codes::Currency'=> '3.14',
            'Locale::Codes::Language'=> '3.14',
            'Locale::Codes::Script' => '3.14',
            'Locale::Constants'     => '3.14',
            'Locale::Country'       => '3.14',
            'Locale::Currency'      => '3.14',
            'Locale::Language'      => '3.14',
            'Locale::Maketext'      => '1.16',
            'Locale::Script'        => '3.14',
            'Math::BigFloat'        => '1.63',
            'Math::BigInt'          => '1.97',
            'Math::BigInt::Calc'    => '0.55',
            'Math::BigInt::CalcEmu' => '0.07',
            'Module::CoreList'      => '2.40',
            'NDBM_File'             => '1.09',
            'NEXT'                  => '0.65',
            'ODBM_File'             => '1.08',
            'Opcode'                => '1.16',
            'POSIX'                 => '1.21',
            'PerlIO::encoding'      => '0.13',
            'PerlIO::scalar'        => '0.10',
            'PerlIO::via'           => '0.10',
            'Pod::Man'              => '2.25',
            'Pod::Text'             => '3.15',
            'SDBM_File'             => '1.07',
            'Socket'                => '1.90',
            'Sys::Hostname'         => '1.13',
            'Tie::Hash::NamedCapture'=> '0.07',
            'Unicode::Collate'      => '0.63',
            'Unicode::Collate::Locale'=> '0.63',
            'Unicode::Normalize'    => '1.07',
            'XS::APItest'           => '0.23',
            'XSLoader'              => '0.13',
            'attributes'            => '0.13',
            'charnames'             => '1.16',
            'if'                    => '0.06',
            'mro'                   => '1.04',
            'overload'              => '1.11',
            're'                    => '0.13',
            'sigtrap'               => '1.05',
            'threads'               => '1.81_01',
            'threads::shared'       => '1.34',
        },
        removed => {
            'XS::APItest::KeywordRPN'=> 1,
        }
    },
    5.013007 => {
        delta_from => 5.013006,
        changed => {
            'Archive::Extract'      => '0.46',
            'Archive::Tar'          => '1.72',
            'Archive::Tar::Constant'=> '1.72',
            'Archive::Tar::File'    => '1.72',
            'AutoLoader'            => '5.71',
            'B'                     => '1.26',
            'B::Concise'            => '0.81',
            'B::Deparse'            => '1.01',
            'CGI'                   => '3.50',
            'CPAN'                  => '1.94_62',
            'CPANPLUS'              => '0.9010',
            'CPANPLUS::Dist::Build' => '0.50',
            'CPANPLUS::Dist::Build::Constants'=> '0.50',
            'CPANPLUS::Internals'   => '0.9010',
            'CPANPLUS::Shell::Default'=> '0.9010',
            'Data::Dumper'          => '2.130_01',
            'DynaLoader'            => '1.11',
            'ExtUtils::Constant'    => '0.23',
            'ExtUtils::Constant::ProxySubs'=> '0.08',
            'Fcntl'                 => '1.10',
            'File::Fetch'           => '0.28',
            'File::Glob'            => '1.10',
            'File::stat'            => '1.04',
            'GDBM_File'             => '1.12',
            'Hash::Util'            => '0.10',
            'Hash::Util::FieldHash' => '1.06',
            'I18N::Langinfo'        => '0.07',
            'Locale::Maketext'      => '1.17',
            'Locale::Maketext::Guts'=> '1.17',
            'Locale::Maketext::GutsLoader'=> '1.17',
            'MIME::Base64'          => '3.10',
            'MIME::QuotedPrint'     => '3.10',
            'Math::BigFloat'        => '1.99_01',
            'Math::BigInt'          => '1.99_01',
            'Math::BigInt::Calc'    => '1.99_01',
            'Math::BigInt::CalcEmu' => '1.99_01',
            'Math::BigInt::FastCalc'=> '0.24_01',
            'Math::BigRat'          => '0.26_01',
            'Module::CoreList'      => '2.41',
            'NDBM_File'             => '1.10',
            'ODBM_File'             => '1.09',
            'Opcode'                => '1.17',
            'POSIX'                 => '1.22',
            'Pod::Simple'           => '3.15',
            'Pod::Simple::BlackBox' => '3.15',
            'Pod::Simple::Checker'  => '3.15',
            'Pod::Simple::Debug'    => '3.15',
            'Pod::Simple::DumpAsText'=> '3.15',
            'Pod::Simple::DumpAsXML'=> '3.15',
            'Pod::Simple::HTML'     => '3.15',
            'Pod::Simple::HTMLBatch'=> '3.15',
            'Pod::Simple::LinkSection'=> '3.15',
            'Pod::Simple::Methody'  => '3.15',
            'Pod::Simple::Progress' => '3.15',
            'Pod::Simple::PullParser'=> '3.15',
            'Pod::Simple::PullParserEndToken'=> '3.15',
            'Pod::Simple::PullParserStartToken'=> '3.15',
            'Pod::Simple::PullParserTextToken'=> '3.15',
            'Pod::Simple::PullParserToken'=> '3.15',
            'Pod::Simple::RTF'      => '3.15',
            'Pod::Simple::Search'   => '3.15',
            'Pod::Simple::SimpleTree'=> '3.15',
            'Pod::Simple::Text'     => '3.15',
            'Pod::Simple::TextContent'=> '3.15',
            'Pod::Simple::TiedOutFH'=> '3.15',
            'Pod::Simple::Transcode'=> '3.15',
            'Pod::Simple::TranscodeDumb'=> '3.15',
            'Pod::Simple::TranscodeSmart'=> '3.15',
            'Pod::Simple::XHTML'    => '3.15',
            'Pod::Simple::XMLOutStream'=> '3.15',
            'SDBM_File'             => '1.08',
            'Safe'                  => '2.29',
            'SelfLoader'            => '1.18',
            'Socket'                => '1.91',
            'Storable'              => '2.24',
            'Sys::Hostname'         => '1.14',
            'Unicode'               => '6.0.0',
            'Unicode::Collate'      => '0.67',
            'Unicode::Collate::CJK::Big5'=> '0.65',
            'Unicode::Collate::CJK::GB2312'=> '0.65',
            'Unicode::Collate::CJK::JISX0208'=> '0.64',
            'Unicode::Collate::CJK::Korean'=> '0.66',
            'Unicode::Collate::CJK::Pinyin'=> '0.65',
            'Unicode::Collate::CJK::Stroke'=> '0.65',
            'Unicode::Collate::Locale'=> '0.67',
            'XS::APItest'           => '0.26',
            'XS::Typemap'           => '0.04',
            'charnames'             => '1.17',
            'mro'                   => '1.05',
            'parent'                => '0.224',
            're'                    => '0.14',
            'threads'               => '1.81_02',
        },
        removed => {
        }
    },
    5.013008 => {
        delta_from => 5.013007,
        changed => {
            'Archive::Tar'          => '1.74',
            'Archive::Tar::Constant'=> '1.74',
            'Archive::Tar::File'    => '1.74',
            'B'                     => '1.27',
            'B::Concise'            => '0.82',
            'B::Deparse'            => '1.02',
            'Carp::Heavy'           => '1.17',
            'Cwd'                   => '3.35',
            'Data::Dumper'          => '2.130_02',
            'Devel::Peek'           => '1.06',
            'Devel::SelfStubber'    => '1.05',
            'Digest::SHA'           => '5.50',
            'Dumpvalue'             => '1.15',
            'DynaLoader'            => '1.12',
            'Env'                   => '1.02',
            'Exporter::Heavy'       => '5.64_01',
            'ExtUtils::CBuilder'    => '0.280201',
            'ExtUtils::CBuilder::Base'=> '0.280201',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.280201',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.280201',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.280201',
            'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280201',
            'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280201',
            'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280201',
            'ExtUtils::CBuilder::Platform::aix'=> '0.280201',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.280201',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.280201',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280201',
            'ExtUtils::CBuilder::Platform::os2'=> '0.280201',
            'ExtUtils::Constant::Utils'=> '0.03',
            'ExtUtils::Embed'       => '1.30',
            'ExtUtils::ParseXS'     => '2.2208',
            'Fatal'                 => '2.1001',
            'Fcntl'                 => '1.11',
            'File::CheckTree'       => '4.41',
            'File::Glob'            => '1.11',
            'GDBM_File'             => '1.13',
            'Hash::Util::FieldHash' => '1.07',
            'I18N::Collate'         => '1.02',
            'IO'                    => '1.25_03',
            'IPC::Cmd'              => '0.66',
            'IPC::Open3'            => '1.08',
            'Locale::Codes'         => '3.15',
            'Locale::Codes::Country'=> '3.15',
            'Locale::Codes::Currency'=> '3.15',
            'Locale::Codes::Language'=> '3.15',
            'Locale::Codes::Script' => '3.15',
            'Locale::Constants'     => '3.15',
            'Locale::Country'       => '3.15',
            'Locale::Currency'      => '3.15',
            'Locale::Language'      => '3.15',
            'Locale::Script'        => '3.15',
            'MIME::Base64'          => '3.13',
            'MIME::QuotedPrint'     => '3.13',
            'Math::BigFloat'        => '1.99_02',
            'Math::BigInt'          => '1.99_02',
            'Math::BigInt::Calc'    => '1.99_02',
            'Math::BigInt::CalcEmu' => '1.99_02',
            'Memoize'               => '1.02',
            'Memoize::AnyDBM_File'  => '1.02',
            'Memoize::Expire'       => '1.02',
            'Memoize::ExpireFile'   => '1.02',
            'Memoize::ExpireTest'   => '1.02',
            'Memoize::NDBM_File'    => '1.02',
            'Memoize::SDBM_File'    => '1.02',
            'Memoize::Storable'     => '1.02',
            'Module::CoreList'      => '2.43',
            'NDBM_File'             => '1.11',
            'Net::Ping'             => '2.37',
            'ODBM_File'             => '1.10',
            'Opcode'                => '1.18',
            'POSIX'                 => '1.23',
            'PerlIO::encoding'      => '0.14',
            'PerlIO::scalar'        => '0.11',
            'PerlIO::via'           => '0.11',
            'SDBM_File'             => '1.09',
            'Socket'                => '1.92',
            'Storable'              => '2.25',
            'Time::HiRes'           => '1.9721_01',
            'Unicode::Collate'      => '0.6801',
            'Unicode::Collate::Locale'=> '0.68',
            'Unicode::Normalize'    => '1.08',
            'Unicode::UCD'          => '0.30',
            'Win32'                 => '0.41',
            'XS::APItest'           => '0.27',
            'autodie'               => '2.1001',
            'autodie::exception'    => '2.1001',
            'autodie::exception::system'=> '2.1001',
            'autodie::hints'        => '2.1001',
            'feature'               => '1.19',
            'if'                    => '0.0601',
            'mro'                   => '1.06',
            'overload'              => '1.12',
            're'                    => '0.15',
            'threads'               => '1.81_03',
            'threads::shared'       => '1.35',
            'version'               => '0.86',
        },
        removed => {
        }
    },
    5.013009 => {
        delta_from => 5.013008,
        changed => {
            'Archive::Extract'      => '0.48',
            'Archive::Tar'          => '1.76',
            'Archive::Tar::Constant'=> '1.76',
            'Archive::Tar::File'    => '1.76',
            'B::Concise'            => '0.83',
            'B::Deparse'            => '1.03',
            'B::Lint'               => '1.13',
            'Benchmark'             => '1.12',
            'CGI'                   => '3.51',
            'CGI::Carp'             => '3.51',
            'CGI::Cookie'           => '1.30',
            'CGI::Push'             => '1.05',
            'CGI::Util'             => '3.51',
            'CPAN'                  => '1.94_63',
            'CPAN::HTTP::Client'    => '1.94',
            'CPAN::HTTP::Credentials'=> '1.94',
            'CPAN::Meta::YAML'      => '0.003',
            'CPANPLUS'              => '0.9011',
            'CPANPLUS::Dist::Build' => '0.52',
            'CPANPLUS::Dist::Build::Constants'=> '0.52',
            'CPANPLUS::Internals'   => '0.9011',
            'CPANPLUS::Shell::Default'=> '0.9011',
            'Carp::Heavy'           => '1.19',
            'Compress::Raw::Bzip2'  => '2.033',
            'Compress::Raw::Zlib'   => '2.033',
            'Compress::Zlib'        => '2.033',
            'Cwd'                   => '3.36',
            'DBM_Filter'            => '0.04',
            'DB_File'               => '1.821',
            'Devel::Peek'           => '1.07',
            'DirHandle'             => '1.04',
            'Dumpvalue'             => '1.16',
            'Encode'                => '2.42',
            'Encode::Alias'         => '2.13',
            'Encode::MIME::Header'  => '2.13',
            'Exporter::Heavy'       => '5.64_03',
            'ExtUtils::Install'     => '1.56',
            'ExtUtils::ParseXS'     => '2.2209',
            'File::Basename'        => '2.80',
            'File::Copy'            => '2.21',
            'File::DosGlob'         => '1.04',
            'File::Fetch'           => '0.32',
            'File::Find'            => '1.19',
            'File::Spec::Mac'       => '3.34',
            'File::Spec::VMS'       => '3.34',
            'File::stat'            => '1.05',
            'HTTP::Tiny'            => '0.009',
            'Hash::Util::FieldHash' => '1.08',
            'IO::Compress::Adapter::Bzip2'=> '2.033',
            'IO::Compress::Adapter::Deflate'=> '2.033',
            'IO::Compress::Adapter::Identity'=> '2.033',
            'IO::Compress::Base'    => '2.033',
            'IO::Compress::Base::Common'=> '2.033',
            'IO::Compress::Bzip2'   => '2.033',
            'IO::Compress::Deflate' => '2.033',
            'IO::Compress::Gzip'    => '2.033',
            'IO::Compress::Gzip::Constants'=> '2.033',
            'IO::Compress::RawDeflate'=> '2.033',
            'IO::Compress::Zip'     => '2.033',
            'IO::Compress::Zip::Constants'=> '2.033',
            'IO::Compress::Zlib::Constants'=> '2.033',
            'IO::Compress::Zlib::Extra'=> '2.033',
            'IO::Handle'            => '1.29',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.033',
            'IO::Uncompress::Adapter::Identity'=> '2.033',
            'IO::Uncompress::Adapter::Inflate'=> '2.033',
            'IO::Uncompress::AnyInflate'=> '2.033',
            'IO::Uncompress::AnyUncompress'=> '2.033',
            'IO::Uncompress::Base'  => '2.033',
            'IO::Uncompress::Bunzip2'=> '2.033',
            'IO::Uncompress::Gunzip'=> '2.033',
            'IO::Uncompress::Inflate'=> '2.033',
            'IO::Uncompress::RawInflate'=> '2.033',
            'IO::Uncompress::Unzip' => '2.033',
            'IPC::Cmd'              => '0.68',
            'IPC::Open3'            => '1.09',
            'JSON::PP'              => '2.27103',
            'JSON::PP::Boolean'     => undef,
            'Locale::Maketext'      => '1.18',
            'Log::Message'          => '0.04',
            'Log::Message::Config'  => '0.04',
            'Log::Message::Handlers'=> '0.04',
            'Log::Message::Item'    => '0.04',
            'Log::Message::Simple'  => '0.08',
            'Math::BigFloat'        => '1.99_03',
            'Math::BigInt'          => '1.99_03',
            'Math::BigInt::Calc'    => '1.99_03',
            'Math::BigInt::FastCalc'=> '0.24_02',
            'Math::BigRat'          => '0.26_02',
            'Module::CoreList'      => '2.42_01',
            'Module::Load::Conditional'=> '0.40',
            'Module::Metadata'      => '1.000003',
            'Net::Ping'             => '2.38',
            'OS2::Process'          => '1.05',
            'Object::Accessor'      => '0.38',
            'POSIX'                 => '1.24',
            'Params::Check'         => '0.28',
            'Perl::OSType'          => '1.002',
            'Pod::LaTeX'            => '0.59',
            'Pod::Perldoc'          => '3.15_03',
            'Socket'                => '1.93',
            'Storable'              => '2.26',
            'Sys::Hostname'         => '1.15',
            'Term::UI'              => '0.24',
            'Thread::Queue'         => '2.12',
            'Thread::Semaphore'     => '2.12',
            'Time::Local'           => '1.2000',
            'UNIVERSAL'             => '1.08',
            'Unicode::Normalize'    => '1.10',
            'Win32'                 => '0.44',
            'bigint'                => '0.26',
            'bignum'                => '0.26',
            'bigrat'                => '0.26',
            'charnames'             => '1.18',
            'diagnostics'           => '1.21',
            're'                    => '0.16',
            'threads'               => '1.83',
            'threads::shared'       => '1.36',
            'version'               => '0.88',
        },
        removed => {
        }
    },
    5.01301 => {
        delta_from => 5.013009,
        changed => {
            'Attribute::Handlers'   => '0.89',
            'B'                     => '1.28',
            'B::Showlex'            => '1.03',
            'CGI'                   => '3.52',
            'CPAN'                  => '1.94_65',
            'CPAN::Distribution'    => '1.9601',
            'CPAN::FTP::netrc'      => '1.01',
            'CPAN::FirstTime'       => '5.5303',
            'CPAN::HandleConfig'    => '5.5003',
            'CPAN::Meta'            => '2.110440',
            'CPAN::Meta::Converter' => '2.110440',
            'CPAN::Meta::Feature'   => '2.110440',
            'CPAN::Meta::History'   => '2.110440',
            'CPAN::Meta::Prereqs'   => '2.110440',
            'CPAN::Meta::Spec'      => '2.110440',
            'CPAN::Meta::Validator' => '2.110440',
            'CPAN::Shell'           => '5.5002',
            'CPANPLUS'              => '0.9101',
            'CPANPLUS::Internals'   => '0.9101',
            'CPANPLUS::Shell::Default'=> '0.9101',
            'Carp'                  => '1.20',
            'Carp::Heavy'           => '1.20',
            'Cwd'                   => '3.37',
            'Devel::DProf'          => '20110217.00',
            'DynaLoader'            => '1.13',
            'ExtUtils::CBuilder'    => '0.280202',
            'ExtUtils::CBuilder::Base'=> '0.280202',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.280202',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.280202',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.280202',
            'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280202',
            'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280202',
            'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280202',
            'ExtUtils::CBuilder::Platform::aix'=> '0.280202',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.280202',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.280202',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280202',
            'ExtUtils::CBuilder::Platform::os2'=> '0.280202',
            'File::Copy'            => '2.22',
            'Filter::Simple'        => '0.86',
            'HTTP::Tiny'            => '0.010',
            'I18N::LangTags::Detect'=> '1.05',
            'IO::Select'            => '1.18',
            'IPC::Cmd'              => '0.70',
            'Locale::Maketext'      => '1.19',
            'Math::BigFloat'        => '1.992',
            'Math::BigInt'          => '1.992',
            'Math::BigInt::Calc'    => '1.992',
            'Math::BigInt::CalcEmu' => '1.992',
            'Module::Build'         => '0.37_05',
            'Module::Build::Base'   => '0.37_05',
            'Module::Build::Compat' => '0.37_05',
            'Module::Build::Config' => '0.37_05',
            'Module::Build::Cookbook'=> '0.37_05',
            'Module::Build::Dumper' => '0.37_05',
            'Module::Build::ModuleInfo'=> '0.37_05',
            'Module::Build::Notes'  => '0.37_05',
            'Module::Build::PPMMaker'=> '0.37_05',
            'Module::Build::Platform::Amiga'=> '0.37_05',
            'Module::Build::Platform::Default'=> '0.37_05',
            'Module::Build::Platform::EBCDIC'=> '0.37_05',
            'Module::Build::Platform::MPEiX'=> '0.37_05',
            'Module::Build::Platform::MacOS'=> '0.37_05',
            'Module::Build::Platform::RiscOS'=> '0.37_05',
            'Module::Build::Platform::Unix'=> '0.37_05',
            'Module::Build::Platform::VMS'=> '0.37_05',
            'Module::Build::Platform::VOS'=> '0.37_05',
            'Module::Build::Platform::Windows'=> '0.37_05',
            'Module::Build::Platform::aix'=> '0.37_05',
            'Module::Build::Platform::cygwin'=> '0.37_05',
            'Module::Build::Platform::darwin'=> '0.37_05',
            'Module::Build::Platform::os2'=> '0.37_05',
            'Module::Build::PodParser'=> '0.37_05',
            'Module::Build::Version'=> '0.87',
            'Module::Build::YAML'   => '1.41',
            'Module::CoreList'      => '2.45',
            'Module::Load::Conditional'=> '0.44',
            'Module::Metadata'      => '1.000004',
            'OS2::Process'          => '1.06',
            'Parse::CPAN::Meta'     => '1.4401',
            'Pod::Html'             => '1.1',
            'Socket'                => '1.94',
            'Term::UI'              => '0.26',
            'Unicode::Collate'      => '0.72',
            'Unicode::Collate::Locale'=> '0.71',
            'Unicode::UCD'          => '0.31',
            'VMS::DCLsym'           => '1.05',
            'Version::Requirements' => '0.101020',
            'bigrat'                => '0.27',
            'deprecate'             => '0.02',
            'diagnostics'           => '1.22',
            'inc::latest'           => '0.37_05',
            'overload'              => '1.13',
            're'                    => '0.17',
            'utf8'                  => '1.09',
            'warnings'              => '1.12',
        },
        removed => {
        }
    },
    5.013011 => {
        delta_from => 5.01301,
        changed => {
            'App::Prove'            => '3.23',
            'App::Prove::State'     => '3.23',
            'App::Prove::State::Result'=> '3.23',
            'App::Prove::State::Result::Test'=> '3.23',
            'B'                     => '1.29',
            'CPAN'                  => '1.9600',
            'CPAN::Author'          => '5.5001',
            'CPAN::CacheMgr'        => '5.5001',
            'CPAN::Distribution'    => '1.9602',
            'CPAN::Exception::blocked_urllist'=> '1.001',
            'CPAN::HTTP::Client'    => '1.9600',
            'CPAN::HTTP::Credentials'=> '1.9600',
            'CPAN::Index'           => '1.9600',
            'CPAN::LWP::UserAgent'  => '1.9600',
            'CPAN::Mirrors'         => '1.9600',
            'CPAN::Module'          => '5.5001',
            'CPANPLUS'              => '0.9103',
            'CPANPLUS::Dist::Build' => '0.54',
            'CPANPLUS::Dist::Build::Constants'=> '0.54',
            'CPANPLUS::Internals'   => '0.9103',
            'CPANPLUS::Shell::Default'=> '0.9103',
            'Cwd'                   => '3.36',
            'Devel::DProf'          => '20110228.00',
            'Digest::SHA'           => '5.61',
            'ExtUtils::Command'     => '1.17',
            'File::Basename'        => '2.81',
            'File::Copy'            => '2.21',
            'File::Glob'            => '1.12',
            'GDBM_File'             => '1.14',
            'HTTP::Tiny'            => '0.011',
            'Hash::Util'            => '0.11',
            'Hash::Util::FieldHash' => '1.09',
            'I18N::Langinfo'        => '0.08',
            'IO'                    => '1.25_04',
            'IO::Dir'               => '1.08',
            'IO::File'              => '1.15',
            'IO::Handle'            => '1.30',
            'IO::Pipe'              => '1.14',
            'IO::Poll'              => '0.08',
            'IO::Select'            => '1.20',
            'JSON::PP'              => '2.27105',
            'Locale::Codes'         => '3.16',
            'Locale::Codes::Country'=> '3.16',
            'Locale::Codes::Currency'=> '3.16',
            'Locale::Codes::Language'=> '3.16',
            'Locale::Codes::Script' => '3.16',
            'Locale::Constants'     => '3.16',
            'Locale::Country'       => '3.16',
            'Locale::Currency'      => '3.16',
            'Locale::Language'      => '3.16',
            'Locale::Script'        => '3.16',
            'Math::BigFloat'        => '1.993',
            'Math::BigInt'          => '1.994',
            'Math::BigInt::Calc'    => '1.993',
            'Math::BigInt::CalcEmu' => '1.993',
            'Math::BigInt::FastCalc'=> '0.28',
            'Module::Build'         => '0.3800',
            'Module::Build::Base'   => '0.3800',
            'Module::Build::Compat' => '0.3800',
            'Module::Build::Config' => '0.3800',
            'Module::Build::Cookbook'=> '0.3800',
            'Module::Build::Dumper' => '0.3800',
            'Module::Build::ModuleInfo'=> '0.3800',
            'Module::Build::Notes'  => '0.3800',
            'Module::Build::PPMMaker'=> '0.3800',
            'Module::Build::Platform::Amiga'=> '0.3800',
            'Module::Build::Platform::Default'=> '0.3800',
            'Module::Build::Platform::EBCDIC'=> '0.3800',
            'Module::Build::Platform::MPEiX'=> '0.3800',
            'Module::Build::Platform::MacOS'=> '0.3800',
            'Module::Build::Platform::RiscOS'=> '0.3800',
            'Module::Build::Platform::Unix'=> '0.3800',
            'Module::Build::Platform::VMS'=> '0.3800',
            'Module::Build::Platform::VOS'=> '0.3800',
            'Module::Build::Platform::Windows'=> '0.3800',
            'Module::Build::Platform::aix'=> '0.3800',
            'Module::Build::Platform::cygwin'=> '0.3800',
            'Module::Build::Platform::darwin'=> '0.3800',
            'Module::Build::Platform::os2'=> '0.3800',
            'Module::Build::PodParser'=> '0.3800',
            'Module::CoreList'      => '2.46',
            'NDBM_File'             => '1.12',
            'Pod::Simple'           => '3.16',
            'Pod::Simple::BlackBox' => '3.16',
            'Pod::Simple::Checker'  => '3.16',
            'Pod::Simple::Debug'    => '3.16',
            'Pod::Simple::DumpAsText'=> '3.16',
            'Pod::Simple::DumpAsXML'=> '3.16',
            'Pod::Simple::HTML'     => '3.16',
            'Pod::Simple::HTMLBatch'=> '3.16',
            'Pod::Simple::LinkSection'=> '3.16',
            'Pod::Simple::Methody'  => '3.16',
            'Pod::Simple::Progress' => '3.16',
            'Pod::Simple::PullParser'=> '3.16',
            'Pod::Simple::PullParserEndToken'=> '3.16',
            'Pod::Simple::PullParserStartToken'=> '3.16',
            'Pod::Simple::PullParserTextToken'=> '3.16',
            'Pod::Simple::PullParserToken'=> '3.16',
            'Pod::Simple::RTF'      => '3.16',
            'Pod::Simple::Search'   => '3.16',
            'Pod::Simple::SimpleTree'=> '3.16',
            'Pod::Simple::Text'     => '3.16',
            'Pod::Simple::TextContent'=> '3.16',
            'Pod::Simple::TiedOutFH'=> '3.16',
            'Pod::Simple::Transcode'=> '3.16',
            'Pod::Simple::TranscodeDumb'=> '3.16',
            'Pod::Simple::TranscodeSmart'=> '3.16',
            'Pod::Simple::XHTML'    => '3.16',
            'Pod::Simple::XMLOutStream'=> '3.16',
            'Storable'              => '2.27',
            'Sys::Hostname'         => '1.16',
            'TAP::Base'             => '3.23',
            'TAP::Formatter::Base'  => '3.23',
            'TAP::Formatter::Color' => '3.23',
            'TAP::Formatter::Console'=> '3.23',
            'TAP::Formatter::Console::ParallelSession'=> '3.23',
            'TAP::Formatter::Console::Session'=> '3.23',
            'TAP::Formatter::File'  => '3.23',
            'TAP::Formatter::File::Session'=> '3.23',
            'TAP::Formatter::Session'=> '3.23',
            'TAP::Harness'          => '3.23',
            'TAP::Object'           => '3.23',
            'TAP::Parser'           => '3.23',
            'TAP::Parser::Aggregator'=> '3.23',
            'TAP::Parser::Grammar'  => '3.23',
            'TAP::Parser::Iterator' => '3.23',
            'TAP::Parser::Iterator::Array'=> '3.23',
            'TAP::Parser::Iterator::Process'=> '3.23',
            'TAP::Parser::Iterator::Stream'=> '3.23',
            'TAP::Parser::IteratorFactory'=> '3.23',
            'TAP::Parser::Multiplexer'=> '3.23',
            'TAP::Parser::Result'   => '3.23',
            'TAP::Parser::Result::Bailout'=> '3.23',
            'TAP::Parser::Result::Comment'=> '3.23',
            'TAP::Parser::Result::Plan'=> '3.23',
            'TAP::Parser::Result::Pragma'=> '3.23',
            'TAP::Parser::Result::Test'=> '3.23',
            'TAP::Parser::Result::Unknown'=> '3.23',
            'TAP::Parser::Result::Version'=> '3.23',
            'TAP::Parser::Result::YAML'=> '3.23',
            'TAP::Parser::ResultFactory'=> '3.23',
            'TAP::Parser::Scheduler'=> '3.23',
            'TAP::Parser::Scheduler::Job'=> '3.23',
            'TAP::Parser::Scheduler::Spinner'=> '3.23',
            'TAP::Parser::Source'   => '3.23',
            'TAP::Parser::SourceHandler'=> '3.23',
            'TAP::Parser::SourceHandler::Executable'=> '3.23',
            'TAP::Parser::SourceHandler::File'=> '3.23',
            'TAP::Parser::SourceHandler::Handle'=> '3.23',
            'TAP::Parser::SourceHandler::Perl'=> '3.23',
            'TAP::Parser::SourceHandler::RawTAP'=> '3.23',
            'TAP::Parser::Utils'    => '3.23',
            'TAP::Parser::YAMLish::Reader'=> '3.23',
            'TAP::Parser::YAMLish::Writer'=> '3.23',
            'Test::Builder'         => '0.98',
            'Test::Builder::Module' => '0.98',
            'Test::Builder::Tester' => '1.22',
            'Test::Builder::Tester::Color'=> '1.22',
            'Test::Harness'         => '3.23',
            'Test::More'            => '0.98',
            'Test::Simple'          => '0.98',
            'Tie::Hash::NamedCapture'=> '0.08',
            'Tie::RefHash'          => '1.39',
            'Unicode::Collate'      => '0.73',
            'Unicode::Collate::Locale'=> '0.73',
            'Unicode::UCD'          => '0.32',
            'XS::Typemap'           => '0.05',
            'attributes'            => '0.14',
            'base'                  => '2.16',
            'inc::latest'           => '0.3800',
            'mro'                   => '1.07',
            'parent'                => '0.225',
        },
        removed => {
        }
    },
    5.014 => {
        delta_from => 5.013011,
        changed => {
            'ExtUtils::CBuilder'    => '0.280203',
            'ExtUtils::CBuilder::Base'=> '0.280203',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.280203',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.280203',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.280203',
            'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280203',
            'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280203',
            'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280203',
            'ExtUtils::CBuilder::Platform::aix'=> '0.280203',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.280203',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.280203',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280203',
            'ExtUtils::CBuilder::Platform::os2'=> '0.280203',
            'ExtUtils::ParseXS'     => '2.2210',
            'File::Basename'        => '2.82',
            'HTTP::Tiny'            => '0.012',
            'IO::Handle'            => '1.31',
            'Module::CoreList'      => '2.49',
            'PerlIO'                => '1.07',
            'Pod::Html'             => '1.11',
            'XS::APItest'           => '0.28',
            'bigint'                => '0.27',
            'bignum'                => '0.27',
            'bigrat'                => '0.28',
            'constant'              => '1.21',
            'feature'               => '1.20',
            're'                    => '0.18',
            'threads::shared'       => '1.37',
        },
        removed => {
        }
    },
    5.014001 => {
        delta_from => 5.014,
        changed => {
            'B::Deparse'            => '1.04',
            'Module::CoreList'      => '2.49_01',
            'Pod::Perldoc'          => '3.15_04',
        },
        removed => {
        }
    },
    5.014002 => {
        delta_from => 5.014001,
        changed => {
            'CPAN'                  => '1.9600_01',
            'CPAN::Distribution'    => '1.9602_01',
            'Devel::DProf::dprof::V'=> undef,
            'Encode'                => '2.42_01',
            'File::Glob'            => '1.13',
            'Module::CoreList'      => '2.49_02',
            'PerlIO::scalar'        => '0.11_01',
            'Time::Piece::Seconds'  => undef,
        },
        removed => {
        }
    },
    5.014003 => {
        delta_from => 5.014002,
        changed => {
            'Digest'                => '1.16_01',
            'IPC::Open3'            => '1.09_01',
            'Module::CoreList'      => '2.49_04',
        },
        removed => {
        }
    },
    5.014004 => {
        delta_from => 5.014003,
        changed => {
            'Encode'                => '2.42_02',
            'IPC::Open3'            => '1.0901',
            'Module::CoreList'      => '2.49_06',
        },
        removed => {
        }
    },
    5.015 => {
        delta_from => 5.014001,
        changed => {
            'Archive::Extract'      => '0.52',
            'Attribute::Handlers'   => '0.91',
            'B'                     => '1.30',
            'B::Concise'            => '0.84',
            'B::Deparse'            => '1.05',
            'Benchmark'             => '1.13',
            'CGI'                   => '3.54',
            'CGI::Util'             => '3.53',
            'CPAN::Meta'            => '2.110930',
            'CPAN::Meta::Converter' => '2.110930',
            'CPAN::Meta::Feature'   => '2.110930',
            'CPAN::Meta::History'   => '2.110930',
            'CPAN::Meta::Prereqs'   => '2.110930',
            'CPAN::Meta::Spec'      => '2.110930',
            'CPAN::Meta::Validator' => '2.110930',
            'CPANPLUS'              => '0.9105',
            'CPANPLUS::Dist::Build' => '0.56',
            'CPANPLUS::Dist::Build::Constants'=> '0.56',
            'CPANPLUS::Internals'   => '0.9105',
            'CPANPLUS::Shell::Default'=> '0.9105',
            'Compress::Raw::Bzip2'  => '2.035',
            'Compress::Raw::Zlib'   => '2.035',
            'Compress::Zlib'        => '2.035',
            'DB_File'               => '1.822',
            'Data::Dumper'          => '2.131',
            'Devel::Peek'           => '1.08',
            'Digest::SHA'           => '5.62',
            'Encode'                => '2.43',
            'Encode::Alias'         => '2.14',
            'ExtUtils::CBuilder'    => '0.280204',
            'ExtUtils::CBuilder::Base'=> '0.280204',
            'Fatal'                 => '2.10',
            'File::Spec::Win32'     => '3.34',
            'Filter::Simple'        => '0.87',
            'Filter::Util::Call'    => '1.39',
            'FindBin'               => '1.51',
            'Hash::Util::FieldHash' => '1.10',
            'I18N::LangTags'        => '0.36',
            'IO::Compress::Adapter::Bzip2'=> '2.035',
            'IO::Compress::Adapter::Deflate'=> '2.035',
            'IO::Compress::Adapter::Identity'=> '2.035',
            'IO::Compress::Base'    => '2.035',
            'IO::Compress::Base::Common'=> '2.035',
            'IO::Compress::Bzip2'   => '2.035',
            'IO::Compress::Deflate' => '2.035',
            'IO::Compress::Gzip'    => '2.035',
            'IO::Compress::Gzip::Constants'=> '2.035',
            'IO::Compress::RawDeflate'=> '2.035',
            'IO::Compress::Zip'     => '2.035',
            'IO::Compress::Zip::Constants'=> '2.035',
            'IO::Compress::Zlib::Constants'=> '2.035',
            'IO::Compress::Zlib::Extra'=> '2.035',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.035',
            'IO::Uncompress::Adapter::Identity'=> '2.035',
            'IO::Uncompress::Adapter::Inflate'=> '2.035',
            'IO::Uncompress::AnyInflate'=> '2.035',
            'IO::Uncompress::AnyUncompress'=> '2.035',
            'IO::Uncompress::Base'  => '2.035',
            'IO::Uncompress::Bunzip2'=> '2.035',
            'IO::Uncompress::Gunzip'=> '2.035',
            'IO::Uncompress::Inflate'=> '2.035',
            'IO::Uncompress::RawInflate'=> '2.035',
            'IO::Uncompress::Unzip' => '2.035',
            'IPC::Open2'            => '1.04',
            'IPC::Open3'            => '1.11',
            'JSON::PP'              => '2.27200',
            'Math::BigFloat'        => '1.994',
            'Math::BigInt'          => '1.995',
            'Math::Complex'         => '1.57',
            'Math::Trig'            => '1.21',
            'Module::CoreList'      => '2.51',
            'ODBM_File'             => '1.11',
            'Object::Accessor'      => '0.42',
            'Opcode'                => '1.19',
            'PerlIO::encoding'      => '0.15',
            'PerlIO::scalar'        => '0.12',
            'Pod::Perldoc'          => '3.15_05',
            'Storable'              => '2.28',
            'Sys::Syslog'           => '0.29',
            'Time::HiRes'           => '1.9722',
            'Unicode::Collate'      => '0.76',
            'Unicode::Collate::CJK::Pinyin'=> '0.76',
            'Unicode::Collate::CJK::Stroke'=> '0.76',
            'Unicode::Collate::Locale'=> '0.76',
            'Unicode::Normalize'    => '1.12',
            'XS::APItest'           => '0.29',
            'XSLoader'              => '0.15',
            'autodie'               => '2.10',
            'autodie::exception'    => '2.10',
            'autodie::exception::system'=> '2.10',
            'autodie::hints'        => '2.10',
            'base'                  => '2.17',
            'charnames'             => '1.22',
            'constant'              => '1.22',
            'feature'               => '1.21',
            'mro'                   => '1.08',
            'overload'              => '1.14',
            'threads::shared'       => '1.38',
            'vmsish'                => '1.03',
        },
        removed => {
            'Devel::DProf'          => 1,
            'Shell'                 => 1,
        }
    },
    5.015001 => {
        delta_from => 5.015,
        changed => {
            'B::Deparse'            => '1.06',
            'CGI'                   => '3.55',
            'CPAN::Meta'            => '2.110930001',
            'CPAN::Meta::Converter' => '2.110930001',
            'CPANPLUS'              => '0.9108',
            'CPANPLUS::Internals'   => '0.9108',
            'CPANPLUS::Shell::Default'=> '0.9108',
            'Carp'                  => '1.21',
            'Carp::Heavy'           => '1.21',
            'Compress::Raw::Bzip2'  => '2.037',
            'Compress::Raw::Zlib'   => '2.037',
            'Compress::Zlib'        => '2.037',
            'Cwd'                   => '3.37',
            'Env'                   => '1.03',
            'ExtUtils::Command::MM' => '6.58',
            'ExtUtils::Liblist'     => '6.58',
            'ExtUtils::Liblist::Kid'=> '6.58',
            'ExtUtils::MM'          => '6.58',
            'ExtUtils::MM_AIX'      => '6.58',
            'ExtUtils::MM_Any'      => '6.58',
            'ExtUtils::MM_BeOS'     => '6.58',
            'ExtUtils::MM_Cygwin'   => '6.58',
            'ExtUtils::MM_DOS'      => '6.58',
            'ExtUtils::MM_Darwin'   => '6.58',
            'ExtUtils::MM_MacOS'    => '6.58',
            'ExtUtils::MM_NW5'      => '6.58',
            'ExtUtils::MM_OS2'      => '6.58',
            'ExtUtils::MM_QNX'      => '6.58',
            'ExtUtils::MM_UWIN'     => '6.58',
            'ExtUtils::MM_Unix'     => '6.58',
            'ExtUtils::MM_VMS'      => '6.58',
            'ExtUtils::MM_VOS'      => '6.58',
            'ExtUtils::MM_Win32'    => '6.58',
            'ExtUtils::MM_Win95'    => '6.58',
            'ExtUtils::MY'          => '6.58',
            'ExtUtils::MakeMaker'   => '6.58',
            'ExtUtils::MakeMaker::Config'=> '6.58',
            'ExtUtils::Mkbootstrap' => '6.58',
            'ExtUtils::Mksymlists'  => '6.58',
            'ExtUtils::ParseXS'     => '3.00_01',
            'ExtUtils::ParseXS::Constants'=> undef,
            'ExtUtils::ParseXS::CountLines'=> undef,
            'ExtUtils::ParseXS::Utilities'=> undef,
            'ExtUtils::Typemaps'    => '1.00',
            'ExtUtils::Typemaps::InputMap'=> undef,
            'ExtUtils::Typemaps::OutputMap'=> undef,
            'ExtUtils::Typemaps::Type'=> '0.05',
            'ExtUtils::testlib'     => '6.58',
            'File::Basename'        => '2.83',
            'File::Find'            => '1.20',
            'HTTP::Tiny'            => '0.013',
            'I18N::Langinfo'        => '0.08_02',
            'IO::Compress::Adapter::Bzip2'=> '2.037',
            'IO::Compress::Adapter::Deflate'=> '2.037',
            'IO::Compress::Adapter::Identity'=> '2.037',
            'IO::Compress::Base'    => '2.037',
            'IO::Compress::Base::Common'=> '2.037',
            'IO::Compress::Bzip2'   => '2.037',
            'IO::Compress::Deflate' => '2.037',
            'IO::Compress::Gzip'    => '2.037',
            'IO::Compress::Gzip::Constants'=> '2.037',
            'IO::Compress::RawDeflate'=> '2.037',
            'IO::Compress::Zip'     => '2.037',
            'IO::Compress::Zip::Constants'=> '2.037',
            'IO::Compress::Zlib::Constants'=> '2.037',
            'IO::Compress::Zlib::Extra'=> '2.037',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.037',
            'IO::Uncompress::Adapter::Identity'=> '2.037',
            'IO::Uncompress::Adapter::Inflate'=> '2.037',
            'IO::Uncompress::AnyInflate'=> '2.037',
            'IO::Uncompress::AnyUncompress'=> '2.037',
            'IO::Uncompress::Base'  => '2.037',
            'IO::Uncompress::Bunzip2'=> '2.037',
            'IO::Uncompress::Gunzip'=> '2.037',
            'IO::Uncompress::Inflate'=> '2.037',
            'IO::Uncompress::RawInflate'=> '2.037',
            'IO::Uncompress::Unzip' => '2.037',
            'IPC::Cmd'              => '0.72',
            'Locale::Codes'         => '3.17',
            'Locale::Codes::Constants'=> '3.17',
            'Locale::Codes::Country'=> '3.17',
            'Locale::Codes::Country_Codes'=> '3.17',
            'Locale::Codes::Currency'=> '3.17',
            'Locale::Codes::Currency_Codes'=> '3.17',
            'Locale::Codes::LangExt'=> '3.17',
            'Locale::Codes::LangExt_Codes'=> '3.17',
            'Locale::Codes::LangVar'=> '3.17',
            'Locale::Codes::LangVar_Codes'=> '3.17',
            'Locale::Codes::Language'=> '3.17',
            'Locale::Codes::Language_Codes'=> '3.17',
            'Locale::Codes::Script' => '3.17',
            'Locale::Codes::Script_Codes'=> '3.17',
            'Locale::Country'       => '3.17',
            'Locale::Currency'      => '3.17',
            'Locale::Language'      => '3.17',
            'Locale::Script'        => '3.17',
            'Math::BigFloat::Trace' => '0.28',
            'Math::BigInt::FastCalc'=> '0.29',
            'Math::BigInt::Trace'   => '0.28',
            'Math::BigRat'          => '0.2602',
            'Math::Complex'         => '1.58',
            'Math::Trig'            => '1.22',
            'Module::CoreList'      => '2.54',
            'OS2::Process'          => '1.07',
            'Pod::Perldoc'          => '3.15_06',
            'Pod::Simple'           => '3.18',
            'Pod::Simple::BlackBox' => '3.18',
            'Pod::Simple::Checker'  => '3.18',
            'Pod::Simple::Debug'    => '3.18',
            'Pod::Simple::DumpAsText'=> '3.18',
            'Pod::Simple::DumpAsXML'=> '3.18',
            'Pod::Simple::HTML'     => '3.18',
            'Pod::Simple::HTMLBatch'=> '3.18',
            'Pod::Simple::LinkSection'=> '3.18',
            'Pod::Simple::Methody'  => '3.18',
            'Pod::Simple::Progress' => '3.18',
            'Pod::Simple::PullParser'=> '3.18',
            'Pod::Simple::PullParserEndToken'=> '3.18',
            'Pod::Simple::PullParserStartToken'=> '3.18',
            'Pod::Simple::PullParserTextToken'=> '3.18',
            'Pod::Simple::PullParserToken'=> '3.18',
            'Pod::Simple::RTF'      => '3.18',
            'Pod::Simple::Search'   => '3.18',
            'Pod::Simple::SimpleTree'=> '3.18',
            'Pod::Simple::Text'     => '3.18',
            'Pod::Simple::TextContent'=> '3.18',
            'Pod::Simple::TiedOutFH'=> '3.18',
            'Pod::Simple::Transcode'=> '3.18',
            'Pod::Simple::TranscodeDumb'=> '3.18',
            'Pod::Simple::TranscodeSmart'=> '3.18',
            'Pod::Simple::XHTML'    => '3.18',
            'Pod::Simple::XMLOutStream'=> '3.18',
            'Storable'              => '2.31',
            'Sys::Syslog::Win32'    => undef,
            'Time::HiRes'           => '1.9724',
            'Unicode::Collate'      => '0.77',
            'Unicode::UCD'          => '0.33',
            'Win32API::File'        => '0.1200',
            'XS::APItest'           => '0.30',
            'attributes'            => '0.15',
            'bigint'                => '0.28',
            'bignum'                => '0.28',
            'charnames'             => '1.23',
            'diagnostics'           => '1.23',
            'feature'               => '1.22',
            'overload'              => '1.15',
            'perlfaq'               => '5.015000',
            'threads'               => '1.84',
            'version'               => '0.93',
        },
        removed => {
            'ExtUtils::MakeMaker::YAML'=> 1,
            'Locale::Constants'     => 1,
            'Sys::Syslog::win32::Win32'=> 1,
        }
    },
    5.015002 => {
        delta_from => 5.015001,
        changed => {
            'Attribute::Handlers'   => '0.92',
            'B'                     => '1.31',
            'B::Concise'            => '0.85',
            'B::Deparse'            => '1.07',
            'B::Terse'              => '1.06',
            'B::Xref'               => '1.03',
            'CPAN'                  => '1.9800',
            'CPAN::Exception::yaml_process_error'=> '5.5',
            'CPAN::Meta'            => '2.112150',
            'CPAN::Meta::Converter' => '2.112150',
            'CPAN::Meta::Feature'   => '2.112150',
            'CPAN::Meta::History'   => '2.112150',
            'CPAN::Meta::Prereqs'   => '2.112150',
            'CPAN::Meta::Spec'      => '2.112150',
            'CPAN::Meta::Validator' => '2.112150',
            'CPANPLUS'              => '0.9109',
            'CPANPLUS::Internals'   => '0.9109',
            'CPANPLUS::Shell::Default'=> '0.9109',
            'DB_File'               => '1.824',
            'Data::Dumper'          => '2.132',
            'Encode'                => '2.44',
            'Encode::Alias'         => '2.15',
            'Encode::Encoder'       => '2.02',
            'Encode::Guess'         => '2.05',
            'ExtUtils::Command::MM' => '6.59',
            'ExtUtils::Install'     => '1.57',
            'ExtUtils::Installed'   => '1.999002',
            'ExtUtils::Liblist'     => '6.59',
            'ExtUtils::Liblist::Kid'=> '6.59',
            'ExtUtils::MM'          => '6.59',
            'ExtUtils::MM_AIX'      => '6.59',
            'ExtUtils::MM_Any'      => '6.59',
            'ExtUtils::MM_BeOS'     => '6.59',
            'ExtUtils::MM_Cygwin'   => '6.59',
            'ExtUtils::MM_DOS'      => '6.59',
            'ExtUtils::MM_Darwin'   => '6.59',
            'ExtUtils::MM_MacOS'    => '6.59',
            'ExtUtils::MM_NW5'      => '6.59',
            'ExtUtils::MM_OS2'      => '6.59',
            'ExtUtils::MM_QNX'      => '6.59',
            'ExtUtils::MM_UWIN'     => '6.59',
            'ExtUtils::MM_Unix'     => '6.59',
            'ExtUtils::MM_VMS'      => '6.59',
            'ExtUtils::MM_VOS'      => '6.59',
            'ExtUtils::MM_Win32'    => '6.59',
            'ExtUtils::MM_Win95'    => '6.59',
            'ExtUtils::MY'          => '6.59',
            'ExtUtils::MakeMaker'   => '6.59',
            'ExtUtils::MakeMaker::Config'=> '6.59',
            'ExtUtils::Manifest'    => '1.60',
            'ExtUtils::Mkbootstrap' => '6.59',
            'ExtUtils::Mksymlists'  => '6.59',
            'ExtUtils::ParseXS'     => '3.03_01',
            'ExtUtils::Typemaps'    => '1.01',
            'ExtUtils::testlib'     => '6.59',
            'File::Spec'            => '3.34',
            'File::Spec::Mac'       => '3.35',
            'File::Spec::Unix'      => '3.34',
            'File::Spec::VMS'       => '3.35',
            'File::Spec::Win32'     => '3.35',
            'I18N::LangTags'        => '0.37',
            'IO'                    => '1.25_05',
            'IO::Handle'            => '1.32',
            'IO::Socket'            => '1.33',
            'IO::Socket::INET'      => '1.32',
            'IPC::Open3'            => '1.12',
            'Math::BigFloat'        => '1.995',
            'Math::BigFloat::Trace' => '0.29',
            'Math::BigInt'          => '1.996',
            'Math::BigInt::Trace'   => '0.29',
            'Module::Build'         => '0.39_01',
            'Module::Build::Base'   => '0.39_01',
            'Module::Build::Compat' => '0.39_01',
            'Module::Build::Config' => '0.39_01',
            'Module::Build::Cookbook'=> '0.39_01',
            'Module::Build::Dumper' => '0.39_01',
            'Module::Build::ModuleInfo'=> '0.39_01',
            'Module::Build::Notes'  => '0.39_01',
            'Module::Build::PPMMaker'=> '0.39_01',
            'Module::Build::Platform::Amiga'=> '0.39_01',
            'Module::Build::Platform::Default'=> '0.39_01',
            'Module::Build::Platform::EBCDIC'=> '0.39_01',
            'Module::Build::Platform::MPEiX'=> '0.39_01',
            'Module::Build::Platform::MacOS'=> '0.39_01',
            'Module::Build::Platform::RiscOS'=> '0.39_01',
            'Module::Build::Platform::Unix'=> '0.39_01',
            'Module::Build::Platform::VMS'=> '0.39_01',
            'Module::Build::Platform::VOS'=> '0.39_01',
            'Module::Build::Platform::Windows'=> '0.39_01',
            'Module::Build::Platform::aix'=> '0.39_01',
            'Module::Build::Platform::cygwin'=> '0.39_01',
            'Module::Build::Platform::darwin'=> '0.39_01',
            'Module::Build::Platform::os2'=> '0.39_01',
            'Module::Build::PodParser'=> '0.39_01',
            'Module::CoreList'      => '2.55',
            'Module::Load'          => '0.20',
            'Module::Metadata'      => '1.000005_01',
            'Opcode'                => '1.20',
            'Params::Check'         => '0.32',
            'PerlIO::via'           => '0.12',
            'Term::ANSIColor'       => '3.01',
            'Unicode::Collate'      => '0.78',
            'Unicode::Normalize'    => '1.13',
            'Unicode::UCD'          => '0.34',
            'bigint'                => '0.29',
            'bignum'                => '0.29',
            'bigrat'                => '0.29',
            'diagnostics'           => '1.24',
            'fields'                => '2.16',
            'inc::latest'           => '0.39_01',
        },
        removed => {
        }
    },
    5.015003 => {
        delta_from => 5.015002,
        changed => {
            'AnyDBM_File'           => '1.01',
            'Archive::Extract'      => '0.56',
            'Archive::Tar'          => '1.78',
            'Archive::Tar::Constant'=> '1.78',
            'Archive::Tar::File'    => '1.78',
            'Attribute::Handlers'   => '0.93',
            'B'                     => '1.32',
            'B::Concise'            => '0.86',
            'B::Deparse'            => '1.08',
            'CPAN::Meta'            => '2.112621',
            'CPAN::Meta::Converter' => '2.112621',
            'CPAN::Meta::Feature'   => '2.112621',
            'CPAN::Meta::History'   => '2.112621',
            'CPAN::Meta::Prereqs'   => '2.112621',
            'CPAN::Meta::Spec'      => '2.112621',
            'CPAN::Meta::Validator' => '2.112621',
            'CPAN::Meta::YAML'      => '0.004',
            'CPANPLUS'              => '0.9111',
            'CPANPLUS::Dist::Build' => '0.58',
            'CPANPLUS::Dist::Build::Constants'=> '0.58',
            'CPANPLUS::Internals'   => '0.9111',
            'CPANPLUS::Shell::Default'=> '0.9111',
            'Carp'                  => '1.23',
            'Carp::Heavy'           => '1.23',
            'Data::Dumper'          => '2.134',
            'Devel::PPPort'         => '3.20',
            'Errno'                 => '1.14',
            'Exporter'              => '5.65',
            'Exporter::Heavy'       => '5.65',
            'ExtUtils::ParseXS'     => '3.04_04',
            'ExtUtils::ParseXS::Constants'=> '3.04_04',
            'ExtUtils::ParseXS::CountLines'=> '3.04_04',
            'ExtUtils::ParseXS::Utilities'=> '3.04_04',
            'ExtUtils::Typemaps'    => '1.02',
            'File::Glob'            => '1.13',
            'Filter::Simple'        => '0.88',
            'IO'                    => '1.25_06',
            'IO::Handle'            => '1.33',
            'Locale::Codes'         => '3.18',
            'Locale::Codes::Constants'=> '3.18',
            'Locale::Codes::Country'=> '3.18',
            'Locale::Codes::Country_Codes'=> '3.18',
            'Locale::Codes::Currency'=> '3.18',
            'Locale::Codes::Currency_Codes'=> '3.18',
            'Locale::Codes::LangExt'=> '3.18',
            'Locale::Codes::LangExt_Codes'=> '3.18',
            'Locale::Codes::LangVar'=> '3.18',
            'Locale::Codes::LangVar_Codes'=> '3.18',
            'Locale::Codes::Language'=> '3.18',
            'Locale::Codes::Language_Codes'=> '3.18',
            'Locale::Codes::Script' => '3.18',
            'Locale::Codes::Script_Codes'=> '3.18',
            'Locale::Country'       => '3.18',
            'Locale::Currency'      => '3.18',
            'Locale::Language'      => '3.18',
            'Locale::Script'        => '3.18',
            'Math::BigFloat'        => '1.997',
            'Math::BigInt'          => '1.997',
            'Math::BigInt::Calc'    => '1.997',
            'Math::BigInt::CalcEmu' => '1.997',
            'Math::BigInt::FastCalc'=> '0.30',
            'Math::BigRat'          => '0.2603',
            'Module::CoreList'      => '2.56',
            'Module::Load::Conditional'=> '0.46',
            'Module::Metadata'      => '1.000007',
            'ODBM_File'             => '1.12',
            'POSIX'                 => '1.26',
            'Pod::Perldoc'          => '3.15_07',
            'Pod::Simple'           => '3.19',
            'Pod::Simple::BlackBox' => '3.19',
            'Pod::Simple::Checker'  => '3.19',
            'Pod::Simple::Debug'    => '3.19',
            'Pod::Simple::DumpAsText'=> '3.19',
            'Pod::Simple::DumpAsXML'=> '3.19',
            'Pod::Simple::HTML'     => '3.19',
            'Pod::Simple::HTMLBatch'=> '3.19',
            'Pod::Simple::LinkSection'=> '3.19',
            'Pod::Simple::Methody'  => '3.19',
            'Pod::Simple::Progress' => '3.19',
            'Pod::Simple::PullParser'=> '3.19',
            'Pod::Simple::PullParserEndToken'=> '3.19',
            'Pod::Simple::PullParserStartToken'=> '3.19',
            'Pod::Simple::PullParserTextToken'=> '3.19',
            'Pod::Simple::PullParserToken'=> '3.19',
            'Pod::Simple::RTF'      => '3.19',
            'Pod::Simple::Search'   => '3.19',
            'Pod::Simple::SimpleTree'=> '3.19',
            'Pod::Simple::Text'     => '3.19',
            'Pod::Simple::TextContent'=> '3.19',
            'Pod::Simple::TiedOutFH'=> '3.19',
            'Pod::Simple::Transcode'=> '3.19',
            'Pod::Simple::TranscodeDumb'=> '3.19',
            'Pod::Simple::TranscodeSmart'=> '3.19',
            'Pod::Simple::XHTML'    => '3.19',
            'Pod::Simple::XMLOutStream'=> '3.19',
            'Search::Dict'          => '1.04',
            'Socket'                => '1.94_01',
            'Storable'              => '2.32',
            'Text::Abbrev'          => '1.02',
            'Tie::Array'            => '1.05',
            'UNIVERSAL'             => '1.09',
            'Unicode::UCD'          => '0.35',
            'XS::APItest'           => '0.31',
            'XSLoader'              => '0.16',
            'attributes'            => '0.16',
            'diagnostics'           => '1.25',
            'open'                  => '1.09',
            'perlfaq'               => '5.0150034',
            'threads'               => '1.85',
            'threads::shared'       => '1.40',
        },
        removed => {
        }
    },
    5.015004 => {
        delta_from => 5.015003,
        changed => {
            'Archive::Tar'          => '1.80',
            'Archive::Tar::Constant'=> '1.80',
            'Archive::Tar::File'    => '1.80',
            'Digest'                => '1.17',
            'DynaLoader'            => '1.14',
            'ExtUtils::Command::MM' => '6.61_01',
            'ExtUtils::Liblist'     => '6.61_01',
            'ExtUtils::Liblist::Kid'=> '6.61_01',
            'ExtUtils::MM'          => '6.61_01',
            'ExtUtils::MM_AIX'      => '6.61_01',
            'ExtUtils::MM_Any'      => '6.61_01',
            'ExtUtils::MM_BeOS'     => '6.61_01',
            'ExtUtils::MM_Cygwin'   => '6.61_01',
            'ExtUtils::MM_DOS'      => '6.61_01',
            'ExtUtils::MM_Darwin'   => '6.61_01',
            'ExtUtils::MM_MacOS'    => '6.61_01',
            'ExtUtils::MM_NW5'      => '6.61_01',
            'ExtUtils::MM_OS2'      => '6.61_01',
            'ExtUtils::MM_QNX'      => '6.61_01',
            'ExtUtils::MM_UWIN'     => '6.61_01',
            'ExtUtils::MM_Unix'     => '6.61_01',
            'ExtUtils::MM_VMS'      => '6.61_01',
            'ExtUtils::MM_VOS'      => '6.61_01',
            'ExtUtils::MM_Win32'    => '6.61_01',
            'ExtUtils::MM_Win95'    => '6.61_01',
            'ExtUtils::MY'          => '6.61_01',
            'ExtUtils::MakeMaker'   => '6.61_01',
            'ExtUtils::MakeMaker::Config'=> '6.61_01',
            'ExtUtils::Mkbootstrap' => '6.61_01',
            'ExtUtils::Mksymlists'  => '6.61_01',
            'ExtUtils::ParseXS'     => '3.05',
            'ExtUtils::ParseXS::Constants'=> '3.05',
            'ExtUtils::ParseXS::CountLines'=> '3.05',
            'ExtUtils::ParseXS::Utilities'=> '3.05',
            'ExtUtils::testlib'     => '6.61_01',
            'File::DosGlob'         => '1.05',
            'Module::CoreList'      => '2.57',
            'Module::Load'          => '0.22',
            'Unicode::Collate'      => '0.80',
            'Unicode::Collate::Locale'=> '0.80',
            'Unicode::UCD'          => '0.36',
            'XS::APItest'           => '0.32',
            'XS::Typemap'           => '0.07',
            'attributes'            => '0.17',
            'base'                  => '2.18',
            'constant'              => '1.23',
            'mro'                   => '1.09',
            'open'                  => '1.10',
            'perlfaq'               => '5.0150035',
        },
        removed => {
        }
    },
    5.015005 => {
        delta_from => 5.015004,
        changed => {
            'Archive::Extract'      => '0.58',
            'B::Concise'            => '0.87',
            'B::Deparse'            => '1.09',
            'CGI'                   => '3.58',
            'CGI::Fast'             => '1.09',
            'CPANPLUS'              => '0.9112',
            'CPANPLUS::Dist::Build' => '0.60',
            'CPANPLUS::Dist::Build::Constants'=> '0.60',
            'CPANPLUS::Internals'   => '0.9112',
            'CPANPLUS::Shell::Default'=> '0.9112',
            'Compress::Raw::Bzip2'  => '2.042',
            'Compress::Raw::Zlib'   => '2.042',
            'Compress::Zlib'        => '2.042',
            'Digest::SHA'           => '5.63',
            'Errno'                 => '1.15',
            'ExtUtils::Command::MM' => '6.63_02',
            'ExtUtils::Liblist'     => '6.63_02',
            'ExtUtils::Liblist::Kid'=> '6.63_02',
            'ExtUtils::MM'          => '6.63_02',
            'ExtUtils::MM_AIX'      => '6.63_02',
            'ExtUtils::MM_Any'      => '6.63_02',
            'ExtUtils::MM_BeOS'     => '6.63_02',
            'ExtUtils::MM_Cygwin'   => '6.63_02',
            'ExtUtils::MM_DOS'      => '6.63_02',
            'ExtUtils::MM_Darwin'   => '6.63_02',
            'ExtUtils::MM_MacOS'    => '6.63_02',
            'ExtUtils::MM_NW5'      => '6.63_02',
            'ExtUtils::MM_OS2'      => '6.63_02',
            'ExtUtils::MM_QNX'      => '6.63_02',
            'ExtUtils::MM_UWIN'     => '6.63_02',
            'ExtUtils::MM_Unix'     => '6.63_02',
            'ExtUtils::MM_VMS'      => '6.63_02',
            'ExtUtils::MM_VOS'      => '6.63_02',
            'ExtUtils::MM_Win32'    => '6.63_02',
            'ExtUtils::MM_Win95'    => '6.63_02',
            'ExtUtils::MY'          => '6.63_02',
            'ExtUtils::MakeMaker'   => '6.63_02',
            'ExtUtils::MakeMaker::Config'=> '6.63_02',
            'ExtUtils::Mkbootstrap' => '6.63_02',
            'ExtUtils::Mksymlists'  => '6.63_02',
            'ExtUtils::testlib'     => '6.63_02',
            'File::DosGlob'         => '1.06',
            'File::Glob'            => '1.14',
            'HTTP::Tiny'            => '0.016',
            'IO::Compress::Adapter::Bzip2'=> '2.042',
            'IO::Compress::Adapter::Deflate'=> '2.042',
            'IO::Compress::Adapter::Identity'=> '2.042',
            'IO::Compress::Base'    => '2.042',
            'IO::Compress::Base::Common'=> '2.042',
            'IO::Compress::Bzip2'   => '2.042',
            'IO::Compress::Deflate' => '2.042',
            'IO::Compress::Gzip'    => '2.042',
            'IO::Compress::Gzip::Constants'=> '2.042',
            'IO::Compress::RawDeflate'=> '2.042',
            'IO::Compress::Zip'     => '2.042',
            'IO::Compress::Zip::Constants'=> '2.042',
            'IO::Compress::Zlib::Constants'=> '2.042',
            'IO::Compress::Zlib::Extra'=> '2.042',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.042',
            'IO::Uncompress::Adapter::Identity'=> '2.042',
            'IO::Uncompress::Adapter::Inflate'=> '2.042',
            'IO::Uncompress::AnyInflate'=> '2.042',
            'IO::Uncompress::AnyUncompress'=> '2.042',
            'IO::Uncompress::Base'  => '2.042',
            'IO::Uncompress::Bunzip2'=> '2.042',
            'IO::Uncompress::Gunzip'=> '2.042',
            'IO::Uncompress::Inflate'=> '2.042',
            'IO::Uncompress::RawInflate'=> '2.042',
            'IO::Uncompress::Unzip' => '2.042',
            'Locale::Maketext'      => '1.20',
            'Locale::Maketext::Guts'=> '1.20',
            'Locale::Maketext::GutsLoader'=> '1.20',
            'Module::CoreList'      => '2.58',
            'Opcode'                => '1.21',
            'Socket'                => '1.94_02',
            'Storable'              => '2.33',
            'UNIVERSAL'             => '1.10',
            'Unicode::Collate'      => '0.85',
            'Unicode::Collate::CJK::Pinyin'=> '0.85',
            'Unicode::Collate::CJK::Stroke'=> '0.85',
            'Unicode::Collate::Locale'=> '0.85',
            'Unicode::UCD'          => '0.37',
            'XS::APItest'           => '0.33',
            'arybase'               => '0.01',
            'charnames'             => '1.24',
            'feature'               => '1.23',
            'perlfaq'               => '5.0150036',
            'strict'                => '1.05',
            'unicore::Name'         => undef,
        },
        removed => {
        }
    },
    5.015006 => {
        delta_from => 5.015005,
        changed => {
            'Archive::Tar'          => '1.82',
            'Archive::Tar::Constant'=> '1.82',
            'Archive::Tar::File'    => '1.82',
            'AutoLoader'            => '5.72',
            'B::Concise'            => '0.88',
            'B::Debug'              => '1.17',
            'B::Deparse'            => '1.10',
            'CPAN::Meta::YAML'      => '0.005',
            'CPANPLUS'              => '0.9113',
            'CPANPLUS::Internals'   => '0.9113',
            'CPANPLUS::Shell::Default'=> '0.9113',
            'Carp'                  => '1.24',
            'Compress::Raw::Bzip2'  => '2.045',
            'Compress::Raw::Zlib'   => '2.045',
            'Compress::Zlib'        => '2.045',
            'Cwd'                   => '3.38',
            'DB'                    => '1.04',
            'Data::Dumper'          => '2.135_01',
            'Digest::SHA'           => '5.70',
            'Dumpvalue'             => '1.17',
            'Exporter'              => '5.66',
            'Exporter::Heavy'       => '5.66',
            'ExtUtils::CBuilder'    => '0.280205',
            'ExtUtils::CBuilder::Platform::os2'=> '0.280204',
            'ExtUtils::Packlist'    => '1.45',
            'ExtUtils::ParseXS'     => '3.08',
            'ExtUtils::ParseXS::Constants'=> '3.08',
            'ExtUtils::ParseXS::CountLines'=> '3.08',
            'ExtUtils::ParseXS::Utilities'=> '3.08',
            'File::Basename'        => '2.84',
            'File::Glob'            => '1.15',
            'File::Spec::Unix'      => '3.35',
            'Getopt::Std'           => '1.07',
            'I18N::LangTags'        => '0.38',
            'IO::Compress::Adapter::Bzip2'=> '2.045',
            'IO::Compress::Adapter::Deflate'=> '2.045',
            'IO::Compress::Adapter::Identity'=> '2.045',
            'IO::Compress::Base'    => '2.046',
            'IO::Compress::Base::Common'=> '2.045',
            'IO::Compress::Bzip2'   => '2.045',
            'IO::Compress::Deflate' => '2.045',
            'IO::Compress::Gzip'    => '2.045',
            'IO::Compress::Gzip::Constants'=> '2.045',
            'IO::Compress::RawDeflate'=> '2.045',
            'IO::Compress::Zip'     => '2.046',
            'IO::Compress::Zip::Constants'=> '2.045',
            'IO::Compress::Zlib::Constants'=> '2.045',
            'IO::Compress::Zlib::Extra'=> '2.045',
            'IO::Dir'               => '1.09',
            'IO::File'              => '1.16',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.045',
            'IO::Uncompress::Adapter::Identity'=> '2.045',
            'IO::Uncompress::Adapter::Inflate'=> '2.045',
            'IO::Uncompress::AnyInflate'=> '2.045',
            'IO::Uncompress::AnyUncompress'=> '2.045',
            'IO::Uncompress::Base'  => '2.046',
            'IO::Uncompress::Bunzip2'=> '2.045',
            'IO::Uncompress::Gunzip'=> '2.045',
            'IO::Uncompress::Inflate'=> '2.045',
            'IO::Uncompress::RawInflate'=> '2.045',
            'IO::Uncompress::Unzip' => '2.046',
            'Locale::Codes'         => '3.20',
            'Locale::Codes::Constants'=> '3.20',
            'Locale::Codes::Country'=> '3.20',
            'Locale::Codes::Country_Codes'=> '3.20',
            'Locale::Codes::Country_Retired'=> '3.20',
            'Locale::Codes::Currency'=> '3.20',
            'Locale::Codes::Currency_Codes'=> '3.20',
            'Locale::Codes::Currency_Retired'=> '3.20',
            'Locale::Codes::LangExt'=> '3.20',
            'Locale::Codes::LangExt_Codes'=> '3.20',
            'Locale::Codes::LangExt_Retired'=> '3.20',
            'Locale::Codes::LangFam'=> '3.20',
            'Locale::Codes::LangFam_Codes'=> '3.20',
            'Locale::Codes::LangFam_Retired'=> '3.20',
            'Locale::Codes::LangVar'=> '3.20',
            'Locale::Codes::LangVar_Codes'=> '3.20',
            'Locale::Codes::LangVar_Retired'=> '3.20',
            'Locale::Codes::Language'=> '3.20',
            'Locale::Codes::Language_Codes'=> '3.20',
            'Locale::Codes::Language_Retired'=> '3.20',
            'Locale::Codes::Script' => '3.20',
            'Locale::Codes::Script_Codes'=> '3.20',
            'Locale::Codes::Script_Retired'=> '3.20',
            'Locale::Country'       => '3.20',
            'Locale::Currency'      => '3.20',
            'Locale::Language'      => '3.20',
            'Locale::Maketext'      => '1.21',
            'Locale::Script'        => '3.20',
            'Module::CoreList'      => '2.59',
            'Module::Loaded'        => '0.08',
            'Opcode'                => '1.22',
            'POSIX'                 => '1.27',
            'Pod::Html'             => '1.12',
            'Pod::LaTeX'            => '0.60',
            'Pod::Perldoc'          => '3.15_08',
            'Safe'                  => '2.30',
            'SelfLoader'            => '1.20',
            'Socket'                => '1.97',
            'Storable'              => '2.34',
            'UNIVERSAL'             => '1.11',
            'Unicode::Collate'      => '0.87',
            'Unicode::Collate::Locale'=> '0.87',
            'XS::APItest'           => '0.34',
            'arybase'               => '0.02',
            'charnames'             => '1.27',
            'diagnostics'           => '1.26',
            'feature'               => '1.24',
            'if'                    => '0.0602',
            'overload'              => '1.16',
            'sigtrap'               => '1.06',
            'strict'                => '1.06',
            'threads'               => '1.86',
            'version'               => '0.96',
        },
        removed => {
        }
    },
    5.015007 => {
        delta_from => 5.015006,
        changed => {
            'B'                     => '1.33',
            'B::Deparse'            => '1.11',
            'CGI'                   => '3.59',
            'CPAN::Meta'            => '2.113640',
            'CPAN::Meta::Converter' => '2.113640',
            'CPAN::Meta::Feature'   => '2.113640',
            'CPAN::Meta::History'   => '2.113640',
            'CPAN::Meta::Prereqs'   => '2.113640',
            'CPAN::Meta::Requirements'=> '2.113640',
            'CPAN::Meta::Spec'      => '2.113640',
            'CPAN::Meta::Validator' => '2.113640',
            'CPANPLUS'              => '0.9116',
            'CPANPLUS::Internals'   => '0.9116',
            'CPANPLUS::Shell::Default'=> '0.9116',
            'Cwd'                   => '3.39_01',
            'Data::Dumper'          => '2.135_03',
            'Devel::InnerPackage'   => '0.4',
            'ExtUtils::CBuilder::Base'=> '0.280205',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.280205',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.280205',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.280205',
            'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280205',
            'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280205',
            'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280205',
            'ExtUtils::CBuilder::Platform::aix'=> '0.280205',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.280205',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.280205',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280205',
            'ExtUtils::CBuilder::Platform::os2'=> '0.280205',
            'ExtUtils::Manifest'    => '1.61',
            'ExtUtils::Packlist'    => '1.46',
            'ExtUtils::ParseXS'     => '3.12',
            'ExtUtils::ParseXS::Constants'=> '3.12',
            'ExtUtils::ParseXS::CountLines'=> '3.12',
            'ExtUtils::ParseXS::Utilities'=> '3.12',
            'ExtUtils::Typemaps'    => '1.03',
            'ExtUtils::Typemaps::Cmd'=> undef,
            'ExtUtils::Typemaps::Type'=> '0.06',
            'File::Glob'            => '1.16',
            'File::Spec'            => '3.39_01',
            'File::Spec::Cygwin'    => '3.39_01',
            'File::Spec::Epoc'      => '3.39_01',
            'File::Spec::Functions' => '3.39_01',
            'File::Spec::Mac'       => '3.39_01',
            'File::Spec::OS2'       => '3.39_01',
            'File::Spec::Unix'      => '3.39_01',
            'File::Spec::VMS'       => '3.39_01',
            'File::Spec::Win32'     => '3.39_01',
            'IO::Dir'               => '1.10',
            'IO::Pipe'              => '1.15',
            'IO::Poll'              => '0.09',
            'IO::Select'            => '1.21',
            'IO::Socket'            => '1.34',
            'IO::Socket::INET'      => '1.33',
            'IO::Socket::UNIX'      => '1.24',
            'Locale::Maketext'      => '1.22',
            'Math::BigInt'          => '1.998',
            'Module::CoreList'      => '2.60',
            'Module::Pluggable'     => '4.0',
            'POSIX'                 => '1.28',
            'PerlIO::scalar'        => '0.13',
            'Pod::Html'             => '1.13',
            'Pod::Perldoc'          => '3.15_15',
            'Pod::Perldoc::BaseTo'  => '3.15_15',
            'Pod::Perldoc::GetOptsOO'=> '3.15_15',
            'Pod::Perldoc::ToANSI'  => '3.15_15',
            'Pod::Perldoc::ToChecker'=> '3.15_15',
            'Pod::Perldoc::ToMan'   => '3.15_15',
            'Pod::Perldoc::ToNroff' => '3.15_15',
            'Pod::Perldoc::ToPod'   => '3.15_15',
            'Pod::Perldoc::ToRtf'   => '3.15_15',
            'Pod::Perldoc::ToTerm'  => '3.15_15',
            'Pod::Perldoc::ToText'  => '3.15_15',
            'Pod::Perldoc::ToTk'    => '3.15_15',
            'Pod::Perldoc::ToXml'   => '3.15_15',
            'Term::UI'              => '0.30',
            'Tie::File'             => '0.98',
            'Unicode::UCD'          => '0.39',
            'Version::Requirements' => '0.101021',
            'XS::APItest'           => '0.35',
            '_charnames'            => '1.28',
            'arybase'               => '0.03',
            'autouse'               => '1.07',
            'charnames'             => '1.28',
            'diagnostics'           => '1.27',
            'feature'               => '1.25',
            'overload'              => '1.17',
            'overloading'           => '0.02',
            'perlfaq'               => '5.0150038',
        },
        removed => {
        }
    },
    5.015008 => {
        delta_from => 5.015007,
        changed => {
            'B'                     => '1.34',
            'B::Deparse'            => '1.12',
            'CPAN::Meta'            => '2.120351',
            'CPAN::Meta::Converter' => '2.120351',
            'CPAN::Meta::Feature'   => '2.120351',
            'CPAN::Meta::History'   => '2.120351',
            'CPAN::Meta::Prereqs'   => '2.120351',
            'CPAN::Meta::Requirements'=> '2.120351',
            'CPAN::Meta::Spec'      => '2.120351',
            'CPAN::Meta::Validator' => '2.120351',
            'CPAN::Meta::YAML'      => '0.007',
            'CPANPLUS'              => '0.9118',
            'CPANPLUS::Dist::Build' => '0.62',
            'CPANPLUS::Dist::Build::Constants'=> '0.62',
            'CPANPLUS::Internals'   => '0.9118',
            'CPANPLUS::Shell::Default'=> '0.9118',
            'Carp'                  => '1.25',
            'Carp::Heavy'           => '1.25',
            'Compress::Raw::Bzip2'  => '2.048',
            'Compress::Raw::Zlib'   => '2.048',
            'Compress::Zlib'        => '2.048',
            'Cwd'                   => '3.39_02',
            'DB_File'               => '1.826',
            'Data::Dumper'          => '2.135_05',
            'English'               => '1.05',
            'ExtUtils::Install'     => '1.58',
            'ExtUtils::ParseXS'     => '3.16',
            'ExtUtils::ParseXS::Constants'=> '3.16',
            'ExtUtils::ParseXS::CountLines'=> '3.16',
            'ExtUtils::ParseXS::Utilities'=> '3.16',
            'ExtUtils::Typemaps'    => '3.16',
            'ExtUtils::Typemaps::Cmd'=> '3.16',
            'ExtUtils::Typemaps::InputMap'=> '3.16',
            'ExtUtils::Typemaps::OutputMap'=> '3.16',
            'ExtUtils::Typemaps::Type'=> '3.16',
            'File::Copy'            => '2.23',
            'File::Glob'            => '1.17',
            'File::Spec'            => '3.39_02',
            'File::Spec::Cygwin'    => '3.39_02',
            'File::Spec::Epoc'      => '3.39_02',
            'File::Spec::Functions' => '3.39_02',
            'File::Spec::Mac'       => '3.39_02',
            'File::Spec::OS2'       => '3.39_02',
            'File::Spec::Unix'      => '3.39_02',
            'File::Spec::VMS'       => '3.39_02',
            'File::Spec::Win32'     => '3.39_02',
            'Filter::Util::Call'    => '1.40',
            'IO::Compress::Adapter::Bzip2'=> '2.048',
            'IO::Compress::Adapter::Deflate'=> '2.048',
            'IO::Compress::Adapter::Identity'=> '2.048',
            'IO::Compress::Base'    => '2.048',
            'IO::Compress::Base::Common'=> '2.048',
            'IO::Compress::Bzip2'   => '2.048',
            'IO::Compress::Deflate' => '2.048',
            'IO::Compress::Gzip'    => '2.048',
            'IO::Compress::Gzip::Constants'=> '2.048',
            'IO::Compress::RawDeflate'=> '2.048',
            'IO::Compress::Zip'     => '2.048',
            'IO::Compress::Zip::Constants'=> '2.048',
            'IO::Compress::Zlib::Constants'=> '2.048',
            'IO::Compress::Zlib::Extra'=> '2.048',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.048',
            'IO::Uncompress::Adapter::Identity'=> '2.048',
            'IO::Uncompress::Adapter::Inflate'=> '2.048',
            'IO::Uncompress::AnyInflate'=> '2.048',
            'IO::Uncompress::AnyUncompress'=> '2.048',
            'IO::Uncompress::Base'  => '2.048',
            'IO::Uncompress::Bunzip2'=> '2.048',
            'IO::Uncompress::Gunzip'=> '2.048',
            'IO::Uncompress::Inflate'=> '2.048',
            'IO::Uncompress::RawInflate'=> '2.048',
            'IO::Uncompress::Unzip' => '2.048',
            'IPC::Cmd'              => '0.76',
            'Math::Complex'         => '1.59',
            'Math::Trig'            => '1.23',
            'Module::Metadata'      => '1.000009',
            'Opcode'                => '1.23',
            'POSIX'                 => '1.30',
            'Parse::CPAN::Meta'     => '1.4402',
            'PerlIO::mmap'          => '0.010',
            'Pod::Checker'          => '1.51',
            'Pod::Find'             => '1.51',
            'Pod::Functions'        => '1.05',
            'Pod::Html'             => '1.14',
            'Pod::InputObjects'     => '1.51',
            'Pod::ParseUtils'       => '1.51',
            'Pod::Parser'           => '1.51',
            'Pod::PlainText'        => '2.05',
            'Pod::Select'           => '1.51',
            'Pod::Usage'            => '1.51',
            'Safe'                  => '2.31',
            'Socket'                => '1.98',
            'Term::Cap'             => '1.13',
            'Term::ReadLine'        => '1.08',
            'Time::HiRes'           => '1.9725',
            'Unicode'               => '6.1.0',
            'Unicode::UCD'          => '0.41',
            'Version::Requirements' => '0.101022',
            'XS::APItest'           => '0.36',
            'XS::Typemap'           => '0.08',
            '_charnames'            => '1.29',
            'arybase'               => '0.04',
            'charnames'             => '1.29',
            'diagnostics'           => '1.28',
            'feature'               => '1.26',
            'locale'                => '1.01',
            'overload'              => '1.18',
            'perlfaq'               => '5.0150039',
            're'                    => '0.19',
            'subs'                  => '1.01',
            'warnings'              => '1.13',
        },
        removed => {
        }
    },
    5.015009 => {
        delta_from => 5.015008,
        changed => {
            'B::Deparse'            => '1.13',
            'B::Lint'               => '1.14',
            'B::Lint::Debug'        => '1.14',
            'CPAN::Meta'            => '2.120630',
            'CPAN::Meta::Converter' => '2.120630',
            'CPAN::Meta::Feature'   => '2.120630',
            'CPAN::Meta::History'   => '2.120630',
            'CPAN::Meta::Prereqs'   => '2.120630',
            'CPAN::Meta::Requirements'=> '2.120630',
            'CPAN::Meta::Spec'      => '2.120630',
            'CPAN::Meta::Validator' => '2.120630',
            'CPANPLUS'              => '0.9121',
            'CPANPLUS::Internals'   => '0.9121',
            'CPANPLUS::Shell::Default'=> '0.9121',
            'Data::Dumper'          => '2.135_06',
            'Digest::SHA'           => '5.71',
            'ExtUtils::CBuilder'    => '0.280206',
            'ExtUtils::CBuilder::Base'=> '0.280206',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.280206',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.280206',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.280206',
            'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280206',
            'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280206',
            'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280206',
            'ExtUtils::CBuilder::Platform::aix'=> '0.280206',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.280206',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.280206',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280206',
            'ExtUtils::CBuilder::Platform::os2'=> '0.280206',
            'HTTP::Tiny'            => '0.017',
            'Locale::Codes'         => '3.21',
            'Locale::Codes::Constants'=> '3.21',
            'Locale::Codes::Country'=> '3.21',
            'Locale::Codes::Country_Codes'=> '3.21',
            'Locale::Codes::Country_Retired'=> '3.21',
            'Locale::Codes::Currency'=> '3.21',
            'Locale::Codes::Currency_Codes'=> '3.21',
            'Locale::Codes::Currency_Retired'=> '3.21',
            'Locale::Codes::LangExt'=> '3.21',
            'Locale::Codes::LangExt_Codes'=> '3.21',
            'Locale::Codes::LangExt_Retired'=> '3.21',
            'Locale::Codes::LangFam'=> '3.21',
            'Locale::Codes::LangFam_Codes'=> '3.21',
            'Locale::Codes::LangFam_Retired'=> '3.21',
            'Locale::Codes::LangVar'=> '3.21',
            'Locale::Codes::LangVar_Codes'=> '3.21',
            'Locale::Codes::LangVar_Retired'=> '3.21',
            'Locale::Codes::Language'=> '3.21',
            'Locale::Codes::Language_Codes'=> '3.21',
            'Locale::Codes::Language_Retired'=> '3.21',
            'Locale::Codes::Script' => '3.21',
            'Locale::Codes::Script_Codes'=> '3.21',
            'Locale::Codes::Script_Retired'=> '3.21',
            'Locale::Country'       => '3.21',
            'Locale::Currency'      => '3.21',
            'Locale::Language'      => '3.21',
            'Locale::Script'        => '3.21',
            'Module::CoreList'      => '2.65',
            'Pod::Html'             => '1.1501',
            'Pod::Perldoc'          => '3.17',
            'Pod::Perldoc::BaseTo'  => '3.17',
            'Pod::Perldoc::GetOptsOO'=> '3.17',
            'Pod::Perldoc::ToANSI'  => '3.17',
            'Pod::Perldoc::ToChecker'=> '3.17',
            'Pod::Perldoc::ToMan'   => '3.17',
            'Pod::Perldoc::ToNroff' => '3.17',
            'Pod::Perldoc::ToPod'   => '3.17',
            'Pod::Perldoc::ToRtf'   => '3.17',
            'Pod::Perldoc::ToTerm'  => '3.17',
            'Pod::Perldoc::ToText'  => '3.17',
            'Pod::Perldoc::ToTk'    => '3.17',
            'Pod::Perldoc::ToXml'   => '3.17',
            'Pod::Simple'           => '3.20',
            'Pod::Simple::BlackBox' => '3.20',
            'Pod::Simple::Checker'  => '3.20',
            'Pod::Simple::Debug'    => '3.20',
            'Pod::Simple::DumpAsText'=> '3.20',
            'Pod::Simple::DumpAsXML'=> '3.20',
            'Pod::Simple::HTML'     => '3.20',
            'Pod::Simple::HTMLBatch'=> '3.20',
            'Pod::Simple::LinkSection'=> '3.20',
            'Pod::Simple::Methody'  => '3.20',
            'Pod::Simple::Progress' => '3.20',
            'Pod::Simple::PullParser'=> '3.20',
            'Pod::Simple::PullParserEndToken'=> '3.20',
            'Pod::Simple::PullParserStartToken'=> '3.20',
            'Pod::Simple::PullParserTextToken'=> '3.20',
            'Pod::Simple::PullParserToken'=> '3.20',
            'Pod::Simple::RTF'      => '3.20',
            'Pod::Simple::Search'   => '3.20',
            'Pod::Simple::SimpleTree'=> '3.20',
            'Pod::Simple::Text'     => '3.20',
            'Pod::Simple::TextContent'=> '3.20',
            'Pod::Simple::TiedOutFH'=> '3.20',
            'Pod::Simple::Transcode'=> '3.20',
            'Pod::Simple::TranscodeDumb'=> '3.20',
            'Pod::Simple::TranscodeSmart'=> '3.20',
            'Pod::Simple::XHTML'    => '3.20',
            'Pod::Simple::XMLOutStream'=> '3.20',
            'Socket'                => '2.000',
            'Term::ReadLine'        => '1.09',
            'Unicode::Collate'      => '0.89',
            'Unicode::Collate::CJK::Korean'=> '0.88',
            'Unicode::Collate::Locale'=> '0.89',
            'Unicode::Normalize'    => '1.14',
            'Unicode::UCD'          => '0.42',
            'XS::APItest'           => '0.37',
            'arybase'               => '0.05',
            'attributes'            => '0.18',
            'charnames'             => '1.30',
            'feature'               => '1.27',
        },
        removed => {
        }
    },
    5.016 => {
        delta_from => 5.015009,
        changed => {
            'B::Concise'            => '0.89',
            'B::Deparse'            => '1.14',
            'Carp'                  => '1.26',
            'Carp::Heavy'           => '1.26',
            'IO::Socket'            => '1.35',
            'Module::CoreList'      => '2.66',
            'PerlIO::scalar'        => '0.14',
            'Pod::Html'             => '1.1502',
            'Safe'                  => '2.31_01',
            'Socket'                => '2.001',
            'Unicode::UCD'          => '0.43',
            'XS::APItest'           => '0.38',
            '_charnames'            => '1.31',
            'attributes'            => '0.19',
            'strict'                => '1.07',
            'version'               => '0.99',
        },
        removed => {
        }
    },
    5.016001 => {
        delta_from => 5.016,
        changed => {
            'B'                     => '1.35',
            'B::Deparse'            => '1.14_01',
            'List::Util'            => '1.25',
            'List::Util::PP'        => '1.25',
            'List::Util::XS'        => '1.25',
            'Module::CoreList'      => '2.70',
            'PerlIO::scalar'        => '0.14_01',
            'Scalar::Util'          => '1.25',
            'Scalar::Util::PP'      => '1.25',
            're'                    => '0.19_01',
        },
        removed => {
        }
    },
    5.016002 => {
        delta_from => 5.016001,
        changed => {
            'Module::CoreList'      => '2.76',
        },
        removed => {
        }
    },
    5.016003 => {
        delta_from => 5.016002,
        changed => {
            'Encode'                => '2.44_01',
            'Module::CoreList'      => '2.76_02',
            'XS::APItest'           => '0.39',
        },
        removed => {
        }
    },
    5.017 => {
        delta_from => 5.016,
        changed => {
            'B'                     => '1.35',
            'B::Concise'            => '0.90',
            'ExtUtils::ParseXS'     => '3.17',
            'ExtUtils::ParseXS::Utilities'=> '3.17',
            'File::DosGlob'         => '1.07',
            'File::Find'            => '1.21',
            'File::stat'            => '1.06',
            'Hash::Util'            => '0.12',
            'IO::Socket'            => '1.34',
            'Module::CoreList'      => '2.67',
            'Pod::Functions'        => '1.06',
            'Storable'              => '2.35',
            'XS::APItest'           => '0.39',
            'diagnostics'           => '1.29',
            'feature'               => '1.28',
            'overload'              => '1.19',
            'utf8'                  => '1.10',
        },
        removed => {
            'Version::Requirements' => 1,
        }
    },
    5.017001 => {
        delta_from => 5.017,
        changed => {
            'App::Prove'            => '3.25',
            'App::Prove::State'     => '3.25',
            'App::Prove::State::Result'=> '3.25',
            'App::Prove::State::Result::Test'=> '3.25',
            'Archive::Extract'      => '0.60',
            'Archive::Tar'          => '1.88',
            'Archive::Tar::Constant'=> '1.88',
            'Archive::Tar::File'    => '1.88',
            'B'                     => '1.36',
            'B::Deparse'            => '1.15',
            'CPAN::Meta'            => '2.120921',
            'CPAN::Meta::Converter' => '2.120921',
            'CPAN::Meta::Feature'   => '2.120921',
            'CPAN::Meta::History'   => '2.120921',
            'CPAN::Meta::Prereqs'   => '2.120921',
            'CPAN::Meta::Requirements'=> '2.122',
            'CPAN::Meta::Spec'      => '2.120921',
            'CPAN::Meta::Validator' => '2.120921',
            'CPAN::Meta::YAML'      => '0.008',
            'CPANPLUS'              => '0.9130',
            'CPANPLUS::Config::HomeEnv'=> '0.04',
            'CPANPLUS::Internals'   => '0.9130',
            'CPANPLUS::Shell::Default'=> '0.9130',
            'Class::Struct'         => '0.64',
            'Compress::Raw::Bzip2'  => '2.052',
            'Compress::Raw::Zlib'   => '2.054',
            'Compress::Zlib'        => '2.052',
            'Digest::MD5'           => '2.52',
            'DynaLoader'            => '1.15',
            'ExtUtils::CBuilder'    => '0.280208',
            'ExtUtils::CBuilder::Base'=> '0.280208',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.280208',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.280208',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.280208',
            'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280208',
            'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280208',
            'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280208',
            'ExtUtils::CBuilder::Platform::aix'=> '0.280208',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.280208',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.280208',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280208',
            'ExtUtils::CBuilder::Platform::os2'=> '0.280208',
            'Fatal'                 => '2.11',
            'File::DosGlob'         => '1.08',
            'File::Fetch'           => '0.34',
            'File::Spec::Unix'      => '3.39_03',
            'Filter::Util::Call'    => '1.45',
            'HTTP::Tiny'            => '0.022',
            'IO'                    => '1.25_07',
            'IO::Compress::Adapter::Bzip2'=> '2.052',
            'IO::Compress::Adapter::Deflate'=> '2.052',
            'IO::Compress::Adapter::Identity'=> '2.052',
            'IO::Compress::Base'    => '2.052',
            'IO::Compress::Base::Common'=> '2.052',
            'IO::Compress::Bzip2'   => '2.052',
            'IO::Compress::Deflate' => '2.052',
            'IO::Compress::Gzip'    => '2.052',
            'IO::Compress::Gzip::Constants'=> '2.052',
            'IO::Compress::RawDeflate'=> '2.052',
            'IO::Compress::Zip'     => '2.052',
            'IO::Compress::Zip::Constants'=> '2.052',
            'IO::Compress::Zlib::Constants'=> '2.052',
            'IO::Compress::Zlib::Extra'=> '2.052',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.052',
            'IO::Uncompress::Adapter::Identity'=> '2.052',
            'IO::Uncompress::Adapter::Inflate'=> '2.052',
            'IO::Uncompress::AnyInflate'=> '2.052',
            'IO::Uncompress::AnyUncompress'=> '2.052',
            'IO::Uncompress::Base'  => '2.052',
            'IO::Uncompress::Bunzip2'=> '2.052',
            'IO::Uncompress::Gunzip'=> '2.052',
            'IO::Uncompress::Inflate'=> '2.052',
            'IO::Uncompress::RawInflate'=> '2.052',
            'IO::Uncompress::Unzip' => '2.052',
            'IPC::Cmd'              => '0.78',
            'List::Util'            => '1.25',
            'List::Util::XS'        => '1.25',
            'Locale::Codes'         => '3.22',
            'Locale::Codes::Constants'=> '3.22',
            'Locale::Codes::Country'=> '3.22',
            'Locale::Codes::Country_Codes'=> '3.22',
            'Locale::Codes::Country_Retired'=> '3.22',
            'Locale::Codes::Currency'=> '3.22',
            'Locale::Codes::Currency_Codes'=> '3.22',
            'Locale::Codes::Currency_Retired'=> '3.22',
            'Locale::Codes::LangExt'=> '3.22',
            'Locale::Codes::LangExt_Codes'=> '3.22',
            'Locale::Codes::LangExt_Retired'=> '3.22',
            'Locale::Codes::LangFam'=> '3.22',
            'Locale::Codes::LangFam_Codes'=> '3.22',
            'Locale::Codes::LangFam_Retired'=> '3.22',
            'Locale::Codes::LangVar'=> '3.22',
            'Locale::Codes::LangVar_Codes'=> '3.22',
            'Locale::Codes::LangVar_Retired'=> '3.22',
            'Locale::Codes::Language'=> '3.22',
            'Locale::Codes::Language_Codes'=> '3.22',
            'Locale::Codes::Language_Retired'=> '3.22',
            'Locale::Codes::Script' => '3.22',
            'Locale::Codes::Script_Codes'=> '3.22',
            'Locale::Codes::Script_Retired'=> '3.22',
            'Locale::Country'       => '3.22',
            'Locale::Currency'      => '3.22',
            'Locale::Language'      => '3.22',
            'Locale::Script'        => '3.22',
            'Memoize'               => '1.03',
            'Memoize::AnyDBM_File'  => '1.03',
            'Memoize::Expire'       => '1.03',
            'Memoize::ExpireFile'   => '1.03',
            'Memoize::ExpireTest'   => '1.03',
            'Memoize::NDBM_File'    => '1.03',
            'Memoize::SDBM_File'    => '1.03',
            'Memoize::Storable'     => '1.03',
            'Module::Build'         => '0.40',
            'Module::Build::Base'   => '0.40',
            'Module::Build::Compat' => '0.40',
            'Module::Build::Config' => '0.40',
            'Module::Build::Cookbook'=> '0.40',
            'Module::Build::Dumper' => '0.40',
            'Module::Build::ModuleInfo'=> '0.40',
            'Module::Build::Notes'  => '0.40',
            'Module::Build::PPMMaker'=> '0.40',
            'Module::Build::Platform::Amiga'=> '0.40',
            'Module::Build::Platform::Default'=> '0.40',
            'Module::Build::Platform::EBCDIC'=> '0.40',
            'Module::Build::Platform::MPEiX'=> '0.40',
            'Module::Build::Platform::MacOS'=> '0.40',
            'Module::Build::Platform::RiscOS'=> '0.40',
            'Module::Build::Platform::Unix'=> '0.40',
            'Module::Build::Platform::VMS'=> '0.40',
            'Module::Build::Platform::VOS'=> '0.40',
            'Module::Build::Platform::Windows'=> '0.40',
            'Module::Build::Platform::aix'=> '0.40',
            'Module::Build::Platform::cygwin'=> '0.40',
            'Module::Build::Platform::darwin'=> '0.40',
            'Module::Build::Platform::os2'=> '0.40',
            'Module::Build::PodParser'=> '0.40',
            'Module::CoreList'      => '2.68',
            'Module::Load::Conditional'=> '0.50',
            'Object::Accessor'      => '0.44',
            'POSIX'                 => '1.31',
            'Params::Check'         => '0.36',
            'Parse::CPAN::Meta'     => '1.4404',
            'PerlIO::mmap'          => '0.011',
            'PerlIO::via::QuotedPrint'=> '0.07',
            'Pod::Html'             => '1.16',
            'Pod::Man'              => '2.26',
            'Pod::Text'             => '3.16',
            'Safe'                  => '2.33_01',
            'Scalar::Util'          => '1.25',
            'Search::Dict'          => '1.07',
            'Storable'              => '2.36',
            'TAP::Base'             => '3.25',
            'TAP::Formatter::Base'  => '3.25',
            'TAP::Formatter::Color' => '3.25',
            'TAP::Formatter::Console'=> '3.25',
            'TAP::Formatter::Console::ParallelSession'=> '3.25',
            'TAP::Formatter::Console::Session'=> '3.25',
            'TAP::Formatter::File'  => '3.25',
            'TAP::Formatter::File::Session'=> '3.25',
            'TAP::Formatter::Session'=> '3.25',
            'TAP::Harness'          => '3.25',
            'TAP::Object'           => '3.25',
            'TAP::Parser'           => '3.25',
            'TAP::Parser::Aggregator'=> '3.25',
            'TAP::Parser::Grammar'  => '3.25',
            'TAP::Parser::Iterator' => '3.25',
            'TAP::Parser::Iterator::Array'=> '3.25',
            'TAP::Parser::Iterator::Process'=> '3.25',
            'TAP::Parser::Iterator::Stream'=> '3.25',
            'TAP::Parser::IteratorFactory'=> '3.25',
            'TAP::Parser::Multiplexer'=> '3.25',
            'TAP::Parser::Result'   => '3.25',
            'TAP::Parser::Result::Bailout'=> '3.25',
            'TAP::Parser::Result::Comment'=> '3.25',
            'TAP::Parser::Result::Plan'=> '3.25',
            'TAP::Parser::Result::Pragma'=> '3.25',
            'TAP::Parser::Result::Test'=> '3.25',
            'TAP::Parser::Result::Unknown'=> '3.25',
            'TAP::Parser::Result::Version'=> '3.25',
            'TAP::Parser::Result::YAML'=> '3.25',
            'TAP::Parser::ResultFactory'=> '3.25',
            'TAP::Parser::Scheduler'=> '3.25',
            'TAP::Parser::Scheduler::Job'=> '3.25',
            'TAP::Parser::Scheduler::Spinner'=> '3.25',
            'TAP::Parser::Source'   => '3.25',
            'TAP::Parser::SourceHandler'=> '3.25',
            'TAP::Parser::SourceHandler::Executable'=> '3.25',
            'TAP::Parser::SourceHandler::File'=> '3.25',
            'TAP::Parser::SourceHandler::Handle'=> '3.25',
            'TAP::Parser::SourceHandler::Perl'=> '3.25',
            'TAP::Parser::SourceHandler::RawTAP'=> '3.25',
            'TAP::Parser::Utils'    => '3.25',
            'TAP::Parser::YAMLish::Reader'=> '3.25',
            'TAP::Parser::YAMLish::Writer'=> '3.25',
            'Term::ANSIColor'       => '3.02',
            'Test::Harness'         => '3.25',
            'Unicode'               => '6.2.0',
            'Unicode::UCD'          => '0.44',
            'XS::APItest'           => '0.40',
            '_charnames'            => '1.32',
            'attributes'            => '0.2',
            'autodie'               => '2.11',
            'autodie::exception'    => '2.11',
            'autodie::exception::system'=> '2.11',
            'autodie::hints'        => '2.11',
            'bigint'                => '0.30',
            'charnames'             => '1.32',
            'feature'               => '1.29',
            'inc::latest'           => '0.40',
            'perlfaq'               => '5.0150040',
            're'                    => '0.20',
        },
        removed => {
            'List::Util::PP'        => 1,
            'Scalar::Util::PP'      => 1,
        }
    },
    5.017002 => {
        delta_from => 5.017001,
        changed => {
            'App::Prove'            => '3.25_01',
            'App::Prove::State'     => '3.25_01',
            'App::Prove::State::Result'=> '3.25_01',
            'App::Prove::State::Result::Test'=> '3.25_01',
            'B::Concise'            => '0.91',
            'Compress::Raw::Bzip2'  => '2.05201',
            'Compress::Raw::Zlib'   => '2.05401',
            'Exporter'              => '5.67',
            'Exporter::Heavy'       => '5.67',
            'Fatal'                 => '2.12',
            'File::Fetch'           => '0.36',
            'File::stat'            => '1.07',
            'IO'                    => '1.25_08',
            'IO::Socket'            => '1.35',
            'Module::CoreList'      => '2.69',
            'PerlIO::scalar'        => '0.15',
            'Socket'                => '2.002',
            'Storable'              => '2.37',
            'TAP::Base'             => '3.25_01',
            'TAP::Formatter::Base'  => '3.25_01',
            'TAP::Formatter::Color' => '3.25_01',
            'TAP::Formatter::Console'=> '3.25_01',
            'TAP::Formatter::Console::ParallelSession'=> '3.25_01',
            'TAP::Formatter::Console::Session'=> '3.25_01',
            'TAP::Formatter::File'  => '3.25_01',
            'TAP::Formatter::File::Session'=> '3.25_01',
            'TAP::Formatter::Session'=> '3.25_01',
            'TAP::Harness'          => '3.25_01',
            'TAP::Object'           => '3.25_01',
            'TAP::Parser'           => '3.25_01',
            'TAP::Parser::Aggregator'=> '3.25_01',
            'TAP::Parser::Grammar'  => '3.25_01',
            'TAP::Parser::Iterator' => '3.25_01',
            'TAP::Parser::Iterator::Array'=> '3.25_01',
            'TAP::Parser::Iterator::Process'=> '3.25_01',
            'TAP::Parser::Iterator::Stream'=> '3.25_01',
            'TAP::Parser::IteratorFactory'=> '3.25_01',
            'TAP::Parser::Multiplexer'=> '3.25_01',
            'TAP::Parser::Result'   => '3.25_01',
            'TAP::Parser::Result::Bailout'=> '3.25_01',
            'TAP::Parser::Result::Comment'=> '3.25_01',
            'TAP::Parser::Result::Plan'=> '3.25_01',
            'TAP::Parser::Result::Pragma'=> '3.25_01',
            'TAP::Parser::Result::Test'=> '3.25_01',
            'TAP::Parser::Result::Unknown'=> '3.25_01',
            'TAP::Parser::Result::Version'=> '3.25_01',
            'TAP::Parser::Result::YAML'=> '3.25_01',
            'TAP::Parser::ResultFactory'=> '3.25_01',
            'TAP::Parser::Scheduler'=> '3.25_01',
            'TAP::Parser::Scheduler::Job'=> '3.25_01',
            'TAP::Parser::Scheduler::Spinner'=> '3.25_01',
            'TAP::Parser::Source'   => '3.25_01',
            'TAP::Parser::SourceHandler'=> '3.25_01',
            'TAP::Parser::SourceHandler::Executable'=> '3.25_01',
            'TAP::Parser::SourceHandler::File'=> '3.25_01',
            'TAP::Parser::SourceHandler::Handle'=> '3.25_01',
            'TAP::Parser::SourceHandler::Perl'=> '3.25_01',
            'TAP::Parser::SourceHandler::RawTAP'=> '3.25_01',
            'TAP::Parser::Utils'    => '3.25_01',
            'TAP::Parser::YAMLish::Reader'=> '3.25_01',
            'TAP::Parser::YAMLish::Writer'=> '3.25_01',
            'Test::Harness'         => '3.25_01',
            'Tie::StdHandle'        => '4.3',
            'XS::APItest'           => '0.41',
            'autodie'               => '2.12',
            'autodie::exception'    => '2.12',
            'autodie::exception::system'=> '2.12',
            'autodie::hints'        => '2.12',
            'diagnostics'           => '1.30',
            'overload'              => '1.20',
            're'                    => '0.21',
            'vars'                  => '1.03',
        },
        removed => {
        }
    },
    5.017003 => {
        delta_from => 5.017002,
        changed => {
            'B'                     => '1.37',
            'B::Concise'            => '0.92',
            'B::Debug'              => '1.18',
            'B::Deparse'            => '1.16',
            'CGI'                   => '3.60',
            'Compress::Raw::Bzip2'  => '2.055',
            'Compress::Raw::Zlib'   => '2.056',
            'Compress::Zlib'        => '2.055',
            'Data::Dumper'          => '2.135_07',
            'Devel::Peek'           => '1.09',
            'Encode'                => '2.47',
            'Encode::Alias'         => '2.16',
            'Encode::GSM0338'       => '2.02',
            'Encode::Unicode::UTF7' => '2.06',
            'IO::Compress::Adapter::Bzip2'=> '2.055',
            'IO::Compress::Adapter::Deflate'=> '2.055',
            'IO::Compress::Adapter::Identity'=> '2.055',
            'IO::Compress::Base'    => '2.055',
            'IO::Compress::Base::Common'=> '2.055',
            'IO::Compress::Bzip2'   => '2.055',
            'IO::Compress::Deflate' => '2.055',
            'IO::Compress::Gzip'    => '2.055',
            'IO::Compress::Gzip::Constants'=> '2.055',
            'IO::Compress::RawDeflate'=> '2.055',
            'IO::Compress::Zip'     => '2.055',
            'IO::Compress::Zip::Constants'=> '2.055',
            'IO::Compress::Zlib::Constants'=> '2.055',
            'IO::Compress::Zlib::Extra'=> '2.055',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.055',
            'IO::Uncompress::Adapter::Identity'=> '2.055',
            'IO::Uncompress::Adapter::Inflate'=> '2.055',
            'IO::Uncompress::AnyInflate'=> '2.055',
            'IO::Uncompress::AnyUncompress'=> '2.055',
            'IO::Uncompress::Base'  => '2.055',
            'IO::Uncompress::Bunzip2'=> '2.055',
            'IO::Uncompress::Gunzip'=> '2.055',
            'IO::Uncompress::Inflate'=> '2.055',
            'IO::Uncompress::RawInflate'=> '2.055',
            'IO::Uncompress::Unzip' => '2.055',
            'Module::Build'         => '0.4003',
            'Module::Build::Base'   => '0.4003',
            'Module::Build::Compat' => '0.4003',
            'Module::Build::Config' => '0.4003',
            'Module::Build::Cookbook'=> '0.4003',
            'Module::Build::Dumper' => '0.4003',
            'Module::Build::ModuleInfo'=> '0.4003',
            'Module::Build::Notes'  => '0.4003',
            'Module::Build::PPMMaker'=> '0.4003',
            'Module::Build::Platform::Amiga'=> '0.4003',
            'Module::Build::Platform::Default'=> '0.4003',
            'Module::Build::Platform::EBCDIC'=> '0.4003',
            'Module::Build::Platform::MPEiX'=> '0.4003',
            'Module::Build::Platform::MacOS'=> '0.4003',
            'Module::Build::Platform::RiscOS'=> '0.4003',
            'Module::Build::Platform::Unix'=> '0.4003',
            'Module::Build::Platform::VMS'=> '0.4003',
            'Module::Build::Platform::VOS'=> '0.4003',
            'Module::Build::Platform::Windows'=> '0.4003',
            'Module::Build::Platform::aix'=> '0.4003',
            'Module::Build::Platform::cygwin'=> '0.4003',
            'Module::Build::Platform::darwin'=> '0.4003',
            'Module::Build::Platform::os2'=> '0.4003',
            'Module::Build::PodParser'=> '0.4003',
            'Module::CoreList'      => '2.71',
            'Module::CoreList::TieHashDelta'=> '2.71',
            'Module::Load::Conditional'=> '0.54',
            'Module::Metadata'      => '1.000011',
            'Module::Pluggable'     => '4.3',
            'Module::Pluggable::Object'=> '4.3',
            'Pod::Simple'           => '3.23',
            'Pod::Simple::BlackBox' => '3.23',
            'Pod::Simple::Checker'  => '3.23',
            'Pod::Simple::Debug'    => '3.23',
            'Pod::Simple::DumpAsText'=> '3.23',
            'Pod::Simple::DumpAsXML'=> '3.23',
            'Pod::Simple::HTML'     => '3.23',
            'Pod::Simple::HTMLBatch'=> '3.23',
            'Pod::Simple::LinkSection'=> '3.23',
            'Pod::Simple::Methody'  => '3.23',
            'Pod::Simple::Progress' => '3.23',
            'Pod::Simple::PullParser'=> '3.23',
            'Pod::Simple::PullParserEndToken'=> '3.23',
            'Pod::Simple::PullParserStartToken'=> '3.23',
            'Pod::Simple::PullParserTextToken'=> '3.23',
            'Pod::Simple::PullParserToken'=> '3.23',
            'Pod::Simple::RTF'      => '3.23',
            'Pod::Simple::Search'   => '3.23',
            'Pod::Simple::SimpleTree'=> '3.23',
            'Pod::Simple::Text'     => '3.23',
            'Pod::Simple::TextContent'=> '3.23',
            'Pod::Simple::TiedOutFH'=> '3.23',
            'Pod::Simple::Transcode'=> '3.23',
            'Pod::Simple::TranscodeDumb'=> '3.23',
            'Pod::Simple::TranscodeSmart'=> '3.23',
            'Pod::Simple::XHTML'    => '3.23',
            'Pod::Simple::XMLOutStream'=> '3.23',
            'Socket'                => '2.004',
            'Storable'              => '2.38',
            'Sys::Syslog'           => '0.31',
            'Term::ReadLine'        => '1.10',
            'Text::Tabs'            => '2012.0818',
            'Text::Wrap'            => '2012.0818',
            'Time::Local'           => '1.2300',
            'Unicode::UCD'          => '0.45',
            'Win32'                 => '0.45',
            'Win32CORE'             => '0.03',
            'XS::APItest'           => '0.42',
            'inc::latest'           => '0.4003',
            'perlfaq'               => '5.0150041',
            're'                    => '0.22',
        },
        removed => {
        }
    },
    5.017004 => {
        delta_from => 5.017003,
        changed => {
            'Archive::Tar'          => '1.90',
            'Archive::Tar::Constant'=> '1.90',
            'Archive::Tar::File'    => '1.90',
            'B'                     => '1.38',
            'B::Concise'            => '0.93',
            'B::Deparse'            => '1.17',
            'B::Xref'               => '1.04',
            'CPANPLUS'              => '0.9131',
            'CPANPLUS::Internals'   => '0.9131',
            'CPANPLUS::Shell::Default'=> '0.9131',
            'DB_File'               => '1.827',
            'Devel::Peek'           => '1.10',
            'DynaLoader'            => '1.16',
            'Errno'                 => '1.16',
            'ExtUtils::ParseXS'     => '3.18',
            'ExtUtils::ParseXS::Constants'=> '3.18',
            'ExtUtils::ParseXS::CountLines'=> '3.18',
            'ExtUtils::ParseXS::Utilities'=> '3.18',
            'File::Copy'            => '2.24',
            'File::Find'            => '1.22',
            'IPC::Open3'            => '1.13',
            'Locale::Codes'         => '3.23',
            'Locale::Codes::Constants'=> '3.23',
            'Locale::Codes::Country'=> '3.23',
            'Locale::Codes::Country_Codes'=> '3.23',
            'Locale::Codes::Country_Retired'=> '3.23',
            'Locale::Codes::Currency'=> '3.23',
            'Locale::Codes::Currency_Codes'=> '3.23',
            'Locale::Codes::Currency_Retired'=> '3.23',
            'Locale::Codes::LangExt'=> '3.23',
            'Locale::Codes::LangExt_Codes'=> '3.23',
            'Locale::Codes::LangExt_Retired'=> '3.23',
            'Locale::Codes::LangFam'=> '3.23',
            'Locale::Codes::LangFam_Codes'=> '3.23',
            'Locale::Codes::LangFam_Retired'=> '3.23',
            'Locale::Codes::LangVar'=> '3.23',
            'Locale::Codes::LangVar_Codes'=> '3.23',
            'Locale::Codes::LangVar_Retired'=> '3.23',
            'Locale::Codes::Language'=> '3.23',
            'Locale::Codes::Language_Codes'=> '3.23',
            'Locale::Codes::Language_Retired'=> '3.23',
            'Locale::Codes::Script' => '3.23',
            'Locale::Codes::Script_Codes'=> '3.23',
            'Locale::Codes::Script_Retired'=> '3.23',
            'Locale::Country'       => '3.23',
            'Locale::Currency'      => '3.23',
            'Locale::Language'      => '3.23',
            'Locale::Script'        => '3.23',
            'Math::BigFloat::Trace' => '0.30',
            'Math::BigInt::Trace'   => '0.30',
            'Module::CoreList'      => '2.73',
            'Module::CoreList::TieHashDelta'=> '2.73',
            'Opcode'                => '1.24',
            'Socket'                => '2.006',
            'Storable'              => '2.39',
            'Sys::Syslog'           => '0.32',
            'Unicode::UCD'          => '0.46',
            'XS::APItest'           => '0.43',
            'bignum'                => '0.30',
            'bigrat'                => '0.30',
            'constant'              => '1.24',
            'feature'               => '1.30',
            'threads::shared'       => '1.41',
            'version'               => '0.9901',
            'warnings'              => '1.14',
        },
        removed => {
        }
    },
    5.017005 => {
        delta_from => 5.017004,
        changed => {
            'AutoLoader'            => '5.73',
            'B'                     => '1.39',
            'B::Deparse'            => '1.18',
            'CPANPLUS'              => '0.9133',
            'CPANPLUS::Internals'   => '0.9133',
            'CPANPLUS::Shell::Default'=> '0.9133',
            'Carp'                  => '1.27',
            'Carp::Heavy'           => '1.27',
            'Data::Dumper'          => '2.136',
            'Digest::SHA'           => '5.72',
            'ExtUtils::CBuilder'    => '0.280209',
            'ExtUtils::CBuilder::Base'=> '0.280209',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.280209',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.280209',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.280209',
            'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280209',
            'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280209',
            'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280209',
            'ExtUtils::CBuilder::Platform::aix'=> '0.280209',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.280209',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.280209',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280209',
            'ExtUtils::CBuilder::Platform::os2'=> '0.280209',
            'File::Copy'            => '2.25',
            'File::Glob'            => '1.18',
            'HTTP::Tiny'            => '0.024',
            'Module::CoreList'      => '2.75',
            'Module::CoreList::TieHashDelta'=> '2.75',
            'PerlIO::encoding'      => '0.16',
            'Unicode::Collate'      => '0.90',
            'Unicode::Collate::Locale'=> '0.90',
            'Unicode::Normalize'    => '1.15',
            'Win32CORE'             => '0.04',
            'XS::APItest'           => '0.44',
            'attributes'            => '0.21',
            'bigint'                => '0.31',
            'bignum'                => '0.31',
            'bigrat'                => '0.31',
            'feature'               => '1.31',
            'threads::shared'       => '1.42',
            'warnings'              => '1.15',
        },
        removed => {
        }
    },
    5.017006 => {
        delta_from => 5.017005,
        changed => {
            'B'                     => '1.40',
            'B::Concise'            => '0.94',
            'B::Deparse'            => '1.19',
            'B::Xref'               => '1.05',
            'CGI'                   => '3.63',
            'CGI::Util'             => '3.62',
            'CPAN'                  => '1.99_51',
            'CPANPLUS::Dist::Build' => '0.64',
            'CPANPLUS::Dist::Build::Constants'=> '0.64',
            'Carp'                  => '1.28',
            'Carp::Heavy'           => '1.28',
            'Compress::Raw::Bzip2'  => '2.058',
            'Compress::Raw::Zlib'   => '2.058',
            'Compress::Zlib'        => '2.058',
            'Data::Dumper'          => '2.137',
            'Digest::SHA'           => '5.73',
            'DynaLoader'            => '1.17',
            'Env'                   => '1.04',
            'Errno'                 => '1.17',
            'ExtUtils::Manifest'    => '1.62',
            'ExtUtils::Typemaps'    => '3.18',
            'ExtUtils::Typemaps::Cmd'=> '3.18',
            'ExtUtils::Typemaps::InputMap'=> '3.18',
            'ExtUtils::Typemaps::OutputMap'=> '3.18',
            'ExtUtils::Typemaps::Type'=> '3.18',
            'Fatal'                 => '2.13',
            'File::Find'            => '1.23',
            'Hash::Util'            => '0.13',
            'IO::Compress::Adapter::Bzip2'=> '2.058',
            'IO::Compress::Adapter::Deflate'=> '2.058',
            'IO::Compress::Adapter::Identity'=> '2.058',
            'IO::Compress::Base'    => '2.058',
            'IO::Compress::Base::Common'=> '2.058',
            'IO::Compress::Bzip2'   => '2.058',
            'IO::Compress::Deflate' => '2.058',
            'IO::Compress::Gzip'    => '2.058',
            'IO::Compress::Gzip::Constants'=> '2.058',
            'IO::Compress::RawDeflate'=> '2.058',
            'IO::Compress::Zip'     => '2.058',
            'IO::Compress::Zip::Constants'=> '2.058',
            'IO::Compress::Zlib::Constants'=> '2.058',
            'IO::Compress::Zlib::Extra'=> '2.058',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.058',
            'IO::Uncompress::Adapter::Identity'=> '2.058',
            'IO::Uncompress::Adapter::Inflate'=> '2.058',
            'IO::Uncompress::AnyInflate'=> '2.058',
            'IO::Uncompress::AnyUncompress'=> '2.058',
            'IO::Uncompress::Base'  => '2.058',
            'IO::Uncompress::Bunzip2'=> '2.058',
            'IO::Uncompress::Gunzip'=> '2.058',
            'IO::Uncompress::Inflate'=> '2.058',
            'IO::Uncompress::RawInflate'=> '2.058',
            'IO::Uncompress::Unzip' => '2.058',
            'Module::CoreList'      => '2.78',
            'Module::CoreList::TieHashDelta'=> '2.77',
            'Module::Pluggable'     => '4.5',
            'Module::Pluggable::Object'=> '4.5',
            'Opcode'                => '1.25',
            'Sys::Hostname'         => '1.17',
            'Term::UI'              => '0.32',
            'Thread::Queue'         => '3.01',
            'Tie::Hash::NamedCapture'=> '0.09',
            'Unicode::Collate'      => '0.93',
            'Unicode::Collate::CJK::Korean'=> '0.93',
            'Unicode::Collate::Locale'=> '0.93',
            'Unicode::Normalize'    => '1.16',
            'Unicode::UCD'          => '0.47',
            'XS::APItest'           => '0.46',
            '_charnames'            => '1.33',
            'autodie'               => '2.13',
            'autodie::exception'    => '2.13',
            'autodie::exception::system'=> '2.13',
            'autodie::hints'        => '2.13',
            'charnames'             => '1.33',
            're'                    => '0.23',
        },
        removed => {
        }
    },
    5.017007 => {
        delta_from => 5.017006,
        changed => {
            'B'                     => '1.41',
            'CPANPLUS::Dist::Build' => '0.68',
            'CPANPLUS::Dist::Build::Constants'=> '0.68',
            'Compress::Raw::Bzip2'  => '2.059',
            'Compress::Raw::Zlib'   => '2.059',
            'Compress::Zlib'        => '2.059',
            'Cwd'                   => '3.39_03',
            'Data::Dumper'          => '2.139',
            'Devel::Peek'           => '1.11',
            'Digest::SHA'           => '5.80',
            'DynaLoader'            => '1.18',
            'English'               => '1.06',
            'Errno'                 => '1.18',
            'ExtUtils::Command::MM' => '6.64',
            'ExtUtils::Liblist'     => '6.64',
            'ExtUtils::Liblist::Kid'=> '6.64',
            'ExtUtils::MM'          => '6.64',
            'ExtUtils::MM_AIX'      => '6.64',
            'ExtUtils::MM_Any'      => '6.64',
            'ExtUtils::MM_BeOS'     => '6.64',
            'ExtUtils::MM_Cygwin'   => '6.64',
            'ExtUtils::MM_DOS'      => '6.64',
            'ExtUtils::MM_Darwin'   => '6.64',
            'ExtUtils::MM_MacOS'    => '6.64',
            'ExtUtils::MM_NW5'      => '6.64',
            'ExtUtils::MM_OS2'      => '6.64',
            'ExtUtils::MM_QNX'      => '6.64',
            'ExtUtils::MM_UWIN'     => '6.64',
            'ExtUtils::MM_Unix'     => '6.64',
            'ExtUtils::MM_VMS'      => '6.64',
            'ExtUtils::MM_VOS'      => '6.64',
            'ExtUtils::MM_Win32'    => '6.64',
            'ExtUtils::MM_Win95'    => '6.64',
            'ExtUtils::MY'          => '6.64',
            'ExtUtils::MakeMaker'   => '6.64',
            'ExtUtils::MakeMaker::Config'=> '6.64',
            'ExtUtils::Mkbootstrap' => '6.64',
            'ExtUtils::Mksymlists'  => '6.64',
            'ExtUtils::testlib'     => '6.64',
            'File::DosGlob'         => '1.09',
            'File::Glob'            => '1.19',
            'GDBM_File'             => '1.15',
            'IO::Compress::Adapter::Bzip2'=> '2.059',
            'IO::Compress::Adapter::Deflate'=> '2.059',
            'IO::Compress::Adapter::Identity'=> '2.059',
            'IO::Compress::Base'    => '2.059',
            'IO::Compress::Base::Common'=> '2.059',
            'IO::Compress::Bzip2'   => '2.059',
            'IO::Compress::Deflate' => '2.059',
            'IO::Compress::Gzip'    => '2.059',
            'IO::Compress::Gzip::Constants'=> '2.059',
            'IO::Compress::RawDeflate'=> '2.059',
            'IO::Compress::Zip'     => '2.059',
            'IO::Compress::Zip::Constants'=> '2.059',
            'IO::Compress::Zlib::Constants'=> '2.059',
            'IO::Compress::Zlib::Extra'=> '2.059',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.059',
            'IO::Uncompress::Adapter::Identity'=> '2.059',
            'IO::Uncompress::Adapter::Inflate'=> '2.059',
            'IO::Uncompress::AnyInflate'=> '2.059',
            'IO::Uncompress::AnyUncompress'=> '2.059',
            'IO::Uncompress::Base'  => '2.059',
            'IO::Uncompress::Bunzip2'=> '2.059',
            'IO::Uncompress::Gunzip'=> '2.059',
            'IO::Uncompress::Inflate'=> '2.059',
            'IO::Uncompress::RawInflate'=> '2.059',
            'IO::Uncompress::Unzip' => '2.059',
            'List::Util'            => '1.26',
            'List::Util::XS'        => '1.26',
            'Locale::Codes'         => '3.24',
            'Locale::Codes::Constants'=> '3.24',
            'Locale::Codes::Country'=> '3.24',
            'Locale::Codes::Country_Codes'=> '3.24',
            'Locale::Codes::Country_Retired'=> '3.24',
            'Locale::Codes::Currency'=> '3.24',
            'Locale::Codes::Currency_Codes'=> '3.24',
            'Locale::Codes::Currency_Retired'=> '3.24',
            'Locale::Codes::LangExt'=> '3.24',
            'Locale::Codes::LangExt_Codes'=> '3.24',
            'Locale::Codes::LangExt_Retired'=> '3.24',
            'Locale::Codes::LangFam'=> '3.24',
            'Locale::Codes::LangFam_Codes'=> '3.24',
            'Locale::Codes::LangFam_Retired'=> '3.24',
            'Locale::Codes::LangVar'=> '3.24',
            'Locale::Codes::LangVar_Codes'=> '3.24',
            'Locale::Codes::LangVar_Retired'=> '3.24',
            'Locale::Codes::Language'=> '3.24',
            'Locale::Codes::Language_Codes'=> '3.24',
            'Locale::Codes::Language_Retired'=> '3.24',
            'Locale::Codes::Script' => '3.24',
            'Locale::Codes::Script_Codes'=> '3.24',
            'Locale::Codes::Script_Retired'=> '3.24',
            'Locale::Country'       => '3.24',
            'Locale::Currency'      => '3.24',
            'Locale::Language'      => '3.24',
            'Locale::Maketext'      => '1.23',
            'Locale::Script'        => '3.24',
            'Module::CoreList'      => '2.79',
            'Module::CoreList::TieHashDelta'=> '2.79',
            'POSIX'                 => '1.32',
            'Scalar::Util'          => '1.26',
            'Socket'                => '2.006_001',
            'Storable'              => '2.40',
            'Term::ReadLine'        => '1.11',
            'Unicode::Collate'      => '0.96',
            'Unicode::Collate::CJK::Stroke'=> '0.94',
            'Unicode::Collate::CJK::Zhuyin'=> '0.94',
            'Unicode::Collate::Locale'=> '0.96',
            'XS::APItest'           => '0.48',
            'XS::Typemap'           => '0.09',
            '_charnames'            => '1.34',
            'charnames'             => '1.34',
            'feature'               => '1.32',
            'mro'                   => '1.10',
            'sigtrap'               => '1.07',
            'sort'                  => '2.02',
        },
        removed => {
        }
    },
    5.017008 => {
        delta_from => 5.017007,
        changed => {
            'Archive::Extract'      => '0.62',
            'B'                     => '1.42',
            'B::Concise'            => '0.95',
            'Compress::Raw::Bzip2'  => '2.060',
            'Compress::Raw::Zlib'   => '2.060',
            'Compress::Zlib'        => '2.060',
            'Cwd'                   => '3.40',
            'Data::Dumper'          => '2.141',
            'Digest::SHA'           => '5.81',
            'ExtUtils::Install'     => '1.59',
            'File::Fetch'           => '0.38',
            'File::Path'            => '2.09',
            'File::Spec'            => '3.40',
            'File::Spec::Cygwin'    => '3.40',
            'File::Spec::Epoc'      => '3.40',
            'File::Spec::Functions' => '3.40',
            'File::Spec::Mac'       => '3.40',
            'File::Spec::OS2'       => '3.40',
            'File::Spec::Unix'      => '3.40',
            'File::Spec::VMS'       => '3.40',
            'File::Spec::Win32'     => '3.40',
            'HTTP::Tiny'            => '0.025',
            'Hash::Util'            => '0.14',
            'I18N::LangTags'        => '0.39',
            'I18N::LangTags::List'  => '0.39',
            'I18N::Langinfo'        => '0.09',
            'IO'                    => '1.26',
            'IO::Compress::Adapter::Bzip2'=> '2.060',
            'IO::Compress::Adapter::Deflate'=> '2.060',
            'IO::Compress::Adapter::Identity'=> '2.060',
            'IO::Compress::Base'    => '2.060',
            'IO::Compress::Base::Common'=> '2.060',
            'IO::Compress::Bzip2'   => '2.060',
            'IO::Compress::Deflate' => '2.060',
            'IO::Compress::Gzip'    => '2.060',
            'IO::Compress::Gzip::Constants'=> '2.060',
            'IO::Compress::RawDeflate'=> '2.060',
            'IO::Compress::Zip'     => '2.060',
            'IO::Compress::Zip::Constants'=> '2.060',
            'IO::Compress::Zlib::Constants'=> '2.060',
            'IO::Compress::Zlib::Extra'=> '2.060',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.060',
            'IO::Uncompress::Adapter::Identity'=> '2.060',
            'IO::Uncompress::Adapter::Inflate'=> '2.060',
            'IO::Uncompress::AnyInflate'=> '2.060',
            'IO::Uncompress::AnyUncompress'=> '2.060',
            'IO::Uncompress::Base'  => '2.060',
            'IO::Uncompress::Bunzip2'=> '2.060',
            'IO::Uncompress::Gunzip'=> '2.060',
            'IO::Uncompress::Inflate'=> '2.060',
            'IO::Uncompress::RawInflate'=> '2.060',
            'IO::Uncompress::Unzip' => '2.060',
            'List::Util'            => '1.27',
            'List::Util::XS'        => '1.27',
            'Module::CoreList'      => '2.80',
            'Module::CoreList::TieHashDelta'=> '2.80',
            'Pod::Html'             => '1.17',
            'Pod::LaTeX'            => '0.61',
            'Pod::Man'              => '2.27',
            'Pod::Text'             => '3.17',
            'Pod::Text::Color'      => '2.07',
            'Pod::Text::Overstrike' => '2.05',
            'Pod::Text::Termcap'    => '2.07',
            'Safe'                  => '2.34',
            'Scalar::Util'          => '1.27',
            'Socket'                => '2.009',
            'Term::ANSIColor'       => '4.02',
            'Test'                  => '1.26',
            'Unicode::Collate'      => '0.97',
            'XS::APItest'           => '0.51',
            'XS::Typemap'           => '0.10',
            '_charnames'            => '1.35',
            'charnames'             => '1.35',
            'constant'              => '1.25',
            'diagnostics'           => '1.31',
            'threads::shared'       => '1.43',
            'warnings'              => '1.16',
        },
        removed => {
        }
    },
    5.017009 => {
        delta_from => 5.017008,
        changed => {
            'App::Cpan'             => '1.60_02',
            'App::Prove'            => '3.26',
            'App::Prove::State'     => '3.26',
            'App::Prove::State::Result'=> '3.26',
            'App::Prove::State::Result::Test'=> '3.26',
            'Archive::Extract'      => '0.68',
            'Attribute::Handlers'   => '0.94',
            'B::Lint'               => '1.17',
            'B::Lint::Debug'        => '1.17',
            'Benchmark'             => '1.14',
            'CPAN'                  => '2.00',
            'CPAN::Distribution'    => '2.00',
            'CPAN::FirstTime'       => '5.5304',
            'CPAN::Nox'             => '5.5001',
            'CPANPLUS'              => '0.9135',
            'CPANPLUS::Backend'     => '0.9135',
            'CPANPLUS::Backend::RV' => '0.9135',
            'CPANPLUS::Config'      => '0.9135',
            'CPANPLUS::Config::HomeEnv'=> '0.9135',
            'CPANPLUS::Configure'   => '0.9135',
            'CPANPLUS::Configure::Setup'=> '0.9135',
            'CPANPLUS::Dist'        => '0.9135',
            'CPANPLUS::Dist::Autobundle'=> '0.9135',
            'CPANPLUS::Dist::Base'  => '0.9135',
            'CPANPLUS::Dist::Build' => '0.70',
            'CPANPLUS::Dist::Build::Constants'=> '0.70',
            'CPANPLUS::Dist::MM'    => '0.9135',
            'CPANPLUS::Dist::Sample'=> '0.9135',
            'CPANPLUS::Error'       => '0.9135',
            'CPANPLUS::Internals'   => '0.9135',
            'CPANPLUS::Internals::Constants'=> '0.9135',
            'CPANPLUS::Internals::Constants::Report'=> '0.9135',
            'CPANPLUS::Internals::Extract'=> '0.9135',
            'CPANPLUS::Internals::Fetch'=> '0.9135',
            'CPANPLUS::Internals::Report'=> '0.9135',
            'CPANPLUS::Internals::Search'=> '0.9135',
            'CPANPLUS::Internals::Source'=> '0.9135',
            'CPANPLUS::Internals::Source::Memory'=> '0.9135',
            'CPANPLUS::Internals::Source::SQLite'=> '0.9135',
            'CPANPLUS::Internals::Source::SQLite::Tie'=> '0.9135',
            'CPANPLUS::Internals::Utils'=> '0.9135',
            'CPANPLUS::Internals::Utils::Autoflush'=> '0.9135',
            'CPANPLUS::Module'      => '0.9135',
            'CPANPLUS::Module::Author'=> '0.9135',
            'CPANPLUS::Module::Author::Fake'=> '0.9135',
            'CPANPLUS::Module::Checksums'=> '0.9135',
            'CPANPLUS::Module::Fake'=> '0.9135',
            'CPANPLUS::Module::Signature'=> '0.9135',
            'CPANPLUS::Selfupdate'  => '0.9135',
            'CPANPLUS::Shell'       => '0.9135',
            'CPANPLUS::Shell::Classic'=> '0.9135',
            'CPANPLUS::Shell::Default'=> '0.9135',
            'CPANPLUS::Shell::Default::Plugins::CustomSource'=> '0.9135',
            'CPANPLUS::Shell::Default::Plugins::Remote'=> '0.9135',
            'CPANPLUS::Shell::Default::Plugins::Source'=> '0.9135',
            'Config'                => '5.017009',
            'Config::Perl::V'       => '0.17',
            'DBM_Filter'            => '0.05',
            'Data::Dumper'          => '2.142',
            'Digest::SHA'           => '5.82',
            'Encode'                => '2.48',
            'ExtUtils::Installed'   => '1.999003',
            'ExtUtils::Manifest'    => '1.63',
            'ExtUtils::ParseXS::Utilities'=> '3.19',
            'ExtUtils::Typemaps'    => '3.19',
            'File::CheckTree'       => '4.42',
            'File::DosGlob'         => '1.10',
            'File::Temp'            => '0.22_90',
            'Filter::Simple'        => '0.89',
            'IO'                    => '1.27',
            'Log::Message'          => '0.06',
            'Log::Message::Config'  => '0.06',
            'Log::Message::Handlers'=> '0.06',
            'Log::Message::Item'    => '0.06',
            'Log::Message::Simple'  => '0.10',
            'Math::BigInt'          => '1.999',
            'Module::CoreList'      => '2.82',
            'Module::CoreList::TieHashDelta'=> '2.82',
            'Module::Load'          => '0.24',
            'Module::Pluggable'     => '4.6',
            'Module::Pluggable::Object'=> '4.6',
            'OS2::DLL'              => '1.05',
            'OS2::ExtAttr'          => '0.03',
            'OS2::Process'          => '1.08',
            'Object::Accessor'      => '0.46',
            'PerlIO::scalar'        => '0.16',
            'Pod::Checker'          => '1.60',
            'Pod::Find'             => '1.60',
            'Pod::Html'             => '1.18',
            'Pod::InputObjects'     => '1.60',
            'Pod::ParseUtils'       => '1.60',
            'Pod::Parser'           => '1.60',
            'Pod::Perldoc'          => '3.19',
            'Pod::Perldoc::BaseTo'  => '3.19',
            'Pod::Perldoc::GetOptsOO'=> '3.19',
            'Pod::Perldoc::ToANSI'  => '3.19',
            'Pod::Perldoc::ToChecker'=> '3.19',
            'Pod::Perldoc::ToMan'   => '3.19',
            'Pod::Perldoc::ToNroff' => '3.19',
            'Pod::Perldoc::ToPod'   => '3.19',
            'Pod::Perldoc::ToRtf'   => '3.19',
            'Pod::Perldoc::ToTerm'  => '3.19',
            'Pod::Perldoc::ToText'  => '3.19',
            'Pod::Perldoc::ToTk'    => '3.19',
            'Pod::Perldoc::ToXml'   => '3.19',
            'Pod::PlainText'        => '2.06',
            'Pod::Select'           => '1.60',
            'Pod::Usage'            => '1.61',
            'SelfLoader'            => '1.21',
            'TAP::Base'             => '3.26',
            'TAP::Formatter::Base'  => '3.26',
            'TAP::Formatter::Color' => '3.26',
            'TAP::Formatter::Console'=> '3.26',
            'TAP::Formatter::Console::ParallelSession'=> '3.26',
            'TAP::Formatter::Console::Session'=> '3.26',
            'TAP::Formatter::File'  => '3.26',
            'TAP::Formatter::File::Session'=> '3.26',
            'TAP::Formatter::Session'=> '3.26',
            'TAP::Harness'          => '3.26',
            'TAP::Object'           => '3.26',
            'TAP::Parser'           => '3.26',
            'TAP::Parser::Aggregator'=> '3.26',
            'TAP::Parser::Grammar'  => '3.26',
            'TAP::Parser::Iterator' => '3.26',
            'TAP::Parser::Iterator::Array'=> '3.26',
            'TAP::Parser::Iterator::Process'=> '3.26',
            'TAP::Parser::Iterator::Stream'=> '3.26',
            'TAP::Parser::IteratorFactory'=> '3.26',
            'TAP::Parser::Multiplexer'=> '3.26',
            'TAP::Parser::Result'   => '3.26',
            'TAP::Parser::Result::Bailout'=> '3.26',
            'TAP::Parser::Result::Comment'=> '3.26',
            'TAP::Parser::Result::Plan'=> '3.26',
            'TAP::Parser::Result::Pragma'=> '3.26',
            'TAP::Parser::Result::Test'=> '3.26',
            'TAP::Parser::Result::Unknown'=> '3.26',
            'TAP::Parser::Result::Version'=> '3.26',
            'TAP::Parser::Result::YAML'=> '3.26',
            'TAP::Parser::ResultFactory'=> '3.26',
            'TAP::Parser::Scheduler'=> '3.26',
            'TAP::Parser::Scheduler::Job'=> '3.26',
            'TAP::Parser::Scheduler::Spinner'=> '3.26',
            'TAP::Parser::Source'   => '3.26',
            'TAP::Parser::SourceHandler'=> '3.26',
            'TAP::Parser::SourceHandler::Executable'=> '3.26',
            'TAP::Parser::SourceHandler::File'=> '3.26',
            'TAP::Parser::SourceHandler::Handle'=> '3.26',
            'TAP::Parser::SourceHandler::Perl'=> '3.26',
            'TAP::Parser::SourceHandler::RawTAP'=> '3.26',
            'TAP::Parser::Utils'    => '3.26',
            'TAP::Parser::YAMLish::Reader'=> '3.26',
            'TAP::Parser::YAMLish::Writer'=> '3.26',
            'Term::UI'              => '0.34',
            'Test::Harness'         => '3.26',
            'Text::Soundex'         => '3.04',
            'Thread::Queue'         => '3.02',
            'Unicode::UCD'          => '0.50',
            'Win32'                 => '0.46',
            'Win32API::File'        => '0.1201',
            '_charnames'            => '1.36',
            'arybase'               => '0.06',
            'bigint'                => '0.32',
            'bignum'                => '0.32',
            'charnames'             => '1.36',
            'filetest'              => '1.03',
            'locale'                => '1.02',
            'overload'              => '1.21',
            'warnings'              => '1.17',
        },
        removed => {
        }
    },
    5.017010 => {
        delta_from => 5.017009,
        changed => {
            'Benchmark'             => '1.15',
            'Config'                => '5.017009',
            'Data::Dumper'          => '2.145',
            'Digest::SHA'           => '5.84',
            'Encode'                => '2.49',
            'ExtUtils::Command::MM' => '6.65_01',
            'ExtUtils::Liblist'     => '6.65_01',
            'ExtUtils::Liblist::Kid'=> '6.65_01',
            'ExtUtils::MM'          => '6.65_01',
            'ExtUtils::MM_AIX'      => '6.65_01',
            'ExtUtils::MM_Any'      => '6.65_01',
            'ExtUtils::MM_BeOS'     => '6.65_01',
            'ExtUtils::MM_Cygwin'   => '6.65_01',
            'ExtUtils::MM_DOS'      => '6.65_01',
            'ExtUtils::MM_Darwin'   => '6.65_01',
            'ExtUtils::MM_MacOS'    => '6.65_01',
            'ExtUtils::MM_NW5'      => '6.65_01',
            'ExtUtils::MM_OS2'      => '6.65_01',
            'ExtUtils::MM_QNX'      => '6.65_01',
            'ExtUtils::MM_UWIN'     => '6.65_01',
            'ExtUtils::MM_Unix'     => '6.65_01',
            'ExtUtils::MM_VMS'      => '6.65_01',
            'ExtUtils::MM_VOS'      => '6.65_01',
            'ExtUtils::MM_Win32'    => '6.65_01',
            'ExtUtils::MM_Win95'    => '6.65_01',
            'ExtUtils::MY'          => '6.65_01',
            'ExtUtils::MakeMaker'   => '6.65_01',
            'ExtUtils::MakeMaker::Config'=> '6.65_01',
            'ExtUtils::Mkbootstrap' => '6.65_01',
            'ExtUtils::Mksymlists'  => '6.65_01',
            'ExtUtils::testlib'     => '6.65_01',
            'File::Copy'            => '2.26',
            'File::Temp'            => '0.23',
            'Getopt::Long'          => '2.39',
            'Hash::Util'            => '0.15',
            'I18N::Langinfo'        => '0.10',
            'IPC::Cmd'              => '0.80',
            'JSON::PP'              => '2.27202',
            'Locale::Codes'         => '3.25',
            'Locale::Codes::Constants'=> '3.25',
            'Locale::Codes::Country'=> '3.25',
            'Locale::Codes::Country_Codes'=> '3.25',
            'Locale::Codes::Country_Retired'=> '3.25',
            'Locale::Codes::Currency'=> '3.25',
            'Locale::Codes::Currency_Codes'=> '3.25',
            'Locale::Codes::Currency_Retired'=> '3.25',
            'Locale::Codes::LangExt'=> '3.25',
            'Locale::Codes::LangExt_Codes'=> '3.25',
            'Locale::Codes::LangExt_Retired'=> '3.25',
            'Locale::Codes::LangFam'=> '3.25',
            'Locale::Codes::LangFam_Codes'=> '3.25',
            'Locale::Codes::LangFam_Retired'=> '3.25',
            'Locale::Codes::LangVar'=> '3.25',
            'Locale::Codes::LangVar_Codes'=> '3.25',
            'Locale::Codes::LangVar_Retired'=> '3.25',
            'Locale::Codes::Language'=> '3.25',
            'Locale::Codes::Language_Codes'=> '3.25',
            'Locale::Codes::Language_Retired'=> '3.25',
            'Locale::Codes::Script' => '3.25',
            'Locale::Codes::Script_Codes'=> '3.25',
            'Locale::Codes::Script_Retired'=> '3.25',
            'Locale::Country'       => '3.25',
            'Locale::Currency'      => '3.25',
            'Locale::Language'      => '3.25',
            'Locale::Script'        => '3.25',
            'Math::BigFloat'        => '1.998',
            'Math::BigFloat::Trace' => '0.32',
            'Math::BigInt'          => '1.9991',
            'Math::BigInt::CalcEmu' => '1.998',
            'Math::BigInt::Trace'   => '0.32',
            'Math::BigRat'          => '0.2604',
            'Module::CoreList'      => '2.84',
            'Module::CoreList::TieHashDelta'=> '2.84',
            'Module::Pluggable'     => '4.7',
            'Net::Ping'             => '2.41',
            'Perl::OSType'          => '1.003',
            'Pod::Simple'           => '3.26',
            'Pod::Simple::BlackBox' => '3.26',
            'Pod::Simple::Checker'  => '3.26',
            'Pod::Simple::Debug'    => '3.26',
            'Pod::Simple::DumpAsText'=> '3.26',
            'Pod::Simple::DumpAsXML'=> '3.26',
            'Pod::Simple::HTML'     => '3.26',
            'Pod::Simple::HTMLBatch'=> '3.26',
            'Pod::Simple::LinkSection'=> '3.26',
            'Pod::Simple::Methody'  => '3.26',
            'Pod::Simple::Progress' => '3.26',
            'Pod::Simple::PullParser'=> '3.26',
            'Pod::Simple::PullParserEndToken'=> '3.26',
            'Pod::Simple::PullParserStartToken'=> '3.26',
            'Pod::Simple::PullParserTextToken'=> '3.26',
            'Pod::Simple::PullParserToken'=> '3.26',
            'Pod::Simple::RTF'      => '3.26',
            'Pod::Simple::Search'   => '3.26',
            'Pod::Simple::SimpleTree'=> '3.26',
            'Pod::Simple::Text'     => '3.26',
            'Pod::Simple::TextContent'=> '3.26',
            'Pod::Simple::TiedOutFH'=> '3.26',
            'Pod::Simple::Transcode'=> '3.26',
            'Pod::Simple::TranscodeDumb'=> '3.26',
            'Pod::Simple::TranscodeSmart'=> '3.26',
            'Pod::Simple::XHTML'    => '3.26',
            'Pod::Simple::XMLOutStream'=> '3.26',
            'Safe'                  => '2.35',
            'Term::ReadLine'        => '1.12',
            'Text::ParseWords'      => '3.28',
            'Tie::File'             => '0.99',
            'Unicode::UCD'          => '0.51',
            'Win32'                 => '0.47',
            'bigint'                => '0.33',
            'bignum'                => '0.33',
            'bigrat'                => '0.33',
            'constant'              => '1.27',
            'perlfaq'               => '5.0150042',
            'version'               => '0.9902',
        },
        removed => {
        }
    },
    5.017011 => {
        delta_from => 5.017010,
        changed => {
            'App::Cpan'             => '1.61',
            'B::Deparse'            => '1.20',
            'Config'                => '5.017009',
            'Exporter'              => '5.68',
            'Exporter::Heavy'       => '5.68',
            'ExtUtils::CBuilder'    => '0.280210',
            'ExtUtils::Command::MM' => '6.66',
            'ExtUtils::Liblist'     => '6.66',
            'ExtUtils::Liblist::Kid'=> '6.66',
            'ExtUtils::MM'          => '6.66',
            'ExtUtils::MM_AIX'      => '6.66',
            'ExtUtils::MM_Any'      => '6.66',
            'ExtUtils::MM_BeOS'     => '6.66',
            'ExtUtils::MM_Cygwin'   => '6.66',
            'ExtUtils::MM_DOS'      => '6.66',
            'ExtUtils::MM_Darwin'   => '6.66',
            'ExtUtils::MM_MacOS'    => '6.66',
            'ExtUtils::MM_NW5'      => '6.66',
            'ExtUtils::MM_OS2'      => '6.66',
            'ExtUtils::MM_QNX'      => '6.66',
            'ExtUtils::MM_UWIN'     => '6.66',
            'ExtUtils::MM_Unix'     => '6.66',
            'ExtUtils::MM_VMS'      => '6.66',
            'ExtUtils::MM_VOS'      => '6.66',
            'ExtUtils::MM_Win32'    => '6.66',
            'ExtUtils::MM_Win95'    => '6.66',
            'ExtUtils::MY'          => '6.66',
            'ExtUtils::MakeMaker'   => '6.66',
            'ExtUtils::MakeMaker::Config'=> '6.66',
            'ExtUtils::Mkbootstrap' => '6.66',
            'ExtUtils::Mksymlists'  => '6.66',
            'ExtUtils::testlib'     => '6.66',
            'File::Glob'            => '1.20',
            'IO'                    => '1.28',
            'Module::CoreList'      => '2.87',
            'Module::CoreList::TieHashDelta'=> '2.87',
            'Storable'              => '2.41',
            'bigint'                => '0.34',
            'mro'                   => '1.11',
            'overload'              => '1.22',
            'warnings'              => '1.18',
        },
        removed => {
        }
    },
    5.018000 => {
        delta_from => 5.017011,
        changed => {
            'Carp'                  => '1.29',
            'Carp::Heavy'           => '1.29',
            'Config'                => '5.018000',
            'Hash::Util'            => '0.16',
            'IO::Handle'            => '1.34',
            'IO::Socket'            => '1.36',
            'Module::CoreList'      => '2.89',
            'Module::CoreList::TieHashDelta'=> '2.89',
            'Pod::Simple'           => '3.28',
            'Pod::Simple::BlackBox' => '3.28',
            'Pod::Simple::Checker'  => '3.28',
            'Pod::Simple::Debug'    => '3.28',
            'Pod::Simple::DumpAsText'=> '3.28',
            'Pod::Simple::DumpAsXML'=> '3.28',
            'Pod::Simple::HTML'     => '3.28',
            'Pod::Simple::HTMLBatch'=> '3.28',
            'Pod::Simple::LinkSection'=> '3.28',
            'Pod::Simple::Methody'  => '3.28',
            'Pod::Simple::Progress' => '3.28',
            'Pod::Simple::PullParser'=> '3.28',
            'Pod::Simple::PullParserEndToken'=> '3.28',
            'Pod::Simple::PullParserStartToken'=> '3.28',
            'Pod::Simple::PullParserTextToken'=> '3.28',
            'Pod::Simple::PullParserToken'=> '3.28',
            'Pod::Simple::RTF'      => '3.28',
            'Pod::Simple::Search'   => '3.28',
            'Pod::Simple::SimpleTree'=> '3.28',
            'Pod::Simple::Text'     => '3.28',
            'Pod::Simple::TextContent'=> '3.28',
            'Pod::Simple::TiedOutFH'=> '3.28',
            'Pod::Simple::Transcode'=> '3.28',
            'Pod::Simple::TranscodeDumb'=> '3.28',
            'Pod::Simple::TranscodeSmart'=> '3.28',
            'Pod::Simple::XHTML'    => '3.28',
            'Pod::Simple::XMLOutStream'=> '3.28',
        },
        removed => {
        }
    },
    5.018001 => {
        delta_from => 5.018000,
        changed => {
            'B'                     => '1.42_01',
            'Config'                => '5.018001',
            'Digest::SHA'           => '5.84_01',
            'Module::CoreList'      => '2.96',
            'Module::CoreList::TieHashDelta'=> '2.96',
            'Module::CoreList::Utils'=> '2.96',
        },
        removed => {
           'VMS::Filespec'         => 1,
        }
    },
    5.018002 => {
        delta_from => 5.018001,
        changed => {
            'B'                     => '1.42_02',
            'B::Concise'            => '0.95_01',
            'Config'                => '5.018002',
            'File::Glob'            => '1.20_01',
            'Module::CoreList'      => '3.03',
            'Module::CoreList::TieHashDelta'=> '3.03',
            'Module::CoreList::Utils'=> '3.03',
        },
    },
    5.018003 => {
        delta_from => 5.018002,
        changed => {
            'Config'                => '5.018003',
            'Digest::SHA'           => '5.84_02',
            'Module::CoreList'      => '3.12',
            'Module::CoreList::TieHashDelta'=> '3.12',
            'Module::CoreList::Utils'=> '3.12',
        },
    },
    5.018004 => {
        delta_from => 5.018003,
        changed => {
            'Config'                => '5.018004',
            'Module::CoreList'      => '3.13',
            'Module::CoreList::TieHashDelta'=> '3.13',
            'Module::CoreList::Utils'=> '3.13',
        },
    },
    5.019000 => {
        delta_from => 5.018000,
        changed => {
            'Config'                => '5.019000',
            'Getopt::Std'           => '1.08',
            'Module::CoreList'      => '2.91',
            'Module::CoreList::TieHashDelta'=> '2.91',
            'Storable'              => '2.42',
            'feature'               => '1.33',
            'utf8'                  => '1.11',
        },
        removed => {
           'Archive::Extract'      => 1,
           'B::Lint'               => 1,
           'B::Lint::Debug'        => 1,
           'CPANPLUS'              => 1,
           'CPANPLUS::Backend'     => 1,
           'CPANPLUS::Backend::RV' => 1,
           'CPANPLUS::Config'      => 1,
           'CPANPLUS::Config::HomeEnv'=> 1,
           'CPANPLUS::Configure'   => 1,
           'CPANPLUS::Configure::Setup'=> 1,
           'CPANPLUS::Dist'        => 1,
           'CPANPLUS::Dist::Autobundle'=> 1,
           'CPANPLUS::Dist::Base'  => 1,
           'CPANPLUS::Dist::Build' => 1,
           'CPANPLUS::Dist::Build::Constants'=> 1,
           'CPANPLUS::Dist::MM'    => 1,
           'CPANPLUS::Dist::Sample'=> 1,
           'CPANPLUS::Error'       => 1,
           'CPANPLUS::Internals'   => 1,
           'CPANPLUS::Internals::Constants'=> 1,
           'CPANPLUS::Internals::Constants::Report'=> 1,
           'CPANPLUS::Internals::Extract'=> 1,
           'CPANPLUS::Internals::Fetch'=> 1,
           'CPANPLUS::Internals::Report'=> 1,
           'CPANPLUS::Internals::Search'=> 1,
           'CPANPLUS::Internals::Source'=> 1,
           'CPANPLUS::Internals::Source::Memory'=> 1,
           'CPANPLUS::Internals::Source::SQLite'=> 1,
           'CPANPLUS::Internals::Source::SQLite::Tie'=> 1,
           'CPANPLUS::Internals::Utils'=> 1,
           'CPANPLUS::Internals::Utils::Autoflush'=> 1,
           'CPANPLUS::Module'      => 1,
           'CPANPLUS::Module::Author'=> 1,
           'CPANPLUS::Module::Author::Fake'=> 1,
           'CPANPLUS::Module::Checksums'=> 1,
           'CPANPLUS::Module::Fake'=> 1,
           'CPANPLUS::Module::Signature'=> 1,
           'CPANPLUS::Selfupdate'  => 1,
           'CPANPLUS::Shell'       => 1,
           'CPANPLUS::Shell::Classic'=> 1,
           'CPANPLUS::Shell::Default'=> 1,
           'CPANPLUS::Shell::Default::Plugins::CustomSource'=> 1,
           'CPANPLUS::Shell::Default::Plugins::Remote'=> 1,
           'CPANPLUS::Shell::Default::Plugins::Source'=> 1,
           'Devel::InnerPackage'   => 1,
           'File::CheckTree'       => 1,
           'Log::Message'          => 1,
           'Log::Message::Config'  => 1,
           'Log::Message::Handlers'=> 1,
           'Log::Message::Item'    => 1,
           'Log::Message::Simple'  => 1,
           'Module::Pluggable'     => 1,
           'Module::Pluggable::Object'=> 1,
           'Object::Accessor'      => 1,
           'Pod::LaTeX'            => 1,
           'Term::UI'              => 1,
           'Term::UI::History'     => 1,
           'Text::Soundex'         => 1,
        }
    },
    5.019001 => {
        delta_from => 5.019000,
        changed => {
            'App::Prove'            => '3.28',
            'App::Prove::State'     => '3.28',
            'App::Prove::State::Result'=> '3.28',
            'App::Prove::State::Result::Test'=> '3.28',
            'Archive::Tar'          => '1.92',
            'Archive::Tar::Constant'=> '1.92',
            'Archive::Tar::File'    => '1.92',
            'Attribute::Handlers'   => '0.95',
            'B'                     => '1.43',
            'B::Concise'            => '0.96',
            'B::Deparse'            => '1.21',
            'B::Showlex'            => '1.04',
            'Benchmark'             => '1.16',
            'CPAN::Meta'            => '2.131560',
            'CPAN::Meta::Converter' => '2.131560',
            'CPAN::Meta::Feature'   => '2.131560',
            'CPAN::Meta::History'   => '2.131560',
            'CPAN::Meta::Prereqs'   => '2.131560',
            'CPAN::Meta::Spec'      => '2.131560',
            'CPAN::Meta::Validator' => '2.131560',
            'Carp'                  => '1.30',
            'Carp::Heavy'           => '1.30',
            'Compress::Raw::Bzip2'  => '2.061',
            'Compress::Raw::Zlib'   => '2.061',
            'Compress::Zlib'        => '2.061',
            'Config'                => '5.019001',
            'Config::Perl::V'       => '0.18',
            'Cwd'                   => '3.41',
            'DB'                    => '1.06',
            'DB_File'               => '1.828',
            'Data::Dumper'          => '2.146',
            'Encode'                => '2.51',
            'Encode::CN::HZ'        => '2.06',
            'Encode::GSM0338'       => '2.03',
            'Encode::Unicode::UTF7' => '2.07',
            'ExtUtils::CBuilder::Base'=> '0.280210',
            'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280210',
            'ExtUtils::Command::MM' => '6.68',
            'ExtUtils::Install'     => '1.60',
            'ExtUtils::Liblist'     => '6.68',
            'ExtUtils::Liblist::Kid'=> '6.68',
            'ExtUtils::MM'          => '6.68',
            'ExtUtils::MM_AIX'      => '6.68',
            'ExtUtils::MM_Any'      => '6.68',
            'ExtUtils::MM_BeOS'     => '6.68',
            'ExtUtils::MM_Cygwin'   => '6.68',
            'ExtUtils::MM_DOS'      => '6.68',
            'ExtUtils::MM_Darwin'   => '6.68',
            'ExtUtils::MM_MacOS'    => '6.68',
            'ExtUtils::MM_NW5'      => '6.68',
            'ExtUtils::MM_OS2'      => '6.68',
            'ExtUtils::MM_QNX'      => '6.68',
            'ExtUtils::MM_UWIN'     => '6.68',
            'ExtUtils::MM_Unix'     => '6.68',
            'ExtUtils::MM_VMS'      => '6.68',
            'ExtUtils::MM_VOS'      => '6.68',
            'ExtUtils::MM_Win32'    => '6.68',
            'ExtUtils::MM_Win95'    => '6.68',
            'ExtUtils::MY'          => '6.68',
            'ExtUtils::MakeMaker'   => '6.68',
            'ExtUtils::MakeMaker::Config'=> '6.68',
            'ExtUtils::Mkbootstrap' => '6.68',
            'ExtUtils::Mksymlists'  => '6.68',
            'ExtUtils::ParseXS'     => '3.19',
            'ExtUtils::testlib'     => '6.68',
            'Fatal'                 => '2.19',
            'File::Copy'            => '2.27',
            'File::DosGlob'         => '1.11',
            'File::Fetch'           => '0.42',
            'File::Find'            => '1.24',
            'File::Spec'            => '3.41',
            'File::Spec::Cygwin'    => '3.41',
            'File::Spec::Epoc'      => '3.41',
            'File::Spec::Mac'       => '3.41',
            'File::Spec::OS2'       => '3.41',
            'File::Spec::Unix'      => '3.41',
            'File::Spec::VMS'       => '3.41',
            'File::Spec::Win32'     => '3.41',
            'File::Temp'            => '0.2301',
            'Filter::Simple'        => '0.90',
            'Filter::Util::Call'    => '1.49',
            'Getopt::Long'          => '2.4',
            'HTTP::Tiny'            => '0.031',
            'Hash::Util::FieldHash' => '1.11',
            'IO::Compress::Adapter::Bzip2'=> '2.061',
            'IO::Compress::Adapter::Deflate'=> '2.061',
            'IO::Compress::Adapter::Identity'=> '2.061',
            'IO::Compress::Base'    => '2.061',
            'IO::Compress::Base::Common'=> '2.061',
            'IO::Compress::Bzip2'   => '2.061',
            'IO::Compress::Deflate' => '2.061',
            'IO::Compress::Gzip'    => '2.061',
            'IO::Compress::Gzip::Constants'=> '2.061',
            'IO::Compress::RawDeflate'=> '2.061',
            'IO::Compress::Zip'     => '2.061',
            'IO::Compress::Zip::Constants'=> '2.061',
            'IO::Compress::Zlib::Constants'=> '2.061',
            'IO::Compress::Zlib::Extra'=> '2.061',
            'IO::Handle'            => '1.35',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.061',
            'IO::Uncompress::Adapter::Identity'=> '2.061',
            'IO::Uncompress::Adapter::Inflate'=> '2.061',
            'IO::Uncompress::AnyInflate'=> '2.061',
            'IO::Uncompress::AnyUncompress'=> '2.061',
            'IO::Uncompress::Base'  => '2.061',
            'IO::Uncompress::Bunzip2'=> '2.061',
            'IO::Uncompress::Gunzip'=> '2.061',
            'IO::Uncompress::Inflate'=> '2.061',
            'IO::Uncompress::RawInflate'=> '2.061',
            'IO::Uncompress::Unzip' => '2.061',
            'IPC::Open3'            => '1.14',
            'Locale::Codes'         => '3.26',
            'Locale::Codes::Constants'=> '3.26',
            'Locale::Codes::Country'=> '3.26',
            'Locale::Codes::Country_Codes'=> '3.26',
            'Locale::Codes::Country_Retired'=> '3.26',
            'Locale::Codes::Currency'=> '3.26',
            'Locale::Codes::Currency_Codes'=> '3.26',
            'Locale::Codes::Currency_Retired'=> '3.26',
            'Locale::Codes::LangExt'=> '3.26',
            'Locale::Codes::LangExt_Codes'=> '3.26',
            'Locale::Codes::LangExt_Retired'=> '3.26',
            'Locale::Codes::LangFam'=> '3.26',
            'Locale::Codes::LangFam_Codes'=> '3.26',
            'Locale::Codes::LangFam_Retired'=> '3.26',
            'Locale::Codes::LangVar'=> '3.26',
            'Locale::Codes::LangVar_Codes'=> '3.26',
            'Locale::Codes::LangVar_Retired'=> '3.26',
            'Locale::Codes::Language'=> '3.26',
            'Locale::Codes::Language_Codes'=> '3.26',
            'Locale::Codes::Language_Retired'=> '3.26',
            'Locale::Codes::Script' => '3.26',
            'Locale::Codes::Script_Codes'=> '3.26',
            'Locale::Codes::Script_Retired'=> '3.26',
            'Locale::Country'       => '3.26',
            'Locale::Currency'      => '3.26',
            'Locale::Language'      => '3.26',
            'Locale::Maketext'      => '1.24',
            'Locale::Script'        => '3.26',
            'Math::BigFloat'        => '1.999',
            'Math::BigInt'          => '1.9992',
            'Math::BigInt::Calc'    => '1.998',
            'Math::BigInt::CalcEmu' => '1.9991',
            'Math::BigRat'          => '0.2606',
            'Module::Build'         => '0.4005',
            'Module::Build::Base'   => '0.4005',
            'Module::Build::Compat' => '0.4005',
            'Module::Build::Config' => '0.4005',
            'Module::Build::Cookbook'=> '0.4005',
            'Module::Build::Dumper' => '0.4005',
            'Module::Build::ModuleInfo'=> '0.4005',
            'Module::Build::Notes'  => '0.4005',
            'Module::Build::PPMMaker'=> '0.4005',
            'Module::Build::Platform::Amiga'=> '0.4005',
            'Module::Build::Platform::Default'=> '0.4005',
            'Module::Build::Platform::EBCDIC'=> '0.4005',
            'Module::Build::Platform::MPEiX'=> '0.4005',
            'Module::Build::Platform::MacOS'=> '0.4005',
            'Module::Build::Platform::RiscOS'=> '0.4005',
            'Module::Build::Platform::Unix'=> '0.4005',
            'Module::Build::Platform::VMS'=> '0.4005',
            'Module::Build::Platform::VOS'=> '0.4005',
            'Module::Build::Platform::Windows'=> '0.4005',
            'Module::Build::Platform::aix'=> '0.4005',
            'Module::Build::Platform::cygwin'=> '0.4005',
            'Module::Build::Platform::darwin'=> '0.4005',
            'Module::Build::Platform::os2'=> '0.4005',
            'Module::Build::PodParser'=> '0.4005',
            'Module::CoreList'      => '2.92',
            'Module::CoreList::TieHashDelta'=> '2.92',
            'Module::CoreList::Utils'=> '2.92',
            'Module::Metadata'      => '1.000014',
            'Net::Ping'             => '2.42',
            'OS2::Process'          => '1.09',
            'POSIX'                 => '1.33',
            'Pod::Find'             => '1.61',
            'Pod::Html'             => '1.19',
            'Pod::InputObjects'     => '1.61',
            'Pod::ParseUtils'       => '1.61',
            'Pod::Parser'           => '1.61',
            'Pod::Perldoc'          => '3.20',
            'Pod::Perldoc::BaseTo'  => '3.20',
            'Pod::Perldoc::GetOptsOO'=> '3.20',
            'Pod::Perldoc::ToANSI'  => '3.20',
            'Pod::Perldoc::ToChecker'=> '3.20',
            'Pod::Perldoc::ToMan'   => '3.20',
            'Pod::Perldoc::ToNroff' => '3.20',
            'Pod::Perldoc::ToPod'   => '3.20',
            'Pod::Perldoc::ToRtf'   => '3.20',
            'Pod::Perldoc::ToTerm'  => '3.20',
            'Pod::Perldoc::ToText'  => '3.20',
            'Pod::Perldoc::ToTk'    => '3.20',
            'Pod::Perldoc::ToXml'   => '3.20',
            'Pod::Select'           => '1.61',
            'Pod::Usage'            => '1.63',
            'Safe'                  => '2.36',
            'Storable'              => '2.43',
            'Sys::Hostname'         => '1.18',
            'Sys::Syslog'           => '0.33',
            'TAP::Base'             => '3.28',
            'TAP::Formatter::Base'  => '3.28',
            'TAP::Formatter::Color' => '3.28',
            'TAP::Formatter::Console'=> '3.28',
            'TAP::Formatter::Console::ParallelSession'=> '3.28',
            'TAP::Formatter::Console::Session'=> '3.28',
            'TAP::Formatter::File'  => '3.28',
            'TAP::Formatter::File::Session'=> '3.28',
            'TAP::Formatter::Session'=> '3.28',
            'TAP::Harness'          => '3.28',
            'TAP::Object'           => '3.28',
            'TAP::Parser'           => '3.28',
            'TAP::Parser::Aggregator'=> '3.28',
            'TAP::Parser::Grammar'  => '3.28',
            'TAP::Parser::Iterator' => '3.28',
            'TAP::Parser::Iterator::Array'=> '3.28',
            'TAP::Parser::Iterator::Process'=> '3.28',
            'TAP::Parser::Iterator::Stream'=> '3.28',
            'TAP::Parser::IteratorFactory'=> '3.28',
            'TAP::Parser::Multiplexer'=> '3.28',
            'TAP::Parser::Result'   => '3.28',
            'TAP::Parser::Result::Bailout'=> '3.28',
            'TAP::Parser::Result::Comment'=> '3.28',
            'TAP::Parser::Result::Plan'=> '3.28',
            'TAP::Parser::Result::Pragma'=> '3.28',
            'TAP::Parser::Result::Test'=> '3.28',
            'TAP::Parser::Result::Unknown'=> '3.28',
            'TAP::Parser::Result::Version'=> '3.28',
            'TAP::Parser::Result::YAML'=> '3.28',
            'TAP::Parser::ResultFactory'=> '3.28',
            'TAP::Parser::Scheduler'=> '3.28',
            'TAP::Parser::Scheduler::Job'=> '3.28',
            'TAP::Parser::Scheduler::Spinner'=> '3.28',
            'TAP::Parser::Source'   => '3.28',
            'TAP::Parser::SourceHandler'=> '3.28',
            'TAP::Parser::SourceHandler::Executable'=> '3.28',
            'TAP::Parser::SourceHandler::File'=> '3.28',
            'TAP::Parser::SourceHandler::Handle'=> '3.28',
            'TAP::Parser::SourceHandler::Perl'=> '3.28',
            'TAP::Parser::SourceHandler::RawTAP'=> '3.28',
            'TAP::Parser::Utils'    => '3.28',
            'TAP::Parser::YAMLish::Reader'=> '3.28',
            'TAP::Parser::YAMLish::Writer'=> '3.28',
            'Term::ReadLine'        => '1.13',
            'Test::Harness'         => '3.28',
            'Text::Tabs'            => '2013.0523',
            'Text::Wrap'            => '2013.0523',
            'Thread'                => '3.04',
            'Tie::File'             => '1.00',
            'Time::Piece'           => '1.2002',
            'Unicode::Collate'      => '0.98',
            'Unicode::UCD'          => '0.53',
            'XS::APItest'           => '0.53',
            '_charnames'            => '1.37',
            'autodie'               => '2.19',
            'autodie::exception'    => '2.19',
            'autodie::exception::system'=> '2.19',
            'autodie::hints'        => '2.19',
            'autodie::skip'         => '2.19',
            'bigint'                => '0.35',
            'charnames'             => '1.38',
            'encoding'              => '2.12',
            'inc::latest'           => '0.4005',
            'mro'                   => '1.12',
            'perlfaq'               => '5.0150043',
            're'                    => '0.25',
            'threads'               => '1.87',
            'threads::shared'       => '1.44',
            'utf8'                  => '1.12',
        },
        removed => {
        }
    },
    5.019002 => {
        delta_from => 5.019001,
        changed => {
            'B'                     => '1.44',
            'B::Concise'            => '0.98',
            'B::Deparse'            => '1.22',
            'Benchmark'             => '1.17',
            'Class::Struct'         => '0.65',
            'Config'                => '5.019002',
            'DB'                    => '1.07',
            'DBM_Filter'            => '0.06',
            'DBM_Filter::compress'  => '0.03',
            'DBM_Filter::encode'    => '0.03',
            'DBM_Filter::int32'     => '0.03',
            'DBM_Filter::null'      => '0.03',
            'DBM_Filter::utf8'      => '0.03',
            'DB_File'               => '1.829',
            'Data::Dumper'          => '2.147',
            'Devel::Peek'           => '1.12',
            'Digest::MD5'           => '2.53',
            'Digest::SHA'           => '5.85',
            'English'               => '1.07',
            'Errno'                 => '1.19',
            'ExtUtils::Embed'       => '1.31',
            'ExtUtils::Miniperl'    => '1',
            'ExtUtils::ParseXS'     => '3.21',
            'ExtUtils::ParseXS::Constants'=> '3.21',
            'ExtUtils::ParseXS::CountLines'=> '3.21',
            'ExtUtils::ParseXS::Eval'=> '3.19',
            'ExtUtils::ParseXS::Utilities'=> '3.21',
            'ExtUtils::Typemaps'    => '3.21',
            'ExtUtils::Typemaps::Cmd'=> '3.21',
            'ExtUtils::Typemaps::InputMap'=> '3.21',
            'ExtUtils::Typemaps::OutputMap'=> '3.21',
            'ExtUtils::Typemaps::Type'=> '3.21',
            'ExtUtils::XSSymSet'    => '1.3',
            'Fatal'                 => '2.20',
            'File::Basename'        => '2.85',
            'File::Spec::VMS'       => '3.43',
            'File::Spec::Win32'     => '3.42',
            'Getopt::Long'          => '2.41',
            'Getopt::Std'           => '1.09',
            'HTTP::Tiny'            => '0.034',
            'Hash::Util::FieldHash' => '1.12',
            'I18N::Langinfo'        => '0.11',
            'IO::Socket::INET'      => '1.34',
            'IO::Socket::UNIX'      => '1.25',
            'IPC::Cmd'              => '0.82',
            'MIME::Base64'          => '3.14',
            'Module::CoreList'      => '2.94',
            'Module::CoreList::TieHashDelta'=> '2.94',
            'Module::CoreList::Utils'=> '2.94',
            'POSIX'                 => '1.34',
            'Params::Check'         => '0.38',
            'Parse::CPAN::Meta'     => '1.4405',
            'Pod::Functions'        => '1.07',
            'Pod::Html'             => '1.2',
            'Safe'                  => '2.37',
            'Socket'                => '2.010',
            'Storable'              => '2.45',
            'Text::ParseWords'      => '3.29',
            'Tie::Array'            => '1.06',
            'Tie::Hash'             => '1.05',
            'Tie::Scalar'           => '1.03',
            'Time::Piece'           => '1.21',
            'Time::Seconds'         => '1.21',
            'XS::APItest'           => '0.54',
            'autodie'               => '2.20',
            'autodie::exception'    => '2.20',
            'autodie::exception::system'=> '2.20',
            'autodie::hints'        => '2.20',
            'autodie::skip'         => '2.20',
            'base'                  => '2.19',
            'deprecate'             => '0.03',
            'if'                    => '0.0603',
            'integer'               => '1.01',
            'strict'                => '1.08',
            'subs'                  => '1.02',
            'vmsish'                => '1.04',
        },
        removed => {
        }
    },
    5.019003 => {
        delta_from => 5.019002,
        changed => {
            'B'                     => '1.45',
            'CPAN::Meta'            => '2.132140',
            'CPAN::Meta::Converter' => '2.132140',
            'CPAN::Meta::Feature'   => '2.132140',
            'CPAN::Meta::History'   => '2.132140',
            'CPAN::Meta::Prereqs'   => '2.132140',
            'CPAN::Meta::Spec'      => '2.132140',
            'CPAN::Meta::Validator' => '2.132140',
            'Carp'                  => '1.31',
            'Carp::Heavy'           => '1.31',
            'Compress::Raw::Bzip2'  => '2.062',
            'Compress::Raw::Zlib'   => '2.062',
            'Compress::Zlib'        => '2.062',
            'Config'                => '5.019003',
            'Config::Perl::V'       => '0.19',
            'Cwd'                   => '3.44',
            'Data::Dumper'          => '2.148',
            'Devel::PPPort'         => '3.21',
            'Devel::Peek'           => '1.13',
            'DynaLoader'            => '1.19',
            'Encode'                => '2.52',
            'Encode::Alias'         => '2.17',
            'Encode::Encoding'      => '2.06',
            'Encode::GSM0338'       => '2.04',
            'Encode::MIME::Header'  => '2.14',
            'Encode::Unicode'       => '2.08',
            'English'               => '1.08',
            'Exporter'              => '5.69',
            'Exporter::Heavy'       => '5.69',
            'ExtUtils::Command::MM' => '6.72',
            'ExtUtils::Liblist'     => '6.72',
            'ExtUtils::Liblist::Kid'=> '6.72',
            'ExtUtils::MM'          => '6.72',
            'ExtUtils::MM_AIX'      => '6.72',
            'ExtUtils::MM_Any'      => '6.72',
            'ExtUtils::MM_BeOS'     => '6.72',
            'ExtUtils::MM_Cygwin'   => '6.72',
            'ExtUtils::MM_DOS'      => '6.72',
            'ExtUtils::MM_Darwin'   => '6.72',
            'ExtUtils::MM_MacOS'    => '6.72',
            'ExtUtils::MM_NW5'      => '6.72',
            'ExtUtils::MM_OS2'      => '6.72',
            'ExtUtils::MM_QNX'      => '6.72',
            'ExtUtils::MM_UWIN'     => '6.72',
            'ExtUtils::MM_Unix'     => '6.72',
            'ExtUtils::MM_VMS'      => '6.72',
            'ExtUtils::MM_VOS'      => '6.72',
            'ExtUtils::MM_Win32'    => '6.72',
            'ExtUtils::MM_Win95'    => '6.72',
            'ExtUtils::MY'          => '6.72',
            'ExtUtils::MakeMaker'   => '6.72',
            'ExtUtils::MakeMaker::Config'=> '6.72',
            'ExtUtils::Mkbootstrap' => '6.72',
            'ExtUtils::Mksymlists'  => '6.72',
            'ExtUtils::ParseXS::Eval'=> '3.21',
            'ExtUtils::testlib'     => '6.72',
            'File::Spec'            => '3.44',
            'File::Spec::Cygwin'    => '3.44',
            'File::Spec::Epoc'      => '3.44',
            'File::Spec::Functions' => '3.44',
            'File::Spec::Mac'       => '3.44',
            'File::Spec::OS2'       => '3.44',
            'File::Spec::Unix'      => '3.44',
            'File::Spec::VMS'       => '3.44',
            'File::Spec::Win32'     => '3.44',
            'Getopt::Std'           => '1.10',
            'IO::Compress::Adapter::Bzip2'=> '2.062',
            'IO::Compress::Adapter::Deflate'=> '2.062',
            'IO::Compress::Adapter::Identity'=> '2.062',
            'IO::Compress::Base'    => '2.062',
            'IO::Compress::Base::Common'=> '2.062',
            'IO::Compress::Bzip2'   => '2.062',
            'IO::Compress::Deflate' => '2.062',
            'IO::Compress::Gzip'    => '2.062',
            'IO::Compress::Gzip::Constants'=> '2.062',
            'IO::Compress::RawDeflate'=> '2.062',
            'IO::Compress::Zip'     => '2.062',
            'IO::Compress::Zip::Constants'=> '2.062',
            'IO::Compress::Zlib::Constants'=> '2.062',
            'IO::Compress::Zlib::Extra'=> '2.062',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.062',
            'IO::Uncompress::Adapter::Identity'=> '2.062',
            'IO::Uncompress::Adapter::Inflate'=> '2.062',
            'IO::Uncompress::AnyInflate'=> '2.062',
            'IO::Uncompress::AnyUncompress'=> '2.062',
            'IO::Uncompress::Base'  => '2.062',
            'IO::Uncompress::Bunzip2'=> '2.062',
            'IO::Uncompress::Gunzip'=> '2.062',
            'IO::Uncompress::Inflate'=> '2.062',
            'IO::Uncompress::RawInflate'=> '2.062',
            'IO::Uncompress::Unzip' => '2.062',
            'IPC::Cmd'              => '0.84',
            'IPC::Msg'              => '2.04',
            'IPC::Open3'            => '1.15',
            'IPC::Semaphore'        => '2.04',
            'IPC::SharedMem'        => '2.04',
            'IPC::SysV'             => '2.04',
            'List::Util'            => '1.31',
            'List::Util::XS'        => '1.31',
            'Math::BigFloat::Trace' => '0.36',
            'Math::BigInt::Trace'   => '0.36',
            'Module::Build'         => '0.4007',
            'Module::Build::Base'   => '0.4007',
            'Module::Build::Compat' => '0.4007',
            'Module::Build::Config' => '0.4007',
            'Module::Build::Cookbook'=> '0.4007',
            'Module::Build::Dumper' => '0.4007',
            'Module::Build::ModuleInfo'=> '0.4007',
            'Module::Build::Notes'  => '0.4007',
            'Module::Build::PPMMaker'=> '0.4007',
            'Module::Build::Platform::Default'=> '0.4007',
            'Module::Build::Platform::MacOS'=> '0.4007',
            'Module::Build::Platform::Unix'=> '0.4007',
            'Module::Build::Platform::VMS'=> '0.4007',
            'Module::Build::Platform::VOS'=> '0.4007',
            'Module::Build::Platform::Windows'=> '0.4007',
            'Module::Build::Platform::aix'=> '0.4007',
            'Module::Build::Platform::cygwin'=> '0.4007',
            'Module::Build::Platform::darwin'=> '0.4007',
            'Module::Build::Platform::os2'=> '0.4007',
            'Module::Build::PodParser'=> '0.4007',
            'Module::CoreList'      => '2.97',
            'Module::CoreList::TieHashDelta'=> '2.97',
            'Module::CoreList::Utils'=> '2.97',
            'Net::Cmd'              => '2.30',
            'Net::Config'           => '1.12',
            'Net::Domain'           => '2.22',
            'Net::FTP'              => '2.78',
            'Net::FTP::dataconn'    => '0.12',
            'Net::NNTP'             => '2.25',
            'Net::Netrc'            => '2.14',
            'Net::POP3'             => '2.30',
            'Net::SMTP'             => '2.32',
            'PerlIO'                => '1.08',
            'Pod::Functions'        => '1.08',
            'Scalar::Util'          => '1.31',
            'Socket'                => '2.011',
            'Storable'              => '2.46',
            'Time::HiRes'           => '1.9726',
            'Time::Piece'           => '1.22',
            'Time::Seconds'         => '1.22',
            'XS::APItest'           => '0.55',
            'bigint'                => '0.36',
            'bignum'                => '0.36',
            'bigrat'                => '0.36',
            'constant'              => '1.28',
            'diagnostics'           => '1.32',
            'inc::latest'           => '0.4007',
            'mro'                   => '1.13',
            'parent'                => '0.226',
            'utf8'                  => '1.13',
            'version'               => '0.9903',
        },
        removed => {
           'Module::Build::Platform::Amiga'=> 1,
           'Module::Build::Platform::EBCDIC'=> 1,
           'Module::Build::Platform::MPEiX'=> 1,
           'Module::Build::Platform::RiscOS'=> 1,
        }
    },
    5.019004 => {
        delta_from => 5.019003,
        changed => {
            'B'                     => '1.46',
            'B::Concise'            => '0.99',
            'B::Deparse'            => '1.23',
            'CPAN'                  => '2.03',
            'CPAN::Meta'            => '2.132620',
            'CPAN::Meta::Converter' => '2.132620',
            'CPAN::Meta::Feature'   => '2.132620',
            'CPAN::Meta::History'   => '2.132620',
            'CPAN::Meta::Prereqs'   => '2.132620',
            'CPAN::Meta::Requirements'=> '2.123',
            'CPAN::Meta::Spec'      => '2.132620',
            'CPAN::Meta::Validator' => '2.132620',
            'Carp'                  => '1.32',
            'Carp::Heavy'           => '1.32',
            'Config'                => '5.019004',
            'Data::Dumper'          => '2.149',
            'Devel::Peek'           => '1.14',
            'DynaLoader'            => '1.20',
            'Encode'                => '2.55',
            'Encode::Alias'         => '2.18',
            'Encode::CN::HZ'        => '2.07',
            'Encode::Encoder'       => '2.03',
            'Encode::Encoding'      => '2.07',
            'Encode::GSM0338'       => '2.05',
            'Encode::Guess'         => '2.06',
            'Encode::JP::JIS7'      => '2.05',
            'Encode::KR::2022_KR'   => '2.03',
            'Encode::MIME::Header'  => '2.15',
            'Encode::MIME::Header::ISO_2022_JP'=> '1.04',
            'Encode::Unicode'       => '2.09',
            'Encode::Unicode::UTF7' => '2.08',
            'Errno'                 => '1.20',
            'Exporter'              => '5.70',
            'Exporter::Heavy'       => '5.70',
            'ExtUtils::CBuilder'    => '0.280212',
            'ExtUtils::CBuilder::Base'=> '0.280212',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.280212',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.280212',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.280212',
            'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280212',
            'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280212',
            'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280212',
            'ExtUtils::CBuilder::Platform::aix'=> '0.280212',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.280212',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.280212',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280212',
            'ExtUtils::CBuilder::Platform::os2'=> '0.280212',
            'ExtUtils::Command'     => '1.18',
            'ExtUtils::Command::MM' => '6.76',
            'ExtUtils::Liblist'     => '6.76',
            'ExtUtils::Liblist::Kid'=> '6.76',
            'ExtUtils::MM'          => '6.76',
            'ExtUtils::MM_AIX'      => '6.76',
            'ExtUtils::MM_Any'      => '6.76',
            'ExtUtils::MM_BeOS'     => '6.76',
            'ExtUtils::MM_Cygwin'   => '6.76',
            'ExtUtils::MM_DOS'      => '6.76',
            'ExtUtils::MM_Darwin'   => '6.76',
            'ExtUtils::MM_MacOS'    => '6.76',
            'ExtUtils::MM_NW5'      => '6.76',
            'ExtUtils::MM_OS2'      => '6.76',
            'ExtUtils::MM_QNX'      => '6.76',
            'ExtUtils::MM_UWIN'     => '6.76',
            'ExtUtils::MM_Unix'     => '6.76',
            'ExtUtils::MM_VMS'      => '6.76',
            'ExtUtils::MM_VOS'      => '6.76',
            'ExtUtils::MM_Win32'    => '6.76',
            'ExtUtils::MM_Win95'    => '6.76',
            'ExtUtils::MY'          => '6.76',
            'ExtUtils::MakeMaker'   => '6.76',
            'ExtUtils::MakeMaker::Config'=> '6.76',
            'ExtUtils::Mkbootstrap' => '6.76',
            'ExtUtils::Mksymlists'  => '6.76',
            'ExtUtils::ParseXS'     => '3.23',
            'ExtUtils::ParseXS::Constants'=> '3.23',
            'ExtUtils::ParseXS::CountLines'=> '3.23',
            'ExtUtils::ParseXS::Eval'=> '3.23',
            'ExtUtils::ParseXS::Utilities'=> '3.23',
            'ExtUtils::Typemaps'    => '3.23',
            'ExtUtils::Typemaps::Cmd'=> '3.23',
            'ExtUtils::Typemaps::InputMap'=> '3.23',
            'ExtUtils::Typemaps::OutputMap'=> '3.23',
            'ExtUtils::Typemaps::Type'=> '3.23',
            'ExtUtils::testlib'     => '6.76',
            'Fatal'                 => '2.21',
            'File::Copy'            => '2.28',
            'File::Find'            => '1.25',
            'File::Glob'            => '1.21',
            'FileCache'             => '1.09',
            'HTTP::Tiny'            => '0.035',
            'Hash::Util::FieldHash' => '1.13',
            'I18N::LangTags'        => '0.40',
            'IO'                    => '1.29',
            'IO::Socket'            => '1.37',
            'IPC::Open3'            => '1.16',
            'JSON::PP'              => '2.27202_01',
            'List::Util'            => '1.32',
            'List::Util::XS'        => '1.32',
            'Locale::Codes'         => '3.27',
            'Locale::Codes::Constants'=> '3.27',
            'Locale::Codes::Country'=> '3.27',
            'Locale::Codes::Country_Codes'=> '3.27',
            'Locale::Codes::Country_Retired'=> '3.27',
            'Locale::Codes::Currency'=> '3.27',
            'Locale::Codes::Currency_Codes'=> '3.27',
            'Locale::Codes::Currency_Retired'=> '3.27',
            'Locale::Codes::LangExt'=> '3.27',
            'Locale::Codes::LangExt_Codes'=> '3.27',
            'Locale::Codes::LangExt_Retired'=> '3.27',
            'Locale::Codes::LangFam'=> '3.27',
            'Locale::Codes::LangFam_Codes'=> '3.27',
            'Locale::Codes::LangFam_Retired'=> '3.27',
            'Locale::Codes::LangVar'=> '3.27',
            'Locale::Codes::LangVar_Codes'=> '3.27',
            'Locale::Codes::LangVar_Retired'=> '3.27',
            'Locale::Codes::Language'=> '3.27',
            'Locale::Codes::Language_Codes'=> '3.27',
            'Locale::Codes::Language_Retired'=> '3.27',
            'Locale::Codes::Script' => '3.27',
            'Locale::Codes::Script_Codes'=> '3.27',
            'Locale::Codes::Script_Retired'=> '3.27',
            'Locale::Country'       => '3.27',
            'Locale::Currency'      => '3.27',
            'Locale::Language'      => '3.27',
            'Locale::Script'        => '3.27',
            'Math::BigFloat'        => '1.9991',
            'Math::BigInt'          => '1.9993',
            'Math::BigInt::FastCalc'=> '0.31',
            'Module::CoreList'      => '2.99',
            'Module::CoreList::TieHashDelta'=> '2.99',
            'Module::CoreList::Utils'=> '2.99',
            'Module::Load::Conditional'=> '0.58',
            'Module::Metadata'      => '1.000018',
            'Opcode'                => '1.26',
            'POSIX'                 => '1.35',
            'Parse::CPAN::Meta'     => '1.4407',
            'Perl::OSType'          => '1.005',
            'Pod::Html'             => '1.21',
            'Scalar::Util'          => '1.32',
            'Socket'                => '2.012',
            'Storable'              => '2.47',
            'Term::ReadLine'        => '1.14',
            'Test::Builder'         => '0.98_06',
            'Test::Builder::Module' => '0.98_06',
            'Test::More'            => '0.98_06',
            'Test::Simple'          => '0.98_06',
            'Time::Piece'           => '1.23',
            'Time::Seconds'         => '1.23',
            'Unicode::Collate'      => '0.99',
            'Unicode::UCD'          => '0.54',
            'XS::APItest'           => '0.56',
            'XS::Typemap'           => '0.11',
            '_charnames'            => '1.39',
            'autodie'               => '2.21',
            'autodie::exception'    => '2.21',
            'autodie::exception::system'=> '2.21',
            'autodie::hints'        => '2.21',
            'autodie::skip'         => '2.21',
            'charnames'             => '1.39',
            'diagnostics'           => '1.33',
            'mro'                   => '1.14',
            'parent'                => '0.228',
            'perlfaq'               => '5.0150044',
            're'                    => '0.26',
            'version'               => '0.9904',
            'warnings'              => '1.19',
        },
        removed => {
        }
    },
    5.019005 => {
        delta_from => 5.019004,
        changed => {
            'App::Prove'            => '3.29',
            'App::Prove::State'     => '3.29',
            'App::Prove::State::Result'=> '3.29',
            'App::Prove::State::Result::Test'=> '3.29',
            'CPAN::Meta'            => '2.132830',
            'CPAN::Meta::Converter' => '2.132830',
            'CPAN::Meta::Feature'   => '2.132830',
            'CPAN::Meta::History'   => '2.132830',
            'CPAN::Meta::Prereqs'   => '2.132830',
            'CPAN::Meta::Requirements'=> '2.125',
            'CPAN::Meta::Spec'      => '2.132830',
            'CPAN::Meta::Validator' => '2.132830',
            'CPAN::Meta::YAML'      => '0.010',
            'Config'                => '5.019005',
            'Cwd'                   => '3.45',
            'ExtUtils::Command::MM' => '6.80',
            'ExtUtils::Install'     => '1.61',
            'ExtUtils::Liblist'     => '6.80',
            'ExtUtils::Liblist::Kid'=> '6.80',
            'ExtUtils::MM'          => '6.80',
            'ExtUtils::MM_AIX'      => '6.80',
            'ExtUtils::MM_Any'      => '6.80',
            'ExtUtils::MM_BeOS'     => '6.80',
            'ExtUtils::MM_Cygwin'   => '6.80',
            'ExtUtils::MM_DOS'      => '6.80',
            'ExtUtils::MM_Darwin'   => '6.80',
            'ExtUtils::MM_MacOS'    => '6.80',
            'ExtUtils::MM_NW5'      => '6.80',
            'ExtUtils::MM_OS2'      => '6.80',
            'ExtUtils::MM_QNX'      => '6.80',
            'ExtUtils::MM_UWIN'     => '6.80',
            'ExtUtils::MM_Unix'     => '6.80',
            'ExtUtils::MM_VMS'      => '6.80',
            'ExtUtils::MM_VOS'      => '6.80',
            'ExtUtils::MM_Win32'    => '6.80',
            'ExtUtils::MM_Win95'    => '6.80',
            'ExtUtils::MY'          => '6.80',
            'ExtUtils::MakeMaker'   => '6.80',
            'ExtUtils::MakeMaker::Config'=> '6.80',
            'ExtUtils::Mkbootstrap' => '6.80',
            'ExtUtils::Mksymlists'  => '6.80',
            'ExtUtils::testlib'     => '6.80',
            'Fatal'                 => '2.22',
            'File::Fetch'           => '0.44',
            'File::Glob'            => '1.22',
            'File::Spec'            => '3.45',
            'File::Spec::Cygwin'    => '3.45',
            'File::Spec::Epoc'      => '3.45',
            'File::Spec::Functions' => '3.45',
            'File::Spec::Mac'       => '3.45',
            'File::Spec::OS2'       => '3.45',
            'File::Spec::Unix'      => '3.45',
            'File::Spec::VMS'       => '3.45',
            'File::Spec::Win32'     => '3.45',
            'File::Temp'            => '0.2304',
            'Getopt::Long'          => '2.42',
            'HTTP::Tiny'            => '0.036',
            'IPC::Cmd'              => '0.84_01',
            'JSON::PP'              => '2.27203',
            'List::Util'            => '1.35',
            'List::Util::XS'        => '1.35',
            'Module::CoreList'      => '3.00',
            'Module::CoreList::TieHashDelta'=> '3.00',
            'Module::CoreList::Utils'=> '3.00',
            'Module::Metadata'      => '1.000019',
            'Parse::CPAN::Meta'     => '1.4409',
            'Perl::OSType'          => '1.006',
            'PerlIO::scalar'        => '0.17',
            'Pod::Man'              => '2.28',
            'Pod::Text'             => '3.18',
            'Pod::Text::Termcap'    => '2.08',
            'Scalar::Util'          => '1.35',
            'TAP::Base'             => '3.29',
            'TAP::Formatter::Base'  => '3.29',
            'TAP::Formatter::Color' => '3.29',
            'TAP::Formatter::Console'=> '3.29',
            'TAP::Formatter::Console::ParallelSession'=> '3.29',
            'TAP::Formatter::Console::Session'=> '3.29',
            'TAP::Formatter::File'  => '3.29',
            'TAP::Formatter::File::Session'=> '3.29',
            'TAP::Formatter::Session'=> '3.29',
            'TAP::Harness'          => '3.29',
            'TAP::Harness::Env'     => '3.29',
            'TAP::Object'           => '3.29',
            'TAP::Parser'           => '3.29',
            'TAP::Parser::Aggregator'=> '3.29',
            'TAP::Parser::Grammar'  => '3.29',
            'TAP::Parser::Iterator' => '3.29',
            'TAP::Parser::Iterator::Array'=> '3.29',
            'TAP::Parser::Iterator::Process'=> '3.29',
            'TAP::Parser::Iterator::Stream'=> '3.29',
            'TAP::Parser::IteratorFactory'=> '3.29',
            'TAP::Parser::Multiplexer'=> '3.29',
            'TAP::Parser::Result'   => '3.29',
            'TAP::Parser::Result::Bailout'=> '3.29',
            'TAP::Parser::Result::Comment'=> '3.29',
            'TAP::Parser::Result::Plan'=> '3.29',
            'TAP::Parser::Result::Pragma'=> '3.29',
            'TAP::Parser::Result::Test'=> '3.29',
            'TAP::Parser::Result::Unknown'=> '3.29',
            'TAP::Parser::Result::Version'=> '3.29',
            'TAP::Parser::Result::YAML'=> '3.29',
            'TAP::Parser::ResultFactory'=> '3.29',
            'TAP::Parser::Scheduler'=> '3.29',
            'TAP::Parser::Scheduler::Job'=> '3.29',
            'TAP::Parser::Scheduler::Spinner'=> '3.29',
            'TAP::Parser::Source'   => '3.29',
            'TAP::Parser::SourceHandler'=> '3.29',
            'TAP::Parser::SourceHandler::Executable'=> '3.29',
            'TAP::Parser::SourceHandler::File'=> '3.29',
            'TAP::Parser::SourceHandler::Handle'=> '3.29',
            'TAP::Parser::SourceHandler::Perl'=> '3.29',
            'TAP::Parser::SourceHandler::RawTAP'=> '3.29',
            'TAP::Parser::YAMLish::Reader'=> '3.29',
            'TAP::Parser::YAMLish::Writer'=> '3.29',
            'Test::Builder'         => '0.99',
            'Test::Builder::Module' => '0.99',
            'Test::Builder::Tester' => '1.23_002',
            'Test::Builder::Tester::Color'=> '1.23_002',
            'Test::Harness'         => '3.29',
            'Test::More'            => '0.99',
            'Test::Simple'          => '0.99',
            'Unicode'               => '6.3.0',
            'Unicode::Normalize'    => '1.17',
            'Unicode::UCD'          => '0.55',
            'attributes'            => '0.22',
            'autodie'               => '2.22',
            'autodie::exception'    => '2.22',
            'autodie::exception::system'=> '2.22',
            'autodie::hints'        => '2.22',
            'autodie::skip'         => '2.22',
            'feature'               => '1.34',
            'threads'               => '1.89',
            'warnings'              => '1.20',
        },
        removed => {
            'TAP::Parser::Utils'    => 1,
        }
    },
    5.019006 => {
        delta_from => 5.019005,
        changed => {
            'App::Prove'            => '3.30',
            'App::Prove::State'     => '3.30',
            'App::Prove::State::Result'=> '3.30',
            'App::Prove::State::Result::Test'=> '3.30',
            'Archive::Tar'          => '1.96',
            'Archive::Tar::Constant'=> '1.96',
            'Archive::Tar::File'    => '1.96',
            'AutoLoader'            => '5.74',
            'B'                     => '1.47',
            'B::Concise'            => '0.991',
            'B::Debug'              => '1.19',
            'B::Deparse'            => '1.24',
            'Benchmark'             => '1.18',
            'Compress::Raw::Bzip2'  => '2.063',
            'Compress::Raw::Zlib'   => '2.063',
            'Compress::Zlib'        => '2.063',
            'Config'                => '5.019006',
            'DB_File'               => '1.831',
            'Devel::Peek'           => '1.15',
            'DynaLoader'            => '1.21',
            'Errno'                 => '1.20_01',
            'ExtUtils::Command::MM' => '6.82',
            'ExtUtils::Liblist'     => '6.82',
            'ExtUtils::Liblist::Kid'=> '6.82',
            'ExtUtils::MM'          => '6.82',
            'ExtUtils::MM_AIX'      => '6.82',
            'ExtUtils::MM_Any'      => '6.82',
            'ExtUtils::MM_BeOS'     => '6.82',
            'ExtUtils::MM_Cygwin'   => '6.82',
            'ExtUtils::MM_DOS'      => '6.82',
            'ExtUtils::MM_Darwin'   => '6.82',
            'ExtUtils::MM_MacOS'    => '6.82',
            'ExtUtils::MM_NW5'      => '6.82',
            'ExtUtils::MM_OS2'      => '6.82',
            'ExtUtils::MM_QNX'      => '6.82',
            'ExtUtils::MM_UWIN'     => '6.82',
            'ExtUtils::MM_Unix'     => '6.82',
            'ExtUtils::MM_VMS'      => '6.82',
            'ExtUtils::MM_VOS'      => '6.82',
            'ExtUtils::MM_Win32'    => '6.82',
            'ExtUtils::MM_Win95'    => '6.82',
            'ExtUtils::MY'          => '6.82',
            'ExtUtils::MakeMaker'   => '6.82',
            'ExtUtils::MakeMaker::Config'=> '6.82',
            'ExtUtils::Mkbootstrap' => '6.82',
            'ExtUtils::Mksymlists'  => '6.82',
            'ExtUtils::testlib'     => '6.82',
            'File::DosGlob'         => '1.12',
            'File::Find'            => '1.26',
            'File::Glob'            => '1.23',
            'HTTP::Tiny'            => '0.038',
            'IO'                    => '1.30',
            'IO::Compress::Adapter::Bzip2'=> '2.063',
            'IO::Compress::Adapter::Deflate'=> '2.063',
            'IO::Compress::Adapter::Identity'=> '2.063',
            'IO::Compress::Base'    => '2.063',
            'IO::Compress::Base::Common'=> '2.063',
            'IO::Compress::Bzip2'   => '2.063',
            'IO::Compress::Deflate' => '2.063',
            'IO::Compress::Gzip'    => '2.063',
            'IO::Compress::Gzip::Constants'=> '2.063',
            'IO::Compress::RawDeflate'=> '2.063',
            'IO::Compress::Zip'     => '2.063',
            'IO::Compress::Zip::Constants'=> '2.063',
            'IO::Compress::Zlib::Constants'=> '2.063',
            'IO::Compress::Zlib::Extra'=> '2.063',
            'IO::Select'            => '1.22',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.063',
            'IO::Uncompress::Adapter::Identity'=> '2.063',
            'IO::Uncompress::Adapter::Inflate'=> '2.063',
            'IO::Uncompress::AnyInflate'=> '2.063',
            'IO::Uncompress::AnyUncompress'=> '2.063',
            'IO::Uncompress::Base'  => '2.063',
            'IO::Uncompress::Bunzip2'=> '2.063',
            'IO::Uncompress::Gunzip'=> '2.063',
            'IO::Uncompress::Inflate'=> '2.063',
            'IO::Uncompress::RawInflate'=> '2.063',
            'IO::Uncompress::Unzip' => '2.063',
            'IPC::Cmd'              => '0.90',
            'Locale::Maketext'      => '1.25',
            'Module::Build'         => '0.4202',
            'Module::Build::Base'   => '0.4202',
            'Module::Build::Compat' => '0.4202',
            'Module::Build::Config' => '0.4202',
            'Module::Build::Cookbook'=> '0.4202',
            'Module::Build::Dumper' => '0.4202',
            'Module::Build::ModuleInfo'=> '0.4202',
            'Module::Build::Notes'  => '0.4202',
            'Module::Build::PPMMaker'=> '0.4202',
            'Module::Build::Platform::Default'=> '0.4202',
            'Module::Build::Platform::MacOS'=> '0.4202',
            'Module::Build::Platform::Unix'=> '0.4202',
            'Module::Build::Platform::VMS'=> '0.4202',
            'Module::Build::Platform::VOS'=> '0.4202',
            'Module::Build::Platform::Windows'=> '0.4202',
            'Module::Build::Platform::aix'=> '0.4202',
            'Module::Build::Platform::cygwin'=> '0.4202',
            'Module::Build::Platform::darwin'=> '0.4202',
            'Module::Build::Platform::os2'=> '0.4202',
            'Module::Build::PodParser'=> '0.4202',
            'Module::CoreList'      => '3.01',
            'Module::CoreList::TieHashDelta'=> '3.01',
            'Module::CoreList::Utils'=> '3.01',
            'Opcode'                => '1.27',
            'POSIX'                 => '1.36',
            'Package::Constants'    => '0.04',
            'PerlIO::scalar'        => '0.18',
            'PerlIO::via'           => '0.13',
            'SDBM_File'             => '1.10',
            'Socket'                => '2.013',
            'TAP::Base'             => '3.30',
            'TAP::Formatter::Base'  => '3.30',
            'TAP::Formatter::Color' => '3.30',
            'TAP::Formatter::Console'=> '3.30',
            'TAP::Formatter::Console::ParallelSession'=> '3.30',
            'TAP::Formatter::Console::Session'=> '3.30',
            'TAP::Formatter::File'  => '3.30',
            'TAP::Formatter::File::Session'=> '3.30',
            'TAP::Formatter::Session'=> '3.30',
            'TAP::Harness'          => '3.30',
            'TAP::Harness::Env'     => '3.30',
            'TAP::Object'           => '3.30',
            'TAP::Parser'           => '3.30',
            'TAP::Parser::Aggregator'=> '3.30',
            'TAP::Parser::Grammar'  => '3.30',
            'TAP::Parser::Iterator' => '3.30',
            'TAP::Parser::Iterator::Array'=> '3.30',
            'TAP::Parser::Iterator::Process'=> '3.30',
            'TAP::Parser::Iterator::Stream'=> '3.30',
            'TAP::Parser::IteratorFactory'=> '3.30',
            'TAP::Parser::Multiplexer'=> '3.30',
            'TAP::Parser::Result'   => '3.30',
            'TAP::Parser::Result::Bailout'=> '3.30',
            'TAP::Parser::Result::Comment'=> '3.30',
            'TAP::Parser::Result::Plan'=> '3.30',
            'TAP::Parser::Result::Pragma'=> '3.30',
            'TAP::Parser::Result::Test'=> '3.30',
            'TAP::Parser::Result::Unknown'=> '3.30',
            'TAP::Parser::Result::Version'=> '3.30',
            'TAP::Parser::Result::YAML'=> '3.30',
            'TAP::Parser::ResultFactory'=> '3.30',
            'TAP::Parser::Scheduler'=> '3.30',
            'TAP::Parser::Scheduler::Job'=> '3.30',
            'TAP::Parser::Scheduler::Spinner'=> '3.30',
            'TAP::Parser::Source'   => '3.30',
            'TAP::Parser::SourceHandler'=> '3.30',
            'TAP::Parser::SourceHandler::Executable'=> '3.30',
            'TAP::Parser::SourceHandler::File'=> '3.30',
            'TAP::Parser::SourceHandler::Handle'=> '3.30',
            'TAP::Parser::SourceHandler::Perl'=> '3.30',
            'TAP::Parser::SourceHandler::RawTAP'=> '3.30',
            'TAP::Parser::YAMLish::Reader'=> '3.30',
            'TAP::Parser::YAMLish::Writer'=> '3.30',
            'Term::Cap'             => '1.15',
            'Test::Builder'         => '1.001002',
            'Test::Builder::Module' => '1.001002',
            'Test::Harness'         => '3.30',
            'Test::More'            => '1.001002',
            'Test::Simple'          => '1.001002',
            'Tie::StdHandle'        => '4.4',
            'Unicode::Collate'      => '1.02',
            'Unicode::Collate::CJK::Korean'=> '1.02',
            'Unicode::Collate::Locale'=> '1.02',
            'XS::APItest'           => '0.57',
            'XS::Typemap'           => '0.12',
            'arybase'               => '0.07',
            'bignum'                => '0.37',
            'constant'              => '1.29',
            'fields'                => '2.17',
            'inc::latest'           => '0.4202',
            'threads'               => '1.90',
            'threads::shared'       => '1.45',
        },
        removed => {
        }
    },
    5.019007 => {
        delta_from => 5.019006,
        changed => {
            'CGI'                   => '3.64',
            'CGI::Apache'           => '1.02',
            'CGI::Carp'             => '3.64',
            'CGI::Cookie'           => '1.31',
            'CGI::Fast'             => '1.10',
            'CGI::Pretty'           => '3.64',
            'CGI::Push'             => '1.06',
            'CGI::Switch'           => '1.02',
            'CGI::Util'             => '3.64',
            'CPAN::Meta'            => '2.133380',
            'CPAN::Meta::Converter' => '2.133380',
            'CPAN::Meta::Feature'   => '2.133380',
            'CPAN::Meta::History'   => '2.133380',
            'CPAN::Meta::Prereqs'   => '2.133380',
            'CPAN::Meta::Spec'      => '2.133380',
            'CPAN::Meta::Validator' => '2.133380',
            'Config'                => '5.019007',
            'Data::Dumper'          => '2.150',
            'DynaLoader'            => '1.22',
            'ExtUtils::Command::MM' => '6.84',
            'ExtUtils::Liblist'     => '6.84',
            'ExtUtils::Liblist::Kid'=> '6.84',
            'ExtUtils::MM'          => '6.84',
            'ExtUtils::MM_AIX'      => '6.84',
            'ExtUtils::MM_Any'      => '6.84',
            'ExtUtils::MM_BeOS'     => '6.84',
            'ExtUtils::MM_Cygwin'   => '6.84',
            'ExtUtils::MM_DOS'      => '6.84',
            'ExtUtils::MM_Darwin'   => '6.84',
            'ExtUtils::MM_MacOS'    => '6.84',
            'ExtUtils::MM_NW5'      => '6.84',
            'ExtUtils::MM_OS2'      => '6.84',
            'ExtUtils::MM_QNX'      => '6.84',
            'ExtUtils::MM_UWIN'     => '6.84',
            'ExtUtils::MM_Unix'     => '6.84',
            'ExtUtils::MM_VMS'      => '6.84',
            'ExtUtils::MM_VOS'      => '6.84',
            'ExtUtils::MM_Win32'    => '6.84',
            'ExtUtils::MM_Win95'    => '6.84',
            'ExtUtils::MY'          => '6.84',
            'ExtUtils::MakeMaker'   => '6.84',
            'ExtUtils::MakeMaker::Config'=> '6.84',
            'ExtUtils::Mkbootstrap' => '6.84',
            'ExtUtils::Mksymlists'  => '6.84',
            'ExtUtils::testlib'     => '6.84',
            'File::Fetch'           => '0.46',
            'HTTP::Tiny'            => '0.039',
            'Locale::Codes'         => '3.28',
            'Locale::Codes::Constants'=> '3.28',
            'Locale::Codes::Country'=> '3.28',
            'Locale::Codes::Country_Codes'=> '3.28',
            'Locale::Codes::Country_Retired'=> '3.28',
            'Locale::Codes::Currency'=> '3.28',
            'Locale::Codes::Currency_Codes'=> '3.28',
            'Locale::Codes::Currency_Retired'=> '3.28',
            'Locale::Codes::LangExt'=> '3.28',
            'Locale::Codes::LangExt_Codes'=> '3.28',
            'Locale::Codes::LangExt_Retired'=> '3.28',
            'Locale::Codes::LangFam'=> '3.28',
            'Locale::Codes::LangFam_Codes'=> '3.28',
            'Locale::Codes::LangFam_Retired'=> '3.28',
            'Locale::Codes::LangVar'=> '3.28',
            'Locale::Codes::LangVar_Codes'=> '3.28',
            'Locale::Codes::LangVar_Retired'=> '3.28',
            'Locale::Codes::Language'=> '3.28',
            'Locale::Codes::Language_Codes'=> '3.28',
            'Locale::Codes::Language_Retired'=> '3.28',
            'Locale::Codes::Script' => '3.28',
            'Locale::Codes::Script_Codes'=> '3.28',
            'Locale::Codes::Script_Retired'=> '3.28',
            'Locale::Country'       => '3.28',
            'Locale::Currency'      => '3.28',
            'Locale::Language'      => '3.28',
            'Locale::Script'        => '3.28',
            'Module::Build'         => '0.4203',
            'Module::Build::Base'   => '0.4203',
            'Module::Build::Compat' => '0.4203',
            'Module::Build::Config' => '0.4203',
            'Module::Build::Cookbook'=> '0.4203',
            'Module::Build::Dumper' => '0.4203',
            'Module::Build::ModuleInfo'=> '0.4203',
            'Module::Build::Notes'  => '0.4203',
            'Module::Build::PPMMaker'=> '0.4203',
            'Module::Build::Platform::Default'=> '0.4203',
            'Module::Build::Platform::MacOS'=> '0.4203',
            'Module::Build::Platform::Unix'=> '0.4203',
            'Module::Build::Platform::VMS'=> '0.4203',
            'Module::Build::Platform::VOS'=> '0.4203',
            'Module::Build::Platform::Windows'=> '0.4203',
            'Module::Build::Platform::aix'=> '0.4203',
            'Module::Build::Platform::cygwin'=> '0.4203',
            'Module::Build::Platform::darwin'=> '0.4203',
            'Module::Build::Platform::os2'=> '0.4203',
            'Module::Build::PodParser'=> '0.4203',
            'Module::CoreList'      => '3.02',
            'Module::CoreList::TieHashDelta'=> '3.02',
            'Module::CoreList::Utils'=> '3.02',
            'POSIX'                 => '1.37',
            'PerlIO::encoding'      => '0.17',
            'PerlIO::via'           => '0.14',
            'SDBM_File'             => '1.11',
            'Storable'              => '2.48',
            'Time::Piece'           => '1.24',
            'Time::Seconds'         => '1.24',
            'Unicode::Collate'      => '1.04',
            'Win32'                 => '0.48',
            'XS::APItest'           => '0.58',
            'base'                  => '2.20',
            'constant'              => '1.30',
            'inc::latest'           => '0.4203',
            'threads'               => '1.91',
        },
        removed => {
        }
    },
    5.019008 => {
        delta_from => 5.019007,
        changed => {
            'Config'                => '5.019008',
            'DynaLoader'            => '1.24',
            'Encode'                => '2.57',
            'Errno'                 => '1.20_02',
            'ExtUtils::CBuilder'    => '0.280213',
            'ExtUtils::CBuilder::Base'=> '0.280213',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.280213',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.280213',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.280213',
            'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280213',
            'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280213',
            'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280213',
            'ExtUtils::CBuilder::Platform::aix'=> '0.280213',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.280213',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.280213',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280213',
            'ExtUtils::CBuilder::Platform::os2'=> '0.280213',
            'ExtUtils::Command::MM' => '6.86',
            'ExtUtils::Liblist'     => '6.86',
            'ExtUtils::Liblist::Kid'=> '6.86',
            'ExtUtils::MM'          => '6.86',
            'ExtUtils::MM_AIX'      => '6.86',
            'ExtUtils::MM_Any'      => '6.86',
            'ExtUtils::MM_BeOS'     => '6.86',
            'ExtUtils::MM_Cygwin'   => '6.86',
            'ExtUtils::MM_DOS'      => '6.86',
            'ExtUtils::MM_Darwin'   => '6.86',
            'ExtUtils::MM_MacOS'    => '6.86',
            'ExtUtils::MM_NW5'      => '6.86',
            'ExtUtils::MM_OS2'      => '6.86',
            'ExtUtils::MM_QNX'      => '6.86',
            'ExtUtils::MM_UWIN'     => '6.86',
            'ExtUtils::MM_Unix'     => '6.86',
            'ExtUtils::MM_VMS'      => '6.86',
            'ExtUtils::MM_VOS'      => '6.86',
            'ExtUtils::MM_Win32'    => '6.86',
            'ExtUtils::MM_Win95'    => '6.86',
            'ExtUtils::MY'          => '6.86',
            'ExtUtils::MakeMaker'   => '6.86',
            'ExtUtils::MakeMaker::Config'=> '6.86',
            'ExtUtils::Mkbootstrap' => '6.86',
            'ExtUtils::Mksymlists'  => '6.86',
            'ExtUtils::testlib'     => '6.86',
            'File::Copy'            => '2.29',
            'Hash::Util::FieldHash' => '1.14',
            'IO::Socket::IP'        => '0.26',
            'IO::Socket::UNIX'      => '1.26',
            'List::Util'            => '1.36',
            'List::Util::XS'        => '1.36',
            'Module::Build'         => '0.4204',
            'Module::Build::Base'   => '0.4204',
            'Module::Build::Compat' => '0.4204',
            'Module::Build::Config' => '0.4204',
            'Module::Build::Cookbook'=> '0.4204',
            'Module::Build::Dumper' => '0.4204',
            'Module::Build::ModuleInfo'=> '0.4204',
            'Module::Build::Notes'  => '0.4204',
            'Module::Build::PPMMaker'=> '0.4204',
            'Module::Build::Platform::Default'=> '0.4204',
            'Module::Build::Platform::MacOS'=> '0.4204',
            'Module::Build::Platform::Unix'=> '0.4204',
            'Module::Build::Platform::VMS'=> '0.4204',
            'Module::Build::Platform::VOS'=> '0.4204',
            'Module::Build::Platform::Windows'=> '0.4204',
            'Module::Build::Platform::aix'=> '0.4204',
            'Module::Build::Platform::cygwin'=> '0.4204',
            'Module::Build::Platform::darwin'=> '0.4204',
            'Module::Build::Platform::os2'=> '0.4204',
            'Module::Build::PodParser'=> '0.4204',
            'Module::CoreList'      => '3.04',
            'Module::CoreList::TieHashDelta'=> '3.04',
            'Module::CoreList::Utils'=> '3.04',
            'Module::Load'          => '0.28',
            'Module::Load::Conditional'=> '0.60',
            'Net::Config'           => '1.13',
            'Net::FTP::A'           => '1.19',
            'POSIX'                 => '1.38_01',
            'Perl::OSType'          => '1.007',
            'PerlIO::encoding'      => '0.18',
            'Pod::Perldoc'          => '3.21',
            'Pod::Perldoc::BaseTo'  => '3.21',
            'Pod::Perldoc::GetOptsOO'=> '3.21',
            'Pod::Perldoc::ToANSI'  => '3.21',
            'Pod::Perldoc::ToChecker'=> '3.21',
            'Pod::Perldoc::ToMan'   => '3.21',
            'Pod::Perldoc::ToNroff' => '3.21',
            'Pod::Perldoc::ToPod'   => '3.21',
            'Pod::Perldoc::ToRtf'   => '3.21',
            'Pod::Perldoc::ToTerm'  => '3.21',
            'Pod::Perldoc::ToText'  => '3.21',
            'Pod::Perldoc::ToTk'    => '3.21',
            'Pod::Perldoc::ToXml'   => '3.21',
            'Scalar::Util'          => '1.36',
            'Time::Piece'           => '1.27',
            'Time::Seconds'         => '1.27',
            'Unicode::UCD'          => '0.57',
            'XS::APItest'           => '0.59',
            'XSLoader'              => '0.17',
            'base'                  => '2.21',
            'constant'              => '1.31',
            'inc::latest'           => '0.4204',
            'threads::shared'       => '1.46',
            'version'               => '0.9907',
            'version::regex'        => '0.9907',
            'version::vpp'          => '0.9907',
            'warnings'              => '1.21',
        },
        removed => {
        }
    },
    5.019009 => {
        delta_from => 5.019008,
        changed => {
            'B'                     => '1.48',
            'B::Concise'            => '0.992',
            'B::Deparse'            => '1.25',
            'CGI'                   => '3.65',
            'CPAN::Meta::YAML'      => '0.011',
            'Compress::Raw::Bzip2'  => '2.064',
            'Compress::Raw::Zlib'   => '2.065',
            'Compress::Zlib'        => '2.064',
            'Config'                => '5.019009',
            'Config::Perl::V'       => '0.20',
            'Cwd'                   => '3.47',
            'Devel::Peek'           => '1.16',
            'Digest::SHA'           => '5.87',
            'DynaLoader'            => '1.25',
            'English'               => '1.09',
            'ExtUtils::CBuilder'    => '0.280216',
            'ExtUtils::CBuilder::Base'=> '0.280216',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.280216',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.280216',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.280216',
            'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280216',
            'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280216',
            'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280216',
            'ExtUtils::CBuilder::Platform::aix'=> '0.280216',
            'ExtUtils::CBuilder::Platform::android'=> '0.280216',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.280216',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.280216',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280216',
            'ExtUtils::CBuilder::Platform::os2'=> '0.280216',
            'ExtUtils::Command::MM' => '6.88',
            'ExtUtils::Embed'       => '1.32',
            'ExtUtils::Install'     => '1.62',
            'ExtUtils::Installed'   => '1.999004',
            'ExtUtils::Liblist'     => '6.88',
            'ExtUtils::Liblist::Kid'=> '6.88',
            'ExtUtils::MM'          => '6.88',
            'ExtUtils::MM_AIX'      => '6.88',
            'ExtUtils::MM_Any'      => '6.88',
            'ExtUtils::MM_BeOS'     => '6.88',
            'ExtUtils::MM_Cygwin'   => '6.88',
            'ExtUtils::MM_DOS'      => '6.88',
            'ExtUtils::MM_Darwin'   => '6.88',
            'ExtUtils::MM_MacOS'    => '6.88',
            'ExtUtils::MM_NW5'      => '6.88',
            'ExtUtils::MM_OS2'      => '6.88',
            'ExtUtils::MM_QNX'      => '6.88',
            'ExtUtils::MM_UWIN'     => '6.88',
            'ExtUtils::MM_Unix'     => '6.88',
            'ExtUtils::MM_VMS'      => '6.88',
            'ExtUtils::MM_VOS'      => '6.88',
            'ExtUtils::MM_Win32'    => '6.88',
            'ExtUtils::MM_Win95'    => '6.88',
            'ExtUtils::MY'          => '6.88',
            'ExtUtils::MakeMaker'   => '6.88',
            'ExtUtils::MakeMaker::Config'=> '6.88',
            'ExtUtils::Mkbootstrap' => '6.88',
            'ExtUtils::Mksymlists'  => '6.88',
            'ExtUtils::Packlist'    => '1.47',
            'ExtUtils::testlib'     => '6.88',
            'Fatal'                 => '2.23',
            'File::Fetch'           => '0.48',
            'File::Spec'            => '3.47',
            'File::Spec::Cygwin'    => '3.47',
            'File::Spec::Epoc'      => '3.47',
            'File::Spec::Functions' => '3.47',
            'File::Spec::Mac'       => '3.47',
            'File::Spec::OS2'       => '3.47',
            'File::Spec::Unix'      => '3.47',
            'File::Spec::VMS'       => '3.47',
            'File::Spec::Win32'     => '3.47',
            'HTTP::Tiny'            => '0.042',
            'IO::Compress::Adapter::Bzip2'=> '2.064',
            'IO::Compress::Adapter::Deflate'=> '2.064',
            'IO::Compress::Adapter::Identity'=> '2.064',
            'IO::Compress::Base'    => '2.064',
            'IO::Compress::Base::Common'=> '2.064',
            'IO::Compress::Bzip2'   => '2.064',
            'IO::Compress::Deflate' => '2.064',
            'IO::Compress::Gzip'    => '2.064',
            'IO::Compress::Gzip::Constants'=> '2.064',
            'IO::Compress::RawDeflate'=> '2.064',
            'IO::Compress::Zip'     => '2.064',
            'IO::Compress::Zip::Constants'=> '2.064',
            'IO::Compress::Zlib::Constants'=> '2.064',
            'IO::Compress::Zlib::Extra'=> '2.064',
            'IO::Socket::INET'      => '1.35',
            'IO::Socket::IP'        => '0.28',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.064',
            'IO::Uncompress::Adapter::Identity'=> '2.064',
            'IO::Uncompress::Adapter::Inflate'=> '2.064',
            'IO::Uncompress::AnyInflate'=> '2.064',
            'IO::Uncompress::AnyUncompress'=> '2.064',
            'IO::Uncompress::Base'  => '2.064',
            'IO::Uncompress::Bunzip2'=> '2.064',
            'IO::Uncompress::Gunzip'=> '2.064',
            'IO::Uncompress::Inflate'=> '2.064',
            'IO::Uncompress::RawInflate'=> '2.064',
            'IO::Uncompress::Unzip' => '2.064',
            'IPC::Cmd'              => '0.92',
            'List::Util'            => '1.38',
            'List::Util::XS'        => '1.38',
            'Locale::Codes'         => '3.29',
            'Locale::Codes::Constants'=> '3.29',
            'Locale::Codes::Country'=> '3.29',
            'Locale::Codes::Country_Codes'=> '3.29',
            'Locale::Codes::Country_Retired'=> '3.29',
            'Locale::Codes::Currency'=> '3.29',
            'Locale::Codes::Currency_Codes'=> '3.29',
            'Locale::Codes::Currency_Retired'=> '3.29',
            'Locale::Codes::LangExt'=> '3.29',
            'Locale::Codes::LangExt_Codes'=> '3.29',
            'Locale::Codes::LangExt_Retired'=> '3.29',
            'Locale::Codes::LangFam'=> '3.29',
            'Locale::Codes::LangFam_Codes'=> '3.29',
            'Locale::Codes::LangFam_Retired'=> '3.29',
            'Locale::Codes::LangVar'=> '3.29',
            'Locale::Codes::LangVar_Codes'=> '3.29',
            'Locale::Codes::LangVar_Retired'=> '3.29',
            'Locale::Codes::Language'=> '3.29',
            'Locale::Codes::Language_Codes'=> '3.29',
            'Locale::Codes::Language_Retired'=> '3.29',
            'Locale::Codes::Script' => '3.29',
            'Locale::Codes::Script_Codes'=> '3.29',
            'Locale::Codes::Script_Retired'=> '3.29',
            'Locale::Country'       => '3.29',
            'Locale::Currency'      => '3.29',
            'Locale::Language'      => '3.29',
            'Locale::Script'        => '3.29',
            'Module::Build'         => '0.4205',
            'Module::Build::Base'   => '0.4205',
            'Module::Build::Compat' => '0.4205',
            'Module::Build::Config' => '0.4205',
            'Module::Build::Cookbook'=> '0.4205',
            'Module::Build::Dumper' => '0.4205',
            'Module::Build::ModuleInfo'=> '0.4205',
            'Module::Build::Notes'  => '0.4205',
            'Module::Build::PPMMaker'=> '0.4205',
            'Module::Build::Platform::Default'=> '0.4205',
            'Module::Build::Platform::MacOS'=> '0.4205',
            'Module::Build::Platform::Unix'=> '0.4205',
            'Module::Build::Platform::VMS'=> '0.4205',
            'Module::Build::Platform::VOS'=> '0.4205',
            'Module::Build::Platform::Windows'=> '0.4205',
            'Module::Build::Platform::aix'=> '0.4205',
            'Module::Build::Platform::cygwin'=> '0.4205',
            'Module::Build::Platform::darwin'=> '0.4205',
            'Module::Build::Platform::os2'=> '0.4205',
            'Module::Build::PodParser'=> '0.4205',
            'Module::CoreList'      => '3.06',
            'Module::CoreList::TieHashDelta'=> '3.06',
            'Module::CoreList::Utils'=> '3.06',
            'Module::Load'          => '0.30',
            'Module::Load::Conditional'=> '0.62',
            'Net::Domain'           => '2.23',
            'Net::FTP'              => '2.79',
            'Net::NNTP'             => '2.26',
            'Net::POP3'             => '2.31',
            'Net::Ping'             => '2.43',
            'Net::SMTP'             => '2.33',
            'POSIX'                 => '1.38_02',
            'Parse::CPAN::Meta'     => '1.4413',
            'Pod::Escapes'          => '1.06',
            'Pod::Find'             => '1.62',
            'Pod::InputObjects'     => '1.62',
            'Pod::ParseUtils'       => '1.62',
            'Pod::Parser'           => '1.62',
            'Pod::Select'           => '1.62',
            'Scalar::Util'          => '1.38',
            'autodie'               => '2.23',
            'autodie::exception'    => '2.23',
            'autodie::exception::system'=> '2.23',
            'autodie::hints'        => '2.23',
            'autodie::skip'         => '2.23',
            'diagnostics'           => '1.34',
            'feature'               => '1.35',
            'inc::latest'           => '0.4205',
            'locale'                => '1.03',
            'mro'                   => '1.15',
            'threads'               => '1.92',
            'version'               => '0.9908',
            'version::regex'        => '0.9908',
            'version::vpp'          => '0.9908',
            'warnings'              => '1.22',
        },
        removed => {
        }
    },
    5.01901 => {
        delta_from => 5.019009,
        changed => {
            'App::Cpan'             => '1.62',
            'Attribute::Handlers'   => '0.96',
            'B::Deparse'            => '1.26',
            'CPAN'                  => '2.04',
            'CPAN::Bundle'          => '5.5001',
            'CPAN::Complete'        => '5.5001',
            'CPAN::Distribution'    => '2.01',
            'CPAN::Distroprefs'     => '6.0001',
            'CPAN::FirstTime'       => '5.5305',
            'CPAN::Meta'            => '2.140640',
            'CPAN::Meta::Converter' => '2.140640',
            'CPAN::Meta::Feature'   => '2.140640',
            'CPAN::Meta::History'   => '2.140640',
            'CPAN::Meta::Prereqs'   => '2.140640',
            'CPAN::Meta::Spec'      => '2.140640',
            'CPAN::Meta::Validator' => '2.140640',
            'CPAN::Meta::YAML'      => '0.012',
            'CPAN::Queue'           => '5.5002',
            'CPAN::Shell'           => '5.5003',
            'CPAN::Tarzip'          => '5.5012',
            'CPAN::Version'         => '5.5003',
            'Carp'                  => '1.33',
            'Carp::Heavy'           => '1.33',
            'Config'                => '5.019010',
            'Data::Dumper'          => '2.151',
            'Devel::PPPort'         => '3.22',
            'Digest::SHA'           => '5.88',
            'ExtUtils::Command::MM' => '6.92',
            'ExtUtils::Install'     => '1.63',
            'ExtUtils::Installed'   => '1.999005',
            'ExtUtils::Liblist'     => '6.92',
            'ExtUtils::Liblist::Kid'=> '6.92',
            'ExtUtils::MM'          => '6.92',
            'ExtUtils::MM_AIX'      => '6.92',
            'ExtUtils::MM_Any'      => '6.92',
            'ExtUtils::MM_BeOS'     => '6.92',
            'ExtUtils::MM_Cygwin'   => '6.92',
            'ExtUtils::MM_DOS'      => '6.92',
            'ExtUtils::MM_Darwin'   => '6.92',
            'ExtUtils::MM_MacOS'    => '6.92',
            'ExtUtils::MM_NW5'      => '6.92',
            'ExtUtils::MM_OS2'      => '6.92',
            'ExtUtils::MM_QNX'      => '6.92',
            'ExtUtils::MM_UWIN'     => '6.92',
            'ExtUtils::MM_Unix'     => '6.92',
            'ExtUtils::MM_VMS'      => '6.92',
            'ExtUtils::MM_VOS'      => '6.92',
            'ExtUtils::MM_Win32'    => '6.92',
            'ExtUtils::MM_Win95'    => '6.92',
            'ExtUtils::MY'          => '6.92',
            'ExtUtils::MakeMaker'   => '6.92',
            'ExtUtils::MakeMaker::Config'=> '6.92',
            'ExtUtils::Mkbootstrap' => '6.92',
            'ExtUtils::Mksymlists'  => '6.92',
            'ExtUtils::Packlist'    => '1.48',
            'ExtUtils::ParseXS'     => '3.24',
            'ExtUtils::ParseXS::Constants'=> '3.24',
            'ExtUtils::ParseXS::CountLines'=> '3.24',
            'ExtUtils::ParseXS::Eval'=> '3.24',
            'ExtUtils::ParseXS::Utilities'=> '3.24',
            'ExtUtils::Typemaps'    => '3.24',
            'ExtUtils::Typemaps::Cmd'=> '3.24',
            'ExtUtils::Typemaps::InputMap'=> '3.24',
            'ExtUtils::Typemaps::OutputMap'=> '3.24',
            'ExtUtils::Typemaps::Type'=> '3.24',
            'ExtUtils::testlib'     => '6.92',
            'File::Find'            => '1.27',
            'Filter::Simple'        => '0.91',
            'HTTP::Tiny'            => '0.043',
            'Hash::Util::FieldHash' => '1.15',
            'IO'                    => '1.31',
            'IO::Socket::IP'        => '0.29',
            'Locale::Codes'         => '3.30',
            'Locale::Codes::Constants'=> '3.30',
            'Locale::Codes::Country'=> '3.30',
            'Locale::Codes::Country_Codes'=> '3.30',
            'Locale::Codes::Country_Retired'=> '3.30',
            'Locale::Codes::Currency'=> '3.30',
            'Locale::Codes::Currency_Codes'=> '3.30',
            'Locale::Codes::Currency_Retired'=> '3.30',
            'Locale::Codes::LangExt'=> '3.30',
            'Locale::Codes::LangExt_Codes'=> '3.30',
            'Locale::Codes::LangExt_Retired'=> '3.30',
            'Locale::Codes::LangFam'=> '3.30',
            'Locale::Codes::LangFam_Codes'=> '3.30',
            'Locale::Codes::LangFam_Retired'=> '3.30',
            'Locale::Codes::LangVar'=> '3.30',
            'Locale::Codes::LangVar_Codes'=> '3.30',
            'Locale::Codes::LangVar_Retired'=> '3.30',
            'Locale::Codes::Language'=> '3.30',
            'Locale::Codes::Language_Codes'=> '3.30',
            'Locale::Codes::Language_Retired'=> '3.30',
            'Locale::Codes::Script' => '3.30',
            'Locale::Codes::Script_Codes'=> '3.30',
            'Locale::Codes::Script_Retired'=> '3.30',
            'Locale::Country'       => '3.30',
            'Locale::Currency'      => '3.30',
            'Locale::Language'      => '3.30',
            'Locale::Script'        => '3.30',
            'Module::CoreList'      => '3.09',
            'Module::CoreList::TieHashDelta'=> '3.09',
            'Module::CoreList::Utils'=> '3.09',
            'Module::Load'          => '0.32',
            'POSIX'                 => '1.38_03',
            'Parse::CPAN::Meta'     => '1.4414',
            'Pod::Perldoc'          => '3.23',
            'Pod::Perldoc::BaseTo'  => '3.23',
            'Pod::Perldoc::GetOptsOO'=> '3.23',
            'Pod::Perldoc::ToANSI'  => '3.23',
            'Pod::Perldoc::ToChecker'=> '3.23',
            'Pod::Perldoc::ToMan'   => '3.23',
            'Pod::Perldoc::ToNroff' => '3.23',
            'Pod::Perldoc::ToPod'   => '3.23',
            'Pod::Perldoc::ToRtf'   => '3.23',
            'Pod::Perldoc::ToTerm'  => '3.23',
            'Pod::Perldoc::ToText'  => '3.23',
            'Pod::Perldoc::ToTk'    => '3.23',
            'Pod::Perldoc::ToXml'   => '3.23',
            'Thread::Queue'         => '3.05',
            'XS::APItest'           => '0.60',
            'XS::Typemap'           => '0.13',
            'autouse'               => '1.08',
            'base'                  => '2.22',
            'charnames'             => '1.40',
            'feature'               => '1.36',
            'mro'                   => '1.16',
            'threads'               => '1.93',
            'warnings'              => '1.23',
            'warnings::register'    => '1.03',
        },
        removed => {
        }
    },
    5.019011 => {
        delta_from => 5.01901,
        changed => {
            'CPAN'                  => '2.05',
            'CPAN::Distribution'    => '2.02',
            'CPAN::FirstTime'       => '5.5306',
            'CPAN::Shell'           => '5.5004',
            'Carp'                  => '1.3301',
            'Carp::Heavy'           => '1.3301',
            'Config'                => '5.019011',
            'ExtUtils::Command::MM' => '6.94',
            'ExtUtils::Install'     => '1.67',
            'ExtUtils::Liblist'     => '6.94',
            'ExtUtils::Liblist::Kid'=> '6.94',
            'ExtUtils::MM'          => '6.94',
            'ExtUtils::MM_AIX'      => '6.94',
            'ExtUtils::MM_Any'      => '6.94',
            'ExtUtils::MM_BeOS'     => '6.94',
            'ExtUtils::MM_Cygwin'   => '6.94',
            'ExtUtils::MM_DOS'      => '6.94',
            'ExtUtils::MM_Darwin'   => '6.94',
            'ExtUtils::MM_MacOS'    => '6.94',
            'ExtUtils::MM_NW5'      => '6.94',
            'ExtUtils::MM_OS2'      => '6.94',
            'ExtUtils::MM_QNX'      => '6.94',
            'ExtUtils::MM_UWIN'     => '6.94',
            'ExtUtils::MM_Unix'     => '6.94',
            'ExtUtils::MM_VMS'      => '6.94',
            'ExtUtils::MM_VOS'      => '6.94',
            'ExtUtils::MM_Win32'    => '6.94',
            'ExtUtils::MM_Win95'    => '6.94',
            'ExtUtils::MY'          => '6.94',
            'ExtUtils::MakeMaker'   => '6.94',
            'ExtUtils::MakeMaker::Config'=> '6.94',
            'ExtUtils::Mkbootstrap' => '6.94',
            'ExtUtils::Mksymlists'  => '6.94',
            'ExtUtils::testlib'     => '6.94',
            'Module::CoreList'      => '3.10',
            'Module::CoreList::TieHashDelta'=> '3.10',
            'Module::CoreList::Utils'=> '3.10',
            'PerlIO'                => '1.09',
            'Storable'              => '2.49',
            'Win32'                 => '0.49',
            'experimental'          => '0.007',
        },
        removed => {
        }
    },
    5.020000 => {
        delta_from => 5.019011,
        changed => {
            'Config'                => '5.02',
            'Devel::PPPort'         => '3.21',
            'Encode'                => '2.60',
            'Errno'                 => '1.20_03',
            'ExtUtils::Command::MM' => '6.98',
            'ExtUtils::Liblist'     => '6.98',
            'ExtUtils::Liblist::Kid'=> '6.98',
            'ExtUtils::MM'          => '6.98',
            'ExtUtils::MM_AIX'      => '6.98',
            'ExtUtils::MM_Any'      => '6.98',
            'ExtUtils::MM_BeOS'     => '6.98',
            'ExtUtils::MM_Cygwin'   => '6.98',
            'ExtUtils::MM_DOS'      => '6.98',
            'ExtUtils::MM_Darwin'   => '6.98',
            'ExtUtils::MM_MacOS'    => '6.98',
            'ExtUtils::MM_NW5'      => '6.98',
            'ExtUtils::MM_OS2'      => '6.98',
            'ExtUtils::MM_QNX'      => '6.98',
            'ExtUtils::MM_UWIN'     => '6.98',
            'ExtUtils::MM_Unix'     => '6.98',
            'ExtUtils::MM_VMS'      => '6.98',
            'ExtUtils::MM_VOS'      => '6.98',
            'ExtUtils::MM_Win32'    => '6.98',
            'ExtUtils::MM_Win95'    => '6.98',
            'ExtUtils::MY'          => '6.98',
            'ExtUtils::MakeMaker'   => '6.98',
            'ExtUtils::MakeMaker::Config'=> '6.98',
            'ExtUtils::Miniperl'    => '1.01',
            'ExtUtils::Mkbootstrap' => '6.98',
            'ExtUtils::Mksymlists'  => '6.98',
            'ExtUtils::testlib'     => '6.98',
        },
        removed => {
        }
    },
    5.021000 => {
        delta_from => 5.020000,
        changed => {
            'Module::CoreList'      => '5.021001',
            'Module::CoreList::TieHashDelta'=> '5.021001',
            'Module::CoreList::Utils'=> '5.021001',
            'feature'               => '1.37',
        },
        removed => {
            'CGI'                   => 1,
            'CGI::Apache'           => 1,
            'CGI::Carp'             => 1,
            'CGI::Cookie'           => 1,
            'CGI::Fast'             => 1,
            'CGI::Pretty'           => 1,
            'CGI::Push'             => 1,
            'CGI::Switch'           => 1,
            'CGI::Util'             => 1,
            'Module::Build'         => 1,
            'Module::Build::Base'   => 1,
            'Module::Build::Compat' => 1,
            'Module::Build::Config' => 1,
            'Module::Build::ConfigData'=> 1,
            'Module::Build::Cookbook'=> 1,
            'Module::Build::Dumper' => 1,
            'Module::Build::ModuleInfo'=> 1,
            'Module::Build::Notes'  => 1,
            'Module::Build::PPMMaker'=> 1,
            'Module::Build::Platform::Default'=> 1,
            'Module::Build::Platform::MacOS'=> 1,
            'Module::Build::Platform::Unix'=> 1,
            'Module::Build::Platform::VMS'=> 1,
            'Module::Build::Platform::VOS'=> 1,
            'Module::Build::Platform::Windows'=> 1,
            'Module::Build::Platform::aix'=> 1,
            'Module::Build::Platform::cygwin'=> 1,
            'Module::Build::Platform::darwin'=> 1,
            'Module::Build::Platform::os2'=> 1,
            'Module::Build::PodParser'=> 1,
            'Module::Build::Version'=> 1,
            'Module::Build::YAML'   => 1,
            'Package::Constants'    => 1,
            'inc::latest'           => 1,
        }
    },
    5.021001 => {
        delta_from => 5.021000,
        changed => {
            'App::Prove'            => '3.32',
            'App::Prove::State'     => '3.32',
            'App::Prove::State::Result'=> '3.32',
            'App::Prove::State::Result::Test'=> '3.32',
            'Archive::Tar'          => '2.00',
            'Archive::Tar::Constant'=> '2.00',
            'Archive::Tar::File'    => '2.00',
            'B'                     => '1.49',
            'B::Deparse'            => '1.27',
            'Benchmark'             => '1.19',
            'CPAN::Meta'            => '2.141520',
            'CPAN::Meta::Converter' => '2.141520',
            'CPAN::Meta::Feature'   => '2.141520',
            'CPAN::Meta::History'   => '2.141520',
            'CPAN::Meta::Prereqs'   => '2.141520',
            'CPAN::Meta::Spec'      => '2.141520',
            'CPAN::Meta::Validator' => '2.141520',
            'Carp'                  => '1.34',
            'Carp::Heavy'           => '1.34',
            'Config'                => '5.021001',
            'Cwd'                   => '3.48',
            'Data::Dumper'          => '2.152',
            'Devel::PPPort'         => '3.24',
            'Devel::Peek'           => '1.17',
            'Digest::SHA'           => '5.92',
            'DynaLoader'            => '1.26',
            'Encode'                => '2.62',
            'Errno'                 => '1.20_04',
            'Exporter'              => '5.71',
            'Exporter::Heavy'       => '5.71',
            'ExtUtils::Install'     => '1.68',
            'ExtUtils::Miniperl'    => '1.02',
            'ExtUtils::ParseXS'     => '3.25',
            'ExtUtils::ParseXS::Constants'=> '3.25',
            'ExtUtils::ParseXS::CountLines'=> '3.25',
            'ExtUtils::ParseXS::Eval'=> '3.25',
            'ExtUtils::ParseXS::Utilities'=> '3.25',
            'ExtUtils::Typemaps'    => '3.25',
            'ExtUtils::Typemaps::Cmd'=> '3.25',
            'ExtUtils::Typemaps::InputMap'=> '3.25',
            'ExtUtils::Typemaps::OutputMap'=> '3.25',
            'ExtUtils::Typemaps::Type'=> '3.25',
            'Fatal'                 => '2.25',
            'File::Spec'            => '3.48',
            'File::Spec::Cygwin'    => '3.48',
            'File::Spec::Epoc'      => '3.48',
            'File::Spec::Functions' => '3.48',
            'File::Spec::Mac'       => '3.48',
            'File::Spec::OS2'       => '3.48',
            'File::Spec::Unix'      => '3.48',
            'File::Spec::VMS'       => '3.48',
            'File::Spec::Win32'     => '3.48',
            'Hash::Util'            => '0.17',
            'IO'                    => '1.32',
            'List::Util'            => '1.39',
            'List::Util::XS'        => '1.39',
            'Locale::Codes'         => '3.31',
            'Locale::Codes::Constants'=> '3.31',
            'Locale::Codes::Country'=> '3.31',
            'Locale::Codes::Country_Codes'=> '3.31',
            'Locale::Codes::Country_Retired'=> '3.31',
            'Locale::Codes::Currency'=> '3.31',
            'Locale::Codes::Currency_Codes'=> '3.31',
            'Locale::Codes::Currency_Retired'=> '3.31',
            'Locale::Codes::LangExt'=> '3.31',
            'Locale::Codes::LangExt_Codes'=> '3.31',
            'Locale::Codes::LangExt_Retired'=> '3.31',
            'Locale::Codes::LangFam'=> '3.31',
            'Locale::Codes::LangFam_Codes'=> '3.31',
            'Locale::Codes::LangFam_Retired'=> '3.31',
            'Locale::Codes::LangVar'=> '3.31',
            'Locale::Codes::LangVar_Codes'=> '3.31',
            'Locale::Codes::LangVar_Retired'=> '3.31',
            'Locale::Codes::Language'=> '3.31',
            'Locale::Codes::Language_Codes'=> '3.31',
            'Locale::Codes::Language_Retired'=> '3.31',
            'Locale::Codes::Script' => '3.31',
            'Locale::Codes::Script_Codes'=> '3.31',
            'Locale::Codes::Script_Retired'=> '3.31',
            'Locale::Country'       => '3.31',
            'Locale::Currency'      => '3.31',
            'Locale::Language'      => '3.31',
            'Locale::Script'        => '3.31',
            'Math::BigFloat'        => '1.9994',
            'Math::BigInt'          => '1.9995',
            'Math::BigInt::Calc'    => '1.9994',
            'Math::BigInt::CalcEmu' => '1.9994',
            'Math::BigRat'          => '0.2608',
            'Module::CoreList'      => '5.021001_01',
            'Module::CoreList::TieHashDelta'=> '5.021001_01',
            'Module::CoreList::Utils'=> '5.021001_01',
            'Module::Metadata'      => '1.000024',
            'NDBM_File'             => '1.13',
            'Net::Config'           => '1.14',
            'Net::SMTP'             => '2.34',
            'Net::Time'             => '2.11',
            'OS2::Process'          => '1.10',
            'POSIX'                 => '1.40',
            'PerlIO::encoding'      => '0.19',
            'PerlIO::mmap'          => '0.013',
            'PerlIO::scalar'        => '0.19',
            'PerlIO::via'           => '0.15',
            'Pod::Html'             => '1.22',
            'Scalar::Util'          => '1.39',
            'SelfLoader'            => '1.22',
            'Socket'                => '2.014',
            'Storable'              => '2.51',
            'TAP::Base'             => '3.32',
            'TAP::Formatter::Base'  => '3.32',
            'TAP::Formatter::Color' => '3.32',
            'TAP::Formatter::Console'=> '3.32',
            'TAP::Formatter::Console::ParallelSession'=> '3.32',
            'TAP::Formatter::Console::Session'=> '3.32',
            'TAP::Formatter::File'  => '3.32',
            'TAP::Formatter::File::Session'=> '3.32',
            'TAP::Formatter::Session'=> '3.32',
            'TAP::Harness'          => '3.32',
            'TAP::Harness::Env'     => '3.32',
            'TAP::Object'           => '3.32',
            'TAP::Parser'           => '3.32',
            'TAP::Parser::Aggregator'=> '3.32',
            'TAP::Parser::Grammar'  => '3.32',
            'TAP::Parser::Iterator' => '3.32',
            'TAP::Parser::Iterator::Array'=> '3.32',
            'TAP::Parser::Iterator::Process'=> '3.32',
            'TAP::Parser::Iterator::Stream'=> '3.32',
            'TAP::Parser::IteratorFactory'=> '3.32',
            'TAP::Parser::Multiplexer'=> '3.32',
            'TAP::Parser::Result'   => '3.32',
            'TAP::Parser::Result::Bailout'=> '3.32',
            'TAP::Parser::Result::Comment'=> '3.32',
            'TAP::Parser::Result::Plan'=> '3.32',
            'TAP::Parser::Result::Pragma'=> '3.32',
            'TAP::Parser::Result::Test'=> '3.32',
            'TAP::Parser::Result::Unknown'=> '3.32',
            'TAP::Parser::Result::Version'=> '3.32',
            'TAP::Parser::Result::YAML'=> '3.32',
            'TAP::Parser::ResultFactory'=> '3.32',
            'TAP::Parser::Scheduler'=> '3.32',
            'TAP::Parser::Scheduler::Job'=> '3.32',
            'TAP::Parser::Scheduler::Spinner'=> '3.32',
            'TAP::Parser::Source'   => '3.32',
            'TAP::Parser::SourceHandler'=> '3.32',
            'TAP::Parser::SourceHandler::Executable'=> '3.32',
            'TAP::Parser::SourceHandler::File'=> '3.32',
            'TAP::Parser::SourceHandler::Handle'=> '3.32',
            'TAP::Parser::SourceHandler::Perl'=> '3.32',
            'TAP::Parser::SourceHandler::RawTAP'=> '3.32',
            'TAP::Parser::YAMLish::Reader'=> '3.32',
            'TAP::Parser::YAMLish::Writer'=> '3.32',
            'Term::ANSIColor'       => '4.03',
            'Test::Builder'         => '1.001003',
            'Test::Builder::Module' => '1.001003',
            'Test::Builder::Tester' => '1.23_003',
            'Test::Harness'         => '3.32',
            'Test::More'            => '1.001003',
            'Test::Simple'          => '1.001003',
            'Tie::File'             => '1.01',
            'Unicode'               => '7.0.0',
            'Unicode::Collate'      => '1.07',
            'Unicode::Normalize'    => '1.18',
            'Unicode::UCD'          => '0.58',
            'XS::APItest'           => '0.61',
            '_charnames'            => '1.41',
            'autodie'               => '2.25',
            'autodie::Scope::Guard' => '2.25',
            'autodie::Scope::GuardStack'=> '2.25',
            'autodie::ScopeUtil'    => '2.25',
            'autodie::exception'    => '2.25',
            'autodie::exception::system'=> '2.25',
            'autodie::hints'        => '2.25',
            'autodie::skip'         => '2.25',
            'charnames'             => '1.41',
            'locale'                => '1.04',
            'threads'               => '1.94',
            'utf8'                  => '1.14',
            'warnings'              => '1.24',
        },
        removed => {
        }
    },
    5.021002 => {
        delta_from => 5.021001,
        changed => {
            'B'                     => '1.50',
            'Config'                => '5.021002',
            'Cwd'                   => '3.49',
            'Devel::Peek'           => '1.18',
            'ExtUtils::Manifest'    => '1.64',
            'File::Copy'            => '2.30',
            'File::Spec'            => '3.49',
            'File::Spec::Cygwin'    => '3.49',
            'File::Spec::Epoc'      => '3.49',
            'File::Spec::Functions' => '3.49',
            'File::Spec::Mac'       => '3.49',
            'File::Spec::OS2'       => '3.49',
            'File::Spec::Unix'      => '3.49',
            'File::Spec::VMS'       => '3.49',
            'File::Spec::Win32'     => '3.49',
            'Filter::Simple'        => '0.92',
            'Hash::Util'            => '0.18',
            'IO'                    => '1.33',
            'IO::Socket::IP'        => '0.31',
            'IPC::Open3'            => '1.17',
            'Math::BigFloat'        => '1.9996',
            'Math::BigInt'          => '1.9996',
            'Math::BigInt::Calc'    => '1.9996',
            'Math::BigInt::CalcEmu' => '1.9996',
            'Module::CoreList'      => '5.021002',
            'Module::CoreList::TieHashDelta'=> '5.021002',
            'Module::CoreList::Utils'=> '5.021002',
            'POSIX'                 => '1.41',
            'Pod::Usage'            => '1.64',
            'XS::APItest'           => '0.62',
            'arybase'               => '0.08',
            'experimental'          => '0.008',
            'threads'               => '1.95',
            'warnings'              => '1.26',
        },
        removed => {
        }
    },
    5.021003 => {
        delta_from => 5.021002,
        changed => {
            'B::Debug'              => '1.21',
            'CPAN::Meta'            => '2.142060',
            'CPAN::Meta::Converter' => '2.142060',
            'CPAN::Meta::Feature'   => '2.142060',
            'CPAN::Meta::History'   => '2.142060',
            'CPAN::Meta::Merge'     => '2.142060',
            'CPAN::Meta::Prereqs'   => '2.142060',
            'CPAN::Meta::Requirements'=> '2.126',
            'CPAN::Meta::Spec'      => '2.142060',
            'CPAN::Meta::Validator' => '2.142060',
            'Config'                => '5.021003',
            'Config::Perl::V'       => '0.22',
            'ExtUtils::CBuilder'    => '0.280217',
            'ExtUtils::CBuilder::Base'=> '0.280217',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.280217',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.280217',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.280217',
            'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280217',
            'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280217',
            'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280217',
            'ExtUtils::CBuilder::Platform::aix'=> '0.280217',
            'ExtUtils::CBuilder::Platform::android'=> '0.280217',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.280217',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.280217',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280217',
            'ExtUtils::CBuilder::Platform::os2'=> '0.280217',
            'ExtUtils::Manifest'    => '1.65',
            'HTTP::Tiny'            => '0.047',
            'IPC::Open3'            => '1.18',
            'Module::CoreList'      => '5.021003',
            'Module::CoreList::TieHashDelta'=> '5.021003',
            'Module::CoreList::Utils'=> '5.021003',
            'Opcode'                => '1.28',
            'POSIX'                 => '1.42',
            'Safe'                  => '2.38',
            'Socket'                => '2.015',
            'Sys::Hostname'         => '1.19',
            'UNIVERSAL'             => '1.12',
            'XS::APItest'           => '0.63',
            'perlfaq'               => '5.0150045',
        },
        removed => {
        }
    },
    5.020001 => {
        delta_from => 5.020000,
        changed => {
            'Config'                => '5.020001',
            'Config::Perl::V'       => '0.22',
            'Cwd'                   => '3.48',
            'Exporter'              => '5.71',
            'Exporter::Heavy'       => '5.71',
            'ExtUtils::CBuilder'    => '0.280217',
            'ExtUtils::CBuilder::Base'=> '0.280217',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.280217',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.280217',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.280217',
            'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280217',
            'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280217',
            'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280217',
            'ExtUtils::CBuilder::Platform::aix'=> '0.280217',
            'ExtUtils::CBuilder::Platform::android'=> '0.280217',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.280217',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.280217',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280217',
            'ExtUtils::CBuilder::Platform::os2'=> '0.280217',
            'File::Copy'            => '2.30',
            'File::Spec'            => '3.48',
            'File::Spec::Cygwin'    => '3.48',
            'File::Spec::Epoc'      => '3.48',
            'File::Spec::Functions' => '3.48',
            'File::Spec::Mac'       => '3.48',
            'File::Spec::OS2'       => '3.48',
            'File::Spec::Unix'      => '3.48',
            'File::Spec::VMS'       => '3.48',
            'File::Spec::Win32'     => '3.48',
            'Module::CoreList'      => '5.020001',
            'Module::CoreList::TieHashDelta'=> '5.020001',
            'Module::CoreList::Utils'=> '5.020001',
            'PerlIO::via'           => '0.15',
            'Unicode::UCD'          => '0.58',
            'XS::APItest'           => '0.60_01',
            'utf8'                  => '1.13_01',
            'version'               => '0.9909',
            'version::regex'        => '0.9909',
            'version::vpp'          => '0.9909',
        },
        removed => {
        }
    },
    5.021004 => {
        delta_from => 5.021003,
        changed => {
            'App::Prove'            => '3.33',
            'App::Prove::State'     => '3.33',
            'App::Prove::State::Result'=> '3.33',
            'App::Prove::State::Result::Test'=> '3.33',
            'Archive::Tar'          => '2.02',
            'Archive::Tar::Constant'=> '2.02',
            'Archive::Tar::File'    => '2.02',
            'Attribute::Handlers'   => '0.97',
            'B'                     => '1.51',
            'B::Concise'            => '0.993',
            'B::Deparse'            => '1.28',
            'B::Op_private'         => '5.021004',
            'CPAN::Meta::Requirements'=> '2.128',
            'Config'                => '5.021004',
            'Cwd'                   => '3.50',
            'Data::Dumper'          => '2.154',
            'ExtUtils::CBuilder'    => '0.280219',
            'ExtUtils::CBuilder::Base'=> '0.280219',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.280219',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.280219',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.280219',
            'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280219',
            'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280219',
            'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280219',
            'ExtUtils::CBuilder::Platform::aix'=> '0.280219',
            'ExtUtils::CBuilder::Platform::android'=> '0.280219',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.280219',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.280219',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280219',
            'ExtUtils::CBuilder::Platform::os2'=> '0.280219',
            'ExtUtils::Install'     => '2.04',
            'ExtUtils::Installed'   => '2.04',
            'ExtUtils::Liblist::Kid'=> '6.98_01',
            'ExtUtils::Manifest'    => '1.68',
            'ExtUtils::Packlist'    => '2.04',
            'File::Find'            => '1.28',
            'File::Spec'            => '3.50',
            'File::Spec::Cygwin'    => '3.50',
            'File::Spec::Epoc'      => '3.50',
            'File::Spec::Functions' => '3.50',
            'File::Spec::Mac'       => '3.50',
            'File::Spec::OS2'       => '3.50',
            'File::Spec::Unix'      => '3.50',
            'File::Spec::VMS'       => '3.50',
            'File::Spec::Win32'     => '3.50',
            'Getopt::Std'           => '1.11',
            'HTTP::Tiny'            => '0.049',
            'IO'                    => '1.34',
            'IO::Socket::IP'        => '0.32',
            'List::Util'            => '1.41',
            'List::Util::XS'        => '1.41',
            'Locale::Codes'         => '3.32',
            'Locale::Codes::Constants'=> '3.32',
            'Locale::Codes::Country'=> '3.32',
            'Locale::Codes::Country_Codes'=> '3.32',
            'Locale::Codes::Country_Retired'=> '3.32',
            'Locale::Codes::Currency'=> '3.32',
            'Locale::Codes::Currency_Codes'=> '3.32',
            'Locale::Codes::Currency_Retired'=> '3.32',
            'Locale::Codes::LangExt'=> '3.32',
            'Locale::Codes::LangExt_Codes'=> '3.32',
            'Locale::Codes::LangExt_Retired'=> '3.32',
            'Locale::Codes::LangFam'=> '3.32',
            'Locale::Codes::LangFam_Codes'=> '3.32',
            'Locale::Codes::LangFam_Retired'=> '3.32',
            'Locale::Codes::LangVar'=> '3.32',
            'Locale::Codes::LangVar_Codes'=> '3.32',
            'Locale::Codes::LangVar_Retired'=> '3.32',
            'Locale::Codes::Language'=> '3.32',
            'Locale::Codes::Language_Codes'=> '3.32',
            'Locale::Codes::Language_Retired'=> '3.32',
            'Locale::Codes::Script' => '3.32',
            'Locale::Codes::Script_Codes'=> '3.32',
            'Locale::Codes::Script_Retired'=> '3.32',
            'Locale::Country'       => '3.32',
            'Locale::Currency'      => '3.32',
            'Locale::Language'      => '3.32',
            'Locale::Script'        => '3.32',
            'Math::BigFloat'        => '1.9997',
            'Math::BigInt'          => '1.9997',
            'Math::BigInt::Calc'    => '1.9997',
            'Math::BigInt::CalcEmu' => '1.9997',
            'Module::CoreList'      => '5.20140920',
            'Module::CoreList::TieHashDelta'=> '5.20140920',
            'Module::CoreList::Utils'=> '5.20140920',
            'POSIX'                 => '1.43',
            'Pod::Perldoc'          => '3.24',
            'Pod::Perldoc::BaseTo'  => '3.24',
            'Pod::Perldoc::GetOptsOO'=> '3.24',
            'Pod::Perldoc::ToANSI'  => '3.24',
            'Pod::Perldoc::ToChecker'=> '3.24',
            'Pod::Perldoc::ToMan'   => '3.24',
            'Pod::Perldoc::ToNroff' => '3.24',
            'Pod::Perldoc::ToPod'   => '3.24',
            'Pod::Perldoc::ToRtf'   => '3.24',
            'Pod::Perldoc::ToTerm'  => '3.24',
            'Pod::Perldoc::ToText'  => '3.24',
            'Pod::Perldoc::ToTk'    => '3.24',
            'Pod::Perldoc::ToXml'   => '3.24',
            'Scalar::Util'          => '1.41',
            'Sub::Util'             => '1.41',
            'TAP::Base'             => '3.33',
            'TAP::Formatter::Base'  => '3.33',
            'TAP::Formatter::Color' => '3.33',
            'TAP::Formatter::Console'=> '3.33',
            'TAP::Formatter::Console::ParallelSession'=> '3.33',
            'TAP::Formatter::Console::Session'=> '3.33',
            'TAP::Formatter::File'  => '3.33',
            'TAP::Formatter::File::Session'=> '3.33',
            'TAP::Formatter::Session'=> '3.33',
            'TAP::Harness'          => '3.33',
            'TAP::Harness::Env'     => '3.33',
            'TAP::Object'           => '3.33',
            'TAP::Parser'           => '3.33',
            'TAP::Parser::Aggregator'=> '3.33',
            'TAP::Parser::Grammar'  => '3.33',
            'TAP::Parser::Iterator' => '3.33',
            'TAP::Parser::Iterator::Array'=> '3.33',
            'TAP::Parser::Iterator::Process'=> '3.33',
            'TAP::Parser::Iterator::Stream'=> '3.33',
            'TAP::Parser::IteratorFactory'=> '3.33',
            'TAP::Parser::Multiplexer'=> '3.33',
            'TAP::Parser::Result'   => '3.33',
            'TAP::Parser::Result::Bailout'=> '3.33',
            'TAP::Parser::Result::Comment'=> '3.33',
            'TAP::Parser::Result::Plan'=> '3.33',
            'TAP::Parser::Result::Pragma'=> '3.33',
            'TAP::Parser::Result::Test'=> '3.33',
            'TAP::Parser::Result::Unknown'=> '3.33',
            'TAP::Parser::Result::Version'=> '3.33',
            'TAP::Parser::Result::YAML'=> '3.33',
            'TAP::Parser::ResultFactory'=> '3.33',
            'TAP::Parser::Scheduler'=> '3.33',
            'TAP::Parser::Scheduler::Job'=> '3.33',
            'TAP::Parser::Scheduler::Spinner'=> '3.33',
            'TAP::Parser::Source'   => '3.33',
            'TAP::Parser::SourceHandler'=> '3.33',
            'TAP::Parser::SourceHandler::Executable'=> '3.33',
            'TAP::Parser::SourceHandler::File'=> '3.33',
            'TAP::Parser::SourceHandler::Handle'=> '3.33',
            'TAP::Parser::SourceHandler::Perl'=> '3.33',
            'TAP::Parser::SourceHandler::RawTAP'=> '3.33',
            'TAP::Parser::YAMLish::Reader'=> '3.33',
            'TAP::Parser::YAMLish::Writer'=> '3.33',
            'Term::ReadLine'        => '1.15',
            'Test::Builder'         => '1.001006',
            'Test::Builder::Module' => '1.001006',
            'Test::Builder::Tester' => '1.24',
            'Test::Builder::Tester::Color'=> '1.24',
            'Test::Harness'         => '3.33',
            'Test::More'            => '1.001006',
            'Test::Simple'          => '1.001006',
            'Time::Piece'           => '1.29',
            'Time::Seconds'         => '1.29',
            'XS::APItest'           => '0.64',
            '_charnames'            => '1.42',
            'attributes'            => '0.23',
            'bigint'                => '0.37',
            'bignum'                => '0.38',
            'bigrat'                => '0.37',
            'constant'              => '1.32',
            'experimental'          => '0.010',
            'overload'              => '1.23',
            'threads'               => '1.96',
            'version'               => '0.9909',
            'version::regex'        => '0.9909',
            'version::vpp'          => '0.9909',
        },
        removed => {
        }
    },
    5.021005 => {
        delta_from => 5.021004,
        changed => {
            'B'                     => '1.52',
            'B::Concise'            => '0.994',
            'B::Debug'              => '1.22',
            'B::Deparse'            => '1.29',
            'B::Op_private'         => '5.021005',
            'CPAN::Meta'            => '2.142690',
            'CPAN::Meta::Converter' => '2.142690',
            'CPAN::Meta::Feature'   => '2.142690',
            'CPAN::Meta::History'   => '2.142690',
            'CPAN::Meta::Merge'     => '2.142690',
            'CPAN::Meta::Prereqs'   => '2.142690',
            'CPAN::Meta::Spec'      => '2.142690',
            'CPAN::Meta::Validator' => '2.142690',
            'Compress::Raw::Bzip2'  => '2.066',
            'Compress::Raw::Zlib'   => '2.066',
            'Compress::Zlib'        => '2.066',
            'Config'                => '5.021005',
            'Cwd'                   => '3.51',
            'DynaLoader'            => '1.27',
            'Errno'                 => '1.21',
            'ExtUtils::CBuilder'    => '0.280220',
            'ExtUtils::CBuilder::Base'=> '0.280220',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.280220',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.280220',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.280220',
            'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280220',
            'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280220',
            'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280220',
            'ExtUtils::CBuilder::Platform::aix'=> '0.280220',
            'ExtUtils::CBuilder::Platform::android'=> '0.280220',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.280220',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.280220',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280220',
            'ExtUtils::CBuilder::Platform::os2'=> '0.280220',
            'ExtUtils::Miniperl'    => '1.03',
            'Fcntl'                 => '1.13',
            'File::Find'            => '1.29',
            'File::Spec'            => '3.51',
            'File::Spec::Cygwin'    => '3.51',
            'File::Spec::Epoc'      => '3.51',
            'File::Spec::Functions' => '3.51',
            'File::Spec::Mac'       => '3.51',
            'File::Spec::OS2'       => '3.51',
            'File::Spec::Unix'      => '3.51',
            'File::Spec::VMS'       => '3.51',
            'File::Spec::Win32'     => '3.51',
            'HTTP::Tiny'            => '0.050',
            'IO::Compress::Adapter::Bzip2'=> '2.066',
            'IO::Compress::Adapter::Deflate'=> '2.066',
            'IO::Compress::Adapter::Identity'=> '2.066',
            'IO::Compress::Base'    => '2.066',
            'IO::Compress::Base::Common'=> '2.066',
            'IO::Compress::Bzip2'   => '2.066',
            'IO::Compress::Deflate' => '2.066',
            'IO::Compress::Gzip'    => '2.066',
            'IO::Compress::Gzip::Constants'=> '2.066',
            'IO::Compress::RawDeflate'=> '2.066',
            'IO::Compress::Zip'     => '2.066',
            'IO::Compress::Zip::Constants'=> '2.066',
            'IO::Compress::Zlib::Constants'=> '2.066',
            'IO::Compress::Zlib::Extra'=> '2.066',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.066',
            'IO::Uncompress::Adapter::Identity'=> '2.066',
            'IO::Uncompress::Adapter::Inflate'=> '2.066',
            'IO::Uncompress::AnyInflate'=> '2.066',
            'IO::Uncompress::AnyUncompress'=> '2.066',
            'IO::Uncompress::Base'  => '2.066',
            'IO::Uncompress::Bunzip2'=> '2.066',
            'IO::Uncompress::Gunzip'=> '2.066',
            'IO::Uncompress::Inflate'=> '2.066',
            'IO::Uncompress::RawInflate'=> '2.066',
            'IO::Uncompress::Unzip' => '2.066',
            'JSON::PP'              => '2.27300',
            'Module::CoreList'      => '5.20141020',
            'Module::CoreList::TieHashDelta'=> '5.20141020',
            'Module::CoreList::Utils'=> '5.20141020',
            'Net::Cmd'              => '3.02',
            'Net::Config'           => '3.02',
            'Net::Domain'           => '3.02',
            'Net::FTP'              => '3.02',
            'Net::FTP::A'           => '3.02',
            'Net::FTP::E'           => '3.02',
            'Net::FTP::I'           => '3.02',
            'Net::FTP::L'           => '3.02',
            'Net::FTP::dataconn'    => '3.02',
            'Net::NNTP'             => '3.02',
            'Net::Netrc'            => '3.02',
            'Net::POP3'             => '3.02',
            'Net::SMTP'             => '3.02',
            'Net::Time'             => '3.02',
            'Opcode'                => '1.29',
            'POSIX'                 => '1.45',
            'Socket'                => '2.016',
            'Test::Builder'         => '1.001008',
            'Test::Builder::Module' => '1.001008',
            'Test::More'            => '1.001008',
            'Test::Simple'          => '1.001008',
            'XS::APItest'           => '0.65',
            'XSLoader'              => '0.18',
            'attributes'            => '0.24',
            'experimental'          => '0.012',
            'feature'               => '1.38',
            'perlfaq'               => '5.0150046',
            're'                    => '0.27',
            'threads::shared'       => '1.47',
            'warnings'              => '1.28',
            'warnings::register'    => '1.04',
        },
        removed => {
        }
    },
    5.021006 => {
        delta_from => 5.021005,
        changed => {
            'App::Prove'            => '3.34',
            'App::Prove::State'     => '3.34',
            'App::Prove::State::Result'=> '3.34',
            'App::Prove::State::Result::Test'=> '3.34',
            'B'                     => '1.53',
            'B::Concise'            => '0.995',
            'B::Deparse'            => '1.30',
            'B::Op_private'         => '5.021006',
            'CPAN::Meta'            => '2.143240',
            'CPAN::Meta::Converter' => '2.143240',
            'CPAN::Meta::Feature'   => '2.143240',
            'CPAN::Meta::History'   => '2.143240',
            'CPAN::Meta::Merge'     => '2.143240',
            'CPAN::Meta::Prereqs'   => '2.143240',
            'CPAN::Meta::Requirements'=> '2.130',
            'CPAN::Meta::Spec'      => '2.143240',
            'CPAN::Meta::Validator' => '2.143240',
            'Config'                => '5.021006',
            'Devel::Peek'           => '1.19',
            'Digest::SHA'           => '5.93',
            'DynaLoader'            => '1.28',
            'Encode'                => '2.64',
            'Exporter'              => '5.72',
            'Exporter::Heavy'       => '5.72',
            'ExtUtils::Command::MM' => '7.02',
            'ExtUtils::Liblist'     => '7.02',
            'ExtUtils::Liblist::Kid'=> '7.02',
            'ExtUtils::MM'          => '7.02',
            'ExtUtils::MM_AIX'      => '7.02',
            'ExtUtils::MM_Any'      => '7.02',
            'ExtUtils::MM_BeOS'     => '7.02',
            'ExtUtils::MM_Cygwin'   => '7.02',
            'ExtUtils::MM_DOS'      => '7.02',
            'ExtUtils::MM_Darwin'   => '7.02',
            'ExtUtils::MM_MacOS'    => '7.02',
            'ExtUtils::MM_NW5'      => '7.02',
            'ExtUtils::MM_OS2'      => '7.02',
            'ExtUtils::MM_QNX'      => '7.02',
            'ExtUtils::MM_UWIN'     => '7.02',
            'ExtUtils::MM_Unix'     => '7.02',
            'ExtUtils::MM_VMS'      => '7.02',
            'ExtUtils::MM_VOS'      => '7.02',
            'ExtUtils::MM_Win32'    => '7.02',
            'ExtUtils::MM_Win95'    => '7.02',
            'ExtUtils::MY'          => '7.02',
            'ExtUtils::MakeMaker'   => '7.02',
            'ExtUtils::MakeMaker::Config'=> '7.02',
            'ExtUtils::MakeMaker::Locale'=> '7.02',
            'ExtUtils::MakeMaker::version'=> '7.02',
            'ExtUtils::MakeMaker::version::regex'=> '7.02',
            'ExtUtils::MakeMaker::version::vpp'=> '7.02',
            'ExtUtils::Manifest'    => '1.69',
            'ExtUtils::Mkbootstrap' => '7.02',
            'ExtUtils::Mksymlists'  => '7.02',
            'ExtUtils::ParseXS'     => '3.26',
            'ExtUtils::ParseXS::Constants'=> '3.26',
            'ExtUtils::ParseXS::CountLines'=> '3.26',
            'ExtUtils::ParseXS::Eval'=> '3.26',
            'ExtUtils::ParseXS::Utilities'=> '3.26',
            'ExtUtils::testlib'     => '7.02',
            'File::Spec::VMS'       => '3.52',
            'HTTP::Tiny'            => '0.051',
            'I18N::Langinfo'        => '0.12',
            'IO::Socket'            => '1.38',
            'Module::CoreList'      => '5.20141120',
            'Module::CoreList::TieHashDelta'=> '5.20141120',
            'Module::CoreList::Utils'=> '5.20141120',
            'POSIX'                 => '1.46',
            'PerlIO::encoding'      => '0.20',
            'PerlIO::scalar'        => '0.20',
            'TAP::Base'             => '3.34',
            'TAP::Formatter::Base'  => '3.34',
            'TAP::Formatter::Color' => '3.34',
            'TAP::Formatter::Console'=> '3.34',
            'TAP::Formatter::Console::ParallelSession'=> '3.34',
            'TAP::Formatter::Console::Session'=> '3.34',
            'TAP::Formatter::File'  => '3.34',
            'TAP::Formatter::File::Session'=> '3.34',
            'TAP::Formatter::Session'=> '3.34',
            'TAP::Harness'          => '3.34',
            'TAP::Harness::Env'     => '3.34',
            'TAP::Object'           => '3.34',
            'TAP::Parser'           => '3.34',
            'TAP::Parser::Aggregator'=> '3.34',
            'TAP::Parser::Grammar'  => '3.34',
            'TAP::Parser::Iterator' => '3.34',
            'TAP::Parser::Iterator::Array'=> '3.34',
            'TAP::Parser::Iterator::Process'=> '3.34',
            'TAP::Parser::Iterator::Stream'=> '3.34',
            'TAP::Parser::IteratorFactory'=> '3.34',
            'TAP::Parser::Multiplexer'=> '3.34',
            'TAP::Parser::Result'   => '3.34',
            'TAP::Parser::Result::Bailout'=> '3.34',
            'TAP::Parser::Result::Comment'=> '3.34',
            'TAP::Parser::Result::Plan'=> '3.34',
            'TAP::Parser::Result::Pragma'=> '3.34',
            'TAP::Parser::Result::Test'=> '3.34',
            'TAP::Parser::Result::Unknown'=> '3.34',
            'TAP::Parser::Result::Version'=> '3.34',
            'TAP::Parser::Result::YAML'=> '3.34',
            'TAP::Parser::ResultFactory'=> '3.34',
            'TAP::Parser::Scheduler'=> '3.34',
            'TAP::Parser::Scheduler::Job'=> '3.34',
            'TAP::Parser::Scheduler::Spinner'=> '3.34',
            'TAP::Parser::Source'   => '3.34',
            'TAP::Parser::SourceHandler'=> '3.34',
            'TAP::Parser::SourceHandler::Executable'=> '3.34',
            'TAP::Parser::SourceHandler::File'=> '3.34',
            'TAP::Parser::SourceHandler::Handle'=> '3.34',
            'TAP::Parser::SourceHandler::Perl'=> '3.34',
            'TAP::Parser::SourceHandler::RawTAP'=> '3.34',
            'TAP::Parser::YAMLish::Reader'=> '3.34',
            'TAP::Parser::YAMLish::Writer'=> '3.34',
            'Test::Builder'         => '1.301001_075',
            'Test::Builder::Module' => '1.301001_075',
            'Test::Builder::Tester' => '1.301001_075',
            'Test::Builder::Tester::Color'=> '1.301001_075',
            'Test::Harness'         => '3.34',
            'Test::More'            => '1.301001_075',
            'Test::More::DeepCheck' => undef,
            'Test::More::DeepCheck::Strict'=> undef,
            'Test::More::DeepCheck::Tolerant'=> undef,
            'Test::More::Tools'     => undef,
            'Test::MostlyLike'      => undef,
            'Test::Simple'          => '1.301001_075',
            'Test::Stream'          => '1.301001_075',
            'Test::Stream::ArrayBase'=> undef,
            'Test::Stream::ArrayBase::Meta'=> undef,
            'Test::Stream::Carp'    => undef,
            'Test::Stream::Context' => undef,
            'Test::Stream::Event'   => undef,
            'Test::Stream::Event::Bail'=> undef,
            'Test::Stream::Event::Child'=> undef,
            'Test::Stream::Event::Diag'=> undef,
            'Test::Stream::Event::Finish'=> undef,
            'Test::Stream::Event::Note'=> undef,
            'Test::Stream::Event::Ok'=> undef,
            'Test::Stream::Event::Plan'=> undef,
            'Test::Stream::Event::Subtest'=> undef,
            'Test::Stream::ExitMagic'=> undef,
            'Test::Stream::ExitMagic::Context'=> undef,
            'Test::Stream::Exporter'=> undef,
            'Test::Stream::Exporter::Meta'=> undef,
            'Test::Stream::IOSets'  => undef,
            'Test::Stream::Meta'    => undef,
            'Test::Stream::PackageUtil'=> undef,
            'Test::Stream::Tester'  => undef,
            'Test::Stream::Tester::Checks'=> undef,
            'Test::Stream::Tester::Checks::Event'=> undef,
            'Test::Stream::Tester::Events'=> undef,
            'Test::Stream::Tester::Events::Event'=> undef,
            'Test::Stream::Tester::Grab'=> undef,
            'Test::Stream::Threads' => undef,
            'Test::Stream::Toolset' => undef,
            'Test::Stream::Util'    => undef,
            'Test::Tester'          => '1.301001_075',
            'Test::Tester::Capture' => undef,
            'Test::use::ok'         => '1.301001_075',
            'Unicode::UCD'          => '0.59',
            'XS::APItest'           => '0.68',
            'XSLoader'              => '0.19',
            'experimental'          => '0.013',
            'locale'                => '1.05',
            'ok'                    => '1.301001_075',
            'overload'              => '1.24',
            're'                    => '0.28',
            'warnings'              => '1.29',
        },
        removed => {
        }
    },
    5.021007 => {
        delta_from => 5.021006,
        changed => {
            'Archive::Tar'          => '2.04',
            'Archive::Tar::Constant'=> '2.04',
            'Archive::Tar::File'    => '2.04',
            'B'                     => '1.54',
            'B::Concise'            => '0.996',
            'B::Deparse'            => '1.31',
            'B::Op_private'         => '5.021007',
            'B::Showlex'            => '1.05',
            'Compress::Raw::Bzip2'  => '2.067',
            'Compress::Raw::Zlib'   => '2.067',
            'Compress::Zlib'        => '2.067',
            'Config'                => '5.021007',
            'Cwd'                   => '3.54',
            'DB_File'               => '1.834',
            'Data::Dumper'          => '2.155',
            'Devel::PPPort'         => '3.25',
            'Devel::Peek'           => '1.20',
            'DynaLoader'            => '1.29',
            'Encode'                => '2.67',
            'Errno'                 => '1.22',
            'ExtUtils::CBuilder'    => '0.280221',
            'ExtUtils::CBuilder::Base'=> '0.280221',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.280221',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.280221',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.280221',
            'ExtUtils::CBuilder::Platform::aix'=> '0.280221',
            'ExtUtils::CBuilder::Platform::android'=> '0.280221',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.280221',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.280221',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280221',
            'ExtUtils::CBuilder::Platform::os2'=> '0.280221',
            'ExtUtils::Command::MM' => '7.04',
            'ExtUtils::Liblist'     => '7.04',
            'ExtUtils::Liblist::Kid'=> '7.04',
            'ExtUtils::MM'          => '7.04',
            'ExtUtils::MM_AIX'      => '7.04',
            'ExtUtils::MM_Any'      => '7.04',
            'ExtUtils::MM_BeOS'     => '7.04',
            'ExtUtils::MM_Cygwin'   => '7.04',
            'ExtUtils::MM_DOS'      => '7.04',
            'ExtUtils::MM_Darwin'   => '7.04',
            'ExtUtils::MM_MacOS'    => '7.04',
            'ExtUtils::MM_NW5'      => '7.04',
            'ExtUtils::MM_OS2'      => '7.04',
            'ExtUtils::MM_QNX'      => '7.04',
            'ExtUtils::MM_UWIN'     => '7.04',
            'ExtUtils::MM_Unix'     => '7.04',
            'ExtUtils::MM_VMS'      => '7.04',
            'ExtUtils::MM_VOS'      => '7.04',
            'ExtUtils::MM_Win32'    => '7.04',
            'ExtUtils::MM_Win95'    => '7.04',
            'ExtUtils::MY'          => '7.04',
            'ExtUtils::MakeMaker'   => '7.04',
            'ExtUtils::MakeMaker::Config'=> '7.04',
            'ExtUtils::MakeMaker::Locale'=> '7.04',
            'ExtUtils::MakeMaker::version'=> '7.04',
            'ExtUtils::MakeMaker::version::regex'=> '7.04',
            'ExtUtils::MakeMaker::version::vpp'=> '7.04',
            'ExtUtils::Mkbootstrap' => '7.04',
            'ExtUtils::Mksymlists'  => '7.04',
            'ExtUtils::ParseXS'     => '3.27',
            'ExtUtils::ParseXS::Constants'=> '3.27',
            'ExtUtils::ParseXS::CountLines'=> '3.27',
            'ExtUtils::ParseXS::Eval'=> '3.27',
            'ExtUtils::ParseXS::Utilities'=> '3.27',
            'ExtUtils::testlib'     => '7.04',
            'File::Spec'            => '3.53',
            'File::Spec::Cygwin'    => '3.54',
            'File::Spec::Epoc'      => '3.54',
            'File::Spec::Functions' => '3.54',
            'File::Spec::Mac'       => '3.54',
            'File::Spec::OS2'       => '3.54',
            'File::Spec::Unix'      => '3.54',
            'File::Spec::VMS'       => '3.54',
            'File::Spec::Win32'     => '3.54',
            'Filter::Util::Call'    => '1.51',
            'HTTP::Tiny'            => '0.053',
            'IO'                    => '1.35',
            'IO::Compress::Adapter::Bzip2'=> '2.067',
            'IO::Compress::Adapter::Deflate'=> '2.067',
            'IO::Compress::Adapter::Identity'=> '2.067',
            'IO::Compress::Base'    => '2.067',
            'IO::Compress::Base::Common'=> '2.067',
            'IO::Compress::Bzip2'   => '2.067',
            'IO::Compress::Deflate' => '2.067',
            'IO::Compress::Gzip'    => '2.067',
            'IO::Compress::Gzip::Constants'=> '2.067',
            'IO::Compress::RawDeflate'=> '2.067',
            'IO::Compress::Zip'     => '2.067',
            'IO::Compress::Zip::Constants'=> '2.067',
            'IO::Compress::Zlib::Constants'=> '2.067',
            'IO::Compress::Zlib::Extra'=> '2.067',
            'IO::Socket::IP'        => '0.34',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.067',
            'IO::Uncompress::Adapter::Identity'=> '2.067',
            'IO::Uncompress::Adapter::Inflate'=> '2.067',
            'IO::Uncompress::AnyInflate'=> '2.067',
            'IO::Uncompress::AnyUncompress'=> '2.067',
            'IO::Uncompress::Base'  => '2.067',
            'IO::Uncompress::Bunzip2'=> '2.067',
            'IO::Uncompress::Gunzip'=> '2.067',
            'IO::Uncompress::Inflate'=> '2.067',
            'IO::Uncompress::RawInflate'=> '2.067',
            'IO::Uncompress::Unzip' => '2.067',
            'Locale::Codes'         => '3.33',
            'Locale::Codes::Constants'=> '3.33',
            'Locale::Codes::Country'=> '3.33',
            'Locale::Codes::Country_Codes'=> '3.33',
            'Locale::Codes::Country_Retired'=> '3.33',
            'Locale::Codes::Currency'=> '3.33',
            'Locale::Codes::Currency_Codes'=> '3.33',
            'Locale::Codes::Currency_Retired'=> '3.33',
            'Locale::Codes::LangExt'=> '3.33',
            'Locale::Codes::LangExt_Codes'=> '3.33',
            'Locale::Codes::LangExt_Retired'=> '3.33',
            'Locale::Codes::LangFam'=> '3.33',
            'Locale::Codes::LangFam_Codes'=> '3.33',
            'Locale::Codes::LangFam_Retired'=> '3.33',
            'Locale::Codes::LangVar'=> '3.33',
            'Locale::Codes::LangVar_Codes'=> '3.33',
            'Locale::Codes::LangVar_Retired'=> '3.33',
            'Locale::Codes::Language'=> '3.33',
            'Locale::Codes::Language_Codes'=> '3.33',
            'Locale::Codes::Language_Retired'=> '3.33',
            'Locale::Codes::Script' => '3.33',
            'Locale::Codes::Script_Codes'=> '3.33',
            'Locale::Codes::Script_Retired'=> '3.33',
            'Locale::Country'       => '3.33',
            'Locale::Currency'      => '3.33',
            'Locale::Language'      => '3.33',
            'Locale::Maketext'      => '1.26',
            'Locale::Script'        => '3.33',
            'Module::CoreList'      => '5.20141220',
            'Module::CoreList::TieHashDelta'=> '5.20141220',
            'Module::CoreList::Utils'=> '5.20141220',
            'NDBM_File'             => '1.14',
            'Net::Cmd'              => '3.04',
            'Net::Config'           => '3.04',
            'Net::Domain'           => '3.04',
            'Net::FTP'              => '3.04',
            'Net::FTP::A'           => '3.04',
            'Net::FTP::E'           => '3.04',
            'Net::FTP::I'           => '3.04',
            'Net::FTP::L'           => '3.04',
            'Net::FTP::dataconn'    => '3.04',
            'Net::NNTP'             => '3.04',
            'Net::Netrc'            => '3.04',
            'Net::POP3'             => '3.04',
            'Net::SMTP'             => '3.04',
            'Net::Time'             => '3.04',
            'Opcode'                => '1.30',
            'POSIX'                 => '1.48',
            'PerlIO::scalar'        => '0.21',
            'Pod::Escapes'          => '1.07',
            'SDBM_File'             => '1.12',
            'Storable'              => '2.52',
            'Sys::Hostname'         => '1.20',
            'Test::Builder'         => '1.301001_090',
            'Test::Builder::Module' => '1.301001_090',
            'Test::Builder::Tester' => '1.301001_090',
            'Test::Builder::Tester::Color'=> '1.301001_090',
            'Test::CanFork'         => undef,
            'Test::CanThread'       => undef,
            'Test::More'            => '1.301001_090',
            'Test::Simple'          => '1.301001_090',
            'Test::Stream'          => '1.301001_090',
            'Test::Stream::API'     => undef,
            'Test::Stream::ForceExit'=> undef,
            'Test::Stream::Subtest' => undef,
            'Test::Tester'          => '1.301001_090',
            'Test::use::ok'         => '1.301001_090',
            'Unicode::Collate'      => '1.09',
            'Unicode::Collate::CJK::Big5'=> '1.09',
            'Unicode::Collate::CJK::GB2312'=> '1.09',
            'Unicode::Collate::CJK::JISX0208'=> '1.09',
            'Unicode::Collate::CJK::Korean'=> '1.09',
            'Unicode::Collate::CJK::Pinyin'=> '1.09',
            'Unicode::Collate::CJK::Stroke'=> '1.09',
            'Unicode::Collate::CJK::Zhuyin'=> '1.09',
            'Unicode::Collate::Locale'=> '1.09',
            'XS::APItest'           => '0.69',
            'XSLoader'              => '0.20',
            '_charnames'            => '1.43',
            'arybase'               => '0.09',
            'charnames'             => '1.43',
            'feature'               => '1.39',
            'mro'                   => '1.17',
            'ok'                    => '1.301001_090',
            'strict'                => '1.09',
            'threads'               => '1.96_001',
        },
        removed => {
        }
    },
    5.021008 => {
        delta_from => 5.021007,
        changed => {
            'App::Prove'            => '3.35',
            'App::Prove::State'     => '3.35',
            'App::Prove::State::Result'=> '3.35',
            'App::Prove::State::Result::Test'=> '3.35',
            'B'                     => '1.55',
            'B::Deparse'            => '1.32',
            'B::Op_private'         => '5.021008',
            'CPAN::Meta::Requirements'=> '2.131',
            'Compress::Raw::Bzip2'  => '2.068',
            'Compress::Raw::Zlib'   => '2.068',
            'Compress::Zlib'        => '2.068',
            'Config'                => '5.021008',
            'DB_File'               => '1.835',
            'Data::Dumper'          => '2.156',
            'Devel::PPPort'         => '3.28',
            'Devel::Peek'           => '1.21',
            'Digest::MD5'           => '2.54',
            'Digest::SHA'           => '5.95',
            'DynaLoader'            => '1.30',
            'ExtUtils::Command'     => '1.20',
            'ExtUtils::Manifest'    => '1.70',
            'Fatal'                 => '2.26',
            'File::Glob'            => '1.24',
            'Filter::Util::Call'    => '1.54',
            'Getopt::Long'          => '2.43',
            'IO::Compress::Adapter::Bzip2'=> '2.068',
            'IO::Compress::Adapter::Deflate'=> '2.068',
            'IO::Compress::Adapter::Identity'=> '2.068',
            'IO::Compress::Base'    => '2.068',
            'IO::Compress::Base::Common'=> '2.068',
            'IO::Compress::Bzip2'   => '2.068',
            'IO::Compress::Deflate' => '2.068',
            'IO::Compress::Gzip'    => '2.068',
            'IO::Compress::Gzip::Constants'=> '2.068',
            'IO::Compress::RawDeflate'=> '2.068',
            'IO::Compress::Zip'     => '2.068',
            'IO::Compress::Zip::Constants'=> '2.068',
            'IO::Compress::Zlib::Constants'=> '2.068',
            'IO::Compress::Zlib::Extra'=> '2.068',
            'IO::Socket::IP'        => '0.36',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.068',
            'IO::Uncompress::Adapter::Identity'=> '2.068',
            'IO::Uncompress::Adapter::Inflate'=> '2.068',
            'IO::Uncompress::AnyInflate'=> '2.068',
            'IO::Uncompress::AnyUncompress'=> '2.068',
            'IO::Uncompress::Base'  => '2.068',
            'IO::Uncompress::Bunzip2'=> '2.068',
            'IO::Uncompress::Gunzip'=> '2.068',
            'IO::Uncompress::Inflate'=> '2.068',
            'IO::Uncompress::RawInflate'=> '2.068',
            'IO::Uncompress::Unzip' => '2.068',
            'MIME::Base64'          => '3.15',
            'Module::CoreList'      => '5.20150220',
            'Module::CoreList::TieHashDelta'=> '5.20150220',
            'Module::CoreList::Utils'=> '5.20150220',
            'Module::Load::Conditional'=> '0.64',
            'Module::Metadata'      => '1.000026',
            'Net::Cmd'              => '3.05',
            'Net::Config'           => '3.05',
            'Net::Domain'           => '3.05',
            'Net::FTP'              => '3.05',
            'Net::FTP::A'           => '3.05',
            'Net::FTP::E'           => '3.05',
            'Net::FTP::I'           => '3.05',
            'Net::FTP::L'           => '3.05',
            'Net::FTP::dataconn'    => '3.05',
            'Net::NNTP'             => '3.05',
            'Net::Netrc'            => '3.05',
            'Net::POP3'             => '3.05',
            'Net::SMTP'             => '3.05',
            'Net::Time'             => '3.05',
            'Opcode'                => '1.31',
            'POSIX'                 => '1.49',
            'PerlIO::encoding'      => '0.21',
            'Pod::Simple'           => '3.29',
            'Pod::Simple::BlackBox' => '3.29',
            'Pod::Simple::Checker'  => '3.29',
            'Pod::Simple::Debug'    => '3.29',
            'Pod::Simple::DumpAsText'=> '3.29',
            'Pod::Simple::DumpAsXML'=> '3.29',
            'Pod::Simple::HTML'     => '3.29',
            'Pod::Simple::HTMLBatch'=> '3.29',
            'Pod::Simple::LinkSection'=> '3.29',
            'Pod::Simple::Methody'  => '3.29',
            'Pod::Simple::Progress' => '3.29',
            'Pod::Simple::PullParser'=> '3.29',
            'Pod::Simple::PullParserEndToken'=> '3.29',
            'Pod::Simple::PullParserStartToken'=> '3.29',
            'Pod::Simple::PullParserTextToken'=> '3.29',
            'Pod::Simple::PullParserToken'=> '3.29',
            'Pod::Simple::RTF'      => '3.29',
            'Pod::Simple::Search'   => '3.29',
            'Pod::Simple::SimpleTree'=> '3.29',
            'Pod::Simple::Text'     => '3.29',
            'Pod::Simple::TextContent'=> '3.29',
            'Pod::Simple::TiedOutFH'=> '3.29',
            'Pod::Simple::Transcode'=> '3.29',
            'Pod::Simple::TranscodeDumb'=> '3.29',
            'Pod::Simple::TranscodeSmart'=> '3.29',
            'Pod::Simple::XHTML'    => '3.29',
            'Pod::Simple::XMLOutStream'=> '3.29',
            'SDBM_File'             => '1.13',
            'Safe'                  => '2.39',
            'TAP::Base'             => '3.35',
            'TAP::Formatter::Base'  => '3.35',
            'TAP::Formatter::Color' => '3.35',
            'TAP::Formatter::Console'=> '3.35',
            'TAP::Formatter::Console::ParallelSession'=> '3.35',
            'TAP::Formatter::Console::Session'=> '3.35',
            'TAP::Formatter::File'  => '3.35',
            'TAP::Formatter::File::Session'=> '3.35',
            'TAP::Formatter::Session'=> '3.35',
            'TAP::Harness'          => '3.35',
            'TAP::Harness::Env'     => '3.35',
            'TAP::Object'           => '3.35',
            'TAP::Parser'           => '3.35',
            'TAP::Parser::Aggregator'=> '3.35',
            'TAP::Parser::Grammar'  => '3.35',
            'TAP::Parser::Iterator' => '3.35',
            'TAP::Parser::Iterator::Array'=> '3.35',
            'TAP::Parser::Iterator::Process'=> '3.35',
            'TAP::Parser::Iterator::Stream'=> '3.35',
            'TAP::Parser::IteratorFactory'=> '3.35',
            'TAP::Parser::Multiplexer'=> '3.35',
            'TAP::Parser::Result'   => '3.35',
            'TAP::Parser::Result::Bailout'=> '3.35',
            'TAP::Parser::Result::Comment'=> '3.35',
            'TAP::Parser::Result::Plan'=> '3.35',
            'TAP::Parser::Result::Pragma'=> '3.35',
            'TAP::Parser::Result::Test'=> '3.35',
            'TAP::Parser::Result::Unknown'=> '3.35',
            'TAP::Parser::Result::Version'=> '3.35',
            'TAP::Parser::Result::YAML'=> '3.35',
            'TAP::Parser::ResultFactory'=> '3.35',
            'TAP::Parser::Scheduler'=> '3.35',
            'TAP::Parser::Scheduler::Job'=> '3.35',
            'TAP::Parser::Scheduler::Spinner'=> '3.35',
            'TAP::Parser::Source'   => '3.35',
            'TAP::Parser::SourceHandler'=> '3.35',
            'TAP::Parser::SourceHandler::Executable'=> '3.35',
            'TAP::Parser::SourceHandler::File'=> '3.35',
            'TAP::Parser::SourceHandler::Handle'=> '3.35',
            'TAP::Parser::SourceHandler::Perl'=> '3.35',
            'TAP::Parser::SourceHandler::RawTAP'=> '3.35',
            'TAP::Parser::YAMLish::Reader'=> '3.35',
            'TAP::Parser::YAMLish::Writer'=> '3.35',
            'Test::Builder'         => '1.301001_097',
            'Test::Builder::Module' => '1.301001_097',
            'Test::Builder::Tester' => '1.301001_097',
            'Test::Builder::Tester::Color'=> '1.301001_097',
            'Test::Harness'         => '3.35',
            'Test::More'            => '1.301001_097',
            'Test::Simple'          => '1.301001_097',
            'Test::Stream'          => '1.301001_097',
            'Test::Stream::Block'   => undef,
            'Test::Tester'          => '1.301001_097',
            'Test::Tester::CaptureRunner'=> undef,
            'Test::Tester::Delegate'=> undef,
            'Test::use::ok'         => '1.301001_097',
            'Unicode::Collate'      => '1.10',
            'Unicode::Collate::CJK::Big5'=> '1.10',
            'Unicode::Collate::CJK::GB2312'=> '1.10',
            'Unicode::Collate::CJK::JISX0208'=> '1.10',
            'Unicode::Collate::CJK::Korean'=> '1.10',
            'Unicode::Collate::CJK::Pinyin'=> '1.10',
            'Unicode::Collate::CJK::Stroke'=> '1.10',
            'Unicode::Collate::CJK::Zhuyin'=> '1.10',
            'Unicode::Collate::Locale'=> '1.10',
            'VMS::DCLsym'           => '1.06',
            'XS::APItest'           => '0.70',
            'arybase'               => '0.10',
            'attributes'            => '0.25',
            'autodie'               => '2.26',
            'autodie::Scope::Guard' => '2.26',
            'autodie::Scope::GuardStack'=> '2.26',
            'autodie::ScopeUtil'    => '2.26',
            'autodie::exception'    => '2.26',
            'autodie::exception::system'=> '2.26',
            'autodie::hints'        => '2.26',
            'autodie::skip'         => '2.26',
            'ok'                    => '1.301001_097',
            're'                    => '0.30',
            'warnings'              => '1.30',
        },
        removed => {
        }
    },
    5.020002 => {
        delta_from => 5.020001,
        changed => {
            'CPAN::Author'          => '5.5002',
            'CPAN::CacheMgr'        => '5.5002',
            'CPAN::FTP'             => '5.5006',
            'CPAN::HTTP::Client'    => '1.9601',
            'CPAN::HandleConfig'    => '5.5005',
            'CPAN::Index'           => '1.9601',
            'CPAN::LWP::UserAgent'  => '1.9601',
            'CPAN::Mirrors'         => '1.9601',
            'Config'                => '5.020002',
            'Cwd'                   => '3.48_01',
            'Data::Dumper'          => '2.151_01',
            'Errno'                 => '1.20_05',
            'File::Spec'            => '3.48_01',
            'File::Spec::Cygwin'    => '3.48_01',
            'File::Spec::Epoc'      => '3.48_01',
            'File::Spec::Functions' => '3.48_01',
            'File::Spec::Mac'       => '3.48_01',
            'File::Spec::OS2'       => '3.48_01',
            'File::Spec::Unix'      => '3.48_01',
            'File::Spec::VMS'       => '3.48_01',
            'File::Spec::Win32'     => '3.48_01',
            'IO::Socket'            => '1.38',
            'Module::CoreList'      => '5.20150214',
            'Module::CoreList::TieHashDelta'=> '5.20150214',
            'Module::CoreList::Utils'=> '5.20150214',
            'PerlIO::scalar'        => '0.18_01',
            'Pod::PlainText'        => '2.07',
            'Storable'              => '2.49_01',
            'VMS::DCLsym'           => '1.05_01',
            'VMS::Stdio'            => '2.41',
            'attributes'            => '0.23',
            'feature'               => '1.36_01',
        },
        removed => {
        }
    },
    5.021009 => {
        delta_from => 5.021008,
        changed => {
            'B'                     => '1.56',
            'B::Debug'              => '1.23',
            'B::Deparse'            => '1.33',
            'B::Op_private'         => '5.021009',
            'Benchmark'             => '1.20',
            'CPAN::Author'          => '5.5002',
            'CPAN::CacheMgr'        => '5.5002',
            'CPAN::FTP'             => '5.5006',
            'CPAN::HTTP::Client'    => '1.9601',
            'CPAN::HandleConfig'    => '5.5005',
            'CPAN::Index'           => '1.9601',
            'CPAN::LWP::UserAgent'  => '1.9601',
            'CPAN::Meta::Requirements'=> '2.132',
            'CPAN::Mirrors'         => '1.9601',
            'Carp'                  => '1.35',
            'Carp::Heavy'           => '1.35',
            'Config'                => '5.021009',
            'Config::Perl::V'       => '0.23',
            'Data::Dumper'          => '2.157',
            'Devel::Peek'           => '1.22',
            'DynaLoader'            => '1.31',
            'Encode'                => '2.70',
            'Encode::MIME::Header'  => '2.16',
            'Errno'                 => '1.23',
            'ExtUtils::Miniperl'    => '1.04',
            'HTTP::Tiny'            => '0.054',
            'Module::CoreList'      => '5.20150220',
            'Module::CoreList::TieHashDelta'=> '5.20150220',
            'Module::CoreList::Utils'=> '5.20150220',
            'Opcode'                => '1.32',
            'POSIX'                 => '1.51',
            'Perl::OSType'          => '1.008',
            'PerlIO::scalar'        => '0.22',
            'Pod::Find'             => '1.63',
            'Pod::InputObjects'     => '1.63',
            'Pod::ParseUtils'       => '1.63',
            'Pod::Parser'           => '1.63',
            'Pod::Perldoc'          => '3.25',
            'Pod::Perldoc::BaseTo'  => '3.25',
            'Pod::Perldoc::GetOptsOO'=> '3.25',
            'Pod::Perldoc::ToANSI'  => '3.25',
            'Pod::Perldoc::ToChecker'=> '3.25',
            'Pod::Perldoc::ToMan'   => '3.25',
            'Pod::Perldoc::ToNroff' => '3.25',
            'Pod::Perldoc::ToPod'   => '3.25',
            'Pod::Perldoc::ToRtf'   => '3.25',
            'Pod::Perldoc::ToTerm'  => '3.25',
            'Pod::Perldoc::ToText'  => '3.25',
            'Pod::Perldoc::ToTk'    => '3.25',
            'Pod::Perldoc::ToXml'   => '3.25',
            'Pod::PlainText'        => '2.07',
            'Pod::Select'           => '1.63',
            'Socket'                => '2.018',
            'Storable'              => '2.53',
            'Test::Builder'         => '1.301001_098',
            'Test::Builder::Module' => '1.301001_098',
            'Test::Builder::Tester' => '1.301001_098',
            'Test::Builder::Tester::Color'=> '1.301001_098',
            'Test::More'            => '1.301001_098',
            'Test::Simple'          => '1.301001_098',
            'Test::Stream'          => '1.301001_098',
            'Test::Tester'          => '1.301001_098',
            'Test::use::ok'         => '1.301001_098',
            'Unicode::Collate'      => '1.11',
            'Unicode::Collate::CJK::Big5'=> '1.11',
            'Unicode::Collate::CJK::GB2312'=> '1.11',
            'Unicode::Collate::CJK::JISX0208'=> '1.11',
            'Unicode::Collate::CJK::Korean'=> '1.11',
            'Unicode::Collate::CJK::Pinyin'=> '1.11',
            'Unicode::Collate::CJK::Stroke'=> '1.11',
            'Unicode::Collate::CJK::Zhuyin'=> '1.11',
            'Unicode::Collate::Locale'=> '1.11',
            'Unicode::UCD'          => '0.61',
            'VMS::Stdio'            => '2.41',
            'Win32'                 => '0.51',
            'Win32API::File'        => '0.1202',
            'attributes'            => '0.26',
            'bigint'                => '0.39',
            'bignum'                => '0.39',
            'bigrat'                => '0.39',
            'constant'              => '1.33',
            'encoding'              => '2.13',
            'feature'               => '1.40',
            'ok'                    => '1.301001_098',
            'overload'              => '1.25',
            'perlfaq'               => '5.021009',
            're'                    => '0.31',
            'threads::shared'       => '1.48',
            'warnings'              => '1.31',
        },
        removed => {
        }
    },
    5.021010 => {
        delta_from => 5.021009,
        changed => {
            'App::Cpan'             => '1.63',
            'B'                     => '1.57',
            'B::Deparse'            => '1.34',
            'B::Op_private'         => '5.021010',
            'Benchmark'             => '1.2',
            'CPAN'                  => '2.10',
            'CPAN::Distribution'    => '2.04',
            'CPAN::FirstTime'       => '5.5307',
            'CPAN::HTTP::Credentials'=> '1.9601',
            'CPAN::HandleConfig'    => '5.5006',
            'CPAN::Meta'            => '2.150001',
            'CPAN::Meta::Converter' => '2.150001',
            'CPAN::Meta::Feature'   => '2.150001',
            'CPAN::Meta::History'   => '2.150001',
            'CPAN::Meta::Merge'     => '2.150001',
            'CPAN::Meta::Prereqs'   => '2.150001',
            'CPAN::Meta::Spec'      => '2.150001',
            'CPAN::Meta::Validator' => '2.150001',
            'CPAN::Module'          => '5.5002',
            'CPAN::Plugin'          => '0.95',
            'CPAN::Plugin::Specfile'=> '0.01',
            'CPAN::Shell'           => '5.5005',
            'Carp'                  => '1.36',
            'Carp::Heavy'           => '1.36',
            'Config'                => '5.02101',
            'Cwd'                   => '3.55',
            'DB'                    => '1.08',
            'Data::Dumper'          => '2.158',
            'Devel::PPPort'         => '3.31',
            'DynaLoader'            => '1.32',
            'Encode'                => '2.72',
            'Encode::Alias'         => '2.19',
            'File::Spec'            => '3.55',
            'File::Spec::Cygwin'    => '3.55',
            'File::Spec::Epoc'      => '3.55',
            'File::Spec::Functions' => '3.55',
            'File::Spec::Mac'       => '3.55',
            'File::Spec::OS2'       => '3.55',
            'File::Spec::Unix'      => '3.55',
            'File::Spec::VMS'       => '3.55',
            'File::Spec::Win32'     => '3.55',
            'Getopt::Long'          => '2.45',
            'Locale::Codes'         => '3.34',
            'Locale::Codes::Constants'=> '3.34',
            'Locale::Codes::Country'=> '3.34',
            'Locale::Codes::Country_Codes'=> '3.34',
            'Locale::Codes::Country_Retired'=> '3.34',
            'Locale::Codes::Currency'=> '3.34',
            'Locale::Codes::Currency_Codes'=> '3.34',
            'Locale::Codes::Currency_Retired'=> '3.34',
            'Locale::Codes::LangExt'=> '3.34',
            'Locale::Codes::LangExt_Codes'=> '3.34',
            'Locale::Codes::LangExt_Retired'=> '3.34',
            'Locale::Codes::LangFam'=> '3.34',
            'Locale::Codes::LangFam_Codes'=> '3.34',
            'Locale::Codes::LangFam_Retired'=> '3.34',
            'Locale::Codes::LangVar'=> '3.34',
            'Locale::Codes::LangVar_Codes'=> '3.34',
            'Locale::Codes::LangVar_Retired'=> '3.34',
            'Locale::Codes::Language'=> '3.34',
            'Locale::Codes::Language_Codes'=> '3.34',
            'Locale::Codes::Language_Retired'=> '3.34',
            'Locale::Codes::Script' => '3.34',
            'Locale::Codes::Script_Codes'=> '3.34',
            'Locale::Codes::Script_Retired'=> '3.34',
            'Locale::Country'       => '3.34',
            'Locale::Currency'      => '3.34',
            'Locale::Language'      => '3.34',
            'Locale::Script'        => '3.34',
            'Module::CoreList'      => '5.20150320',
            'Module::CoreList::TieHashDelta'=> '5.20150320',
            'Module::CoreList::Utils'=> '5.20150320',
            'POSIX'                 => '1.52',
            'Pod::Functions'        => '1.09',
            'Term::Complete'        => '1.403',
            'Test::Builder'         => '1.001014',
            'Test::Builder::IO::Scalar'=> '2.113',
            'Test::Builder::Module' => '1.001014',
            'Test::Builder::Tester' => '1.28',
            'Test::Builder::Tester::Color'=> '1.290001',
            'Test::More'            => '1.001014',
            'Test::Simple'          => '1.001014',
            'Test::Tester'          => '0.114',
            'Test::use::ok'         => '0.16',
            'Text::Balanced'        => '2.03',
            'Text::ParseWords'      => '3.30',
            'Unicode::Collate'      => '1.12',
            'Unicode::Collate::CJK::Big5'=> '1.12',
            'Unicode::Collate::CJK::GB2312'=> '1.12',
            'Unicode::Collate::CJK::JISX0208'=> '1.12',
            'Unicode::Collate::CJK::Korean'=> '1.12',
            'Unicode::Collate::CJK::Pinyin'=> '1.12',
            'Unicode::Collate::CJK::Stroke'=> '1.12',
            'Unicode::Collate::CJK::Zhuyin'=> '1.12',
            'Unicode::Collate::Locale'=> '1.12',
            'XS::APItest'           => '0.71',
            'encoding'              => '2.14',
            'locale'                => '1.06',
            'meta_notation'         => undef,
            'ok'                    => '0.16',
            'parent'                => '0.232',
            're'                    => '0.32',
            'sigtrap'               => '1.08',
            'threads'               => '2.01',
            'utf8'                  => '1.15',
        },
        removed => {
            'Test::CanFork'         => 1,
            'Test::CanThread'       => 1,
            'Test::More::DeepCheck' => 1,
            'Test::More::DeepCheck::Strict'=> 1,
            'Test::More::DeepCheck::Tolerant'=> 1,
            'Test::More::Tools'     => 1,
            'Test::MostlyLike'      => 1,
            'Test::Stream'          => 1,
            'Test::Stream::API'     => 1,
            'Test::Stream::ArrayBase'=> 1,
            'Test::Stream::ArrayBase::Meta'=> 1,
            'Test::Stream::Block'   => 1,
            'Test::Stream::Carp'    => 1,
            'Test::Stream::Context' => 1,
            'Test::Stream::Event'   => 1,
            'Test::Stream::Event::Bail'=> 1,
            'Test::Stream::Event::Child'=> 1,
            'Test::Stream::Event::Diag'=> 1,
            'Test::Stream::Event::Finish'=> 1,
            'Test::Stream::Event::Note'=> 1,
            'Test::Stream::Event::Ok'=> 1,
            'Test::Stream::Event::Plan'=> 1,
            'Test::Stream::Event::Subtest'=> 1,
            'Test::Stream::ExitMagic'=> 1,
            'Test::Stream::ExitMagic::Context'=> 1,
            'Test::Stream::Exporter'=> 1,
            'Test::Stream::Exporter::Meta'=> 1,
            'Test::Stream::ForceExit'=> 1,
            'Test::Stream::IOSets'  => 1,
            'Test::Stream::Meta'    => 1,
            'Test::Stream::PackageUtil'=> 1,
            'Test::Stream::Subtest' => 1,
            'Test::Stream::Tester'  => 1,
            'Test::Stream::Tester::Checks'=> 1,
            'Test::Stream::Tester::Checks::Event'=> 1,
            'Test::Stream::Tester::Events'=> 1,
            'Test::Stream::Tester::Events::Event'=> 1,
            'Test::Stream::Tester::Grab'=> 1,
            'Test::Stream::Threads' => 1,
            'Test::Stream::Toolset' => 1,
            'Test::Stream::Util'    => 1,
        }
    },
    5.021011 => {
        delta_from => 5.021010,
        changed => {
            'B'                     => '1.58',
            'B::Deparse'            => '1.35',
            'B::Op_private'         => '5.021011',
            'CPAN'                  => '2.11',
            'Config'                => '5.021011',
            'Config::Perl::V'       => '0.24',
            'Cwd'                   => '3.56',
            'ExtUtils::Miniperl'    => '1.05',
            'ExtUtils::ParseXS'     => '3.28',
            'ExtUtils::ParseXS::Constants'=> '3.28',
            'ExtUtils::ParseXS::CountLines'=> '3.28',
            'ExtUtils::ParseXS::Eval'=> '3.28',
            'ExtUtils::ParseXS::Utilities'=> '3.28',
            'ExtUtils::Typemaps'    => '3.28',
            'ExtUtils::Typemaps::Cmd'=> '3.28',
            'ExtUtils::Typemaps::InputMap'=> '3.28',
            'ExtUtils::Typemaps::OutputMap'=> '3.28',
            'ExtUtils::Typemaps::Type'=> '3.28',
            'File::Spec'            => '3.56',
            'File::Spec::Cygwin'    => '3.56',
            'File::Spec::Epoc'      => '3.56',
            'File::Spec::Functions' => '3.56',
            'File::Spec::Mac'       => '3.56',
            'File::Spec::OS2'       => '3.56',
            'File::Spec::Unix'      => '3.56',
            'File::Spec::VMS'       => '3.56',
            'File::Spec::Win32'     => '3.56',
            'IO::Socket::IP'        => '0.37',
            'Module::CoreList'      => '5.20150420',
            'Module::CoreList::TieHashDelta'=> '5.20150420',
            'Module::CoreList::Utils'=> '5.20150420',
            'PerlIO::mmap'          => '0.014',
            'XS::APItest'           => '0.72',
            'attributes'            => '0.27',
            'if'                    => '0.0604',
            'utf8'                  => '1.16',
            'warnings'              => '1.32',
        },
        removed => {
        }
    },
    5.022000 => {
        delta_from => 5.021011,
        changed => {
            'B::Op_private'         => '5.022000',
            'Config'                => '5.022',
            'ExtUtils::Command::MM' => '7.04_01',
            'ExtUtils::Liblist'     => '7.04_01',
            'ExtUtils::Liblist::Kid'=> '7.04_01',
            'ExtUtils::MM'          => '7.04_01',
            'ExtUtils::MM_AIX'      => '7.04_01',
            'ExtUtils::MM_Any'      => '7.04_01',
            'ExtUtils::MM_BeOS'     => '7.04_01',
            'ExtUtils::MM_Cygwin'   => '7.04_01',
            'ExtUtils::MM_DOS'      => '7.04_01',
            'ExtUtils::MM_Darwin'   => '7.04_01',
            'ExtUtils::MM_MacOS'    => '7.04_01',
            'ExtUtils::MM_NW5'      => '7.04_01',
            'ExtUtils::MM_OS2'      => '7.04_01',
            'ExtUtils::MM_QNX'      => '7.04_01',
            'ExtUtils::MM_UWIN'     => '7.04_01',
            'ExtUtils::MM_Unix'     => '7.04_01',
            'ExtUtils::MM_VMS'      => '7.04_01',
            'ExtUtils::MM_VOS'      => '7.04_01',
            'ExtUtils::MM_Win32'    => '7.04_01',
            'ExtUtils::MM_Win95'    => '7.04_01',
            'ExtUtils::MY'          => '7.04_01',
            'ExtUtils::MakeMaker'   => '7.04_01',
            'ExtUtils::MakeMaker::Config'=> '7.04_01',
            'ExtUtils::MakeMaker::Locale'=> '7.04_01',
            'ExtUtils::MakeMaker::version'=> '7.04_01',
            'ExtUtils::MakeMaker::version::regex'=> '7.04_01',
            'ExtUtils::MakeMaker::version::vpp'=> '7.04_01',
            'ExtUtils::Mkbootstrap' => '7.04_01',
            'ExtUtils::Mksymlists'  => '7.04_01',
            'ExtUtils::testlib'     => '7.04_01',
            'Module::CoreList'      => '5.20150520',
            'Module::CoreList::TieHashDelta'=> '5.20150520',
            'Module::CoreList::Utils'=> '5.20150520',
            'POSIX'                 => '1.53',
            'PerlIO::via::QuotedPrint'=> '0.08',
            'overload'              => '1.26',
            'utf8'                  => '1.17',
        },
        removed => {
        }
    },
    5.023000 => {
        delta_from => 5.022000,
        changed => {
            'B::Op_private'         => '5.023000',
            'CPAN::Meta'            => '2.150005',
            'CPAN::Meta::Converter' => '2.150005',
            'CPAN::Meta::Feature'   => '2.150005',
            'CPAN::Meta::History'   => '2.150005',
            'CPAN::Meta::Merge'     => '2.150005',
            'CPAN::Meta::Prereqs'   => '2.150005',
            'CPAN::Meta::Requirements'=> '2.133',
            'CPAN::Meta::Spec'      => '2.150005',
            'CPAN::Meta::Validator' => '2.150005',
            'CPAN::Meta::YAML'      => '0.016',
            'Config'                => '5.023',
            'Encode'                => '2.73',
            'ExtUtils::CBuilder'    => '0.280223',
            'ExtUtils::CBuilder::Base'=> '0.280223',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.280223',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.280223',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.280223',
            'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280223',
            'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280223',
            'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280223',
            'ExtUtils::CBuilder::Platform::aix'=> '0.280223',
            'ExtUtils::CBuilder::Platform::android'=> '0.280223',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.280223',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.280223',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280223',
            'ExtUtils::CBuilder::Platform::os2'=> '0.280223',
            'Fatal'                 => '2.27',
            'Getopt::Long'          => '2.46',
            'HTTP::Tiny'            => '0.056',
            'List::Util'            => '1.42_01',
            'List::Util::XS'        => '1.42_01',
            'Locale::Codes'         => '3.35',
            'Locale::Codes::Constants'=> '3.35',
            'Locale::Codes::Country'=> '3.35',
            'Locale::Codes::Country_Codes'=> '3.35',
            'Locale::Codes::Country_Retired'=> '3.35',
            'Locale::Codes::Currency'=> '3.35',
            'Locale::Codes::Currency_Codes'=> '3.35',
            'Locale::Codes::Currency_Retired'=> '3.35',
            'Locale::Codes::LangExt'=> '3.35',
            'Locale::Codes::LangExt_Codes'=> '3.35',
            'Locale::Codes::LangExt_Retired'=> '3.35',
            'Locale::Codes::LangFam'=> '3.35',
            'Locale::Codes::LangFam_Codes'=> '3.35',
            'Locale::Codes::LangFam_Retired'=> '3.35',
            'Locale::Codes::LangVar'=> '3.35',
            'Locale::Codes::LangVar_Codes'=> '3.35',
            'Locale::Codes::LangVar_Retired'=> '3.35',
            'Locale::Codes::Language'=> '3.35',
            'Locale::Codes::Language_Codes'=> '3.35',
            'Locale::Codes::Language_Retired'=> '3.35',
            'Locale::Codes::Script' => '3.35',
            'Locale::Codes::Script_Codes'=> '3.35',
            'Locale::Codes::Script_Retired'=> '3.35',
            'Locale::Country'       => '3.35',
            'Locale::Currency'      => '3.35',
            'Locale::Language'      => '3.35',
            'Locale::Script'        => '3.35',
            'Math::BigFloat'        => '1.999701',
            'Math::BigInt'          => '1.999701',
            'Math::BigInt::Calc'    => '1.999701',
            'Math::BigInt::CalcEmu' => '1.999701',
            'Math::BigRat'          => '0.260801',
            'Module::CoreList'      => '5.20150620',
            'Module::CoreList::TieHashDelta'=> '5.20150620',
            'Module::CoreList::Utils'=> '5.20150620',
            'Module::Metadata'      => '1.000027',
            'Net::Cmd'              => '3.06',
            'Net::Config'           => '3.06',
            'Net::Domain'           => '3.06',
            'Net::FTP'              => '3.06',
            'Net::FTP::A'           => '3.06',
            'Net::FTP::E'           => '3.06',
            'Net::FTP::I'           => '3.06',
            'Net::FTP::L'           => '3.06',
            'Net::FTP::dataconn'    => '3.06',
            'Net::NNTP'             => '3.06',
            'Net::Netrc'            => '3.06',
            'Net::POP3'             => '3.06',
            'Net::SMTP'             => '3.06',
            'Net::Time'             => '3.06',
            'POSIX'                 => '1.54',
            'Parse::CPAN::Meta'     => '1.4417',
            'Pod::Simple'           => '3.30',
            'Pod::Simple::BlackBox' => '3.30',
            'Pod::Simple::Checker'  => '3.30',
            'Pod::Simple::Debug'    => '3.30',
            'Pod::Simple::DumpAsText'=> '3.30',
            'Pod::Simple::DumpAsXML'=> '3.30',
            'Pod::Simple::HTML'     => '3.30',
            'Pod::Simple::HTMLBatch'=> '3.30',
            'Pod::Simple::LinkSection'=> '3.30',
            'Pod::Simple::Methody'  => '3.30',
            'Pod::Simple::Progress' => '3.30',
            'Pod::Simple::PullParser'=> '3.30',
            'Pod::Simple::PullParserEndToken'=> '3.30',
            'Pod::Simple::PullParserStartToken'=> '3.30',
            'Pod::Simple::PullParserTextToken'=> '3.30',
            'Pod::Simple::PullParserToken'=> '3.30',
            'Pod::Simple::RTF'      => '3.30',
            'Pod::Simple::Search'   => '3.30',
            'Pod::Simple::SimpleTree'=> '3.30',
            'Pod::Simple::Text'     => '3.30',
            'Pod::Simple::TextContent'=> '3.30',
            'Pod::Simple::TiedOutFH'=> '3.30',
            'Pod::Simple::Transcode'=> '3.30',
            'Pod::Simple::TranscodeDumb'=> '3.30',
            'Pod::Simple::TranscodeSmart'=> '3.30',
            'Pod::Simple::XHTML'    => '3.30',
            'Pod::Simple::XMLOutStream'=> '3.30',
            'Pod::Usage'            => '1.67',
            'Scalar::Util'          => '1.42_01',
            'Socket'                => '2.019',
            'Sub::Util'             => '1.42_01',
            'Time::Piece'           => '1.30',
            'Time::Seconds'         => '1.30',
            'UNIVERSAL'             => '1.13',
            'Unicode'               => '8.0.0',
            'XS::APItest'           => '0.73',
            'autodie'               => '2.27',
            'autodie::Scope::Guard' => '2.27',
            'autodie::Scope::GuardStack'=> '2.27',
            'autodie::Util'         => '2.27',
            'autodie::exception'    => '2.27',
            'autodie::exception::system'=> '2.27',
            'autodie::hints'        => '2.27',
            'autodie::skip'         => '2.27',
            'encoding'              => '2.15',
            'feature'               => '1.41',
            'parent'                => '0.234',
            'threads'               => '2.02',
        },
        removed => {
        }
    },
    5.023001 => {
        delta_from => 5.023000,
        changed => {
            'B::Op_private'         => '5.023001',
            'Config'                => '5.023001',
            'DynaLoader'            => '1.33',
            'Encode'                => '2.75',
            'Encode::MIME::Header'  => '2.17',
            'Encode::Unicode'       => '2.13',
            'Fatal'                 => '2.29',
            'File::Path'            => '2.11',
            'Getopt::Long'          => '2.47',
            'I18N::Langinfo'        => '0.13',
            'IPC::Open3'            => '1.19',
            'Module::CoreList'      => '5.20150720',
            'Module::CoreList::TieHashDelta'=> '5.20150720',
            'Module::CoreList::Utils'=> '5.20150720',
            'Net::Cmd'              => '3.07',
            'Net::Config'           => '3.07',
            'Net::Domain'           => '3.07',
            'Net::FTP'              => '3.07',
            'Net::FTP::A'           => '3.07',
            'Net::FTP::E'           => '3.07',
            'Net::FTP::I'           => '3.07',
            'Net::FTP::L'           => '3.07',
            'Net::FTP::dataconn'    => '3.07',
            'Net::NNTP'             => '3.07',
            'Net::Netrc'            => '3.07',
            'Net::POP3'             => '3.07',
            'Net::SMTP'             => '3.07',
            'Net::Time'             => '3.07',
            'Opcode'                => '1.33',
            'POSIX'                 => '1.55',
            'PerlIO::scalar'        => '0.23',
            'Socket'                => '2.020',
            'Storable'              => '2.54',
            'Unicode::Collate'      => '1.14',
            'Unicode::Collate::CJK::Big5'=> '1.14',
            'Unicode::Collate::CJK::GB2312'=> '1.14',
            'Unicode::Collate::CJK::JISX0208'=> '1.14',
            'Unicode::Collate::CJK::Korean'=> '1.14',
            'Unicode::Collate::CJK::Pinyin'=> '1.14',
            'Unicode::Collate::CJK::Stroke'=> '1.14',
            'Unicode::Collate::CJK::Zhuyin'=> '1.14',
            'Unicode::Collate::Locale'=> '1.14',
            'Unicode::Normalize'    => '1.19',
            'XS::APItest'           => '0.74',
            'XS::Typemap'           => '0.14',
            'autodie'               => '2.29',
            'autodie::Scope::Guard' => '2.29',
            'autodie::Scope::GuardStack'=> '2.29',
            'autodie::Util'         => '2.29',
            'autodie::exception'    => '2.29',
            'autodie::exception::system'=> '2.29',
            'autodie::hints'        => '2.29',
            'autodie::skip'         => '2.29',
            'encoding'              => '2.16',
            'feature'               => '1.42',
            'warnings'              => '1.33',
        },
        removed => {
            'autodie::ScopeUtil'    => 1,
        }
    },
    5.023002 => {
        delta_from => 5.023001,
        changed => {
            'Attribute::Handlers'   => '0.99',
            'B::Op_private'         => '5.023002',
            'CPAN::Meta::YAML'      => '0.017',
            'Config'                => '5.023002',
            'Cwd'                   => '3.57',
            'Encode'                => '2.76',
            'ExtUtils::ParseXS'     => '3.29',
            'ExtUtils::ParseXS::Constants'=> '3.29',
            'ExtUtils::ParseXS::CountLines'=> '3.29',
            'ExtUtils::ParseXS::Eval'=> '3.29',
            'ExtUtils::ParseXS::Utilities'=> '3.29',
            'ExtUtils::Typemaps'    => '3.29',
            'File::Find'            => '1.30',
            'File::Spec'            => '3.57',
            'File::Spec::Cygwin'    => '3.57',
            'File::Spec::Epoc'      => '3.57',
            'File::Spec::Functions' => '3.57',
            'File::Spec::Mac'       => '3.57',
            'File::Spec::OS2'       => '3.57',
            'File::Spec::Unix'      => '3.57',
            'File::Spec::VMS'       => '3.57',
            'File::Spec::Win32'     => '3.57',
            'Filter::Util::Call'    => '1.55',
            'Hash::Util'            => '0.19',
            'Module::CoreList'      => '5.20150820',
            'Module::CoreList::TieHashDelta'=> '5.20150820',
            'Module::CoreList::Utils'=> '5.20150820',
            'POSIX'                 => '1.56',
            'Term::Cap'             => '1.17',
            'Unicode::UCD'          => '0.62',
            'perlfaq'               => '5.021010',
        },
        removed => {
        }
    },
    5.020003 => {
        delta_from => 5.020002,
        changed => {
            'Config'                => '5.020003',
            'Errno'                 => '1.20_06',
            'Module::CoreList'      => '5.20150912',
            'Module::CoreList::TieHashDelta'=> '5.20150912',
            'Module::CoreList::Utils'=> '5.20150912',
        },
        removed => {
        }
    },
    5.023003 => {
        delta_from => 5.023002,
        changed => {
            'Amiga::ARexx'          => '0.02',
            'Amiga::Exec'           => '0.01',
            'B'                     => '1.59',
            'B::Op_private'         => '5.023003',
            'Carp'                  => '1.37',
            'Carp::Heavy'           => '1.37',
            'Compress::Raw::Zlib'   => '2.068_01',
            'Config'                => '5.023003',
            'Cwd'                   => '3.58',
            'DynaLoader'            => '1.34',
            'Encode'                => '2.77',
            'Encode::Unicode'       => '2.14',
            'English'               => '1.10',
            'Errno'                 => '1.24',
            'ExtUtils::Command'     => '7.10',
            'ExtUtils::Command::MM' => '7.10',
            'ExtUtils::Liblist'     => '7.10',
            'ExtUtils::Liblist::Kid'=> '7.10',
            'ExtUtils::MM'          => '7.10',
            'ExtUtils::MM_AIX'      => '7.10',
            'ExtUtils::MM_Any'      => '7.10',
            'ExtUtils::MM_BeOS'     => '7.10',
            'ExtUtils::MM_Cygwin'   => '7.10',
            'ExtUtils::MM_DOS'      => '7.10',
            'ExtUtils::MM_Darwin'   => '7.10',
            'ExtUtils::MM_MacOS'    => '7.10',
            'ExtUtils::MM_NW5'      => '7.10',
            'ExtUtils::MM_OS2'      => '7.10',
            'ExtUtils::MM_QNX'      => '7.10',
            'ExtUtils::MM_UWIN'     => '7.10',
            'ExtUtils::MM_Unix'     => '7.10',
            'ExtUtils::MM_VMS'      => '7.10',
            'ExtUtils::MM_VOS'      => '7.10',
            'ExtUtils::MM_Win32'    => '7.10',
            'ExtUtils::MM_Win95'    => '7.10',
            'ExtUtils::MY'          => '7.10',
            'ExtUtils::MakeMaker'   => '7.10',
            'ExtUtils::MakeMaker::Config'=> '7.10',
            'ExtUtils::MakeMaker::Locale'=> '7.10',
            'ExtUtils::MakeMaker::version'=> '7.10',
            'ExtUtils::MakeMaker::version::regex'=> '7.10',
            'ExtUtils::MakeMaker::version::vpp'=> '7.10',
            'ExtUtils::Mkbootstrap' => '7.10',
            'ExtUtils::Mksymlists'  => '7.10',
            'ExtUtils::ParseXS'     => '3.30',
            'ExtUtils::ParseXS::Constants'=> '3.30',
            'ExtUtils::ParseXS::CountLines'=> '3.30',
            'ExtUtils::ParseXS::Eval'=> '3.30',
            'ExtUtils::ParseXS::Utilities'=> '3.30',
            'ExtUtils::Typemaps'    => '3.30',
            'ExtUtils::Typemaps::Cmd'=> '3.30',
            'ExtUtils::Typemaps::InputMap'=> '3.30',
            'ExtUtils::Typemaps::OutputMap'=> '3.30',
            'ExtUtils::Typemaps::Type'=> '3.30',
            'ExtUtils::testlib'     => '7.10',
            'File::Find'            => '1.31',
            'File::Glob'            => '1.25',
            'File::Spec'            => '3.58',
            'File::Spec::AmigaOS'   => '3.58',
            'File::Spec::Cygwin'    => '3.58',
            'File::Spec::Epoc'      => '3.58',
            'File::Spec::Functions' => '3.58',
            'File::Spec::Mac'       => '3.58',
            'File::Spec::OS2'       => '3.58',
            'File::Spec::Unix'      => '3.58',
            'File::Spec::VMS'       => '3.58',
            'File::Spec::Win32'     => '3.58',
            'Hash::Util::FieldHash' => '1.17',
            'Locale::Codes'         => '3.36',
            'Locale::Codes::Constants'=> '3.36',
            'Locale::Codes::Country'=> '3.36',
            'Locale::Codes::Country_Codes'=> '3.36',
            'Locale::Codes::Country_Retired'=> '3.36',
            'Locale::Codes::Currency'=> '3.36',
            'Locale::Codes::Currency_Codes'=> '3.36',
            'Locale::Codes::Currency_Retired'=> '3.36',
            'Locale::Codes::LangExt'=> '3.36',
            'Locale::Codes::LangExt_Codes'=> '3.36',
            'Locale::Codes::LangExt_Retired'=> '3.36',
            'Locale::Codes::LangFam'=> '3.36',
            'Locale::Codes::LangFam_Codes'=> '3.36',
            'Locale::Codes::LangFam_Retired'=> '3.36',
            'Locale::Codes::LangVar'=> '3.36',
            'Locale::Codes::LangVar_Codes'=> '3.36',
            'Locale::Codes::LangVar_Retired'=> '3.36',
            'Locale::Codes::Language'=> '3.36',
            'Locale::Codes::Language_Codes'=> '3.36',
            'Locale::Codes::Language_Retired'=> '3.36',
            'Locale::Codes::Script' => '3.36',
            'Locale::Codes::Script_Codes'=> '3.36',
            'Locale::Codes::Script_Retired'=> '3.36',
            'Locale::Country'       => '3.36',
            'Locale::Currency'      => '3.36',
            'Locale::Language'      => '3.36',
            'Locale::Script'        => '3.36',
            'Math::BigFloat::Trace' => '0.40',
            'Math::BigInt::Trace'   => '0.40',
            'Module::CoreList'      => '5.20150920',
            'Module::CoreList::TieHashDelta'=> '5.20150920',
            'Module::CoreList::Utils'=> '5.20150920',
            'OS2::DLL'              => '1.06',
            'OS2::ExtAttr'          => '0.04',
            'OS2::Process'          => '1.11',
            'OS2::REXX'             => '1.05',
            'POSIX'                 => '1.57',
            'Pod::Perldoc'          => '3.25_01',
            'Socket'                => '2.020_01',
            'Test'                  => '1.27',
            'Thread::Queue'         => '3.06',
            'Time::HiRes'           => '1.9727_02',
            'Unicode::UCD'          => '0.63',
            'Win32'                 => '0.52',
            'XS::APItest'           => '0.75',
            'bigint'                => '0.40',
            'bignum'                => '0.40',
            'bigrat'                => '0.40',
            'encoding'              => '2.17',
            'experimental'          => '0.014',
            'if'                    => '0.0605',
            'locale'                => '1.07',
            'mro'                   => '1.18',
            'threads'               => '2.03',
        },
        removed => {
        }
    },
    5.023004 => {
        delta_from => 5.023003,
        changed => {
            'B'                     => '1.60',
            'B::Op_private'         => '5.023004',
            'Compress::Raw::Bzip2'  => '2.069',
            'Compress::Raw::Zlib'   => '2.069',
            'Compress::Zlib'        => '2.069',
            'Config'                => '5.023004',
            'Devel::PPPort'         => '3.32',
            'DynaLoader'            => '1.35',
            'Encode'                => '2.78',
            'ExtUtils::CBuilder'    => '0.280224',
            'ExtUtils::CBuilder::Base'=> '0.280224',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.280224',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.280224',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.280224',
            'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280224',
            'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280224',
            'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280224',
            'ExtUtils::CBuilder::Platform::aix'=> '0.280224',
            'ExtUtils::CBuilder::Platform::android'=> '0.280224',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.280224',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.280224',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280224',
            'ExtUtils::CBuilder::Platform::os2'=> '0.280224',
            'File::Path'            => '2.12',
            'IO'                    => '1.36',
            'IO::Compress::Adapter::Bzip2'=> '2.069',
            'IO::Compress::Adapter::Deflate'=> '2.069',
            'IO::Compress::Adapter::Identity'=> '2.069',
            'IO::Compress::Base'    => '2.069',
            'IO::Compress::Base::Common'=> '2.069',
            'IO::Compress::Bzip2'   => '2.069',
            'IO::Compress::Deflate' => '2.069',
            'IO::Compress::Gzip'    => '2.069',
            'IO::Compress::Gzip::Constants'=> '2.069',
            'IO::Compress::RawDeflate'=> '2.069',
            'IO::Compress::Zip'     => '2.069',
            'IO::Compress::Zip::Constants'=> '2.069',
            'IO::Compress::Zlib::Constants'=> '2.069',
            'IO::Compress::Zlib::Extra'=> '2.069',
            'IO::Poll'              => '0.10',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.069',
            'IO::Uncompress::Adapter::Identity'=> '2.069',
            'IO::Uncompress::Adapter::Inflate'=> '2.069',
            'IO::Uncompress::AnyInflate'=> '2.069',
            'IO::Uncompress::AnyUncompress'=> '2.069',
            'IO::Uncompress::Base'  => '2.069',
            'IO::Uncompress::Bunzip2'=> '2.069',
            'IO::Uncompress::Gunzip'=> '2.069',
            'IO::Uncompress::Inflate'=> '2.069',
            'IO::Uncompress::RawInflate'=> '2.069',
            'IO::Uncompress::Unzip' => '2.069',
            'Math::BigFloat'        => '1.999704',
            'Math::BigFloat::Trace' => '0.41',
            'Math::BigInt'          => '1.999704',
            'Math::BigInt::Calc'    => '1.999704',
            'Math::BigInt::CalcEmu' => '1.999704',
            'Math::BigInt::FastCalc'=> '0.34',
            'Math::BigInt::Trace'   => '0.41',
            'Module::CoreList'      => '5.20151020',
            'Module::CoreList::TieHashDelta'=> '5.20151020',
            'Module::CoreList::Utils'=> '5.20151020',
            'Module::Metadata'      => '1.000029',
            'POSIX'                 => '1.58',
            'Perl::OSType'          => '1.009',
            'PerlIO::encoding'      => '0.22',
            'Socket'                => '2.020_02',
            'Unicode::Normalize'    => '1.21',
            'XS::APItest'           => '0.76',
            'bigint'                => '0.41',
            'bignum'                => '0.41',
            'bigrat'                => '0.41',
            'experimental'          => '0.016',
            'if'                    => '0.0606',
            'warnings'              => '1.35',
        },
        removed => {
        }
    },
    5.023005 => {
        delta_from => 5.023004,
        changed => {
            'B'                     => '1.61',
            'B::Op_private'         => '5.023005',
            'Carp'                  => '1.38',
            'Carp::Heavy'           => '1.38',
            'Config'                => '5.023005',
            'Config::Perl::V'       => '0.25',
            'Cwd'                   => '3.59',
            'Devel::Peek'           => '1.23',
            'Dumpvalue'             => '1.18',
            'DynaLoader'            => '1.36',
            'File::Find'            => '1.32',
            'File::Spec'            => '3.59',
            'File::Spec::AmigaOS'   => '3.59',
            'File::Spec::Cygwin'    => '3.59',
            'File::Spec::Epoc'      => '3.59',
            'File::Spec::Functions' => '3.59',
            'File::Spec::Mac'       => '3.59',
            'File::Spec::OS2'       => '3.59',
            'File::Spec::Unix'      => '3.59',
            'File::Spec::VMS'       => '3.59',
            'File::Spec::Win32'     => '3.59',
            'Getopt::Long'          => '2.48',
            'Hash::Util::FieldHash' => '1.18',
            'IPC::Open3'            => '1.20',
            'Math::BigFloat'        => '1.999710',
            'Math::BigInt'          => '1.999710',
            'Math::BigInt::Calc'    => '1.999710',
            'Math::BigInt::CalcEmu' => '1.999710',
            'Math::BigInt::FastCalc'=> '0.37',
            'Module::CoreList'      => '5.20151120',
            'Module::CoreList::TieHashDelta'=> '5.20151120',
            'Module::CoreList::Utils'=> '5.20151120',
            'Module::Metadata'      => '1.000030',
            'POSIX'                 => '1.59',
            'PerlIO::encoding'      => '0.23',
            'PerlIO::mmap'          => '0.015',
            'PerlIO::scalar'        => '0.24',
            'PerlIO::via'           => '0.16',
            'Pod::Simple'           => '3.32',
            'Pod::Simple::BlackBox' => '3.32',
            'Pod::Simple::Checker'  => '3.32',
            'Pod::Simple::Debug'    => '3.32',
            'Pod::Simple::DumpAsText'=> '3.32',
            'Pod::Simple::DumpAsXML'=> '3.32',
            'Pod::Simple::HTML'     => '3.32',
            'Pod::Simple::HTMLBatch'=> '3.32',
            'Pod::Simple::LinkSection'=> '3.32',
            'Pod::Simple::Methody'  => '3.32',
            'Pod::Simple::Progress' => '3.32',
            'Pod::Simple::PullParser'=> '3.32',
            'Pod::Simple::PullParserEndToken'=> '3.32',
            'Pod::Simple::PullParserStartToken'=> '3.32',
            'Pod::Simple::PullParserTextToken'=> '3.32',
            'Pod::Simple::PullParserToken'=> '3.32',
            'Pod::Simple::RTF'      => '3.32',
            'Pod::Simple::Search'   => '3.32',
            'Pod::Simple::SimpleTree'=> '3.32',
            'Pod::Simple::Text'     => '3.32',
            'Pod::Simple::TextContent'=> '3.32',
            'Pod::Simple::TiedOutFH'=> '3.32',
            'Pod::Simple::Transcode'=> '3.32',
            'Pod::Simple::TranscodeDumb'=> '3.32',
            'Pod::Simple::TranscodeSmart'=> '3.32',
            'Pod::Simple::XHTML'    => '3.32',
            'Pod::Simple::XMLOutStream'=> '3.32',
            'Thread::Queue'         => '3.07',
            'Tie::Scalar'           => '1.04',
            'Time::HiRes'           => '1.9728',
            'Time::Piece'           => '1.31',
            'Time::Seconds'         => '1.31',
            'Unicode::Normalize'    => '1.23',
            'XSLoader'              => '0.21',
            'arybase'               => '0.11',
            'base'                  => '2.22_01',
            'fields'                => '2.22_01',
            'threads'               => '2.04',
            'threads::shared'       => '1.49',
        },
        removed => {
            'ExtUtils::MakeMaker::version::vpp'=> 1,
            'version::vpp'          => 1,
        }
    },
    5.022001 => {
        delta_from => 5.022,
        changed => {
            'B::Op_private'         => '5.022001',
            'Config'                => '5.022001',
            'Module::CoreList'      => '5.20151213',
            'Module::CoreList::TieHashDelta'=> '5.20151213',
            'Module::CoreList::Utils'=> '5.20151213',
            'POSIX'                 => '1.53_01',
            'PerlIO::scalar'        => '0.23',
            'Storable'              => '2.53_01',
            'Win32'                 => '0.52',
            'warnings'              => '1.34',
        },
        removed => {
        }
    },
    5.023006 => {
        delta_from => 5.023005,
        changed => {
            'B::Deparse'            => '1.36',
            'B::Op_private'         => '5.023006',
            'Benchmark'             => '1.21',
            'CPAN::Meta::Requirements'=> '2.140',
            'CPAN::Meta::YAML'      => '0.018',
            'Config'                => '5.023006',
            'Cwd'                   => '3.60',
            'Data::Dumper'          => '2.159',
            'DynaLoader'            => '1.37',
            'File::Spec'            => '3.60',
            'File::Spec::AmigaOS'   => '3.60',
            'File::Spec::Cygwin'    => '3.60',
            'File::Spec::Epoc'      => '3.60',
            'File::Spec::Functions' => '3.60',
            'File::Spec::Mac'       => '3.60',
            'File::Spec::OS2'       => '3.60',
            'File::Spec::Unix'      => '3.60',
            'File::Spec::VMS'       => '3.60',
            'File::Spec::Win32'     => '3.60',
            'Hash::Util::FieldHash' => '1.19',
            'Locale::Codes'         => '3.37',
            'Locale::Codes::Constants'=> '3.37',
            'Locale::Codes::Country'=> '3.37',
            'Locale::Codes::Country_Codes'=> '3.37',
            'Locale::Codes::Country_Retired'=> '3.37',
            'Locale::Codes::Currency'=> '3.37',
            'Locale::Codes::Currency_Codes'=> '3.37',
            'Locale::Codes::Currency_Retired'=> '3.37',
            'Locale::Codes::LangExt'=> '3.37',
            'Locale::Codes::LangExt_Codes'=> '3.37',
            'Locale::Codes::LangExt_Retired'=> '3.37',
            'Locale::Codes::LangFam'=> '3.37',
            'Locale::Codes::LangFam_Codes'=> '3.37',
            'Locale::Codes::LangFam_Retired'=> '3.37',
            'Locale::Codes::LangVar'=> '3.37',
            'Locale::Codes::LangVar_Codes'=> '3.37',
            'Locale::Codes::LangVar_Retired'=> '3.37',
            'Locale::Codes::Language'=> '3.37',
            'Locale::Codes::Language_Codes'=> '3.37',
            'Locale::Codes::Language_Retired'=> '3.37',
            'Locale::Codes::Script' => '3.37',
            'Locale::Codes::Script_Codes'=> '3.37',
            'Locale::Codes::Script_Retired'=> '3.37',
            'Locale::Country'       => '3.37',
            'Locale::Currency'      => '3.37',
            'Locale::Language'      => '3.37',
            'Locale::Script'        => '3.37',
            'Math::BigInt::FastCalc'=> '0.38',
            'Module::CoreList'      => '5.20151220',
            'Module::CoreList::TieHashDelta'=> '5.20151220',
            'Module::CoreList::Utils'=> '5.20151220',
            'Module::Metadata'      => '1.000031',
            'Opcode'                => '1.34',
            'PerlIO::mmap'          => '0.016',
            'Pod::Perldoc'          => '3.25_02',
            'SDBM_File'             => '1.14',
            'Term::ANSIColor'       => '4.04',
            'Test'                  => '1.28',
            'Unicode::Normalize'    => '1.24',
            'XS::APItest'           => '0.77',
            'base'                  => '2.23',
            'encoding::warnings'    => '0.12',
            'fields'                => '2.23',
            'locale'                => '1.08',
            'strict'                => '1.10',
            'threads'               => '2.05',
            'threads::shared'       => '1.50',
            'utf8'                  => '1.18',
        },
        removed => {
        }
    },
    5.023007 => {
        delta_from => 5.023006,
        changed => {
            'App::Prove'            => '3.36',
            'App::Prove::State'     => '3.36',
            'App::Prove::State::Result'=> '3.36',
            'App::Prove::State::Result::Test'=> '3.36',
            'B'                     => '1.62',
            'B::Deparse'            => '1.37',
            'B::Op_private'         => '5.023007',
            'Benchmark'             => '1.22',
            'Config'                => '5.023007',
            'Cwd'                   => '3.62',
            'Data::Dumper'          => '2.160',
            'ExtUtils::ParseXS'     => '3.31',
            'ExtUtils::ParseXS::Constants'=> '3.31',
            'ExtUtils::ParseXS::CountLines'=> '3.31',
            'ExtUtils::ParseXS::Eval'=> '3.31',
            'ExtUtils::ParseXS::Utilities'=> '3.31',
            'ExtUtils::Typemaps'    => '3.31',
            'ExtUtils::Typemaps::Cmd'=> '3.31',
            'ExtUtils::Typemaps::InputMap'=> '3.31',
            'ExtUtils::Typemaps::OutputMap'=> '3.31',
            'ExtUtils::Typemaps::Type'=> '3.31',
            'File::Find'            => '1.33',
            'File::Spec'            => '3.62',
            'File::Spec::AmigaOS'   => '3.62',
            'File::Spec::Cygwin'    => '3.62',
            'File::Spec::Epoc'      => '3.62',
            'File::Spec::Functions' => '3.62',
            'File::Spec::Mac'       => '3.62',
            'File::Spec::OS2'       => '3.62',
            'File::Spec::Unix'      => '3.62',
            'File::Spec::VMS'       => '3.62',
            'File::Spec::Win32'     => '3.62',
            'Math::BigFloat'        => '1.999715',
            'Math::BigFloat::Trace' => '0.42',
            'Math::BigInt'          => '1.999715',
            'Math::BigInt::Calc'    => '1.999715',
            'Math::BigInt::CalcEmu' => '1.999715',
            'Math::BigInt::FastCalc'=> '0.40',
            'Math::BigInt::Trace'   => '0.42',
            'Math::BigRat'          => '0.260802',
            'Module::CoreList'      => '5.20160120',
            'Module::CoreList::TieHashDelta'=> '5.20160120',
            'Module::CoreList::Utils'=> '5.20160120',
            'Net::Cmd'              => '3.08',
            'Net::Config'           => '3.08',
            'Net::Domain'           => '3.08',
            'Net::FTP'              => '3.08',
            'Net::FTP::A'           => '3.08',
            'Net::FTP::E'           => '3.08',
            'Net::FTP::I'           => '3.08',
            'Net::FTP::L'           => '3.08',
            'Net::FTP::dataconn'    => '3.08',
            'Net::NNTP'             => '3.08',
            'Net::Netrc'            => '3.08',
            'Net::POP3'             => '3.08',
            'Net::SMTP'             => '3.08',
            'Net::Time'             => '3.08',
            'Pod::Man'              => '4.04',
            'Pod::ParseLink'        => '4.04',
            'Pod::Text'             => '4.04',
            'Pod::Text::Color'      => '4.04',
            'Pod::Text::Overstrike' => '4.04',
            'Pod::Text::Termcap'    => '4.04',
            'Pod::Usage'            => '1.68',
            'TAP::Base'             => '3.36',
            'TAP::Formatter::Base'  => '3.36',
            'TAP::Formatter::Color' => '3.36',
            'TAP::Formatter::Console'=> '3.36',
            'TAP::Formatter::Console::ParallelSession'=> '3.36',
            'TAP::Formatter::Console::Session'=> '3.36',
            'TAP::Formatter::File'  => '3.36',
            'TAP::Formatter::File::Session'=> '3.36',
            'TAP::Formatter::Session'=> '3.36',
            'TAP::Harness'          => '3.36',
            'TAP::Harness::Env'     => '3.36',
            'TAP::Object'           => '3.36',
            'TAP::Parser'           => '3.36',
            'TAP::Parser::Aggregator'=> '3.36',
            'TAP::Parser::Grammar'  => '3.36',
            'TAP::Parser::Iterator' => '3.36',
            'TAP::Parser::Iterator::Array'=> '3.36',
            'TAP::Parser::Iterator::Process'=> '3.36',
            'TAP::Parser::Iterator::Stream'=> '3.36',
            'TAP::Parser::IteratorFactory'=> '3.36',
            'TAP::Parser::Multiplexer'=> '3.36',
            'TAP::Parser::Result'   => '3.36',
            'TAP::Parser::Result::Bailout'=> '3.36',
            'TAP::Parser::Result::Comment'=> '3.36',
            'TAP::Parser::Result::Plan'=> '3.36',
            'TAP::Parser::Result::Pragma'=> '3.36',
            'TAP::Parser::Result::Test'=> '3.36',
            'TAP::Parser::Result::Unknown'=> '3.36',
            'TAP::Parser::Result::Version'=> '3.36',
            'TAP::Parser::Result::YAML'=> '3.36',
            'TAP::Parser::ResultFactory'=> '3.36',
            'TAP::Parser::Scheduler'=> '3.36',
            'TAP::Parser::Scheduler::Job'=> '3.36',
            'TAP::Parser::Scheduler::Spinner'=> '3.36',
            'TAP::Parser::Source'   => '3.36',
            'TAP::Parser::SourceHandler'=> '3.36',
            'TAP::Parser::SourceHandler::Executable'=> '3.36',
            'TAP::Parser::SourceHandler::File'=> '3.36',
            'TAP::Parser::SourceHandler::Handle'=> '3.36',
            'TAP::Parser::SourceHandler::Perl'=> '3.36',
            'TAP::Parser::SourceHandler::RawTAP'=> '3.36',
            'TAP::Parser::YAMLish::Reader'=> '3.36',
            'TAP::Parser::YAMLish::Writer'=> '3.36',
            'Test::Harness'         => '3.36',
            'Unicode::Normalize'    => '1.25',
            'Unicode::UCD'          => '0.64',
            'XS::APItest'           => '0.78',
            'bigint'                => '0.42',
            'bignum'                => '0.42',
            'bigrat'                => '0.42',
            'utf8'                  => '1.19',
        },
        removed => {
        }
    },
    5.023008 => {
        delta_from => 5.023007,
        changed => {
            'B::Op_private'         => '5.023008',
            'Config'                => '5.023008',
            'Cwd'                   => '3.63',
            'DynaLoader'            => '1.38',
            'Encode'                => '2.80',
            'Encode::Alias'         => '2.20',
            'Encode::MIME::Header'  => '2.19',
            'Encode::Unicode'       => '2.15',
            'ExtUtils::CBuilder'    => '0.280225',
            'ExtUtils::CBuilder::Base'=> '0.280225',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.280225',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.280225',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.280225',
            'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280225',
            'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280225',
            'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280225',
            'ExtUtils::CBuilder::Platform::aix'=> '0.280225',
            'ExtUtils::CBuilder::Platform::android'=> '0.280225',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.280225',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.280225',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280225',
            'ExtUtils::CBuilder::Platform::os2'=> '0.280225',
            'ExtUtils::Command::MM' => '7.10_01',
            'ExtUtils::Liblist'     => '7.10_01',
            'ExtUtils::Liblist::Kid'=> '7.10_01',
            'ExtUtils::MM'          => '7.10_01',
            'ExtUtils::MM_AIX'      => '7.10_01',
            'ExtUtils::MM_Any'      => '7.10_01',
            'ExtUtils::MM_BeOS'     => '7.10_01',
            'ExtUtils::MM_Cygwin'   => '7.10_01',
            'ExtUtils::MM_DOS'      => '7.10_01',
            'ExtUtils::MM_Darwin'   => '7.10_01',
            'ExtUtils::MM_MacOS'    => '7.10_01',
            'ExtUtils::MM_NW5'      => '7.10_01',
            'ExtUtils::MM_OS2'      => '7.10_01',
            'ExtUtils::MM_QNX'      => '7.10_01',
            'ExtUtils::MM_UWIN'     => '7.10_01',
            'ExtUtils::MM_Unix'     => '7.10_01',
            'ExtUtils::MM_VMS'      => '7.10_01',
            'ExtUtils::MM_VOS'      => '7.10_01',
            'ExtUtils::MM_Win32'    => '7.10_01',
            'ExtUtils::MM_Win95'    => '7.10_01',
            'ExtUtils::MY'          => '7.10_01',
            'ExtUtils::MakeMaker'   => '7.10_01',
            'ExtUtils::MakeMaker::Config'=> '7.10_01',
            'ExtUtils::MakeMaker::version'=> '7.10_01',
            'ExtUtils::MakeMaker::version::regex'=> '7.10_01',
            'ExtUtils::Mkbootstrap' => '7.10_01',
            'ExtUtils::Mksymlists'  => '7.10_01',
            'ExtUtils::testlib'     => '7.10_01',
            'File::Spec'            => '3.63',
            'File::Spec::AmigaOS'   => '3.63',
            'File::Spec::Cygwin'    => '3.63',
            'File::Spec::Epoc'      => '3.63',
            'File::Spec::Functions' => '3.63',
            'File::Spec::Mac'       => '3.63',
            'File::Spec::OS2'       => '3.63',
            'File::Spec::Unix'      => '3.63',
            'File::Spec::VMS'       => '3.63',
            'File::Spec::Win32'     => '3.63',
            'IPC::Msg'              => '2.05',
            'IPC::Semaphore'        => '2.05',
            'IPC::SharedMem'        => '2.05',
            'IPC::SysV'             => '2.05',
            'Module::CoreList'      => '5.20160121',
            'Module::CoreList::TieHashDelta'=> '5.20160121',
            'Module::CoreList::Utils'=> '5.20160121',
            'ODBM_File'             => '1.13',
            'POSIX'                 => '1.63',
            'PerlIO::encoding'      => '0.24',
            'Pod::Man'              => '4.06',
            'Pod::ParseLink'        => '4.06',
            'Pod::Text'             => '4.06',
            'Pod::Text::Color'      => '4.06',
            'Pod::Text::Overstrike' => '4.06',
            'Pod::Text::Termcap'    => '4.06',
            'Storable'              => '2.55',
            'Time::HiRes'           => '1.9730',
            'XS::APItest'           => '0.79',
        },
        removed => {
        }
    },
    5.023009 => {
        delta_from => 5.023008,
        changed => {
            'Amiga::ARexx'          => '0.04',
            'Amiga::Exec'           => '0.02',
            'B::Op_private'         => '5.023009',
            'Carp'                  => '1.40',
            'Carp::Heavy'           => '1.40',
            'Config'                => '5.023009',
            'Errno'                 => '1.25',
            'ExtUtils::Embed'       => '1.33',
            'File::Find'            => '1.34',
            'File::Glob'            => '1.26',
            'File::Spec::AmigaOS'   => ';.64',
            'IPC::Msg'              => '2.06_01',
            'IPC::Semaphore'        => '2.06_01',
            'IPC::SharedMem'        => '2.06_01',
            'IPC::SysV'             => '2.06_01',
            'List::Util'            => '1.42_02',
            'List::Util::XS'        => '1.42_02',
            'Module::CoreList'      => '5.20160320',
            'Module::CoreList::TieHashDelta'=> '5.20160320',
            'Module::CoreList::Utils'=> '5.20160320',
            'POSIX'                 => '1.64',
            'Pod::Functions'        => '1.10',
            'Scalar::Util'          => '1.42_02',
            'SelfLoader'            => '1.23',
            'Socket'                => '2.020_03',
            'Storable'              => '2.56',
            'Sub::Util'             => '1.42_02',
            'Thread::Queue'         => '3.08',
            'Tie::File'             => '1.02',
            'Time::HiRes'           => '1.9732',
            'Win32API::File'        => '0.1203',
            'XS::APItest'           => '0.80',
            'autouse'               => '1.11',
            'bytes'                 => '1.05',
            'strict'                => '1.11',
            'threads'               => '2.06',
            'version'               => '0.9916',
            'version::regex'        => '0.9916',
            'warnings'              => '1.36',
        },
        removed => {
            'Win32API::File::ExtUtils::Myconst2perl'=> 1,
        }
    },
    5.022002 => {
        delta_from => 5.022001,
        changed => {
            'B::Op_private'         => '5.022002',
            'Config'                => '5.022002',
            'Cwd'                   => '3.56_01',
            'File::Spec'            => '3.56_01',
            'File::Spec::Cygwin'    => '3.56_01',
            'File::Spec::Epoc'      => '3.56_01',
            'File::Spec::Functions' => '3.56_01',
            'File::Spec::Mac'       => '3.56_01',
            'File::Spec::OS2'       => '3.56_01',
            'File::Spec::Unix'      => '3.56_01',
            'File::Spec::VMS'       => '3.56_01',
            'File::Spec::Win32'     => '3.56_01',
            'Module::CoreList'      => '5.20160429',
            'Module::CoreList::TieHashDelta'=> '5.20160429',
            'Module::CoreList::Utils'=> '5.20160429',
            'XS::APItest'           => '0.72_01',
        },
        removed => {
        }
    },
    5.024000 => {
        delta_from => 5.023009,
        changed => {
            'B::Op_private'         => '5.024000',
            'Config'                => '5.024',
            'File::Copy'            => '2.31',
            'File::Path'            => '2.12_01',
            'File::Spec::AmigaOS'   => '3.64',
            'IO::Handle'            => '1.36',
            'Module::CoreList'      => '5.20160506',
            'Module::CoreList::TieHashDelta'=> '5.20160506',
            'Module::CoreList::Utils'=> '5.20160506',
            'ODBM_File'             => '1.14',
            'POSIX'                 => '1.65',
            'Pod::Man'              => '4.07',
            'Pod::ParseLink'        => '4.07',
            'Pod::Text'             => '4.07',
            'Pod::Text::Color'      => '4.07',
            'Pod::Text::Overstrike' => '4.07',
            'Pod::Text::Termcap'    => '4.07',
            'Thread::Queue'         => '3.09',
            'Time::HiRes'           => '1.9733',
            'threads'               => '2.07',
            'threads::shared'       => '1.51',
            'locale'                => '1.09',
        },
        removed => {
        }
    },
    5.025000 => {
        delta_from => 5.024,
        changed => {
            'B::Op_private'         => '5.025000',
            'Config'                => '5.025',
            'Module::CoreList'      => '5.20160507',
            'Module::CoreList::TieHashDelta'=> '5.20160507',
            'Module::CoreList::Utils'=> '5.20160507',
            'feature'               => '1.43',
        },
        removed => {
        }
    },
    5.025001 => {
        delta_from => 5.025,
        changed => {
            'Archive::Tar'          => '2.08',
            'Archive::Tar::Constant'=> '2.08',
            'Archive::Tar::File'    => '2.08',
            'B::Op_private'         => '5.025001',
            'Carp'                  => '1.41',
            'Carp::Heavy'           => '1.41',
            'Config'                => '5.025001',
            'Config::Perl::V'       => '0.26',
            'DB_File'               => '1.838',
            'Digest::MD5'           => '2.55',
            'IPC::Cmd'              => '0.94',
            'IPC::Msg'              => '2.07',
            'IPC::Semaphore'        => '2.07',
            'IPC::SharedMem'        => '2.07',
            'IPC::SysV'             => '2.07',
            'List::Util'            => '1.45_01',
            'List::Util::XS'        => '1.45_01',
            'Locale::Codes'         => '3.38',
            'Locale::Codes::Constants'=> '3.38',
            'Locale::Codes::Country'=> '3.38',
            'Locale::Codes::Country_Codes'=> '3.38',
            'Locale::Codes::Country_Retired'=> '3.38',
            'Locale::Codes::Currency'=> '3.38',
            'Locale::Codes::Currency_Codes'=> '3.38',
            'Locale::Codes::Currency_Retired'=> '3.38',
            'Locale::Codes::LangExt'=> '3.38',
            'Locale::Codes::LangExt_Codes'=> '3.38',
            'Locale::Codes::LangExt_Retired'=> '3.38',
            'Locale::Codes::LangFam'=> '3.38',
            'Locale::Codes::LangFam_Codes'=> '3.38',
            'Locale::Codes::LangFam_Retired'=> '3.38',
            'Locale::Codes::LangVar'=> '3.38',
            'Locale::Codes::LangVar_Codes'=> '3.38',
            'Locale::Codes::LangVar_Retired'=> '3.38',
            'Locale::Codes::Language'=> '3.38',
            'Locale::Codes::Language_Codes'=> '3.38',
            'Locale::Codes::Language_Retired'=> '3.38',
            'Locale::Codes::Script' => '3.38',
            'Locale::Codes::Script_Codes'=> '3.38',
            'Locale::Codes::Script_Retired'=> '3.38',
            'Locale::Country'       => '3.38',
            'Locale::Currency'      => '3.38',
            'Locale::Language'      => '3.38',
            'Locale::Maketext'      => '1.27',
            'Locale::Script'        => '3.38',
            'Module::CoreList'      => '5.20160520',
            'Module::CoreList::TieHashDelta'=> '5.20160520',
            'Module::CoreList::Utils'=> '5.20160520',
            'Module::Metadata'      => '1.000032',
            'POSIX'                 => '1.69',
            'Scalar::Util'          => '1.45_01',
            'Sub::Util'             => '1.45_01',
            'Sys::Syslog'           => '0.34',
            'Term::ANSIColor'       => '4.05',
            'Test2'                 => '1.302015',
            'Test2::API'            => '1.302015',
            'Test2::API::Breakage'  => '1.302015',
            'Test2::API::Context'   => '1.302015',
            'Test2::API::Instance'  => '1.302015',
            'Test2::API::Stack'     => '1.302015',
            'Test2::Event'          => '1.302015',
            'Test2::Event::Bail'    => '1.302015',
            'Test2::Event::Diag'    => '1.302015',
            'Test2::Event::Exception'=> '1.302015',
            'Test2::Event::Note'    => '1.302015',
            'Test2::Event::Ok'      => '1.302015',
            'Test2::Event::Plan'    => '1.302015',
            'Test2::Event::Skip'    => '1.302015',
            'Test2::Event::Subtest' => '1.302015',
            'Test2::Event::Waiting' => '1.302015',
            'Test2::Formatter'      => '1.302015',
            'Test2::Formatter::TAP' => '1.302015',
            'Test2::Hub'            => '1.302015',
            'Test2::Hub::Interceptor'=> '1.302015',
            'Test2::Hub::Interceptor::Terminator'=> '1.302015',
            'Test2::Hub::Subtest'   => '1.302015',
            'Test2::IPC'            => '1.302015',
            'Test2::IPC::Driver'    => '1.302015',
            'Test2::IPC::Driver::Files'=> '1.302015',
            'Test2::Util'           => '1.302015',
            'Test2::Util::ExternalMeta'=> '1.302015',
            'Test2::Util::HashBase' => '1.302015',
            'Test2::Util::Trace'    => '1.302015',
            'Test::Builder'         => '1.302015',
            'Test::Builder::Formatter'=> '1.302015',
            'Test::Builder::Module' => '1.302015',
            'Test::Builder::Tester' => '1.302015',
            'Test::Builder::Tester::Color'=> '1.302015',
            'Test::Builder::TodoDiag'=> '1.302015',
            'Test::More'            => '1.302015',
            'Test::Simple'          => '1.302015',
            'Test::Tester'          => '1.302015',
            'Test::Tester::Capture' => '1.302015',
            'Test::Tester::CaptureRunner'=> '1.302015',
            'Test::Tester::Delegate'=> '1.302015',
            'Test::use::ok'         => '1.302015',
            'XS::APItest'           => '0.81',
            '_charnames'            => '1.44',
            'charnames'             => '1.44',
            'ok'                    => '1.302015',
            'perlfaq'               => '5.021011',
            're'                    => '0.33',
            'threads'               => '2.08',
            'threads::shared'       => '1.52',
        },
        removed => {
        }
    },
    5.025002 => {
        delta_from => 5.025001,
        changed => {
            'App::Cpan'             => '1.64',
            'B::Op_private'         => '5.025002',
            'CPAN'                  => '2.14',
            'CPAN::Distribution'    => '2.12',
            'CPAN::FTP'             => '5.5007',
            'CPAN::FirstTime'       => '5.5309',
            'CPAN::HandleConfig'    => '5.5007',
            'CPAN::Index'           => '2.12',
            'CPAN::Mirrors'         => '2.12',
            'CPAN::Plugin'          => '0.96',
            'CPAN::Shell'           => '5.5006',
            'Config'                => '5.025002',
            'Cwd'                   => '3.64',
            'Devel::Peek'           => '1.24',
            'DynaLoader'            => '1.39',
            'ExtUtils::Command'     => '7.18',
            'ExtUtils::Command::MM' => '7.18',
            'ExtUtils::Liblist'     => '7.18',
            'ExtUtils::Liblist::Kid'=> '7.18',
            'ExtUtils::MM'          => '7.18',
            'ExtUtils::MM_AIX'      => '7.18',
            'ExtUtils::MM_Any'      => '7.18',
            'ExtUtils::MM_BeOS'     => '7.18',
            'ExtUtils::MM_Cygwin'   => '7.18',
            'ExtUtils::MM_DOS'      => '7.18',
            'ExtUtils::MM_Darwin'   => '7.18',
            'ExtUtils::MM_MacOS'    => '7.18',
            'ExtUtils::MM_NW5'      => '7.18',
            'ExtUtils::MM_OS2'      => '7.18',
            'ExtUtils::MM_QNX'      => '7.18',
            'ExtUtils::MM_UWIN'     => '7.18',
            'ExtUtils::MM_Unix'     => '7.18',
            'ExtUtils::MM_VMS'      => '7.18',
            'ExtUtils::MM_VOS'      => '7.18',
            'ExtUtils::MM_Win32'    => '7.18',
            'ExtUtils::MM_Win95'    => '7.18',
            'ExtUtils::MY'          => '7.18',
            'ExtUtils::MakeMaker'   => '7.18',
            'ExtUtils::MakeMaker::Config'=> '7.18',
            'ExtUtils::MakeMaker::Locale'=> '7.18',
            'ExtUtils::MakeMaker::version'=> '7.18',
            'ExtUtils::MakeMaker::version::regex'=> '7.18',
            'ExtUtils::Miniperl'    => '1.06',
            'ExtUtils::Mkbootstrap' => '7.18',
            'ExtUtils::Mksymlists'  => '7.18',
            'ExtUtils::ParseXS'     => '3.32',
            'ExtUtils::ParseXS::Constants'=> '3.32',
            'ExtUtils::ParseXS::CountLines'=> '3.32',
            'ExtUtils::ParseXS::Eval'=> '3.32',
            'ExtUtils::ParseXS::Utilities'=> '3.32',
            'ExtUtils::Typemaps'    => '3.32',
            'ExtUtils::Typemaps::Cmd'=> '3.32',
            'ExtUtils::Typemaps::InputMap'=> '3.32',
            'ExtUtils::Typemaps::OutputMap'=> '3.32',
            'ExtUtils::Typemaps::Type'=> '3.32',
            'ExtUtils::testlib'     => '7.18',
            'File::Copy'            => '2.32',
            'File::Glob'            => '1.27',
            'File::Spec'            => '3.64',
            'File::Spec::Cygwin'    => '3.64',
            'File::Spec::Epoc'      => '3.64',
            'File::Spec::Functions' => '3.64',
            'File::Spec::Mac'       => '3.64',
            'File::Spec::OS2'       => '3.64',
            'File::Spec::Unix'      => '3.64',
            'File::Spec::VMS'       => '3.64',
            'File::Spec::Win32'     => '3.64',
            'FileHandle'            => '2.03',
            'Getopt::Long'          => '2.49',
            'HTTP::Tiny'            => '0.058',
            'JSON::PP'              => '2.27400',
            'Locale::Codes'         => '3.39',
            'Locale::Codes::Constants'=> '3.39',
            'Locale::Codes::Country'=> '3.39',
            'Locale::Codes::Country_Codes'=> '3.39',
            'Locale::Codes::Country_Retired'=> '3.39',
            'Locale::Codes::Currency'=> '3.39',
            'Locale::Codes::Currency_Codes'=> '3.39',
            'Locale::Codes::Currency_Retired'=> '3.39',
            'Locale::Codes::LangExt'=> '3.39',
            'Locale::Codes::LangExt_Codes'=> '3.39',
            'Locale::Codes::LangExt_Retired'=> '3.39',
            'Locale::Codes::LangFam'=> '3.39',
            'Locale::Codes::LangFam_Codes'=> '3.39',
            'Locale::Codes::LangFam_Retired'=> '3.39',
            'Locale::Codes::LangVar'=> '3.39',
            'Locale::Codes::LangVar_Codes'=> '3.39',
            'Locale::Codes::LangVar_Retired'=> '3.39',
            'Locale::Codes::Language'=> '3.39',
            'Locale::Codes::Language_Codes'=> '3.39',
            'Locale::Codes::Language_Retired'=> '3.39',
            'Locale::Codes::Script' => '3.39',
            'Locale::Codes::Script_Codes'=> '3.39',
            'Locale::Codes::Script_Retired'=> '3.39',
            'Locale::Country'       => '3.39',
            'Locale::Currency'      => '3.39',
            'Locale::Language'      => '3.39',
            'Locale::Script'        => '3.39',
            'Module::CoreList'      => '5.20160620',
            'Module::CoreList::TieHashDelta'=> '5.20160620',
            'Module::CoreList::Utils'=> '5.20160620',
            'Opcode'                => '1.35',
            'POSIX'                 => '1.70',
            'Pod::Checker'          => '1.73',
            'Pod::Functions'        => '1.11',
            'Pod::Usage'            => '1.69',
            'Test2'                 => '1.302026',
            'Test2::API'            => '1.302026',
            'Test2::API::Breakage'  => '1.302026',
            'Test2::API::Context'   => '1.302026',
            'Test2::API::Instance'  => '1.302026',
            'Test2::API::Stack'     => '1.302026',
            'Test2::Event'          => '1.302026',
            'Test2::Event::Bail'    => '1.302026',
            'Test2::Event::Diag'    => '1.302026',
            'Test2::Event::Exception'=> '1.302026',
            'Test2::Event::Generic' => '1.302026',
            'Test2::Event::Note'    => '1.302026',
            'Test2::Event::Ok'      => '1.302026',
            'Test2::Event::Plan'    => '1.302026',
            'Test2::Event::Skip'    => '1.302026',
            'Test2::Event::Subtest' => '1.302026',
            'Test2::Event::Waiting' => '1.302026',
            'Test2::Formatter'      => '1.302026',
            'Test2::Formatter::TAP' => '1.302026',
            'Test2::Hub'            => '1.302026',
            'Test2::Hub::Interceptor'=> '1.302026',
            'Test2::Hub::Interceptor::Terminator'=> '1.302026',
            'Test2::Hub::Subtest'   => '1.302026',
            'Test2::IPC'            => '1.302026',
            'Test2::IPC::Driver'    => '1.302026',
            'Test2::IPC::Driver::Files'=> '1.302026',
            'Test2::Util'           => '1.302026',
            'Test2::Util::ExternalMeta'=> '1.302026',
            'Test2::Util::HashBase' => '1.302026',
            'Test2::Util::Trace'    => '1.302026',
            'Test::Builder'         => '1.302026',
            'Test::Builder::Formatter'=> '1.302026',
            'Test::Builder::Module' => '1.302026',
            'Test::Builder::Tester' => '1.302026',
            'Test::Builder::Tester::Color'=> '1.302026',
            'Test::Builder::TodoDiag'=> '1.302026',
            'Test::More'            => '1.302026',
            'Test::Simple'          => '1.302026',
            'Test::Tester'          => '1.302026',
            'Test::Tester::Capture' => '1.302026',
            'Test::Tester::CaptureRunner'=> '1.302026',
            'Test::Tester::Delegate'=> '1.302026',
            'Test::use::ok'         => '1.302026',
            'Thread::Queue'         => '3.11',
            'Time::HiRes'           => '1.9734',
            'Unicode::UCD'          => '0.65',
            'VMS::DCLsym'           => '1.07',
            'XS::APItest'           => '0.82',
            'diagnostics'           => '1.35',
            'feature'               => '1.44',
            'ok'                    => '1.302026',
            'threads'               => '2.09',
        },
        removed => {
        }
    },
    5.025003 => {
        delta_from => 5.025002,
        changed => {
            'B::Op_private'         => '5.025003',
            'Config'                => '5.025003',
            'Data::Dumper'          => '2.161',
            'Devel::PPPort'         => '3.35',
            'Encode'                => '2.84',
            'Encode::MIME::Header'  => '2.23',
            'Encode::MIME::Header::ISO_2022_JP'=> '1.07',
            'ExtUtils::ParseXS'     => '3.33',
            'ExtUtils::ParseXS::Constants'=> '3.33',
            'ExtUtils::ParseXS::CountLines'=> '3.33',
            'ExtUtils::ParseXS::Eval'=> '3.33',
            'ExtUtils::ParseXS::Utilities'=> '3.33',
            'ExtUtils::Typemaps'    => '3.33',
            'ExtUtils::Typemaps::Cmd'=> '3.33',
            'ExtUtils::Typemaps::InputMap'=> '3.33',
            'ExtUtils::Typemaps::OutputMap'=> '3.33',
            'ExtUtils::Typemaps::Type'=> '3.33',
            'Hash::Util'            => '0.20',
            'Math::BigFloat'        => '1.999726',
            'Math::BigFloat::Trace' => '0.43',
            'Math::BigInt'          => '1.999726',
            'Math::BigInt::Calc'    => '1.999726',
            'Math::BigInt::CalcEmu' => '1.999726',
            'Math::BigInt::FastCalc'=> '0.42',
            'Math::BigInt::Trace'   => '0.43',
            'Math::BigRat'          => '0.260804',
            'Module::CoreList'      => '5.20160720',
            'Module::CoreList::TieHashDelta'=> '5.20160720',
            'Module::CoreList::Utils'=> '5.20160720',
            'Net::Cmd'              => '3.09',
            'Net::Config'           => '3.09',
            'Net::Domain'           => '3.09',
            'Net::FTP'              => '3.09',
            'Net::FTP::A'           => '3.09',
            'Net::FTP::E'           => '3.09',
            'Net::FTP::I'           => '3.09',
            'Net::FTP::L'           => '3.09',
            'Net::FTP::dataconn'    => '3.09',
            'Net::NNTP'             => '3.09',
            'Net::Netrc'            => '3.09',
            'Net::POP3'             => '3.09',
            'Net::SMTP'             => '3.09',
            'Net::Time'             => '3.09',
            'Parse::CPAN::Meta'     => '1.4422',
            'Perl::OSType'          => '1.010',
            'Test2'                 => '1.302045',
            'Test2::API'            => '1.302045',
            'Test2::API::Breakage'  => '1.302045',
            'Test2::API::Context'   => '1.302045',
            'Test2::API::Instance'  => '1.302045',
            'Test2::API::Stack'     => '1.302045',
            'Test2::Event'          => '1.302045',
            'Test2::Event::Bail'    => '1.302045',
            'Test2::Event::Diag'    => '1.302045',
            'Test2::Event::Exception'=> '1.302045',
            'Test2::Event::Generic' => '1.302045',
            'Test2::Event::Info'    => '1.302045',
            'Test2::Event::Note'    => '1.302045',
            'Test2::Event::Ok'      => '1.302045',
            'Test2::Event::Plan'    => '1.302045',
            'Test2::Event::Skip'    => '1.302045',
            'Test2::Event::Subtest' => '1.302045',
            'Test2::Event::Waiting' => '1.302045',
            'Test2::Formatter'      => '1.302045',
            'Test2::Formatter::TAP' => '1.302045',
            'Test2::Hub'            => '1.302045',
            'Test2::Hub::Interceptor'=> '1.302045',
            'Test2::Hub::Interceptor::Terminator'=> '1.302045',
            'Test2::Hub::Subtest'   => '1.302045',
            'Test2::IPC'            => '1.302045',
            'Test2::IPC::Driver'    => '1.302045',
            'Test2::IPC::Driver::Files'=> '1.302045',
            'Test2::Util'           => '1.302045',
            'Test2::Util::ExternalMeta'=> '1.302045',
            'Test2::Util::HashBase' => '1.302045',
            'Test2::Util::Trace'    => '1.302045',
            'Test::Builder'         => '1.302045',
            'Test::Builder::Formatter'=> '1.302045',
            'Test::Builder::Module' => '1.302045',
            'Test::Builder::Tester' => '1.302045',
            'Test::Builder::Tester::Color'=> '1.302045',
            'Test::Builder::TodoDiag'=> '1.302045',
            'Test::More'            => '1.302045',
            'Test::Simple'          => '1.302045',
            'Test::Tester'          => '1.302045',
            'Test::Tester::Capture' => '1.302045',
            'Test::Tester::CaptureRunner'=> '1.302045',
            'Test::Tester::Delegate'=> '1.302045',
            'Test::use::ok'         => '1.302045',
            'Time::HiRes'           => '1.9739',
            'Unicode'               => '9.0.0',
            'Unicode::UCD'          => '0.66',
            'XSLoader'              => '0.22',
            'bigint'                => '0.43',
            'bignum'                => '0.43',
            'bigrat'                => '0.43',
            'encoding'              => '2.17_01',
            'encoding::warnings'    => '0.13',
            'feature'               => '1.45',
            'ok'                    => '1.302045',
            'version'               => '0.9917',
            'version::regex'        => '0.9917',
            'warnings'              => '1.37',
        },
        removed => {
        }
    },
    5.025004 => {
        delta_from => 5.025003,
        changed => {
            'App::Cpan'             => '1.64_01',
            'App::Prove'            => '3.36_01',
            'App::Prove::State'     => '3.36_01',
            'App::Prove::State::Result'=> '3.36_01',
            'App::Prove::State::Result::Test'=> '3.36_01',
            'Archive::Tar'          => '2.10',
            'Archive::Tar::Constant'=> '2.10',
            'Archive::Tar::File'    => '2.10',
            'B'                     => '1.63',
            'B::Concise'            => '0.998',
            'B::Deparse'            => '1.38',
            'B::Op_private'         => '5.025004',
            'CPAN'                  => '2.14_01',
            'CPAN::Meta'            => '2.150010',
            'CPAN::Meta::Converter' => '2.150010',
            'CPAN::Meta::Feature'   => '2.150010',
            'CPAN::Meta::History'   => '2.150010',
            'CPAN::Meta::Merge'     => '2.150010',
            'CPAN::Meta::Prereqs'   => '2.150010',
            'CPAN::Meta::Spec'      => '2.150010',
            'CPAN::Meta::Validator' => '2.150010',
            'Carp'                  => '1.42',
            'Carp::Heavy'           => '1.42',
            'Compress::Zlib'        => '2.069_01',
            'Config'                => '5.025004',
            'Config::Perl::V'       => '0.27',
            'Cwd'                   => '3.65',
            'Digest'                => '1.17_01',
            'Digest::SHA'           => '5.96',
            'Encode'                => '2.86',
            'Errno'                 => '1.26',
            'ExtUtils::Command'     => '7.24',
            'ExtUtils::Command::MM' => '7.24',
            'ExtUtils::Liblist'     => '7.24',
            'ExtUtils::Liblist::Kid'=> '7.24',
            'ExtUtils::MM'          => '7.24',
            'ExtUtils::MM_AIX'      => '7.24',
            'ExtUtils::MM_Any'      => '7.24',
            'ExtUtils::MM_BeOS'     => '7.24',
            'ExtUtils::MM_Cygwin'   => '7.24',
            'ExtUtils::MM_DOS'      => '7.24',
            'ExtUtils::MM_Darwin'   => '7.24',
            'ExtUtils::MM_MacOS'    => '7.24',
            'ExtUtils::MM_NW5'      => '7.24',
            'ExtUtils::MM_OS2'      => '7.24',
            'ExtUtils::MM_QNX'      => '7.24',
            'ExtUtils::MM_UWIN'     => '7.24',
            'ExtUtils::MM_Unix'     => '7.24',
            'ExtUtils::MM_VMS'      => '7.24',
            'ExtUtils::MM_VOS'      => '7.24',
            'ExtUtils::MM_Win32'    => '7.24',
            'ExtUtils::MM_Win95'    => '7.24',
            'ExtUtils::MY'          => '7.24',
            'ExtUtils::MakeMaker'   => '7.24',
            'ExtUtils::MakeMaker::Config'=> '7.24',
            'ExtUtils::MakeMaker::Locale'=> '7.24',
            'ExtUtils::MakeMaker::version'=> '7.24',
            'ExtUtils::MakeMaker::version::regex'=> '7.24',
            'ExtUtils::Mkbootstrap' => '7.24',
            'ExtUtils::Mksymlists'  => '7.24',
            'ExtUtils::testlib'     => '7.24',
            'File::Fetch'           => '0.52',
            'File::Spec'            => '3.65',
            'File::Spec::AmigaOS'   => '3.65',
            'File::Spec::Cygwin'    => '3.65',
            'File::Spec::Epoc'      => '3.65',
            'File::Spec::Functions' => '3.65',
            'File::Spec::Mac'       => '3.65',
            'File::Spec::OS2'       => '3.65',
            'File::Spec::Unix'      => '3.65',
            'File::Spec::VMS'       => '3.65',
            'File::Spec::Win32'     => '3.65',
            'HTTP::Tiny'            => '0.064',
            'Hash::Util'            => '0.21',
            'I18N::LangTags'        => '0.41',
            'I18N::LangTags::Detect'=> '1.06',
            'IO'                    => '1.37',
            'IO::Compress::Adapter::Bzip2'=> '2.069_01',
            'IO::Compress::Adapter::Deflate'=> '2.069_01',
            'IO::Compress::Adapter::Identity'=> '2.069_01',
            'IO::Compress::Base'    => '2.069_01',
            'IO::Compress::Base::Common'=> '2.069_01',
            'IO::Compress::Bzip2'   => '2.069_01',
            'IO::Compress::Deflate' => '2.069_01',
            'IO::Compress::Gzip'    => '2.069_01',
            'IO::Compress::Gzip::Constants'=> '2.069_01',
            'IO::Compress::RawDeflate'=> '2.069_01',
            'IO::Compress::Zip'     => '2.069_01',
            'IO::Compress::Zip::Constants'=> '2.069_01',
            'IO::Compress::Zlib::Constants'=> '2.069_01',
            'IO::Compress::Zlib::Extra'=> '2.069_01',
            'IO::Socket::IP'        => '0.38',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.069_01',
            'IO::Uncompress::Adapter::Identity'=> '2.069_01',
            'IO::Uncompress::Adapter::Inflate'=> '2.069_01',
            'IO::Uncompress::AnyInflate'=> '2.069_01',
            'IO::Uncompress::AnyUncompress'=> '2.069_01',
            'IO::Uncompress::Base'  => '2.069_01',
            'IO::Uncompress::Bunzip2'=> '2.069_01',
            'IO::Uncompress::Gunzip'=> '2.069_01',
            'IO::Uncompress::Inflate'=> '2.069_01',
            'IO::Uncompress::RawInflate'=> '2.069_01',
            'IO::Uncompress::Unzip' => '2.069_01',
            'IPC::Cmd'              => '0.96',
            'JSON::PP'              => '2.27400_01',
            'Locale::Maketext'      => '1.28',
            'Locale::Maketext::Simple'=> '0.21_01',
            'Math::BigFloat::Trace' => '0.43_01',
            'Math::BigInt::Trace'   => '0.43_01',
            'Memoize'               => '1.03_01',
            'Module::CoreList'      => '5.20160820',
            'Module::CoreList::TieHashDelta'=> '5.20160820',
            'Module::CoreList::Utils'=> '5.20160820',
            'Module::Load::Conditional'=> '0.68',
            'Module::Metadata'      => '1.000033',
            'NEXT'                  => '0.67',
            'Net::Cmd'              => '3.10',
            'Net::Config'           => '3.10',
            'Net::Domain'           => '3.10',
            'Net::FTP'              => '3.10',
            'Net::FTP::A'           => '3.10',
            'Net::FTP::E'           => '3.10',
            'Net::FTP::I'           => '3.10',
            'Net::FTP::L'           => '3.10',
            'Net::FTP::dataconn'    => '3.10',
            'Net::NNTP'             => '3.10',
            'Net::Netrc'            => '3.10',
            'Net::POP3'             => '3.10',
            'Net::Ping'             => '2.44',
            'Net::SMTP'             => '3.10',
            'Net::Time'             => '3.10',
            'Opcode'                => '1.37',
            'POSIX'                 => '1.71',
            'Parse::CPAN::Meta'     => '2.150010',
            'Pod::Html'             => '1.2201',
            'Pod::Perldoc'          => '3.27',
            'Pod::Perldoc::BaseTo'  => '3.27',
            'Pod::Perldoc::GetOptsOO'=> '3.27',
            'Pod::Perldoc::ToANSI'  => '3.27',
            'Pod::Perldoc::ToChecker'=> '3.27',
            'Pod::Perldoc::ToMan'   => '3.27',
            'Pod::Perldoc::ToNroff' => '3.27',
            'Pod::Perldoc::ToPod'   => '3.27',
            'Pod::Perldoc::ToRtf'   => '3.27',
            'Pod::Perldoc::ToTerm'  => '3.27',
            'Pod::Perldoc::ToText'  => '3.27',
            'Pod::Perldoc::ToTk'    => '3.27',
            'Pod::Perldoc::ToXml'   => '3.27',
            'Storable'              => '2.57',
            'Sys::Syslog'           => '0.34_01',
            'TAP::Base'             => '3.36_01',
            'TAP::Formatter::Base'  => '3.36_01',
            'TAP::Formatter::Color' => '3.36_01',
            'TAP::Formatter::Console'=> '3.36_01',
            'TAP::Formatter::Console::ParallelSession'=> '3.36_01',
            'TAP::Formatter::Console::Session'=> '3.36_01',
            'TAP::Formatter::File'  => '3.36_01',
            'TAP::Formatter::File::Session'=> '3.36_01',
            'TAP::Formatter::Session'=> '3.36_01',
            'TAP::Harness'          => '3.36_01',
            'TAP::Harness::Env'     => '3.36_01',
            'TAP::Object'           => '3.36_01',
            'TAP::Parser'           => '3.36_01',
            'TAP::Parser::Aggregator'=> '3.36_01',
            'TAP::Parser::Grammar'  => '3.36_01',
            'TAP::Parser::Iterator' => '3.36_01',
            'TAP::Parser::Iterator::Array'=> '3.36_01',
            'TAP::Parser::Iterator::Process'=> '3.36_01',
            'TAP::Parser::Iterator::Stream'=> '3.36_01',
            'TAP::Parser::IteratorFactory'=> '3.36_01',
            'TAP::Parser::Multiplexer'=> '3.36_01',
            'TAP::Parser::Result'   => '3.36_01',
            'TAP::Parser::Result::Bailout'=> '3.36_01',
            'TAP::Parser::Result::Comment'=> '3.36_01',
            'TAP::Parser::Result::Plan'=> '3.36_01',
            'TAP::Parser::Result::Pragma'=> '3.36_01',
            'TAP::Parser::Result::Test'=> '3.36_01',
            'TAP::Parser::Result::Unknown'=> '3.36_01',
            'TAP::Parser::Result::Version'=> '3.36_01',
            'TAP::Parser::Result::YAML'=> '3.36_01',
            'TAP::Parser::ResultFactory'=> '3.36_01',
            'TAP::Parser::Scheduler'=> '3.36_01',
            'TAP::Parser::Scheduler::Job'=> '3.36_01',
            'TAP::Parser::Scheduler::Spinner'=> '3.36_01',
            'TAP::Parser::Source'   => '3.36_01',
            'TAP::Parser::SourceHandler'=> '3.36_01',
            'TAP::Parser::SourceHandler::Executable'=> '3.36_01',
            'TAP::Parser::SourceHandler::File'=> '3.36_01',
            'TAP::Parser::SourceHandler::Handle'=> '3.36_01',
            'TAP::Parser::SourceHandler::Perl'=> '3.36_01',
            'TAP::Parser::SourceHandler::RawTAP'=> '3.36_01',
            'TAP::Parser::YAMLish::Reader'=> '3.36_01',
            'TAP::Parser::YAMLish::Writer'=> '3.36_01',
            'Test'                  => '1.29',
            'Test2'                 => '1.302052',
            'Test2::API'            => '1.302052',
            'Test2::API::Breakage'  => '1.302052',
            'Test2::API::Context'   => '1.302052',
            'Test2::API::Instance'  => '1.302052',
            'Test2::API::Stack'     => '1.302052',
            'Test2::Event'          => '1.302052',
            'Test2::Event::Bail'    => '1.302052',
            'Test2::Event::Diag'    => '1.302052',
            'Test2::Event::Exception'=> '1.302052',
            'Test2::Event::Generic' => '1.302052',
            'Test2::Event::Info'    => '1.302052',
            'Test2::Event::Note'    => '1.302052',
            'Test2::Event::Ok'      => '1.302052',
            'Test2::Event::Plan'    => '1.302052',
            'Test2::Event::Skip'    => '1.302052',
            'Test2::Event::Subtest' => '1.302052',
            'Test2::Event::Waiting' => '1.302052',
            'Test2::Formatter'      => '1.302052',
            'Test2::Formatter::TAP' => '1.302052',
            'Test2::Hub'            => '1.302052',
            'Test2::Hub::Interceptor'=> '1.302052',
            'Test2::Hub::Interceptor::Terminator'=> '1.302052',
            'Test2::Hub::Subtest'   => '1.302052',
            'Test2::IPC'            => '1.302052',
            'Test2::IPC::Driver'    => '1.302052',
            'Test2::IPC::Driver::Files'=> '1.302052',
            'Test2::Util'           => '1.302052',
            'Test2::Util::ExternalMeta'=> '1.302052',
            'Test2::Util::HashBase' => '1.302052',
            'Test2::Util::Trace'    => '1.302052',
            'Test::Builder'         => '1.302052',
            'Test::Builder::Formatter'=> '1.302052',
            'Test::Builder::Module' => '1.302052',
            'Test::Builder::Tester' => '1.302052',
            'Test::Builder::Tester::Color'=> '1.302052',
            'Test::Builder::TodoDiag'=> '1.302052',
            'Test::Harness'         => '3.36_01',
            'Test::More'            => '1.302052',
            'Test::Simple'          => '1.302052',
            'Test::Tester'          => '1.302052',
            'Test::Tester::Capture' => '1.302052',
            'Test::Tester::CaptureRunner'=> '1.302052',
            'Test::Tester::Delegate'=> '1.302052',
            'Test::use::ok'         => '1.302052',
            'Tie::Hash::NamedCapture'=> '0.10',
            'Time::Local'           => '1.24',
            'XS::APItest'           => '0.83',
            'arybase'               => '0.12',
            'base'                  => '2.24',
            'bigint'                => '0.43_01',
            'bignum'                => '0.43_01',
            'bigrat'                => '0.43_01',
            'encoding'              => '2.18',
            'ok'                    => '1.302052',
        },
        removed => {
        }
    },
    5.025005 => {
        delta_from => 5.025004,
        changed => {
            'B::Op_private'         => '5.025005',
            'Config'                => '5.025005',
            'Filter::Simple'        => '0.93',
            'Locale::Codes'         => '3.40',
            'Locale::Codes::Constants'=> '3.40',
            'Locale::Codes::Country'=> '3.40',
            'Locale::Codes::Country_Codes'=> '3.40',
            'Locale::Codes::Country_Retired'=> '3.40',
            'Locale::Codes::Currency'=> '3.40',
            'Locale::Codes::Currency_Codes'=> '3.40',
            'Locale::Codes::Currency_Retired'=> '3.40',
            'Locale::Codes::LangExt'=> '3.40',
            'Locale::Codes::LangExt_Codes'=> '3.40',
            'Locale::Codes::LangExt_Retired'=> '3.40',
            'Locale::Codes::LangFam'=> '3.40',
            'Locale::Codes::LangFam_Codes'=> '3.40',
            'Locale::Codes::LangFam_Retired'=> '3.40',
            'Locale::Codes::LangVar'=> '3.40',
            'Locale::Codes::LangVar_Codes'=> '3.40',
            'Locale::Codes::LangVar_Retired'=> '3.40',
            'Locale::Codes::Language'=> '3.40',
            'Locale::Codes::Language_Codes'=> '3.40',
            'Locale::Codes::Language_Retired'=> '3.40',
            'Locale::Codes::Script' => '3.40',
            'Locale::Codes::Script_Codes'=> '3.40',
            'Locale::Codes::Script_Retired'=> '3.40',
            'Locale::Country'       => '3.40',
            'Locale::Currency'      => '3.40',
            'Locale::Language'      => '3.40',
            'Locale::Script'        => '3.40',
            'Module::CoreList'      => '5.20160920',
            'Module::CoreList::TieHashDelta'=> '5.20160920',
            'Module::CoreList::Utils'=> '5.20160920',
            'POSIX'                 => '1.72',
            'Sys::Syslog'           => '0.35',
            'Test2'                 => '1.302056',
            'Test2::API'            => '1.302056',
            'Test2::API::Breakage'  => '1.302056',
            'Test2::API::Context'   => '1.302056',
            'Test2::API::Instance'  => '1.302056',
            'Test2::API::Stack'     => '1.302056',
            'Test2::Event'          => '1.302056',
            'Test2::Event::Bail'    => '1.302056',
            'Test2::Event::Diag'    => '1.302056',
            'Test2::Event::Exception'=> '1.302056',
            'Test2::Event::Generic' => '1.302056',
            'Test2::Event::Info'    => '1.302056',
            'Test2::Event::Note'    => '1.302056',
            'Test2::Event::Ok'      => '1.302056',
            'Test2::Event::Plan'    => '1.302056',
            'Test2::Event::Skip'    => '1.302056',
            'Test2::Event::Subtest' => '1.302056',
            'Test2::Event::Waiting' => '1.302056',
            'Test2::Formatter'      => '1.302056',
            'Test2::Formatter::TAP' => '1.302056',
            'Test2::Hub'            => '1.302056',
            'Test2::Hub::Interceptor'=> '1.302056',
            'Test2::Hub::Interceptor::Terminator'=> '1.302056',
            'Test2::Hub::Subtest'   => '1.302056',
            'Test2::IPC'            => '1.302056',
            'Test2::IPC::Driver'    => '1.302056',
            'Test2::IPC::Driver::Files'=> '1.302056',
            'Test2::Util'           => '1.302056',
            'Test2::Util::ExternalMeta'=> '1.302056',
            'Test2::Util::HashBase' => '1.302056',
            'Test2::Util::Trace'    => '1.302056',
            'Test::Builder'         => '1.302056',
            'Test::Builder::Formatter'=> '1.302056',
            'Test::Builder::Module' => '1.302056',
            'Test::Builder::Tester' => '1.302056',
            'Test::Builder::Tester::Color'=> '1.302056',
            'Test::Builder::TodoDiag'=> '1.302056',
            'Test::More'            => '1.302056',
            'Test::Simple'          => '1.302056',
            'Test::Tester'          => '1.302056',
            'Test::Tester::Capture' => '1.302056',
            'Test::Tester::CaptureRunner'=> '1.302056',
            'Test::Tester::Delegate'=> '1.302056',
            'Test::use::ok'         => '1.302056',
            'Thread::Semaphore'     => '2.13',
            'XS::APItest'           => '0.84',
            'XSLoader'              => '0.24',
            'ok'                    => '1.302056',
        },
        removed => {
        }
    },
    5.025006 => {
        delta_from => 5.025005,
        changed => {
            'Archive::Tar'          => '2.14',
            'Archive::Tar::Constant'=> '2.14',
            'Archive::Tar::File'    => '2.14',
            'B'                     => '1.64',
            'B::Concise'            => '0.999',
            'B::Deparse'            => '1.39',
            'B::Op_private'         => '5.025006',
            'Config'                => '5.025006',
            'Data::Dumper'          => '2.162',
            'Devel::Peek'           => '1.25',
            'HTTP::Tiny'            => '0.070',
            'List::Util'            => '1.46',
            'List::Util::XS'        => '1.46',
            'Module::CoreList'      => '5.20161020',
            'Module::CoreList::TieHashDelta'=> '5.20161020',
            'Module::CoreList::Utils'=> '5.20161020',
            'Net::Ping'             => '2.51',
            'OS2::DLL'              => '1.07',
            'Opcode'                => '1.38',
            'POSIX'                 => '1.73',
            'PerlIO::encoding'      => '0.25',
            'Pod::Man'              => '4.08',
            'Pod::ParseLink'        => '4.08',
            'Pod::Text'             => '4.08',
            'Pod::Text::Color'      => '4.08',
            'Pod::Text::Overstrike' => '4.08',
            'Pod::Text::Termcap'    => '4.08',
            'Scalar::Util'          => '1.46',
            'Storable'              => '2.58',
            'Sub::Util'             => '1.46',
            'Test2'                 => '1.302059',
            'Test2::API'            => '1.302059',
            'Test2::API::Breakage'  => '1.302059',
            'Test2::API::Context'   => '1.302059',
            'Test2::API::Instance'  => '1.302059',
            'Test2::API::Stack'     => '1.302059',
            'Test2::Event'          => '1.302059',
            'Test2::Event::Bail'    => '1.302059',
            'Test2::Event::Diag'    => '1.302059',
            'Test2::Event::Exception'=> '1.302059',
            'Test2::Event::Generic' => '1.302059',
            'Test2::Event::Info'    => '1.302059',
            'Test2::Event::Note'    => '1.302059',
            'Test2::Event::Ok'      => '1.302059',
            'Test2::Event::Plan'    => '1.302059',
            'Test2::Event::Skip'    => '1.302059',
            'Test2::Event::Subtest' => '1.302059',
            'Test2::Event::Waiting' => '1.302059',
            'Test2::Formatter'      => '1.302059',
            'Test2::Formatter::TAP' => '1.302059',
            'Test2::Hub'            => '1.302059',
            'Test2::Hub::Interceptor'=> '1.302059',
            'Test2::Hub::Interceptor::Terminator'=> '1.302059',
            'Test2::Hub::Subtest'   => '1.302059',
            'Test2::IPC'            => '1.302059',
            'Test2::IPC::Driver'    => '1.302059',
            'Test2::IPC::Driver::Files'=> '1.302059',
            'Test2::Util'           => '1.302059',
            'Test2::Util::ExternalMeta'=> '1.302059',
            'Test2::Util::HashBase' => '1.302059',
            'Test2::Util::Trace'    => '1.302059',
            'Test::Builder'         => '1.302059',
            'Test::Builder::Formatter'=> '1.302059',
            'Test::Builder::Module' => '1.302059',
            'Test::Builder::Tester' => '1.302059',
            'Test::Builder::Tester::Color'=> '1.302059',
            'Test::Builder::TodoDiag'=> '1.302059',
            'Test::More'            => '1.302059',
            'Test::Simple'          => '1.302059',
            'Test::Tester'          => '1.302059',
            'Test::Tester::Capture' => '1.302059',
            'Test::Tester::CaptureRunner'=> '1.302059',
            'Test::Tester::Delegate'=> '1.302059',
            'Test::use::ok'         => '1.302059',
            'Time::HiRes'           => '1.9740_01',
            'VMS::Stdio'            => '2.42',
            'XS::APItest'           => '0.86',
            'attributes'            => '0.28',
            'mro'                   => '1.19',
            'ok'                    => '1.302059',
            'overload'              => '1.27',
            'parent'                => '0.236',
        },
        removed => {
        }
    },
    5.025007 => {
        delta_from => 5.025006,
        changed => {
            'Archive::Tar'          => '2.18',
            'Archive::Tar::Constant'=> '2.18',
            'Archive::Tar::File'    => '2.18',
            'B'                     => '1.65',
            'B::Op_private'         => '5.025007',
            'Config'                => '5.025007',
            'Cwd'                   => '3.66',
            'Data::Dumper'          => '2.165',
            'Devel::Peek'           => '1.26',
            'DynaLoader'            => '1.40',
            'Errno'                 => '1.27',
            'ExtUtils::ParseXS::Utilities'=> '3.34',
            'File::Spec'            => '3.66',
            'File::Spec::AmigaOS'   => '3.66',
            'File::Spec::Cygwin'    => '3.66',
            'File::Spec::Epoc'      => '3.66',
            'File::Spec::Functions' => '3.66',
            'File::Spec::Mac'       => '3.66',
            'File::Spec::OS2'       => '3.66',
            'File::Spec::Unix'      => '3.66',
            'File::Spec::VMS'       => '3.66',
            'File::Spec::Win32'     => '3.66',
            'Hash::Util'            => '0.22',
            'JSON::PP'              => '2.27400_02',
            'List::Util'            => '1.46_02',
            'List::Util::XS'        => '1.46_02',
            'Math::BigFloat'        => '1.999727',
            'Math::BigInt'          => '1.999727',
            'Math::BigInt::Calc'    => '1.999727',
            'Math::BigInt::CalcEmu' => '1.999727',
            'Math::Complex'         => '1.5901',
            'Module::CoreList'      => '5.20161120',
            'Module::CoreList::TieHashDelta'=> '5.20161120',
            'Module::CoreList::Utils'=> '5.20161120',
            'Net::Ping'             => '2.55',
            'Opcode'                => '1.39',
            'POSIX'                 => '1.75',
            'Pod::Man'              => '4.09',
            'Pod::ParseLink'        => '4.09',
            'Pod::Text'             => '4.09',
            'Pod::Text::Color'      => '4.09',
            'Pod::Text::Overstrike' => '4.09',
            'Pod::Text::Termcap'    => '4.09',
            'Scalar::Util'          => '1.46_02',
            'Storable'              => '2.59',
            'Sub::Util'             => '1.46_02',
            'Term::ANSIColor'       => '4.06',
            'Test2'                 => '1.302062',
            'Test2::API'            => '1.302062',
            'Test2::API::Breakage'  => '1.302062',
            'Test2::API::Context'   => '1.302062',
            'Test2::API::Instance'  => '1.302062',
            'Test2::API::Stack'     => '1.302062',
            'Test2::Event'          => '1.302062',
            'Test2::Event::Bail'    => '1.302062',
            'Test2::Event::Diag'    => '1.302062',
            'Test2::Event::Exception'=> '1.302062',
            'Test2::Event::Generic' => '1.302062',
            'Test2::Event::Info'    => '1.302062',
            'Test2::Event::Note'    => '1.302062',
            'Test2::Event::Ok'      => '1.302062',
            'Test2::Event::Plan'    => '1.302062',
            'Test2::Event::Skip'    => '1.302062',
            'Test2::Event::Subtest' => '1.302062',
            'Test2::Event::Waiting' => '1.302062',
            'Test2::Formatter'      => '1.302062',
            'Test2::Formatter::TAP' => '1.302062',
            'Test2::Hub'            => '1.302062',
            'Test2::Hub::Interceptor'=> '1.302062',
            'Test2::Hub::Interceptor::Terminator'=> '1.302062',
            'Test2::Hub::Subtest'   => '1.302062',
            'Test2::IPC'            => '1.302062',
            'Test2::IPC::Driver'    => '1.302062',
            'Test2::IPC::Driver::Files'=> '1.302062',
            'Test2::Util'           => '1.302062',
            'Test2::Util::ExternalMeta'=> '1.302062',
            'Test2::Util::HashBase' => '1.302062',
            'Test2::Util::Trace'    => '1.302062',
            'Test::Builder'         => '1.302062',
            'Test::Builder::Formatter'=> '1.302062',
            'Test::Builder::Module' => '1.302062',
            'Test::Builder::Tester' => '1.302062',
            'Test::Builder::Tester::Color'=> '1.302062',
            'Test::Builder::TodoDiag'=> '1.302062',
            'Test::More'            => '1.302062',
            'Test::Simple'          => '1.302062',
            'Test::Tester'          => '1.302062',
            'Test::Tester::Capture' => '1.302062',
            'Test::Tester::CaptureRunner'=> '1.302062',
            'Test::Tester::Delegate'=> '1.302062',
            'Test::use::ok'         => '1.302062',
            'Time::HiRes'           => '1.9740_03',
            'Unicode::Collate'      => '1.18',
            'Unicode::Collate::CJK::Big5'=> '1.18',
            'Unicode::Collate::CJK::GB2312'=> '1.18',
            'Unicode::Collate::CJK::JISX0208'=> '1.18',
            'Unicode::Collate::CJK::Korean'=> '1.18',
            'Unicode::Collate::CJK::Pinyin'=> '1.18',
            'Unicode::Collate::CJK::Stroke'=> '1.18',
            'Unicode::Collate::CJK::Zhuyin'=> '1.18',
            'Unicode::Collate::Locale'=> '1.18',
            'Unicode::UCD'          => '0.67',
            'XS::APItest'           => '0.87',
            'XS::Typemap'           => '0.15',
            'mro'                   => '1.20',
            'ok'                    => '1.302062',
            'threads'               => '2.10',
        },
        removed => {
        }
    },
    5.025008 => {
        delta_from => 5.025007,
        changed => {
            'Archive::Tar'          => '2.24',
            'Archive::Tar::Constant'=> '2.24',
            'Archive::Tar::File'    => '2.24',
            'B::Debug'              => '1.24',
            'B::Op_private'         => '5.025008',
            'Config'                => '5.025008',
            'Data::Dumper'          => '2.166',
            'Encode'                => '2.88',
            'Encode::Alias'         => '2.21',
            'Encode::CN::HZ'        => '2.08',
            'Encode::MIME::Header'  => '2.24',
            'Encode::MIME::Name'    => '1.02',
            'Encode::Unicode'       => '2.1501',
            'IO'                    => '1.38',
            'Locale::Codes'         => '3.42',
            'Locale::Codes::Constants'=> '3.42',
            'Locale::Codes::Country'=> '3.42',
            'Locale::Codes::Country_Codes'=> '3.42',
            'Locale::Codes::Country_Retired'=> '3.42',
            'Locale::Codes::Currency'=> '3.42',
            'Locale::Codes::Currency_Codes'=> '3.42',
            'Locale::Codes::Currency_Retired'=> '3.42',
            'Locale::Codes::LangExt'=> '3.42',
            'Locale::Codes::LangExt_Codes'=> '3.42',
            'Locale::Codes::LangExt_Retired'=> '3.42',
            'Locale::Codes::LangFam'=> '3.42',
            'Locale::Codes::LangFam_Codes'=> '3.42',
            'Locale::Codes::LangFam_Retired'=> '3.42',
            'Locale::Codes::LangVar'=> '3.42',
            'Locale::Codes::LangVar_Codes'=> '3.42',
            'Locale::Codes::LangVar_Retired'=> '3.42',
            'Locale::Codes::Language'=> '3.42',
            'Locale::Codes::Language_Codes'=> '3.42',
            'Locale::Codes::Language_Retired'=> '3.42',
            'Locale::Codes::Script' => '3.42',
            'Locale::Codes::Script_Codes'=> '3.42',
            'Locale::Codes::Script_Retired'=> '3.42',
            'Locale::Country'       => '3.42',
            'Locale::Currency'      => '3.42',
            'Locale::Language'      => '3.42',
            'Locale::Script'        => '3.42',
            'Math::BigFloat'        => '1.999806',
            'Math::BigFloat::Trace' => '0.47',
            'Math::BigInt'          => '1.999806',
            'Math::BigInt::Calc'    => '1.999806',
            'Math::BigInt::CalcEmu' => '1.999806',
            'Math::BigInt::FastCalc'=> '0.5005',
            'Math::BigInt::Lib'     => '1.999806',
            'Math::BigInt::Trace'   => '0.47',
            'Math::BigRat'          => '0.2611',
            'Module::CoreList'      => '5.20161220',
            'Module::CoreList::TieHashDelta'=> '5.20161220',
            'Module::CoreList::Utils'=> '5.20161220',
            'POSIX'                 => '1.76',
            'PerlIO::scalar'        => '0.25',
            'Pod::Simple'           => '3.35',
            'Pod::Simple::BlackBox' => '3.35',
            'Pod::Simple::Checker'  => '3.35',
            'Pod::Simple::Debug'    => '3.35',
            'Pod::Simple::DumpAsText'=> '3.35',
            'Pod::Simple::DumpAsXML'=> '3.35',
            'Pod::Simple::HTML'     => '3.35',
            'Pod::Simple::HTMLBatch'=> '3.35',
            'Pod::Simple::LinkSection'=> '3.35',
            'Pod::Simple::Methody'  => '3.35',
            'Pod::Simple::Progress' => '3.35',
            'Pod::Simple::PullParser'=> '3.35',
            'Pod::Simple::PullParserEndToken'=> '3.35',
            'Pod::Simple::PullParserStartToken'=> '3.35',
            'Pod::Simple::PullParserTextToken'=> '3.35',
            'Pod::Simple::PullParserToken'=> '3.35',
            'Pod::Simple::RTF'      => '3.35',
            'Pod::Simple::Search'   => '3.35',
            'Pod::Simple::SimpleTree'=> '3.35',
            'Pod::Simple::Text'     => '3.35',
            'Pod::Simple::TextContent'=> '3.35',
            'Pod::Simple::TiedOutFH'=> '3.35',
            'Pod::Simple::Transcode'=> '3.35',
            'Pod::Simple::TranscodeDumb'=> '3.35',
            'Pod::Simple::TranscodeSmart'=> '3.35',
            'Pod::Simple::XHTML'    => '3.35',
            'Pod::Simple::XMLOutStream'=> '3.35',
            'Test2'                 => '1.302073',
            'Test2::API'            => '1.302073',
            'Test2::API::Breakage'  => '1.302073',
            'Test2::API::Context'   => '1.302073',
            'Test2::API::Instance'  => '1.302073',
            'Test2::API::Stack'     => '1.302073',
            'Test2::Event'          => '1.302073',
            'Test2::Event::Bail'    => '1.302073',
            'Test2::Event::Diag'    => '1.302073',
            'Test2::Event::Encoding'=> '1.302073',
            'Test2::Event::Exception'=> '1.302073',
            'Test2::Event::Generic' => '1.302073',
            'Test2::Event::Info'    => '1.302073',
            'Test2::Event::Note'    => '1.302073',
            'Test2::Event::Ok'      => '1.302073',
            'Test2::Event::Plan'    => '1.302073',
            'Test2::Event::Skip'    => '1.302073',
            'Test2::Event::Subtest' => '1.302073',
            'Test2::Event::TAP::Version'=> '1.302073',
            'Test2::Event::Waiting' => '1.302073',
            'Test2::Formatter'      => '1.302073',
            'Test2::Formatter::TAP' => '1.302073',
            'Test2::Hub'            => '1.302073',
            'Test2::Hub::Interceptor'=> '1.302073',
            'Test2::Hub::Interceptor::Terminator'=> '1.302073',
            'Test2::Hub::Subtest'   => '1.302073',
            'Test2::IPC'            => '1.302073',
            'Test2::IPC::Driver'    => '1.302073',
            'Test2::IPC::Driver::Files'=> '1.302073',
            'Test2::Tools::Tiny'    => '1.302073',
            'Test2::Util'           => '1.302073',
            'Test2::Util::ExternalMeta'=> '1.302073',
            'Test2::Util::HashBase' => '0.002',
            'Test2::Util::Trace'    => '1.302073',
            'Test::Builder'         => '1.302073',
            'Test::Builder::Formatter'=> '1.302073',
            'Test::Builder::Module' => '1.302073',
            'Test::Builder::Tester' => '1.302073',
            'Test::Builder::Tester::Color'=> '1.302073',
            'Test::Builder::TodoDiag'=> '1.302073',
            'Test::More'            => '1.302073',
            'Test::Simple'          => '1.302073',
            'Test::Tester'          => '1.302073',
            'Test::Tester::Capture' => '1.302073',
            'Test::Tester::CaptureRunner'=> '1.302073',
            'Test::Tester::Delegate'=> '1.302073',
            'Test::use::ok'         => '1.302073',
            'Time::HiRes'           => '1.9741',
            'Time::Local'           => '1.25',
            'Unicode::Collate'      => '1.19',
            'Unicode::Collate::CJK::Big5'=> '1.19',
            'Unicode::Collate::CJK::GB2312'=> '1.19',
            'Unicode::Collate::CJK::JISX0208'=> '1.19',
            'Unicode::Collate::CJK::Korean'=> '1.19',
            'Unicode::Collate::CJK::Pinyin'=> '1.19',
            'Unicode::Collate::CJK::Stroke'=> '1.19',
            'Unicode::Collate::CJK::Zhuyin'=> '1.19',
            'Unicode::Collate::Locale'=> '1.19',
            'bigint'                => '0.47',
            'bignum'                => '0.47',
            'bigrat'                => '0.47',
            'encoding'              => '2.19',
            'ok'                    => '1.302073',
        },
        removed => {
        }
    },
    5.022003 => {
        delta_from => 5.022002,
        changed => {
            'App::Cpan'             => '1.63_01',
            'App::Prove'            => '3.35_01',
            'App::Prove::State'     => '3.35_01',
            'App::Prove::State::Result'=> '3.35_01',
            'App::Prove::State::Result::Test'=> '3.35_01',
            'Archive::Tar'          => '2.04_01',
            'Archive::Tar::Constant'=> '2.04_01',
            'Archive::Tar::File'    => '2.04_01',
            'B::Op_private'         => '5.022003',
            'CPAN'                  => '2.11_01',
            'Compress::Zlib'        => '2.068_001',
            'Config'                => '5.022003',
            'Cwd'                   => '3.56_02',
            'Digest'                => '1.17_01',
            'Digest::SHA'           => '5.95_01',
            'Encode'                => '2.72_01',
            'ExtUtils::Command'     => '1.20_01',
            'ExtUtils::Command::MM' => '7.04_02',
            'ExtUtils::Liblist'     => '7.04_02',
            'ExtUtils::Liblist::Kid'=> '7.04_02',
            'ExtUtils::MM'          => '7.04_02',
            'ExtUtils::MM_AIX'      => '7.04_02',
            'ExtUtils::MM_Any'      => '7.04_02',
            'ExtUtils::MM_BeOS'     => '7.04_02',
            'ExtUtils::MM_Cygwin'   => '7.04_02',
            'ExtUtils::MM_DOS'      => '7.04_02',
            'ExtUtils::MM_Darwin'   => '7.04_02',
            'ExtUtils::MM_MacOS'    => '7.04_02',
            'ExtUtils::MM_NW5'      => '7.04_02',
            'ExtUtils::MM_OS2'      => '7.04_02',
            'ExtUtils::MM_QNX'      => '7.04_02',
            'ExtUtils::MM_UWIN'     => '7.04_02',
            'ExtUtils::MM_Unix'     => '7.04_02',
            'ExtUtils::MM_VMS'      => '7.04_02',
            'ExtUtils::MM_VOS'      => '7.04_02',
            'ExtUtils::MM_Win32'    => '7.04_02',
            'ExtUtils::MM_Win95'    => '7.04_02',
            'ExtUtils::MY'          => '7.04_02',
            'ExtUtils::MakeMaker'   => '7.04_02',
            'ExtUtils::MakeMaker::Config'=> '7.04_02',
            'ExtUtils::Mkbootstrap' => '7.04_02',
            'ExtUtils::Mksymlists'  => '7.04_02',
            'ExtUtils::testlib'     => '7.04_02',
            'File::Fetch'           => '0.48_01',
            'File::Spec'            => '3.56_02',
            'File::Spec::Cygwin'    => '3.56_02',
            'File::Spec::Epoc'      => '3.56_02',
            'File::Spec::Functions' => '3.56_02',
            'File::Spec::Mac'       => '3.56_02',
            'File::Spec::OS2'       => '3.56_02',
            'File::Spec::Unix'      => '3.56_02',
            'File::Spec::VMS'       => '3.56_02',
            'File::Spec::Win32'     => '3.56_02',
            'HTTP::Tiny'            => '0.054_01',
            'I18N::LangTags::Detect'=> '1.05_01',
            'IO'                    => '1.35_01',
            'IO::Compress::Adapter::Bzip2'=> '2.068_001',
            'IO::Compress::Adapter::Deflate'=> '2.068_001',
            'IO::Compress::Adapter::Identity'=> '2.068_001',
            'IO::Compress::Base'    => '2.068_001',
            'IO::Compress::Base::Common'=> '2.068_001',
            'IO::Compress::Bzip2'   => '2.068_001',
            'IO::Compress::Deflate' => '2.068_001',
            'IO::Compress::Gzip'    => '2.068_001',
            'IO::Compress::Gzip::Constants'=> '2.068_001',
            'IO::Compress::RawDeflate'=> '2.068_001',
            'IO::Compress::Zip'     => '2.068_001',
            'IO::Compress::Zip::Constants'=> '2.068_001',
            'IO::Compress::Zlib::Constants'=> '2.068_001',
            'IO::Compress::Zlib::Extra'=> '2.068_001',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.068_001',
            'IO::Uncompress::Adapter::Identity'=> '2.068_001',
            'IO::Uncompress::Adapter::Inflate'=> '2.068_001',
            'IO::Uncompress::AnyInflate'=> '2.068_001',
            'IO::Uncompress::AnyUncompress'=> '2.068_001',
            'IO::Uncompress::Base'  => '2.068_001',
            'IO::Uncompress::Bunzip2'=> '2.068_001',
            'IO::Uncompress::Gunzip'=> '2.068_001',
            'IO::Uncompress::Inflate'=> '2.068_001',
            'IO::Uncompress::RawInflate'=> '2.068_001',
            'IO::Uncompress::Unzip' => '2.068_001',
            'IPC::Cmd'              => '0.92_01',
            'JSON::PP'              => '2.27300_01',
            'Locale::Maketext'      => '1.26_01',
            'Locale::Maketext::Simple'=> '0.21_01',
            'Memoize'               => '1.03_01',
            'Module::CoreList'      => '5.20170114_22',
            'Module::CoreList::TieHashDelta'=> '5.20170114_22',
            'Module::CoreList::Utils'=> '5.20170114_22',
            'Module::Metadata::corpus::BOMTest::UTF16BE'=> undef,
            'Module::Metadata::corpus::BOMTest::UTF16LE'=> undef,
            'Module::Metadata::corpus::BOMTest::UTF8'=> '1',
            'Net::Cmd'              => '3.05_01',
            'Net::Config'           => '3.05_01',
            'Net::Domain'           => '3.05_01',
            'Net::FTP'              => '3.05_01',
            'Net::FTP::A'           => '3.05_01',
            'Net::FTP::E'           => '3.05_01',
            'Net::FTP::I'           => '3.05_01',
            'Net::FTP::L'           => '3.05_01',
            'Net::FTP::dataconn'    => '3.05_01',
            'Net::NNTP'             => '3.05_01',
            'Net::Netrc'            => '3.05_01',
            'Net::POP3'             => '3.05_01',
            'Net::Ping'             => '2.43_01',
            'Net::SMTP'             => '3.05_01',
            'Net::Time'             => '3.05_01',
            'Parse::CPAN::Meta'     => '1.4414_001',
            'Pod::Html'             => '1.2201',
            'Pod::Perldoc'          => '3.25_01',
            'Storable'              => '2.53_02',
            'Sys::Syslog'           => '0.33_01',
            'TAP::Base'             => '3.35_01',
            'TAP::Formatter::Base'  => '3.35_01',
            'TAP::Formatter::Color' => '3.35_01',
            'TAP::Formatter::Console'=> '3.35_01',
            'TAP::Formatter::Console::ParallelSession'=> '3.35_01',
            'TAP::Formatter::Console::Session'=> '3.35_01',
            'TAP::Formatter::File'  => '3.35_01',
            'TAP::Formatter::File::Session'=> '3.35_01',
            'TAP::Formatter::Session'=> '3.35_01',
            'TAP::Harness'          => '3.35_01',
            'TAP::Harness::Env'     => '3.35_01',
            'TAP::Object'           => '3.35_01',
            'TAP::Parser'           => '3.35_01',
            'TAP::Parser::Aggregator'=> '3.35_01',
            'TAP::Parser::Grammar'  => '3.35_01',
            'TAP::Parser::Iterator' => '3.35_01',
            'TAP::Parser::Iterator::Array'=> '3.35_01',
            'TAP::Parser::Iterator::Process'=> '3.35_01',
            'TAP::Parser::Iterator::Stream'=> '3.35_01',
            'TAP::Parser::IteratorFactory'=> '3.35_01',
            'TAP::Parser::Multiplexer'=> '3.35_01',
            'TAP::Parser::Result'   => '3.35_01',
            'TAP::Parser::Result::Bailout'=> '3.35_01',
            'TAP::Parser::Result::Comment'=> '3.35_01',
            'TAP::Parser::Result::Plan'=> '3.35_01',
            'TAP::Parser::Result::Pragma'=> '3.35_01',
            'TAP::Parser::Result::Test'=> '3.35_01',
            'TAP::Parser::Result::Unknown'=> '3.35_01',
            'TAP::Parser::Result::Version'=> '3.35_01',
            'TAP::Parser::Result::YAML'=> '3.35_01',
            'TAP::Parser::ResultFactory'=> '3.35_01',
            'TAP::Parser::Scheduler'=> '3.35_01',
            'TAP::Parser::Scheduler::Job'=> '3.35_01',
            'TAP::Parser::Scheduler::Spinner'=> '3.35_01',
            'TAP::Parser::Source'   => '3.35_01',
            'TAP::Parser::SourceHandler'=> '3.35_01',
            'TAP::Parser::SourceHandler::Executable'=> '3.35_01',
            'TAP::Parser::SourceHandler::File'=> '3.35_01',
            'TAP::Parser::SourceHandler::Handle'=> '3.35_01',
            'TAP::Parser::SourceHandler::Perl'=> '3.35_01',
            'TAP::Parser::SourceHandler::RawTAP'=> '3.35_01',
            'TAP::Parser::YAMLish::Reader'=> '3.35_01',
            'TAP::Parser::YAMLish::Writer'=> '3.35_01',
            'Test'                  => '1.26_01',
            'Test::Harness'         => '3.35_01',
            'XSLoader'              => '0.20_01',
            'bigint'                => '0.39_01',
            'bignum'                => '0.39_01',
            'bigrat'                => '0.39_01',
        },
        removed => {
        }
    },
    5.024001 => {
        delta_from => 5.024000,
        changed => {
            'App::Cpan'             => '1.63_01',
            'App::Prove'            => '3.36_01',
            'App::Prove::State'     => '3.36_01',
            'App::Prove::State::Result'=> '3.36_01',
            'App::Prove::State::Result::Test'=> '3.36_01',
            'Archive::Tar'          => '2.04_01',
            'Archive::Tar::Constant'=> '2.04_01',
            'Archive::Tar::File'    => '2.04_01',
            'B::Op_private'         => '5.024001',
            'CPAN'                  => '2.11_01',
            'Compress::Zlib'        => '2.069_001',
            'Config'                => '5.024001',
            'Cwd'                   => '3.63_01',
            'Digest'                => '1.17_01',
            'Digest::SHA'           => '5.95_01',
            'Encode'                => '2.80_01',
            'ExtUtils::Command'     => '7.10_02',
            'ExtUtils::Command::MM' => '7.10_02',
            'ExtUtils::Liblist'     => '7.10_02',
            'ExtUtils::Liblist::Kid'=> '7.10_02',
            'ExtUtils::MM'          => '7.10_02',
            'ExtUtils::MM_AIX'      => '7.10_02',
            'ExtUtils::MM_Any'      => '7.10_02',
            'ExtUtils::MM_BeOS'     => '7.10_02',
            'ExtUtils::MM_Cygwin'   => '7.10_02',
            'ExtUtils::MM_DOS'      => '7.10_02',
            'ExtUtils::MM_Darwin'   => '7.10_02',
            'ExtUtils::MM_MacOS'    => '7.10_02',
            'ExtUtils::MM_NW5'      => '7.10_02',
            'ExtUtils::MM_OS2'      => '7.10_02',
            'ExtUtils::MM_QNX'      => '7.10_02',
            'ExtUtils::MM_UWIN'     => '7.10_02',
            'ExtUtils::MM_Unix'     => '7.10_02',
            'ExtUtils::MM_VMS'      => '7.10_02',
            'ExtUtils::MM_VOS'      => '7.10_02',
            'ExtUtils::MM_Win32'    => '7.10_02',
            'ExtUtils::MM_Win95'    => '7.10_02',
            'ExtUtils::MY'          => '7.10_02',
            'ExtUtils::MakeMaker'   => '7.10_02',
            'ExtUtils::MakeMaker::Config'=> '7.10_02',
            'ExtUtils::Mkbootstrap' => '7.10_02',
            'ExtUtils::Mksymlists'  => '7.10_02',
            'ExtUtils::testlib'     => '7.10_02',
            'File::Fetch'           => '0.48_01',
            'File::Spec'            => '3.63_01',
            'File::Spec::Cygwin'    => '3.63_01',
            'File::Spec::Epoc'      => '3.63_01',
            'File::Spec::Functions' => '3.63_01',
            'File::Spec::Mac'       => '3.63_01',
            'File::Spec::OS2'       => '3.63_01',
            'File::Spec::Unix'      => '3.63_01',
            'File::Spec::VMS'       => '3.63_01',
            'File::Spec::Win32'     => '3.63_01',
            'HTTP::Tiny'            => '0.056_001',
            'I18N::LangTags::Detect'=> '1.05_01',
            'IO'                    => '1.36_01',
            'IO::Compress::Adapter::Bzip2'=> '2.069_001',
            'IO::Compress::Adapter::Deflate'=> '2.069_001',
            'IO::Compress::Adapter::Identity'=> '2.069_001',
            'IO::Compress::Base'    => '2.069_001',
            'IO::Compress::Base::Common'=> '2.069_001',
            'IO::Compress::Bzip2'   => '2.069_001',
            'IO::Compress::Deflate' => '2.069_001',
            'IO::Compress::Gzip'    => '2.069_001',
            'IO::Compress::Gzip::Constants'=> '2.069_001',
            'IO::Compress::RawDeflate'=> '2.069_001',
            'IO::Compress::Zip'     => '2.069_001',
            'IO::Compress::Zip::Constants'=> '2.069_001',
            'IO::Compress::Zlib::Constants'=> '2.069_001',
            'IO::Compress::Zlib::Extra'=> '2.069_001',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.069_001',
            'IO::Uncompress::Adapter::Identity'=> '2.069_001',
            'IO::Uncompress::Adapter::Inflate'=> '2.069_001',
            'IO::Uncompress::AnyInflate'=> '2.069_001',
            'IO::Uncompress::AnyUncompress'=> '2.069_001',
            'IO::Uncompress::Base'  => '2.069_001',
            'IO::Uncompress::Bunzip2'=> '2.069_001',
            'IO::Uncompress::Gunzip'=> '2.069_001',
            'IO::Uncompress::Inflate'=> '2.069_001',
            'IO::Uncompress::RawInflate'=> '2.069_001',
            'IO::Uncompress::Unzip' => '2.069_001',
            'IPC::Cmd'              => '0.92_01',
            'JSON::PP'              => '2.27300_01',
            'Locale::Maketext'      => '1.26_01',
            'Locale::Maketext::Simple'=> '0.21_01',
            'Math::BigFloat::Trace' => '0.42_01',
            'Math::BigInt::Trace'   => '0.42_01',
            'Memoize'               => '1.03_01',
            'Module::CoreList'      => '5.20170114_24',
            'Module::CoreList::TieHashDelta'=> '5.20170114_24',
            'Module::CoreList::Utils'=> '5.20170114_24',
            'Module::Metadata::corpus::BOMTest::UTF16BE'=> undef,
            'Module::Metadata::corpus::BOMTest::UTF16LE'=> undef,
            'Module::Metadata::corpus::BOMTest::UTF8'=> '1',
            'Net::Cmd'              => '3.08_01',
            'Net::Config'           => '3.08_01',
            'Net::Domain'           => '3.08_01',
            'Net::FTP'              => '3.08_01',
            'Net::FTP::A'           => '3.08_01',
            'Net::FTP::E'           => '3.08_01',
            'Net::FTP::I'           => '3.08_01',
            'Net::FTP::L'           => '3.08_01',
            'Net::FTP::dataconn'    => '3.08_01',
            'Net::NNTP'             => '3.08_01',
            'Net::Netrc'            => '3.08_01',
            'Net::POP3'             => '3.08_01',
            'Net::Ping'             => '2.43_01',
            'Net::SMTP'             => '3.08_01',
            'Net::Time'             => '3.08_01',
            'Parse::CPAN::Meta'     => '1.4417_001',
            'Pod::Html'             => '1.2201',
            'Pod::Perldoc'          => '3.25_03',
            'Storable'              => '2.56_01',
            'Sys::Syslog'           => '0.33_01',
            'TAP::Base'             => '3.36_01',
            'TAP::Formatter::Base'  => '3.36_01',
            'TAP::Formatter::Color' => '3.36_01',
            'TAP::Formatter::Console'=> '3.36_01',
            'TAP::Formatter::Console::ParallelSession'=> '3.36_01',
            'TAP::Formatter::Console::Session'=> '3.36_01',
            'TAP::Formatter::File'  => '3.36_01',
            'TAP::Formatter::File::Session'=> '3.36_01',
            'TAP::Formatter::Session'=> '3.36_01',
            'TAP::Harness'          => '3.36_01',
            'TAP::Harness::Env'     => '3.36_01',
            'TAP::Object'           => '3.36_01',
            'TAP::Parser'           => '3.36_01',
            'TAP::Parser::Aggregator'=> '3.36_01',
            'TAP::Parser::Grammar'  => '3.36_01',
            'TAP::Parser::Iterator' => '3.36_01',
            'TAP::Parser::Iterator::Array'=> '3.36_01',
            'TAP::Parser::Iterator::Process'=> '3.36_01',
            'TAP::Parser::Iterator::Stream'=> '3.36_01',
            'TAP::Parser::IteratorFactory'=> '3.36_01',
            'TAP::Parser::Multiplexer'=> '3.36_01',
            'TAP::Parser::Result'   => '3.36_01',
            'TAP::Parser::Result::Bailout'=> '3.36_01',
            'TAP::Parser::Result::Comment'=> '3.36_01',
            'TAP::Parser::Result::Plan'=> '3.36_01',
            'TAP::Parser::Result::Pragma'=> '3.36_01',
            'TAP::Parser::Result::Test'=> '3.36_01',
            'TAP::Parser::Result::Unknown'=> '3.36_01',
            'TAP::Parser::Result::Version'=> '3.36_01',
            'TAP::Parser::Result::YAML'=> '3.36_01',
            'TAP::Parser::ResultFactory'=> '3.36_01',
            'TAP::Parser::Scheduler'=> '3.36_01',
            'TAP::Parser::Scheduler::Job'=> '3.36_01',
            'TAP::Parser::Scheduler::Spinner'=> '3.36_01',
            'TAP::Parser::Source'   => '3.36_01',
            'TAP::Parser::SourceHandler'=> '3.36_01',
            'TAP::Parser::SourceHandler::Executable'=> '3.36_01',
            'TAP::Parser::SourceHandler::File'=> '3.36_01',
            'TAP::Parser::SourceHandler::Handle'=> '3.36_01',
            'TAP::Parser::SourceHandler::Perl'=> '3.36_01',
            'TAP::Parser::SourceHandler::RawTAP'=> '3.36_01',
            'TAP::Parser::YAMLish::Reader'=> '3.36_01',
            'TAP::Parser::YAMLish::Writer'=> '3.36_01',
            'Test'                  => '1.28_01',
            'Test::Harness'         => '3.36_01',
            'XSLoader'              => '0.22',
            'bigint'                => '0.42_01',
            'bignum'                => '0.42_01',
            'bigrat'                => '0.42_01',
        },
        removed => {
        }
    },
    5.025009 => {
        delta_from => 5.025008,
        changed => {
            'App::Cpan'             => '1.66',
            'B::Deparse'            => '1.40',
            'B::Op_private'         => '5.025009',
            'B::Terse'              => '1.07',
            'B::Xref'               => '1.06',
            'CPAN'                  => '2.16',
            'CPAN::Bundle'          => '5.5002',
            'CPAN::Distribution'    => '2.16',
            'CPAN::Exception::RecursiveDependency'=> '5.5001',
            'CPAN::FTP'             => '5.5008',
            'CPAN::FirstTime'       => '5.5310',
            'CPAN::HandleConfig'    => '5.5008',
            'CPAN::Module'          => '5.5003',
            'Compress::Raw::Bzip2'  => '2.070',
            'Compress::Raw::Zlib'   => '2.070',
            'Config'                => '5.025009',
            'DB_File'               => '1.840',
            'Data::Dumper'          => '2.167',
            'Devel::SelfStubber'    => '1.06',
            'DynaLoader'            => '1.41',
            'Errno'                 => '1.28',
            'ExtUtils::Embed'       => '1.34',
            'File::Glob'            => '1.28',
            'I18N::LangTags'        => '0.42',
            'Module::CoreList'      => '5.20170120',
            'Module::CoreList::TieHashDelta'=> '5.20170120',
            'Module::CoreList::Utils'=> '5.20170120',
            'OS2::Process'          => '1.12',
            'PerlIO::scalar'        => '0.26',
            'Pod::Html'             => '1.2202',
            'Storable'              => '2.61',
            'Symbol'                => '1.08',
            'Term::ReadLine'        => '1.16',
            'Test'                  => '1.30',
            'Unicode::UCD'          => '0.68',
            'VMS::DCLsym'           => '1.08',
            'XS::APItest'           => '0.88',
            'XSLoader'              => '0.26',
            'attributes'            => '0.29',
            'diagnostics'           => '1.36',
            'feature'               => '1.46',
            'lib'                   => '0.64',
            'overload'              => '1.28',
            're'                    => '0.34',
            'threads'               => '2.12',
            'threads::shared'       => '1.54',
        },
        removed => {
        }
    },
    5.025010 => {
        delta_from => 5.025009,
        changed => {
            'B'                     => '1.68',
            'B::Op_private'         => '5.025010',
            'CPAN'                  => '2.17',
            'CPAN::Distribution'    => '2.17',
            'Config'                => '5.02501',
            'Getopt::Std'           => '1.12',
            'Module::CoreList'      => '5.20170220',
            'Module::CoreList::TieHashDelta'=> '5.20170220',
            'Module::CoreList::Utils'=> '5.20170220',
            'PerlIO'                => '1.10',
            'Storable'              => '2.62',
            'Thread::Queue'         => '3.12',
            'feature'               => '1.47',
            'open'                  => '1.11',
            'threads'               => '2.13',
        },
        removed => {
        }
    },
    5.025011 => {
        delta_from => 5.025010,
        changed => {
            'App::Prove'            => '3.38',
            'App::Prove::State'     => '3.38',
            'App::Prove::State::Result'=> '3.38',
            'App::Prove::State::Result::Test'=> '3.38',
            'B::Op_private'         => '5.025011',
            'Compress::Raw::Bzip2'  => '2.074',
            'Compress::Raw::Zlib'   => '2.074',
            'Compress::Zlib'        => '2.074',
            'Config'                => '5.025011',
            'Config::Perl::V'       => '0.28',
            'Cwd'                   => '3.67',
            'ExtUtils::ParseXS'     => '3.34',
            'ExtUtils::ParseXS::Constants'=> '3.34',
            'ExtUtils::ParseXS::CountLines'=> '3.34',
            'ExtUtils::ParseXS::Eval'=> '3.34',
            'ExtUtils::Typemaps'    => '3.34',
            'ExtUtils::Typemaps::Cmd'=> '3.34',
            'ExtUtils::Typemaps::InputMap'=> '3.34',
            'ExtUtils::Typemaps::OutputMap'=> '3.34',
            'ExtUtils::Typemaps::Type'=> '3.34',
            'File::Spec'            => '3.67',
            'File::Spec::AmigaOS'   => '3.67',
            'File::Spec::Cygwin'    => '3.67',
            'File::Spec::Epoc'      => '3.67',
            'File::Spec::Functions' => '3.67',
            'File::Spec::Mac'       => '3.67',
            'File::Spec::OS2'       => '3.67',
            'File::Spec::Unix'      => '3.67',
            'File::Spec::VMS'       => '3.67',
            'File::Spec::Win32'     => '3.67',
            'IO::Compress::Adapter::Bzip2'=> '2.074',
            'IO::Compress::Adapter::Deflate'=> '2.074',
            'IO::Compress::Adapter::Identity'=> '2.074',
            'IO::Compress::Base'    => '2.074',
            'IO::Compress::Base::Common'=> '2.074',
            'IO::Compress::Bzip2'   => '2.074',
            'IO::Compress::Deflate' => '2.074',
            'IO::Compress::Gzip'    => '2.074',
            'IO::Compress::Gzip::Constants'=> '2.074',
            'IO::Compress::RawDeflate'=> '2.074',
            'IO::Compress::Zip'     => '2.074',
            'IO::Compress::Zip::Constants'=> '2.074',
            'IO::Compress::Zlib::Constants'=> '2.074',
            'IO::Compress::Zlib::Extra'=> '2.074',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.074',
            'IO::Uncompress::Adapter::Identity'=> '2.074',
            'IO::Uncompress::Adapter::Inflate'=> '2.074',
            'IO::Uncompress::AnyInflate'=> '2.074',
            'IO::Uncompress::AnyUncompress'=> '2.074',
            'IO::Uncompress::Base'  => '2.074',
            'IO::Uncompress::Bunzip2'=> '2.074',
            'IO::Uncompress::Gunzip'=> '2.074',
            'IO::Uncompress::Inflate'=> '2.074',
            'IO::Uncompress::RawInflate'=> '2.074',
            'IO::Uncompress::Unzip' => '2.074',
            'Module::CoreList'      => '5.20170320',
            'Module::CoreList::TieHashDelta'=> '5.20170230',
            'Module::CoreList::Utils'=> '5.20170320',
            'Pod::Perldoc'          => '3.28',
            'Pod::Perldoc::BaseTo'  => '3.28',
            'Pod::Perldoc::GetOptsOO'=> '3.28',
            'Pod::Perldoc::ToANSI'  => '3.28',
            'Pod::Perldoc::ToChecker'=> '3.28',
            'Pod::Perldoc::ToMan'   => '3.28',
            'Pod::Perldoc::ToNroff' => '3.28',
            'Pod::Perldoc::ToPod'   => '3.28',
            'Pod::Perldoc::ToRtf'   => '3.28',
            'Pod::Perldoc::ToTerm'  => '3.28',
            'Pod::Perldoc::ToText'  => '3.28',
            'Pod::Perldoc::ToTk'    => '3.28',
            'Pod::Perldoc::ToXml'   => '3.28',
            'TAP::Base'             => '3.38',
            'TAP::Formatter::Base'  => '3.38',
            'TAP::Formatter::Color' => '3.38',
            'TAP::Formatter::Console'=> '3.38',
            'TAP::Formatter::Console::ParallelSession'=> '3.38',
            'TAP::Formatter::Console::Session'=> '3.38',
            'TAP::Formatter::File'  => '3.38',
            'TAP::Formatter::File::Session'=> '3.38',
            'TAP::Formatter::Session'=> '3.38',
            'TAP::Harness'          => '3.38',
            'TAP::Harness::Env'     => '3.38',
            'TAP::Object'           => '3.38',
            'TAP::Parser'           => '3.38',
            'TAP::Parser::Aggregator'=> '3.38',
            'TAP::Parser::Grammar'  => '3.38',
            'TAP::Parser::Iterator' => '3.38',
            'TAP::Parser::Iterator::Array'=> '3.38',
            'TAP::Parser::Iterator::Process'=> '3.38',
            'TAP::Parser::Iterator::Stream'=> '3.38',
            'TAP::Parser::IteratorFactory'=> '3.38',
            'TAP::Parser::Multiplexer'=> '3.38',
            'TAP::Parser::Result'   => '3.38',
            'TAP::Parser::Result::Bailout'=> '3.38',
            'TAP::Parser::Result::Comment'=> '3.38',
            'TAP::Parser::Result::Plan'=> '3.38',
            'TAP::Parser::Result::Pragma'=> '3.38',
            'TAP::Parser::Result::Test'=> '3.38',
            'TAP::Parser::Result::Unknown'=> '3.38',
            'TAP::Parser::Result::Version'=> '3.38',
            'TAP::Parser::Result::YAML'=> '3.38',
            'TAP::Parser::ResultFactory'=> '3.38',
            'TAP::Parser::Scheduler'=> '3.38',
            'TAP::Parser::Scheduler::Job'=> '3.38',
            'TAP::Parser::Scheduler::Spinner'=> '3.38',
            'TAP::Parser::Source'   => '3.38',
            'TAP::Parser::SourceHandler'=> '3.38',
            'TAP::Parser::SourceHandler::Executable'=> '3.38',
            'TAP::Parser::SourceHandler::File'=> '3.38',
            'TAP::Parser::SourceHandler::Handle'=> '3.38',
            'TAP::Parser::SourceHandler::Perl'=> '3.38',
            'TAP::Parser::SourceHandler::RawTAP'=> '3.38',
            'TAP::Parser::YAMLish::Reader'=> '3.38',
            'TAP::Parser::YAMLish::Writer'=> '3.38',
            'Test::Harness'         => '3.38',
            'VMS::Stdio'            => '2.41',
            'threads'               => '2.15',
            'threads::shared'       => '1.55',
        },
        removed => {
        }
    },
    5.025012 => {
        delta_from => 5.025011,
        changed => {
            'B::Op_private'         => '5.025012',
            'CPAN'                  => '2.18',
            'CPAN::Bundle'          => '5.5003',
            'CPAN::Distribution'    => '2.18',
            'Config'                => '5.025012',
            'DynaLoader'            => '1.42',
            'Module::CoreList'      => '5.20170420',
            'Module::CoreList::TieHashDelta'=> '5.20170420',
            'Module::CoreList::Utils'=> '5.20170420',
            'Safe'                  => '2.40',
            'XSLoader'              => '0.27',
            'base'                  => '2.25',
            'threads::shared'       => '1.56',
        },
        removed => {
        }
    },
    5.026000 => {
        delta_from => 5.025012,
        changed => {
            'B::Op_private'         => '5.026000',
            'Config'                => '5.026',
            'Module::CoreList'      => '5.20170530',
            'Module::CoreList::TieHashDelta'=> '5.20170530',
            'Module::CoreList::Utils'=> '5.20170530',
        },
        removed => {
        }
    },
    5.027000 => {
        delta_from => 5.026000,
        changed => {
            'Attribute::Handlers'   => '1.00',
            'B::Concise'            => '1.000',
            'B::Deparse'            => '1.41',
            'B::Op_private'         => '5.027000',
            'Config'                => '5.027',
            'Module::CoreList'      => '5.20170531',
            'Module::CoreList::TieHashDelta'=> '5.20170531',
            'Module::CoreList::Utils'=> '5.20170531',
            'O'                     => '1.02',
            'attributes'            => '0.3',
            'feature'               => '1.48',
        },
        removed => {
        }
    },
    5.027001 => {
        delta_from => 5.027,
        changed => {
            'App::Prove'            => '3.39',
            'App::Prove::State'     => '3.39',
            'App::Prove::State::Result'=> '3.39',
            'App::Prove::State::Result::Test'=> '3.39',
            'Archive::Tar'          => '2.26',
            'Archive::Tar::Constant'=> '2.26',
            'Archive::Tar::File'    => '2.26',
            'B::Op_private'         => '5.027001',
            'B::Terse'              => '1.08',
            'Config'                => '5.027001',
            'Devel::PPPort'         => '3.36',
            'DirHandle'             => '1.05',
            'ExtUtils::Command'     => '7.30',
            'ExtUtils::Command::MM' => '7.30',
            'ExtUtils::Install'     => '2.14',
            'ExtUtils::Installed'   => '2.14',
            'ExtUtils::Liblist'     => '7.30',
            'ExtUtils::Liblist::Kid'=> '7.30',
            'ExtUtils::MM'          => '7.30',
            'ExtUtils::MM_AIX'      => '7.30',
            'ExtUtils::MM_Any'      => '7.30',
            'ExtUtils::MM_BeOS'     => '7.30',
            'ExtUtils::MM_Cygwin'   => '7.30',
            'ExtUtils::MM_DOS'      => '7.30',
            'ExtUtils::MM_Darwin'   => '7.30',
            'ExtUtils::MM_MacOS'    => '7.30',
            'ExtUtils::MM_NW5'      => '7.30',
            'ExtUtils::MM_OS2'      => '7.30',
            'ExtUtils::MM_QNX'      => '7.30',
            'ExtUtils::MM_UWIN'     => '7.30',
            'ExtUtils::MM_Unix'     => '7.30',
            'ExtUtils::MM_VMS'      => '7.30',
            'ExtUtils::MM_VOS'      => '7.30',
            'ExtUtils::MM_Win32'    => '7.30',
            'ExtUtils::MM_Win95'    => '7.30',
            'ExtUtils::MY'          => '7.30',
            'ExtUtils::MakeMaker'   => '7.30',
            'ExtUtils::MakeMaker::Config'=> '7.30',
            'ExtUtils::MakeMaker::Locale'=> '7.30',
            'ExtUtils::MakeMaker::version'=> '7.30',
            'ExtUtils::MakeMaker::version::regex'=> '7.30',
            'ExtUtils::Mkbootstrap' => '7.30',
            'ExtUtils::Mksymlists'  => '7.30',
            'ExtUtils::Packlist'    => '2.14',
            'ExtUtils::testlib'     => '7.30',
            'File::Path'            => '2.14',
            'Filter::Util::Call'    => '1.57',
            'GDBM_File'             => '1.16',
            'Getopt::Long'          => '2.5',
            'IO::Socket::IP'        => '0.39',
            'IPC::Cmd'              => '0.98',
            'JSON::PP'              => '2.94',
            'JSON::PP::Boolean'     => '2.94',
            'Locale::Codes'         => '3.52',
            'Locale::Codes::Constants'=> '3.52',
            'Locale::Codes::Country'=> '3.52',
            'Locale::Codes::Country_Codes'=> '3.52',
            'Locale::Codes::Country_Retired'=> '3.52',
            'Locale::Codes::Currency'=> '3.52',
            'Locale::Codes::Currency_Codes'=> '3.52',
            'Locale::Codes::Currency_Retired'=> '3.52',
            'Locale::Codes::LangExt'=> '3.52',
            'Locale::Codes::LangExt_Codes'=> '3.52',
            'Locale::Codes::LangExt_Retired'=> '3.52',
            'Locale::Codes::LangFam'=> '3.52',
            'Locale::Codes::LangFam_Codes'=> '3.52',
            'Locale::Codes::LangFam_Retired'=> '3.52',
            'Locale::Codes::LangVar'=> '3.52',
            'Locale::Codes::LangVar_Codes'=> '3.52',
            'Locale::Codes::LangVar_Retired'=> '3.52',
            'Locale::Codes::Language'=> '3.52',
            'Locale::Codes::Language_Codes'=> '3.52',
            'Locale::Codes::Language_Retired'=> '3.52',
            'Locale::Codes::Script' => '3.52',
            'Locale::Codes::Script_Codes'=> '3.52',
            'Locale::Codes::Script_Retired'=> '3.52',
            'Locale::Country'       => '3.52',
            'Locale::Currency'      => '3.52',
            'Locale::Language'      => '3.52',
            'Locale::Script'        => '3.52',
            'Module::CoreList'      => '5.20170621',
            'Module::CoreList::TieHashDelta'=> '5.20170621',
            'Module::CoreList::Utils'=> '5.20170621',
            'PerlIO::scalar'        => '0.27',
            'PerlIO::via'           => '0.17',
            'Storable'              => '2.63',
            'TAP::Base'             => '3.39',
            'TAP::Formatter::Base'  => '3.39',
            'TAP::Formatter::Color' => '3.39',
            'TAP::Formatter::Console'=> '3.39',
            'TAP::Formatter::Console::ParallelSession'=> '3.39',
            'TAP::Formatter::Console::Session'=> '3.39',
            'TAP::Formatter::File'  => '3.39',
            'TAP::Formatter::File::Session'=> '3.39',
            'TAP::Formatter::Session'=> '3.39',
            'TAP::Harness'          => '3.39',
            'TAP::Harness::Env'     => '3.39',
            'TAP::Object'           => '3.39',
            'TAP::Parser'           => '3.39',
            'TAP::Parser::Aggregator'=> '3.39',
            'TAP::Parser::Grammar'  => '3.39',
            'TAP::Parser::Iterator' => '3.39',
            'TAP::Parser::Iterator::Array'=> '3.39',
            'TAP::Parser::Iterator::Process'=> '3.39',
            'TAP::Parser::Iterator::Stream'=> '3.39',
            'TAP::Parser::IteratorFactory'=> '3.39',
            'TAP::Parser::Multiplexer'=> '3.39',
            'TAP::Parser::Result'   => '3.39',
            'TAP::Parser::Result::Bailout'=> '3.39',
            'TAP::Parser::Result::Comment'=> '3.39',
            'TAP::Parser::Result::Plan'=> '3.39',
            'TAP::Parser::Result::Pragma'=> '3.39',
            'TAP::Parser::Result::Test'=> '3.39',
            'TAP::Parser::Result::Unknown'=> '3.39',
            'TAP::Parser::Result::Version'=> '3.39',
            'TAP::Parser::Result::YAML'=> '3.39',
            'TAP::Parser::ResultFactory'=> '3.39',
            'TAP::Parser::Scheduler'=> '3.39',
            'TAP::Parser::Scheduler::Job'=> '3.39',
            'TAP::Parser::Scheduler::Spinner'=> '3.39',
            'TAP::Parser::Source'   => '3.39',
            'TAP::Parser::SourceHandler'=> '3.39',
            'TAP::Parser::SourceHandler::Executable'=> '3.39',
            'TAP::Parser::SourceHandler::File'=> '3.39',
            'TAP::Parser::SourceHandler::Handle'=> '3.39',
            'TAP::Parser::SourceHandler::Perl'=> '3.39',
            'TAP::Parser::SourceHandler::RawTAP'=> '3.39',
            'TAP::Parser::YAMLish::Reader'=> '3.39',
            'TAP::Parser::YAMLish::Writer'=> '3.39',
            'Test::Harness'         => '3.39',
            'XS::APItest'           => '0.89',
            '_charnames'            => '1.45',
            'charnames'             => '1.45',
            'if'                    => '0.0607',
            'mro'                   => '1.21',
            'threads'               => '2.16',
            'threads::shared'       => '1.57',
            'version'               => '0.9918',
            'version::regex'        => '0.9918',
        },
        removed => {
        }
    },
    5.022004 => {
        delta_from => 5.022003,
        changed => {
            'B::Op_private'         => '5.022004',
            'Config'                => '5.022004',
            'Module::CoreList'      => '5.20170715_22',
            'Module::CoreList::TieHashDelta'=> '5.20170715_22',
            'Module::CoreList::Utils'=> '5.20170715_22',
            'base'                  => '2.22_01',
        },
        removed => {
        }
    },
    5.024002 => {
        delta_from => 5.024001,
        changed => {
            'B::Op_private'         => '5.024002',
            'Config'                => '5.024002',
            'Module::CoreList'      => '5.20170715_24',
            'Module::CoreList::TieHashDelta'=> '5.20170715_24',
            'Module::CoreList::Utils'=> '5.20170715_24',
            'base'                  => '2.23_01',
        },
        removed => {
        }
    },
    5.027002 => {
        delta_from => 5.027001,
        changed => {
            'B::Op_private'         => '5.027002',
            'Carp'                  => '1.43',
            'Carp::Heavy'           => '1.43',
            'Config'                => '5.027002',
            'Cwd'                   => '3.68',
            'Encode'                => '2.92',
            'Encode::Alias'         => '2.23',
            'Encode::CN::HZ'        => '2.09',
            'Encode::Encoding'      => '2.08',
            'Encode::GSM0338'       => '2.07',
            'Encode::Guess'         => '2.07',
            'Encode::JP::JIS7'      => '2.07',
            'Encode::KR::2022_KR'   => '2.04',
            'Encode::MIME::Header'  => '2.27',
            'Encode::MIME::Header::ISO_2022_JP'=> '1.09',
            'Encode::Unicode'       => '2.16',
            'Encode::Unicode::UTF7' => '2.10',
            'ExtUtils::CBuilder'    => '0.280228',
            'ExtUtils::CBuilder::Base'=> '0.280228',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.280228',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.280228',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.280228',
            'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280228',
            'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280228',
            'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280228',
            'ExtUtils::CBuilder::Platform::aix'=> '0.280228',
            'ExtUtils::CBuilder::Platform::android'=> '0.280228',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.280228',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.280228',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280228',
            'ExtUtils::CBuilder::Platform::os2'=> '0.280228',
            'File::Glob'            => '1.29',
            'File::Spec'            => '3.68',
            'File::Spec::AmigaOS'   => '3.68',
            'File::Spec::Cygwin'    => '3.68',
            'File::Spec::Epoc'      => '3.68',
            'File::Spec::Functions' => '3.68',
            'File::Spec::Mac'       => '3.68',
            'File::Spec::OS2'       => '3.68',
            'File::Spec::Unix'      => '3.68',
            'File::Spec::VMS'       => '3.68',
            'File::Spec::Win32'     => '3.68',
            'List::Util'            => '1.48',
            'List::Util::XS'        => '1.48',
            'Math::BigRat'          => '0.2613',
            'Module::CoreList'      => '5.20170720',
            'Module::CoreList::TieHashDelta'=> '5.20170720',
            'Module::CoreList::Utils'=> '5.20170720',
            'Opcode'                => '1.40',
            'POSIX'                 => '1.77',
            'PerlIO::scalar'        => '0.29',
            'Scalar::Util'          => '1.48',
            'Sub::Util'             => '1.48',
            'Time::HiRes'           => '1.9743',
            'Time::Piece'           => '1.3201',
            'Time::Seconds'         => '1.3201',
            'Unicode'               => '10.0.0',
            'XS::APItest'           => '0.90',
            'arybase'               => '0.13',
            'encoding'              => '2.20',
            'feature'               => '1.49',
            're'                    => '0.35',
        },
        removed => {
        }
    },
    5.027003 => {
        delta_from => 5.027002,
        changed => {
            'B'                     => '1.69',
            'B::Concise'            => '1.001',
            'B::Debug'              => '1.25',
            'B::Deparse'            => '1.42',
            'B::Op_private'         => '5.027003',
            'Config'                => '5.027003',
            'Data::Dumper'          => '2.167_02',
            'Devel::Peek'           => '1.27',
            'ExtUtils::Constant'    => '0.24',
            'ExtUtils::Constant::Base'=> '0.06',
            'ExtUtils::Constant::ProxySubs'=> '0.09',
            'ExtUtils::Constant::Utils'=> '0.04',
            'ExtUtils::ParseXS'     => '3.35',
            'ExtUtils::ParseXS::Constants'=> '3.35',
            'ExtUtils::ParseXS::CountLines'=> '3.35',
            'ExtUtils::ParseXS::Eval'=> '3.35',
            'ExtUtils::ParseXS::Utilities'=> '3.35',
            'ExtUtils::Typemaps'    => '3.35',
            'ExtUtils::Typemaps::Cmd'=> '3.35',
            'ExtUtils::Typemaps::InputMap'=> '3.35',
            'ExtUtils::Typemaps::OutputMap'=> '3.35',
            'ExtUtils::Typemaps::Type'=> '3.35',
            'Filter::Simple'        => '0.94',
            'Module::CoreList'      => '5.20170821',
            'Module::CoreList::TieHashDelta'=> '5.20170821',
            'Module::CoreList::Utils'=> '5.20170821',
            'SelfLoader'            => '1.24',
            'Storable'              => '2.64',
            'XS::APItest'           => '0.91',
            'base'                  => '2.26',
            'threads'               => '2.17',
            'utf8'                  => '1.20',
        },
        removed => {
        }
    },
    5.027004 => {
        delta_from => 5.027003,
        changed => {
            'B::Op_private'         => '5.027004',
            'Config'                => '5.027004',
            'File::Glob'            => '1.30',
            'I18N::Langinfo'        => '0.14',
            'Module::CoreList'      => '5.20170920',
            'Module::CoreList::TieHashDelta'=> '5.20170920',
            'Module::CoreList::Utils'=> '5.20170920',
            'Term::ReadLine'        => '1.17',
            'VMS::Stdio'            => '2.42',
            'XS::APItest'           => '0.92',
            'attributes'            => '0.31',
            'sort'                  => '2.03',
            'threads'               => '2.18',
        },
        removed => {
        }
    },
    5.024003 => {
        delta_from => 5.024002,
        changed => {
            'B::Op_private'         => '5.024003',
            'Config'                => '5.024003',
            'Module::CoreList'      => '5.20170922_24',
            'Module::CoreList::TieHashDelta'=> '5.20170922_24',
            'Module::CoreList::Utils'=> '5.20170922_24',
            'POSIX'                 => '1.65_01',
            'Time::HiRes'           => '1.9741',
        },
        removed => {
        }
    },
    5.026001 => {
        delta_from => 5.026000,
        changed => {
            'B::Op_private'         => '5.026001',
            'Config'                => '5.026001',
            'Module::CoreList'      => '5.20170922_26',
            'Module::CoreList::TieHashDelta'=> '5.20170922_26',
            'Module::CoreList::Utils'=> '5.20170922_26',
            '_charnames'            => '1.45',
            'base'                  => '2.26',
            'charnames'             => '1.45',
        },
        removed => {
        }
    },
    5.027005 => {
        delta_from => 5.027004,
        changed => {
            'B'                     => '1.70',
            'B::Concise'            => '1.002',
            'B::Deparse'            => '1.43',
            'B::Op_private'         => '5.027005',
            'B::Xref'               => '1.07',
            'Config'                => '5.027005',
            'Config::Perl::V'       => '0.29',
            'Digest::SHA'           => '5.98',
            'Encode'                => '2.93',
            'Encode::CN::HZ'        => '2.10',
            'Encode::JP::JIS7'      => '2.08',
            'Encode::MIME::Header'  => '2.28',
            'Encode::MIME::Name'    => '1.03',
            'File::Fetch'           => '0.54',
            'File::Path'            => '2.15',
            'List::Util'            => '1.49',
            'List::Util::XS'        => '1.49',
            'Locale::Codes'         => '3.54',
            'Locale::Codes::Constants'=> '3.54',
            'Locale::Codes::Country'=> '3.54',
            'Locale::Codes::Country_Codes'=> '3.54',
            'Locale::Codes::Country_Retired'=> '3.54',
            'Locale::Codes::Currency'=> '3.54',
            'Locale::Codes::Currency_Codes'=> '3.54',
            'Locale::Codes::Currency_Retired'=> '3.54',
            'Locale::Codes::LangExt'=> '3.54',
            'Locale::Codes::LangExt_Codes'=> '3.54',
            'Locale::Codes::LangExt_Retired'=> '3.54',
            'Locale::Codes::LangFam'=> '3.54',
            'Locale::Codes::LangFam_Codes'=> '3.54',
            'Locale::Codes::LangFam_Retired'=> '3.54',
            'Locale::Codes::LangVar'=> '3.54',
            'Locale::Codes::LangVar_Codes'=> '3.54',
            'Locale::Codes::LangVar_Retired'=> '3.54',
            'Locale::Codes::Language'=> '3.54',
            'Locale::Codes::Language_Codes'=> '3.54',
            'Locale::Codes::Language_Retired'=> '3.54',
            'Locale::Codes::Script' => '3.54',
            'Locale::Codes::Script_Codes'=> '3.54',
            'Locale::Codes::Script_Retired'=> '3.54',
            'Locale::Country'       => '3.54',
            'Locale::Currency'      => '3.54',
            'Locale::Language'      => '3.54',
            'Locale::Script'        => '3.54',
            'Math::BigFloat'        => '1.999811',
            'Math::BigInt'          => '1.999811',
            'Math::BigInt::Calc'    => '1.999811',
            'Math::BigInt::CalcEmu' => '1.999811',
            'Math::BigInt::FastCalc'=> '0.5006',
            'Math::BigInt::Lib'     => '1.999811',
            'Module::CoreList'      => '5.20171020',
            'Module::CoreList::TieHashDelta'=> '5.20171020',
            'Module::CoreList::Utils'=> '5.20171020',
            'NEXT'                  => '0.67_01',
            'POSIX'                 => '1.78',
            'Pod::Perldoc'          => '3.2801',
            'Scalar::Util'          => '1.49',
            'Sub::Util'             => '1.49',
            'Sys::Hostname'         => '1.21',
            'Test2'                 => '1.302103',
            'Test2::API'            => '1.302103',
            'Test2::API::Breakage'  => '1.302103',
            'Test2::API::Context'   => '1.302103',
            'Test2::API::Instance'  => '1.302103',
            'Test2::API::Stack'     => '1.302103',
            'Test2::Event'          => '1.302103',
            'Test2::Event::Bail'    => '1.302103',
            'Test2::Event::Diag'    => '1.302103',
            'Test2::Event::Encoding'=> '1.302103',
            'Test2::Event::Exception'=> '1.302103',
            'Test2::Event::Fail'    => '1.302103',
            'Test2::Event::Generic' => '1.302103',
            'Test2::Event::Note'    => '1.302103',
            'Test2::Event::Ok'      => '1.302103',
            'Test2::Event::Pass'    => '1.302103',
            'Test2::Event::Plan'    => '1.302103',
            'Test2::Event::Skip'    => '1.302103',
            'Test2::Event::Subtest' => '1.302103',
            'Test2::Event::TAP::Version'=> '1.302103',
            'Test2::Event::Waiting' => '1.302103',
            'Test2::EventFacet'     => '1.302103',
            'Test2::EventFacet::About'=> '1.302103',
            'Test2::EventFacet::Amnesty'=> '1.302103',
            'Test2::EventFacet::Assert'=> '1.302103',
            'Test2::EventFacet::Control'=> '1.302103',
            'Test2::EventFacet::Error'=> '1.302103',
            'Test2::EventFacet::Info'=> '1.302103',
            'Test2::EventFacet::Meta'=> '1.302103',
            'Test2::EventFacet::Parent'=> '1.302103',
            'Test2::EventFacet::Plan'=> '1.302103',
            'Test2::EventFacet::Trace'=> '1.302103',
            'Test2::Formatter'      => '1.302103',
            'Test2::Formatter::TAP' => '1.302103',
            'Test2::Hub'            => '1.302103',
            'Test2::Hub::Interceptor'=> '1.302103',
            'Test2::Hub::Interceptor::Terminator'=> '1.302103',
            'Test2::Hub::Subtest'   => '1.302103',
            'Test2::IPC'            => '1.302103',
            'Test2::IPC::Driver'    => '1.302103',
            'Test2::IPC::Driver::Files'=> '1.302103',
            'Test2::Tools::Tiny'    => '1.302103',
            'Test2::Util'           => '1.302103',
            'Test2::Util::ExternalMeta'=> '1.302103',
            'Test2::Util::Facets2Legacy'=> '1.302103',
            'Test2::Util::HashBase' => '0.005',
            'Test2::Util::Trace'    => '1.302103',
            'Test::Builder'         => '1.302103',
            'Test::Builder::Formatter'=> '1.302103',
            'Test::Builder::IO::Scalar'=> '2.114',
            'Test::Builder::Module' => '1.302103',
            'Test::Builder::Tester' => '1.302103',
            'Test::Builder::Tester::Color'=> '1.302103',
            'Test::Builder::TodoDiag'=> '1.302103',
            'Test::More'            => '1.302103',
            'Test::Simple'          => '1.302103',
            'Test::Tester'          => '1.302103',
            'Test::Tester::Capture' => '1.302103',
            'Test::Tester::CaptureRunner'=> '1.302103',
            'Test::Tester::Delegate'=> '1.302103',
            'Test::use::ok'         => '1.302103',
            'Time::HiRes'           => '1.9746',
            'Time::Piece'           => '1.3202',
            'Time::Seconds'         => '1.3202',
            'arybase'               => '0.14',
            'encoding'              => '2.21',
            'ok'                    => '1.302103',
        },
        removed => {
            'Test2::Event::Info'    => 1,
        }
    },
    5.027006 => {
        delta_from => 5.027005,
        changed => {
            'Attribute::Handlers'   => '1.01',
            'B'                     => '1.72',
            'B::Concise'            => '1.003',
            'B::Deparse'            => '1.45',
            'B::Op_private'         => '5.027006',
            'Carp'                  => '1.44',
            'Carp::Heavy'           => '1.44',
            'Compress::Raw::Zlib'   => '2.075',
            'Config'                => '5.027006',
            'Config::Extensions'    => '0.02',
            'Cwd'                   => '3.70',
            'DynaLoader'            => '1.44',
            'ExtUtils::CBuilder'    => '0.280229',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.280229',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.280229',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.280229',
            'ExtUtils::CBuilder::Platform::aix'=> '0.280229',
            'ExtUtils::CBuilder::Platform::android'=> '0.280229',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.280229',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.280229',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280229',
            'ExtUtils::CBuilder::Platform::os2'=> '0.280229',
            'ExtUtils::Embed'       => '1.35',
            'ExtUtils::Miniperl'    => '1.07',
            'ExtUtils::ParseXS'     => '3.36',
            'ExtUtils::ParseXS::Constants'=> '3.36',
            'ExtUtils::ParseXS::CountLines'=> '3.36',
            'ExtUtils::ParseXS::Eval'=> '3.36',
            'ExtUtils::ParseXS::Utilities'=> '3.36',
            'ExtUtils::Typemaps'    => '3.36',
            'ExtUtils::Typemaps::Cmd'=> '3.36',
            'ExtUtils::Typemaps::InputMap'=> '3.36',
            'ExtUtils::Typemaps::OutputMap'=> '3.36',
            'ExtUtils::Typemaps::Type'=> '3.36',
            'ExtUtils::XSSymSet'    => '1.4',
            'File::Copy'            => '2.33',
            'File::Spec'            => '3.69',
            'File::Spec::AmigaOS'   => '3.69',
            'File::Spec::Cygwin'    => '3.69',
            'File::Spec::Epoc'      => '3.69',
            'File::Spec::Functions' => '3.69',
            'File::Spec::Mac'       => '3.69',
            'File::Spec::OS2'       => '3.69',
            'File::Spec::Unix'      => '3.69',
            'File::Spec::VMS'       => '3.69',
            'File::Spec::Win32'     => '3.69',
            'File::stat'            => '1.08',
            'FileCache'             => '1.10',
            'Filter::Simple'        => '0.95',
            'Hash::Util::FieldHash' => '1.20',
            'I18N::LangTags'        => '0.43',
            'I18N::LangTags::Detect'=> '1.07',
            'I18N::LangTags::List'  => '0.40',
            'I18N::Langinfo'        => '0.15',
            'IO::Handle'            => '1.37',
            'IO::Select'            => '1.23',
            'Locale::Maketext'      => '1.29',
            'Module::CoreList'      => '5.20171120',
            'Module::CoreList::TieHashDelta'=> '5.20171120',
            'Module::CoreList::Utils'=> '5.20171120',
            'Net::Cmd'              => '3.11',
            'Net::Config'           => '3.11',
            'Net::Domain'           => '3.11',
            'Net::FTP'              => '3.11',
            'Net::FTP::A'           => '3.11',
            'Net::FTP::E'           => '3.11',
            'Net::FTP::I'           => '3.11',
            'Net::FTP::L'           => '3.11',
            'Net::FTP::dataconn'    => '3.11',
            'Net::NNTP'             => '3.11',
            'Net::Netrc'            => '3.11',
            'Net::POP3'             => '3.11',
            'Net::Ping'             => '2.62',
            'Net::SMTP'             => '3.11',
            'Net::Time'             => '3.11',
            'Net::hostent'          => '1.02',
            'Net::netent'           => '1.01',
            'Net::protoent'         => '1.01',
            'Net::servent'          => '1.02',
            'O'                     => '1.03',
            'ODBM_File'             => '1.15',
            'Opcode'                => '1.41',
            'POSIX'                 => '1.80',
            'Pod::Html'             => '1.2203',
            'SelfLoader'            => '1.25',
            'Socket'                => '2.020_04',
            'Storable'              => '2.65',
            'Test'                  => '1.31',
            'Test2'                 => '1.302111',
            'Test2::API'            => '1.302111',
            'Test2::API::Breakage'  => '1.302111',
            'Test2::API::Context'   => '1.302111',
            'Test2::API::Instance'  => '1.302111',
            'Test2::API::Stack'     => '1.302111',
            'Test2::Event'          => '1.302111',
            'Test2::Event::Bail'    => '1.302111',
            'Test2::Event::Diag'    => '1.302111',
            'Test2::Event::Encoding'=> '1.302111',
            'Test2::Event::Exception'=> '1.302111',
            'Test2::Event::Fail'    => '1.302111',
            'Test2::Event::Generic' => '1.302111',
            'Test2::Event::Note'    => '1.302111',
            'Test2::Event::Ok'      => '1.302111',
            'Test2::Event::Pass'    => '1.302111',
            'Test2::Event::Plan'    => '1.302111',
            'Test2::Event::Skip'    => '1.302111',
            'Test2::Event::Subtest' => '1.302111',
            'Test2::Event::TAP::Version'=> '1.302111',
            'Test2::Event::Waiting' => '1.302111',
            'Test2::EventFacet'     => '1.302111',
            'Test2::EventFacet::About'=> '1.302111',
            'Test2::EventFacet::Amnesty'=> '1.302111',
            'Test2::EventFacet::Assert'=> '1.302111',
            'Test2::EventFacet::Control'=> '1.302111',
            'Test2::EventFacet::Error'=> '1.302111',
            'Test2::EventFacet::Info'=> '1.302111',
            'Test2::EventFacet::Meta'=> '1.302111',
            'Test2::EventFacet::Parent'=> '1.302111',
            'Test2::EventFacet::Plan'=> '1.302111',
            'Test2::EventFacet::Trace'=> '1.302111',
            'Test2::Formatter'      => '1.302111',
            'Test2::Formatter::TAP' => '1.302111',
            'Test2::Hub'            => '1.302111',
            'Test2::Hub::Interceptor'=> '1.302111',
            'Test2::Hub::Interceptor::Terminator'=> '1.302111',
            'Test2::Hub::Subtest'   => '1.302111',
            'Test2::IPC'            => '1.302111',
            'Test2::IPC::Driver'    => '1.302111',
            'Test2::IPC::Driver::Files'=> '1.302111',
            'Test2::Tools::Tiny'    => '1.302111',
            'Test2::Util'           => '1.302111',
            'Test2::Util::ExternalMeta'=> '1.302111',
            'Test2::Util::Facets2Legacy'=> '1.302111',
            'Test2::Util::HashBase' => '1.302111',
            'Test2::Util::Trace'    => '1.302111',
            'Test::Builder'         => '1.302111',
            'Test::Builder::Formatter'=> '1.302111',
            'Test::Builder::Module' => '1.302111',
            'Test::Builder::Tester' => '1.302111',
            'Test::Builder::Tester::Color'=> '1.302111',
            'Test::Builder::TodoDiag'=> '1.302111',
            'Test::More'            => '1.302111',
            'Test::Simple'          => '1.302111',
            'Test::Tester'          => '1.302111',
            'Test::Tester::Capture' => '1.302111',
            'Test::Tester::CaptureRunner'=> '1.302111',
            'Test::Tester::Delegate'=> '1.302111',
            'Test::use::ok'         => '1.302111',
            'Tie::Array'            => '1.07',
            'Tie::StdHandle'        => '4.5',
            'Time::HiRes'           => '1.9747',
            'Time::gmtime'          => '1.04',
            'Time::localtime'       => '1.03',
            'Unicode::Collate'      => '1.23',
            'Unicode::Collate::CJK::Big5'=> '1.23',
            'Unicode::Collate::CJK::GB2312'=> '1.23',
            'Unicode::Collate::CJK::JISX0208'=> '1.23',
            'Unicode::Collate::CJK::Korean'=> '1.23',
            'Unicode::Collate::CJK::Pinyin'=> '1.23',
            'Unicode::Collate::CJK::Stroke'=> '1.23',
            'Unicode::Collate::CJK::Zhuyin'=> '1.23',
            'Unicode::Collate::Locale'=> '1.23',
            'Unicode::Normalize'    => '1.26',
            'User::grent'           => '1.02',
            'User::pwent'           => '1.01',
            'VMS::DCLsym'           => '1.09',
            'VMS::Stdio'            => '2.44',
            'XS::APItest'           => '0.93',
            'XS::Typemap'           => '0.16',
            'XSLoader'              => '0.28',
            'attributes'            => '0.32',
            'base'                  => '2.27',
            'blib'                  => '1.07',
            'experimental'          => '0.017',
            'fields'                => '2.24',
            'ok'                    => '1.302111',
            're'                    => '0.36',
            'sort'                  => '2.04',
            'threads'               => '2.19',
            'warnings'              => '1.38',
        },
        removed => {
        }
    },
    5.027007 => {
        delta_from => 5.027006,
        changed => {
            'App::Cpan'             => '1.67',
            'B'                     => '1.73',
            'B::Debug'              => '1.26',
            'B::Deparse'            => '1.46',
            'B::Op_private'         => '5.027007',
            'CPAN'                  => '2.20',
            'CPAN::Distribution'    => '2.19',
            'CPAN::FTP'             => '5.5011',
            'CPAN::FirstTime'       => '5.5311',
            'CPAN::Shell'           => '5.5007',
            'Carp'                  => '1.45',
            'Carp::Heavy'           => '1.45',
            'Compress::Raw::Zlib'   => '2.076',
            'Config'                => '5.027007',
            'Cwd'                   => '3.71',
            'Data::Dumper'          => '2.169',
            'Devel::PPPort'         => '3.37',
            'Digest::SHA'           => '6.00',
            'DynaLoader'            => '1.45',
            'ExtUtils::CBuilder'    => '0.280230',
            'ExtUtils::CBuilder::Base'=> '0.280230',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.280230',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.280230',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.280230',
            'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280230',
            'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280230',
            'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280230',
            'ExtUtils::CBuilder::Platform::aix'=> '0.280230',
            'ExtUtils::CBuilder::Platform::android'=> '0.280230',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.280230',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.280230',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280230',
            'ExtUtils::CBuilder::Platform::os2'=> '0.280230',
            'ExtUtils::Typemaps'    => '3.37',
            'File::Fetch'           => '0.56',
            'File::Spec'            => '3.71',
            'File::Spec::AmigaOS'   => '3.71',
            'File::Spec::Cygwin'    => '3.71',
            'File::Spec::Epoc'      => '3.71',
            'File::Spec::Functions' => '3.71',
            'File::Spec::Mac'       => '3.71',
            'File::Spec::OS2'       => '3.71',
            'File::Spec::Unix'      => '3.71',
            'File::Spec::VMS'       => '3.71',
            'File::Spec::Win32'     => '3.71',
            'Filter::Util::Call'    => '1.58',
            'GDBM_File'             => '1.17',
            'JSON::PP'              => '2.97000',
            'JSON::PP::Boolean'     => '2.97000',
            'Locale::Codes'         => '3.55',
            'Locale::Codes::Constants'=> '3.55',
            'Locale::Codes::Country'=> '3.55',
            'Locale::Codes::Country_Codes'=> '3.55',
            'Locale::Codes::Country_Retired'=> '3.55',
            'Locale::Codes::Currency'=> '3.55',
            'Locale::Codes::Currency_Codes'=> '3.55',
            'Locale::Codes::Currency_Retired'=> '3.55',
            'Locale::Codes::LangExt'=> '3.55',
            'Locale::Codes::LangExt_Codes'=> '3.55',
            'Locale::Codes::LangExt_Retired'=> '3.55',
            'Locale::Codes::LangFam'=> '3.55',
            'Locale::Codes::LangFam_Codes'=> '3.55',
            'Locale::Codes::LangFam_Retired'=> '3.55',
            'Locale::Codes::LangVar'=> '3.55',
            'Locale::Codes::LangVar_Codes'=> '3.55',
            'Locale::Codes::LangVar_Retired'=> '3.55',
            'Locale::Codes::Language'=> '3.55',
            'Locale::Codes::Language_Codes'=> '3.55',
            'Locale::Codes::Language_Retired'=> '3.55',
            'Locale::Codes::Script' => '3.55',
            'Locale::Codes::Script_Codes'=> '3.55',
            'Locale::Codes::Script_Retired'=> '3.55',
            'Locale::Country'       => '3.55',
            'Locale::Currency'      => '3.55',
            'Locale::Language'      => '3.55',
            'Locale::Script'        => '3.55',
            'Module::CoreList'      => '5.20171220',
            'Module::CoreList::TieHashDelta'=> '5.20171220',
            'Module::CoreList::Utils'=> '5.20171220',
            'Opcode'                => '1.42',
            'POSIX'                 => '1.81',
            'Pod::Functions'        => '1.12',
            'Pod::Html'             => '1.23',
            'Sys::Hostname'         => '1.22',
            'Test2'                 => '1.302120',
            'Test2::API'            => '1.302120',
            'Test2::API::Breakage'  => '1.302120',
            'Test2::API::Context'   => '1.302120',
            'Test2::API::Instance'  => '1.302120',
            'Test2::API::Stack'     => '1.302120',
            'Test2::Event'          => '1.302120',
            'Test2::Event::Bail'    => '1.302120',
            'Test2::Event::Diag'    => '1.302120',
            'Test2::Event::Encoding'=> '1.302120',
            'Test2::Event::Exception'=> '1.302120',
            'Test2::Event::Fail'    => '1.302120',
            'Test2::Event::Generic' => '1.302120',
            'Test2::Event::Note'    => '1.302120',
            'Test2::Event::Ok'      => '1.302120',
            'Test2::Event::Pass'    => '1.302120',
            'Test2::Event::Plan'    => '1.302120',
            'Test2::Event::Skip'    => '1.302120',
            'Test2::Event::Subtest' => '1.302120',
            'Test2::Event::TAP::Version'=> '1.302120',
            'Test2::Event::Waiting' => '1.302120',
            'Test2::EventFacet'     => '1.302120',
            'Test2::EventFacet::About'=> '1.302120',
            'Test2::EventFacet::Amnesty'=> '1.302120',
            'Test2::EventFacet::Assert'=> '1.302120',
            'Test2::EventFacet::Control'=> '1.302120',
            'Test2::EventFacet::Error'=> '1.302120',
            'Test2::EventFacet::Info'=> '1.302120',
            'Test2::EventFacet::Meta'=> '1.302120',
            'Test2::EventFacet::Parent'=> '1.302120',
            'Test2::EventFacet::Plan'=> '1.302120',
            'Test2::EventFacet::Trace'=> '1.302120',
            'Test2::Formatter'      => '1.302120',
            'Test2::Formatter::TAP' => '1.302120',
            'Test2::Hub'            => '1.302120',
            'Test2::Hub::Interceptor'=> '1.302120',
            'Test2::Hub::Interceptor::Terminator'=> '1.302120',
            'Test2::Hub::Subtest'   => '1.302120',
            'Test2::IPC'            => '1.302120',
            'Test2::IPC::Driver'    => '1.302120',
            'Test2::IPC::Driver::Files'=> '1.302120',
            'Test2::Tools::Tiny'    => '1.302120',
            'Test2::Util'           => '1.302120',
            'Test2::Util::ExternalMeta'=> '1.302120',
            'Test2::Util::Facets2Legacy'=> '1.302120',
            'Test2::Util::HashBase' => '1.302120',
            'Test2::Util::Trace'    => '1.302120',
            'Test::Builder'         => '1.302120',
            'Test::Builder::Formatter'=> '1.302120',
            'Test::Builder::Module' => '1.302120',
            'Test::Builder::Tester' => '1.302120',
            'Test::Builder::Tester::Color'=> '1.302120',
            'Test::Builder::TodoDiag'=> '1.302120',
            'Test::More'            => '1.302120',
            'Test::Simple'          => '1.302120',
            'Test::Tester'          => '1.302120',
            'Test::Tester::Capture' => '1.302120',
            'Test::Tester::CaptureRunner'=> '1.302120',
            'Test::Tester::Delegate'=> '1.302120',
            'Test::use::ok'         => '1.302120',
            'Time::HiRes'           => '1.9748',
            'Time::Piece'           => '1.3203',
            'Time::Seconds'         => '1.3203',
            'Unicode::Collate'      => '1.25',
            'Unicode::Collate::CJK::Big5'=> '1.25',
            'Unicode::Collate::CJK::GB2312'=> '1.25',
            'Unicode::Collate::CJK::JISX0208'=> '1.25',
            'Unicode::Collate::CJK::Korean'=> '1.25',
            'Unicode::Collate::CJK::Pinyin'=> '1.25',
            'Unicode::Collate::CJK::Stroke'=> '1.25',
            'Unicode::Collate::CJK::Zhuyin'=> '1.25',
            'Unicode::Collate::Locale'=> '1.25',
            'Unicode::UCD'          => '0.69',
            'XS::APItest'           => '0.94',
            'XSLoader'              => '0.29',
            'arybase'               => '0.15',
            'autodie::exception'    => '2.29001',
            'autodie::hints'        => '2.29001',
            'experimental'          => '0.019',
            'feature'               => '1.50',
            'ok'                    => '1.302120',
            'overload'              => '1.29',
            'threads'               => '2.21',
            'threads::shared'       => '1.58',
            'warnings'              => '1.39',
        },
        removed => {
        }
    },
    5.027008 => {
        delta_from => 5.027007,
        changed => {
            'B'                     => '1.74',
            'B::Deparse'            => '1.47',
            'B::Op_private'         => '5.027008',
            'Config'                => '5.027008',
            'Cwd'                   => '3.72',
            'Data::Dumper'          => '2.170',
            'Devel::PPPort'         => '3.38',
            'Digest::SHA'           => '6.01',
            'Encode'                => '2.94',
            'Encode::Alias'         => '2.24',
            'ExtUtils::Miniperl'    => '1.08',
            'File::Spec'            => '3.72',
            'File::Spec::AmigaOS'   => '3.72',
            'File::Spec::Cygwin'    => '3.72',
            'File::Spec::Epoc'      => '3.72',
            'File::Spec::Functions' => '3.72',
            'File::Spec::Mac'       => '3.72',
            'File::Spec::OS2'       => '3.72',
            'File::Spec::Unix'      => '3.72',
            'File::Spec::VMS'       => '3.72',
            'File::Spec::Win32'     => '3.72',
            'JSON::PP'              => '2.97001',
            'JSON::PP::Boolean'     => '2.97001',
            'Module::CoreList'      => '5.20180120',
            'Module::CoreList::TieHashDelta'=> '5.20180120',
            'Module::CoreList::Utils'=> '5.20180120',
            'Opcode'                => '1.43',
            'Pod::Functions'        => '1.13',
            'Pod::Html'             => '1.24',
            'Pod::Man'              => '4.10',
            'Pod::ParseLink'        => '4.10',
            'Pod::Text'             => '4.10',
            'Pod::Text::Color'      => '4.10',
            'Pod::Text::Overstrike' => '4.10',
            'Pod::Text::Termcap'    => '4.10',
            'Socket'                => '2.027',
            'Time::HiRes'           => '1.9752',
            'Unicode::UCD'          => '0.70',
            'XS::APItest'           => '0.95',
            'XSLoader'              => '0.30',
            'autodie::exception'    => '2.29002',
            'feature'               => '1.51',
            'overload'              => '1.30',
            'utf8'                  => '1.21',
            'warnings'              => '1.40',
        },
        removed => {
        }
    },
    5.027009 => {
        delta_from => 5.027008,
        changed => {
            'B::Op_private'         => '5.027009',
            'Carp'                  => '1.46',
            'Carp::Heavy'           => '1.46',
            'Config'                => '5.027009',
            'Cwd'                   => '3.74',
            'Devel::PPPort'         => '3.39',
            'Encode'                => '2.96',
            'Encode::Unicode'       => '2.17',
            'Errno'                 => '1.29',
            'ExtUtils::Command'     => '7.32',
            'ExtUtils::Command::MM' => '7.32',
            'ExtUtils::Liblist'     => '7.32',
            'ExtUtils::Liblist::Kid'=> '7.32',
            'ExtUtils::MM'          => '7.32',
            'ExtUtils::MM_AIX'      => '7.32',
            'ExtUtils::MM_Any'      => '7.32',
            'ExtUtils::MM_BeOS'     => '7.32',
            'ExtUtils::MM_Cygwin'   => '7.32',
            'ExtUtils::MM_DOS'      => '7.32',
            'ExtUtils::MM_Darwin'   => '7.32',
            'ExtUtils::MM_MacOS'    => '7.32',
            'ExtUtils::MM_NW5'      => '7.32',
            'ExtUtils::MM_OS2'      => '7.32',
            'ExtUtils::MM_QNX'      => '7.32',
            'ExtUtils::MM_UWIN'     => '7.32',
            'ExtUtils::MM_Unix'     => '7.32',
            'ExtUtils::MM_VMS'      => '7.32',
            'ExtUtils::MM_VOS'      => '7.32',
            'ExtUtils::MM_Win32'    => '7.32',
            'ExtUtils::MM_Win95'    => '7.32',
            'ExtUtils::MY'          => '7.32',
            'ExtUtils::MakeMaker'   => '7.32',
            'ExtUtils::MakeMaker::Config'=> '7.32',
            'ExtUtils::MakeMaker::Locale'=> '7.32',
            'ExtUtils::MakeMaker::version'=> '7.32',
            'ExtUtils::MakeMaker::version::regex'=> '7.32',
            'ExtUtils::Mkbootstrap' => '7.32',
            'ExtUtils::Mksymlists'  => '7.32',
            'ExtUtils::ParseXS'     => '3.38',
            'ExtUtils::ParseXS::Constants'=> '3.38',
            'ExtUtils::ParseXS::CountLines'=> '3.38',
            'ExtUtils::ParseXS::Eval'=> '3.38',
            'ExtUtils::ParseXS::Utilities'=> '3.38',
            'ExtUtils::Typemaps'    => '3.38',
            'ExtUtils::Typemaps::Cmd'=> '3.38',
            'ExtUtils::Typemaps::InputMap'=> '3.38',
            'ExtUtils::Typemaps::OutputMap'=> '3.38',
            'ExtUtils::Typemaps::Type'=> '3.38',
            'ExtUtils::testlib'     => '7.32',
            'File::Spec'            => '3.74',
            'File::Spec::AmigaOS'   => '3.74',
            'File::Spec::Cygwin'    => '3.74',
            'File::Spec::Epoc'      => '3.74',
            'File::Spec::Functions' => '3.74',
            'File::Spec::Mac'       => '3.74',
            'File::Spec::OS2'       => '3.74',
            'File::Spec::Unix'      => '3.74',
            'File::Spec::VMS'       => '3.74',
            'File::Spec::Win32'     => '3.74',
            'IPC::Cmd'              => '1.00',
            'Math::BigFloat::Trace' => '0.49',
            'Math::BigInt::Trace'   => '0.49',
            'Module::CoreList'      => '5.20180220',
            'Module::CoreList::Utils'=> '5.20180220',
            'POSIX'                 => '1.82',
            'PerlIO::encoding'      => '0.26',
            'Storable'              => '3.06',
            'Storable::Limit'       => undef,
            'Test2'                 => '1.302122',
            'Test2::API'            => '1.302122',
            'Test2::API::Breakage'  => '1.302122',
            'Test2::API::Context'   => '1.302122',
            'Test2::API::Instance'  => '1.302122',
            'Test2::API::Stack'     => '1.302122',
            'Test2::Event'          => '1.302122',
            'Test2::Event::Bail'    => '1.302122',
            'Test2::Event::Diag'    => '1.302122',
            'Test2::Event::Encoding'=> '1.302122',
            'Test2::Event::Exception'=> '1.302122',
            'Test2::Event::Fail'    => '1.302122',
            'Test2::Event::Generic' => '1.302122',
            'Test2::Event::Note'    => '1.302122',
            'Test2::Event::Ok'      => '1.302122',
            'Test2::Event::Pass'    => '1.302122',
            'Test2::Event::Plan'    => '1.302122',
            'Test2::Event::Skip'    => '1.302122',
            'Test2::Event::Subtest' => '1.302122',
            'Test2::Event::TAP::Version'=> '1.302122',
            'Test2::Event::Waiting' => '1.302122',
            'Test2::EventFacet'     => '1.302122',
            'Test2::EventFacet::About'=> '1.302122',
            'Test2::EventFacet::Amnesty'=> '1.302122',
            'Test2::EventFacet::Assert'=> '1.302122',
            'Test2::EventFacet::Control'=> '1.302122',
            'Test2::EventFacet::Error'=> '1.302122',
            'Test2::EventFacet::Info'=> '1.302122',
            'Test2::EventFacet::Meta'=> '1.302122',
            'Test2::EventFacet::Parent'=> '1.302122',
            'Test2::EventFacet::Plan'=> '1.302122',
            'Test2::EventFacet::Render'=> '1.302122',
            'Test2::EventFacet::Trace'=> '1.302122',
            'Test2::Formatter'      => '1.302122',
            'Test2::Formatter::TAP' => '1.302122',
            'Test2::Hub'            => '1.302122',
            'Test2::Hub::Interceptor'=> '1.302122',
            'Test2::Hub::Interceptor::Terminator'=> '1.302122',
            'Test2::Hub::Subtest'   => '1.302122',
            'Test2::IPC'            => '1.302122',
            'Test2::IPC::Driver'    => '1.302122',
            'Test2::IPC::Driver::Files'=> '1.302122',
            'Test2::Tools::Tiny'    => '1.302122',
            'Test2::Util'           => '1.302122',
            'Test2::Util::ExternalMeta'=> '1.302122',
            'Test2::Util::Facets2Legacy'=> '1.302122',
            'Test2::Util::HashBase' => '1.302122',
            'Test2::Util::Trace'    => '1.302122',
            'Test::Builder'         => '1.302122',
            'Test::Builder::Formatter'=> '1.302122',
            'Test::Builder::Module' => '1.302122',
            'Test::Builder::Tester' => '1.302122',
            'Test::Builder::Tester::Color'=> '1.302122',
            'Test::Builder::TodoDiag'=> '1.302122',
            'Test::More'            => '1.302122',
            'Test::Simple'          => '1.302122',
            'Test::Tester'          => '1.302122',
            'Test::Tester::Capture' => '1.302122',
            'Test::Tester::CaptureRunner'=> '1.302122',
            'Test::Tester::Delegate'=> '1.302122',
            'Test::use::ok'         => '1.302122',
            'Time::HiRes'           => '1.9753',
            'XS::APItest'           => '0.96',
            'bigint'                => '0.49',
            'bignum'                => '0.49',
            'bigrat'                => '0.49',
            'encoding'              => '2.22',
            'if'                    => '0.0608',
            'mro'                   => '1.22',
            'ok'                    => '1.302122',
            'threads'               => '2.22',
            'warnings'              => '1.41',
        },
        removed => {
            'Module::CoreList::TieHashDelta'=> 1,
        }
    },
    5.027010 => {
        delta_from => 5.027009,
        changed => {
            'App::Prove'            => '3.42',
            'App::Prove::State'     => '3.42',
            'App::Prove::State::Result'=> '3.42',
            'App::Prove::State::Result::Test'=> '3.42',
            'B::Deparse'            => '1.48',
            'B::Op_private'         => '5.027010',
            'Carp'                  => '1.49',
            'Carp::Heavy'           => '1.49',
            'Config'                => '5.02701',
            'Encode'                => '2.97',
            'ExtUtils::Command'     => '7.34',
            'ExtUtils::Command::MM' => '7.34',
            'ExtUtils::Liblist'     => '7.34',
            'ExtUtils::Liblist::Kid'=> '7.34',
            'ExtUtils::MM'          => '7.34',
            'ExtUtils::MM_AIX'      => '7.34',
            'ExtUtils::MM_Any'      => '7.34',
            'ExtUtils::MM_BeOS'     => '7.34',
            'ExtUtils::MM_Cygwin'   => '7.34',
            'ExtUtils::MM_DOS'      => '7.34',
            'ExtUtils::MM_Darwin'   => '7.34',
            'ExtUtils::MM_MacOS'    => '7.34',
            'ExtUtils::MM_NW5'      => '7.34',
            'ExtUtils::MM_OS2'      => '7.34',
            'ExtUtils::MM_QNX'      => '7.34',
            'ExtUtils::MM_UWIN'     => '7.34',
            'ExtUtils::MM_Unix'     => '7.34',
            'ExtUtils::MM_VMS'      => '7.34',
            'ExtUtils::MM_VOS'      => '7.34',
            'ExtUtils::MM_Win32'    => '7.34',
            'ExtUtils::MM_Win95'    => '7.34',
            'ExtUtils::MY'          => '7.34',
            'ExtUtils::MakeMaker'   => '7.34',
            'ExtUtils::MakeMaker::Config'=> '7.34',
            'ExtUtils::MakeMaker::Locale'=> '7.34',
            'ExtUtils::MakeMaker::version'=> '7.34',
            'ExtUtils::MakeMaker::version::regex'=> '7.34',
            'ExtUtils::Mkbootstrap' => '7.34',
            'ExtUtils::Mksymlists'  => '7.34',
            'ExtUtils::ParseXS'     => '3.39',
            'ExtUtils::ParseXS::Constants'=> '3.39',
            'ExtUtils::ParseXS::CountLines'=> '3.39',
            'ExtUtils::ParseXS::Eval'=> '3.39',
            'ExtUtils::ParseXS::Utilities'=> '3.39',
            'ExtUtils::testlib'     => '7.34',
            'File::Glob'            => '1.31',
            'I18N::Langinfo'        => '0.16',
            'List::Util'            => '1.50',
            'List::Util::XS'        => '1.50',
            'Locale::Codes'         => '3.56',
            'Locale::Codes::Constants'=> '3.56',
            'Locale::Codes::Country'=> '3.56',
            'Locale::Codes::Country_Codes'=> '3.56',
            'Locale::Codes::Country_Retired'=> '3.56',
            'Locale::Codes::Currency'=> '3.56',
            'Locale::Codes::Currency_Codes'=> '3.56',
            'Locale::Codes::Currency_Retired'=> '3.56',
            'Locale::Codes::LangExt'=> '3.56',
            'Locale::Codes::LangExt_Codes'=> '3.56',
            'Locale::Codes::LangExt_Retired'=> '3.56',
            'Locale::Codes::LangFam'=> '3.56',
            'Locale::Codes::LangFam_Codes'=> '3.56',
            'Locale::Codes::LangFam_Retired'=> '3.56',
            'Locale::Codes::LangVar'=> '3.56',
            'Locale::Codes::LangVar_Codes'=> '3.56',
            'Locale::Codes::LangVar_Retired'=> '3.56',
            'Locale::Codes::Language'=> '3.56',
            'Locale::Codes::Language_Codes'=> '3.56',
            'Locale::Codes::Language_Retired'=> '3.56',
            'Locale::Codes::Script' => '3.56',
            'Locale::Codes::Script_Codes'=> '3.56',
            'Locale::Codes::Script_Retired'=> '3.56',
            'Locale::Country'       => '3.56',
            'Locale::Currency'      => '3.56',
            'Locale::Language'      => '3.56',
            'Locale::Script'        => '3.56',
            'Module::CoreList'      => '5.20180221',
            'Module::CoreList::Utils'=> '5.20180221',
            'POSIX'                 => '1.83',
            'Scalar::Util'          => '1.50',
            'Sub::Util'             => '1.50',
            'TAP::Base'             => '3.42',
            'TAP::Formatter::Base'  => '3.42',
            'TAP::Formatter::Color' => '3.42',
            'TAP::Formatter::Console'=> '3.42',
            'TAP::Formatter::Console::ParallelSession'=> '3.42',
            'TAP::Formatter::Console::Session'=> '3.42',
            'TAP::Formatter::File'  => '3.42',
            'TAP::Formatter::File::Session'=> '3.42',
            'TAP::Formatter::Session'=> '3.42',
            'TAP::Harness'          => '3.42',
            'TAP::Harness::Env'     => '3.42',
            'TAP::Object'           => '3.42',
            'TAP::Parser'           => '3.42',
            'TAP::Parser::Aggregator'=> '3.42',
            'TAP::Parser::Grammar'  => '3.42',
            'TAP::Parser::Iterator' => '3.42',
            'TAP::Parser::Iterator::Array'=> '3.42',
            'TAP::Parser::Iterator::Process'=> '3.42',
            'TAP::Parser::Iterator::Stream'=> '3.42',
            'TAP::Parser::IteratorFactory'=> '3.42',
            'TAP::Parser::Multiplexer'=> '3.42',
            'TAP::Parser::Result'   => '3.42',
            'TAP::Parser::Result::Bailout'=> '3.42',
            'TAP::Parser::Result::Comment'=> '3.42',
            'TAP::Parser::Result::Plan'=> '3.42',
            'TAP::Parser::Result::Pragma'=> '3.42',
            'TAP::Parser::Result::Test'=> '3.42',
            'TAP::Parser::Result::Unknown'=> '3.42',
            'TAP::Parser::Result::Version'=> '3.42',
            'TAP::Parser::Result::YAML'=> '3.42',
            'TAP::Parser::ResultFactory'=> '3.42',
            'TAP::Parser::Scheduler'=> '3.42',
            'TAP::Parser::Scheduler::Job'=> '3.42',
            'TAP::Parser::Scheduler::Spinner'=> '3.42',
            'TAP::Parser::Source'   => '3.42',
            'TAP::Parser::SourceHandler'=> '3.42',
            'TAP::Parser::SourceHandler::Executable'=> '3.42',
            'TAP::Parser::SourceHandler::File'=> '3.42',
            'TAP::Parser::SourceHandler::Handle'=> '3.42',
            'TAP::Parser::SourceHandler::Perl'=> '3.42',
            'TAP::Parser::SourceHandler::RawTAP'=> '3.42',
            'TAP::Parser::YAMLish::Reader'=> '3.42',
            'TAP::Parser::YAMLish::Writer'=> '3.42',
            'Test2'                 => '1.302133',
            'Test2::API'            => '1.302133',
            'Test2::API::Breakage'  => '1.302133',
            'Test2::API::Context'   => '1.302133',
            'Test2::API::Instance'  => '1.302133',
            'Test2::API::Stack'     => '1.302133',
            'Test2::Event'          => '1.302133',
            'Test2::Event::Bail'    => '1.302133',
            'Test2::Event::Diag'    => '1.302133',
            'Test2::Event::Encoding'=> '1.302133',
            'Test2::Event::Exception'=> '1.302133',
            'Test2::Event::Fail'    => '1.302133',
            'Test2::Event::Generic' => '1.302133',
            'Test2::Event::Note'    => '1.302133',
            'Test2::Event::Ok'      => '1.302133',
            'Test2::Event::Pass'    => '1.302133',
            'Test2::Event::Plan'    => '1.302133',
            'Test2::Event::Skip'    => '1.302133',
            'Test2::Event::Subtest' => '1.302133',
            'Test2::Event::TAP::Version'=> '1.302133',
            'Test2::Event::V2'      => '1.302133',
            'Test2::Event::Waiting' => '1.302133',
            'Test2::EventFacet'     => '1.302133',
            'Test2::EventFacet::About'=> '1.302133',
            'Test2::EventFacet::Amnesty'=> '1.302133',
            'Test2::EventFacet::Assert'=> '1.302133',
            'Test2::EventFacet::Control'=> '1.302133',
            'Test2::EventFacet::Error'=> '1.302133',
            'Test2::EventFacet::Hub'=> '1.302133',
            'Test2::EventFacet::Info'=> '1.302133',
            'Test2::EventFacet::Meta'=> '1.302133',
            'Test2::EventFacet::Parent'=> '1.302133',
            'Test2::EventFacet::Plan'=> '1.302133',
            'Test2::EventFacet::Render'=> '1.302133',
            'Test2::EventFacet::Trace'=> '1.302133',
            'Test2::Formatter'      => '1.302133',
            'Test2::Formatter::TAP' => '1.302133',
            'Test2::Hub'            => '1.302133',
            'Test2::Hub::Interceptor'=> '1.302133',
            'Test2::Hub::Interceptor::Terminator'=> '1.302133',
            'Test2::Hub::Subtest'   => '1.302133',
            'Test2::IPC'            => '1.302133',
            'Test2::IPC::Driver'    => '1.302133',
            'Test2::IPC::Driver::Files'=> '1.302133',
            'Test2::Tools::Tiny'    => '1.302133',
            'Test2::Util'           => '1.302133',
            'Test2::Util::ExternalMeta'=> '1.302133',
            'Test2::Util::Facets2Legacy'=> '1.302133',
            'Test2::Util::HashBase' => '1.302133',
            'Test2::Util::Trace'    => '1.302133',
            'Test::Builder'         => '1.302133',
            'Test::Builder::Formatter'=> '1.302133',
            'Test::Builder::Module' => '1.302133',
            'Test::Builder::Tester' => '1.302133',
            'Test::Builder::Tester::Color'=> '1.302133',
            'Test::Builder::TodoDiag'=> '1.302133',
            'Test::Harness'         => '3.42',
            'Test::More'            => '1.302133',
            'Test::Simple'          => '1.302133',
            'Test::Tester'          => '1.302133',
            'Test::Tester::Capture' => '1.302133',
            'Test::Tester::CaptureRunner'=> '1.302133',
            'Test::Tester::Delegate'=> '1.302133',
            'Test::use::ok'         => '1.302133',
            'Time::HiRes'           => '1.9757',
            'Time::Piece'           => '1.3204',
            'Time::Seconds'         => '1.3204',
            'attributes'            => '0.33',
            'ok'                    => '1.302133',
            'warnings'              => '1.42',
        },
        removed => {
        }
    },
    5.024004 => {
        delta_from => 5.024003,
        changed => {
            'B::Op_private'         => '5.024004',
            'Config'                => '5.024004',
            'Module::CoreList'      => '5.20180414_24',
            'Module::CoreList::TieHashDelta'=> '5.20180414_24',
            'Module::CoreList::Utils'=> '5.20180414_24',
        },
        removed => {
        }
    },
    5.026002 => {
        delta_from => 5.026001,
        changed => {
            'B::Op_private'         => '5.026002',
            'Config'                => '5.026002',
            'Module::CoreList'      => '5.20180414_26',
            'Module::CoreList::TieHashDelta'=> '5.20180414_26',
            'Module::CoreList::Utils'=> '5.20180414_26',
            'PerlIO::via'           => '0.17',
            'Term::ReadLine'        => '1.17',
            'Unicode::UCD'          => '0.69',
        },
        removed => {
        }
    },
    5.027011 => {
        delta_from => 5.027010,
        changed => {
            'B::Op_private'         => '5.027011',
            'Carp'                  => '1.50',
            'Carp::Heavy'           => '1.50',
            'Config'                => '5.027011',
            'Devel::PPPort'         => '3.40',
            'Exporter'              => '5.73',
            'Exporter::Heavy'       => '5.73',
            'ExtUtils::Constant'    => '0.25',
            'I18N::Langinfo'        => '0.17',
            'IO'                    => '1.39',
            'IO::Dir'               => '1.39',
            'IO::File'              => '1.39',
            'IO::Handle'            => '1.39',
            'IO::Pipe'              => '1.39',
            'IO::Poll'              => '1.39',
            'IO::Seekable'          => '1.39',
            'IO::Select'            => '1.39',
            'IO::Socket'            => '1.39',
            'IO::Socket::INET'      => '1.39',
            'IO::Socket::UNIX'      => '1.39',
            'Module::CoreList'      => '5.20180420',
            'Module::CoreList::Utils'=> '5.20180420',
            'POSIX'                 => '1.84',
            'Time::HiRes'           => '1.9759',
            'XS::APItest'           => '0.97',
            'bytes'                 => '1.06',
            'subs'                  => '1.03',
            'vars'                  => '1.04',
            'version'               => '0.9923',
            'version::regex'        => '0.9923',
        },
        removed => {
        }
    },
    5.028000 => {
        delta_from => 5.027011,
        changed => {
            'Archive::Tar'          => '2.30',
            'Archive::Tar::Constant'=> '2.30',
            'Archive::Tar::File'    => '2.30',
            'B::Op_private'         => '5.028000',
            'Config'                => '5.028',
            'Module::CoreList'      => '5.20180622',
            'Module::CoreList::Utils'=> '5.20180622',
            'Storable'              => '3.08',
            'XS::APItest'           => '0.98',
            'feature'               => '1.52',
        },
        removed => {
        }
    },
    5.029000 => {
        delta_from => 5.028,
        changed => {
            'B::Op_private'         => '5.029000',
            'Config'                => '5.029',
            'Module::CoreList'      => '5.20180626',
            'Module::CoreList::Utils'=> '5.20180626',
            'Unicode::UCD'          => '0.71',
            'XS::APItest'           => '0.99',
            'feature'               => '1.53',
        },
        removed => {
        }
    },
    5.029001 => {
        delta_from => 5.029000,
        changed => {
            'B::Op_private'         => '5.029001',
            'Compress::Raw::Bzip2'  => '2.081',
            'Compress::Raw::Zlib'   => '2.081',
            'Compress::Zlib'        => '2.081',
            'Config'                => '5.029001',
            'Config::Perl::V'       => '0.30',
            'DB_File'               => '1.842',
            'Devel::PPPort'         => '3.42',
            'Digest::SHA'           => '6.02',
            'ExtUtils::Manifest'    => '1.71',
            'File::GlobMapper'      => '1.001',
            'File::Temp'            => '0.2308',
            'IO::Compress::Adapter::Bzip2'=> '2.081',
            'IO::Compress::Adapter::Deflate'=> '2.081',
            'IO::Compress::Adapter::Identity'=> '2.081',
            'IO::Compress::Base'    => '2.081',
            'IO::Compress::Base::Common'=> '2.081',
            'IO::Compress::Bzip2'   => '2.081',
            'IO::Compress::Deflate' => '2.081',
            'IO::Compress::Gzip'    => '2.081',
            'IO::Compress::Gzip::Constants'=> '2.081',
            'IO::Compress::RawDeflate'=> '2.081',
            'IO::Compress::Zip'     => '2.081',
            'IO::Compress::Zip::Constants'=> '2.081',
            'IO::Compress::Zlib::Constants'=> '2.081',
            'IO::Compress::Zlib::Extra'=> '2.081',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.081',
            'IO::Uncompress::Adapter::Identity'=> '2.081',
            'IO::Uncompress::Adapter::Inflate'=> '2.081',
            'IO::Uncompress::AnyInflate'=> '2.081',
            'IO::Uncompress::AnyUncompress'=> '2.081',
            'IO::Uncompress::Base'  => '2.081',
            'IO::Uncompress::Bunzip2'=> '2.081',
            'IO::Uncompress::Gunzip'=> '2.081',
            'IO::Uncompress::Inflate'=> '2.081',
            'IO::Uncompress::RawInflate'=> '2.081',
            'IO::Uncompress::Unzip' => '2.081',
            'IPC::Cmd'              => '1.02',
            'Locale::Codes'         => '3.57',
            'Locale::Codes::Constants'=> '3.57',
            'Locale::Codes::Country'=> '3.57',
            'Locale::Codes::Country_Codes'=> '3.57',
            'Locale::Codes::Country_Retired'=> '3.57',
            'Locale::Codes::Currency'=> '3.57',
            'Locale::Codes::Currency_Codes'=> '3.57',
            'Locale::Codes::Currency_Retired'=> '3.57',
            'Locale::Codes::LangExt'=> '3.57',
            'Locale::Codes::LangExt_Codes'=> '3.57',
            'Locale::Codes::LangExt_Retired'=> '3.57',
            'Locale::Codes::LangFam'=> '3.57',
            'Locale::Codes::LangFam_Codes'=> '3.57',
            'Locale::Codes::LangFam_Retired'=> '3.57',
            'Locale::Codes::LangVar'=> '3.57',
            'Locale::Codes::LangVar_Codes'=> '3.57',
            'Locale::Codes::LangVar_Retired'=> '3.57',
            'Locale::Codes::Language'=> '3.57',
            'Locale::Codes::Language_Codes'=> '3.57',
            'Locale::Codes::Language_Retired'=> '3.57',
            'Locale::Codes::Script' => '3.57',
            'Locale::Codes::Script_Codes'=> '3.57',
            'Locale::Codes::Script_Retired'=> '3.57',
            'Locale::Country'       => '3.57',
            'Locale::Currency'      => '3.57',
            'Locale::Language'      => '3.57',
            'Locale::Script'        => '3.57',
            'Math::BigFloat'        => '1.999813',
            'Math::BigFloat::Trace' => '0.50',
            'Math::BigInt'          => '1.999813',
            'Math::BigInt::Calc'    => '1.999813',
            'Math::BigInt::CalcEmu' => '1.999813',
            'Math::BigInt::FastCalc'=> '0.5007',
            'Math::BigInt::Lib'     => '1.999813',
            'Math::BigInt::Trace'   => '0.50',
            'Math::BigRat'          => '0.2614',
            'Module::CoreList'      => '5.20180720',
            'Module::CoreList::Utils'=> '5.20180720',
            'Pod::Man'              => '4.11',
            'Pod::ParseLink'        => '4.11',
            'Pod::Text'             => '4.11',
            'Pod::Text::Color'      => '4.11',
            'Pod::Text::Overstrike' => '4.11',
            'Pod::Text::Termcap'    => '4.11',
            'Storable'              => '3.11',
            'Test2'                 => '1.302138',
            'Test2::API'            => '1.302138',
            'Test2::API::Breakage'  => '1.302138',
            'Test2::API::Context'   => '1.302138',
            'Test2::API::Instance'  => '1.302138',
            'Test2::API::Stack'     => '1.302138',
            'Test2::Event'          => '1.302138',
            'Test2::Event::Bail'    => '1.302138',
            'Test2::Event::Diag'    => '1.302138',
            'Test2::Event::Encoding'=> '1.302138',
            'Test2::Event::Exception'=> '1.302138',
            'Test2::Event::Fail'    => '1.302138',
            'Test2::Event::Generic' => '1.302138',
            'Test2::Event::Note'    => '1.302138',
            'Test2::Event::Ok'      => '1.302138',
            'Test2::Event::Pass'    => '1.302138',
            'Test2::Event::Plan'    => '1.302138',
            'Test2::Event::Skip'    => '1.302138',
            'Test2::Event::Subtest' => '1.302138',
            'Test2::Event::TAP::Version'=> '1.302138',
            'Test2::Event::V2'      => '1.302138',
            'Test2::Event::Waiting' => '1.302138',
            'Test2::EventFacet'     => '1.302138',
            'Test2::EventFacet::About'=> '1.302138',
            'Test2::EventFacet::Amnesty'=> '1.302138',
            'Test2::EventFacet::Assert'=> '1.302138',
            'Test2::EventFacet::Control'=> '1.302138',
            'Test2::EventFacet::Error'=> '1.302138',
            'Test2::EventFacet::Hub'=> '1.302138',
            'Test2::EventFacet::Info'=> '1.302138',
            'Test2::EventFacet::Meta'=> '1.302138',
            'Test2::EventFacet::Parent'=> '1.302138',
            'Test2::EventFacet::Plan'=> '1.302138',
            'Test2::EventFacet::Render'=> '1.302138',
            'Test2::EventFacet::Trace'=> '1.302138',
            'Test2::Formatter'      => '1.302138',
            'Test2::Formatter::TAP' => '1.302138',
            'Test2::Hub'            => '1.302138',
            'Test2::Hub::Interceptor'=> '1.302138',
            'Test2::Hub::Interceptor::Terminator'=> '1.302138',
            'Test2::Hub::Subtest'   => '1.302138',
            'Test2::IPC'            => '1.302138',
            'Test2::IPC::Driver'    => '1.302138',
            'Test2::IPC::Driver::Files'=> '1.302138',
            'Test2::Tools::Tiny'    => '1.302138',
            'Test2::Util'           => '1.302138',
            'Test2::Util::ExternalMeta'=> '1.302138',
            'Test2::Util::Facets2Legacy'=> '1.302138',
            'Test2::Util::HashBase' => '1.302138',
            'Test2::Util::Trace'    => '1.302138',
            'Test::Builder'         => '1.302138',
            'Test::Builder::Formatter'=> '1.302138',
            'Test::Builder::Module' => '1.302138',
            'Test::Builder::Tester' => '1.302138',
            'Test::Builder::Tester::Color'=> '1.302138',
            'Test::Builder::TodoDiag'=> '1.302138',
            'Test::More'            => '1.302138',
            'Test::Simple'          => '1.302138',
            'Test::Tester'          => '1.302138',
            'Test::Tester::Capture' => '1.302138',
            'Test::Tester::CaptureRunner'=> '1.302138',
            'Test::Tester::Delegate'=> '1.302138',
            'Test::use::ok'         => '1.302138',
            'Thread::Queue'         => '3.13',
            'Time::Local'           => '1.28',
            'bigint'                => '0.50',
            'bignum'                => '0.50',
            'bigrat'                => '0.50',
            'experimental'          => '0.020',
            'ok'                    => '1.302138',
            'parent'                => '0.237',
            'perlfaq'               => '5.20180605',
            'version'               => '0.9924',
            'version::regex'        => '0.9924',
        },
        removed => {
        }
    },
    5.029002 => {
        delta_from => 5.029001,
        changed => {
            'B::Op_private'         => '5.029002',
            'Config'                => '5.029002',
            'Config::Extensions'    => '0.03',
            'Cwd'                   => '3.75',
            'Data::Dumper'          => '2.171',
            'Filter::Util::Call'    => '1.59',
            'HTTP::Tiny'            => '0.076',
            'Module::CoreList'      => '5.20180820',
            'Module::CoreList::Utils'=> '5.20180820',
            'PerlIO::scalar'        => '0.30',
            'Storable'              => '3.12',
            'Test2'                 => '1.302140',
            'Test2::API'            => '1.302140',
            'Test2::API::Breakage'  => '1.302140',
            'Test2::API::Context'   => '1.302140',
            'Test2::API::Instance'  => '1.302140',
            'Test2::API::Stack'     => '1.302140',
            'Test2::Event'          => '1.302140',
            'Test2::Event::Bail'    => '1.302140',
            'Test2::Event::Diag'    => '1.302140',
            'Test2::Event::Encoding'=> '1.302140',
            'Test2::Event::Exception'=> '1.302140',
            'Test2::Event::Fail'    => '1.302140',
            'Test2::Event::Generic' => '1.302140',
            'Test2::Event::Note'    => '1.302140',
            'Test2::Event::Ok'      => '1.302140',
            'Test2::Event::Pass'    => '1.302140',
            'Test2::Event::Plan'    => '1.302140',
            'Test2::Event::Skip'    => '1.302140',
            'Test2::Event::Subtest' => '1.302140',
            'Test2::Event::TAP::Version'=> '1.302140',
            'Test2::Event::V2'      => '1.302140',
            'Test2::Event::Waiting' => '1.302140',
            'Test2::EventFacet'     => '1.302140',
            'Test2::EventFacet::About'=> '1.302140',
            'Test2::EventFacet::Amnesty'=> '1.302140',
            'Test2::EventFacet::Assert'=> '1.302140',
            'Test2::EventFacet::Control'=> '1.302140',
            'Test2::EventFacet::Error'=> '1.302140',
            'Test2::EventFacet::Hub'=> '1.302140',
            'Test2::EventFacet::Info'=> '1.302140',
            'Test2::EventFacet::Meta'=> '1.302140',
            'Test2::EventFacet::Parent'=> '1.302140',
            'Test2::EventFacet::Plan'=> '1.302140',
            'Test2::EventFacet::Render'=> '1.302140',
            'Test2::EventFacet::Trace'=> '1.302140',
            'Test2::Formatter'      => '1.302140',
            'Test2::Formatter::TAP' => '1.302140',
            'Test2::Hub'            => '1.302140',
            'Test2::Hub::Interceptor'=> '1.302140',
            'Test2::Hub::Interceptor::Terminator'=> '1.302140',
            'Test2::Hub::Subtest'   => '1.302140',
            'Test2::IPC'            => '1.302140',
            'Test2::IPC::Driver'    => '1.302140',
            'Test2::IPC::Driver::Files'=> '1.302140',
            'Test2::Tools::Tiny'    => '1.302140',
            'Test2::Util'           => '1.302140',
            'Test2::Util::ExternalMeta'=> '1.302140',
            'Test2::Util::Facets2Legacy'=> '1.302140',
            'Test2::Util::HashBase' => '1.302140',
            'Test2::Util::Trace'    => '1.302140',
            'Test::Builder'         => '1.302140',
            'Test::Builder::Formatter'=> '1.302140',
            'Test::Builder::Module' => '1.302140',
            'Test::Builder::Tester' => '1.302140',
            'Test::Builder::Tester::Color'=> '1.302140',
            'Test::Builder::TodoDiag'=> '1.302140',
            'Test::More'            => '1.302140',
            'Test::Simple'          => '1.302140',
            'Test::Tester'          => '1.302140',
            'Test::Tester::Capture' => '1.302140',
            'Test::Tester::CaptureRunner'=> '1.302140',
            'Test::Tester::Delegate'=> '1.302140',
            'Test::use::ok'         => '1.302140',
            'Time::HiRes'           => '1.9760',
            'Time::Piece'           => '1.33',
            'Time::Seconds'         => '1.33',
            'Unicode'               => '11.0.0',
            'ok'                    => '1.302140',
            'warnings'              => '1.43',
        },
        removed => {
        }
    },
    5.029003 => {
        delta_from => 5.029002,
        changed => {
            'Archive::Tar'          => '2.32',
            'Archive::Tar::Constant'=> '2.32',
            'Archive::Tar::File'    => '2.32',
            'B::Op_private'         => '5.029003',
            'Config'                => '5.029003',
            'Data::Dumper'          => '2.172',
            'Devel::PPPort'         => '3.43',
            'File::Path'            => '2.16',
            'File::Spec'            => '3.75',
            'File::Spec::AmigaOS'   => '3.75',
            'File::Spec::Cygwin'    => '3.75',
            'File::Spec::Epoc'      => '3.75',
            'File::Spec::Functions' => '3.75',
            'File::Spec::Mac'       => '3.75',
            'File::Spec::OS2'       => '3.75',
            'File::Spec::Unix'      => '3.75',
            'File::Spec::VMS'       => '3.75',
            'File::Spec::Win32'     => '3.75',
            'Module::CoreList'      => '5.20180920',
            'Module::CoreList::Utils'=> '5.20180920',
            'POSIX'                 => '1.85',
            'Storable'              => '3.13',
            'User::grent'           => '1.03',
            'perlfaq'               => '5.20180915',
        },
        removed => {
            'Locale::Codes'         => 1,
            'Locale::Codes::Constants'=> 1,
            'Locale::Codes::Country'=> 1,
            'Locale::Codes::Country_Codes'=> 1,
            'Locale::Codes::Country_Retired'=> 1,
            'Locale::Codes::Currency'=> 1,
            'Locale::Codes::Currency_Codes'=> 1,
            'Locale::Codes::Currency_Retired'=> 1,
            'Locale::Codes::LangExt'=> 1,
            'Locale::Codes::LangExt_Codes'=> 1,
            'Locale::Codes::LangExt_Retired'=> 1,
            'Locale::Codes::LangFam'=> 1,
            'Locale::Codes::LangFam_Codes'=> 1,
            'Locale::Codes::LangFam_Retired'=> 1,
            'Locale::Codes::LangVar'=> 1,
            'Locale::Codes::LangVar_Codes'=> 1,
            'Locale::Codes::LangVar_Retired'=> 1,
            'Locale::Codes::Language'=> 1,
            'Locale::Codes::Language_Codes'=> 1,
            'Locale::Codes::Language_Retired'=> 1,
            'Locale::Codes::Script' => 1,
            'Locale::Codes::Script_Codes'=> 1,
            'Locale::Codes::Script_Retired'=> 1,
            'Locale::Country'       => 1,
            'Locale::Currency'      => 1,
            'Locale::Language'      => 1,
            'Locale::Script'        => 1,
        }
    },
    5.029004 => {
        delta_from => 5.029003,
        changed => {
            'App::Cpan'             => '1.671',
            'B'                     => '1.75',
            'B::Concise'            => '1.004',
            'B::Deparse'            => '1.49',
            'B::Op_private'         => '5.029004',
            'B::Terse'              => '1.09',
            'CPAN'                  => '2.21',
            'CPAN::Distribution'    => '2.21',
            'CPAN::Mirrors'         => '2.21',
            'CPAN::Plugin'          => '0.97',
            'CPAN::Shell'           => '5.5008',
            'Config'                => '5.029004',
            'Devel::Peek'           => '1.28',
            'File::Copy'            => '2.34',
            'File::Glob'            => '1.32',
            'Math::BigFloat::Trace' => '0.51',
            'Math::BigInt::Trace'   => '0.51',
            'Module::CoreList'      => '5.20181020',
            'Module::CoreList::Utils'=> '5.20181020',
            'Unicode::UCD'          => '0.72',
            'bigint'                => '0.51',
            'bignum'                => '0.51',
            'bigrat'                => '0.51',
            'bytes'                 => '1.07',
            'feature'               => '1.54',
            'sigtrap'               => '1.09',
            'vars'                  => '1.05',
        },
        removed => {
            'B::Debug'              => 1,
            'arybase'               => 1,
        }
    },
    5.029005 => {
        delta_from => 5.029004,
        changed => {
            'B::Op_private'         => '5.029005',
            'Config'                => '5.029005',
            'Cwd'                   => '3.76',
            'Data::Dumper'          => '2.173',
            'Errno'                 => '1.30',
            'File::Spec'            => '3.76',
            'File::Spec::AmigaOS'   => '3.76',
            'File::Spec::Cygwin'    => '3.76',
            'File::Spec::Epoc'      => '3.76',
            'File::Spec::Functions' => '3.76',
            'File::Spec::Mac'       => '3.76',
            'File::Spec::OS2'       => '3.76',
            'File::Spec::Unix'      => '3.76',
            'File::Spec::VMS'       => '3.76',
            'File::Spec::Win32'     => '3.76',
            'GDBM_File'             => '1.18',
            'Module::CoreList'      => '5.20181120',
            'Module::CoreList::Utils'=> '5.20181120',
            'NDBM_File'             => '1.15',
            'ODBM_File'             => '1.16',
            'SDBM_File'             => '1.15',
            're'                    => '0.37',
        },
        removed => {
        }
    },
    5.026003 => {
        delta_from => 5.026002,
        changed => {
            'Archive::Tar'          => '2.24_01',
            'B::Op_private'         => '5.026003',
            'Config'                => '5.026003',
            'Module::CoreList'      => '5.20181129_26',
            'Module::CoreList::TieHashDelta'=> '5.20181129_26',
            'Module::CoreList::Utils'=> '5.20181129_26',
        },
        removed => {
        }
    },
    5.028001 => {
        delta_from => 5.028,
        changed => {
            'B::Op_private'         => '5.028001',
            'Config'                => '5.028001',
            'Module::CoreList'      => '5.20181129_28',
            'Module::CoreList::Utils'=> '5.20181129_28',
        },
        removed => {
        }
    },
    5.029006 => {
        delta_from => 5.029005,
        changed => {
            'B::Op_private'         => '5.029006',
            'Config'                => '5.029006',
            'Config::Perl::V'       => '0.32',
            'ExtUtils::ParseXS'     => '3.40',
            'ExtUtils::ParseXS::Constants'=> '3.40',
            'ExtUtils::ParseXS::CountLines'=> '3.40',
            'ExtUtils::ParseXS::Eval'=> '3.40',
            'ExtUtils::ParseXS::Utilities'=> '3.40',
            'File::Find'            => '1.35',
            'Module::CoreList'      => '5.20181218',
            'Module::CoreList::Utils'=> '5.20181218',
            'POSIX'                 => '1.86',
            'Storable'              => '3.14',
            'Test2'                 => '1.302141',
            'Test2::API'            => '1.302141',
            'Test2::API::Breakage'  => '1.302141',
            'Test2::API::Context'   => '1.302141',
            'Test2::API::Instance'  => '1.302141',
            'Test2::API::Stack'     => '1.302141',
            'Test2::Event'          => '1.302141',
            'Test2::Event::Bail'    => '1.302141',
            'Test2::Event::Diag'    => '1.302141',
            'Test2::Event::Encoding'=> '1.302141',
            'Test2::Event::Exception'=> '1.302141',
            'Test2::Event::Fail'    => '1.302141',
            'Test2::Event::Generic' => '1.302141',
            'Test2::Event::Note'    => '1.302141',
            'Test2::Event::Ok'      => '1.302141',
            'Test2::Event::Pass'    => '1.302141',
            'Test2::Event::Plan'    => '1.302141',
            'Test2::Event::Skip'    => '1.302141',
            'Test2::Event::Subtest' => '1.302141',
            'Test2::Event::TAP::Version'=> '1.302141',
            'Test2::Event::V2'      => '1.302141',
            'Test2::Event::Waiting' => '1.302141',
            'Test2::EventFacet'     => '1.302141',
            'Test2::EventFacet::About'=> '1.302141',
            'Test2::EventFacet::Amnesty'=> '1.302141',
            'Test2::EventFacet::Assert'=> '1.302141',
            'Test2::EventFacet::Control'=> '1.302141',
            'Test2::EventFacet::Error'=> '1.302141',
            'Test2::EventFacet::Hub'=> '1.302141',
            'Test2::EventFacet::Info'=> '1.302141',
            'Test2::EventFacet::Meta'=> '1.302141',
            'Test2::EventFacet::Parent'=> '1.302141',
            'Test2::EventFacet::Plan'=> '1.302141',
            'Test2::EventFacet::Render'=> '1.302141',
            'Test2::EventFacet::Trace'=> '1.302141',
            'Test2::Formatter'      => '1.302141',
            'Test2::Formatter::TAP' => '1.302141',
            'Test2::Hub'            => '1.302141',
            'Test2::Hub::Interceptor'=> '1.302141',
            'Test2::Hub::Interceptor::Terminator'=> '1.302141',
            'Test2::Hub::Subtest'   => '1.302141',
            'Test2::IPC'            => '1.302141',
            'Test2::IPC::Driver'    => '1.302141',
            'Test2::IPC::Driver::Files'=> '1.302141',
            'Test2::Tools::Tiny'    => '1.302141',
            'Test2::Util'           => '1.302141',
            'Test2::Util::ExternalMeta'=> '1.302141',
            'Test2::Util::Facets2Legacy'=> '1.302141',
            'Test2::Util::HashBase' => '1.302141',
            'Test2::Util::Trace'    => '1.302141',
            'Test::Builder'         => '1.302141',
            'Test::Builder::Formatter'=> '1.302141',
            'Test::Builder::Module' => '1.302141',
            'Test::Builder::Tester' => '1.302141',
            'Test::Builder::Tester::Color'=> '1.302141',
            'Test::Builder::TodoDiag'=> '1.302141',
            'Test::More'            => '1.302141',
            'Test::Simple'          => '1.302141',
            'Test::Tester'          => '1.302141',
            'Test::Tester::Capture' => '1.302141',
            'Test::Tester::CaptureRunner'=> '1.302141',
            'Test::Tester::Delegate'=> '1.302141',
            'Test::use::ok'         => '1.302141',
            'ok'                    => '1.302141',
            'threads::shared'       => '1.59',
        },
        removed => {
            'Storable::Limit'       => 1,
        }
    },
    5.029007 => {
        delta_from => 5.029006,
        changed => {
            'App::Cpan'             => '1.672',
            'B::Op_private'         => '5.029007',
            'CPAN'                  => '2.22',
            'CPAN::Distribution'    => '2.22',
            'CPAN::Plugin::Specfile'=> '0.02',
            'Compress::Raw::Bzip2'  => '2.084',
            'Compress::Raw::Zlib'   => '2.084',
            'Compress::Zlib'        => '2.084',
            'Config'                => '5.029007',
            'Cwd'                   => '3.77',
            'DB_File'               => '1.843',
            'File::Find'            => '1.36',
            'File::Spec'            => '3.77',
            'File::Spec::AmigaOS'   => '3.77',
            'File::Spec::Cygwin'    => '3.77',
            'File::Spec::Epoc'      => '3.77',
            'File::Spec::Functions' => '3.77',
            'File::Spec::Mac'       => '3.77',
            'File::Spec::OS2'       => '3.77',
            'File::Spec::Unix'      => '3.77',
            'File::Spec::VMS'       => '3.77',
            'File::Spec::Win32'     => '3.77',
            'File::Temp'            => '0.2309',
            'IO::Compress::Adapter::Bzip2'=> '2.084',
            'IO::Compress::Adapter::Deflate'=> '2.084',
            'IO::Compress::Adapter::Identity'=> '2.084',
            'IO::Compress::Base'    => '2.084',
            'IO::Compress::Base::Common'=> '2.084',
            'IO::Compress::Bzip2'   => '2.084',
            'IO::Compress::Deflate' => '2.084',
            'IO::Compress::Gzip'    => '2.084',
            'IO::Compress::Gzip::Constants'=> '2.084',
            'IO::Compress::RawDeflate'=> '2.084',
            'IO::Compress::Zip'     => '2.084',
            'IO::Compress::Zip::Constants'=> '2.084',
            'IO::Compress::Zlib::Constants'=> '2.084',
            'IO::Compress::Zlib::Extra'=> '2.084',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.084',
            'IO::Uncompress::Adapter::Identity'=> '2.084',
            'IO::Uncompress::Adapter::Inflate'=> '2.084',
            'IO::Uncompress::AnyInflate'=> '2.084',
            'IO::Uncompress::AnyUncompress'=> '2.084',
            'IO::Uncompress::Base'  => '2.084',
            'IO::Uncompress::Bunzip2'=> '2.084',
            'IO::Uncompress::Gunzip'=> '2.084',
            'IO::Uncompress::Inflate'=> '2.084',
            'IO::Uncompress::RawInflate'=> '2.084',
            'IO::Uncompress::Unzip' => '2.084',
            'Math::BigFloat'        => '1.999816',
            'Math::BigInt'          => '1.999816',
            'Math::BigInt::Calc'    => '1.999816',
            'Math::BigInt::FastCalc'=> '0.5008',
            'Math::BigInt::Lib'     => '1.999816',
            'Module::CoreList'      => '5.20190120',
            'Module::CoreList::Utils'=> '5.20190120',
            'Test2'                 => '1.302160',
            'Test2::API'            => '1.302160',
            'Test2::API::Breakage'  => '1.302160',
            'Test2::API::Context'   => '1.302160',
            'Test2::API::Instance'  => '1.302160',
            'Test2::API::Stack'     => '1.302160',
            'Test2::Event'          => '1.302160',
            'Test2::Event::Bail'    => '1.302160',
            'Test2::Event::Diag'    => '1.302160',
            'Test2::Event::Encoding'=> '1.302160',
            'Test2::Event::Exception'=> '1.302160',
            'Test2::Event::Fail'    => '1.302160',
            'Test2::Event::Generic' => '1.302160',
            'Test2::Event::Note'    => '1.302160',
            'Test2::Event::Ok'      => '1.302160',
            'Test2::Event::Pass'    => '1.302160',
            'Test2::Event::Plan'    => '1.302160',
            'Test2::Event::Skip'    => '1.302160',
            'Test2::Event::Subtest' => '1.302160',
            'Test2::Event::TAP::Version'=> '1.302160',
            'Test2::Event::V2'      => '1.302160',
            'Test2::Event::Waiting' => '1.302160',
            'Test2::EventFacet'     => '1.302160',
            'Test2::EventFacet::About'=> '1.302160',
            'Test2::EventFacet::Amnesty'=> '1.302160',
            'Test2::EventFacet::Assert'=> '1.302160',
            'Test2::EventFacet::Control'=> '1.302160',
            'Test2::EventFacet::Error'=> '1.302160',
            'Test2::EventFacet::Hub'=> '1.302160',
            'Test2::EventFacet::Info'=> '1.302160',
            'Test2::EventFacet::Info::Table'=> undef,
            'Test2::EventFacet::Meta'=> '1.302160',
            'Test2::EventFacet::Parent'=> '1.302160',
            'Test2::EventFacet::Plan'=> '1.302160',
            'Test2::EventFacet::Render'=> '1.302160',
            'Test2::EventFacet::Trace'=> '1.302160',
            'Test2::Formatter'      => '1.302160',
            'Test2::Formatter::TAP' => '1.302160',
            'Test2::Hub'            => '1.302160',
            'Test2::Hub::Interceptor'=> '1.302160',
            'Test2::Hub::Interceptor::Terminator'=> '1.302160',
            'Test2::Hub::Subtest'   => '1.302160',
            'Test2::IPC'            => '1.302160',
            'Test2::IPC::Driver'    => '1.302160',
            'Test2::IPC::Driver::Files'=> '1.302160',
            'Test2::Tools::Tiny'    => '1.302160',
            'Test2::Util'           => '1.302160',
            'Test2::Util::ExternalMeta'=> '1.302160',
            'Test2::Util::Facets2Legacy'=> '1.302160',
            'Test2::Util::HashBase' => '1.302160',
            'Test2::Util::Trace'    => '1.302160',
            'Test::Builder'         => '1.302160',
            'Test::Builder::Formatter'=> '1.302160',
            'Test::Builder::Module' => '1.302160',
            'Test::Builder::Tester' => '1.302160',
            'Test::Builder::Tester::Color'=> '1.302160',
            'Test::Builder::TodoDiag'=> '1.302160',
            'Test::More'            => '1.302160',
            'Test::Simple'          => '1.302160',
            'Test::Tester'          => '1.302160',
            'Test::Tester::Capture' => '1.302160',
            'Test::Tester::CaptureRunner'=> '1.302160',
            'Test::Tester::Delegate'=> '1.302160',
            'Test::use::ok'         => '1.302160',
            'Unicode::Collate'      => '1.27',
            'Unicode::Collate::CJK::Big5'=> '1.27',
            'Unicode::Collate::CJK::GB2312'=> '1.27',
            'Unicode::Collate::CJK::JISX0208'=> '1.27',
            'Unicode::Collate::CJK::Korean'=> '1.27',
            'Unicode::Collate::CJK::Pinyin'=> '1.27',
            'Unicode::Collate::CJK::Stroke'=> '1.27',
            'Unicode::Collate::CJK::Zhuyin'=> '1.27',
            'Unicode::Collate::Locale'=> '1.27',
            'lib'                   => '0.65',
            'ok'                    => '1.302160',
        },
        removed => {
            'Math::BigInt::CalcEmu' => 1,
        }
    },
    5.029008 => {
        delta_from => 5.029007,
        changed => {
            'B'                     => '1.76',
            'B::Op_private'         => '5.029008',
            'Config'                => '5.029008',
            'Devel::PPPort'         => '3.44',
            'Encode'                => '3.00',
            'Encode::Unicode'       => '2.18',
            'ExtUtils::Miniperl'    => '1.09',
            'IO'                    => '1.40',
            'IO::Dir'               => '1.40',
            'IO::File'              => '1.40',
            'IO::Handle'            => '1.40',
            'IO::Pipe'              => '1.40',
            'IO::Poll'              => '1.40',
            'IO::Seekable'          => '1.40',
            'IO::Select'            => '1.40',
            'IO::Socket'            => '1.40',
            'IO::Socket::INET'      => '1.40',
            'IO::Socket::UNIX'      => '1.40',
            'JSON::PP'              => '4.00',
            'JSON::PP::Boolean'     => '4.00',
            'Module::CoreList'      => '5.20190220',
            'Module::CoreList::Utils'=> '5.20190220',
            'Module::Load'          => '0.34',
            'Net::Ping'             => '2.71',
            'POSIX'                 => '1.87',
            'Test2'                 => '1.302162',
            'Test2::API'            => '1.302162',
            'Test2::API::Breakage'  => '1.302162',
            'Test2::API::Context'   => '1.302162',
            'Test2::API::Instance'  => '1.302162',
            'Test2::API::Stack'     => '1.302162',
            'Test2::Event'          => '1.302162',
            'Test2::Event::Bail'    => '1.302162',
            'Test2::Event::Diag'    => '1.302162',
            'Test2::Event::Encoding'=> '1.302162',
            'Test2::Event::Exception'=> '1.302162',
            'Test2::Event::Fail'    => '1.302162',
            'Test2::Event::Generic' => '1.302162',
            'Test2::Event::Note'    => '1.302162',
            'Test2::Event::Ok'      => '1.302162',
            'Test2::Event::Pass'    => '1.302162',
            'Test2::Event::Plan'    => '1.302162',
            'Test2::Event::Skip'    => '1.302162',
            'Test2::Event::Subtest' => '1.302162',
            'Test2::Event::TAP::Version'=> '1.302162',
            'Test2::Event::V2'      => '1.302162',
            'Test2::Event::Waiting' => '1.302162',
            'Test2::EventFacet'     => '1.302162',
            'Test2::EventFacet::About'=> '1.302162',
            'Test2::EventFacet::Amnesty'=> '1.302162',
            'Test2::EventFacet::Assert'=> '1.302162',
            'Test2::EventFacet::Control'=> '1.302162',
            'Test2::EventFacet::Error'=> '1.302162',
            'Test2::EventFacet::Hub'=> '1.302162',
            'Test2::EventFacet::Info'=> '1.302162',
            'Test2::EventFacet::Meta'=> '1.302162',
            'Test2::EventFacet::Parent'=> '1.302162',
            'Test2::EventFacet::Plan'=> '1.302162',
            'Test2::EventFacet::Render'=> '1.302162',
            'Test2::EventFacet::Trace'=> '1.302162',
            'Test2::Formatter'      => '1.302162',
            'Test2::Formatter::TAP' => '1.302162',
            'Test2::Hub'            => '1.302162',
            'Test2::Hub::Interceptor'=> '1.302162',
            'Test2::Hub::Interceptor::Terminator'=> '1.302162',
            'Test2::Hub::Subtest'   => '1.302162',
            'Test2::IPC'            => '1.302162',
            'Test2::IPC::Driver'    => '1.302162',
            'Test2::IPC::Driver::Files'=> '1.302162',
            'Test2::Tools::Tiny'    => '1.302162',
            'Test2::Util'           => '1.302162',
            'Test2::Util::ExternalMeta'=> '1.302162',
            'Test2::Util::Facets2Legacy'=> '1.302162',
            'Test2::Util::HashBase' => '1.302162',
            'Test2::Util::Trace'    => '1.302162',
            'Test::Builder'         => '1.302162',
            'Test::Builder::Formatter'=> '1.302162',
            'Test::Builder::Module' => '1.302162',
            'Test::Builder::Tester' => '1.302162',
            'Test::Builder::Tester::Color'=> '1.302162',
            'Test::Builder::TodoDiag'=> '1.302162',
            'Test::More'            => '1.302162',
            'Test::Simple'          => '1.302162',
            'Test::Tester'          => '1.302162',
            'Test::Tester::Capture' => '1.302162',
            'Test::Tester::CaptureRunner'=> '1.302162',
            'Test::Tester::Delegate'=> '1.302162',
            'Test::use::ok'         => '1.302162',
            'XS::APItest'           => '1.00',
            'deprecate'             => '0.04',
            'ok'                    => '1.302162',
            'perlfaq'               => '5.20190126',
        },
        removed => {
        }
    },
    5.029009 => {
        delta_from => 5.029008,
        changed => {
            'B::Op_private'         => '5.029009',
            'Config'                => '5.029009',
            'Devel::PPPort'         => '3.45',
            'Encode'                => '3.01',
            'ExtUtils::Manifest'    => '1.72',
            'JSON::PP'              => '4.02',
            'JSON::PP::Boolean'     => '4.02',
            'Module::CoreList'      => '5.20190320',
            'Module::CoreList::Utils'=> '5.20190320',
            'PerlIO::encoding'      => '0.27',
            'Unicode'               => '12.0.0',
            'threads::shared'       => '1.60',
            'utf8'                  => '1.22',
            'warnings'              => '1.44',
        },
        removed => {
        }
    },
    5.028002 => {
        delta_from => 5.028001,
        changed => {
            'B::Op_private'         => '5.028002',
            'Config'                => '5.028002',
            'Module::CoreList'      => '5.20190419',
            'Module::CoreList::Utils'=> '5.20190419',
            'PerlIO::scalar'        => '0.30',
            'Storable'              => '3.08_01',
        },
        removed => {
        }
    },
    5.029010 => {
        delta_from => 5.029009,
        changed => {
            'B::Op_private'         => '5.029010',
            'Config'                => '5.02901',
            'Cwd'                   => '3.78',
            'Data::Dumper'          => '2.174',
            'ExtUtils::CBuilder'    => '0.280231',
            'ExtUtils::CBuilder::Base'=> '0.280231',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.280231',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.280231',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.280231',
            'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280231',
            'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280231',
            'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280231',
            'ExtUtils::CBuilder::Platform::aix'=> '0.280231',
            'ExtUtils::CBuilder::Platform::android'=> '0.280231',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.280231',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.280231',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280231',
            'ExtUtils::CBuilder::Platform::os2'=> '0.280231',
            'File::Spec'            => '3.78',
            'File::Spec::AmigaOS'   => '3.78',
            'File::Spec::Cygwin'    => '3.78',
            'File::Spec::Epoc'      => '3.78',
            'File::Spec::Functions' => '3.78',
            'File::Spec::Mac'       => '3.78',
            'File::Spec::OS2'       => '3.78',
            'File::Spec::Unix'      => '3.78',
            'File::Spec::VMS'       => '3.78',
            'File::Spec::Win32'     => '3.78',
            'I18N::Langinfo'        => '0.18',
            'Module::CoreList'      => '5.20190420',
            'Module::CoreList::Utils'=> '5.20190420',
            'Module::Metadata'      => '1.000036',
            'POSIX'                 => '1.88',
            'Storable'              => '3.15',
            'Unicode'               => '12.1.0',
        },
        removed => {
        }
    },
    5.030000 => {
        delta_from => 5.02901,
        changed => {
            'B::Op_private'         => '5.030000',
            'Config'                => '5.03',
            'Devel::PPPort'         => '3.52',
            'Module::CoreList'      => '5.20190522',
            'Module::CoreList::Utils'=> '5.20190522',
            'XS::Typemap'           => '0.17',
        },
        removed => {
        }
    },
    5.031000 => {
        delta_from => 5.03,
        changed => {
            'B::Op_private'         => '5.031000',
            'Config'                => '5.031',
            'Module::CoreList'      => '5.20190524',
            'Module::CoreList::Utils'=> '5.20190524',
            'Pod::Simple'           => '3.36',
            'Pod::Simple::BlackBox' => '3.36',
            'Pod::Simple::Checker'  => '3.36',
            'Pod::Simple::Debug'    => '3.36',
            'Pod::Simple::DumpAsText'=> '3.36',
            'Pod::Simple::DumpAsXML'=> '3.36',
            'Pod::Simple::HTML'     => '3.36',
            'Pod::Simple::HTMLBatch'=> '3.36',
            'Pod::Simple::JustPod'  => undef,
            'Pod::Simple::LinkSection'=> '3.36',
            'Pod::Simple::Methody'  => '3.36',
            'Pod::Simple::Progress' => '3.36',
            'Pod::Simple::PullParser'=> '3.36',
            'Pod::Simple::PullParserEndToken'=> '3.36',
            'Pod::Simple::PullParserStartToken'=> '3.36',
            'Pod::Simple::PullParserTextToken'=> '3.36',
            'Pod::Simple::PullParserToken'=> '3.36',
            'Pod::Simple::RTF'      => '3.36',
            'Pod::Simple::Search'   => '3.36',
            'Pod::Simple::SimpleTree'=> '3.36',
            'Pod::Simple::Text'     => '3.36',
            'Pod::Simple::TextContent'=> '3.36',
            'Pod::Simple::TiedOutFH'=> '3.36',
            'Pod::Simple::Transcode'=> '3.36',
            'Pod::Simple::TranscodeDumb'=> '3.36',
            'Pod::Simple::TranscodeSmart'=> '3.36',
            'Pod::Simple::XHTML'    => '3.36',
            'Pod::Simple::XMLOutStream'=> '3.36',
            'Socket'                => '2.029',
            'feature'               => '1.55',
        },
        removed => {
        }
    },
    5.031001 => {
        delta_from => 5.031000,
        changed => {
            'App::Cpan'             => '1.675',
            'B::Op_private'         => '5.031001',
            'CPAN'                  => '2.27',
            'CPAN::Bundle'          => '5.5005',
            'CPAN::Distribution'    => '2.27',
            'CPAN::FTP'             => '5.5012',
            'CPAN::FirstTime'       => '5.5314',
            'CPAN::HandleConfig'    => '5.5011',
            'CPAN::Mirrors'         => '2.27',
            'CPAN::Queue'           => '5.5003',
            'CPAN::Shell'           => '5.5009',
            'CPAN::Tarzip'          => '5.5013',
            'Class::Struct'         => '0.66',
            'Compress::Raw::Bzip2'  => '2.086',
            'Compress::Raw::Zlib'   => '2.086',
            'Compress::Zlib'        => '2.086',
            'Config'                => '5.031001',
            'DB_File'               => '1.852',
            'Devel::PPPort'         => '3.53',
            'ExtUtils::CBuilder'    => '0.280232',
            'ExtUtils::Command'     => '7.36',
            'ExtUtils::Command::MM' => '7.36',
            'ExtUtils::Liblist'     => '7.36',
            'ExtUtils::Liblist::Kid'=> '7.36',
            'ExtUtils::MM'          => '7.36',
            'ExtUtils::MM_AIX'      => '7.36',
            'ExtUtils::MM_Any'      => '7.36',
            'ExtUtils::MM_BeOS'     => '7.36',
            'ExtUtils::MM_Cygwin'   => '7.36',
            'ExtUtils::MM_DOS'      => '7.36',
            'ExtUtils::MM_Darwin'   => '7.36',
            'ExtUtils::MM_MacOS'    => '7.36',
            'ExtUtils::MM_NW5'      => '7.36',
            'ExtUtils::MM_OS2'      => '7.36',
            'ExtUtils::MM_QNX'      => '7.36',
            'ExtUtils::MM_UWIN'     => '7.36',
            'ExtUtils::MM_Unix'     => '7.36',
            'ExtUtils::MM_VMS'      => '7.36',
            'ExtUtils::MM_VOS'      => '7.36',
            'ExtUtils::MM_Win32'    => '7.36',
            'ExtUtils::MM_Win95'    => '7.36',
            'ExtUtils::MY'          => '7.36',
            'ExtUtils::MakeMaker'   => '7.36',
            'ExtUtils::MakeMaker::Config'=> '7.36',
            'ExtUtils::MakeMaker::Locale'=> '7.36',
            'ExtUtils::MakeMaker::version'=> '7.36',
            'ExtUtils::MakeMaker::version::regex'=> '7.36',
            'ExtUtils::Mkbootstrap' => '7.36',
            'ExtUtils::Mksymlists'  => '7.36',
            'ExtUtils::testlib'     => '7.36',
            'File::Spec::Win32'     => '3.79',
            'I18N::LangTags'        => '0.44',
            'IO'                    => '1.41',
            'IO::Compress::Adapter::Bzip2'=> '2.086',
            'IO::Compress::Adapter::Deflate'=> '2.086',
            'IO::Compress::Adapter::Identity'=> '2.086',
            'IO::Compress::Base'    => '2.086',
            'IO::Compress::Base::Common'=> '2.086',
            'IO::Compress::Bzip2'   => '2.086',
            'IO::Compress::Deflate' => '2.086',
            'IO::Compress::Gzip'    => '2.086',
            'IO::Compress::Gzip::Constants'=> '2.086',
            'IO::Compress::RawDeflate'=> '2.086',
            'IO::Compress::Zip'     => '2.086',
            'IO::Compress::Zip::Constants'=> '2.086',
            'IO::Compress::Zlib::Constants'=> '2.086',
            'IO::Compress::Zlib::Extra'=> '2.086',
            'IO::Dir'               => '1.41',
            'IO::File'              => '1.41',
            'IO::Handle'            => '1.41',
            'IO::Pipe'              => '1.41',
            'IO::Poll'              => '1.41',
            'IO::Seekable'          => '1.41',
            'IO::Select'            => '1.41',
            'IO::Socket'            => '1.41',
            'IO::Socket::INET'      => '1.41',
            'IO::Socket::UNIX'      => '1.41',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.086',
            'IO::Uncompress::Adapter::Identity'=> '2.086',
            'IO::Uncompress::Adapter::Inflate'=> '2.086',
            'IO::Uncompress::AnyInflate'=> '2.086',
            'IO::Uncompress::AnyUncompress'=> '2.086',
            'IO::Uncompress::Base'  => '2.086',
            'IO::Uncompress::Bunzip2'=> '2.086',
            'IO::Uncompress::Gunzip'=> '2.086',
            'IO::Uncompress::Inflate'=> '2.086',
            'IO::Uncompress::RawInflate'=> '2.086',
            'IO::Uncompress::Unzip' => '2.086',
            'Module::CoreList'      => '5.20190620',
            'Module::CoreList::Utils'=> '5.20190620',
            'POSIX'                 => '1.89',
            'Pod::Man'              => '4.12',
            'Pod::ParseLink'        => '4.12',
            'Pod::Simple'           => '3.38',
            'Pod::Simple::BlackBox' => '3.38',
            'Pod::Simple::Checker'  => '3.38',
            'Pod::Simple::Debug'    => '3.38',
            'Pod::Simple::DumpAsText'=> '3.38',
            'Pod::Simple::DumpAsXML'=> '3.38',
            'Pod::Simple::HTML'     => '3.38',
            'Pod::Simple::HTMLBatch'=> '3.38',
            'Pod::Simple::LinkSection'=> '3.38',
            'Pod::Simple::Methody'  => '3.38',
            'Pod::Simple::Progress' => '3.38',
            'Pod::Simple::PullParser'=> '3.38',
            'Pod::Simple::PullParserEndToken'=> '3.38',
            'Pod::Simple::PullParserStartToken'=> '3.38',
            'Pod::Simple::PullParserTextToken'=> '3.38',
            'Pod::Simple::PullParserToken'=> '3.38',
            'Pod::Simple::RTF'      => '3.38',
            'Pod::Simple::Search'   => '3.38',
            'Pod::Simple::SimpleTree'=> '3.38',
            'Pod::Simple::Text'     => '3.38',
            'Pod::Simple::TextContent'=> '3.38',
            'Pod::Simple::TiedOutFH'=> '3.38',
            'Pod::Simple::Transcode'=> '3.38',
            'Pod::Simple::TranscodeDumb'=> '3.38',
            'Pod::Simple::TranscodeSmart'=> '3.38',
            'Pod::Simple::XHTML'    => '3.38',
            'Pod::Simple::XMLOutStream'=> '3.38',
            'Pod::Text'             => '4.12',
            'Pod::Text::Color'      => '4.12',
            'Pod::Text::Overstrike' => '4.12',
            'Pod::Text::Termcap'    => '4.12',
            'SelfLoader'            => '1.26',
            'Storable'              => '3.16',
            'Sys::Hostname'         => '1.23',
            'Test2'                 => '1.302164',
            'Test2::API'            => '1.302164',
            'Test2::API::Breakage'  => '1.302164',
            'Test2::API::Context'   => '1.302164',
            'Test2::API::Instance'  => '1.302164',
            'Test2::API::Stack'     => '1.302164',
            'Test2::Event'          => '1.302164',
            'Test2::Event::Bail'    => '1.302164',
            'Test2::Event::Diag'    => '1.302164',
            'Test2::Event::Encoding'=> '1.302164',
            'Test2::Event::Exception'=> '1.302164',
            'Test2::Event::Fail'    => '1.302164',
            'Test2::Event::Generic' => '1.302164',
            'Test2::Event::Note'    => '1.302164',
            'Test2::Event::Ok'      => '1.302164',
            'Test2::Event::Pass'    => '1.302164',
            'Test2::Event::Plan'    => '1.302164',
            'Test2::Event::Skip'    => '1.302164',
            'Test2::Event::Subtest' => '1.302164',
            'Test2::Event::TAP::Version'=> '1.302164',
            'Test2::Event::V2'      => '1.302164',
            'Test2::Event::Waiting' => '1.302164',
            'Test2::EventFacet'     => '1.302164',
            'Test2::EventFacet::About'=> '1.302164',
            'Test2::EventFacet::Amnesty'=> '1.302164',
            'Test2::EventFacet::Assert'=> '1.302164',
            'Test2::EventFacet::Control'=> '1.302164',
            'Test2::EventFacet::Error'=> '1.302164',
            'Test2::EventFacet::Hub'=> '1.302164',
            'Test2::EventFacet::Info'=> '1.302164',
            'Test2::EventFacet::Info::Table'=> '1.302164',
            'Test2::EventFacet::Meta'=> '1.302164',
            'Test2::EventFacet::Parent'=> '1.302164',
            'Test2::EventFacet::Plan'=> '1.302164',
            'Test2::EventFacet::Render'=> '1.302164',
            'Test2::EventFacet::Trace'=> '1.302164',
            'Test2::Formatter'      => '1.302164',
            'Test2::Formatter::TAP' => '1.302164',
            'Test2::Hub'            => '1.302164',
            'Test2::Hub::Interceptor'=> '1.302164',
            'Test2::Hub::Interceptor::Terminator'=> '1.302164',
            'Test2::Hub::Subtest'   => '1.302164',
            'Test2::IPC'            => '1.302164',
            'Test2::IPC::Driver'    => '1.302164',
            'Test2::IPC::Driver::Files'=> '1.302164',
            'Test2::Tools::Tiny'    => '1.302164',
            'Test2::Util'           => '1.302164',
            'Test2::Util::ExternalMeta'=> '1.302164',
            'Test2::Util::Facets2Legacy'=> '1.302164',
            'Test2::Util::HashBase' => '1.302164',
            'Test2::Util::Trace'    => '1.302164',
            'Test::Builder'         => '1.302164',
            'Test::Builder::Formatter'=> '1.302164',
            'Test::Builder::Module' => '1.302164',
            'Test::Builder::Tester' => '1.302164',
            'Test::Builder::Tester::Color'=> '1.302164',
            'Test::Builder::TodoDiag'=> '1.302164',
            'Test::More'            => '1.302164',
            'Test::Simple'          => '1.302164',
            'Test::Tester'          => '1.302164',
            'Test::Tester::Capture' => '1.302164',
            'Test::Tester::CaptureRunner'=> '1.302164',
            'Test::Tester::Delegate'=> '1.302164',
            'Test::use::ok'         => '1.302164',
            'Tie::File'             => '1.03',
            'Tie::Hash::NamedCapture'=> '0.11',
            'Time::HiRes'           => '1.9761',
            'Unicode::Normalize'    => '1.27',
            'Unicode::UCD'          => '0.73',
            'XS::APItest'           => '1.01',
            'ok'                    => '1.302164',
            'overload'              => '1.31',
            'warnings'              => '1.45',
        },
        removed => {
            'Pod::Find'             => 1,
            'Pod::InputObjects'     => 1,
            'Pod::ParseUtils'       => 1,
            'Pod::Parser'           => 1,
            'Pod::PlainText'        => 1,
            'Pod::Select'           => 1,
        }
    },
    5.031002 => {
        delta_from => 5.031001,
        changed => {
            'B::Op_private'         => '5.031002',
            'Config'                => '5.031002',
            'Devel::PPPort'         => '3.54',
            'Exporter'              => '5.74',
            'Exporter::Heavy'       => '5.74',
            'IPC::Cmd'              => '1.04',
            'JSON::PP'              => '4.04',
            'JSON::PP::Boolean'     => '4.04',
            'Module::CoreList'      => '5.20190720',
            'Module::CoreList::Utils'=> '5.20190720',
            'Opcode'                => '1.44',
            'PerlIO::encoding'      => '0.28',
            'Pod::Simple'           => '3.39',
            'Pod::Simple::BlackBox' => '3.39',
            'Pod::Simple::Checker'  => '3.39',
            'Pod::Simple::Debug'    => '3.39',
            'Pod::Simple::DumpAsText'=> '3.39',
            'Pod::Simple::DumpAsXML'=> '3.39',
            'Pod::Simple::HTML'     => '3.39',
            'Pod::Simple::HTMLBatch'=> '3.39',
            'Pod::Simple::LinkSection'=> '3.39',
            'Pod::Simple::Methody'  => '3.39',
            'Pod::Simple::Progress' => '3.39',
            'Pod::Simple::PullParser'=> '3.39',
            'Pod::Simple::PullParserEndToken'=> '3.39',
            'Pod::Simple::PullParserStartToken'=> '3.39',
            'Pod::Simple::PullParserTextToken'=> '3.39',
            'Pod::Simple::PullParserToken'=> '3.39',
            'Pod::Simple::RTF'      => '3.39',
            'Pod::Simple::Search'   => '3.39',
            'Pod::Simple::SimpleTree'=> '3.39',
            'Pod::Simple::Text'     => '3.39',
            'Pod::Simple::TextContent'=> '3.39',
            'Pod::Simple::TiedOutFH'=> '3.39',
            'Pod::Simple::Transcode'=> '3.39',
            'Pod::Simple::TranscodeDumb'=> '3.39',
            'Pod::Simple::TranscodeSmart'=> '3.39',
            'Pod::Simple::XHTML'    => '3.39',
            'Pod::Simple::XMLOutStream'=> '3.39',
            'threads::shared'       => '1.61',
        },
        removed => {
        }
    },
    5.031003 => {
        delta_from => 5.031002,
        changed => {
            'B::Op_private'         => '5.031003',
            'Compress::Raw::Bzip2'  => '2.087',
            'Compress::Raw::Zlib'   => '2.087',
            'Compress::Zlib'        => '2.087',
            'Config'                => '5.031003',
            'Devel::PPPort'         => '3.55',
            'File::Find'            => '1.37',
            'Getopt::Long'          => '2.51',
            'I18N::LangTags::Detect'=> '1.08',
            'IO::Compress::Adapter::Bzip2'=> '2.087',
            'IO::Compress::Adapter::Deflate'=> '2.087',
            'IO::Compress::Adapter::Identity'=> '2.087',
            'IO::Compress::Base'    => '2.087',
            'IO::Compress::Base::Common'=> '2.087',
            'IO::Compress::Bzip2'   => '2.087',
            'IO::Compress::Deflate' => '2.087',
            'IO::Compress::Gzip'    => '2.087',
            'IO::Compress::Gzip::Constants'=> '2.087',
            'IO::Compress::RawDeflate'=> '2.087',
            'IO::Compress::Zip'     => '2.087',
            'IO::Compress::Zip::Constants'=> '2.087',
            'IO::Compress::Zlib::Constants'=> '2.087',
            'IO::Compress::Zlib::Extra'=> '2.087',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.087',
            'IO::Uncompress::Adapter::Identity'=> '2.087',
            'IO::Uncompress::Adapter::Inflate'=> '2.087',
            'IO::Uncompress::AnyInflate'=> '2.087',
            'IO::Uncompress::AnyUncompress'=> '2.087',
            'IO::Uncompress::Base'  => '2.087',
            'IO::Uncompress::Bunzip2'=> '2.087',
            'IO::Uncompress::Gunzip'=> '2.087',
            'IO::Uncompress::Inflate'=> '2.087',
            'IO::Uncompress::RawInflate'=> '2.087',
            'IO::Uncompress::Unzip' => '2.087',
            'Module::CoreList'      => '5.20190820',
            'Module::CoreList::Utils'=> '5.20190820',
            'PerlIO::via'           => '0.18',
            'Storable'              => '3.17',
            'Test2'                 => '1.302166',
            'Test2::API'            => '1.302166',
            'Test2::API::Breakage'  => '1.302166',
            'Test2::API::Context'   => '1.302166',
            'Test2::API::Instance'  => '1.302166',
            'Test2::API::Stack'     => '1.302166',
            'Test2::Event'          => '1.302166',
            'Test2::Event::Bail'    => '1.302166',
            'Test2::Event::Diag'    => '1.302166',
            'Test2::Event::Encoding'=> '1.302166',
            'Test2::Event::Exception'=> '1.302166',
            'Test2::Event::Fail'    => '1.302166',
            'Test2::Event::Generic' => '1.302166',
            'Test2::Event::Note'    => '1.302166',
            'Test2::Event::Ok'      => '1.302166',
            'Test2::Event::Pass'    => '1.302166',
            'Test2::Event::Plan'    => '1.302166',
            'Test2::Event::Skip'    => '1.302166',
            'Test2::Event::Subtest' => '1.302166',
            'Test2::Event::TAP::Version'=> '1.302166',
            'Test2::Event::V2'      => '1.302166',
            'Test2::Event::Waiting' => '1.302166',
            'Test2::EventFacet'     => '1.302166',
            'Test2::EventFacet::About'=> '1.302166',
            'Test2::EventFacet::Amnesty'=> '1.302166',
            'Test2::EventFacet::Assert'=> '1.302166',
            'Test2::EventFacet::Control'=> '1.302166',
            'Test2::EventFacet::Error'=> '1.302166',
            'Test2::EventFacet::Hub'=> '1.302166',
            'Test2::EventFacet::Info'=> '1.302166',
            'Test2::EventFacet::Info::Table'=> '1.302166',
            'Test2::EventFacet::Meta'=> '1.302166',
            'Test2::EventFacet::Parent'=> '1.302166',
            'Test2::EventFacet::Plan'=> '1.302166',
            'Test2::EventFacet::Render'=> '1.302166',
            'Test2::EventFacet::Trace'=> '1.302166',
            'Test2::Formatter'      => '1.302166',
            'Test2::Formatter::TAP' => '1.302166',
            'Test2::Hub'            => '1.302166',
            'Test2::Hub::Interceptor'=> '1.302166',
            'Test2::Hub::Interceptor::Terminator'=> '1.302166',
            'Test2::Hub::Subtest'   => '1.302166',
            'Test2::IPC'            => '1.302166',
            'Test2::IPC::Driver'    => '1.302166',
            'Test2::IPC::Driver::Files'=> '1.302166',
            'Test2::Tools::Tiny'    => '1.302166',
            'Test2::Util'           => '1.302166',
            'Test2::Util::ExternalMeta'=> '1.302166',
            'Test2::Util::Facets2Legacy'=> '1.302166',
            'Test2::Util::HashBase' => '1.302166',
            'Test2::Util::Trace'    => '1.302166',
            'Test::Builder'         => '1.302166',
            'Test::Builder::Formatter'=> '1.302166',
            'Test::Builder::Module' => '1.302166',
            'Test::Builder::Tester' => '1.302166',
            'Test::Builder::Tester::Color'=> '1.302166',
            'Test::Builder::TodoDiag'=> '1.302166',
            'Test::More'            => '1.302166',
            'Test::Simple'          => '1.302166',
            'Test::Tester'          => '1.302166',
            'Test::Tester::Capture' => '1.302166',
            'Test::Tester::CaptureRunner'=> '1.302166',
            'Test::Tester::Delegate'=> '1.302166',
            'Test::use::ok'         => '1.302166',
            'Thread'                => '3.05',
            'Time::HiRes'           => '1.9762',
            'Win32'                 => '0.53',
            'XS::APItest'           => '1.02',
            'ok'                    => '1.302166',
        },
        removed => {
        }
    },
    5.031004 => {
        delta_from => 5.031003,
        changed => {
            'B::Op_private'         => '5.031004',
            'Config'                => '5.031004',
            'ExtUtils::Command'     => '7.38',
            'ExtUtils::Command::MM' => '7.38',
            'ExtUtils::Liblist'     => '7.38',
            'ExtUtils::Liblist::Kid'=> '7.38',
            'ExtUtils::MM'          => '7.38',
            'ExtUtils::MM_AIX'      => '7.38',
            'ExtUtils::MM_Any'      => '7.38',
            'ExtUtils::MM_BeOS'     => '7.38',
            'ExtUtils::MM_Cygwin'   => '7.38',
            'ExtUtils::MM_DOS'      => '7.38',
            'ExtUtils::MM_Darwin'   => '7.38',
            'ExtUtils::MM_MacOS'    => '7.38',
            'ExtUtils::MM_NW5'      => '7.38',
            'ExtUtils::MM_OS2'      => '7.38',
            'ExtUtils::MM_QNX'      => '7.38',
            'ExtUtils::MM_UWIN'     => '7.38',
            'ExtUtils::MM_Unix'     => '7.38',
            'ExtUtils::MM_VMS'      => '7.38',
            'ExtUtils::MM_VOS'      => '7.38',
            'ExtUtils::MM_Win32'    => '7.38',
            'ExtUtils::MM_Win95'    => '7.38',
            'ExtUtils::MY'          => '7.38',
            'ExtUtils::MakeMaker'   => '7.38',
            'ExtUtils::MakeMaker::Config'=> '7.38',
            'ExtUtils::MakeMaker::Locale'=> '7.38',
            'ExtUtils::MakeMaker::version'=> '7.38',
            'ExtUtils::MakeMaker::version::regex'=> '7.38',
            'ExtUtils::Mkbootstrap' => '7.38',
            'ExtUtils::Mksymlists'  => '7.38',
            'ExtUtils::testlib'     => '7.38',
            'I18N::Langinfo'        => '0.19',
            'List::Util'            => '1.52',
            'List::Util::XS'        => '1.52',
            'Module::CoreList'      => '5.20190920',
            'Module::CoreList::Utils'=> '5.20190920',
            'Module::Metadata'      => '1.000037',
            'Scalar::Util'          => '1.52',
            'Sub::Util'             => '1.52',
            'Test2'                 => '1.302168',
            'Test2::API'            => '1.302168',
            'Test2::API::Breakage'  => '1.302168',
            'Test2::API::Context'   => '1.302168',
            'Test2::API::Instance'  => '1.302168',
            'Test2::API::Stack'     => '1.302168',
            'Test2::Event'          => '1.302168',
            'Test2::Event::Bail'    => '1.302168',
            'Test2::Event::Diag'    => '1.302168',
            'Test2::Event::Encoding'=> '1.302168',
            'Test2::Event::Exception'=> '1.302168',
            'Test2::Event::Fail'    => '1.302168',
            'Test2::Event::Generic' => '1.302168',
            'Test2::Event::Note'    => '1.302168',
            'Test2::Event::Ok'      => '1.302168',
            'Test2::Event::Pass'    => '1.302168',
            'Test2::Event::Plan'    => '1.302168',
            'Test2::Event::Skip'    => '1.302168',
            'Test2::Event::Subtest' => '1.302168',
            'Test2::Event::TAP::Version'=> '1.302168',
            'Test2::Event::V2'      => '1.302168',
            'Test2::Event::Waiting' => '1.302168',
            'Test2::EventFacet'     => '1.302168',
            'Test2::EventFacet::About'=> '1.302168',
            'Test2::EventFacet::Amnesty'=> '1.302168',
            'Test2::EventFacet::Assert'=> '1.302168',
            'Test2::EventFacet::Control'=> '1.302168',
            'Test2::EventFacet::Error'=> '1.302168',
            'Test2::EventFacet::Hub'=> '1.302168',
            'Test2::EventFacet::Info'=> '1.302168',
            'Test2::EventFacet::Info::Table'=> '1.302168',
            'Test2::EventFacet::Meta'=> '1.302168',
            'Test2::EventFacet::Parent'=> '1.302168',
            'Test2::EventFacet::Plan'=> '1.302168',
            'Test2::EventFacet::Render'=> '1.302168',
            'Test2::EventFacet::Trace'=> '1.302168',
            'Test2::Formatter'      => '1.302168',
            'Test2::Formatter::TAP' => '1.302168',
            'Test2::Hub'            => '1.302168',
            'Test2::Hub::Interceptor'=> '1.302168',
            'Test2::Hub::Interceptor::Terminator'=> '1.302168',
            'Test2::Hub::Subtest'   => '1.302168',
            'Test2::IPC'            => '1.302168',
            'Test2::IPC::Driver'    => '1.302168',
            'Test2::IPC::Driver::Files'=> '1.302168',
            'Test2::Tools::Tiny'    => '1.302168',
            'Test2::Util'           => '1.302168',
            'Test2::Util::ExternalMeta'=> '1.302168',
            'Test2::Util::Facets2Legacy'=> '1.302168',
            'Test2::Util::HashBase' => '1.302168',
            'Test2::Util::Trace'    => '1.302168',
            'Test::Builder'         => '1.302168',
            'Test::Builder::Formatter'=> '1.302168',
            'Test::Builder::Module' => '1.302168',
            'Test::Builder::Tester' => '1.302168',
            'Test::Builder::Tester::Color'=> '1.302168',
            'Test::Builder::TodoDiag'=> '1.302168',
            'Test::More'            => '1.302168',
            'Test::Simple'          => '1.302168',
            'Test::Tester'          => '1.302168',
            'Test::Tester::Capture' => '1.302168',
            'Test::Tester::CaptureRunner'=> '1.302168',
            'Test::Tester::Delegate'=> '1.302168',
            'Test::use::ok'         => '1.302168',
            'Time::HiRes'           => '1.9763',
            'XS::APItest'           => '1.03',
            'ok'                    => '1.302168',
            're'                    => '0.38',
        },
        removed => {
        }
    },
    5.031005 => {
        delta_from => 5.031004,
        changed => {
            'B'                     => '1.77',
            'B::Deparse'            => '1.50',
            'B::Op_private'         => '5.031005',
            'Config'                => '5.031005',
            'Devel::PPPort'         => '3.54',
            'Digest::MD5'           => '2.55_01',
            'Dumpvalue'             => '1.21',
            'ExtUtils::CBuilder'    => '0.280233',
            'Math::BigFloat'        => '1.999817_01',
            'Math::BigInt'          => '1.999817_01',
            'Math::BigInt::Calc'    => '1.999817_01',
            'Math::BigInt::FastCalc'=> '0.5009',
            'Math::BigInt::Lib'     => '1.999817_01',
            'Module::CoreList'      => '5.20191020',
            'Module::CoreList::Utils'=> '5.20191020',
            'Safe'                  => '2.41',
            'Time::HiRes'           => '1.9764',
            'XS::APItest'           => '1.04',
            'threads'               => '2.23',
        },
        removed => {
        }
    },
    5.030001 => {
        delta_from => 5.030000,
        changed => {
            'B::Op_private'         => '5.030001',
            'Config'                => '5.030001',
            'Module::CoreList'      => '5.20191110',
            'Module::CoreList::Utils'=> '5.20191110',
        },
        removed => {
        }
    },
    5.031006 => {
        delta_from => 5.031005,
        changed => {
            'B::Deparse'            => '1.51',
            'B::Op_private'         => '5.031006',
            'Compress::Raw::Bzip2'  => '2.090',
            'Compress::Raw::Zlib'   => '2.090',
            'Compress::Zlib'        => '2.090',
            'Config'                => '5.031006',
            'Devel::PPPort'         => '3.55',
            'DynaLoader'            => '1.46',
            'IO::Compress::Adapter::Bzip2'=> '2.090',
            'IO::Compress::Adapter::Deflate'=> '2.090',
            'IO::Compress::Adapter::Identity'=> '2.090',
            'IO::Compress::Base'    => '2.090',
            'IO::Compress::Base::Common'=> '2.090',
            'IO::Compress::Bzip2'   => '2.090',
            'IO::Compress::Deflate' => '2.090',
            'IO::Compress::Gzip'    => '2.090',
            'IO::Compress::Gzip::Constants'=> '2.090',
            'IO::Compress::RawDeflate'=> '2.090',
            'IO::Compress::Zip'     => '2.090',
            'IO::Compress::Zip::Constants'=> '2.090',
            'IO::Compress::Zlib::Constants'=> '2.090',
            'IO::Compress::Zlib::Extra'=> '2.090',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.090',
            'IO::Uncompress::Adapter::Identity'=> '2.090',
            'IO::Uncompress::Adapter::Inflate'=> '2.090',
            'IO::Uncompress::AnyInflate'=> '2.090',
            'IO::Uncompress::AnyUncompress'=> '2.090',
            'IO::Uncompress::Base'  => '2.090',
            'IO::Uncompress::Bunzip2'=> '2.090',
            'IO::Uncompress::Gunzip'=> '2.090',
            'IO::Uncompress::Inflate'=> '2.090',
            'IO::Uncompress::RawInflate'=> '2.090',
            'IO::Uncompress::Unzip' => '2.090',
            'List::Util'            => '1.53',
            'List::Util::XS'        => '1.53',
            'Math::BigFloat'        => '1.999818',
            'Math::BigInt'          => '1.999818',
            'Math::BigInt::Calc'    => '1.999818',
            'Math::BigInt::Lib'     => '1.999818',
            'Module::CoreList'      => '5.20191120',
            'Module::CoreList::Utils'=> '5.20191120',
            'Module::Load::Conditional'=> '0.70',
            'POSIX'                 => '1.90',
            'Pod::Simple'           => '3.40',
            'Pod::Simple::BlackBox' => '3.40',
            'Pod::Simple::Checker'  => '3.40',
            'Pod::Simple::Debug'    => '3.40',
            'Pod::Simple::DumpAsText'=> '3.40',
            'Pod::Simple::DumpAsXML'=> '3.40',
            'Pod::Simple::HTML'     => '3.40',
            'Pod::Simple::HTMLBatch'=> '3.40',
            'Pod::Simple::LinkSection'=> '3.40',
            'Pod::Simple::Methody'  => '3.40',
            'Pod::Simple::Progress' => '3.40',
            'Pod::Simple::PullParser'=> '3.40',
            'Pod::Simple::PullParserEndToken'=> '3.40',
            'Pod::Simple::PullParserStartToken'=> '3.40',
            'Pod::Simple::PullParserTextToken'=> '3.40',
            'Pod::Simple::PullParserToken'=> '3.40',
            'Pod::Simple::RTF'      => '3.40',
            'Pod::Simple::Search'   => '3.40',
            'Pod::Simple::SimpleTree'=> '3.40',
            'Pod::Simple::Text'     => '3.40',
            'Pod::Simple::TextContent'=> '3.40',
            'Pod::Simple::TiedOutFH'=> '3.40',
            'Pod::Simple::Transcode'=> '3.40',
            'Pod::Simple::TranscodeDumb'=> '3.40',
            'Pod::Simple::TranscodeSmart'=> '3.40',
            'Pod::Simple::XHTML'    => '3.40',
            'Pod::Simple::XMLOutStream'=> '3.40',
            'Scalar::Util'          => '1.53',
            'Sub::Util'             => '1.53',
            'Sys::Syslog'           => '0.36',
            'Test2'                 => '1.302169',
            'Test2::API'            => '1.302169',
            'Test2::API::Breakage'  => '1.302169',
            'Test2::API::Context'   => '1.302169',
            'Test2::API::Instance'  => '1.302169',
            'Test2::API::Stack'     => '1.302169',
            'Test2::Event'          => '1.302169',
            'Test2::Event::Bail'    => '1.302169',
            'Test2::Event::Diag'    => '1.302169',
            'Test2::Event::Encoding'=> '1.302169',
            'Test2::Event::Exception'=> '1.302169',
            'Test2::Event::Fail'    => '1.302169',
            'Test2::Event::Generic' => '1.302169',
            'Test2::Event::Note'    => '1.302169',
            'Test2::Event::Ok'      => '1.302169',
            'Test2::Event::Pass'    => '1.302169',
            'Test2::Event::Plan'    => '1.302169',
            'Test2::Event::Skip'    => '1.302169',
            'Test2::Event::Subtest' => '1.302169',
            'Test2::Event::TAP::Version'=> '1.302169',
            'Test2::Event::V2'      => '1.302169',
            'Test2::Event::Waiting' => '1.302169',
            'Test2::EventFacet'     => '1.302169',
            'Test2::EventFacet::About'=> '1.302169',
            'Test2::EventFacet::Amnesty'=> '1.302169',
            'Test2::EventFacet::Assert'=> '1.302169',
            'Test2::EventFacet::Control'=> '1.302169',
            'Test2::EventFacet::Error'=> '1.302169',
            'Test2::EventFacet::Hub'=> '1.302169',
            'Test2::EventFacet::Info'=> '1.302169',
            'Test2::EventFacet::Info::Table'=> '1.302169',
            'Test2::EventFacet::Meta'=> '1.302169',
            'Test2::EventFacet::Parent'=> '1.302169',
            'Test2::EventFacet::Plan'=> '1.302169',
            'Test2::EventFacet::Render'=> '1.302169',
            'Test2::EventFacet::Trace'=> '1.302169',
            'Test2::Formatter'      => '1.302169',
            'Test2::Formatter::TAP' => '1.302169',
            'Test2::Hub'            => '1.302169',
            'Test2::Hub::Interceptor'=> '1.302169',
            'Test2::Hub::Interceptor::Terminator'=> '1.302169',
            'Test2::Hub::Subtest'   => '1.302169',
            'Test2::IPC'            => '1.302169',
            'Test2::IPC::Driver'    => '1.302169',
            'Test2::IPC::Driver::Files'=> '1.302169',
            'Test2::Tools::Tiny'    => '1.302169',
            'Test2::Util'           => '1.302169',
            'Test2::Util::ExternalMeta'=> '1.302169',
            'Test2::Util::Facets2Legacy'=> '1.302169',
            'Test2::Util::HashBase' => '1.302169',
            'Test2::Util::Trace'    => '1.302169',
            'Test::Builder'         => '1.302169',
            'Test::Builder::Formatter'=> '1.302169',
            'Test::Builder::Module' => '1.302169',
            'Test::Builder::Tester' => '1.302169',
            'Test::Builder::Tester::Color'=> '1.302169',
            'Test::Builder::TodoDiag'=> '1.302169',
            'Test::More'            => '1.302169',
            'Test::Simple'          => '1.302169',
            'Test::Tester'          => '1.302169',
            'Test::Tester::Capture' => '1.302169',
            'Test::Tester::CaptureRunner'=> '1.302169',
            'Test::Tester::Delegate'=> '1.302169',
            'Test::use::ok'         => '1.302169',
            'Tie::StdHandle'        => '4.6',
            'Unicode::UCD'          => '0.74',
            'Win32API::File'        => '0.1203_01',
            'feature'               => '1.56',
            'mro'                   => '1.23',
            'ok'                    => '1.302169',
            'perlfaq'               => '5.20191102',
        },
        removed => {
        }
    },
    5.031007 => {
        delta_from => 5.031006,
        changed => {
            'B'                     => '1.78',
            'B::Deparse'            => '1.52',
            'B::Op_private'         => '5.031007',
            'Compress::Raw::Bzip2'  => '2.093',
            'Compress::Raw::Zlib'   => '2.093',
            'Compress::Zlib'        => '2.093',
            'Config'                => '5.031007',
            'Devel::PPPort'         => '3.56',
            'English'               => '1.11',
            'ExtUtils::Command'     => '7.42',
            'ExtUtils::Command::MM' => '7.42',
            'ExtUtils::Liblist'     => '7.42',
            'ExtUtils::Liblist::Kid'=> '7.42',
            'ExtUtils::MM'          => '7.42',
            'ExtUtils::MM_AIX'      => '7.42',
            'ExtUtils::MM_Any'      => '7.42',
            'ExtUtils::MM_BeOS'     => '7.42',
            'ExtUtils::MM_Cygwin'   => '7.42',
            'ExtUtils::MM_DOS'      => '7.42',
            'ExtUtils::MM_Darwin'   => '7.42',
            'ExtUtils::MM_MacOS'    => '7.42',
            'ExtUtils::MM_NW5'      => '7.42',
            'ExtUtils::MM_OS2'      => '7.42',
            'ExtUtils::MM_QNX'      => '7.42',
            'ExtUtils::MM_UWIN'     => '7.42',
            'ExtUtils::MM_Unix'     => '7.42',
            'ExtUtils::MM_VMS'      => '7.42',
            'ExtUtils::MM_VOS'      => '7.42',
            'ExtUtils::MM_Win32'    => '7.42',
            'ExtUtils::MM_Win95'    => '7.42',
            'ExtUtils::MY'          => '7.42',
            'ExtUtils::MakeMaker'   => '7.42',
            'ExtUtils::MakeMaker::Config'=> '7.42',
            'ExtUtils::MakeMaker::Locale'=> '7.42',
            'ExtUtils::MakeMaker::version'=> '7.42',
            'ExtUtils::MakeMaker::version::regex'=> '7.42',
            'ExtUtils::Mkbootstrap' => '7.42',
            'ExtUtils::Mksymlists'  => '7.42',
            'ExtUtils::testlib'     => '7.42',
            'File::stat'            => '1.09',
            'Filter::Simple'        => '0.96',
            'IO::Compress::Adapter::Bzip2'=> '2.093',
            'IO::Compress::Adapter::Deflate'=> '2.093',
            'IO::Compress::Adapter::Identity'=> '2.093',
            'IO::Compress::Base'    => '2.093',
            'IO::Compress::Base::Common'=> '2.093',
            'IO::Compress::Bzip2'   => '2.093',
            'IO::Compress::Deflate' => '2.093',
            'IO::Compress::Gzip'    => '2.093',
            'IO::Compress::Gzip::Constants'=> '2.093',
            'IO::Compress::RawDeflate'=> '2.093',
            'IO::Compress::Zip'     => '2.093',
            'IO::Compress::Zip::Constants'=> '2.093',
            'IO::Compress::Zlib::Constants'=> '2.093',
            'IO::Compress::Zlib::Extra'=> '2.093',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.093',
            'IO::Uncompress::Adapter::Identity'=> '2.093',
            'IO::Uncompress::Adapter::Inflate'=> '2.093',
            'IO::Uncompress::AnyInflate'=> '2.093',
            'IO::Uncompress::AnyUncompress'=> '2.093',
            'IO::Uncompress::Base'  => '2.093',
            'IO::Uncompress::Bunzip2'=> '2.093',
            'IO::Uncompress::Gunzip'=> '2.093',
            'IO::Uncompress::Inflate'=> '2.093',
            'IO::Uncompress::RawInflate'=> '2.093',
            'IO::Uncompress::Unzip' => '2.093',
            'Module::CoreList'      => '5.20191220',
            'Module::CoreList::Utils'=> '5.20191220',
            'Net::Ping'             => '2.72',
            'Opcode'                => '1.45',
            'Storable'              => '3.18',
            'Test2'                 => '1.302170',
            'Test2::API'            => '1.302170',
            'Test2::API::Breakage'  => '1.302170',
            'Test2::API::Context'   => '1.302170',
            'Test2::API::Instance'  => '1.302170',
            'Test2::API::Stack'     => '1.302170',
            'Test2::Event'          => '1.302170',
            'Test2::Event::Bail'    => '1.302170',
            'Test2::Event::Diag'    => '1.302170',
            'Test2::Event::Encoding'=> '1.302170',
            'Test2::Event::Exception'=> '1.302170',
            'Test2::Event::Fail'    => '1.302170',
            'Test2::Event::Generic' => '1.302170',
            'Test2::Event::Note'    => '1.302170',
            'Test2::Event::Ok'      => '1.302170',
            'Test2::Event::Pass'    => '1.302170',
            'Test2::Event::Plan'    => '1.302170',
            'Test2::Event::Skip'    => '1.302170',
            'Test2::Event::Subtest' => '1.302170',
            'Test2::Event::TAP::Version'=> '1.302170',
            'Test2::Event::V2'      => '1.302170',
            'Test2::Event::Waiting' => '1.302170',
            'Test2::EventFacet'     => '1.302170',
            'Test2::EventFacet::About'=> '1.302170',
            'Test2::EventFacet::Amnesty'=> '1.302170',
            'Test2::EventFacet::Assert'=> '1.302170',
            'Test2::EventFacet::Control'=> '1.302170',
            'Test2::EventFacet::Error'=> '1.302170',
            'Test2::EventFacet::Hub'=> '1.302170',
            'Test2::EventFacet::Info'=> '1.302170',
            'Test2::EventFacet::Info::Table'=> '1.302170',
            'Test2::EventFacet::Meta'=> '1.302170',
            'Test2::EventFacet::Parent'=> '1.302170',
            'Test2::EventFacet::Plan'=> '1.302170',
            'Test2::EventFacet::Render'=> '1.302170',
            'Test2::EventFacet::Trace'=> '1.302170',
            'Test2::Formatter'      => '1.302170',
            'Test2::Formatter::TAP' => '1.302170',
            'Test2::Hub'            => '1.302170',
            'Test2::Hub::Interceptor'=> '1.302170',
            'Test2::Hub::Interceptor::Terminator'=> '1.302170',
            'Test2::Hub::Subtest'   => '1.302170',
            'Test2::IPC'            => '1.302170',
            'Test2::IPC::Driver'    => '1.302170',
            'Test2::IPC::Driver::Files'=> '1.302170',
            'Test2::Tools::Tiny'    => '1.302170',
            'Test2::Util'           => '1.302170',
            'Test2::Util::ExternalMeta'=> '1.302170',
            'Test2::Util::Facets2Legacy'=> '1.302170',
            'Test2::Util::HashBase' => '1.302170',
            'Test2::Util::Trace'    => '1.302170',
            'Test::Builder'         => '1.302170',
            'Test::Builder::Formatter'=> '1.302170',
            'Test::Builder::Module' => '1.302170',
            'Test::Builder::Tester' => '1.302170',
            'Test::Builder::Tester::Color'=> '1.302170',
            'Test::Builder::TodoDiag'=> '1.302170',
            'Test::More'            => '1.302170',
            'Test::Simple'          => '1.302170',
            'Test::Tester'          => '1.302170',
            'Test::Tester::Capture' => '1.302170',
            'Test::Tester::CaptureRunner'=> '1.302170',
            'Test::Tester::Delegate'=> '1.302170',
            'Test::use::ok'         => '1.302170',
            'Tie::Hash::NamedCapture'=> '0.13',
            'VMS::Stdio'            => '2.45',
            'XS::APItest'           => '1.05',
            'feature'               => '1.57',
            'ok'                    => '1.302170',
            'warnings'              => '1.46',
        },
        removed => {
        }
    },
    5.031008 => {
        delta_from => 5.031007,
        changed => {
            'B::Op_private'         => '5.031008',
            'Config'                => '5.031008',
            'DB_File'               => '1.853',
            'Encode'                => '3.02',
            'ExtUtils::Command'     => '7.44',
            'ExtUtils::Command::MM' => '7.44',
            'ExtUtils::Liblist'     => '7.44',
            'ExtUtils::Liblist::Kid'=> '7.44',
            'ExtUtils::MM'          => '7.44',
            'ExtUtils::MM_AIX'      => '7.44',
            'ExtUtils::MM_Any'      => '7.44',
            'ExtUtils::MM_BeOS'     => '7.44',
            'ExtUtils::MM_Cygwin'   => '7.44',
            'ExtUtils::MM_DOS'      => '7.44',
            'ExtUtils::MM_Darwin'   => '7.44',
            'ExtUtils::MM_MacOS'    => '7.44',
            'ExtUtils::MM_NW5'      => '7.44',
            'ExtUtils::MM_OS2'      => '7.44',
            'ExtUtils::MM_QNX'      => '7.44',
            'ExtUtils::MM_UWIN'     => '7.44',
            'ExtUtils::MM_Unix'     => '7.44',
            'ExtUtils::MM_VMS'      => '7.44',
            'ExtUtils::MM_VOS'      => '7.44',
            'ExtUtils::MM_Win32'    => '7.44',
            'ExtUtils::MM_Win95'    => '7.44',
            'ExtUtils::MY'          => '7.44',
            'ExtUtils::MakeMaker'   => '7.44',
            'ExtUtils::MakeMaker::Config'=> '7.44',
            'ExtUtils::MakeMaker::Locale'=> '7.44',
            'ExtUtils::MakeMaker::version'=> '7.44',
            'ExtUtils::MakeMaker::version::regex'=> '7.44',
            'ExtUtils::Mkbootstrap' => '7.44',
            'ExtUtils::Mksymlists'  => '7.44',
            'ExtUtils::testlib'     => '7.44',
            'Fatal'                 => '2.32',
            'Hash::Util'            => '0.23',
            'IO'                    => '1.42',
            'IO::Handle'            => '1.42',
            'IO::Socket'            => '1.42',
            'Module::CoreList'      => '5.20200120',
            'Module::CoreList::Utils'=> '5.20200120',
            'POSIX'                 => '1.91',
            'Pod::Man'              => '4.14',
            'Pod::ParseLink'        => '4.14',
            'Pod::Text'             => '4.14',
            'Pod::Text::Color'      => '4.14',
            'Pod::Text::Overstrike' => '4.14',
            'Pod::Text::Termcap'    => '4.14',
            'Term::ANSIColor'       => '5.01',
            'Test2'                 => '1.302171',
            'Test2::API'            => '1.302171',
            'Test2::API::Breakage'  => '1.302171',
            'Test2::API::Context'   => '1.302171',
            'Test2::API::Instance'  => '1.302171',
            'Test2::API::Stack'     => '1.302171',
            'Test2::Event'          => '1.302171',
            'Test2::Event::Bail'    => '1.302171',
            'Test2::Event::Diag'    => '1.302171',
            'Test2::Event::Encoding'=> '1.302171',
            'Test2::Event::Exception'=> '1.302171',
            'Test2::Event::Fail'    => '1.302171',
            'Test2::Event::Generic' => '1.302171',
            'Test2::Event::Note'    => '1.302171',
            'Test2::Event::Ok'      => '1.302171',
            'Test2::Event::Pass'    => '1.302171',
            'Test2::Event::Plan'    => '1.302171',
            'Test2::Event::Skip'    => '1.302171',
            'Test2::Event::Subtest' => '1.302171',
            'Test2::Event::TAP::Version'=> '1.302171',
            'Test2::Event::V2'      => '1.302171',
            'Test2::Event::Waiting' => '1.302171',
            'Test2::EventFacet'     => '1.302171',
            'Test2::EventFacet::About'=> '1.302171',
            'Test2::EventFacet::Amnesty'=> '1.302171',
            'Test2::EventFacet::Assert'=> '1.302171',
            'Test2::EventFacet::Control'=> '1.302171',
            'Test2::EventFacet::Error'=> '1.302171',
            'Test2::EventFacet::Hub'=> '1.302171',
            'Test2::EventFacet::Info'=> '1.302171',
            'Test2::EventFacet::Info::Table'=> '1.302171',
            'Test2::EventFacet::Meta'=> '1.302171',
            'Test2::EventFacet::Parent'=> '1.302171',
            'Test2::EventFacet::Plan'=> '1.302171',
            'Test2::EventFacet::Render'=> '1.302171',
            'Test2::EventFacet::Trace'=> '1.302171',
            'Test2::Formatter'      => '1.302171',
            'Test2::Formatter::TAP' => '1.302171',
            'Test2::Hub'            => '1.302171',
            'Test2::Hub::Interceptor'=> '1.302171',
            'Test2::Hub::Interceptor::Terminator'=> '1.302171',
            'Test2::Hub::Subtest'   => '1.302171',
            'Test2::IPC'            => '1.302171',
            'Test2::IPC::Driver'    => '1.302171',
            'Test2::IPC::Driver::Files'=> '1.302171',
            'Test2::Tools::Tiny'    => '1.302171',
            'Test2::Util'           => '1.302171',
            'Test2::Util::ExternalMeta'=> '1.302171',
            'Test2::Util::Facets2Legacy'=> '1.302171',
            'Test2::Util::HashBase' => '1.302171',
            'Test2::Util::Trace'    => '1.302171',
            'Test::Builder'         => '1.302171',
            'Test::Builder::Formatter'=> '1.302171',
            'Test::Builder::Module' => '1.302171',
            'Test::Builder::Tester' => '1.302171',
            'Test::Builder::Tester::Color'=> '1.302171',
            'Test::Builder::TodoDiag'=> '1.302171',
            'Test::More'            => '1.302171',
            'Test::Simple'          => '1.302171',
            'Test::Tester'          => '1.302171',
            'Test::Tester::Capture' => '1.302171',
            'Test::Tester::CaptureRunner'=> '1.302171',
            'Test::Tester::Delegate'=> '1.302171',
            'Test::use::ok'         => '1.302171',
            'XS::APItest'           => '1.06',
            'autodie'               => '2.32',
            'autodie::Scope::Guard' => '2.32',
            'autodie::Scope::GuardStack'=> '2.32',
            'autodie::Util'         => '2.32',
            'autodie::exception'    => '2.32',
            'autodie::exception::system'=> '2.32',
            'autodie::hints'        => '2.32',
            'autodie::skip'         => '2.32',
            'ok'                    => '1.302171',
        },
        removed => {
        }
    },
    5.031009 => {
        delta_from => 5.031008,
        changed => {
            'Archive::Tar'          => '2.36',
            'Archive::Tar::Constant'=> '2.36',
            'Archive::Tar::File'    => '2.36',
            'B'                     => '1.80',
            'B::Op_private'         => '5.031009',
            'Config'                => '5.031009',
            'Devel::PPPort'         => '3.57',
            'Encode'                => '3.03',
            'ExtUtils::CBuilder'    => '0.280234',
            'ExtUtils::CBuilder::Base'=> '0.280234',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.280234',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.280234',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.280234',
            'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280234',
            'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280234',
            'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280234',
            'ExtUtils::CBuilder::Platform::aix'=> '0.280234',
            'ExtUtils::CBuilder::Platform::android'=> '0.280234',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.280234',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.280234',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280234',
            'ExtUtils::CBuilder::Platform::os2'=> '0.280234',
            'IO'                    => '1.43',
            'IO::Select'            => '1.42',
            'IO::Socket'            => '1.43',
            'Module::CoreList'      => '5.20200220',
            'Module::CoreList::Utils'=> '5.20200220',
            'POSIX'                 => '1.92',
            'Pod::Html'             => '1.25',
            'Storable'              => '3.19',
            'Tie::File'             => '1.06',
            'Unicode'               => '13.0.0',
            'XS::APItest'           => '1.07',
            '_charnames'            => '1.46',
            'charnames'             => '1.46',
            'diagnostics'           => '1.37',
            'feature'               => '1.58',
            'parent'                => '0.238',
            'perlfaq'               => '5.20200125',
            'threads'               => '2.24',
        },
        removed => {
        }
    },
    5.030002 => {
        delta_from => 5.030001,
        changed => {
            'B::Op_private'         => '5.030002',
            'Compress::Raw::Bzip2'  => '2.089',
            'Config'                => '5.030002',
            'Module::CoreList'      => '5.20200314',
            'Module::CoreList::Utils'=> '5.20200314',
        },
        removed => {
        }
    },
    5.031010 => {
        delta_from => 5.031009,
        changed => {
            'B::Op_private'         => '5.031010',
            'Config'                => '5.03101',
        },
        removed => {
        }
    },
    5.031011 => {
        delta_from => 5.031010,
        changed => {
            'B::Deparse'            => '1.53',
            'B::Op_private'         => '5.031011',
            'Config'                => '5.031011',
            'DynaLoader'            => '1.47',
            'Encode'                => '3.04',
            'IPC::Open2'            => '1.05',
            'IPC::Open3'            => '1.21',
            'Module::CoreList'      => '5.20200428',
            'Module::CoreList::Utils'=> '5.20200428',
            'Opcode'                => '1.47',
            'POSIX'                 => '1.93',
            'PerlIO'                => '1.11',
            'Storable'              => '3.20',
            'Test2'                 => '1.302175',
            'Test2::API'            => '1.302175',
            'Test2::API::Breakage'  => '1.302175',
            'Test2::API::Context'   => '1.302175',
            'Test2::API::Instance'  => '1.302175',
            'Test2::API::Stack'     => '1.302175',
            'Test2::Event'          => '1.302175',
            'Test2::Event::Bail'    => '1.302175',
            'Test2::Event::Diag'    => '1.302175',
            'Test2::Event::Encoding'=> '1.302175',
            'Test2::Event::Exception'=> '1.302175',
            'Test2::Event::Fail'    => '1.302175',
            'Test2::Event::Generic' => '1.302175',
            'Test2::Event::Note'    => '1.302175',
            'Test2::Event::Ok'      => '1.302175',
            'Test2::Event::Pass'    => '1.302175',
            'Test2::Event::Plan'    => '1.302175',
            'Test2::Event::Skip'    => '1.302175',
            'Test2::Event::Subtest' => '1.302175',
            'Test2::Event::TAP::Version'=> '1.302175',
            'Test2::Event::V2'      => '1.302175',
            'Test2::Event::Waiting' => '1.302175',
            'Test2::EventFacet'     => '1.302175',
            'Test2::EventFacet::About'=> '1.302175',
            'Test2::EventFacet::Amnesty'=> '1.302175',
            'Test2::EventFacet::Assert'=> '1.302175',
            'Test2::EventFacet::Control'=> '1.302175',
            'Test2::EventFacet::Error'=> '1.302175',
            'Test2::EventFacet::Hub'=> '1.302175',
            'Test2::EventFacet::Info'=> '1.302175',
            'Test2::EventFacet::Info::Table'=> '1.302175',
            'Test2::EventFacet::Meta'=> '1.302175',
            'Test2::EventFacet::Parent'=> '1.302175',
            'Test2::EventFacet::Plan'=> '1.302175',
            'Test2::EventFacet::Render'=> '1.302175',
            'Test2::EventFacet::Trace'=> '1.302175',
            'Test2::Formatter'      => '1.302175',
            'Test2::Formatter::TAP' => '1.302175',
            'Test2::Hub'            => '1.302175',
            'Test2::Hub::Interceptor'=> '1.302175',
            'Test2::Hub::Interceptor::Terminator'=> '1.302175',
            'Test2::Hub::Subtest'   => '1.302175',
            'Test2::IPC'            => '1.302175',
            'Test2::IPC::Driver'    => '1.302175',
            'Test2::IPC::Driver::Files'=> '1.302175',
            'Test2::Tools::Tiny'    => '1.302175',
            'Test2::Util'           => '1.302175',
            'Test2::Util::ExternalMeta'=> '1.302175',
            'Test2::Util::Facets2Legacy'=> '1.302175',
            'Test2::Util::HashBase' => '1.302175',
            'Test2::Util::Trace'    => '1.302175',
            'Test::Builder'         => '1.302175',
            'Test::Builder::Formatter'=> '1.302175',
            'Test::Builder::Module' => '1.302175',
            'Test::Builder::Tester' => '1.302175',
            'Test::Builder::Tester::Color'=> '1.302175',
            'Test::Builder::TodoDiag'=> '1.302175',
            'Test::More'            => '1.302175',
            'Test::Simple'          => '1.302175',
            'Test::Tester'          => '1.302175',
            'Test::Tester::Capture' => '1.302175',
            'Test::Tester::CaptureRunner'=> '1.302175',
            'Test::Tester::Delegate'=> '1.302175',
            'Test::use::ok'         => '1.302175',
            'Time::Piece'           => '1.3401',
            'Time::Seconds'         => '1.3401',
            'Unicode::UCD'          => '0.75',
            'XS::APItest'           => '1.09',
            '_charnames'            => '1.47',
            'charnames'             => '1.47',
            'ok'                    => '1.302175',
            'open'                  => '1.12',
            're'                    => '0.39',
            'warnings'              => '1.47',
        },
        removed => {
        }
    },
    5.028003 => {
        delta_from => 5.028002,
        changed => {
            'B::Op_private'         => '5.028003',
            'Config'                => '5.028003',
            'Module::CoreList'      => '5.20200601_28',
            'Module::CoreList::Utils'=> '5.20200601_28',
        },
        removed => {
        }
    },
    5.030003 => {
        delta_from => 5.030002,
        changed => {
            'B::Op_private'         => '5.030003',
            'Config'                => '5.030003',
            'Module::CoreList'      => '5.20200601_30',
            'Module::CoreList::Utils'=> '5.20200601_30',
        },
        removed => {
        }
    },
    5.032000 => {
        delta_from => 5.031011,
        changed => {
            'B::Deparse'            => '1.54',
            'B::Op_private'         => '5.032000',
            'Benchmark'             => '1.23',
            'Config'                => '5.032',
            'Encode'                => '3.06',
            'Encode::Guess'         => '2.08',
            'File::Glob'            => '1.33',
            'List::Util'            => '1.55',
            'List::Util::XS'        => '1.55',
            'Module::CoreList'      => '5.20200620',
            'Module::CoreList::Utils'=> '5.20200620',
            'POSIX'                 => '1.94',
            'Scalar::Util'          => '1.55',
            'Storable'              => '3.21',
            'Sub::Util'             => '1.55',
            'Thread::Queue'         => '3.14',
            'Tie::Scalar'           => '1.05',
            '_charnames'            => '1.48',
            'charnames'             => '1.48',
            'encoding'              => '3.00',
            'perlfaq'               => '5.20200523',
            're'                    => '0.40',
            'threads'               => '2.25',
        },
        removed => {
        }
    },
    5.033000 => {
        delta_from => 5.032000,
        changed => {
            'B::Op_private'         => '5.033000',
            'Config'                => '5.033',
            'Module::CoreList'      => '5.20200717',
            'Module::CoreList::Utils'=> '5.20200717',
            'feature'               => '1.59',
        },
        removed => {
        }
    },
    5.033001 => {
        delta_from => 5.033,
        changed => {
            'B'                     => '1.81',
            'B::Deparse'            => '1.55',
            'B::Op_private'         => '5.033001',
            'Config'                => '5.033001',
            'Data::Dumper'          => '2.175',
            'Devel::PPPort'         => '3.60',
            'Devel::Peek'           => '1.29',
            'DynaLoader'            => '1.48',
            'Errno'                 => '1.31',
            'Exporter'              => '5.75',
            'Exporter::Heavy'       => '5.75',
            'ExtUtils::Miniperl'    => '1.10',
            'ExtUtils::PL2Bat'      => '0.002',
            'ExtUtils::ParseXS'     => '3.41',
            'ExtUtils::ParseXS::Constants'=> '3.41',
            'ExtUtils::ParseXS::CountLines'=> '3.41',
            'ExtUtils::ParseXS::Eval'=> '3.41',
            'ExtUtils::ParseXS::Utilities'=> '3.41',
            'Fcntl'                 => '1.14',
            'File::Path'            => '2.17',
            'Hash::Util'            => '0.24',
            'Hash::Util::FieldHash' => '1.21',
            'IO'                    => '1.44',
            'IO::Socket'            => '1.44',
            'IO::Socket::UNIX'      => '1.42',
            'IPC::Msg'              => '2.08',
            'IPC::Semaphore'        => '2.08',
            'IPC::SharedMem'        => '2.08',
            'IPC::SysV'             => '2.08',
            'JSON::PP'              => '4.05',
            'JSON::PP::Boolean'     => '4.05',
            'Math::Complex'         => '1.5902',
            'Module::CoreList'      => '5.20200820',
            'Module::CoreList::Utils'=> '5.20200820',
            'Net::Ping'             => '2.73_01',
            'POSIX'                 => '1.95',
            'PerlIO::mmap'          => '0.017',
            'Pod::Usage'            => '1.70',
            'Safe'                  => '2.42',
            'Socket'                => '2.030',
            'Storable'              => '3.22',
            'Time::HiRes'           => '1.9765',
            'Unicode::Normalize'    => '1.28',
            'XS::APItest'           => '1.11',
            'XS::Typemap'           => '0.18',
            'feature'               => '1.60',
            'mro'                   => '1.24',
            'strict'                => '1.12',
            'threads'               => '2.26',
            'threads::shared'       => '1.62',
            'warnings'              => '1.48',
        },
        removed => {
            'Moped::Msg'            => 1,
        }
    },
    5.033002 => {
        delta_from => 5.033001,
        changed => {
            'Archive::Tar'          => '2.38',
            'Archive::Tar::Constant'=> '2.38',
            'Archive::Tar::File'    => '2.38',
            'B::Op_private'         => '5.033002',
            'Compress::Raw::Bzip2'  => '2.096',
            'Compress::Raw::Zlib'   => '2.096',
            'Compress::Zlib'        => '2.096',
            'Config'                => '5.033002',
            'DB_File'               => '1.854',
            'Env'                   => '1.05',
            'Errno'                 => '1.32',
            'ExtUtils::Install'     => '2.18',
            'ExtUtils::Installed'   => '2.18',
            'ExtUtils::Packlist'    => '2.18',
            'Filter::Util::Call'    => '1.60',
            'IO::Compress::Adapter::Bzip2'=> '2.096',
            'IO::Compress::Adapter::Deflate'=> '2.096',
            'IO::Compress::Adapter::Identity'=> '2.096',
            'IO::Compress::Base'    => '2.096',
            'IO::Compress::Base::Common'=> '2.096',
            'IO::Compress::Bzip2'   => '2.096',
            'IO::Compress::Deflate' => '2.096',
            'IO::Compress::Gzip'    => '2.096',
            'IO::Compress::Gzip::Constants'=> '2.096',
            'IO::Compress::RawDeflate'=> '2.096',
            'IO::Compress::Zip'     => '2.096',
            'IO::Compress::Zip::Constants'=> '2.096',
            'IO::Compress::Zlib::Constants'=> '2.096',
            'IO::Compress::Zlib::Extra'=> '2.096',
            'IO::Socket::IP'        => '0.41',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.096',
            'IO::Uncompress::Adapter::Identity'=> '2.096',
            'IO::Uncompress::Adapter::Inflate'=> '2.096',
            'IO::Uncompress::AnyInflate'=> '2.096',
            'IO::Uncompress::AnyUncompress'=> '2.096',
            'IO::Uncompress::Base'  => '2.096',
            'IO::Uncompress::Bunzip2'=> '2.096',
            'IO::Uncompress::Gunzip'=> '2.096',
            'IO::Uncompress::Inflate'=> '2.096',
            'IO::Uncompress::RawInflate'=> '2.096',
            'IO::Uncompress::Unzip' => '2.096',
            'IO::Zlib'              => '1.11',
            'Module::CoreList'      => '5.20200920',
            'Module::CoreList::Utils'=> '5.20200920',
            'Module::Load::Conditional'=> '0.74',
            'Opcode'                => '1.48',
            'PerlIO::scalar'        => '0.31',
            'Safe'                  => '2.43',
            'Test2'                 => '1.302181',
            'Test2::API'            => '1.302181',
            'Test2::API::Breakage'  => '1.302181',
            'Test2::API::Context'   => '1.302181',
            'Test2::API::Instance'  => '1.302181',
            'Test2::API::InterceptResult'=> '1.302181',
            'Test2::API::InterceptResult::Event'=> '1.302181',
            'Test2::API::InterceptResult::Facet'=> '1.302181',
            'Test2::API::InterceptResult::Hub'=> '1.302181',
            'Test2::API::InterceptResult::Squasher'=> '1.302181',
            'Test2::API::Stack'     => '1.302181',
            'Test2::Event'          => '1.302181',
            'Test2::Event::Bail'    => '1.302181',
            'Test2::Event::Diag'    => '1.302181',
            'Test2::Event::Encoding'=> '1.302181',
            'Test2::Event::Exception'=> '1.302181',
            'Test2::Event::Fail'    => '1.302181',
            'Test2::Event::Generic' => '1.302181',
            'Test2::Event::Note'    => '1.302181',
            'Test2::Event::Ok'      => '1.302181',
            'Test2::Event::Pass'    => '1.302181',
            'Test2::Event::Plan'    => '1.302181',
            'Test2::Event::Skip'    => '1.302181',
            'Test2::Event::Subtest' => '1.302181',
            'Test2::Event::TAP::Version'=> '1.302181',
            'Test2::Event::V2'      => '1.302181',
            'Test2::Event::Waiting' => '1.302181',
            'Test2::EventFacet'     => '1.302181',
            'Test2::EventFacet::About'=> '1.302181',
            'Test2::EventFacet::Amnesty'=> '1.302181',
            'Test2::EventFacet::Assert'=> '1.302181',
            'Test2::EventFacet::Control'=> '1.302181',
            'Test2::EventFacet::Error'=> '1.302181',
            'Test2::EventFacet::Hub'=> '1.302181',
            'Test2::EventFacet::Info'=> '1.302181',
            'Test2::EventFacet::Info::Table'=> '1.302181',
            'Test2::EventFacet::Meta'=> '1.302181',
            'Test2::EventFacet::Parent'=> '1.302181',
            'Test2::EventFacet::Plan'=> '1.302181',
            'Test2::EventFacet::Render'=> '1.302181',
            'Test2::EventFacet::Trace'=> '1.302181',
            'Test2::Formatter'      => '1.302181',
            'Test2::Formatter::TAP' => '1.302181',
            'Test2::Hub'            => '1.302181',
            'Test2::Hub::Interceptor'=> '1.302181',
            'Test2::Hub::Interceptor::Terminator'=> '1.302181',
            'Test2::Hub::Subtest'   => '1.302181',
            'Test2::IPC'            => '1.302181',
            'Test2::IPC::Driver'    => '1.302181',
            'Test2::IPC::Driver::Files'=> '1.302181',
            'Test2::Tools::Tiny'    => '1.302181',
            'Test2::Util'           => '1.302181',
            'Test2::Util::ExternalMeta'=> '1.302181',
            'Test2::Util::Facets2Legacy'=> '1.302181',
            'Test2::Util::HashBase' => '1.302181',
            'Test2::Util::Trace'    => '1.302181',
            'Test::Builder'         => '1.302181',
            'Test::Builder::Formatter'=> '1.302181',
            'Test::Builder::Module' => '1.302181',
            'Test::Builder::Tester' => '1.302181',
            'Test::Builder::Tester::Color'=> '1.302181',
            'Test::Builder::TodoDiag'=> '1.302181',
            'Test::More'            => '1.302181',
            'Test::Simple'          => '1.302181',
            'Test::Tester'          => '1.302181',
            'Test::Tester::Capture' => '1.302181',
            'Test::Tester::CaptureRunner'=> '1.302181',
            'Test::Tester::Delegate'=> '1.302181',
            'Test::use::ok'         => '1.302181',
            'ok'                    => '1.302181',
            'overload'              => '1.32',
        },
        removed => {
        }
    },
    5.033003 => {
        delta_from => 5.033002,
        changed => {
            'Amiga::ARexx'          => '0.05',
            'App::Cpan'             => '1.676',
            'B::Op_private'         => '5.033003',
            'CPAN'                  => '2.28',
            'CPAN::FTP'             => '5.5013',
            'CPAN::FirstTime'       => '5.5315',
            'Config'                => '5.033003',
            'DB_File'               => '1.855',
            'Data::Dumper'          => '2.176',
            'Devel::PPPort'         => '3.62',
            'Devel::Peek'           => '1.30',
            'Digest'                => '1.19',
            'Digest::MD5'           => '2.58',
            'Digest::base'          => '1.19',
            'Digest::file'          => '1.19',
            'Encode'                => '3.07',
            'Encode::GSM0338'       => '2.08',
            'Errno'                 => '1.33',
            'Exporter'              => '5.76',
            'Exporter::Heavy'       => '5.76',
            'ExtUtils::Command'     => '7.48',
            'ExtUtils::Command::MM' => '7.48',
            'ExtUtils::Liblist'     => '7.48',
            'ExtUtils::Liblist::Kid'=> '7.48',
            'ExtUtils::MM'          => '7.48',
            'ExtUtils::MM_AIX'      => '7.48',
            'ExtUtils::MM_Any'      => '7.48',
            'ExtUtils::MM_BeOS'     => '7.48',
            'ExtUtils::MM_Cygwin'   => '7.48',
            'ExtUtils::MM_DOS'      => '7.48',
            'ExtUtils::MM_Darwin'   => '7.48',
            'ExtUtils::MM_MacOS'    => '7.48',
            'ExtUtils::MM_NW5'      => '7.48',
            'ExtUtils::MM_OS2'      => '7.48',
            'ExtUtils::MM_OS390'    => '7.48',
            'ExtUtils::MM_QNX'      => '7.48',
            'ExtUtils::MM_UWIN'     => '7.48',
            'ExtUtils::MM_Unix'     => '7.48',
            'ExtUtils::MM_VMS'      => '7.48',
            'ExtUtils::MM_VOS'      => '7.48',
            'ExtUtils::MM_Win32'    => '7.48',
            'ExtUtils::MM_Win95'    => '7.48',
            'ExtUtils::MY'          => '7.48',
            'ExtUtils::MakeMaker'   => '7.48',
            'ExtUtils::MakeMaker::Config'=> '7.48',
            'ExtUtils::MakeMaker::Locale'=> '7.48',
            'ExtUtils::MakeMaker::version'=> '7.48',
            'ExtUtils::MakeMaker::version::regex'=> '7.48',
            'ExtUtils::Mkbootstrap' => '7.48',
            'ExtUtils::Mksymlists'  => '7.48',
            'ExtUtils::PL2Bat'      => '0.003',
            'ExtUtils::testlib'     => '7.48',
            'File::Temp'            => '0.2311',
            'FindBin'               => '1.52',
            'Getopt::Long'          => '2.52',
            'Getopt::Std'           => '1.13',
            'I18N::LangTags'        => '0.45',
            'MIME::Base64'          => '3.16',
            'MIME::QuotedPrint'     => '3.16',
            'Module::CoreList'      => '5.20201020',
            'Module::CoreList::Utils'=> '5.20201020',
            'Module::Load'          => '0.36',
            'Pod::Checker'          => '1.74',
            'Pod::Simple'           => '3.41',
            'Pod::Simple::BlackBox' => '3.41',
            'Pod::Simple::Checker'  => '3.41',
            'Pod::Simple::Debug'    => '3.41',
            'Pod::Simple::DumpAsText'=> '3.41',
            'Pod::Simple::DumpAsXML'=> '3.41',
            'Pod::Simple::HTML'     => '3.41',
            'Pod::Simple::HTMLBatch'=> '3.41',
            'Pod::Simple::LinkSection'=> '3.41',
            'Pod::Simple::Methody'  => '3.41',
            'Pod::Simple::Progress' => '3.41',
            'Pod::Simple::PullParser'=> '3.41',
            'Pod::Simple::PullParserEndToken'=> '3.41',
            'Pod::Simple::PullParserStartToken'=> '3.41',
            'Pod::Simple::PullParserTextToken'=> '3.41',
            'Pod::Simple::PullParserToken'=> '3.41',
            'Pod::Simple::RTF'      => '3.41',
            'Pod::Simple::Search'   => '3.41',
            'Pod::Simple::SimpleTree'=> '3.41',
            'Pod::Simple::Text'     => '3.41',
            'Pod::Simple::TextContent'=> '3.41',
            'Pod::Simple::TiedOutFH'=> '3.41',
            'Pod::Simple::Transcode'=> '3.41',
            'Pod::Simple::TranscodeDumb'=> '3.41',
            'Pod::Simple::TranscodeSmart'=> '3.41',
            'Pod::Simple::XHTML'    => '3.41',
            'Pod::Simple::XMLOutStream'=> '3.41',
            'Pod::Usage'            => '2.01',
            'Storable'              => '3.23',
            'Symbol'                => '1.09',
            'Test2'                 => '1.302182',
            'Test2::API'            => '1.302182',
            'Test2::API::Breakage'  => '1.302182',
            'Test2::API::Context'   => '1.302182',
            'Test2::API::Instance'  => '1.302182',
            'Test2::API::InterceptResult'=> '1.302182',
            'Test2::API::InterceptResult::Event'=> '1.302182',
            'Test2::API::InterceptResult::Facet'=> '1.302182',
            'Test2::API::InterceptResult::Hub'=> '1.302182',
            'Test2::API::InterceptResult::Squasher'=> '1.302182',
            'Test2::API::Stack'     => '1.302182',
            'Test2::Event'          => '1.302182',
            'Test2::Event::Bail'    => '1.302182',
            'Test2::Event::Diag'    => '1.302182',
            'Test2::Event::Encoding'=> '1.302182',
            'Test2::Event::Exception'=> '1.302182',
            'Test2::Event::Fail'    => '1.302182',
            'Test2::Event::Generic' => '1.302182',
            'Test2::Event::Note'    => '1.302182',
            'Test2::Event::Ok'      => '1.302182',
            'Test2::Event::Pass'    => '1.302182',
            'Test2::Event::Plan'    => '1.302182',
            'Test2::Event::Skip'    => '1.302182',
            'Test2::Event::Subtest' => '1.302182',
            'Test2::Event::TAP::Version'=> '1.302182',
            'Test2::Event::V2'      => '1.302182',
            'Test2::Event::Waiting' => '1.302182',
            'Test2::EventFacet'     => '1.302182',
            'Test2::EventFacet::About'=> '1.302182',
            'Test2::EventFacet::Amnesty'=> '1.302182',
            'Test2::EventFacet::Assert'=> '1.302182',
            'Test2::EventFacet::Control'=> '1.302182',
            'Test2::EventFacet::Error'=> '1.302182',
            'Test2::EventFacet::Hub'=> '1.302182',
            'Test2::EventFacet::Info'=> '1.302182',
            'Test2::EventFacet::Info::Table'=> '1.302182',
            'Test2::EventFacet::Meta'=> '1.302182',
            'Test2::EventFacet::Parent'=> '1.302182',
            'Test2::EventFacet::Plan'=> '1.302182',
            'Test2::EventFacet::Render'=> '1.302182',
            'Test2::EventFacet::Trace'=> '1.302182',
            'Test2::Formatter'      => '1.302182',
            'Test2::Formatter::TAP' => '1.302182',
            'Test2::Hub'            => '1.302182',
            'Test2::Hub::Interceptor'=> '1.302182',
            'Test2::Hub::Interceptor::Terminator'=> '1.302182',
            'Test2::Hub::Subtest'   => '1.302182',
            'Test2::IPC'            => '1.302182',
            'Test2::IPC::Driver'    => '1.302182',
            'Test2::IPC::Driver::Files'=> '1.302182',
            'Test2::Tools::Tiny'    => '1.302182',
            'Test2::Util'           => '1.302182',
            'Test2::Util::ExternalMeta'=> '1.302182',
            'Test2::Util::Facets2Legacy'=> '1.302182',
            'Test2::Util::HashBase' => '1.302182',
            'Test2::Util::Trace'    => '1.302182',
            'Test::Builder'         => '1.302182',
            'Test::Builder::Formatter'=> '1.302182',
            'Test::Builder::Module' => '1.302182',
            'Test::Builder::Tester' => '1.302182',
            'Test::Builder::Tester::Color'=> '1.302182',
            'Test::Builder::TodoDiag'=> '1.302182',
            'Test::More'            => '1.302182',
            'Test::Simple'          => '1.302182',
            'Test::Tester'          => '1.302182',
            'Test::Tester::Capture' => '1.302182',
            'Test::Tester::CaptureRunner'=> '1.302182',
            'Test::Tester::Delegate'=> '1.302182',
            'Test::use::ok'         => '1.302182',
            'Tie::RefHash'          => '1.40',
            'Time::Local'           => '1.30',
            'Unicode::Collate'      => '1.29',
            'Unicode::Collate::CJK::Big5'=> '1.29',
            'Unicode::Collate::CJK::GB2312'=> '1.29',
            'Unicode::Collate::CJK::JISX0208'=> '1.29',
            'Unicode::Collate::CJK::Korean'=> '1.29',
            'Unicode::Collate::CJK::Pinyin'=> '1.29',
            'Unicode::Collate::CJK::Stroke'=> '1.29',
            'Unicode::Collate::CJK::Zhuyin'=> '1.29',
            'Unicode::Collate::Locale'=> '1.29',
            'Win32'                 => '0.54',
            'XS::APItest'           => '1.12',
            'bytes'                 => '1.08',
            'experimental'          => '0.022',
            'feature'               => '1.61',
            'if'                    => '0.0609',
            'locale'                => '1.10',
            'mro'                   => '1.25',
            'ok'                    => '1.302182',
            'overload'              => '1.33',
            're'                    => '0.41',
            'subs'                  => '1.04',
            'utf8'                  => '1.24',
            'version'               => '0.9928',
            'version::regex'        => '0.9928',
        },
        removed => {
        }
    },
    5.033004 => {
        delta_from => 5.033003,
        changed => {
            'B'                     => '1.82',
            'B::Op_private'         => '5.033004',
            'Config'                => '5.033004',
            'Cwd'                   => '3.79',
            'ExtUtils::CBuilder'    => '0.280235',
            'ExtUtils::CBuilder::Base'=> '0.280235',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.280235',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.280235',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.280235',
            'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280235',
            'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280235',
            'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280235',
            'ExtUtils::CBuilder::Platform::aix'=> '0.280235',
            'ExtUtils::CBuilder::Platform::android'=> '0.280235',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.280235',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.280235',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280235',
            'ExtUtils::CBuilder::Platform::os2'=> '0.280235',
            'ExtUtils::Command'     => '7.56',
            'ExtUtils::Command::MM' => '7.56',
            'ExtUtils::Liblist'     => '7.56',
            'ExtUtils::Liblist::Kid'=> '7.56',
            'ExtUtils::MM'          => '7.56',
            'ExtUtils::MM_AIX'      => '7.56',
            'ExtUtils::MM_Any'      => '7.56',
            'ExtUtils::MM_BeOS'     => '7.56',
            'ExtUtils::MM_Cygwin'   => '7.56',
            'ExtUtils::MM_DOS'      => '7.56',
            'ExtUtils::MM_Darwin'   => '7.56',
            'ExtUtils::MM_MacOS'    => '7.56',
            'ExtUtils::MM_NW5'      => '7.56',
            'ExtUtils::MM_OS2'      => '7.56',
            'ExtUtils::MM_OS390'    => '7.56',
            'ExtUtils::MM_QNX'      => '7.56',
            'ExtUtils::MM_UWIN'     => '7.56',
            'ExtUtils::MM_Unix'     => '7.56',
            'ExtUtils::MM_VMS'      => '7.56',
            'ExtUtils::MM_VOS'      => '7.56',
            'ExtUtils::MM_Win32'    => '7.56',
            'ExtUtils::MM_Win95'    => '7.56',
            'ExtUtils::MY'          => '7.56',
            'ExtUtils::MakeMaker'   => '7.56',
            'ExtUtils::MakeMaker::Config'=> '7.56',
            'ExtUtils::MakeMaker::Locale'=> '7.56',
            'ExtUtils::MakeMaker::version'=> '7.56',
            'ExtUtils::MakeMaker::version::regex'=> '7.56',
            'ExtUtils::Mkbootstrap' => '7.56',
            'ExtUtils::Mksymlists'  => '7.56',
            'ExtUtils::testlib'     => '7.56',
            'File::Fetch'           => '1.00',
            'File::Path'            => '2.18',
            'File::Spec'            => '3.79',
            'File::Spec::AmigaOS'   => '3.79',
            'File::Spec::Cygwin'    => '3.79',
            'File::Spec::Epoc'      => '3.79',
            'File::Spec::Functions' => '3.79',
            'File::Spec::Mac'       => '3.79',
            'File::Spec::OS2'       => '3.79',
            'File::Spec::Unix'      => '3.79',
            'File::Spec::VMS'       => '3.79',
            'IPC::Msg'              => '2.09',
            'IPC::Semaphore'        => '2.09',
            'IPC::SharedMem'        => '2.09',
            'IPC::SysV'             => '2.09',
            'Module::CoreList'      => '5.20201120',
            'Module::CoreList::Utils'=> '5.20201120',
            'Net::Ping'             => '2.74',
            'Pod::Html'             => '1.26',
            'Pod::Simple'           => '3.42',
            'Pod::Simple::BlackBox' => '3.42',
            'Pod::Simple::Checker'  => '3.42',
            'Pod::Simple::Debug'    => '3.42',
            'Pod::Simple::DumpAsText'=> '3.42',
            'Pod::Simple::DumpAsXML'=> '3.42',
            'Pod::Simple::HTML'     => '3.42',
            'Pod::Simple::HTMLBatch'=> '3.42',
            'Pod::Simple::LinkSection'=> '3.42',
            'Pod::Simple::Methody'  => '3.42',
            'Pod::Simple::Progress' => '3.42',
            'Pod::Simple::PullParser'=> '3.42',
            'Pod::Simple::PullParserEndToken'=> '3.42',
            'Pod::Simple::PullParserStartToken'=> '3.42',
            'Pod::Simple::PullParserTextToken'=> '3.42',
            'Pod::Simple::PullParserToken'=> '3.42',
            'Pod::Simple::RTF'      => '3.42',
            'Pod::Simple::Search'   => '3.42',
            'Pod::Simple::SimpleTree'=> '3.42',
            'Pod::Simple::Text'     => '3.42',
            'Pod::Simple::TextContent'=> '3.42',
            'Pod::Simple::TiedOutFH'=> '3.42',
            'Pod::Simple::Transcode'=> '3.42',
            'Pod::Simple::TranscodeDumb'=> '3.42',
            'Pod::Simple::TranscodeSmart'=> '3.42',
            'Pod::Simple::XHTML'    => '3.42',
            'Pod::Simple::XMLOutStream'=> '3.42',
            'Test2'                 => '1.302183',
            'Test2::API'            => '1.302183',
            'Test2::API::Breakage'  => '1.302183',
            'Test2::API::Context'   => '1.302183',
            'Test2::API::Instance'  => '1.302183',
            'Test2::API::InterceptResult'=> '1.302183',
            'Test2::API::InterceptResult::Event'=> '1.302183',
            'Test2::API::InterceptResult::Facet'=> '1.302183',
            'Test2::API::InterceptResult::Hub'=> '1.302183',
            'Test2::API::InterceptResult::Squasher'=> '1.302183',
            'Test2::API::Stack'     => '1.302183',
            'Test2::Event'          => '1.302183',
            'Test2::Event::Bail'    => '1.302183',
            'Test2::Event::Diag'    => '1.302183',
            'Test2::Event::Encoding'=> '1.302183',
            'Test2::Event::Exception'=> '1.302183',
            'Test2::Event::Fail'    => '1.302183',
            'Test2::Event::Generic' => '1.302183',
            'Test2::Event::Note'    => '1.302183',
            'Test2::Event::Ok'      => '1.302183',
            'Test2::Event::Pass'    => '1.302183',
            'Test2::Event::Plan'    => '1.302183',
            'Test2::Event::Skip'    => '1.302183',
            'Test2::Event::Subtest' => '1.302183',
            'Test2::Event::TAP::Version'=> '1.302183',
            'Test2::Event::V2'      => '1.302183',
            'Test2::Event::Waiting' => '1.302183',
            'Test2::EventFacet'     => '1.302183',
            'Test2::EventFacet::About'=> '1.302183',
            'Test2::EventFacet::Amnesty'=> '1.302183',
            'Test2::EventFacet::Assert'=> '1.302183',
            'Test2::EventFacet::Control'=> '1.302183',
            'Test2::EventFacet::Error'=> '1.302183',
            'Test2::EventFacet::Hub'=> '1.302183',
            'Test2::EventFacet::Info'=> '1.302183',
            'Test2::EventFacet::Info::Table'=> '1.302183',
            'Test2::EventFacet::Meta'=> '1.302183',
            'Test2::EventFacet::Parent'=> '1.302183',
            'Test2::EventFacet::Plan'=> '1.302183',
            'Test2::EventFacet::Render'=> '1.302183',
            'Test2::EventFacet::Trace'=> '1.302183',
            'Test2::Formatter'      => '1.302183',
            'Test2::Formatter::TAP' => '1.302183',
            'Test2::Hub'            => '1.302183',
            'Test2::Hub::Interceptor'=> '1.302183',
            'Test2::Hub::Interceptor::Terminator'=> '1.302183',
            'Test2::Hub::Subtest'   => '1.302183',
            'Test2::IPC'            => '1.302183',
            'Test2::IPC::Driver'    => '1.302183',
            'Test2::IPC::Driver::Files'=> '1.302183',
            'Test2::Tools::Tiny'    => '1.302183',
            'Test2::Util'           => '1.302183',
            'Test2::Util::ExternalMeta'=> '1.302183',
            'Test2::Util::Facets2Legacy'=> '1.302183',
            'Test2::Util::HashBase' => '1.302183',
            'Test2::Util::Trace'    => '1.302183',
            'Test::Builder'         => '1.302183',
            'Test::Builder::Formatter'=> '1.302183',
            'Test::Builder::Module' => '1.302183',
            'Test::Builder::Tester' => '1.302183',
            'Test::Builder::Tester::Color'=> '1.302183',
            'Test::Builder::TodoDiag'=> '1.302183',
            'Test::More'            => '1.302183',
            'Test::Simple'          => '1.302183',
            'Test::Tester'          => '1.302183',
            'Test::Tester::Capture' => '1.302183',
            'Test::Tester::CaptureRunner'=> '1.302183',
            'Test::Tester::Delegate'=> '1.302183',
            'Test::use::ok'         => '1.302183',
            'XS::APItest'           => '1.13',
            'ok'                    => '1.302183',
            'perlfaq'               => '5.20201107',
        },
        removed => {
        }
    },
    5.033005 => {
        delta_from => 5.033004,
        changed => {
            'App::Prove'            => '3.43',
            'App::Prove::State'     => '3.43',
            'App::Prove::State::Result'=> '3.43',
            'App::Prove::State::Result::Test'=> '3.43',
            'B::Op_private'         => '5.033005',
            'Carp'                  => '1.51',
            'Carp::Heavy'           => '1.51',
            'Config'                => '5.033005',
            'Config::Perl::V'       => '0.33',
            'Cwd'                   => '3.80',
            'DynaLoader'            => '1.49',
            'Encode'                => '3.08',
            'Encode::GSM0338'       => '2.09',
            'ExtUtils::Install'     => '2.20',
            'ExtUtils::Installed'   => '2.20',
            'ExtUtils::Packlist'    => '2.20',
            'ExtUtils::ParseXS'     => '3.42',
            'ExtUtils::ParseXS::Constants'=> '3.42',
            'ExtUtils::ParseXS::CountLines'=> '3.42',
            'ExtUtils::ParseXS::Eval'=> '3.42',
            'ExtUtils::ParseXS::Utilities'=> '3.42',
            'File::Copy'            => '2.35',
            'File::Find'            => '1.38',
            'File::Spec'            => '3.80',
            'File::Spec::AmigaOS'   => '3.80',
            'File::Spec::Cygwin'    => '3.80',
            'File::Spec::Epoc'      => '3.80',
            'File::Spec::Functions' => '3.80',
            'File::Spec::Mac'       => '3.80',
            'File::Spec::OS2'       => '3.80',
            'File::Spec::Unix'      => '3.80',
            'File::Spec::VMS'       => '3.80',
            'File::Spec::Win32'     => '3.80',
            'Module::CoreList'      => '5.20201220',
            'Module::CoreList::Utils'=> '5.20201220',
            'Net::Cmd'              => '3.12',
            'Net::Config'           => '3.12',
            'Net::Domain'           => '3.12',
            'Net::FTP'              => '3.12',
            'Net::FTP::A'           => '3.12',
            'Net::FTP::E'           => '3.12',
            'Net::FTP::I'           => '3.12',
            'Net::FTP::L'           => '3.12',
            'Net::FTP::dataconn'    => '3.12',
            'Net::NNTP'             => '3.12',
            'Net::Netrc'            => '3.12',
            'Net::POP3'             => '3.12',
            'Net::SMTP'             => '3.12',
            'Net::Time'             => '3.12',
            'ODBM_File'             => '1.17',
            'Opcode'                => '1.49',
            'POSIX'                 => '1.96',
            'PerlIO::via::QuotedPrint'=> '0.09',
            'TAP::Base'             => '3.43',
            'TAP::Formatter::Base'  => '3.43',
            'TAP::Formatter::Color' => '3.43',
            'TAP::Formatter::Console'=> '3.43',
            'TAP::Formatter::Console::ParallelSession'=> '3.43',
            'TAP::Formatter::Console::Session'=> '3.43',
            'TAP::Formatter::File'  => '3.43',
            'TAP::Formatter::File::Session'=> '3.43',
            'TAP::Formatter::Session'=> '3.43',
            'TAP::Harness'          => '3.43',
            'TAP::Harness::Env'     => '3.43',
            'TAP::Object'           => '3.43',
            'TAP::Parser'           => '3.43',
            'TAP::Parser::Aggregator'=> '3.43',
            'TAP::Parser::Grammar'  => '3.43',
            'TAP::Parser::Iterator' => '3.43',
            'TAP::Parser::Iterator::Array'=> '3.43',
            'TAP::Parser::Iterator::Process'=> '3.43',
            'TAP::Parser::Iterator::Stream'=> '3.43',
            'TAP::Parser::IteratorFactory'=> '3.43',
            'TAP::Parser::Multiplexer'=> '3.43',
            'TAP::Parser::Result'   => '3.43',
            'TAP::Parser::Result::Bailout'=> '3.43',
            'TAP::Parser::Result::Comment'=> '3.43',
            'TAP::Parser::Result::Plan'=> '3.43',
            'TAP::Parser::Result::Pragma'=> '3.43',
            'TAP::Parser::Result::Test'=> '3.43',
            'TAP::Parser::Result::Unknown'=> '3.43',
            'TAP::Parser::Result::Version'=> '3.43',
            'TAP::Parser::Result::YAML'=> '3.43',
            'TAP::Parser::ResultFactory'=> '3.43',
            'TAP::Parser::Scheduler'=> '3.43',
            'TAP::Parser::Scheduler::Job'=> '3.43',
            'TAP::Parser::Scheduler::Spinner'=> '3.43',
            'TAP::Parser::Source'   => '3.43',
            'TAP::Parser::SourceHandler'=> '3.43',
            'TAP::Parser::SourceHandler::Executable'=> '3.43',
            'TAP::Parser::SourceHandler::File'=> '3.43',
            'TAP::Parser::SourceHandler::Handle'=> '3.43',
            'TAP::Parser::SourceHandler::Perl'=> '3.43',
            'TAP::Parser::SourceHandler::RawTAP'=> '3.43',
            'TAP::Parser::YAMLish::Reader'=> '3.43',
            'TAP::Parser::YAMLish::Writer'=> '3.43',
            'Test::Harness'         => '3.43',
            'Text::Balanced'        => '2.04',
            'Time::HiRes'           => '1.9766',
            'XS::APItest'           => '1.14',
            'warnings'              => '1.49',
        },
        removed => {
        }
    },
    5.033006 => {
        delta_from => 5.033005,
        changed => {
            'B::Op_private'         => '5.033006',
            'Carp'                  => '1.52',
            'Carp::Heavy'           => '1.52',
            'Compress::Raw::Bzip2'  => '2.100',
            'Compress::Raw::Zlib'   => '2.100',
            'Compress::Zlib'        => '2.100',
            'Config'                => '5.033006',
            'DynaLoader'            => '1.50',
            'ExtUtils::Command'     => '7.58',
            'ExtUtils::Command::MM' => '7.58',
            'ExtUtils::Liblist'     => '7.58',
            'ExtUtils::Liblist::Kid'=> '7.58',
            'ExtUtils::MM'          => '7.58',
            'ExtUtils::MM_AIX'      => '7.58',
            'ExtUtils::MM_Any'      => '7.58',
            'ExtUtils::MM_BeOS'     => '7.58',
            'ExtUtils::MM_Cygwin'   => '7.58',
            'ExtUtils::MM_DOS'      => '7.58',
            'ExtUtils::MM_Darwin'   => '7.58',
            'ExtUtils::MM_MacOS'    => '7.58',
            'ExtUtils::MM_NW5'      => '7.58',
            'ExtUtils::MM_OS2'      => '7.58',
            'ExtUtils::MM_OS390'    => '7.58',
            'ExtUtils::MM_QNX'      => '7.58',
            'ExtUtils::MM_UWIN'     => '7.58',
            'ExtUtils::MM_Unix'     => '7.58',
            'ExtUtils::MM_VMS'      => '7.58',
            'ExtUtils::MM_VOS'      => '7.58',
            'ExtUtils::MM_Win32'    => '7.58',
            'ExtUtils::MM_Win95'    => '7.58',
            'ExtUtils::MY'          => '7.58',
            'ExtUtils::MakeMaker'   => '7.58',
            'ExtUtils::MakeMaker::Config'=> '7.58',
            'ExtUtils::MakeMaker::Locale'=> '7.58',
            'ExtUtils::MakeMaker::version'=> '7.58',
            'ExtUtils::MakeMaker::version::regex'=> '7.58',
            'ExtUtils::Manifest'    => '1.73',
            'ExtUtils::Mkbootstrap' => '7.58',
            'ExtUtils::Mksymlists'  => '7.58',
            'ExtUtils::testlib'     => '7.58',
            'GDBM_File'             => '1.19',
            'IO'                    => '1.45',
            'IO::Compress::Adapter::Bzip2'=> '2.100',
            'IO::Compress::Adapter::Deflate'=> '2.100',
            'IO::Compress::Adapter::Identity'=> '2.100',
            'IO::Compress::Base'    => '2.100',
            'IO::Compress::Base::Common'=> '2.100',
            'IO::Compress::Bzip2'   => '2.100',
            'IO::Compress::Deflate' => '2.100',
            'IO::Compress::Gzip'    => '2.100',
            'IO::Compress::Gzip::Constants'=> '2.100',
            'IO::Compress::RawDeflate'=> '2.100',
            'IO::Compress::Zip'     => '2.100',
            'IO::Compress::Zip::Constants'=> '2.100',
            'IO::Compress::Zlib::Constants'=> '2.100',
            'IO::Compress::Zlib::Extra'=> '2.100',
            'IO::Dir'               => '1.45',
            'IO::File'              => '1.45',
            'IO::Handle'            => '1.45',
            'IO::Pipe'              => '1.45',
            'IO::Poll'              => '1.45',
            'IO::Seekable'          => '1.45',
            'IO::Select'            => '1.45',
            'IO::Socket'            => '1.45',
            'IO::Socket::INET'      => '1.45',
            'IO::Socket::UNIX'      => '1.45',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.100',
            'IO::Uncompress::Adapter::Identity'=> '2.100',
            'IO::Uncompress::Adapter::Inflate'=> '2.100',
            'IO::Uncompress::AnyInflate'=> '2.100',
            'IO::Uncompress::AnyUncompress'=> '2.100',
            'IO::Uncompress::Base'  => '2.100',
            'IO::Uncompress::Bunzip2'=> '2.100',
            'IO::Uncompress::Gunzip'=> '2.100',
            'IO::Uncompress::Inflate'=> '2.100',
            'IO::Uncompress::RawInflate'=> '2.100',
            'IO::Uncompress::Unzip' => '2.100',
            'Module::CoreList'      => '5.20210120',
            'Module::CoreList::Utils'=> '5.20210120',
            'Net::Cmd'              => '3.13',
            'Net::Config'           => '3.13',
            'Net::Domain'           => '3.13',
            'Net::FTP'              => '3.13',
            'Net::FTP::A'           => '3.13',
            'Net::FTP::E'           => '3.13',
            'Net::FTP::I'           => '3.13',
            'Net::FTP::L'           => '3.13',
            'Net::FTP::dataconn'    => '3.13',
            'Net::NNTP'             => '3.13',
            'Net::Netrc'            => '3.13',
            'Net::POP3'             => '3.13',
            'Net::SMTP'             => '3.13',
            'Net::Time'             => '3.13',
            'POSIX'                 => '1.97',
            'Socket'                => '2.031',
            'XS::APItest'           => '1.15',
            'feature'               => '1.62',
            'warnings'              => '1.50',
        },
        removed => {
        }
    },
    5.032001 => {
        delta_from => 5.032000,
        changed => {
            'B::Op_private'         => '5.032001',
            'Config'                => '5.032001',
            'Data::Dumper'          => '2.174_01',
            'DynaLoader'            => '1.47_01',
            'ExtUtils::Liblist::Kid'=> '7.44_01',
            'Module::CoreList'      => '5.20210123',
            'Module::CoreList::Utils'=> '5.20210123',
            'Opcode'                => '1.48',
            'Safe'                  => '2.41_01',
            'Win32API::File::inc::ExtUtils::Myconst2perl'=> '1',
        },
        removed => {
        }
    },
    5.033007 => {
        delta_from => 5.033006,
        changed => {
            'B::Deparse'            => '1.56',
            'B::Op_private'         => '5.033007',
            'Config'                => '5.033007',
            'ExtUtils::CBuilder'    => '0.280236',
            'ExtUtils::CBuilder::Base'=> '0.280236',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.280236',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.280236',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.280236',
            'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280236',
            'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280236',
            'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280236',
            'ExtUtils::CBuilder::Platform::aix'=> '0.280236',
            'ExtUtils::CBuilder::Platform::android'=> '0.280236',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.280236',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.280236',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280236',
            'ExtUtils::CBuilder::Platform::os2'=> '0.280236',
            'ExtUtils::Command'     => '7.60',
            'ExtUtils::Command::MM' => '7.60',
            'ExtUtils::Liblist'     => '7.60',
            'ExtUtils::Liblist::Kid'=> '7.60',
            'ExtUtils::MM'          => '7.60',
            'ExtUtils::MM_AIX'      => '7.60',
            'ExtUtils::MM_Any'      => '7.60',
            'ExtUtils::MM_BeOS'     => '7.60',
            'ExtUtils::MM_Cygwin'   => '7.60',
            'ExtUtils::MM_DOS'      => '7.60',
            'ExtUtils::MM_Darwin'   => '7.60',
            'ExtUtils::MM_MacOS'    => '7.60',
            'ExtUtils::MM_NW5'      => '7.60',
            'ExtUtils::MM_OS2'      => '7.60',
            'ExtUtils::MM_OS390'    => '7.60',
            'ExtUtils::MM_QNX'      => '7.60',
            'ExtUtils::MM_UWIN'     => '7.60',
            'ExtUtils::MM_Unix'     => '7.60',
            'ExtUtils::MM_VMS'      => '7.60',
            'ExtUtils::MM_VOS'      => '7.60',
            'ExtUtils::MM_Win32'    => '7.60',
            'ExtUtils::MM_Win95'    => '7.60',
            'ExtUtils::MY'          => '7.60',
            'ExtUtils::MakeMaker'   => '7.60',
            'ExtUtils::MakeMaker::Config'=> '7.60',
            'ExtUtils::MakeMaker::Locale'=> '7.60',
            'ExtUtils::MakeMaker::version'=> '7.60',
            'ExtUtils::MakeMaker::version::regex'=> '7.60',
            'ExtUtils::Mkbootstrap' => '7.60',
            'ExtUtils::Mksymlists'  => '7.60',
            'ExtUtils::PL2Bat'      => '0.004',
            'ExtUtils::testlib'     => '7.60',
            'Fatal'                 => '2.34',
            'File::Find'            => '1.39',
            'Hash::Util'            => '0.25',
            'IO'                    => '1.46',
            'IO::Dir'               => '1.46',
            'IO::File'              => '1.46',
            'IO::Handle'            => '1.46',
            'IO::Pipe'              => '1.46',
            'IO::Poll'              => '1.46',
            'IO::Seekable'          => '1.46',
            'IO::Select'            => '1.46',
            'IO::Socket'            => '1.46',
            'IO::Socket::INET'      => '1.46',
            'IO::Socket::UNIX'      => '1.46',
            'JSON::PP'              => '4.06',
            'JSON::PP::Boolean'     => '4.06',
            'Module::CoreList'      => '5.20210220',
            'Module::CoreList::Utils'=> '5.20210220',
            'Opcode'                => '1.50',
            'PerlIO::encoding'      => '0.30',
            'Time::HiRes'           => '1.9767',
            'autodie'               => '2.34',
            'autodie::Scope::Guard' => '2.34',
            'autodie::Scope::GuardStack'=> '2.34',
            'autodie::Util'         => '2.34',
            'autodie::exception'    => '2.34',
            'autodie::exception::system'=> '2.34',
            'autodie::hints'        => '2.34',
            'autodie::skip'         => '2.34',
            'feature'               => '1.63',
            'mro'                   => '1.25_001',
            'warnings'              => '1.51',
        },
        removed => {
        }
    },
    5.033008 => {
        delta_from => 5.033007,
        changed => {
            'B::Op_private'         => '5.033008',
            'Compress::Raw::Bzip2'  => '2.101',
            'Compress::Raw::Zlib'   => '2.101',
            'Compress::Zlib'        => '2.102',
            'Config'                => '5.033008',
            'Data::Dumper'          => '2.177',
            'IO::Compress::Adapter::Bzip2'=> '2.102',
            'IO::Compress::Adapter::Deflate'=> '2.102',
            'IO::Compress::Adapter::Identity'=> '2.102',
            'IO::Compress::Base'    => '2.102',
            'IO::Compress::Base::Common'=> '2.102',
            'IO::Compress::Bzip2'   => '2.102',
            'IO::Compress::Deflate' => '2.102',
            'IO::Compress::Gzip'    => '2.102',
            'IO::Compress::Gzip::Constants'=> '2.102',
            'IO::Compress::RawDeflate'=> '2.102',
            'IO::Compress::Zip'     => '2.102',
            'IO::Compress::Zip::Constants'=> '2.102',
            'IO::Compress::Zlib::Constants'=> '2.102',
            'IO::Compress::Zlib::Extra'=> '2.102',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.102',
            'IO::Uncompress::Adapter::Identity'=> '2.102',
            'IO::Uncompress::Adapter::Inflate'=> '2.102',
            'IO::Uncompress::AnyInflate'=> '2.102',
            'IO::Uncompress::AnyUncompress'=> '2.102',
            'IO::Uncompress::Base'  => '2.102',
            'IO::Uncompress::Bunzip2'=> '2.102',
            'IO::Uncompress::Gunzip'=> '2.102',
            'IO::Uncompress::Inflate'=> '2.102',
            'IO::Uncompress::RawInflate'=> '2.102',
            'IO::Uncompress::Unzip' => '2.102',
            'Module::CoreList'      => '5.20210320',
            'Module::CoreList::Utils'=> '5.20210320',
            'Pod::Html'             => '1.27',
            'Win32'                 => '0.57',
        },
        removed => {
        }
    },
    5.033009 => {
        delta_from => 5.033008,
        changed => {
            'B::Op_private'         => '5.033009',
            'Config'                => '5.033009',
            'Data::Dumper'          => '2.178',
            'ExtUtils::Command'     => '7.62',
            'ExtUtils::Command::MM' => '7.62',
            'ExtUtils::Liblist'     => '7.62',
            'ExtUtils::Liblist::Kid'=> '7.62',
            'ExtUtils::MM'          => '7.62',
            'ExtUtils::MM_AIX'      => '7.62',
            'ExtUtils::MM_Any'      => '7.62',
            'ExtUtils::MM_BeOS'     => '7.62',
            'ExtUtils::MM_Cygwin'   => '7.62',
            'ExtUtils::MM_DOS'      => '7.62',
            'ExtUtils::MM_Darwin'   => '7.62',
            'ExtUtils::MM_MacOS'    => '7.62',
            'ExtUtils::MM_NW5'      => '7.62',
            'ExtUtils::MM_OS2'      => '7.62',
            'ExtUtils::MM_OS390'    => '7.62',
            'ExtUtils::MM_QNX'      => '7.62',
            'ExtUtils::MM_UWIN'     => '7.62',
            'ExtUtils::MM_Unix'     => '7.62',
            'ExtUtils::MM_VMS'      => '7.62',
            'ExtUtils::MM_VOS'      => '7.62',
            'ExtUtils::MM_Win32'    => '7.62',
            'ExtUtils::MM_Win95'    => '7.62',
            'ExtUtils::MY'          => '7.62',
            'ExtUtils::MakeMaker'   => '7.62',
            'ExtUtils::MakeMaker::Config'=> '7.62',
            'ExtUtils::MakeMaker::Locale'=> '7.62',
            'ExtUtils::MakeMaker::version'=> '7.62',
            'ExtUtils::MakeMaker::version::regex'=> '7.62',
            'ExtUtils::Mkbootstrap' => '7.62',
            'ExtUtils::Mksymlists'  => '7.62',
            'ExtUtils::ParseXS'     => '3.43',
            'ExtUtils::ParseXS::Constants'=> '3.43',
            'ExtUtils::ParseXS::CountLines'=> '3.43',
            'ExtUtils::ParseXS::Eval'=> '3.43',
            'ExtUtils::ParseXS::Utilities'=> '3.43',
            'ExtUtils::Typemaps'    => '3.43',
            'ExtUtils::Typemaps::Cmd'=> '3.43',
            'ExtUtils::Typemaps::InputMap'=> '3.43',
            'ExtUtils::Typemaps::OutputMap'=> '3.43',
            'ExtUtils::Typemaps::Type'=> '3.43',
            'ExtUtils::testlib'     => '7.62',
            'Module::CoreList'      => '5.20210420',
            'Module::CoreList::Utils'=> '5.20210420',
            'NEXT'                  => '0.68',
            'XS::APItest'           => '1.16',
            'feature'               => '1.64',
            'perlfaq'               => '5.20210411',
        },
        removed => {
        }
    },
    5.034000 => {
        delta_from => 5.033009,
        changed => {
            'B::Op_private'         => '5.034000',
            'Config'                => '5.034',
            'Data::Dumper'          => '2.179',
            'Module::CoreList'      => '5.20210520',
            'Module::CoreList::Utils'=> '5.20210520',
            'experimental'          => '0.024',
        },
        removed => {
        }
    },
    5.035000 => {
        delta_from => 5.034,
        changed => {
            'B::Op_private'         => '5.035000',
            'Config'                => '5.035',
            'Module::CoreList'      => '5.20210521',
            'Module::CoreList::Utils'=> '5.20210521',
            'feature'               => '1.65',
        },
        removed => {
        }
    },
    5.035001 => {
        delta_from => 5.035000,
        changed => {
            'B::Deparse'            => '1.57',
            'B::Op_private'         => '5.035001',
            'Config'                => '5.035001',
            'Cwd'                   => '3.81',
            'Data::Dumper'          => '2.181',
            'File::Copy'            => '2.36',
            'File::Glob'            => '1.35',
            'File::Spec'            => '3.81',
            'File::Spec::AmigaOS'   => '3.81',
            'File::Spec::Cygwin'    => '3.81',
            'File::Spec::Epoc'      => '3.81',
            'File::Spec::Functions' => '3.81',
            'File::Spec::Mac'       => '3.81',
            'File::Spec::OS2'       => '3.81',
            'File::Spec::Unix'      => '3.81',
            'File::Spec::VMS'       => '3.81',
            'File::Spec::Win32'     => '3.81',
            'File::stat'            => '1.10',
            'IO'                    => '1.47',
            'IO::Dir'               => '1.47',
            'IO::File'              => '1.47',
            'IO::Handle'            => '1.47',
            'IO::Pipe'              => '1.47',
            'IO::Poll'              => '1.47',
            'IO::Seekable'          => '1.47',
            'IO::Select'            => '1.47',
            'IO::Socket'            => '1.47',
            'IO::Socket::INET'      => '1.47',
            'IO::Socket::UNIX'      => '1.47',
            'List::Util'            => '1.56',
            'List::Util::XS'        => '1.56',
            'Module::CoreList'      => '5.20210620',
            'Module::CoreList::Utils'=> '5.20210620',
            'Opcode'                => '1.51',
            'POSIX'                 => '1.98',
            'Scalar::Util'          => '1.56',
            'Socket'                => '2.032',
            'Sub::Util'             => '1.56',
            'Test2'                 => '1.302185',
            'Test2::API'            => '1.302185',
            'Test2::API::Breakage'  => '1.302185',
            'Test2::API::Context'   => '1.302185',
            'Test2::API::Instance'  => '1.302185',
            'Test2::API::InterceptResult'=> '1.302185',
            'Test2::API::InterceptResult::Event'=> '1.302185',
            'Test2::API::InterceptResult::Facet'=> '1.302185',
            'Test2::API::InterceptResult::Hub'=> '1.302185',
            'Test2::API::InterceptResult::Squasher'=> '1.302185',
            'Test2::API::Stack'     => '1.302185',
            'Test2::Event'          => '1.302185',
            'Test2::Event::Bail'    => '1.302185',
            'Test2::Event::Diag'    => '1.302185',
            'Test2::Event::Encoding'=> '1.302185',
            'Test2::Event::Exception'=> '1.302185',
            'Test2::Event::Fail'    => '1.302185',
            'Test2::Event::Generic' => '1.302185',
            'Test2::Event::Note'    => '1.302185',
            'Test2::Event::Ok'      => '1.302185',
            'Test2::Event::Pass'    => '1.302185',
            'Test2::Event::Plan'    => '1.302185',
            'Test2::Event::Skip'    => '1.302185',
            'Test2::Event::Subtest' => '1.302185',
            'Test2::Event::TAP::Version'=> '1.302185',
            'Test2::Event::V2'      => '1.302185',
            'Test2::Event::Waiting' => '1.302185',
            'Test2::EventFacet'     => '1.302185',
            'Test2::EventFacet::About'=> '1.302185',
            'Test2::EventFacet::Amnesty'=> '1.302185',
            'Test2::EventFacet::Assert'=> '1.302185',
            'Test2::EventFacet::Control'=> '1.302185',
            'Test2::EventFacet::Error'=> '1.302185',
            'Test2::EventFacet::Hub'=> '1.302185',
            'Test2::EventFacet::Info'=> '1.302185',
            'Test2::EventFacet::Info::Table'=> '1.302185',
            'Test2::EventFacet::Meta'=> '1.302185',
            'Test2::EventFacet::Parent'=> '1.302185',
            'Test2::EventFacet::Plan'=> '1.302185',
            'Test2::EventFacet::Render'=> '1.302185',
            'Test2::EventFacet::Trace'=> '1.302185',
            'Test2::Formatter'      => '1.302185',
            'Test2::Formatter::TAP' => '1.302185',
            'Test2::Hub'            => '1.302185',
            'Test2::Hub::Interceptor'=> '1.302185',
            'Test2::Hub::Interceptor::Terminator'=> '1.302185',
            'Test2::Hub::Subtest'   => '1.302185',
            'Test2::IPC'            => '1.302185',
            'Test2::IPC::Driver'    => '1.302185',
            'Test2::IPC::Driver::Files'=> '1.302185',
            'Test2::Tools::Tiny'    => '1.302185',
            'Test2::Util'           => '1.302185',
            'Test2::Util::ExternalMeta'=> '1.302185',
            'Test2::Util::Facets2Legacy'=> '1.302185',
            'Test2::Util::HashBase' => '1.302185',
            'Test2::Util::Trace'    => '1.302185',
            'Test::Builder'         => '1.302185',
            'Test::Builder::Formatter'=> '1.302185',
            'Test::Builder::Module' => '1.302185',
            'Test::Builder::Tester' => '1.302185',
            'Test::Builder::Tester::Color'=> '1.302185',
            'Test::Builder::TodoDiag'=> '1.302185',
            'Test::More'            => '1.302185',
            'Test::Simple'          => '1.302185',
            'Test::Tester'          => '1.302185',
            'Test::Tester::Capture' => '1.302185',
            'Test::Tester::CaptureRunner'=> '1.302185',
            'Test::Tester::Delegate'=> '1.302185',
            'Test::use::ok'         => '1.302185',
            'Unicode::Collate'      => '1.30',
            'Unicode::Collate::CJK::Big5'=> '1.30',
            'Unicode::Collate::CJK::GB2312'=> '1.30',
            'Unicode::Collate::CJK::JISX0208'=> '1.30',
            'Unicode::Collate::CJK::Korean'=> '1.30',
            'Unicode::Collate::CJK::Pinyin'=> '1.30',
            'Unicode::Collate::CJK::Stroke'=> '1.30',
            'Unicode::Collate::CJK::Zhuyin'=> '1.30',
            'Unicode::Collate::Locale'=> '1.30',
            'Unicode::UCD'          => '0.76',
            'XS::APItest'           => '1.17',
            'feature'               => '1.66',
            'ok'                    => '1.302185',
            'open'                  => '1.13',
            'perlfaq'               => '5.20210520',
            'version'               => '0.9929',
            'version::regex'        => '0.9929',
        },
        removed => {
        }
    },
    5.035002 => {
        delta_from => 5.035001,
        changed => {
            'Amiga::ARexx'          => '0.06',
            'Amiga::Exec'           => '0.03',
            'B::Concise'            => '1.005',
            'B::Op_private'         => '5.035002',
            'Config'                => '5.035002',
            'Cwd'                   => '3.82',
            'DB_File'               => '1.856',
            'Data::Dumper'          => '2.183',
            'Devel::PPPort'         => '3.63',
            'Devel::Peek'           => '1.31',
            'DynaLoader'            => '1.51',
            'Encode'                => '3.10',
            'Encode::JP'            => '2.05',
            'Errno'                 => '1.34',
            'ExtUtils::Miniperl'    => '1.11',
            'Fcntl'                 => '1.15',
            'File::Find'            => '1.39_01',
            'File::Glob'            => '1.36',
            'File::stat'            => '1.11',
            'GDBM_File'             => '1.20',
            'Hash::Util'            => '0.26',
            'Hash::Util::FieldHash' => '1.22',
            'I18N::Langinfo'        => '0.20',
            'IPC::Open2'            => '1.06',
            'IPC::Open3'            => '1.22',
            'Math::BigFloat'        => '1.999823',
            'Math::BigFloat::Trace' => '0.53',
            'Math::BigInt'          => '1.999823',
            'Math::BigInt::Calc'    => '1.999823',
            'Math::BigInt::FastCalc'=> '0.5010',
            'Math::BigInt::Lib'     => '1.999823',
            'Math::BigInt::Trace'   => '0.53',
            'Math::BigRat'          => '0.2617',
            'Module::CoreList'      => '5.20210723',
            'Module::CoreList::Utils'=> '5.20210723',
            'Opcode'                => '1.52',
            'PerlIO'                => '1.12',
            'Pod::Functions'        => '1.14',
            'Pod::Html'             => '1.31',
            'Pod::Html::Util'       => '1.31',
            'Pod::Simple'           => '3.43',
            'Pod::Simple::BlackBox' => '3.43',
            'Pod::Simple::Checker'  => '3.43',
            'Pod::Simple::Debug'    => '3.43',
            'Pod::Simple::DumpAsText'=> '3.43',
            'Pod::Simple::DumpAsXML'=> '3.43',
            'Pod::Simple::HTML'     => '3.43',
            'Pod::Simple::HTMLBatch'=> '3.43',
            'Pod::Simple::LinkSection'=> '3.43',
            'Pod::Simple::Methody'  => '3.43',
            'Pod::Simple::Progress' => '3.43',
            'Pod::Simple::PullParser'=> '3.43',
            'Pod::Simple::PullParserEndToken'=> '3.43',
            'Pod::Simple::PullParserStartToken'=> '3.43',
            'Pod::Simple::PullParserTextToken'=> '3.43',
            'Pod::Simple::PullParserToken'=> '3.43',
            'Pod::Simple::RTF'      => '3.43',
            'Pod::Simple::Search'   => '3.43',
            'Pod::Simple::SimpleTree'=> '3.43',
            'Pod::Simple::Text'     => '3.43',
            'Pod::Simple::TextContent'=> '3.43',
            'Pod::Simple::TiedOutFH'=> '3.43',
            'Pod::Simple::Transcode'=> '3.43',
            'Pod::Simple::TranscodeDumb'=> '3.43',
            'Pod::Simple::TranscodeSmart'=> '3.43',
            'Pod::Simple::XHTML'    => '3.43',
            'Pod::Simple::XMLOutStream'=> '3.43',
            'Sys::Hostname'         => '1.24',
            'Text::Tabs'            => '2021.0717',
            'Text::Wrap'            => '2021.0717',
            'Time::HiRes'           => '1.9768',
            'Unicode::UCD'          => '0.77',
            'VMS::Filespec'         => '1.13',
            'VMS::Stdio'            => '2.46',
            'XSLoader'              => '0.31',
            'bigint'                => '0.53',
            'bignum'                => '0.53',
            'bigrat'                => '0.53',
            'if'                    => '0.0610',
            'threads'               => '2.27',
            'warnings'              => '1.52',
        },
        removed => {
        }
    },
    5.035003 => {
        delta_from => 5.035002,
        changed => {
            'Archive::Tar'          => '2.40',
            'Archive::Tar::Constant'=> '2.40',
            'Archive::Tar::File'    => '2.40',
            'B::Op_private'         => '5.035003',
            'Config'                => '5.035003',
            'Encode'                => '3.12',
            'Encode::GSM0338'       => '2.10',
            'Encode::Unicode'       => '2.19',
            'Module::CoreList'      => '5.20210820',
            'Module::CoreList::Utils'=> '5.20210820',
            'Net::hostent'          => '1.03',
            'Opcode'                => '1.53',
            'POSIX'                 => '1.99',
            'Pod::Html'             => '1.32',
            'Pod::Html::Util'       => '1.32',
            'Storable'              => '3.24',
            'Text::Tabs'            => '2021.0804',
            'Text::Wrap'            => '2021.0804',
            'Time::HiRes'           => '1.9769',
            'Unicode::Normalize'    => '1.30',
            'XS::APItest'           => '1.18',
            'diagnostics'           => '1.38',
            'feature'               => '1.67',
            'sort'                  => '2.05',
            'threads::shared'       => '1.63',
            'warnings'              => '1.53',
        },
        removed => {
        }
    },
    5.035004 => {
        delta_from => 5.035003,
        changed => {
            'B'                     => '1.83',
            'B::Deparse'            => '1.58',
            'B::Op_private'         => '5.035004',
            'Config'                => '5.035004',
            'Devel::Peek'           => '1.32',
            'Exporter'              => '5.77',
            'Exporter::Heavy'       => '5.77',
            'ExtUtils::ParseXS'     => '3.44',
            'ExtUtils::ParseXS::Constants'=> '3.44',
            'ExtUtils::ParseXS::CountLines'=> '3.44',
            'ExtUtils::ParseXS::Eval'=> '3.44',
            'ExtUtils::ParseXS::Utilities'=> '3.44',
            'ExtUtils::Typemaps'    => '3.44',
            'ExtUtils::Typemaps::Cmd'=> '3.44',
            'ExtUtils::Typemaps::InputMap'=> '3.44',
            'ExtUtils::Typemaps::OutputMap'=> '3.44',
            'ExtUtils::Typemaps::Type'=> '3.44',
            'Hash::Util::FieldHash' => '1.23',
            'IO'                    => '1.48',
            'IO::Dir'               => '1.48',
            'IO::File'              => '1.48',
            'IO::Handle'            => '1.48',
            'IO::Pipe'              => '1.48',
            'IO::Poll'              => '1.48',
            'IO::Seekable'          => '1.48',
            'IO::Select'            => '1.48',
            'IO::Socket'            => '1.48',
            'IO::Socket::INET'      => '1.48',
            'IO::Socket::UNIX'      => '1.48',
            'List::Util'            => '1.56_001',
            'List::Util::XS'        => '1.56_001',
            'Module::CoreList'      => '5.20210920',
            'Module::CoreList::Utils'=> '5.20210920',
            'Opcode'                => '1.54',
            'Pod::Html'             => '1.33',
            'Pod::Html::Util'       => '1.33',
            'Scalar::Util'          => '1.56_001',
            'Storable'              => '3.25',
            'Sub::Util'             => '1.56_001',
            'Text::Tabs'            => '2021.0814',
            'Text::Wrap'            => '2021.0814',
            'UNIVERSAL'             => '1.14',
            'Unicode'               => '14.0.0',
            'Unicode::Collate'      => '1.31',
            'Unicode::Collate::CJK::Big5'=> '1.31',
            'Unicode::Collate::CJK::GB2312'=> '1.31',
            'Unicode::Collate::CJK::JISX0208'=> '1.31',
            'Unicode::Collate::CJK::Korean'=> '1.31',
            'Unicode::Collate::CJK::Pinyin'=> '1.31',
            'Unicode::Collate::CJK::Stroke'=> '1.31',
            'Unicode::Collate::CJK::Zhuyin'=> '1.31',
            'Unicode::Collate::Locale'=> '1.31',
            'Unicode::UCD'          => '0.78',
            'XS::APItest'           => '1.19',
            'XS::Typemap'           => '0.19',
            'attributes'            => '0.34',
            'feature'               => '1.68',
            'mro'                   => '1.26',
            'threads::shared'       => '1.64',
            'warnings'              => '1.54',
        },
        removed => {
        }
    },
    5.035005 => {
        delta_from => 5.035004,
        changed => {
            'B::Concise'            => '1.006',
            'B::Deparse'            => '1.59',
            'B::Op_private'         => '5.035005',
            'Config'                => '5.035005',
            'Digest'                => '1.20',
            'Digest::base'          => '1.20',
            'Digest::file'          => '1.20',
            'DynaLoader'            => '1.52',
            'Encode'                => '3.16',
            'Errno'                 => '1.35',
            'File::Copy'            => '2.37',
            'File::Spec::Unix'      => '3.82',
            'FindBin'               => '1.53',
            'GDBM_File'             => '1.21',
            'HTTP::Tiny'            => '0.078',
            'I18N::Langinfo'        => '0.21',
            'IO::Dir'               => '1.49',
            'IO::Pipe'              => '1.49',
            'IO::Poll'              => '1.49',
            'IO::Select'            => '1.49',
            'IO::Socket'            => '1.49',
            'IO::Socket::INET'      => '1.49',
            'IO::Socket::UNIX'      => '1.49',
            'List::Util'            => '1.60',
            'List::Util::XS'        => '1.60',
            'Math::BigRat::Trace'   => '0.63',
            'Module::CoreList'      => '5.20211020',
            'Module::CoreList::Utils'=> '5.20211020',
            'POSIX'                 => '2.01',
            'Scalar::Util'          => '1.60',
            'Sub::Util'             => '1.60',
            'Test2'                 => '1.302188',
            'Test2::API'            => '1.302188',
            'Test2::API::Breakage'  => '1.302188',
            'Test2::API::Context'   => '1.302188',
            'Test2::API::Instance'  => '1.302188',
            'Test2::API::InterceptResult'=> '1.302188',
            'Test2::API::InterceptResult::Event'=> '1.302188',
            'Test2::API::InterceptResult::Facet'=> '1.302188',
            'Test2::API::InterceptResult::Hub'=> '1.302188',
            'Test2::API::InterceptResult::Squasher'=> '1.302188',
            'Test2::API::Stack'     => '1.302188',
            'Test2::Event'          => '1.302188',
            'Test2::Event::Bail'    => '1.302188',
            'Test2::Event::Diag'    => '1.302188',
            'Test2::Event::Encoding'=> '1.302188',
            'Test2::Event::Exception'=> '1.302188',
            'Test2::Event::Fail'    => '1.302188',
            'Test2::Event::Generic' => '1.302188',
            'Test2::Event::Note'    => '1.302188',
            'Test2::Event::Ok'      => '1.302188',
            'Test2::Event::Pass'    => '1.302188',
            'Test2::Event::Plan'    => '1.302188',
            'Test2::Event::Skip'    => '1.302188',
            'Test2::Event::Subtest' => '1.302188',
            'Test2::Event::TAP::Version'=> '1.302188',
            'Test2::Event::V2'      => '1.302188',
            'Test2::Event::Waiting' => '1.302188',
            'Test2::EventFacet'     => '1.302188',
            'Test2::EventFacet::About'=> '1.302188',
            'Test2::EventFacet::Amnesty'=> '1.302188',
            'Test2::EventFacet::Assert'=> '1.302188',
            'Test2::EventFacet::Control'=> '1.302188',
            'Test2::EventFacet::Error'=> '1.302188',
            'Test2::EventFacet::Hub'=> '1.302188',
            'Test2::EventFacet::Info'=> '1.302188',
            'Test2::EventFacet::Info::Table'=> '1.302188',
            'Test2::EventFacet::Meta'=> '1.302188',
            'Test2::EventFacet::Parent'=> '1.302188',
            'Test2::EventFacet::Plan'=> '1.302188',
            'Test2::EventFacet::Render'=> '1.302188',
            'Test2::EventFacet::Trace'=> '1.302188',
            'Test2::Formatter'      => '1.302188',
            'Test2::Formatter::TAP' => '1.302188',
            'Test2::Hub'            => '1.302188',
            'Test2::Hub::Interceptor'=> '1.302188',
            'Test2::Hub::Interceptor::Terminator'=> '1.302188',
            'Test2::Hub::Subtest'   => '1.302188',
            'Test2::IPC'            => '1.302188',
            'Test2::IPC::Driver'    => '1.302188',
            'Test2::IPC::Driver::Files'=> '1.302188',
            'Test2::Tools::Tiny'    => '1.302188',
            'Test2::Util'           => '1.302188',
            'Test2::Util::ExternalMeta'=> '1.302188',
            'Test2::Util::Facets2Legacy'=> '1.302188',
            'Test2::Util::HashBase' => '1.302188',
            'Test2::Util::Trace'    => '1.302188',
            'Test::Builder'         => '1.302188',
            'Test::Builder::Formatter'=> '1.302188',
            'Test::Builder::Module' => '1.302188',
            'Test::Builder::Tester' => '1.302188',
            'Test::Builder::Tester::Color'=> '1.302188',
            'Test::Builder::TodoDiag'=> '1.302188',
            'Test::More'            => '1.302188',
            'Test::Simple'          => '1.302188',
            'Test::Tester'          => '1.302188',
            'Test::Tester::Capture' => '1.302188',
            'Test::Tester::CaptureRunner'=> '1.302188',
            'Test::Tester::Delegate'=> '1.302188',
            'Test::use::ok'         => '1.302188',
            'Tie::Handle'           => '4.3',
            'Tie::Hash'             => '1.06',
            'Tie::Scalar'           => '1.06',
            'XS::APItest'           => '1.20',
            'experimental'          => '0.025',
            'ok'                    => '1.302188',
            'warnings'              => '1.55',
        },
        removed => {
        }
    },
    5.035006 => {
        delta_from => 5.035005,
        changed => {
            'B::Op_private'         => '5.035006',
            'Config'                => '5.035006',
            'File::Glob'            => '1.37',
            'File::stat'            => '1.12',
            'GDBM_File'             => '1.22',
            'HTTP::Tiny'            => '0.080',
            'Math::BigFloat'        => '1.999827',
            'Math::BigFloat::Trace' => '0.63',
            'Math::BigInt'          => '1.999827',
            'Math::BigInt::Calc'    => '1.999827',
            'Math::BigInt::FastCalc'=> '0.5012',
            'Math::BigInt::Lib'     => '1.999827',
            'Math::BigInt::Trace'   => '0.63',
            'Math::BigRat'          => '0.2620',
            'Module::CoreList'      => '5.20211120',
            'Module::CoreList::Utils'=> '5.20211120',
            'POSIX'                 => '2.02',
            'bigint'                => '0.63',
            'bignum'                => '0.63',
            'bigrat'                => '0.63',
            'diagnostics'           => '1.39',
            'feature'               => '1.69',
            'warnings'              => '1.56',
        },
        removed => {
        }
    },
    5.035007 => {
        delta_from => 5.035006,
        changed => {
            'B::Deparse'            => '1.60',
            'B::Op_private'         => '5.035007',
            'CPAN'                  => '2.29',
            'CPAN::Distribution'    => '2.29',
            'CPAN::FTP'             => '5.5014',
            'CPAN::FirstTime'       => '5.5316',
            'CPAN::HandleConfig'    => '5.5012',
            'CPAN::Index'           => '2.29',
            'Config'                => '5.035007',
            'Cwd'                   => '3.83',
            'ExtUtils::Command'     => '7.64',
            'ExtUtils::Command::MM' => '7.64',
            'ExtUtils::Liblist'     => '7.64',
            'ExtUtils::Liblist::Kid'=> '7.64',
            'ExtUtils::MM'          => '7.64',
            'ExtUtils::MM_AIX'      => '7.64',
            'ExtUtils::MM_Any'      => '7.64',
            'ExtUtils::MM_BeOS'     => '7.64',
            'ExtUtils::MM_Cygwin'   => '7.64',
            'ExtUtils::MM_DOS'      => '7.64',
            'ExtUtils::MM_Darwin'   => '7.64',
            'ExtUtils::MM_MacOS'    => '7.64',
            'ExtUtils::MM_NW5'      => '7.64',
            'ExtUtils::MM_OS2'      => '7.64',
            'ExtUtils::MM_OS390'    => '7.64',
            'ExtUtils::MM_QNX'      => '7.64',
            'ExtUtils::MM_UWIN'     => '7.64',
            'ExtUtils::MM_Unix'     => '7.64',
            'ExtUtils::MM_VMS'      => '7.64',
            'ExtUtils::MM_VOS'      => '7.64',
            'ExtUtils::MM_Win32'    => '7.64',
            'ExtUtils::MM_Win95'    => '7.64',
            'ExtUtils::MY'          => '7.64',
            'ExtUtils::MakeMaker'   => '7.64',
            'ExtUtils::MakeMaker::Config'=> '7.64',
            'ExtUtils::MakeMaker::Locale'=> '7.64',
            'ExtUtils::MakeMaker::version'=> '7.64',
            'ExtUtils::MakeMaker::version::regex'=> '7.64',
            'ExtUtils::Mkbootstrap' => '7.64',
            'ExtUtils::Mksymlists'  => '7.64',
            'ExtUtils::testlib'     => '7.64',
            'File::Compare'         => '1.1007',
            'File::Copy'            => '2.38',
            'File::Spec'            => '3.83',
            'File::Spec::AmigaOS'   => '3.83',
            'File::Spec::Cygwin'    => '3.83',
            'File::Spec::Epoc'      => '3.83',
            'File::Spec::Functions' => '3.83',
            'File::Spec::Mac'       => '3.83',
            'File::Spec::OS2'       => '3.83',
            'File::Spec::Unix'      => '3.83',
            'File::Spec::VMS'       => '3.83',
            'File::Spec::Win32'     => '3.83',
            'Hash::Util'            => '0.27',
            'Hash::Util::FieldHash' => '1.24',
            'IO'                    => '1.49',
            'JSON::PP'              => '4.07',
            'JSON::PP::Boolean'     => '4.07',
            'Math::BigFloat'        => '1.999828',
            'Math::BigInt'          => '1.999828',
            'Math::BigInt::Calc'    => '1.999828',
            'Math::BigInt::Lib'     => '1.999828',
            'Module::CoreList'      => '5.20211220',
            'Module::CoreList::Utils'=> '5.20211220',
            'Opcode'                => '1.55',
            'builtin'               => '0.001',
            'overload'              => '1.34',
        },
        removed => {
        }
    },
    5.035008 => {
        delta_from => 5.035007,
        changed => {
            'B::Deparse'            => '1.61',
            'B::Op_private'         => '5.035008',
            'Config'                => '5.035008',
            'Data::Dumper'          => '2.184',
            'Errno'                 => '1.36',
            'File::Fetch'           => '1.04',
            'File::Find'            => '1.40',
            'Hash::Util::FieldHash' => '1.25',
            'Locale::Maketext'      => '1.30',
            'Math::BigFloat'        => '1.999829',
            'Math::BigFloat::Trace' => '0.64',
            'Math::BigInt'          => '1.999829',
            'Math::BigInt::Calc'    => '1.999829',
            'Math::BigInt::Lib'     => '1.999829',
            'Math::BigInt::Trace'   => '0.64',
            'Math::BigRat::Trace'   => '0.64',
            'Module::CoreList'      => '5.20220120',
            'Module::CoreList::Utils'=> '5.20220120',
            'NEXT'                  => '0.69',
            'POSIX'                 => '2.03',
            'Win32'                 => '0.58',
            '_charnames'            => '1.49',
            'bigint'                => '0.64',
            'bignum'                => '0.64',
            'bigrat'                => '0.64',
            'charnames'             => '1.49',
        },
        removed => {
        }
    },
    5.035009 => {
        delta_from => 5.035008,
        changed => {
            'App::Cpan'             => '1.678',
            'B::Deparse'            => '1.62',
            'B::Op_private'         => '5.035009',
            'CPAN'                  => '2.33',
            'CPAN::Distribution'    => '2.33',
            'CPAN::FTP'             => '5.5016',
            'CPAN::FirstTime'       => '5.5317',
            'Config'                => '5.035009',
            'Devel::PPPort'         => '3.64',
            'File::Copy'            => '2.39',
            'Hash::Util'            => '0.28',
            'Hash::Util::FieldHash' => '1.26',
            'List::Util'            => '1.61',
            'List::Util::XS'        => '1.61',
            'Module::CoreList'      => '5.20220220',
            'Module::CoreList::Utils'=> '5.20220220',
            'Opcode'                => '1.56',
            'Scalar::Util'          => '1.61',
            'Sub::Util'             => '1.61',
            'Tie::SubstrHash'       => '1.01',
            'XS::APItest'           => '1.21',
            '_charnames'            => '1.50',
            'builtin'               => '0.002',
            'charnames'             => '1.50',
            'experimental'          => '0.027',
            'feature'               => '1.70',
            'overload'              => '1.35',
            're'                    => '0.42',
            'sigtrap'               => '1.10',
            'warnings'              => '1.57',
        },
        removed => {
        }
    },
    5.034001 => {
        delta_from => 5.034000,
        changed => {
            'B::Deparse'            => '1.57',
            'B::Op_private'         => '5.034001',
            'Config'                => '5.034001',
            'Encode'                => '3.08_01',
            'GDBM_File'             => '1.19_01',
            'Module::CoreList'      => '5.20220313',
            'Module::CoreList::Utils'=> '5.20220313',
        },
        removed => {
        }
    },
    5.035010 => {
        delta_from => 5.035009,
        changed => {
            'Attribute::Handlers'   => '1.02',
            'B::Deparse'            => '1.63',
            'B::Op_private'         => '5.035010',
            'Config'                => '5.03501',
            'Cwd'                   => '3.84',
            'DB_File'               => '1.857',
            'Devel::PPPort'         => '3.68',
            'ExtUtils::ParseXS'     => '3.45',
            'ExtUtils::ParseXS::Constants'=> '3.45',
            'ExtUtils::ParseXS::CountLines'=> '3.45',
            'ExtUtils::ParseXS::Eval'=> '3.45',
            'ExtUtils::ParseXS::Utilities'=> '3.45',
            'ExtUtils::Typemaps'    => '3.45',
            'ExtUtils::Typemaps::Cmd'=> '3.45',
            'ExtUtils::Typemaps::InputMap'=> '3.45',
            'ExtUtils::Typemaps::OutputMap'=> '3.45',
            'ExtUtils::Typemaps::Type'=> '3.45',
            'File::Spec'            => '3.84',
            'File::Spec::AmigaOS'   => '3.84',
            'File::Spec::Cygwin'    => '3.84',
            'File::Spec::Epoc'      => '3.84',
            'File::Spec::Functions' => '3.84',
            'File::Spec::Mac'       => '3.84',
            'File::Spec::OS2'       => '3.84',
            'File::Spec::Unix'      => '3.84',
            'File::Spec::VMS'       => '3.84',
            'File::Spec::Win32'     => '3.84',
            'GDBM_File'             => '1.23',
            'List::Util'            => '1.62',
            'List::Util::XS'        => '1.62',
            'Module::CoreList'      => '5.20220320',
            'Module::CoreList::Utils'=> '5.20220320',
            'Opcode'                => '1.57',
            'Scalar::Util'          => '1.62',
            'Sub::Util'             => '1.62',
            'Test2'                 => '1.302190',
            'Test2::API'            => '1.302190',
            'Test2::API::Breakage'  => '1.302190',
            'Test2::API::Context'   => '1.302190',
            'Test2::API::Instance'  => '1.302190',
            'Test2::API::InterceptResult'=> '1.302190',
            'Test2::API::InterceptResult::Event'=> '1.302190',
            'Test2::API::InterceptResult::Facet'=> '1.302190',
            'Test2::API::InterceptResult::Hub'=> '1.302190',
            'Test2::API::InterceptResult::Squasher'=> '1.302190',
            'Test2::API::Stack'     => '1.302190',
            'Test2::Event'          => '1.302190',
            'Test2::Event::Bail'    => '1.302190',
            'Test2::Event::Diag'    => '1.302190',
            'Test2::Event::Encoding'=> '1.302190',
            'Test2::Event::Exception'=> '1.302190',
            'Test2::Event::Fail'    => '1.302190',
            'Test2::Event::Generic' => '1.302190',
            'Test2::Event::Note'    => '1.302190',
            'Test2::Event::Ok'      => '1.302190',
            'Test2::Event::Pass'    => '1.302190',
            'Test2::Event::Plan'    => '1.302190',
            'Test2::Event::Skip'    => '1.302190',
            'Test2::Event::Subtest' => '1.302190',
            'Test2::Event::TAP::Version'=> '1.302190',
            'Test2::Event::V2'      => '1.302190',
            'Test2::Event::Waiting' => '1.302190',
            'Test2::EventFacet'     => '1.302190',
            'Test2::EventFacet::About'=> '1.302190',
            'Test2::EventFacet::Amnesty'=> '1.302190',
            'Test2::EventFacet::Assert'=> '1.302190',
            'Test2::EventFacet::Control'=> '1.302190',
            'Test2::EventFacet::Error'=> '1.302190',
            'Test2::EventFacet::Hub'=> '1.302190',
            'Test2::EventFacet::Info'=> '1.302190',
            'Test2::EventFacet::Info::Table'=> '1.302190',
            'Test2::EventFacet::Meta'=> '1.302190',
            'Test2::EventFacet::Parent'=> '1.302190',
            'Test2::EventFacet::Plan'=> '1.302190',
            'Test2::EventFacet::Render'=> '1.302190',
            'Test2::EventFacet::Trace'=> '1.302190',
            'Test2::Formatter'      => '1.302190',
            'Test2::Formatter::TAP' => '1.302190',
            'Test2::Hub'            => '1.302190',
            'Test2::Hub::Interceptor'=> '1.302190',
            'Test2::Hub::Interceptor::Terminator'=> '1.302190',
            'Test2::Hub::Subtest'   => '1.302190',
            'Test2::IPC'            => '1.302190',
            'Test2::IPC::Driver'    => '1.302190',
            'Test2::IPC::Driver::Files'=> '1.302190',
            'Test2::Tools::Tiny'    => '1.302190',
            'Test2::Util'           => '1.302190',
            'Test2::Util::ExternalMeta'=> '1.302190',
            'Test2::Util::Facets2Legacy'=> '1.302190',
            'Test2::Util::HashBase' => '1.302190',
            'Test2::Util::Trace'    => '1.302190',
            'Test::Builder'         => '1.302190',
            'Test::Builder::Formatter'=> '1.302190',
            'Test::Builder::Module' => '1.302190',
            'Test::Builder::Tester' => '1.302190',
            'Test::Builder::Tester::Color'=> '1.302190',
            'Test::Builder::TodoDiag'=> '1.302190',
            'Test::More'            => '1.302190',
            'Test::Simple'          => '1.302190',
            'Test::Tester'          => '1.302190',
            'Test::Tester::Capture' => '1.302190',
            'Test::Tester::CaptureRunner'=> '1.302190',
            'Test::Tester::Delegate'=> '1.302190',
            'Test::use::ok'         => '1.302190',
            'XS::APItest'           => '1.22',
            'builtin'               => '0.004',
            'experimental'          => '0.028',
            'feature'               => '1.71',
            'ok'                    => '1.302190',
            'warnings'              => '1.58',
        },
        removed => {
        }
    },
    5.035011 => {
        delta_from => 5.03501,
        changed => {
            'App::Prove'            => '3.44',
            'App::Prove::State'     => '3.44',
            'App::Prove::State::Result'=> '3.44',
            'App::Prove::State::Result::Test'=> '3.44',
            'B::Deparse'            => '1.64',
            'B::Op_private'         => '5.035011',
            'Compress::Raw::Bzip2'  => '2.103',
            'Compress::Raw::Zlib'   => '2.103',
            'Compress::Zlib'        => '2.106',
            'Config'                => '5.035011',
            'Encode'                => '3.17',
            'Encode::Unicode'       => '2.20',
            'ExtUtils::Constant::Base'=> '0.07',
            'IO'                    => '1.49_01',
            'IO::Compress::Adapter::Bzip2'=> '2.106',
            'IO::Compress::Adapter::Deflate'=> '2.106',
            'IO::Compress::Adapter::Identity'=> '2.106',
            'IO::Compress::Base'    => '2.106',
            'IO::Compress::Base::Common'=> '2.106',
            'IO::Compress::Bzip2'   => '2.106',
            'IO::Compress::Deflate' => '2.106',
            'IO::Compress::Gzip'    => '2.106',
            'IO::Compress::Gzip::Constants'=> '2.106',
            'IO::Compress::RawDeflate'=> '2.106',
            'IO::Compress::Zip'     => '2.106',
            'IO::Compress::Zip::Constants'=> '2.106',
            'IO::Compress::Zlib::Constants'=> '2.106',
            'IO::Compress::Zlib::Extra'=> '2.106',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.106',
            'IO::Uncompress::Adapter::Identity'=> '2.106',
            'IO::Uncompress::Adapter::Inflate'=> '2.106',
            'IO::Uncompress::AnyInflate'=> '2.106',
            'IO::Uncompress::AnyUncompress'=> '2.106',
            'IO::Uncompress::Base'  => '2.106',
            'IO::Uncompress::Bunzip2'=> '2.106',
            'IO::Uncompress::Gunzip'=> '2.106',
            'IO::Uncompress::Inflate'=> '2.106',
            'IO::Uncompress::RawInflate'=> '2.106',
            'IO::Uncompress::Unzip' => '2.106',
            'Locale::Maketext'      => '1.31',
            'Math::BigFloat'        => '1.999830',
            'Math::BigFloat::Trace' => '0.65',
            'Math::BigInt'          => '1.999830',
            'Math::BigInt::Calc'    => '1.999830',
            'Math::BigInt::Lib'     => '1.999830',
            'Math::BigInt::Trace'   => '0.65',
            'Math::BigRat'          => '0.2621',
            'Math::BigRat::Trace'   => '0.65',
            'Module::CoreList'      => '5.20220420',
            'Module::CoreList::Utils'=> '5.20220420',
            'Net::Cmd'              => '3.14',
            'Net::Config'           => '3.14',
            'Net::Domain'           => '3.14',
            'Net::FTP'              => '3.14',
            'Net::FTP::A'           => '3.14',
            'Net::FTP::E'           => '3.14',
            'Net::FTP::I'           => '3.14',
            'Net::FTP::L'           => '3.14',
            'Net::FTP::dataconn'    => '3.14',
            'Net::NNTP'             => '3.14',
            'Net::Netrc'            => '3.14',
            'Net::POP3'             => '3.14',
            'Net::SMTP'             => '3.14',
            'Net::Time'             => '3.14',
            'Socket'                => '2.033',
            'Storable'              => '3.26',
            'TAP::Base'             => '3.44',
            'TAP::Formatter::Base'  => '3.44',
            'TAP::Formatter::Color' => '3.44',
            'TAP::Formatter::Console'=> '3.44',
            'TAP::Formatter::Console::ParallelSession'=> '3.44',
            'TAP::Formatter::Console::Session'=> '3.44',
            'TAP::Formatter::File'  => '3.44',
            'TAP::Formatter::File::Session'=> '3.44',
            'TAP::Formatter::Session'=> '3.44',
            'TAP::Harness'          => '3.44',
            'TAP::Harness::Env'     => '3.44',
            'TAP::Object'           => '3.44',
            'TAP::Parser'           => '3.44',
            'TAP::Parser::Aggregator'=> '3.44',
            'TAP::Parser::Grammar'  => '3.44',
            'TAP::Parser::Iterator' => '3.44',
            'TAP::Parser::Iterator::Array'=> '3.44',
            'TAP::Parser::Iterator::Process'=> '3.44',
            'TAP::Parser::Iterator::Stream'=> '3.44',
            'TAP::Parser::IteratorFactory'=> '3.44',
            'TAP::Parser::Multiplexer'=> '3.44',
            'TAP::Parser::Result'   => '3.44',
            'TAP::Parser::Result::Bailout'=> '3.44',
            'TAP::Parser::Result::Comment'=> '3.44',
            'TAP::Parser::Result::Plan'=> '3.44',
            'TAP::Parser::Result::Pragma'=> '3.44',
            'TAP::Parser::Result::Test'=> '3.44',
            'TAP::Parser::Result::Unknown'=> '3.44',
            'TAP::Parser::Result::Version'=> '3.44',
            'TAP::Parser::Result::YAML'=> '3.44',
            'TAP::Parser::ResultFactory'=> '3.44',
            'TAP::Parser::Scheduler'=> '3.44',
            'TAP::Parser::Scheduler::Job'=> '3.44',
            'TAP::Parser::Scheduler::Spinner'=> '3.44',
            'TAP::Parser::Source'   => '3.44',
            'TAP::Parser::SourceHandler'=> '3.44',
            'TAP::Parser::SourceHandler::Executable'=> '3.44',
            'TAP::Parser::SourceHandler::File'=> '3.44',
            'TAP::Parser::SourceHandler::Handle'=> '3.44',
            'TAP::Parser::SourceHandler::Perl'=> '3.44',
            'TAP::Parser::SourceHandler::RawTAP'=> '3.44',
            'TAP::Parser::YAMLish::Reader'=> '3.44',
            'TAP::Parser::YAMLish::Writer'=> '3.44',
            'Test::Harness'         => '3.44',
            'Text::ParseWords'      => '3.31',
            'Time::HiRes'           => '1.9770',
            'Unicode::Normalize'    => '1.31',
            'bigfloat'              => '0.65',
            'bigint'                => '0.65',
            'bignum'                => '0.65',
            'bigrat'                => '0.65',
            'builtin'               => '0.005',
            're'                    => '0.43',
        },
        removed => {
        }
    },
    5.036000 => {
        delta_from => 5.035011,
        changed => {
            'Amiga::Exec'           => '0.04',
            'B::Op_private'         => '5.036000',
            'Compress::Raw::Zlib'   => '2.105',
            'Config'                => '5.036',
            'IO'                    => '1.50',
            'Module::CoreList'      => '5.20220520',
            'Module::CoreList::Utils'=> '5.20220520',
            'Win32'                 => '0.59',
            'builtin'               => '0.006',
            'feature'               => '1.72',
        },
        removed => {
        }
    },
    5.037000 => {
        delta_from => 5.036000,
        changed => {
            'feature'               => '1.73',
            'Module::CoreList'      => '5.20220527',
            'Module::CoreList::Utils'=> '5.20220527',
        },
        removed => {
        }
    },
    5.037001 => {
        delta_from => 5.037000,
        changed => {
            'B'                     => '1.84',
            'B::Op_private'         => '5.037001',
            'Carp'                  => '1.53',
            'Carp::Heavy'           => '1.53',
            'Config'                => '5.037001',
            'Cwd'                   => '3.85',
            'Data::Dumper'          => '2.185',
            'ExtUtils::CBuilder'    => '0.280237',
            'ExtUtils::CBuilder::Base'=> '0.280237',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.280237',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.280237',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.280237',
            'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280237',
            'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280237',
            'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280237',
            'ExtUtils::CBuilder::Platform::aix'=> '0.280237',
            'ExtUtils::CBuilder::Platform::android'=> '0.280237',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.280237',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.280237',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280237',
            'ExtUtils::CBuilder::Platform::os2'=> '0.280237',
            'ExtUtils::Miniperl'    => '1.12',
            'File::Spec'            => '3.85',
            'File::Spec::AmigaOS'   => '3.85',
            'File::Spec::Cygwin'    => '3.85',
            'File::Spec::Epoc'      => '3.85',
            'File::Spec::Functions' => '3.85',
            'File::Spec::Mac'       => '3.85',
            'File::Spec::OS2'       => '3.85',
            'File::Spec::Unix'      => '3.85',
            'File::Spec::VMS'       => '3.85',
            'File::Spec::Win32'     => '3.85',
            'FileHandle'            => '2.04',
            'GDBM_File'             => '1.24',
            'IO::Handle'            => '1.49',
            'IO::Pipe'              => '1.50',
            'IO::Socket::INET'      => '1.50',
            'IO::Socket::UNIX'      => '1.50',
            'Module::CoreList'      => '5.20220620',
            'Module::CoreList::Utils'=> '5.20220620',
            'ODBM_File'             => '1.18',
            'OS2::REXX'             => '1.06',
            'Opcode'                => '1.58',
            'POSIX'                 => '2.04',
            'SDBM_File'             => '1.16',
            'Unicode::Normalize'    => '1.32',
            'XS::APItest'           => '1.23',
            'builtin'               => '0.007',
            'encoding::warnings'    => '0.14',
            'feature'               => '1.74',
            'threads'               => '2.28',
        },
        removed => {
        }
    },
    5.037002 => {
        delta_from => 5.037001,
        changed => {
            'B'                     => '1.85',
            'B::Concise'            => '1.007',
            'B::Deparse'            => '1.65',
            'B::Op_private'         => '5.037002',
            'CPAN'                  => '2.34',
            'CPAN::Distribution'    => '2.34',
            'Compress::Raw::Bzip2'  => '2.201',
            'Compress::Raw::Zlib'   => '2.202',
            'Compress::Zlib'        => '2.201',
            'Config'                => '5.037002',
            'Cwd'                   => '3.86',
            'DB_File'               => '1.858',
            'Data::Dumper'          => '2.186',
            'Encode'                => '3.18',
            'Encode::MIME::Header'  => '2.29',
            'File::Glob'            => '1.38',
            'File::Spec'            => '3.86',
            'File::Spec::AmigaOS'   => '3.86',
            'File::Spec::Cygwin'    => '3.86',
            'File::Spec::Epoc'      => '3.86',
            'File::Spec::Functions' => '3.86',
            'File::Spec::Mac'       => '3.86',
            'File::Spec::OS2'       => '3.86',
            'File::Spec::Unix'      => '3.86',
            'File::Spec::VMS'       => '3.86',
            'File::Spec::Win32'     => '3.86',
            'Filter::Util::Call'    => '1.61',
            'Hash::Util'            => '0.29',
            'I18N::LangTags::List'  => '0.41',
            'IO::Compress::Adapter::Bzip2'=> '2.201',
            'IO::Compress::Adapter::Deflate'=> '2.201',
            'IO::Compress::Adapter::Identity'=> '2.201',
            'IO::Compress::Base'    => '2.201',
            'IO::Compress::Base::Common'=> '2.201',
            'IO::Compress::Bzip2'   => '2.201',
            'IO::Compress::Deflate' => '2.201',
            'IO::Compress::Gzip'    => '2.201',
            'IO::Compress::Gzip::Constants'=> '2.201',
            'IO::Compress::RawDeflate'=> '2.201',
            'IO::Compress::Zip'     => '2.201',
            'IO::Compress::Zip::Constants'=> '2.201',
            'IO::Compress::Zlib::Constants'=> '2.201',
            'IO::Compress::Zlib::Extra'=> '2.201',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.201',
            'IO::Uncompress::Adapter::Identity'=> '2.201',
            'IO::Uncompress::Adapter::Inflate'=> '2.201',
            'IO::Uncompress::AnyInflate'=> '2.201',
            'IO::Uncompress::AnyUncompress'=> '2.201',
            'IO::Uncompress::Base'  => '2.201',
            'IO::Uncompress::Bunzip2'=> '2.201',
            'IO::Uncompress::Gunzip'=> '2.201',
            'IO::Uncompress::Inflate'=> '2.201',
            'IO::Uncompress::RawInflate'=> '2.201',
            'IO::Uncompress::Unzip' => '2.201',
            'JSON::PP'              => '4.10',
            'JSON::PP::Boolean'     => '4.10',
            'Math::BigFloat'        => '1.999837',
            'Math::BigFloat::Trace' => '0.66',
            'Math::BigInt'          => '1.999837',
            'Math::BigInt::Calc'    => '1.999837',
            'Math::BigInt::FastCalc'=> '0.5013',
            'Math::BigInt::Lib'     => '1.999837',
            'Math::BigInt::Trace'   => '0.66',
            'Math::BigRat'          => '0.2624',
            'Math::BigRat::Trace'   => '0.66',
            'Module::CoreList'      => '5.20220720',
            'Module::CoreList::Utils'=> '5.20220720',
            'Opcode'                => '1.59',
            'PerlIO::via::QuotedPrint'=> '0.10',
            'Pod::Checker'          => '1.75',
            'Pod::Usage'            => '2.03',
            'Socket'                => '2.035',
            'Storable'              => '3.27',
            'Test2'                 => '1.302191',
            'Test2::API'            => '1.302191',
            'Test2::API::Breakage'  => '1.302191',
            'Test2::API::Context'   => '1.302191',
            'Test2::API::Instance'  => '1.302191',
            'Test2::API::InterceptResult'=> '1.302191',
            'Test2::API::InterceptResult::Event'=> '1.302191',
            'Test2::API::InterceptResult::Facet'=> '1.302191',
            'Test2::API::InterceptResult::Hub'=> '1.302191',
            'Test2::API::InterceptResult::Squasher'=> '1.302191',
            'Test2::API::Stack'     => '1.302191',
            'Test2::Event'          => '1.302191',
            'Test2::Event::Bail'    => '1.302191',
            'Test2::Event::Diag'    => '1.302191',
            'Test2::Event::Encoding'=> '1.302191',
            'Test2::Event::Exception'=> '1.302191',
            'Test2::Event::Fail'    => '1.302191',
            'Test2::Event::Generic' => '1.302191',
            'Test2::Event::Note'    => '1.302191',
            'Test2::Event::Ok'      => '1.302191',
            'Test2::Event::Pass'    => '1.302191',
            'Test2::Event::Plan'    => '1.302191',
            'Test2::Event::Skip'    => '1.302191',
            'Test2::Event::Subtest' => '1.302191',
            'Test2::Event::TAP::Version'=> '1.302191',
            'Test2::Event::V2'      => '1.302191',
            'Test2::Event::Waiting' => '1.302191',
            'Test2::EventFacet'     => '1.302191',
            'Test2::EventFacet::About'=> '1.302191',
            'Test2::EventFacet::Amnesty'=> '1.302191',
            'Test2::EventFacet::Assert'=> '1.302191',
            'Test2::EventFacet::Control'=> '1.302191',
            'Test2::EventFacet::Error'=> '1.302191',
            'Test2::EventFacet::Hub'=> '1.302191',
            'Test2::EventFacet::Info'=> '1.302191',
            'Test2::EventFacet::Info::Table'=> '1.302191',
            'Test2::EventFacet::Meta'=> '1.302191',
            'Test2::EventFacet::Parent'=> '1.302191',
            'Test2::EventFacet::Plan'=> '1.302191',
            'Test2::EventFacet::Render'=> '1.302191',
            'Test2::EventFacet::Trace'=> '1.302191',
            'Test2::Formatter'      => '1.302191',
            'Test2::Formatter::TAP' => '1.302191',
            'Test2::Hub'            => '1.302191',
            'Test2::Hub::Interceptor'=> '1.302191',
            'Test2::Hub::Interceptor::Terminator'=> '1.302191',
            'Test2::Hub::Subtest'   => '1.302191',
            'Test2::IPC'            => '1.302191',
            'Test2::IPC::Driver'    => '1.302191',
            'Test2::IPC::Driver::Files'=> '1.302191',
            'Test2::Tools::Tiny'    => '1.302191',
            'Test2::Util'           => '1.302191',
            'Test2::Util::ExternalMeta'=> '1.302191',
            'Test2::Util::Facets2Legacy'=> '1.302191',
            'Test2::Util::HashBase' => '1.302191',
            'Test2::Util::Trace'    => '1.302191',
            'Test::Builder'         => '1.302191',
            'Test::Builder::Formatter'=> '1.302191',
            'Test::Builder::Module' => '1.302191',
            'Test::Builder::Tester' => '1.302191',
            'Test::Builder::Tester::Color'=> '1.302191',
            'Test::Builder::TodoDiag'=> '1.302191',
            'Test::More'            => '1.302191',
            'Test::Simple'          => '1.302191',
            'Test::Tester'          => '1.302191',
            'Test::Tester::Capture' => '1.302191',
            'Test::Tester::CaptureRunner'=> '1.302191',
            'Test::Tester::Delegate'=> '1.302191',
            'Test::use::ok'         => '1.302191',
            'Text::Balanced'        => '2.06',
            'XS::APItest'           => '1.24',
            'bigfloat'              => '0.66',
            'bigint'                => '0.66',
            'bignum'                => '0.66',
            'bigrat'                => '0.66',
            'builtin'               => '0.008',
            'feature'               => '1.75',
            'ok'                    => '1.302191',
            'threads::shared'       => '1.65',
        },
        removed => {
        }
    },
    5.037003 => {
        delta_from => 5.037002,
        changed => {
            'B'                     => '1.86',
            'B::Deparse'            => '1.68',
            'B::Op_private'         => '5.037003',
            'Config'                => '5.037003',
            'Digest::SHA'           => '6.03',
            'DynaLoader'            => '1.53',
            'Encode'                => '3.19',
            'Encode::Alias'         => '2.25',
            'ExtUtils::PL2Bat'      => '0.005',
            'File::Find'            => '1.41',
            'Filter::Util::Call'    => '1.64',
            'HTTP::Tiny'            => '0.082',
            'JSON::PP'              => '4.11',
            'JSON::PP::Boolean'     => '4.11',
            'List::Util'            => '1.63',
            'List::Util::XS'        => '1.63',
            'Memoize'               => '1.10',
            'Memoize::AnyDBM_File'  => '1.10',
            'Memoize::Expire'       => '1.10',
            'Memoize::NDBM_File'    => '1.10',
            'Memoize::SDBM_File'    => '1.10',
            'Memoize::Storable'     => '1.10',
            'Module::CoreList'      => '5.20220820',
            'Module::CoreList::Utils'=> '5.20220820',
            'NDBM_File'             => '1.16',
            'Opcode'                => '1.60',
            'Scalar::Util'          => '1.63',
            'Socket'                => '2.036',
            'Sub::Util'             => '1.63',
            'XS::APItest'           => '1.25',
            'attributes'            => '0.35',
            'threads'               => '2.29',
        },
        removed => {
            'Memoize::ExpireFile'   => 1,
            'Memoize::ExpireTest'   => 1,
        }
    },
    5.037004 => {
        delta_from => 5.037003,
        changed => {
            'B::Deparse'            => '1.69',
            'B::Op_private'         => '5.037004',
            'Carp'                  => '1.54',
            'Carp::Heavy'           => '1.54',
            'Class::Struct'         => '0.67',
            'Config'                => '5.037004',
            'Config::Perl::V'       => '0.34',
            'Errno'                 => '1.37',
            'ExtUtils::ParseXS'     => '3.46',
            'ExtUtils::ParseXS::Constants'=> '3.46',
            'ExtUtils::ParseXS::CountLines'=> '3.46',
            'ExtUtils::ParseXS::Eval'=> '3.46',
            'ExtUtils::ParseXS::Utilities'=> '3.46',
            'ExtUtils::Typemaps'    => '3.46',
            'ExtUtils::Typemaps::Cmd'=> '3.46',
            'ExtUtils::Typemaps::InputMap'=> '3.46',
            'ExtUtils::Typemaps::OutputMap'=> '3.46',
            'ExtUtils::Typemaps::Type'=> '3.46',
            'File::Basename'        => '2.86',
            'File::Copy'            => '2.40',
            'File::Spec'            => '3.87',
            'File::stat'            => '1.13',
            'FileHandle'            => '2.05',
            'Hash::Util'            => '0.30',
            'I18N::Langinfo'        => '0.22',
            'IO'                    => '1.51',
            'IO::Dir'               => '1.51',
            'IO::File'              => '1.51',
            'IO::Handle'            => '1.51',
            'IO::Pipe'              => '1.51',
            'IO::Poll'              => '1.51',
            'IO::Seekable'          => '1.51',
            'IO::Select'            => '1.51',
            'IO::Socket'            => '1.51',
            'IO::Socket::INET'      => '1.51',
            'IO::Socket::UNIX'      => '1.51',
            'Locale::Maketext'      => '1.32',
            'Module::CoreList'      => '5.20220920',
            'Module::CoreList::Utils'=> '5.20220920',
            'Net::protoent'         => '1.02',
            'Net::servent'          => '1.03',
            'Opcode'                => '1.61',
            'POSIX'                 => '2.06',
            'Safe'                  => '2.44',
            'Sys::Hostname'         => '1.25',
            'Time::HiRes'           => '1.9771',
            'User::grent'           => '1.04',
            'User::pwent'           => '1.02',
            'XS::APItest'           => '1.26',
            'XSLoader'              => '0.32',
            'feature'               => '1.76',
        },
        removed => {
        }
    },
    5.037005 => {
        delta_from => 5.037004,
        changed => {
            'B::Deparse'            => '1.70',
            'B::Op_private'         => '5.037005',
            'Config'                => '5.037005',
            'JSON::PP'              => '4.12',
            'JSON::PP::Boolean'     => '4.12',
            'Math::Complex'         => '1.5903',
            'Math::Trig'            => '1.2301',
            'Memoize'               => '1.14',
            'Memoize::AnyDBM_File'  => '1.14',
            'Memoize::Expire'       => '1.14',
            'Memoize::NDBM_File'    => '1.14',
            'Memoize::SDBM_File'    => '1.14',
            'Memoize::Storable'     => '1.14',
            'Module::CoreList'      => '5.20221020',
            'Module::CoreList::Utils'=> '5.20221020',
            'Net::Ping'             => '2.75',
            'POSIX'                 => '2.07',
            'Unicode'               => '15.0.0',
            'threads'               => '2.31',
            'warnings'              => '1.59',
        },
        removed => {
        }
    },
    5.037006 => {
        delta_from => 5.037005,
        changed => {
            'Attribute::Handlers'   => '1.03',
            'B'                     => '1.87',
            'B::Deparse'            => '1.71',
            'B::Op_private'         => '5.037006',
            'Config'                => '5.037006',
            'Data::Dumper'          => '2.187',
            'Devel::PPPort'         => '3.69',
            'ExtUtils::CBuilder'    => '0.280238',
            'ExtUtils::CBuilder::Base'=> '0.280238',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.280238',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.280238',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.280238',
            'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280238',
            'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280238',
            'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280238',
            'ExtUtils::CBuilder::Platform::aix'=> '0.280238',
            'ExtUtils::CBuilder::Platform::android'=> '0.280238',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.280238',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.280238',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280238',
            'ExtUtils::CBuilder::Platform::os2'=> '0.280238',
            'ExtUtils::ParseXS'     => '3.48',
            'ExtUtils::ParseXS::Constants'=> '3.48',
            'ExtUtils::ParseXS::CountLines'=> '3.48',
            'ExtUtils::ParseXS::Eval'=> '3.48',
            'ExtUtils::ParseXS::Utilities'=> '3.48',
            'ExtUtils::Typemaps'    => '3.48',
            'ExtUtils::Typemaps::Cmd'=> '3.48',
            'ExtUtils::Typemaps::InputMap'=> '3.48',
            'ExtUtils::Typemaps::OutputMap'=> '3.48',
            'ExtUtils::Typemaps::Type'=> '3.48',
            'Getopt::Long'          => '2.54',
            'Memoize'               => '1.15',
            'Memoize::AnyDBM_File'  => '1.15',
            'Memoize::Expire'       => '1.15',
            'Memoize::NDBM_File'    => '1.15',
            'Memoize::SDBM_File'    => '1.15',
            'Memoize::Storable'     => '1.15',
            'Module::CoreList'      => '5.20221120',
            'Module::CoreList::Utils'=> '5.20221120',
            'Opcode'                => '1.62',
            'POSIX'                 => '2.08',
            'Storable'              => '3.28',
            'Time::HiRes'           => '1.9772',
            'XS::APItest'           => '1.27',
            'experimental'          => '0.029',
            'feature'               => '1.77',
            'overload'              => '1.36',
            'threads'               => '2.32',
            'utf8'                  => '1.25',
            'warnings'              => '1.61',
        },
        removed => {
        }
    },
    5.037007 => {
        delta_from => 5.037006,
        changed => {
            'B::Deparse'            => '1.72',
            'B::Op_private'         => '5.037007',
            'Config'                => '5.037007',
            'Cwd'                   => '3.88',
            'ExtUtils::Miniperl'    => '1.13',
            'ExtUtils::ParseXS'     => '3.49',
            'ExtUtils::ParseXS::Constants'=> '3.49',
            'ExtUtils::ParseXS::CountLines'=> '3.49',
            'ExtUtils::ParseXS::Eval'=> '3.49',
            'ExtUtils::ParseXS::Utilities'=> '3.49',
            'ExtUtils::Typemaps'    => '3.49',
            'ExtUtils::Typemaps::Cmd'=> '3.49',
            'ExtUtils::Typemaps::InputMap'=> '3.49',
            'ExtUtils::Typemaps::OutputMap'=> '3.49',
            'ExtUtils::Typemaps::Type'=> '3.49',
            'File::Glob'            => '1.39',
            'File::Spec'            => '3.88',
            'File::Spec::AmigaOS'   => '3.88',
            'File::Spec::Cygwin'    => '3.88',
            'File::Spec::Epoc'      => '3.88',
            'File::Spec::Functions' => '3.88',
            'File::Spec::Mac'       => '3.88',
            'File::Spec::OS2'       => '3.88',
            'File::Spec::Unix'      => '3.88',
            'File::Spec::VMS'       => '3.88',
            'File::Spec::Win32'     => '3.88',
            'Module::CoreList'      => '5.20221220',
            'Module::CoreList::Utils'=> '5.20221220',
            'Opcode'                => '1.63',
            'POSIX'                 => '2.10',
            'Pod::Html'             => '1.34',
            'Pod::Html::Util'       => '1.34',
            'Pod::Man'              => '5.00',
            'Pod::ParseLink'        => '5.00',
            'Pod::Text'             => '5.00',
            'Pod::Text::Color'      => '5.00',
            'Pod::Text::Overstrike' => '5.00',
            'Pod::Text::Termcap'    => '5.00',
            'XS::APItest'           => '1.28',
            'experimental'          => '0.030',
            'feature'               => '1.78',
            'parent'                => '0.239',
            'threads'               => '2.33',
        },
        removed => {
        }
    },
    5.037008 => {
        delta_from => 5.037007,
        changed => {
            'B::Op_private'         => '5.037008',
            'Config'                => '5.037008',
            'Config::Perl::V'       => '0.35',
            'ExtUtils::Command'     => '7.66',
            'ExtUtils::Command::MM' => '7.66',
            'ExtUtils::Install'     => '2.22',
            'ExtUtils::Installed'   => '2.22',
            'ExtUtils::Liblist'     => '7.66',
            'ExtUtils::Liblist::Kid'=> '7.66',
            'ExtUtils::MM'          => '7.66',
            'ExtUtils::MM_AIX'      => '7.66',
            'ExtUtils::MM_Any'      => '7.66',
            'ExtUtils::MM_BeOS'     => '7.66',
            'ExtUtils::MM_Cygwin'   => '7.66',
            'ExtUtils::MM_DOS'      => '7.66',
            'ExtUtils::MM_Darwin'   => '7.66',
            'ExtUtils::MM_MacOS'    => '7.66',
            'ExtUtils::MM_NW5'      => '7.66',
            'ExtUtils::MM_OS2'      => '7.66',
            'ExtUtils::MM_OS390'    => '7.66',
            'ExtUtils::MM_QNX'      => '7.66',
            'ExtUtils::MM_UWIN'     => '7.66',
            'ExtUtils::MM_Unix'     => '7.66',
            'ExtUtils::MM_VMS'      => '7.66',
            'ExtUtils::MM_VOS'      => '7.66',
            'ExtUtils::MM_Win32'    => '7.66',
            'ExtUtils::MM_Win95'    => '7.66',
            'ExtUtils::MY'          => '7.66',
            'ExtUtils::MakeMaker'   => '7.66',
            'ExtUtils::MakeMaker::Config'=> '7.66',
            'ExtUtils::MakeMaker::Locale'=> '7.66',
            'ExtUtils::MakeMaker::version'=> '7.66',
            'ExtUtils::MakeMaker::version::regex'=> '7.66',
            'ExtUtils::Mkbootstrap' => '7.66',
            'ExtUtils::Mksymlists'  => '7.66',
            'ExtUtils::Packlist'    => '2.22',
            'ExtUtils::testlib'     => '7.66',
            'File::Find'            => '1.42',
            'IO::Zlib'              => '1.14',
            'JSON::PP'              => '4.16',
            'JSON::PP::Boolean'     => '4.16',
            'Math::Complex'         => '1.6',
            'Math::Trig'            => '1.6',
            'Module::CoreList'      => '5.20230120',
            'Module::CoreList::Utils'=> '5.20230120',
            'POSIX'                 => '2.11',
            'Pod::Man'              => '5.01',
            'Pod::ParseLink'        => '5.01',
            'Pod::Text'             => '5.01',
            'Pod::Text::Color'      => '5.01',
            'Pod::Text::Overstrike' => '5.01',
            'Pod::Text::Termcap'    => '5.01',
            'Storable'              => '3.29',
            'XS::APItest'           => '1.30',
            'feature'               => '1.79',
            're'                    => '0.44',
            'threads'               => '2.34',
        },
        removed => {
        }
    },
    5.037009 => {
        delta_from => 5.037008,
        changed => {
            'B'                     => '1.88',
            'B::Op_private'         => '5.037009',
            'Compress::Raw::Bzip2'  => '2.204',
            'Compress::Raw::Zlib'   => '2.204',
            'Compress::Zlib'        => '2.204',
            'Config'                => '5.037009',
            'Devel::PPPort'         => '3.70',
            'Devel::Peek'           => '1.33',
            'Fatal'                 => '2.36',
            'File::Find'            => '1.43',
            'IO::Compress::Adapter::Bzip2'=> '2.204',
            'IO::Compress::Adapter::Deflate'=> '2.204',
            'IO::Compress::Adapter::Identity'=> '2.204',
            'IO::Compress::Base'    => '2.204',
            'IO::Compress::Base::Common'=> '2.204',
            'IO::Compress::Bzip2'   => '2.204',
            'IO::Compress::Deflate' => '2.204',
            'IO::Compress::Gzip'    => '2.204',
            'IO::Compress::Gzip::Constants'=> '2.204',
            'IO::Compress::RawDeflate'=> '2.204',
            'IO::Compress::Zip'     => '2.204',
            'IO::Compress::Zip::Constants'=> '2.204',
            'IO::Compress::Zlib::Constants'=> '2.204',
            'IO::Compress::Zlib::Extra'=> '2.204',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.204',
            'IO::Uncompress::Adapter::Identity'=> '2.204',
            'IO::Uncompress::Adapter::Inflate'=> '2.204',
            'IO::Uncompress::AnyInflate'=> '2.204',
            'IO::Uncompress::AnyUncompress'=> '2.204',
            'IO::Uncompress::Base'  => '2.204',
            'IO::Uncompress::Bunzip2'=> '2.204',
            'IO::Uncompress::Gunzip'=> '2.204',
            'IO::Uncompress::Inflate'=> '2.204',
            'IO::Uncompress::RawInflate'=> '2.204',
            'IO::Uncompress::Unzip' => '2.204',
            'Math::Complex'         => '1.61',
            'Math::Trig'            => '1.61',
            'Memoize'               => '1.16',
            'Memoize::AnyDBM_File'  => '1.16',
            'Memoize::Expire'       => '1.16',
            'Memoize::NDBM_File'    => '1.16',
            'Memoize::SDBM_File'    => '1.16',
            'Memoize::Storable'     => '1.16',
            'Module::CoreList'      => '5.20230220',
            'Module::CoreList::Utils'=> '5.20230220',
            'Opcode'                => '1.64',
            'Term::Cap'             => '1.18',
            'Test2'                 => '1.302192',
            'Test2::API'            => '1.302192',
            'Test2::API::Breakage'  => '1.302192',
            'Test2::API::Context'   => '1.302192',
            'Test2::API::Instance'  => '1.302192',
            'Test2::API::InterceptResult'=> '1.302192',
            'Test2::API::InterceptResult::Event'=> '1.302192',
            'Test2::API::InterceptResult::Facet'=> '1.302192',
            'Test2::API::InterceptResult::Hub'=> '1.302192',
            'Test2::API::InterceptResult::Squasher'=> '1.302192',
            'Test2::API::Stack'     => '1.302192',
            'Test2::Event'          => '1.302192',
            'Test2::Event::Bail'    => '1.302192',
            'Test2::Event::Diag'    => '1.302192',
            'Test2::Event::Encoding'=> '1.302192',
            'Test2::Event::Exception'=> '1.302192',
            'Test2::Event::Fail'    => '1.302192',
            'Test2::Event::Generic' => '1.302192',
            'Test2::Event::Note'    => '1.302192',
            'Test2::Event::Ok'      => '1.302192',
            'Test2::Event::Pass'    => '1.302192',
            'Test2::Event::Plan'    => '1.302192',
            'Test2::Event::Skip'    => '1.302192',
            'Test2::Event::Subtest' => '1.302192',
            'Test2::Event::TAP::Version'=> '1.302192',
            'Test2::Event::V2'      => '1.302192',
            'Test2::Event::Waiting' => '1.302192',
            'Test2::EventFacet'     => '1.302192',
            'Test2::EventFacet::About'=> '1.302192',
            'Test2::EventFacet::Amnesty'=> '1.302192',
            'Test2::EventFacet::Assert'=> '1.302192',
            'Test2::EventFacet::Control'=> '1.302192',
            'Test2::EventFacet::Error'=> '1.302192',
            'Test2::EventFacet::Hub'=> '1.302192',
            'Test2::EventFacet::Info'=> '1.302192',
            'Test2::EventFacet::Info::Table'=> '1.302192',
            'Test2::EventFacet::Meta'=> '1.302192',
            'Test2::EventFacet::Parent'=> '1.302192',
            'Test2::EventFacet::Plan'=> '1.302192',
            'Test2::EventFacet::Render'=> '1.302192',
            'Test2::EventFacet::Trace'=> '1.302192',
            'Test2::Formatter'      => '1.302192',
            'Test2::Formatter::TAP' => '1.302192',
            'Test2::Hub'            => '1.302192',
            'Test2::Hub::Interceptor'=> '1.302192',
            'Test2::Hub::Interceptor::Terminator'=> '1.302192',
            'Test2::Hub::Subtest'   => '1.302192',
            'Test2::IPC'            => '1.302192',
            'Test2::IPC::Driver'    => '1.302192',
            'Test2::IPC::Driver::Files'=> '1.302192',
            'Test2::Tools::Tiny'    => '1.302192',
            'Test2::Util'           => '1.302192',
            'Test2::Util::ExternalMeta'=> '1.302192',
            'Test2::Util::Facets2Legacy'=> '1.302192',
            'Test2::Util::HashBase' => '1.302192',
            'Test2::Util::Trace'    => '1.302192',
            'Test::Builder'         => '1.302192',
            'Test::Builder::Formatter'=> '1.302192',
            'Test::Builder::Module' => '1.302192',
            'Test::Builder::Tester' => '1.302192',
            'Test::Builder::Tester::Color'=> '1.302192',
            'Test::Builder::TodoDiag'=> '1.302192',
            'Test::More'            => '1.302192',
            'Test::Simple'          => '1.302192',
            'Test::Tester'          => '1.302192',
            'Test::Tester::Capture' => '1.302192',
            'Test::Tester::CaptureRunner'=> '1.302192',
            'Test::Tester::Delegate'=> '1.302192',
            'Test::use::ok'         => '1.302192',
            'Tie::File'             => '1.07',
            'UNIVERSAL'             => '1.15',
            'autodie'               => '2.36',
            'autodie::Scope::Guard' => '2.36',
            'autodie::Scope::GuardStack'=> '2.36',
            'autodie::Util'         => '2.36',
            'autodie::exception'    => '2.36',
            'autodie::exception::system'=> '2.36',
            'autodie::hints'        => '2.36',
            'autodie::skip'         => '2.36',
            'experimental'          => '0.031',
            'feature'               => '1.80',
            'mro'                   => '1.28',
            'ok'                    => '1.302192',
            'parent'                => '0.241',
            'stable'                => '0.031',
            'warnings'              => '1.62',
        },
        removed => {
        }
    },
    5.037010 => {
        delta_from => 5.037009,
        changed => {
            'B::Op_private'         => '5.037010',
            'Benchmark'             => '1.24',
            'Class::Struct'         => '0.68',
            'Config'                => '5.03701',
            'Config::Perl::V'       => '0.36',
            'Cwd'                   => '3.89',
            'Data::Dumper'          => '2.188',
            'Digest::SHA'           => '6.04',
            'Env'                   => '1.06',
            'Math::Complex'         => '1.62',
            'Math::Trig'            => '1.62',
            'Module::CoreList'      => '5.20230320',
            'Module::CoreList::Utils'=> '5.20230320',
            'Net::Cmd'              => '3.15',
            'Net::Config'           => '3.15',
            'Net::Domain'           => '3.15',
            'Net::FTP'              => '3.15',
            'Net::FTP::A'           => '3.15',
            'Net::FTP::E'           => '3.15',
            'Net::FTP::I'           => '3.15',
            'Net::FTP::L'           => '3.15',
            'Net::FTP::dataconn'    => '3.15',
            'Net::NNTP'             => '3.15',
            'Net::Netrc'            => '3.15',
            'Net::POP3'             => '3.15',
            'Net::SMTP'             => '3.15',
            'Net::Time'             => '3.15',
            'POSIX'                 => '2.12',
            'Storable'              => '3.31',
            'Test2'                 => '1.302194',
            'Test2::API'            => '1.302194',
            'Test2::API::Breakage'  => '1.302194',
            'Test2::API::Context'   => '1.302194',
            'Test2::API::Instance'  => '1.302194',
            'Test2::API::InterceptResult'=> '1.302194',
            'Test2::API::InterceptResult::Event'=> '1.302194',
            'Test2::API::InterceptResult::Facet'=> '1.302194',
            'Test2::API::InterceptResult::Hub'=> '1.302194',
            'Test2::API::InterceptResult::Squasher'=> '1.302194',
            'Test2::API::Stack'     => '1.302194',
            'Test2::Event'          => '1.302194',
            'Test2::Event::Bail'    => '1.302194',
            'Test2::Event::Diag'    => '1.302194',
            'Test2::Event::Encoding'=> '1.302194',
            'Test2::Event::Exception'=> '1.302194',
            'Test2::Event::Fail'    => '1.302194',
            'Test2::Event::Generic' => '1.302194',
            'Test2::Event::Note'    => '1.302194',
            'Test2::Event::Ok'      => '1.302194',
            'Test2::Event::Pass'    => '1.302194',
            'Test2::Event::Plan'    => '1.302194',
            'Test2::Event::Skip'    => '1.302194',
            'Test2::Event::Subtest' => '1.302194',
            'Test2::Event::TAP::Version'=> '1.302194',
            'Test2::Event::V2'      => '1.302194',
            'Test2::Event::Waiting' => '1.302194',
            'Test2::EventFacet'     => '1.302194',
            'Test2::EventFacet::About'=> '1.302194',
            'Test2::EventFacet::Amnesty'=> '1.302194',
            'Test2::EventFacet::Assert'=> '1.302194',
            'Test2::EventFacet::Control'=> '1.302194',
            'Test2::EventFacet::Error'=> '1.302194',
            'Test2::EventFacet::Hub'=> '1.302194',
            'Test2::EventFacet::Info'=> '1.302194',
            'Test2::EventFacet::Info::Table'=> '1.302194',
            'Test2::EventFacet::Meta'=> '1.302194',
            'Test2::EventFacet::Parent'=> '1.302194',
            'Test2::EventFacet::Plan'=> '1.302194',
            'Test2::EventFacet::Render'=> '1.302194',
            'Test2::EventFacet::Trace'=> '1.302194',
            'Test2::Formatter'      => '1.302194',
            'Test2::Formatter::TAP' => '1.302194',
            'Test2::Hub'            => '1.302194',
            'Test2::Hub::Interceptor'=> '1.302194',
            'Test2::Hub::Interceptor::Terminator'=> '1.302194',
            'Test2::Hub::Subtest'   => '1.302194',
            'Test2::IPC'            => '1.302194',
            'Test2::IPC::Driver'    => '1.302194',
            'Test2::IPC::Driver::Files'=> '1.302194',
            'Test2::Tools::Tiny'    => '1.302194',
            'Test2::Util'           => '1.302194',
            'Test2::Util::ExternalMeta'=> '1.302194',
            'Test2::Util::Facets2Legacy'=> '1.302194',
            'Test2::Util::HashBase' => '1.302194',
            'Test2::Util::Trace'    => '1.302194',
            'Test::Builder'         => '1.302194',
            'Test::Builder::Formatter'=> '1.302194',
            'Test::Builder::Module' => '1.302194',
            'Test::Builder::Tester' => '1.302194',
            'Test::Builder::Tester::Color'=> '1.302194',
            'Test::Builder::TodoDiag'=> '1.302194',
            'Test::More'            => '1.302194',
            'Test::Simple'          => '1.302194',
            'Test::Tester'          => '1.302194',
            'Test::Tester::Capture' => '1.302194',
            'Test::Tester::CaptureRunner'=> '1.302194',
            'Test::Tester::Delegate'=> '1.302194',
            'Test::use::ok'         => '1.302194',
            'Time::HiRes'           => '1.9774',
            'XS::APItest'           => '1.32',
            'feature'               => '1.81',
            'ok'                    => '1.302194',
            'overload'              => '1.37',
            'threads'               => '2.35',
            'threads::shared'       => '1.67',
            'warnings'              => '1.63',
            'warnings::register'    => '1.05',
        },
        removed => {
        }
    },
    5.037011 => {
        delta_from => 5.037010,
        changed => {
            'B::Deparse'            => '1.73',
            'B::Op_private'         => '5.037011',
            'Config'                => '5.037011',
            'Devel::PPPort'         => '3.71',
            'ExtUtils::Command'     => '7.70',
            'ExtUtils::Command::MM' => '7.70',
            'ExtUtils::Liblist'     => '7.70',
            'ExtUtils::Liblist::Kid'=> '7.70',
            'ExtUtils::MM'          => '7.70',
            'ExtUtils::MM_AIX'      => '7.70',
            'ExtUtils::MM_Any'      => '7.70',
            'ExtUtils::MM_BeOS'     => '7.70',
            'ExtUtils::MM_Cygwin'   => '7.70',
            'ExtUtils::MM_DOS'      => '7.70',
            'ExtUtils::MM_Darwin'   => '7.70',
            'ExtUtils::MM_MacOS'    => '7.70',
            'ExtUtils::MM_NW5'      => '7.70',
            'ExtUtils::MM_OS2'      => '7.70',
            'ExtUtils::MM_OS390'    => '7.70',
            'ExtUtils::MM_QNX'      => '7.70',
            'ExtUtils::MM_UWIN'     => '7.70',
            'ExtUtils::MM_Unix'     => '7.70',
            'ExtUtils::MM_VMS'      => '7.70',
            'ExtUtils::MM_VOS'      => '7.70',
            'ExtUtils::MM_Win32'    => '7.70',
            'ExtUtils::MM_Win95'    => '7.70',
            'ExtUtils::MY'          => '7.70',
            'ExtUtils::MakeMaker'   => '7.70',
            'ExtUtils::MakeMaker::Config'=> '7.70',
            'ExtUtils::MakeMaker::Locale'=> '7.70',
            'ExtUtils::MakeMaker::version'=> '7.70',
            'ExtUtils::MakeMaker::version::regex'=> '7.70',
            'ExtUtils::Mkbootstrap' => '7.70',
            'ExtUtils::Mksymlists'  => '7.70',
            'ExtUtils::ParseXS'     => '3.50',
            'ExtUtils::ParseXS::Constants'=> '3.50',
            'ExtUtils::ParseXS::CountLines'=> '3.50',
            'ExtUtils::ParseXS::Eval'=> '3.50',
            'ExtUtils::ParseXS::Utilities'=> '3.50',
            'ExtUtils::testlib'     => '7.70',
            'File::Copy'            => '2.41',
            'Locale::Maketext'      => '1.33',
            'Module::CoreList'      => '5.20230420',
            'Module::CoreList::Utils'=> '5.20230420',
            'Net::Ping'             => '2.76',
            'feature'               => '1.82',
            'threads'               => '2.36',
            'threads::shared'       => '1.68',
            'warnings'              => '1.64',
        },
        removed => {
        }
    },
    5.036001 => {
        delta_from => 5.036000,
        changed => {
            'B::Op_private'         => '5.036001',
            'Config'                => '5.036001',
            'Module::CoreList'      => '5.20230423',
            'Module::CoreList::Utils'=> '5.20230423',
        },
        removed => {
        }
    },
    5.038000 => {
        delta_from => 5.037011,
        changed => {
            'B::Deparse'            => '1.74',
            'B::Op_private'         => '5.038000',
            'CPAN'                  => '2.36',
            'CPAN::HTTP::Client'    => '1.9602',
            'Compress::Raw::Bzip2'  => '2.204_001',
            'Compress::Raw::Zlib'   => '2.204_001',
            'Config'                => '5.038',
            'Digest::MD5'           => '2.58_01',
            'DynaLoader'            => '1.54',
            'ExtUtils::ParseXS'     => '3.51',
            'ExtUtils::ParseXS::Constants'=> '3.51',
            'ExtUtils::ParseXS::CountLines'=> '3.51',
            'ExtUtils::ParseXS::Eval'=> '3.51',
            'ExtUtils::ParseXS::Utilities'=> '3.51',
            'ExtUtils::Typemaps'    => '3.51',
            'ExtUtils::Typemaps::Cmd'=> '3.51',
            'ExtUtils::Typemaps::InputMap'=> '3.51',
            'ExtUtils::Typemaps::OutputMap'=> '3.51',
            'ExtUtils::Typemaps::Type'=> '3.51',
            'File::Glob'            => '1.40',
            'HTTP::Tiny'            => '0.086',
            'IO'                    => '1.52',
            'IO::Dir'               => '1.52',
            'IO::File'              => '1.52',
            'IO::Handle'            => '1.52',
            'IO::Pipe'              => '1.52',
            'IO::Poll'              => '1.52',
            'IO::Seekable'          => '1.52',
            'IO::Select'            => '1.52',
            'IO::Socket'            => '1.52',
            'IO::Socket::INET'      => '1.52',
            'IO::Socket::IP'        => '0.41_01',
            'IO::Socket::UNIX'      => '1.52',
            'MIME::Base64'          => '3.16_01',
            'MIME::QuotedPrint'     => '3.16_01',
            'Module::CoreList'      => '5.20230520',
            'Module::CoreList::Utils'=> '5.20230520',
            'POSIX'                 => '2.13',
            'SDBM_File'             => '1.17',
            'Storable'              => '3.32',
            'Time::HiRes'           => '1.9775',
            'Time::Piece'           => '1.3401_01',
            'warnings'              => '1.65',
        },
        removed => {
        }
    },
    5.039001 => {
        delta_from => 5.038000,
        changed => {
            'B::Op_private'         => '5.039001',
            'CPAN::Meta::Requirements'=> '2.143',
            'CPAN::Meta::Requirements::Range'=> '2.143',
            'Compress::Raw::Bzip2'  => '2.205',
            'Compress::Raw::Zlib'   => '2.205',
            'Compress::Zlib'        => '2.205',
            'Config'                => '5.039001',
            'Errno'                 => '1.38',
            'ExtUtils::CBuilder'    => '0.280239',
            'ExtUtils::CBuilder::Base'=> '0.280239',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.280239',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.280239',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.280239',
            'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280239',
            'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280239',
            'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280239',
            'ExtUtils::CBuilder::Platform::aix'=> '0.280239',
            'ExtUtils::CBuilder::Platform::android'=> '0.280239',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.280239',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.280239',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280239',
            'ExtUtils::CBuilder::Platform::os2'=> '0.280239',
            'ExtUtils::Manifest'    => '1.75',
            'IO::Compress::Adapter::Bzip2'=> '2.205',
            'IO::Compress::Adapter::Deflate'=> '2.205',
            'IO::Compress::Adapter::Identity'=> '2.205',
            'IO::Compress::Base'    => '2.205',
            'IO::Compress::Base::Common'=> '2.205',
            'IO::Compress::Bzip2'   => '2.205',
            'IO::Compress::Deflate' => '2.205',
            'IO::Compress::Gzip'    => '2.205',
            'IO::Compress::Gzip::Constants'=> '2.205',
            'IO::Compress::RawDeflate'=> '2.205',
            'IO::Compress::Zip'     => '2.205',
            'IO::Compress::Zip::Constants'=> '2.205',
            'IO::Compress::Zlib::Constants'=> '2.205',
            'IO::Compress::Zlib::Extra'=> '2.205',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.205',
            'IO::Uncompress::Adapter::Identity'=> '2.205',
            'IO::Uncompress::Adapter::Inflate'=> '2.205',
            'IO::Uncompress::AnyInflate'=> '2.205',
            'IO::Uncompress::AnyUncompress'=> '2.205',
            'IO::Uncompress::Base'  => '2.205',
            'IO::Uncompress::Bunzip2'=> '2.205',
            'IO::Uncompress::Gunzip'=> '2.205',
            'IO::Uncompress::Inflate'=> '2.205',
            'IO::Uncompress::RawInflate'=> '2.205',
            'IO::Uncompress::Unzip' => '2.205',
            'Math::BigFloat'        => '1.999839',
            'Math::BigInt'          => '1.999839',
            'Math::BigInt::Calc'    => '1.999839',
            'Math::BigInt::FastCalc'=> '0.5014',
            'Math::BigInt::Lib'     => '1.999839',
            'Module::CoreList'      => '5.20230720',
            'Module::CoreList::Utils'=> '5.20230720',
            'Module::Metadata'      => '1.000038',
            'POSIX'                 => '2.14',
            'Socket'                => '2.037',
            'Test2'                 => '1.302195',
            'Test2::API'            => '1.302195',
            'Test2::API::Breakage'  => '1.302195',
            'Test2::API::Context'   => '1.302195',
            'Test2::API::Instance'  => '1.302195',
            'Test2::API::InterceptResult'=> '1.302195',
            'Test2::API::InterceptResult::Event'=> '1.302195',
            'Test2::API::InterceptResult::Facet'=> '1.302195',
            'Test2::API::InterceptResult::Hub'=> '1.302195',
            'Test2::API::InterceptResult::Squasher'=> '1.302195',
            'Test2::API::Stack'     => '1.302195',
            'Test2::Event'          => '1.302195',
            'Test2::Event::Bail'    => '1.302195',
            'Test2::Event::Diag'    => '1.302195',
            'Test2::Event::Encoding'=> '1.302195',
            'Test2::Event::Exception'=> '1.302195',
            'Test2::Event::Fail'    => '1.302195',
            'Test2::Event::Generic' => '1.302195',
            'Test2::Event::Note'    => '1.302195',
            'Test2::Event::Ok'      => '1.302195',
            'Test2::Event::Pass'    => '1.302195',
            'Test2::Event::Plan'    => '1.302195',
            'Test2::Event::Skip'    => '1.302195',
            'Test2::Event::Subtest' => '1.302195',
            'Test2::Event::TAP::Version'=> '1.302195',
            'Test2::Event::V2'      => '1.302195',
            'Test2::Event::Waiting' => '1.302195',
            'Test2::EventFacet'     => '1.302195',
            'Test2::EventFacet::About'=> '1.302195',
            'Test2::EventFacet::Amnesty'=> '1.302195',
            'Test2::EventFacet::Assert'=> '1.302195',
            'Test2::EventFacet::Control'=> '1.302195',
            'Test2::EventFacet::Error'=> '1.302195',
            'Test2::EventFacet::Hub'=> '1.302195',
            'Test2::EventFacet::Info'=> '1.302195',
            'Test2::EventFacet::Info::Table'=> '1.302195',
            'Test2::EventFacet::Meta'=> '1.302195',
            'Test2::EventFacet::Parent'=> '1.302195',
            'Test2::EventFacet::Plan'=> '1.302195',
            'Test2::EventFacet::Render'=> '1.302195',
            'Test2::EventFacet::Trace'=> '1.302195',
            'Test2::Formatter'      => '1.302195',
            'Test2::Formatter::TAP' => '1.302195',
            'Test2::Hub'            => '1.302195',
            'Test2::Hub::Interceptor'=> '1.302195',
            'Test2::Hub::Interceptor::Terminator'=> '1.302195',
            'Test2::Hub::Subtest'   => '1.302195',
            'Test2::IPC'            => '1.302195',
            'Test2::IPC::Driver'    => '1.302195',
            'Test2::IPC::Driver::Files'=> '1.302195',
            'Test2::Tools::Tiny'    => '1.302195',
            'Test2::Util'           => '1.302195',
            'Test2::Util::ExternalMeta'=> '1.302195',
            'Test2::Util::Facets2Legacy'=> '1.302195',
            'Test2::Util::HashBase' => '1.302195',
            'Test2::Util::Trace'    => '1.302195',
            'Test::Builder'         => '1.302195',
            'Test::Builder::Formatter'=> '1.302195',
            'Test::Builder::Module' => '1.302195',
            'Test::Builder::Tester' => '1.302195',
            'Test::Builder::Tester::Color'=> '1.302195',
            'Test::Builder::TodoDiag'=> '1.302195',
            'Test::More'            => '1.302195',
            'Test::Simple'          => '1.302195',
            'Test::Tester'          => '1.302195',
            'Test::Tester::Capture' => '1.302195',
            'Test::Tester::CaptureRunner'=> '1.302195',
            'Test::Tester::Delegate'=> '1.302195',
            'Test::use::ok'         => '1.302195',
            'Text::Tabs'            => '2023.0511',
            'Text::Wrap'            => '2023.0511',
            'Time::HiRes'           => '1.9776',
            'Time::Local'           => '1.35',
            'UNIVERSAL'             => '1.16',
            'feature'               => '1.83',
            'ok'                    => '1.302195',
            'perlfaq'               => '5.20230701',
            'threads'               => '2.37',
            'warnings'              => '1.66',
            'warnings::register'    => '1.06',
        },
        removed => {
        }
    },
    5.039002 => {
        delta_from => 5.039001,
        changed => {
            'App::Prove'            => '3.47',
            'App::Prove::State'     => '3.47',
            'App::Prove::State::Result'=> '3.47',
            'App::Prove::State::Result::Test'=> '3.47',
            'B::Op_private'         => '5.039002',
            'Compress::Raw::Bzip2'  => '2.206',
            'Compress::Raw::Zlib'   => '2.206',
            'Compress::Zlib'        => '2.206',
            'Config'                => '5.039002',
            'Cwd'                   => '3.90',
            'Devel::Peek'           => '1.34',
            'ExtUtils::Miniperl'    => '1.14',
            'File::Spec'            => '3.90',
            'File::Spec::AmigaOS'   => '3.90',
            'File::Spec::Cygwin'    => '3.90',
            'File::Spec::Epoc'      => '3.90',
            'File::Spec::Functions' => '3.90',
            'File::Spec::Mac'       => '3.90',
            'File::Spec::OS2'       => '3.90',
            'File::Spec::Unix'      => '3.90',
            'File::Spec::VMS'       => '3.90',
            'File::Spec::Win32'     => '3.90',
            'HTTP::Tiny'            => '0.088',
            'IO::Compress::Adapter::Bzip2'=> '2.206',
            'IO::Compress::Adapter::Deflate'=> '2.206',
            'IO::Compress::Adapter::Identity'=> '2.206',
            'IO::Compress::Base'    => '2.206',
            'IO::Compress::Base::Common'=> '2.206',
            'IO::Compress::Bzip2'   => '2.206',
            'IO::Compress::Deflate' => '2.206',
            'IO::Compress::Gzip'    => '2.206',
            'IO::Compress::Gzip::Constants'=> '2.206',
            'IO::Compress::RawDeflate'=> '2.206',
            'IO::Compress::Zip'     => '2.206',
            'IO::Compress::Zip::Constants'=> '2.206',
            'IO::Compress::Zlib::Constants'=> '2.206',
            'IO::Compress::Zlib::Extra'=> '2.206',
            'IO::Socket::IP'        => '0.42',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.206',
            'IO::Uncompress::Adapter::Identity'=> '2.206',
            'IO::Uncompress::Adapter::Inflate'=> '2.206',
            'IO::Uncompress::AnyInflate'=> '2.206',
            'IO::Uncompress::AnyUncompress'=> '2.206',
            'IO::Uncompress::Base'  => '2.206',
            'IO::Uncompress::Bunzip2'=> '2.206',
            'IO::Uncompress::Gunzip'=> '2.206',
            'IO::Uncompress::Inflate'=> '2.206',
            'IO::Uncompress::RawInflate'=> '2.206',
            'IO::Uncompress::Unzip' => '2.206',
            'Module::CoreList'      => '5.20230820',
            'Module::CoreList::Utils'=> '5.20230820',
            'NDBM_File'             => '1.17',
            'Opcode'                => '1.65',
            'POSIX'                 => '2.15',
            'PerlIO::scalar'        => '0.32',
            'PerlIO::via'           => '0.19',
            'Pod::Html'             => '1.35',
            'Pod::Html::Util'       => '1.35',
            'Pod::Simple'           => '3.45',
            'Pod::Simple::BlackBox' => '3.45',
            'Pod::Simple::Checker'  => '3.45',
            'Pod::Simple::Debug'    => '3.45',
            'Pod::Simple::DumpAsText'=> '3.45',
            'Pod::Simple::DumpAsXML'=> '3.45',
            'Pod::Simple::HTML'     => '3.45',
            'Pod::Simple::HTMLBatch'=> '3.45',
            'Pod::Simple::HTMLLegacy'=> '5.02',
            'Pod::Simple::LinkSection'=> '3.45',
            'Pod::Simple::Methody'  => '3.45',
            'Pod::Simple::Progress' => '3.45',
            'Pod::Simple::PullParser'=> '3.45',
            'Pod::Simple::PullParserEndToken'=> '3.45',
            'Pod::Simple::PullParserStartToken'=> '3.45',
            'Pod::Simple::PullParserTextToken'=> '3.45',
            'Pod::Simple::PullParserToken'=> '3.45',
            'Pod::Simple::RTF'      => '3.45',
            'Pod::Simple::Search'   => '3.45',
            'Pod::Simple::SimpleTree'=> '3.45',
            'Pod::Simple::Text'     => '3.45',
            'Pod::Simple::TextContent'=> '3.45',
            'Pod::Simple::TiedOutFH'=> '3.45',
            'Pod::Simple::Transcode'=> '3.45',
            'Pod::Simple::TranscodeDumb'=> '3.45',
            'Pod::Simple::TranscodeSmart'=> '3.45',
            'Pod::Simple::XHTML'    => '3.45',
            'Pod::Simple::XMLOutStream'=> '3.45',
            'Safe'                  => '2.45',
            'TAP::Base'             => '3.47',
            'TAP::Formatter::Base'  => '3.47',
            'TAP::Formatter::Color' => '3.47',
            'TAP::Formatter::Console'=> '3.47',
            'TAP::Formatter::Console::ParallelSession'=> '3.47',
            'TAP::Formatter::Console::Session'=> '3.47',
            'TAP::Formatter::File'  => '3.47',
            'TAP::Formatter::File::Session'=> '3.47',
            'TAP::Formatter::Session'=> '3.47',
            'TAP::Harness'          => '3.47',
            'TAP::Harness::Env'     => '3.47',
            'TAP::Object'           => '3.47',
            'TAP::Parser'           => '3.47',
            'TAP::Parser::Aggregator'=> '3.47',
            'TAP::Parser::Grammar'  => '3.47',
            'TAP::Parser::Iterator' => '3.47',
            'TAP::Parser::Iterator::Array'=> '3.47',
            'TAP::Parser::Iterator::Process'=> '3.47',
            'TAP::Parser::Iterator::Stream'=> '3.47',
            'TAP::Parser::IteratorFactory'=> '3.47',
            'TAP::Parser::Multiplexer'=> '3.47',
            'TAP::Parser::Result'   => '3.47',
            'TAP::Parser::Result::Bailout'=> '3.47',
            'TAP::Parser::Result::Comment'=> '3.47',
            'TAP::Parser::Result::Plan'=> '3.47',
            'TAP::Parser::Result::Pragma'=> '3.47',
            'TAP::Parser::Result::Test'=> '3.47',
            'TAP::Parser::Result::Unknown'=> '3.47',
            'TAP::Parser::Result::Version'=> '3.47',
            'TAP::Parser::Result::YAML'=> '3.47',
            'TAP::Parser::ResultFactory'=> '3.47',
            'TAP::Parser::Scheduler'=> '3.47',
            'TAP::Parser::Scheduler::Job'=> '3.47',
            'TAP::Parser::Scheduler::Spinner'=> '3.47',
            'TAP::Parser::Source'   => '3.47',
            'TAP::Parser::SourceHandler'=> '3.47',
            'TAP::Parser::SourceHandler::Executable'=> '3.47',
            'TAP::Parser::SourceHandler::File'=> '3.47',
            'TAP::Parser::SourceHandler::Handle'=> '3.47',
            'TAP::Parser::SourceHandler::Perl'=> '3.47',
            'TAP::Parser::SourceHandler::RawTAP'=> '3.47',
            'TAP::Parser::YAMLish::Reader'=> '3.47',
            'TAP::Parser::YAMLish::Writer'=> '3.47',
            'Test::Harness'         => '3.47',
            'XS::APItest'           => '1.33',
            'builtin'               => '0.009',
            'feature'               => '1.84',
            'perlfaq'               => '5.20230812',
            'strict'                => '1.13',
            'threads'               => '2.38',
            'warnings'              => '1.67',
        },
        removed => {
        }
    },
    5.039003 => {
        delta_from => 5.039002,
        changed => {
            'B'                     => '1.89',
            'B::Op_private'         => '5.039003',
            'Config'                => '5.039003',
            'DB_File'               => '1.859',
            'Data::Dumper'          => '2.189',
            'Devel::PPPort'         => '3.72',
            'ExtUtils::CBuilder'    => '0.280240',
            'ExtUtils::CBuilder::Base'=> '0.280240',
            'ExtUtils::CBuilder::Platform::Unix'=> '0.280240',
            'ExtUtils::CBuilder::Platform::VMS'=> '0.280240',
            'ExtUtils::CBuilder::Platform::Windows'=> '0.280240',
            'ExtUtils::CBuilder::Platform::Windows::BCC'=> '0.280240',
            'ExtUtils::CBuilder::Platform::Windows::GCC'=> '0.280240',
            'ExtUtils::CBuilder::Platform::Windows::MSVC'=> '0.280240',
            'ExtUtils::CBuilder::Platform::aix'=> '0.280240',
            'ExtUtils::CBuilder::Platform::android'=> '0.280240',
            'ExtUtils::CBuilder::Platform::cygwin'=> '0.280240',
            'ExtUtils::CBuilder::Platform::darwin'=> '0.280240',
            'ExtUtils::CBuilder::Platform::dec_osf'=> '0.280240',
            'ExtUtils::CBuilder::Platform::os2'=> '0.280240',
            'File::Compare'         => '1.1008',
            'File::Spec::AmigaOS'   => '3.91',
            'File::Spec::Cygwin'    => '3.91',
            'File::Spec::Epoc'      => '3.91',
            'File::Spec::Functions' => '3.91',
            'File::Spec::Mac'       => '3.91',
            'File::Spec::OS2'       => '3.91',
            'File::Spec::Unix'      => '3.91',
            'File::Spec::VMS'       => '3.91',
            'File::Spec::Win32'     => '3.91',
            'FindBin'               => '1.54',
            'Getopt::Std'           => '1.14',
            'IO'                    => '1.53',
            'IO::Dir'               => '1.53',
            'IO::File'              => '1.53',
            'IO::Handle'            => '1.53',
            'IO::Pipe'              => '1.53',
            'IO::Poll'              => '1.53',
            'IO::Seekable'          => '1.53',
            'IO::Select'            => '1.53',
            'IO::Socket'            => '1.53',
            'IO::Socket::INET'      => '1.53',
            'IO::Socket::UNIX'      => '1.53',
            'Module::CoreList'      => '5.20230920',
            'Module::CoreList::Utils'=> '5.20230920',
            'builtin'               => '0.010',
            'fields'                => '2.25',
            'threads'               => '2.39',
            'threads::shared'       => '1.69',
        },
        removed => {
        }
    },
    5.039004 => {
        delta_from => 5.039003,
        changed => {
            'App::Prove'            => '3.48',
            'App::Prove::State'     => '3.48',
            'App::Prove::State::Result'=> '3.48',
            'App::Prove::State::Result::Test'=> '3.48',
            'B::Op_private'         => '5.039004',
            'Config'                => '5.039004',
            'File::Find'            => '1.44',
            'File::stat'            => '1.14',
            'Math::BigFloat'        => '1.999842',
            'Math::BigInt'          => '1.999842',
            'Math::BigInt::Calc'    => '1.999842',
            'Math::BigInt::FastCalc'=> '0.5015',
            'Math::BigInt::Lib'     => '1.999842',
            'Module::CoreList'      => '5.20231025',
            'Module::CoreList::Utils'=> '5.20231025',
            'Net::hostent'          => '1.04',
            'Net::netent'           => '1.02',
            'Net::protoent'         => '1.03',
            'Net::servent'          => '1.04',
            'POSIX'                 => '2.16',
            'TAP::Base'             => '3.48',
            'TAP::Formatter::Base'  => '3.48',
            'TAP::Formatter::Color' => '3.48',
            'TAP::Formatter::Console'=> '3.48',
            'TAP::Formatter::Console::ParallelSession'=> '3.48',
            'TAP::Formatter::Console::Session'=> '3.48',
            'TAP::Formatter::File'  => '3.48',
            'TAP::Formatter::File::Session'=> '3.48',
            'TAP::Formatter::Session'=> '3.48',
            'TAP::Harness'          => '3.48',
            'TAP::Harness::Env'     => '3.48',
            'TAP::Object'           => '3.48',
            'TAP::Parser'           => '3.48',
            'TAP::Parser::Aggregator'=> '3.48',
            'TAP::Parser::Grammar'  => '3.48',
            'TAP::Parser::Iterator' => '3.48',
            'TAP::Parser::Iterator::Array'=> '3.48',
            'TAP::Parser::Iterator::Process'=> '3.48',
            'TAP::Parser::Iterator::Stream'=> '3.48',
            'TAP::Parser::IteratorFactory'=> '3.48',
            'TAP::Parser::Multiplexer'=> '3.48',
            'TAP::Parser::Result'   => '3.48',
            'TAP::Parser::Result::Bailout'=> '3.48',
            'TAP::Parser::Result::Comment'=> '3.48',
            'TAP::Parser::Result::Plan'=> '3.48',
            'TAP::Parser::Result::Pragma'=> '3.48',
            'TAP::Parser::Result::Test'=> '3.48',
            'TAP::Parser::Result::Unknown'=> '3.48',
            'TAP::Parser::Result::Version'=> '3.48',
            'TAP::Parser::Result::YAML'=> '3.48',
            'TAP::Parser::ResultFactory'=> '3.48',
            'TAP::Parser::Scheduler'=> '3.48',
            'TAP::Parser::Scheduler::Job'=> '3.48',
            'TAP::Parser::Scheduler::Spinner'=> '3.48',
            'TAP::Parser::Source'   => '3.48',
            'TAP::Parser::SourceHandler'=> '3.48',
            'TAP::Parser::SourceHandler::Executable'=> '3.48',
            'TAP::Parser::SourceHandler::File'=> '3.48',
            'TAP::Parser::SourceHandler::Handle'=> '3.48',
            'TAP::Parser::SourceHandler::Perl'=> '3.48',
            'TAP::Parser::SourceHandler::RawTAP'=> '3.48',
            'TAP::Parser::YAMLish::Reader'=> '3.48',
            'TAP::Parser::YAMLish::Writer'=> '3.48',
            'Term::Table'           => '0.017',
            'Term::Table::Cell'     => '0.017',
            'Term::Table::CellStack'=> '0.017',
            'Term::Table::HashBase' => '0.017',
            'Term::Table::LineBreak'=> '0.017',
            'Term::Table::Spacer'   => '0.017',
            'Term::Table::Util'     => '0.017',
            'Test2::AsyncSubtest'   => '0.000156',
            'Test2::AsyncSubtest::Event::Attach'=> '0.000156',
            'Test2::AsyncSubtest::Event::Detach'=> '0.000156',
            'Test2::AsyncSubtest::Formatter'=> '0.000156',
            'Test2::AsyncSubtest::Hub'=> '0.000156',
            'Test2::Bundle'         => '0.000156',
            'Test2::Bundle::Extended'=> '0.000156',
            'Test2::Bundle::More'   => '0.000156',
            'Test2::Bundle::Simple' => '0.000156',
            'Test2::Compare'        => '0.000156',
            'Test2::Compare::Array' => '0.000156',
            'Test2::Compare::Bag'   => '0.000156',
            'Test2::Compare::Base'  => '0.000156',
            'Test2::Compare::Bool'  => '0.000156',
            'Test2::Compare::Custom'=> '0.000156',
            'Test2::Compare::DeepRef'=> '0.000156',
            'Test2::Compare::Delta' => '0.000156',
            'Test2::Compare::Event' => '0.000156',
            'Test2::Compare::EventMeta'=> '0.000156',
            'Test2::Compare::Float' => '0.000156',
            'Test2::Compare::Hash'  => '0.000156',
            'Test2::Compare::Isa'   => '0.000156',
            'Test2::Compare::Meta'  => '0.000156',
            'Test2::Compare::Negatable'=> '0.000156',
            'Test2::Compare::Number'=> '0.000156',
            'Test2::Compare::Object'=> '0.000156',
            'Test2::Compare::OrderedSubset'=> '0.000156',
            'Test2::Compare::Pattern'=> '0.000156',
            'Test2::Compare::Ref'   => '0.000156',
            'Test2::Compare::Regex' => '0.000156',
            'Test2::Compare::Scalar'=> '0.000156',
            'Test2::Compare::Set'   => '0.000156',
            'Test2::Compare::String'=> '0.000156',
            'Test2::Compare::Undef' => '0.000156',
            'Test2::Compare::Wildcard'=> '0.000156',
            'Test2::Manual'         => '0.000156',
            'Test2::Manual::Anatomy'=> '0.000156',
            'Test2::Manual::Anatomy::API'=> '0.000156',
            'Test2::Manual::Anatomy::Context'=> '0.000156',
            'Test2::Manual::Anatomy::EndToEnd'=> '0.000156',
            'Test2::Manual::Anatomy::Event'=> '0.000156',
            'Test2::Manual::Anatomy::Hubs'=> '0.000156',
            'Test2::Manual::Anatomy::IPC'=> '0.000156',
            'Test2::Manual::Anatomy::Utilities'=> '0.000156',
            'Test2::Manual::Concurrency'=> '0.000156',
            'Test2::Manual::Contributing'=> '0.000156',
            'Test2::Manual::Testing'=> '0.000156',
            'Test2::Manual::Testing::Introduction'=> '0.000156',
            'Test2::Manual::Testing::Migrating'=> '0.000156',
            'Test2::Manual::Testing::Planning'=> '0.000156',
            'Test2::Manual::Testing::Todo'=> '0.000156',
            'Test2::Manual::Tooling'=> '0.000156',
            'Test2::Manual::Tooling::FirstTool'=> '0.000156',
            'Test2::Manual::Tooling::Formatter'=> '0.000156',
            'Test2::Manual::Tooling::Nesting'=> '0.000156',
            'Test2::Manual::Tooling::Plugin::TestExit'=> '0.000156',
            'Test2::Manual::Tooling::Plugin::TestingDone'=> '0.000156',
            'Test2::Manual::Tooling::Plugin::ToolCompletes'=> '0.000156',
            'Test2::Manual::Tooling::Plugin::ToolStarts'=> '0.000156',
            'Test2::Manual::Tooling::Subtest'=> '0.000156',
            'Test2::Manual::Tooling::TestBuilder'=> '0.000156',
            'Test2::Manual::Tooling::Testing'=> '0.000156',
            'Test2::Mock'           => '0.000156',
            'Test2::Plugin'         => '0.000156',
            'Test2::Plugin::BailOnFail'=> '0.000156',
            'Test2::Plugin::DieOnFail'=> '0.000156',
            'Test2::Plugin::ExitSummary'=> '0.000156',
            'Test2::Plugin::SRand'  => '0.000156',
            'Test2::Plugin::Times'  => '0.000156',
            'Test2::Plugin::UTF8'   => '0.000156',
            'Test2::Require'        => '0.000156',
            'Test2::Require::AuthorTesting'=> '0.000156',
            'Test2::Require::EnvVar'=> '0.000156',
            'Test2::Require::Fork'  => '0.000156',
            'Test2::Require::Module'=> '0.000156',
            'Test2::Require::Perl'  => '0.000156',
            'Test2::Require::RealFork'=> '0.000156',
            'Test2::Require::Threads'=> '0.000156',
            'Test2::Suite'          => '0.000156',
            'Test2::Todo'           => '0.000156',
            'Test2::Tools'          => '0.000156',
            'Test2::Tools::AsyncSubtest'=> '0.000156',
            'Test2::Tools::Basic'   => '0.000156',
            'Test2::Tools::Class'   => '0.000156',
            'Test2::Tools::ClassicCompare'=> '0.000156',
            'Test2::Tools::Compare' => '0.000156',
            'Test2::Tools::Defer'   => '0.000156',
            'Test2::Tools::Encoding'=> '0.000156',
            'Test2::Tools::Event'   => '0.000156',
            'Test2::Tools::Exception'=> '0.000156',
            'Test2::Tools::Exports' => '0.000156',
            'Test2::Tools::GenTemp' => '0.000156',
            'Test2::Tools::Grab'    => '0.000156',
            'Test2::Tools::Mock'    => '0.000156',
            'Test2::Tools::Ref'     => '0.000156',
            'Test2::Tools::Refcount'=> '0.000156',
            'Test2::Tools::Spec'    => '0.000156',
            'Test2::Tools::Subtest' => '0.000156',
            'Test2::Tools::Target'  => '0.000156',
            'Test2::Tools::Tester'  => '0.000156',
            'Test2::Tools::Warnings'=> '0.000156',
            'Test2::Util::Grabber'  => '0.000156',
            'Test2::Util::Guard'    => '0.000156',
            'Test2::Util::Importer' => '0.000156',
            'Test2::Util::Ref'      => '0.000156',
            'Test2::Util::Stash'    => '0.000156',
            'Test2::Util::Sub'      => '0.000156',
            'Test2::Util::Table'    => '0.000156',
            'Test2::Util::Table::Cell'=> '0.000156',
            'Test2::Util::Table::LineBreak'=> '0.000156',
            'Test2::Util::Term'     => '0.000156',
            'Test2::Util::Times'    => '0.000156',
            'Test2::V0'             => '0.000156',
            'Test2::Workflow'       => '0.000156',
            'Test2::Workflow::BlockBase'=> '0.000156',
            'Test2::Workflow::Build'=> '0.000156',
            'Test2::Workflow::Runner'=> '0.000156',
            'Test2::Workflow::Task' => '0.000156',
            'Test2::Workflow::Task::Action'=> '0.000156',
            'Test2::Workflow::Task::Group'=> '0.000156',
            'Test::Harness'         => '3.48',
            'Time::gmtime'          => '1.05',
            'Time::localtime'       => '1.04',
            'Time::tm'              => '1.01',
            'User::grent'           => '1.05',
            'User::pwent'           => '1.03',
            'XS::APItest'           => '1.34',
            'XS::Typemap'           => '0.20',
            'builtin'               => '0.011',
            'feature'               => '1.85',
            'version'               => '0.9930',
            'version::regex'        => '0.9930',
        },
        removed => {
        }
    },
    5.039005 => {
        delta_from => 5.039004,
        changed => {
            'B::Op_private'         => '5.039005',
            'Benchmark'             => '1.25',
            'Config'                => '5.039005',
            'Encode'                => '3.20',
            'Getopt::Long'          => '2.57',
            'Getopt::Long::Parser'  => '2.57',
            'IO'                    => '1.54',
            'IO::Dir'               => '1.54',
            'IO::File'              => '1.54',
            'IO::Handle'            => '1.54',
            'IO::Pipe'              => '1.54',
            'IO::Poll'              => '1.54',
            'IO::Seekable'          => '1.54',
            'IO::Select'            => '1.54',
            'IO::Socket'            => '1.54',
            'IO::Socket::INET'      => '1.54',
            'IO::Socket::UNIX'      => '1.54',
            'Math::BigFloat'        => '2.001000',
            'Math::BigInt'          => '2.001000',
            'Math::BigInt::Calc'    => '2.001000',
            'Math::BigInt::Lib'     => '2.001000',
            'Math::BigRat'          => '2.001000',
            'Module::CoreList'      => '5.20231120',
            'Module::CoreList::Utils'=> '5.20231120',
            'POSIX'                 => '2.17',
            'Term::Table'           => '0.018',
            'Term::Table::Cell'     => '0.018',
            'Term::Table::CellStack'=> '0.018',
            'Term::Table::HashBase' => '0.018',
            'Term::Table::LineBreak'=> '0.018',
            'Term::Table::Spacer'   => '0.018',
            'Term::Table::Util'     => '0.018',
            'Test2::AsyncSubtest'   => '0.000159',
            'Test2::AsyncSubtest::Event::Attach'=> '0.000159',
            'Test2::AsyncSubtest::Event::Detach'=> '0.000159',
            'Test2::AsyncSubtest::Formatter'=> '0.000159',
            'Test2::AsyncSubtest::Hub'=> '0.000159',
            'Test2::Bundle'         => '0.000159',
            'Test2::Bundle::Extended'=> '0.000159',
            'Test2::Bundle::More'   => '0.000159',
            'Test2::Bundle::Simple' => '0.000159',
            'Test2::Compare'        => '0.000159',
            'Test2::Compare::Array' => '0.000159',
            'Test2::Compare::Bag'   => '0.000159',
            'Test2::Compare::Base'  => '0.000159',
            'Test2::Compare::Bool'  => '0.000159',
            'Test2::Compare::Custom'=> '0.000159',
            'Test2::Compare::DeepRef'=> '0.000159',
            'Test2::Compare::Delta' => '0.000159',
            'Test2::Compare::Event' => '0.000159',
            'Test2::Compare::EventMeta'=> '0.000159',
            'Test2::Compare::Float' => '0.000159',
            'Test2::Compare::Hash'  => '0.000159',
            'Test2::Compare::Isa'   => '0.000159',
            'Test2::Compare::Meta'  => '0.000159',
            'Test2::Compare::Negatable'=> '0.000159',
            'Test2::Compare::Number'=> '0.000159',
            'Test2::Compare::Object'=> '0.000159',
            'Test2::Compare::OrderedSubset'=> '0.000159',
            'Test2::Compare::Pattern'=> '0.000159',
            'Test2::Compare::Ref'   => '0.000159',
            'Test2::Compare::Regex' => '0.000159',
            'Test2::Compare::Scalar'=> '0.000159',
            'Test2::Compare::Set'   => '0.000159',
            'Test2::Compare::String'=> '0.000159',
            'Test2::Compare::Undef' => '0.000159',
            'Test2::Compare::Wildcard'=> '0.000159',
            'Test2::Manual'         => '0.000159',
            'Test2::Manual::Anatomy'=> '0.000159',
            'Test2::Manual::Anatomy::API'=> '0.000159',
            'Test2::Manual::Anatomy::Context'=> '0.000159',
            'Test2::Manual::Anatomy::EndToEnd'=> '0.000159',
            'Test2::Manual::Anatomy::Event'=> '0.000159',
            'Test2::Manual::Anatomy::Hubs'=> '0.000159',
            'Test2::Manual::Anatomy::IPC'=> '0.000159',
            'Test2::Manual::Anatomy::Utilities'=> '0.000159',
            'Test2::Manual::Concurrency'=> '0.000159',
            'Test2::Manual::Contributing'=> '0.000159',
            'Test2::Manual::Testing'=> '0.000159',
            'Test2::Manual::Testing::Introduction'=> '0.000159',
            'Test2::Manual::Testing::Migrating'=> '0.000159',
            'Test2::Manual::Testing::Planning'=> '0.000159',
            'Test2::Manual::Testing::Todo'=> '0.000159',
            'Test2::Manual::Tooling'=> '0.000159',
            'Test2::Manual::Tooling::FirstTool'=> '0.000159',
            'Test2::Manual::Tooling::Formatter'=> '0.000159',
            'Test2::Manual::Tooling::Nesting'=> '0.000159',
            'Test2::Manual::Tooling::Plugin::TestExit'=> '0.000159',
            'Test2::Manual::Tooling::Plugin::TestingDone'=> '0.000159',
            'Test2::Manual::Tooling::Plugin::ToolCompletes'=> '0.000159',
            'Test2::Manual::Tooling::Plugin::ToolStarts'=> '0.000159',
            'Test2::Manual::Tooling::Subtest'=> '0.000159',
            'Test2::Manual::Tooling::TestBuilder'=> '0.000159',
            'Test2::Manual::Tooling::Testing'=> '0.000159',
            'Test2::Mock'           => '0.000159',
            'Test2::Plugin'         => '0.000159',
            'Test2::Plugin::BailOnFail'=> '0.000159',
            'Test2::Plugin::DieOnFail'=> '0.000159',
            'Test2::Plugin::ExitSummary'=> '0.000159',
            'Test2::Plugin::SRand'  => '0.000159',
            'Test2::Plugin::Times'  => '0.000159',
            'Test2::Plugin::UTF8'   => '0.000159',
            'Test2::Require'        => '0.000159',
            'Test2::Require::AuthorTesting'=> '0.000159',
            'Test2::Require::EnvVar'=> '0.000159',
            'Test2::Require::Fork'  => '0.000159',
            'Test2::Require::Module'=> '0.000159',
            'Test2::Require::Perl'  => '0.000159',
            'Test2::Require::RealFork'=> '0.000159',
            'Test2::Require::Threads'=> '0.000159',
            'Test2::Suite'          => '0.000159',
            'Test2::Todo'           => '0.000159',
            'Test2::Tools'          => '0.000159',
            'Test2::Tools::AsyncSubtest'=> '0.000159',
            'Test2::Tools::Basic'   => '0.000159',
            'Test2::Tools::Class'   => '0.000159',
            'Test2::Tools::ClassicCompare'=> '0.000159',
            'Test2::Tools::Compare' => '0.000159',
            'Test2::Tools::Defer'   => '0.000159',
            'Test2::Tools::Encoding'=> '0.000159',
            'Test2::Tools::Event'   => '0.000159',
            'Test2::Tools::Exception'=> '0.000159',
            'Test2::Tools::Exports' => '0.000159',
            'Test2::Tools::GenTemp' => '0.000159',
            'Test2::Tools::Grab'    => '0.000159',
            'Test2::Tools::Mock'    => '0.000159',
            'Test2::Tools::Ref'     => '0.000159',
            'Test2::Tools::Refcount'=> '0.000159',
            'Test2::Tools::Spec'    => '0.000159',
            'Test2::Tools::Subtest' => '0.000159',
            'Test2::Tools::Target'  => '0.000159',
            'Test2::Tools::Tester'  => '0.000159',
            'Test2::Tools::Warnings'=> '0.000159',
            'Test2::Util::Grabber'  => '0.000159',
            'Test2::Util::Guard'    => '0.000159',
            'Test2::Util::Importer' => '0.000159',
            'Test2::Util::Ref'      => '0.000159',
            'Test2::Util::Stash'    => '0.000159',
            'Test2::Util::Sub'      => '0.000159',
            'Test2::Util::Table'    => '0.000159',
            'Test2::Util::Table::Cell'=> '0.000159',
            'Test2::Util::Table::LineBreak'=> '0.000159',
            'Test2::Util::Term'     => '0.000159',
            'Test2::Util::Times'    => '0.000159',
            'Test2::V0'             => '0.000159',
            'Test2::Workflow'       => '0.000159',
            'Test2::Workflow::BlockBase'=> '0.000159',
            'Test2::Workflow::Build'=> '0.000159',
            'Test2::Workflow::Runner'=> '0.000159',
            'Test2::Workflow::Task' => '0.000159',
            'Test2::Workflow::Task::Action'=> '0.000159',
            'Test2::Workflow::Task::Group'=> '0.000159',
            'builtin'               => '0.012',
            'locale'                => '1.11',
        },
        removed => {
        }
    },
    5.034002 => {
        delta_from => 5.034001,
        changed => {
            'B::Op_private'         => '5.034002',
            'Config'                => '5.034002',
            'Module::CoreList'      => '5.20231125',
            'Module::CoreList::Utils'=> '5.20231125',
        },
        removed => {
        }
    },
    5.036002 => {
        delta_from => 5.036001,
        changed => {
            'B::Op_private'         => '5.036002',
            'Config'                => '5.036002',
            'Module::CoreList'      => '5.20231125',
            'Module::CoreList::Utils'=> '5.20231125',
        },
        removed => {
        }
    },
    5.038001 => {
        delta_from => 5.038000,
        changed => {
            'B::Op_private'         => '5.038001',
            'Config'                => '5.038001',
            'Module::CoreList'      => '5.20231125',
            'Module::CoreList::Utils'=> '5.20231125',
        },
        removed => {
        }
    },
    5.034003 => {
        delta_from => 5.034002,
        changed => {
            'B::Op_private'         => '5.034003',
            'Config'                => '5.034003',
            'Module::CoreList'      => '5.20231129',
            'Module::CoreList::Utils'=> '5.20231129',
        },
        removed => {
        }
    },
    5.036003 => {
        delta_from => 5.036002,
        changed => {
            'B::Op_private'         => '5.036003',
            'Config'                => '5.036003',
            'Module::CoreList'      => '5.20231129',
            'Module::CoreList::Utils'=> '5.20231129',
        },
        removed => {
        }
    },
    5.038002 => {
        delta_from => 5.038001,
        changed => {
            'B::Op_private'         => '5.038002',
            'Config'                => '5.038002',
            'Module::CoreList'      => '5.20231129',
            'Module::CoreList::Utils'=> '5.20231129',
        },
        removed => {
        }
    },
    5.039006 => {
        delta_from => 5.039005,
        changed => {
            'Archive::Tar'          => '3.02_001',
            'Archive::Tar::Constant'=> '3.02_001',
            'Archive::Tar::File'    => '3.02_001',
            'B::Op_private'         => '5.039006',
            'Config'                => '5.039006',
            'Fatal'                 => '2.37',
            'Fcntl'                 => '1.16',
            'File::Glob'            => '1.41',
            'IO'                    => '1.55',
            'IO::Dir'               => '1.55',
            'IO::File'              => '1.55',
            'IO::Handle'            => '1.55',
            'IO::Pipe'              => '1.55',
            'IO::Poll'              => '1.55',
            'IO::Seekable'          => '1.55',
            'IO::Select'            => '1.55',
            'IO::Socket'            => '1.55',
            'IO::Socket::INET'      => '1.55',
            'IO::Socket::UNIX'      => '1.55',
            'Math::BigFloat'        => '2.003001',
            'Math::BigFloat::Trace' => '0.67',
            'Math::BigInt'          => '2.003001',
            'Math::BigInt::Calc'    => '2.003001',
            'Math::BigInt::FastCalc'=> '0.5016',
            'Math::BigInt::Lib'     => '2.003001',
            'Math::BigInt::Trace'   => '0.67',
            'Math::BigRat'          => '2.003001',
            'Math::BigRat::Trace'   => '0.67',
            'Module::CoreList'      => '5.20231230',
            'Module::CoreList::Utils'=> '5.20231230',
            'Test2'                 => '1.302198',
            'Test2::API'            => '1.302198',
            'Test2::API::Breakage'  => '1.302198',
            'Test2::API::Context'   => '1.302198',
            'Test2::API::Instance'  => '1.302198',
            'Test2::API::InterceptResult'=> '1.302198',
            'Test2::API::InterceptResult::Event'=> '1.302198',
            'Test2::API::InterceptResult::Facet'=> '1.302198',
            'Test2::API::InterceptResult::Hub'=> '1.302198',
            'Test2::API::InterceptResult::Squasher'=> '1.302198',
            'Test2::API::Stack'     => '1.302198',
            'Test2::Event'          => '1.302198',
            'Test2::Event::Bail'    => '1.302198',
            'Test2::Event::Diag'    => '1.302198',
            'Test2::Event::Encoding'=> '1.302198',
            'Test2::Event::Exception'=> '1.302198',
            'Test2::Event::Fail'    => '1.302198',
            'Test2::Event::Generic' => '1.302198',
            'Test2::Event::Note'    => '1.302198',
            'Test2::Event::Ok'      => '1.302198',
            'Test2::Event::Pass'    => '1.302198',
            'Test2::Event::Plan'    => '1.302198',
            'Test2::Event::Skip'    => '1.302198',
            'Test2::Event::Subtest' => '1.302198',
            'Test2::Event::TAP::Version'=> '1.302198',
            'Test2::Event::V2'      => '1.302198',
            'Test2::Event::Waiting' => '1.302198',
            'Test2::EventFacet'     => '1.302198',
            'Test2::EventFacet::About'=> '1.302198',
            'Test2::EventFacet::Amnesty'=> '1.302198',
            'Test2::EventFacet::Assert'=> '1.302198',
            'Test2::EventFacet::Control'=> '1.302198',
            'Test2::EventFacet::Error'=> '1.302198',
            'Test2::EventFacet::Hub'=> '1.302198',
            'Test2::EventFacet::Info'=> '1.302198',
            'Test2::EventFacet::Info::Table'=> '1.302198',
            'Test2::EventFacet::Meta'=> '1.302198',
            'Test2::EventFacet::Parent'=> '1.302198',
            'Test2::EventFacet::Plan'=> '1.302198',
            'Test2::EventFacet::Render'=> '1.302198',
            'Test2::EventFacet::Trace'=> '1.302198',
            'Test2::Formatter'      => '1.302198',
            'Test2::Formatter::TAP' => '1.302198',
            'Test2::Hub'            => '1.302198',
            'Test2::Hub::Interceptor'=> '1.302198',
            'Test2::Hub::Interceptor::Terminator'=> '1.302198',
            'Test2::Hub::Subtest'   => '1.302198',
            'Test2::IPC'            => '1.302198',
            'Test2::IPC::Driver'    => '1.302198',
            'Test2::IPC::Driver::Files'=> '1.302198',
            'Test2::Tools::Tiny'    => '1.302198',
            'Test2::Util'           => '1.302198',
            'Test2::Util::ExternalMeta'=> '1.302198',
            'Test2::Util::Facets2Legacy'=> '1.302198',
            'Test2::Util::HashBase' => '1.302198',
            'Test2::Util::Trace'    => '1.302198',
            'Test::Builder'         => '1.302198',
            'Test::Builder::Formatter'=> '1.302198',
            'Test::Builder::Module' => '1.302198',
            'Test::Builder::Tester' => '1.302198',
            'Test::Builder::Tester::Color'=> '1.302198',
            'Test::Builder::TodoDiag'=> '1.302198',
            'Test::More'            => '1.302198',
            'Test::Simple'          => '1.302198',
            'Test::Tester'          => '1.302198',
            'Test::Tester::Capture' => '1.302198',
            'Test::Tester::CaptureRunner'=> '1.302198',
            'Test::Tester::Delegate'=> '1.302198',
            'Test::use::ok'         => '1.302198',
            'autodie'               => '2.37',
            'autodie::Scope::Guard' => '2.37',
            'autodie::Scope::GuardStack'=> '2.37',
            'autodie::Util'         => '2.37',
            'autodie::exception'    => '2.37',
            'autodie::exception::system'=> '2.37',
            'autodie::hints'        => '2.37',
            'autodie::skip'         => '2.37',
            'bigfloat'              => '0.67',
            'bigint'                => '0.67',
            'bignum'                => '0.67',
            'bigrat'                => '0.67',
            'diagnostics'           => '1.40',
            'feature'               => '1.86',
            'ok'                    => '1.302198',
            're'                    => '0.45',
            'threads'               => '2.40',
        },
        removed => {
        }
    },
    5.039007 => {
        delta_from => 5.039006,
        changed => {
            'B::Op_private'         => '5.039007',
            'Config'                => '5.039007',
            'Exporter'              => '5.78',
            'Exporter::Heavy'       => '5.78',
            'Hash::Util'            => '0.31',
            'I18N::Langinfo'        => '0.23',
            'Math::BigFloat'        => '2.003002',
            'Math::BigInt'          => '2.003002',
            'Math::BigInt::Calc'    => '2.003002',
            'Math::BigInt::FastCalc'=> '0.5018',
            'Math::BigInt::Lib'     => '2.003002',
            'Math::BigRat'          => '2.003002',
            'Module::CoreList'      => '5.20240120',
            'Module::CoreList::Utils'=> '5.20240120',
            'Pod::Checker'          => '1.76',
            'SelfLoader'            => '1.27',
        },
        removed => {
        }
    },
    5.039008 => {
        delta_from => 5.039007,
        changed => {
            'B::Deparse'            => '1.75',
            'B::Op_private'         => '5.039008',
            'Config'                => '5.039008',
            'DynaLoader'            => '1.55',
            'File::Glob'            => '1.42',
            'Hash::Util'            => '0.32',
            'Hash::Util::FieldHash' => '1.27',
            'I18N::Langinfo'        => '0.24',
            'Module::CoreList'      => '5.20240223',
            'Module::CoreList::Utils'=> '5.20240223',
            'POSIX'                 => '2.18',
            'PerlIO::encoding'      => '0.31',
            'Pod::Checker'          => '1.77',
            'Safe'                  => '2.46',
            'Tie::File'             => '1.08',
            'XS::APItest'           => '1.35',
            'attributes'            => '0.36',
            'builtin'               => '0.014',
            'mro'                   => '1.29',
            'perlfaq'               => '5.20240218',
            'warnings'              => '1.68',
        },
        removed => {
        }
    },
    5.039009 => {
        delta_from => 5.039008,
        changed => {
            'B::Op_private'         => '5.039009',
            'Compress::Raw::Bzip2'  => '2.210',
            'Compress::Raw::Zlib'   => '2.209',
            'Compress::Zlib'        => '2.207',
            'Config'                => '5.039009',
            'DynaLoader'            => '1.56',
            'Encode'                => '3.21',
            'Fcntl'                 => '1.17',
            'IO::Compress::Adapter::Bzip2'=> '2.207',
            'IO::Compress::Adapter::Deflate'=> '2.207',
            'IO::Compress::Adapter::Identity'=> '2.207',
            'IO::Compress::Base'    => '2.207',
            'IO::Compress::Base::Common'=> '2.207',
            'IO::Compress::Bzip2'   => '2.207',
            'IO::Compress::Deflate' => '2.207',
            'IO::Compress::Gzip'    => '2.207',
            'IO::Compress::Gzip::Constants'=> '2.207',
            'IO::Compress::RawDeflate'=> '2.207',
            'IO::Compress::Zip'     => '2.207',
            'IO::Compress::Zip::Constants'=> '2.207',
            'IO::Compress::Zlib::Constants'=> '2.207',
            'IO::Compress::Zlib::Extra'=> '2.207',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.207',
            'IO::Uncompress::Adapter::Identity'=> '2.207',
            'IO::Uncompress::Adapter::Inflate'=> '2.207',
            'IO::Uncompress::AnyInflate'=> '2.207',
            'IO::Uncompress::AnyUncompress'=> '2.207',
            'IO::Uncompress::Base'  => '2.207',
            'IO::Uncompress::Bunzip2'=> '2.207',
            'IO::Uncompress::Gunzip'=> '2.207',
            'IO::Uncompress::Inflate'=> '2.207',
            'IO::Uncompress::RawInflate'=> '2.207',
            'IO::Uncompress::Unzip' => '2.207',
            'IO::Zlib'              => '1.15',
            'Module::CoreList'      => '5.20240320',
            'Module::CoreList::Utils'=> '5.20240320',
            'Text::Tabs'            => '2024.001',
            'Text::Wrap'            => '2024.001',
            'Tie::File'             => '1.09',
            'Time::HiRes'           => '1.9777',
            'bytes'                 => '1.09',
            're'                    => '0.47',
        },
        removed => {
        }
    },
    5.039010 => {
        delta_from => 5.039009,
        changed => {
            'B::Deparse'            => '1.76',
            'B::Op_private'         => '5.039010',
            'Compress::Raw::Bzip2'  => '2.212',
            'Compress::Raw::Zlib'   => '2.212',
            'Compress::Zlib'        => '2.212',
            'Config'                => '5.03901',
            'IO::Compress::Adapter::Bzip2'=> '2.212',
            'IO::Compress::Adapter::Deflate'=> '2.212',
            'IO::Compress::Adapter::Identity'=> '2.212',
            'IO::Compress::Base'    => '2.212',
            'IO::Compress::Base::Common'=> '2.212',
            'IO::Compress::Bzip2'   => '2.212',
            'IO::Compress::Deflate' => '2.212',
            'IO::Compress::Gzip'    => '2.212',
            'IO::Compress::Gzip::Constants'=> '2.212',
            'IO::Compress::RawDeflate'=> '2.212',
            'IO::Compress::Zip'     => '2.212',
            'IO::Compress::Zip::Constants'=> '2.212',
            'IO::Compress::Zlib::Constants'=> '2.212',
            'IO::Compress::Zlib::Extra'=> '2.212',
            'IO::Uncompress::Adapter::Bunzip2'=> '2.212',
            'IO::Uncompress::Adapter::Identity'=> '2.212',
            'IO::Uncompress::Adapter::Inflate'=> '2.212',
            'IO::Uncompress::AnyInflate'=> '2.212',
            'IO::Uncompress::AnyUncompress'=> '2.212',
            'IO::Uncompress::Base'  => '2.212',
            'IO::Uncompress::Bunzip2'=> '2.212',
            'IO::Uncompress::Gunzip'=> '2.212',
            'IO::Uncompress::Inflate'=> '2.212',
            'IO::Uncompress::RawInflate'=> '2.212',
            'IO::Uncompress::Unzip' => '2.212',
            'Module::CoreList'      => '5.20240420',
            'Module::CoreList::Utils'=> '5.20240420',
            'POSIX'                 => '2.19',
            'Pod::Man'              => '5.01_01',
            'Pod::ParseLink'        => '5.01_01',
            'Pod::Text'             => '5.01_01',
            'Pod::Text::Color'      => '5.01_01',
            'Pod::Text::Overstrike' => '5.01_01',
            'Pod::Text::Termcap'    => '5.01_01',
            'Socket'                => '2.038',
            'Test2'                 => '1.302199',
            'Test2::API'            => '1.302199',
            'Test2::