/**
 * GNU LibreJS - A browser add-on to block nonfree nontrivial JavaScript.
 * *
 * Copyright (C) 2011, 2012 Loic J. Duros
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see  <http://www.gnu.org/licenses/>.
 *
 */

/** 
 * This module checks http responses by mime type and returns a
 * modified response.
 */

var {Cc, Ci, Cu, Cm, Cr} = require("chrome");
var xpcom = require("xpcom");
Cu.import("resource://gre/modules/XPCOMUtils.jsm"); 

var jsChecker = require("js_checker/js_checker");

const types = require("js_checker/constant_types");
var checkTypes = types.checkTypes;

// check if scripts embedded dynamically have a jsWebLabel entry indexed by referrer.
var jsWebLabelEntries = require("html_script_finder/js_web_labels").jsWebLabelEntries;

var htmlParser = require("html_script_finder/html_parser");

var removedScripts = require("script_entries/removed_scripts").removedScripts;
var allowedRef = require('http_observer/allowed_referrers').allowedReferrers;

var acceptedScripts = require("script_entries/accepted_scripts").acceptedScripts;

// node.js url module. Makes it easier to resolve 
// urls in that datauri loaded dom
var urlHandler = require("url_handler/url_handler");


var jsMimeTypeRe = /.*(javascript|ecmascript).*/i;
var htmlMimeTypeRe = /.*(xhtml\+xml|html|multipart\/x-mixed-replace).*/i;


var processResponseObject = {
    data: null,
    myParser: null,
    url: null,
    scriptFinder: null,
    jsCheckString: null,
    referrer: null,
    contentType: null,
    resInfo: null,
    listener: null,
    req: null,

    /**
     * starts the handling of a new response.
     */
    init: function (listener, resInfo) {
	this.resInfo = resInfo;
	this.req = resInfo.request;
	this.listener = listener;
	this.setData();
	this.setContentType();
	this.setUrls();	
    },
    
    /**
     * genBinaryOutput
     * Set or reset binaryOutputStream and storageStream.
     */
    genBinaryOutput: function () {
	this.storageStream = Cc["@mozilla.org/storagestream;1"].createInstance(Ci.nsIStorageStream);
        this.binaryOutputStream = Cc["@mozilla.org/binaryoutputstream;1"].createInstance(Ci.nsIBinaryOutputStream);
    },

    /**
     * Join the data gathered from onDataAvailable.
     */
    setData: function () {

	this.data = this.listener.receivedData.join('');

	// Prevents the http response body from being empty,
	// which would throw an error.
	if (this.data == '' || this.data == undefined) {
	    this.data = " ";
	}

    },

    /**
     * Set a standardized lowercase mime type.
     */
    setContentType: function() {
	if (this.req.contentType != undefined) {
	    this.contentType = String(this.req.contentType).toLowerCase();
	}
    },

    /**
     * setUrls
     * Set the current URL of the response, and
     * set referrer if applicable.
     */
    setUrls: function() {
	if (this.req.URI != undefined) {
	    this.fragment = urlHandler.getFragment(this.req.URI.spec);
	    //console.log('fragment is', this.fragment);
	    this.url = urlHandler.removeFragment(this.req.URI.spec);
	}
	if (this.req.referrer != undefined) {
	    this.referrerFragment = urlHandler.getFragment(this.req.referrer.spec);
	    this.referrer = urlHandler.removeFragment(this.req.referrer.spec);
	}
    },
 
    /**
     * processHTML
     * Modifies a string of html
     */
    processHTML: function() {

	var charset = this.req.contentCharset, myParser;
	//console.log('charset is', charset);
	//console.log('responseStatus for', this.url, 'is', this.req.responseStatus);

	// send data to htmlParser, and pass on modified data to
	// originalListener.
	myParser = htmlParser.htmlParser().parse(this.data, 
						 charset, 
						 this.url,
						 this.fragment,
						 this.req.responseStatus,
						 this.htmlParseCallback.bind(this));
    },

    /**
      *
      * htmlParseCallback
      *
      * Passed on the callback result to
      * the originalListener.
      *
      */
    htmlParseCallback: function(result) {
	
	var len = result.length;

	try {

	    this.listener.originalListener.onDataAvailable(this.req, 
							   this.resInfo.context, 
							   result.newInputStream(0), 0, len);
	} catch (e) {

	    this.req.cancel(this.req.NS_BINDING_ABORTED);

	}

	this.listener.originalListener.onStopRequest(this.req, 
						     this.resInfo.context, this.resInfo.statusCode);
    },

    /**
     * processJS
     * Process and modify a string of JavaScript.
     */
    processJS: function() {
	var checker, check, jsCheckString,
	    that = this;
	//var start = Date.now(), end;

	try {
	    
	    // make sure script isn't already listed as free
	    // in a JS web labels table.
	    if (this.checkJsWebLabelsForScript()) { 

		// this is free. we are done.
		this.jsListenerCallback();
		return; 

	    }
	    
	    // analyze javascript in response.
	    checker = jsChecker.jsChecker();
	    check = checker.searchJs(this.data, function () {
					 that.processJsCallback(checker);
				     });



	} catch(e) {

	    // any error is considered nontrivial.
	    //console.log('js error in js app, removing script', e);

	    // modify data that will be sent to the browser.
	    this.data = '// LibreJS: JS mime-type. Script removed.';
	    this.jsListenerCallback();
	}

    },

    /**
     * checkJsWebLabelsForScript
     * 
     * check whether script that's been received has an entry
     * in a js web labels table (lookup referrer.)
     *  
     */
    checkJsWebLabelsForScript: function () {

	//console.log('checking script', this.url);
	//console.log('current list is', JSON.stringify(jsWebLabelEntries));
	if (jsWebLabelEntries[this.referrer] != undefined) {

	    var scriptList = jsWebLabelEntries[this.referrer],
	    i = 0,
            len = scriptList.length;
	    
	    for (; i < len; i++) {

		if (scriptList[i].fileUrl === this.url &&
		    scriptList[i].free === true) {

		    //console.log(this.url, "is free and dynamic!");

		    var scriptObj = {inline: false,
				     contents: this.url};

		    acceptedScripts.addAScript(this.req.referrer.spec, scriptObj);

		    return true;

		}

	    }
	    
 	    
	}
	
    },
    
    processJsCallback: function(checker) {
	try {
	    var scriptObj;
	    
	    var jsCheckString = checker.parseTree.freeTrivialCheck;
	    // for testing only.
	    // var jsCheckString = checkTypes.FREE;
    
	    if (jsCheckString === checkTypes.NONTRIVIAL_LOCAL ||
		jsCheckString === checkTypes.NONTRIVIAL_GLOBAL) {
		scriptObj = {inline: false,
			     contents: 'this script embedded dynamically was removed: '+ this.url,
			     removalReason: 'nontrivial'};

		removedScripts.addAScript(this.req.referrer.spec, scriptObj);
		// modify data that will be sent to the browser.
		this.data = '// LibreJS: JS mime-type. Script removed.';

		this.jsListenerCallback();

	    } else if (jsCheckString === checkTypes.FREE ||
		       jsCheckString === checkTypes.FREE_NONTRIVIAL_GLOBAL ||
		       jsCheckString === checkTypes.TRIVIAL ||
		       jsCheckString === checkTypes.TRIVIAL_DEFINES_FUNCTION) {

		//console.log('found a free script', this.req.referrer.spec);

		scriptObj = {inline: false,
			     contents: this.url};
		acceptedScripts.addAScript(this.req.referrer.spec, scriptObj);

		this.jsListenerCallback();

	    }

	    //var end = Date.now();
	    //console.log('exec time', this.url, ' -- ', end - start);
	} catch (x) {
	    console.log('error', x);
	}
    },

    /**
     * ProcessAllTypes
     * Calls processHTML or JS if it finds an appropriate content
     * type.  For everything else it just passes on the data to the
     * original listener.
     */
    processAllTypes: function() {
	// toggle xlibrejs if X-LibreJS is set.

	// process HTML 
	if ((htmlMimeTypeRe.test(this.contentType) ||
	     this.req.contentType === undefined) &&
	   !(allowedRef.urlInAllowedReferrers(this.url))) {
	    this.processHTML();
	    return;
	} 

	else {
	    // process external JS files that are called from another
	    // file (and hence have a referrer).

	    if (this.referrer != undefined &&
		jsMimeTypeRe.test(this.contentType) &&
		!allowedRef.urlInAllowedReferrers(this.referrer) &&
	        !(acceptedScripts.isFound(this.referrer, {inline: false, contents: this.url})) &&
	       !(acceptedScripts.isFound(this.referrer, {inline:false, contents:this.req.originalURI.spec}))) {
		
		//console.log('process js triggered for', this.url);
		this.processJS();

	    } else {
		this.jsListenerCallback();
	    }

	}

    },
    
    jsListenerCallback: function () {

	var len = this.data.length;
	
	this.genBinaryOutput();
	
	this.storageStream.init(8192, len, null);
	this.binaryOutputStream.setOutputStream(this.storageStream.getOutputStream(0));
	this.binaryOutputStream.writeBytes(this.data, len);
	
	try {
	    this.listener.originalListener.onDataAvailable(this.req, 
							   this.resInfo.context, 
							   this.storageStream.newInputStream(0), 
							   0, len);
	} catch (e) {
	    this.req.cancel(this.req.NS_BINDING_ABORTED);
	}

	this.listener.originalListener.onStopRequest(this.req, 
						     this.resInfo.context, 
						     this.resInfo.statusCode);

    }
    
    
};

// creates an instance of jsCheckerObject.
exports.ProcessResponse = function (listener, resInfo) {
    //console.log('triggered');
    var procResponse = Object.create(processResponseObject);
    procResponse.init(listener, resInfo);
    return procResponse;
};