"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerQueryAssistRoutes = registerQueryAssistRoutes;
var _configSchema = require("@osd/config-schema");
var _errors = require("../../../../../src/core/server/opensearch/client/errors");
var _query_assist = require("../../../common/constants/query_assist");
var _generate_field_context = require("../../common/helpers/query_assist/generate_field_context");
var _agents = require("./utils/agents");
var _constants = require("./utils/constants");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function registerQueryAssistRoutes(router) {
  /**
   * Returns whether the PPL agent is configured.
   */
  router.get({
    path: _query_assist.QUERY_ASSIST_API.CONFIGURED,
    validate: {}
  }, async (context, request, response) => {
    const client = context.core.opensearch.client.asCurrentUser;
    try {
      // if the call does not throw any error, then the agent is properly configured
      await (0, _agents.getAgentIdByConfig)(client, _constants.AGENT_CONFIGS.PPL_AGENT);
      return response.ok({
        body: {
          configured: true
        }
      });
    } catch (error) {
      return response.ok({
        body: {
          configured: false,
          error: error.message
        }
      });
    }
  });
  router.post({
    path: _query_assist.QUERY_ASSIST_API.GENERATE_PPL,
    validate: {
      body: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        question: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const client = context.core.opensearch.client.asCurrentUser;
    try {
      const pplRequest = await (0, _agents.getAgentIdAndRequest)({
        client,
        configName: _constants.AGENT_CONFIGS.PPL_AGENT,
        body: {
          parameters: {
            index: request.body.index,
            question: request.body.question
          }
        }
      });
      if (!pplRequest.body.inference_results[0].output[0].result) throw new Error('Generated PPL query not found.');
      const result = JSON.parse(pplRequest.body.inference_results[0].output[0].result);
      const ppl = result.ppl.replace(/[\r\n]/g, ' ').trim().replace(/ISNOTNULL/g, 'isnotnull') // https://github.com/opensearch-project/sql/issues/2431
      .replace(/\bSPAN\(/g, 'span('); // https://github.com/opensearch-project/dashboards-observability/issues/759
      return response.ok({
        body: ppl
      });
    } catch (error) {
      if ((0, _errors.isResponseError)(error) && error.statusCode === 400 && error.body.includes(_query_assist.ERROR_DETAILS.GUARDRAILS_TRIGGERED)) {
        return response.badRequest({
          body: _query_assist.ERROR_DETAILS.GUARDRAILS_TRIGGERED
        });
      }
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: _query_assist.QUERY_ASSIST_API.SUMMARIZE,
    validate: {
      body: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        question: _configSchema.schema.string(),
        query: _configSchema.schema.maybe(_configSchema.schema.string()),
        response: _configSchema.schema.string(),
        isError: _configSchema.schema.boolean()
      })
    }
  }, async (context, request, response) => {
    const client = context.core.opensearch.client.asCurrentUser;
    const {
      index,
      question,
      query,
      response: _response,
      isError
    } = request.body;
    const queryResponse = JSON.stringify(_response);
    let summaryRequest;
    try {
      var _summaryRequest$body$;
      if (!isError) {
        summaryRequest = await (0, _agents.getAgentIdAndRequest)({
          client,
          configName: _constants.AGENT_CONFIGS.RESPONSE_SUMMARY_AGENT,
          body: {
            parameters: {
              index,
              question,
              query,
              response: queryResponse
            }
          }
        });
      } else {
        const [mappings, sampleDoc] = await Promise.all([client.indices.getMapping({
          index
        }), client.search({
          index,
          size: 1
        })]);
        const fields = (0, _generate_field_context.generateFieldContext)(mappings, sampleDoc);
        summaryRequest = await (0, _agents.getAgentIdAndRequest)({
          client,
          configName: _constants.AGENT_CONFIGS.ERROR_SUMMARY_AGENT,
          body: {
            parameters: {
              index,
              question,
              query,
              response: queryResponse,
              fields
            }
          }
        });
      }
      const summary = summaryRequest.body.inference_results[0].output[0].result;
      if (!summary) throw new Error('Generated summary not found.');
      const suggestedQuestions = Array.from((((_summaryRequest$body$ = summaryRequest.body.inference_results[0].output[1]) === null || _summaryRequest$body$ === void 0 ? void 0 : _summaryRequest$body$.result) || '').matchAll(/<question>((.|[\r\n])+?)<\/question>/g)).map(m => m[1]);
      return response.ok({
        body: {
          summary,
          suggestedQuestions
        }
      });
    } catch (error) {
      if ((0, _errors.isResponseError)(error) && error.statusCode === 400 && error.body.includes(_query_assist.ERROR_DETAILS.GUARDRAILS_TRIGGERED)) {
        return response.badRequest({
          body: _query_assist.ERROR_DETAILS.GUARDRAILS_TRIGGERED
        });
      }
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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