#!/usr/bin/python

# (c) 2018-2025, NetApp, Inc
# GNU General Public License v3.0+ (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)

from __future__ import absolute_import, division, print_function

__metaclass__ = type

DOCUMENTATION = '''
module: na_ontap_vscan_on_demand_task
short_description: NetApp ONTAP Vscan on demand task configuration.
extends_documentation_fragment:
    - netapp.ontap.netapp.na_ontap
version_added: 2.8.0
author: NetApp Ansible Team (@carchi8py) <ng-ansible-team@netapp.com>
description:
- Configure on demand task for Vscan
options:
  state:
    description:
    - Whether a Vscan on demand task is present or not
    choices: ['present', 'absent']
    type: str
    default: present

  vserver:
    description:
    - the name of the data vserver to use.
    required: true
    type: str

  cross_junction:
    description:
    - Specifies whether the On-Demand task is allowed to cross volume junctions
    - This option is not supported with REST.
    - This option defaults to False for ZAPI.
    type: bool

  directory_recursion:
    description:
    - Specifies whether the On-Demand task is allowed to recursively scan through sub-directories.
    - This option is not supported with REST.
    - This option defaults to False for ZAPI.
    type: bool

  file_ext_to_exclude:
    description:
    - File-Extensions for which scanning must not be performed.
    - File whose extension matches with both inclusion and exclusion list is not considered for scanning.
    type: list
    elements: str

  file_ext_to_include:
    description:
    - File extensions for which scanning is considered.
    - The default value is '*', which means that all files are considered for scanning except those which are excluded from scanning.
    - File whose extension matches with both inclusion and exclusion list is not considered for scanning.
    type: list
    elements: str

  max_file_size:
    description:
    - Max file-size (in bytes) allowed for scanning. The default value of 10737418240 (10GB) is taken if not provided at the time of creating a task.
    type: int

  paths_to_exclude:
    description:
    - File-paths for which scanning must not be performed.
    type: list
    elements: str

  report_directory:
    description:
    - Path from the vserver root where task report is created. The path must be a directory and provided in unix-format from the root of the Vserver.
    - Example /vol1/on-demand-reports.
    type: str

  report_log_level:
    description:
    - Log level for the On-Demand report.
    - This option is not supported with REST.
    - This option defaults to 'error' for ZAPI.
    choices: ['verbose', 'info', 'error']
    type: str

  request_timeout:
    description:
    - Total request-service time-limit in seconds. If the virus-scanner does not respond within the provided time, scan will be timedout.
    - This option is not supported with REST.
    type: str

  scan_files_with_no_ext:
    description:
    - Specifies whether files without any extension are considered for scanning or not.
    type: bool
    default: True

  scan_paths:
    description:
    - List of paths that need to be scanned. The path must be provided in unix-format and from the root of the Vserver.
    - Example /vol1/large_files.
    type: list
    elements: str

  scan_priority:
    description:
    - Priority of the On-Demand scan requests generated by this task.
    - This option is not supported with REST.
    - This option default to 'low' for ZAPI
    choices: ['low', 'normal']
    type: str

  schedule:
    description:
    - Schedule of the task. The task will be run as per the schedule.
    - For running the task immediately, vscan-on-demand-task-run api must be used after creating a task.
    type: str

  task_name:
    description:
    - Name of the task.
    type: str
    required: True
'''

EXAMPLES = """
- name: Create Vscan On Demand Task
  netapp.ontap.na_ontap_vscan_on_demand_task:
    state: present
    username: '{{ netapp_username }}'
    password: '{{ netapp_password }}'
    hostname: '{{ netapp_hostname }}'
    vserver: carchi-vsim2
    task_name: carchiOnDemand
    scan_paths: /
    report_directory: /
    file_ext_to_exclude: ['py', 'yml']
    max_file_size: 10737418241
    paths_to_exclude: ['/tmp', '/var']
    report_log_level: info
    request_timeout: 60

- name: Delete Vscan On Demand Task
  netapp.ontap.na_ontap_vscan_on_demand_task:
    state: absent
    username: '{{ netapp_username }}'
    password: '{{ netapp_password }}'
    hostname: '{{ netapp_hostname }}'
    vserver: carchi-vsim2
    task_name: carchiOnDemand
"""

RETURN = """

"""

import traceback

from ansible.module_utils.basic import AnsibleModule
from ansible.module_utils._text import to_native
import ansible_collections.netapp.ontap.plugins.module_utils.netapp as netapp_utils
from ansible_collections.netapp.ontap.plugins.module_utils.netapp_module import NetAppModule
from ansible_collections.netapp.ontap.plugins.module_utils.netapp import OntapRestAPI
from ansible_collections.netapp.ontap.plugins.module_utils import rest_generic

HAS_NETAPP_LIB = netapp_utils.has_netapp_lib()


class NetAppOntapVscanOnDemandTask:
    def __init__(self):
        self.svm_uuid = None
        self.argument_spec = netapp_utils.na_ontap_host_argument_spec()
        self.argument_spec.update(dict(
            state=dict(required=False, type='str', choices=['present', 'absent'], default='present'),
            vserver=dict(required=True, type='str'),
            cross_junction=dict(required=False, type='bool'),
            directory_recursion=dict(required=False, type='bool'),
            file_ext_to_exclude=dict(required=False, type='list', elements='str'),
            file_ext_to_include=dict(required=False, type='list', elements='str'),
            max_file_size=dict(required=False, type="int"),
            paths_to_exclude=dict(required=False, type='list', elements='str'),
            report_directory=dict(required=False, type='str'),
            report_log_level=dict(required=False, type='str', choices=['verbose', 'info', 'error']),
            request_timeout=dict(required=False, type='str'),
            scan_files_with_no_ext=dict(required=False, type='bool', default=True),
            scan_paths=dict(required=False, type='list', elements='str'),
            scan_priority=dict(required=False, type='str', choices=['low', 'normal']),
            schedule=dict(required=False, type="str"),
            task_name=dict(required=True, type="str")
        ))
        self.module = AnsibleModule(
            argument_spec=self.argument_spec,
            supports_check_mode=True,
            required_if=[
                ["state", "present", ["report_directory", "scan_paths"]]
            ]
        )
        self.na_helper = NetAppModule()
        self.parameters = self.na_helper.set_parameters(self.module.params)

        self.rest_api = OntapRestAPI(self.module)
        unsupported_rest_properties = ['cross_junction', 'directory_recursion', 'report_log_level', 'request_timeout',
                                       'scan_priority']
        self.use_rest = self.rest_api.is_rest_supported_properties(self.parameters, unsupported_rest_properties)
        if not self.use_rest:
            if self.parameters.get('cross_junction') is None:
                self.parameters['cross_junction'] = False
            if self.parameters.get('directory_recursion') is None:
                self.parameters['directory_recursion'] = False
            if not netapp_utils.has_netapp_lib():
                self.module.fail_json(msg=netapp_utils.netapp_lib_is_required())
            else:
                self.server = netapp_utils.setup_na_ontap_zapi(module=self.module, vserver=self.parameters['vserver'])

    def get_demand_task(self):
        """
        Get a demand task
        :return: A vscan-on-demand-task-info or None
        """
        if self.use_rest:
            self.get_svm_uuid()
            return self.get_demand_task_rest()
        demand_task_iter = netapp_utils.zapi.NaElement("vscan-on-demand-task-get-iter")
        demand_task_info = netapp_utils.zapi.NaElement("vscan-on-demand-task-info")
        demand_task_info.add_new_child('task-name', self.parameters['task_name'])
        query = netapp_utils.zapi.NaElement('query')
        query.add_child_elem(demand_task_info)
        demand_task_iter.add_child_elem(query)
        try:
            result = self.server.invoke_successfully(demand_task_iter, True)
        except netapp_utils.zapi.NaApiError as error:
            self.module.fail_json(msg='Error searching for Vscan on demand task %s: %s' %
                                      (self.parameters['task_name'], to_native(error)),
                                  exception=traceback.format_exc())
        if result.get_child_by_name('num-records') and int(result.get_child_content('num-records')) >= 1:
            return result.get_child_by_name('attributes-list').get_child_by_name('vscan-on-demand-task-info')
        return None

    def create_demand_task(self):
        """
        Create a Demand Task
        :return: None
        """
        if self.use_rest:
            return self.create_demand_task_rest()
        demand_task_obj = netapp_utils.zapi.NaElement("vscan-on-demand-task-create")
        # Required items first
        demand_task_obj.add_new_child('report-directory', self.parameters['report_directory'])
        demand_task_obj.add_new_child('task-name', self.parameters['task_name'])
        scan_paths = netapp_utils.zapi.NaElement("scan-paths")
        for scan_path in self.parameters['scan_paths']:
            scan_paths.add_new_child('string', scan_path)
        demand_task_obj.add_child_elem(scan_paths)
        # Optional items next
        if self.parameters.get('cross_junction'):
            demand_task_obj.add_new_child('cross-junction', str(self.parameters['cross_junction']).lower())
        if self.parameters.get('directory_recursion'):
            demand_task_obj.add_new_child('directory-recursion', str(self.parameters['directory_recursion']).lower())
        if self.parameters.get('file_ext_to_exclude'):
            ext_to_exclude_obj = netapp_utils.zapi.NaElement('file-ext-to-exclude')
            for exclude_file in self.parameters['file_ext_to_exclude']:
                ext_to_exclude_obj.add_new_child('file-extension', exclude_file)
            demand_task_obj.add_child_elem(ext_to_exclude_obj)
        if self.parameters.get('file_ext_to_include'):
            ext_to_include_obj = netapp_utils.zapi.NaElement('file-ext-to-include')
            for include_file in self.parameters['file_ext_to_exclude']:
                ext_to_include_obj.add_child_elem(include_file)
            demand_task_obj.add_child_elem(ext_to_include_obj)
        if self.parameters.get('max_file_size'):
            demand_task_obj.add_new_child('max-file-size', str(self.parameters['max_file_size']))
        if self.parameters.get('paths_to_exclude'):
            exclude_paths = netapp_utils.zapi.NaElement('paths-to-exclude')
            for path in self.parameters['paths_to_exclude']:
                exclude_paths.add_new_child('string', path)
            demand_task_obj.add_child_elem(exclude_paths)
        if self.parameters.get('report_log_level'):
            demand_task_obj.add_new_child('report-log-level', self.parameters['report_log_level'])
        if self.parameters.get('request_timeout'):
            demand_task_obj.add_new_child('request-timeout', self.parameters['request_timeout'])
        if self.parameters.get('scan_files_with_no_ext'):
            demand_task_obj.add_new_child('scan-files-with-no-ext',
                                          str(self.parameters['scan_files_with_no_ext']).lower())
        if self.parameters.get('scan_priority'):
            demand_task_obj.add_new_child('scan-priority', self.parameters['scan_priority'].lower())
        if self.parameters.get('schedule'):
            demand_task_obj.add_new_child('schedule', self.parameters['schedule'])
        try:
            self.server.invoke_successfully(demand_task_obj, True)
        except netapp_utils.zapi.NaApiError as error:
            self.module.fail_json(msg='Error creating on demand task %s: %s' %
                                      (self.parameters['task_name'], to_native(error)),
                                  exception=traceback.format_exc())

    def delete_demand_task(self):
        """
        Delete a Demand Task"
        :return:
        """
        if self.use_rest:
            return self.delete_demand_task_rest()
        demand_task_obj = netapp_utils.zapi.NaElement('vscan-on-demand-task-delete')
        demand_task_obj.add_new_child('task-name', self.parameters['task_name'])
        try:
            self.server.invoke_successfully(demand_task_obj, True)
        except netapp_utils.zapi.NaApiError as error:
            self.module.fail_json(msg='Error deleting on demand task, %s: %s' %
                                      (self.parameters['task_name'], to_native(error)),
                                  exception=traceback.format_exc())

    def get_svm_uuid(self):
        api = 'svm/svms'
        query = {'name': self.parameters['vserver']}
        record, error = rest_generic.get_one_record(self.rest_api, api, query)
        if error:
            self.module.fail_json(msg='Error fetching svm uuid: %s' % to_native(error))
        if not record:
            self.module.fail_json(msg='Could not find svm uuid for %s' % self.parameters['vserver'])
        self.svm_uuid = record['uuid']

    def get_demand_task_rest(self):
        api = 'protocols/vscan/%s/on-demand-policies' % self.svm_uuid
        query = {'name': self.parameters['task_name'],
                 'fields': 'scope.exclude_extensions,'
                           'scope.include_extensions,'
                           'scope.max_file_size,'
                           'scope.exclude_paths,'
                           'log_path,'
                           'scope.scan_without_extension,'
                           'scan_paths,'
                           'schedule.name,'
                           'name'
                 }
        record, error = rest_generic.get_one_record(self.rest_api, api, query)
        if error:
            self.module.fail_json(msg='Error fetching on demand task %s: %s' % (self.parameters['task_name'], to_native(error)))
        if record:
            return self.format_on_demand_task(record)
        return None

    def format_on_demand_task(self, record):
        return {
            'task_name': record['name'],
            'file_ext_to_exclude': self.na_helper.safe_get(record, ['scope', 'exclude_extensions']),
            'file_ext_to_include': self.na_helper.safe_get(record, ['scope', 'include_extensions']),
            'max_file_size': self.na_helper.safe_get(record, ['scope', 'max_file_size']),
            'paths_to_exclude': self.na_helper.safe_get(record, ['scope', 'exclude_paths']),
            'report_directory': self.na_helper.safe_get(record, ['log_path']),
            'scan_files_with_no_ext': self.na_helper.safe_get(record, ['scope', 'scan_without_extension']),
            'scan_paths': self.na_helper.safe_get(record, ['scan_paths']),
            'schedule': self.na_helper.safe_get(record, ['schedule', 'name']),
        }

    def create_demand_task_rest(self):
        api = 'protocols/vscan/%s/on-demand-policies' % self.svm_uuid
        body = {
            'name': self.parameters['task_name'],
            'log_path': self.parameters['report_directory'],
            'scan_paths': self.parameters['scan_paths'],
        }
        if self.parameters.get('file_ext_to_exclude'):
            body['scope.exclude_extensions'] = self.parameters['file_ext_to_exclude']
        if self.parameters.get('file_ext_to_include'):
            body['scope.include_extensions'] = self.parameters['file_ext_to_include']
        if self.parameters.get('max_file_size'):
            body['scope.max_file_size'] = self.parameters['max_file_size']
        if self.parameters.get('paths_to_exclude'):
            body['scope.exclude_paths'] = self.parameters['paths_to_exclude']
        if self.parameters.get('scan_files_with_no_ext'):
            body['scope.scan_without_extension'] = self.parameters['scan_files_with_no_ext']
        if self.parameters.get('schedule'):
            body['schedule.name'] = self.parameters['schedule']
        dummy, error = rest_generic.post_async(self.rest_api, api, body)
        if error:
            self.module.fail_json(msg='Error creating on demand task %s: %s' % (self.parameters['task_name'], to_native(error)))

    def delete_demand_task_rest(self):
        api = 'protocols/vscan/%s/on-demand-policies' % self.svm_uuid
        dummy, error = rest_generic.delete_async(self.rest_api, api, self.parameters['task_name'])
        if error:
            self.module.fail_json(msg='Error deleting on demand task %s: %s' % (self.parameters['task_name'], to_native(error)))

    def apply(self):
        current = self.get_demand_task()
        cd_action = self.na_helper.get_cd_action(current, self.parameters)
        if self.na_helper.changed:
            if self.module.check_mode:
                pass
            else:
                if cd_action == 'create':
                    self.create_demand_task()
                elif cd_action == 'delete':
                    self.delete_demand_task()
        result = netapp_utils.generate_result(self.na_helper.changed, cd_action)
        self.module.exit_json(**result)


def main():
    """
    Execute action from playbook
    """
    command = NetAppOntapVscanOnDemandTask()
    command.apply()


if __name__ == '__main__':
    main()
