# -*- coding: utf-8 -*-
# Copyright 2025 Google LLC
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
import logging
import json  # type: ignore

from google.auth.transport.requests import AuthorizedSession  # type: ignore
from google.auth import credentials as ga_credentials  # type: ignore
from google.api_core import exceptions as core_exceptions
from google.api_core import retry as retries
from google.api_core import rest_helpers
from google.api_core import rest_streaming
from google.api_core import gapic_v1
import cloudsdk.google.protobuf

from cloudsdk.google.protobuf import json_format

from requests import __version__ as requests_version
import dataclasses
from typing import Any, Callable, Dict, List, Optional, Sequence, Tuple, Union
import warnings


from google.api import httpbody_pb2  # type: ignore
from googlecloudsdk.generated_clients.gapic_clients.aiplatform_v1beta1.types import prediction_service
from google.longrunning import operations_pb2  # type: ignore


from .rest_base import _BasePredictionServiceRestTransport
from .base import DEFAULT_CLIENT_INFO as BASE_DEFAULT_CLIENT_INFO

try:
    OptionalRetry = Union[retries.Retry, gapic_v1.method._MethodDefault, None]
except AttributeError:  # pragma: NO COVER
    OptionalRetry = Union[retries.Retry, object, None]  # type: ignore

try:
    from google.api_core import client_logging  # type: ignore
    CLIENT_LOGGING_SUPPORTED = True  # pragma: NO COVER
except ImportError:  # pragma: NO COVER
    CLIENT_LOGGING_SUPPORTED = False

_LOGGER = logging.getLogger(__name__)

DEFAULT_CLIENT_INFO = gapic_v1.client_info.ClientInfo(
    gapic_version=BASE_DEFAULT_CLIENT_INFO.gapic_version,
    grpc_version=None,
    rest_version=f"requests@{requests_version}",
)

if hasattr(DEFAULT_CLIENT_INFO, "protobuf_runtime_version"):  # pragma: NO COVER
    DEFAULT_CLIENT_INFO.protobuf_runtime_version = cloudsdk.google.protobuf.__version__


class PredictionServiceRestInterceptor:
    """Interceptor for PredictionService.

    Interceptors are used to manipulate requests, request metadata, and responses
    in arbitrary ways.
    Example use cases include:
    * Logging
    * Verifying requests according to service or custom semantics
    * Stripping extraneous information from responses

    These use cases and more can be enabled by injecting an
    instance of a custom subclass when constructing the PredictionServiceRestTransport.

    .. code-block:: python
        class MyCustomPredictionServiceInterceptor(PredictionServiceRestInterceptor):
            def pre_chat_completions(self, request, metadata):
                logging.log(f"Received request: {request}")
                return request, metadata

            def post_chat_completions(self, response):
                logging.log(f"Received response: {response}")
                return response

            def pre_count_tokens(self, request, metadata):
                logging.log(f"Received request: {request}")
                return request, metadata

            def post_count_tokens(self, response):
                logging.log(f"Received response: {response}")
                return response

            def pre_direct_predict(self, request, metadata):
                logging.log(f"Received request: {request}")
                return request, metadata

            def post_direct_predict(self, response):
                logging.log(f"Received response: {response}")
                return response

            def pre_direct_raw_predict(self, request, metadata):
                logging.log(f"Received request: {request}")
                return request, metadata

            def post_direct_raw_predict(self, response):
                logging.log(f"Received response: {response}")
                return response

            def pre_embed_content(self, request, metadata):
                logging.log(f"Received request: {request}")
                return request, metadata

            def post_embed_content(self, response):
                logging.log(f"Received response: {response}")
                return response

            def pre_explain(self, request, metadata):
                logging.log(f"Received request: {request}")
                return request, metadata

            def post_explain(self, response):
                logging.log(f"Received response: {response}")
                return response

            def pre_fetch_predict_operation(self, request, metadata):
                logging.log(f"Received request: {request}")
                return request, metadata

            def post_fetch_predict_operation(self, response):
                logging.log(f"Received response: {response}")
                return response

            def pre_generate_content(self, request, metadata):
                logging.log(f"Received request: {request}")
                return request, metadata

            def post_generate_content(self, response):
                logging.log(f"Received response: {response}")
                return response

            def pre_invoke(self, request, metadata):
                logging.log(f"Received request: {request}")
                return request, metadata

            def post_invoke(self, response):
                logging.log(f"Received response: {response}")
                return response

            def pre_predict(self, request, metadata):
                logging.log(f"Received request: {request}")
                return request, metadata

            def post_predict(self, response):
                logging.log(f"Received response: {response}")
                return response

            def pre_predict_long_running(self, request, metadata):
                logging.log(f"Received request: {request}")
                return request, metadata

            def post_predict_long_running(self, response):
                logging.log(f"Received response: {response}")
                return response

            def pre_raw_predict(self, request, metadata):
                logging.log(f"Received request: {request}")
                return request, metadata

            def post_raw_predict(self, response):
                logging.log(f"Received response: {response}")
                return response

            def pre_server_streaming_predict(self, request, metadata):
                logging.log(f"Received request: {request}")
                return request, metadata

            def post_server_streaming_predict(self, response):
                logging.log(f"Received response: {response}")
                return response

            def pre_stream_generate_content(self, request, metadata):
                logging.log(f"Received request: {request}")
                return request, metadata

            def post_stream_generate_content(self, response):
                logging.log(f"Received response: {response}")
                return response

            def pre_stream_raw_predict(self, request, metadata):
                logging.log(f"Received request: {request}")
                return request, metadata

            def post_stream_raw_predict(self, response):
                logging.log(f"Received response: {response}")
                return response

        transport = PredictionServiceRestTransport(interceptor=MyCustomPredictionServiceInterceptor())
        client = PredictionServiceClient(transport=transport)


    """
    def pre_chat_completions(self, request: prediction_service.ChatCompletionsRequest, metadata: Sequence[Tuple[str, Union[str, bytes]]]) -> Tuple[prediction_service.ChatCompletionsRequest, Sequence[Tuple[str, Union[str, bytes]]]]:
        """Pre-rpc interceptor for chat_completions

        Override in a subclass to manipulate the request or metadata
        before they are sent to the PredictionService server.
        """
        return request, metadata

    def post_chat_completions(self, response: rest_streaming.ResponseIterator) -> rest_streaming.ResponseIterator:
        """Post-rpc interceptor for chat_completions

        DEPRECATED. Please use the `post_chat_completions_with_metadata`
        interceptor instead.

        Override in a subclass to read or manipulate the response
        after it is returned by the PredictionService server but before
        it is returned to user code. This `post_chat_completions` interceptor runs
        before the `post_chat_completions_with_metadata` interceptor.
        """
        return response

    def post_chat_completions_with_metadata(self, response: rest_streaming.ResponseIterator, metadata: Sequence[Tuple[str, Union[str, bytes]]]) -> Tuple[rest_streaming.ResponseIterator, Sequence[Tuple[str, Union[str, bytes]]]]:
        """Post-rpc interceptor for chat_completions

        Override in a subclass to read or manipulate the response or metadata after it
        is returned by the PredictionService server but before it is returned to user code.

        We recommend only using this `post_chat_completions_with_metadata`
        interceptor in new development instead of the `post_chat_completions` interceptor.
        When both interceptors are used, this `post_chat_completions_with_metadata` interceptor runs after the
        `post_chat_completions` interceptor. The (possibly modified) response returned by
        `post_chat_completions` will be passed to
        `post_chat_completions_with_metadata`.
        """
        return response, metadata

    def pre_count_tokens(self, request: prediction_service.CountTokensRequest, metadata: Sequence[Tuple[str, Union[str, bytes]]]) -> Tuple[prediction_service.CountTokensRequest, Sequence[Tuple[str, Union[str, bytes]]]]:
        """Pre-rpc interceptor for count_tokens

        Override in a subclass to manipulate the request or metadata
        before they are sent to the PredictionService server.
        """
        return request, metadata

    def post_count_tokens(self, response: prediction_service.CountTokensResponse) -> prediction_service.CountTokensResponse:
        """Post-rpc interceptor for count_tokens

        DEPRECATED. Please use the `post_count_tokens_with_metadata`
        interceptor instead.

        Override in a subclass to read or manipulate the response
        after it is returned by the PredictionService server but before
        it is returned to user code. This `post_count_tokens` interceptor runs
        before the `post_count_tokens_with_metadata` interceptor.
        """
        return response

    def post_count_tokens_with_metadata(self, response: prediction_service.CountTokensResponse, metadata: Sequence[Tuple[str, Union[str, bytes]]]) -> Tuple[prediction_service.CountTokensResponse, Sequence[Tuple[str, Union[str, bytes]]]]:
        """Post-rpc interceptor for count_tokens

        Override in a subclass to read or manipulate the response or metadata after it
        is returned by the PredictionService server but before it is returned to user code.

        We recommend only using this `post_count_tokens_with_metadata`
        interceptor in new development instead of the `post_count_tokens` interceptor.
        When both interceptors are used, this `post_count_tokens_with_metadata` interceptor runs after the
        `post_count_tokens` interceptor. The (possibly modified) response returned by
        `post_count_tokens` will be passed to
        `post_count_tokens_with_metadata`.
        """
        return response, metadata

    def pre_direct_predict(self, request: prediction_service.DirectPredictRequest, metadata: Sequence[Tuple[str, Union[str, bytes]]]) -> Tuple[prediction_service.DirectPredictRequest, Sequence[Tuple[str, Union[str, bytes]]]]:
        """Pre-rpc interceptor for direct_predict

        Override in a subclass to manipulate the request or metadata
        before they are sent to the PredictionService server.
        """
        return request, metadata

    def post_direct_predict(self, response: prediction_service.DirectPredictResponse) -> prediction_service.DirectPredictResponse:
        """Post-rpc interceptor for direct_predict

        DEPRECATED. Please use the `post_direct_predict_with_metadata`
        interceptor instead.

        Override in a subclass to read or manipulate the response
        after it is returned by the PredictionService server but before
        it is returned to user code. This `post_direct_predict` interceptor runs
        before the `post_direct_predict_with_metadata` interceptor.
        """
        return response

    def post_direct_predict_with_metadata(self, response: prediction_service.DirectPredictResponse, metadata: Sequence[Tuple[str, Union[str, bytes]]]) -> Tuple[prediction_service.DirectPredictResponse, Sequence[Tuple[str, Union[str, bytes]]]]:
        """Post-rpc interceptor for direct_predict

        Override in a subclass to read or manipulate the response or metadata after it
        is returned by the PredictionService server but before it is returned to user code.

        We recommend only using this `post_direct_predict_with_metadata`
        interceptor in new development instead of the `post_direct_predict` interceptor.
        When both interceptors are used, this `post_direct_predict_with_metadata` interceptor runs after the
        `post_direct_predict` interceptor. The (possibly modified) response returned by
        `post_direct_predict` will be passed to
        `post_direct_predict_with_metadata`.
        """
        return response, metadata

    def pre_direct_raw_predict(self, request: prediction_service.DirectRawPredictRequest, metadata: Sequence[Tuple[str, Union[str, bytes]]]) -> Tuple[prediction_service.DirectRawPredictRequest, Sequence[Tuple[str, Union[str, bytes]]]]:
        """Pre-rpc interceptor for direct_raw_predict

        Override in a subclass to manipulate the request or metadata
        before they are sent to the PredictionService server.
        """
        return request, metadata

    def post_direct_raw_predict(self, response: prediction_service.DirectRawPredictResponse) -> prediction_service.DirectRawPredictResponse:
        """Post-rpc interceptor for direct_raw_predict

        DEPRECATED. Please use the `post_direct_raw_predict_with_metadata`
        interceptor instead.

        Override in a subclass to read or manipulate the response
        after it is returned by the PredictionService server but before
        it is returned to user code. This `post_direct_raw_predict` interceptor runs
        before the `post_direct_raw_predict_with_metadata` interceptor.
        """
        return response

    def post_direct_raw_predict_with_metadata(self, response: prediction_service.DirectRawPredictResponse, metadata: Sequence[Tuple[str, Union[str, bytes]]]) -> Tuple[prediction_service.DirectRawPredictResponse, Sequence[Tuple[str, Union[str, bytes]]]]:
        """Post-rpc interceptor for direct_raw_predict

        Override in a subclass to read or manipulate the response or metadata after it
        is returned by the PredictionService server but before it is returned to user code.

        We recommend only using this `post_direct_raw_predict_with_metadata`
        interceptor in new development instead of the `post_direct_raw_predict` interceptor.
        When both interceptors are used, this `post_direct_raw_predict_with_metadata` interceptor runs after the
        `post_direct_raw_predict` interceptor. The (possibly modified) response returned by
        `post_direct_raw_predict` will be passed to
        `post_direct_raw_predict_with_metadata`.
        """
        return response, metadata

    def pre_embed_content(self, request: prediction_service.EmbedContentRequest, metadata: Sequence[Tuple[str, Union[str, bytes]]]) -> Tuple[prediction_service.EmbedContentRequest, Sequence[Tuple[str, Union[str, bytes]]]]:
        """Pre-rpc interceptor for embed_content

        Override in a subclass to manipulate the request or metadata
        before they are sent to the PredictionService server.
        """
        return request, metadata

    def post_embed_content(self, response: prediction_service.EmbedContentResponse) -> prediction_service.EmbedContentResponse:
        """Post-rpc interceptor for embed_content

        DEPRECATED. Please use the `post_embed_content_with_metadata`
        interceptor instead.

        Override in a subclass to read or manipulate the response
        after it is returned by the PredictionService server but before
        it is returned to user code. This `post_embed_content` interceptor runs
        before the `post_embed_content_with_metadata` interceptor.
        """
        return response

    def post_embed_content_with_metadata(self, response: prediction_service.EmbedContentResponse, metadata: Sequence[Tuple[str, Union[str, bytes]]]) -> Tuple[prediction_service.EmbedContentResponse, Sequence[Tuple[str, Union[str, bytes]]]]:
        """Post-rpc interceptor for embed_content

        Override in a subclass to read or manipulate the response or metadata after it
        is returned by the PredictionService server but before it is returned to user code.

        We recommend only using this `post_embed_content_with_metadata`
        interceptor in new development instead of the `post_embed_content` interceptor.
        When both interceptors are used, this `post_embed_content_with_metadata` interceptor runs after the
        `post_embed_content` interceptor. The (possibly modified) response returned by
        `post_embed_content` will be passed to
        `post_embed_content_with_metadata`.
        """
        return response, metadata

    def pre_explain(self, request: prediction_service.ExplainRequest, metadata: Sequence[Tuple[str, Union[str, bytes]]]) -> Tuple[prediction_service.ExplainRequest, Sequence[Tuple[str, Union[str, bytes]]]]:
        """Pre-rpc interceptor for explain

        Override in a subclass to manipulate the request or metadata
        before they are sent to the PredictionService server.
        """
        return request, metadata

    def post_explain(self, response: prediction_service.ExplainResponse) -> prediction_service.ExplainResponse:
        """Post-rpc interceptor for explain

        DEPRECATED. Please use the `post_explain_with_metadata`
        interceptor instead.

        Override in a subclass to read or manipulate the response
        after it is returned by the PredictionService server but before
        it is returned to user code. This `post_explain` interceptor runs
        before the `post_explain_with_metadata` interceptor.
        """
        return response

    def post_explain_with_metadata(self, response: prediction_service.ExplainResponse, metadata: Sequence[Tuple[str, Union[str, bytes]]]) -> Tuple[prediction_service.ExplainResponse, Sequence[Tuple[str, Union[str, bytes]]]]:
        """Post-rpc interceptor for explain

        Override in a subclass to read or manipulate the response or metadata after it
        is returned by the PredictionService server but before it is returned to user code.

        We recommend only using this `post_explain_with_metadata`
        interceptor in new development instead of the `post_explain` interceptor.
        When both interceptors are used, this `post_explain_with_metadata` interceptor runs after the
        `post_explain` interceptor. The (possibly modified) response returned by
        `post_explain` will be passed to
        `post_explain_with_metadata`.
        """
        return response, metadata

    def pre_fetch_predict_operation(self, request: prediction_service.FetchPredictOperationRequest, metadata: Sequence[Tuple[str, Union[str, bytes]]]) -> Tuple[prediction_service.FetchPredictOperationRequest, Sequence[Tuple[str, Union[str, bytes]]]]:
        """Pre-rpc interceptor for fetch_predict_operation

        Override in a subclass to manipulate the request or metadata
        before they are sent to the PredictionService server.
        """
        return request, metadata

    def post_fetch_predict_operation(self, response: operations_pb2.Operation) -> operations_pb2.Operation:
        """Post-rpc interceptor for fetch_predict_operation

        DEPRECATED. Please use the `post_fetch_predict_operation_with_metadata`
        interceptor instead.

        Override in a subclass to read or manipulate the response
        after it is returned by the PredictionService server but before
        it is returned to user code. This `post_fetch_predict_operation` interceptor runs
        before the `post_fetch_predict_operation_with_metadata` interceptor.
        """
        return response

    def post_fetch_predict_operation_with_metadata(self, response: operations_pb2.Operation, metadata: Sequence[Tuple[str, Union[str, bytes]]]) -> Tuple[operations_pb2.Operation, Sequence[Tuple[str, Union[str, bytes]]]]:
        """Post-rpc interceptor for fetch_predict_operation

        Override in a subclass to read or manipulate the response or metadata after it
        is returned by the PredictionService server but before it is returned to user code.

        We recommend only using this `post_fetch_predict_operation_with_metadata`
        interceptor in new development instead of the `post_fetch_predict_operation` interceptor.
        When both interceptors are used, this `post_fetch_predict_operation_with_metadata` interceptor runs after the
        `post_fetch_predict_operation` interceptor. The (possibly modified) response returned by
        `post_fetch_predict_operation` will be passed to
        `post_fetch_predict_operation_with_metadata`.
        """
        return response, metadata

    def pre_generate_content(self, request: prediction_service.GenerateContentRequest, metadata: Sequence[Tuple[str, Union[str, bytes]]]) -> Tuple[prediction_service.GenerateContentRequest, Sequence[Tuple[str, Union[str, bytes]]]]:
        """Pre-rpc interceptor for generate_content

        Override in a subclass to manipulate the request or metadata
        before they are sent to the PredictionService server.
        """
        return request, metadata

    def post_generate_content(self, response: prediction_service.GenerateContentResponse) -> prediction_service.GenerateContentResponse:
        """Post-rpc interceptor for generate_content

        DEPRECATED. Please use the `post_generate_content_with_metadata`
        interceptor instead.

        Override in a subclass to read or manipulate the response
        after it is returned by the PredictionService server but before
        it is returned to user code. This `post_generate_content` interceptor runs
        before the `post_generate_content_with_metadata` interceptor.
        """
        return response

    def post_generate_content_with_metadata(self, response: prediction_service.GenerateContentResponse, metadata: Sequence[Tuple[str, Union[str, bytes]]]) -> Tuple[prediction_service.GenerateContentResponse, Sequence[Tuple[str, Union[str, bytes]]]]:
        """Post-rpc interceptor for generate_content

        Override in a subclass to read or manipulate the response or metadata after it
        is returned by the PredictionService server but before it is returned to user code.

        We recommend only using this `post_generate_content_with_metadata`
        interceptor in new development instead of the `post_generate_content` interceptor.
        When both interceptors are used, this `post_generate_content_with_metadata` interceptor runs after the
        `post_generate_content` interceptor. The (possibly modified) response returned by
        `post_generate_content` will be passed to
        `post_generate_content_with_metadata`.
        """
        return response, metadata

    def pre_invoke(self, request: prediction_service.InvokeRequest, metadata: Sequence[Tuple[str, Union[str, bytes]]]) -> Tuple[prediction_service.InvokeRequest, Sequence[Tuple[str, Union[str, bytes]]]]:
        """Pre-rpc interceptor for invoke

        Override in a subclass to manipulate the request or metadata
        before they are sent to the PredictionService server.
        """
        return request, metadata

    def post_invoke(self, response: rest_streaming.ResponseIterator) -> rest_streaming.ResponseIterator:
        """Post-rpc interceptor for invoke

        DEPRECATED. Please use the `post_invoke_with_metadata`
        interceptor instead.

        Override in a subclass to read or manipulate the response
        after it is returned by the PredictionService server but before
        it is returned to user code. This `post_invoke` interceptor runs
        before the `post_invoke_with_metadata` interceptor.
        """
        return response

    def post_invoke_with_metadata(self, response: rest_streaming.ResponseIterator, metadata: Sequence[Tuple[str, Union[str, bytes]]]) -> Tuple[rest_streaming.ResponseIterator, Sequence[Tuple[str, Union[str, bytes]]]]:
        """Post-rpc interceptor for invoke

        Override in a subclass to read or manipulate the response or metadata after it
        is returned by the PredictionService server but before it is returned to user code.

        We recommend only using this `post_invoke_with_metadata`
        interceptor in new development instead of the `post_invoke` interceptor.
        When both interceptors are used, this `post_invoke_with_metadata` interceptor runs after the
        `post_invoke` interceptor. The (possibly modified) response returned by
        `post_invoke` will be passed to
        `post_invoke_with_metadata`.
        """
        return response, metadata

    def pre_predict(self, request: prediction_service.PredictRequest, metadata: Sequence[Tuple[str, Union[str, bytes]]]) -> Tuple[prediction_service.PredictRequest, Sequence[Tuple[str, Union[str, bytes]]]]:
        """Pre-rpc interceptor for predict

        Override in a subclass to manipulate the request or metadata
        before they are sent to the PredictionService server.
        """
        return request, metadata

    def post_predict(self, response: prediction_service.PredictResponse) -> prediction_service.PredictResponse:
        """Post-rpc interceptor for predict

        DEPRECATED. Please use the `post_predict_with_metadata`
        interceptor instead.

        Override in a subclass to read or manipulate the response
        after it is returned by the PredictionService server but before
        it is returned to user code. This `post_predict` interceptor runs
        before the `post_predict_with_metadata` interceptor.
        """
        return response

    def post_predict_with_metadata(self, response: prediction_service.PredictResponse, metadata: Sequence[Tuple[str, Union[str, bytes]]]) -> Tuple[prediction_service.PredictResponse, Sequence[Tuple[str, Union[str, bytes]]]]:
        """Post-rpc interceptor for predict

        Override in a subclass to read or manipulate the response or metadata after it
        is returned by the PredictionService server but before it is returned to user code.

        We recommend only using this `post_predict_with_metadata`
        interceptor in new development instead of the `post_predict` interceptor.
        When both interceptors are used, this `post_predict_with_metadata` interceptor runs after the
        `post_predict` interceptor. The (possibly modified) response returned by
        `post_predict` will be passed to
        `post_predict_with_metadata`.
        """
        return response, metadata

    def pre_predict_long_running(self, request: prediction_service.PredictLongRunningRequest, metadata: Sequence[Tuple[str, Union[str, bytes]]]) -> Tuple[prediction_service.PredictLongRunningRequest, Sequence[Tuple[str, Union[str, bytes]]]]:
        """Pre-rpc interceptor for predict_long_running

        Override in a subclass to manipulate the request or metadata
        before they are sent to the PredictionService server.
        """
        return request, metadata

    def post_predict_long_running(self, response: operations_pb2.Operation) -> operations_pb2.Operation:
        """Post-rpc interceptor for predict_long_running

        DEPRECATED. Please use the `post_predict_long_running_with_metadata`
        interceptor instead.

        Override in a subclass to read or manipulate the response
        after it is returned by the PredictionService server but before
        it is returned to user code. This `post_predict_long_running` interceptor runs
        before the `post_predict_long_running_with_metadata` interceptor.
        """
        return response

    def post_predict_long_running_with_metadata(self, response: operations_pb2.Operation, metadata: Sequence[Tuple[str, Union[str, bytes]]]) -> Tuple[operations_pb2.Operation, Sequence[Tuple[str, Union[str, bytes]]]]:
        """Post-rpc interceptor for predict_long_running

        Override in a subclass to read or manipulate the response or metadata after it
        is returned by the PredictionService server but before it is returned to user code.

        We recommend only using this `post_predict_long_running_with_metadata`
        interceptor in new development instead of the `post_predict_long_running` interceptor.
        When both interceptors are used, this `post_predict_long_running_with_metadata` interceptor runs after the
        `post_predict_long_running` interceptor. The (possibly modified) response returned by
        `post_predict_long_running` will be passed to
        `post_predict_long_running_with_metadata`.
        """
        return response, metadata

    def pre_raw_predict(self, request: prediction_service.RawPredictRequest, metadata: Sequence[Tuple[str, Union[str, bytes]]]) -> Tuple[prediction_service.RawPredictRequest, Sequence[Tuple[str, Union[str, bytes]]]]:
        """Pre-rpc interceptor for raw_predict

        Override in a subclass to manipulate the request or metadata
        before they are sent to the PredictionService server.
        """
        return request, metadata

    def post_raw_predict(self, response: httpbody_pb2.HttpBody) -> httpbody_pb2.HttpBody:
        """Post-rpc interceptor for raw_predict

        DEPRECATED. Please use the `post_raw_predict_with_metadata`
        interceptor instead.

        Override in a subclass to read or manipulate the response
        after it is returned by the PredictionService server but before
        it is returned to user code. This `post_raw_predict` interceptor runs
        before the `post_raw_predict_with_metadata` interceptor.
        """
        return response

    def post_raw_predict_with_metadata(self, response: httpbody_pb2.HttpBody, metadata: Sequence[Tuple[str, Union[str, bytes]]]) -> Tuple[httpbody_pb2.HttpBody, Sequence[Tuple[str, Union[str, bytes]]]]:
        """Post-rpc interceptor for raw_predict

        Override in a subclass to read or manipulate the response or metadata after it
        is returned by the PredictionService server but before it is returned to user code.

        We recommend only using this `post_raw_predict_with_metadata`
        interceptor in new development instead of the `post_raw_predict` interceptor.
        When both interceptors are used, this `post_raw_predict_with_metadata` interceptor runs after the
        `post_raw_predict` interceptor. The (possibly modified) response returned by
        `post_raw_predict` will be passed to
        `post_raw_predict_with_metadata`.
        """
        return response, metadata

    def pre_server_streaming_predict(self, request: prediction_service.StreamingPredictRequest, metadata: Sequence[Tuple[str, Union[str, bytes]]]) -> Tuple[prediction_service.StreamingPredictRequest, Sequence[Tuple[str, Union[str, bytes]]]]:
        """Pre-rpc interceptor for server_streaming_predict

        Override in a subclass to manipulate the request or metadata
        before they are sent to the PredictionService server.
        """
        return request, metadata

    def post_server_streaming_predict(self, response: rest_streaming.ResponseIterator) -> rest_streaming.ResponseIterator:
        """Post-rpc interceptor for server_streaming_predict

        DEPRECATED. Please use the `post_server_streaming_predict_with_metadata`
        interceptor instead.

        Override in a subclass to read or manipulate the response
        after it is returned by the PredictionService server but before
        it is returned to user code. This `post_server_streaming_predict` interceptor runs
        before the `post_server_streaming_predict_with_metadata` interceptor.
        """
        return response

    def post_server_streaming_predict_with_metadata(self, response: rest_streaming.ResponseIterator, metadata: Sequence[Tuple[str, Union[str, bytes]]]) -> Tuple[rest_streaming.ResponseIterator, Sequence[Tuple[str, Union[str, bytes]]]]:
        """Post-rpc interceptor for server_streaming_predict

        Override in a subclass to read or manipulate the response or metadata after it
        is returned by the PredictionService server but before it is returned to user code.

        We recommend only using this `post_server_streaming_predict_with_metadata`
        interceptor in new development instead of the `post_server_streaming_predict` interceptor.
        When both interceptors are used, this `post_server_streaming_predict_with_metadata` interceptor runs after the
        `post_server_streaming_predict` interceptor. The (possibly modified) response returned by
        `post_server_streaming_predict` will be passed to
        `post_server_streaming_predict_with_metadata`.
        """
        return response, metadata

    def pre_stream_generate_content(self, request: prediction_service.GenerateContentRequest, metadata: Sequence[Tuple[str, Union[str, bytes]]]) -> Tuple[prediction_service.GenerateContentRequest, Sequence[Tuple[str, Union[str, bytes]]]]:
        """Pre-rpc interceptor for stream_generate_content

        Override in a subclass to manipulate the request or metadata
        before they are sent to the PredictionService server.
        """
        return request, metadata

    def post_stream_generate_content(self, response: rest_streaming.ResponseIterator) -> rest_streaming.ResponseIterator:
        """Post-rpc interceptor for stream_generate_content

        DEPRECATED. Please use the `post_stream_generate_content_with_metadata`
        interceptor instead.

        Override in a subclass to read or manipulate the response
        after it is returned by the PredictionService server but before
        it is returned to user code. This `post_stream_generate_content` interceptor runs
        before the `post_stream_generate_content_with_metadata` interceptor.
        """
        return response

    def post_stream_generate_content_with_metadata(self, response: rest_streaming.ResponseIterator, metadata: Sequence[Tuple[str, Union[str, bytes]]]) -> Tuple[rest_streaming.ResponseIterator, Sequence[Tuple[str, Union[str, bytes]]]]:
        """Post-rpc interceptor for stream_generate_content

        Override in a subclass to read or manipulate the response or metadata after it
        is returned by the PredictionService server but before it is returned to user code.

        We recommend only using this `post_stream_generate_content_with_metadata`
        interceptor in new development instead of the `post_stream_generate_content` interceptor.
        When both interceptors are used, this `post_stream_generate_content_with_metadata` interceptor runs after the
        `post_stream_generate_content` interceptor. The (possibly modified) response returned by
        `post_stream_generate_content` will be passed to
        `post_stream_generate_content_with_metadata`.
        """
        return response, metadata

    def pre_stream_raw_predict(self, request: prediction_service.StreamRawPredictRequest, metadata: Sequence[Tuple[str, Union[str, bytes]]]) -> Tuple[prediction_service.StreamRawPredictRequest, Sequence[Tuple[str, Union[str, bytes]]]]:
        """Pre-rpc interceptor for stream_raw_predict

        Override in a subclass to manipulate the request or metadata
        before they are sent to the PredictionService server.
        """
        return request, metadata

    def post_stream_raw_predict(self, response: rest_streaming.ResponseIterator) -> rest_streaming.ResponseIterator:
        """Post-rpc interceptor for stream_raw_predict

        DEPRECATED. Please use the `post_stream_raw_predict_with_metadata`
        interceptor instead.

        Override in a subclass to read or manipulate the response
        after it is returned by the PredictionService server but before
        it is returned to user code. This `post_stream_raw_predict` interceptor runs
        before the `post_stream_raw_predict_with_metadata` interceptor.
        """
        return response

    def post_stream_raw_predict_with_metadata(self, response: rest_streaming.ResponseIterator, metadata: Sequence[Tuple[str, Union[str, bytes]]]) -> Tuple[rest_streaming.ResponseIterator, Sequence[Tuple[str, Union[str, bytes]]]]:
        """Post-rpc interceptor for stream_raw_predict

        Override in a subclass to read or manipulate the response or metadata after it
        is returned by the PredictionService server but before it is returned to user code.

        We recommend only using this `post_stream_raw_predict_with_metadata`
        interceptor in new development instead of the `post_stream_raw_predict` interceptor.
        When both interceptors are used, this `post_stream_raw_predict_with_metadata` interceptor runs after the
        `post_stream_raw_predict` interceptor. The (possibly modified) response returned by
        `post_stream_raw_predict` will be passed to
        `post_stream_raw_predict_with_metadata`.
        """
        return response, metadata


@dataclasses.dataclass
class PredictionServiceRestStub:
    _session: AuthorizedSession
    _host: str
    _interceptor: PredictionServiceRestInterceptor


class PredictionServiceRestTransport(_BasePredictionServiceRestTransport):
    """REST backend synchronous transport for PredictionService.

    A service for online predictions and explanations.

    This class defines the same methods as the primary client, so the
    primary client can load the underlying transport implementation
    and call it.

    It sends JSON representations of protocol buffers over HTTP/1.1
    """

    def __init__(self, *,
            host: str = 'aiplatform.googleapis.com',
            credentials: Optional[ga_credentials.Credentials] = None,
            credentials_file: Optional[str] = None,
            scopes: Optional[Sequence[str]] = None,
            client_cert_source_for_mtls: Optional[Callable[[
                ], Tuple[bytes, bytes]]] = None,
            quota_project_id: Optional[str] = None,
            client_info: gapic_v1.client_info.ClientInfo = DEFAULT_CLIENT_INFO,
            always_use_jwt_access: Optional[bool] = False,
            url_scheme: str = 'https',
            interceptor: Optional[PredictionServiceRestInterceptor] = None,
            api_audience: Optional[str] = None,
            ) -> None:
        """Instantiate the transport.

       NOTE: This REST transport functionality is currently in a beta
       state (preview). We welcome your feedback via a GitHub issue in
       this library's repository. Thank you!

        Args:
            host (Optional[str]):
                 The hostname to connect to (default: 'aiplatform.googleapis.com').
            credentials (Optional[google.auth.credentials.Credentials]): The
                authorization credentials to attach to requests. These
                credentials identify the application to the service; if none
                are specified, the client will attempt to ascertain the
                credentials from the environment.

            credentials_file (Optional[str]): Deprecated. A file with credentials that can
                be loaded with :func:`google.auth.load_credentials_from_file`.
                This argument is ignored if ``channel`` is provided. This argument will be
                removed in the next major version of this library.
            scopes (Optional(Sequence[str])): A list of scopes. This argument is
                ignored if ``channel`` is provided.
            client_cert_source_for_mtls (Callable[[], Tuple[bytes, bytes]]): Client
                certificate to configure mutual TLS HTTP channel. It is ignored
                if ``channel`` is provided.
            quota_project_id (Optional[str]): An optional project to use for billing
                and quota.
            client_info (google.api_core.gapic_v1.client_info.ClientInfo):
                The client info used to send a user-agent string along with
                API requests. If ``None``, then default info will be used.
                Generally, you only need to set this if you are developing
                your own client library.
            always_use_jwt_access (Optional[bool]): Whether self signed JWT should
                be used for service account credentials.
            url_scheme: the protocol scheme for the API endpoint.  Normally
                "https", but for testing or local servers,
                "http" can be specified.
        """
        # Run the base constructor
        # TODO(yon-mg): resolve other ctor params i.e. scopes, quota, etc.
        # TODO: When custom host (api_endpoint) is set, `scopes` must *also* be set on the
        # credentials object
        super().__init__(
            host=host,
            credentials=credentials,
            client_info=client_info,
            always_use_jwt_access=always_use_jwt_access,
            url_scheme=url_scheme,
            api_audience=api_audience
        )
        self._session = AuthorizedSession(
            self._credentials, default_host=self.DEFAULT_HOST)
        if client_cert_source_for_mtls:
            self._session.configure_mtls_channel(client_cert_source_for_mtls)
        self._interceptor = interceptor or PredictionServiceRestInterceptor()
        self._prep_wrapped_messages(client_info)

    class _ChatCompletions(_BasePredictionServiceRestTransport._BaseChatCompletions, PredictionServiceRestStub):
        def __hash__(self):
            return hash("PredictionServiceRestTransport.ChatCompletions")

        @staticmethod
        def _get_response(
            host,
            metadata,
            query_params,
            session,
            timeout,
            transcoded_request,
            body=None):

            uri = transcoded_request['uri']
            method = transcoded_request['method']
            headers = dict(metadata)
            headers['Content-Type'] = 'application/json'
            response = getattr(session, method)(
                "{host}{uri}".format(host=host, uri=uri),
                timeout=timeout,
                headers=headers,
                params=rest_helpers.flatten_query_params(query_params, strict=True),
                data=body,
                stream=True,
                )
            return response

        def __call__(self,
                request: prediction_service.ChatCompletionsRequest, *,
                retry: OptionalRetry=gapic_v1.method.DEFAULT,
                timeout: Optional[float]=None,
                metadata: Sequence[Tuple[str, Union[str, bytes]]]=(),
                ) -> rest_streaming.ResponseIterator:
            r"""Call the chat completions method over HTTP.

            Args:
                request (~.prediction_service.ChatCompletionsRequest):
                    The request object. Request message for [PredictionService.ChatCompletions]
                retry (google.api_core.retry.Retry): Designation of what errors, if any,
                    should be retried.
                timeout (float): The timeout for this request.
                metadata (Sequence[Tuple[str, Union[str, bytes]]]): Key/value pairs which should be
                    sent along with the request as metadata. Normally, each value must be of type `str`,
                    but for metadata keys ending with the suffix `-bin`, the corresponding values must
                    be of type `bytes`.

            Returns:
                ~.httpbody_pb2.HttpBody:
                    Message that represents an arbitrary HTTP body. It
                should only be used for payload formats that can't be
                represented as JSON, such as raw binary or an HTML page.

                This message can be used both in streaming and
                non-streaming API methods in the request as well as the
                response.

                It can be used as a top-level request field, which is
                convenient if one wants to extract parameters from
                either the URL or HTTP template into the request fields
                and also want access to the raw HTTP body.

                Example:

                ::

                    message GetResourceRequest {
                      // A unique request id.
                      string request_id = 1;

                      // The raw HTTP body is bound to this field.
                      google.api.HttpBody http_body = 2;

                    }

                    service ResourceService {
                      rpc GetResource(GetResourceRequest)
                        returns (google.api.HttpBody);
                      rpc UpdateResource(google.api.HttpBody)
                        returns (google.protobuf.Empty);

                    }

                Example with streaming methods:

                ::

                    service CaldavService {
                      rpc GetCalendar(stream google.api.HttpBody)
                        returns (stream google.api.HttpBody);
                      rpc UpdateCalendar(stream google.api.HttpBody)
                        returns (stream google.api.HttpBody);

                    }

                Use of this type only changes how the request and
                response bodies are handled, all other features will
                continue to work unchanged.

            """

            http_options = _BasePredictionServiceRestTransport._BaseChatCompletions._get_http_options()

            request, metadata = self._interceptor.pre_chat_completions(request, metadata)
            transcoded_request = _BasePredictionServiceRestTransport._BaseChatCompletions._get_transcoded_request(http_options, request)

            body = _BasePredictionServiceRestTransport._BaseChatCompletions._get_request_body_json(transcoded_request)

            # Jsonify the query params
            query_params = _BasePredictionServiceRestTransport._BaseChatCompletions._get_query_params_json(transcoded_request)

            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(logging.DEBUG):  # pragma: NO COVER
                request_url = "{host}{uri}".format(host=self._host, uri=transcoded_request['uri'])
                method = transcoded_request['method']
                try:
                    request_payload = json_format.MessageToJson(request)
                except:
                    request_payload = None
                http_request = {
                  "payload": request_payload,
                  "requestMethod": method,
                  "requestUrl": request_url,
                  "headers": dict(metadata),
                }
                _LOGGER.debug(
                    f"Sending request for google.cloud.aiplatform_v1beta1.PredictionServiceClient.ChatCompletions",
                    extra = {
                        "serviceName": "google.cloud.aiplatform.v1beta1.PredictionService",
                        "rpcName": "ChatCompletions",
                        "httpRequest": http_request,
                        "metadata": http_request["headers"],
                    },
                )

            # Send the request
            response = PredictionServiceRestTransport._ChatCompletions._get_response(self._host, metadata, query_params, self._session, timeout, transcoded_request, body)

            # In case of error, raise the appropriate core_exceptions.GoogleAPICallError exception
            # subclass.
            if response.status_code >= 400:
                raise core_exceptions.from_http_response(response)

            # Return the response
            resp = rest_streaming.ResponseIterator(response, httpbody_pb2.HttpBody)

            resp = self._interceptor.post_chat_completions(resp)
            response_metadata = [(k, str(v)) for k, v in response.headers.items()]
            resp, _ = self._interceptor.post_chat_completions_with_metadata(resp, response_metadata)
            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(logging.DEBUG):  # pragma: NO COVER
                http_response = {
                "headers":  dict(response.headers),
                "status": response.status_code,
                }
                _LOGGER.debug(
                    "Received response for google.cloud.aiplatform_v1beta1.PredictionServiceClient.chat_completions",
                    extra = {
                        "serviceName": "google.cloud.aiplatform.v1beta1.PredictionService",
                        "rpcName": "ChatCompletions",
                        "metadata": http_response["headers"],
                        "httpResponse": http_response,
                    },
                )
            return resp

    class _CountTokens(_BasePredictionServiceRestTransport._BaseCountTokens, PredictionServiceRestStub):
        def __hash__(self):
            return hash("PredictionServiceRestTransport.CountTokens")

        @staticmethod
        def _get_response(
            host,
            metadata,
            query_params,
            session,
            timeout,
            transcoded_request,
            body=None):

            uri = transcoded_request['uri']
            method = transcoded_request['method']
            headers = dict(metadata)
            headers['Content-Type'] = 'application/json'
            response = getattr(session, method)(
                "{host}{uri}".format(host=host, uri=uri),
                timeout=timeout,
                headers=headers,
                params=rest_helpers.flatten_query_params(query_params, strict=True),
                data=body,
                )
            return response

        def __call__(self,
                request: prediction_service.CountTokensRequest, *,
                retry: OptionalRetry=gapic_v1.method.DEFAULT,
                timeout: Optional[float]=None,
                metadata: Sequence[Tuple[str, Union[str, bytes]]]=(),
                ) -> prediction_service.CountTokensResponse:
            r"""Call the count tokens method over HTTP.

            Args:
                request (~.prediction_service.CountTokensRequest):
                    The request object. Request message for
                [PredictionService.CountTokens][google.cloud.aiplatform.v1beta1.PredictionService.CountTokens].
                retry (google.api_core.retry.Retry): Designation of what errors, if any,
                    should be retried.
                timeout (float): The timeout for this request.
                metadata (Sequence[Tuple[str, Union[str, bytes]]]): Key/value pairs which should be
                    sent along with the request as metadata. Normally, each value must be of type `str`,
                    but for metadata keys ending with the suffix `-bin`, the corresponding values must
                    be of type `bytes`.

            Returns:
                ~.prediction_service.CountTokensResponse:
                    Response message for
                [PredictionService.CountTokens][google.cloud.aiplatform.v1beta1.PredictionService.CountTokens].

            """

            http_options = _BasePredictionServiceRestTransport._BaseCountTokens._get_http_options()

            request, metadata = self._interceptor.pre_count_tokens(request, metadata)
            transcoded_request = _BasePredictionServiceRestTransport._BaseCountTokens._get_transcoded_request(http_options, request)

            body = _BasePredictionServiceRestTransport._BaseCountTokens._get_request_body_json(transcoded_request)

            # Jsonify the query params
            query_params = _BasePredictionServiceRestTransport._BaseCountTokens._get_query_params_json(transcoded_request)

            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(logging.DEBUG):  # pragma: NO COVER
                request_url = "{host}{uri}".format(host=self._host, uri=transcoded_request['uri'])
                method = transcoded_request['method']
                try:
                    request_payload = type(request).to_json(request)
                except:
                    request_payload = None
                http_request = {
                  "payload": request_payload,
                  "requestMethod": method,
                  "requestUrl": request_url,
                  "headers": dict(metadata),
                }
                _LOGGER.debug(
                    f"Sending request for google.cloud.aiplatform_v1beta1.PredictionServiceClient.CountTokens",
                    extra = {
                        "serviceName": "google.cloud.aiplatform.v1beta1.PredictionService",
                        "rpcName": "CountTokens",
                        "httpRequest": http_request,
                        "metadata": http_request["headers"],
                    },
                )

            # Send the request
            response = PredictionServiceRestTransport._CountTokens._get_response(self._host, metadata, query_params, self._session, timeout, transcoded_request, body)

            # In case of error, raise the appropriate core_exceptions.GoogleAPICallError exception
            # subclass.
            if response.status_code >= 400:
                raise core_exceptions.from_http_response(response)

            # Return the response
            resp = prediction_service.CountTokensResponse()
            pb_resp = prediction_service.CountTokensResponse.pb(resp)

            json_format.Parse(response.content, pb_resp, ignore_unknown_fields=True)

            resp = self._interceptor.post_count_tokens(resp)
            response_metadata = [(k, str(v)) for k, v in response.headers.items()]
            resp, _ = self._interceptor.post_count_tokens_with_metadata(resp, response_metadata)
            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(logging.DEBUG):  # pragma: NO COVER
                try:
                    response_payload = prediction_service.CountTokensResponse.to_json(response)
                except:
                    response_payload = None
                http_response = {
                "payload": response_payload,
                "headers":  dict(response.headers),
                "status": response.status_code,
                }
                _LOGGER.debug(
                    "Received response for google.cloud.aiplatform_v1beta1.PredictionServiceClient.count_tokens",
                    extra = {
                        "serviceName": "google.cloud.aiplatform.v1beta1.PredictionService",
                        "rpcName": "CountTokens",
                        "metadata": http_response["headers"],
                        "httpResponse": http_response,
                    },
                )
            return resp

    class _DirectPredict(_BasePredictionServiceRestTransport._BaseDirectPredict, PredictionServiceRestStub):
        def __hash__(self):
            return hash("PredictionServiceRestTransport.DirectPredict")

        @staticmethod
        def _get_response(
            host,
            metadata,
            query_params,
            session,
            timeout,
            transcoded_request,
            body=None):

            uri = transcoded_request['uri']
            method = transcoded_request['method']
            headers = dict(metadata)
            headers['Content-Type'] = 'application/json'
            response = getattr(session, method)(
                "{host}{uri}".format(host=host, uri=uri),
                timeout=timeout,
                headers=headers,
                params=rest_helpers.flatten_query_params(query_params, strict=True),
                data=body,
                )
            return response

        def __call__(self,
                request: prediction_service.DirectPredictRequest, *,
                retry: OptionalRetry=gapic_v1.method.DEFAULT,
                timeout: Optional[float]=None,
                metadata: Sequence[Tuple[str, Union[str, bytes]]]=(),
                ) -> prediction_service.DirectPredictResponse:
            r"""Call the direct predict method over HTTP.

            Args:
                request (~.prediction_service.DirectPredictRequest):
                    The request object. Request message for
                [PredictionService.DirectPredict][google.cloud.aiplatform.v1beta1.PredictionService.DirectPredict].
                retry (google.api_core.retry.Retry): Designation of what errors, if any,
                    should be retried.
                timeout (float): The timeout for this request.
                metadata (Sequence[Tuple[str, Union[str, bytes]]]): Key/value pairs which should be
                    sent along with the request as metadata. Normally, each value must be of type `str`,
                    but for metadata keys ending with the suffix `-bin`, the corresponding values must
                    be of type `bytes`.

            Returns:
                ~.prediction_service.DirectPredictResponse:
                    Response message for
                [PredictionService.DirectPredict][google.cloud.aiplatform.v1beta1.PredictionService.DirectPredict].

            """

            http_options = _BasePredictionServiceRestTransport._BaseDirectPredict._get_http_options()

            request, metadata = self._interceptor.pre_direct_predict(request, metadata)
            transcoded_request = _BasePredictionServiceRestTransport._BaseDirectPredict._get_transcoded_request(http_options, request)

            body = _BasePredictionServiceRestTransport._BaseDirectPredict._get_request_body_json(transcoded_request)

            # Jsonify the query params
            query_params = _BasePredictionServiceRestTransport._BaseDirectPredict._get_query_params_json(transcoded_request)

            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(logging.DEBUG):  # pragma: NO COVER
                request_url = "{host}{uri}".format(host=self._host, uri=transcoded_request['uri'])
                method = transcoded_request['method']
                try:
                    request_payload = type(request).to_json(request)
                except:
                    request_payload = None
                http_request = {
                  "payload": request_payload,
                  "requestMethod": method,
                  "requestUrl": request_url,
                  "headers": dict(metadata),
                }
                _LOGGER.debug(
                    f"Sending request for google.cloud.aiplatform_v1beta1.PredictionServiceClient.DirectPredict",
                    extra = {
                        "serviceName": "google.cloud.aiplatform.v1beta1.PredictionService",
                        "rpcName": "DirectPredict",
                        "httpRequest": http_request,
                        "metadata": http_request["headers"],
                    },
                )

            # Send the request
            response = PredictionServiceRestTransport._DirectPredict._get_response(self._host, metadata, query_params, self._session, timeout, transcoded_request, body)

            # In case of error, raise the appropriate core_exceptions.GoogleAPICallError exception
            # subclass.
            if response.status_code >= 400:
                raise core_exceptions.from_http_response(response)

            # Return the response
            resp = prediction_service.DirectPredictResponse()
            pb_resp = prediction_service.DirectPredictResponse.pb(resp)

            json_format.Parse(response.content, pb_resp, ignore_unknown_fields=True)

            resp = self._interceptor.post_direct_predict(resp)
            response_metadata = [(k, str(v)) for k, v in response.headers.items()]
            resp, _ = self._interceptor.post_direct_predict_with_metadata(resp, response_metadata)
            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(logging.DEBUG):  # pragma: NO COVER
                try:
                    response_payload = prediction_service.DirectPredictResponse.to_json(response)
                except:
                    response_payload = None
                http_response = {
                "payload": response_payload,
                "headers":  dict(response.headers),
                "status": response.status_code,
                }
                _LOGGER.debug(
                    "Received response for google.cloud.aiplatform_v1beta1.PredictionServiceClient.direct_predict",
                    extra = {
                        "serviceName": "google.cloud.aiplatform.v1beta1.PredictionService",
                        "rpcName": "DirectPredict",
                        "metadata": http_response["headers"],
                        "httpResponse": http_response,
                    },
                )
            return resp

    class _DirectRawPredict(_BasePredictionServiceRestTransport._BaseDirectRawPredict, PredictionServiceRestStub):
        def __hash__(self):
            return hash("PredictionServiceRestTransport.DirectRawPredict")

        @staticmethod
        def _get_response(
            host,
            metadata,
            query_params,
            session,
            timeout,
            transcoded_request,
            body=None):

            uri = transcoded_request['uri']
            method = transcoded_request['method']
            headers = dict(metadata)
            headers['Content-Type'] = 'application/json'
            response = getattr(session, method)(
                "{host}{uri}".format(host=host, uri=uri),
                timeout=timeout,
                headers=headers,
                params=rest_helpers.flatten_query_params(query_params, strict=True),
                data=body,
                )
            return response

        def __call__(self,
                request: prediction_service.DirectRawPredictRequest, *,
                retry: OptionalRetry=gapic_v1.method.DEFAULT,
                timeout: Optional[float]=None,
                metadata: Sequence[Tuple[str, Union[str, bytes]]]=(),
                ) -> prediction_service.DirectRawPredictResponse:
            r"""Call the direct raw predict method over HTTP.

            Args:
                request (~.prediction_service.DirectRawPredictRequest):
                    The request object. Request message for
                [PredictionService.DirectRawPredict][google.cloud.aiplatform.v1beta1.PredictionService.DirectRawPredict].
                retry (google.api_core.retry.Retry): Designation of what errors, if any,
                    should be retried.
                timeout (float): The timeout for this request.
                metadata (Sequence[Tuple[str, Union[str, bytes]]]): Key/value pairs which should be
                    sent along with the request as metadata. Normally, each value must be of type `str`,
                    but for metadata keys ending with the suffix `-bin`, the corresponding values must
                    be of type `bytes`.

            Returns:
                ~.prediction_service.DirectRawPredictResponse:
                    Response message for
                [PredictionService.DirectRawPredict][google.cloud.aiplatform.v1beta1.PredictionService.DirectRawPredict].

            """

            http_options = _BasePredictionServiceRestTransport._BaseDirectRawPredict._get_http_options()

            request, metadata = self._interceptor.pre_direct_raw_predict(request, metadata)
            transcoded_request = _BasePredictionServiceRestTransport._BaseDirectRawPredict._get_transcoded_request(http_options, request)

            body = _BasePredictionServiceRestTransport._BaseDirectRawPredict._get_request_body_json(transcoded_request)

            # Jsonify the query params
            query_params = _BasePredictionServiceRestTransport._BaseDirectRawPredict._get_query_params_json(transcoded_request)

            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(logging.DEBUG):  # pragma: NO COVER
                request_url = "{host}{uri}".format(host=self._host, uri=transcoded_request['uri'])
                method = transcoded_request['method']
                try:
                    request_payload = type(request).to_json(request)
                except:
                    request_payload = None
                http_request = {
                  "payload": request_payload,
                  "requestMethod": method,
                  "requestUrl": request_url,
                  "headers": dict(metadata),
                }
                _LOGGER.debug(
                    f"Sending request for google.cloud.aiplatform_v1beta1.PredictionServiceClient.DirectRawPredict",
                    extra = {
                        "serviceName": "google.cloud.aiplatform.v1beta1.PredictionService",
                        "rpcName": "DirectRawPredict",
                        "httpRequest": http_request,
                        "metadata": http_request["headers"],
                    },
                )

            # Send the request
            response = PredictionServiceRestTransport._DirectRawPredict._get_response(self._host, metadata, query_params, self._session, timeout, transcoded_request, body)

            # In case of error, raise the appropriate core_exceptions.GoogleAPICallError exception
            # subclass.
            if response.status_code >= 400:
                raise core_exceptions.from_http_response(response)

            # Return the response
            resp = prediction_service.DirectRawPredictResponse()
            pb_resp = prediction_service.DirectRawPredictResponse.pb(resp)

            json_format.Parse(response.content, pb_resp, ignore_unknown_fields=True)

            resp = self._interceptor.post_direct_raw_predict(resp)
            response_metadata = [(k, str(v)) for k, v in response.headers.items()]
            resp, _ = self._interceptor.post_direct_raw_predict_with_metadata(resp, response_metadata)
            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(logging.DEBUG):  # pragma: NO COVER
                try:
                    response_payload = prediction_service.DirectRawPredictResponse.to_json(response)
                except:
                    response_payload = None
                http_response = {
                "payload": response_payload,
                "headers":  dict(response.headers),
                "status": response.status_code,
                }
                _LOGGER.debug(
                    "Received response for google.cloud.aiplatform_v1beta1.PredictionServiceClient.direct_raw_predict",
                    extra = {
                        "serviceName": "google.cloud.aiplatform.v1beta1.PredictionService",
                        "rpcName": "DirectRawPredict",
                        "metadata": http_response["headers"],
                        "httpResponse": http_response,
                    },
                )
            return resp

    class _EmbedContent(_BasePredictionServiceRestTransport._BaseEmbedContent, PredictionServiceRestStub):
        def __hash__(self):
            return hash("PredictionServiceRestTransport.EmbedContent")

        @staticmethod
        def _get_response(
            host,
            metadata,
            query_params,
            session,
            timeout,
            transcoded_request,
            body=None):

            uri = transcoded_request['uri']
            method = transcoded_request['method']
            headers = dict(metadata)
            headers['Content-Type'] = 'application/json'
            response = getattr(session, method)(
                "{host}{uri}".format(host=host, uri=uri),
                timeout=timeout,
                headers=headers,
                params=rest_helpers.flatten_query_params(query_params, strict=True),
                data=body,
                )
            return response

        def __call__(self,
                request: prediction_service.EmbedContentRequest, *,
                retry: OptionalRetry=gapic_v1.method.DEFAULT,
                timeout: Optional[float]=None,
                metadata: Sequence[Tuple[str, Union[str, bytes]]]=(),
                ) -> prediction_service.EmbedContentResponse:
            r"""Call the embed content method over HTTP.

            Args:
                request (~.prediction_service.EmbedContentRequest):
                    The request object. Request message for
                [PredictionService.EmbedContent][google.cloud.aiplatform.v1beta1.PredictionService.EmbedContent].
                retry (google.api_core.retry.Retry): Designation of what errors, if any,
                    should be retried.
                timeout (float): The timeout for this request.
                metadata (Sequence[Tuple[str, Union[str, bytes]]]): Key/value pairs which should be
                    sent along with the request as metadata. Normally, each value must be of type `str`,
                    but for metadata keys ending with the suffix `-bin`, the corresponding values must
                    be of type `bytes`.

            Returns:
                ~.prediction_service.EmbedContentResponse:
                    Response message for
                [PredictionService.EmbedContent][google.cloud.aiplatform.v1beta1.PredictionService.EmbedContent].

            """

            http_options = _BasePredictionServiceRestTransport._BaseEmbedContent._get_http_options()

            request, metadata = self._interceptor.pre_embed_content(request, metadata)
            transcoded_request = _BasePredictionServiceRestTransport._BaseEmbedContent._get_transcoded_request(http_options, request)

            body = _BasePredictionServiceRestTransport._BaseEmbedContent._get_request_body_json(transcoded_request)

            # Jsonify the query params
            query_params = _BasePredictionServiceRestTransport._BaseEmbedContent._get_query_params_json(transcoded_request)

            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(logging.DEBUG):  # pragma: NO COVER
                request_url = "{host}{uri}".format(host=self._host, uri=transcoded_request['uri'])
                method = transcoded_request['method']
                try:
                    request_payload = type(request).to_json(request)
                except:
                    request_payload = None
                http_request = {
                  "payload": request_payload,
                  "requestMethod": method,
                  "requestUrl": request_url,
                  "headers": dict(metadata),
                }
                _LOGGER.debug(
                    f"Sending request for google.cloud.aiplatform_v1beta1.PredictionServiceClient.EmbedContent",
                    extra = {
                        "serviceName": "google.cloud.aiplatform.v1beta1.PredictionService",
                        "rpcName": "EmbedContent",
                        "httpRequest": http_request,
                        "metadata": http_request["headers"],
                    },
                )

            # Send the request
            response = PredictionServiceRestTransport._EmbedContent._get_response(self._host, metadata, query_params, self._session, timeout, transcoded_request, body)

            # In case of error, raise the appropriate core_exceptions.GoogleAPICallError exception
            # subclass.
            if response.status_code >= 400:
                raise core_exceptions.from_http_response(response)

            # Return the response
            resp = prediction_service.EmbedContentResponse()
            pb_resp = prediction_service.EmbedContentResponse.pb(resp)

            json_format.Parse(response.content, pb_resp, ignore_unknown_fields=True)

            resp = self._interceptor.post_embed_content(resp)
            response_metadata = [(k, str(v)) for k, v in response.headers.items()]
            resp, _ = self._interceptor.post_embed_content_with_metadata(resp, response_metadata)
            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(logging.DEBUG):  # pragma: NO COVER
                try:
                    response_payload = prediction_service.EmbedContentResponse.to_json(response)
                except:
                    response_payload = None
                http_response = {
                "payload": response_payload,
                "headers":  dict(response.headers),
                "status": response.status_code,
                }
                _LOGGER.debug(
                    "Received response for google.cloud.aiplatform_v1beta1.PredictionServiceClient.embed_content",
                    extra = {
                        "serviceName": "google.cloud.aiplatform.v1beta1.PredictionService",
                        "rpcName": "EmbedContent",
                        "metadata": http_response["headers"],
                        "httpResponse": http_response,
                    },
                )
            return resp

    class _Explain(_BasePredictionServiceRestTransport._BaseExplain, PredictionServiceRestStub):
        def __hash__(self):
            return hash("PredictionServiceRestTransport.Explain")

        @staticmethod
        def _get_response(
            host,
            metadata,
            query_params,
            session,
            timeout,
            transcoded_request,
            body=None):

            uri = transcoded_request['uri']
            method = transcoded_request['method']
            headers = dict(metadata)
            headers['Content-Type'] = 'application/json'
            response = getattr(session, method)(
                "{host}{uri}".format(host=host, uri=uri),
                timeout=timeout,
                headers=headers,
                params=rest_helpers.flatten_query_params(query_params, strict=True),
                data=body,
                )
            return response

        def __call__(self,
                request: prediction_service.ExplainRequest, *,
                retry: OptionalRetry=gapic_v1.method.DEFAULT,
                timeout: Optional[float]=None,
                metadata: Sequence[Tuple[str, Union[str, bytes]]]=(),
                ) -> prediction_service.ExplainResponse:
            r"""Call the explain method over HTTP.

            Args:
                request (~.prediction_service.ExplainRequest):
                    The request object. Request message for
                [PredictionService.Explain][google.cloud.aiplatform.v1beta1.PredictionService.Explain].
                retry (google.api_core.retry.Retry): Designation of what errors, if any,
                    should be retried.
                timeout (float): The timeout for this request.
                metadata (Sequence[Tuple[str, Union[str, bytes]]]): Key/value pairs which should be
                    sent along with the request as metadata. Normally, each value must be of type `str`,
                    but for metadata keys ending with the suffix `-bin`, the corresponding values must
                    be of type `bytes`.

            Returns:
                ~.prediction_service.ExplainResponse:
                    Response message for
                [PredictionService.Explain][google.cloud.aiplatform.v1beta1.PredictionService.Explain].

            """

            http_options = _BasePredictionServiceRestTransport._BaseExplain._get_http_options()

            request, metadata = self._interceptor.pre_explain(request, metadata)
            transcoded_request = _BasePredictionServiceRestTransport._BaseExplain._get_transcoded_request(http_options, request)

            body = _BasePredictionServiceRestTransport._BaseExplain._get_request_body_json(transcoded_request)

            # Jsonify the query params
            query_params = _BasePredictionServiceRestTransport._BaseExplain._get_query_params_json(transcoded_request)

            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(logging.DEBUG):  # pragma: NO COVER
                request_url = "{host}{uri}".format(host=self._host, uri=transcoded_request['uri'])
                method = transcoded_request['method']
                try:
                    request_payload = type(request).to_json(request)
                except:
                    request_payload = None
                http_request = {
                  "payload": request_payload,
                  "requestMethod": method,
                  "requestUrl": request_url,
                  "headers": dict(metadata),
                }
                _LOGGER.debug(
                    f"Sending request for google.cloud.aiplatform_v1beta1.PredictionServiceClient.Explain",
                    extra = {
                        "serviceName": "google.cloud.aiplatform.v1beta1.PredictionService",
                        "rpcName": "Explain",
                        "httpRequest": http_request,
                        "metadata": http_request["headers"],
                    },
                )

            # Send the request
            response = PredictionServiceRestTransport._Explain._get_response(self._host, metadata, query_params, self._session, timeout, transcoded_request, body)

            # In case of error, raise the appropriate core_exceptions.GoogleAPICallError exception
            # subclass.
            if response.status_code >= 400:
                raise core_exceptions.from_http_response(response)

            # Return the response
            resp = prediction_service.ExplainResponse()
            pb_resp = prediction_service.ExplainResponse.pb(resp)

            json_format.Parse(response.content, pb_resp, ignore_unknown_fields=True)

            resp = self._interceptor.post_explain(resp)
            response_metadata = [(k, str(v)) for k, v in response.headers.items()]
            resp, _ = self._interceptor.post_explain_with_metadata(resp, response_metadata)
            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(logging.DEBUG):  # pragma: NO COVER
                try:
                    response_payload = prediction_service.ExplainResponse.to_json(response)
                except:
                    response_payload = None
                http_response = {
                "payload": response_payload,
                "headers":  dict(response.headers),
                "status": response.status_code,
                }
                _LOGGER.debug(
                    "Received response for google.cloud.aiplatform_v1beta1.PredictionServiceClient.explain",
                    extra = {
                        "serviceName": "google.cloud.aiplatform.v1beta1.PredictionService",
                        "rpcName": "Explain",
                        "metadata": http_response["headers"],
                        "httpResponse": http_response,
                    },
                )
            return resp

    class _FetchPredictOperation(_BasePredictionServiceRestTransport._BaseFetchPredictOperation, PredictionServiceRestStub):
        def __hash__(self):
            return hash("PredictionServiceRestTransport.FetchPredictOperation")

        @staticmethod
        def _get_response(
            host,
            metadata,
            query_params,
            session,
            timeout,
            transcoded_request,
            body=None):

            uri = transcoded_request['uri']
            method = transcoded_request['method']
            headers = dict(metadata)
            headers['Content-Type'] = 'application/json'
            response = getattr(session, method)(
                "{host}{uri}".format(host=host, uri=uri),
                timeout=timeout,
                headers=headers,
                params=rest_helpers.flatten_query_params(query_params, strict=True),
                data=body,
                )
            return response

        def __call__(self,
                request: prediction_service.FetchPredictOperationRequest, *,
                retry: OptionalRetry=gapic_v1.method.DEFAULT,
                timeout: Optional[float]=None,
                metadata: Sequence[Tuple[str, Union[str, bytes]]]=(),
                ) -> operations_pb2.Operation:
            r"""Call the fetch predict operation method over HTTP.

            Args:
                request (~.prediction_service.FetchPredictOperationRequest):
                    The request object. Request message for
                [PredictionService.FetchPredictOperation][google.cloud.aiplatform.v1beta1.PredictionService.FetchPredictOperation].
                retry (google.api_core.retry.Retry): Designation of what errors, if any,
                    should be retried.
                timeout (float): The timeout for this request.
                metadata (Sequence[Tuple[str, Union[str, bytes]]]): Key/value pairs which should be
                    sent along with the request as metadata. Normally, each value must be of type `str`,
                    but for metadata keys ending with the suffix `-bin`, the corresponding values must
                    be of type `bytes`.

            Returns:
                ~.operations_pb2.Operation:
                    This resource represents a
                long-running operation that is the
                result of a network API call.

            """

            http_options = _BasePredictionServiceRestTransport._BaseFetchPredictOperation._get_http_options()

            request, metadata = self._interceptor.pre_fetch_predict_operation(request, metadata)
            transcoded_request = _BasePredictionServiceRestTransport._BaseFetchPredictOperation._get_transcoded_request(http_options, request)

            body = _BasePredictionServiceRestTransport._BaseFetchPredictOperation._get_request_body_json(transcoded_request)

            # Jsonify the query params
            query_params = _BasePredictionServiceRestTransport._BaseFetchPredictOperation._get_query_params_json(transcoded_request)

            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(logging.DEBUG):  # pragma: NO COVER
                request_url = "{host}{uri}".format(host=self._host, uri=transcoded_request['uri'])
                method = transcoded_request['method']
                try:
                    request_payload = json_format.MessageToJson(request)
                except:
                    request_payload = None
                http_request = {
                  "payload": request_payload,
                  "requestMethod": method,
                  "requestUrl": request_url,
                  "headers": dict(metadata),
                }
                _LOGGER.debug(
                    f"Sending request for google.cloud.aiplatform_v1beta1.PredictionServiceClient.FetchPredictOperation",
                    extra = {
                        "serviceName": "google.cloud.aiplatform.v1beta1.PredictionService",
                        "rpcName": "FetchPredictOperation",
                        "httpRequest": http_request,
                        "metadata": http_request["headers"],
                    },
                )

            # Send the request
            response = PredictionServiceRestTransport._FetchPredictOperation._get_response(self._host, metadata, query_params, self._session, timeout, transcoded_request, body)

            # In case of error, raise the appropriate core_exceptions.GoogleAPICallError exception
            # subclass.
            if response.status_code >= 400:
                raise core_exceptions.from_http_response(response)

            # Return the response
            resp = operations_pb2.Operation()
            pb_resp = resp

            json_format.Parse(response.content, pb_resp, ignore_unknown_fields=True)

            resp = self._interceptor.post_fetch_predict_operation(resp)
            response_metadata = [(k, str(v)) for k, v in response.headers.items()]
            resp, _ = self._interceptor.post_fetch_predict_operation_with_metadata(resp, response_metadata)
            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(logging.DEBUG):  # pragma: NO COVER
                try:
                    response_payload = json_format.MessageToJson(resp)
                except:
                    response_payload = None
                http_response = {
                "payload": response_payload,
                "headers":  dict(response.headers),
                "status": response.status_code,
                }
                _LOGGER.debug(
                    "Received response for google.cloud.aiplatform_v1beta1.PredictionServiceClient.fetch_predict_operation",
                    extra = {
                        "serviceName": "google.cloud.aiplatform.v1beta1.PredictionService",
                        "rpcName": "FetchPredictOperation",
                        "metadata": http_response["headers"],
                        "httpResponse": http_response,
                    },
                )
            return resp

    class _GenerateContent(_BasePredictionServiceRestTransport._BaseGenerateContent, PredictionServiceRestStub):
        def __hash__(self):
            return hash("PredictionServiceRestTransport.GenerateContent")

        @staticmethod
        def _get_response(
            host,
            metadata,
            query_params,
            session,
            timeout,
            transcoded_request,
            body=None):

            uri = transcoded_request['uri']
            method = transcoded_request['method']
            headers = dict(metadata)
            headers['Content-Type'] = 'application/json'
            response = getattr(session, method)(
                "{host}{uri}".format(host=host, uri=uri),
                timeout=timeout,
                headers=headers,
                params=rest_helpers.flatten_query_params(query_params, strict=True),
                data=body,
                )
            return response

        def __call__(self,
                request: prediction_service.GenerateContentRequest, *,
                retry: OptionalRetry=gapic_v1.method.DEFAULT,
                timeout: Optional[float]=None,
                metadata: Sequence[Tuple[str, Union[str, bytes]]]=(),
                ) -> prediction_service.GenerateContentResponse:
            r"""Call the generate content method over HTTP.

            Args:
                request (~.prediction_service.GenerateContentRequest):
                    The request object. Request message for [PredictionService.GenerateContent].
                retry (google.api_core.retry.Retry): Designation of what errors, if any,
                    should be retried.
                timeout (float): The timeout for this request.
                metadata (Sequence[Tuple[str, Union[str, bytes]]]): Key/value pairs which should be
                    sent along with the request as metadata. Normally, each value must be of type `str`,
                    but for metadata keys ending with the suffix `-bin`, the corresponding values must
                    be of type `bytes`.

            Returns:
                ~.prediction_service.GenerateContentResponse:
                    Response message for
                [PredictionService.GenerateContent].

            """

            http_options = _BasePredictionServiceRestTransport._BaseGenerateContent._get_http_options()

            request, metadata = self._interceptor.pre_generate_content(request, metadata)
            transcoded_request = _BasePredictionServiceRestTransport._BaseGenerateContent._get_transcoded_request(http_options, request)

            body = _BasePredictionServiceRestTransport._BaseGenerateContent._get_request_body_json(transcoded_request)

            # Jsonify the query params
            query_params = _BasePredictionServiceRestTransport._BaseGenerateContent._get_query_params_json(transcoded_request)

            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(logging.DEBUG):  # pragma: NO COVER
                request_url = "{host}{uri}".format(host=self._host, uri=transcoded_request['uri'])
                method = transcoded_request['method']
                try:
                    request_payload = type(request).to_json(request)
                except:
                    request_payload = None
                http_request = {
                  "payload": request_payload,
                  "requestMethod": method,
                  "requestUrl": request_url,
                  "headers": dict(metadata),
                }
                _LOGGER.debug(
                    f"Sending request for google.cloud.aiplatform_v1beta1.PredictionServiceClient.GenerateContent",
                    extra = {
                        "serviceName": "google.cloud.aiplatform.v1beta1.PredictionService",
                        "rpcName": "GenerateContent",
                        "httpRequest": http_request,
                        "metadata": http_request["headers"],
                    },
                )

            # Send the request
            response = PredictionServiceRestTransport._GenerateContent._get_response(self._host, metadata, query_params, self._session, timeout, transcoded_request, body)

            # In case of error, raise the appropriate core_exceptions.GoogleAPICallError exception
            # subclass.
            if response.status_code >= 400:
                raise core_exceptions.from_http_response(response)

            # Return the response
            resp = prediction_service.GenerateContentResponse()
            pb_resp = prediction_service.GenerateContentResponse.pb(resp)

            json_format.Parse(response.content, pb_resp, ignore_unknown_fields=True)

            resp = self._interceptor.post_generate_content(resp)
            response_metadata = [(k, str(v)) for k, v in response.headers.items()]
            resp, _ = self._interceptor.post_generate_content_with_metadata(resp, response_metadata)
            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(logging.DEBUG):  # pragma: NO COVER
                try:
                    response_payload = prediction_service.GenerateContentResponse.to_json(response)
                except:
                    response_payload = None
                http_response = {
                "payload": response_payload,
                "headers":  dict(response.headers),
                "status": response.status_code,
                }
                _LOGGER.debug(
                    "Received response for google.cloud.aiplatform_v1beta1.PredictionServiceClient.generate_content",
                    extra = {
                        "serviceName": "google.cloud.aiplatform.v1beta1.PredictionService",
                        "rpcName": "GenerateContent",
                        "metadata": http_response["headers"],
                        "httpResponse": http_response,
                    },
                )
            return resp

    class _Invoke(_BasePredictionServiceRestTransport._BaseInvoke, PredictionServiceRestStub):
        def __hash__(self):
            return hash("PredictionServiceRestTransport.Invoke")

        @staticmethod
        def _get_response(
            host,
            metadata,
            query_params,
            session,
            timeout,
            transcoded_request,
            body=None):

            uri = transcoded_request['uri']
            method = transcoded_request['method']
            headers = dict(metadata)
            headers['Content-Type'] = 'application/json'
            response = getattr(session, method)(
                "{host}{uri}".format(host=host, uri=uri),
                timeout=timeout,
                headers=headers,
                params=rest_helpers.flatten_query_params(query_params, strict=True),
                data=body,
                stream=True,
                )
            return response

        def __call__(self,
                request: prediction_service.InvokeRequest, *,
                retry: OptionalRetry=gapic_v1.method.DEFAULT,
                timeout: Optional[float]=None,
                metadata: Sequence[Tuple[str, Union[str, bytes]]]=(),
                ) -> rest_streaming.ResponseIterator:
            r"""Call the invoke method over HTTP.

            Args:
                request (~.prediction_service.InvokeRequest):
                    The request object. Request message for
                [PredictionService.Invoke][google.cloud.aiplatform.v1beta1.PredictionService.Invoke].
                retry (google.api_core.retry.Retry): Designation of what errors, if any,
                    should be retried.
                timeout (float): The timeout for this request.
                metadata (Sequence[Tuple[str, Union[str, bytes]]]): Key/value pairs which should be
                    sent along with the request as metadata. Normally, each value must be of type `str`,
                    but for metadata keys ending with the suffix `-bin`, the corresponding values must
                    be of type `bytes`.

            Returns:
                ~.httpbody_pb2.HttpBody:
                    Message that represents an arbitrary HTTP body. It
                should only be used for payload formats that can't be
                represented as JSON, such as raw binary or an HTML page.

                This message can be used both in streaming and
                non-streaming API methods in the request as well as the
                response.

                It can be used as a top-level request field, which is
                convenient if one wants to extract parameters from
                either the URL or HTTP template into the request fields
                and also want access to the raw HTTP body.

                Example:

                ::

                    message GetResourceRequest {
                      // A unique request id.
                      string request_id = 1;

                      // The raw HTTP body is bound to this field.
                      google.api.HttpBody http_body = 2;

                    }

                    service ResourceService {
                      rpc GetResource(GetResourceRequest)
                        returns (google.api.HttpBody);
                      rpc UpdateResource(google.api.HttpBody)
                        returns (google.protobuf.Empty);

                    }

                Example with streaming methods:

                ::

                    service CaldavService {
                      rpc GetCalendar(stream google.api.HttpBody)
                        returns (stream google.api.HttpBody);
                      rpc UpdateCalendar(stream google.api.HttpBody)
                        returns (stream google.api.HttpBody);

                    }

                Use of this type only changes how the request and
                response bodies are handled, all other features will
                continue to work unchanged.

            """

            http_options = _BasePredictionServiceRestTransport._BaseInvoke._get_http_options()

            request, metadata = self._interceptor.pre_invoke(request, metadata)
            transcoded_request = _BasePredictionServiceRestTransport._BaseInvoke._get_transcoded_request(http_options, request)

            body = _BasePredictionServiceRestTransport._BaseInvoke._get_request_body_json(transcoded_request)

            # Jsonify the query params
            query_params = _BasePredictionServiceRestTransport._BaseInvoke._get_query_params_json(transcoded_request)

            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(logging.DEBUG):  # pragma: NO COVER
                request_url = "{host}{uri}".format(host=self._host, uri=transcoded_request['uri'])
                method = transcoded_request['method']
                try:
                    request_payload = json_format.MessageToJson(request)
                except:
                    request_payload = None
                http_request = {
                  "payload": request_payload,
                  "requestMethod": method,
                  "requestUrl": request_url,
                  "headers": dict(metadata),
                }
                _LOGGER.debug(
                    f"Sending request for google.cloud.aiplatform_v1beta1.PredictionServiceClient.Invoke",
                    extra = {
                        "serviceName": "google.cloud.aiplatform.v1beta1.PredictionService",
                        "rpcName": "Invoke",
                        "httpRequest": http_request,
                        "metadata": http_request["headers"],
                    },
                )

            # Send the request
            response = PredictionServiceRestTransport._Invoke._get_response(self._host, metadata, query_params, self._session, timeout, transcoded_request, body)

            # In case of error, raise the appropriate core_exceptions.GoogleAPICallError exception
            # subclass.
            if response.status_code >= 400:
                raise core_exceptions.from_http_response(response)

            # Return the response
            resp = rest_streaming.ResponseIterator(response, httpbody_pb2.HttpBody)

            resp = self._interceptor.post_invoke(resp)
            response_metadata = [(k, str(v)) for k, v in response.headers.items()]
            resp, _ = self._interceptor.post_invoke_with_metadata(resp, response_metadata)
            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(logging.DEBUG):  # pragma: NO COVER
                http_response = {
                "headers":  dict(response.headers),
                "status": response.status_code,
                }
                _LOGGER.debug(
                    "Received response for google.cloud.aiplatform_v1beta1.PredictionServiceClient.invoke",
                    extra = {
                        "serviceName": "google.cloud.aiplatform.v1beta1.PredictionService",
                        "rpcName": "Invoke",
                        "metadata": http_response["headers"],
                        "httpResponse": http_response,
                    },
                )
            return resp

    class _Predict(_BasePredictionServiceRestTransport._BasePredict, PredictionServiceRestStub):
        def __hash__(self):
            return hash("PredictionServiceRestTransport.Predict")

        @staticmethod
        def _get_response(
            host,
            metadata,
            query_params,
            session,
            timeout,
            transcoded_request,
            body=None):

            uri = transcoded_request['uri']
            method = transcoded_request['method']
            headers = dict(metadata)
            headers['Content-Type'] = 'application/json'
            response = getattr(session, method)(
                "{host}{uri}".format(host=host, uri=uri),
                timeout=timeout,
                headers=headers,
                params=rest_helpers.flatten_query_params(query_params, strict=True),
                data=body,
                )
            return response

        def __call__(self,
                request: prediction_service.PredictRequest, *,
                retry: OptionalRetry=gapic_v1.method.DEFAULT,
                timeout: Optional[float]=None,
                metadata: Sequence[Tuple[str, Union[str, bytes]]]=(),
                ) -> prediction_service.PredictResponse:
            r"""Call the predict method over HTTP.

            Args:
                request (~.prediction_service.PredictRequest):
                    The request object. Request message for
                [PredictionService.Predict][google.cloud.aiplatform.v1beta1.PredictionService.Predict].
                retry (google.api_core.retry.Retry): Designation of what errors, if any,
                    should be retried.
                timeout (float): The timeout for this request.
                metadata (Sequence[Tuple[str, Union[str, bytes]]]): Key/value pairs which should be
                    sent along with the request as metadata. Normally, each value must be of type `str`,
                    but for metadata keys ending with the suffix `-bin`, the corresponding values must
                    be of type `bytes`.

            Returns:
                ~.prediction_service.PredictResponse:
                    Response message for
                [PredictionService.Predict][google.cloud.aiplatform.v1beta1.PredictionService.Predict].

            """

            http_options = _BasePredictionServiceRestTransport._BasePredict._get_http_options()

            request, metadata = self._interceptor.pre_predict(request, metadata)
            transcoded_request = _BasePredictionServiceRestTransport._BasePredict._get_transcoded_request(http_options, request)

            body = _BasePredictionServiceRestTransport._BasePredict._get_request_body_json(transcoded_request)

            # Jsonify the query params
            query_params = _BasePredictionServiceRestTransport._BasePredict._get_query_params_json(transcoded_request)

            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(logging.DEBUG):  # pragma: NO COVER
                request_url = "{host}{uri}".format(host=self._host, uri=transcoded_request['uri'])
                method = transcoded_request['method']
                try:
                    request_payload = type(request).to_json(request)
                except:
                    request_payload = None
                http_request = {
                  "payload": request_payload,
                  "requestMethod": method,
                  "requestUrl": request_url,
                  "headers": dict(metadata),
                }
                _LOGGER.debug(
                    f"Sending request for google.cloud.aiplatform_v1beta1.PredictionServiceClient.Predict",
                    extra = {
                        "serviceName": "google.cloud.aiplatform.v1beta1.PredictionService",
                        "rpcName": "Predict",
                        "httpRequest": http_request,
                        "metadata": http_request["headers"],
                    },
                )

            # Send the request
            response = PredictionServiceRestTransport._Predict._get_response(self._host, metadata, query_params, self._session, timeout, transcoded_request, body)

            # In case of error, raise the appropriate core_exceptions.GoogleAPICallError exception
            # subclass.
            if response.status_code >= 400:
                raise core_exceptions.from_http_response(response)

            # Return the response
            resp = prediction_service.PredictResponse()
            pb_resp = prediction_service.PredictResponse.pb(resp)

            json_format.Parse(response.content, pb_resp, ignore_unknown_fields=True)

            resp = self._interceptor.post_predict(resp)
            response_metadata = [(k, str(v)) for k, v in response.headers.items()]
            resp, _ = self._interceptor.post_predict_with_metadata(resp, response_metadata)
            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(logging.DEBUG):  # pragma: NO COVER
                try:
                    response_payload = prediction_service.PredictResponse.to_json(response)
                except:
                    response_payload = None
                http_response = {
                "payload": response_payload,
                "headers":  dict(response.headers),
                "status": response.status_code,
                }
                _LOGGER.debug(
                    "Received response for google.cloud.aiplatform_v1beta1.PredictionServiceClient.predict",
                    extra = {
                        "serviceName": "google.cloud.aiplatform.v1beta1.PredictionService",
                        "rpcName": "Predict",
                        "metadata": http_response["headers"],
                        "httpResponse": http_response,
                    },
                )
            return resp

    class _PredictLongRunning(_BasePredictionServiceRestTransport._BasePredictLongRunning, PredictionServiceRestStub):
        def __hash__(self):
            return hash("PredictionServiceRestTransport.PredictLongRunning")

        @staticmethod
        def _get_response(
            host,
            metadata,
            query_params,
            session,
            timeout,
            transcoded_request,
            body=None):

            uri = transcoded_request['uri']
            method = transcoded_request['method']
            headers = dict(metadata)
            headers['Content-Type'] = 'application/json'
            response = getattr(session, method)(
                "{host}{uri}".format(host=host, uri=uri),
                timeout=timeout,
                headers=headers,
                params=rest_helpers.flatten_query_params(query_params, strict=True),
                data=body,
                )
            return response

        def __call__(self,
                request: prediction_service.PredictLongRunningRequest, *,
                retry: OptionalRetry=gapic_v1.method.DEFAULT,
                timeout: Optional[float]=None,
                metadata: Sequence[Tuple[str, Union[str, bytes]]]=(),
                ) -> operations_pb2.Operation:
            r"""Call the predict long running method over HTTP.

            Args:
                request (~.prediction_service.PredictLongRunningRequest):
                    The request object. Request message for
                [PredictionService.PredictLongRunning][google.cloud.aiplatform.v1beta1.PredictionService.PredictLongRunning].
                retry (google.api_core.retry.Retry): Designation of what errors, if any,
                    should be retried.
                timeout (float): The timeout for this request.
                metadata (Sequence[Tuple[str, Union[str, bytes]]]): Key/value pairs which should be
                    sent along with the request as metadata. Normally, each value must be of type `str`,
                    but for metadata keys ending with the suffix `-bin`, the corresponding values must
                    be of type `bytes`.

            Returns:
                ~.operations_pb2.Operation:
                    This resource represents a
                long-running operation that is the
                result of a network API call.

            """

            http_options = _BasePredictionServiceRestTransport._BasePredictLongRunning._get_http_options()

            request, metadata = self._interceptor.pre_predict_long_running(request, metadata)
            transcoded_request = _BasePredictionServiceRestTransport._BasePredictLongRunning._get_transcoded_request(http_options, request)

            body = _BasePredictionServiceRestTransport._BasePredictLongRunning._get_request_body_json(transcoded_request)

            # Jsonify the query params
            query_params = _BasePredictionServiceRestTransport._BasePredictLongRunning._get_query_params_json(transcoded_request)

            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(logging.DEBUG):  # pragma: NO COVER
                request_url = "{host}{uri}".format(host=self._host, uri=transcoded_request['uri'])
                method = transcoded_request['method']
                try:
                    request_payload = json_format.MessageToJson(request)
                except:
                    request_payload = None
                http_request = {
                  "payload": request_payload,
                  "requestMethod": method,
                  "requestUrl": request_url,
                  "headers": dict(metadata),
                }
                _LOGGER.debug(
                    f"Sending request for google.cloud.aiplatform_v1beta1.PredictionServiceClient.PredictLongRunning",
                    extra = {
                        "serviceName": "google.cloud.aiplatform.v1beta1.PredictionService",
                        "rpcName": "PredictLongRunning",
                        "httpRequest": http_request,
                        "metadata": http_request["headers"],
                    },
                )

            # Send the request
            response = PredictionServiceRestTransport._PredictLongRunning._get_response(self._host, metadata, query_params, self._session, timeout, transcoded_request, body)

            # In case of error, raise the appropriate core_exceptions.GoogleAPICallError exception
            # subclass.
            if response.status_code >= 400:
                raise core_exceptions.from_http_response(response)

            # Return the response
            resp = operations_pb2.Operation()
            pb_resp = resp

            json_format.Parse(response.content, pb_resp, ignore_unknown_fields=True)

            resp = self._interceptor.post_predict_long_running(resp)
            response_metadata = [(k, str(v)) for k, v in response.headers.items()]
            resp, _ = self._interceptor.post_predict_long_running_with_metadata(resp, response_metadata)
            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(logging.DEBUG):  # pragma: NO COVER
                try:
                    response_payload = json_format.MessageToJson(resp)
                except:
                    response_payload = None
                http_response = {
                "payload": response_payload,
                "headers":  dict(response.headers),
                "status": response.status_code,
                }
                _LOGGER.debug(
                    "Received response for google.cloud.aiplatform_v1beta1.PredictionServiceClient.predict_long_running",
                    extra = {
                        "serviceName": "google.cloud.aiplatform.v1beta1.PredictionService",
                        "rpcName": "PredictLongRunning",
                        "metadata": http_response["headers"],
                        "httpResponse": http_response,
                    },
                )
            return resp

    class _RawPredict(_BasePredictionServiceRestTransport._BaseRawPredict, PredictionServiceRestStub):
        def __hash__(self):
            return hash("PredictionServiceRestTransport.RawPredict")

        @staticmethod
        def _get_response(
            host,
            metadata,
            query_params,
            session,
            timeout,
            transcoded_request,
            body=None):

            uri = transcoded_request['uri']
            method = transcoded_request['method']
            headers = dict(metadata)
            headers['Content-Type'] = 'application/json'
            response = getattr(session, method)(
                "{host}{uri}".format(host=host, uri=uri),
                timeout=timeout,
                headers=headers,
                params=rest_helpers.flatten_query_params(query_params, strict=True),
                data=body,
                )
            return response

        def __call__(self,
                request: prediction_service.RawPredictRequest, *,
                retry: OptionalRetry=gapic_v1.method.DEFAULT,
                timeout: Optional[float]=None,
                metadata: Sequence[Tuple[str, Union[str, bytes]]]=(),
                ) -> httpbody_pb2.HttpBody:
            r"""Call the raw predict method over HTTP.

            Args:
                request (~.prediction_service.RawPredictRequest):
                    The request object. Request message for
                [PredictionService.RawPredict][google.cloud.aiplatform.v1beta1.PredictionService.RawPredict].
                retry (google.api_core.retry.Retry): Designation of what errors, if any,
                    should be retried.
                timeout (float): The timeout for this request.
                metadata (Sequence[Tuple[str, Union[str, bytes]]]): Key/value pairs which should be
                    sent along with the request as metadata. Normally, each value must be of type `str`,
                    but for metadata keys ending with the suffix `-bin`, the corresponding values must
                    be of type `bytes`.

            Returns:
                ~.httpbody_pb2.HttpBody:
                    Message that represents an arbitrary HTTP body. It
                should only be used for payload formats that can't be
                represented as JSON, such as raw binary or an HTML page.

                This message can be used both in streaming and
                non-streaming API methods in the request as well as the
                response.

                It can be used as a top-level request field, which is
                convenient if one wants to extract parameters from
                either the URL or HTTP template into the request fields
                and also want access to the raw HTTP body.

                Example:

                ::

                    message GetResourceRequest {
                      // A unique request id.
                      string request_id = 1;

                      // The raw HTTP body is bound to this field.
                      google.api.HttpBody http_body = 2;

                    }

                    service ResourceService {
                      rpc GetResource(GetResourceRequest)
                        returns (google.api.HttpBody);
                      rpc UpdateResource(google.api.HttpBody)
                        returns (google.protobuf.Empty);

                    }

                Example with streaming methods:

                ::

                    service CaldavService {
                      rpc GetCalendar(stream google.api.HttpBody)
                        returns (stream google.api.HttpBody);
                      rpc UpdateCalendar(stream google.api.HttpBody)
                        returns (stream google.api.HttpBody);

                    }

                Use of this type only changes how the request and
                response bodies are handled, all other features will
                continue to work unchanged.

            """

            http_options = _BasePredictionServiceRestTransport._BaseRawPredict._get_http_options()

            request, metadata = self._interceptor.pre_raw_predict(request, metadata)
            transcoded_request = _BasePredictionServiceRestTransport._BaseRawPredict._get_transcoded_request(http_options, request)

            body = _BasePredictionServiceRestTransport._BaseRawPredict._get_request_body_json(transcoded_request)

            # Jsonify the query params
            query_params = _BasePredictionServiceRestTransport._BaseRawPredict._get_query_params_json(transcoded_request)

            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(logging.DEBUG):  # pragma: NO COVER
                request_url = "{host}{uri}".format(host=self._host, uri=transcoded_request['uri'])
                method = transcoded_request['method']
                try:
                    request_payload = json_format.MessageToJson(request)
                except:
                    request_payload = None
                http_request = {
                  "payload": request_payload,
                  "requestMethod": method,
                  "requestUrl": request_url,
                  "headers": dict(metadata),
                }
                _LOGGER.debug(
                    f"Sending request for google.cloud.aiplatform_v1beta1.PredictionServiceClient.RawPredict",
                    extra = {
                        "serviceName": "google.cloud.aiplatform.v1beta1.PredictionService",
                        "rpcName": "RawPredict",
                        "httpRequest": http_request,
                        "metadata": http_request["headers"],
                    },
                )

            # Send the request
            response = PredictionServiceRestTransport._RawPredict._get_response(self._host, metadata, query_params, self._session, timeout, transcoded_request, body)

            # In case of error, raise the appropriate core_exceptions.GoogleAPICallError exception
            # subclass.
            if response.status_code >= 400:
                raise core_exceptions.from_http_response(response)

            # Return the response
            resp = httpbody_pb2.HttpBody()
            pb_resp = resp

            json_format.Parse(response.content, pb_resp, ignore_unknown_fields=True)

            resp = self._interceptor.post_raw_predict(resp)
            response_metadata = [(k, str(v)) for k, v in response.headers.items()]
            resp, _ = self._interceptor.post_raw_predict_with_metadata(resp, response_metadata)
            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(logging.DEBUG):  # pragma: NO COVER
                try:
                    response_payload = json_format.MessageToJson(resp)
                except:
                    response_payload = None
                http_response = {
                "payload": response_payload,
                "headers":  dict(response.headers),
                "status": response.status_code,
                }
                _LOGGER.debug(
                    "Received response for google.cloud.aiplatform_v1beta1.PredictionServiceClient.raw_predict",
                    extra = {
                        "serviceName": "google.cloud.aiplatform.v1beta1.PredictionService",
                        "rpcName": "RawPredict",
                        "metadata": http_response["headers"],
                        "httpResponse": http_response,
                    },
                )
            return resp

    class _ServerStreamingPredict(_BasePredictionServiceRestTransport._BaseServerStreamingPredict, PredictionServiceRestStub):
        def __hash__(self):
            return hash("PredictionServiceRestTransport.ServerStreamingPredict")

        @staticmethod
        def _get_response(
            host,
            metadata,
            query_params,
            session,
            timeout,
            transcoded_request,
            body=None):

            uri = transcoded_request['uri']
            method = transcoded_request['method']
            headers = dict(metadata)
            headers['Content-Type'] = 'application/json'
            response = getattr(session, method)(
                "{host}{uri}".format(host=host, uri=uri),
                timeout=timeout,
                headers=headers,
                params=rest_helpers.flatten_query_params(query_params, strict=True),
                data=body,
                stream=True,
                )
            return response

        def __call__(self,
                request: prediction_service.StreamingPredictRequest, *,
                retry: OptionalRetry=gapic_v1.method.DEFAULT,
                timeout: Optional[float]=None,
                metadata: Sequence[Tuple[str, Union[str, bytes]]]=(),
                ) -> rest_streaming.ResponseIterator:
            r"""Call the server streaming predict method over HTTP.

            Args:
                request (~.prediction_service.StreamingPredictRequest):
                    The request object. Request message for
                [PredictionService.StreamingPredict][google.cloud.aiplatform.v1beta1.PredictionService.StreamingPredict].

                The first message must contain
                [endpoint][google.cloud.aiplatform.v1beta1.StreamingPredictRequest.endpoint]
                field and optionally [input][]. The subsequent messages
                must contain [input][].
                retry (google.api_core.retry.Retry): Designation of what errors, if any,
                    should be retried.
                timeout (float): The timeout for this request.
                metadata (Sequence[Tuple[str, Union[str, bytes]]]): Key/value pairs which should be
                    sent along with the request as metadata. Normally, each value must be of type `str`,
                    but for metadata keys ending with the suffix `-bin`, the corresponding values must
                    be of type `bytes`.

            Returns:
                ~.prediction_service.StreamingPredictResponse:
                    Response message for
                [PredictionService.StreamingPredict][google.cloud.aiplatform.v1beta1.PredictionService.StreamingPredict].

            """

            http_options = _BasePredictionServiceRestTransport._BaseServerStreamingPredict._get_http_options()

            request, metadata = self._interceptor.pre_server_streaming_predict(request, metadata)
            transcoded_request = _BasePredictionServiceRestTransport._BaseServerStreamingPredict._get_transcoded_request(http_options, request)

            body = _BasePredictionServiceRestTransport._BaseServerStreamingPredict._get_request_body_json(transcoded_request)

            # Jsonify the query params
            query_params = _BasePredictionServiceRestTransport._BaseServerStreamingPredict._get_query_params_json(transcoded_request)

            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(logging.DEBUG):  # pragma: NO COVER
                request_url = "{host}{uri}".format(host=self._host, uri=transcoded_request['uri'])
                method = transcoded_request['method']
                try:
                    request_payload = type(request).to_json(request)
                except:
                    request_payload = None
                http_request = {
                  "payload": request_payload,
                  "requestMethod": method,
                  "requestUrl": request_url,
                  "headers": dict(metadata),
                }
                _LOGGER.debug(
                    f"Sending request for google.cloud.aiplatform_v1beta1.PredictionServiceClient.ServerStreamingPredict",
                    extra = {
                        "serviceName": "google.cloud.aiplatform.v1beta1.PredictionService",
                        "rpcName": "ServerStreamingPredict",
                        "httpRequest": http_request,
                        "metadata": http_request["headers"],
                    },
                )

            # Send the request
            response = PredictionServiceRestTransport._ServerStreamingPredict._get_response(self._host, metadata, query_params, self._session, timeout, transcoded_request, body)

            # In case of error, raise the appropriate core_exceptions.GoogleAPICallError exception
            # subclass.
            if response.status_code >= 400:
                raise core_exceptions.from_http_response(response)

            # Return the response
            resp = rest_streaming.ResponseIterator(response, prediction_service.StreamingPredictResponse)

            resp = self._interceptor.post_server_streaming_predict(resp)
            response_metadata = [(k, str(v)) for k, v in response.headers.items()]
            resp, _ = self._interceptor.post_server_streaming_predict_with_metadata(resp, response_metadata)
            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(logging.DEBUG):  # pragma: NO COVER
                http_response = {
                "headers":  dict(response.headers),
                "status": response.status_code,
                }
                _LOGGER.debug(
                    "Received response for google.cloud.aiplatform_v1beta1.PredictionServiceClient.server_streaming_predict",
                    extra = {
                        "serviceName": "google.cloud.aiplatform.v1beta1.PredictionService",
                        "rpcName": "ServerStreamingPredict",
                        "metadata": http_response["headers"],
                        "httpResponse": http_response,
                    },
                )
            return resp

    class _StreamDirectPredict(_BasePredictionServiceRestTransport._BaseStreamDirectPredict, PredictionServiceRestStub):
        def __hash__(self):
            return hash("PredictionServiceRestTransport.StreamDirectPredict")

        def __call__(self,
                request: prediction_service.StreamDirectPredictRequest, *,
                retry: OptionalRetry=gapic_v1.method.DEFAULT,
                timeout: Optional[float]=None,
                metadata: Sequence[Tuple[str, Union[str, bytes]]]=(),
                ) -> rest_streaming.ResponseIterator:
            raise NotImplementedError(
                "Method StreamDirectPredict is not available over REST transport"
            )
    class _StreamDirectRawPredict(_BasePredictionServiceRestTransport._BaseStreamDirectRawPredict, PredictionServiceRestStub):
        def __hash__(self):
            return hash("PredictionServiceRestTransport.StreamDirectRawPredict")

        def __call__(self,
                request: prediction_service.StreamDirectRawPredictRequest, *,
                retry: OptionalRetry=gapic_v1.method.DEFAULT,
                timeout: Optional[float]=None,
                metadata: Sequence[Tuple[str, Union[str, bytes]]]=(),
                ) -> rest_streaming.ResponseIterator:
            raise NotImplementedError(
                "Method StreamDirectRawPredict is not available over REST transport"
            )
    class _StreamGenerateContent(_BasePredictionServiceRestTransport._BaseStreamGenerateContent, PredictionServiceRestStub):
        def __hash__(self):
            return hash("PredictionServiceRestTransport.StreamGenerateContent")

        @staticmethod
        def _get_response(
            host,
            metadata,
            query_params,
            session,
            timeout,
            transcoded_request,
            body=None):

            uri = transcoded_request['uri']
            method = transcoded_request['method']
            headers = dict(metadata)
            headers['Content-Type'] = 'application/json'
            response = getattr(session, method)(
                "{host}{uri}".format(host=host, uri=uri),
                timeout=timeout,
                headers=headers,
                params=rest_helpers.flatten_query_params(query_params, strict=True),
                data=body,
                stream=True,
                )
            return response

        def __call__(self,
                request: prediction_service.GenerateContentRequest, *,
                retry: OptionalRetry=gapic_v1.method.DEFAULT,
                timeout: Optional[float]=None,
                metadata: Sequence[Tuple[str, Union[str, bytes]]]=(),
                ) -> rest_streaming.ResponseIterator:
            r"""Call the stream generate content method over HTTP.

            Args:
                request (~.prediction_service.GenerateContentRequest):
                    The request object. Request message for [PredictionService.GenerateContent].
                retry (google.api_core.retry.Retry): Designation of what errors, if any,
                    should be retried.
                timeout (float): The timeout for this request.
                metadata (Sequence[Tuple[str, Union[str, bytes]]]): Key/value pairs which should be
                    sent along with the request as metadata. Normally, each value must be of type `str`,
                    but for metadata keys ending with the suffix `-bin`, the corresponding values must
                    be of type `bytes`.

            Returns:
                ~.prediction_service.GenerateContentResponse:
                    Response message for
                [PredictionService.GenerateContent].

            """

            http_options = _BasePredictionServiceRestTransport._BaseStreamGenerateContent._get_http_options()

            request, metadata = self._interceptor.pre_stream_generate_content(request, metadata)
            transcoded_request = _BasePredictionServiceRestTransport._BaseStreamGenerateContent._get_transcoded_request(http_options, request)

            body = _BasePredictionServiceRestTransport._BaseStreamGenerateContent._get_request_body_json(transcoded_request)

            # Jsonify the query params
            query_params = _BasePredictionServiceRestTransport._BaseStreamGenerateContent._get_query_params_json(transcoded_request)

            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(logging.DEBUG):  # pragma: NO COVER
                request_url = "{host}{uri}".format(host=self._host, uri=transcoded_request['uri'])
                method = transcoded_request['method']
                try:
                    request_payload = type(request).to_json(request)
                except:
                    request_payload = None
                http_request = {
                  "payload": request_payload,
                  "requestMethod": method,
                  "requestUrl": request_url,
                  "headers": dict(metadata),
                }
                _LOGGER.debug(
                    f"Sending request for google.cloud.aiplatform_v1beta1.PredictionServiceClient.StreamGenerateContent",
                    extra = {
                        "serviceName": "google.cloud.aiplatform.v1beta1.PredictionService",
                        "rpcName": "StreamGenerateContent",
                        "httpRequest": http_request,
                        "metadata": http_request["headers"],
                    },
                )

            # Send the request
            response = PredictionServiceRestTransport._StreamGenerateContent._get_response(self._host, metadata, query_params, self._session, timeout, transcoded_request, body)

            # In case of error, raise the appropriate core_exceptions.GoogleAPICallError exception
            # subclass.
            if response.status_code >= 400:
                raise core_exceptions.from_http_response(response)

            # Return the response
            resp = rest_streaming.ResponseIterator(response, prediction_service.GenerateContentResponse)

            resp = self._interceptor.post_stream_generate_content(resp)
            response_metadata = [(k, str(v)) for k, v in response.headers.items()]
            resp, _ = self._interceptor.post_stream_generate_content_with_metadata(resp, response_metadata)
            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(logging.DEBUG):  # pragma: NO COVER
                http_response = {
                "headers":  dict(response.headers),
                "status": response.status_code,
                }
                _LOGGER.debug(
                    "Received response for google.cloud.aiplatform_v1beta1.PredictionServiceClient.stream_generate_content",
                    extra = {
                        "serviceName": "google.cloud.aiplatform.v1beta1.PredictionService",
                        "rpcName": "StreamGenerateContent",
                        "metadata": http_response["headers"],
                        "httpResponse": http_response,
                    },
                )
            return resp

    class _StreamingPredict(_BasePredictionServiceRestTransport._BaseStreamingPredict, PredictionServiceRestStub):
        def __hash__(self):
            return hash("PredictionServiceRestTransport.StreamingPredict")

        def __call__(self,
                request: prediction_service.StreamingPredictRequest, *,
                retry: OptionalRetry=gapic_v1.method.DEFAULT,
                timeout: Optional[float]=None,
                metadata: Sequence[Tuple[str, Union[str, bytes]]]=(),
                ) -> rest_streaming.ResponseIterator:
            raise NotImplementedError(
                "Method StreamingPredict is not available over REST transport"
            )
    class _StreamingRawPredict(_BasePredictionServiceRestTransport._BaseStreamingRawPredict, PredictionServiceRestStub):
        def __hash__(self):
            return hash("PredictionServiceRestTransport.StreamingRawPredict")

        def __call__(self,
                request: prediction_service.StreamingRawPredictRequest, *,
                retry: OptionalRetry=gapic_v1.method.DEFAULT,
                timeout: Optional[float]=None,
                metadata: Sequence[Tuple[str, Union[str, bytes]]]=(),
                ) -> rest_streaming.ResponseIterator:
            raise NotImplementedError(
                "Method StreamingRawPredict is not available over REST transport"
            )
    class _StreamRawPredict(_BasePredictionServiceRestTransport._BaseStreamRawPredict, PredictionServiceRestStub):
        def __hash__(self):
            return hash("PredictionServiceRestTransport.StreamRawPredict")

        @staticmethod
        def _get_response(
            host,
            metadata,
            query_params,
            session,
            timeout,
            transcoded_request,
            body=None):

            uri = transcoded_request['uri']
            method = transcoded_request['method']
            headers = dict(metadata)
            headers['Content-Type'] = 'application/json'
            response = getattr(session, method)(
                "{host}{uri}".format(host=host, uri=uri),
                timeout=timeout,
                headers=headers,
                params=rest_helpers.flatten_query_params(query_params, strict=True),
                data=body,
                stream=True,
                )
            return response

        def __call__(self,
                request: prediction_service.StreamRawPredictRequest, *,
                retry: OptionalRetry=gapic_v1.method.DEFAULT,
                timeout: Optional[float]=None,
                metadata: Sequence[Tuple[str, Union[str, bytes]]]=(),
                ) -> rest_streaming.ResponseIterator:
            r"""Call the stream raw predict method over HTTP.

            Args:
                request (~.prediction_service.StreamRawPredictRequest):
                    The request object. Request message for
                [PredictionService.StreamRawPredict][google.cloud.aiplatform.v1beta1.PredictionService.StreamRawPredict].
                retry (google.api_core.retry.Retry): Designation of what errors, if any,
                    should be retried.
                timeout (float): The timeout for this request.
                metadata (Sequence[Tuple[str, Union[str, bytes]]]): Key/value pairs which should be
                    sent along with the request as metadata. Normally, each value must be of type `str`,
                    but for metadata keys ending with the suffix `-bin`, the corresponding values must
                    be of type `bytes`.

            Returns:
                ~.httpbody_pb2.HttpBody:
                    Message that represents an arbitrary HTTP body. It
                should only be used for payload formats that can't be
                represented as JSON, such as raw binary or an HTML page.

                This message can be used both in streaming and
                non-streaming API methods in the request as well as the
                response.

                It can be used as a top-level request field, which is
                convenient if one wants to extract parameters from
                either the URL or HTTP template into the request fields
                and also want access to the raw HTTP body.

                Example:

                ::

                    message GetResourceRequest {
                      // A unique request id.
                      string request_id = 1;

                      // The raw HTTP body is bound to this field.
                      google.api.HttpBody http_body = 2;

                    }

                    service ResourceService {
                      rpc GetResource(GetResourceRequest)
                        returns (google.api.HttpBody);
                      rpc UpdateResource(google.api.HttpBody)
                        returns (google.protobuf.Empty);

                    }

                Example with streaming methods:

                ::

                    service CaldavService {
                      rpc GetCalendar(stream google.api.HttpBody)
                        returns (stream google.api.HttpBody);
                      rpc UpdateCalendar(stream google.api.HttpBody)
                        returns (stream google.api.HttpBody);

                    }

                Use of this type only changes how the request and
                response bodies are handled, all other features will
                continue to work unchanged.

            """

            http_options = _BasePredictionServiceRestTransport._BaseStreamRawPredict._get_http_options()

            request, metadata = self._interceptor.pre_stream_raw_predict(request, metadata)
            transcoded_request = _BasePredictionServiceRestTransport._BaseStreamRawPredict._get_transcoded_request(http_options, request)

            body = _BasePredictionServiceRestTransport._BaseStreamRawPredict._get_request_body_json(transcoded_request)

            # Jsonify the query params
            query_params = _BasePredictionServiceRestTransport._BaseStreamRawPredict._get_query_params_json(transcoded_request)

            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(logging.DEBUG):  # pragma: NO COVER
                request_url = "{host}{uri}".format(host=self._host, uri=transcoded_request['uri'])
                method = transcoded_request['method']
                try:
                    request_payload = json_format.MessageToJson(request)
                except:
                    request_payload = None
                http_request = {
                  "payload": request_payload,
                  "requestMethod": method,
                  "requestUrl": request_url,
                  "headers": dict(metadata),
                }
                _LOGGER.debug(
                    f"Sending request for google.cloud.aiplatform_v1beta1.PredictionServiceClient.StreamRawPredict",
                    extra = {
                        "serviceName": "google.cloud.aiplatform.v1beta1.PredictionService",
                        "rpcName": "StreamRawPredict",
                        "httpRequest": http_request,
                        "metadata": http_request["headers"],
                    },
                )

            # Send the request
            response = PredictionServiceRestTransport._StreamRawPredict._get_response(self._host, metadata, query_params, self._session, timeout, transcoded_request, body)

            # In case of error, raise the appropriate core_exceptions.GoogleAPICallError exception
            # subclass.
            if response.status_code >= 400:
                raise core_exceptions.from_http_response(response)

            # Return the response
            resp = rest_streaming.ResponseIterator(response, httpbody_pb2.HttpBody)

            resp = self._interceptor.post_stream_raw_predict(resp)
            response_metadata = [(k, str(v)) for k, v in response.headers.items()]
            resp, _ = self._interceptor.post_stream_raw_predict_with_metadata(resp, response_metadata)
            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(logging.DEBUG):  # pragma: NO COVER
                http_response = {
                "headers":  dict(response.headers),
                "status": response.status_code,
                }
                _LOGGER.debug(
                    "Received response for google.cloud.aiplatform_v1beta1.PredictionServiceClient.stream_raw_predict",
                    extra = {
                        "serviceName": "google.cloud.aiplatform.v1beta1.PredictionService",
                        "rpcName": "StreamRawPredict",
                        "metadata": http_response["headers"],
                        "httpResponse": http_response,
                    },
                )
            return resp

    @property
    def chat_completions(self) -> Callable[
            [prediction_service.ChatCompletionsRequest],
            httpbody_pb2.HttpBody]:
        # The return type is fine, but mypy isn't sophisticated enough to determine what's going on here.
        # In C++ this would require a dynamic_cast
        return self._ChatCompletions(self._session, self._host, self._interceptor) # type: ignore

    @property
    def count_tokens(self) -> Callable[
            [prediction_service.CountTokensRequest],
            prediction_service.CountTokensResponse]:
        # The return type is fine, but mypy isn't sophisticated enough to determine what's going on here.
        # In C++ this would require a dynamic_cast
        return self._CountTokens(self._session, self._host, self._interceptor) # type: ignore

    @property
    def direct_predict(self) -> Callable[
            [prediction_service.DirectPredictRequest],
            prediction_service.DirectPredictResponse]:
        # The return type is fine, but mypy isn't sophisticated enough to determine what's going on here.
        # In C++ this would require a dynamic_cast
        return self._DirectPredict(self._session, self._host, self._interceptor) # type: ignore

    @property
    def direct_raw_predict(self) -> Callable[
            [prediction_service.DirectRawPredictRequest],
            prediction_service.DirectRawPredictResponse]:
        # The return type is fine, but mypy isn't sophisticated enough to determine what's going on here.
        # In C++ this would require a dynamic_cast
        return self._DirectRawPredict(self._session, self._host, self._interceptor) # type: ignore

    @property
    def embed_content(self) -> Callable[
            [prediction_service.EmbedContentRequest],
            prediction_service.EmbedContentResponse]:
        # The return type is fine, but mypy isn't sophisticated enough to determine what's going on here.
        # In C++ this would require a dynamic_cast
        return self._EmbedContent(self._session, self._host, self._interceptor) # type: ignore

    @property
    def explain(self) -> Callable[
            [prediction_service.ExplainRequest],
            prediction_service.ExplainResponse]:
        # The return type is fine, but mypy isn't sophisticated enough to determine what's going on here.
        # In C++ this would require a dynamic_cast
        return self._Explain(self._session, self._host, self._interceptor) # type: ignore

    @property
    def fetch_predict_operation(self) -> Callable[
            [prediction_service.FetchPredictOperationRequest],
            operations_pb2.Operation]:
        # The return type is fine, but mypy isn't sophisticated enough to determine what's going on here.
        # In C++ this would require a dynamic_cast
        return self._FetchPredictOperation(self._session, self._host, self._interceptor) # type: ignore

    @property
    def generate_content(self) -> Callable[
            [prediction_service.GenerateContentRequest],
            prediction_service.GenerateContentResponse]:
        # The return type is fine, but mypy isn't sophisticated enough to determine what's going on here.
        # In C++ this would require a dynamic_cast
        return self._GenerateContent(self._session, self._host, self._interceptor) # type: ignore

    @property
    def invoke(self) -> Callable[
            [prediction_service.InvokeRequest],
            httpbody_pb2.HttpBody]:
        # The return type is fine, but mypy isn't sophisticated enough to determine what's going on here.
        # In C++ this would require a dynamic_cast
        return self._Invoke(self._session, self._host, self._interceptor) # type: ignore

    @property
    def predict(self) -> Callable[
            [prediction_service.PredictRequest],
            prediction_service.PredictResponse]:
        # The return type is fine, but mypy isn't sophisticated enough to determine what's going on here.
        # In C++ this would require a dynamic_cast
        return self._Predict(self._session, self._host, self._interceptor) # type: ignore

    @property
    def predict_long_running(self) -> Callable[
            [prediction_service.PredictLongRunningRequest],
            operations_pb2.Operation]:
        # The return type is fine, but mypy isn't sophisticated enough to determine what's going on here.
        # In C++ this would require a dynamic_cast
        return self._PredictLongRunning(self._session, self._host, self._interceptor) # type: ignore

    @property
    def raw_predict(self) -> Callable[
            [prediction_service.RawPredictRequest],
            httpbody_pb2.HttpBody]:
        # The return type is fine, but mypy isn't sophisticated enough to determine what's going on here.
        # In C++ this would require a dynamic_cast
        return self._RawPredict(self._session, self._host, self._interceptor) # type: ignore

    @property
    def server_streaming_predict(self) -> Callable[
            [prediction_service.StreamingPredictRequest],
            prediction_service.StreamingPredictResponse]:
        # The return type is fine, but mypy isn't sophisticated enough to determine what's going on here.
        # In C++ this would require a dynamic_cast
        return self._ServerStreamingPredict(self._session, self._host, self._interceptor) # type: ignore

    @property
    def stream_direct_predict(self) -> Callable[
            [prediction_service.StreamDirectPredictRequest],
            prediction_service.StreamDirectPredictResponse]:
        # The return type is fine, but mypy isn't sophisticated enough to determine what's going on here.
        # In C++ this would require a dynamic_cast
        return self._StreamDirectPredict(self._session, self._host, self._interceptor) # type: ignore

    @property
    def stream_direct_raw_predict(self) -> Callable[
            [prediction_service.StreamDirectRawPredictRequest],
            prediction_service.StreamDirectRawPredictResponse]:
        # The return type is fine, but mypy isn't sophisticated enough to determine what's going on here.
        # In C++ this would require a dynamic_cast
        return self._StreamDirectRawPredict(self._session, self._host, self._interceptor) # type: ignore

    @property
    def stream_generate_content(self) -> Callable[
            [prediction_service.GenerateContentRequest],
            prediction_service.GenerateContentResponse]:
        # The return type is fine, but mypy isn't sophisticated enough to determine what's going on here.
        # In C++ this would require a dynamic_cast
        return self._StreamGenerateContent(self._session, self._host, self._interceptor) # type: ignore

    @property
    def streaming_predict(self) -> Callable[
            [prediction_service.StreamingPredictRequest],
            prediction_service.StreamingPredictResponse]:
        # The return type is fine, but mypy isn't sophisticated enough to determine what's going on here.
        # In C++ this would require a dynamic_cast
        return self._StreamingPredict(self._session, self._host, self._interceptor) # type: ignore

    @property
    def streaming_raw_predict(self) -> Callable[
            [prediction_service.StreamingRawPredictRequest],
            prediction_service.StreamingRawPredictResponse]:
        # The return type is fine, but mypy isn't sophisticated enough to determine what's going on here.
        # In C++ this would require a dynamic_cast
        return self._StreamingRawPredict(self._session, self._host, self._interceptor) # type: ignore

    @property
    def stream_raw_predict(self) -> Callable[
            [prediction_service.StreamRawPredictRequest],
            httpbody_pb2.HttpBody]:
        # The return type is fine, but mypy isn't sophisticated enough to determine what's going on here.
        # In C++ this would require a dynamic_cast
        return self._StreamRawPredict(self._session, self._host, self._interceptor) # type: ignore

    @property
    def kind(self) -> str:
        return "rest"

    def close(self):
        self._session.close()


__all__=(
    'PredictionServiceRestTransport',
)
