// This file is part of the AspectC++ compiler 'ac++'.
// Copyright (C) 1999-2003  The 'ac++' developers (see aspectc.org)
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

// Helper functions that access system interfaces

#include "SysCall.h"

#include <sys/stat.h>
#include <iostream>

namespace SysCall {

  // get the UNIX modification time of a file
  // returns 'false' on any error, otherwise result will be set
  bool file_modification_time(const std::filesystem::path &file, std::time_t &result) {
    struct stat st;
    // call to 'stat' can be replaced by a std::filesystem function in C++20
    if (stat(file.string().c_str(), &st) != 0) // calling string() to avoid wide characters on Windows
      return false;
    result = st.st_mtime;
    return true;
  }

  // create the specified directory hierarchy; return false on error
  bool make_directory_hierarchy(const std::filesystem::path &dir) {
    // just a wrapper for filesystem library
    std::error_code ec;
    std::filesystem::create_directories(dir, ec );
    return !ec; // ec is true in case of an any error
  }

#if defined (WIN32)
  // replace all '\' in a string (filename path) with '/'
  void make_unix_path(std::string &path) {
    for (std::size_t i = 0; i < path.length(); ++i)
      if (path[i] == '\\')
        path[i] = '/';
  }
#endif

  // build a canonical representation of a file path
  // - the resulting path is absolute and contains neither ".", "..", nor soft links
  // - on Windows all '\' are replace by '/'
  // returns 'false' on any error, e.g. file does not exist
  bool make_canonical_path(const std::filesystem::path &path, std::string &result) {
    // just a wrapper for filesystem library
    std::error_code ec;
    result = std::filesystem::canonical(path, ec).string();
#if defined (WIN32)
    if (!ec)
      make_unix_path(result);
#endif
    return !ec; // ec is true in case of an any error
  }

}
