/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2025 Univ. Grenoble Alpes, CNRS, Grenoble INP - UGA, TIMC, 38000 Grenoble, France
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/
#include "RemoveTransformation.h"

// CamiTK includes
#include <TransformationManager.h>
#include <TransformationExplorer.h>
#include <Application.h>
#include <Log.h>

#include <QMessageBox>

using namespace camitk;

// -------------------- init --------------------
void RemoveTransformation::init() {
    currentTransformation = nullptr;
    TransformationExplorer* transfoExplorer = dynamic_cast<TransformationExplorer*>(Application::getViewer("Transformation Explorer"));
    if (transfoExplorer != nullptr) {
        connect(transfoExplorer, &TransformationExplorer::currentTransformationChanged, this, &RemoveTransformation::setTransformation);
    }
}

void RemoveTransformation::setTransformation(const Transformation* tr) {
    currentTransformation = tr;
}

// -------------------- process --------------------
Action::ApplyStatus RemoveTransformation::process() {
    if (currentTransformation == nullptr) {
        CAMITK_WARNING("Please first select a Transformation to remove.");
    }
    else {
        QString transformationInformation;
        if (TransformationManager::isCompositeTransformation(currentTransformation)) {
            transformationInformation = "composite";
        }
        else {
            if (TransformationManager::isDefaultIdentityToWorld(currentTransformation)) {
                transformationInformation = "default identity to world";
            }
            else {
                if (TransformationManager::isInverseTransformation(currentTransformation)) {
                    transformationInformation = "inverse";
                }
                else {
                    transformationInformation = "user defined";
                }
            }
        }
        std::shared_ptr<Transformation> tr = TransformationManager::getTransformationOwnership(currentTransformation);
        if (tr != nullptr) {
            QString name = currentTransformation->getName();
            if (QMessageBox::question(nullptr, "Confirm remove Transformation", QString("Are you sure you want to delete Transformation %1?").arg(name)) == QMessageBox::Yes) {
                if (TransformationManager::removeTransformation(tr)) {
                    currentTransformation = nullptr;
                    refreshApplication();
                    return SUCCESS;
                }
                else {
                    CAMITK_WARNING(QString("Transformation '%1' cannot be removed as it is used/owned by a Component.").arg(currentTransformation->getName()));
                }
            }
        }
        else {
            if (transformationInformation == "user defined") {
                // we tried to get the ownership but failed, currentTransformation is not alive anymore
                transformationInformation = "invalid (probably deleted)";
            }
            CAMITK_WARNING(QString("Transformation '%1' cannot be removed as it is %2.").arg(currentTransformation->getName()).arg(transformationInformation));
        }
    }

    return ERROR;
}

// -------------------- targetDefined --------------------
void RemoveTransformation::targetDefined() {
}

// -------------------- parameterChanged --------------------
void RemoveTransformation::parameterChanged(QString parameterName) {
}

