/*
 * vim:ts=4:sw=4:expandtab
 *
 * i3 - an improved tiling window manager
 * © 2009 Michael Stapelberg and contributors (see also: LICENSE)
 *
 * config_parser.c: handwritten parser to parse configuration directives.
 *
 * See also src/commands_parser.c for rationale on why we use a custom parser.
 *
 * This parser works VERY MUCH like src/commands_parser.c, so read that first.
 * The differences are:
 *
 * 1. config_parser supports the 'number' token type (in addition to 'word' and
 *    'string'). Numbers are referred to using &num (like $str).
 *
 * 2. Criteria are not executed immediately, they are just stored.
 *
 * 3. config_parser recognizes \n and \r as 'end' token, while commands_parser
 *    ignores them.
 *
 * 4. config_parser skips the current line on invalid inputs and follows the
 *    nearest <error> token.
 *
 */
#include "all.h"

#include <fcntl.h>
#include <stdbool.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <libgen.h>

#include <xcb/xcb_xrm.h>

xcb_xrm_database_t *database = NULL;

#ifndef TEST_PARSER
pid_t config_error_nagbar_pid = -1;
#endif

/*******************************************************************************
 * The data structures used for parsing. Essentially the current state and a
 * list of tokens for that state.
 *
 * The GENERATED_* files are generated by generate-commands-parser.pl with the
 * input parser-specs/configs.spec.
 ******************************************************************************/

#include "GENERATED_config_enums.h"

typedef struct token {
    char *name;
    char *identifier;
    /* This might be __CALL */
    cmdp_state next_state;
    union {
        uint16_t call_identifier;
    } extra;
} cmdp_token;

typedef struct tokenptr {
    cmdp_token *array;
    int n;
} cmdp_token_ptr;

#include "GENERATED_config_tokens.h"

/*******************************************************************************
 * The parser itself.
 ******************************************************************************/

#include "GENERATED_config_call.h"

static void next_state(const cmdp_token *token, struct parser_ctx *ctx) {
    cmdp_state _next_state = token->next_state;

    if (token->next_state == __CALL) {
        struct ConfigResultIR subcommand_output = {
            .ctx = ctx,
        };
        GENERATED_call(&(ctx->current_match), ctx->stack, token->extra.call_identifier, &subcommand_output);
        if (subcommand_output.has_errors) {
            ctx->has_errors = true;
        }
        _next_state = subcommand_output.next_state;
        parser_clear_stack(ctx->stack);
    }

    ctx->state = _next_state;
    if (ctx->state == INITIAL) {
        parser_clear_stack(ctx->stack);
    }

    /* See if we are jumping back to a state in which we were in previously
     * (statelist contains INITIAL) and just move statelist_idx accordingly. */
    for (int i = 0; i < ctx->statelist_idx; i++) {
        if ((cmdp_state)(ctx->statelist[i]) != _next_state) {
            continue;
        }
        ctx->statelist_idx = i + 1;
        return;
    }

    /* Otherwise, the state is new and we add it to the list */
    ctx->statelist[ctx->statelist_idx++] = _next_state;
}

/*
 * Returns a pointer to the start of the line (one byte after the previous \r,
 * \n) or the start of the input, if this is the first line.
 *
 */
static const char *start_of_line(const char *walk, const char *beginning) {
    while (walk >= beginning && *walk != '\n' && *walk != '\r') {
        walk--;
    }

    return walk + 1;
}

/*
 * Copies the line and terminates it at the next \n, if any.
 *
 * The caller has to free() the result.
 *
 */
static char *single_line(const char *start) {
    char *result = sstrdup(start);
    char *end = strchr(result, '\n');
    if (end != NULL) {
        *end = '\0';
    }
    return result;
}

static void parse_config(struct parser_ctx *ctx, const char *input, struct context *context) {
    /* Dump the entire config file into the debug log. We cannot just use
     * DLOG("%s", input); because one log message must not exceed 4 KiB. */
    const char *dumpwalk = input;
    int linecnt = 1;
    while (*dumpwalk != '\0') {
        const char *next_nl = strchr(dumpwalk, '\n');
        if (next_nl != NULL) {
            DLOG("CONFIG(line %3d): %.*s\n", linecnt, (int)(next_nl - dumpwalk), dumpwalk);
            dumpwalk = next_nl + 1;
        } else {
            DLOG("CONFIG(line %3d): %s\n", linecnt, dumpwalk);
            break;
        }
        linecnt++;
    }
    ctx->state = INITIAL;
    for (int i = 0; i < 10; i++) {
        ctx->statelist[i] = INITIAL;
    }
    ctx->statelist_idx = 1;

    const char *walk = input;
    const size_t len = strlen(input);
    int c;
    const cmdp_token *token;
    bool token_handled;
    linecnt = 1;

#ifndef TEST_PARSER
    struct ConfigResultIR subcommand_output = {
        .ctx = ctx,
    };
    cfg_criteria_init(&(ctx->current_match), &subcommand_output, INITIAL);
#endif

    /* The "<=" operator is intentional: We also handle the terminating 0-byte
     * explicitly by looking for an 'end' token. */
    while ((size_t)(walk - input) <= len) {
        /* Skip whitespace before every token, newlines are relevant since they
         * separate configuration directives. */
        while ((*walk == ' ' || *walk == '\t') && *walk != '\0') {
            walk++;
        }

        const cmdp_token_ptr *ptr = &(tokens[ctx->state]);
        token_handled = false;
        for (c = 0; c < ptr->n; c++) {
            token = &(ptr->array[c]);

            /* A literal. */
            if (token->name[0] == '\'') {
                if (strncasecmp(walk, token->name + 1, strlen(token->name) - 1) == 0) {
                    if (token->identifier != NULL) {
                        parser_push_string(ctx->stack, token->identifier, token->name + 1);
                    }
                    walk += strlen(token->name) - 1;
                    next_state(token, ctx);
                    token_handled = true;
                    break;
                }
                continue;
            }

            if (strcmp(token->name, "number") == 0) {
                /* Handle numbers. We only accept decimal numbers for now. */
                char *end = NULL;
                errno = 0;
                const long int num = strtol(walk, &end, 10);
                if ((errno == ERANGE && (num == LONG_MIN || num == LONG_MAX)) ||
                    (errno != 0 && num == 0)) {
                    continue;
                }

                /* No valid numbers found */
                if (end == walk) {
                    continue;
                }

                if (token->identifier != NULL) {
                    parser_push_long(ctx->stack, token->identifier, num);
                }

                /* Set walk to the first non-number character */
                walk = end;
                next_state(token, ctx);
                token_handled = true;
                break;
            }

            if (strcmp(token->name, "string") == 0 ||
                strcmp(token->name, "word") == 0) {
                const char *beginning = walk;
                /* Handle quoted strings (or words). */
                if (*walk == '"') {
                    beginning++;
                    walk++;
                    while (*walk != '\0' && (*walk != '"' || *(walk - 1) == '\\')) {
                        walk++;
                    }
                } else {
                    if (token->name[0] == 's') {
                        while (*walk != '\0' && *walk != '\r' && *walk != '\n') {
                            walk++;
                        }
                    } else {
                        /* For a word, the delimiters are white space (' ' or
                         * '\t'), closing square bracket (]), comma (,) and
                         * semicolon (;). */
                        while (*walk != ' ' && *walk != '\t' &&
                               *walk != ']' && *walk != ',' &&
                               *walk != ';' && *walk != '\r' &&
                               *walk != '\n' && *walk != '\0') {
                            walk++;
                        }
                    }
                }
                if (walk != beginning) {
                    char *str = scalloc(walk - beginning + 1, 1);
                    /* We copy manually to handle escaping of characters. */
                    int inpos, outpos;
                    for (inpos = 0, outpos = 0;
                         inpos < (walk - beginning);
                         inpos++, outpos++) {
                        /* We only handle escaped double quotes to not break
                         * backwards compatibility with people using \w in
                         * regular expressions etc. */
                        if (beginning[inpos] == '\\' && beginning[inpos + 1] == '"') {
                            inpos++;
                        }
                        str[outpos] = beginning[inpos];
                    }
                    if (token->identifier) {
                        parser_push_string(ctx->stack, token->identifier, str);
                    }
                    free(str);
                    /* If we are at the end of a quoted string, skip the ending
                     * double quote. */
                    if (*walk == '"') {
                        walk++;
                    }
                    next_state(token, ctx);
                    token_handled = true;
                    break;
                }
            }

            if (strcmp(token->name, "line") == 0) {
                while (*walk != '\0' && *walk != '\n' && *walk != '\r') {
                    walk++;
                }
                next_state(token, ctx);
                token_handled = true;
                linecnt++;
                walk++;
                break;
            }

            if (strcmp(token->name, "end") == 0) {
                if (*walk == '\0' || *walk == '\n' || *walk == '\r') {
                    next_state(token, ctx);
                    token_handled = true;
                    /* To make sure we start with an appropriate matching
                     * datastructure for commands which do *not* specify any
                     * criteria, we re-initialize the criteria system after
                     * every command. */
#ifndef TEST_PARSER
                    cfg_criteria_init(&(ctx->current_match), &subcommand_output, INITIAL);
#endif
                    linecnt++;
                    walk++;
                    break;
                }
            }
        }

        if (!token_handled) {
            /* Figure out how much memory we will need to fill in the names of
             * all tokens afterward. */
            int tokenlen = 0;
            for (c = 0; c < ptr->n; c++) {
                tokenlen += strlen(ptr->array[c].name) + strlen("'', ");
            }

            /* Build up a decent error message. We include the problem, the
             * full input, and underline the position where the parser
             * currently is. */
            char *errormessage;
            char *possible_tokens = smalloc(tokenlen + 1);
            char *tokenwalk = possible_tokens;
            for (c = 0; c < ptr->n; c++) {
                token = &(ptr->array[c]);
                if (token->name[0] == '\'') {
                    /* A literal is copied to the error message enclosed with
                     * single quotes. */
                    *tokenwalk++ = '\'';
                    strcpy(tokenwalk, token->name + 1);
                    tokenwalk += strlen(token->name + 1);
                    *tokenwalk++ = '\'';
                } else {
                    /* Skip error tokens in error messages, they are used
                     * internally only and might confuse users. */
                    if (strcmp(token->name, "error") == 0) {
                        continue;
                    }
                    /* Any other token is copied to the error message enclosed
                     * with angle brackets. */
                    *tokenwalk++ = '<';
                    strcpy(tokenwalk, token->name);
                    tokenwalk += strlen(token->name);
                    *tokenwalk++ = '>';
                }
                if (c < (ptr->n - 1)) {
                    *tokenwalk++ = ',';
                    *tokenwalk++ = ' ';
                }
            }
            *tokenwalk = '\0';
            sasprintf(&errormessage, "Expected one of these tokens: %s",
                      possible_tokens);
            free(possible_tokens);

            /* Go back to the beginning of the line */
            const char *error_line = start_of_line(walk, input);

            /* Contains the same amount of characters as 'input' has, but with
             * the unparsable part highlighted using ^ characters. */
            char *position = scalloc(strlen(error_line) + 1, 1);
            const char *copywalk;
            for (copywalk = error_line;
                 *copywalk != '\n' && *copywalk != '\r' && *copywalk != '\0';
                 copywalk++) {
                position[(copywalk - error_line)] = (copywalk >= walk ? '^' : (*copywalk == '\t' ? '\t' : ' '));
            }
            position[(copywalk - error_line)] = '\0';

            ELOG("CONFIG: %s\n", errormessage);
            ELOG("CONFIG: (in file %s)\n", context->filename);
            char *error_copy = single_line(error_line);

            /* Print context lines *before* the error, if any. */
            if (linecnt > 1) {
                const char *context_p1_start = start_of_line(error_line - 2, input);
                char *context_p1_line = single_line(context_p1_start);
                if (linecnt > 2) {
                    const char *context_p2_start = start_of_line(context_p1_start - 2, input);
                    char *context_p2_line = single_line(context_p2_start);
                    ELOG("CONFIG: Line %3d: %s\n", linecnt - 2, context_p2_line);
                    free(context_p2_line);
                }
                ELOG("CONFIG: Line %3d: %s\n", linecnt - 1, context_p1_line);
                free(context_p1_line);
            }
            ELOG("CONFIG: Line %3d: %s\n", linecnt, error_copy);
            ELOG("CONFIG:           %s\n", position);
            free(error_copy);
            /* Print context lines *after* the error, if any. */
            for (int i = 0; i < 2; i++) {
                const char *error_line_end = strchr(error_line, '\n');
                if (error_line_end != NULL && *(error_line_end + 1) != '\0') {
                    error_line = error_line_end + 1;
                    error_copy = single_line(error_line);
                    ELOG("CONFIG: Line %3d: %s\n", linecnt + i + 1, error_copy);
                    free(error_copy);
                }
            }

            context->has_errors = true;

            /* Skip the rest of this line, but continue parsing. */
            while ((size_t)(walk - input) <= len && *walk != '\n') {
                walk++;
            }

            free(position);
            free(errormessage);
            parser_clear_stack(ctx->stack);

            /* To figure out in which state to go (e.g. MODE or INITIAL),
             * we find the nearest state which contains an <error> token
             * and follow that one. */
            bool error_token_found = false;
            for (int i = ctx->statelist_idx - 1; (i >= 0) && !error_token_found; i--) {
                const cmdp_token_ptr *errptr = &(tokens[ctx->statelist[i]]);
                for (int j = 0; j < errptr->n; j++) {
                    if (strcmp(errptr->array[j].name, "error") != 0) {
                        continue;
                    }
                    next_state(&(errptr->array[j]), ctx);
                    error_token_found = true;
                    break;
                }
            }

            assert(error_token_found);
        }
    }
}

/*******************************************************************************
 * Code for building the stand-alone binary test.commands_parser which is used
 * by t/187-commands-parser.t.
 ******************************************************************************/

#ifdef TEST_PARSER

/*
 * Logs the given message to stdout while prefixing the current time to it,
 * but only if debug logging was activated.
 * This is to be called by DLOG() which includes filename/linenumber
 *
 */
void debuglog(char *fmt, ...) {
    va_list args;

    va_start(args, fmt);
    fprintf(stdout, "# ");
    vfprintf(stdout, fmt, args);
    va_end(args);
}

void errorlog(char *fmt, ...) {
    va_list args;

    va_start(args, fmt);
    vfprintf(stderr, fmt, args);
    va_end(args);
}

static int criteria_next_state;

void cfg_criteria_init(I3_CFG, int _state) {
    criteria_next_state = _state;
}

void cfg_criteria_add(I3_CFG, const char *ctype, const char *cvalue) {
}

void cfg_criteria_pop_state(I3_CFG) {
    result->next_state = criteria_next_state;
}

int main(int argc, char *argv[]) {
    if (argc < 2) {
        fprintf(stderr, "Syntax: %s <command>\n", argv[0]);
        return 1;
    }
    struct stack stack;
    memset(&stack, '\0', sizeof(struct stack));
    struct parser_ctx ctx = {
        .use_nagbar = false,
        .stack = &stack,
    };
    SLIST_INIT(&(ctx.variables));
    struct context context;
    context.filename = "<stdin>";
    parse_config(&ctx, argv[1], &context);
}

#else

/**
 * Launch nagbar to indicate errors in the configuration file.
 */
void start_config_error_nagbar(const char *configpath, const bool has_errors) {
    char *editaction, *pageraction;
    sasprintf(&editaction, "i3-sensible-editor \"%s\" && i3-msg reload\n", configpath);
    sasprintf(&pageraction, "i3-sensible-pager \"%s\"\n", errorfilename);
    char *argv[] = {
        NULL, /* will be replaced by the executable path */
        "-f",
        (config.font.pattern ? config.font.pattern : "fixed"),
        "-t",
        (has_errors ? "error" : "warning"),
        "-m",
        (has_errors ? "You have an error in your i3 config file!" : "Your config is outdated. Please fix the warnings to make sure everything works."),
        "-b",
        "edit config",
        editaction,
        (errorfilename ? "-b" : NULL),
        (has_errors ? "show errors" : "show warnings"),
        pageraction,
        NULL};

    start_nagbar(&config_error_nagbar_pid, argv);
    free(editaction);
    free(pageraction);
}

/*
 * Inserts or updates a variable assignment depending on whether it already exists.
 *
 */
static void upsert_variable(struct variables_head *variables, char *key, char *value) {
    struct Variable *current;
    SLIST_FOREACH (current, variables, variables) {
        if (strcmp(current->key, key) != 0) {
            continue;
        }

        DLOG("Updated variable: %s = %s -> %s\n", key, current->value, value);
        FREE(current->value);
        current->value = sstrdup(value);
        return;
    }

    DLOG("Defined new variable: %s = %s\n", key, value);
    struct Variable *new = scalloc(1, sizeof(struct Variable));
    struct Variable *test = NULL, *loc = NULL;
    new->key = sstrdup(key);
    new->value = sstrdup(value);
    /* ensure that the correct variable is matched in case of one being
     * the prefix of another */
    SLIST_FOREACH (test, variables, variables) {
        if (strlen(new->key) >= strlen(test->key)) {
            break;
        }
        loc = test;
    }

    if (loc == NULL) {
        SLIST_INSERT_HEAD(variables, new, variables);
    } else {
        SLIST_INSERT_AFTER(loc, new, variables);
    }
}

static char *get_resource(const char *name) {
    if (conn == NULL) {
        return NULL;
    }

    /* Load the resource database lazily. */
    if (database == NULL) {
        database = xcb_xrm_database_from_default(conn);

        if (database == NULL) {
            ELOG("Failed to open the resource database.\n");

            /* Load an empty database so we don't keep trying to load the
             * default database over and over again. */
            database = xcb_xrm_database_from_string("");

            return NULL;
        }
    }

    char *resource;
    xcb_xrm_resource_get_string(database, name, NULL, &resource);
    return resource;
}

/*
 * Releases the memory of all variables in ctx.
 *
 */
void free_variables(struct parser_ctx *ctx) {
    while (!SLIST_EMPTY(&(ctx->variables))) {
        struct Variable *current = SLIST_FIRST(&(ctx->variables));
        FREE(current->key);
        FREE(current->value);
        SLIST_REMOVE_HEAD(&(ctx->variables), variables);
        FREE(current);
    }
}

static bool try_chdir(const char *path) {
    bool result = true;
    char *dir = NULL;
    /* dirname(3) might modify the buffer, so make a copy: */
    char *dirbuf = sstrdup(path);
    if ((dir = dirname(dirbuf)) != NULL) {
        LOG("Changing working directory to config file directory %s\n", dir);
        if (chdir(dir) == -1) {
            ELOG("chdir(%s) failed: %s\n", dir, strerror(errno));
            result = false;
        }
    }
    FREE(dirbuf);
    return result;
}

static parse_file_result_t parse_file_inner(struct parser_ctx *ctx, const char *f, IncludedFile *included_file) {
    if (!try_chdir(f)) {
        return PARSE_FILE_FAILED;
    }

    int fd;
    if ((fd = open(f, O_RDONLY)) == -1) {
        return PARSE_FILE_FAILED;
    }

    struct stat stbuf;
    if (fstat(fd, &stbuf) == -1) {
        close(fd);
        return PARSE_FILE_FAILED;
    }

    FILE *fstr;
    if ((fstr = fdopen(fd, "r")) == NULL) {
        return PARSE_FILE_FAILED;
    }

    included_file->raw_contents = scalloc(stbuf.st_size + 1, 1);
    if ((ssize_t)fread(included_file->raw_contents, 1, stbuf.st_size, fstr) != stbuf.st_size) {
        fclose(fstr);
        return PARSE_FILE_FAILED;
    }
    rewind(fstr);

    char buffer[4096], key[512], value[4096], *continuation = NULL;
    bool invalid_sets = false;
    char *buf = scalloc(stbuf.st_size + 1, 1);
    while (!feof(fstr)) {
        if (!continuation) {
            continuation = buffer;
        }
        if (fgets(continuation, sizeof(buffer) - (continuation - buffer), fstr) == NULL) {
            if (feof(fstr)) {
                break;
            }
            fclose(fstr);
            return PARSE_FILE_FAILED;
        }
        if (buffer[strlen(buffer) - 1] != '\n' && !feof(fstr)) {
            ELOG("Your line continuation is too long, it exceeds %zd bytes\n", sizeof(buffer));
        }

        /* sscanf implicitly strips whitespace. */
        value[0] = '\0';
        const bool skip_line = (sscanf(buffer, "%511s %4095[^\n]", key, value) < 1 || strlen(key) < 3);
        const bool comment = (key[0] == '#');
        value[4095] = '\n';

        continuation = strstr(buffer, "\\\n");
        if (continuation) {
            if (!comment) {
                continue;
            }
            DLOG("line continuation in comment is ignored: \"%.*s\"\n", (int)strlen(buffer) - 1, buffer);
            continuation = NULL;
        }

        strcpy(buf + strlen(buf), buffer);

        /* Skip comments and empty lines. */
        if (skip_line || comment) {
            continue;
        }

        if (strcasecmp(key, "set") == 0 && *value != '\0') {
            char v_key[512];
            char v_value[4096] = {'\0'};

            if (sscanf(value, "%511s %4095[^\n]", v_key, v_value) < 1) {
                ELOG("Failed to parse variable specification '%s', skipping it.\n", value);
                invalid_sets = true;
                continue;
            }

            if (v_key[0] != '$') {
                ELOG("Malformed variable assignment, name has to start with $\n");
                invalid_sets = true;
                continue;
            }

            upsert_variable(&(ctx->variables), v_key, v_value);
            continue;
        } else if (strcasecmp(key, "set_from_resource") == 0) {
            char res_name[512] = {'\0'};
            char v_key[512];
            char fallback[4096] = {'\0'};

            /* Ensure that this string is terminated. For example, a user might
             * want a variable to be empty if the resource can't be found and
             * uses
             *   set_from_resource $foo i3wm.foo
             * Without explicitly terminating the string first, sscanf() will
             * leave it uninitialized, causing garbage in the config.*/
            fallback[0] = '\0';

            if (sscanf(value, "%511s %511s %4095[^\n]", v_key, res_name, fallback) < 1) {
                ELOG("Failed to parse resource specification '%s', skipping it.\n", value);
                invalid_sets = true;
                continue;
            }

            if (v_key[0] != '$') {
                ELOG("Malformed variable assignment, name has to start with $\n");
                invalid_sets = true;
                continue;
            }

            char *res_value = get_resource(res_name);
            if (res_value == NULL) {
                DLOG("Could not get resource '%s', using fallback '%s'.\n", res_name, fallback);
                res_value = sstrdup(fallback);
            }

            upsert_variable(&(ctx->variables), v_key, res_value);
            FREE(res_value);
            continue;
        }
    }
    fclose(fstr); /* No need to close(fd) */

    if (database != NULL) {
        xcb_xrm_database_free(database);
        /* Explicitly set the database to NULL again in case the config gets reloaded. */
        database = NULL;
    }

    /* For every custom variable, see how often it occurs in the file and
     * how much extra bytes it requires when replaced. */
    struct Variable *current;
    int extra_bytes = 0;
    /* We need to copy the buffer because we need to invalidate the
     * variables (otherwise we will count them twice, which is bad when
     * 'extra' is negative) */
    char *bufcopy = sstrdup(buf);
    SLIST_FOREACH (current, &(ctx->variables), variables) {
        const int extra = (strlen(current->value) - strlen(current->key));
        for (char *next = bufcopy;
             next < (bufcopy + stbuf.st_size) &&
             (next = strcasestr(next, current->key)) != NULL;) {
            /* We need to invalidate variables completely (otherwise we may count
             * the same variable more than once, thus causing buffer overflow or
             * allocation failure) with spaces (variable names cannot contain spaces) */
            const char *end = next + strlen(current->key);
            while (next < end) {
                *next++ = ' ';
            }
            extra_bytes += extra;
        }
    }
    FREE(bufcopy);

    /* Then, allocate a new buffer and copy the file over to the new one,
     * but replace occurrences of our variables */
    const char *walk = buf;
    char *new = scalloc(stbuf.st_size + extra_bytes + 1, 1);
    char *destwalk = new;
    while (walk < (buf + stbuf.st_size)) {
        /* Find the next variable */
        SLIST_FOREACH (current, &(ctx->variables), variables) {
            current->next_match = strcasestr(walk, current->key);
        }
        const struct Variable *nearest = NULL;
        int distance = stbuf.st_size;
        SLIST_FOREACH (current, &(ctx->variables), variables) {
            if (current->next_match == NULL) {
                continue;
            }
            if ((current->next_match - walk) < distance) {
                distance = (current->next_match - walk);
                nearest = current;
            }
        }
        if (nearest == NULL) {
            /* If there are no more variables, we just copy the rest */
            strncpy(destwalk, walk, (buf + stbuf.st_size) - walk);
            destwalk += (buf + stbuf.st_size) - walk;
            *destwalk = '\0';
            break;
        } else {
            /* Copy until the next variable, then copy its value */
            strncpy(destwalk, walk, distance);
            strcpy(destwalk + distance, nearest->value);
            walk += distance + strlen(nearest->key);
            destwalk += distance + strlen(nearest->value);
        }
    }

    included_file->variable_replaced_contents = sstrdup(new);

    struct context *context = scalloc(1, sizeof(struct context));
    context->filename = f;
    parse_config(ctx, new, context);
    if (ctx->has_errors) {
        context->has_errors = true;
    }

    check_for_duplicate_bindings(context);

    if (ctx->use_nagbar && (context->has_errors || context->has_warnings || invalid_sets)) {
        ELOG("FYI: You are using i3 version %s\n", i3_version);
        start_config_error_nagbar(f, context->has_errors || invalid_sets);
    }

    const bool has_errors = context->has_errors;

    FREE(context->line_copy);
    free(context);
    free(new);
    free(buf);

    if (has_errors) {
        return PARSE_FILE_CONFIG_ERRORS;
    }
    return PARSE_FILE_SUCCESS;
}

/*
 * Parses the given file by first replacing the variables, then calling
 * parse_config and possibly launching i3-nagbar.
 *
 */
parse_file_result_t parse_file(struct parser_ctx *ctx, const char *f, IncludedFile *included_file) {
    char *old_dir = getcwd(NULL, 0);

    parse_file_result_t result = parse_file_inner(ctx, f, included_file);

    if (chdir(old_dir) == -1) {
        ELOG("chdir(%s) failed: %s\n", old_dir, strerror(errno));
        result = PARSE_FILE_FAILED;
    }
    free(old_dir);
    return result;
}

#endif
