!***********************************************************************
! This file is part of OpenMolcas.                                     *
!                                                                      *
! OpenMolcas is free software; you can redistribute it and/or modify   *
! it under the terms of the GNU Lesser General Public License, v. 2.1. *
! OpenMolcas is distributed in the hope that it will be useful, but it *
! is provided "as is" and without any express or implied warranties.   *
! For more details see the full text of the license in the file        *
! LICENSE or in <http://www.gnu.org/licenses/>.                        *
!***********************************************************************

module jcoeff

use Definitions, only: wp, iwp

implicit none
private

!-----------------------------------------------------------------------
! iLn -- number of the lanthanide:
!         iLn = 1;  ==> Gd;
!         iLn = 2;  ==> Tb;
!         iLn = 3;  ==> Dy;
!         iLn = 4;  ==> Ho;
!         iLn = 5;  ==> Er;
!         iLn = 6;  ==> Tm;
!         iLn = 7;  ==> Yb;
! iLS -- number of the LS term;
!          1 <= iLS <= 4
!  iJ -- number of the intermediate J term
!          1 <= iJ <= 17 ( 2iJ+1 )
!
!  iK --  0 <= iK  <= 7
! ika -- -5 <= ika <= 5
!  If --  0 <= iP  <= 1
! iph -- -1 <= iph <= 1
!
!  Jx( iLn, iLS, iJ,    iK, ika,   iP, iph );
!  dE( iLn, iLS, iJ )
!-----------------------------------------------------------------------

real(kind=wp), protected :: dE(5,4,17), Jx(1:5,1:4,1:17,0:7,-5:5,0:1,-1:1)
logical(kind=iwp), protected :: initialized = .false.

public :: dE, init_Jx, Jx

contains

subroutine init_Jx()

  use Constants, only: Zero

  if (initialized) return

  dE(:,:,:) = Zero
  Jx(:,:,:,:,:,:,:) = Zero

  !--------------------
  !    Gd2+
  ! J     energy (cm-1)
  !--------------------
  ! 6.0        0.000000
  ! 5.0     1468.764132
  ! 4.0     2692.734237
  ! 3.0     3671.910326
  ! 2.0     4406.292389
  ! 1.0     4895.880434
  ! 0.0     5140.674455
  dE(1,1,13) = Zero
  dE(1,1,11) = 1468.764132_wp
  dE(1,1,9) = 2692.734237_wp
  dE(1,1,7) = 3671.910326_wp
  dE(1,1,5) = 4406.292389_wp
  dE(1,1,3) = 4895.880434_wp
  dE(1,1,1) = 5140.674455_wp
  ! coefficients for Gd
  ! iLn=1; iLS=1, iJ=6 (2.0_wp*iJ+1)
  Jx(1,1,13,0,0,0,0) = -13.0_wp/98.0_wp
  Jx(1,1,13,4,0,0,0) = 39.0_wp/6776.0_wp
  Jx(1,1,13,4,-4,0,0) = (39.0_wp*sqrt(5.0_wp))/(6776.0_wp*sqrt(14.0_wp))
  Jx(1,1,13,4,4,0,0) = (39.0_wp*sqrt(5.0_wp))/(6776.0_wp*sqrt(14.0_wp))
  Jx(1,1,13,6,0,0,0) = 1.0_wp/284592.0_wp
  Jx(1,1,13,6,-4,0,0) = -1.0_wp/(40656.0_wp*sqrt(14.0_wp))
  Jx(1,1,13,6,4,0,0) = -1.0_wp/(40656.0_wp*sqrt(14.0_wp))

  !Jx(1,1,13,1,0,1,0) = 1.0_wp
  !Jx(1,1,13,1,-1,1,1) = -1.0_wp
  !Jx(1,1,13,1,1,1,-1) = -1.0_wp
  Jx(1,1,13,1,0,1,0) = 13.0_wp/98.0_wp
  Jx(1,1,13,1,-1,1,1) = -13.0_wp/98.0_wp
  Jx(1,1,13,1,1,1,-1) = -13.0_wp/98.0_wp

  Jx(1,1,13,3,0,1,0) = -13.0_wp/5082.0_wp
  Jx(1,1,13,3,-1,1,1) = -13.0_wp/(3388.0_wp*sqrt(6.0_wp))
  Jx(1,1,13,3,1,1,-1) = -13.0_wp/(3388.0_wp*sqrt(6.0_wp))
  Jx(1,1,13,3,3,1,1) = (-13.0_wp*sqrt(5.0_wp))/(10164.0_wp*sqrt(2.0_wp))
  Jx(1,1,13,3,-3,1,-1) = (-13.0_wp*sqrt(5.0_wp))/(10164.0_wp*sqrt(2.0_wp))
  Jx(1,1,13,5,0,1,0) = -269.0_wp/84084.0_wp
  Jx(1,1,13,5,-4,1,0) = (-23.0_wp*sqrt(5.0_wp))/(12012.0_wp*sqrt(14.0_wp))
  Jx(1,1,13,5,4,1,0) = (-23.0_wp*sqrt(5.0_wp))/(12012.0_wp*sqrt(14.0_wp))
  Jx(1,1,13,5,-1,1,1) = (215.0_wp*sqrt(5.0_wp))/(112112.0_wp*sqrt(3.0_wp))
  Jx(1,1,13,5,1,1,-1) = (215.0_wp*sqrt(5.0_wp))/(112112.0_wp*sqrt(3.0_wp))
  Jx(1,1,13,5,3,1,1) = (31.0_wp*sqrt(5.0_wp))/(48048.0_wp*sqrt(14.0_wp))
  Jx(1,1,13,5,-3,1,-1) = (31.0_wp*sqrt(5.0_wp))/(48048.0_wp*sqrt(14.0_wp))
  Jx(1,1,13,5,-5,1,1) = 461.0_wp/(48048.0_wp*sqrt(14.0_wp))
  Jx(1,1,13,5,5,1,-1) = 461.0_wp/(48048.0_wp*sqrt(14.0_wp))
  Jx(1,1,13,7,0,1,0) = -1.0_wp/528528.0_wp
  Jx(1,1,13,7,-4,1,0) = 1.0_wp/(16016.0_wp*sqrt(462.0_wp))
  Jx(1,1,13,7,4,1,0) = 1.0_wp/(16016.0_wp*sqrt(462.0_wp))
  Jx(1,1,13,7,-1,1,1) = 1.0_wp/(264264.0_wp*sqrt(7.0_wp))
  Jx(1,1,13,7,1,1,-1) = 1.0_wp/(264264.0_wp*sqrt(7.0_wp))
  Jx(1,1,13,7,3,1,1) = -1.0_wp/(176176.0_wp*sqrt(21.0_wp))
  Jx(1,1,13,7,-3,1,-1) = -1.0_wp/(176176.0_wp*sqrt(21.0_wp))
  Jx(1,1,13,7,-5,1,1) = -1.0_wp/(16016.0_wp*sqrt(231.0_wp))
  Jx(1,1,13,7,5,1,-1) = -1.0_wp/(16016.0_wp*sqrt(231.0_wp))
  ! coefficients for Gd
  ! iLn=1; iLS=1, iJ=5 (2.0_wp*iJ+1)
  Jx(1,1,11,0,0,0,0) = -11.0_wp/98.0_wp
  Jx(1,1,11,4,0,0,0) = -17.0_wp/616.0_wp
  Jx(1,1,11,4,-4,0,0) = (-17.0_wp*sqrt(5.0_wp))/(616.0_wp*sqrt(14.0_wp))
  Jx(1,1,11,4,4,0,0) = (-17.0_wp*sqrt(5.0_wp))/(616.0_wp*sqrt(14.0_wp))
  Jx(1,1,11,6,0,0,0) = -1.0_wp/25872.0_wp
  Jx(1,1,11,6,-4,0,0) = 1.0_wp/(3696.0_wp*sqrt(14.0_wp))
  Jx(1,1,11,6,4,0,0) = 1.0_wp/(3696.0_wp*sqrt(14.0_wp))

  !Jx(1,1,11,1,0,1,0) = 1.0_wp
  !Jx(1,1,11,1,-1,1,1) = -1.0_wp
  !Jx(1,1,11,1,1,1,-1) = -1.0_wp
  Jx(1,1,11,1,0,1,0) = 11.0_wp/98.0_wp
  Jx(1,1,11,1,-1,1,1) = -11.0_wp/98.0_wp
  Jx(1,1,11,1,1,1,-1) = -11.0_wp/98.0_wp

  Jx(1,1,11,3,0,1,0) = 17.0_wp/1386.0_wp
  Jx(1,1,11,3,-1,1,1) = 17.0_wp/(924.0_wp*sqrt(6.0_wp))
  Jx(1,1,11,3,1,1,-1) = 17.0_wp/(924.0_wp*sqrt(6.0_wp))
  Jx(1,1,11,3,3,1,1) = (17.0_wp*sqrt(5.0_wp))/(2772.0_wp*sqrt(2.0_wp))
  Jx(1,1,11,3,-3,1,-1) = (17.0_wp*sqrt(5.0_wp))/(2772.0_wp*sqrt(2.0_wp))
  Jx(1,1,11,5,0,1,0) = 88.0_wp/5733.0_wp
  Jx(1,1,11,5,-4,1,0) = (5.0_wp*sqrt(5.0_wp))/(546.0_wp*sqrt(14.0_wp))
  Jx(1,1,11,5,4,1,0) = (5.0_wp*sqrt(5.0_wp))/(546.0_wp*sqrt(14.0_wp))
  Jx(1,1,11,5,-1,1,1) = (-281.0_wp*sqrt(5.0_wp))/(30576.0_wp*sqrt(3.0_wp))
  Jx(1,1,11,5,1,1,-1) = (-281.0_wp*sqrt(5.0_wp))/(30576.0_wp*sqrt(3.0_wp))
  Jx(1,1,11,5,3,1,1) = (-41.0_wp*sqrt(5.0_wp))/(13104.0_wp*sqrt(14.0_wp))
  Jx(1,1,11,5,-3,1,-1) = (-41.0_wp*sqrt(5.0_wp))/(13104.0_wp*sqrt(14.0_wp))
  Jx(1,1,11,5,-5,1,1) = -67.0_wp/(1456.0_wp*sqrt(14.0_wp))
  Jx(1,1,11,5,5,1,-1) = -67.0_wp/(1456.0_wp*sqrt(14.0_wp))
  Jx(1,1,11,7,0,1,0) = 1.0_wp/48048.0_wp
  Jx(1,1,11,7,-4,1,0) = -1.0_wp/(1456.0_wp*sqrt(462.0_wp))
  Jx(1,1,11,7,4,1,0) = -1.0_wp/(1456.0_wp*sqrt(462.0_wp))
  Jx(1,1,11,7,-1,1,1) = -1.0_wp/(24024.0_wp*sqrt(7.0_wp))
  Jx(1,1,11,7,1,1,-1) = -1.0_wp/(24024.0_wp*sqrt(7.0_wp))
  Jx(1,1,11,7,3,1,1) = 1.0_wp/(16016.0_wp*sqrt(21.0_wp))
  Jx(1,1,11,7,-3,1,-1) = 1.0_wp/(16016.0_wp*sqrt(21.0_wp))
  Jx(1,1,11,7,-5,1,1) = 1.0_wp/(1456.0_wp*sqrt(231.0_wp))
  Jx(1,1,11,7,5,1,-1) = 1.0_wp/(1456.0_wp*sqrt(231.0_wp))
  ! coefficients for Gd
  ! iLn=1; iLS=1, iJ=4 (2.0_wp*iJ+1)
  Jx(1,1,9,0,0,0,0) = -9.0_wp/98.0_wp
  Jx(1,1,9,4,0,0,0) = 291.0_wp/6776.0_wp
  Jx(1,1,9,4,-4,0,0) = (291.0_wp*sqrt(5.0_wp))/(6776.0_wp*sqrt(14.0_wp))
  Jx(1,1,9,4,4,0,0) = (291.0_wp*sqrt(5.0_wp))/(6776.0_wp*sqrt(14.0_wp))
  Jx(1,1,9,6,0,0,0) = 9.0_wp/47432.0_wp
  Jx(1,1,9,6,-4,0,0) = -9.0_wp/(6776.0_wp*sqrt(14.0_wp))
  Jx(1,1,9,6,4,0,0) = -9.0_wp/(6776.0_wp*sqrt(14.0_wp))

  !Jx(1,1,9,1,0,1,0) = 1.0_wp
  !Jx(1,1,9,1,-1,1,1) = -1.0_wp
  !Jx(1,1,9,1,1,1,-1) = -1.0_wp
  Jx(1,1,9,1,0,1,0) = 9.0_wp/98.0_wp
  Jx(1,1,9,1,-1,1,1) = -9.0_wp/98.0_wp
  Jx(1,1,9,1,1,1,-1) = -9.0_wp/98.0_wp

  Jx(1,1,9,3,0,1,0) = -97.0_wp/5082.0_wp
  Jx(1,1,9,3,-1,1,1) = -97.0_wp/(3388.0_wp*sqrt(6.0_wp))
  Jx(1,1,9,3,1,1,-1) = -97.0_wp/(3388.0_wp*sqrt(6.0_wp))
  Jx(1,1,9,3,3,1,1) = (-97.0_wp*sqrt(5.0_wp))/(10164.0_wp*sqrt(2.0_wp))
  Jx(1,1,9,3,-3,1,-1) = (-97.0_wp*sqrt(5.0_wp))/(10164.0_wp*sqrt(2.0_wp))
  Jx(1,1,9,5,0,1,0) = -4027.0_wp/168168.0_wp
  Jx(1,1,9,5,-4,1,0) = (-113.0_wp*sqrt(5.0_wp))/(8008.0_wp*sqrt(14.0_wp))
  Jx(1,1,9,5,4,1,0) = (-113.0_wp*sqrt(5.0_wp))/(8008.0_wp*sqrt(14.0_wp))
  Jx(1,1,9,5,-1,1,1) = (100.0_wp*sqrt(5.0_wp))/(7007.0_wp*sqrt(3.0_wp))
  Jx(1,1,9,5,1,1,-1) = (100.0_wp*sqrt(5.0_wp))/(7007.0_wp*sqrt(3.0_wp))
  Jx(1,1,9,5,3,1,1) = (61.0_wp*sqrt(5.0_wp))/(12012.0_wp*sqrt(14.0_wp))
  Jx(1,1,9,5,-3,1,-1) = (61.0_wp*sqrt(5.0_wp))/(12012.0_wp*sqrt(14.0_wp))
  Jx(1,1,9,5,-5,1,1) = 41.0_wp/(572.0_wp*sqrt(14.0_wp))
  Jx(1,1,9,5,5,1,-1) = 41.0_wp/(572.0_wp*sqrt(14.0_wp))
  Jx(1,1,9,7,0,1,0) = -9.0_wp/88088.0_wp
  Jx(1,1,9,7,-4,1,0) = (9.0_wp*sqrt(3.0_wp))/(8008.0_wp*sqrt(154.0_wp))
  Jx(1,1,9,7,4,1,0) = (9.0_wp*sqrt(3.0_wp))/(8008.0_wp*sqrt(154.0_wp))
  Jx(1,1,9,7,-1,1,1) = 9.0_wp/(44044.0_wp*sqrt(7.0_wp))
  Jx(1,1,9,7,1,1,-1) = 9.0_wp/(44044.0_wp*sqrt(7.0_wp))
  Jx(1,1,9,7,3,1,1) = (-9.0_wp*sqrt(3.0_wp))/(88088.0_wp*sqrt(7.0_wp))
  Jx(1,1,9,7,-3,1,-1) = (-9.0_wp*sqrt(3.0_wp))/(88088.0_wp*sqrt(7.0_wp))
  Jx(1,1,9,7,-5,1,1) = (-9.0_wp*sqrt(3.0_wp))/(8008.0_wp*sqrt(77.0_wp))
  Jx(1,1,9,7,5,1,-1) = (-9.0_wp*sqrt(3.0_wp))/(8008.0_wp*sqrt(77.0_wp))
  ! coefficients for Gd
  ! iLn=1; iLS=1, iJ=3 (2.0_wp*iJ+1)
  Jx(1,1,7,0,0,0,0) = -1.0_wp/14.0_wp
  Jx(1,1,7,4,0,0,0) = -1.0_wp/88.0_wp
  Jx(1,1,7,4,-4,0,0) = (-sqrt(5.0_wp))/(88.0_wp*sqrt(14.0_wp))
  Jx(1,1,7,4,4,0,0) = (-sqrt(5.0_wp))/(88.0_wp*sqrt(14.0_wp))
  Jx(1,1,7,6,0,0,0) = -1.0_wp/1848.0_wp
  Jx(1,1,7,6,-4,0,0) = 1.0_wp/(264.0_wp*sqrt(14.0_wp))
  Jx(1,1,7,6,4,0,0) = 1.0_wp/(264.0_wp*sqrt(14.0_wp))

  !Jx(1,1,7,1,0,1,0) = 1.0_wp
  !Jx(1,1,7,1,-1,1,1) = -1.0_wp
  !Jx(1,1,7,1,1,1,-1) = -1.0_wp
  Jx(1,1,7,1,0,1,0) = 1.0_wp/14.0_wp
  Jx(1,1,7,1,-1,1,1) = -1.0_wp/14.0_wp
  Jx(1,1,7,1,1,1,-1) = -1.0_wp/14.0_wp

  Jx(1,1,7,3,0,1,0) = 1.0_wp/198.0_wp
  Jx(1,1,7,3,-1,1,1) = 1.0_wp/(132.0_wp*sqrt(6.0_wp))
  Jx(1,1,7,3,1,1,-1) = 1.0_wp/(132.0_wp*sqrt(6.0_wp))
  Jx(1,1,7,3,3,1,1) = (sqrt(5.0_wp))/(396.0_wp*sqrt(2.0_wp))
  Jx(1,1,7,3,-3,1,-1) = (sqrt(5.0_wp))/(396.0_wp*sqrt(2.0_wp))
  Jx(1,1,7,5,0,1,0) = 43.0_wp/6552.0_wp
  Jx(1,1,7,5,-4,1,0) = (sqrt(5.0_wp))/(312.0_wp*sqrt(14.0_wp))
  Jx(1,1,7,5,4,1,0) = (sqrt(5.0_wp))/(312.0_wp*sqrt(14.0_wp))
  Jx(1,1,7,5,-1,1,1) = (-sqrt(5.0_wp))/(273.0_wp*sqrt(3.0_wp))
  Jx(1,1,7,5,1,1,-1) = (-sqrt(5.0_wp))/(273.0_wp*sqrt(3.0_wp))
  Jx(1,1,7,5,3,1,1) = (-sqrt(5.0_wp))/(468.0_wp*sqrt(14.0_wp))
  Jx(1,1,7,5,-3,1,-1) = (-sqrt(5.0_wp))/(468.0_wp*sqrt(14.0_wp))
  Jx(1,1,7,5,-5,1,1) = -1.0_wp/(52.0_wp*sqrt(14.0_wp))
  Jx(1,1,7,5,5,1,-1) = -1.0_wp/(52.0_wp*sqrt(14.0_wp))
  Jx(1,1,7,7,0,1,0) = 1.0_wp/3432.0_wp
  Jx(1,1,7,7,-4,1,0) = -1.0_wp/(104.0_wp*sqrt(462.0_wp))
  Jx(1,1,7,7,4,1,0) = -1.0_wp/(104.0_wp*sqrt(462.0_wp))
  Jx(1,1,7,7,-1,1,1) = -1.0_wp/(1716.0_wp*sqrt(7.0_wp))
  Jx(1,1,7,7,1,1,-1) = -1.0_wp/(1716.0_wp*sqrt(7.0_wp))
  Jx(1,1,7,7,3,1,1) = 1.0_wp/(1144.0_wp*sqrt(21.0_wp))
  Jx(1,1,7,7,-3,1,-1) = 1.0_wp/(1144.0_wp*sqrt(21.0_wp))
  Jx(1,1,7,7,-5,1,1) = 1.0_wp/(104.0_wp*sqrt(231.0_wp))
  Jx(1,1,7,7,5,1,-1) = 1.0_wp/(104.0_wp*sqrt(231.0_wp))
  ! coefficients for Gd
  ! iLn=1; iLS=1, iJ=2 (2.0_wp*iJ+1)
  Jx(1,1,5,0,0,0,0) = -5.0_wp/98.0_wp
  Jx(1,1,5,4,0,0,0) = -15.0_wp/616.0_wp
  Jx(1,1,5,4,-4,0,0) = (-15.0_wp*sqrt(5.0_wp))/(616.0_wp*sqrt(14.0_wp))
  Jx(1,1,5,4,4,0,0) = (-15.0_wp*sqrt(5.0_wp))/(616.0_wp*sqrt(14.0_wp))
  Jx(1,1,5,6,0,0,0) = 25.0_wp/25872.0_wp
  Jx(1,1,5,6,-4,0,0) = -25.0_wp/(3696.0_wp*sqrt(14.0_wp))
  Jx(1,1,5,6,4,0,0) = -25.0_wp/(3696.0_wp*sqrt(14.0_wp))

  !Jx(1,1,5,1,0,1,0) = 1.0_wp
  !Jx(1,1,5,1,-1,1,1) = -1.0_wp
  !Jx(1,1,5,1,1,1,-1) = -1.0_wp
  Jx(1,1,5,1,0,1,0) = 5.0_wp/98.0_wp
  Jx(1,1,5,1,-1,1,1) = -5.0_wp/98.0_wp
  Jx(1,1,5,1,1,1,-1) = -5.0_wp/98.0_wp

  Jx(1,1,5,3,0,1,0) = 5.0_wp/462.0_wp
  Jx(1,1,5,3,-1,1,1) = 5.0_wp/(308.0_wp*sqrt(6.0_wp))
  Jx(1,1,5,3,1,1,-1) = 5.0_wp/(308.0_wp*sqrt(6.0_wp))
  Jx(1,1,5,3,3,1,1) = (5.0_wp*sqrt(5.0_wp))/(924.0_wp*sqrt(2.0_wp))
  Jx(1,1,5,3,-3,1,-1) = (5.0_wp*sqrt(5.0_wp))/(924.0_wp*sqrt(2.0_wp))
  Jx(1,1,5,5,0,1,0) = 25.0_wp/1911.0_wp
  Jx(1,1,5,5,-4,1,0) = (5.0_wp*sqrt(5.0_wp))/(546.0_wp*sqrt(14.0_wp))
  Jx(1,1,5,5,4,1,0) = (5.0_wp*sqrt(5.0_wp))/(546.0_wp*sqrt(14.0_wp))
  Jx(1,1,5,5,-1,1,1) = (-85.0_wp*sqrt(5.0_wp))/(10192.0_wp*sqrt(3.0_wp))
  Jx(1,1,5,5,1,1,-1) = (-85.0_wp*sqrt(5.0_wp))/(10192.0_wp*sqrt(3.0_wp))
  Jx(1,1,5,5,3,1,1) = (-5.0_wp*sqrt(5.0_wp))/(4368.0_wp*sqrt(14.0_wp))
  Jx(1,1,5,5,-3,1,-1) = (-5.0_wp*sqrt(5.0_wp))/(4368.0_wp*sqrt(14.0_wp))
  Jx(1,1,5,5,-5,1,1) = -25.0_wp/(624.0_wp*sqrt(14.0_wp))
  Jx(1,1,5,5,5,1,-1) = -25.0_wp/(624.0_wp*sqrt(14.0_wp))
  Jx(1,1,5,7,0,1,0) = -25.0_wp/48048.0_wp
  Jx(1,1,5,7,-4,1,0) = 25.0_wp/(1456.0_wp*sqrt(462.0_wp))
  Jx(1,1,5,7,4,1,0) = 25.0_wp/(1456.0_wp*sqrt(462.0_wp))
  Jx(1,1,5,7,-1,1,1) = 25.0_wp/(24024.0_wp*sqrt(7.0_wp))
  Jx(1,1,5,7,1,1,-1) = 25.0_wp/(24024.0_wp*sqrt(7.0_wp))
  Jx(1,1,5,7,3,1,1) = -25.0_wp/(16016.0_wp*sqrt(21.0_wp))
  Jx(1,1,5,7,-3,1,-1) = -25.0_wp/(16016.0_wp*sqrt(21.0_wp))
  Jx(1,1,5,7,-5,1,1) = -25.0_wp/(1456.0_wp*sqrt(231.0_wp))
  Jx(1,1,5,7,5,1,-1) = -25.0_wp/(1456.0_wp*sqrt(231.0_wp))
  ! coefficients for Gd
  ! iLn=1; iLS=1, iJ=1 (2.0_wp*iJ+1)
  Jx(1,1,3,0,0,0,0) = -3.0_wp/98.0_wp
  Jx(1,1,3,4,0,0,0) = 3.0_wp/616.0_wp
  Jx(1,1,3,4,-4,0,0) = (3.0_wp*sqrt(5.0_wp))/(616.0_wp*sqrt(14.0_wp))
  Jx(1,1,3,4,4,0,0) = (3.0_wp*sqrt(5.0_wp))/(616.0_wp*sqrt(14.0_wp))
  Jx(1,1,3,6,0,0,0) = -9.0_wp/8624.0_wp
  Jx(1,1,3,6,-4,0,0) = 9.0_wp/(1232.0_wp*sqrt(14.0_wp))
  Jx(1,1,3,6,4,0,0) = 9.0_wp/(1232.0_wp*sqrt(14.0_wp))

  !Jx(1,1,3,1,0,1,0) = 1.0_wp
  !Jx(1,1,3,1,-1,1,1) = -1.0_wp
  !Jx(1,1,3,1,1,1,-1) = -1.0_wp
  Jx(1,1,3,1,0,1,0) = 3.0_wp/98.0_wp
  Jx(1,1,3,1,-1,1,1) = -3.0_wp/98.0_wp
  Jx(1,1,3,1,1,1,-1) = -3.0_wp/98.0_wp

  Jx(1,1,3,3,0,1,0) = -1.0_wp/462.0_wp
  Jx(1,1,3,3,-1,1,1) = -1.0_wp/(308.0_wp*sqrt(6.0_wp))
  Jx(1,1,3,3,1,1,-1) = -1.0_wp/(308.0_wp*sqrt(6.0_wp))
  Jx(1,1,3,3,3,1,1) = (-sqrt(5.0_wp))/(924.0_wp*sqrt(2.0_wp))
  Jx(1,1,3,3,-3,1,-1) = (-sqrt(5.0_wp))/(924.0_wp*sqrt(2.0_wp))
  Jx(1,1,3,5,0,1,0) = -17.0_wp/7644.0_wp
  Jx(1,1,3,5,-4,1,0) = (-sqrt(5.0_wp))/(364.0_wp*sqrt(14.0_wp))
  Jx(1,1,3,5,4,1,0) = (-sqrt(5.0_wp))/(364.0_wp*sqrt(14.0_wp))
  Jx(1,1,3,5,-1,1,1) = (19.0_wp*sqrt(5.0_wp))/(10192.0_wp*sqrt(3.0_wp))
  Jx(1,1,3,5,1,1,-1) = (19.0_wp*sqrt(5.0_wp))/(10192.0_wp*sqrt(3.0_wp))
  Jx(1,1,3,5,3,1,1) = (-5.0_wp*sqrt(5.0_wp))/(4368.0_wp*sqrt(14.0_wp))
  Jx(1,1,3,5,-3,1,-1) = (-5.0_wp*sqrt(5.0_wp))/(4368.0_wp*sqrt(14.0_wp))
  Jx(1,1,3,5,-5,1,1) = 11.0_wp/(1456.0_wp*sqrt(14.0_wp))
  Jx(1,1,3,5,5,1,-1) = 11.0_wp/(1456.0_wp*sqrt(14.0_wp))
  Jx(1,1,3,7,0,1,0) = 9.0_wp/16016.0_wp
  Jx(1,1,3,7,-4,1,0) = (-9.0_wp*sqrt(3.0_wp))/(1456.0_wp*sqrt(154.0_wp))
  Jx(1,1,3,7,4,1,0) = (-9.0_wp*sqrt(3.0_wp))/(1456.0_wp*sqrt(154.0_wp))
  Jx(1,1,3,7,-1,1,1) = -9.0_wp/(8008.0_wp*sqrt(7.0_wp))
  Jx(1,1,3,7,1,1,-1) = -9.0_wp/(8008.0_wp*sqrt(7.0_wp))
  Jx(1,1,3,7,3,1,1) = (9.0_wp*sqrt(3.0_wp))/(16016.0_wp*sqrt(7.0_wp))
  Jx(1,1,3,7,-3,1,-1) = (9.0_wp*sqrt(3.0_wp))/(16016.0_wp*sqrt(7.0_wp))
  Jx(1,1,3,7,-5,1,1) = (9.0_wp*sqrt(3.0_wp))/(1456.0_wp*sqrt(77.0_wp))
  Jx(1,1,3,7,5,1,-1) = (9.0_wp*sqrt(3.0_wp))/(1456.0_wp*sqrt(77.0_wp))
  ! coefficients for Gd
  ! iLn=1; iLS=1, iJ=0 (2.0_wp*iJ+1)
  Jx(1,1,1,0,0,0,0) = -1.0_wp/98.0_wp
  Jx(1,1,1,4,0,0,0) = 3.0_wp/308.0_wp
  Jx(1,1,1,4,-4,0,0) = (3.0_wp*sqrt(5.0_wp))/(308.0_wp*sqrt(14.0_wp))
  Jx(1,1,1,4,4,0,0) = (3.0_wp*sqrt(5.0_wp))/(308.0_wp*sqrt(14.0_wp))
  Jx(1,1,1,6,0,0,0) = 1.0_wp/2156.0_wp
  Jx(1,1,1,6,-4,0,0) = -1.0_wp/(308.0_wp*sqrt(14.0_wp))
  Jx(1,1,1,6,4,0,0) = -1.0_wp/(308.0_wp*sqrt(14.0_wp))

  !Jx(1,1,1,1,0,1,0) = 1.0_wp
  !Jx(1,1,1,1,-1,1,1) = -1.0_wp
  !Jx(1,1,1,1,1,1,-1) = -1.0_wp
  Jx(1,1,1,1,0,1,0) = 1.0_wp/98.0_wp
  Jx(1,1,1,1,-1,1,1) = -1.0_wp/98.0_wp
  Jx(1,1,1,1,1,1,-1) = -1.0_wp/98.0_wp

  Jx(1,1,1,3,0,1,0) = -1.0_wp/231.0_wp
  Jx(1,1,1,3,-1,1,1) = -1.0_wp/(154.0_wp*sqrt(6.0_wp))
  Jx(1,1,1,3,1,1,-1) = -1.0_wp/(154.0_wp*sqrt(6.0_wp))
  Jx(1,1,1,3,3,1,1) = (-sqrt(5.0_wp))/(462.0_wp*sqrt(2.0_wp))
  Jx(1,1,1,3,-3,1,-1) = (-sqrt(5.0_wp))/(462.0_wp*sqrt(2.0_wp))
  Jx(1,1,1,5,0,1,0) = -43.0_wp/7644.0_wp
  Jx(1,1,1,5,-4,1,0) = (-sqrt(5.0_wp))/(364.0_wp*sqrt(14.0_wp))
  Jx(1,1,1,5,4,1,0) = (-sqrt(5.0_wp))/(364.0_wp*sqrt(14.0_wp))
  Jx(1,1,1,5,-1,1,1) = (2.0_wp*sqrt(5.0_wp))/(637.0_wp*sqrt(3.0_wp))
  Jx(1,1,1,5,1,1,-1) = (2.0_wp*sqrt(5.0_wp))/(637.0_wp*sqrt(3.0_wp))
  Jx(1,1,1,5,3,1,1) = (sqrt(5.0_wp))/(546.0_wp*sqrt(14.0_wp))
  Jx(1,1,1,5,-3,1,-1) = (sqrt(5.0_wp))/(546.0_wp*sqrt(14.0_wp))
  Jx(1,1,1,5,-5,1,1) = 3.0_wp/(182.0_wp*sqrt(14.0_wp))
  Jx(1,1,1,5,5,1,-1) = 3.0_wp/(182.0_wp*sqrt(14.0_wp))
  Jx(1,1,1,7,0,1,0) = -1.0_wp/4004.0_wp
  Jx(1,1,1,7,-4,1,0) = (sqrt(3.0_wp))/(364.0_wp*sqrt(154.0_wp))
  Jx(1,1,1,7,4,1,0) = (sqrt(3.0_wp))/(364.0_wp*sqrt(154.0_wp))
  Jx(1,1,1,7,-1,1,1) = 1.0_wp/(2002.0_wp*sqrt(7.0_wp))
  Jx(1,1,1,7,1,1,-1) = 1.0_wp/(2002.0_wp*sqrt(7.0_wp))
  Jx(1,1,1,7,3,1,1) = (-sqrt(3.0_wp))/(4004.0_wp*sqrt(7.0_wp))
  Jx(1,1,1,7,-3,1,-1) = (-sqrt(3.0_wp))/(4004.0_wp*sqrt(7.0_wp))
  Jx(1,1,1,7,-5,1,1) = (-sqrt(3.0_wp))/(364.0_wp*sqrt(77.0_wp))
  Jx(1,1,1,7,5,1,-1) = (-sqrt(3.0_wp))/(364.0_wp*sqrt(77.0_wp))
  !---------------------------------------------------------------------

  !--------------------
  !    Tb2+
  ! J     energy (cm-1)
  !--------------------
  ! 7.5        0.000000
  ! 6.5     2479.137826
  ! 5.5     4627.723931
  ! 4.5     6445.758336
  ! 3.5     7933.241025
  ! 2.5     9090.172009
  !
  ! 5.5     8476.381751
  ! 4.5    10294.416151
  ! 3.5    11781.898838
  ! 2.5    12938.829825
  ! 1.5    13765.209097
  ! 0.5    14261.036662
  !
  ! 3.5    35149.092017
  ! 2.5    36306.023001
  ! 1.5    37132.402274
  dE(2,1,16) = Zero
  dE(2,1,14) = 2479.137826_wp
  dE(2,1,12) = 4627.723931_wp
  dE(2,1,10) = 6445.758336_wp
  dE(2,1,8) = 7933.241025_wp
  dE(1,1,6) = 9090.172009_wp

  dE(2,2,12) = 8476.381751_wp
  dE(2,2,10) = 10294.416151_wp
  dE(2,2,8) = 11781.898838_wp
  dE(2,2,6) = 12938.829825_wp
  dE(2,2,4) = 13765.209097_wp
  dE(1,2,2) = 14261.036662_wp

  dE(2,3,8) = 35149.092017_wp
  dE(2,3,6) = 36306.023001_wp
  dE(2,3,4) = 37132.402274_wp
  ! coefficients for Tb
  ! iLn=2; iLS=1, iJ=15/2 (2.0_wp*iJ+1)
  Jx(2,1,16,0,0,0,0) = -34.0_wp/231.0_wp
  Jx(2,1,16,4,0,0,0) = -2329.0_wp/22022.0_wp
  Jx(2,1,16,4,-4,0,0) = (-2329.0_wp*sqrt(5.0_wp))/(22022.0_wp*sqrt(14.0_wp))
  Jx(2,1,16,4,4,0,0) = (-2329.0_wp*sqrt(5.0_wp))/(22022.0_wp*sqrt(14.0_wp))
  Jx(2,1,16,6,0,0,0) = -18.0_wp/11011.0_wp
  Jx(2,1,16,6,-4,0,0) = (9.0_wp*sqrt(2.0_wp))/(1573.0_wp*sqrt(7.0_wp))
  Jx(2,1,16,6,4,0,0) = (9.0_wp*sqrt(2.0_wp))/(1573.0_wp*sqrt(7.0_wp))
  Jx(2,1,16,1,0,1,0) = 34.0_wp/231.0_wp
  Jx(2,1,16,1,-1,1,1) = -34.0_wp/231.0_wp
  Jx(2,1,16,1,1,1,-1) = -34.0_wp/231.0_wp
  Jx(2,1,16,3,0,1,0) = 4658.0_wp/99099.0_wp
  Jx(2,1,16,3,-1,1,1) = 2329.0_wp/(33033.0_wp*sqrt(6.0_wp))
  Jx(2,1,16,3,1,1,-1) = 2329.0_wp/(33033.0_wp*sqrt(6.0_wp))
  Jx(2,1,16,3,3,1,1) = (2329.0_wp*sqrt(5.0_wp))/(99099.0_wp*sqrt(2.0_wp))
  Jx(2,1,16,3,-3,1,-1) = (2329.0_wp*sqrt(5.0_wp))/(99099.0_wp*sqrt(2.0_wp))
  Jx(2,1,16,5,0,1,0) = 13939.0_wp/234234.0_wp
  Jx(2,1,16,5,-4,1,0) = (2615.0_wp*sqrt(5.0_wp))/(78078.0_wp*sqrt(14.0_wp))
  Jx(2,1,16,5,4,1,0) = (2615.0_wp*sqrt(5.0_wp))/(78078.0_wp*sqrt(14.0_wp))
  Jx(2,1,16,5,-1,1,1) = (-389.0_wp*sqrt(5.0_wp))/(11154.0_wp*sqrt(3.0_wp))
  Jx(2,1,16,5,1,1,-1) = (-389.0_wp*sqrt(5.0_wp))/(11154.0_wp*sqrt(3.0_wp))
  Jx(2,1,16,5,3,1,1) = (-3371.0_wp*sqrt(5.0_wp))/(234234.0_wp*sqrt(14.0_wp))
  Jx(2,1,16,5,-3,1,-1) = (-3371.0_wp*sqrt(5.0_wp))/(234234.0_wp*sqrt(14.0_wp))
  Jx(2,1,16,5,-5,1,1) = (-13831)/(78078.0_wp*sqrt(14.0_wp))
  Jx(2,1,16,5,5,1,-1) = (-13831)/(78078.0_wp*sqrt(14.0_wp))
  Jx(2,1,16,7,0,1,0) = 18.0_wp/20449.0_wp
  Jx(2,1,16,7,-4,1,0) = (-9.0_wp*sqrt(6.0_wp))/(1859.0_wp*sqrt(77.0_wp))
  Jx(2,1,16,7,4,1,0) = (-9.0_wp*sqrt(6.0_wp))/(1859.0_wp*sqrt(77.0_wp))
  Jx(2,1,16,7,-1,1,1) = -36.0_wp/(20449.0_wp*sqrt(7.0_wp))
  Jx(2,1,16,7,1,1,-1) = -36.0_wp/(20449.0_wp*sqrt(7.0_wp))
  Jx(2,1,16,7,3,1,1) = (18.0_wp*sqrt(3.0_wp))/(20449.0_wp*sqrt(7.0_wp))
  Jx(2,1,16,7,-3,1,-1) = (18.0_wp*sqrt(3.0_wp))/(20449.0_wp*sqrt(7.0_wp))
  Jx(2,1,16,7,-5,1,1) = (18.0_wp*sqrt(3.0_wp))/(1859.0_wp*sqrt(77.0_wp))
  Jx(2,1,16,7,5,1,-1) = (18.0_wp*sqrt(3.0_wp))/(1859.0_wp*sqrt(77.0_wp))
  ! coefficients for Tb
  ! iLn=2; iLS=1, iJ=13/2 (2.0_wp*iJ+1)
  Jx(2,1,14,0,0,0,0) = -2.0_wp/33.0_wp
  Jx(2,1,14,4,0,0,0) = 145.0_wp/3146.0_wp
  Jx(2,1,14,4,-4,0,0) = (145.0_wp*sqrt(5.0_wp))/(3146.0_wp*sqrt(14.0_wp))
  Jx(2,1,14,4,4,0,0) = (145.0_wp*sqrt(5.0_wp))/(3146.0_wp*sqrt(14.0_wp))
  Jx(2,1,14,6,0,0,0) = 9.0_wp/3146.0_wp
  Jx(2,1,14,6,-4,0,0) = (-9.0_wp*sqrt(7.0_wp))/(3146.0_wp*sqrt(2.0_wp))
  Jx(2,1,14,6,4,0,0) = (-9.0_wp*sqrt(7.0_wp))/(3146.0_wp*sqrt(2.0_wp))
  Jx(2,1,14,1,0,1,0) = 2.0_wp/33.0_wp
  Jx(2,1,14,1,-1,1,1) = -2.0_wp/33.0_wp
  Jx(2,1,14,1,1,1,-1) = -2.0_wp/33.0_wp
  Jx(2,1,14,3,0,1,0) = -290.0_wp/14157.0_wp
  Jx(2,1,14,3,-1,1,1) = -145.0_wp/(4719.0_wp*sqrt(6.0_wp))
  Jx(2,1,14,3,1,1,-1) = -145.0_wp/(4719.0_wp*sqrt(6.0_wp))
  Jx(2,1,14,3,3,1,1) = (-145.0_wp*sqrt(5.0_wp))/(14157.0_wp*sqrt(2.0_wp))
  Jx(2,1,14,3,-3,1,-1) = (-145.0_wp*sqrt(5.0_wp))/(14157.0_wp*sqrt(2.0_wp))
  Jx(2,1,14,5,0,1,0) = -901.0_wp/33462.0_wp
  Jx(2,1,14,5,-4,1,0) = (-137.0_wp*sqrt(5.0_wp))/(11154.0_wp*sqrt(14.0_wp))
  Jx(2,1,14,5,4,1,0) = (-137.0_wp*sqrt(5.0_wp))/(11154.0_wp*sqrt(14.0_wp))
  Jx(2,1,14,5,-1,1,1) = (82.0_wp*sqrt(5.0_wp))/(5577.0_wp*sqrt(3.0_wp))
  Jx(2,1,14,5,1,1,-1) = (82.0_wp*sqrt(5.0_wp))/(5577.0_wp*sqrt(3.0_wp))
  Jx(2,1,14,5,3,1,1) = (163.0_wp*sqrt(5.0_wp))/(16731.0_wp*sqrt(14.0_wp))
  Jx(2,1,14,5,-3,1,-1) = (163.0_wp*sqrt(5.0_wp))/(16731.0_wp*sqrt(14.0_wp))
  Jx(2,1,14,5,-5,1,1) = 437.0_wp/(5577.0_wp*sqrt(14.0_wp))
  Jx(2,1,14,5,5,1,-1) = 437.0_wp/(5577.0_wp*sqrt(14.0_wp))
  Jx(2,1,14,7,0,1,0) = -63.0_wp/40898.0_wp
  Jx(2,1,14,7,-4,1,0) = (9.0_wp*sqrt(21.0_wp))/(3718.0_wp*sqrt(22.0_wp))
  Jx(2,1,14,7,4,1,0) = (9.0_wp*sqrt(21.0_wp))/(3718.0_wp*sqrt(22.0_wp))
  Jx(2,1,14,7,-1,1,1) = (9.0_wp*sqrt(7.0_wp))/20449.0_wp
  Jx(2,1,14,7,1,1,-1) = (9.0_wp*sqrt(7.0_wp))/20449.0_wp
  Jx(2,1,14,7,3,1,1) = (-9.0_wp*sqrt(21.0_wp))/40898.0_wp
  Jx(2,1,14,7,-3,1,-1) = (-9.0_wp*sqrt(21.0_wp))/40898.0_wp
  Jx(2,1,14,7,-5,1,1) = (-9.0_wp*sqrt(21.0_wp))/(3718.0_wp*sqrt(11.0_wp))
  Jx(2,1,14,7,5,1,-1) = (-9.0_wp*sqrt(21.0_wp))/(3718.0_wp*sqrt(11.0_wp))
  ! coefficients for Tb
  ! iLn=2; iLS=1, iJ=11/2 (2.0_wp*iJ+1)
  Jx(2,1,12,0,0,0,0) = -10.0_wp/693.0_wp
  Jx(2,1,12,4,0,0,0) = 200.0_wp/33033.0_wp
  Jx(2,1,12,4,-4,0,0) = (100.0_wp*sqrt(10.0_wp))/(33033.0_wp*sqrt(7.0_wp))
  Jx(2,1,12,4,4,0,0) = (100.0_wp*sqrt(10.0_wp))/(33033.0_wp*sqrt(7.0_wp))
  Jx(2,1,12,6,0,0,0) = -160.0_wp/99099.0_wp
  Jx(2,1,12,6,-4,0,0) = (80.0_wp*sqrt(2.0_wp))/(14157.0_wp*sqrt(7.0_wp))
  Jx(2,1,12,6,4,0,0) = (80.0_wp*sqrt(2.0_wp))/(14157.0_wp*sqrt(7.0_wp))
  Jx(2,1,12,1,0,1,0) = 10.0_wp/693.0_wp
  Jx(2,1,12,1,-1,1,1) = -10.0_wp/693.0_wp
  Jx(2,1,12,1,1,1,-1) = -10.0_wp/693.0_wp
  Jx(2,1,12,3,0,1,0) = -800.0_wp/297297.0_wp
  Jx(2,1,12,3,-1,1,1) = (-200.0_wp*sqrt(2.0_wp))/(99099.0_wp*sqrt(3.0_wp))
  Jx(2,1,12,3,1,1,-1) = (-200.0_wp*sqrt(2.0_wp))/(99099.0_wp*sqrt(3.0_wp))
  Jx(2,1,12,3,3,1,1) = (-200.0_wp*sqrt(10.0_wp))/297297.0_wp
  Jx(2,1,12,3,-3,1,-1) = (-200.0_wp*sqrt(10.0_wp))/297297.0_wp
  Jx(2,1,12,5,0,1,0) = -920.0_wp/351351.0_wp
  Jx(2,1,12,5,-4,1,0) = (-20.0_wp*sqrt(10.0_wp))/(10647.0_wp*sqrt(7.0_wp))
  Jx(2,1,12,5,4,1,0) = (-20.0_wp*sqrt(10.0_wp))/(10647.0_wp*sqrt(7.0_wp))
  Jx(2,1,12,5,-1,1,1) = (40.0_wp*sqrt(5.0_wp))/(16731.0_wp*sqrt(3.0_wp))
  Jx(2,1,12,5,1,1,-1) = (40.0_wp*sqrt(5.0_wp))/(16731.0_wp*sqrt(3.0_wp))
  Jx(2,1,12,5,3,1,1) = (-340.0_wp*sqrt(10.0_wp))/(351351.0_wp*sqrt(7.0_wp))
  Jx(2,1,12,5,-3,1,-1) = (-340.0_wp*sqrt(10.0_wp))/(351351.0_wp*sqrt(7.0_wp))
  Jx(2,1,12,5,-5,1,1) = (60.0_wp*sqrt(2.0_wp))/(13013.0_wp*sqrt(7.0_wp))
  Jx(2,1,12,5,5,1,-1) = (60.0_wp*sqrt(2.0_wp))/(13013.0_wp*sqrt(7.0_wp))
  Jx(2,1,12,7,0,1,0) = 160.0_wp/184041.0_wp
  Jx(2,1,12,7,-4,1,0) = (-80.0_wp*sqrt(2.0_wp))/(5577.0_wp*sqrt(231.0_wp))
  Jx(2,1,12,7,4,1,0) = (-80.0_wp*sqrt(2.0_wp))/(5577.0_wp*sqrt(231.0_wp))
  Jx(2,1,12,7,-1,1,1) = -320.0_wp/(184041.0_wp*sqrt(7.0_wp))
  Jx(2,1,12,7,1,1,-1) = -320.0_wp/(184041.0_wp*sqrt(7.0_wp))
  Jx(2,1,12,7,3,1,1) = 160.0_wp/(61347.0_wp*sqrt(21.0_wp))
  Jx(2,1,12,7,-3,1,-1) = 160.0_wp/(61347.0_wp*sqrt(21.0_wp))
  Jx(2,1,12,7,-5,1,1) = 160.0_wp/(5577.0_wp*sqrt(231.0_wp))
  Jx(2,1,12,7,5,1,-1) = 160.0_wp/(5577.0_wp*sqrt(231.0_wp))
  ! coefficients for Tb
  ! iLn=2; iLS=1, iJ=9/2 (2.0_wp*iJ+1)
  Jx(2,1,10,0,0,0,0) = -5.0_wp/2376.0_wp
  Jx(2,1,10,4,0,0,0) = -995.0_wp/679536.0_wp
  Jx(2,1,10,4,-4,0,0) = (-995.0_wp*sqrt(5.0_wp))/(679536.0_wp*sqrt(14.0_wp))
  Jx(2,1,10,4,4,0,0) = (-995.0_wp*sqrt(5.0_wp))/(679536.0_wp*sqrt(14.0_wp))
  Jx(2,1,10,6,0,0,0) = 25.0_wp/75504.0_wp
  Jx(2,1,10,6,-4,0,0) = (-25.0_wp*sqrt(7.0_wp))/(75504.0_wp*sqrt(2.0_wp))
  Jx(2,1,10,6,4,0,0) = (-25.0_wp*sqrt(7.0_wp))/(75504.0_wp*sqrt(2.0_wp))
  Jx(2,1,10,1,0,1,0) = 5.0_wp/2376.0_wp
  Jx(2,1,10,1,-1,1,1) = -5.0_wp/2376.0_wp
  Jx(2,1,10,1,1,1,-1) = -5.0_wp/2376.0_wp
  Jx(2,1,10,3,0,1,0) = 995.0_wp/1528956.0_wp
  Jx(2,1,10,3,-1,1,1) = 995.0_wp/(1019304.0_wp*sqrt(6.0_wp))
  Jx(2,1,10,3,1,1,-1) = 995.0_wp/(1019304.0_wp*sqrt(6.0_wp))
  Jx(2,1,10,3,3,1,1) = (995.0_wp*sqrt(5.0_wp))/(3057912.0_wp*sqrt(2.0_wp))
  Jx(2,1,10,3,-3,1,-1) = (995.0_wp*sqrt(5.0_wp))/(3057912.0_wp*sqrt(2.0_wp))
  Jx(2,1,10,5,0,1,0) = 4775.0_wp/7227792.0_wp
  Jx(2,1,10,5,-4,1,0) = (185.0_wp*sqrt(5.0_wp))/(219024.0_wp*sqrt(14.0_wp))
  Jx(2,1,10,5,4,1,0) = (185.0_wp*sqrt(5.0_wp))/(219024.0_wp*sqrt(14.0_wp))
  Jx(2,1,10,5,-1,1,1) = (-85.0_wp*sqrt(5.0_wp))/(150579.0_wp*sqrt(3.0_wp))
  Jx(2,1,10,5,1,1,-1) = (-85.0_wp*sqrt(5.0_wp))/(150579.0_wp*sqrt(3.0_wp))
  Jx(2,1,10,5,3,1,1) = (1345.0_wp*sqrt(5.0_wp))/(3613896.0_wp*sqrt(14.0_wp))
  Jx(2,1,10,5,-3,1,-1) = (1345.0_wp*sqrt(5.0_wp))/(3613896.0_wp*sqrt(14.0_wp))
  Jx(2,1,10,5,-5,1,1) = -2725.0_wp/(1204632.0_wp*sqrt(14.0_wp))
  Jx(2,1,10,5,5,1,-1) = -2725.0_wp/(1204632.0_wp*sqrt(14.0_wp))
  Jx(2,1,10,7,0,1,0) = -175.0_wp/981552.0_wp
  Jx(2,1,10,7,-4,1,0) = (25.0_wp*sqrt(7.0_wp))/(29744.0_wp*sqrt(66.0_wp))
  Jx(2,1,10,7,4,1,0) = (25.0_wp*sqrt(7.0_wp))/(29744.0_wp*sqrt(66.0_wp))
  Jx(2,1,10,7,-1,1,1) = (25.0_wp*sqrt(7.0_wp))/490776.0_wp
  Jx(2,1,10,7,1,1,-1) = (25.0_wp*sqrt(7.0_wp))/490776.0_wp
  Jx(2,1,10,7,3,1,1) = (-25.0_wp*sqrt(7.0_wp))/(327184.0_wp*sqrt(3.0_wp))
  Jx(2,1,10,7,-3,1,-1) = (-25.0_wp*sqrt(7.0_wp))/(327184.0_wp*sqrt(3.0_wp))
  Jx(2,1,10,7,-5,1,1) = (-25.0_wp*sqrt(7.0_wp))/(29744.0_wp*sqrt(33.0_wp))
  Jx(2,1,10,7,5,1,-1) = (-25.0_wp*sqrt(7.0_wp))/(29744.0_wp*sqrt(33.0_wp))
  ! coefficients for Tb
  ! iLn=2; iLS=1, iJ=7/2 (2.0_wp*iJ+1)
  Jx(2,1,8,0,0,0,0) = -65.0_wp/407484.0_wp
  Jx(2,1,8,4,0,0,0) = -155.0_wp/1280664.0_wp
  Jx(2,1,8,4,-4,0,0) = (-155.0_wp*sqrt(5.0_wp))/(1280664.0_wp*sqrt(14.0_wp))
  Jx(2,1,8,4,4,0,0) = (-155.0_wp*sqrt(5.0_wp))/(1280664.0_wp*sqrt(14.0_wp))
  Jx(2,1,8,6,0,0,0) = -5.0_wp/249018.0_wp
  Jx(2,1,8,6,-4,0,0) = 5.0_wp/(35574.0_wp*sqrt(14.0_wp))
  Jx(2,1,8,6,4,0,0) = 5.0_wp/(35574.0_wp*sqrt(14.0_wp))
  Jx(2,1,8,1,0,1,0) = 65.0_wp/407484.0_wp
  Jx(2,1,8,1,-1,1,1) = -65.0_wp/407484.0_wp
  Jx(2,1,8,1,1,1,-1) = -65.0_wp/407484.0_wp
  Jx(2,1,8,3,0,1,0) = 155.0_wp/2881494.0_wp
  Jx(2,1,8,3,-1,1,1) = 155.0_wp/(1920996.0_wp*sqrt(6.0_wp))
  Jx(2,1,8,3,1,1,-1) = 155.0_wp/(1920996.0_wp*sqrt(6.0_wp))
  Jx(2,1,8,3,3,1,1) = (155.0_wp*sqrt(5.0_wp))/(5762988.0_wp*sqrt(2.0_wp))
  Jx(2,1,8,3,-3,1,-1) = (155.0_wp*sqrt(5.0_wp))/(5762988.0_wp*sqrt(2.0_wp))
  Jx(2,1,8,5,0,1,0) = 7295.0_wp/95351256.0_wp
  Jx(2,1,8,5,-4,1,0) = (85.0_wp*sqrt(5.0_wp))/(4540536.0_wp*sqrt(14.0_wp))
  Jx(2,1,8,5,4,1,0) = (85.0_wp*sqrt(5.0_wp))/(4540536.0_wp*sqrt(14.0_wp))
  Jx(2,1,8,5,-1,1,1) = (-1135.0_wp*sqrt(5.0_wp))/(31783752.0_wp*sqrt(3.0_wp))
  Jx(2,1,8,5,1,1,-1) = (-1135.0_wp*sqrt(5.0_wp))/(31783752.0_wp*sqrt(3.0_wp))
  Jx(2,1,8,5,3,1,1) = (-625.0_wp*sqrt(5.0_wp))/(13621608.0_wp*sqrt(14.0_wp))
  Jx(2,1,8,5,-3,1,-1) = (-625.0_wp*sqrt(5.0_wp))/(13621608.0_wp*sqrt(14.0_wp))
  Jx(2,1,8,5,-5,1,1) = -965.0_wp/(4540536.0_wp*sqrt(14.0_wp))
  Jx(2,1,8,5,5,1,-1) = -965.0_wp/(4540536.0_wp*sqrt(14.0_wp))
  Jx(2,1,8,7,0,1,0) = 5.0_wp/462462.0_wp
  Jx(2,1,8,7,-4,1,0) = -5.0_wp/(14014.0_wp*sqrt(462.0_wp))
  Jx(2,1,8,7,4,1,0) = -5.0_wp/(14014.0_wp*sqrt(462.0_wp))
  Jx(2,1,8,7,-1,1,1) = -5.0_wp/(231231.0_wp*sqrt(7.0_wp))
  Jx(2,1,8,7,1,1,-1) = -5.0_wp/(231231.0_wp*sqrt(7.0_wp))
  Jx(2,1,8,7,3,1,1) = 5.0_wp/(154154.0_wp*sqrt(21.0_wp))
  Jx(2,1,8,7,-3,1,-1) = 5.0_wp/(154154.0_wp*sqrt(21.0_wp))
  Jx(2,1,8,7,-5,1,1) = 5.0_wp/(14014.0_wp*sqrt(231.0_wp))
  Jx(2,1,8,7,5,1,-1) = 5.0_wp/(14014.0_wp*sqrt(231.0_wp))
  ! coefficients for Tb
  ! iLn=2; iLS=1, iJ=5/2 (2.0_wp*iJ+1)
  Jx(2,1,6,0,0,0,0) = -1.0_wp/271656.0_wp
  Jx(2,1,6,4,0,0,0) = 1.0_wp/284592.0_wp
  Jx(2,1,6,4,-4,0,0) = (sqrt(5.0_wp))/(284592.0_wp*sqrt(14.0_wp))
  Jx(2,1,6,4,4,0,0) = (sqrt(5.0_wp))/(284592.0_wp*sqrt(14.0_wp))
  Jx(2,1,6,6,0,0,0) = 1.0_wp/5976432.0_wp
  Jx(2,1,6,6,-4,0,0) = -1.0_wp/(853776.0_wp*sqrt(14.0_wp))
  Jx(2,1,6,6,4,0,0) = -1.0_wp/(853776.0_wp*sqrt(14.0_wp))
  Jx(2,1,6,1,0,1,0) = 1.0_wp/271656.0_wp
  Jx(2,1,6,1,-1,1,1) = -1.0_wp/271656.0_wp
  Jx(2,1,6,1,1,1,-1) = -1.0_wp/271656.0_wp
  Jx(2,1,6,3,0,1,0) = -1.0_wp/640332.0_wp
  Jx(2,1,6,3,-1,1,1) = -1.0_wp/(426888.0_wp*sqrt(6.0_wp))
  Jx(2,1,6,3,1,1,-1) = -1.0_wp/(426888.0_wp*sqrt(6.0_wp))
  Jx(2,1,6,3,3,1,1) = (-sqrt(5.0_wp))/(1280664.0_wp*sqrt(2.0_wp))
  Jx(2,1,6,3,-3,1,-1) = (-sqrt(5.0_wp))/(1280664.0_wp*sqrt(2.0_wp))
  Jx(2,1,6,5,0,1,0) = -43.0_wp/21189168.0_wp
  Jx(2,1,6,5,-4,1,0) = (-sqrt(5.0_wp))/(1009008.0_wp*sqrt(14.0_wp))
  Jx(2,1,6,5,4,1,0) = (-sqrt(5.0_wp))/(1009008.0_wp*sqrt(14.0_wp))
  Jx(2,1,6,5,-1,1,1) = (sqrt(5.0_wp))/(882882.0_wp*sqrt(3.0_wp))
  Jx(2,1,6,5,1,1,-1) = (sqrt(5.0_wp))/(882882.0_wp*sqrt(3.0_wp))
  Jx(2,1,6,5,3,1,1) = (sqrt(5.0_wp))/(1513512.0_wp*sqrt(14.0_wp))
  Jx(2,1,6,5,-3,1,-1) = (sqrt(5.0_wp))/(1513512.0_wp*sqrt(14.0_wp))
  Jx(2,1,6,5,-5,1,1) = 1.0_wp/(168168.0_wp*sqrt(14.0_wp))
  Jx(2,1,6,5,5,1,-1) = 1.0_wp/(168168.0_wp*sqrt(14.0_wp))
  Jx(2,1,6,7,0,1,0) = -1.0_wp/11099088.0_wp
  Jx(2,1,6,7,-4,1,0) = 1.0_wp/(336336.0_wp*sqrt(462.0_wp))
  Jx(2,1,6,7,4,1,0) = 1.0_wp/(336336.0_wp*sqrt(462.0_wp))
  Jx(2,1,6,7,-1,1,1) = 1.0_wp/(5549544.0_wp*sqrt(7.0_wp))
  Jx(2,1,6,7,1,1,-1) = 1.0_wp/(5549544.0_wp*sqrt(7.0_wp))
  Jx(2,1,6,7,3,1,1) = -1.0_wp/(3699696.0_wp*sqrt(21.0_wp))
  Jx(2,1,6,7,-3,1,-1) = -1.0_wp/(3699696.0_wp*sqrt(21.0_wp))
  Jx(2,1,6,7,-5,1,1) = -1.0_wp/(336336.0_wp*sqrt(231.0_wp))
  Jx(2,1,6,7,5,1,-1) = -1.0_wp/(336336.0_wp*sqrt(231.0_wp))

  ! coefficients for Tb
  ! iLn=2; iLS=2, iJ=11/2 (2.0_wp*iJ+1)
  Jx(2,2,12,0,0,0,0) = -13.0_wp/198.0_wp
  Jx(2,2,12,4,0,0,0) = 10.0_wp/363.0_wp
  Jx(2,2,12,4,-4,0,0) = (5.0_wp*sqrt(10.0_wp))/(363.0_wp*sqrt(7.0_wp))
  Jx(2,2,12,4,4,0,0) = (5.0_wp*sqrt(10.0_wp))/(363.0_wp*sqrt(7.0_wp))
  Jx(2,2,12,6,0,0,0) = -8.0_wp/1089.0_wp
  Jx(2,2,12,6,-4,0,0) = (4.0_wp*sqrt(14.0_wp))/1089.0_wp
  Jx(2,2,12,6,4,0,0) = (4.0_wp*sqrt(14.0_wp))/1089.0_wp
  Jx(2,2,12,1,0,1,0) = 13.0_wp/198.0_wp
  Jx(2,2,12,1,-1,1,1) = -13.0_wp/198.0_wp
  Jx(2,2,12,1,1,1,-1) = -13.0_wp/198.0_wp
  Jx(2,2,12,3,0,1,0) = -40.0_wp/3267.0_wp
  Jx(2,2,12,3,-1,1,1) = (-10.0_wp*sqrt(2.0_wp))/(1089.0_wp*sqrt(3.0_wp))
  Jx(2,2,12,3,1,1,-1) = (-10.0_wp*sqrt(2.0_wp))/(1089.0_wp*sqrt(3.0_wp))
  Jx(2,2,12,3,3,1,1) = (-10.0_wp*sqrt(10.0_wp))/3267.0_wp
  Jx(2,2,12,3,-3,1,-1) = (-10.0_wp*sqrt(10.0_wp))/3267.0_wp
  Jx(2,2,12,5,0,1,0) = -46.0_wp/3861.0_wp
  Jx(2,2,12,5,-4,1,0) = (-sqrt(10.0_wp))/(117.0_wp*sqrt(7.0_wp))
  Jx(2,2,12,5,4,1,0) = (-sqrt(10.0_wp))/(117.0_wp*sqrt(7.0_wp))
  Jx(2,2,12,5,-1,1,1) = (14.0_wp*sqrt(5.0_wp))/(1287.0_wp*sqrt(3.0_wp))
  Jx(2,2,12,5,1,1,-1) = (14.0_wp*sqrt(5.0_wp))/(1287.0_wp*sqrt(3.0_wp))
  Jx(2,2,12,5,3,1,1) = (-17.0_wp*sqrt(10.0_wp))/(3861.0_wp*sqrt(7.0_wp))
  Jx(2,2,12,5,-3,1,-1) = (-17.0_wp*sqrt(10.0_wp))/(3861.0_wp*sqrt(7.0_wp))
  Jx(2,2,12,5,-5,1,1) = (3.0_wp*sqrt(2.0_wp))/(143.0_wp*sqrt(7.0_wp))
  Jx(2,2,12,5,5,1,-1) = (3.0_wp*sqrt(2.0_wp))/(143.0_wp*sqrt(7.0_wp))
  Jx(2,2,12,7,0,1,0) = 56.0_wp/14157.0_wp
  Jx(2,2,12,7,-4,1,0) = (-4.0_wp*sqrt(14.0_wp))/(429.0_wp*sqrt(33.0_wp))
  Jx(2,2,12,7,4,1,0) = (-4.0_wp*sqrt(14.0_wp))/(429.0_wp*sqrt(33.0_wp))
  Jx(2,2,12,7,-1,1,1) = (-16.0_wp*sqrt(7.0_wp))/14157.0_wp
  Jx(2,2,12,7,1,1,-1) = (-16.0_wp*sqrt(7.0_wp))/14157.0_wp
  Jx(2,2,12,7,3,1,1) = (8.0_wp*sqrt(7.0_wp))/(4719.0_wp*sqrt(3.0_wp))
  Jx(2,2,12,7,-3,1,-1) = (8.0_wp*sqrt(7.0_wp))/(4719.0_wp*sqrt(3.0_wp))
  Jx(2,2,12,7,-5,1,1) = (8.0_wp*sqrt(7.0_wp))/(429.0_wp*sqrt(33.0_wp))
  Jx(2,2,12,7,5,1,-1) = (8.0_wp*sqrt(7.0_wp))/(429.0_wp*sqrt(33.0_wp))
  ! coefficients for Tb
  ! iLn=2; iLS=2, iJ=9/2 (2.0_wp*iJ+1)
  Jx(2,2,10,0,0,0,0) = -65.0_wp/1188.0_wp
  Jx(2,2,10,4,0,0,0) = -995.0_wp/26136.0_wp
  Jx(2,2,10,4,-4,0,0) = (-995.0_wp*sqrt(5.0_wp))/(26136.0_wp*sqrt(14.0_wp))
  Jx(2,2,10,4,4,0,0) = (-995.0_wp*sqrt(5.0_wp))/(26136.0_wp*sqrt(14.0_wp))
  Jx(2,2,10,6,0,0,0) = 25.0_wp/2904.0_wp
  Jx(2,2,10,6,-4,0,0) = (-25.0_wp*sqrt(7.0_wp))/(2904.0_wp*sqrt(2.0_wp))
  Jx(2,2,10,6,4,0,0) = (-25.0_wp*sqrt(7.0_wp))/(2904.0_wp*sqrt(2.0_wp))
  Jx(2,2,10,1,0,1,0) = 65.0_wp/1188.0_wp
  Jx(2,2,10,1,-1,1,1) = -65.0_wp/1188.0_wp
  Jx(2,2,10,1,1,1,-1) = -65.0_wp/1188.0_wp
  Jx(2,2,10,3,0,1,0) = 995.0_wp/58806.0_wp
  Jx(2,2,10,3,-1,1,1) = 995.0_wp/(39204.0_wp*sqrt(6.0_wp))
  Jx(2,2,10,3,1,1,-1) = 995.0_wp/(39204.0_wp*sqrt(6.0_wp))
  Jx(2,2,10,3,3,1,1) = (995.0_wp*sqrt(5.0_wp))/(117612.0_wp*sqrt(2.0_wp))
  Jx(2,2,10,3,-3,1,-1) = (995.0_wp*sqrt(5.0_wp))/(117612.0_wp*sqrt(2.0_wp))
  Jx(2,2,10,5,0,1,0) = 4775.0_wp/277992.0_wp
  Jx(2,2,10,5,-4,1,0) = (185.0_wp*sqrt(5.0_wp))/(8424.0_wp*sqrt(14.0_wp))
  Jx(2,2,10,5,4,1,0) = (185.0_wp*sqrt(5.0_wp))/(8424.0_wp*sqrt(14.0_wp))
  Jx(2,2,10,5,-1,1,1) = (-170.0_wp*sqrt(5.0_wp))/(11583.0_wp*sqrt(3.0_wp))
  Jx(2,2,10,5,1,1,-1) = (-170.0_wp*sqrt(5.0_wp))/(11583.0_wp*sqrt(3.0_wp))
  Jx(2,2,10,5,3,1,1) = (1345.0_wp*sqrt(5.0_wp))/(138996.0_wp*sqrt(14.0_wp))
  Jx(2,2,10,5,-3,1,-1) = (1345.0_wp*sqrt(5.0_wp))/(138996.0_wp*sqrt(14.0_wp))
  Jx(2,2,10,5,-5,1,1) = -2725.0_wp/(46332.0_wp*sqrt(14.0_wp))
  Jx(2,2,10,5,5,1,-1) = -2725.0_wp/(46332.0_wp*sqrt(14.0_wp))
  Jx(2,2,10,7,0,1,0) = -175.0_wp/37752.0_wp
  Jx(2,2,10,7,-4,1,0) = (25.0_wp*sqrt(7.0_wp))/(1144.0_wp*sqrt(66.0_wp))
  Jx(2,2,10,7,4,1,0) = (25.0_wp*sqrt(7.0_wp))/(1144.0_wp*sqrt(66.0_wp))
  Jx(2,2,10,7,-1,1,1) = (25.0_wp*sqrt(7.0_wp))/18876.0_wp
  Jx(2,2,10,7,1,1,-1) = (25.0_wp*sqrt(7.0_wp))/18876.0_wp
  Jx(2,2,10,7,3,1,1) = (-25.0_wp*sqrt(7.0_wp))/(12584.0_wp*sqrt(3.0_wp))
  Jx(2,2,10,7,-3,1,-1) = (-25.0_wp*sqrt(7.0_wp))/(12584.0_wp*sqrt(3.0_wp))
  Jx(2,2,10,7,-5,1,1) = (-25.0_wp*sqrt(7.0_wp))/(1144.0_wp*sqrt(33.0_wp))
  Jx(2,2,10,7,5,1,-1) = (-25.0_wp*sqrt(7.0_wp))/(1144.0_wp*sqrt(33.0_wp))
  ! coefficients for Tb
  ! iLn=2; iLS=2, iJ=7/2 (2.0_wp*iJ+1)
  Jx(2,2,8,0,0,0,0) = -26.0_wp/1323.0_wp
  Jx(2,2,8,4,0,0,0) = -31.0_wp/2079.0_wp
  Jx(2,2,8,4,-4,0,0) = (-31.0_wp*sqrt(5.0_wp))/(2079.0_wp*sqrt(14.0_wp))
  Jx(2,2,8,4,4,0,0) = (-31.0_wp*sqrt(5.0_wp))/(2079.0_wp*sqrt(14.0_wp))
  Jx(2,2,8,6,0,0,0) = -4.0_wp/1617.0_wp
  Jx(2,2,8,6,-4,0,0) = (2.0_wp*sqrt(2.0_wp))/(231.0_wp*sqrt(7.0_wp))
  Jx(2,2,8,6,4,0,0) = (2.0_wp*sqrt(2.0_wp))/(231.0_wp*sqrt(7.0_wp))
  Jx(2,2,8,1,0,1,0) = 26.0_wp/1323.0_wp
  Jx(2,2,8,1,-1,1,1) = -26.0_wp/1323.0_wp
  Jx(2,2,8,1,1,1,-1) = -26.0_wp/1323.0_wp
  Jx(2,2,8,3,0,1,0) = 124.0_wp/18711.0_wp
  Jx(2,2,8,3,-1,1,1) = (31.0_wp*sqrt(2.0_wp))/(6237.0_wp*sqrt(3.0_wp))
  Jx(2,2,8,3,1,1,-1) = (31.0_wp*sqrt(2.0_wp))/(6237.0_wp*sqrt(3.0_wp))
  Jx(2,2,8,3,3,1,1) = (31.0_wp*sqrt(10.0_wp))/18711.0_wp
  Jx(2,2,8,3,-3,1,-1) = (31.0_wp*sqrt(10.0_wp))/18711.0_wp
  Jx(2,2,8,5,0,1,0) = 1459.0_wp/154791.0_wp
  Jx(2,2,8,5,-4,1,0) = (17.0_wp*sqrt(5.0_wp))/(7371.0_wp*sqrt(14.0_wp))
  Jx(2,2,8,5,4,1,0) = (17.0_wp*sqrt(5.0_wp))/(7371.0_wp*sqrt(14.0_wp))
  Jx(2,2,8,5,-1,1,1) = (-227.0_wp*sqrt(5.0_wp))/(51597.0_wp*sqrt(3.0_wp))
  Jx(2,2,8,5,1,1,-1) = (-227.0_wp*sqrt(5.0_wp))/(51597.0_wp*sqrt(3.0_wp))
  Jx(2,2,8,5,3,1,1) = (-125.0_wp*sqrt(5.0_wp))/(22113.0_wp*sqrt(14.0_wp))
  Jx(2,2,8,5,-3,1,-1) = (-125.0_wp*sqrt(5.0_wp))/(22113.0_wp*sqrt(14.0_wp))
  Jx(2,2,8,5,-5,1,1) = -193.0_wp/(7371.0_wp*sqrt(14.0_wp))
  Jx(2,2,8,5,5,1,-1) = -193.0_wp/(7371.0_wp*sqrt(14.0_wp))
  Jx(2,2,8,7,0,1,0) = 4.0_wp/3003.0_wp
  Jx(2,2,8,7,-4,1,0) = (-2.0_wp*sqrt(2.0_wp))/(91.0_wp*sqrt(231.0_wp))
  Jx(2,2,8,7,4,1,0) = (-2.0_wp*sqrt(2.0_wp))/(91.0_wp*sqrt(231.0_wp))
  Jx(2,2,8,7,-1,1,1) = -8.0_wp/(3003.0_wp*sqrt(7.0_wp))
  Jx(2,2,8,7,1,1,-1) = -8.0_wp/(3003.0_wp*sqrt(7.0_wp))
  Jx(2,2,8,7,3,1,1) = 4.0_wp/(1001.0_wp*sqrt(21.0_wp))
  Jx(2,2,8,7,-3,1,-1) = 4.0_wp/(1001.0_wp*sqrt(21.0_wp))
  Jx(2,2,8,7,-5,1,1) = 4.0_wp/(91.0_wp*sqrt(231.0_wp))
  Jx(2,2,8,7,5,1,-1) = 4.0_wp/(91.0_wp*sqrt(231.0_wp))
  ! coefficients for Tb
  ! iLn=2; iLS=2, iJ=5/2 (2.0_wp*iJ+1)
  Jx(2,2,6,0,0,0,0) = -5.0_wp/1764.0_wp
  Jx(2,2,6,4,0,0,0) = 5.0_wp/1848.0_wp
  Jx(2,2,6,4,-4,0,0) = (5.0_wp*sqrt(5.0_wp))/(1848.0_wp*sqrt(14.0_wp))
  Jx(2,2,6,4,4,0,0) = (5.0_wp*sqrt(5.0_wp))/(1848.0_wp*sqrt(14.0_wp))
  Jx(2,2,6,6,0,0,0) = 5.0_wp/38808.0_wp
  Jx(2,2,6,6,-4,0,0) = -5.0_wp/(5544.0_wp*sqrt(14.0_wp))
  Jx(2,2,6,6,4,0,0) = -5.0_wp/(5544.0_wp*sqrt(14.0_wp))
  Jx(2,2,6,1,0,1,0) = 5.0_wp/1764.0_wp
  Jx(2,2,6,1,-1,1,1) = -5.0_wp/1764.0_wp
  Jx(2,2,6,1,1,1,-1) = -5.0_wp/1764.0_wp
  Jx(2,2,6,3,0,1,0) = -5.0_wp/4158.0_wp
  Jx(2,2,6,3,-1,1,1) = -5.0_wp/(2772.0_wp*sqrt(6.0_wp))
  Jx(2,2,6,3,1,1,-1) = -5.0_wp/(2772.0_wp*sqrt(6.0_wp))
  Jx(2,2,6,3,3,1,1) = (-5.0_wp*sqrt(5.0_wp))/(8316.0_wp*sqrt(2.0_wp))
  Jx(2,2,6,3,-3,1,-1) = (-5.0_wp*sqrt(5.0_wp))/(8316.0_wp*sqrt(2.0_wp))
  Jx(2,2,6,5,0,1,0) = -215.0_wp/137592.0_wp
  Jx(2,2,6,5,-4,1,0) = (-5.0_wp*sqrt(5.0_wp))/(6552.0_wp*sqrt(14.0_wp))
  Jx(2,2,6,5,4,1,0) = (-5.0_wp*sqrt(5.0_wp))/(6552.0_wp*sqrt(14.0_wp))
  Jx(2,2,6,5,-1,1,1) = (5.0_wp*sqrt(5.0_wp))/(5733.0_wp*sqrt(3.0_wp))
  Jx(2,2,6,5,1,1,-1) = (5.0_wp*sqrt(5.0_wp))/(5733.0_wp*sqrt(3.0_wp))
  Jx(2,2,6,5,3,1,1) = (5.0_wp*sqrt(5.0_wp))/(9828.0_wp*sqrt(14.0_wp))
  Jx(2,2,6,5,-3,1,-1) = (5.0_wp*sqrt(5.0_wp))/(9828.0_wp*sqrt(14.0_wp))
  Jx(2,2,6,5,-5,1,1) = 5.0_wp/(1092.0_wp*sqrt(14.0_wp))
  Jx(2,2,6,5,5,1,-1) = 5.0_wp/(1092.0_wp*sqrt(14.0_wp))
  Jx(2,2,6,7,0,1,0) = -5.0_wp/72072.0_wp
  Jx(2,2,6,7,-4,1,0) = 5.0_wp/(2184.0_wp*sqrt(462.0_wp))
  Jx(2,2,6,7,4,1,0) = 5.0_wp/(2184.0_wp*sqrt(462.0_wp))
  Jx(2,2,6,7,-1,1,1) = 5.0_wp/(36036.0_wp*sqrt(7.0_wp))
  Jx(2,2,6,7,1,1,-1) = 5.0_wp/(36036.0_wp*sqrt(7.0_wp))
  Jx(2,2,6,7,3,1,1) = -5.0_wp/(24024.0_wp*sqrt(21.0_wp))
  Jx(2,2,6,7,-3,1,-1) = -5.0_wp/(24024.0_wp*sqrt(21.0_wp))
  Jx(2,2,6,7,-5,1,1) = -5.0_wp/(2184.0_wp*sqrt(231.0_wp))
  Jx(2,2,6,7,5,1,-1) = -5.0_wp/(2184.0_wp*sqrt(231.0_wp))

  ! coefficients for Tb
  ! iLn=2; iLS=3, iJ=7/2 (2.0_wp*iJ+1)
  Jx(2,3,8,0,0,0,0) = -39.0_wp/1372.0_wp
  Jx(2,3,8,4,0,0,0) = -93.0_wp/4312.0_wp
  Jx(2,3,8,4,-4,0,0) = (-93.0_wp*sqrt(5.0_wp))/(4312.0_wp*sqrt(14.0_wp))
  Jx(2,3,8,4,4,0,0) = (-93.0_wp*sqrt(5.0_wp))/(4312.0_wp*sqrt(14.0_wp))
  Jx(2,3,8,6,0,0,0) = -27.0_wp/7546.0_wp
  Jx(2,3,8,6,-4,0,0) = 27.0_wp/(1078.0_wp*sqrt(14.0_wp))
  Jx(2,3,8,6,4,0,0) = 27.0_wp/(1078.0_wp*sqrt(14.0_wp))
  Jx(2,3,8,1,0,1,0) = 39.0_wp/1372.0_wp
  Jx(2,3,8,1,-1,1,1) = -39.0_wp/1372.0_wp
  Jx(2,3,8,1,1,1,-1) = -39.0_wp/1372.0_wp
  Jx(2,3,8,3,0,1,0) = 31.0_wp/3234.0_wp
  Jx(2,3,8,3,-1,1,1) = 31.0_wp/(2156.0_wp*sqrt(6.0_wp))
  Jx(2,3,8,3,1,1,-1) = 31.0_wp/(2156.0_wp*sqrt(6.0_wp))
  Jx(2,3,8,3,3,1,1) = (31.0_wp*sqrt(5.0_wp))/(6468.0_wp*sqrt(2.0_wp))
  Jx(2,3,8,3,-3,1,-1) = (31.0_wp*sqrt(5.0_wp))/(6468.0_wp*sqrt(2.0_wp))
  Jx(2,3,8,5,0,1,0) = 1459.0_wp/107016.0_wp
  Jx(2,3,8,5,-4,1,0) = (17.0_wp*sqrt(5.0_wp))/(5096.0_wp*sqrt(14.0_wp))
  Jx(2,3,8,5,4,1,0) = (17.0_wp*sqrt(5.0_wp))/(5096.0_wp*sqrt(14.0_wp))
  Jx(2,3,8,5,-1,1,1) = (-227.0_wp*sqrt(5.0_wp))/(35672.0_wp*sqrt(3.0_wp))
  Jx(2,3,8,5,1,1,-1) = (-227.0_wp*sqrt(5.0_wp))/(35672.0_wp*sqrt(3.0_wp))
  Jx(2,3,8,5,3,1,1) = (-125.0_wp*sqrt(5.0_wp))/(15288.0_wp*sqrt(14.0_wp))
  Jx(2,3,8,5,-3,1,-1) = (-125.0_wp*sqrt(5.0_wp))/(15288.0_wp*sqrt(14.0_wp))
  Jx(2,3,8,5,-5,1,1) = -193.0_wp/(5096.0_wp*sqrt(14.0_wp))
  Jx(2,3,8,5,5,1,-1) = -193.0_wp/(5096.0_wp*sqrt(14.0_wp))
  Jx(2,3,8,7,0,1,0) = 27.0_wp/14014.0_wp
  Jx(2,3,8,7,-4,1,0) = (-27.0_wp*sqrt(3.0_wp))/(1274.0_wp*sqrt(154.0_wp))
  Jx(2,3,8,7,4,1,0) = (-27.0_wp*sqrt(3.0_wp))/(1274.0_wp*sqrt(154.0_wp))
  Jx(2,3,8,7,-1,1,1) = -27.0_wp/(7007.0_wp*sqrt(7.0_wp))
  Jx(2,3,8,7,1,1,-1) = -27.0_wp/(7007.0_wp*sqrt(7.0_wp))
  Jx(2,3,8,7,3,1,1) = (27.0_wp*sqrt(3.0_wp))/(14014.0_wp*sqrt(7.0_wp))
  Jx(2,3,8,7,-3,1,-1) = (27.0_wp*sqrt(3.0_wp))/(14014.0_wp*sqrt(7.0_wp))
  Jx(2,3,8,7,-5,1,1) = (27.0_wp*sqrt(3.0_wp))/(1274.0_wp*sqrt(77.0_wp))
  Jx(2,3,8,7,5,1,-1) = (27.0_wp*sqrt(3.0_wp))/(1274.0_wp*sqrt(77.0_wp))
  ! coefficients for Tb
  ! iLn=2; iLS=3, iJ=5/2 (2.0_wp*iJ+1)
  Jx(2,3,6,0,0,0,0) = -45.0_wp/1372.0_wp
  Jx(2,3,6,4,0,0,0) = 135.0_wp/4312.0_wp
  Jx(2,3,6,4,-4,0,0) = (135.0_wp*sqrt(5.0_wp))/(4312.0_wp*sqrt(14.0_wp))
  Jx(2,3,6,4,4,0,0) = (135.0_wp*sqrt(5.0_wp))/(4312.0_wp*sqrt(14.0_wp))
  Jx(2,3,6,6,0,0,0) = 45.0_wp/30184.0_wp
  Jx(2,3,6,6,-4,0,0) = -45.0_wp/(4312.0_wp*sqrt(14.0_wp))
  Jx(2,3,6,6,4,0,0) = -45.0_wp/(4312.0_wp*sqrt(14.0_wp))
  Jx(2,3,6,1,0,1,0) = 45.0_wp/1372.0_wp
  Jx(2,3,6,1,-1,1,1) = -45.0_wp/1372.0_wp
  Jx(2,3,6,1,1,1,-1) = -45.0_wp/1372.0_wp
  Jx(2,3,6,3,0,1,0) = -15.0_wp/1078.0_wp
  Jx(2,3,6,3,-1,1,1) = (-15.0_wp*sqrt(3.0_wp))/(2156.0_wp*sqrt(2.0_wp))
  Jx(2,3,6,3,1,1,-1) = (-15.0_wp*sqrt(3.0_wp))/(2156.0_wp*sqrt(2.0_wp))
  Jx(2,3,6,3,3,1,1) = (-15.0_wp*sqrt(5.0_wp))/(2156.0_wp*sqrt(2.0_wp))
  Jx(2,3,6,3,-3,1,-1) = (-15.0_wp*sqrt(5.0_wp))/(2156.0_wp*sqrt(2.0_wp))
  Jx(2,3,6,5,0,1,0) = -645.0_wp/35672.0_wp
  Jx(2,3,6,5,-4,1,0) = (-45.0_wp*sqrt(5.0_wp))/(5096.0_wp*sqrt(14.0_wp))
  Jx(2,3,6,5,4,1,0) = (-45.0_wp*sqrt(5.0_wp))/(5096.0_wp*sqrt(14.0_wp))
  Jx(2,3,6,5,-1,1,1) = (15.0_wp*sqrt(15.0_wp))/4459.0_wp
  Jx(2,3,6,5,1,1,-1) = (15.0_wp*sqrt(15.0_wp))/4459.0_wp
  Jx(2,3,6,5,3,1,1) = (15.0_wp*sqrt(5.0_wp))/(2548.0_wp*sqrt(14.0_wp))
  Jx(2,3,6,5,-3,1,-1) = (15.0_wp*sqrt(5.0_wp))/(2548.0_wp*sqrt(14.0_wp))
  Jx(2,3,6,5,-5,1,1) = 135.0_wp/(2548.0_wp*sqrt(14.0_wp))
  Jx(2,3,6,5,5,1,-1) = 135.0_wp/(2548.0_wp*sqrt(14.0_wp))
  Jx(2,3,6,7,0,1,0) = -45.0_wp/56056.0_wp
  Jx(2,3,6,7,-4,1,0) = (45.0_wp*sqrt(3.0_wp))/(5096.0_wp*sqrt(154.0_wp))
  Jx(2,3,6,7,4,1,0) = (45.0_wp*sqrt(3.0_wp))/(5096.0_wp*sqrt(154.0_wp))
  Jx(2,3,6,7,-1,1,1) = 45.0_wp/(28028.0_wp*sqrt(7.0_wp))
  Jx(2,3,6,7,1,1,-1) = 45.0_wp/(28028.0_wp*sqrt(7.0_wp))
  Jx(2,3,6,7,3,1,1) = (-45.0_wp*sqrt(3.0_wp))/(56056.0_wp*sqrt(7.0_wp))
  Jx(2,3,6,7,-3,1,-1) = (-45.0_wp*sqrt(3.0_wp))/(56056.0_wp*sqrt(7.0_wp))
  Jx(2,3,6,7,-5,1,1) = (-45.0_wp*sqrt(3.0_wp))/(5096.0_wp*sqrt(77.0_wp))
  Jx(2,3,6,7,5,1,-1) = (-45.0_wp*sqrt(3.0_wp))/(5096.0_wp*sqrt(77.0_wp))
  !---------------------------------------------------------------------

  !--------------------
  !    Dy2+
  ! J     energy (cm-1)
  !--------------------
  ! 8.0        0.000000
  ! 7.0     3695.732363
  ! 6.0     6929.498167
  ! 5.0     9701.297441
  ! 4.0    12011.130158
  !
  ! 5.0    19110.154549
  ! 4.0    21419.987263
  ! 3.0    23267.853448
  ! 2.0    24653.753075
  ! 1.0    25577.686169
  !
  ! 6.0    27522.424450
  ! 5.0    30294.223753
  ! 4.0    32604.056435
  ! 3.0    34451.922647
  ! 2.0    35837.822260
  !
  ! 4.0    45708.645925
  ! 3.0    47556.512094
  ! 2.0    48942.411735
  ! 1.0    49866.344819
  ! 0.0    50328.311366
  dE(3,1,17) = Zero
  dE(3,1,15) = 3695.732363_wp
  dE(3,1,13) = 6929.498167_wp
  dE(3,1,11) = 9701.297441_wp
  dE(3,1,9) = 12011.130158_wp

  dE(3,2,11) = 19110.154549_wp
  dE(3,2,9) = 21419.987263_wp
  dE(3,2,7) = 23267.853448_wp
  dE(3,2,5) = 24653.753075_wp
  dE(3,2,3) = 25577.686169_wp

  dE(3,3,13) = 27522.424450_wp
  dE(3,3,11) = 30294.223753_wp
  dE(3,3,9) = 32604.056435_wp
  dE(3,3,7) = 34451.922647_wp
  dE(3,3,5) = 35837.822260_wp

  dE(3,4,9) = 45708.645925_wp
  dE(3,4,7) = 47556.512094_wp
  dE(3,4,5) = 48942.411735_wp
  dE(3,4,3) = 49866.344819_wp
  dE(3,4,1) = 50328.311366_wp
  ! coefficients for Dy
  ! iLn=3; iLS=1, iJ=8 (2.0_wp*iJ+1)
  Jx(3,1,17,0,0,0,0) = -969.0_wp/6370.0_wp
  Jx(3,1,17,4,0,0,0) = 881.0_wp/7280.0_wp
  Jx(3,1,17,4,-4,0,0) = 881.0_wp/(1456.0_wp*sqrt(70.0_wp))
  Jx(3,1,17,4,4,0,0) = 881.0_wp/(1456.0_wp*sqrt(70.0_wp))
  Jx(3,1,17,6,0,0,0) = 135.0_wp/10192.0_wp
  Jx(3,1,17,6,-4,0,0) = -135.0_wp/(1456.0_wp*sqrt(14.0_wp))
  Jx(3,1,17,6,4,0,0) = -135.0_wp/(1456.0_wp*sqrt(14.0_wp))
  Jx(3,1,17,1,0,1,0) = 969.0_wp/6370.0_wp
  Jx(3,1,17,1,-1,1,1) = -969.0_wp/6370.0_wp
  Jx(3,1,17,1,1,1,-1) = -969.0_wp/6370.0_wp
  Jx(3,1,17,3,0,1,0) = -881.0_wp/16380.0_wp
  Jx(3,1,17,3,-1,1,1) = -881.0_wp/(10920.0_wp*sqrt(6.0_wp))
  Jx(3,1,17,3,1,1,-1) = -881.0_wp/(10920.0_wp*sqrt(6.0_wp))
  Jx(3,1,17,3,3,1,1) = -881.0_wp/(6552.0_wp*sqrt(10.0_wp))
  Jx(3,1,17,3,-3,1,-1) = -881.0_wp/(6552.0_wp*sqrt(10.0_wp))
  Jx(3,1,17,5,0,1,0) = -87461.0_wp/1192464.0_wp
  Jx(3,1,17,5,-4,1,0) = -7403.0_wp/(56784.0_wp*sqrt(70.0_wp))
  Jx(3,1,17,5,4,1,0) = -7403.0_wp/(56784.0_wp*sqrt(70.0_wp))
  Jx(3,1,17,5,-1,1,1) = 4631.0_wp/(24843.0_wp*sqrt(15.0_wp))
  Jx(3,1,17,5,1,1,-1) = 4631.0_wp/(24843.0_wp*sqrt(15.0_wp))
  Jx(3,1,17,5,3,1,1) = 14839.0_wp/(85176.0_wp*sqrt(70.0_wp))
  Jx(3,1,17,5,-3,1,-1) = 14839.0_wp/(85176.0_wp*sqrt(70.0_wp))
  Jx(3,1,17,5,-5,1,1) = (121.0_wp*sqrt(7.0_wp))/(4056.0_wp*sqrt(2.0_wp))
  Jx(3,1,17,5,5,1,-1) = (121.0_wp*sqrt(7.0_wp))/(4056.0_wp*sqrt(2.0_wp))
  Jx(3,1,17,7,0,1,0) = -135.0_wp/18928.0_wp
  Jx(3,1,17,7,-4,1,0) = (135.0_wp*sqrt(33.0_wp))/(18928.0_wp*sqrt(14.0_wp))
  Jx(3,1,17,7,4,1,0) = (135.0_wp*sqrt(33.0_wp))/(18928.0_wp*sqrt(14.0_wp))
  Jx(3,1,17,7,-1,1,1) = 135.0_wp/(9464.0_wp*sqrt(7.0_wp))
  Jx(3,1,17,7,1,1,-1) = 135.0_wp/(9464.0_wp*sqrt(7.0_wp))
  Jx(3,1,17,7,3,1,1) = (-135.0_wp*sqrt(3.0_wp))/(18928.0_wp*sqrt(7.0_wp))
  Jx(3,1,17,7,-3,1,-1) = (-135.0_wp*sqrt(3.0_wp))/(18928.0_wp*sqrt(7.0_wp))
  Jx(3,1,17,7,-5,1,1) = (-135.0_wp*sqrt(33.0_wp))/(18928.0_wp*sqrt(7.0_wp))
  Jx(3,1,17,7,5,1,-1) = (-135.0_wp*sqrt(33.0_wp))/(18928.0_wp*sqrt(7.0_wp))
  ! coefficients for Dy
  ! iLn=3; iLS=1, iJ=7 (2.0_wp*iJ+1)
  Jx(3,1,15,0,0,0,0) = -51.0_wp/1274.0_wp
  Jx(3,1,15,4,0,0,0) = 3.0_wp/112.0_wp
  Jx(3,1,15,4,-4,0,0) = (3.0_wp*sqrt(5.0_wp))/(112.0_wp*sqrt(14.0_wp))
  Jx(3,1,15,4,4,0,0) = (3.0_wp*sqrt(5.0_wp))/(112.0_wp*sqrt(14.0_wp))
  Jx(3,1,15,6,0,0,0) = -75.0_wp/10192.0_wp
  Jx(3,1,15,6,-4,0,0) = 75.0_wp/(1456.0_wp*sqrt(14.0_wp))
  Jx(3,1,15,6,4,0,0) = 75.0_wp/(1456.0_wp*sqrt(14.0_wp))
  Jx(3,1,15,1,0,1,0) = 51.0_wp/1274.0_wp
  Jx(3,1,15,1,-1,1,1) = -51.0_wp/1274.0_wp
  Jx(3,1,15,1,1,1,-1) = -51.0_wp/1274.0_wp
  Jx(3,1,15,3,0,1,0) = -1.0_wp/84.0_wp
  Jx(3,1,15,3,-1,1,1) = -1.0_wp/(56.0_wp*sqrt(6.0_wp))
  Jx(3,1,15,3,1,1,-1) = -1.0_wp/(56.0_wp*sqrt(6.0_wp))
  Jx(3,1,15,3,3,1,1) = (-sqrt(5.0_wp))/(168.0_wp*sqrt(2.0_wp))
  Jx(3,1,15,3,-3,1,-1) = (-sqrt(5.0_wp))/(168.0_wp*sqrt(2.0_wp))
  Jx(3,1,15,5,0,1,0) = -4565.0_wp/397488.0_wp
  Jx(3,1,15,5,-4,1,0) = (-319.0_wp*sqrt(5.0_wp))/(18928.0_wp*sqrt(14.0_wp))
  Jx(3,1,15,5,4,1,0) = (-319.0_wp*sqrt(5.0_wp))/(18928.0_wp*sqrt(14.0_wp))
  Jx(3,1,15,5,-1,1,1) = (88.0_wp*sqrt(5.0_wp))/(8281.0_wp*sqrt(3.0_wp))
  Jx(3,1,15,5,1,1,-1) = (88.0_wp*sqrt(5.0_wp))/(8281.0_wp*sqrt(3.0_wp))
  Jx(3,1,15,5,3,1,1) = (-253.0_wp*sqrt(5.0_wp))/(28392.0_wp*sqrt(14.0_wp))
  Jx(3,1,15,5,-3,1,-1) = (-253.0_wp*sqrt(5.0_wp))/(28392.0_wp*sqrt(14.0_wp))
  Jx(3,1,15,5,-5,1,1) = 55.0_wp/(1352.0_wp*sqrt(14.0_wp))
  Jx(3,1,15,5,5,1,-1) = 55.0_wp/(1352.0_wp*sqrt(14.0_wp))
  Jx(3,1,15,7,0,1,0) = 75.0_wp/18928.0_wp
  Jx(3,1,15,7,-4,1,0) = (-75.0_wp*sqrt(33.0_wp))/(18928.0_wp*sqrt(14.0_wp))
  Jx(3,1,15,7,4,1,0) = (-75.0_wp*sqrt(33.0_wp))/(18928.0_wp*sqrt(14.0_wp))
  Jx(3,1,15,7,-1,1,1) = -75.0_wp/(9464.0_wp*sqrt(7.0_wp))
  Jx(3,1,15,7,1,1,-1) = -75.0_wp/(9464.0_wp*sqrt(7.0_wp))
  Jx(3,1,15,7,3,1,1) = (75.0_wp*sqrt(3.0_wp))/(18928.0_wp*sqrt(7.0_wp))
  Jx(3,1,15,7,-3,1,-1) = (75.0_wp*sqrt(3.0_wp))/(18928.0_wp*sqrt(7.0_wp))
  Jx(3,1,15,7,-5,1,1) = (75.0_wp*sqrt(33.0_wp))/(18928.0_wp*sqrt(7.0_wp))
  Jx(3,1,15,7,5,1,-1) = (75.0_wp*sqrt(33.0_wp))/(18928.0_wp*sqrt(7.0_wp))
  ! coefficients for Dy
  ! iLn=3; iLS=1, iJ=6 (2.0_wp*iJ+1)
  Jx(3,1,13,0,0,0,0) = -136.0_wp/29645.0_wp
  Jx(3,1,13,4,0,0,0) = -137.0_wp/46585.0_wp
  Jx(3,1,13,4,-4,0,0) = -137.0_wp/(9317.0_wp*sqrt(70.0_wp))
  Jx(3,1,13,4,4,0,0) = -137.0_wp/(9317.0_wp*sqrt(70.0_wp))
  Jx(3,1,13,6,0,0,0) = 135.0_wp/130438.0_wp
  Jx(3,1,13,6,-4,0,0) = -135.0_wp/(18634.0_wp*sqrt(14.0_wp))
  Jx(3,1,13,6,4,0,0) = -135.0_wp/(18634.0_wp*sqrt(14.0_wp))
  Jx(3,1,13,1,0,1,0) = 136.0_wp/29645.0_wp
  Jx(3,1,13,1,-1,1,1) = -136.0_wp/29645.0_wp
  Jx(3,1,13,1,1,1,-1) = -136.0_wp/29645.0_wp
  Jx(3,1,13,3,0,1,0) = 548.0_wp/419265.0_wp
  Jx(3,1,13,3,-1,1,1) = (137.0_wp*sqrt(2.0_wp))/(139755.0_wp*sqrt(3.0_wp))
  Jx(3,1,13,3,1,1,-1) = (137.0_wp*sqrt(2.0_wp))/(139755.0_wp*sqrt(3.0_wp))
  Jx(3,1,13,3,3,1,1) = (137.0_wp*sqrt(2.0_wp))/(83853.0_wp*sqrt(5.0_wp))
  Jx(3,1,13,3,-3,1,-1) = (137.0_wp*sqrt(2.0_wp))/(83853.0_wp*sqrt(5.0_wp))
  Jx(3,1,13,5,0,1,0) = 802.0_wp/693693.0_wp
  Jx(3,1,13,5,-4,1,0) = (173.0_wp*sqrt(2.0_wp))/(33033.0_wp*sqrt(35.0_wp))
  Jx(3,1,13,5,4,1,0) = (173.0_wp*sqrt(2.0_wp))/(33033.0_wp*sqrt(35.0_wp))
  Jx(3,1,13,5,-1,1,1) = -2819.0_wp/(462462.0_wp*sqrt(15.0_wp))
  Jx(3,1,13,5,1,1,-1) = -2819.0_wp/(462462.0_wp*sqrt(15.0_wp))
  Jx(3,1,13,5,3,1,1) = 1333.0_wp/(198198.0_wp*sqrt(70.0_wp))
  Jx(3,1,13,5,-3,1,-1) = 1333.0_wp/(198198.0_wp*sqrt(70.0_wp))
  Jx(3,1,13,5,-5,1,1) = -41.0_wp/(9438.0_wp*sqrt(14.0_wp))
  Jx(3,1,13,5,5,1,-1) = -41.0_wp/(9438.0_wp*sqrt(14.0_wp))
  Jx(3,1,13,7,0,1,0) = -135.0_wp/242242.0_wp
  Jx(3,1,13,7,-4,1,0) = (135.0_wp*sqrt(3.0_wp))/(22022.0_wp*sqrt(154.0_wp))
  Jx(3,1,13,7,4,1,0) = (135.0_wp*sqrt(3.0_wp))/(22022.0_wp*sqrt(154.0_wp))
  Jx(3,1,13,7,-1,1,1) = 135.0_wp/(121121.0_wp*sqrt(7.0_wp))
  Jx(3,1,13,7,1,1,-1) = 135.0_wp/(121121.0_wp*sqrt(7.0_wp))
  Jx(3,1,13,7,3,1,1) = (-135.0_wp*sqrt(3.0_wp))/(242242.0_wp*sqrt(7.0_wp))
  Jx(3,1,13,7,-3,1,-1) = (-135.0_wp*sqrt(3.0_wp))/(242242.0_wp*sqrt(7.0_wp))
  Jx(3,1,13,7,-5,1,1) = (-135.0_wp*sqrt(3.0_wp))/(22022.0_wp*sqrt(77.0_wp))
  Jx(3,1,13,7,5,1,-1) = (-135.0_wp*sqrt(3.0_wp))/(22022.0_wp*sqrt(77.0_wp))
  ! coefficients for Dy
  ! iLn=3; iLS=1, iJ=5 (2.0_wp*iJ+1)
  Jx(3,1,11,0,0,0,0) = -8.0_wp/35035.0_wp
  Jx(3,1,11,4,0,0,0) = -1.0_wp/4235.0_wp
  Jx(3,1,11,4,-4,0,0) = -1.0_wp/(847.0_wp*sqrt(70.0_wp))
  Jx(3,1,11,4,4,0,0) = -1.0_wp/(847.0_wp*sqrt(70.0_wp))
  Jx(3,1,11,6,0,0,0) = -27.0_wp/770770.0_wp
  Jx(3,1,11,6,-4,0,0) = 27.0_wp/(110110.0_wp*sqrt(14.0_wp))
  Jx(3,1,11,6,4,0,0) = 27.0_wp/(110110.0_wp*sqrt(14.0_wp))
  Jx(3,1,11,1,0,1,0) = 8.0_wp/35035.0_wp
  Jx(3,1,11,1,-1,1,1) = -8.0_wp/35035.0_wp
  Jx(3,1,11,1,1,1,-1) = -8.0_wp/35035.0_wp
  Jx(3,1,11,3,0,1,0) = 4.0_wp/38115.0_wp
  Jx(3,1,11,3,-1,1,1) = (sqrt(2.0_wp))/(12705.0_wp*sqrt(3.0_wp))
  Jx(3,1,11,3,1,1,-1) = (sqrt(2.0_wp))/(12705.0_wp*sqrt(3.0_wp))
  Jx(3,1,11,3,3,1,1) = (sqrt(2.0_wp))/(7623.0_wp*sqrt(5.0_wp))
  Jx(3,1,11,3,-3,1,-1) = (sqrt(2.0_wp))/(7623.0_wp*sqrt(5.0_wp))
  Jx(3,1,11,5,0,1,0) = 604.0_wp/4099095.0_wp
  Jx(3,1,11,5,-4,1,0) = (4.0_wp*sqrt(2.0_wp))/(39039.0_wp*sqrt(35.0_wp))
  Jx(3,1,11,5,4,1,0) = (4.0_wp*sqrt(2.0_wp))/(39039.0_wp*sqrt(35.0_wp))
  Jx(3,1,11,5,-1,1,1) = -193.0_wp/(546546.0_wp*sqrt(15.0_wp))
  Jx(3,1,11,5,1,1,-1) = -193.0_wp/(546546.0_wp*sqrt(15.0_wp))
  Jx(3,1,11,5,3,1,1) = -97.0_wp/(234234.0_wp*sqrt(70.0_wp))
  Jx(3,1,11,5,-3,1,-1) = -97.0_wp/(234234.0_wp*sqrt(70.0_wp))
  Jx(3,1,11,5,-5,1,1) = (-23)/(55770.0_wp*sqrt(14.0_wp))
  Jx(3,1,11,5,5,1,-1) = (-23)/(55770.0_wp*sqrt(14.0_wp))
  Jx(3,1,11,7,0,1,0) = 27.0_wp/1431430.0_wp
  Jx(3,1,11,7,-4,1,0) = (-27.0_wp*sqrt(3.0_wp))/(130130.0_wp*sqrt(154.0_wp))
  Jx(3,1,11,7,4,1,0) = (-27.0_wp*sqrt(3.0_wp))/(130130.0_wp*sqrt(154.0_wp))
  Jx(3,1,11,7,-1,1,1) = -27.0_wp/(715715.0_wp*sqrt(7.0_wp))
  Jx(3,1,11,7,1,1,-1) = -27.0_wp/(715715.0_wp*sqrt(7.0_wp))
  Jx(3,1,11,7,3,1,1) = (27.0_wp*sqrt(3.0_wp))/(1431430.0_wp*sqrt(7.0_wp))
  Jx(3,1,11,7,-3,1,-1) = (27.0_wp*sqrt(3.0_wp))/(1431430.0_wp*sqrt(7.0_wp))
  Jx(3,1,11,7,-5,1,1) = (27.0_wp*sqrt(3.0_wp))/(130130.0_wp*sqrt(77.0_wp))
  Jx(3,1,11,7,5,1,-1) = (27.0_wp*sqrt(3.0_wp))/(130130.0_wp*sqrt(77.0_wp))
  ! coefficients for Dy
  ! iLn=3; iLS=1, iJ=4 (2.0_wp*iJ+1)
  Jx(3,1,9,0,0,0,0) = -1.0_wp/330330.0_wp
  Jx(3,1,9,4,0,0,0) = 1.0_wp/346060.0_wp
  Jx(3,1,9,4,-4,0,0) = 1.0_wp/(69212.0_wp*sqrt(70.0_wp))
  Jx(3,1,9,4,4,0,0) = 1.0_wp/(69212.0_wp*sqrt(70.0_wp))
  Jx(3,1,9,6,0,0,0) = 1.0_wp/7267260.0_wp
  Jx(3,1,9,6,-4,0,0) = -1.0_wp/(1038180.0_wp*sqrt(14.0_wp))
  Jx(3,1,9,6,4,0,0) = -1.0_wp/(1038180.0_wp*sqrt(14.0_wp))
  Jx(3,1,9,1,0,1,0) = 1.0_wp/330330.0_wp
  Jx(3,1,9,1,-1,1,1) = -1.0_wp/330330.0_wp
  Jx(3,1,9,1,1,1,-1) = -1.0_wp/330330.0_wp
  Jx(3,1,9,3,0,1,0) = -1.0_wp/778635.0_wp
  Jx(3,1,9,3,-1,1,1) = -1.0_wp/(519090.0_wp*sqrt(6.0_wp))
  Jx(3,1,9,3,1,1,-1) = -1.0_wp/(519090.0_wp*sqrt(6.0_wp))
  Jx(3,1,9,3,3,1,1) = -1.0_wp/(311454.0_wp*sqrt(10.0_wp))
  Jx(3,1,9,3,-3,1,-1) = -1.0_wp/(311454.0_wp*sqrt(10.0_wp))
  Jx(3,1,9,5,0,1,0) = -43.0_wp/25765740.0_wp
  Jx(3,1,9,5,-4,1,0) = -1.0_wp/(245388.0_wp*sqrt(70.0_wp))
  Jx(3,1,9,5,4,1,0) = -1.0_wp/(245388.0_wp*sqrt(70.0_wp))
  Jx(3,1,9,5,-1,1,1) = 2.0_wp/(429429.0_wp*sqrt(15.0_wp))
  Jx(3,1,9,5,1,1,-1) = 2.0_wp/(429429.0_wp*sqrt(15.0_wp))
  Jx(3,1,9,5,3,1,1) = 1.0_wp/(368082.0_wp*sqrt(70.0_wp))
  Jx(3,1,9,5,-3,1,-1) = 1.0_wp/(368082.0_wp*sqrt(70.0_wp))
  Jx(3,1,9,5,-5,1,1) = 1.0_wp/(204490.0_wp*sqrt(14.0_wp))
  Jx(3,1,9,5,5,1,-1) = 1.0_wp/(204490.0_wp*sqrt(14.0_wp))
  Jx(3,1,9,7,0,1,0) = -1.0_wp/13496340.0_wp
  Jx(3,1,9,7,-4,1,0) = 1.0_wp/(408980.0_wp*sqrt(462.0_wp))
  Jx(3,1,9,7,4,1,0) = 1.0_wp/(408980.0_wp*sqrt(462.0_wp))
  Jx(3,1,9,7,-1,1,1) = 1.0_wp/(6748170.0_wp*sqrt(7.0_wp))
  Jx(3,1,9,7,1,1,-1) = 1.0_wp/(6748170.0_wp*sqrt(7.0_wp))
  Jx(3,1,9,7,3,1,1) = -1.0_wp/(4498780.0_wp*sqrt(21.0_wp))
  Jx(3,1,9,7,-3,1,-1) = -1.0_wp/(4498780.0_wp*sqrt(21.0_wp))
  Jx(3,1,9,7,-5,1,1) = -1.0_wp/(408980.0_wp*sqrt(231.0_wp))
  Jx(3,1,9,7,5,1,-1) = -1.0_wp/(408980.0_wp*sqrt(231.0_wp))

  ! coefficients for Dy
  ! iLn=3; iLS=2, iJ=6 (2.0_wp*iJ+1)
  Jx(3,2,13,0,0,0,0) = -170.0_wp/2541.0_wp
  Jx(3,2,13,4,0,0,0) = -685.0_wp/15972.0_wp
  Jx(3,2,13,4,-4,0,0) = (-685.0_wp*sqrt(5.0_wp))/(15972.0_wp*sqrt(14.0_wp))
  Jx(3,2,13,4,4,0,0) = (-685.0_wp*sqrt(5.0_wp))/(15972.0_wp*sqrt(14.0_wp))
  Jx(3,2,13,6,0,0,0) = 1125.0_wp/74536.0_wp
  Jx(3,2,13,6,-4,0,0) = -1125.0_wp/(10648.0_wp*sqrt(14.0_wp))
  Jx(3,2,13,6,4,0,0) = -1125.0_wp/(10648.0_wp*sqrt(14.0_wp))
  Jx(3,2,13,1,0,1,0) = 170.0_wp/2541.0_wp
  Jx(3,2,13,1,-1,1,1) = -170.0_wp/2541.0_wp
  Jx(3,2,13,1,1,1,-1) = -170.0_wp/2541.0_wp
  Jx(3,2,13,3,0,1,0) = 685.0_wp/35937.0_wp
  Jx(3,2,13,3,-1,1,1) = 685.0_wp/(23958.0_wp*sqrt(6.0_wp))
  Jx(3,2,13,3,1,1,-1) = 685.0_wp/(23958.0_wp*sqrt(6.0_wp))
  Jx(3,2,13,3,3,1,1) = (685.0_wp*sqrt(5.0_wp))/(71874.0_wp*sqrt(2.0_wp))
  Jx(3,2,13,3,-3,1,-1) = (685.0_wp*sqrt(5.0_wp))/(71874.0_wp*sqrt(2.0_wp))
  Jx(3,2,13,5,0,1,0) = 10025.0_wp/594594.0_wp
  Jx(3,2,13,5,-4,1,0) = (865.0_wp*sqrt(5.0_wp))/(28314.0_wp*sqrt(14.0_wp))
  Jx(3,2,13,5,4,1,0) = (865.0_wp*sqrt(5.0_wp))/(28314.0_wp*sqrt(14.0_wp))
  Jx(3,2,13,5,-1,1,1) = (-14095.0_wp*sqrt(5.0_wp))/(792792.0_wp*sqrt(3.0_wp))
  Jx(3,2,13,5,1,1,-1) = (-14095.0_wp*sqrt(5.0_wp))/(792792.0_wp*sqrt(3.0_wp))
  Jx(3,2,13,5,3,1,1) = (6665.0_wp*sqrt(5.0_wp))/(339768.0_wp*sqrt(14.0_wp))
  Jx(3,2,13,5,-3,1,-1) = (6665.0_wp*sqrt(5.0_wp))/(339768.0_wp*sqrt(14.0_wp))
  Jx(3,2,13,5,-5,1,1) = (-1025.0_wp*sqrt(7.0_wp))/(113256.0_wp*sqrt(2.0_wp))
  Jx(3,2,13,5,5,1,-1) = (-1025.0_wp*sqrt(7.0_wp))/(113256.0_wp*sqrt(2.0_wp))
  Jx(3,2,13,7,0,1,0) = -1125.0_wp/138424.0_wp
  Jx(3,2,13,7,-4,1,0) = (1125.0_wp*sqrt(3.0_wp))/(12584.0_wp*sqrt(154.0_wp))
  Jx(3,2,13,7,4,1,0) = (1125.0_wp*sqrt(3.0_wp))/(12584.0_wp*sqrt(154.0_wp))
  Jx(3,2,13,7,-1,1,1) = 1125.0_wp/(69212.0_wp*sqrt(7.0_wp))
  Jx(3,2,13,7,1,1,-1) = 1125.0_wp/(69212.0_wp*sqrt(7.0_wp))
  Jx(3,2,13,7,3,1,1) = (-1125.0_wp*sqrt(3.0_wp))/(138424.0_wp*sqrt(7.0_wp))
  Jx(3,2,13,7,-3,1,-1) = (-1125.0_wp*sqrt(3.0_wp))/(138424.0_wp*sqrt(7.0_wp))
  Jx(3,2,13,7,-5,1,1) = (-1125.0_wp*sqrt(3.0_wp))/(12584.0_wp*sqrt(77.0_wp))
  Jx(3,2,13,7,5,1,-1) = (-1125.0_wp*sqrt(3.0_wp))/(12584.0_wp*sqrt(77.0_wp))
  ! coefficients for Dy
  ! iLn=3; iLS=2, iJ=5 (2.0_wp*iJ+1)
  Jx(3,2,11,0,0,0,0) = -5.0_wp/231.0_wp
  Jx(3,2,11,4,0,0,0) = -65.0_wp/2904.0_wp
  Jx(3,2,11,4,-4,0,0) = (-65.0_wp*sqrt(5.0_wp))/(2904.0_wp*sqrt(14.0_wp))
  Jx(3,2,11,4,4,0,0) = (-65.0_wp*sqrt(5.0_wp))/(2904.0_wp*sqrt(14.0_wp))
  Jx(3,2,11,6,0,0,0) = -45.0_wp/13552.0_wp
  Jx(3,2,11,6,-4,0,0) = 45.0_wp/(1936.0_wp*sqrt(14.0_wp))
  Jx(3,2,11,6,4,0,0) = 45.0_wp/(1936.0_wp*sqrt(14.0_wp))
  Jx(3,2,11,1,0,1,0) = 5.0_wp/231.0_wp
  Jx(3,2,11,1,-1,1,1) = -5.0_wp/231.0_wp
  Jx(3,2,11,1,1,1,-1) = -5.0_wp/231.0_wp
  Jx(3,2,11,3,0,1,0) = 65.0_wp/6534.0_wp
  Jx(3,2,11,3,-1,1,1) = 65.0_wp/(4356.0_wp*sqrt(6.0_wp))
  Jx(3,2,11,3,1,1,-1) = 65.0_wp/(4356.0_wp*sqrt(6.0_wp))
  Jx(3,2,11,3,3,1,1) = (65.0_wp*sqrt(5.0_wp))/(13068.0_wp*sqrt(2.0_wp))
  Jx(3,2,11,3,-3,1,-1) = (65.0_wp*sqrt(5.0_wp))/(13068.0_wp*sqrt(2.0_wp))
  Jx(3,2,11,5,0,1,0) = 755.0_wp/54054.0_wp
  Jx(3,2,11,5,-4,1,0) = (5.0_wp*sqrt(5.0_wp))/(1287.0_wp*sqrt(14.0_wp))
  Jx(3,2,11,5,4,1,0) = (5.0_wp*sqrt(5.0_wp))/(1287.0_wp*sqrt(14.0_wp))
  Jx(3,2,11,5,-1,1,1) = (-965.0_wp*sqrt(5.0_wp))/(144144.0_wp*sqrt(3.0_wp))
  Jx(3,2,11,5,1,1,-1) = (-965.0_wp*sqrt(5.0_wp))/(144144.0_wp*sqrt(3.0_wp))
  Jx(3,2,11,5,3,1,1) = (-485.0_wp*sqrt(5.0_wp))/(61776.0_wp*sqrt(14.0_wp))
  Jx(3,2,11,5,-3,1,-1) = (-485.0_wp*sqrt(5.0_wp))/(61776.0_wp*sqrt(14.0_wp))
  Jx(3,2,11,5,-5,1,1) = (-115.0_wp*sqrt(7.0_wp))/(20592.0_wp*sqrt(2.0_wp))
  Jx(3,2,11,5,5,1,-1) = (-115.0_wp*sqrt(7.0_wp))/(20592.0_wp*sqrt(2.0_wp))
  Jx(3,2,11,7,0,1,0) = 45.0_wp/25168.0_wp
  Jx(3,2,11,7,-4,1,0) = (-45.0_wp*sqrt(3.0_wp))/(2288.0_wp*sqrt(154.0_wp))
  Jx(3,2,11,7,4,1,0) = (-45.0_wp*sqrt(3.0_wp))/(2288.0_wp*sqrt(154.0_wp))
  Jx(3,2,11,7,-1,1,1) = -45.0_wp/(12584.0_wp*sqrt(7.0_wp))
  Jx(3,2,11,7,1,1,-1) = -45.0_wp/(12584.0_wp*sqrt(7.0_wp))
  Jx(3,2,11,7,3,1,1) = (45.0_wp*sqrt(3.0_wp))/(25168.0_wp*sqrt(7.0_wp))
  Jx(3,2,11,7,-3,1,-1) = (45.0_wp*sqrt(3.0_wp))/(25168.0_wp*sqrt(7.0_wp))
  Jx(3,2,11,7,-5,1,1) = (45.0_wp*sqrt(3.0_wp))/(2288.0_wp*sqrt(77.0_wp))
  Jx(3,2,11,7,5,1,-1) = (45.0_wp*sqrt(3.0_wp))/(2288.0_wp*sqrt(77.0_wp))
  ! coefficients for Dy
  ! iLn=3; iLS=2, iJ=4 (2.0_wp*iJ+1)
  Jx(3,2,9,0,0,0,0) = -45.0_wp/23716.0_wp
  Jx(3,2,9,4,0,0,0) = 135.0_wp/74536.0_wp
  Jx(3,2,9,4,-4,0,0) = (135.0_wp*sqrt(5.0_wp))/(74536.0_wp*sqrt(14.0_wp))
  Jx(3,2,9,4,4,0,0) = (135.0_wp*sqrt(5.0_wp))/(74536.0_wp*sqrt(14.0_wp))
  Jx(3,2,9,6,0,0,0) = 45.0_wp/521752.0_wp
  Jx(3,2,9,6,-4,0,0) = -45.0_wp/(74536.0_wp*sqrt(14.0_wp))
  Jx(3,2,9,6,4,0,0) = -45.0_wp/(74536.0_wp*sqrt(14.0_wp))
  Jx(3,2,9,1,0,1,0) = 45.0_wp/23716.0_wp
  Jx(3,2,9,1,-1,1,1) = -45.0_wp/23716.0_wp
  Jx(3,2,9,1,1,1,-1) = -45.0_wp/23716.0_wp
  Jx(3,2,9,3,0,1,0) = -15.0_wp/18634.0_wp
  Jx(3,2,9,3,-1,1,1) = (-15.0_wp*sqrt(3.0_wp))/(37268.0_wp*sqrt(2.0_wp))
  Jx(3,2,9,3,1,1,-1) = (-15.0_wp*sqrt(3.0_wp))/(37268.0_wp*sqrt(2.0_wp))
  Jx(3,2,9,3,3,1,1) = (-15.0_wp*sqrt(5.0_wp))/(37268.0_wp*sqrt(2.0_wp))
  Jx(3,2,9,3,-3,1,-1) = (-15.0_wp*sqrt(5.0_wp))/(37268.0_wp*sqrt(2.0_wp))
  Jx(3,2,9,5,0,1,0) = -645.0_wp/616616.0_wp
  Jx(3,2,9,5,-4,1,0) = (-45.0_wp*sqrt(5.0_wp))/(88088.0_wp*sqrt(14.0_wp))
  Jx(3,2,9,5,4,1,0) = (-45.0_wp*sqrt(5.0_wp))/(88088.0_wp*sqrt(14.0_wp))
  Jx(3,2,9,5,-1,1,1) = (15.0_wp*sqrt(15.0_wp))/(77077)
  Jx(3,2,9,5,1,1,-1) = (15.0_wp*sqrt(15.0_wp))/(77077)
  Jx(3,2,9,5,3,1,1) = (15.0_wp*sqrt(5.0_wp))/(44044.0_wp*sqrt(14.0_wp))
  Jx(3,2,9,5,-3,1,-1) = (15.0_wp*sqrt(5.0_wp))/(44044.0_wp*sqrt(14.0_wp))
  Jx(3,2,9,5,-5,1,1) = 135.0_wp/(44044.0_wp*sqrt(14.0_wp))
  Jx(3,2,9,5,5,1,-1) = 135.0_wp/(44044.0_wp*sqrt(14.0_wp))
  Jx(3,2,9,7,0,1,0) = -45.0_wp/968968.0_wp
  Jx(3,2,9,7,-4,1,0) = (45.0_wp*sqrt(3.0_wp))/(88088.0_wp*sqrt(154.0_wp))
  Jx(3,2,9,7,4,1,0) = (45.0_wp*sqrt(3.0_wp))/(88088.0_wp*sqrt(154.0_wp))
  Jx(3,2,9,7,-1,1,1) = 45.0_wp/(484484.0_wp*sqrt(7.0_wp))
  Jx(3,2,9,7,1,1,-1) = 45.0_wp/(484484.0_wp*sqrt(7.0_wp))
  Jx(3,2,9,7,3,1,1) = (-45.0_wp*sqrt(3.0_wp))/(968968.0_wp*sqrt(7.0_wp))
  Jx(3,2,9,7,-3,1,-1) = (-45.0_wp*sqrt(3.0_wp))/(968968.0_wp*sqrt(7.0_wp))
  Jx(3,2,9,7,-5,1,1) = (-45.0_wp*sqrt(3.0_wp))/(88088.0_wp*sqrt(77.0_wp))
  Jx(3,2,9,7,5,1,-1) = (-45.0_wp*sqrt(3.0_wp))/(88088.0_wp*sqrt(77.0_wp))

  ! coefficients for Dy
  ! iLn=3; iLS=3, iJ=5 (2.0_wp*iJ+1)
  Jx(3,3,11,0,0,0,0) = -1.0_wp/35.0_wp
  Jx(3,3,11,4,0,0,0) = -13.0_wp/440.0_wp
  Jx(3,3,11,4,-4,0,0) = -13.0_wp/(88.0_wp*sqrt(70.0_wp))
  Jx(3,3,11,4,4,0,0) = -13.0_wp/(88.0_wp*sqrt(70.0_wp))
  Jx(3,3,11,6,0,0,0) = -27.0_wp/6160.0_wp
  Jx(3,3,11,6,-4,0,0) = 27.0_wp/(880.0_wp*sqrt(14.0_wp))
  Jx(3,3,11,6,4,0,0) = 27.0_wp/(880.0_wp*sqrt(14.0_wp))
  Jx(3,3,11,1,0,1,0) = 1.0_wp/35.0_wp
  Jx(3,3,11,1,-1,1,1) = -1.0_wp/35.0_wp
  Jx(3,3,11,1,1,1,-1) = -1.0_wp/35.0_wp
  Jx(3,3,11,3,0,1,0) = 13.0_wp/990.0_wp
  Jx(3,3,11,3,-1,1,1) = 13.0_wp/(660.0_wp*sqrt(6.0_wp))
  Jx(3,3,11,3,1,1,-1) = 13.0_wp/(660.0_wp*sqrt(6.0_wp))
  Jx(3,3,11,3,3,1,1) = 13.0_wp/(396.0_wp*sqrt(10.0_wp))
  Jx(3,3,11,3,-3,1,-1) = 13.0_wp/(396.0_wp*sqrt(10.0_wp))
  Jx(3,3,11,5,0,1,0) = 151.0_wp/8190.0_wp
  Jx(3,3,11,5,-4,1,0) = 1.0_wp/(39.0_wp*sqrt(70.0_wp))
  Jx(3,3,11,5,4,1,0) = 1.0_wp/(39.0_wp*sqrt(70.0_wp))
  Jx(3,3,11,5,-1,1,1) = -193.0_wp/(4368.0_wp*sqrt(15.0_wp))
  Jx(3,3,11,5,1,1,-1) = -193.0_wp/(4368.0_wp*sqrt(15.0_wp))
  Jx(3,3,11,5,3,1,1) = -97.0_wp/(1872.0_wp*sqrt(70.0_wp))
  Jx(3,3,11,5,-3,1,-1) = -97.0_wp/(1872.0_wp*sqrt(70.0_wp))
  Jx(3,3,11,5,-5,1,1) = (-23.0_wp*sqrt(7.0_wp))/(3120.0_wp*sqrt(2.0_wp))
  Jx(3,3,11,5,5,1,-1) = (-23.0_wp*sqrt(7.0_wp))/(3120.0_wp*sqrt(2.0_wp))
  Jx(3,3,11,7,0,1,0) = 27.0_wp/11440.0_wp
  Jx(3,3,11,7,-4,1,0) = (-27.0_wp*sqrt(3.0_wp))/(1040.0_wp*sqrt(154.0_wp))
  Jx(3,3,11,7,4,1,0) = (-27.0_wp*sqrt(3.0_wp))/(1040.0_wp*sqrt(154.0_wp))
  Jx(3,3,11,7,-1,1,1) = -27.0_wp/(5720.0_wp*sqrt(7.0_wp))
  Jx(3,3,11,7,1,1,-1) = -27.0_wp/(5720.0_wp*sqrt(7.0_wp))
  Jx(3,3,11,7,3,1,1) = (27.0_wp*sqrt(3.0_wp))/(11440.0_wp*sqrt(7.0_wp))
  Jx(3,3,11,7,-3,1,-1) = (27.0_wp*sqrt(3.0_wp))/(11440.0_wp*sqrt(7.0_wp))
  Jx(3,3,11,7,-5,1,1) = (27.0_wp*sqrt(3.0_wp))/(1040.0_wp*sqrt(77.0_wp))
  Jx(3,3,11,7,5,1,-1) = (27.0_wp*sqrt(3.0_wp))/(1040.0_wp*sqrt(77.0_wp))
  ! coefficients for Dy
  ! iLn=3; iLS=3, iJ=4 (2.0_wp*iJ+1)
  Jx(3,3,9,0,0,0,0) = -1.0_wp/140.0_wp
  Jx(3,3,9,4,0,0,0) = 3.0_wp/440.0_wp
  Jx(3,3,9,4,-4,0,0) = 3.0_wp/(88.0_wp*sqrt(70.0_wp))
  Jx(3,3,9,4,4,0,0) = 3.0_wp/(88.0_wp*sqrt(70.0_wp))
  Jx(3,3,9,6,0,0,0) = 1.0_wp/3080.0_wp
  Jx(3,3,9,6,-4,0,0) = -1.0_wp/(440.0_wp*sqrt(14.0_wp))
  Jx(3,3,9,6,4,0,0) = -1.0_wp/(440.0_wp*sqrt(14.0_wp))
  Jx(3,3,9,1,0,1,0) = 1.0_wp/140.0_wp
  Jx(3,3,9,1,-1,1,1) = -1.0_wp/140.0_wp
  Jx(3,3,9,1,1,1,-1) = -1.0_wp/140.0_wp
  Jx(3,3,9,3,0,1,0) = -1.0_wp/330.0_wp
  Jx(3,3,9,3,-1,1,1) = -1.0_wp/(220.0_wp*sqrt(6.0_wp))
  Jx(3,3,9,3,1,1,-1) = -1.0_wp/(220.0_wp*sqrt(6.0_wp))
  Jx(3,3,9,3,3,1,1) = -1.0_wp/(132.0_wp*sqrt(10.0_wp))
  Jx(3,3,9,3,-3,1,-1) = -1.0_wp/(132.0_wp*sqrt(10.0_wp))
  Jx(3,3,9,5,0,1,0) = -43.0_wp/10920.0_wp
  Jx(3,3,9,5,-4,1,0) = -1.0_wp/(104.0_wp*sqrt(70.0_wp))
  Jx(3,3,9,5,4,1,0) = -1.0_wp/(104.0_wp*sqrt(70.0_wp))
  Jx(3,3,9,5,-1,1,1) = 1.0_wp/(91.0_wp*sqrt(15.0_wp))
  Jx(3,3,9,5,1,1,-1) = 1.0_wp/(91.0_wp*sqrt(15.0_wp))
  Jx(3,3,9,5,3,1,1) = 1.0_wp/(156.0_wp*sqrt(70.0_wp))
  Jx(3,3,9,5,-3,1,-1) = 1.0_wp/(156.0_wp*sqrt(70.0_wp))
  Jx(3,3,9,5,-5,1,1) = 3.0_wp/(260.0_wp*sqrt(14.0_wp))
  Jx(3,3,9,5,5,1,-1) = 3.0_wp/(260.0_wp*sqrt(14.0_wp))
  Jx(3,3,9,7,0,1,0) = -1.0_wp/5720.0_wp
  Jx(3,3,9,7,-4,1,0) = (sqrt(3.0_wp))/(520.0_wp*sqrt(154.0_wp))
  Jx(3,3,9,7,4,1,0) = (sqrt(3.0_wp))/(520.0_wp*sqrt(154.0_wp))
  Jx(3,3,9,7,-1,1,1) = 1.0_wp/(2860.0_wp*sqrt(7.0_wp))
  Jx(3,3,9,7,1,1,-1) = 1.0_wp/(2860.0_wp*sqrt(7.0_wp))
  Jx(3,3,9,7,3,1,1) = (-sqrt(3.0_wp))/(5720.0_wp*sqrt(7.0_wp))
  Jx(3,3,9,7,-3,1,-1) = (-sqrt(3.0_wp))/(5720.0_wp*sqrt(7.0_wp))
  Jx(3,3,9,7,-5,1,1) = (-sqrt(3.0_wp))/(520.0_wp*sqrt(77.0_wp))
  Jx(3,3,9,7,5,1,-1) = (-sqrt(3.0_wp))/(520.0_wp*sqrt(77.0_wp))

  ! coefficients for Dy
  ! iLn=3; iLS=4, iJ=4 (2.0_wp*iJ+1)
  Jx(3,4,9,0,0,0,0) = -5.0_wp/147.0_wp
  Jx(3,4,9,4,0,0,0) = 5.0_wp/154.0_wp
  Jx(3,4,9,4,-4,0,0) = (5.0_wp*sqrt(5.0_wp))/(154.0_wp*sqrt(14.0_wp))
  Jx(3,4,9,4,4,0,0) = (5.0_wp*sqrt(5.0_wp))/(154.0_wp*sqrt(14.0_wp))
  Jx(3,4,9,6,0,0,0) = 5.0_wp/3234.0_wp
  Jx(3,4,9,6,-4,0,0) = -5.0_wp/(462.0_wp*sqrt(14.0_wp))
  Jx(3,4,9,6,4,0,0) = -5.0_wp/(462.0_wp*sqrt(14.0_wp))
  Jx(3,4,9,1,0,1,0) = 5.0_wp/147.0_wp
  Jx(3,4,9,1,-1,1,1) = -5.0_wp/147.0_wp
  Jx(3,4,9,1,1,1,-1) = -5.0_wp/147.0_wp
  Jx(3,4,9,3,0,1,0) = -10.0_wp/693.0_wp
  Jx(3,4,9,3,-1,1,1) = -5.0_wp/(231.0_wp*sqrt(6.0_wp))
  Jx(3,4,9,3,1,1,-1) = -5.0_wp/(231.0_wp*sqrt(6.0_wp))
  Jx(3,4,9,3,3,1,1) = (-5.0_wp*sqrt(5.0_wp))/(693.0_wp*sqrt(2.0_wp))
  Jx(3,4,9,3,-3,1,-1) = (-5.0_wp*sqrt(5.0_wp))/(693.0_wp*sqrt(2.0_wp))
  Jx(3,4,9,5,0,1,0) = -215.0_wp/11466.0_wp
  Jx(3,4,9,5,-4,1,0) = (-5.0_wp*sqrt(5.0_wp))/(546.0_wp*sqrt(14.0_wp))
  Jx(3,4,9,5,4,1,0) = (-5.0_wp*sqrt(5.0_wp))/(546.0_wp*sqrt(14.0_wp))
  Jx(3,4,9,5,-1,1,1) = (20.0_wp*sqrt(5.0_wp))/(1911.0_wp*sqrt(3.0_wp))
  Jx(3,4,9,5,1,1,-1) = (20.0_wp*sqrt(5.0_wp))/(1911.0_wp*sqrt(3.0_wp))
  Jx(3,4,9,5,3,1,1) = (5.0_wp*sqrt(5.0_wp))/(819.0_wp*sqrt(14.0_wp))
  Jx(3,4,9,5,-3,1,-1) = (5.0_wp*sqrt(5.0_wp))/(819.0_wp*sqrt(14.0_wp))
  Jx(3,4,9,5,-5,1,1) = 5.0_wp/(91.0_wp*sqrt(14.0_wp))
  Jx(3,4,9,5,5,1,-1) = 5.0_wp/(91.0_wp*sqrt(14.0_wp))
  Jx(3,4,9,7,0,1,0) = -5.0_wp/6006.0_wp
  Jx(3,4,9,7,-4,1,0) = 5.0_wp/(182.0_wp*sqrt(462.0_wp))
  Jx(3,4,9,7,4,1,0) = 5.0_wp/(182.0_wp*sqrt(462.0_wp))
  Jx(3,4,9,7,-1,1,1) = 5.0_wp/(3003.0_wp*sqrt(7.0_wp))
  Jx(3,4,9,7,1,1,-1) = 5.0_wp/(3003.0_wp*sqrt(7.0_wp))
  Jx(3,4,9,7,3,1,1) = -5.0_wp/(2002.0_wp*sqrt(21.0_wp))
  Jx(3,4,9,7,-3,1,-1) = -5.0_wp/(2002.0_wp*sqrt(21.0_wp))
  Jx(3,4,9,7,-5,1,1) = -5.0_wp/(182.0_wp*sqrt(231.0_wp))
  Jx(3,4,9,7,5,1,-1) = -5.0_wp/(182.0_wp*sqrt(231.0_wp))
  !---------------------------------------------------------------------

  !--------------------
  !    Ho2+
  ! J     energy (cm-1)
  !--------------------
  ! 7.5        0.000000
  ! 6.5     5147.916186
  ! 5.5     9609.443529
  ! 4.5    13384.582065
  !
  ! 4.5    19854.490832
  ! 3.5    22943.240537
  ! 2.5    25345.601421
  ! 1.5    27061.573481
  !
  ! 5.5    28405.331097
  ! 4.5    32180.469634
  ! 3.5    35269.219331
  ! 2.5    37671.580219
  !
  ! 3.5    47230.238695
  ! 2.5    49632.599582
  ! 1.5    51348.571637
  ! 0.5    52378.154873
  dE(4,1,16) = Zero
  dE(4,1,14) = 5147.916186_wp
  dE(4,1,12) = 9609.443529_wp
  dE(4,1,10) = 13384.582065_wp

  dE(4,2,10) = 19854.490832_wp
  dE(4,2,8) = 22943.240537_wp
  dE(4,2,6) = 25345.601421_wp
  dE(4,2,4) = 27061.573481_wp

  dE(4,3,12) = 28405.331097_wp
  dE(4,3,10) = 32180.469634_wp
  dE(4,3,8) = 35269.219331_wp
  dE(4,3,6) = 37671.580219_wp

  dE(4,4,8) = 47230.238695_wp
  dE(4,4,6) = 49632.599582_wp
  dE(4,4,4) = 51348.571637_wp
  dE(4,4,2) = 52378.154873_wp
  ! coefficients for Ho
  ! iLn=4; iLS=1, iJ=15/2 (2.0_wp*iJ+1)
  Jx(4,1,16,0,0,0,0) = -969.0_wp/6370.0_wp
  Jx(4,1,16,4,0,0,0) = 51.0_wp/455.0_wp
  Jx(4,1,16,4,-4,0,0) = 51.0_wp/(91.0_wp*sqrt(70.0_wp))
  Jx(4,1,16,4,4,0,0) = 51.0_wp/(91.0_wp*sqrt(70.0_wp))
  Jx(4,1,16,6,0,0,0) = -20.0_wp/637.0_wp
  Jx(4,1,16,6,-4,0,0) = (10.0_wp*sqrt(2.0_wp))/(91.0_wp*sqrt(7.0_wp))
  Jx(4,1,16,6,4,0,0) = (10.0_wp*sqrt(2.0_wp))/(91.0_wp*sqrt(7.0_wp))
  Jx(4,1,16,1,0,1,0) = 969.0_wp/6370.0_wp
  Jx(4,1,16,1,-1,1,1) = -969.0_wp/6370.0_wp
  Jx(4,1,16,1,1,1,-1) = -969.0_wp/6370.0_wp
  Jx(4,1,16,3,0,1,0) = -68.0_wp/1365.0_wp
  Jx(4,1,16,3,-1,1,1) = (-17.0_wp*sqrt(2.0_wp))/(455.0_wp*sqrt(3.0_wp))
  Jx(4,1,16,3,1,1,-1) = (-17.0_wp*sqrt(2.0_wp))/(455.0_wp*sqrt(3.0_wp))
  Jx(4,1,16,3,3,1,1) = (-17.0_wp*sqrt(2.0_wp))/(273.0_wp*sqrt(5.0_wp))
  Jx(4,1,16,3,-3,1,-1) = (-17.0_wp*sqrt(2.0_wp))/(273.0_wp*sqrt(5.0_wp))
  Jx(4,1,16,5,0,1,0) = -1187.0_wp/24843.0_wp
  Jx(4,1,16,5,-4,1,0) = -421.0_wp/(1183.0_wp*sqrt(70.0_wp))
  Jx(4,1,16,5,4,1,0) = -421.0_wp/(1183.0_wp*sqrt(70.0_wp))
  Jx(4,1,16,5,-1,1,1) = 1847.0_wp/(8281.0_wp*sqrt(15.0_wp))
  Jx(4,1,16,5,1,1,-1) = 1847.0_wp/(8281.0_wp*sqrt(15.0_wp))
  Jx(4,1,16,5,3,1,1) = -97.0_wp/(507.0_wp*sqrt(70.0_wp))
  Jx(4,1,16,5,-3,1,-1) = -97.0_wp/(507.0_wp*sqrt(70.0_wp))
  Jx(4,1,16,5,-5,1,1) = 201.0_wp/(1183.0_wp*sqrt(14.0_wp))
  Jx(4,1,16,5,5,1,-1) = 201.0_wp/(1183.0_wp*sqrt(14.0_wp))
  Jx(4,1,16,7,0,1,0) = 20.0_wp/1183.0_wp
  Jx(4,1,16,7,-4,1,0) = (-10.0_wp*sqrt(66.0_wp))/(1183.0_wp*sqrt(7.0_wp))
  Jx(4,1,16,7,4,1,0) = (-10.0_wp*sqrt(66.0_wp))/(1183.0_wp*sqrt(7.0_wp))
  Jx(4,1,16,7,-1,1,1) = -40.0_wp/(1183.0_wp*sqrt(7.0_wp))
  Jx(4,1,16,7,1,1,-1) = -40.0_wp/(1183.0_wp*sqrt(7.0_wp))
  Jx(4,1,16,7,3,1,1) = (20.0_wp*sqrt(3.0_wp))/(1183.0_wp*sqrt(7.0_wp))
  Jx(4,1,16,7,-3,1,-1) = (20.0_wp*sqrt(3.0_wp))/(1183.0_wp*sqrt(7.0_wp))
  Jx(4,1,16,7,-5,1,1) = (20.0_wp*sqrt(33.0_wp))/(1183.0_wp*sqrt(7.0_wp))
  Jx(4,1,16,7,5,1,-1) = (20.0_wp*sqrt(33.0_wp))/(1183.0_wp*sqrt(7.0_wp))
  ! coefficients for Ho
  ! iLn=4; iLS=1, iJ=13/2 (2.0_wp*iJ+1)
  Jx(4,1,14,0,0,0,0) = -51.0_wp/1820.0_wp
  Jx(4,1,14,4,0,0,0) = -9.0_wp/520.0_wp
  Jx(4,1,14,4,-4,0,0) = -9.0_wp/(104.0_wp*sqrt(70.0_wp))
  Jx(4,1,14,4,4,0,0) = -9.0_wp/(104.0_wp*sqrt(70.0_wp))
  Jx(4,1,14,6,0,0,0) = 5.0_wp/728.0_wp
  Jx(4,1,14,6,-4,0,0) = -5.0_wp/(104.0_wp*sqrt(14.0_wp))
  Jx(4,1,14,6,4,0,0) = -5.0_wp/(104.0_wp*sqrt(14.0_wp))
  Jx(4,1,14,1,0,1,0) = 51.0_wp/1820.0_wp
  Jx(4,1,14,1,-1,1,1) = -51.0_wp/1820.0_wp
  Jx(4,1,14,1,1,1,-1) = -51.0_wp/1820.0_wp
  Jx(4,1,14,3,0,1,0) = 1.0_wp/130.0_wp
  Jx(4,1,14,3,-1,1,1) = (sqrt(3.0_wp))/(260.0_wp*sqrt(2.0_wp))
  Jx(4,1,14,3,1,1,-1) = (sqrt(3.0_wp))/(260.0_wp*sqrt(2.0_wp))
  Jx(4,1,14,3,3,1,1) = 1.0_wp/(52.0_wp*sqrt(10.0_wp))
  Jx(4,1,14,3,-3,1,-1) = 1.0_wp/(52.0_wp*sqrt(10.0_wp))
  Jx(4,1,14,5,0,1,0) = 61.0_wp/9464.0_wp
  Jx(4,1,14,5,-4,1,0) = 89.0_wp/(1352.0_wp*sqrt(70.0_wp))
  Jx(4,1,14,5,4,1,0) = 89.0_wp/(1352.0_wp*sqrt(70.0_wp))
  Jx(4,1,14,5,-1,1,1) = (-29.0_wp*sqrt(3.0_wp))/(2366.0_wp*sqrt(5.0_wp))
  Jx(4,1,14,5,1,1,-1) = (-29.0_wp*sqrt(3.0_wp))/(2366.0_wp*sqrt(5.0_wp))
  Jx(4,1,14,5,3,1,1) = 31.0_wp/(676.0_wp*sqrt(70.0_wp))
  Jx(4,1,14,5,-3,1,-1) = 31.0_wp/(676.0_wp*sqrt(70.0_wp))
  Jx(4,1,14,5,-5,1,1) = -17.0_wp/(676.0_wp*sqrt(14.0_wp))
  Jx(4,1,14,5,5,1,-1) = -17.0_wp/(676.0_wp*sqrt(14.0_wp))
  Jx(4,1,14,7,0,1,0) = -5.0_wp/1352.0_wp
  Jx(4,1,14,7,-4,1,0) = (5.0_wp*sqrt(33.0_wp))/(1352.0_wp*sqrt(14.0_wp))
  Jx(4,1,14,7,4,1,0) = (5.0_wp*sqrt(33.0_wp))/(1352.0_wp*sqrt(14.0_wp))
  Jx(4,1,14,7,-1,1,1) = 5.0_wp/(676.0_wp*sqrt(7.0_wp))
  Jx(4,1,14,7,1,1,-1) = 5.0_wp/(676.0_wp*sqrt(7.0_wp))
  Jx(4,1,14,7,3,1,1) = (-5.0_wp*sqrt(3.0_wp))/(1352.0_wp*sqrt(7.0_wp))
  Jx(4,1,14,7,-3,1,-1) = (-5.0_wp*sqrt(3.0_wp))/(1352.0_wp*sqrt(7.0_wp))
  Jx(4,1,14,7,-5,1,1) = (-5.0_wp*sqrt(33.0_wp))/(1352.0_wp*sqrt(7.0_wp))
  Jx(4,1,14,7,5,1,-1) = (-5.0_wp*sqrt(33.0_wp))/(1352.0_wp*sqrt(7.0_wp))
  ! coefficients for Ho
  ! iLn=4; iLS=1, iJ=11/2 (2.0_wp*iJ+1)
  Jx(4,1,12,0,0,0,0) = -255.0_wp/154154.0_wp
  Jx(4,1,12,4,0,0,0) = -885.0_wp/484484.0_wp
  Jx(4,1,12,4,-4,0,0) = (-885.0_wp*sqrt(5.0_wp))/(484484.0_wp*sqrt(14.0_wp))
  Jx(4,1,12,4,4,0,0) = (-885.0_wp*sqrt(5.0_wp))/(484484.0_wp*sqrt(14.0_wp))
  Jx(4,1,12,6,0,0,0) = -225.0_wp/847847.0_wp
  Jx(4,1,12,6,-4,0,0) = 225.0_wp/(121121.0_wp*sqrt(14.0_wp))
  Jx(4,1,12,6,4,0,0) = 225.0_wp/(121121.0_wp*sqrt(14.0_wp))
  Jx(4,1,12,1,0,1,0) = 255.0_wp/154154.0_wp
  Jx(4,1,12,1,-1,1,1) = -255.0_wp/154154.0_wp
  Jx(4,1,12,1,1,1,-1) = -255.0_wp/154154.0_wp
  Jx(4,1,12,3,0,1,0) = 295.0_wp/363363.0_wp
  Jx(4,1,12,3,-1,1,1) = 295.0_wp/(242242.0_wp*sqrt(6.0_wp))
  Jx(4,1,12,3,1,1,-1) = 295.0_wp/(242242.0_wp*sqrt(6.0_wp))
  Jx(4,1,12,3,3,1,1) = (295.0_wp*sqrt(5.0_wp))/(726726.0_wp*sqrt(2.0_wp))
  Jx(4,1,12,3,-3,1,-1) = (295.0_wp*sqrt(5.0_wp))/(726726.0_wp*sqrt(2.0_wp))
  Jx(4,1,12,5,0,1,0) = 13675.0_wp/12024012.0_wp
  Jx(4,1,12,5,-4,1,0) = (185.0_wp*sqrt(5.0_wp))/(572572.0_wp*sqrt(14.0_wp))
  Jx(4,1,12,5,4,1,0) = (185.0_wp*sqrt(5.0_wp))/(572572.0_wp*sqrt(14.0_wp))
  Jx(4,1,12,5,-1,1,1) = (-2195.0_wp*sqrt(5.0_wp))/(4008004.0_wp*sqrt(3.0_wp))
  Jx(4,1,12,5,1,1,-1) = (-2195.0_wp*sqrt(5.0_wp))/(4008004.0_wp*sqrt(3.0_wp))
  Jx(4,1,12,5,3,1,1) = (-155.0_wp*sqrt(5.0_wp))/(245388.0_wp*sqrt(14.0_wp))
  Jx(4,1,12,5,-3,1,-1) = (-155.0_wp*sqrt(5.0_wp))/(245388.0_wp*sqrt(14.0_wp))
  Jx(4,1,12,5,-5,1,1) = -1825.0_wp/(572572.0_wp*sqrt(14.0_wp))
  Jx(4,1,12,5,5,1,-1) = -1825.0_wp/(572572.0_wp*sqrt(14.0_wp))
  Jx(4,1,12,7,0,1,0) = 225.0_wp/1574573.0_wp
  Jx(4,1,12,7,-4,1,0) = (-225.0_wp*sqrt(3.0_wp))/(143143.0_wp*sqrt(154.0_wp))
  Jx(4,1,12,7,4,1,0) = (-225.0_wp*sqrt(3.0_wp))/(143143.0_wp*sqrt(154.0_wp))
  Jx(4,1,12,7,-1,1,1) = -450.0_wp/(1574573.0_wp*sqrt(7.0_wp))
  Jx(4,1,12,7,1,1,-1) = -450.0_wp/(1574573.0_wp*sqrt(7.0_wp))
  Jx(4,1,12,7,3,1,1) = (225.0_wp*sqrt(3.0_wp))/(1574573.0_wp*sqrt(7.0_wp))
  Jx(4,1,12,7,-3,1,-1) = (225.0_wp*sqrt(3.0_wp))/(1574573.0_wp*sqrt(7.0_wp))
  Jx(4,1,12,7,-5,1,1) = (225.0_wp*sqrt(3.0_wp))/(143143.0_wp*sqrt(77.0_wp))
  Jx(4,1,12,7,5,1,-1) = (225.0_wp*sqrt(3.0_wp))/(143143.0_wp*sqrt(77.0_wp))
  ! coefficients for Ho
  ! iLn=4; iLS=1, iJ=9/2 (2.0_wp*iJ+1)
  Jx(4,1,10,0,0,0,0) = -1.0_wp/44044.0_wp
  Jx(4,1,10,4,0,0,0) = 3.0_wp/138424.0_wp
  Jx(4,1,10,4,-4,0,0) = (3.0_wp*sqrt(5.0_wp))/(138424.0_wp*sqrt(14.0_wp))
  Jx(4,1,10,4,4,0,0) = (3.0_wp*sqrt(5.0_wp))/(138424.0_wp*sqrt(14.0_wp))
  Jx(4,1,10,6,0,0,0) = 1.0_wp/968968.0_wp
  Jx(4,1,10,6,-4,0,0) = -1.0_wp/(138424.0_wp*sqrt(14.0_wp))
  Jx(4,1,10,6,4,0,0) = -1.0_wp/(138424.0_wp*sqrt(14.0_wp))
  Jx(4,1,10,1,0,1,0) = 1.0_wp/44044.0_wp
  Jx(4,1,10,1,-1,1,1) = -1.0_wp/44044.0_wp
  Jx(4,1,10,1,1,1,-1) = -1.0_wp/44044.0_wp
  Jx(4,1,10,3,0,1,0) = -1.0_wp/103818.0_wp
  Jx(4,1,10,3,-1,1,1) = -1.0_wp/(69212.0_wp*sqrt(6.0_wp))
  Jx(4,1,10,3,1,1,-1) = -1.0_wp/(69212.0_wp*sqrt(6.0_wp))
  Jx(4,1,10,3,3,1,1) = (-sqrt(5.0_wp))/(207636.0_wp*sqrt(2.0_wp))
  Jx(4,1,10,3,-3,1,-1) = (-sqrt(5.0_wp))/(207636.0_wp*sqrt(2.0_wp))
  Jx(4,1,10,5,0,1,0) = -43.0_wp/3435432.0_wp
  Jx(4,1,10,5,-4,1,0) = (-sqrt(5.0_wp))/(163592.0_wp*sqrt(14.0_wp))
  Jx(4,1,10,5,4,1,0) = (-sqrt(5.0_wp))/(163592.0_wp*sqrt(14.0_wp))
  Jx(4,1,10,5,-1,1,1) = (sqrt(5.0_wp))/(143143.0_wp*sqrt(3.0_wp))
  Jx(4,1,10,5,1,1,-1) = (sqrt(5.0_wp))/(143143.0_wp*sqrt(3.0_wp))
  Jx(4,1,10,5,3,1,1) = (sqrt(5.0_wp))/(245388.0_wp*sqrt(14.0_wp))
  Jx(4,1,10,5,-3,1,-1) = (sqrt(5.0_wp))/(245388.0_wp*sqrt(14.0_wp))
  Jx(4,1,10,5,-5,1,1) = 3.0_wp/(81796.0_wp*sqrt(14.0_wp))
  Jx(4,1,10,5,5,1,-1) = 3.0_wp/(81796.0_wp*sqrt(14.0_wp))
  Jx(4,1,10,7,0,1,0) = -1.0_wp/1799512.0_wp
  Jx(4,1,10,7,-4,1,0) = (sqrt(3.0_wp))/(163592.0_wp*sqrt(154.0_wp))
  Jx(4,1,10,7,4,1,0) = (sqrt(3.0_wp))/(163592.0_wp*sqrt(154.0_wp))
  Jx(4,1,10,7,-1,1,1) = 1.0_wp/(899756.0_wp*sqrt(7.0_wp))
  Jx(4,1,10,7,1,1,-1) = 1.0_wp/(899756.0_wp*sqrt(7.0_wp))
  Jx(4,1,10,7,3,1,1) = (-sqrt(3.0_wp))/(1799512.0_wp*sqrt(7.0_wp))
  Jx(4,1,10,7,-3,1,-1) = (-sqrt(3.0_wp))/(1799512.0_wp*sqrt(7.0_wp))
  Jx(4,1,10,7,-5,1,1) = (-sqrt(3.0_wp))/(163592.0_wp*sqrt(77.0_wp))
  Jx(4,1,10,7,5,1,-1) = (-sqrt(3.0_wp))/(163592.0_wp*sqrt(77.0_wp))

  ! coefficients for Ho
  ! iLn=4; iLS=2, iJ=11/2 (2.0_wp*iJ+1)
  Jx(4,2,12,0,0,0,0) = -51.0_wp/847.0_wp
  Jx(4,2,12,4,0,0,0) = -177.0_wp/2662.0_wp
  Jx(4,2,12,4,-4,0,0) = (-177.0_wp*sqrt(5.0_wp))/(2662.0_wp*sqrt(14.0_wp))
  Jx(4,2,12,4,4,0,0) = (-177.0_wp*sqrt(5.0_wp))/(2662.0_wp*sqrt(14.0_wp))
  Jx(4,2,12,6,0,0,0) = -90.0_wp/9317.0_wp
  Jx(4,2,12,6,-4,0,0) = (45.0_wp*sqrt(2.0_wp))/(1331.0_wp*sqrt(7.0_wp))
  Jx(4,2,12,6,4,0,0) = (45.0_wp*sqrt(2.0_wp))/(1331.0_wp*sqrt(7.0_wp))
  Jx(4,2,12,1,0,1,0) = 51.0_wp/847.0_wp
  Jx(4,2,12,1,-1,1,1) = -51.0_wp/847.0_wp
  Jx(4,2,12,1,1,1,-1) = -51.0_wp/847.0_wp
  Jx(4,2,12,3,0,1,0) = 118.0_wp/3993.0_wp
  Jx(4,2,12,3,-1,1,1) = 59.0_wp/(1331.0_wp*sqrt(6.0_wp))
  Jx(4,2,12,3,1,1,-1) = 59.0_wp/(1331.0_wp*sqrt(6.0_wp))
  Jx(4,2,12,3,3,1,1) = (59.0_wp*sqrt(5.0_wp))/(3993.0_wp*sqrt(2.0_wp))
  Jx(4,2,12,3,-3,1,-1) = (59.0_wp*sqrt(5.0_wp))/(3993.0_wp*sqrt(2.0_wp))
  Jx(4,2,12,5,0,1,0) = 2735.0_wp/66066.0_wp
  Jx(4,2,12,5,-4,1,0) = (37.0_wp*sqrt(5.0_wp))/(3146.0_wp*sqrt(14.0_wp))
  Jx(4,2,12,5,4,1,0) = (37.0_wp*sqrt(5.0_wp))/(3146.0_wp*sqrt(14.0_wp))
  Jx(4,2,12,5,-1,1,1) = (-439.0_wp*sqrt(5.0_wp))/(22022.0_wp*sqrt(3.0_wp))
  Jx(4,2,12,5,1,1,-1) = (-439.0_wp*sqrt(5.0_wp))/(22022.0_wp*sqrt(3.0_wp))
  Jx(4,2,12,5,3,1,1) = (-31.0_wp*sqrt(35.0_wp))/(9438.0_wp*sqrt(2.0_wp))
  Jx(4,2,12,5,-3,1,-1) = (-31.0_wp*sqrt(35.0_wp))/(9438.0_wp*sqrt(2.0_wp))
  Jx(4,2,12,5,-5,1,1) = -365.0_wp/(3146.0_wp*sqrt(14.0_wp))
  Jx(4,2,12,5,5,1,-1) = -365.0_wp/(3146.0_wp*sqrt(14.0_wp))
  Jx(4,2,12,7,0,1,0) = 90.0_wp/17303.0_wp
  Jx(4,2,12,7,-4,1,0) = (-45.0_wp*sqrt(6.0_wp))/(1573.0_wp*sqrt(77.0_wp))
  Jx(4,2,12,7,4,1,0) = (-45.0_wp*sqrt(6.0_wp))/(1573.0_wp*sqrt(77.0_wp))
  Jx(4,2,12,7,-1,1,1) = -180.0_wp/(17303.0_wp*sqrt(7.0_wp))
  Jx(4,2,12,7,1,1,-1) = -180.0_wp/(17303.0_wp*sqrt(7.0_wp))
  Jx(4,2,12,7,3,1,1) = (90.0_wp*sqrt(3.0_wp))/(17303.0_wp*sqrt(7.0_wp))
  Jx(4,2,12,7,-3,1,-1) = (90.0_wp*sqrt(3.0_wp))/(17303.0_wp*sqrt(7.0_wp))
  Jx(4,2,12,7,-5,1,1) = (90.0_wp*sqrt(3.0_wp))/(1573.0_wp*sqrt(77.0_wp))
  Jx(4,2,12,7,5,1,-1) = (90.0_wp*sqrt(3.0_wp))/(1573.0_wp*sqrt(77.0_wp))
  ! coefficients for Ho
  ! iLn=4; iLS=2, iJ=9/2 (2.0_wp*iJ+1)
  Jx(4,2,10,0,0,0,0) = -27.0_wp/3388.0_wp
  Jx(4,2,10,4,0,0,0) = 81.0_wp/10648.0_wp
  Jx(4,2,10,4,-4,0,0) = (81.0_wp*sqrt(5.0_wp))/(10648.0_wp*sqrt(14.0_wp))
  Jx(4,2,10,4,4,0,0) = (81.0_wp*sqrt(5.0_wp))/(10648.0_wp*sqrt(14.0_wp))
  Jx(4,2,10,6,0,0,0) = 27.0_wp/74536.0_wp
  Jx(4,2,10,6,-4,0,0) = -27.0_wp/(10648.0_wp*sqrt(14.0_wp))
  Jx(4,2,10,6,4,0,0) = -27.0_wp/(10648.0_wp*sqrt(14.0_wp))
  Jx(4,2,10,1,0,1,0) = 27.0_wp/3388.0_wp
  Jx(4,2,10,1,-1,1,1) = -27.0_wp/3388.0_wp
  Jx(4,2,10,1,1,1,-1) = -27.0_wp/3388.0_wp
  Jx(4,2,10,3,0,1,0) = -9.0_wp/2662.0_wp
  Jx(4,2,10,3,-1,1,1) = (-9.0_wp*sqrt(3.0_wp))/(5324.0_wp*sqrt(2.0_wp))
  Jx(4,2,10,3,1,1,-1) = (-9.0_wp*sqrt(3.0_wp))/(5324.0_wp*sqrt(2.0_wp))
  Jx(4,2,10,3,3,1,1) = (-9.0_wp*sqrt(5.0_wp))/(5324.0_wp*sqrt(2.0_wp))
  Jx(4,2,10,3,-3,1,-1) = (-9.0_wp*sqrt(5.0_wp))/(5324.0_wp*sqrt(2.0_wp))
  Jx(4,2,10,5,0,1,0) = -387.0_wp/88088.0_wp
  Jx(4,2,10,5,-4,1,0) = (-27.0_wp*sqrt(5.0_wp))/(12584.0_wp*sqrt(14.0_wp))
  Jx(4,2,10,5,4,1,0) = (-27.0_wp*sqrt(5.0_wp))/(12584.0_wp*sqrt(14.0_wp))
  Jx(4,2,10,5,-1,1,1) = (9.0_wp*sqrt(15.0_wp))/11011.0_wp
  Jx(4,2,10,5,1,1,-1) = (9.0_wp*sqrt(15.0_wp))/11011.0_wp
  Jx(4,2,10,5,3,1,1) = (9.0_wp*sqrt(5.0_wp))/(6292.0_wp*sqrt(14.0_wp))
  Jx(4,2,10,5,-3,1,-1) = (9.0_wp*sqrt(5.0_wp))/(6292.0_wp*sqrt(14.0_wp))
  Jx(4,2,10,5,-5,1,1) = 81.0_wp/(6292.0_wp*sqrt(14.0_wp))
  Jx(4,2,10,5,5,1,-1) = 81.0_wp/(6292.0_wp*sqrt(14.0_wp))
  Jx(4,2,10,7,0,1,0) = -27.0_wp/138424.0_wp
  Jx(4,2,10,7,-4,1,0) = (27.0_wp*sqrt(3.0_wp))/(12584.0_wp*sqrt(154.0_wp))
  Jx(4,2,10,7,4,1,0) = (27.0_wp*sqrt(3.0_wp))/(12584.0_wp*sqrt(154.0_wp))
  Jx(4,2,10,7,-1,1,1) = 27.0_wp/(69212.0_wp*sqrt(7.0_wp))
  Jx(4,2,10,7,1,1,-1) = 27.0_wp/(69212.0_wp*sqrt(7.0_wp))
  Jx(4,2,10,7,3,1,1) = (-27.0_wp*sqrt(3.0_wp))/(138424.0_wp*sqrt(7.0_wp))
  Jx(4,2,10,7,-3,1,-1) = (-27.0_wp*sqrt(3.0_wp))/(138424.0_wp*sqrt(7.0_wp))
  Jx(4,2,10,7,-5,1,1) = (-27.0_wp*sqrt(3.0_wp))/(12584.0_wp*sqrt(77.0_wp))
  Jx(4,2,10,7,5,1,-1) = (-27.0_wp*sqrt(3.0_wp))/(12584.0_wp*sqrt(77.0_wp))

  ! coefficients for Ho
  ! iLn=4; iLS=3, iJ=9/2 (2.0_wp*iJ+1)
  Jx(4,3,10,0,0,0,0) = -1.0_wp/28.0_wp
  Jx(4,3,10,4,0,0,0) = 3.0_wp/88.0_wp
  Jx(4,3,10,4,-4,0,0) = (3.0_wp*sqrt(5.0_wp))/(88.0_wp*sqrt(14.0_wp))
  Jx(4,3,10,4,4,0,0) = (3.0_wp*sqrt(5.0_wp))/(88.0_wp*sqrt(14.0_wp))
  Jx(4,3,10,6,0,0,0) = 1.0_wp/616.0_wp
  Jx(4,3,10,6,-4,0,0) = -1.0_wp/(88.0_wp*sqrt(14.0_wp))
  Jx(4,3,10,6,4,0,0) = -1.0_wp/(88.0_wp*sqrt(14.0_wp))
  Jx(4,3,10,1,0,1,0) = 1.0_wp/28.0_wp
  Jx(4,3,10,1,-1,1,1) = -1.0_wp/28.0_wp
  Jx(4,3,10,1,1,1,-1) = -1.0_wp/28.0_wp
  Jx(4,3,10,3,0,1,0) = -1.0_wp/66.0_wp
  Jx(4,3,10,3,-1,1,1) = -1.0_wp/(44.0_wp*sqrt(6.0_wp))
  Jx(4,3,10,3,1,1,-1) = -1.0_wp/(44.0_wp*sqrt(6.0_wp))
  Jx(4,3,10,3,3,1,1) = (-sqrt(5.0_wp))/(132.0_wp*sqrt(2.0_wp))
  Jx(4,3,10,3,-3,1,-1) = (-sqrt(5.0_wp))/(132.0_wp*sqrt(2.0_wp))
  Jx(4,3,10,5,0,1,0) = -43.0_wp/2184.0_wp
  Jx(4,3,10,5,-4,1,0) = (-sqrt(5.0_wp))/(104.0_wp*sqrt(14.0_wp))
  Jx(4,3,10,5,4,1,0) = (-sqrt(5.0_wp))/(104.0_wp*sqrt(14.0_wp))
  Jx(4,3,10,5,-1,1,1) = (sqrt(5.0_wp))/(91.0_wp*sqrt(3.0_wp))
  Jx(4,3,10,5,1,1,-1) = (sqrt(5.0_wp))/(91.0_wp*sqrt(3.0_wp))
  Jx(4,3,10,5,3,1,1) = (sqrt(5.0_wp))/(156.0_wp*sqrt(14.0_wp))
  Jx(4,3,10,5,-3,1,-1) = (sqrt(5.0_wp))/(156.0_wp*sqrt(14.0_wp))
  Jx(4,3,10,5,-5,1,1) = 3.0_wp/(52.0_wp*sqrt(14.0_wp))
  Jx(4,3,10,5,5,1,-1) = 3.0_wp/(52.0_wp*sqrt(14.0_wp))
  Jx(4,3,10,7,0,1,0) = -1.0_wp/1144.0_wp
  Jx(4,3,10,7,-4,1,0) = (sqrt(3.0_wp))/(104.0_wp*sqrt(154.0_wp))
  Jx(4,3,10,7,4,1,0) = (sqrt(3.0_wp))/(104.0_wp*sqrt(154.0_wp))
  Jx(4,3,10,7,-1,1,1) = 1.0_wp/(572.0_wp*sqrt(7.0_wp))
  Jx(4,3,10,7,1,1,-1) = 1.0_wp/(572.0_wp*sqrt(7.0_wp))
  Jx(4,3,10,7,3,1,1) = (-sqrt(3.0_wp))/(1144.0_wp*sqrt(7.0_wp))
  Jx(4,3,10,7,-3,1,-1) = (-sqrt(3.0_wp))/(1144.0_wp*sqrt(7.0_wp))
  Jx(4,3,10,7,-5,1,1) = (-sqrt(3.0_wp))/(104.0_wp*sqrt(77.0_wp))
  Jx(4,3,10,7,5,1,-1) = (-sqrt(3.0_wp))/(104.0_wp*sqrt(77.0_wp))
  !---------------------------------------------------------------------

  !--------------------
  !    Er2+
  ! J     energy (cm-1)
  !--------------------
  ! 6.0        0.000000
  ! 5.0     6863.330975
  ! 4.0     9661.800385
  !
  ! 4.0    12582.773438
  ! 3.0    14237.354352
  ! 2.0    17669.019858
  dE(5,1,13) = Zero
  dE(5,1,11) = 6863.330975_wp
  dE(5,1,9) = 9661.800385_wp

  dE(5,2,9) = 12582.773438_wp
  dE(5,2,7) = 14237.354352_wp
  dE(5,2,5) = 17669.019858_wp
  ! coefficients for Er
  ! iLn=5; iLS=1, iJ=6 (2.0_wp*iJ+1)
  Jx(5,1,13,0,0,0,0) = -34.0_wp/231.0_wp
  Jx(5,1,13,4,0,0,0) = -137.0_wp/1452.0_wp
  Jx(5,1,13,4,-4,0,0) = (-137.0_wp*sqrt(5.0_wp))/(1452.0_wp*sqrt(14.0_wp))
  Jx(5,1,13,4,4,0,0) = (-137.0_wp*sqrt(5.0_wp))/(1452.0_wp*sqrt(14.0_wp))
  Jx(5,1,13,6,0,0,0) = 225.0_wp/6776.0_wp
  Jx(5,1,13,6,-4,0,0) = -225.0_wp/(968.0_wp*sqrt(14.0_wp))
  Jx(5,1,13,6,4,0,0) = -225.0_wp/(968.0_wp*sqrt(14.0_wp))
  Jx(5,1,13,1,0,1,0) = 34.0_wp/231.0_wp
  Jx(5,1,13,1,-1,1,1) = -34.0_wp/231.0_wp
  Jx(5,1,13,1,1,1,-1) = -34.0_wp/231.0_wp
  Jx(5,1,13,3,0,1,0) = 137.0_wp/3267.0_wp
  Jx(5,1,13,3,-1,1,1) = 137.0_wp/(2178.0_wp*sqrt(6.0_wp))
  Jx(5,1,13,3,1,1,-1) = 137.0_wp/(2178.0_wp*sqrt(6.0_wp))
  Jx(5,1,13,3,3,1,1) = (137.0_wp*sqrt(5.0_wp))/(6534.0_wp*sqrt(2.0_wp))
  Jx(5,1,13,3,-3,1,-1) = (137.0_wp*sqrt(5.0_wp))/(6534.0_wp*sqrt(2.0_wp))
  Jx(5,1,13,5,0,1,0) = 2005.0_wp/54054.0_wp
  Jx(5,1,13,5,-4,1,0) = (173.0_wp*sqrt(5.0_wp))/(2574.0_wp*sqrt(14.0_wp))
  Jx(5,1,13,5,4,1,0) = (173.0_wp*sqrt(5.0_wp))/(2574.0_wp*sqrt(14.0_wp))
  Jx(5,1,13,5,-1,1,1) = (-2819.0_wp*sqrt(5.0_wp))/(72072.0_wp*sqrt(3.0_wp))
  Jx(5,1,13,5,1,1,-1) = (-2819.0_wp*sqrt(5.0_wp))/(72072.0_wp*sqrt(3.0_wp))
  Jx(5,1,13,5,3,1,1) = (1333.0_wp*sqrt(5.0_wp))/(30888.0_wp*sqrt(14.0_wp))
  Jx(5,1,13,5,-3,1,-1) = (1333.0_wp*sqrt(5.0_wp))/(30888.0_wp*sqrt(14.0_wp))
  Jx(5,1,13,5,-5,1,1) = (-205.0_wp*sqrt(7.0_wp))/(10296.0_wp*sqrt(2.0_wp))
  Jx(5,1,13,5,5,1,-1) = (-205.0_wp*sqrt(7.0_wp))/(10296.0_wp*sqrt(2.0_wp))
  Jx(5,1,13,7,0,1,0) = -225.0_wp/12584.0_wp
  Jx(5,1,13,7,-4,1,0) = (225.0_wp*sqrt(3.0_wp))/(1144.0_wp*sqrt(154.0_wp))
  Jx(5,1,13,7,4,1,0) = (225.0_wp*sqrt(3.0_wp))/(1144.0_wp*sqrt(154.0_wp))
  Jx(5,1,13,7,-1,1,1) = 225.0_wp/(6292.0_wp*sqrt(7.0_wp))
  Jx(5,1,13,7,1,1,-1) = 225.0_wp/(6292.0_wp*sqrt(7.0_wp))
  Jx(5,1,13,7,3,1,1) = (-225.0_wp*sqrt(3.0_wp))/(12584.0_wp*sqrt(7.0_wp))
  Jx(5,1,13,7,-3,1,-1) = (-225.0_wp*sqrt(3.0_wp))/(12584.0_wp*sqrt(7.0_wp))
  Jx(5,1,13,7,-5,1,1) = (-225.0_wp*sqrt(3.0_wp))/(1144.0_wp*sqrt(77.0_wp))
  Jx(5,1,13,7,5,1,-1) = (-225.0_wp*sqrt(3.0_wp))/(1144.0_wp*sqrt(77.0_wp))
  ! coefficients for Er
  ! iLn=5; iLS=1, iJ=5 (2.0_wp*iJ+1)
  Jx(5,1,11,0,0,0,0) = -2.0_wp/105.0_wp
  Jx(5,1,11,4,0,0,0) = -13.0_wp/660.0_wp
  Jx(5,1,11,4,-4,0,0) = -13.0_wp/(132.0_wp*sqrt(70.0_wp))
  Jx(5,1,11,4,4,0,0) = -13.0_wp/(132.0_wp*sqrt(70.0_wp))
  Jx(5,1,11,6,0,0,0) = -9.0_wp/3080.0_wp
  Jx(5,1,11,6,-4,0,0) = 9.0_wp/(440.0_wp*sqrt(14.0_wp))
  Jx(5,1,11,6,4,0,0) = 9.0_wp/(440.0_wp*sqrt(14.0_wp))
  Jx(5,1,11,1,0,1,0) = 2.0_wp/105.0_wp
  Jx(5,1,11,1,-1,1,1) = -2.0_wp/105.0_wp
  Jx(5,1,11,1,1,1,-1) = -2.0_wp/105.0_wp
  Jx(5,1,11,3,0,1,0) = 13.0_wp/1485.0_wp
  Jx(5,1,11,3,-1,1,1) = 13.0_wp/(990.0_wp*sqrt(6.0_wp))
  Jx(5,1,11,3,1,1,-1) = 13.0_wp/(990.0_wp*sqrt(6.0_wp))
  Jx(5,1,11,3,3,1,1) = 13.0_wp/(594.0_wp*sqrt(10.0_wp))
  Jx(5,1,11,3,-3,1,-1) = 13.0_wp/(594.0_wp*sqrt(10.0_wp))
  Jx(5,1,11,5,0,1,0) = 151.0_wp/12285.0_wp
  Jx(5,1,11,5,-4,1,0) = (sqrt(2.0_wp))/(117.0_wp*sqrt(35.0_wp))
  Jx(5,1,11,5,4,1,0) = (sqrt(2.0_wp))/(117.0_wp*sqrt(35.0_wp))
  Jx(5,1,11,5,-1,1,1) = -193.0_wp/(6552.0_wp*sqrt(15.0_wp))
  Jx(5,1,11,5,1,1,-1) = -193.0_wp/(6552.0_wp*sqrt(15.0_wp))
  Jx(5,1,11,5,3,1,1) = -97.0_wp/(2808.0_wp*sqrt(70.0_wp))
  Jx(5,1,11,5,-3,1,-1) = -97.0_wp/(2808.0_wp*sqrt(70.0_wp))
  Jx(5,1,11,5,-5,1,1) = (-23.0_wp*sqrt(7.0_wp))/(4680.0_wp*sqrt(2.0_wp))
  Jx(5,1,11,5,5,1,-1) = (-23.0_wp*sqrt(7.0_wp))/(4680.0_wp*sqrt(2.0_wp))
  Jx(5,1,11,7,0,1,0) = 9.0_wp/5720.0_wp
  Jx(5,1,11,7,-4,1,0) = (-9.0_wp*sqrt(3.0_wp))/(520.0_wp*sqrt(154.0_wp))
  Jx(5,1,11,7,4,1,0) = (-9.0_wp*sqrt(3.0_wp))/(520.0_wp*sqrt(154.0_wp))
  Jx(5,1,11,7,-1,1,1) = -9.0_wp/(2860.0_wp*sqrt(7.0_wp))
  Jx(5,1,11,7,1,1,-1) = -9.0_wp/(2860.0_wp*sqrt(7.0_wp))
  Jx(5,1,11,7,3,1,1) = (9.0_wp*sqrt(3.0_wp))/(5720.0_wp*sqrt(7.0_wp))
  Jx(5,1,11,7,-3,1,-1) = (9.0_wp*sqrt(3.0_wp))/(5720.0_wp*sqrt(7.0_wp))
  Jx(5,1,11,7,-5,1,1) = (9.0_wp*sqrt(3.0_wp))/(520.0_wp*sqrt(77.0_wp))
  Jx(5,1,11,7,5,1,-1) = (9.0_wp*sqrt(3.0_wp))/(520.0_wp*sqrt(77.0_wp))
  ! coefficients for Er
  ! iLn=5; iLS=1, iJ=4 (2.0_wp*iJ+1)
  Jx(5,1,9,0,0,0,0) = -1.0_wp/2310.0_wp
  Jx(5,1,9,4,0,0,0) = 1.0_wp/2420.0_wp
  Jx(5,1,9,4,-4,0,0) = 1.0_wp/(484.0_wp*sqrt(70.0_wp))
  Jx(5,1,9,4,4,0,0) = 1.0_wp/(484.0_wp*sqrt(70.0_wp))
  Jx(5,1,9,6,0,0,0) = 1.0_wp/50820.0_wp
  Jx(5,1,9,6,-4,0,0) = -1.0_wp/(7260.0_wp*sqrt(14.0_wp))
  Jx(5,1,9,6,4,0,0) = -1.0_wp/(7260.0_wp*sqrt(14.0_wp))
  Jx(5,1,9,1,0,1,0) = 1.0_wp/2310.0_wp
  Jx(5,1,9,1,-1,1,1) = -1.0_wp/2310.0_wp
  Jx(5,1,9,1,1,1,-1) = -1.0_wp/2310.0_wp
  Jx(5,1,9,3,0,1,0) = -1.0_wp/5445.0_wp
  Jx(5,1,9,3,-1,1,1) = -1.0_wp/(3630.0_wp*sqrt(6.0_wp))
  Jx(5,1,9,3,1,1,-1) = -1.0_wp/(3630.0_wp*sqrt(6.0_wp))
  Jx(5,1,9,3,3,1,1) = -1.0_wp/(2178.0_wp*sqrt(10.0_wp))
  Jx(5,1,9,3,-3,1,-1) = -1.0_wp/(2178.0_wp*sqrt(10.0_wp))
  Jx(5,1,9,5,0,1,0) = -43.0_wp/180180.0_wp
  Jx(5,1,9,5,-4,1,0) = -1.0_wp/(1716.0_wp*sqrt(70.0_wp))
  Jx(5,1,9,5,4,1,0) = -1.0_wp/(1716.0_wp*sqrt(70.0_wp))
  Jx(5,1,9,5,-1,1,1) = 2.0_wp/(3003.0_wp*sqrt(15.0_wp))
  Jx(5,1,9,5,1,1,-1) = 2.0_wp/(3003.0_wp*sqrt(15.0_wp))
  Jx(5,1,9,5,3,1,1) = 1.0_wp/(2574.0_wp*sqrt(70.0_wp))
  Jx(5,1,9,5,-3,1,-1) = 1.0_wp/(2574.0_wp*sqrt(70.0_wp))
  Jx(5,1,9,5,-5,1,1) = 1.0_wp/(1430.0_wp*sqrt(14.0_wp))
  Jx(5,1,9,5,5,1,-1) = 1.0_wp/(1430.0_wp*sqrt(14.0_wp))
  Jx(5,1,9,7,0,1,0) = -1.0_wp/94380.0_wp
  Jx(5,1,9,7,-4,1,0) = 1.0_wp/(2860.0_wp*sqrt(462.0_wp))
  Jx(5,1,9,7,4,1,0) = 1.0_wp/(2860.0_wp*sqrt(462.0_wp))
  Jx(5,1,9,7,-1,1,1) = 1.0_wp/(47190.0_wp*sqrt(7.0_wp))
  Jx(5,1,9,7,1,1,-1) = 1.0_wp/(47190.0_wp*sqrt(7.0_wp))
  Jx(5,1,9,7,3,1,1) = -1.0_wp/(31460.0_wp*sqrt(21.0_wp))
  Jx(5,1,9,7,-3,1,-1) = -1.0_wp/(31460.0_wp*sqrt(21.0_wp))
  Jx(5,1,9,7,-5,1,1) = -1.0_wp/(2860.0_wp*sqrt(231.0_wp))
  Jx(5,1,9,7,5,1,-1) = -1.0_wp/(2860.0_wp*sqrt(231.0_wp))

  ! coefficients for Er
  ! iLn=5; iLS=2, iJ=4 (2.0_wp*iJ+1)
  Jx(5,2,9,0,0,0,0) = -1.0_wp/21.0_wp
  Jx(5,2,9,4,0,0,0) = 1.0_wp/22.0_wp
  Jx(5,2,9,4,-4,0,0) = (sqrt(5.0_wp))/(22.0_wp*sqrt(14.0_wp))
  Jx(5,2,9,4,4,0,0) = (sqrt(5.0_wp))/(22.0_wp*sqrt(14.0_wp))
  Jx(5,2,9,6,0,0,0) = 1.0_wp/462.0_wp
  Jx(5,2,9,6,-4,0,0) = -1.0_wp/(66.0_wp*sqrt(14.0_wp))
  Jx(5,2,9,6,4,0,0) = -1.0_wp/(66.0_wp*sqrt(14.0_wp))
  Jx(5,2,9,1,0,1,0) = 1.0_wp/21.0_wp
  Jx(5,2,9,1,-1,1,1) = -1.0_wp/21.0_wp
  Jx(5,2,9,1,1,1,-1) = -1.0_wp/21.0_wp
  Jx(5,2,9,3,0,1,0) = -2.0_wp/99.0_wp
  Jx(5,2,9,3,-1,1,1) = -1.0_wp/(33.0_wp*sqrt(6.0_wp))
  Jx(5,2,9,3,1,1,-1) = -1.0_wp/(33.0_wp*sqrt(6.0_wp))
  Jx(5,2,9,3,3,1,1) = (-sqrt(5.0_wp))/(99.0_wp*sqrt(2.0_wp))
  Jx(5,2,9,3,-3,1,-1) = (-sqrt(5.0_wp))/(99.0_wp*sqrt(2.0_wp))
  Jx(5,2,9,5,0,1,0) = -43.0_wp/1638.0_wp
  Jx(5,2,9,5,-4,1,0) = (-sqrt(5.0_wp))/(78.0_wp*sqrt(14.0_wp))
  Jx(5,2,9,5,4,1,0) = (-sqrt(5.0_wp))/(78.0_wp*sqrt(14.0_wp))
  Jx(5,2,9,5,-1,1,1) = (4.0_wp*sqrt(5.0_wp))/(273.0_wp*sqrt(3.0_wp))
  Jx(5,2,9,5,1,1,-1) = (4.0_wp*sqrt(5.0_wp))/(273.0_wp*sqrt(3.0_wp))
  Jx(5,2,9,5,3,1,1) = (sqrt(5.0_wp))/(117.0_wp*sqrt(14.0_wp))
  Jx(5,2,9,5,-3,1,-1) = (sqrt(5.0_wp))/(117.0_wp*sqrt(14.0_wp))
  Jx(5,2,9,5,-5,1,1) = 1.0_wp/(13.0_wp*sqrt(14.0_wp))
  Jx(5,2,9,5,5,1,-1) = 1.0_wp/(13.0_wp*sqrt(14.0_wp))
  Jx(5,2,9,7,0,1,0) = -1.0_wp/858.0_wp
  Jx(5,2,9,7,-4,1,0) = 1.0_wp/(26.0_wp*sqrt(462.0_wp))
  Jx(5,2,9,7,4,1,0) = 1.0_wp/(26.0_wp*sqrt(462.0_wp))
  Jx(5,2,9,7,-1,1,1) = 1.0_wp/(429.0_wp*sqrt(7.0_wp))
  Jx(5,2,9,7,1,1,-1) = 1.0_wp/(429.0_wp*sqrt(7.0_wp))
  Jx(5,2,9,7,3,1,1) = -1.0_wp/(286.0_wp*sqrt(21.0_wp))
  Jx(5,2,9,7,-3,1,-1) = -1.0_wp/(286.0_wp*sqrt(21.0_wp))
  Jx(5,2,9,7,-5,1,1) = -1.0_wp/(26.0_wp*sqrt(231.0_wp))
  Jx(5,2,9,7,5,1,-1) = -1.0_wp/(26.0_wp*sqrt(231.0_wp))
  !---------------------------------------------------------------------

  initialized = .true.

end subroutine init_Jx

end module jcoeff
