/*
 * This file is part of OpenTTD.
 * OpenTTD is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, version 2.
 * OpenTTD is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details. You should have received a copy of the GNU General Public License along with OpenTTD. If not, see <https://www.gnu.org/licenses/old-licenses/gpl-2.0>.
 */

/** @file table/townname.h Namepart tables for the town name generator */

#include "../core/enum_type.hpp"

static const std::string_view _name_original_english_1[] = {
	"Great ",
	"Little ",
	"New ",
	"Fort ",
};

static const std::string_view _name_original_english_2[] = {
	"Wr",
	"B",
	"C",
	"Ch",
	"Br",
	"D",
	"Dr",
	"F",
	"Fr",
	"Fl",
	"G",
	"Gr",
	"H",
	"L",
	"M",
	"N",
	"P",
	"Pr",
	"Pl",
	"R",
	"S",
	"S",
	"Sl",
	"T",
	"Tr",
	"W",
};

static const std::string_view _name_original_english_3[] = {
	"ar",
	"a",
	"e",
	"in",
	"on",
	"u",
	"un",
	"en",
};

static const std::string_view _name_original_english_4[] = {
	"n",
	"ning",
	"ding",
	"d",
	"",
	"t",
	"fing",
};

static const std::string_view _name_original_english_5[] = {
	"ville",
	"ham",
	"field",
	"ton",
	"town",
	"bridge",
	"bury",
	"wood",
	"ford",
	"hall",
	"ston",
	"way",
	"stone",
	"borough",
	"ley",
	"head",
	"bourne",
	"pool",
	"worth",
	"hill",
	"well",
	"hattan",
	"burg",
};

static const std::string_view _name_original_english_6[] = {
	"-on-sea",
	" Bay",
	" Market",
	" Cross",
	" Bridge",
	" Falls",
	" City",
	" Ridge",
	" Springs",
};

static const std::string_view _name_additional_english_prefix[] = {
	"Great ",
	"Little ",
	"New ",
	"Fort ",
	"St. ",
	"Old ",
};

static const std::string_view _name_additional_english_1a[] = {
	"Pen",
	"Lough",
	"Stam",
	"Aber",
	"Acc",
	"Ex",
	"Ax",
	"Bre",
	"Cum",
	"Dun",
	"Fin",
	"Inver",
	"Kin",
	"Mon",
	"Nan",
	"Nant",
	"Pit",
	"Pol",
	"Pont",
	"Strath",
	"Tre",
	"Tilly",
	"Beck",
	"Canter",
	"Bath",
	"Liver",
	"Mal",
	"Ox",
	"Bletch",
	"Maccles",
	"Grim",
	"Wind",
	"Sher",
	"Gates",
	"Orp",
	"Brom",
	"Lewis",
	"Whit",
	"White",
	"Worm",
	"Tyne",
	"Avon",
	"Stan",
};

static const std::string_view _name_additional_english_1b1[] = {
	"Wr",
	"B",
	"C",
	"Ch",
	"Br",
	"D",
	"Dr",
	"F",
	"Fr",
	"Fl",
	"G",
	"Gr",
	"H",
	"L",
	"M",
	"N",
	"P",
	"Pr",
	"Pl",
	"R",
	"S",
	"S",
	"Sl",
	"T",
	"Tr",
	"W",
};

static const std::string_view _name_additional_english_1b2[] = {
	"ar",
	"a",
	"e",
	"in",
	"on",
	"u",
	"o",
	"ee",
	"es",
	"ea",
	"un",
	"en",
};

static const std::string_view _name_additional_english_1b3a[] = {
	"n",
	"d",
	"",
	"t",
	"",
	"",
};

static const std::string_view _name_additional_english_1b3b[] = {
	"ning",
	"ding",
	"fing",
};

static const std::string_view _name_additional_english_2[] = {
	"ville",
	"ham",
	"field",
	"ton",
	"town",
	"borough",
	"bridge",
	"bury",
	"wood",
	"ditch",
	"ford",
	"hall",
	"dean",
	"leigh",
	"dore",
	"ston",
	"stow",
	"church",
	"wich",
	"low",
	"way",
	"stone",
	"minster",
	"ley",
	"head",
	"bourne",
	"pool",
	"worth",
	"hill",
	"well",
	"hattan",
	"burg",
	"berg",
	"burgh",
	"port",
	"stoke",
	"haven",
	"stable",
	"stock",
	"side",
	"brook",
	"don",
	"den",
	"down",
	"nor",
	"grove",
	"combe",
	"by",
	"say",
	"ney",
	"chester",
	"dale",
	"ness",
	"shaw",
	"thwaite",
};

static const std::string_view _name_additional_english_3[] = {
	"-on-sea",
	" Bay",
	" Market",
	" Beeches",
	" Common",
	" Park",
	" Heath",
	" Marsh",
	" Green",
	" Castle",
	" End",
	" Rivers",
	" Cross",
	" Bridge",
	" Falls",
	" City",
	" Ridge",
	" Springs",
};

static const std::string_view _name_austrian_a1[] = {
	"Bad ",
	"Deutsch ",
	"Gross ",
	"Klein ",
	"Markt ",
	"Maria ",
};

static const std::string_view _name_austrian_a2[] = {
	"Aus",
	"Alten",
	"Braun",
	"V\u00f6sl",
	"Mittern",
	"Nuss",
	"Neu",
	"Walters",
	"Breiten",
	"Eisen",
	"Feld",
	"Mittern",
	"Gall",
	"Obern",
	"Grat",
	"Heiligen",
	"Hof",
	"Holla",
	"Stein",
	"Eber",
	"Eggen",
	"Enzers",
	"Frauen",
	"Herren",
	"Hof",
	"H\u00fctt",
	"Kaisers",
	"K\u00f6nigs",
	"Knittel",
	"Lang",
	"Ober",
	"Ollers",
	"Pfaffen",
	"Potten",
	"Salz",
	"Schwarz",
	"Stocker",
	"Unter",
	"Utten",
	"V\u00f6sen",
	"Vill",
	"Weissen",
};

static const std::string_view _name_austrian_a3[] = {
	"see",
	"bach",
	"dorf",
	"ach",
	"stein",
	"hofen",
	"au",
	"ach",
	"kirch",
	"kirchen",
	"kreuz",
	"brunn",
	"siedl",
	"markt",
	"wang",
	"haag",
};

static const std::string_view _name_austrian_a4[] = {
	"Bruck",
	"Brunn",
	"Gams",
	"Grein",
	"Ried",
	"Faak",
	"Zell",
	"Spital",
	"Kirchberg",
	"Saal",
	"Taferl",
	"Wald",
};

static const std::string_view _name_austrian_a5[] = {
	"St. ",
	"Sankt ",
};

static const std::string_view _name_austrian_a6[] = {
	"Aegyd",
	"Andr\u00e4",
	"Georgen",
	"Jakob",
	"Johann",
	"Leonhard",
	"Marein",
	"Lorenzen",
	"Margarethen",
	"Martin",
	"Michael",
	"Nikolai",
	"Oswald",
	"Peter",
	"P\u00f6lten",
	"Stefan",
	"Stephan",
	"Thomas",
	"Veit",
	"Wolfgang",
};

static const std::string_view _name_austrian_f1[] = {
	" an der ",
	" ob der ",
};

static const std::string_view _name_austrian_f2[] = {
	"Donau",
	"Steyr",
	"Lafnitz",
	"Leitha",
	"Thaya",
	"Gail",
	"Drau",
	"Salzach",
	"Ybbs",
	"Traisen",
	"Enns",
	"Mur",
	"Ill",
};

static const std::string_view _name_austrian_b1[] = {
	" am ",
};

static const std::string_view _name_austrian_b2[] = {
	"Brenner",
	"Dachstein",
	"Gebirge",
	"Grossglockner",
	"Hausruck",
	"Semmering",
	"Wagram",
	"Wechsel",
	"Wilden Kaiser",
	"Ziller",
};

static const std::string_view _name_german_real[] = {
	"Berlin",
	"Bonn",
	"Bremen",
	"Cottbus",
	"Chemnitz",
	"Dortmund",
	"Dresden",
	"Erfurt",
	"Erlangen",
	"Essen",
	"Fulda",
	"Gera",
	"Kassel",
	"Kiel",
	"K\u00f6ln",
	"L\u00fcbeck",
	"Magdeburg",
	"M\u00fcnchen",
	"Potsdam",
	"Stuttgart",
	"Wiesbaden",
};

static const std::string_view _name_german_pre[] = {
	"Bad ",
	"Klein ",
	"Neu ",
};

static const std::string_view _name_german_1[] = {
	"Alb",
	"Als",
	"Ander",
	"Arns",
	"Bruns",
	"Bam",
	"Biele",
	"Cloppen",
	"Co",
	"Duis",
	"D\u00fcssel",
	"Dannen",
	"Elb",
	"Els",
	"Elster",
	"Eichen",
	"Ems",
	"Fahr",
	"Falken",
	"Flens",
	"Frank",
	"Frei",
	"Freuden",
	"Fried",
	"F\u00fcrsten",
	"Hahn",
	"Ham",
	"Harz",
	"Heidel",
	"Hers",
	"Herz",
	"Holz",
	"Hildes",
	"Inns",
	"Ilsen",
	"Ingols",
	"Kel",
	"Kies",
	"Korn",
	"Kor",
	"Kreuz",
	"Kulm",
	"Langen",
	"Lim",
	"Lohr",
	"L\u00fcne",
	"Mel",
	"Michels",
	"M\u00fchl",
	"Naum",
	"Nest",
	"Nord",
	"Nort",
	"Nien",
	"Nidda",
	"Nieder",
	"N\u00fcrn",
	"Ober",
	"Offen",
	"Osna",
	"Olden",
	"Ols",
	"Oranien",
	"Pader",
	"Quedlin",
	"Quer",
	"Ravens",
	"Regens",
	"Rott",
	"Ros",
	"R\u00fcssels",
	"Saal",
	"Saar",
	"Salz",
	"Sch\u00f6ne",
	"Schwein",
	"Sonder",
	"Sonnen",
	"Stein",
	"Strals",
	"Straus",
	"S\u00fcd",
	"Ton",
	"Unter",
	"Ur",
	"Vor",
	"Wald",
	"War",
	"Wert",
	"Wester",
	"Witten",
	"Wolfs",
	"W\u00fcrz",
};

static const std::string_view _name_german_2[] = {
	"bach",
	"berg",
	"br\u00fcck",
	"br\u00fccken",
	"burg",
	"dorf",
	"feld",
	"furt",
	"hausen",
	"haven",
	"heim",
	"horst",
	"mund",
	"m\u00fcnster",
	"stadt",
	"wald",
};

static const std::string_view _name_german_3_an_der[] = {
	" an der ",
};

static const std::string_view _name_german_3_am[] = {
	" am ",
};

static const std::string_view _name_german_4_an_der[] = {
	"Oder",
	"Spree",
	"Donau",
	"Saale",
	"Elbe",
};

static const std::string_view _name_german_4_am[] = {
	"Main",
};

static const std::string_view _name_spanish_real[] = {
	"Caracas",
	"Maracay",
	"Maracaibo",
	"Valencia",
	"El Dorado",
	"Morrocoy",
	"Cata",
	"Cataito",
	"Ciudad Bol\u00edvar",
	"Barquisimeto",
	"M\u00e9rida",
	"Puerto Ordaz",
	"Santa Elena",
	"San Juan",
	"San Luis",
	"San Rafael",
	"Santiago",
	"Barcelona",
	"Barinas",
	"San Crist\u00f3bal",
	"San Francisco",
	"San Mart\u00edn",
	"Guayana",
	"San Carlos",
	"El Lim\u00f3n",
	"Coro",
	"Coro Coro",
	"Puerto Ayacucho",
	"Elorza",
	"Arismendi",
	"Trujillo",
	"Car\u00fapano",
	"Anaco",
	"Lima",
	"Cuzco",
	"Iquitos",
	"Callao",
	"Huacho",
	"Caman\u00e1",
	"Puerto Chala",
	"Santa Cruz",
	"Quito",
	"Cuenca",
	"Medell\u00edn",
	"Tulc\u00e1n",
	"Esmeraldas",
	"Ibarra",
	"San Lorenzo",
	"Macas",
	"Mora\u00f1a",
	"Machala",
	"Zamora",
	"Latacunga",
	"Tena",
	"Cochabamba",
	"Ascensi\u00f3n",
	"Magdalena",
	"Santa Ana",
	"Manoa",
	"Sucre",
	"Oruro",
	"Uyuni",
	"Potos\u00ed",
	"Tupiza",
	"La Quiaca",
	"Yacu\u00edba",
	"San Borja",
	"Fuerte Olimpo",
	"Fort\u00edn Esteros",
	"Campo Grande",
	"Bogot\u00e1",
	"El Banco",
	"Zaragoza",
	"Neiva",
	"Mariano",
	"Cali",
	"La Palma",
	"Andoas",
	"Barranca",
	"Montevideo",
	"Valdivia",
	"Arica",
	"Temuco",
	"Tocopilla",
	"Mendoza",
	"Santa Rosa",
};

static const std::string_view _name_french_real[] = {
	"Agincourt",
	"Lille",
	"Dinan",
	"Aubusson",
	"Rodez",
	"Bergerac",
	"Bordeaux",
	"Bayonne",
	"Montpellier",
	"Mont\u00e9limar",
	"Valence",
	"Digne",
	"Nice",
	"Cannes",
	"St. Tropez",
	"Marseille",
	"Narbonne",
	"S\u00e8te",
	"Aurillac",
	"Gu\u00e9ret",
	"Le Creusot",
	"Nevers",
	"Auxerre",
	"Versailles",
	"Meaux",
	"Ch\u00e2lons",
	"Compi\u00e8gne",
	"Metz",
	"Chaumont",
	"Langres",
	"Bourg",
	"Lyon",
	"Vienne",
	"Grenoble",
	"Toulon",
	"Rennes",
	"Le Mans",
	"Angers",
	"Nantes",
	"Ch\u00e2teauroux",
	"Orl\u00e9ans",
	"Lisieux",
	"Cherbourg",
	"Morlaix",
	"Cognac",
	"Agen",
	"Tulle",
	"Blois",
	"Troyes",
	"Charolles",
	"Toulouse",
	"Chamb\u00e9ry",
	"Tours",
	"St. Brieuc",
	"St. Malo",
	"La Rochelle",
	"St. Flour",
	"Le Puy",
	"Vichy",
	"St. Valery",
	"Beaujolais",
	"Reims",
	"Albi",
	"Paris",
	"Biarritz",
	"B\u00e9ziers",
	"N\u00eemes",
	"Chamonix",
	"Angoul\u00e8me",
	"Alen\u00e7on",
};

static const std::string_view _name_silly_1[] = {
	"Binky",
	"Blubber",
	"Bumble",
	"Crinkle",
	"Crusty",
	"Dangle",
	"Dribble",
	"Flippety",
	"Google",
	"Muffin",
	"Nosey",
	"Pinker",
	"Quack",
	"Rumble",
	"Sleepy",
	"Sliggles",
	"Snooze",
	"Teddy",
	"Tinkle",
	"Twister",
	"Pinker",
	"Hippo",
	"Itchy",
	"Jelly",
	"Jingle",
	"Jolly",
	"Kipper",
	"Lazy",
	"Frogs",
	"Mouse",
	"Quack",
	"Cheeky",
	"Lumpy",
	"Grumpy",
	"Mangle",
	"Fiddle",
	"Slugs",
	"Noodles",
	"Poodles",
	"Shiver",
	"Rumble",
	"Pixie",
	"Puddle",
	"Riddle",
	"Rattle",
	"Rickety",
	"Waffle",
	"Sagging",
	"Sausage",
	"Egg",
	"Sleepy",
	"Scatter",
	"Scramble",
	"Silly",
	"Simple",
	"Trickle",
	"Slippery",
	"Slimey",
	"Slumber",
	"Soggy",
	"Sliggles",
	"Splutter",
	"Sulky",
	"Swindle",
	"Swivel",
	"Tasty",
	"Tangle",
	"Toggle",
	"Trotting",
	"Tumble",
	"Snooze",
	"Water",
	"Windy",
	"Amble",
	"Bubble",
	"Cheery",
	"Cheese",
	"Cockle",
	"Cracker",
	"Crumple",
	"Teddy",
	"Evil",
	"Fairy",
	"Falling",
	"Fishy",
	"Fizzle",
	"Frosty",
	"Griddle",
};

static const std::string_view _name_silly_2[] = {
	"ton",
	"bury",
	"bottom",
	"ville",
	"well",
	"weed",
	"worth",
	"wig",
	"wick",
	"wood",
	"pool",
	"head",
	"burg",
	"gate",
	"bridge",
};

static const std::string_view _name_swedish_1[] = {
	"Gamla ",
	"Lilla ",
	"Nya ",
	"Stora ",
};

static const std::string_view _name_swedish_2[] = {
	"Boll",
	"Bor",
	"Ed",
	"En",
	"Erik",
	"Es",
	"Fin",
	"Fisk",
	"Gr\u00f6n",
	"Hag",
	"Halm",
	"Karl",
	"Kram",
	"Kung",
	"Land",
	"Lid",
	"Lin",
	"Mal",
	"Malm",
	"Marie",
	"Ner",
	"Norr",
	"Oskar",
	"Sand",
	"Skog",
	"Stock",
	"Stor",
	"Str\u00f6m",
	"Sund",
	"S\u00f6der",
	"Tall",
	"Tratt",
	"Troll",
	"Upp",
	"Var",
	"V\u00e4ster",
	"\u00c4ngel",
	"\u00d6ster",
};

static const std::string_view _name_swedish_2a[] = {
	"B",
	"Br",
	"D",
	"Dr",
	"Dv",
	"F",
	"Fj",
	"Fl",
	"Fr",
	"G",
	"Gl",
	"Gn",
	"Gr",
	"H",
	"J",
	"K",
	"Kl",
	"Kn",
	"Kr",
	"Kv",
	"L",
	"M",
	"N",
	"P",
	"Pl",
	"Pr",
	"R",
	"S",
	"Sk",
	"Skr",
	"Sl",
	"Sn",
	"Sp",
	"Spr",
	"St",
	"Str",
	"Sv",
	"T",
	"Tr",
	"Tv",
	"V",
	"Vr",
};

static const std::string_view _name_swedish_2b[] = {
	"a",
	"e",
	"i",
	"o",
	"u",
	"y",
	"\u00e5",
	"\u00e4",
	"\u00f6",
};

static const std::string_view _name_swedish_2c[] = {
	"ck",
	"d",
	"dd",
	"g",
	"gg",
	"l",
	"ld",
	"m",
	"n",
	"nd",
	"ng",
	"nn",
	"p",
	"pp",
	"r",
	"rd",
	"rk",
	"rp",
	"rr",
	"rt",
	"s",
	"sk",
	"st",
	"t",
	"tt",
	"v",
};

static const std::string_view _name_swedish_3[] = {
	"arp",
	"berg",
	"boda",
	"borg",
	"bro",
	"bukten",
	"by",
	"byn",
	"fors",
	"hammar",
	"hamn",
	"holm",
	"hus",
	"h\u00e4ttan",
	"kulle",
	"k\u00f6ping",
	"lund",
	"l\u00f6v",
	"sala",
	"skrona",
	"sl\u00e4tt",
	"sp\u00e5ng",
	"stad",
	"sund",
	"svall",
	"svik",
	"s\u00e5ker",
	"udde",
	"valla",
	"viken",
	"\u00e4lv",
	"\u00e5s",
};

static const std::string_view _name_dutch_1[] = {
	"Nieuw ",
	"Oud ",
	"Groot ",
	"Zuid ",
	"Noord ",
	"Oost ",
	"West ",
	"Klein ",
};

static const std::string_view _name_dutch_2[] = {
	"Hoog",
	"Laag",
	"Zuider",
	"Zuid",
	"Ooster",
	"Oost",
	"Wester",
	"West",
	"Hoofd",
	"Midden",
	"Eind",
	"Amster",
	"Amstel",
	"Dord",
	"Rotter",
	"Haar",
	"Til",
	"Enk",
	"Dok",
	"Veen",
	"Leidsch",
	"Lely",
	"En",
	"Kaats",
	"U",
	"Maas",
	"Mar",
	"Bla",
	"Al",
	"Alk",
	"Eer",
	"Drie",
	"Ter",
	"Groes",
	"Goes",
	"Soest",
	"Coe",
	"Uit",
	"Zwaag",
	"Hellen",
	"Slie",
	"IJ",
	"Grubben",
	"Groen",
	"Lek",
	"Ridder",
	"Schie",
	"Olde",
	"Roose",
	"Haar",
	"Til",
	"Loos",
	"Hil",
};

static const std::string_view _name_dutch_3[] = {
	"Drog",
	"Nat",
	"Valk",
	"Bob",
	"Dedem",
	"Kollum",
	"Best",
	"Hoend",
	"Leeuw",
	"Graaf",
	"Uithuis",
	"Purm",
	"Hard",
	"Hell",
	"Werk",
	"Spijk",
	"Vink",
	"Wams",
	"Heerhug",
	"Koning",
};

static const std::string_view _name_dutch_4[] = {
	"e",
	"er",
	"el",
	"en",
	"o",
	"s",
};

static const std::string_view _name_dutch_5[] = {
	"stad",
	"vorst",
	"dorp",
	"dam",
	"beek",
	"doorn",
	"zijl",
	"zijlen",
	"lo",
	"muiden",
	"meden",
	"vliet",
	"nisse",
	"daal",
	"vorden",
	"vaart",
	"mond",
	"zaal",
	"water",
	"duinen",
	"heuvel",
	"geest",
	"kerk",
	"meer",
	"maar",
	"hoorn",
	"rade",
	"wijk",
	"berg",
	"heim",
	"sum",
	"richt",
	"burg",
	"recht",
	"drecht",
	"trecht",
	"tricht",
	"dricht",
	"lum",
	"rum",
	"halen",
	"oever",
	"wolde",
	"veen",
	"hoven",
	"gast",
	"kum",
	"hage",
	"dijk",
	"zwaag",
	"pomp",
	"huizen",
	"bergen",
	"schede",
	"mere",
	"end",
};

static const std::string_view _name_finnish_real[] = {
	"Aijala",
	"Kisko",
	"Espoo",
	"Helsinki",
	"Tapiola",
	"J\u00e4rvel\u00e4",
	"Lahti",
	"Kotka",
	"Hamina",
	"Loviisa",
	"Kouvola",
	"Tampere",
	"Oulu",
	"Salo",
	"Malmi",
	"Pelto",
	"Koski",
	"Iisalmi",
	"Raisio",
	"Taavetti",
	"Joensuu",
	"Imatra",
	"Tapanila",
	"Pasila",
	"Turku",
	"Kupittaa",
	"Vaasa",
	"Pori",
	"Rauma",
	"Kolari",
	"Lieksa",
};

static const std::string_view _name_finnish_1[] = {
	"Hiekka",
	"Haapa",
	"Mylly",
	"Sauna",
	"Uusi",
	"Vanha",
	"Kes\u00e4",
	"Kuusi",
	"Pelto",
	"Tuomi",
	"Terva",
	"Olki",
	"Hein\u00e4",
	"Sein\u00e4",
	"Rova",
	"Koivu",
	"Kokko",
	"M\u00e4nty",
	"Pihlaja",
	"Pet\u00e4j\u00e4",
	"Kielo",
	"Kauha",
	"Viita",
	"Kivi",
	"Riihi",
	"\u00c4\u00e4ne",
	"Niini",
};

static const std::string_view _name_finnish_2[] = {
	"Lappeen",
	"Lohjan",
	"Savon",
	"Lapin",
	"Pit\u00e4j\u00e4n",
	"Martin",
	"Kuusan",
	"Kemi",
	"Keri",
	"H\u00e4meen",
	"Kangas",
};

static const std::string_view _name_finnish_3[] = {
	"harju",
	"linna",
	"j\u00e4rvi",
	"kallio",
	"m\u00e4ki",
	"nummi",
	"joki",
	"kyl\u00e4",
	"lampi",
	"lahti",
	"mets\u00e4",
	"suo",
	"laakso",
	"niitty",
	"luoto",
	"hovi",
	"ranta",
	"koski",
	"salo",
};

static const std::string_view _name_polish_1_m[] = {
	"Wielki ",
	"Ma\u0142y ",
	"Z\u0142y ",
	"Dobry ",
	"Nowy ",
	"Stary ",
	"Z\u0142oty ",
	"Zielony ",
	"Bia\u0142y ",
	"Modry ",
	"D\u0119bowy ",
};

static const std::string_view _name_polish_1_f[] = {
	"Wielka ",
	"Ma\u0142a ",
	"Z\u0142a ",
	"Dobra ",
	"Nowa ",
	"Stara ",
	"Z\u0142ota ",
	"Zielona ",
	"Bia\u0142a ",
	"Modra ",
	"D\u0119bowa ",
};

static const std::string_view _name_polish_1_n[] = {
	"Wielkie ",
	"Ma\u0142e ",
	"Z\u0142e ",
	"Dobre ",
	"Nowe ",
	"Stare ",
	"Z\u0142ote ",
	"Zielone ",
	"Bia\u0142e ",
	"Modre ",
	"D\u0119bowe ",
};

static const std::string_view _name_polish_2_o[] = {
	"Frombork",
	"Gniezno",
	"Olsztyn",
	"Toru\u0144",
	"Bydgoszcz",
	"Terespol",
	"Krak\u00f3w",
	"Pozna\u0144",
	"Wroc\u0142aw",
	"Katowice",
	"Cieszyn",
	"Bytom",
	"Frombork",
	"Hel",
	"Konin",
	"Lublin",
	"Malbork",
	"Sopot",
	"Sosnowiec",
	"Gda\u0144sk",
	"Gdynia",
	"Sieradz",
	"Sandomierz",
	"Szczyrk",
	"Szczytno",
	"Szczecin",
	"Zakopane",
	"Szklarska Por\u0119ba",
	"Bochnia",
	"Golub-Dobrzy\u0144",
	"Chojnice",
	"Ostrowiec",
	"Otwock",
	"Wolsztyn",
};

static const std::string_view _name_polish_2_m[] = {
	"Jarocin",
	"Gogolin",
	"Tomasz\u00f3w",
	"Piotrk\u00f3w",
	"Lidzbark",
	"Rypin",
	"Radzymin",
	"Wo\u0142omin",
	"Pruszk\u00f3w",
	"Olsztynek",
	"Rypin",
	"Cisek",
	"Krotoszyn",
	"Stoczek",
	"Lubin",
	"Lubicz",
	"Milicz",
	"Targ",
	"Ostr\u00f3w",
	"Ozimek",
	"Puck",
	"Rzepin",
	"Siewierz",
	"Stargard",
	"Starogard",
	"Turek",
	"Tymbark",
	"Wolsztyn",
	"Strzepcz",
	"Strzebielin",
	"Sochaczew",
	"Gr\u0119bocin",
	"Gniew",
	"Lubliniec",
	"Lubasz",
	"Lutomiersk",
	"Niemodlin",
	"Przeworsk",
	"Ursus",
	"Tyczyn",
	"Sztum",
	"Szczebrzeszyn",
	"Wolin",
	"Wrzeszcz",
	"Zgierz",
	"Zieleniec",
	"Drobin",
	"Garwolin",
};

static const std::string_view _name_polish_2_f[] = {
	"Szprotawa",
	"Pogorzelica",
	"Mot\u0142awa",
	"Lubawa",
	"Nidzica",
	"Kruszwica",
	"Bierawa",
	"Brodnica",
	"Chojna",
	"Krzepica",
	"Ruda",
	"Rumia",
	"Tuchola",
	"Trzebinia",
	"Ustka",
	"Warszawa",
	"Bobowa",
	"Dukla",
	"Krynica",
	"Murowana",
	"Niemcza",
	"Zaspa",
	"Zawoja",
	"Wola",
	"Limanowa",
	"Rabka",
	"Skawina",
	"Pilawa",
};

static const std::string_view _name_polish_2_n[] = {
	"Lipsko",
	"Pilzno",
	"Przodkowo",
	"Strzelno",
	"Leszno",
	"Jaworzno",
	"Choszczno",
	"Mogilno",
	"Luzino",
	"Miasto",
	"Dziadowo",
	"Kowalewo",
	"Legionowo",
	"Miastko",
	"Zabrze",
	"Zawiercie",
	"Kochanowo",
	"Miechucino",
	"Mirachowo",
	"Robakowo",
	"Kosakowo",
	"Borne",
	"Braniewo",
	"Sulinowo",
	"Chmielno",
	"Jastrz\u0119bie",
	"Gryfino",
	"Koronowo",
	"Lubichowo",
	"Opoczno",
};

static const std::string_view _name_polish_3_m[] = {
	" Wybudowanie",
	" \u015awi\u0119tokrzyski",
	" G\u00f3rski",
	" Morski",
	" Zdr\u00f3j",
	" Wody",
	" Bajoro",
	" Kraje\u0144ski",
	" \u015al\u0105ski",
	" Mazowiecki",
	" Pomorski",
	" Wielki",
	" Maly",
	" Warmi\u0144ski",
	" Mazurski",
	" Mniejszy",
	" Wi\u0119kszy",
	" G\u00f3rny",
	" Dolny",
	" Wielki",
	" Stary",
	" Nowy",
	" Wielkopolski",
	" Wzg\u00f3rze",
	" Mosty",
	" Kujawski",
	" Ma\u0142opolski",
	" Podlaski",
	" Lesny",
};

static const std::string_view _name_polish_3_f[] = {
	" Wybudowanie",
	" \u015awi\u0119tokrzyska",
	" G\u00f3rska",
	" Morska",
	" Zdr\u00f3j",
	" Woda",
	" Bajoro",
	" Kraje\u0144ska",
	" \u015al\u0105ska",
	" Mazowiecka",
	" Pomorska",
	" Wielka",
	" Ma\u0142a",
	" Warmi\u0144ska",
	" Mazurska",
	" Mniejsza",
	" Wi\u0119ksza",
	" G\u00f3rna",
	" Dolna",
	" Wielka",
	" Stara",
	" Nowa",
	" Wielkopolska",
	" Wzg\u00f3rza",
	" Mosty",
	" Kujawska",
	" Malopolska",
	" Podlaska",
	" Le\u015bna",
};

static const std::string_view _name_polish_3_n[] = {
	" Wybudowanie",
	" \u015awietokrzyskie",
	" G\u00f3rskie",
	" Morskie",
	" Zdr\u00f3j",
	" Wody",
	" Bajoro",
	" Kraje\u0144skie",
	" \u015al\u0105skie",
	" Mazowieckie",
	" Pomorskie",
	" Wielkie",
	" Ma\u0142e",
	" Warmi\u0144skie ",
	" Mazurskie ",
	" Mniejsze",
	" Wi\u0119ksze",
	" G\u00f3rne",
	" Dolne",
	" Wielkie",
	" Stare",
	" Nowe",
	" Wielkopolskie",
	" Wzg\u00f3rze",
	" Mosty",
	" Kujawskie",
	" Ma\u0142opolskie",
	" Podlaskie",
	" Le\u015bne",
};

static const std::string_view _name_czech_real[] = {
	"A\u0161",
	"Bene\u0161ov",
	"Beroun",
	"Bezdru\u017eice",
	"Blansko",
	"B\u0159eclav",
	"Brno",
	"Brunt\u00e1l",
	"\u010cesk\u00e1 L\u00edpa",
	"\u010cesk\u00e9 Bud\u011bjovice",
	"\u010cesk\u00fd Krumlov",
	"D\u011b\u010d\u00edn",
	"Doma\u017elice",
	"Dub\u00ed",
	"Fr\u00fddek-M\u00edstek",
	"Havl\u00ed\u010dk\u016fv Brod",
	"Hodon\u00edn",
	"Hradec Kr\u00e1lov\u00e9",
	"Humpolec",
	"Cheb",
	"Chomutov",
	"Chrudim",
	"Jablonec nad Nisou",
	"Jesen\u00edk",
	"Ji\u010d\u00edn",
	"Jihlava",
	"Jind\u0159ich\u016fv Hradec",
	"Karlovy Vary",
	"Karvin\u00e1",
	"Kladno",
	"Klatovy",
	"Kol\u00edn",
	"Kosmonosy",
	"Krom\u011b\u0159\u00ed\u017e",
	"Kutn\u00e1 Hora",
	"Liberec",
	"Litom\u011b\u0159ice",
	"Louny",
	"Man\u011bt\u00edn",
	"M\u011bln\u00edk",
	"Mlad\u00e1 Boleslav",
	"Most",
	"N\u00e1chod",
	"Nov\u00fd Ji\u010d\u00edn",
	"Nymburk",
	"Olomouc",
	"Opava",
	"Or\u00e1\u010dov",
	"Ostrava",
	"Pardubice",
	"Pelh\u0159imov",
	"Pol\u017eice",
	"P\u00edsek",
	"Plze\u0148",
	"Praha",
	"Prachatice",
	"P\u0159erov",
	"P\u0159\u00edbram",
	"Prost\u011bjov",
	"Rakovn\u00edk",
	"Rokycany",
	"Rudn\u00e1",
	"Rychnov nad Kn\u011b\u017enou",
	"Semily",
	"Sokolov",
	"Strakonice",
	"St\u0159edokluky",
	"\u0160umperk",
	"Svitavy",
	"T\u00e1bor",
	"Tachov",
	"Teplice",
	"T\u0159eb\u00ed\u010d",
	"Trutnov",
	"Uhersk\u00e9 Hradi\u0161t\u011b",
	"\u00dast\u00ed nad Labem",
	"\u00dast\u00ed nad Orlic\u00ed",
	"Vset\u00edn",
	"Vy\u0161kov",
	"\u017d\u010f\u00e1r nad S\u00e1zavou",
	"Zl\u00edn",
	"Znojmo",
};


/* The advanced hyperintelligent Czech town names generator!
 * The tables and MakeCzechTownName() is (c) Petr Baudis 2005 (GPL'd)
 * Feel free to ask me about anything unclear or if you need help
 * with cloning this for your own language. */

/* Sing., pl. */
enum CzechGender : uint8_t {
	CZG_SMASC,
	CZG_SFEM,
	CZG_SNEUT,
	CZG_PMASC,
	CZG_PFEM,
	CZG_PNEUT,
	/* Special for substantive stems - the ending chooses the gender. */
	CZG_FREE,
	/* Like CZG_FREE, but disallow CZG_SNEUT. */
	CZG_NFREE
};

enum CzechPattern : uint8_t {
	CZP_JARNI,
	CZP_MLADY,
	CZP_PRIVL
};

/* [CzechGender][CzechPattern] - suffixes for adjectives */
static const std::string_view _name_czech_patmod[][3] = {
	/* CZG_SMASC */ { "\u00ed", "\u00fd", "uv" },
	/* CZG_SFEM */  { "\u00ed", "\u00e1", "ova" },
	/* CZG_SNEUT */ { "\u00ed", "\u00e9", "ovo" },
	/* CZG_PMASC */ { "\u00ed", "\u00e9", "ovy" },
	/* CZG_PFEM */  { "\u00ed", "\u00e9", "ovy" },
	/* CZG_PNEUT */ { "\u00ed", "\u00e1", "ova" }
};

/* This way the substantive can choose only some adjectives/endings:
 * At least one of these flags must be satisfied: */
enum class CzechAllowFlag : uint8_t {
	Short,
	Middle,
	Long,
};

using CzechAllowFlags = EnumBitSet<CzechAllowFlag, uint8_t>;

static constexpr CzechAllowFlags CZA_ALL = {CzechAllowFlag::Short, CzechAllowFlag::Middle, CzechAllowFlag::Long};

/* All these flags must be satisfied (in the stem->others direction): */
enum class CzechChooseFlag : uint8_t {
	Colour,
	Postfix, // Matched if postfix was inserted.
	NoPostfix, // Matched if no postfix was inserted.
};

using CzechChooseFlags = EnumBitSet<CzechChooseFlag, uint8_t>;

static constexpr CzechChooseFlags CZC_ANY = {CzechChooseFlag::Colour, CzechChooseFlag::Postfix, CzechChooseFlag::NoPostfix};

struct CzechNameSubst {
	CzechGender gender;
	CzechAllowFlags allow;
	CzechChooseFlags choose;
	std::string_view name;
};

struct CzechNameAdj {
	CzechPattern pattern;
	CzechChooseFlags choose;
	std::string_view name;
};

/* Some of items which should be common are doubled. */
static const CzechNameAdj _name_czech_adj[] = {
	{ CZP_JARNI, CZC_ANY, "Horn" },
	{ CZP_JARNI, CZC_ANY, "Horn" },
	{ CZP_JARNI, CZC_ANY, "Doln" },
	{ CZP_JARNI, CZC_ANY, "Doln" },
	{ CZP_JARNI, CZC_ANY, "P\u0159edn" },
	{ CZP_JARNI, CZC_ANY, "Zadn" },
	{ CZP_JARNI, CZC_ANY, "Kosteln" },
	{ CZP_JARNI, CZC_ANY, "Havran" },
	{ CZP_JARNI, CZC_ANY, "\u0158\u00ed\u010dn" },
	{ CZP_JARNI, CZC_ANY, "Jezern" },
	{ CZP_MLADY, CZC_ANY, "Velk" },
	{ CZP_MLADY, CZC_ANY, "Velk" },
	{ CZP_MLADY, CZC_ANY, "Mal" },
	{ CZP_MLADY, CZC_ANY, "Mal" },
	{ CZP_MLADY, CZC_ANY, "Vysok" },
	{ CZP_MLADY, CZC_ANY, "\u010cesk" },
	{ CZP_MLADY, CZC_ANY, "Moravsk" },
	{ CZP_MLADY, CZC_ANY, "Slov\u00e1ck" },
	{ CZP_MLADY, CZC_ANY, "Slezsk" },
	{ CZP_MLADY, CZC_ANY, "Uhersk" },
	{ CZP_MLADY, CZC_ANY, "Star" },
	{ CZP_MLADY, CZC_ANY, "Star" },
	{ CZP_MLADY, CZC_ANY, "Nov" },
	{ CZP_MLADY, CZC_ANY, "Nov" },
	{ CZP_MLADY, CZC_ANY, "Mlad" },
	{ CZP_MLADY, CZC_ANY, "Kr\u00e1lovsk" },
	{ CZP_MLADY, CZC_ANY, "Kamenn" },
	{ CZP_MLADY, CZC_ANY, "Cihlov" },
	{ CZP_MLADY, CZC_ANY, "Divn" },
	{ CZP_MLADY, CzechChooseFlag::Colour, "\u010cerven" },
	{ CZP_MLADY, CzechChooseFlag::Colour, "\u010cerven" },
	{ CZP_MLADY, CzechChooseFlag::Colour, "\u010cerven" },
	{ CZP_MLADY, CzechChooseFlag::Colour, "Zelen" },
	{ CZP_MLADY, CzechChooseFlag::Colour, "\u017dlut" },
	{ CZP_MLADY, CzechChooseFlag::Colour, "Siv" },
	{ CZP_MLADY, CzechChooseFlag::Colour, "\u0160ed" },
	{ CZP_MLADY, CzechChooseFlag::Colour, "B\u00edl" },
	{ CZP_MLADY, CzechChooseFlag::Colour, "B\u00edl" },
	{ CZP_MLADY, CzechChooseFlag::Colour, "Modr" },
	{ CZP_MLADY, CzechChooseFlag::Colour, "R\u016f\u017eov" },
	{ CZP_MLADY, CzechChooseFlag::Colour, "\u010cern" },
	{ CZP_PRIVL, CZC_ANY, "Kr\u00e1l" },
	{ CZP_PRIVL, CZC_ANY, "Jan" },
	{ CZP_PRIVL, CZC_ANY, "Karl" },
	{ CZP_PRIVL, CZC_ANY, "Kry\u0161tof" },
	{ CZP_PRIVL, CZC_ANY, "Ji\u0159\u00edk" },
	{ CZP_PRIVL, CZC_ANY, "Petr" },
	{ CZP_PRIVL, CZC_ANY, "Sud" },
};

/* Considered a stem for choose/allow matching purposes. */
static const CzechNameSubst _name_czech_subst_full[] = {
	{ CZG_SMASC, CZA_ALL, CzechChooseFlag::Colour, "Sedlec" },
	{ CZG_SMASC, CZA_ALL, CzechChooseFlag::Colour, "Brod" },
	{ CZG_SMASC, CZA_ALL, CzechChooseFlag::Colour, "Brod" },
	{ CZG_SMASC, CZA_ALL, {}, "\u00daval" },
	{ CZG_SMASC, CZA_ALL, CzechChooseFlag::Colour, "\u017d\u010f\u00e1r" },
	{ CZG_SMASC, CZA_ALL, CzechChooseFlag::Colour, "Smrk" },
	{ CZG_SFEM,  CZA_ALL, CzechChooseFlag::Colour, "Hora" },
	{ CZG_SFEM,  CZA_ALL, CzechChooseFlag::Colour, "Lhota" },
	{ CZG_SFEM,  CZA_ALL, CzechChooseFlag::Colour, "Lhota" },
	{ CZG_SFEM,  CZA_ALL, CzechChooseFlag::Colour, "Hlava" },
	{ CZG_SFEM,  CZA_ALL, CzechChooseFlag::Colour, "L\u00edpa" },
	{ CZG_SNEUT, CZA_ALL, CzechChooseFlag::Colour, "Pole" },
	{ CZG_SNEUT, CZA_ALL, CzechChooseFlag::Colour, "\u00dadol\u00ed" },
	{ CZG_PMASC, CZA_ALL, {}, "\u00davaly" },
	{ CZG_PFEM,  CZA_ALL, CzechChooseFlag::Colour, "Luka" },
	{ CZG_PNEUT, CZA_ALL, CzechChooseFlag::Colour, "Pole" },
};

/* TODO: More stems needed. --pasky */
static const CzechNameSubst _name_czech_subst_stem[] = {
	{ CZG_SMASC, {                       CzechAllowFlag::Middle                      }, CzechChooseFlag::Colour, "Kostel" },
	{ CZG_SMASC, {                       CzechAllowFlag::Middle                      }, CzechChooseFlag::Colour, "Kl\u00e1\u0161ter" },
	{ CZG_SMASC, {CzechAllowFlag::Short                                              }, CzechChooseFlag::Colour, "Lhot" },
	{ CZG_SFEM,  {CzechAllowFlag::Short                                              }, CzechChooseFlag::Colour, "Lhot" },
	{ CZG_SFEM,  {CzechAllowFlag::Short                                              }, CzechChooseFlag::Colour, "Hur" },
	{ CZG_FREE,  {                       CzechAllowFlag::Middle, CzechAllowFlag::Long}, {}, "Sedl" },
	{ CZG_FREE,  {CzechAllowFlag::Short, CzechAllowFlag::Middle, CzechAllowFlag::Long}, CzechChooseFlag::Colour, "Hrad" },
	{ CZG_NFREE, {                       CzechAllowFlag::Middle                      }, {}, "Pras" },
	{ CZG_NFREE, {                       CzechAllowFlag::Middle                      }, {}, "Ba\u017e" },
	{ CZG_NFREE, {                       CzechAllowFlag::Middle                      }, {}, "Tes" },
	{ CZG_NFREE, {                       CzechAllowFlag::Middle                      }, {}, "U\u017e" },
	{ CZG_NFREE, {                       CzechAllowFlag::Middle, CzechAllowFlag::Long}, CzechChooseFlag::Postfix, "B\u0159" },
	{ CZG_NFREE, {                       CzechAllowFlag::Middle, CzechAllowFlag::Long}, {}, "Vod" },
	{ CZG_NFREE, {                       CzechAllowFlag::Middle, CzechAllowFlag::Long}, {}, "Jan" },
	{ CZG_NFREE, {                                               CzechAllowFlag::Long}, {}, "Prach" },
	{ CZG_NFREE, {                                               CzechAllowFlag::Long}, {}, "Kunr" },
	{ CZG_NFREE, {                                               CzechAllowFlag::Long}, {}, "Strak" },
	{ CZG_NFREE, {                                               CzechAllowFlag::Long}, {}, "V\u00edt" },
	{ CZG_NFREE, {                                               CzechAllowFlag::Long}, {}, "Vy\u0161" },
	{ CZG_NFREE, {                                               CzechAllowFlag::Long}, {}, "\u017dat" },
	{ CZG_NFREE, {                                               CzechAllowFlag::Long}, {}, "\u017der" },
	{ CZG_NFREE, {                                               CzechAllowFlag::Long}, {}, "St\u0159ed" },
	{ CZG_NFREE, {                                               CzechAllowFlag::Long}, {}, "Harv" },
	{ CZG_NFREE, {                                               CzechAllowFlag::Long}, {}, "Pruh" },
	{ CZG_NFREE, {                                               CzechAllowFlag::Long}, {}, "Tach" },
	{ CZG_NFREE, {                                               CzechAllowFlag::Long}, {}, "P\u00edsn" },
	{ CZG_NFREE, {                                               CzechAllowFlag::Long}, {}, "Jin" },
	{ CZG_NFREE, {                                               CzechAllowFlag::Long}, {}, "Jes" },
	{ CZG_NFREE, {                                               CzechAllowFlag::Long}, {}, "Jar" },
	{ CZG_NFREE, {                                               CzechAllowFlag::Long}, {}, "Sok" },
	{ CZG_NFREE, {                                               CzechAllowFlag::Long}, {}, "Hod" },
	{ CZG_NFREE, {                                               CzechAllowFlag::Long}, {}, "Net" },
	{ CZG_FREE,  {                                               CzechAllowFlag::Long}, {}, "Pra\u017e" },
	{ CZG_FREE,  {                                               CzechAllowFlag::Long}, {}, "Nerat" },
	{ CZG_FREE,  {                                               CzechAllowFlag::Long}, {}, "Kral" },
	{ CZG_FREE,  {                                               CzechAllowFlag::Long}, {}, "Hut" },
	{ CZG_FREE,  {                                               CzechAllowFlag::Long}, CzechChooseFlag::NoPostfix, "Pan" },
	{ CZG_FREE,  {CzechAllowFlag::Short, CzechAllowFlag::Middle, CzechAllowFlag::Long}, CzechChooseFlag::NoPostfix, "Odst\u0159ed" },
	{ CZG_FREE,  {CzechAllowFlag::Short, CzechAllowFlag::Middle, CzechAllowFlag::Long}, CzechChooseFlag::Colour, "Mrat" },
	{ CZG_FREE,  {                                               CzechAllowFlag::Long}, CzechChooseFlag::Colour, "Hlav" },
	{ CZG_FREE,  {CzechAllowFlag::Short, CzechAllowFlag::Middle                      }, {}, "M\u011b\u0159" },
	{ CZG_FREE,  {                       CzechAllowFlag::Middle, CzechAllowFlag::Long}, {}, "Lip" },
};

/* Optional postfix inserted between stem and ending. */
static const std::string_view _name_czech_subst_postfix[] = {
	"av", "an", "at",
	"ov", "on", "ot",
	"ev", "en", "et",
};

/* This array must have the both neutral genders at the end! */
static const CzechNameSubst _name_czech_subst_ending[] = {
	{ CZG_SMASC, {CzechAllowFlag::Short, CzechAllowFlag::Middle                      }, CZC_ANY, "ec" },
	{ CZG_SMASC, {CzechAllowFlag::Short, CzechAllowFlag::Middle                      }, CZC_ANY, "\u00edn" },
	{ CZG_SMASC, {CzechAllowFlag::Short, CzechAllowFlag::Middle, CzechAllowFlag::Long}, CZC_ANY, "ov" },
	{ CZG_SMASC, {CzechAllowFlag::Short,                         CzechAllowFlag::Long}, CZC_ANY, "kov" },
	{ CZG_SMASC, {                                               CzechAllowFlag::Long}, CzechChooseFlag::Postfix, "\u00edn" },
	{ CZG_SMASC, {                                               CzechAllowFlag::Long}, CzechChooseFlag::Postfix, "n\u00edk" },
	{ CZG_SMASC, {                                               CzechAllowFlag::Long}, CZC_ANY, "burk" },
	{ CZG_SFEM,  {CzechAllowFlag::Short                                              }, CZC_ANY, "ka" },
	{ CZG_SFEM,  {                       CzechAllowFlag::Middle                      }, CZC_ANY, "inka" },
	{ CZG_SFEM,  {                       CzechAllowFlag::Middle                      }, CZC_ANY, "n\u00e1" },
	{ CZG_SFEM,  {                                               CzechAllowFlag::Long}, CZC_ANY, "ava" },
	{ CZG_PMASC, {                                               CzechAllowFlag::Long}, CzechChooseFlag::Postfix, "\u00edky" },
	{ CZG_PMASC, {                                               CzechAllowFlag::Long}, CZC_ANY, "upy" },
	{ CZG_PMASC, {                                               CzechAllowFlag::Long}, CZC_ANY, "olupy" },
	{ CZG_PFEM,  {                                               CzechAllowFlag::Long}, CZC_ANY, "avy" },
	{ CZG_PFEM,  {CzechAllowFlag::Short, CzechAllowFlag::Middle, CzechAllowFlag::Long}, CZC_ANY, "ice" },
	{ CZG_PFEM,  {CzechAllowFlag::Short, CzechAllowFlag::Middle, CzechAllowFlag::Long}, CZC_ANY, "i\u010dky" },
	{ CZG_PNEUT, {CzechAllowFlag::Short, CzechAllowFlag::Middle                      }, CZC_ANY, "na" },
	{ CZG_SNEUT, {CzechAllowFlag::Short, CzechAllowFlag::Middle                      }, CZC_ANY, "no" },
	{ CZG_SNEUT, {                                               CzechAllowFlag::Long}, CZC_ANY, "i\u0161t\u011b" },
};

static const std::string_view _name_czech_suffix[] = {
	"nad Cidlinou",
	"nad Dyj\u00ed",
	"nad Jihlavou",
	"nad Labem",
	"nad Lesy",
	"nad Moravou",
	"nad Nisou",
	"nad Odrou",
	"nad Ostravic\u00ed",
	"nad S\u00e1zavou",
	"nad Vltavou",
	"pod Prad\u011bdem",
	"pod Radho\u0161t\u011bm",
	"pod \u0158\u00edpem",
	"pod Sn\u011b\u017ekou",
	"pod \u0160pi\u010d\u00e1kem",
	"pod Sedlem",
	"v \u010cech\u00e1ch",
	"na Morav\u011b",
};



static const std::string_view _name_romanian_real[] = {
	"Adjud",
	"Alba Iulia",
	"Alexandria",
	"Babadag",
	"Bac\u0103u",
	"Baia Mare",
	"B\u0103ile Herculane",
	"B\u0103ile\u0219ti",
	"B\u00e2rlad",
	"Bicaz",
	"Bistri\u021ba",
	"Blaj",
	"Borsec",
	"Boto\u0219ani",
	"Br\u0103ila",
	"Bra\u0219ov",
	"Bucure\u0219ti",
	"Buftea",
	"Buz\u0103u",
	"C\u0103l\u0103ra\u0219i",
	"Caransebe\u0219",
	"Cernavod\u0103",
	"Cluj-Napoca",
	"Constan\u021ba",
	"Covasna",
	"Craiova",
	"Dej",
	"Deva",
	"Dorohoi",
	"Drobeta-Turnu Severin",
	"Dr\u0103g\u0103\u0219ani",
	"F\u0103g\u0103ra\u0219",
	"F\u0103lticeni",
	"Fete\u0219ti",
	"Foc\u0219ani",
	"Gala\u021bi",
	"Gheorgheni",
	"Giurgiu",
	"H\u00e2r\u0219ova",
	"Hunedoara",
	"Hu\u0219i",
	"Ia\u0219i",
	"Isaccea",
	"Lugoj",
	"M\u0103cin",
	"Mangalia",
	"Medgidia",
	"Media\u0219",
	"Miercurea Ciuc",
	"Mizil",
	"Motru",
	"N\u0103s\u0103ud",
	"N\u0103vodari",
	"Odobe\u0219ti",
	"Olteni\u021ba",
	"One\u0219ti",
	"Oradea",
	"Or\u0219ova",
	"Petro\u0219ani",
	"Piatra Neam\u021b",
	"Pite\u0219ti",
	"Ploie\u0219ti",
	"Predeal",
	"R\u00e2mnicu V\u00e2lcea",
	"Reghin",
	"Re\u0219i\u021ba",
	"Roman",
	"Ro\u0219iorii de Vede",
	"Satu Mare",
	"Sebe\u0219",
	"Sf\u00e2ntu Gheorghe",
	"Sibiu",
	"Sighi\u0219oara",
	"Sinaia",
	"Slatina",
	"Slobozia",
	"Sovata",
	"Suceava",
	"Sulina",
	"\u021a\u0103nd\u0103rei",
	"T\u00e2rgovi\u0219te",
	"T\u00e2rgu Jiu",
	"T\u00e2rgu Mure\u0219",
	"Tecuci",
	"Timi\u0219oara",
	"Tulcea",
	"Turda",
	"Turnu M\u0103gurele",
	"Urziceni",
	"Vaslui",
	"Vatra Dornei",
	"Victoria",
	"Videle",
	"Zal\u0103u",
};

static const std::string_view _name_slovak_real[] = {
	"Bratislava",
	"B\u00e1novce nad Bebravou",
	"Bansk\u00e1 Bystrica",
	"Bansk\u00e1 \u0160tiavnica",
	"Bardejov",
	"Brezno",
	"Brezova pod Bradlom",
	"Byt\u010da",
	"\u010cadca",
	"\u010cierna nad Tisou",
	"Detva",
	"Pre\u0161ov",
	"Dolny Kubin",
	"Spi\u0161sk\u00e1 Nov\u00e1 Ves",
	"Dunajsk\u00e1 Streda",
	"Gab\u010d\u00edikovo",
	"Galanta",
	"Gbely",
	"Gelnica",
	"Handlov\u00e1",
	"Hlohovec",
	"Hol\u00ed\u010d",
	"Humenn\u00e9",
	"Hurbanovo",
	"Kezmarok",
	"Kom\u00e1rno",
	"Ko\u0161ice",
	"Kremnica",
	"Krompachy",
	"Kuty",
	"Leopoldov",
	"Levo\u010da",
	"Liptovsk\u00fd Mikul\u00e1\u0161",
	"Lu\u010denec",
	"Malacky",
	"Martin",
	"Medzilaborce",
	"Michalovce",
	"Modra",
	"Myjava",
	"N\u00e1mestovo",
	"Nitra",
	"Nov\u00e1 Ba\u0148a",
	"Nov\u00e9 Mesto nad V\u00e1hom",
	"Nov\u00e9 Z\u00e1mky",
	"Partiz\u00e1nske",
	"Pezinok",
	"Pie\u0161\u0165any",
	"Polt\u00e1r",
	"Poprad",
	"Pova\u017esk\u00e1 Bystrica",
	"Prievidza",
	"P\u00fachov",
	"Rev\u00faca",
	"Rimavsk\u00e1 Sobota",
	"Ro\u017e\u0148ava",
	"Ru\u017eomberok",
	"Sabinov",
	"\u0160a\u013ea",
	"Senec",
	"Senica",
	"Sere\u010f",
	"Skalica",
	"Sl\u00e1dkovi\u00cdovo",
	"Smolenice",
	"Snina",
	"Star\u00e1 \u013dubov\u0148a",
	"Star\u00e1 Tur\u00e1",
	"Str\u00e1\u017eske",
	"Stropkov",
	"Stupava",
	"\u0160t\u00farovo",
	"\u0120ulekovo",
	"Topo\u013e\u010dany",
	"Trebi\u0161ov",
	"Tren\u010d\u00edn",
	"Trnava",
	"Tur\u010dianske Teplice",
	"Tvrdo\u0161\u00edn",
	"Vr\u00e1ble",
	"Vranov nad Top\u013eou",
	"Z\u00e1horsk\u00e1 Bystrica",
	"\u017ddiar",
	"\u013diar nad Hronom",
	"\u017dilina",
	"Zlat\u00e9 Moravce",
	"Zvolen",
};

static const std::string_view _name_norwegian_1[] = {
	"Arna",
	"Aust",
	"Bj\u00f8rk",
	"Bj\u00f8rn",
	"Brand",
	"B\u00f8ver",
	"Drag",
	"Dr\u00f8",
	"Eids",
	"Egge",
	"Fager",
	"Finns",
	"Flat",
	"Foll",
	"Foss",
	"Fugle",
	"Furu",
	"Gaus",
	"Galte",
	"Geir",
	"Gl\u00f8s",
	"Gran",
	"Grind",
	"Grims",
	"Gr\u00f8n",
	"Gr\u00f8t",
	"Gulle",
	"Haka",
	"Hammer",
	"Haug",
	"Hol",
	"Hon",
	"Hop",
	"Hov",
	"Jess",
	"Kabel",
	"Kjerns",
	"Kjerring",
	"Knatte",
	"Krok",
	"K\u00f8y",
	"Lang",
	"Lauv",
	"Leir",
	"Lund",
	"Logn",
	"Lo",
	"Lyng",
	"L\u00f8n",
	"Mesna",
	"Mel",
	"Mo",
	"Nar",
	"Nitte",
	"Nord",
	"Odd",
	"Ola",
	"Otte",
	"Ran",
	"Rev",
	"Rog",
	"Roms",
	"Rosen",
	"Sand",
	"Sau",
	"Sel",
	"Sol",
	"Sjur",
	"Sk\u00e5r",
	"Sl\u00e5tt",
	"Stj\u00f8r",
	"Stor",
	"Svart",
	"Svens",
	"Svin",
	"Sylte",
	"Syn",
	"Tran",
	"Vass",
	"Ved",
	"Vest",
	"Vesle",
	"Vik",
	"V\u00e5g",
};

static const std::string_view _name_norwegian_2[] = {
	"aker",
	"anger",
	"bakken",
	"bekk",
	"berg",
	"botn",
	"breen",
	"bu",
	"bugen",
	"by",
	"bygd",
	"b\u00f8",
	"dal",
	"egga",
	"eid",
	"elv",
	"enga",
	"foss",
	"fjell",
	"fjord",
	"foten",
	"gard",
	"grend",
	"hammer",
	"haug",
	"havn",
	"heim",
	"hella",
	"hovda",
	"h\u00f8a",
	"h\u00f8gda",
	"kampen",
	"kj\u00f8len",
	"kollen",
	"kroken",
	"land",
	"lia",
	"mark",
	"moen",
	"myr",
	"nes",
	"nuten",
	"osen",
	"rike",
	"rud",
	"sand",
	"set",
	"sj\u00f8en",
	"skogen",
	"slette",
	"snipa",
	"stad",
	"stua",
	"stulen",
	"sund",
	"svingen",
	"s\u00e6tra",
	"tinden",
	"tun",
	"vang",
	"vatn",
	"veid",
	"vik",
	"voll",
	"v\u00e5g",
	"um",
	"\u00e5sen",
};

static const std::string_view _name_norwegian_real[] = {
	"Alta",
	"Arendal",
	"Askim",
	"Bergen",
	"Bod\u00f8",
	"Brevik",
	"Bryne",
	"Br\u00f8nn\u00f8ysund",
	"Drammen",
	"Dr\u00f8bak",
	"Egersund",
	"Elverum",
	"Farsund",
	"Fauske",
	"Finnsnes",
	"Flekkefjord",
	"Flora",
	"Fosnav\u00e5g",
	"Fredrikstad",
	"F\u00f8rde",
	"Gj\u00f8vik",
	"Grimstad",
	"Halden",
	"Hamar",
	"Hammerfest",
	"Harstad",
	"Haugesund",
	"Holmestrand",
	"Horten",
	"J\u00f8rpeland",
	"Kirkenes",
	"Kolvereid",
	"Kongsberg",
	"Kongsvinger",
	"Kopervik",
	"Krager\u00f8",
	"Kristiansand",
	"Kristiansund",
	"Langesund",
	"Larvik",
	"Leirvik",
	"Leknes",
	"Levanger",
	"Lillehammer",
	"Lillesand",
	"Lillestr\u00f8m",
	"Lyngdal",
	"L\u00f8renskog",
	"Mandal",
	"Mo i Rana",
	"Molde",
	"Mosj\u00f8en",
	"Moss",
	"Mysen",
	"M\u00e5l\u00f8y",
	"Namsos",
	"Narvik",
	"Notodden",
	"Odda",
	"Oslo",
	"Otta",
	"Porsgrunn",
	"Ringerike",
	"Ris\u00f8r",
	"Rjukan",
	"Sandefjord",
	"Sandnes",
	"Sandnessj\u00f8en",
	"Sandvika",
	"Sarpsborg",
	"Sauda",
	"Ski",
	"Skien",
	"Skudeneshavn",
	"Sortland",
	"Stathelle",
	"Stavanger",
	"Steinkjer",
	"Stj\u00f8rdal",
	"Stokmarknes",
	"Stord",
	"Svelvik",
	"Svolv\u00e6r",
	"Troms\u00f8",
	"Trondheim",
	"Tvedestrand",
	"T\u00f8nsberg",
	"Ulsteinvik",
	"Vads\u00f8",
	"Vard\u00f8",
	"Verdals\u00f8ra",
	"\u00c5krehamn",
	"\u00c5lesund",
	"\u00c5ndalsnes",
};

static const std::string_view _name_hungarian_1[] = {
	"Nagy-",
	"Kis-",
	"Fels\u0151-",
	"Als\u00f3-",
	"\u00daj-",
};

static const std::string_view _name_hungarian_2[] = {
	"Bodrog",
	"Dr\u00e1va",
	"Duna",
	"Hej\u0151",
	"Hern\u00e1d",
	"R\u00e1ba",
	"Saj\u00f3",
	"Szamos",
	"Tisza",
	"Zala",
	"Balaton",
	"Fert\u0151",
	"Bakony",
	"Cserh\u00e1t",
	"Bihar",
	"Hajd\u00fa",
	"J\u00e1sz",
	"Kun",
	"Magyar",
	"N\u00f3gr\u00e1d",
	"Ny\u00edr",
	"Somogy",
	"Sz\u00e9kely",
	"Buda",
	"Gy\u0151r",
	"Pest",
	"Feh\u00e9r",
	"Cser\u00e9p",
	"Erd\u0151",
	"Hegy",
	"Homok",
	"Mez\u0151",
	"Puszta",
	"S\u00e1r",
	"Cs\u00e1sz\u00e1r",
	"Herceg",
	"Kir\u00e1ly",
	"Nemes",
	"P\u00fcsp\u00f6k",
	"Szent",
	"Alm\u00e1s",
	"Szilv\u00e1s",
	"Agg",
	"Aranyos",
	"B\u00e9k\u00e9s",
	"Egyh\u00e1zas",
	"Gagy",
	"Heves",
	"Kapos",
	"T\u00e1pi\u00f3",
	"Torna",
	"Vas",
	"V\u00e1mos",
	"V\u00e1s\u00e1ros",
};

static const std::string_view _name_hungarian_3[] = {
	"ap\u00e1ti",
	"b\u00e1ba",
	"bikk",
	"dob",
	"fa",
	"f\u00f6ld",
	"hegyes",
	"kak",
	"kereszt",
	"k\u00fcrt",
	"lad\u00e1ny",
	"m\u00e9rges",
	"szalonta",
	"telek",
	"vas",
	"v\u00f6lgy",
};

static const std::string_view _name_hungarian_4[] = {
	"alja",
	"egyh\u00e1za",
	"h\u00e1za",
	"\u00far",
	"v\u00e1r",
};

static const std::string_view _name_hungarian_real[] = {
	"Ajka",
	"Asz\u00f3d",
	"Badacsony",
	"Baja",
	"Budapest",
	"Debrecen",
	"Eger",
	"Fony\u00f3d",
	"G\u00f6d\u00f6ll\u0151",
	"Gy\u0151r",
	"Gyula",
	"Karcag",
	"Kecskem\u00e9t",
	"Keszthely",
	"Kisk\u00f6re",
	"Kocsord",
	"Kom\u00e1rom",
	"K\u0151szeg",
	"Mak\u00f3",
	"Moh\u00e1cs",
	"Miskolc",
	"\u00d3zd",
	"Paks",
	"P\u00e1pa",
	"P\u00e9cs",
	"Polg\u00e1r",
	"Sarkad",
	"Si\u00f3fok",
	"Szeged",
	"Szentes",
	"Szolnok",
	"Tihany",
	"Tokaj",
	"V\u00e1c",
	"Z\u00e1hony",
	"Zirc",
};

static const std::string_view _name_swiss_real[] = {
	"Aarau",
	"Aesch",
	"Altdorf",
	"Arosa",
	"Appenzell",
	"Arbon",
	"Altst\u00e4tten",
	"Baar",
	"Baden",
	"Bellinzona",
	"Brig-Glis",
	"Bienne",
	"Bulle",
	"Binningen",
	"Burgdorf",
	"Bern",
	"Basel",
	"B\u00fclach",
	"Carouge",
	"Cham",
	"Chiasso",
	"Chur",
	"Davos",
	"Del\u00e9mont",
	"Dietikon",
	"D\u00fcbendorf",
	"Emmen",
	"Freienbach-Pf\u00e4ffikon",
	"Fribourg",
	"Frauenfeld",
	"Gen\u00e8ve",
	"Glarus",
	"Gossau",
	"Grenchen",
	"Herisau",
	"Horgen",
	"Horw",
	"Illnau-Effretikon",
	"Ittigen",
	"Jona",
	"Kriens",
	"Kloten",
	"K\u00f6niz",
	"Kreuzlingen",
	"K\u00fcsnacht",
	"Agen",
	"Lancy",
	"La Chaux-de-Fonds",
	"Lenzburg",
	"Lugano",
	"Langenthal",
	"Littau",
	"Le Locle",
	"La Neuveville",
	"Locarno",
	"Liestal",
	"La Tour-de-Peilz",
	"Lausanne",
	"Lyss",
	"Luzern",
	"Martigny",
	"M\u00fcnchenstein",
	"Meyrin",
	"Montreux",
	"Monthey",
	"Morges",
	"Murten",
	"Moutier",
	"Muttenz",
	"Neuch\u00e2tel",
	"Neuhausen am Rheinfall",
	"Nyon",
	"Olten",
	"Onex",
	"Opfikon",
	"Ostermundigen",
	"Payerne",
	"Peseux",
	"Prilly",
	"Pully",
	"Rapperswil",
	"Richterswil",
	"Regensdorf",
	"Rheinfelden",
	"Riehen",
	"Renens",
	"Romanshorn",
	"Rorschach",
	"Stans",
	"Schaffhausen",
	"Steffisburg",
	"St. Gallen",
	"Schlieren",
	"Sierre",
	"Solothurn",
	"St. Moritz",
	"Sion",
	"Spiez",
	"St\u00e4fa",
	"Sursee",
	"Schwyz",
	"Thalwil",
	"Th\u00f4nex",
	"Thun",
	"Uster",
	"Uzwil",
	"Vernier",
	"Volketswil",
	"Versoix",
	"Vevey",
	"W\u00e4denswil",
	"Wettingen",
	"Wil",
	"Wallisellen",
	"Winterthur",
	"Wohlen",
	"Worb",
	"Wetzikon",
	"Yverdon-les-Bains",
	"Zollikon",
	"Zofingen",
	"Z\u00fcrich",
	"Zug",
};

static const std::string_view _name_danish_1[] = {
	"Gamle ",
	"Lille ",
	"Nye ",
	"Store ",
	"Kirke ",
	"N\u00f8rre ",
	"Vester ",
	"S\u00f8nder ",
	"\u00d8ster ",
	"Hvide ",
	"H\u00f8je ",
	"Kongens ",
};

static const std::string_view _name_danish_2[] = {
	"Ager",
	"Alle",
	"Aske",
	"Balle",
	"Bede",
	"Birke",
	"Bjerring",
	"Bj\u00e6ver",
	"Blommens",
	"Blok",
	"Bolder",
	"Bred",
	"Charlotten",
	"Christians",
	"Danne",
	"Diana",
	"Es",
	"Fredens",
	"Frederiks",
	"Fugle",
	"F\u00e5re",
	"Gille",
	"Gis",
	"Givs",
	"Glams",
	"Glo",
	"Guld",
	"Had",
	"Haralds",
	"Hassel",
	"Hede",
	"Helle",
	"Hessel",
	"Hjorts",
	"Hol",
	"Horn",
	"Humle",
	"H\u00f8j",
	"H\u00f8r",
	"Is",
	"Jyde",
	"J\u00e6gers",
	"Karls",
	"Klov",
	"Kokke",
	"Kvist",
	"Lang",
	"Lange",
	"Mari",
	"Nord",
	"Ny",
	"Oks",
	"Ring",
	"R\u00f8de",
	"Rung",
	"R\u00f8r",
	"Rud",
	"Saks",
	"Salt",
	"Skam",
	"Silke",
	"Skod",
	"Sk\u00e6l",
	"Sk\u00e6r",
	"Sol",
	"Svend",
	"Svine",
	"Strand",
	"Stubbe",
	"Ting",
	"Tj\u00e6re",
	"Tore",
	"Uger",
	"Ulf",
	"Val",
	"Vand",
	"Vej",
	"Vor",
	"V\u00e6r",
	"\u00d8r",
	"\u00c5l",
};

static const std::string_view _name_danish_3[] = {
	"basse",
	"borg",
	"berg",
	"bro",
	"by",
	"havn",
	"strup",
	"holm",
	"hus",
	"k\u00f8bing",
	"lund",
	"lunde",
	"sund",
	"ovre",
	"h\u00f8j",
	"dal",
	"sted",
	"sten",
	"l\u00f8se",
	"r\u00f8d",
	"magle",
	"s\u00f8",
	"bjerg",
	"b\u00e6k",
	"drup",
	"lev",
	"bo",
	"lyst",
	"feld",
	"skov",
};

static const std::string_view _name_turkish_prefix[] = {
	"Ak\u00e7a",
	"Alt\u0131n",
	"Bah\u00e7e",
	"Boz",
	"B\u00fcy\u00fck",
	"\u00c7ay",
	"Do\u011fu",
	"Eski",
	"G\u00fczel",
	"K\u0131z\u0131l",
	"K\u00fc\u00e7\u00fck",
	"Orta",
	"Sar\u0131",
	"Sultan",
	"Ulu",
	"Yeni",
};

static const std::string_view _name_turkish_middle[] = {
	"aga\u00e7",
	"ayva",
	"\u00e7am",
	"elma",
	"kurt",
	"pazar",
	"yal\u0131",
};

static const std::string_view _name_turkish_suffix[] = {
	"dere",
	"hisar",
	"kale",
	"kaya",
	"kent",
	"k\u00f6y",
	"ova",
	"\u00f6z\u00fc",
	"\u00f6ren",
	"pazar",
	"saray",
	"tepe",
	"yer",
	"yurt",
};

static const std::string_view _name_turkish_real[] = {
	"Adana",
	"Ad\u0131yaman",
	"Afyon",
	"A\u011fr\u0131",
	"Amasya",
	"Antalya",
	"Artvin",
	"Bal\u0131kesir",
	"Bilecik",
	"Bitlis",
	"Bolu",
	"Burdur",
	"Bursa",
	"\u00c7anakkale",
	"\u00c7ank\u0131r\u0131",
	"Denizli",
	"Diyarbak\u0131r",
	"Edirne",
	"Elaz\u0131\u011f",
	"Erzurum",
	"Eski\u015fehir",
	"Giresun",
	"G\u00fcm\u00fc\u015fhane",
	"Hatay",
	"Isparta",
	"\u0130\u00e7el",
	"\u0130stanbul",
	"\u0130zmir",
	"Kars",
	"Kastamonu",
	"Kayseri",
	"Kirklareli",
	"Kocaeli",
	"Konya",
	"K\u00fctahya",
	"Malatya",
	"Manisa",
	"Kahramanmara\u015f",
	"Mardin",
	"Mu\u011fla",
	"Mu\u015f",
	"Nev\u015fehir",
	"Ni\u011fde",
	"Rize",
	"Sakarya",
	"Samsun",
	"Siirt",
	"Sinop",
	"Sivas",
	"Trabzon",
	"\u015eanl\u0131urfa",
	"Van",
	"Yozgat",
	"Zonguldak",
	"Aksaray",
	"Bayburt",
	"Karaman",
	"\u015e\u0131rnak",
	"Bart\u0131n",
	"Ardahan",
	"I\u011fd\u0131r",
	"Yalova",
	"Karab\u00fck",
	"Osmaniye",
	"D\u00fczce",
};

static const std::string_view _name_italian_real[] = {
	"Roma",
	"Milano",
	"Napoli",
	"Torino",
	"Venezia",
	"Firenze",
	"Palermo",
	"Genova",
	"Parma",
	"Bologna",
	"Bari",
	"Cagliari",
	"Sassari",
	"Pisa",
	"Aosta",
	"Brescia",
	"Verona",
	"Bolzano",
	"Padova",
	"Udine",
	"Trieste",
	"Livorno",
	"Ancona",
	"Perugia",
	"Pescara",
	"L'Aquila",
	"Campobasso",
	"Potenza",
	"Cosenza",
	"Reggio Calabria",
	"Catania",
	"Caltanisetta",
	"Agrigento",
	"La Spezia",
	"Modena",
	"Vicenza",
	"Mantova",
	"Cremona",
	"Piacenza",
	"Reggio Emilia",
	"Foggia",
	"Benevento",
	"Salerno",
	"Catanzaro",
	"Lecce",
	"Como",
	"Lecco",
	"Sondrio",
	"Trento",
	"Desenzano",
	"Cuneo",
	"Asti",
	"Lodi",
	"Novara",
	"Biella",
	"Vercelli",
	"Rieti",
	"Nuoro",
	"Oristano",
	"Matera",
	"Taranto",
	"Varese",
	"Bergamo",
	"Pavia",
	"Caserta",
	"Frosinone",
	"Latina",
	"Enna",
	"Ragusa",
	"Siracusa",
	"Pordenone",
	"Imperia",
	"Verbania",
	"Alessandria",
	"Messina",
	"Siena",
	"Arezzo",
	"Grosseto",
};

static const std::string_view _name_italian_pref[] = {
	"Alpe ",
	"Borgo ",
	"Cascina ",
	"Castel ",
	"Fonte ",
	"Forte ",
	"Malga ",
	"Pieve ",
	"Poggio ",
	"Rocca ",
	"Villa ",
	"Villar ",
};

static const std::string_view _name_italian_1m[] = {
	"Bel",
	"Borgo",
	"Bosco",
	"Campo",
	"Capo",
	"Casal",
	"Castel",
	"Colle",
	"Fiume",
	"Fonte",
	"Lago",
	"Mezzo",
	"Monte",
	"Mon",
	"Orto",
	"Passo",
	"Prato",
	"Poggio",
	"Ponte",
	"Pozzo",
	"Sasso",
	"Tra",
	"Tre",
	"Ver",
	"Vico",
};

static const std::string_view _name_italian_1f[] = {
	"Acqua",
	"Bra",
	"Cala",
	"Casa",
	"Chiesa",
	"Citta",
	"Civita",
	"Corte",
	"Costa",
	"Croce",
	"Fontana",
	"Grotta",
	"Guardia",
	"Mezza",
	"Palma",
	"Pietra",
	"Ripa",
	"Rocca",
	"Serra",
	"Torre",
	"Val",
	"Valle",
	"Villa",
};

static const std::string_view _name_italian_2[] = {
	"bell",
	"bianc",
	"cald",
	"chiar",
	"cort",
	"ferrat",
	"fier",
	"fredd",
	"gioios",
	"grec",
	"guzz",
	"lung",
	"long",
	"migli",
	"negr",
	"ner",
	"nov",
	"nuov",
	"ross",
	"rotond",
	"scur",
	"secc",
	"sett",
	"vecchi",
	"ventos",
	"vers",
	"viv",
};

static const std::string_view _name_italian_2i[] = {
	"",
	"breve",
	"brevi",
	"chiari",
	"ferro",
	"fieschi",
	"fiore",
	"fonte",
	"forte",
	"gate",
	"leone",
	"maggiore",
	"minore",
	"mole",
	"monte",
	"poli",
	"scuri",
	"terra",
	"te",
	"torrione",
	"vento",
	"verde",
	"versiere",
};


static const std::string_view _name_italian_3[] = {
	" Marittimo",
	" Marittima",
	" del Capo",
	" del Monte",
	" di Sopra",
	" di Sotto",
	" sui Monti",
	" dei Marmi",
	" dei Sassi",
	" delle Fonti",
	" sui Prati",
	" a Mare",
	" Superiore",
	" Inferiore",
	" Terme",
	" Alta",
	" Bassa",
	" Brianza",
	" Vesuviano",
	" Scrivia",
	" Ticino",
};

static const std::string_view _name_italian_river1[] = {
	" del",
	" sul",
	" al",
	" nel",
};

static const std::string_view _name_italian_river2[] = {
	"l'Adda",
	"l'Adige",
	"le Alpi",
	"l'Arno",
	" Bormida",
	" Brenta",
	"la Dora Baltea",
	" Lambro",
	" Mincio",
	" Naviglio",
	"l'Oglio",
	"l'Olona",
	"l'Ombrone",
	" Panaro",
	" Piave",
	" Po",
	" Reno",
	" Scrivia",
	" Secchia",
	" Serio",
	" Tagliamento",
	" Tanaro",
	" Taro",
	" Ticino",
	" Tevere",
};

static const std::string_view _name_catalan_real[] = {
	"Barcelona",
	"L'Hospitalet de Llobregat",
	"Cerdanyola",
	"Martorell",
	"Badalona",
	"Tarragona",
	"Lleida",
	"Girona",
	"Sabadell",
	"Terrassa",
	"Reus",
	"Valls",
	"Vic",
	"Vielha e Mijaran",
	"Amposta",
	"Tortosa",
	"Berga",
	"Olot",
	"Mollerussa",
	"Banyoles",
	"Figueres",
	"Balaguer",
	"Vilafranca del Pened\u00e8s",
	"La Seu d'Urgell",
	"El Pont de Suert",
	"Igualada",
	"Manresa",
	"Solsona",
	"Les Borges Blanques",
	"Tremp",
	"Sort",
	"Colera",
	"Portbou",
	"El Vendrell",
	"Falset",
	"Ripoll",
	"Cervera",
	"Gandesa",
	"Matar\u00f3",
	"Montblanc",
	"Vilanova i la Geltr\u00fa",
	"T\u00e0rrega",
	"Camprodon",
	"Campdev\u00e0nol",
	"Cambrils",
	"Begur",
	"Setcases",
	"Palafrugell",
	"Begues",
	"El Bruc",
	"Cadaqu\u00e9s",
	"Collbat\u00f3",
	"Cervell\u00f3",
	"Esparreguera",
	"Abrera",
	"Alp",
	"Das",
	"Cercs",
	"Manlleu",
	"El Masnou",
	"Molins de Rei",
	"Monistrol",
	"Rocallaura",
	"Rub\u00ed",
	"Ripollet",
	"Sitges",
	"Roses",
};

static const std::string_view _name_catalan_pref[] = {
	"El Pont de ",
	"Parets de ",
	"Canet de ",
	"Castellar de ",
	"Corbera de ",
	"Arenys de ",
	"Calella de ",
	"La Seu de ",
	"La Bisbal de ",
	"Torroella de ",
	"Port de ",
	"Vilafranca de ",
	"Vilanova de ",
	"Caldes de ",
	"La Conca de ",
	"Olesa de ",
	"La Roca de ",
	"Sant Esteve de ",
	"Sant Andreu de ",
	"Sant Jordi de ",
	"Sant Joan de ",
	"Sant Feliu de ",
	"Sant Quirze de ",
	"Sant Sadurn\u00ed de ",
	"Santa Coloma de ",
	"Santa Margarida de ",
	"Santa Maria de ",
	"Sant Mart\u00ed de ",
	"Sant Pere de ",
	"Sant Juli\u00e0 de ",
	"Sant Vicen\u00e7 de ",

};

static const std::string_view _name_catalan_1m[] = {
	"Torrent",
	"Cami",
	"Mont",
	"Bell",
	"Puig",
	"Riu",
};

static const std::string_view _name_catalan_1f[] = {
	"Pala",
	"Selva",
	"Vall",
	"Serra",
	"Torre",
	"Riba",
	"Cova",
	"Terra",
};

static const std::string_view _name_catalan_2m[] = {
	"alt",
	"baix",
	"fosc",
	"pelat",
	"vent\u00f3s",
	"negre",
	"roig",
	"gr\u00eds",
};

static const std::string_view _name_catalan_2f[] = {
	"baixa",
	"alta",
	"fosca",
	"clara",
	"negra",
	"roja",
	"grisa",
	"freda",
};

static const std::string_view _name_catalan_3[] = {
	" Desp\u00ed",
	" Desvern",
	" del Cam\u00ed",
	" de Mar",
	" de Dalt",
	" de Baix",
	" del Vall\u00e8s",
	" de Bergued\u00e0",
	" de Conflent",
	" de la Plana",
};

static const std::string_view _name_catalan_river1[] = {
	" d'Anoia",
	" de Ter",
	" de Llobregat",
	" d'Ebre",
	" de Segre",
	" de Francol\u00ed",
};
