(function ($$1) {
  $$1 = $$1 && $$1.hasOwnProperty('default') ? $$1['default'] : $$1;

  define("jquery", function() {
    return $;
  });
  define("underscore", function() {
    return {};
  });

  window.exports = {};

  // Copyright (c) Jupyter Development Team.
  // Distributed under the terms of the Modified BSD License.

  define('services/kernels/comm', [
      'jquery',
      'base/js/utils',
  ], function($, utils) {

      //-----------------------------------------------------------------------
      // CommManager class
      //-----------------------------------------------------------------------
      
      var CommManager = function (kernel) {
          this.comms = {};
          this.targets = {};
          if (kernel !== undefined) {
              this.init_kernel(kernel);
          }
      };
      
      CommManager.prototype.init_kernel = function (kernel) {
          /**
           * connect the kernel, and register message handlers
           */
          this.kernel = kernel;
          var msg_types = ['comm_open', 'comm_msg', 'comm_close'];
          for (var i = 0; i < msg_types.length; i++) {
              var msg_type = msg_types[i];
              kernel.register_iopub_handler(msg_type, $.proxy(this[msg_type], this));
          }
      };

      CommManager.prototype.new_comm = function (target_name, data, callbacks, metadata, comm_id, buffers) {
          /**
           * Create a new Comm, register it, and open its Kernel-side counterpart
           * Mimics the auto-registration in `Comm.__init__` in the Jupyter Comm.
           *
           * argument comm_id is optional
           */
          var comm = new Comm(target_name, comm_id);
          this.register_comm(comm);
          comm.open(data, callbacks, metadata, buffers);
          return comm;
      };
      
      CommManager.prototype.register_target = function (target_name, f) {
          /**
           * Register a target function for a given target name
           */
          this.targets[target_name] = f;
      };
      
      CommManager.prototype.unregister_target = function (target_name, f) {
          /**
           * Unregister a target function for a given target name
           */
          delete this.targets[target_name];
      };
      
      CommManager.prototype.register_comm = function (comm) {
          /**
           * Register a comm in the mapping
           */
          this.comms[comm.comm_id] = Promise.resolve(comm);
          comm.kernel = this.kernel;
          return comm.comm_id;
      };
      
      CommManager.prototype.unregister_comm = function (comm) {
          /**
           * Remove a comm from the mapping
           */
          delete this.comms[comm.comm_id];
      };
      
      // comm message handlers
      
      CommManager.prototype.comm_open = function (msg) {
          var content = msg.content;
          var that = this;
          var comm_id = content.comm_id;

          this.comms[comm_id] = utils.load_class(content.target_name, content.target_module, 
              this.targets).then(function(target) {
                  var comm = new Comm(content.target_name, comm_id);
                  comm.kernel = that.kernel;
                  try {
                      var response = target(comm, msg);
                  } catch (e) {
                      comm.close();
                      that.unregister_comm(comm);
                      var wrapped_error = new utils.WrappedError("Exception opening new comm", e);
                      console.error(wrapped_error);
                      return Promise.reject(wrapped_error);
                  }
                  // Regardless of the target return value, we need to
                  // then return the comm
                  return Promise.resolve(response).then(function() {return comm;});
              }, utils.reject('Could not open comm', true));
          return this.comms[comm_id];
      };
      
      CommManager.prototype.comm_close = function(msg) {
          var content = msg.content;
          if (this.comms[content.comm_id] === undefined) {
              console.error('Comm promise not found for comm id ' + content.comm_id);
              return;
          }
          var that = this;
          this.comms[content.comm_id] = this.comms[content.comm_id].then(function(comm) {
              that.unregister_comm(comm);
              try {
                  comm.handle_close(msg);
              } catch (e) {
                  console.log("Exception closing comm: ", e, e.stack, msg);
              }
              // don't return a comm, so that further .then() functions
              // get an undefined comm input
          });
          return this.comms[content.comm_id];
      };
      
      CommManager.prototype.comm_msg = function(msg) {
          var content = msg.content;
          if (this.comms[content.comm_id] === undefined) {
              console.error('Comm promise not found for comm id ' + content.comm_id);
              return;
          }

          this.comms[content.comm_id] = this.comms[content.comm_id].then(function(comm) {
              return (Promise.resolve(comm.handle_msg(msg))
                  .catch(utils.reject('Exception handling comm message'))
                  .then(function() {return comm;}));
          });
          return this.comms[content.comm_id];
      };
      
      //-----------------------------------------------------------------------
      // Comm base class
      //-----------------------------------------------------------------------
      
      var Comm = function (target_name, comm_id) {
          this.target_name = target_name;
          this.comm_id = comm_id || utils.uuid();
          this._msg_callback = this._close_callback = null;
      };
      
      // methods for sending messages
      Comm.prototype.open = function (data, callbacks, metadata, buffers) {
          var content = {
              comm_id : this.comm_id,
              target_name : this.target_name,
              data : data || {},
          };
          return this.kernel.send_shell_message("comm_open", content, callbacks, metadata, buffers);
      };
      
      Comm.prototype.send = function (data, callbacks, metadata, buffers) {
          var content = {
              comm_id : this.comm_id,
              data : data || {},
          };
          return this.kernel.send_shell_message("comm_msg", content, callbacks, metadata, buffers);
      };

      Comm.prototype.close = function (data, callbacks, metadata, buffers) {
          var content = {
              comm_id : this.comm_id,
              data : data || {},
          };
          return this.kernel.send_shell_message("comm_close", content, callbacks, metadata, buffers);
      };
      
      // methods for registering callbacks for incoming messages
      Comm.prototype._register_callback = function (key, callback) {
          this['_' + key + '_callback'] = callback;
      };
      
      Comm.prototype.on_msg = function (callback) {
          this._register_callback('msg', callback);
      };
      
      Comm.prototype.on_close = function (callback) {
          this._register_callback('close', callback);
      };
      
      // methods for handling incoming messages
      
      Comm.prototype._callback = function (key, msg) {
          var callback = this['_' + key + '_callback'];
          if (callback) {
              try {
                  return callback(msg);
              } catch (e) {
                  console.log("Exception in Comm callback", e, e.stack, msg);
              }
          }
      };
      
      Comm.prototype.handle_msg = function (msg) {
          return this._callback('msg', msg);
      };
      
      Comm.prototype.handle_close = function (msg) {
          this._callback('close', msg);
      };
      
      return {
          'CommManager': CommManager,
          'Comm': Comm
      };
  });

  // Copyright (c) Jupyter Development Team.
  // Distributed under the terms of the Modified BSD License.

  define('services/kernels/kernel', [
      'jquery',
      'base/js/utils',
      './comm',
      './serialize',
      'base/js/events'
  ], function($, utils, comm, serialize, events) {

      /**
       * A Kernel class to communicate with the Python kernel. This
       * should generally not be constructed directly, but be created
       * by.  the `Session` object. Once created, this object should be
       * used to communicate with the kernel.
       * 
       * Preliminary documentation for the REST API is at
       * https://github.com/ipython/ipython/wiki/IPEP-16%3A-Notebook-multi-directory-dashboard-and-URL-mapping#kernels-api
       * 
       * Documentation for the messaging specifications is at
       * https://jupyter-client.readthedocs.io/en/stable/messaging.html
       *
       * @class Kernel
       * @param {string} kernel_service_url - the URL to access the kernel REST api
       * @param {string} ws_url - the websockets URL
       * @param {string} name - the kernel type (e.g. python3)
       */
      var Kernel = function (kernel_service_url, ws_url, name) {
          this.events = events;

          this.id = null;
          this.name = name;
          this.ws = null;
          this._stopping = false;

          this.kernel_service_url = kernel_service_url;
          this.kernel_url = null;
          this.ws_url = ws_url || utils.get_body_data("wsUrl");
          if (!this.ws_url) {
              // trailing 's' in https will become wss for secure web sockets
              this.ws_url = location.protocol.replace('http', 'ws') + "//" + location.host;
          }

          this.username = "username";
          this.session_id = utils.uuid();
          this._msg_callbacks = {};
          this._msg_callbacks_overrides = {};
          this._display_id_to_parent_ids = {};
          this._msg_queue = Promise.resolve();
          this.info_reply = {}; // kernel_info_reply stored here after starting

          if (typeof(WebSocket) !== 'undefined') {
              this.WebSocket = WebSocket;
          } else if (typeof(MozWebSocket) !== 'undefined') {
              this.WebSocket = MozWebSocket;
          } else {
              alert('Your browser does not have WebSocket support, please try Chrome, Safari or Firefox ≥ 6. Firefox 4 and 5 are also supported by you have to enable WebSockets in about:config.');
          }
          
          this.bind_events();
          this.init_iopub_handlers();
          this.comm_manager = new comm.CommManager(this);
          
          this.last_msg_id = null;
          this.last_msg_callbacks = {};

          this._autorestart_attempt = 0;
          this._reconnect_attempt = 0;
          this.reconnect_limit = 7;
          
          this._pending_messages = [];
      };

      /**
       * @function _get_msg
       */
      Kernel.prototype._get_msg = function (msg_type, content, metadata, buffers) {
          var msg = {
              header : {
                  msg_id : utils.uuid(),
                  username : this.username,
                  session : this.session_id,
                  msg_type : msg_type,
                  version : "5.2",
              },
              metadata : metadata || {},
              content : content,
              buffers : buffers || [],
              parent_header : {}
          };
          return msg;
      };

      /**
       * @function bind_events
       */
      Kernel.prototype.bind_events = function () {
          var that = this;
          this.events.on('send_input_reply.Kernel', function(evt, data) { 
              that.send_input_reply(data);
          });

          var record_status = function (evt, info) {
              console.log('Kernel: ' + evt.type + ' (' + info.kernel.id + ')');
          };

          this.events.on('kernel_created.Kernel', record_status);
          this.events.on('kernel_reconnecting.Kernel', record_status);
          this.events.on('kernel_connected.Kernel', record_status);
          this.events.on('kernel_starting.Kernel', record_status);
          this.events.on('kernel_restarting.Kernel', record_status);
          this.events.on('kernel_autorestarting.Kernel', record_status);
          this.events.on('kernel_interrupting.Kernel', record_status);
          this.events.on('kernel_disconnected.Kernel', record_status);
          // these are commented out because they are triggered a lot, but can
          // be uncommented for debugging purposes
          //this.events.on('kernel_idle.Kernel', record_status);
          //this.events.on('kernel_busy.Kernel', record_status);
          this.events.on('kernel_ready.Kernel', record_status);
          this.events.on('kernel_killed.Kernel', record_status);
          this.events.on('kernel_dead.Kernel', record_status);

          this.events.on('kernel_ready.Kernel', function () {
              that._autorestart_attempt = 0;
          });
          this.events.on('kernel_connected.Kernel', function () {
              that._reconnect_attempt = 0;
          });
      };

      /**
       * Initialize the iopub handlers.
       *
       * @function init_iopub_handlers
       */
      Kernel.prototype.init_iopub_handlers = function () {
          var output_msg_types = ['stream', 'display_data', 'execute_result', 'error', 'update_display_data'];
          this._iopub_handlers = {};
          this.register_iopub_handler('status', $.proxy(this._handle_status_message, this));
          this.register_iopub_handler('clear_output', $.proxy(this._handle_clear_output, this));
          this.register_iopub_handler('execute_input', $.proxy(this._handle_input_message, this));
          
          for (var i=0; i < output_msg_types.length; i++) {
              this.register_iopub_handler(output_msg_types[i], $.proxy(this._handle_output_message, this));
          }
      };

      /**
       * GET /api/kernels
       *
       * Get the list of running kernels.
       *
       * @function list
       * @param {function} [success] - function executed on ajax success
       * @param {function} [error] - function executed on ajax error
       */
      Kernel.prototype.list = function (success, error) {
          utils.ajax(this.kernel_service_url, {
              processData: false,
              cache: false,
              type: "GET",
              dataType: "json",
              success: success,
              error: this._on_error(error)
          });
      };

      /**
       * POST /api/kernels
       *
       * Start a new kernel.
       *
       * In general this shouldn't be used -- the kernel should be
       * started through the session API. If you use this function and
       * are also using the session API then your session and kernel
       * WILL be out of sync!
       *
       * @function start
       * @param {params} [Object] - parameters to include in the query string
       * @param {function} [success] - function executed on ajax success
       * @param {function} [error] - function executed on ajax error
       */
      Kernel.prototype.start = function (params, success, error) {
          var url = this.kernel_service_url;
          var qs = $.param(params || {}); // query string for sage math stuff
          if (qs !== "") {
              url = url + "?" + qs;
          }

          this.events.trigger('kernel_starting.Kernel', {kernel: this});
          var that = this;
          var on_success = function (data, status, xhr) {
              that.events.trigger('kernel_created.Kernel', {kernel: that});
              that._kernel_created(data);
              if (success) {
                  success(data, status, xhr);
              }
          };

          utils.ajax(url, {
              processData: false,
              cache: false,
              type: "POST",
              data: JSON.stringify({name: this.name}),
              contentType: 'application/json',
              dataType: "json",
              success: this._on_success(on_success),
              error: this._on_error(error)
          });

          return url;
      };

      /**
       * GET /api/kernels/[:kernel_id]
       *
       * Get information about the kernel.
       *
       * @function get_info
       * @param {function} [success] - function executed on ajax success
       * @param {function} [error] - function executed on ajax error
       */
      Kernel.prototype.get_info = function (success, error) {
          utils.ajax(this.kernel_url, {
              processData: false,
              cache: false,
              type: "GET",
              dataType: "json",
              success: this._on_success(success),
              error: this._on_error(error)
          });
      };

      /**
       * DELETE /api/kernels/[:kernel_id]
       *
       * Shutdown the kernel.
       *
       * If you are also using sessions, then this function should NOT be
       * used. Instead, use Session.delete. Otherwise, the session and
       * kernel WILL be out of sync.
       *
       * @function kill
       * @param {function} [success] - function executed on ajax success
       * @param {function} [error] - function executed on ajax error
       */
      Kernel.prototype.kill = function (success, error) {
          this.events.trigger('kernel_killed.Kernel', {kernel: this});
          this._kernel_dead();
          utils.ajax(this.kernel_url, {
              processData: false,
              cache: false,
              type: "DELETE",
              dataType: "json",
              success: this._on_success(success),
              error: this._on_error(error)
          });
      };

      /**
       * POST /api/kernels/[:kernel_id]/interrupt
       *
       * Interrupt the kernel.
       *
       * @function interrupt
       * @param {function} [success] - function executed on ajax success
       * @param {function} [error] - function executed on ajax error
       */
      Kernel.prototype.interrupt = function (success, error) {
          this.events.trigger('kernel_interrupting.Kernel', {kernel: this});

          var that = this;
          var on_success = function (data, status, xhr) {
              /**
               * get kernel info so we know what state the kernel is in
               */
              that.kernel_info();
              if (success) {
                  success(data, status, xhr);
              }
          };

          var url = utils.url_path_join(this.kernel_url, 'interrupt');
          utils.ajax(url, {
              processData: false,
              cache: false,
              type: "POST",
              contentType: false,  // there's no data with this
              dataType: "json",
              success: this._on_success(on_success),
              error: this._on_error(error)
          });
      };

      Kernel.prototype.restart = function (success, error) {
          /**
           * POST /api/kernels/[:kernel_id]/restart
           *
           * Restart the kernel.
           *
           * @function interrupt
           * @param {function} [success] - function executed on ajax success
           * @param {function} [error] - function executed on ajax error
           */
          this.events.trigger('kernel_restarting.Kernel', {kernel: this});
          this.stop_channels();
          this._msg_callbacks = {};
          this._msg_callbacks_overrides = {};
          this._display_id_to_parent_ids = {};

          var that = this;
          var on_success = function (data, status, xhr) {
              that.events.trigger('kernel_created.Kernel', {kernel: that});
              that._kernel_created(data);
              if (success) {
                  success(data, status, xhr);
              }
          };

          var on_error = function (xhr, status, err) {
              that.events.trigger('kernel_failed_restart.Kernel', {kernel: that});
              that._kernel_dead();
              if (error) {
                  error(xhr, status, err);
              }
          };

          var url = utils.url_path_join(this.kernel_url, 'restart');
          utils.ajax(url, {
              processData: false,
              cache: false,
              type: "POST",
              contentType: false,  // there's no data with this
              dataType: "json",
              success: this._on_success(on_success),
              error: this._on_error(on_error)
          });
      };

      Kernel.prototype.reconnect = function () {
          /**
           * Reconnect to a disconnected kernel. This is not actually a
           * standard HTTP request, but useful function nonetheless for
           * reconnecting to the kernel if the connection is somehow lost.
           *
           * @function reconnect
           */
          if (this.is_connected()) {
              this.stop_channels();
          }
          this._reconnect_attempt = this._reconnect_attempt + 1;
          this.events.trigger('kernel_reconnecting.Kernel', {
              kernel: this,
              attempt: this._reconnect_attempt,
          });
          this.start_channels();
      };

      Kernel.prototype._on_success = function (success) {
          /**
           * Handle a successful AJAX request by updating the kernel id and
           * name from the response, and then optionally calling a provided
           * callback.
           *
           * @function _on_success
           * @param {function} success - callback
           */
          var that = this;
          return function (data, status, xhr) {
              if (data) {
                  that.id = data.id;
                  that.name = data.name;
              }
              that.kernel_url = utils.url_path_join(that.kernel_service_url,
                  encodeURIComponent(that.id));
              if (success) {
                  success(data, status, xhr);
              }
          };
      };

      Kernel.prototype._on_error = function (error) {
          /**
           * Handle a failed AJAX request by logging the error message, and
           * then optionally calling a provided callback.
           *
           * @function _on_error
           * @param {function} error - callback
           */
          return function (xhr, status, err) {
              utils.log_ajax_error(xhr, status, err);
              if (error) {
                  error(xhr, status, err);
              }
          };
      };

      Kernel.prototype._kernel_created = function (data) {
          /**
           * Perform necessary tasks once the kernel has been started,
           * including actually connecting to the kernel.
           *
           * @function _kernel_created
           * @param {Object} data - information about the kernel including id
           */
          this.id = data.id;
          this.kernel_url = utils.url_path_join(this.kernel_service_url,
              encodeURIComponent(this.id));
          this.start_channels();
      };

      Kernel.prototype._kernel_connected = function () {
          /**
           * Perform necessary tasks once the connection to the kernel has
           * been established. This includes requesting information about
           * the kernel.
           *
           * @function _kernel_connected
           */
          this.events.trigger('kernel_connected.Kernel', {kernel: this});

          // Send pending messages. We shift the message off the queue
          // after the message is sent so that if there is an exception,
          // the message is still pending.
          while (this._pending_messages.length > 0) {
            this.ws.send(this._pending_messages[0]);
            this._pending_messages.shift();
          }

          // get kernel info so we know what state the kernel is in
          var that = this;
          this.kernel_info(function (reply) {
              that.info_reply = reply.content;
              that.events.trigger('kernel_ready.Kernel', {kernel: that});
          });
      };

      Kernel.prototype._kernel_dead = function () {
          /**
           * Perform necessary tasks after the kernel has died. This closing
           * communication channels to the kernel if they are still somehow
           * open.
           *
           * @function _kernel_dead
           */
          this.stop_channels();
      };

      Kernel.prototype.start_channels = function () {
          /**
           * Start the websocket channels.
           * Will stop and restart them if they already exist.
           *
           * @function start_channels
           */
          var that = this;
          this.stop_channels();
          var ws_host_url = this.ws_url + this.kernel_url;

          console.log("Starting WebSockets:", ws_host_url);

          this.ws = new this.WebSocket([
                  that.ws_url,
                  utils.url_path_join(that.kernel_url, 'channels'),
                  "?session_id=" + that.session_id
              ].join('')
          );
          
          var already_called_onclose = false; // only alert once
          var ws_closed_early = function(evt){
              console.log("WebSocket closed early", evt);
              if (already_called_onclose){
                  return;
              }
              already_called_onclose = true;
              // If the websocket was closed early, that could mean
              // that the kernel is actually dead. Try getting
              // information about the kernel from the API call --
              // if that fails, then assume the kernel is dead,
              // otherwise just follow the typical websocket closed
              // protocol.
              that.get_info(function () {
                  that._ws_closed(ws_host_url, false);
              }, function () {
                  that.events.trigger('kernel_dead.Kernel', {kernel: that});
                  that._kernel_dead();
              });
          };
          var ws_closed_late = function(evt){
              console.log("WebSocket closed unexpectedly", evt);
              if (already_called_onclose){
                  return;
              }
              already_called_onclose = true;
              that._ws_closed(ws_host_url, false);
          };
          var ws_error = function(evt){
              if (already_called_onclose){
                  return;
              }
              already_called_onclose = true;
              that._ws_closed(ws_host_url, true);
          };

          this.ws.onopen = $.proxy(this._ws_opened, this);
          this.ws.onclose = ws_closed_early;
          this.ws.onerror = ws_error;
          // switch from early-close to late-close message after 1s
          setTimeout(function() {
              if (that.ws !== null && !that._stopping) {
                  that.ws.onclose = ws_closed_late;
              }
          }, 1000);
          this.ws.onmessage = $.proxy(this._handle_ws_message, this);
      };

      Kernel.prototype._ws_opened = function (evt) {
          /**
           * Handle a websocket entering the open state,
           * signaling that the kernel is connected when websocket is open.
           *
           * @function _ws_opened
           */
          if (this.is_connected()) {
              // all events ready, trigger started event.
              this._kernel_connected();
          }
      };

      Kernel.prototype._ws_closed = function(ws_url, error) {
          /**
           * Handle a websocket entering the closed state.  If the websocket
           * was not closed due to an error, try to reconnect to the kernel.
           *
           * @function _ws_closed
           * @param {string} ws_url - the websocket url
           * @param {bool} error - whether the connection was closed due to an error
           */
          this.stop_channels();

          this.events.trigger('kernel_disconnected.Kernel', {kernel: this});
          if (error) {
              console.log('WebSocket connection failed: ', ws_url, error);
              this.events.trigger('kernel_connection_failed.Kernel', {
                  kernel: this,
                  ws_url: ws_url,
                  attempt: this._reconnect_attempt,
                  error: error,
              });
          }
          this._schedule_reconnect();
      };
      
      Kernel.prototype._schedule_reconnect = function () {
          /**
           * function to call when kernel connection is lost
           * schedules reconnect, or fires 'connection_dead' if reconnect limit is hit
           */
          if (this._reconnect_attempt < this.reconnect_limit) {
              var timeout = Math.pow(2, this._reconnect_attempt);
              console.log("Connection lost, reconnecting in " + timeout + " seconds.");
              setTimeout($.proxy(this.reconnect, this), 1e3 * timeout);
          } else {
              this.events.trigger('kernel_connection_dead.Kernel', {
                  kernel: this,
                  reconnect_attempt: this._reconnect_attempt,
              });
              console.log("Failed to reconnect, giving up.");
          }
      };
      
      Kernel.prototype.stop_channels = function () {
          /**
           * Close the websocket. After successful close, the value
           * in `this.ws` will be null.
           *
           * @function stop_channels
           */
          var that = this;
          var close = function () {
              that._stopping = false;
              if (that.ws && that.ws.readyState === WebSocket.CLOSED) {
                  that.ws = null;
              }
          };
          if (this.ws !== null) {
              // flag to avoid races with on_close_late
              this._stopping = true;
              if (this.ws.readyState === WebSocket.OPEN) {
                  this.ws.onclose = close;
                  this.ws.close();
              } else {
                  close();
              }
          }
      };

      Kernel.prototype.is_connected = function () {
          /**
           * Check whether there is a connection to the kernel. This
           * function only returns true if websocket has been
           * created and has a state of WebSocket.OPEN.
           *
           * @function is_connected
           * @returns {bool} - whether there is a connection
           */
          // if any channel is not ready, then we're not connected
          if (this.ws === null) {
              return false;
          }
          if (this.ws.readyState !== WebSocket.OPEN) {
              return false;
          }
          return true;
      };

      Kernel.prototype.is_fully_disconnected = function () {
          /**
           * Check whether the connection to the kernel has been completely
           * severed. This function only returns true if all channel objects
           * are null.
           *
           * @function is_fully_disconnected
           * @returns {bool} - whether the kernel is fully disconnected
           */
          return (this.ws === null);
      };
      
      Kernel.prototype._send = function(msg) {
        /**
         * Send a message (if the kernel is connected) or queue the message for future delivery
         *
         * Pending messages will automatically be sent when a kernel becomes connected.
         *
         * @function _send
         * @param msg
         */
        if (this.is_connected()) {
              this.ws.send(msg);
          } else {
              this._pending_messages.push(msg);
          }
      };
      
      Kernel.prototype.send_shell_message = function (msg_type, content, callbacks, metadata, buffers) {
          /**
           * Send a message on the Kernel's shell channel
           *
           * If the kernel is not connected, the message will be buffered.
           * 
           * @function send_shell_message
           */
          var msg = this._get_msg(msg_type, content, metadata, buffers);
          msg.channel = 'shell';
          this.set_callbacks_for_msg(msg.header.msg_id, callbacks);
          this._send(serialize.serialize(msg));
          return msg.header.msg_id;
      };

      Kernel.prototype.kernel_info = function (callback) {
          /**
           * Get kernel info
           *
           * @function kernel_info
           * @param callback {function}
           *
           * When calling this method, pass a callback function that expects one argument.
           * The callback will be passed the complete `kernel_info_reply` message documented
           * [here](https://jupyter-client.readthedocs.io/en/latest/messaging.html#kernel-info)
           */
          var callbacks;
          if (callback) {
              callbacks = { shell : { reply : callback } };
          }
          return this.send_shell_message("kernel_info_request", {}, callbacks);
      };

      Kernel.prototype.comm_info = function (target_name, callback) {
          /**
           * Get comm info
           *
           * @function comm_info
           * @param callback {function}
           *
           * When calling this method, pass a callback function that expects one argument.
           * The callback will be passed the complete `comm_info_reply` message documented
           * [here](https://jupyter-client.readthedocs.io/en/latest/messaging.html#comm_info)
           */
          var callbacks;
          if (callback) {
              callbacks = { shell : { reply : callback } };
          }
          var content = {
              target_name : target_name,
          };
          return this.send_shell_message("comm_info_request", content, callbacks);
      };

      Kernel.prototype.inspect = function (code, cursor_pos, callback) {
          /**
           * Get info on an object
           *
           * When calling this method, pass a callback function that expects one argument.
           * The callback will be passed the complete `inspect_reply` message documented
           * [here](https://jupyter-client.readthedocs.io/en/latest/messaging.html#object-information)
           *
           * @function inspect
           * @param code {string}
           * @param cursor_pos {integer}
           * @param callback {function}
           */
          var callbacks;
          if (callback) {
              callbacks = { shell : { reply : callback } };
          }
          
          var content = {
              code : code,
              cursor_pos : cursor_pos,
              detail_level : 0
          };
          return this.send_shell_message("inspect_request", content, callbacks);
      };

      Kernel.prototype.execute = function (code, callbacks, options) {
          /**
           * Execute given code into kernel, and pass result to callback.
           *
           * @async
           * @function execute
           * @param {string} code
           * @param [callbacks] {Object} With the following keys (all optional)
           *      @param callbacks.shell.reply {function}
           *      @param callbacks.shell.payload.[payload_name] {function}
           *      @param callbacks.iopub.output {function}
           *      @param callbacks.iopub.clear_output {function}
           *      @param callbacks.input {function}
           *      @param callbacks.clear_on_done=true {Boolean}
           * @param {object} [options]
           *      @param [options.silent=false] {Boolean}
           *      @param [options.user_expressions=empty_dict] {Dict}
           *      @param [options.allow_stdin=false] {Boolean} true|false
           *
           * @example
           *
           * The options object should contain the options for the execute
           * call. Its default values are:
           *
           *      options = {
           *        silent : true,
           *        user_expressions : {},
           *        allow_stdin : false
           *      }
           *
           * When calling this method pass a callbacks structure of the
           * form:
           *
           *      callbacks = {
           *       shell : {
           *         reply : execute_reply_callback,
           *         payload : {
           *           set_next_input : set_next_input_callback,
           *         }
           *       },
           *       iopub : {
           *         output : output_callback,
           *         clear_output : clear_output_callback,
           *       },
           *       input : raw_input_callback
           *      }
           *
           * Each callback will be passed the entire message as a single
           * argument.  Payload handlers will be passed the corresponding
           * payload and the execute_reply message.
           */
          var content = {
              code : code,
              silent : true,
              store_history : false,
              user_expressions : {},
              allow_stdin : false
          };
          callbacks = callbacks || {};
          if (callbacks.input !== undefined) {
              content.allow_stdin = true;
          }
          $.extend(true, content, options);
          this.events.trigger('execution_request.Kernel', {kernel: this, content: content});
          return this.send_shell_message("execute_request", content, callbacks);
      };

      /**
       * When calling this method, pass a function to be called with the
       * `complete_reply` message as its only argument when it arrives.
       *
       * `complete_reply` is documented
       * [here](https://jupyter-client.readthedocs.io/en/latest/messaging.html#complete)
       *
       * @function complete
       * @param code {string}
       * @param cursor_pos {integer}
       * @param callback {function}
       */
      Kernel.prototype.complete = function (code, cursor_pos, callback) {
          var callbacks;
          if (callback) {
              callbacks = { shell : { reply : callback } };
          }
          var content = {
              code : code,
              cursor_pos : cursor_pos
          };
          return this.send_shell_message("complete_request", content, callbacks);
      };

      /**
       * @function send_input_reply
       */
      Kernel.prototype.send_input_reply = function (input) {
          var content = {
              value : input
          };
          this.events.trigger('input_reply.Kernel', {kernel: this, content: content});
          var msg = this._get_msg("input_reply", content);
          msg.channel = 'stdin';
          this._send(serialize.serialize(msg));
          return msg.header.msg_id;
      };

      /**
       * @function register_iopub_handler
       */
      Kernel.prototype.register_iopub_handler = function (msg_type, callback) {
          this._iopub_handlers[msg_type] = callback;
      };

      /**
       * Get the iopub handler for a specific message type.
       *
       * @function get_iopub_handler
       */
      Kernel.prototype.get_iopub_handler = function (msg_type) {
          return this._iopub_handlers[msg_type];
      };

      /**
       * Get callbacks for a specific message.
       *
       * @function get_callbacks_for_msg
       */
      Kernel.prototype.get_callbacks_for_msg = function (msg_id) {
          if (msg_id == this.last_msg_id) {
              return this.last_msg_callbacks;
          } else {
              return this._msg_callbacks[msg_id];
          }
      };

      /**
       * Get output callbacks for a specific message.
       *
       * @function get_output_callbacks_for_msg
       *
       * Since output callbacks can be overridden, we first check the override stack.
       */
      Kernel.prototype.get_output_callbacks_for_msg = function (msg_id) {
          return this.get_callbacks_for_msg(this.get_output_callback_id(msg_id));
      };


      /**
       * Get the output callback id for a message
       *
       * Since output callbacks can be redirected, this may not be the same as
       * the msg_id.
       *
       * @function get_output_callback_id
       */
      Kernel.prototype.get_output_callback_id = function (msg_id) {
          var callback_id = msg_id;
          var overrides = this._msg_callbacks_overrides[msg_id];
          if (overrides && overrides.length > 0) {
              callback_id = overrides[overrides.length-1];
          }
          return callback_id
      };

      /**
       * Clear callbacks for a specific message.
       *
       * @function clear_callbacks_for_msg
       */
      Kernel.prototype.clear_callbacks_for_msg = function (msg_id) {
          if (this._msg_callbacks[msg_id] !== undefined ) {
              var callbacks = this._msg_callbacks[msg_id];
              var kernel = this;
              // clear display_id:msg_id map for display_ids associated with this msg_id
              if (!callbacks) return;
              callbacks.display_ids.map(function (display_id) {
                  var msg_ids = kernel._display_id_to_parent_ids[display_id];
                  if (msg_ids) {
                      var idx = msg_ids.indexOf(msg_id);
                      if (idx === -1) {
                          return;
                      }
                      if (msg_ids.length === 1) {
                          delete kernel._display_id_to_parent_ids[display_id];
                      } else {
                          msg_ids.splice(idx, 1);
                          kernel._display_id_to_parent_ids[display_id] = msg_ids;
                      }
                  }
              });
              delete this._msg_callbacks[msg_id];
          }
      };
      
      /**
       * @function _finish_shell
       */
      Kernel.prototype._finish_shell = function (msg_id) {
          var callbacks = this._msg_callbacks[msg_id];
          if (callbacks !== undefined) {
              callbacks.shell_done = true;
              if (callbacks.clear_on_done && callbacks.iopub_done) {
                  this.clear_callbacks_for_msg(msg_id);
              }
          }
      };

      /**
       * @function _finish_iopub
       */
      Kernel.prototype._finish_iopub = function (msg_id) {
          var callbacks = this._msg_callbacks[msg_id];
          if (callbacks !== undefined) {
              callbacks.iopub_done = true;
              if (callbacks.clear_on_done && callbacks.shell_done) {
                  this.clear_callbacks_for_msg(msg_id);
              }
          }
          this.events.trigger('finished_iopub.Kernel', {kernel: this, msg_id: msg_id});
      };
      
      /**
       * Set callbacks for a particular message.
       * Callbacks should be a struct of the following form:
       * shell : {
       * 
       * }
       *
       * If the third parameter is truthy, the callback is set as the last
       * callback registered.
       *
       * @function set_callbacks_for_msg
       */
      Kernel.prototype.set_callbacks_for_msg = function (msg_id, callbacks, save) {
          var remember = save || true;
          if (remember) {
              this.last_msg_id = msg_id;
          }
          if (callbacks) {
              // shallow-copy mapping, because we will modify it at the top level
              var cbcopy = this._msg_callbacks[msg_id] = this.last_msg_callbacks = {};
              cbcopy.shell = callbacks.shell;
              cbcopy.iopub = callbacks.iopub;
              cbcopy.input = callbacks.input;
              cbcopy.clear_on_done = callbacks.clear_on_done;
              cbcopy.shell_done = (!callbacks.shell);
              cbcopy.iopub_done = (!callbacks.iopub);
              cbcopy.display_ids = [];
              if (callbacks.clear_on_done === undefined) {
                  // default to clear-on-done
                  cbcopy.clear_on_done = true;
              }
          } else if (remember) {
              this.last_msg_callbacks = {};
          }
      };

      /**
       * Override output callbacks for a particular msg_id
       */
      Kernel.prototype.output_callback_overrides_push = function(msg_id, callback_id) {
          var output_callbacks = this._msg_callbacks_overrides[msg_id];
          if (!output_callbacks) {
              this._msg_callbacks_overrides[msg_id] = output_callbacks = [];
          }
          output_callbacks.push(callback_id);
      };

      Kernel.prototype.output_callback_overrides_pop = function(msg_id) {
          var callback_ids = this._msg_callbacks_overrides[msg_id];
          if (!callback_ids) {
              console.error("Popping callback overrides, but none registered", msg_id);
              return;
          }
          return callback_ids.pop();
      };

      Kernel.prototype._handle_ws_message = function (e) {
          var that = this;
          this._msg_queue = this._msg_queue.then(function() {
              return serialize.deserialize(e.data);
          }).then(function(msg) {return that._finish_ws_message(msg);})
          .catch(function(error) { console.error("Couldn't process kernel message", error); });
      };

      Kernel.prototype._finish_ws_message = function (msg) {
          switch (msg.channel) {
              case 'shell':
                  return this._handle_shell_reply(msg);
              case 'iopub':
                  return this._handle_iopub_message(msg);
              case 'stdin':
                  return this._handle_input_request(msg);
              default:
                  console.error("unrecognized message channel", msg.channel, msg);
          }
      };
      
      Kernel.prototype._handle_shell_reply = function (reply) {
          this.events.trigger('shell_reply.Kernel', {kernel: this, reply:reply});
          var that = this;
          var content = reply.content;
          var metadata = reply.metadata;
          var parent_id = reply.parent_header.msg_id;
          var callbacks = this.get_callbacks_for_msg(parent_id);
          var promise = Promise.resolve();
          if (!callbacks || !callbacks.shell) {
              return;
          }
          var shell_callbacks = callbacks.shell;
          
          // signal that shell callbacks are done
          this._finish_shell(parent_id);
          
          if (shell_callbacks.reply !== undefined) {
              promise = promise.then(function() {return shell_callbacks.reply(reply);});
          }
          if (content.payload && shell_callbacks.payload) {
              promise = promise.then(function() {
                  return that._handle_payloads(content.payload, shell_callbacks.payload, reply);
              });
          }
          return promise;
      };

      /**
       * @function _handle_payloads
       */
      Kernel.prototype._handle_payloads = function (payloads, payload_callbacks, msg) {
          var promise = [];
          var l = payloads.length;
          // Payloads are handled by triggering events because we don't want the Kernel
          // to depend on the Notebook or Pager classes.
          for (var i=0; i<l; i++) {
              var payload = payloads[i];
              var callback = payload_callbacks[payload.source];
              if (callback) {
                  promise.push(callback(payload, msg));
              }
          }
          return Promise.all(promise);
      };

      /**
       * @function _handle_status_message
       */
      Kernel.prototype._handle_status_message = function (msg) {
          var execution_state = msg.content.execution_state;
          var parent_id = msg.parent_header.msg_id;
          
          // dispatch status msg callbacks, if any
          var callbacks = this.get_callbacks_for_msg(parent_id);
          if (callbacks && callbacks.iopub && callbacks.iopub.status) {
              try {
                  callbacks.iopub.status(msg);
              } catch (e) {
                  console.log("Exception in status msg handler", e, e.stack);
              }
          }
          
          if (execution_state === 'busy') {
              this.events.trigger('kernel_busy.Kernel', {kernel: this});

          } else if (execution_state === 'idle') {
              // signal that iopub callbacks are (probably) done
              // async output may still arrive,
              // but only for the most recent request
              this._finish_iopub(parent_id);
              
              // trigger status_idle event
              this.events.trigger('kernel_idle.Kernel', {kernel: this});

          } else if (execution_state === 'starting') {
              this.events.trigger('kernel_starting.Kernel', {kernel: this});
              var that = this;
              this.kernel_info(function (reply) {
                  that.info_reply = reply.content;
                  that.events.trigger('kernel_ready.Kernel', {kernel: that});
              });

          } else if (execution_state === 'restarting') {
              // autorestarting is distinct from restarting,
              // in that it means the kernel died and the server is restarting it.
              // kernel_restarting sets the notification widget,
              // autorestart shows the more prominent dialog.
              this._autorestart_attempt = this._autorestart_attempt + 1;
              this.events.trigger('kernel_restarting.Kernel', {kernel: this});
              this.events.trigger('kernel_autorestarting.Kernel', {kernel: this, attempt: this._autorestart_attempt});

          } else if (execution_state === 'dead') {
              this.events.trigger('kernel_dead.Kernel', {kernel: this});
              this._kernel_dead();
          }
      };
      
      /**
       * Handle clear_output message
       *
       * @function _handle_clear_output
       */
      Kernel.prototype._handle_clear_output = function (msg) {
          var callbacks = this.get_output_callbacks_for_msg(msg.parent_header.msg_id);
          if (!callbacks || !callbacks.iopub) {
              return;
          }
          var callback = callbacks.iopub.clear_output;
          if (callback) {
              callback(msg);
          }
      };

      /**
       * handle an output message (execute_result, display_data, etc.)
       *
       * @function _handle_output_message
       */
      Kernel.prototype._handle_output_message = function (msg) {
          var that = this;
          var msg_id = msg.parent_header.msg_id;
          var callbacks = this.get_output_callbacks_for_msg(msg_id);
          if (['display_data', 'update_display_data', 'execute_result'].indexOf(msg.header.msg_type) > -1) {
              // display_data messages may re-route based on their display_id
              var display_id = (msg.content.transient || {}).display_id;
              if (display_id) {
                  // it has a display_id
                  var parent_ids = this._display_id_to_parent_ids[display_id];
                  if (parent_ids) {
                      // we've seen it before, update existing outputs with same display_id
                      // by handling display_data as update_display_data
                      var update_msg = $.extend(true, {}, msg);
                      update_msg.header.msg_type = 'update_display_data';

                      parent_ids.map(function (parent_id) {
                          var callbacks = that.get_callbacks_for_msg(parent_id);
                          if (!callbacks) return;
                          var callback = callbacks.iopub.output;
                          if (callback) {
                              callback(update_msg);
                          }
                      });
                  }
                  // we're done here if it's update_display
                  if (msg.header.msg_type === 'update_display_data') {
                      // it's an update, don't proceed to the normal display
                      return;
                  }
                  // regular display_data with id, record it for future updating
                  // in _display_id_to_parent_ids for future lookup
                  if (this._display_id_to_parent_ids[display_id] === undefined) {
                      this._display_id_to_parent_ids[display_id] = [];
                  }
                  var callback_id = this.get_output_callback_id(msg_id);
                  if (this._display_id_to_parent_ids[display_id].indexOf(callback_id) === -1) {
                      this._display_id_to_parent_ids[display_id].push(callback_id);
                  }
                  // and in callbacks for cleanup on clear_callbacks_for_msg
                  if (callbacks && callbacks.display_ids.indexOf(display_id) === -1) {
                      callbacks.display_ids.push(display_id);
                  }
              }
          }


          if (!callbacks || !callbacks.iopub) {
              // The message came from another client. Let the UI decide what to
              // do with it.
              this.events.trigger('received_unsolicited_message.Kernel', msg);
              return;
          }
          var callback = callbacks.iopub.output;
          if (callback) {
              callback(msg);
          }
      };

      /**
       * Handle an input message (execute_input).
       *
       * @function _handle_input message
       */
      Kernel.prototype._handle_input_message = function (msg) {
          var callbacks = this.get_callbacks_for_msg(msg.parent_header.msg_id);
          if (!callbacks) {
              // The message came from another client. Let the UI decide what to
              // do with it.
              this.events.trigger('received_unsolicited_message.Kernel', msg);
          }
      };

      /**
       * Dispatch IOPub messages to respective handlers. Each message
       * type should have a handler.
       *
       * @function _handle_iopub_message
       */
      Kernel.prototype._handle_iopub_message = function (msg) {
          var handler = this.get_iopub_handler(msg.header.msg_type);
          if (handler !== undefined) {
              return handler(msg);
          }
      };

      /**
       * @function _handle_input_request
       */
      Kernel.prototype._handle_input_request = function (request) {
          var header = request.header;
          var content = request.content;
          var metadata = request.metadata;
          var msg_type = header.msg_type;
          if (msg_type !== 'input_request') {
              console.log("Invalid input request!", request);
              return;
          }
          var callbacks = this.get_callbacks_for_msg(request.parent_header.msg_id);
          if (callbacks) {
              if (callbacks.input) {
                  callbacks.input(request);
              }
          }
      };

      return {'Kernel': Kernel};
  });

  // Copyright (c) Jupyter Development Team.
  // Distributed under the terms of the Modified BSD License.

  define('services/kernels/serialize', [
      'underscore',
      ], function (_) {
      
      var _deserialize_array_buffer = function (buf) {
          var data = new DataView(buf);
          // read the header: 1 + nbufs 32b integers
          var nbufs = data.getUint32(0);
          var offsets = [];
          var i;
          for (i = 1; i <= nbufs; i++) {
              offsets.push(data.getUint32(i * 4));
          }
          var json_bytes = new Uint8Array(buf.slice(offsets[0], offsets[1]));
          var msg = JSON.parse(
              (new TextDecoder('utf8')).decode(json_bytes)
          );
          // the remaining chunks are stored as DataViews in msg.buffers
          msg.buffers = [];
          var start, stop;
          for (i = 1; i < nbufs; i++) {
              start = offsets[i];
              stop = offsets[i+1] || buf.byteLength;
              msg.buffers.push(new DataView(buf.slice(start, stop)));
          }
          return msg;
      };
      
      var _deserialize_binary = function(data) {
          /**
           * deserialize the binary message format
           * callback will be called with a message whose buffers attribute
           * will be an array of DataViews.
           */
          if (data instanceof Blob) {
              // data is Blob, have to deserialize from ArrayBuffer in reader callback
              var reader = new FileReader();
              var promise = new Promise(function(resolve, reject) {
                  reader.onload = function () {
                      var msg = _deserialize_array_buffer(this.result);
                      resolve(msg);
                  };
              });
              reader.readAsArrayBuffer(data);
              return promise;
          } else {
              // data is ArrayBuffer, can deserialize directly
              var msg = _deserialize_array_buffer(data);
              return msg;
          }
      };

      var deserialize = function (data) {
          /**
           * deserialize a message and return a promise for the unpacked message
           */
          if (typeof data === "string") {
              // text JSON message
              return Promise.resolve(JSON.parse(data));
          } else {
              // binary message
              return Promise.resolve(_deserialize_binary(data));
          }
      };
      
      var _serialize_binary = function (msg) {
          /**
           * implement the binary serialization protocol
           * serializes JSON message to ArrayBuffer
           */
          msg = _.clone(msg);
          var offsets = [];
          var buffers = [];
          var i;
          for (i = 0; i < msg.buffers.length; i++) {
              // msg.buffers elements could be either views or ArrayBuffers
              // buffers elements are ArrayBuffers
              var b = msg.buffers[i];
              buffers.push(b.buffer instanceof ArrayBuffer ? b.buffer : b);
          }
          delete msg.buffers;
          var json_utf8 = (new TextEncoder('utf8')).encode(JSON.stringify(msg));
          buffers.unshift(json_utf8);
          var nbufs = buffers.length;
          offsets.push(4 * (nbufs + 1));
          for (i = 0; i + 1 < buffers.length; i++) {
              offsets.push(offsets[offsets.length-1] + buffers[i].byteLength);
          }
          var msg_buf = new Uint8Array(
              offsets[offsets.length-1] + buffers[buffers.length-1].byteLength
          );
          // use DataView.setUint32 for network byte-order
          var view = new DataView(msg_buf.buffer);
          // write nbufs to first 4 bytes
          view.setUint32(0, nbufs);
          // write offsets to next 4 * nbufs bytes
          for (i = 0; i < offsets.length; i++) {
              view.setUint32(4 * (i+1), offsets[i]);
          }
          // write all the buffers at their respective offsets
          for (i = 0; i < buffers.length; i++) {
              msg_buf.set(new Uint8Array(buffers[i]), offsets[i]);
          }
          
          // return raw ArrayBuffer
          return msg_buf.buffer;
      };
      
      var serialize = function (msg) {
          if (msg.buffers && msg.buffers.length) {
              return _serialize_binary(msg);
          } else {
              return JSON.stringify(msg);
          }
      };
      
      var exports = {
          deserialize : deserialize,
          serialize: serialize
      };
      return exports;
  });

  define("notebook/js/outputarea", [
    "base/js/keyboard",
    "base/js/events"
  ], function (keyboard_manager, events) {

    function createElement() {
      var area = $('<div/>');
      document.body.append(area[0]);
      return area[0];
    }
    function OutputArea() {
      this.keyboard_manager = keyboard_manager;
      this.events = events;
      this.outputs = [];
      this.element = createElement();
    }

    OutputArea.__mime_types = {};

    OutputArea.prototype.register_mime_type = function (mimetype, append, options) {
      OutputArea.__mime_types[mimetype] = {
        append: append,
        options: options
      };
    };

    OutputArea.prototype.create_output_subarea = function (md, classes, mime) {
      var subarea = $('<div/>').addClass('output_subarea').addClass(classes);
      return subarea;
    };

    OutputArea.prototype.create_output_area = function () {
      return $("<div/>").addClass("output_area");
    };

    OutputArea.prototype.append_mime_type = function (json, element, handle_inserted) {
      let types = Object.keys(OutputArea.__mime_types);
      types.forEach(type => {
        var append = OutputArea.__mime_types[type].append;
        if ((json.data[type] !== undefined) && append) {
          var md = json.metadata || {};
          var value = json.data[type];
          return append.apply(this, [value, md, element, handle_inserted]);
        }
      });
      return null;
    };

    OutputArea.prototype.clear_output = function () {
      this.outputs.length = 0;
      this.element.innerHTML = "";
    };

    OutputArea.prototype.fromJSON = function (outputs, metadata) {
      var len = outputs.length;
      metadata = metadata || {};
      for (var i = 0; i < len; i++) {
        this.append_output(outputs[i]);
      }
    };

    OutputArea.prototype.handle_clear_output = function (msg) {
      this.clear_output(msg.content.wait || false);
    };

    OutputArea.prototype.handle_output = function (msg) {
      var json = {};
      var msg_type = json.output_type = msg.header.msg_type;
      var content = msg.content;
      switch (msg_type) {
        case "display_data":
          json.transient = content.transient;
          json.data = content.data;
          json.metadata = content.metadata;
          break;
        default:
          console.error("unhandled output message", msg);
          return;
      }
      this.append_output(json);
    };

    OutputArea.prototype.append_output = function (json) {
      this.append_display_data(json, this.handle_appended);
      this.outputs.push(json);
      this.events.trigger('output_added.OutputArea', {
        output: json,
        output_area: this,
      });
    };

    OutputArea.prototype.append_display_data = function (json, handle_inserted) {
      var toinsert = this.create_output_area();
      this.element.append(toinsert[0]);
      this.append_mime_type(json, toinsert, handle_inserted);
    };

    const outputarea = {
      OutputArea: OutputArea
    };

    return outputarea;

  });

  define("base/js/events", function () {

    const eventTarget = document.createElement("div");

    const events = {
      on: function (type, handler) {
        eventTarget.addEventListener(type, evt => {
          handler(evt, evt.detail);
        });
      },
      trigger: function (type, data) {
        eventTarget.dispatchEvent(new CustomEvent(type, {detail: data}));
      }
    };

    return events;
  });

  define("base/js/keyboard", function () {
    function KeyboardManager() {
    }
    KeyboardManager.prototype.actions = {
      register: function () {
      }
    };

    KeyboardManager.prototype.register_events = function () {
    };

    const keyboard_manager = new KeyboardManager();

    return keyboard_manager;
  });

  define("base/js/namespace", [
    "services/kernels/kernel",
    "notebook/js/outputarea",
    "base/js/keyboard",
    "base/js/events",
  ], function (kernel, outputarea, keyboard_manager, events) {

    const cells = [{
      output_area: new outputarea.OutputArea()
    }];

    const namespace = {
      notebook: {
        kernel: new kernel.Kernel(),
        keyboard_manager: keyboard_manager,
        container: {
          on: function () {
          }
        },
        metadata: {
          widgets: {}
        },
        get_cells: function () {
          return cells;
        },
        render_cell_output: function (cell) {
          const output_area = cell.output_area;
          const outputs = [...output_area.outputs];
          output_area.clear_output();
          output_area.fromJSON(outputs);
        },
        events: events
      },
      notification_area: {
        widget: function () {
        }
      },
      menubar: {
        actions: {
          register: function () {
          }
        }
      }
    };

    return namespace;
  });

  define("base/js/utils", function () {
    return {
      get_body_data: function () {
        return "";
      },
      uuid: function () {
        return Date.now() + "_" + Math.random();
      },
      load_class: function (class_name, module_name, registry) {
        return new Promise(function (resolve, reject) {
          if (registry && registry[class_name]) {
            resolve(registry[class_name]);
          } else {
            reject(new Error('Class ' + class_name + ' not found in registry '));
          }
        });
      },
      reject: function (message, log) {
        return function (error) {
          return Promise.reject(message + "\n" + error);
        }
      },
      debounse: function debounce(fun, delay) {
        var timerId;
        return function () {
          var args = arguments;
          if (timerId) {
            clearTimeout(timerId);
          }
          timerId = setTimeout(function () {
            fun.apply(null, args);
            timerId = null;
          }, delay);
        }
      }
    };
  });

  const eventTarget = document.createElement("div");

  const INIT_REQUEST = "init";
  const SAVE_STATE_REQUEST = "save_state";
  const COMM_MSG_REQUEST = "comm";
  const WIDGET_MODEL_SET_REQUEST = "widget_model_set";

  const DISPLAY_DATA_RESPONSE = "display_data";
  const KERNEL_STATE_RESPONSE = "kernel_state";
  const COMM_MSG_RESPONSE = "comm";
  const SET_STATE_RESPONSE = "set_state";
  const WIDGET_MODEL_SET_RESPONSE = "widget_model_set";

  const id = location.hash.replace("#", "");

  window.addEventListener("message", event => {
    const data = event.data;
    const message = data.message ? JSON.parse(data.message) : undefined;
    eventTarget.dispatchEvent(new CustomEvent(data.type, {detail: message}));
  });

  const channel = {
    send: function (type, message) {
      window.top.postMessage({
        type: type,
        sender: id,
        message: message,
      }, "*");
    },
    on: function (type, handler) {
      eventTarget.addEventListener(type, evt => {
        handler(evt.detail);
      });
    },
    start: function () {
      this.send(INIT_REQUEST);
    }
  };

  function patch(kernel) {

    kernel.set_kernel_state = function (connected) {
      if (!connected) {
        this.comm_info = function (target_name, callback) {
          callback({content: {comms: {}}});
        };
      }

      this._is_connected = true;
      this._kernel_connected();
    };

    kernel.is_connected = function () {
      return this._is_connected;
    };

    kernel._send = function (msg) {
      channel.send(COMM_MSG_REQUEST, msg);
    };

    channel.on(COMM_MSG_RESPONSE, msg => {
      if (msg && msg.content && msg.content.data) {
        msg.content.data.buffer_paths = [];
      }
      kernel._finish_ws_message(msg);
    });
  }

  function patch$1(WidgetModel, widgetManager) {

    const originalSet = WidgetModel.prototype.set;

    WidgetModel.prototype.set = function (key, val, options) {

      if (key == "value") {
        channel.send(WIDGET_MODEL_SET_REQUEST, JSON.stringify({
          model_id: this.model_id,
          val: val,
          key: key
        }));
      }

      return originalSet.apply(this, arguments);
    };

    channel.on(WIDGET_MODEL_SET_RESPONSE, msg => {
      const model_id = msg.model_id;
      const model_promise = widgetManager.get_model(model_id);

      if (model_promise) {
        model_promise.then(model => {
          originalSet.call(model, msg.key, msg.val);
        });
      }
    });
  }

  var toByteArray_1 = toByteArray;
  var fromByteArray_1 = fromByteArray;

  var lookup = [];
  var revLookup = [];
  var Arr = typeof Uint8Array !== 'undefined' ? Uint8Array : Array;

  var code = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/';
  for (var i = 0, len = code.length; i < len; ++i) {
    lookup[i] = code[i];
    revLookup[code.charCodeAt(i)] = i;
  }

  // Support decoding URL-safe base64 strings, as Node.js does.
  // See: https://en.wikipedia.org/wiki/Base64#URL_applications
  revLookup['-'.charCodeAt(0)] = 62;
  revLookup['_'.charCodeAt(0)] = 63;

  function getLens (b64) {
    var len = b64.length;

    if (len % 4 > 0) {
      throw new Error('Invalid string. Length must be a multiple of 4')
    }

    // Trim off extra bytes after placeholder bytes are found
    // See: https://github.com/beatgammit/base64-js/issues/42
    var validLen = b64.indexOf('=');
    if (validLen === -1) validLen = len;

    var placeHoldersLen = validLen === len
      ? 0
      : 4 - (validLen % 4);

    return [validLen, placeHoldersLen]
  }

  function _byteLength (b64, validLen, placeHoldersLen) {
    return ((validLen + placeHoldersLen) * 3 / 4) - placeHoldersLen
  }

  function toByteArray (b64) {
    var tmp;
    var lens = getLens(b64);
    var validLen = lens[0];
    var placeHoldersLen = lens[1];

    var arr = new Arr(_byteLength(b64, validLen, placeHoldersLen));

    var curByte = 0;

    // if there are placeholders, only get up to the last complete 4 chars
    var len = placeHoldersLen > 0
      ? validLen - 4
      : validLen;

    var i;
    for (i = 0; i < len; i += 4) {
      tmp =
        (revLookup[b64.charCodeAt(i)] << 18) |
        (revLookup[b64.charCodeAt(i + 1)] << 12) |
        (revLookup[b64.charCodeAt(i + 2)] << 6) |
        revLookup[b64.charCodeAt(i + 3)];
      arr[curByte++] = (tmp >> 16) & 0xFF;
      arr[curByte++] = (tmp >> 8) & 0xFF;
      arr[curByte++] = tmp & 0xFF;
    }

    if (placeHoldersLen === 2) {
      tmp =
        (revLookup[b64.charCodeAt(i)] << 2) |
        (revLookup[b64.charCodeAt(i + 1)] >> 4);
      arr[curByte++] = tmp & 0xFF;
    }

    if (placeHoldersLen === 1) {
      tmp =
        (revLookup[b64.charCodeAt(i)] << 10) |
        (revLookup[b64.charCodeAt(i + 1)] << 4) |
        (revLookup[b64.charCodeAt(i + 2)] >> 2);
      arr[curByte++] = (tmp >> 8) & 0xFF;
      arr[curByte++] = tmp & 0xFF;
    }

    return arr
  }

  function tripletToBase64 (num) {
    return lookup[num >> 18 & 0x3F] +
      lookup[num >> 12 & 0x3F] +
      lookup[num >> 6 & 0x3F] +
      lookup[num & 0x3F]
  }

  function encodeChunk (uint8, start, end) {
    var tmp;
    var output = [];
    for (var i = start; i < end; i += 3) {
      tmp =
        ((uint8[i] << 16) & 0xFF0000) +
        ((uint8[i + 1] << 8) & 0xFF00) +
        (uint8[i + 2] & 0xFF);
      output.push(tripletToBase64(tmp));
    }
    return output.join('')
  }

  function fromByteArray (uint8) {
    var tmp;
    var len = uint8.length;
    var extraBytes = len % 3; // if we have 1 byte left, pad 2 bytes
    var parts = [];
    var maxChunkLength = 16383; // must be multiple of 3

    // go through the array every three bytes, we'll deal with trailing stuff later
    for (var i = 0, len2 = len - extraBytes; i < len2; i += maxChunkLength) {
      parts.push(encodeChunk(
        uint8, i, (i + maxChunkLength) > len2 ? len2 : (i + maxChunkLength)
      ));
    }

    // pad the end with zeros, but make sure to not forget the extra bytes
    if (extraBytes === 1) {
      tmp = uint8[len - 1];
      parts.push(
        lookup[tmp >> 2] +
        lookup[(tmp << 4) & 0x3F] +
        '=='
      );
    } else if (extraBytes === 2) {
      tmp = (uint8[len - 2] << 8) + uint8[len - 1];
      parts.push(
        lookup[tmp >> 10] +
        lookup[(tmp >> 4) & 0x3F] +
        lookup[(tmp << 2) & 0x3F] +
        '='
      );
    }

    return parts.join('')
  }

  var commonjsGlobal = typeof globalThis !== 'undefined' ? globalThis : typeof window !== 'undefined' ? window : typeof global !== 'undefined' ? global : typeof self !== 'undefined' ? self : {};

  function createCommonjsModule(fn, basedir, module) {
  	return module = {
  	  path: basedir,
  	  exports: {},
  	  require: function (path, base) {
        return commonjsRequire(path, (base === undefined || base === null) ? module.path : base);
      }
  	}, fn(module, module.exports), module.exports;
  }

  function getCjsExportFromNamespace (n) {
  	return n && n['default'] || n;
  }

  function commonjsRequire () {
  	throw new Error('Dynamic requires are not currently supported by @rollup/plugin-commonjs');
  }

  // Copyright (c) Jupyter Development Team.
  // Distributed under the terms of the Modified BSD License.
  /*-----------------------------------------------------------------------------
  | Copyright (c) 2014-2017, PhosphorJS Contributors
  |
  | Distributed under the terms of the BSD 3-Clause License.
  |
  | The full license is in the file LICENSE, distributed with this software.
  |----------------------------------------------------------------------------*/
  /**
   * The namespace for JSON-specific functions.
   */
  var JSONExt;
  (function (JSONExt) {
      /**
       * A shared frozen empty JSONObject
       */
      JSONExt.emptyObject = Object.freeze({});
      /**
       * A shared frozen empty JSONArray
       */
      JSONExt.emptyArray = Object.freeze([]);
      /**
       * Test whether a JSON value is a primitive.
       *
       * @param value - The JSON value of interest.
       *
       * @returns `true` if the value is a primitive,`false` otherwise.
       */
      function isPrimitive(value) {
          return (value === null ||
              typeof value === 'boolean' ||
              typeof value === 'number' ||
              typeof value === 'string');
      }
      JSONExt.isPrimitive = isPrimitive;
      function isArray(value) {
          return Array.isArray(value);
      }
      JSONExt.isArray = isArray;
      function isObject(value) {
          return !isPrimitive(value) && !isArray(value);
      }
      JSONExt.isObject = isObject;
      /**
       * Compare two JSON values for deep equality.
       *
       * @param first - The first JSON value of interest.
       *
       * @param second - The second JSON value of interest.
       *
       * @returns `true` if the values are equivalent, `false` otherwise.
       */
      function deepEqual(first, second) {
          // Check referential and primitive equality first.
          if (first === second) {
              return true;
          }
          // If one is a primitive, the `===` check ruled out the other.
          if (isPrimitive(first) || isPrimitive(second)) {
              return false;
          }
          // Test whether they are arrays.
          var a1 = isArray(first);
          var a2 = isArray(second);
          // Bail if the types are different.
          if (a1 !== a2) {
              return false;
          }
          // If they are both arrays, compare them.
          if (a1 && a2) {
              return deepArrayEqual(first, second);
          }
          // At this point, they must both be objects.
          return deepObjectEqual(first, second);
      }
      JSONExt.deepEqual = deepEqual;
      /**
       * Create a deep copy of a JSON value.
       *
       * @param value - The JSON value to copy.
       *
       * @returns A deep copy of the given JSON value.
       */
      function deepCopy(value) {
          // Do nothing for primitive values.
          if (isPrimitive(value)) {
              return value;
          }
          // Deep copy an array.
          if (isArray(value)) {
              return deepArrayCopy(value);
          }
          // Deep copy an object.
          return deepObjectCopy(value);
      }
      JSONExt.deepCopy = deepCopy;
      /**
       * Compare two JSON arrays for deep equality.
       */
      function deepArrayEqual(first, second) {
          // Check referential equality first.
          if (first === second) {
              return true;
          }
          // Test the arrays for equal length.
          if (first.length !== second.length) {
              return false;
          }
          // Compare the values for equality.
          for (var i = 0, n = first.length; i < n; ++i) {
              if (!deepEqual(first[i], second[i])) {
                  return false;
              }
          }
          // At this point, the arrays are equal.
          return true;
      }
      /**
       * Compare two JSON objects for deep equality.
       */
      function deepObjectEqual(first, second) {
          // Check referential equality first.
          if (first === second) {
              return true;
          }
          // Check for the first object's keys in the second object.
          for (var key in first) {
              if (first[key] !== undefined && !(key in second)) {
                  return false;
              }
          }
          // Check for the second object's keys in the first object.
          for (var key in second) {
              if (second[key] !== undefined && !(key in first)) {
                  return false;
              }
          }
          // Compare the values for equality.
          for (var key in first) {
              // Get the values.
              var firstValue = first[key];
              var secondValue = second[key];
              // If both are undefined, ignore the key.
              if (firstValue === undefined && secondValue === undefined) {
                  continue;
              }
              // If only one value is undefined, the objects are not equal.
              if (firstValue === undefined || secondValue === undefined) {
                  return false;
              }
              // Compare the values.
              if (!deepEqual(firstValue, secondValue)) {
                  return false;
              }
          }
          // At this point, the objects are equal.
          return true;
      }
      /**
       * Create a deep copy of a JSON array.
       */
      function deepArrayCopy(value) {
          var result = new Array(value.length);
          for (var i = 0, n = value.length; i < n; ++i) {
              result[i] = deepCopy(value[i]);
          }
          return result;
      }
      /**
       * Create a deep copy of a JSON object.
       */
      function deepObjectCopy(value) {
          var result = {};
          for (var key in value) {
              // Ignore undefined values.
              var subvalue = value[key];
              if (subvalue === undefined) {
                  continue;
              }
              result[key] = deepCopy(subvalue);
          }
          return result;
      }
  })(JSONExt || (JSONExt = {}));

  // Copyright (c) Jupyter Development Team.
  // Distributed under the terms of the Modified BSD License.
  /*-----------------------------------------------------------------------------
  | Copyright (c) 2014-2017, PhosphorJS Contributors
  |
  | Distributed under the terms of the BSD 3-Clause License.
  |
  | The full license is in the file LICENSE, distributed with this software.
  |----------------------------------------------------------------------------*/
  /**
   * An object which stores MIME data for general application use.
   *
   * #### Notes
   * This class does not attempt to enforce "correctness" of MIME types
   * and their associated data. Since this class is designed to transfer
   * arbitrary data and objects within the same application, it assumes
   * that the user provides correct and accurate data.
   */
  var MimeData = /** @class */ (function () {
      function MimeData() {
          this._types = [];
          this._values = [];
      }
      /**
       * Get an array of the MIME types contained within the dataset.
       *
       * @returns A new array of the MIME types, in order of insertion.
       */
      MimeData.prototype.types = function () {
          return this._types.slice();
      };
      /**
       * Test whether the dataset has an entry for the given type.
       *
       * @param mime - The MIME type of interest.
       *
       * @returns `true` if the dataset contains a value for the given
       *   MIME type, `false` otherwise.
       */
      MimeData.prototype.hasData = function (mime) {
          return this._types.indexOf(mime) !== -1;
      };
      /**
       * Get the data value for the given MIME type.
       *
       * @param mime - The MIME type of interest.
       *
       * @returns The value for the given MIME type, or `undefined` if
       *   the dataset does not contain a value for the type.
       */
      MimeData.prototype.getData = function (mime) {
          var i = this._types.indexOf(mime);
          return i !== -1 ? this._values[i] : undefined;
      };
      /**
       * Set the data value for the given MIME type.
       *
       * @param mime - The MIME type of interest.
       *
       * @param data - The data value for the given MIME type.
       *
       * #### Notes
       * This will overwrite any previous entry for the MIME type.
       */
      MimeData.prototype.setData = function (mime, data) {
          this.clearData(mime);
          this._types.push(mime);
          this._values.push(data);
      };
      /**
       * Remove the data entry for the given MIME type.
       *
       * @param mime - The MIME type of interest.
       *
       * #### Notes
       * This is a no-op if there is no entry for the given MIME type.
       */
      MimeData.prototype.clearData = function (mime) {
          var i = this._types.indexOf(mime);
          if (i !== -1) {
              this._types.splice(i, 1);
              this._values.splice(i, 1);
          }
      };
      /**
       * Remove all data entries from the dataset.
       */
      MimeData.prototype.clear = function () {
          this._types.length = 0;
          this._values.length = 0;
      };
      return MimeData;
  }());

  // Copyright (c) Jupyter Development Team.
  // Distributed under the terms of the Modified BSD License.
  /*-----------------------------------------------------------------------------
  | Copyright (c) 2014-2017, PhosphorJS Contributors
  |
  | Distributed under the terms of the BSD 3-Clause License.
  |
  | The full license is in the file LICENSE, distributed with this software.
  |----------------------------------------------------------------------------*/
  /**
   * The namespace for random number related functionality.
   */
  var Random;
  (function (Random) {
      /**
       * A function which generates random bytes.
       *
       * @param buffer - The `Uint8Array` to fill with random bytes.
       *
       * #### Notes
       * A cryptographically strong random number generator will be used if
       * available. Otherwise, `Math.random` will be used as a fallback for
       * randomness.
       *
       * The following RNGs are supported, listed in order of precedence:
       *   - `window.crypto.getRandomValues`
       *   - `window.msCrypto.getRandomValues`
       *   - `require('crypto').randomFillSync
       *   - `require('crypto').randomBytes
       *   - `Math.random`
       */
      Random.getRandomValues = (function () {
          // Look up the crypto module if available.
          var crypto = ((typeof window !== 'undefined' && (window.crypto || window.msCrypto)) ||
              (typeof commonjsRequire !== 'undefined' && require('crypto')) || null);
          // Modern browsers and IE 11
          if (crypto && typeof crypto.getRandomValues === 'function') {
              return function getRandomValues(buffer) {
                  return crypto.getRandomValues(buffer);
              };
          }
          // Node 7+
          if (crypto && typeof crypto.randomFillSync === 'function') {
              return function getRandomValues(buffer) {
                  return crypto.randomFillSync(buffer);
              };
          }
          // Node 0.10+
          if (crypto && typeof crypto.randomBytes === 'function') {
              return function getRandomValues(buffer) {
                  var bytes = crypto.randomBytes(buffer.length);
                  for (var i = 0, n = bytes.length; i < n; ++i) {
                      buffer[i] = bytes[i];
                  }
              };
          }
          // Fallback
          return function getRandomValues(buffer) {
              var value = 0;
              for (var i = 0, n = buffer.length; i < n; ++i) {
                  if (i % 4 === 0) {
                      value = Math.random() * 0xFFFFFFFF >>> 0;
                  }
                  buffer[i] = value & 0xFF;
                  value >>>= 8;
              }
          };
      })();
  })(Random || (Random = {}));

  // Copyright (c) Jupyter Development Team.
  // Distributed under the terms of the Modified BSD License.
  /*-----------------------------------------------------------------------------
  | Copyright (c) 2014-2017, PhosphorJS Contributors
  |
  | Distributed under the terms of the BSD 3-Clause License.
  |
  | The full license is in the file LICENSE, distributed with this software.
  |----------------------------------------------------------------------------*/
  /**
   * A runtime object which captures compile-time type information.
   *
   * #### Notes
   * A token captures the compile-time type of an interface or class in
   * an object which can be used at runtime in a type-safe fashion.
   */
  var Token = /** @class */ (function () {
      /**
       * Construct a new token.
       *
       * @param name - A human readable name for the token.
       */
      function Token(name) {
          this.name = name;
          this._tokenStructuralPropertyT = null;
      }
      return Token;
  }());

  // Copyright (c) Jupyter Development Team.
  /**
   * The namespace for UUID related functionality.
   */
  var UUID;
  (function (UUID) {
      /**
       * A function which generates UUID v4 identifiers.
       *
       * @returns A new UUID v4 string.
       *
       * #### Notes
       * This implementation complies with RFC 4122.
       *
       * This uses `Random.getRandomValues()` for random bytes, which in
       * turn will use the underlying `crypto` module of the platform if
       * it is available. The fallback for randomness is `Math.random`.
       */
      UUID.uuid4 = (function () {
          // Create a 16 byte array to hold the random values.
          var bytes = new Uint8Array(16);
          // Create a look up table from bytes to hex strings.
          var lut = new Array(256);
          // Pad the single character hex digits with a leading zero.
          for (var i = 0; i < 16; ++i) {
              lut[i] = '0' + i.toString(16);
          }
          // Populate the rest of the hex digits.
          for (var i = 16; i < 256; ++i) {
              lut[i] = i.toString(16);
          }
          // Return a function which generates the UUID.
          return function uuid4() {
              // Get a new batch of random values.
              Random.getRandomValues(bytes);
              // Set the UUID version number to 4.
              bytes[6] = 0x40 | (bytes[6] & 0x0F);
              // Set the clock sequence bit to the RFC spec.
              bytes[8] = 0x80 | (bytes[8] & 0x3F);
              // Assemble the UUID string.
              return (lut[bytes[0]] +
                  lut[bytes[1]] +
                  lut[bytes[2]] +
                  lut[bytes[3]] +
                  '-' +
                  lut[bytes[4]] +
                  lut[bytes[5]] +
                  '-' +
                  lut[bytes[6]] +
                  lut[bytes[7]] +
                  '-' +
                  lut[bytes[8]] +
                  lut[bytes[9]] +
                  '-' +
                  lut[bytes[10]] +
                  lut[bytes[11]] +
                  lut[bytes[12]] +
                  lut[bytes[13]] +
                  lut[bytes[14]] +
                  lut[bytes[15]]);
          };
      })();
  })(UUID || (UUID = {}));

  /**
   * Removes all key-value entries from the list cache.
   *
   * @private
   * @name clear
   * @memberOf ListCache
   */
  function listCacheClear() {
    this.__data__ = [];
    this.size = 0;
  }

  var _listCacheClear = listCacheClear;

  /**
   * Performs a
   * [`SameValueZero`](http://ecma-international.org/ecma-262/7.0/#sec-samevaluezero)
   * comparison between two values to determine if they are equivalent.
   *
   * @static
   * @memberOf _
   * @since 4.0.0
   * @category Lang
   * @param {*} value The value to compare.
   * @param {*} other The other value to compare.
   * @returns {boolean} Returns `true` if the values are equivalent, else `false`.
   * @example
   *
   * var object = { 'a': 1 };
   * var other = { 'a': 1 };
   *
   * _.eq(object, object);
   * // => true
   *
   * _.eq(object, other);
   * // => false
   *
   * _.eq('a', 'a');
   * // => true
   *
   * _.eq('a', Object('a'));
   * // => false
   *
   * _.eq(NaN, NaN);
   * // => true
   */
  function eq(value, other) {
    return value === other || (value !== value && other !== other);
  }

  var eq_1 = eq;

  /**
   * Gets the index at which the `key` is found in `array` of key-value pairs.
   *
   * @private
   * @param {Array} array The array to inspect.
   * @param {*} key The key to search for.
   * @returns {number} Returns the index of the matched value, else `-1`.
   */
  function assocIndexOf(array, key) {
    var length = array.length;
    while (length--) {
      if (eq_1(array[length][0], key)) {
        return length;
      }
    }
    return -1;
  }

  var _assocIndexOf = assocIndexOf;

  /** Used for built-in method references. */
  var arrayProto = Array.prototype;

  /** Built-in value references. */
  var splice = arrayProto.splice;

  /**
   * Removes `key` and its value from the list cache.
   *
   * @private
   * @name delete
   * @memberOf ListCache
   * @param {string} key The key of the value to remove.
   * @returns {boolean} Returns `true` if the entry was removed, else `false`.
   */
  function listCacheDelete(key) {
    var data = this.__data__,
        index = _assocIndexOf(data, key);

    if (index < 0) {
      return false;
    }
    var lastIndex = data.length - 1;
    if (index == lastIndex) {
      data.pop();
    } else {
      splice.call(data, index, 1);
    }
    --this.size;
    return true;
  }

  var _listCacheDelete = listCacheDelete;

  /**
   * Gets the list cache value for `key`.
   *
   * @private
   * @name get
   * @memberOf ListCache
   * @param {string} key The key of the value to get.
   * @returns {*} Returns the entry value.
   */
  function listCacheGet(key) {
    var data = this.__data__,
        index = _assocIndexOf(data, key);

    return index < 0 ? undefined : data[index][1];
  }

  var _listCacheGet = listCacheGet;

  /**
   * Checks if a list cache value for `key` exists.
   *
   * @private
   * @name has
   * @memberOf ListCache
   * @param {string} key The key of the entry to check.
   * @returns {boolean} Returns `true` if an entry for `key` exists, else `false`.
   */
  function listCacheHas(key) {
    return _assocIndexOf(this.__data__, key) > -1;
  }

  var _listCacheHas = listCacheHas;

  /**
   * Sets the list cache `key` to `value`.
   *
   * @private
   * @name set
   * @memberOf ListCache
   * @param {string} key The key of the value to set.
   * @param {*} value The value to set.
   * @returns {Object} Returns the list cache instance.
   */
  function listCacheSet(key, value) {
    var data = this.__data__,
        index = _assocIndexOf(data, key);

    if (index < 0) {
      ++this.size;
      data.push([key, value]);
    } else {
      data[index][1] = value;
    }
    return this;
  }

  var _listCacheSet = listCacheSet;

  /**
   * Creates an list cache object.
   *
   * @private
   * @constructor
   * @param {Array} [entries] The key-value pairs to cache.
   */
  function ListCache(entries) {
    var index = -1,
        length = entries == null ? 0 : entries.length;

    this.clear();
    while (++index < length) {
      var entry = entries[index];
      this.set(entry[0], entry[1]);
    }
  }

  // Add methods to `ListCache`.
  ListCache.prototype.clear = _listCacheClear;
  ListCache.prototype['delete'] = _listCacheDelete;
  ListCache.prototype.get = _listCacheGet;
  ListCache.prototype.has = _listCacheHas;
  ListCache.prototype.set = _listCacheSet;

  var _ListCache = ListCache;

  /**
   * Removes all key-value entries from the stack.
   *
   * @private
   * @name clear
   * @memberOf Stack
   */
  function stackClear() {
    this.__data__ = new _ListCache;
    this.size = 0;
  }

  var _stackClear = stackClear;

  /**
   * Removes `key` and its value from the stack.
   *
   * @private
   * @name delete
   * @memberOf Stack
   * @param {string} key The key of the value to remove.
   * @returns {boolean} Returns `true` if the entry was removed, else `false`.
   */
  function stackDelete(key) {
    var data = this.__data__,
        result = data['delete'](key);

    this.size = data.size;
    return result;
  }

  var _stackDelete = stackDelete;

  /**
   * Gets the stack value for `key`.
   *
   * @private
   * @name get
   * @memberOf Stack
   * @param {string} key The key of the value to get.
   * @returns {*} Returns the entry value.
   */
  function stackGet(key) {
    return this.__data__.get(key);
  }

  var _stackGet = stackGet;

  /**
   * Checks if a stack value for `key` exists.
   *
   * @private
   * @name has
   * @memberOf Stack
   * @param {string} key The key of the entry to check.
   * @returns {boolean} Returns `true` if an entry for `key` exists, else `false`.
   */
  function stackHas(key) {
    return this.__data__.has(key);
  }

  var _stackHas = stackHas;

  /** Detect free variable `global` from Node.js. */
  var freeGlobal = typeof commonjsGlobal == 'object' && commonjsGlobal && commonjsGlobal.Object === Object && commonjsGlobal;

  var _freeGlobal = freeGlobal;

  /** Detect free variable `self`. */
  var freeSelf = typeof self == 'object' && self && self.Object === Object && self;

  /** Used as a reference to the global object. */
  var root = _freeGlobal || freeSelf || Function('return this')();

  var _root = root;

  /** Built-in value references. */
  var Symbol$1 = _root.Symbol;

  var _Symbol = Symbol$1;

  /** Used for built-in method references. */
  var objectProto = Object.prototype;

  /** Used to check objects for own properties. */
  var hasOwnProperty = objectProto.hasOwnProperty;

  /**
   * Used to resolve the
   * [`toStringTag`](http://ecma-international.org/ecma-262/7.0/#sec-object.prototype.tostring)
   * of values.
   */
  var nativeObjectToString = objectProto.toString;

  /** Built-in value references. */
  var symToStringTag = _Symbol ? _Symbol.toStringTag : undefined;

  /**
   * A specialized version of `baseGetTag` which ignores `Symbol.toStringTag` values.
   *
   * @private
   * @param {*} value The value to query.
   * @returns {string} Returns the raw `toStringTag`.
   */
  function getRawTag(value) {
    var isOwn = hasOwnProperty.call(value, symToStringTag),
        tag = value[symToStringTag];

    try {
      value[symToStringTag] = undefined;
      var unmasked = true;
    } catch (e) {}

    var result = nativeObjectToString.call(value);
    if (unmasked) {
      if (isOwn) {
        value[symToStringTag] = tag;
      } else {
        delete value[symToStringTag];
      }
    }
    return result;
  }

  var _getRawTag = getRawTag;

  /** Used for built-in method references. */
  var objectProto$1 = Object.prototype;

  /**
   * Used to resolve the
   * [`toStringTag`](http://ecma-international.org/ecma-262/7.0/#sec-object.prototype.tostring)
   * of values.
   */
  var nativeObjectToString$1 = objectProto$1.toString;

  /**
   * Converts `value` to a string using `Object.prototype.toString`.
   *
   * @private
   * @param {*} value The value to convert.
   * @returns {string} Returns the converted string.
   */
  function objectToString(value) {
    return nativeObjectToString$1.call(value);
  }

  var _objectToString = objectToString;

  /** `Object#toString` result references. */
  var nullTag = '[object Null]',
      undefinedTag = '[object Undefined]';

  /** Built-in value references. */
  var symToStringTag$1 = _Symbol ? _Symbol.toStringTag : undefined;

  /**
   * The base implementation of `getTag` without fallbacks for buggy environments.
   *
   * @private
   * @param {*} value The value to query.
   * @returns {string} Returns the `toStringTag`.
   */
  function baseGetTag(value) {
    if (value == null) {
      return value === undefined ? undefinedTag : nullTag;
    }
    return (symToStringTag$1 && symToStringTag$1 in Object(value))
      ? _getRawTag(value)
      : _objectToString(value);
  }

  var _baseGetTag = baseGetTag;

  /**
   * Checks if `value` is the
   * [language type](http://www.ecma-international.org/ecma-262/7.0/#sec-ecmascript-language-types)
   * of `Object`. (e.g. arrays, functions, objects, regexes, `new Number(0)`, and `new String('')`)
   *
   * @static
   * @memberOf _
   * @since 0.1.0
   * @category Lang
   * @param {*} value The value to check.
   * @returns {boolean} Returns `true` if `value` is an object, else `false`.
   * @example
   *
   * _.isObject({});
   * // => true
   *
   * _.isObject([1, 2, 3]);
   * // => true
   *
   * _.isObject(_.noop);
   * // => true
   *
   * _.isObject(null);
   * // => false
   */
  function isObject(value) {
    var type = typeof value;
    return value != null && (type == 'object' || type == 'function');
  }

  var isObject_1 = isObject;

  /** `Object#toString` result references. */
  var asyncTag = '[object AsyncFunction]',
      funcTag = '[object Function]',
      genTag = '[object GeneratorFunction]',
      proxyTag = '[object Proxy]';

  /**
   * Checks if `value` is classified as a `Function` object.
   *
   * @static
   * @memberOf _
   * @since 0.1.0
   * @category Lang
   * @param {*} value The value to check.
   * @returns {boolean} Returns `true` if `value` is a function, else `false`.
   * @example
   *
   * _.isFunction(_);
   * // => true
   *
   * _.isFunction(/abc/);
   * // => false
   */
  function isFunction(value) {
    if (!isObject_1(value)) {
      return false;
    }
    // The use of `Object#toString` avoids issues with the `typeof` operator
    // in Safari 9 which returns 'object' for typed arrays and other constructors.
    var tag = _baseGetTag(value);
    return tag == funcTag || tag == genTag || tag == asyncTag || tag == proxyTag;
  }

  var isFunction_1 = isFunction;

  /** Used to detect overreaching core-js shims. */
  var coreJsData = _root['__core-js_shared__'];

  var _coreJsData = coreJsData;

  /** Used to detect methods masquerading as native. */
  var maskSrcKey = (function() {
    var uid = /[^.]+$/.exec(_coreJsData && _coreJsData.keys && _coreJsData.keys.IE_PROTO || '');
    return uid ? ('Symbol(src)_1.' + uid) : '';
  }());

  /**
   * Checks if `func` has its source masked.
   *
   * @private
   * @param {Function} func The function to check.
   * @returns {boolean} Returns `true` if `func` is masked, else `false`.
   */
  function isMasked(func) {
    return !!maskSrcKey && (maskSrcKey in func);
  }

  var _isMasked = isMasked;

  /** Used for built-in method references. */
  var funcProto = Function.prototype;

  /** Used to resolve the decompiled source of functions. */
  var funcToString = funcProto.toString;

  /**
   * Converts `func` to its source code.
   *
   * @private
   * @param {Function} func The function to convert.
   * @returns {string} Returns the source code.
   */
  function toSource(func) {
    if (func != null) {
      try {
        return funcToString.call(func);
      } catch (e) {}
      try {
        return (func + '');
      } catch (e) {}
    }
    return '';
  }

  var _toSource = toSource;

  /**
   * Used to match `RegExp`
   * [syntax characters](http://ecma-international.org/ecma-262/7.0/#sec-patterns).
   */
  var reRegExpChar = /[\\^$.*+?()[\]{}|]/g;

  /** Used to detect host constructors (Safari). */
  var reIsHostCtor = /^\[object .+?Constructor\]$/;

  /** Used for built-in method references. */
  var funcProto$1 = Function.prototype,
      objectProto$2 = Object.prototype;

  /** Used to resolve the decompiled source of functions. */
  var funcToString$1 = funcProto$1.toString;

  /** Used to check objects for own properties. */
  var hasOwnProperty$1 = objectProto$2.hasOwnProperty;

  /** Used to detect if a method is native. */
  var reIsNative = RegExp('^' +
    funcToString$1.call(hasOwnProperty$1).replace(reRegExpChar, '\\$&')
    .replace(/hasOwnProperty|(function).*?(?=\\\()| for .+?(?=\\\])/g, '$1.*?') + '$'
  );

  /**
   * The base implementation of `_.isNative` without bad shim checks.
   *
   * @private
   * @param {*} value The value to check.
   * @returns {boolean} Returns `true` if `value` is a native function,
   *  else `false`.
   */
  function baseIsNative(value) {
    if (!isObject_1(value) || _isMasked(value)) {
      return false;
    }
    var pattern = isFunction_1(value) ? reIsNative : reIsHostCtor;
    return pattern.test(_toSource(value));
  }

  var _baseIsNative = baseIsNative;

  /**
   * Gets the value at `key` of `object`.
   *
   * @private
   * @param {Object} [object] The object to query.
   * @param {string} key The key of the property to get.
   * @returns {*} Returns the property value.
   */
  function getValue(object, key) {
    return object == null ? undefined : object[key];
  }

  var _getValue = getValue;

  /**
   * Gets the native function at `key` of `object`.
   *
   * @private
   * @param {Object} object The object to query.
   * @param {string} key The key of the method to get.
   * @returns {*} Returns the function if it's native, else `undefined`.
   */
  function getNative(object, key) {
    var value = _getValue(object, key);
    return _baseIsNative(value) ? value : undefined;
  }

  var _getNative = getNative;

  /* Built-in method references that are verified to be native. */
  var Map$1 = _getNative(_root, 'Map');

  var _Map = Map$1;

  /* Built-in method references that are verified to be native. */
  var nativeCreate = _getNative(Object, 'create');

  var _nativeCreate = nativeCreate;

  /**
   * Removes all key-value entries from the hash.
   *
   * @private
   * @name clear
   * @memberOf Hash
   */
  function hashClear() {
    this.__data__ = _nativeCreate ? _nativeCreate(null) : {};
    this.size = 0;
  }

  var _hashClear = hashClear;

  /**
   * Removes `key` and its value from the hash.
   *
   * @private
   * @name delete
   * @memberOf Hash
   * @param {Object} hash The hash to modify.
   * @param {string} key The key of the value to remove.
   * @returns {boolean} Returns `true` if the entry was removed, else `false`.
   */
  function hashDelete(key) {
    var result = this.has(key) && delete this.__data__[key];
    this.size -= result ? 1 : 0;
    return result;
  }

  var _hashDelete = hashDelete;

  /** Used to stand-in for `undefined` hash values. */
  var HASH_UNDEFINED = '__lodash_hash_undefined__';

  /** Used for built-in method references. */
  var objectProto$3 = Object.prototype;

  /** Used to check objects for own properties. */
  var hasOwnProperty$2 = objectProto$3.hasOwnProperty;

  /**
   * Gets the hash value for `key`.
   *
   * @private
   * @name get
   * @memberOf Hash
   * @param {string} key The key of the value to get.
   * @returns {*} Returns the entry value.
   */
  function hashGet(key) {
    var data = this.__data__;
    if (_nativeCreate) {
      var result = data[key];
      return result === HASH_UNDEFINED ? undefined : result;
    }
    return hasOwnProperty$2.call(data, key) ? data[key] : undefined;
  }

  var _hashGet = hashGet;

  /** Used for built-in method references. */
  var objectProto$4 = Object.prototype;

  /** Used to check objects for own properties. */
  var hasOwnProperty$3 = objectProto$4.hasOwnProperty;

  /**
   * Checks if a hash value for `key` exists.
   *
   * @private
   * @name has
   * @memberOf Hash
   * @param {string} key The key of the entry to check.
   * @returns {boolean} Returns `true` if an entry for `key` exists, else `false`.
   */
  function hashHas(key) {
    var data = this.__data__;
    return _nativeCreate ? (data[key] !== undefined) : hasOwnProperty$3.call(data, key);
  }

  var _hashHas = hashHas;

  /** Used to stand-in for `undefined` hash values. */
  var HASH_UNDEFINED$1 = '__lodash_hash_undefined__';

  /**
   * Sets the hash `key` to `value`.
   *
   * @private
   * @name set
   * @memberOf Hash
   * @param {string} key The key of the value to set.
   * @param {*} value The value to set.
   * @returns {Object} Returns the hash instance.
   */
  function hashSet(key, value) {
    var data = this.__data__;
    this.size += this.has(key) ? 0 : 1;
    data[key] = (_nativeCreate && value === undefined) ? HASH_UNDEFINED$1 : value;
    return this;
  }

  var _hashSet = hashSet;

  /**
   * Creates a hash object.
   *
   * @private
   * @constructor
   * @param {Array} [entries] The key-value pairs to cache.
   */
  function Hash(entries) {
    var index = -1,
        length = entries == null ? 0 : entries.length;

    this.clear();
    while (++index < length) {
      var entry = entries[index];
      this.set(entry[0], entry[1]);
    }
  }

  // Add methods to `Hash`.
  Hash.prototype.clear = _hashClear;
  Hash.prototype['delete'] = _hashDelete;
  Hash.prototype.get = _hashGet;
  Hash.prototype.has = _hashHas;
  Hash.prototype.set = _hashSet;

  var _Hash = Hash;

  /**
   * Removes all key-value entries from the map.
   *
   * @private
   * @name clear
   * @memberOf MapCache
   */
  function mapCacheClear() {
    this.size = 0;
    this.__data__ = {
      'hash': new _Hash,
      'map': new (_Map || _ListCache),
      'string': new _Hash
    };
  }

  var _mapCacheClear = mapCacheClear;

  /**
   * Checks if `value` is suitable for use as unique object key.
   *
   * @private
   * @param {*} value The value to check.
   * @returns {boolean} Returns `true` if `value` is suitable, else `false`.
   */
  function isKeyable(value) {
    var type = typeof value;
    return (type == 'string' || type == 'number' || type == 'symbol' || type == 'boolean')
      ? (value !== '__proto__')
      : (value === null);
  }

  var _isKeyable = isKeyable;

  /**
   * Gets the data for `map`.
   *
   * @private
   * @param {Object} map The map to query.
   * @param {string} key The reference key.
   * @returns {*} Returns the map data.
   */
  function getMapData(map, key) {
    var data = map.__data__;
    return _isKeyable(key)
      ? data[typeof key == 'string' ? 'string' : 'hash']
      : data.map;
  }

  var _getMapData = getMapData;

  /**
   * Removes `key` and its value from the map.
   *
   * @private
   * @name delete
   * @memberOf MapCache
   * @param {string} key The key of the value to remove.
   * @returns {boolean} Returns `true` if the entry was removed, else `false`.
   */
  function mapCacheDelete(key) {
    var result = _getMapData(this, key)['delete'](key);
    this.size -= result ? 1 : 0;
    return result;
  }

  var _mapCacheDelete = mapCacheDelete;

  /**
   * Gets the map value for `key`.
   *
   * @private
   * @name get
   * @memberOf MapCache
   * @param {string} key The key of the value to get.
   * @returns {*} Returns the entry value.
   */
  function mapCacheGet(key) {
    return _getMapData(this, key).get(key);
  }

  var _mapCacheGet = mapCacheGet;

  /**
   * Checks if a map value for `key` exists.
   *
   * @private
   * @name has
   * @memberOf MapCache
   * @param {string} key The key of the entry to check.
   * @returns {boolean} Returns `true` if an entry for `key` exists, else `false`.
   */
  function mapCacheHas(key) {
    return _getMapData(this, key).has(key);
  }

  var _mapCacheHas = mapCacheHas;

  /**
   * Sets the map `key` to `value`.
   *
   * @private
   * @name set
   * @memberOf MapCache
   * @param {string} key The key of the value to set.
   * @param {*} value The value to set.
   * @returns {Object} Returns the map cache instance.
   */
  function mapCacheSet(key, value) {
    var data = _getMapData(this, key),
        size = data.size;

    data.set(key, value);
    this.size += data.size == size ? 0 : 1;
    return this;
  }

  var _mapCacheSet = mapCacheSet;

  /**
   * Creates a map cache object to store key-value pairs.
   *
   * @private
   * @constructor
   * @param {Array} [entries] The key-value pairs to cache.
   */
  function MapCache(entries) {
    var index = -1,
        length = entries == null ? 0 : entries.length;

    this.clear();
    while (++index < length) {
      var entry = entries[index];
      this.set(entry[0], entry[1]);
    }
  }

  // Add methods to `MapCache`.
  MapCache.prototype.clear = _mapCacheClear;
  MapCache.prototype['delete'] = _mapCacheDelete;
  MapCache.prototype.get = _mapCacheGet;
  MapCache.prototype.has = _mapCacheHas;
  MapCache.prototype.set = _mapCacheSet;

  var _MapCache = MapCache;

  /** Used as the size to enable large array optimizations. */
  var LARGE_ARRAY_SIZE = 200;

  /**
   * Sets the stack `key` to `value`.
   *
   * @private
   * @name set
   * @memberOf Stack
   * @param {string} key The key of the value to set.
   * @param {*} value The value to set.
   * @returns {Object} Returns the stack cache instance.
   */
  function stackSet(key, value) {
    var data = this.__data__;
    if (data instanceof _ListCache) {
      var pairs = data.__data__;
      if (!_Map || (pairs.length < LARGE_ARRAY_SIZE - 1)) {
        pairs.push([key, value]);
        this.size = ++data.size;
        return this;
      }
      data = this.__data__ = new _MapCache(pairs);
    }
    data.set(key, value);
    this.size = data.size;
    return this;
  }

  var _stackSet = stackSet;

  /**
   * Creates a stack cache object to store key-value pairs.
   *
   * @private
   * @constructor
   * @param {Array} [entries] The key-value pairs to cache.
   */
  function Stack(entries) {
    var data = this.__data__ = new _ListCache(entries);
    this.size = data.size;
  }

  // Add methods to `Stack`.
  Stack.prototype.clear = _stackClear;
  Stack.prototype['delete'] = _stackDelete;
  Stack.prototype.get = _stackGet;
  Stack.prototype.has = _stackHas;
  Stack.prototype.set = _stackSet;

  var _Stack = Stack;

  /** Used to stand-in for `undefined` hash values. */
  var HASH_UNDEFINED$2 = '__lodash_hash_undefined__';

  /**
   * Adds `value` to the array cache.
   *
   * @private
   * @name add
   * @memberOf SetCache
   * @alias push
   * @param {*} value The value to cache.
   * @returns {Object} Returns the cache instance.
   */
  function setCacheAdd(value) {
    this.__data__.set(value, HASH_UNDEFINED$2);
    return this;
  }

  var _setCacheAdd = setCacheAdd;

  /**
   * Checks if `value` is in the array cache.
   *
   * @private
   * @name has
   * @memberOf SetCache
   * @param {*} value The value to search for.
   * @returns {number} Returns `true` if `value` is found, else `false`.
   */
  function setCacheHas(value) {
    return this.__data__.has(value);
  }

  var _setCacheHas = setCacheHas;

  /**
   *
   * Creates an array cache object to store unique values.
   *
   * @private
   * @constructor
   * @param {Array} [values] The values to cache.
   */
  function SetCache(values) {
    var index = -1,
        length = values == null ? 0 : values.length;

    this.__data__ = new _MapCache;
    while (++index < length) {
      this.add(values[index]);
    }
  }

  // Add methods to `SetCache`.
  SetCache.prototype.add = SetCache.prototype.push = _setCacheAdd;
  SetCache.prototype.has = _setCacheHas;

  var _SetCache = SetCache;

  /**
   * A specialized version of `_.some` for arrays without support for iteratee
   * shorthands.
   *
   * @private
   * @param {Array} [array] The array to iterate over.
   * @param {Function} predicate The function invoked per iteration.
   * @returns {boolean} Returns `true` if any element passes the predicate check,
   *  else `false`.
   */
  function arraySome(array, predicate) {
    var index = -1,
        length = array == null ? 0 : array.length;

    while (++index < length) {
      if (predicate(array[index], index, array)) {
        return true;
      }
    }
    return false;
  }

  var _arraySome = arraySome;

  /**
   * Checks if a `cache` value for `key` exists.
   *
   * @private
   * @param {Object} cache The cache to query.
   * @param {string} key The key of the entry to check.
   * @returns {boolean} Returns `true` if an entry for `key` exists, else `false`.
   */
  function cacheHas(cache, key) {
    return cache.has(key);
  }

  var _cacheHas = cacheHas;

  /** Used to compose bitmasks for value comparisons. */
  var COMPARE_PARTIAL_FLAG = 1,
      COMPARE_UNORDERED_FLAG = 2;

  /**
   * A specialized version of `baseIsEqualDeep` for arrays with support for
   * partial deep comparisons.
   *
   * @private
   * @param {Array} array The array to compare.
   * @param {Array} other The other array to compare.
   * @param {number} bitmask The bitmask flags. See `baseIsEqual` for more details.
   * @param {Function} customizer The function to customize comparisons.
   * @param {Function} equalFunc The function to determine equivalents of values.
   * @param {Object} stack Tracks traversed `array` and `other` objects.
   * @returns {boolean} Returns `true` if the arrays are equivalent, else `false`.
   */
  function equalArrays(array, other, bitmask, customizer, equalFunc, stack) {
    var isPartial = bitmask & COMPARE_PARTIAL_FLAG,
        arrLength = array.length,
        othLength = other.length;

    if (arrLength != othLength && !(isPartial && othLength > arrLength)) {
      return false;
    }
    // Assume cyclic values are equal.
    var stacked = stack.get(array);
    if (stacked && stack.get(other)) {
      return stacked == other;
    }
    var index = -1,
        result = true,
        seen = (bitmask & COMPARE_UNORDERED_FLAG) ? new _SetCache : undefined;

    stack.set(array, other);
    stack.set(other, array);

    // Ignore non-index properties.
    while (++index < arrLength) {
      var arrValue = array[index],
          othValue = other[index];

      if (customizer) {
        var compared = isPartial
          ? customizer(othValue, arrValue, index, other, array, stack)
          : customizer(arrValue, othValue, index, array, other, stack);
      }
      if (compared !== undefined) {
        if (compared) {
          continue;
        }
        result = false;
        break;
      }
      // Recursively compare arrays (susceptible to call stack limits).
      if (seen) {
        if (!_arraySome(other, function(othValue, othIndex) {
              if (!_cacheHas(seen, othIndex) &&
                  (arrValue === othValue || equalFunc(arrValue, othValue, bitmask, customizer, stack))) {
                return seen.push(othIndex);
              }
            })) {
          result = false;
          break;
        }
      } else if (!(
            arrValue === othValue ||
              equalFunc(arrValue, othValue, bitmask, customizer, stack)
          )) {
        result = false;
        break;
      }
    }
    stack['delete'](array);
    stack['delete'](other);
    return result;
  }

  var _equalArrays = equalArrays;

  /** Built-in value references. */
  var Uint8Array$1 = _root.Uint8Array;

  var _Uint8Array = Uint8Array$1;

  /**
   * Converts `map` to its key-value pairs.
   *
   * @private
   * @param {Object} map The map to convert.
   * @returns {Array} Returns the key-value pairs.
   */
  function mapToArray(map) {
    var index = -1,
        result = Array(map.size);

    map.forEach(function(value, key) {
      result[++index] = [key, value];
    });
    return result;
  }

  var _mapToArray = mapToArray;

  /**
   * Converts `set` to an array of its values.
   *
   * @private
   * @param {Object} set The set to convert.
   * @returns {Array} Returns the values.
   */
  function setToArray(set) {
    var index = -1,
        result = Array(set.size);

    set.forEach(function(value) {
      result[++index] = value;
    });
    return result;
  }

  var _setToArray = setToArray;

  /** Used to compose bitmasks for value comparisons. */
  var COMPARE_PARTIAL_FLAG$1 = 1,
      COMPARE_UNORDERED_FLAG$1 = 2;

  /** `Object#toString` result references. */
  var boolTag = '[object Boolean]',
      dateTag = '[object Date]',
      errorTag = '[object Error]',
      mapTag = '[object Map]',
      numberTag = '[object Number]',
      regexpTag = '[object RegExp]',
      setTag = '[object Set]',
      stringTag = '[object String]',
      symbolTag = '[object Symbol]';

  var arrayBufferTag = '[object ArrayBuffer]',
      dataViewTag = '[object DataView]';

  /** Used to convert symbols to primitives and strings. */
  var symbolProto = _Symbol ? _Symbol.prototype : undefined,
      symbolValueOf = symbolProto ? symbolProto.valueOf : undefined;

  /**
   * A specialized version of `baseIsEqualDeep` for comparing objects of
   * the same `toStringTag`.
   *
   * **Note:** This function only supports comparing values with tags of
   * `Boolean`, `Date`, `Error`, `Number`, `RegExp`, or `String`.
   *
   * @private
   * @param {Object} object The object to compare.
   * @param {Object} other The other object to compare.
   * @param {string} tag The `toStringTag` of the objects to compare.
   * @param {number} bitmask The bitmask flags. See `baseIsEqual` for more details.
   * @param {Function} customizer The function to customize comparisons.
   * @param {Function} equalFunc The function to determine equivalents of values.
   * @param {Object} stack Tracks traversed `object` and `other` objects.
   * @returns {boolean} Returns `true` if the objects are equivalent, else `false`.
   */
  function equalByTag(object, other, tag, bitmask, customizer, equalFunc, stack) {
    switch (tag) {
      case dataViewTag:
        if ((object.byteLength != other.byteLength) ||
            (object.byteOffset != other.byteOffset)) {
          return false;
        }
        object = object.buffer;
        other = other.buffer;

      case arrayBufferTag:
        if ((object.byteLength != other.byteLength) ||
            !equalFunc(new _Uint8Array(object), new _Uint8Array(other))) {
          return false;
        }
        return true;

      case boolTag:
      case dateTag:
      case numberTag:
        // Coerce booleans to `1` or `0` and dates to milliseconds.
        // Invalid dates are coerced to `NaN`.
        return eq_1(+object, +other);

      case errorTag:
        return object.name == other.name && object.message == other.message;

      case regexpTag:
      case stringTag:
        // Coerce regexes to strings and treat strings, primitives and objects,
        // as equal. See http://www.ecma-international.org/ecma-262/7.0/#sec-regexp.prototype.tostring
        // for more details.
        return object == (other + '');

      case mapTag:
        var convert = _mapToArray;

      case setTag:
        var isPartial = bitmask & COMPARE_PARTIAL_FLAG$1;
        convert || (convert = _setToArray);

        if (object.size != other.size && !isPartial) {
          return false;
        }
        // Assume cyclic values are equal.
        var stacked = stack.get(object);
        if (stacked) {
          return stacked == other;
        }
        bitmask |= COMPARE_UNORDERED_FLAG$1;

        // Recursively compare objects (susceptible to call stack limits).
        stack.set(object, other);
        var result = _equalArrays(convert(object), convert(other), bitmask, customizer, equalFunc, stack);
        stack['delete'](object);
        return result;

      case symbolTag:
        if (symbolValueOf) {
          return symbolValueOf.call(object) == symbolValueOf.call(other);
        }
    }
    return false;
  }

  var _equalByTag = equalByTag;

  /**
   * Appends the elements of `values` to `array`.
   *
   * @private
   * @param {Array} array The array to modify.
   * @param {Array} values The values to append.
   * @returns {Array} Returns `array`.
   */
  function arrayPush(array, values) {
    var index = -1,
        length = values.length,
        offset = array.length;

    while (++index < length) {
      array[offset + index] = values[index];
    }
    return array;
  }

  var _arrayPush = arrayPush;

  /**
   * Checks if `value` is classified as an `Array` object.
   *
   * @static
   * @memberOf _
   * @since 0.1.0
   * @category Lang
   * @param {*} value The value to check.
   * @returns {boolean} Returns `true` if `value` is an array, else `false`.
   * @example
   *
   * _.isArray([1, 2, 3]);
   * // => true
   *
   * _.isArray(document.body.children);
   * // => false
   *
   * _.isArray('abc');
   * // => false
   *
   * _.isArray(_.noop);
   * // => false
   */
  var isArray = Array.isArray;

  var isArray_1 = isArray;

  /**
   * The base implementation of `getAllKeys` and `getAllKeysIn` which uses
   * `keysFunc` and `symbolsFunc` to get the enumerable property names and
   * symbols of `object`.
   *
   * @private
   * @param {Object} object The object to query.
   * @param {Function} keysFunc The function to get the keys of `object`.
   * @param {Function} symbolsFunc The function to get the symbols of `object`.
   * @returns {Array} Returns the array of property names and symbols.
   */
  function baseGetAllKeys(object, keysFunc, symbolsFunc) {
    var result = keysFunc(object);
    return isArray_1(object) ? result : _arrayPush(result, symbolsFunc(object));
  }

  var _baseGetAllKeys = baseGetAllKeys;

  /**
   * A specialized version of `_.filter` for arrays without support for
   * iteratee shorthands.
   *
   * @private
   * @param {Array} [array] The array to iterate over.
   * @param {Function} predicate The function invoked per iteration.
   * @returns {Array} Returns the new filtered array.
   */
  function arrayFilter(array, predicate) {
    var index = -1,
        length = array == null ? 0 : array.length,
        resIndex = 0,
        result = [];

    while (++index < length) {
      var value = array[index];
      if (predicate(value, index, array)) {
        result[resIndex++] = value;
      }
    }
    return result;
  }

  var _arrayFilter = arrayFilter;

  /**
   * This method returns a new empty array.
   *
   * @static
   * @memberOf _
   * @since 4.13.0
   * @category Util
   * @returns {Array} Returns the new empty array.
   * @example
   *
   * var arrays = _.times(2, _.stubArray);
   *
   * console.log(arrays);
   * // => [[], []]
   *
   * console.log(arrays[0] === arrays[1]);
   * // => false
   */
  function stubArray() {
    return [];
  }

  var stubArray_1 = stubArray;

  /** Used for built-in method references. */
  var objectProto$5 = Object.prototype;

  /** Built-in value references. */
  var propertyIsEnumerable = objectProto$5.propertyIsEnumerable;

  /* Built-in method references for those with the same name as other `lodash` methods. */
  var nativeGetSymbols = Object.getOwnPropertySymbols;

  /**
   * Creates an array of the own enumerable symbols of `object`.
   *
   * @private
   * @param {Object} object The object to query.
   * @returns {Array} Returns the array of symbols.
   */
  var getSymbols = !nativeGetSymbols ? stubArray_1 : function(object) {
    if (object == null) {
      return [];
    }
    object = Object(object);
    return _arrayFilter(nativeGetSymbols(object), function(symbol) {
      return propertyIsEnumerable.call(object, symbol);
    });
  };

  var _getSymbols = getSymbols;

  /**
   * The base implementation of `_.times` without support for iteratee shorthands
   * or max array length checks.
   *
   * @private
   * @param {number} n The number of times to invoke `iteratee`.
   * @param {Function} iteratee The function invoked per iteration.
   * @returns {Array} Returns the array of results.
   */
  function baseTimes(n, iteratee) {
    var index = -1,
        result = Array(n);

    while (++index < n) {
      result[index] = iteratee(index);
    }
    return result;
  }

  var _baseTimes = baseTimes;

  /**
   * Checks if `value` is object-like. A value is object-like if it's not `null`
   * and has a `typeof` result of "object".
   *
   * @static
   * @memberOf _
   * @since 4.0.0
   * @category Lang
   * @param {*} value The value to check.
   * @returns {boolean} Returns `true` if `value` is object-like, else `false`.
   * @example
   *
   * _.isObjectLike({});
   * // => true
   *
   * _.isObjectLike([1, 2, 3]);
   * // => true
   *
   * _.isObjectLike(_.noop);
   * // => false
   *
   * _.isObjectLike(null);
   * // => false
   */
  function isObjectLike(value) {
    return value != null && typeof value == 'object';
  }

  var isObjectLike_1 = isObjectLike;

  /** `Object#toString` result references. */
  var argsTag = '[object Arguments]';

  /**
   * The base implementation of `_.isArguments`.
   *
   * @private
   * @param {*} value The value to check.
   * @returns {boolean} Returns `true` if `value` is an `arguments` object,
   */
  function baseIsArguments(value) {
    return isObjectLike_1(value) && _baseGetTag(value) == argsTag;
  }

  var _baseIsArguments = baseIsArguments;

  /** Used for built-in method references. */
  var objectProto$6 = Object.prototype;

  /** Used to check objects for own properties. */
  var hasOwnProperty$4 = objectProto$6.hasOwnProperty;

  /** Built-in value references. */
  var propertyIsEnumerable$1 = objectProto$6.propertyIsEnumerable;

  /**
   * Checks if `value` is likely an `arguments` object.
   *
   * @static
   * @memberOf _
   * @since 0.1.0
   * @category Lang
   * @param {*} value The value to check.
   * @returns {boolean} Returns `true` if `value` is an `arguments` object,
   *  else `false`.
   * @example
   *
   * _.isArguments(function() { return arguments; }());
   * // => true
   *
   * _.isArguments([1, 2, 3]);
   * // => false
   */
  var isArguments = _baseIsArguments(function() { return arguments; }()) ? _baseIsArguments : function(value) {
    return isObjectLike_1(value) && hasOwnProperty$4.call(value, 'callee') &&
      !propertyIsEnumerable$1.call(value, 'callee');
  };

  var isArguments_1 = isArguments;

  /**
   * This method returns `false`.
   *
   * @static
   * @memberOf _
   * @since 4.13.0
   * @category Util
   * @returns {boolean} Returns `false`.
   * @example
   *
   * _.times(2, _.stubFalse);
   * // => [false, false]
   */
  function stubFalse() {
    return false;
  }

  var stubFalse_1 = stubFalse;

  var isBuffer_1 = createCommonjsModule(function (module, exports) {
  /** Detect free variable `exports`. */
  var freeExports = exports && !exports.nodeType && exports;

  /** Detect free variable `module`. */
  var freeModule = freeExports && 'object' == 'object' && module && !module.nodeType && module;

  /** Detect the popular CommonJS extension `module.exports`. */
  var moduleExports = freeModule && freeModule.exports === freeExports;

  /** Built-in value references. */
  var Buffer = moduleExports ? _root.Buffer : undefined;

  /* Built-in method references for those with the same name as other `lodash` methods. */
  var nativeIsBuffer = Buffer ? Buffer.isBuffer : undefined;

  /**
   * Checks if `value` is a buffer.
   *
   * @static
   * @memberOf _
   * @since 4.3.0
   * @category Lang
   * @param {*} value The value to check.
   * @returns {boolean} Returns `true` if `value` is a buffer, else `false`.
   * @example
   *
   * _.isBuffer(new Buffer(2));
   * // => true
   *
   * _.isBuffer(new Uint8Array(2));
   * // => false
   */
  var isBuffer = nativeIsBuffer || stubFalse_1;

  module.exports = isBuffer;
  });

  /** Used as references for various `Number` constants. */
  var MAX_SAFE_INTEGER = 9007199254740991;

  /** Used to detect unsigned integer values. */
  var reIsUint = /^(?:0|[1-9]\d*)$/;

  /**
   * Checks if `value` is a valid array-like index.
   *
   * @private
   * @param {*} value The value to check.
   * @param {number} [length=MAX_SAFE_INTEGER] The upper bounds of a valid index.
   * @returns {boolean} Returns `true` if `value` is a valid index, else `false`.
   */
  function isIndex(value, length) {
    var type = typeof value;
    length = length == null ? MAX_SAFE_INTEGER : length;

    return !!length &&
      (type == 'number' ||
        (type != 'symbol' && reIsUint.test(value))) &&
          (value > -1 && value % 1 == 0 && value < length);
  }

  var _isIndex = isIndex;

  /** Used as references for various `Number` constants. */
  var MAX_SAFE_INTEGER$1 = 9007199254740991;

  /**
   * Checks if `value` is a valid array-like length.
   *
   * **Note:** This method is loosely based on
   * [`ToLength`](http://ecma-international.org/ecma-262/7.0/#sec-tolength).
   *
   * @static
   * @memberOf _
   * @since 4.0.0
   * @category Lang
   * @param {*} value The value to check.
   * @returns {boolean} Returns `true` if `value` is a valid length, else `false`.
   * @example
   *
   * _.isLength(3);
   * // => true
   *
   * _.isLength(Number.MIN_VALUE);
   * // => false
   *
   * _.isLength(Infinity);
   * // => false
   *
   * _.isLength('3');
   * // => false
   */
  function isLength(value) {
    return typeof value == 'number' &&
      value > -1 && value % 1 == 0 && value <= MAX_SAFE_INTEGER$1;
  }

  var isLength_1 = isLength;

  /** `Object#toString` result references. */
  var argsTag$1 = '[object Arguments]',
      arrayTag = '[object Array]',
      boolTag$1 = '[object Boolean]',
      dateTag$1 = '[object Date]',
      errorTag$1 = '[object Error]',
      funcTag$1 = '[object Function]',
      mapTag$1 = '[object Map]',
      numberTag$1 = '[object Number]',
      objectTag = '[object Object]',
      regexpTag$1 = '[object RegExp]',
      setTag$1 = '[object Set]',
      stringTag$1 = '[object String]',
      weakMapTag = '[object WeakMap]';

  var arrayBufferTag$1 = '[object ArrayBuffer]',
      dataViewTag$1 = '[object DataView]',
      float32Tag = '[object Float32Array]',
      float64Tag = '[object Float64Array]',
      int8Tag = '[object Int8Array]',
      int16Tag = '[object Int16Array]',
      int32Tag = '[object Int32Array]',
      uint8Tag = '[object Uint8Array]',
      uint8ClampedTag = '[object Uint8ClampedArray]',
      uint16Tag = '[object Uint16Array]',
      uint32Tag = '[object Uint32Array]';

  /** Used to identify `toStringTag` values of typed arrays. */
  var typedArrayTags = {};
  typedArrayTags[float32Tag] = typedArrayTags[float64Tag] =
  typedArrayTags[int8Tag] = typedArrayTags[int16Tag] =
  typedArrayTags[int32Tag] = typedArrayTags[uint8Tag] =
  typedArrayTags[uint8ClampedTag] = typedArrayTags[uint16Tag] =
  typedArrayTags[uint32Tag] = true;
  typedArrayTags[argsTag$1] = typedArrayTags[arrayTag] =
  typedArrayTags[arrayBufferTag$1] = typedArrayTags[boolTag$1] =
  typedArrayTags[dataViewTag$1] = typedArrayTags[dateTag$1] =
  typedArrayTags[errorTag$1] = typedArrayTags[funcTag$1] =
  typedArrayTags[mapTag$1] = typedArrayTags[numberTag$1] =
  typedArrayTags[objectTag] = typedArrayTags[regexpTag$1] =
  typedArrayTags[setTag$1] = typedArrayTags[stringTag$1] =
  typedArrayTags[weakMapTag] = false;

  /**
   * The base implementation of `_.isTypedArray` without Node.js optimizations.
   *
   * @private
   * @param {*} value The value to check.
   * @returns {boolean} Returns `true` if `value` is a typed array, else `false`.
   */
  function baseIsTypedArray(value) {
    return isObjectLike_1(value) &&
      isLength_1(value.length) && !!typedArrayTags[_baseGetTag(value)];
  }

  var _baseIsTypedArray = baseIsTypedArray;

  /**
   * The base implementation of `_.unary` without support for storing metadata.
   *
   * @private
   * @param {Function} func The function to cap arguments for.
   * @returns {Function} Returns the new capped function.
   */
  function baseUnary(func) {
    return function(value) {
      return func(value);
    };
  }

  var _baseUnary = baseUnary;

  var _nodeUtil = createCommonjsModule(function (module, exports) {
  /** Detect free variable `exports`. */
  var freeExports = exports && !exports.nodeType && exports;

  /** Detect free variable `module`. */
  var freeModule = freeExports && 'object' == 'object' && module && !module.nodeType && module;

  /** Detect the popular CommonJS extension `module.exports`. */
  var moduleExports = freeModule && freeModule.exports === freeExports;

  /** Detect free variable `process` from Node.js. */
  var freeProcess = moduleExports && _freeGlobal.process;

  /** Used to access faster Node.js helpers. */
  var nodeUtil = (function() {
    try {
      // Use `util.types` for Node.js 10+.
      var types = freeModule && freeModule.require && freeModule.require('util').types;

      if (types) {
        return types;
      }

      // Legacy `process.binding('util')` for Node.js < 10.
      return freeProcess && freeProcess.binding && freeProcess.binding('util');
    } catch (e) {}
  }());

  module.exports = nodeUtil;
  });

  /* Node.js helper references. */
  var nodeIsTypedArray = _nodeUtil && _nodeUtil.isTypedArray;

  /**
   * Checks if `value` is classified as a typed array.
   *
   * @static
   * @memberOf _
   * @since 3.0.0
   * @category Lang
   * @param {*} value The value to check.
   * @returns {boolean} Returns `true` if `value` is a typed array, else `false`.
   * @example
   *
   * _.isTypedArray(new Uint8Array);
   * // => true
   *
   * _.isTypedArray([]);
   * // => false
   */
  var isTypedArray = nodeIsTypedArray ? _baseUnary(nodeIsTypedArray) : _baseIsTypedArray;

  var isTypedArray_1 = isTypedArray;

  /** Used for built-in method references. */
  var objectProto$7 = Object.prototype;

  /** Used to check objects for own properties. */
  var hasOwnProperty$5 = objectProto$7.hasOwnProperty;

  /**
   * Creates an array of the enumerable property names of the array-like `value`.
   *
   * @private
   * @param {*} value The value to query.
   * @param {boolean} inherited Specify returning inherited property names.
   * @returns {Array} Returns the array of property names.
   */
  function arrayLikeKeys(value, inherited) {
    var isArr = isArray_1(value),
        isArg = !isArr && isArguments_1(value),
        isBuff = !isArr && !isArg && isBuffer_1(value),
        isType = !isArr && !isArg && !isBuff && isTypedArray_1(value),
        skipIndexes = isArr || isArg || isBuff || isType,
        result = skipIndexes ? _baseTimes(value.length, String) : [],
        length = result.length;

    for (var key in value) {
      if ((inherited || hasOwnProperty$5.call(value, key)) &&
          !(skipIndexes && (
             // Safari 9 has enumerable `arguments.length` in strict mode.
             key == 'length' ||
             // Node.js 0.10 has enumerable non-index properties on buffers.
             (isBuff && (key == 'offset' || key == 'parent')) ||
             // PhantomJS 2 has enumerable non-index properties on typed arrays.
             (isType && (key == 'buffer' || key == 'byteLength' || key == 'byteOffset')) ||
             // Skip index properties.
             _isIndex(key, length)
          ))) {
        result.push(key);
      }
    }
    return result;
  }

  var _arrayLikeKeys = arrayLikeKeys;

  /** Used for built-in method references. */
  var objectProto$8 = Object.prototype;

  /**
   * Checks if `value` is likely a prototype object.
   *
   * @private
   * @param {*} value The value to check.
   * @returns {boolean} Returns `true` if `value` is a prototype, else `false`.
   */
  function isPrototype(value) {
    var Ctor = value && value.constructor,
        proto = (typeof Ctor == 'function' && Ctor.prototype) || objectProto$8;

    return value === proto;
  }

  var _isPrototype = isPrototype;

  /**
   * Creates a unary function that invokes `func` with its argument transformed.
   *
   * @private
   * @param {Function} func The function to wrap.
   * @param {Function} transform The argument transform.
   * @returns {Function} Returns the new function.
   */
  function overArg(func, transform) {
    return function(arg) {
      return func(transform(arg));
    };
  }

  var _overArg = overArg;

  /* Built-in method references for those with the same name as other `lodash` methods. */
  var nativeKeys = _overArg(Object.keys, Object);

  var _nativeKeys = nativeKeys;

  /** Used for built-in method references. */
  var objectProto$9 = Object.prototype;

  /** Used to check objects for own properties. */
  var hasOwnProperty$6 = objectProto$9.hasOwnProperty;

  /**
   * The base implementation of `_.keys` which doesn't treat sparse arrays as dense.
   *
   * @private
   * @param {Object} object The object to query.
   * @returns {Array} Returns the array of property names.
   */
  function baseKeys(object) {
    if (!_isPrototype(object)) {
      return _nativeKeys(object);
    }
    var result = [];
    for (var key in Object(object)) {
      if (hasOwnProperty$6.call(object, key) && key != 'constructor') {
        result.push(key);
      }
    }
    return result;
  }

  var _baseKeys = baseKeys;

  /**
   * Checks if `value` is array-like. A value is considered array-like if it's
   * not a function and has a `value.length` that's an integer greater than or
   * equal to `0` and less than or equal to `Number.MAX_SAFE_INTEGER`.
   *
   * @static
   * @memberOf _
   * @since 4.0.0
   * @category Lang
   * @param {*} value The value to check.
   * @returns {boolean} Returns `true` if `value` is array-like, else `false`.
   * @example
   *
   * _.isArrayLike([1, 2, 3]);
   * // => true
   *
   * _.isArrayLike(document.body.children);
   * // => true
   *
   * _.isArrayLike('abc');
   * // => true
   *
   * _.isArrayLike(_.noop);
   * // => false
   */
  function isArrayLike(value) {
    return value != null && isLength_1(value.length) && !isFunction_1(value);
  }

  var isArrayLike_1 = isArrayLike;

  /**
   * Creates an array of the own enumerable property names of `object`.
   *
   * **Note:** Non-object values are coerced to objects. See the
   * [ES spec](http://ecma-international.org/ecma-262/7.0/#sec-object.keys)
   * for more details.
   *
   * @static
   * @since 0.1.0
   * @memberOf _
   * @category Object
   * @param {Object} object The object to query.
   * @returns {Array} Returns the array of property names.
   * @example
   *
   * function Foo() {
   *   this.a = 1;
   *   this.b = 2;
   * }
   *
   * Foo.prototype.c = 3;
   *
   * _.keys(new Foo);
   * // => ['a', 'b'] (iteration order is not guaranteed)
   *
   * _.keys('hi');
   * // => ['0', '1']
   */
  function keys(object) {
    return isArrayLike_1(object) ? _arrayLikeKeys(object) : _baseKeys(object);
  }

  var keys_1 = keys;

  /**
   * Creates an array of own enumerable property names and symbols of `object`.
   *
   * @private
   * @param {Object} object The object to query.
   * @returns {Array} Returns the array of property names and symbols.
   */
  function getAllKeys(object) {
    return _baseGetAllKeys(object, keys_1, _getSymbols);
  }

  var _getAllKeys = getAllKeys;

  /** Used to compose bitmasks for value comparisons. */
  var COMPARE_PARTIAL_FLAG$2 = 1;

  /** Used for built-in method references. */
  var objectProto$a = Object.prototype;

  /** Used to check objects for own properties. */
  var hasOwnProperty$7 = objectProto$a.hasOwnProperty;

  /**
   * A specialized version of `baseIsEqualDeep` for objects with support for
   * partial deep comparisons.
   *
   * @private
   * @param {Object} object The object to compare.
   * @param {Object} other The other object to compare.
   * @param {number} bitmask The bitmask flags. See `baseIsEqual` for more details.
   * @param {Function} customizer The function to customize comparisons.
   * @param {Function} equalFunc The function to determine equivalents of values.
   * @param {Object} stack Tracks traversed `object` and `other` objects.
   * @returns {boolean} Returns `true` if the objects are equivalent, else `false`.
   */
  function equalObjects(object, other, bitmask, customizer, equalFunc, stack) {
    var isPartial = bitmask & COMPARE_PARTIAL_FLAG$2,
        objProps = _getAllKeys(object),
        objLength = objProps.length,
        othProps = _getAllKeys(other),
        othLength = othProps.length;

    if (objLength != othLength && !isPartial) {
      return false;
    }
    var index = objLength;
    while (index--) {
      var key = objProps[index];
      if (!(isPartial ? key in other : hasOwnProperty$7.call(other, key))) {
        return false;
      }
    }
    // Assume cyclic values are equal.
    var stacked = stack.get(object);
    if (stacked && stack.get(other)) {
      return stacked == other;
    }
    var result = true;
    stack.set(object, other);
    stack.set(other, object);

    var skipCtor = isPartial;
    while (++index < objLength) {
      key = objProps[index];
      var objValue = object[key],
          othValue = other[key];

      if (customizer) {
        var compared = isPartial
          ? customizer(othValue, objValue, key, other, object, stack)
          : customizer(objValue, othValue, key, object, other, stack);
      }
      // Recursively compare objects (susceptible to call stack limits).
      if (!(compared === undefined
            ? (objValue === othValue || equalFunc(objValue, othValue, bitmask, customizer, stack))
            : compared
          )) {
        result = false;
        break;
      }
      skipCtor || (skipCtor = key == 'constructor');
    }
    if (result && !skipCtor) {
      var objCtor = object.constructor,
          othCtor = other.constructor;

      // Non `Object` object instances with different constructors are not equal.
      if (objCtor != othCtor &&
          ('constructor' in object && 'constructor' in other) &&
          !(typeof objCtor == 'function' && objCtor instanceof objCtor &&
            typeof othCtor == 'function' && othCtor instanceof othCtor)) {
        result = false;
      }
    }
    stack['delete'](object);
    stack['delete'](other);
    return result;
  }

  var _equalObjects = equalObjects;

  /* Built-in method references that are verified to be native. */
  var DataView$1 = _getNative(_root, 'DataView');

  var _DataView = DataView$1;

  /* Built-in method references that are verified to be native. */
  var Promise$1 = _getNative(_root, 'Promise');

  var _Promise = Promise$1;

  /* Built-in method references that are verified to be native. */
  var Set$1 = _getNative(_root, 'Set');

  var _Set = Set$1;

  /* Built-in method references that are verified to be native. */
  var WeakMap$1 = _getNative(_root, 'WeakMap');

  var _WeakMap = WeakMap$1;

  /** `Object#toString` result references. */
  var mapTag$2 = '[object Map]',
      objectTag$1 = '[object Object]',
      promiseTag = '[object Promise]',
      setTag$2 = '[object Set]',
      weakMapTag$1 = '[object WeakMap]';

  var dataViewTag$2 = '[object DataView]';

  /** Used to detect maps, sets, and weakmaps. */
  var dataViewCtorString = _toSource(_DataView),
      mapCtorString = _toSource(_Map),
      promiseCtorString = _toSource(_Promise),
      setCtorString = _toSource(_Set),
      weakMapCtorString = _toSource(_WeakMap);

  /**
   * Gets the `toStringTag` of `value`.
   *
   * @private
   * @param {*} value The value to query.
   * @returns {string} Returns the `toStringTag`.
   */
  var getTag = _baseGetTag;

  // Fallback for data views, maps, sets, and weak maps in IE 11 and promises in Node.js < 6.
  if ((_DataView && getTag(new _DataView(new ArrayBuffer(1))) != dataViewTag$2) ||
      (_Map && getTag(new _Map) != mapTag$2) ||
      (_Promise && getTag(_Promise.resolve()) != promiseTag) ||
      (_Set && getTag(new _Set) != setTag$2) ||
      (_WeakMap && getTag(new _WeakMap) != weakMapTag$1)) {
    getTag = function(value) {
      var result = _baseGetTag(value),
          Ctor = result == objectTag$1 ? value.constructor : undefined,
          ctorString = Ctor ? _toSource(Ctor) : '';

      if (ctorString) {
        switch (ctorString) {
          case dataViewCtorString: return dataViewTag$2;
          case mapCtorString: return mapTag$2;
          case promiseCtorString: return promiseTag;
          case setCtorString: return setTag$2;
          case weakMapCtorString: return weakMapTag$1;
        }
      }
      return result;
    };
  }

  var _getTag = getTag;

  /** Used to compose bitmasks for value comparisons. */
  var COMPARE_PARTIAL_FLAG$3 = 1;

  /** `Object#toString` result references. */
  var argsTag$2 = '[object Arguments]',
      arrayTag$1 = '[object Array]',
      objectTag$2 = '[object Object]';

  /** Used for built-in method references. */
  var objectProto$b = Object.prototype;

  /** Used to check objects for own properties. */
  var hasOwnProperty$8 = objectProto$b.hasOwnProperty;

  /**
   * A specialized version of `baseIsEqual` for arrays and objects which performs
   * deep comparisons and tracks traversed objects enabling objects with circular
   * references to be compared.
   *
   * @private
   * @param {Object} object The object to compare.
   * @param {Object} other The other object to compare.
   * @param {number} bitmask The bitmask flags. See `baseIsEqual` for more details.
   * @param {Function} customizer The function to customize comparisons.
   * @param {Function} equalFunc The function to determine equivalents of values.
   * @param {Object} [stack] Tracks traversed `object` and `other` objects.
   * @returns {boolean} Returns `true` if the objects are equivalent, else `false`.
   */
  function baseIsEqualDeep(object, other, bitmask, customizer, equalFunc, stack) {
    var objIsArr = isArray_1(object),
        othIsArr = isArray_1(other),
        objTag = objIsArr ? arrayTag$1 : _getTag(object),
        othTag = othIsArr ? arrayTag$1 : _getTag(other);

    objTag = objTag == argsTag$2 ? objectTag$2 : objTag;
    othTag = othTag == argsTag$2 ? objectTag$2 : othTag;

    var objIsObj = objTag == objectTag$2,
        othIsObj = othTag == objectTag$2,
        isSameTag = objTag == othTag;

    if (isSameTag && isBuffer_1(object)) {
      if (!isBuffer_1(other)) {
        return false;
      }
      objIsArr = true;
      objIsObj = false;
    }
    if (isSameTag && !objIsObj) {
      stack || (stack = new _Stack);
      return (objIsArr || isTypedArray_1(object))
        ? _equalArrays(object, other, bitmask, customizer, equalFunc, stack)
        : _equalByTag(object, other, objTag, bitmask, customizer, equalFunc, stack);
    }
    if (!(bitmask & COMPARE_PARTIAL_FLAG$3)) {
      var objIsWrapped = objIsObj && hasOwnProperty$8.call(object, '__wrapped__'),
          othIsWrapped = othIsObj && hasOwnProperty$8.call(other, '__wrapped__');

      if (objIsWrapped || othIsWrapped) {
        var objUnwrapped = objIsWrapped ? object.value() : object,
            othUnwrapped = othIsWrapped ? other.value() : other;

        stack || (stack = new _Stack);
        return equalFunc(objUnwrapped, othUnwrapped, bitmask, customizer, stack);
      }
    }
    if (!isSameTag) {
      return false;
    }
    stack || (stack = new _Stack);
    return _equalObjects(object, other, bitmask, customizer, equalFunc, stack);
  }

  var _baseIsEqualDeep = baseIsEqualDeep;

  /**
   * The base implementation of `_.isEqual` which supports partial comparisons
   * and tracks traversed objects.
   *
   * @private
   * @param {*} value The value to compare.
   * @param {*} other The other value to compare.
   * @param {boolean} bitmask The bitmask flags.
   *  1 - Unordered comparison
   *  2 - Partial comparison
   * @param {Function} [customizer] The function to customize comparisons.
   * @param {Object} [stack] Tracks traversed `value` and `other` objects.
   * @returns {boolean} Returns `true` if the values are equivalent, else `false`.
   */
  function baseIsEqual(value, other, bitmask, customizer, stack) {
    if (value === other) {
      return true;
    }
    if (value == null || other == null || (!isObjectLike_1(value) && !isObjectLike_1(other))) {
      return value !== value && other !== other;
    }
    return _baseIsEqualDeep(value, other, bitmask, customizer, baseIsEqual, stack);
  }

  var _baseIsEqual = baseIsEqual;

  /**
   * Performs a deep comparison between two values to determine if they are
   * equivalent.
   *
   * **Note:** This method supports comparing arrays, array buffers, booleans,
   * date objects, error objects, maps, numbers, `Object` objects, regexes,
   * sets, strings, symbols, and typed arrays. `Object` objects are compared
   * by their own, not inherited, enumerable properties. Functions and DOM
   * nodes are compared by strict equality, i.e. `===`.
   *
   * @static
   * @memberOf _
   * @since 0.1.0
   * @category Lang
   * @param {*} value The value to compare.
   * @param {*} other The other value to compare.
   * @returns {boolean} Returns `true` if the values are equivalent, else `false`.
   * @example
   *
   * var object = { 'a': 1 };
   * var other = { 'a': 1 };
   *
   * _.isEqual(object, other);
   * // => true
   *
   * object === other;
   * // => false
   */
  function isEqual(value, other) {
    return _baseIsEqual(value, other);
  }

  var isEqual_1 = isEqual;

  /** Built-in value references. */
  var getPrototype = _overArg(Object.getPrototypeOf, Object);

  var _getPrototype = getPrototype;

  /** `Object#toString` result references. */
  var objectTag$3 = '[object Object]';

  /** Used for built-in method references. */
  var funcProto$2 = Function.prototype,
      objectProto$c = Object.prototype;

  /** Used to resolve the decompiled source of functions. */
  var funcToString$2 = funcProto$2.toString;

  /** Used to check objects for own properties. */
  var hasOwnProperty$9 = objectProto$c.hasOwnProperty;

  /** Used to infer the `Object` constructor. */
  var objectCtorString = funcToString$2.call(Object);

  /**
   * Checks if `value` is a plain object, that is, an object created by the
   * `Object` constructor or one with a `[[Prototype]]` of `null`.
   *
   * @static
   * @memberOf _
   * @since 0.8.0
   * @category Lang
   * @param {*} value The value to check.
   * @returns {boolean} Returns `true` if `value` is a plain object, else `false`.
   * @example
   *
   * function Foo() {
   *   this.a = 1;
   * }
   *
   * _.isPlainObject(new Foo);
   * // => false
   *
   * _.isPlainObject([1, 2, 3]);
   * // => false
   *
   * _.isPlainObject({ 'x': 0, 'y': 0 });
   * // => true
   *
   * _.isPlainObject(Object.create(null));
   * // => true
   */
  function isPlainObject(value) {
    if (!isObjectLike_1(value) || _baseGetTag(value) != objectTag$3) {
      return false;
    }
    var proto = _getPrototype(value);
    if (proto === null) {
      return true;
    }
    var Ctor = hasOwnProperty$9.call(proto, 'constructor') && proto.constructor;
    return typeof Ctor == 'function' && Ctor instanceof Ctor &&
      funcToString$2.call(Ctor) == objectCtorString;
  }

  var isPlainObject_1 = isPlainObject;

  // Copyright (c) Jupyter Development Team.
  // Distributed under the terms of the Modified BSD License.
  var __extends = (undefined && undefined.__extends) || (function () {
      var extendStatics = function (d, b) {
          extendStatics = Object.setPrototypeOf ||
              ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
              function (d, b) { for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p]; };
          return extendStatics(d, b);
      };
      return function (d, b) {
          extendStatics(d, b);
          function __() { this.constructor = d; }
          d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
      };
  })();
  var __assign = (undefined && undefined.__assign) || function () {
      __assign = Object.assign || function(t) {
          for (var s, i = 1, n = arguments.length; i < n; i++) {
              s = arguments[i];
              for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                  t[p] = s[p];
          }
          return t;
      };
      return __assign.apply(this, arguments);
  };
  /**
   * Find all strings in the first argument that are not in the second.
   */
  function difference(a, b) {
      return a.filter(function (v) { return b.indexOf(v) === -1; });
  }
  /**
   * Compare two objects deeply to see if they are equal.
   */
  function isEqual$1(a, b) {
      return isEqual_1(a, b);
  }
  /**
   * A polyfill for Object.assign
   *
   * This is from code that Typescript 2.4 generates for a polyfill.
   */
  var assign = Object.assign || function (t) {
      for (var i = 1; i < arguments.length; i++) {
          var s = arguments[i];
          for (var p in s) {
              if (Object.prototype.hasOwnProperty.call(s, p)) {
                  t[p] = s[p];
              }
          }
      }
      return t;
  };
  /**
   * Generate a UUID
   *
   * http://www.ietf.org/rfc/rfc4122.txt
   */
  function uuid() {
      return UUID.uuid4();
  }
  /**
   * Wrappable Error class
   *
   * The Error class doesn't actually act on `this`.  Instead it always
   * returns a new instance of Error.  Here we capture that instance so we
   * can apply it's properties to `this`.
   */
  var WrappedError = /** @class */ (function (_super) {
      __extends(WrappedError, _super);
      function WrappedError(message, error) {
          var _this = _super.call(this, message) || this;
          console.warn('WrappedError has been deprecated!');
          // Keep a stack of the original error messages.
          if (error instanceof WrappedError) {
              _this.error_stack = error.error_stack;
          }
          else {
              _this.error_stack = [error];
          }
          _this.error_stack.push(_this);
          return _this;
      }
      return WrappedError;
  }(Error));
  /**
   * Resolve a promiseful dictionary.
   * Returns a single Promise.
   */
  function resolvePromisesDict(d) {
      var keys = Object.keys(d);
      var values = [];
      keys.forEach(function (key) {
          values.push(d[key]);
      });
      return Promise.all(values).then(function (v) {
          var d = {};
          for (var i = 0; i < keys.length; i++) {
              d[keys[i]] = v[i];
          }
          return d;
      });
  }
  /**
   * Creates a wrappable Promise rejection function.
   *
   * Creates a function that logs an error message before rethrowing
   * the original error that caused the promise to reject.
   */
  function reject(message, log) {
      return function promiseRejection(error) {
          if (log) {
              console.error(new Error(message));
          }
          throw error;
      };
  }
  /**
   * Takes an object 'state' and fills in buffer[i] at 'path' buffer_paths[i]
   * where buffer_paths[i] is a list indicating where in the object buffer[i] should
   * be placed
   * Example: state = {a: 1, b: {}, c: [0, null]}
   * buffers = [array1, array2]
   * buffer_paths = [['b', 'data'], ['c', 1]]
   * Will lead to {a: 1, b: {data: array1}, c: [0, array2]}
   */
  function put_buffers(state, buffer_paths, buffers) {
      for (var i = 0; i < buffer_paths.length; i++) {
          var buffer_path = buffer_paths[i];
          // say we want to set state[x][y][z] = buffers[i]
          var obj = state;
          // we first get obj = state[x][y]
          for (var j = 0; j < buffer_path.length - 1; j++) {
              obj = obj[buffer_path[j]];
          }
          // and then set: obj[z] = buffers[i]
          obj[buffer_path[buffer_path.length - 1]] = buffers[i];
      }
  }
  /**
   * The inverse of put_buffers, return an objects with the new state where all buffers(ArrayBuffer)
   * are removed. If a buffer is a member of an object, that object is cloned, and the key removed. If a buffer
   * is an element of an array, that array is cloned, and the element is set to null.
   * See put_buffers for the meaning of buffer_paths
   * Returns an object with the new state (.state) an array with paths to the buffers (.buffer_paths),
   * and the buffers associated to those paths (.buffers).
   */
  function remove_buffers(state) {
      var buffers = [];
      var buffer_paths = [];
      // if we need to remove an object from a list, we need to clone that list, otherwise we may modify
      // the internal state of the widget model
      // however, we do not want to clone everything, for performance
      function remove(obj, path) {
          if (obj.toJSON) {
              // We need to get the JSON form of the object before recursing.
              // See https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/JSON/stringify#toJSON()_behavior
              obj = obj.toJSON();
          }
          if (Array.isArray(obj)) {
              var is_cloned = false;
              for (var i = 0; i < obj.length; i++) {
                  var value = obj[i];
                  if (value) {
                      if (value instanceof ArrayBuffer || ArrayBuffer.isView(value)) {
                          if (!is_cloned) {
                              obj = obj.slice();
                              is_cloned = true;
                          }
                          buffers.push(ArrayBuffer.isView(value) ? value.buffer : value);
                          buffer_paths.push(path.concat([i]));
                          // easier to just keep the array, but clear the entry, otherwise we have to think
                          // about array length, much easier this way
                          obj[i] = null;
                      }
                      else {
                          var new_value = remove(value, path.concat([i]));
                          // only assigned when the value changes, we may serialize objects that don't support assignment
                          if (new_value !== value) {
                              if (!is_cloned) {
                                  obj = obj.slice();
                                  is_cloned = true;
                              }
                              obj[i] = new_value;
                          }
                      }
                  }
              }
          }
          else if (isPlainObject_1(obj)) {
              for (var key in obj) {
                  var is_cloned = false;
                  if (obj.hasOwnProperty(key)) {
                      var value = obj[key];
                      if (value) {
                          if (value instanceof ArrayBuffer || ArrayBuffer.isView(value)) {
                              if (!is_cloned) {
                                  obj = __assign({}, obj);
                                  is_cloned = true;
                              }
                              buffers.push(ArrayBuffer.isView(value) ? value.buffer : value);
                              buffer_paths.push(path.concat([key]));
                              delete obj[key]; // for objects/dicts we just delete them
                          }
                          else {
                              var new_value = remove(value, path.concat([key]));
                              // only assigned when the value changes, we may serialize objects that don't support assignment
                              if (new_value !== value) {
                                  if (!is_cloned) {
                                      obj = __assign({}, obj);
                                      is_cloned = true;
                                  }
                                  obj[key] = new_value;
                              }
                          }
                      }
                  }
              }
          }
          return obj;
      }
      var new_state = remove(state, []);
      return { state: new_state, buffers: buffers, buffer_paths: buffer_paths };
  }
  var hexTable = [
      '00', '01', '02', '03', '04', '05', '06', '07', '08', '09', '0A', '0B', '0C', '0D', '0E', '0F',
      '10', '11', '12', '13', '14', '15', '16', '17', '18', '19', '1A', '1B', '1C', '1D', '1E', '1F',
      '20', '21', '22', '23', '24', '25', '26', '27', '28', '29', '2A', '2B', '2C', '2D', '2E', '2F',
      '30', '31', '32', '33', '34', '35', '36', '37', '38', '39', '3A', '3B', '3C', '3D', '3E', '3F',
      '40', '41', '42', '43', '44', '45', '46', '47', '48', '49', '4A', '4B', '4C', '4D', '4E', '4F',
      '50', '51', '52', '53', '54', '55', '56', '57', '58', '59', '5A', '5B', '5C', '5D', '5E', '5F',
      '60', '61', '62', '63', '64', '65', '66', '67', '68', '69', '6A', '6B', '6C', '6D', '6E', '6F',
      '70', '71', '72', '73', '74', '75', '76', '77', '78', '79', '7A', '7B', '7C', '7D', '7E', '7F',
      '80', '81', '82', '83', '84', '85', '86', '87', '88', '89', '8A', '8B', '8C', '8D', '8E', '8F',
      '90', '91', '92', '93', '94', '95', '96', '97', '98', '99', '9A', '9B', '9C', '9D', '9E', '9F',
      'A0', 'A1', 'A2', 'A3', 'A4', 'A5', 'A6', 'A7', 'A8', 'A9', 'AA', 'AB', 'AC', 'AD', 'AE', 'AF',
      'B0', 'B1', 'B2', 'B3', 'B4', 'B5', 'B6', 'B7', 'B8', 'B9', 'BA', 'BB', 'BC', 'BD', 'BE', 'BF',
      'C0', 'C1', 'C2', 'C3', 'C4', 'C5', 'C6', 'C7', 'C8', 'C9', 'CA', 'CB', 'CC', 'CD', 'CE', 'CF',
      'D0', 'D1', 'D2', 'D3', 'D4', 'D5', 'D6', 'D7', 'D8', 'D9', 'DA', 'DB', 'DC', 'DD', 'DE', 'DF',
      'E0', 'E1', 'E2', 'E3', 'E4', 'E5', 'E6', 'E7', 'E8', 'E9', 'EA', 'EB', 'EC', 'ED', 'EE', 'EF',
      'F0', 'F1', 'F2', 'F3', 'F4', 'F5', 'F6', 'F7', 'F8', 'F9', 'FA', 'FB', 'FC', 'FD', 'FE', 'FF'
  ];
  /**
   * Convert an ArrayBuffer to a hex string.
   */
  function bufferToHex(buffer) {
      var x = new Uint8Array(buffer);
      var s = [];
      for (var i = 0; i < x.length; i++) {
          s.push(hexTable[x[i]]);
      }
      return s.join('');
  }
  /**
   * Convert a hex string to an ArrayBuffer.
   */
  function hexToBuffer(hex) {
      var x = new Uint8Array(hex.length / 2);
      for (var i = 0; i < hex.length; i += 2) {
          x[i / 2] = parseInt(hex.slice(i, i + 2), 16);
      }
      return x.buffer;
  }
  /**
   * Convert an ArrayBuffer to a base64 string.
   */
  function bufferToBase64(buffer) {
      return fromByteArray_1(new Uint8Array(buffer));
  }
  /**
   * Convert a base64 string to an ArrayBuffer.
   */
  function base64ToBuffer(base64) {
      return toByteArray_1(base64).buffer;
  }

  var utils = /*#__PURE__*/Object.freeze({
    difference: difference,
    isEqual: isEqual$1,
    assign: assign,
    uuid: uuid,
    WrappedError: WrappedError,
    resolvePromisesDict: resolvePromisesDict,
    reject: reject,
    put_buffers: put_buffers,
    remove_buffers: remove_buffers,
    bufferToHex: bufferToHex,
    hexToBuffer: hexToBuffer,
    bufferToBase64: bufferToBase64,
    base64ToBuffer: base64ToBuffer
  });

  // Copyright (c) Jupyter Development Team.
  // Distributed under the terms of the Modified BSD License.
  var JUPYTER_WIDGETS_VERSION = '1.2.0';
  var PROTOCOL_VERSION = '2.0.0';

  // Copyright (c) Jupyter Development Team.
  // Distributed under the terms of the Modified BSD License.
  var __assign$1 = (undefined && undefined.__assign) || function () {
      __assign$1 = Object.assign || function(t) {
          for (var s, i = 1, n = arguments.length; i < n; i++) {
              s = arguments[i];
              for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                  t[p] = s[p];
          }
          return t;
      };
      return __assign$1.apply(this, arguments);
  };
  var __awaiter = (undefined && undefined.__awaiter) || function (thisArg, _arguments, P, generator) {
      function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
      return new (P || (P = Promise))(function (resolve, reject) {
          function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
          function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
          function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
          step((generator = generator.apply(thisArg, _arguments || [])).next());
      });
  };
  var __generator = (undefined && undefined.__generator) || function (thisArg, body) {
      var _ = { label: 0, sent: function() { if (t[0] & 1) throw t[1]; return t[1]; }, trys: [], ops: [] }, f, y, t, g;
      return g = { next: verb(0), "throw": verb(1), "return": verb(2) }, typeof Symbol === "function" && (g[Symbol.iterator] = function() { return this; }), g;
      function verb(n) { return function (v) { return step([n, v]); }; }
      function step(op) {
          if (f) throw new TypeError("Generator is already executing.");
          while (_) try {
              if (f = 1, y && (t = op[0] & 2 ? y["return"] : op[0] ? y["throw"] || ((t = y["return"]) && t.call(y), 0) : y.next) && !(t = t.call(y, op[1])).done) return t;
              if (y = 0, t) op = [op[0] & 2, t.value];
              switch (op[0]) {
                  case 0: case 1: t = op; break;
                  case 4: _.label++; return { value: op[1], done: false };
                  case 5: _.label++; y = op[1]; op = [0]; continue;
                  case 7: op = _.ops.pop(); _.trys.pop(); continue;
                  default:
                      if (!(t = _.trys, t = t.length > 0 && t[t.length - 1]) && (op[0] === 6 || op[0] === 2)) { _ = 0; continue; }
                      if (op[0] === 3 && (!t || (op[1] > t[0] && op[1] < t[3]))) { _.label = op[1]; break; }
                      if (op[0] === 6 && _.label < t[1]) { _.label = t[1]; t = op; break; }
                      if (t && _.label < t[2]) { _.label = t[2]; _.ops.push(op); break; }
                      if (t[2]) _.ops.pop();
                      _.trys.pop(); continue;
              }
              op = body.call(thisArg, _);
          } catch (e) { op = [6, e]; y = 0; } finally { f = t = 0; }
          if (op[0] & 5) throw op[1]; return { value: op[0] ? op[1] : void 0, done: true };
      }
  };
  var PROTOCOL_MAJOR_VERSION = PROTOCOL_VERSION.split('.', 1)[0];
  /**
   * Manager abstract base class
   */
  var ManagerBase = /** @class */ (function () {
      function ManagerBase() {
          /**
           * The comm target name to register
           */
          this.comm_target_name = 'jupyter.widget';
          /**
           * Dictionary of model ids and model instance promises
           */
          this._models = Object.create(null);
      }
      /**
       * Display a DOMWidgetView for a particular model.
       */
      ManagerBase.prototype.display_model = function (msg, model, options) {
          var _this = this;
          if (options === void 0) { options = {}; }
          return this.create_view(model, options).then(function (view) { return _this.display_view(msg, view, options); }).catch(reject('Could not create view', true));
      };
      /**
       * Modifies view options. Generally overloaded in custom widget manager
       * implementations.
       */
      ManagerBase.prototype.setViewOptions = function (options) {
          if (options === void 0) { options = {}; }
          return options;
      };
      ManagerBase.prototype.create_view = function (model, options) {
          var _this = this;
          if (options === void 0) { options = {}; }
          var viewPromise = model.state_change = model.state_change.then(function () {
              return _this.loadClass(model.get('_view_name'), model.get('_view_module'), model.get('_view_module_version')).then(function (ViewType) {
                  var view = new ViewType({
                      model: model,
                      options: _this.setViewOptions(options)
                  });
                  view.listenTo(model, 'destroy', view.remove);
                  return Promise.resolve(view.render()).then(function () { return view; });
              }).catch(reject('Could not create a view for model id ' + model.model_id, true));
          });
          var id = uuid();
          model.views[id] = viewPromise;
          viewPromise.then(function (view) {
              view.once('remove', function () { delete view.model.views[id]; }, _this);
          });
          return model.state_change;
      };
      /**
       * callback handlers specific to a view
       */
      ManagerBase.prototype.callbacks = function (view) {
          return {};
      };
      /**
       * Get a promise for a model by model id.
       *
       * #### Notes
       * If a model is not found, undefined is returned (NOT a promise). However,
       * the calling code should also deal with the case where a rejected promise
       * is returned, and should treat that also as a model not found.
       */
      ManagerBase.prototype.get_model = function (model_id) {
          // TODO: Perhaps we should return a Promise.reject if the model is not
          // found. Right now this isn't a true async function because it doesn't
          // always return a promise.
          return this._models[model_id];
      };
      /**
       * Handle when a comm is opened.
       */
      ManagerBase.prototype.handle_comm_open = function (comm, msg) {
          var protocolVersion = (msg.metadata || {}).version || '';
          if (protocolVersion.split('.', 1)[0] !== PROTOCOL_MAJOR_VERSION) {
              var error = "Wrong widget protocol version: received protocol version '" + protocolVersion + "', but was expecting major version '" + PROTOCOL_MAJOR_VERSION + "'";
              console.error(error);
              return Promise.reject(error);
          }
          var data = msg.content.data;
          var buffer_paths = data.buffer_paths || [];
          // Make sure the buffers are DataViews
          var buffers = (msg.buffers || []).map(function (b) {
              if (b instanceof DataView) {
                  return b;
              }
              else {
                  return new DataView(b instanceof ArrayBuffer ? b : b.buffer);
              }
          });
          put_buffers(data.state, buffer_paths, buffers);
          return this.new_model({
              model_name: data.state['_model_name'],
              model_module: data.state['_model_module'],
              model_module_version: data.state['_model_module_version'],
              comm: comm
          }, data.state).catch(reject('Could not create a model.', true));
      };
      /**
       * Create a comm and new widget model.
       * @param  options - same options as new_model but comm is not
       *                          required and additional options are available.
       * @param  serialized_state - serialized model attributes.
       */
      ManagerBase.prototype.new_widget = function (options, serialized_state) {
          var _this = this;
          if (serialized_state === void 0) { serialized_state = {}; }
          var commPromise;
          // we check to make sure the view information is provided, to help catch
          // backwards incompatibility errors.
          if (options.view_name === undefined
              || options.view_module === undefined
              || options.view_module_version === undefined) {
              return Promise.reject('new_widget(...) must be given view information in the options.');
          }
          // If no comm is provided, a new comm is opened for the jupyter.widget
          // target.
          if (options.comm) {
              commPromise = Promise.resolve(options.comm);
          }
          else {
              commPromise = this._create_comm(this.comm_target_name, options.model_id, {
                  state: {
                      _model_module: options.model_module,
                      _model_module_version: options.model_module_version,
                      _model_name: options.model_name,
                      _view_module: options.view_module,
                      _view_module_version: options.view_module_version,
                      _view_name: options.view_name
                  },
              }, { version: PROTOCOL_VERSION });
          }
          // The options dictionary is copied since data will be added to it.
          var options_clone = __assign$1({}, options);
          // Create the model. In the case where the comm promise is rejected a
          // comm-less model is still created with the required model id.
          return commPromise.then(function (comm) {
              // Comm Promise Resolved.
              options_clone.comm = comm;
              var widget_model = _this.new_model(options_clone, serialized_state);
              return widget_model.then(function (model) {
                  model.sync('create', model);
                  return model;
              });
          }, function () {
              // Comm Promise Rejected.
              if (!options_clone.model_id) {
                  options_clone.model_id = uuid();
              }
              return _this.new_model(options_clone, serialized_state);
          });
      };
      ManagerBase.prototype.register_model = function (model_id, modelPromise) {
          var _this = this;
          this._models[model_id] = modelPromise;
          modelPromise.then(function (model) {
              model.once('comm:close', function () {
                  delete _this._models[model_id];
              });
          });
      };
      /**
       * Create and return a promise for a new widget model
       *
       * @param options - the options for creating the model.
       * @param serialized_state - attribute values for the model.
       *
       * @example
       * widget_manager.new_model({
       *      model_name: 'IntSlider',
       *      model_module: '@jupyter-widgets/controls',
       *      model_module_version: '1.0.0',
       *      model_id: 'u-u-i-d'
       * }).then((model) => { console.log('Create success!', model); },
       *  (err) => {console.error(err)});
       *
       */
      ManagerBase.prototype.new_model = function (options, serialized_state) {
          if (serialized_state === void 0) { serialized_state = {}; }
          return __awaiter(this, void 0, void 0, function () {
              var model_id, modelPromise;
              return __generator(this, function (_a) {
                  switch (_a.label) {
                      case 0:
                          if (options.model_id) {
                              model_id = options.model_id;
                          }
                          else if (options.comm) {
                              model_id = options.model_id = options.comm.comm_id;
                          }
                          else {
                              throw new Error('Neither comm nor model_id provided in options object. At least one must exist.');
                          }
                          modelPromise = this._make_model(options, serialized_state);
                          // this call needs to happen before the first `await`, see note in `set_state`:
                          this.register_model(model_id, modelPromise);
                          return [4 /*yield*/, modelPromise];
                      case 1: return [2 /*return*/, _a.sent()];
                  }
              });
          });
      };
      ManagerBase.prototype._make_model = function (options, serialized_state) {
          if (serialized_state === void 0) { serialized_state = {}; }
          return __awaiter(this, void 0, void 0, function () {
              var model_id, model_promise, ModelType, error_1, attributes, modelOptions, widget_model;
              return __generator(this, function (_a) {
                  switch (_a.label) {
                      case 0:
                          model_id = options.model_id;
                          model_promise = this.loadClass(options.model_name, options.model_module, options.model_module_version);
                          _a.label = 1;
                      case 1:
                          _a.trys.push([1, 3, , 4]);
                          return [4 /*yield*/, model_promise];
                      case 2:
                          ModelType = _a.sent();
                          return [3 /*break*/, 4];
                      case 3:
                          error_1 = _a.sent();
                          console.error('Could not instantiate widget');
                          throw error_1;
                      case 4:
                          if (!ModelType) {
                              throw new Error("Cannot find model module " + options.model_module + "@" + options.model_module_version + ", " + options.model_name);
                          }
                          return [4 /*yield*/, ModelType._deserialize_state(serialized_state, this)];
                      case 5:
                          attributes = _a.sent();
                          modelOptions = {
                              widget_manager: this,
                              model_id: model_id,
                              comm: options.comm,
                          };
                          widget_model = new ModelType(attributes, modelOptions);
                          widget_model.name = options.model_name;
                          widget_model.module = options.model_module;
                          return [2 /*return*/, widget_model];
                  }
              });
          });
      };
      /**
       * Close all widgets and empty the widget state.
       * @return Promise that resolves when the widget state is cleared.
       */
      ManagerBase.prototype.clear_state = function () {
          var _this = this;
          return resolvePromisesDict(this._models).then(function (models) {
              Object.keys(models).forEach(function (id) { return models[id].close(); });
              _this._models = Object.create(null);
          });
      };
      /**
       * Asynchronously get the state of the widget manager.
       *
       * This includes all of the widget models, and follows the format given in
       * the @jupyter-widgets/schema package.
       *
       * @param options - The options for what state to return.
       * @returns Promise for a state dictionary
       */
      ManagerBase.prototype.get_state = function (options) {
          var _this = this;
          if (options === void 0) { options = {}; }
          var modelPromises = Object.keys(this._models).map(function (id) { return _this._models[id]; });
          return Promise.all(modelPromises).then(function (models) {
              return serialize_state(models, options);
          });
      };
      /**
       * Set the widget manager state.
       *
       * @param state - a Javascript object conforming to the application/vnd.jupyter.widget-state+json spec.
       *
       * Reconstructs all of the widget models in the state, merges that with the
       * current manager state, and then attempts to redisplay the widgets in the
       * state.
       */
      ManagerBase.prototype.set_state = function (state) {
          var _this = this;
          // Check to make sure that it's the same version we are parsing.
          if (!(state.version_major && state.version_major <= 2)) {
              throw 'Unsupported widget state format';
          }
          var models = state.state;
          // Recreate all the widget models for the given widget manager state.
          var all_models = this._get_comm_info().then(function (live_comms) {
              /* Note: It is currently safe to just loop over the models in any order,
                 given that the following holds (does at the time of writing):
                 1: any call to `new_model` with state registers the model promise (e.g. with `register_model`)
                    synchronously (before it's first `await` statement).
                 2: any calls to a model constructor or the `set_state` method on a model,
                    happens asynchronously (in a `then` clause, or after an `await` statement).

                Without these assumptions, one risks trying to set model state with a reference
                to another model that doesn't exist yet!
              */
              return Promise.all(Object.keys(models).map(function (model_id) {
                  // First put back the binary buffers
                  var decode = { 'base64': base64ToBuffer, 'hex': hexToBuffer };
                  var model = models[model_id];
                  var modelState = model.state;
                  if (model.buffers) {
                      var bufferPaths = model.buffers.map(function (b) { return b.path; });
                      // put_buffers expects buffers to be DataViews
                      var buffers = model.buffers.map(function (b) { return new DataView(decode[b.encoding](b.data)); });
                      put_buffers(model.state, bufferPaths, buffers);
                  }
                  // If the model has already been created, set its state and then
                  // return it.
                  if (_this._models[model_id]) {
                      return _this._models[model_id].then(function (model) {
                          // deserialize state
                          return model.constructor._deserialize_state(modelState || {}, _this).then(function (attributes) {
                              model.set_state(attributes); // case 2
                              return model;
                          });
                      });
                  }
                  var modelCreate = {
                      model_id: model_id,
                      model_name: model.model_name,
                      model_module: model.model_module,
                      model_module_version: model.model_module_version
                  };
                  if (live_comms.hasOwnProperty(model_id)) { // live comm
                      // This connects to an existing comm if it exists, and
                      // should *not* send a comm open message.
                      return _this._create_comm(_this.comm_target_name, model_id).then(function (comm) {
                          modelCreate.comm = comm;
                          return _this.new_model(modelCreate); // No state, so safe wrt. case 1
                      });
                  }
                  else {
                      return _this.new_model(modelCreate, modelState); // case 1
                  }
              }));
          });
          return all_models;
      };
      /**
       * Disconnect the widget manager from the kernel, setting each model's comm
       * as dead.
       */
      ManagerBase.prototype.disconnect = function () {
          var _this = this;
          Object.keys(this._models).forEach(function (i) {
              _this._models[i].then(function (model) { model.comm_live = false; });
          });
      };
      /**
       * Resolve a URL relative to the current notebook location.
       *
       * The default implementation just returns the original url.
       */
      ManagerBase.prototype.resolveUrl = function (url) {
          return Promise.resolve(url);
      };
      /**
       * Filter serialized widget state to remove any ID's already present in manager.
       *
       * @param {*} state Serialized state to filter
       *
       * @returns {*} A copy of the state, with its 'state' attribute filtered
       */
      ManagerBase.prototype.filterExistingModelState = function (serialized_state) {
          var _this = this;
          var models = serialized_state.state;
          models = Object.keys(models)
              .filter(function (model_id) {
              return !_this._models[model_id];
          })
              .reduce(function (res, model_id) {
              res[model_id] = models[model_id];
              return res;
          }, {});
          return __assign$1(__assign$1({}, serialized_state), { state: models });
      };
      return ManagerBase;
  }());
  /**
   * Serialize an array of widget models
   *
   * #### Notes
   * The return value follows the format given in the
   * @jupyter-widgets/schema package.
   */
  function serialize_state(models, options) {
      if (options === void 0) { options = {}; }
      var state = {};
      models.forEach(function (model) {
          var model_id = model.model_id;
          var split = remove_buffers(model.serialize(model.get_state(options.drop_defaults)));
          var buffers = split.buffers.map(function (buffer, index) {
              return {
                  data: bufferToBase64(buffer),
                  path: split.buffer_paths[index],
                  encoding: 'base64'
              };
          });
          state[model_id] = {
              model_name: model.name,
              model_module: model.module,
              model_module_version: model.get('_model_module_version'),
              state: split.state
          };
          // To save space, only include the buffers key if we have buffers
          if (buffers.length > 0) {
              state[model_id].buffers = buffers;
          }
      });
      return { version_major: 2, version_minor: 0, state: state };
  }

  // This file contains a modified version of the set function from the Backbone
  // (see
  // https://github.com/jashkenas/backbone/blob/05fde9e201f7e2137796663081105cd6dad12a98/backbone.js#L460,
  // with changes below marked with an EDIT comment). This file in Backbone has the following license.
  var __assign$2 = (undefined && undefined.__assign) || function () {
      __assign$2 = Object.assign || function(t) {
          for (var s, i = 1, n = arguments.length; i < n; i++) {
              s = arguments[i];
              for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                  t[p] = s[p];
          }
          return t;
      };
      return __assign$2.apply(this, arguments);
  };
  // Set a hash of model attributes on the object, firing `"change"`. This is
  // the core primitive operation of a model, updating the data and notifying
  // anyone who needs to know about the change in state. The heart of the beast.
  // This *MUST* be called with the model as the `this` context.
  function set(key, val, options) {
      /* tslint:disable:no-invalid-this */
      if (key == null) {
          return this;
      }
      // Handle both `"key", value` and `{key: value}` -style arguments.
      var attrs;
      if (typeof key === 'object') {
          attrs = key;
          options = val;
      }
      else {
          (attrs = {})[key] = val;
      }
      options || (options = {});
      // Run validation.
      if (!this._validate(attrs, options)) {
          return false;
      }
      // Extract attributes and options.
      var unset = options.unset;
      var silent = options.silent;
      var changes = [];
      var changing = this._changing;
      this._changing = true;
      if (!changing) {
          // EDIT: changed to use object spread instead of _.clone
          this._previousAttributes = __assign$2({}, this.attributes);
          this.changed = {};
      }
      var current = this.attributes;
      var changed = this.changed;
      var prev = this._previousAttributes;
      // For each `set` attribute, update or delete the current value.
      for (var attr in attrs) {
          val = attrs[attr];
          // EDIT: the following two lines use our isEqual instead of _.isEqual
          if (!isEqual$1(current[attr], val)) {
              changes.push(attr);
          }
          if (!isEqual$1(prev[attr], val)) {
              changed[attr] = val;
          }
          else {
              delete changed[attr];
          }
          unset ? delete current[attr] : current[attr] = val;
      }
      // Update the `id`.
      this.id = this.get(this.idAttribute);
      // Trigger all relevant attribute changes.
      if (!silent) {
          if (changes.length) {
              this._pending = options;
          }
          for (var i = 0; i < changes.length; i++) {
              this.trigger('change:' + changes[i], this, current[changes[i]], options);
          }
      }
      // You might be wondering why there's a `while` loop here. Changes can
      // be recursively nested within `"change"` events.
      if (changing) {
          return this;
      }
      if (!silent) {
          while (this._pending) {
              options = this._pending;
              this._pending = false;
              this.trigger('change', this, options);
          }
      }
      this._pending = false;
      this._changing = false;
      return this;
      /* tslint:enable:no-invalid-this */
  }

  //     Underscore.js 1.10.2
  //     https://underscorejs.org
  //     (c) 2009-2020 Jeremy Ashkenas, DocumentCloud and Investigative Reporters & Editors
  //     Underscore may be freely distributed under the MIT license.

  // Baseline setup
  // --------------

  // Establish the root object, `window` (`self`) in the browser, `global`
  // on the server, or `this` in some virtual machines. We use `self`
  // instead of `window` for `WebWorker` support.
  var root$1 = typeof self == 'object' && self.self === self && self ||
            typeof global == 'object' && global.global === global && global ||
            Function('return this')() ||
            {};

  // Save bytes in the minified (but not gzipped) version:
  var ArrayProto = Array.prototype, ObjProto = Object.prototype;
  var SymbolProto = typeof Symbol !== 'undefined' ? Symbol.prototype : null;

  // Create quick reference variables for speed access to core prototypes.
  var push = ArrayProto.push,
      slice = ArrayProto.slice,
      toString = ObjProto.toString,
      hasOwnProperty$a = ObjProto.hasOwnProperty;

  // All **ECMAScript 5** native function implementations that we hope to use
  // are declared here.
  var nativeIsArray = Array.isArray,
      nativeKeys$1 = Object.keys,
      nativeCreate$1 = Object.create;

  // Create references to these builtin functions because we override them.
  var _isNaN = root$1.isNaN,
      _isFinite = root$1.isFinite;

  // Naked function reference for surrogate-prototype-swapping.
  var Ctor = function(){};

  // The Underscore object. All exported functions below are added to it in the
  // modules/index-all.js using the mixin function.
  function _(obj) {
    if (obj instanceof _) return obj;
    if (!(this instanceof _)) return new _(obj);
    this._wrapped = obj;
  }

  // Current version.
  var VERSION = _.VERSION = '1.10.2';

  // Internal function that returns an efficient (for current engines) version
  // of the passed-in callback, to be repeatedly applied in other Underscore
  // functions.
  function optimizeCb(func, context, argCount) {
    if (context === void 0) return func;
    switch (argCount == null ? 3 : argCount) {
      case 1: return function(value) {
        return func.call(context, value);
      };
      // The 2-argument case is omitted because we’re not using it.
      case 3: return function(value, index, collection) {
        return func.call(context, value, index, collection);
      };
      case 4: return function(accumulator, value, index, collection) {
        return func.call(context, accumulator, value, index, collection);
      };
    }
    return function() {
      return func.apply(context, arguments);
    };
  }

  // An internal function to generate callbacks that can be applied to each
  // element in a collection, returning the desired result — either `identity`,
  // an arbitrary callback, a property matcher, or a property accessor.
  function baseIteratee(value, context, argCount) {
    if (value == null) return identity;
    if (isFunction$1(value)) return optimizeCb(value, context, argCount);
    if (isObject$1(value) && !isArray$1(value)) return matcher(value);
    return property(value);
  }

  // External wrapper for our callback generator. Users may customize
  // `_.iteratee` if they want additional predicate/iteratee shorthand styles.
  // This abstraction hides the internal-only argCount argument.
  _.iteratee = iteratee;
  function iteratee(value, context) {
    return baseIteratee(value, context, Infinity);
  }

  // The function we actually call internally. It invokes _.iteratee if
  // overridden, otherwise baseIteratee.
  function cb(value, context, argCount) {
    if (_.iteratee !== iteratee) return _.iteratee(value, context);
    return baseIteratee(value, context, argCount);
  }

  // Some functions take a variable number of arguments, or a few expected
  // arguments at the beginning and then a variable number of values to operate
  // on. This helper accumulates all remaining arguments past the function’s
  // argument length (or an explicit `startIndex`), into an array that becomes
  // the last argument. Similar to ES6’s "rest parameter".
  function restArguments(func, startIndex) {
    startIndex = startIndex == null ? func.length - 1 : +startIndex;
    return function() {
      var length = Math.max(arguments.length - startIndex, 0),
          rest = Array(length),
          index = 0;
      for (; index < length; index++) {
        rest[index] = arguments[index + startIndex];
      }
      switch (startIndex) {
        case 0: return func.call(this, rest);
        case 1: return func.call(this, arguments[0], rest);
        case 2: return func.call(this, arguments[0], arguments[1], rest);
      }
      var args = Array(startIndex + 1);
      for (index = 0; index < startIndex; index++) {
        args[index] = arguments[index];
      }
      args[startIndex] = rest;
      return func.apply(this, args);
    };
  }

  // An internal function for creating a new object that inherits from another.
  function baseCreate(prototype) {
    if (!isObject$1(prototype)) return {};
    if (nativeCreate$1) return nativeCreate$1(prototype);
    Ctor.prototype = prototype;
    var result = new Ctor;
    Ctor.prototype = null;
    return result;
  }

  function shallowProperty(key) {
    return function(obj) {
      return obj == null ? void 0 : obj[key];
    };
  }

  function _has(obj, path) {
    return obj != null && hasOwnProperty$a.call(obj, path);
  }

  function deepGet(obj, path) {
    var length = path.length;
    for (var i = 0; i < length; i++) {
      if (obj == null) return void 0;
      obj = obj[path[i]];
    }
    return length ? obj : void 0;
  }

  // Helper for collection methods to determine whether a collection
  // should be iterated as an array or as an object.
  // Related: https://people.mozilla.org/~jorendorff/es6-draft.html#sec-tolength
  // Avoids a very nasty iOS 8 JIT bug on ARM-64. #2094
  var MAX_ARRAY_INDEX = Math.pow(2, 53) - 1;
  var getLength = shallowProperty('length');
  function isArrayLike$1(collection) {
    var length = getLength(collection);
    return typeof length == 'number' && length >= 0 && length <= MAX_ARRAY_INDEX;
  }

  // Collection Functions
  // --------------------

  // The cornerstone, an `each` implementation, aka `forEach`.
  // Handles raw objects in addition to array-likes. Treats all
  // sparse array-likes as if they were dense.
  function each(obj, iteratee, context) {
    iteratee = optimizeCb(iteratee, context);
    var i, length;
    if (isArrayLike$1(obj)) {
      for (i = 0, length = obj.length; i < length; i++) {
        iteratee(obj[i], i, obj);
      }
    } else {
      var _keys = keys$1(obj);
      for (i = 0, length = _keys.length; i < length; i++) {
        iteratee(obj[_keys[i]], _keys[i], obj);
      }
    }
    return obj;
  }

  // Return the results of applying the iteratee to each element.
  function map(obj, iteratee, context) {
    iteratee = cb(iteratee, context);
    var _keys = !isArrayLike$1(obj) && keys$1(obj),
        length = (_keys || obj).length,
        results = Array(length);
    for (var index = 0; index < length; index++) {
      var currentKey = _keys ? _keys[index] : index;
      results[index] = iteratee(obj[currentKey], currentKey, obj);
    }
    return results;
  }

  // Create a reducing function iterating left or right.
  function createReduce(dir) {
    // Wrap code that reassigns argument variables in a separate function than
    // the one that accesses `arguments.length` to avoid a perf hit. (#1991)
    var reducer = function(obj, iteratee, memo, initial) {
      var _keys = !isArrayLike$1(obj) && keys$1(obj),
          length = (_keys || obj).length,
          index = dir > 0 ? 0 : length - 1;
      if (!initial) {
        memo = obj[_keys ? _keys[index] : index];
        index += dir;
      }
      for (; index >= 0 && index < length; index += dir) {
        var currentKey = _keys ? _keys[index] : index;
        memo = iteratee(memo, obj[currentKey], currentKey, obj);
      }
      return memo;
    };

    return function(obj, iteratee, memo, context) {
      var initial = arguments.length >= 3;
      return reducer(obj, optimizeCb(iteratee, context, 4), memo, initial);
    };
  }

  // **Reduce** builds up a single result from a list of values, aka `inject`,
  // or `foldl`.
  var reduce = createReduce(1);

  // The right-associative version of reduce, also known as `foldr`.
  var reduceRight = createReduce(-1);

  // Return the first value which passes a truth test.
  function find(obj, predicate, context) {
    var keyFinder = isArrayLike$1(obj) ? findIndex : findKey;
    var key = keyFinder(obj, predicate, context);
    if (key !== void 0 && key !== -1) return obj[key];
  }

  // Return all the elements that pass a truth test.
  function filter(obj, predicate, context) {
    var results = [];
    predicate = cb(predicate, context);
    each(obj, function(value, index, list) {
      if (predicate(value, index, list)) results.push(value);
    });
    return results;
  }

  // Return all the elements for which a truth test fails.
  function reject$1(obj, predicate, context) {
    return filter(obj, negate(cb(predicate)), context);
  }

  // Determine whether all of the elements match a truth test.
  function every(obj, predicate, context) {
    predicate = cb(predicate, context);
    var _keys = !isArrayLike$1(obj) && keys$1(obj),
        length = (_keys || obj).length;
    for (var index = 0; index < length; index++) {
      var currentKey = _keys ? _keys[index] : index;
      if (!predicate(obj[currentKey], currentKey, obj)) return false;
    }
    return true;
  }

  // Determine if at least one element in the object matches a truth test.
  function some(obj, predicate, context) {
    predicate = cb(predicate, context);
    var _keys = !isArrayLike$1(obj) && keys$1(obj),
        length = (_keys || obj).length;
    for (var index = 0; index < length; index++) {
      var currentKey = _keys ? _keys[index] : index;
      if (predicate(obj[currentKey], currentKey, obj)) return true;
    }
    return false;
  }

  // Determine if the array or object contains a given item (using `===`).
  function contains(obj, item, fromIndex, guard) {
    if (!isArrayLike$1(obj)) obj = values(obj);
    if (typeof fromIndex != 'number' || guard) fromIndex = 0;
    return indexOf(obj, item, fromIndex) >= 0;
  }

  // Invoke a method (with arguments) on every item in a collection.
  var invoke = restArguments(function(obj, path, args) {
    var contextPath, func;
    if (isFunction$1(path)) {
      func = path;
    } else if (isArray$1(path)) {
      contextPath = path.slice(0, -1);
      path = path[path.length - 1];
    }
    return map(obj, function(context) {
      var method = func;
      if (!method) {
        if (contextPath && contextPath.length) {
          context = deepGet(context, contextPath);
        }
        if (context == null) return void 0;
        method = context[path];
      }
      return method == null ? method : method.apply(context, args);
    });
  });

  // Convenience version of a common use case of `map`: fetching a property.
  function pluck(obj, key) {
    return map(obj, property(key));
  }

  // Convenience version of a common use case of `filter`: selecting only objects
  // containing specific `key:value` pairs.
  function where(obj, attrs) {
    return filter(obj, matcher(attrs));
  }

  // Convenience version of a common use case of `find`: getting the first object
  // containing specific `key:value` pairs.
  function findWhere(obj, attrs) {
    return find(obj, matcher(attrs));
  }

  // Return the maximum element (or element-based computation).
  function max(obj, iteratee, context) {
    var result = -Infinity, lastComputed = -Infinity,
        value, computed;
    if (iteratee == null || typeof iteratee == 'number' && typeof obj[0] != 'object' && obj != null) {
      obj = isArrayLike$1(obj) ? obj : values(obj);
      for (var i = 0, length = obj.length; i < length; i++) {
        value = obj[i];
        if (value != null && value > result) {
          result = value;
        }
      }
    } else {
      iteratee = cb(iteratee, context);
      each(obj, function(v, index, list) {
        computed = iteratee(v, index, list);
        if (computed > lastComputed || computed === -Infinity && result === -Infinity) {
          result = v;
          lastComputed = computed;
        }
      });
    }
    return result;
  }

  // Return the minimum element (or element-based computation).
  function min(obj, iteratee, context) {
    var result = Infinity, lastComputed = Infinity,
        value, computed;
    if (iteratee == null || typeof iteratee == 'number' && typeof obj[0] != 'object' && obj != null) {
      obj = isArrayLike$1(obj) ? obj : values(obj);
      for (var i = 0, length = obj.length; i < length; i++) {
        value = obj[i];
        if (value != null && value < result) {
          result = value;
        }
      }
    } else {
      iteratee = cb(iteratee, context);
      each(obj, function(v, index, list) {
        computed = iteratee(v, index, list);
        if (computed < lastComputed || computed === Infinity && result === Infinity) {
          result = v;
          lastComputed = computed;
        }
      });
    }
    return result;
  }

  // Shuffle a collection.
  function shuffle(obj) {
    return sample(obj, Infinity);
  }

  // Sample **n** random values from a collection using the modern version of the
  // [Fisher-Yates shuffle](https://en.wikipedia.org/wiki/Fisher–Yates_shuffle).
  // If **n** is not specified, returns a single random element.
  // The internal `guard` argument allows it to work with `map`.
  function sample(obj, n, guard) {
    if (n == null || guard) {
      if (!isArrayLike$1(obj)) obj = values(obj);
      return obj[random(obj.length - 1)];
    }
    var sample = isArrayLike$1(obj) ? clone(obj) : values(obj);
    var length = getLength(sample);
    n = Math.max(Math.min(n, length), 0);
    var last = length - 1;
    for (var index = 0; index < n; index++) {
      var rand = random(index, last);
      var temp = sample[index];
      sample[index] = sample[rand];
      sample[rand] = temp;
    }
    return sample.slice(0, n);
  }

  // Sort the object's values by a criterion produced by an iteratee.
  function sortBy(obj, iteratee, context) {
    var index = 0;
    iteratee = cb(iteratee, context);
    return pluck(map(obj, function(value, key, list) {
      return {
        value: value,
        index: index++,
        criteria: iteratee(value, key, list)
      };
    }).sort(function(left, right) {
      var a = left.criteria;
      var b = right.criteria;
      if (a !== b) {
        if (a > b || a === void 0) return 1;
        if (a < b || b === void 0) return -1;
      }
      return left.index - right.index;
    }), 'value');
  }

  // An internal function used for aggregate "group by" operations.
  function group(behavior, partition) {
    return function(obj, iteratee, context) {
      var result = partition ? [[], []] : {};
      iteratee = cb(iteratee, context);
      each(obj, function(value, index) {
        var key = iteratee(value, index, obj);
        behavior(result, value, key);
      });
      return result;
    };
  }

  // Groups the object's values by a criterion. Pass either a string attribute
  // to group by, or a function that returns the criterion.
  var groupBy = group(function(result, value, key) {
    if (_has(result, key)) result[key].push(value); else result[key] = [value];
  });

  // Indexes the object's values by a criterion, similar to `groupBy`, but for
  // when you know that your index values will be unique.
  var indexBy = group(function(result, value, key) {
    result[key] = value;
  });

  // Counts instances of an object that group by a certain criterion. Pass
  // either a string attribute to count by, or a function that returns the
  // criterion.
  var countBy = group(function(result, value, key) {
    if (_has(result, key)) result[key]++; else result[key] = 1;
  });

  var reStrSymbol = /[^\ud800-\udfff]|[\ud800-\udbff][\udc00-\udfff]|[\ud800-\udfff]/g;
  // Safely create a real, live array from anything iterable.
  function toArray(obj) {
    if (!obj) return [];
    if (isArray$1(obj)) return slice.call(obj);
    if (isString(obj)) {
      // Keep surrogate pair characters together
      return obj.match(reStrSymbol);
    }
    if (isArrayLike$1(obj)) return map(obj, identity);
    return values(obj);
  }

  // Return the number of elements in an object.
  function size(obj) {
    if (obj == null) return 0;
    return isArrayLike$1(obj) ? obj.length : keys$1(obj).length;
  }

  // Split a collection into two arrays: one whose elements all satisfy the given
  // predicate, and one whose elements all do not satisfy the predicate.
  var partition = group(function(result, value, pass) {
    result[pass ? 0 : 1].push(value);
  }, true);

  // Array Functions
  // ---------------

  // Get the first element of an array. Passing **n** will return the first N
  // values in the array. The **guard** check allows it to work with `map`.
  function first(array, n, guard) {
    if (array == null || array.length < 1) return n == null ? void 0 : [];
    if (n == null || guard) return array[0];
    return initial(array, array.length - n);
  }

  // Returns everything but the last entry of the array. Especially useful on
  // the arguments object. Passing **n** will return all the values in
  // the array, excluding the last N.
  function initial(array, n, guard) {
    return slice.call(array, 0, Math.max(0, array.length - (n == null || guard ? 1 : n)));
  }

  // Get the last element of an array. Passing **n** will return the last N
  // values in the array.
  function last(array, n, guard) {
    if (array == null || array.length < 1) return n == null ? void 0 : [];
    if (n == null || guard) return array[array.length - 1];
    return rest(array, Math.max(0, array.length - n));
  }

  // Returns everything but the first entry of the array. Especially useful on
  // the arguments object. Passing an **n** will return the rest N values in the
  // array.
  function rest(array, n, guard) {
    return slice.call(array, n == null || guard ? 1 : n);
  }

  // Trim out all falsy values from an array.
  function compact(array) {
    return filter(array, Boolean);
  }

  // Internal implementation of a recursive `flatten` function.
  function _flatten(input, shallow, strict, output) {
    output = output || [];
    var idx = output.length;
    for (var i = 0, length = getLength(input); i < length; i++) {
      var value = input[i];
      if (isArrayLike$1(value) && (isArray$1(value) || isArguments$1(value))) {
        // Flatten current level of array or arguments object.
        if (shallow) {
          var j = 0, len = value.length;
          while (j < len) output[idx++] = value[j++];
        } else {
          _flatten(value, shallow, strict, output);
          idx = output.length;
        }
      } else if (!strict) {
        output[idx++] = value;
      }
    }
    return output;
  }

  // Flatten out an array, either recursively (by default), or just one level.
  function flatten(array, shallow) {
    return _flatten(array, shallow, false);
  }

  // Return a version of the array that does not contain the specified value(s).
  var without = restArguments(function(array, otherArrays) {
    return difference$1(array, otherArrays);
  });

  // Produce a duplicate-free version of the array. If the array has already
  // been sorted, you have the option of using a faster algorithm.
  // The faster algorithm will not work with an iteratee if the iteratee
  // is not a one-to-one function, so providing an iteratee will disable
  // the faster algorithm.
  function uniq(array, isSorted, iteratee, context) {
    if (!isBoolean(isSorted)) {
      context = iteratee;
      iteratee = isSorted;
      isSorted = false;
    }
    if (iteratee != null) iteratee = cb(iteratee, context);
    var result = [];
    var seen = [];
    for (var i = 0, length = getLength(array); i < length; i++) {
      var value = array[i],
          computed = iteratee ? iteratee(value, i, array) : value;
      if (isSorted && !iteratee) {
        if (!i || seen !== computed) result.push(value);
        seen = computed;
      } else if (iteratee) {
        if (!contains(seen, computed)) {
          seen.push(computed);
          result.push(value);
        }
      } else if (!contains(result, value)) {
        result.push(value);
      }
    }
    return result;
  }

  // Produce an array that contains the union: each distinct element from all of
  // the passed-in arrays.
  var union = restArguments(function(arrays) {
    return uniq(_flatten(arrays, true, true));
  });

  // Produce an array that contains every item shared between all the
  // passed-in arrays.
  function intersection(array) {
    var result = [];
    var argsLength = arguments.length;
    for (var i = 0, length = getLength(array); i < length; i++) {
      var item = array[i];
      if (contains(result, item)) continue;
      var j;
      for (j = 1; j < argsLength; j++) {
        if (!contains(arguments[j], item)) break;
      }
      if (j === argsLength) result.push(item);
    }
    return result;
  }

  // Take the difference between one array and a number of other arrays.
  // Only the elements present in just the first array will remain.
  var difference$1 = restArguments(function(array, rest) {
    rest = _flatten(rest, true, true);
    return filter(array, function(value){
      return !contains(rest, value);
    });
  });

  // Complement of zip. Unzip accepts an array of arrays and groups
  // each array's elements on shared indices.
  function unzip(array) {
    var length = array && max(array, getLength).length || 0;
    var result = Array(length);

    for (var index = 0; index < length; index++) {
      result[index] = pluck(array, index);
    }
    return result;
  }

  // Zip together multiple lists into a single array -- elements that share
  // an index go together.
  var zip = restArguments(unzip);

  // Converts lists into objects. Pass either a single array of `[key, value]`
  // pairs, or two parallel arrays of the same length -- one of keys, and one of
  // the corresponding values. Passing by pairs is the reverse of pairs.
  function object(list, values) {
    var result = {};
    for (var i = 0, length = getLength(list); i < length; i++) {
      if (values) {
        result[list[i]] = values[i];
      } else {
        result[list[i][0]] = list[i][1];
      }
    }
    return result;
  }

  // Generator function to create the findIndex and findLastIndex functions.
  function createPredicateIndexFinder(dir) {
    return function(array, predicate, context) {
      predicate = cb(predicate, context);
      var length = getLength(array);
      var index = dir > 0 ? 0 : length - 1;
      for (; index >= 0 && index < length; index += dir) {
        if (predicate(array[index], index, array)) return index;
      }
      return -1;
    };
  }

  // Returns the first index on an array-like that passes a predicate test.
  var findIndex = createPredicateIndexFinder(1);
  var findLastIndex = createPredicateIndexFinder(-1);

  // Use a comparator function to figure out the smallest index at which
  // an object should be inserted so as to maintain order. Uses binary search.
  function sortedIndex(array, obj, iteratee, context) {
    iteratee = cb(iteratee, context, 1);
    var value = iteratee(obj);
    var low = 0, high = getLength(array);
    while (low < high) {
      var mid = Math.floor((low + high) / 2);
      if (iteratee(array[mid]) < value) low = mid + 1; else high = mid;
    }
    return low;
  }

  // Generator function to create the indexOf and lastIndexOf functions.
  function createIndexFinder(dir, predicateFind, sortedIndex) {
    return function(array, item, idx) {
      var i = 0, length = getLength(array);
      if (typeof idx == 'number') {
        if (dir > 0) {
          i = idx >= 0 ? idx : Math.max(idx + length, i);
        } else {
          length = idx >= 0 ? Math.min(idx + 1, length) : idx + length + 1;
        }
      } else if (sortedIndex && idx && length) {
        idx = sortedIndex(array, item);
        return array[idx] === item ? idx : -1;
      }
      if (item !== item) {
        idx = predicateFind(slice.call(array, i, length), isNaN$1);
        return idx >= 0 ? idx + i : -1;
      }
      for (idx = dir > 0 ? i : length - 1; idx >= 0 && idx < length; idx += dir) {
        if (array[idx] === item) return idx;
      }
      return -1;
    };
  }

  // Return the position of the first occurrence of an item in an array,
  // or -1 if the item is not included in the array.
  // If the array is large and already in sort order, pass `true`
  // for **isSorted** to use binary search.
  var indexOf = createIndexFinder(1, findIndex, sortedIndex);
  var lastIndexOf = createIndexFinder(-1, findLastIndex);

  // Generate an integer Array containing an arithmetic progression. A port of
  // the native Python `range()` function. See
  // [the Python documentation](https://docs.python.org/library/functions.html#range).
  function range(start, stop, step) {
    if (stop == null) {
      stop = start || 0;
      start = 0;
    }
    if (!step) {
      step = stop < start ? -1 : 1;
    }

    var length = Math.max(Math.ceil((stop - start) / step), 0);
    var range = Array(length);

    for (var idx = 0; idx < length; idx++, start += step) {
      range[idx] = start;
    }

    return range;
  }

  // Chunk a single array into multiple arrays, each containing `count` or fewer
  // items.
  function chunk(array, count) {
    if (count == null || count < 1) return [];
    var result = [];
    var i = 0, length = array.length;
    while (i < length) {
      result.push(slice.call(array, i, i += count));
    }
    return result;
  }

  // Function (ahem) Functions
  // ------------------

  // Determines whether to execute a function as a constructor
  // or a normal function with the provided arguments.
  function executeBound(sourceFunc, boundFunc, context, callingContext, args) {
    if (!(callingContext instanceof boundFunc)) return sourceFunc.apply(context, args);
    var self = baseCreate(sourceFunc.prototype);
    var result = sourceFunc.apply(self, args);
    if (isObject$1(result)) return result;
    return self;
  }

  // Create a function bound to a given object (assigning `this`, and arguments,
  // optionally). Delegates to **ECMAScript 5**'s native `Function.bind` if
  // available.
  var bind = restArguments(function(func, context, args) {
    if (!isFunction$1(func)) throw new TypeError('Bind must be called on a function');
    var bound = restArguments(function(callArgs) {
      return executeBound(func, bound, context, this, args.concat(callArgs));
    });
    return bound;
  });

  // Partially apply a function by creating a version that has had some of its
  // arguments pre-filled, without changing its dynamic `this` context. _ acts
  // as a placeholder by default, allowing any combination of arguments to be
  // pre-filled. Set `partial.placeholder` for a custom placeholder argument.
  var partial = restArguments(function(func, boundArgs) {
    var placeholder = partial.placeholder;
    var bound = function() {
      var position = 0, length = boundArgs.length;
      var args = Array(length);
      for (var i = 0; i < length; i++) {
        args[i] = boundArgs[i] === placeholder ? arguments[position++] : boundArgs[i];
      }
      while (position < arguments.length) args.push(arguments[position++]);
      return executeBound(func, bound, this, this, args);
    };
    return bound;
  });

  partial.placeholder = _;

  // Bind a number of an object's methods to that object. Remaining arguments
  // are the method names to be bound. Useful for ensuring that all callbacks
  // defined on an object belong to it.
  var bindAll = restArguments(function(obj, _keys) {
    _keys = _flatten(_keys, false, false);
    var index = _keys.length;
    if (index < 1) throw new Error('bindAll must be passed function names');
    while (index--) {
      var key = _keys[index];
      obj[key] = bind(obj[key], obj);
    }
  });

  // Memoize an expensive function by storing its results.
  function memoize(func, hasher) {
    var memoize = function(key) {
      var cache = memoize.cache;
      var address = '' + (hasher ? hasher.apply(this, arguments) : key);
      if (!_has(cache, address)) cache[address] = func.apply(this, arguments);
      return cache[address];
    };
    memoize.cache = {};
    return memoize;
  }

  // Delays a function for the given number of milliseconds, and then calls
  // it with the arguments supplied.
  var delay = restArguments(function(func, wait, args) {
    return setTimeout(function() {
      return func.apply(null, args);
    }, wait);
  });

  // Defers a function, scheduling it to run after the current call stack has
  // cleared.
  var defer = partial(delay, _, 1);

  // Returns a function, that, when invoked, will only be triggered at most once
  // during a given window of time. Normally, the throttled function will run
  // as much as it can, without ever going more than once per `wait` duration;
  // but if you'd like to disable the execution on the leading edge, pass
  // `{leading: false}`. To disable execution on the trailing edge, ditto.
  function throttle(func, wait, options) {
    var timeout, context, args, result;
    var previous = 0;
    if (!options) options = {};

    var later = function() {
      previous = options.leading === false ? 0 : now();
      timeout = null;
      result = func.apply(context, args);
      if (!timeout) context = args = null;
    };

    var throttled = function() {
      var _now = now();
      if (!previous && options.leading === false) previous = _now;
      var remaining = wait - (_now - previous);
      context = this;
      args = arguments;
      if (remaining <= 0 || remaining > wait) {
        if (timeout) {
          clearTimeout(timeout);
          timeout = null;
        }
        previous = _now;
        result = func.apply(context, args);
        if (!timeout) context = args = null;
      } else if (!timeout && options.trailing !== false) {
        timeout = setTimeout(later, remaining);
      }
      return result;
    };

    throttled.cancel = function() {
      clearTimeout(timeout);
      previous = 0;
      timeout = context = args = null;
    };

    return throttled;
  }

  // Returns a function, that, as long as it continues to be invoked, will not
  // be triggered. The function will be called after it stops being called for
  // N milliseconds. If `immediate` is passed, trigger the function on the
  // leading edge, instead of the trailing.
  function debounce(func, wait, immediate) {
    var timeout, result;

    var later = function(context, args) {
      timeout = null;
      if (args) result = func.apply(context, args);
    };

    var debounced = restArguments(function(args) {
      if (timeout) clearTimeout(timeout);
      if (immediate) {
        var callNow = !timeout;
        timeout = setTimeout(later, wait);
        if (callNow) result = func.apply(this, args);
      } else {
        timeout = delay(later, wait, this, args);
      }

      return result;
    });

    debounced.cancel = function() {
      clearTimeout(timeout);
      timeout = null;
    };

    return debounced;
  }

  // Returns the first function passed as an argument to the second,
  // allowing you to adjust arguments, run code before and after, and
  // conditionally execute the original function.
  function wrap(func, wrapper) {
    return partial(wrapper, func);
  }

  // Returns a negated version of the passed-in predicate.
  function negate(predicate) {
    return function() {
      return !predicate.apply(this, arguments);
    };
  }

  // Returns a function that is the composition of a list of functions, each
  // consuming the return value of the function that follows.
  function compose() {
    var args = arguments;
    var start = args.length - 1;
    return function() {
      var i = start;
      var result = args[start].apply(this, arguments);
      while (i--) result = args[i].call(this, result);
      return result;
    };
  }

  // Returns a function that will only be executed on and after the Nth call.
  function after(times, func) {
    return function() {
      if (--times < 1) {
        return func.apply(this, arguments);
      }
    };
  }

  // Returns a function that will only be executed up to (but not including) the Nth call.
  function before(times, func) {
    var memo;
    return function() {
      if (--times > 0) {
        memo = func.apply(this, arguments);
      }
      if (times <= 1) func = null;
      return memo;
    };
  }

  // Returns a function that will be executed at most one time, no matter how
  // often you call it. Useful for lazy initialization.
  var once = partial(before, 2);

  // Object Functions
  // ----------------

  // Keys in IE < 9 that won't be iterated by `for key in ...` and thus missed.
  var hasEnumBug = !{toString: null}.propertyIsEnumerable('toString');
  var nonEnumerableProps = ['valueOf', 'isPrototypeOf', 'toString',
    'propertyIsEnumerable', 'hasOwnProperty', 'toLocaleString'];

  function collectNonEnumProps(obj, _keys) {
    var nonEnumIdx = nonEnumerableProps.length;
    var constructor = obj.constructor;
    var proto = isFunction$1(constructor) && constructor.prototype || ObjProto;

    // Constructor is a special case.
    var prop = 'constructor';
    if (_has(obj, prop) && !contains(_keys, prop)) _keys.push(prop);

    while (nonEnumIdx--) {
      prop = nonEnumerableProps[nonEnumIdx];
      if (prop in obj && obj[prop] !== proto[prop] && !contains(_keys, prop)) {
        _keys.push(prop);
      }
    }
  }

  // Retrieve the names of an object's own properties.
  // Delegates to **ECMAScript 5**'s native `Object.keys`.
  function keys$1(obj) {
    if (!isObject$1(obj)) return [];
    if (nativeKeys$1) return nativeKeys$1(obj);
    var _keys = [];
    for (var key in obj) if (_has(obj, key)) _keys.push(key);
    // Ahem, IE < 9.
    if (hasEnumBug) collectNonEnumProps(obj, _keys);
    return _keys;
  }

  // Retrieve all the property names of an object.
  function allKeys(obj) {
    if (!isObject$1(obj)) return [];
    var _keys = [];
    for (var key in obj) _keys.push(key);
    // Ahem, IE < 9.
    if (hasEnumBug) collectNonEnumProps(obj, _keys);
    return _keys;
  }

  // Retrieve the values of an object's properties.
  function values(obj) {
    var _keys = keys$1(obj);
    var length = _keys.length;
    var values = Array(length);
    for (var i = 0; i < length; i++) {
      values[i] = obj[_keys[i]];
    }
    return values;
  }

  // Returns the results of applying the iteratee to each element of the object.
  // In contrast to map it returns an object.
  function mapObject(obj, iteratee, context) {
    iteratee = cb(iteratee, context);
    var _keys = keys$1(obj),
        length = _keys.length,
        results = {};
    for (var index = 0; index < length; index++) {
      var currentKey = _keys[index];
      results[currentKey] = iteratee(obj[currentKey], currentKey, obj);
    }
    return results;
  }

  // Convert an object into a list of `[key, value]` pairs.
  // The opposite of object.
  function pairs(obj) {
    var _keys = keys$1(obj);
    var length = _keys.length;
    var pairs = Array(length);
    for (var i = 0; i < length; i++) {
      pairs[i] = [_keys[i], obj[_keys[i]]];
    }
    return pairs;
  }

  // Invert the keys and values of an object. The values must be serializable.
  function invert(obj) {
    var result = {};
    var _keys = keys$1(obj);
    for (var i = 0, length = _keys.length; i < length; i++) {
      result[obj[_keys[i]]] = _keys[i];
    }
    return result;
  }

  // Return a sorted list of the function names available on the object.
  function functions(obj) {
    var names = [];
    for (var key in obj) {
      if (isFunction$1(obj[key])) names.push(key);
    }
    return names.sort();
  }

  // An internal function for creating assigner functions.
  function createAssigner(keysFunc, defaults) {
    return function(obj) {
      var length = arguments.length;
      if (defaults) obj = Object(obj);
      if (length < 2 || obj == null) return obj;
      for (var index = 1; index < length; index++) {
        var source = arguments[index],
            _keys = keysFunc(source),
            l = _keys.length;
        for (var i = 0; i < l; i++) {
          var key = _keys[i];
          if (!defaults || obj[key] === void 0) obj[key] = source[key];
        }
      }
      return obj;
    };
  }

  // Extend a given object with all the properties in passed-in object(s).
  var extend = createAssigner(allKeys);

  // Assigns a given object with all the own properties in the passed-in object(s).
  // (https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/Object/assign)
  var extendOwn = createAssigner(keys$1);

  // Returns the first key on an object that passes a predicate test.
  function findKey(obj, predicate, context) {
    predicate = cb(predicate, context);
    var _keys = keys$1(obj), key;
    for (var i = 0, length = _keys.length; i < length; i++) {
      key = _keys[i];
      if (predicate(obj[key], key, obj)) return key;
    }
  }

  // Internal pick helper function to determine if `obj` has key `key`.
  function keyInObj(value, key, obj) {
    return key in obj;
  }

  // Return a copy of the object only containing the whitelisted properties.
  var pick = restArguments(function(obj, _keys) {
    var result = {}, iteratee = _keys[0];
    if (obj == null) return result;
    if (isFunction$1(iteratee)) {
      if (_keys.length > 1) iteratee = optimizeCb(iteratee, _keys[1]);
      _keys = allKeys(obj);
    } else {
      iteratee = keyInObj;
      _keys = _flatten(_keys, false, false);
      obj = Object(obj);
    }
    for (var i = 0, length = _keys.length; i < length; i++) {
      var key = _keys[i];
      var value = obj[key];
      if (iteratee(value, key, obj)) result[key] = value;
    }
    return result;
  });

  // Return a copy of the object without the blacklisted properties.
  var omit = restArguments(function(obj, _keys) {
    var iteratee = _keys[0], context;
    if (isFunction$1(iteratee)) {
      iteratee = negate(iteratee);
      if (_keys.length > 1) context = _keys[1];
    } else {
      _keys = map(_flatten(_keys, false, false), String);
      iteratee = function(value, key) {
        return !contains(_keys, key);
      };
    }
    return pick(obj, iteratee, context);
  });

  // Fill in a given object with default properties.
  var defaults = createAssigner(allKeys, true);

  // Creates an object that inherits from the given prototype object.
  // If additional properties are provided then they will be added to the
  // created object.
  function create(prototype, props) {
    var result = baseCreate(prototype);
    if (props) extendOwn(result, props);
    return result;
  }

  // Create a (shallow-cloned) duplicate of an object.
  function clone(obj) {
    if (!isObject$1(obj)) return obj;
    return isArray$1(obj) ? obj.slice() : extend({}, obj);
  }

  // Invokes interceptor with the obj, and then returns obj.
  // The primary purpose of this method is to "tap into" a method chain, in
  // order to perform operations on intermediate results within the chain.
  function tap(obj, interceptor) {
    interceptor(obj);
    return obj;
  }

  // Returns whether an object has a given set of `key:value` pairs.
  function isMatch(object, attrs) {
    var _keys = keys$1(attrs), length = _keys.length;
    if (object == null) return !length;
    var obj = Object(object);
    for (var i = 0; i < length; i++) {
      var key = _keys[i];
      if (attrs[key] !== obj[key] || !(key in obj)) return false;
    }
    return true;
  }


  // Internal recursive comparison function for `isEqual`.
  function eq$1(a, b, aStack, bStack) {
    // Identical objects are equal. `0 === -0`, but they aren't identical.
    // See the [Harmony `egal` proposal](https://wiki.ecmascript.org/doku.php?id=harmony:egal).
    if (a === b) return a !== 0 || 1 / a === 1 / b;
    // `null` or `undefined` only equal to itself (strict comparison).
    if (a == null || b == null) return false;
    // `NaN`s are equivalent, but non-reflexive.
    if (a !== a) return b !== b;
    // Exhaust primitive checks
    var type = typeof a;
    if (type !== 'function' && type !== 'object' && typeof b != 'object') return false;
    return deepEq(a, b, aStack, bStack);
  }

  // Internal recursive comparison function for `isEqual`.
  function deepEq(a, b, aStack, bStack) {
    // Unwrap any wrapped objects.
    if (a instanceof _) a = a._wrapped;
    if (b instanceof _) b = b._wrapped;
    // Compare `[[Class]]` names.
    var className = toString.call(a);
    if (className !== toString.call(b)) return false;
    switch (className) {
      // Strings, numbers, regular expressions, dates, and booleans are compared by value.
      case '[object RegExp]':
      // RegExps are coerced to strings for comparison (Note: '' + /a/i === '/a/i')
      case '[object String]':
        // Primitives and their corresponding object wrappers are equivalent; thus, `"5"` is
        // equivalent to `new String("5")`.
        return '' + a === '' + b;
      case '[object Number]':
        // `NaN`s are equivalent, but non-reflexive.
        // Object(NaN) is equivalent to NaN.
        if (+a !== +a) return +b !== +b;
        // An `egal` comparison is performed for other numeric values.
        return +a === 0 ? 1 / +a === 1 / b : +a === +b;
      case '[object Date]':
      case '[object Boolean]':
        // Coerce dates and booleans to numeric primitive values. Dates are compared by their
        // millisecond representations. Note that invalid dates with millisecond representations
        // of `NaN` are not equivalent.
        return +a === +b;
      case '[object Symbol]':
        return SymbolProto.valueOf.call(a) === SymbolProto.valueOf.call(b);
    }

    var areArrays = className === '[object Array]';
    if (!areArrays) {
      if (typeof a != 'object' || typeof b != 'object') return false;

      // Objects with different constructors are not equivalent, but `Object`s or `Array`s
      // from different frames are.
      var aCtor = a.constructor, bCtor = b.constructor;
      if (aCtor !== bCtor && !(isFunction$1(aCtor) && aCtor instanceof aCtor &&
                               isFunction$1(bCtor) && bCtor instanceof bCtor)
                          && ('constructor' in a && 'constructor' in b)) {
        return false;
      }
    }
    // Assume equality for cyclic structures. The algorithm for detecting cyclic
    // structures is adapted from ES 5.1 section 15.12.3, abstract operation `JO`.

    // Initializing stack of traversed objects.
    // It's done here since we only need them for objects and arrays comparison.
    aStack = aStack || [];
    bStack = bStack || [];
    var length = aStack.length;
    while (length--) {
      // Linear search. Performance is inversely proportional to the number of
      // unique nested structures.
      if (aStack[length] === a) return bStack[length] === b;
    }

    // Add the first object to the stack of traversed objects.
    aStack.push(a);
    bStack.push(b);

    // Recursively compare objects and arrays.
    if (areArrays) {
      // Compare array lengths to determine if a deep comparison is necessary.
      length = a.length;
      if (length !== b.length) return false;
      // Deep compare the contents, ignoring non-numeric properties.
      while (length--) {
        if (!eq$1(a[length], b[length], aStack, bStack)) return false;
      }
    } else {
      // Deep compare objects.
      var _keys = keys$1(a), key;
      length = _keys.length;
      // Ensure that both objects contain the same number of properties before comparing deep equality.
      if (keys$1(b).length !== length) return false;
      while (length--) {
        // Deep compare each member
        key = _keys[length];
        if (!(_has(b, key) && eq$1(a[key], b[key], aStack, bStack))) return false;
      }
    }
    // Remove the first object from the stack of traversed objects.
    aStack.pop();
    bStack.pop();
    return true;
  }

  // Perform a deep comparison to check if two objects are equal.
  function isEqual$2(a, b) {
    return eq$1(a, b);
  }

  // Is a given array, string, or object empty?
  // An "empty" object has no enumerable own-properties.
  function isEmpty(obj) {
    if (obj == null) return true;
    if (isArrayLike$1(obj) && (isArray$1(obj) || isString(obj) || isArguments$1(obj))) return obj.length === 0;
    return keys$1(obj).length === 0;
  }

  // Is a given value a DOM element?
  function isElement(obj) {
    return !!(obj && obj.nodeType === 1);
  }

  // Internal function for creating a toString-based type tester.
  function tagTester(name) {
    return function(obj) {
      return toString.call(obj) === '[object ' + name + ']';
    };
  }

  // Is a given value an array?
  // Delegates to ECMA5's native Array.isArray
  var isArray$1 = nativeIsArray || tagTester('Array');

  // Is a given variable an object?
  function isObject$1(obj) {
    var type = typeof obj;
    return type === 'function' || type === 'object' && !!obj;
  }

  // Add some isType methods: isArguments, isFunction, isString, isNumber, isDate, isRegExp, isError, isMap, isWeakMap, isSet, isWeakSet.
  var isArguments$1 = tagTester('Arguments');
  var isFunction$1 = tagTester('Function');
  var isString = tagTester('String');
  var isNumber = tagTester('Number');
  var isDate = tagTester('Date');
  var isRegExp = tagTester('RegExp');
  var isError = tagTester('Error');
  var isSymbol = tagTester('Symbol');
  var isMap = tagTester('Map');
  var isWeakMap = tagTester('WeakMap');
  var isSet = tagTester('Set');
  var isWeakSet = tagTester('WeakSet');

  // Define a fallback version of the method in browsers (ahem, IE < 9), where
  // there isn't any inspectable "Arguments" type.
  (function() {
    if (!isArguments$1(arguments)) {
      isArguments$1 = function(obj) {
        return _has(obj, 'callee');
      };
    }
  }());

  // Optimize `isFunction` if appropriate. Work around some typeof bugs in old v8,
  // IE 11 (#1621), Safari 8 (#1929), and PhantomJS (#2236).
  var nodelist = root$1.document && root$1.document.childNodes;
  if (typeof Int8Array != 'object' && typeof nodelist != 'function') {
    isFunction$1 = function(obj) {
      return typeof obj == 'function' || false;
    };
  }

  // Is a given object a finite number?
  function isFinite(obj) {
    return !isSymbol(obj) && _isFinite(obj) && !_isNaN(parseFloat(obj));
  }

  // Is the given value `NaN`?
  function isNaN$1(obj) {
    return isNumber(obj) && _isNaN(obj);
  }

  // Is a given value a boolean?
  function isBoolean(obj) {
    return obj === true || obj === false || toString.call(obj) === '[object Boolean]';
  }

  // Is a given value equal to null?
  function isNull(obj) {
    return obj === null;
  }

  // Is a given variable undefined?
  function isUndefined(obj) {
    return obj === void 0;
  }

  // Shortcut function for checking if an object has a given property directly
  // on itself (in other words, not on a prototype).
  function has(obj, path) {
    if (!isArray$1(path)) {
      return _has(obj, path);
    }
    var length = path.length;
    for (var i = 0; i < length; i++) {
      var key = path[i];
      if (obj == null || !hasOwnProperty$a.call(obj, key)) {
        return false;
      }
      obj = obj[key];
    }
    return !!length;
  }

  // Utility Functions
  // -----------------

  // Keep the identity function around for default iteratees.
  function identity(value) {
    return value;
  }

  // Predicate-generating functions. Often useful outside of Underscore.
  function constant(value) {
    return function() {
      return value;
    };
  }

  function noop(){}

  // Creates a function that, when passed an object, will traverse that object’s
  // properties down the given `path`, specified as an array of keys or indexes.
  function property(path) {
    if (!isArray$1(path)) {
      return shallowProperty(path);
    }
    return function(obj) {
      return deepGet(obj, path);
    };
  }

  // Generates a function for a given object that returns a given property.
  function propertyOf(obj) {
    if (obj == null) {
      return function(){};
    }
    return function(path) {
      return !isArray$1(path) ? obj[path] : deepGet(obj, path);
    };
  }

  // Returns a predicate for checking whether an object has a given set of
  // `key:value` pairs.
  function matcher(attrs) {
    attrs = extendOwn({}, attrs);
    return function(obj) {
      return isMatch(obj, attrs);
    };
  }

  // Run a function **n** times.
  function times(n, iteratee, context) {
    var accum = Array(Math.max(0, n));
    iteratee = optimizeCb(iteratee, context, 1);
    for (var i = 0; i < n; i++) accum[i] = iteratee(i);
    return accum;
  }

  // Return a random integer between min and max (inclusive).
  function random(min, max) {
    if (max == null) {
      max = min;
      min = 0;
    }
    return min + Math.floor(Math.random() * (max - min + 1));
  }

  // A (possibly faster) way to get the current timestamp as an integer.
  var now = Date.now || function() {
    return new Date().getTime();
  };

  // List of HTML entities for escaping.
  var escapeMap = {
    '&': '&amp;',
    '<': '&lt;',
    '>': '&gt;',
    '"': '&quot;',
    "'": '&#x27;',
    '`': '&#x60;'
  };
  var unescapeMap = invert(escapeMap);

  // Functions for escaping and unescaping strings to/from HTML interpolation.
  function createEscaper(map) {
    var escaper = function(match) {
      return map[match];
    };
    // Regexes for identifying a key that needs to be escaped.
    var source = '(?:' + keys$1(map).join('|') + ')';
    var testRegexp = RegExp(source);
    var replaceRegexp = RegExp(source, 'g');
    return function(string) {
      string = string == null ? '' : '' + string;
      return testRegexp.test(string) ? string.replace(replaceRegexp, escaper) : string;
    };
  }
  var escape = createEscaper(escapeMap);
  var unescape = createEscaper(unescapeMap);

  // Traverses the children of `obj` along `path`. If a child is a function, it
  // is invoked with its parent as context. Returns the value of the final
  // child, or `fallback` if any child is undefined.
  function result(obj, path, fallback) {
    if (!isArray$1(path)) path = [path];
    var length = path.length;
    if (!length) {
      return isFunction$1(fallback) ? fallback.call(obj) : fallback;
    }
    for (var i = 0; i < length; i++) {
      var prop = obj == null ? void 0 : obj[path[i]];
      if (prop === void 0) {
        prop = fallback;
        i = length; // Ensure we don't continue iterating.
      }
      obj = isFunction$1(prop) ? prop.call(obj) : prop;
    }
    return obj;
  }

  // Generate a unique integer id (unique within the entire client session).
  // Useful for temporary DOM ids.
  var idCounter = 0;
  function uniqueId(prefix) {
    var id = ++idCounter + '';
    return prefix ? prefix + id : id;
  }

  // By default, Underscore uses ERB-style template delimiters, change the
  // following template settings to use alternative delimiters.
  var templateSettings = _.templateSettings = {
    evaluate: /<%([\s\S]+?)%>/g,
    interpolate: /<%=([\s\S]+?)%>/g,
    escape: /<%-([\s\S]+?)%>/g
  };

  // When customizing `templateSettings`, if you don't want to define an
  // interpolation, evaluation or escaping regex, we need one that is
  // guaranteed not to match.
  var noMatch = /(.)^/;

  // Certain characters need to be escaped so that they can be put into a
  // string literal.
  var escapes = {
    "'": "'",
    '\\': '\\',
    '\r': 'r',
    '\n': 'n',
    '\u2028': 'u2028',
    '\u2029': 'u2029'
  };

  var escapeRegExp = /\\|'|\r|\n|\u2028|\u2029/g;

  var escapeChar = function(match) {
    return '\\' + escapes[match];
  };

  // JavaScript micro-templating, similar to John Resig's implementation.
  // Underscore templating handles arbitrary delimiters, preserves whitespace,
  // and correctly escapes quotes within interpolated code.
  // NB: `oldSettings` only exists for backwards compatibility.
  function template(text, settings, oldSettings) {
    if (!settings && oldSettings) settings = oldSettings;
    settings = defaults({}, settings, _.templateSettings);

    // Combine delimiters into one regular expression via alternation.
    var matcher = RegExp([
      (settings.escape || noMatch).source,
      (settings.interpolate || noMatch).source,
      (settings.evaluate || noMatch).source
    ].join('|') + '|$', 'g');

    // Compile the template source, escaping string literals appropriately.
    var index = 0;
    var source = "__p+='";
    text.replace(matcher, function(match, escape, interpolate, evaluate, offset) {
      source += text.slice(index, offset).replace(escapeRegExp, escapeChar);
      index = offset + match.length;

      if (escape) {
        source += "'+\n((__t=(" + escape + "))==null?'':_.escape(__t))+\n'";
      } else if (interpolate) {
        source += "'+\n((__t=(" + interpolate + "))==null?'':__t)+\n'";
      } else if (evaluate) {
        source += "';\n" + evaluate + "\n__p+='";
      }

      // Adobe VMs need the match returned to produce the correct offset.
      return match;
    });
    source += "';\n";

    // If a variable is not specified, place data values in local scope.
    if (!settings.variable) source = 'with(obj||{}){\n' + source + '}\n';

    source = "var __t,__p='',__j=Array.prototype.join," +
      "print=function(){__p+=__j.call(arguments,'');};\n" +
      source + 'return __p;\n';

    var render;
    try {
      render = new Function(settings.variable || 'obj', '_', source);
    } catch (e) {
      e.source = source;
      throw e;
    }

    var template = function(data) {
      return render.call(this, data, _);
    };

    // Provide the compiled source as a convenience for precompilation.
    var argument = settings.variable || 'obj';
    template.source = 'function(' + argument + '){\n' + source + '}';

    return template;
  }

  // Add a "chain" function. Start chaining a wrapped Underscore object.
  function chain(obj) {
    var instance = _(obj);
    instance._chain = true;
    return instance;
  }

  // OOP
  // ---------------
  // If Underscore is called as a function, it returns a wrapped object that
  // can be used OO-style. This wrapper holds altered versions of all the
  // underscore functions. Wrapped objects may be chained.

  // Helper function to continue chaining intermediate results.
  function chainResult(instance, obj) {
    return instance._chain ? _(obj).chain() : obj;
  }

  // Add your own custom functions to the Underscore object.
  function mixin(obj) {
    each(functions(obj), function(name) {
      var func = _[name] = obj[name];
      _.prototype[name] = function() {
        var args = [this._wrapped];
        push.apply(args, arguments);
        return chainResult(this, func.apply(_, args));
      };
    });
    return _;
  }

  // Add all mutator Array functions to the wrapper.
  each(['pop', 'push', 'reverse', 'shift', 'sort', 'splice', 'unshift'], function(name) {
    var method = ArrayProto[name];
    _.prototype[name] = function() {
      var obj = this._wrapped;
      method.apply(obj, arguments);
      if ((name === 'shift' || name === 'splice') && obj.length === 0) delete obj[0];
      return chainResult(this, obj);
    };
  });

  // Add all accessor Array functions to the wrapper.
  each(['concat', 'join', 'slice'], function(name) {
    var method = ArrayProto[name];
    _.prototype[name] = function() {
      return chainResult(this, method.apply(this._wrapped, arguments));
    };
  });

  // Extracts the result from a wrapped and chained object.
  _.prototype.value = function() {
    return this._wrapped;
  };

  // Provide unwrapping proxy for some methods used in engine operations
  // such as arithmetic and JSON stringification.
  _.prototype.valueOf = _.prototype.toJSON = _.prototype.value;

  _.prototype.toString = function() {
    return String(this._wrapped);
  };

  var allExports = /*#__PURE__*/Object.freeze({
    'default': _,
    VERSION: VERSION,
    iteratee: iteratee,
    restArguments: restArguments,
    each: each,
    forEach: each,
    map: map,
    collect: map,
    reduce: reduce,
    foldl: reduce,
    inject: reduce,
    reduceRight: reduceRight,
    foldr: reduceRight,
    find: find,
    detect: find,
    filter: filter,
    select: filter,
    reject: reject$1,
    every: every,
    all: every,
    some: some,
    any: some,
    contains: contains,
    includes: contains,
    include: contains,
    invoke: invoke,
    pluck: pluck,
    where: where,
    findWhere: findWhere,
    max: max,
    min: min,
    shuffle: shuffle,
    sample: sample,
    sortBy: sortBy,
    groupBy: groupBy,
    indexBy: indexBy,
    countBy: countBy,
    toArray: toArray,
    size: size,
    partition: partition,
    first: first,
    head: first,
    take: first,
    initial: initial,
    last: last,
    rest: rest,
    tail: rest,
    drop: rest,
    compact: compact,
    flatten: flatten,
    without: without,
    uniq: uniq,
    unique: uniq,
    union: union,
    intersection: intersection,
    difference: difference$1,
    unzip: unzip,
    zip: zip,
    object: object,
    findIndex: findIndex,
    findLastIndex: findLastIndex,
    sortedIndex: sortedIndex,
    indexOf: indexOf,
    lastIndexOf: lastIndexOf,
    range: range,
    chunk: chunk,
    bind: bind,
    partial: partial,
    bindAll: bindAll,
    memoize: memoize,
    delay: delay,
    defer: defer,
    throttle: throttle,
    debounce: debounce,
    wrap: wrap,
    negate: negate,
    compose: compose,
    after: after,
    before: before,
    once: once,
    keys: keys$1,
    allKeys: allKeys,
    values: values,
    mapObject: mapObject,
    pairs: pairs,
    invert: invert,
    functions: functions,
    methods: functions,
    extend: extend,
    extendOwn: extendOwn,
    assign: extendOwn,
    findKey: findKey,
    pick: pick,
    omit: omit,
    defaults: defaults,
    create: create,
    clone: clone,
    tap: tap,
    isMatch: isMatch,
    isEqual: isEqual$2,
    isEmpty: isEmpty,
    isElement: isElement,
    isArray: isArray$1,
    isObject: isObject$1,
    get isArguments () { return isArguments$1; },
    get isFunction () { return isFunction$1; },
    isString: isString,
    isNumber: isNumber,
    isDate: isDate,
    isRegExp: isRegExp,
    isError: isError,
    isSymbol: isSymbol,
    isMap: isMap,
    isWeakMap: isWeakMap,
    isSet: isSet,
    isWeakSet: isWeakSet,
    isFinite: isFinite,
    isNaN: isNaN$1,
    isBoolean: isBoolean,
    isNull: isNull,
    isUndefined: isUndefined,
    has: has,
    identity: identity,
    constant: constant,
    noop: noop,
    property: property,
    propertyOf: propertyOf,
    matcher: matcher,
    matches: matcher,
    times: times,
    random: random,
    now: now,
    escape: escape,
    unescape: unescape,
    result: result,
    uniqueId: uniqueId,
    templateSettings: templateSettings,
    template: template,
    chain: chain,
    mixin: mixin
  });

  // Add all of the Underscore functions to the wrapper object.
  var _$1 = mixin(allExports);
  // Legacy Node.js API
  _$1._ = _$1;

  //     Backbone.js 1.2.3

  //     (c) 2010-2015 Jeremy Ashkenas, DocumentCloud and Investigative Reporters & Editors
  //     Backbone may be freely distributed under the MIT license.
  //     For all details and documentation:
  //     http://backbonejs.org

  (function(factory) {

    // Establish the root object, `window` (`self`) in the browser, or `global` on the server.
    // We use `self` instead of `window` for `WebWorker` support.
    var root = (typeof self == 'object' && self.self == self && self) ||
              (typeof global == 'object' && global.global == global && global);

    // Set up Backbone appropriately for the environment. Start with AMD.
    {
      var _ = _$1, $;
      try { $ = $$1; } catch(e) {}
      factory(root, exports, _, $);

    // Finally, as a browser global.
    }

  }(function(root, Backbone, _, $) {

    // Initial Setup
    // -------------

    // Save the previous value of the `Backbone` variable, so that it can be
    // restored later on, if `noConflict` is used.
    var previousBackbone = root.Backbone;

    // Create a local reference to a common array method we'll want to use later.
    var slice = Array.prototype.slice;

    // Current version of the library. Keep in sync with `package.json`.
    Backbone.VERSION = '1.2.3';

    // For Backbone's purposes, jQuery, Zepto, Ender, or My Library (kidding) owns
    // the `$` variable.
    Backbone.$ = $;

    // Runs Backbone.js in *noConflict* mode, returning the `Backbone` variable
    // to its previous owner. Returns a reference to this Backbone object.
    Backbone.noConflict = function() {
      root.Backbone = previousBackbone;
      return this;
    };

    // Turn on `emulateHTTP` to support legacy HTTP servers. Setting this option
    // will fake `"PATCH"`, `"PUT"` and `"DELETE"` requests via the `_method` parameter and
    // set a `X-Http-Method-Override` header.
    Backbone.emulateHTTP = false;

    // Turn on `emulateJSON` to support legacy servers that can't deal with direct
    // `application/json` requests ... this will encode the body as
    // `application/x-www-form-urlencoded` instead and will send the model in a
    // form param named `model`.
    Backbone.emulateJSON = false;

    // Proxy Backbone class methods to Underscore functions, wrapping the model's
    // `attributes` object or collection's `models` array behind the scenes.
    //
    // collection.filter(function(model) { return model.get('age') > 10 });
    // collection.each(this.addView);
    //
    // `Function#apply` can be slow so we use the method's arg count, if we know it.
    var addMethod = function(length, method, attribute) {
      switch (length) {
        case 1: return function() {
          return _[method](this[attribute]);
        };
        case 2: return function(value) {
          return _[method](this[attribute], value);
        };
        case 3: return function(iteratee, context) {
          return _[method](this[attribute], cb(iteratee, this), context);
        };
        case 4: return function(iteratee, defaultVal, context) {
          return _[method](this[attribute], cb(iteratee, this), defaultVal, context);
        };
        default: return function() {
          var args = slice.call(arguments);
          args.unshift(this[attribute]);
          return _[method].apply(_, args);
        };
      }
    };
    var addUnderscoreMethods = function(Class, methods, attribute) {
      _.each(methods, function(length, method) {
        if (_[method]) Class.prototype[method] = addMethod(length, method, attribute);
      });
    };

    // Support `collection.sortBy('attr')` and `collection.findWhere({id: 1})`.
    var cb = function(iteratee, instance) {
      if (_.isFunction(iteratee)) return iteratee;
      if (_.isObject(iteratee) && !instance._isModel(iteratee)) return modelMatcher(iteratee);
      if (_.isString(iteratee)) return function(model) { return model.get(iteratee); };
      return iteratee;
    };
    var modelMatcher = function(attrs) {
      var matcher = _.matches(attrs);
      return function(model) {
        return matcher(model.attributes);
      };
    };

    // Backbone.Events
    // ---------------

    // A module that can be mixed in to *any object* in order to provide it with
    // a custom event channel. You may bind a callback to an event with `on` or
    // remove with `off`; `trigger`-ing an event fires all callbacks in
    // succession.
    //
    //     var object = {};
    //     _.extend(object, Backbone.Events);
    //     object.on('expand', function(){ alert('expanded'); });
    //     object.trigger('expand');
    //
    var Events = Backbone.Events = {};

    // Regular expression used to split event strings.
    var eventSplitter = /\s+/;

    // Iterates over the standard `event, callback` (as well as the fancy multiple
    // space-separated events `"change blur", callback` and jQuery-style event
    // maps `{event: callback}`).
    var eventsApi = function(iteratee, events, name, callback, opts) {
      var i = 0, names;
      if (name && typeof name === 'object') {
        // Handle event maps.
        if (callback !== void 0 && 'context' in opts && opts.context === void 0) opts.context = callback;
        for (names = _.keys(name); i < names.length ; i++) {
          events = eventsApi(iteratee, events, names[i], name[names[i]], opts);
        }
      } else if (name && eventSplitter.test(name)) {
        // Handle space separated event names by delegating them individually.
        for (names = name.split(eventSplitter); i < names.length; i++) {
          events = iteratee(events, names[i], callback, opts);
        }
      } else {
        // Finally, standard events.
        events = iteratee(events, name, callback, opts);
      }
      return events;
    };

    // Bind an event to a `callback` function. Passing `"all"` will bind
    // the callback to all events fired.
    Events.on = function(name, callback, context) {
      return internalOn(this, name, callback, context);
    };

    // Guard the `listening` argument from the public API.
    var internalOn = function(obj, name, callback, context, listening) {
      obj._events = eventsApi(onApi, obj._events || {}, name, callback, {
          context: context,
          ctx: obj,
          listening: listening
      });

      if (listening) {
        var listeners = obj._listeners || (obj._listeners = {});
        listeners[listening.id] = listening;
      }

      return obj;
    };

    // Inversion-of-control versions of `on`. Tell *this* object to listen to
    // an event in another object... keeping track of what it's listening to
    // for easier unbinding later.
    Events.listenTo =  function(obj, name, callback) {
      if (!obj) return this;
      var id = obj._listenId || (obj._listenId = _.uniqueId('l'));
      var listeningTo = this._listeningTo || (this._listeningTo = {});
      var listening = listeningTo[id];

      // This object is not listening to any other events on `obj` yet.
      // Setup the necessary references to track the listening callbacks.
      if (!listening) {
        var thisId = this._listenId || (this._listenId = _.uniqueId('l'));
        listening = listeningTo[id] = {obj: obj, objId: id, id: thisId, listeningTo: listeningTo, count: 0};
      }

      // Bind callbacks on obj, and keep track of them on listening.
      internalOn(obj, name, callback, this, listening);
      return this;
    };

    // The reducing API that adds a callback to the `events` object.
    var onApi = function(events, name, callback, options) {
      if (callback) {
        var handlers = events[name] || (events[name] = []);
        var context = options.context, ctx = options.ctx, listening = options.listening;
        if (listening) listening.count++;

        handlers.push({ callback: callback, context: context, ctx: context || ctx, listening: listening });
      }
      return events;
    };

    // Remove one or many callbacks. If `context` is null, removes all
    // callbacks with that function. If `callback` is null, removes all
    // callbacks for the event. If `name` is null, removes all bound
    // callbacks for all events.
    Events.off =  function(name, callback, context) {
      if (!this._events) return this;
      this._events = eventsApi(offApi, this._events, name, callback, {
          context: context,
          listeners: this._listeners
      });
      return this;
    };

    // Tell this object to stop listening to either specific events ... or
    // to every object it's currently listening to.
    Events.stopListening =  function(obj, name, callback) {
      var listeningTo = this._listeningTo;
      if (!listeningTo) return this;

      var ids = obj ? [obj._listenId] : _.keys(listeningTo);

      for (var i = 0; i < ids.length; i++) {
        var listening = listeningTo[ids[i]];

        // If listening doesn't exist, this object is not currently
        // listening to obj. Break out early.
        if (!listening) break;

        listening.obj.off(name, callback, this);
      }
      if (_.isEmpty(listeningTo)) this._listeningTo = void 0;

      return this;
    };

    // The reducing API that removes a callback from the `events` object.
    var offApi = function(events, name, callback, options) {
      if (!events) return;

      var i = 0, listening;
      var context = options.context, listeners = options.listeners;

      // Delete all events listeners and "drop" events.
      if (!name && !callback && !context) {
        var ids = _.keys(listeners);
        for (; i < ids.length; i++) {
          listening = listeners[ids[i]];
          delete listeners[listening.id];
          delete listening.listeningTo[listening.objId];
        }
        return;
      }

      var names = name ? [name] : _.keys(events);
      for (; i < names.length; i++) {
        name = names[i];
        var handlers = events[name];

        // Bail out if there are no events stored.
        if (!handlers) break;

        // Replace events if there are any remaining.  Otherwise, clean up.
        var remaining = [];
        for (var j = 0; j < handlers.length; j++) {
          var handler = handlers[j];
          if (
            callback && callback !== handler.callback &&
              callback !== handler.callback._callback ||
                context && context !== handler.context
          ) {
            remaining.push(handler);
          } else {
            listening = handler.listening;
            if (listening && --listening.count === 0) {
              delete listeners[listening.id];
              delete listening.listeningTo[listening.objId];
            }
          }
        }

        // Update tail event if the list has any events.  Otherwise, clean up.
        if (remaining.length) {
          events[name] = remaining;
        } else {
          delete events[name];
        }
      }
      if (_.size(events)) return events;
    };

    // Bind an event to only be triggered a single time. After the first time
    // the callback is invoked, its listener will be removed. If multiple events
    // are passed in using the space-separated syntax, the handler will fire
    // once for each event, not once for a combination of all events.
    Events.once =  function(name, callback, context) {
      // Map the event into a `{event: once}` object.
      var events = eventsApi(onceMap, {}, name, callback, _.bind(this.off, this));
      return this.on(events, void 0, context);
    };

    // Inversion-of-control versions of `once`.
    Events.listenToOnce =  function(obj, name, callback) {
      // Map the event into a `{event: once}` object.
      var events = eventsApi(onceMap, {}, name, callback, _.bind(this.stopListening, this, obj));
      return this.listenTo(obj, events);
    };

    // Reduces the event callbacks into a map of `{event: onceWrapper}`.
    // `offer` unbinds the `onceWrapper` after it has been called.
    var onceMap = function(map, name, callback, offer) {
      if (callback) {
        var once = map[name] = _.once(function() {
          offer(name, once);
          callback.apply(this, arguments);
        });
        once._callback = callback;
      }
      return map;
    };

    // Trigger one or many events, firing all bound callbacks. Callbacks are
    // passed the same arguments as `trigger` is, apart from the event name
    // (unless you're listening on `"all"`, which will cause your callback to
    // receive the true name of the event as the first argument).
    Events.trigger =  function(name) {
      if (!this._events) return this;

      var length = Math.max(0, arguments.length - 1);
      var args = Array(length);
      for (var i = 0; i < length; i++) args[i] = arguments[i + 1];

      eventsApi(triggerApi, this._events, name, void 0, args);
      return this;
    };

    // Handles triggering the appropriate event callbacks.
    var triggerApi = function(objEvents, name, cb, args) {
      if (objEvents) {
        var events = objEvents[name];
        var allEvents = objEvents.all;
        if (events && allEvents) allEvents = allEvents.slice();
        if (events) triggerEvents(events, args);
        if (allEvents) triggerEvents(allEvents, [name].concat(args));
      }
      return objEvents;
    };

    // A difficult-to-believe, but optimized internal dispatch function for
    // triggering events. Tries to keep the usual cases speedy (most internal
    // Backbone events have 3 arguments).
    var triggerEvents = function(events, args) {
      var ev, i = -1, l = events.length, a1 = args[0], a2 = args[1], a3 = args[2];
      switch (args.length) {
        case 0: while (++i < l) (ev = events[i]).callback.call(ev.ctx); return;
        case 1: while (++i < l) (ev = events[i]).callback.call(ev.ctx, a1); return;
        case 2: while (++i < l) (ev = events[i]).callback.call(ev.ctx, a1, a2); return;
        case 3: while (++i < l) (ev = events[i]).callback.call(ev.ctx, a1, a2, a3); return;
        default: while (++i < l) (ev = events[i]).callback.apply(ev.ctx, args); return;
      }
    };

    // Aliases for backwards compatibility.
    Events.bind   = Events.on;
    Events.unbind = Events.off;

    // Allow the `Backbone` object to serve as a global event bus, for folks who
    // want global "pubsub" in a convenient place.
    _.extend(Backbone, Events);

    // Backbone.Model
    // --------------

    // Backbone **Models** are the basic data object in the framework --
    // frequently representing a row in a table in a database on your server.
    // A discrete chunk of data and a bunch of useful, related methods for
    // performing computations and transformations on that data.

    // Create a new model with the specified attributes. A client id (`cid`)
    // is automatically generated and assigned for you.
    var Model = Backbone.Model = function(attributes, options) {
      var attrs = attributes || {};
      options || (options = {});
      this.cid = _.uniqueId(this.cidPrefix);
      this.attributes = {};
      if (options.collection) this.collection = options.collection;
      if (options.parse) attrs = this.parse(attrs, options) || {};
      attrs = _.defaults({}, attrs, _.result(this, 'defaults'));
      this.set(attrs, options);
      this.changed = {};
      this.initialize.apply(this, arguments);
    };

    // Attach all inheritable methods to the Model prototype.
    _.extend(Model.prototype, Events, {

      // A hash of attributes whose current and previous value differ.
      changed: null,

      // The value returned during the last failed validation.
      validationError: null,

      // The default name for the JSON `id` attribute is `"id"`. MongoDB and
      // CouchDB users may want to set this to `"_id"`.
      idAttribute: 'id',

      // The prefix is used to create the client id which is used to identify models locally.
      // You may want to override this if you're experiencing name clashes with model ids.
      cidPrefix: 'c',

      // Initialize is an empty function by default. Override it with your own
      // initialization logic.
      initialize: function(){},

      // Return a copy of the model's `attributes` object.
      toJSON: function(options) {
        return _.clone(this.attributes);
      },

      // Proxy `Backbone.sync` by default -- but override this if you need
      // custom syncing semantics for *this* particular model.
      sync: function() {
        return Backbone.sync.apply(this, arguments);
      },

      // Get the value of an attribute.
      get: function(attr) {
        return this.attributes[attr];
      },

      // Get the HTML-escaped value of an attribute.
      escape: function(attr) {
        return _.escape(this.get(attr));
      },

      // Returns `true` if the attribute contains a value that is not null
      // or undefined.
      has: function(attr) {
        return this.get(attr) != null;
      },

      // Special-cased proxy to underscore's `_.matches` method.
      matches: function(attrs) {
        return !!_.iteratee(attrs, this)(this.attributes);
      },

      // Set a hash of model attributes on the object, firing `"change"`. This is
      // the core primitive operation of a model, updating the data and notifying
      // anyone who needs to know about the change in state. The heart of the beast.
      set: function(key, val, options) {
        if (key == null) return this;

        // Handle both `"key", value` and `{key: value}` -style arguments.
        var attrs;
        if (typeof key === 'object') {
          attrs = key;
          options = val;
        } else {
          (attrs = {})[key] = val;
        }

        options || (options = {});

        // Run validation.
        if (!this._validate(attrs, options)) return false;

        // Extract attributes and options.
        var unset      = options.unset;
        var silent     = options.silent;
        var changes    = [];
        var changing   = this._changing;
        this._changing = true;

        if (!changing) {
          this._previousAttributes = _.clone(this.attributes);
          this.changed = {};
        }

        var current = this.attributes;
        var changed = this.changed;
        var prev    = this._previousAttributes;

        // For each `set` attribute, update or delete the current value.
        for (var attr in attrs) {
          val = attrs[attr];
          if (!_.isEqual(current[attr], val)) changes.push(attr);
          if (!_.isEqual(prev[attr], val)) {
            changed[attr] = val;
          } else {
            delete changed[attr];
          }
          unset ? delete current[attr] : current[attr] = val;
        }

        // Update the `id`.
        this.id = this.get(this.idAttribute);

        // Trigger all relevant attribute changes.
        if (!silent) {
          if (changes.length) this._pending = options;
          for (var i = 0; i < changes.length; i++) {
            this.trigger('change:' + changes[i], this, current[changes[i]], options);
          }
        }

        // You might be wondering why there's a `while` loop here. Changes can
        // be recursively nested within `"change"` events.
        if (changing) return this;
        if (!silent) {
          while (this._pending) {
            options = this._pending;
            this._pending = false;
            this.trigger('change', this, options);
          }
        }
        this._pending = false;
        this._changing = false;
        return this;
      },

      // Remove an attribute from the model, firing `"change"`. `unset` is a noop
      // if the attribute doesn't exist.
      unset: function(attr, options) {
        return this.set(attr, void 0, _.extend({}, options, {unset: true}));
      },

      // Clear all attributes on the model, firing `"change"`.
      clear: function(options) {
        var attrs = {};
        for (var key in this.attributes) attrs[key] = void 0;
        return this.set(attrs, _.extend({}, options, {unset: true}));
      },

      // Determine if the model has changed since the last `"change"` event.
      // If you specify an attribute name, determine if that attribute has changed.
      hasChanged: function(attr) {
        if (attr == null) return !_.isEmpty(this.changed);
        return _.has(this.changed, attr);
      },

      // Return an object containing all the attributes that have changed, or
      // false if there are no changed attributes. Useful for determining what
      // parts of a view need to be updated and/or what attributes need to be
      // persisted to the server. Unset attributes will be set to undefined.
      // You can also pass an attributes object to diff against the model,
      // determining if there *would be* a change.
      changedAttributes: function(diff) {
        if (!diff) return this.hasChanged() ? _.clone(this.changed) : false;
        var old = this._changing ? this._previousAttributes : this.attributes;
        var changed = {};
        for (var attr in diff) {
          var val = diff[attr];
          if (_.isEqual(old[attr], val)) continue;
          changed[attr] = val;
        }
        return _.size(changed) ? changed : false;
      },

      // Get the previous value of an attribute, recorded at the time the last
      // `"change"` event was fired.
      previous: function(attr) {
        if (attr == null || !this._previousAttributes) return null;
        return this._previousAttributes[attr];
      },

      // Get all of the attributes of the model at the time of the previous
      // `"change"` event.
      previousAttributes: function() {
        return _.clone(this._previousAttributes);
      },

      // Fetch the model from the server, merging the response with the model's
      // local attributes. Any changed attributes will trigger a "change" event.
      fetch: function(options) {
        options = _.extend({parse: true}, options);
        var model = this;
        var success = options.success;
        options.success = function(resp) {
          var serverAttrs = options.parse ? model.parse(resp, options) : resp;
          if (!model.set(serverAttrs, options)) return false;
          if (success) success.call(options.context, model, resp, options);
          model.trigger('sync', model, resp, options);
        };
        wrapError(this, options);
        return this.sync('read', this, options);
      },

      // Set a hash of model attributes, and sync the model to the server.
      // If the server returns an attributes hash that differs, the model's
      // state will be `set` again.
      save: function(key, val, options) {
        // Handle both `"key", value` and `{key: value}` -style arguments.
        var attrs;
        if (key == null || typeof key === 'object') {
          attrs = key;
          options = val;
        } else {
          (attrs = {})[key] = val;
        }

        options = _.extend({validate: true, parse: true}, options);
        var wait = options.wait;

        // If we're not waiting and attributes exist, save acts as
        // `set(attr).save(null, opts)` with validation. Otherwise, check if
        // the model will be valid when the attributes, if any, are set.
        if (attrs && !wait) {
          if (!this.set(attrs, options)) return false;
        } else {
          if (!this._validate(attrs, options)) return false;
        }

        // After a successful server-side save, the client is (optionally)
        // updated with the server-side state.
        var model = this;
        var success = options.success;
        var attributes = this.attributes;
        options.success = function(resp) {
          // Ensure attributes are restored during synchronous saves.
          model.attributes = attributes;
          var serverAttrs = options.parse ? model.parse(resp, options) : resp;
          if (wait) serverAttrs = _.extend({}, attrs, serverAttrs);
          if (serverAttrs && !model.set(serverAttrs, options)) return false;
          if (success) success.call(options.context, model, resp, options);
          model.trigger('sync', model, resp, options);
        };
        wrapError(this, options);

        // Set temporary attributes if `{wait: true}` to properly find new ids.
        if (attrs && wait) this.attributes = _.extend({}, attributes, attrs);

        var method = this.isNew() ? 'create' : (options.patch ? 'patch' : 'update');
        if (method === 'patch' && !options.attrs) options.attrs = attrs;
        var xhr = this.sync(method, this, options);

        // Restore attributes.
        this.attributes = attributes;

        return xhr;
      },

      // Destroy this model on the server if it was already persisted.
      // Optimistically removes the model from its collection, if it has one.
      // If `wait: true` is passed, waits for the server to respond before removal.
      destroy: function(options) {
        options = options ? _.clone(options) : {};
        var model = this;
        var success = options.success;
        var wait = options.wait;

        var destroy = function() {
          model.stopListening();
          model.trigger('destroy', model, model.collection, options);
        };

        options.success = function(resp) {
          if (wait) destroy();
          if (success) success.call(options.context, model, resp, options);
          if (!model.isNew()) model.trigger('sync', model, resp, options);
        };

        var xhr = false;
        if (this.isNew()) {
          _.defer(options.success);
        } else {
          wrapError(this, options);
          xhr = this.sync('delete', this, options);
        }
        if (!wait) destroy();
        return xhr;
      },

      // Default URL for the model's representation on the server -- if you're
      // using Backbone's restful methods, override this to change the endpoint
      // that will be called.
      url: function() {
        var base =
          _.result(this, 'urlRoot') ||
          _.result(this.collection, 'url') ||
          urlError();
        if (this.isNew()) return base;
        var id = this.get(this.idAttribute);
        return base.replace(/[^\/]$/, '$&/') + encodeURIComponent(id);
      },

      // **parse** converts a response into the hash of attributes to be `set` on
      // the model. The default implementation is just to pass the response along.
      parse: function(resp, options) {
        return resp;
      },

      // Create a new model with identical attributes to this one.
      clone: function() {
        return new this.constructor(this.attributes);
      },

      // A model is new if it has never been saved to the server, and lacks an id.
      isNew: function() {
        return !this.has(this.idAttribute);
      },

      // Check if the model is currently in a valid state.
      isValid: function(options) {
        return this._validate({}, _.defaults({validate: true}, options));
      },

      // Run validation against the next complete set of model attributes,
      // returning `true` if all is well. Otherwise, fire an `"invalid"` event.
      _validate: function(attrs, options) {
        if (!options.validate || !this.validate) return true;
        attrs = _.extend({}, this.attributes, attrs);
        var error = this.validationError = this.validate(attrs, options) || null;
        if (!error) return true;
        this.trigger('invalid', this, error, _.extend(options, {validationError: error}));
        return false;
      }

    });

    // Underscore methods that we want to implement on the Model, mapped to the
    // number of arguments they take.
    var modelMethods = { keys: 1, values: 1, pairs: 1, invert: 1, pick: 0,
        omit: 0, chain: 1, isEmpty: 1 };

    // Mix in each Underscore method as a proxy to `Model#attributes`.
    addUnderscoreMethods(Model, modelMethods, 'attributes');

    // Backbone.Collection
    // -------------------

    // If models tend to represent a single row of data, a Backbone Collection is
    // more analogous to a table full of data ... or a small slice or page of that
    // table, or a collection of rows that belong together for a particular reason
    // -- all of the messages in this particular folder, all of the documents
    // belonging to this particular author, and so on. Collections maintain
    // indexes of their models, both in order, and for lookup by `id`.

    // Create a new **Collection**, perhaps to contain a specific type of `model`.
    // If a `comparator` is specified, the Collection will maintain
    // its models in sort order, as they're added and removed.
    var Collection = Backbone.Collection = function(models, options) {
      options || (options = {});
      if (options.model) this.model = options.model;
      if (options.comparator !== void 0) this.comparator = options.comparator;
      this._reset();
      this.initialize.apply(this, arguments);
      if (models) this.reset(models, _.extend({silent: true}, options));
    };

    // Default options for `Collection#set`.
    var setOptions = {add: true, remove: true, merge: true};
    var addOptions = {add: true, remove: false};

    // Splices `insert` into `array` at index `at`.
    var splice = function(array, insert, at) {
      at = Math.min(Math.max(at, 0), array.length);
      var tail = Array(array.length - at);
      var length = insert.length;
      for (var i = 0; i < tail.length; i++) tail[i] = array[i + at];
      for (i = 0; i < length; i++) array[i + at] = insert[i];
      for (i = 0; i < tail.length; i++) array[i + length + at] = tail[i];
    };

    // Define the Collection's inheritable methods.
    _.extend(Collection.prototype, Events, {

      // The default model for a collection is just a **Backbone.Model**.
      // This should be overridden in most cases.
      model: Model,

      // Initialize is an empty function by default. Override it with your own
      // initialization logic.
      initialize: function(){},

      // The JSON representation of a Collection is an array of the
      // models' attributes.
      toJSON: function(options) {
        return this.map(function(model) { return model.toJSON(options); });
      },

      // Proxy `Backbone.sync` by default.
      sync: function() {
        return Backbone.sync.apply(this, arguments);
      },

      // Add a model, or list of models to the set. `models` may be Backbone
      // Models or raw JavaScript objects to be converted to Models, or any
      // combination of the two.
      add: function(models, options) {
        return this.set(models, _.extend({merge: false}, options, addOptions));
      },

      // Remove a model, or a list of models from the set.
      remove: function(models, options) {
        options = _.extend({}, options);
        var singular = !_.isArray(models);
        models = singular ? [models] : _.clone(models);
        var removed = this._removeModels(models, options);
        if (!options.silent && removed) this.trigger('update', this, options);
        return singular ? removed[0] : removed;
      },

      // Update a collection by `set`-ing a new list of models, adding new ones,
      // removing models that are no longer present, and merging models that
      // already exist in the collection, as necessary. Similar to **Model#set**,
      // the core operation for updating the data contained by the collection.
      set: function(models, options) {
        if (models == null) return;

        options = _.defaults({}, options, setOptions);
        if (options.parse && !this._isModel(models)) models = this.parse(models, options);

        var singular = !_.isArray(models);
        models = singular ? [models] : models.slice();

        var at = options.at;
        if (at != null) at = +at;
        if (at < 0) at += this.length + 1;

        var set = [];
        var toAdd = [];
        var toRemove = [];
        var modelMap = {};

        var add = options.add;
        var merge = options.merge;
        var remove = options.remove;

        var sort = false;
        var sortable = this.comparator && (at == null) && options.sort !== false;
        var sortAttr = _.isString(this.comparator) ? this.comparator : null;

        // Turn bare objects into model references, and prevent invalid models
        // from being added.
        var model;
        for (var i = 0; i < models.length; i++) {
          model = models[i];

          // If a duplicate is found, prevent it from being added and
          // optionally merge it into the existing model.
          var existing = this.get(model);
          if (existing) {
            if (merge && model !== existing) {
              var attrs = this._isModel(model) ? model.attributes : model;
              if (options.parse) attrs = existing.parse(attrs, options);
              existing.set(attrs, options);
              if (sortable && !sort) sort = existing.hasChanged(sortAttr);
            }
            if (!modelMap[existing.cid]) {
              modelMap[existing.cid] = true;
              set.push(existing);
            }
            models[i] = existing;

          // If this is a new, valid model, push it to the `toAdd` list.
          } else if (add) {
            model = models[i] = this._prepareModel(model, options);
            if (model) {
              toAdd.push(model);
              this._addReference(model, options);
              modelMap[model.cid] = true;
              set.push(model);
            }
          }
        }

        // Remove stale models.
        if (remove) {
          for (i = 0; i < this.length; i++) {
            model = this.models[i];
            if (!modelMap[model.cid]) toRemove.push(model);
          }
          if (toRemove.length) this._removeModels(toRemove, options);
        }

        // See if sorting is needed, update `length` and splice in new models.
        var orderChanged = false;
        var replace = !sortable && add && remove;
        if (set.length && replace) {
          orderChanged = this.length != set.length || _.some(this.models, function(model, index) {
            return model !== set[index];
          });
          this.models.length = 0;
          splice(this.models, set, 0);
          this.length = this.models.length;
        } else if (toAdd.length) {
          if (sortable) sort = true;
          splice(this.models, toAdd, at == null ? this.length : at);
          this.length = this.models.length;
        }

        // Silently sort the collection if appropriate.
        if (sort) this.sort({silent: true});

        // Unless silenced, it's time to fire all appropriate add/sort events.
        if (!options.silent) {
          for (i = 0; i < toAdd.length; i++) {
            if (at != null) options.index = at + i;
            model = toAdd[i];
            model.trigger('add', model, this, options);
          }
          if (sort || orderChanged) this.trigger('sort', this, options);
          if (toAdd.length || toRemove.length) this.trigger('update', this, options);
        }

        // Return the added (or merged) model (or models).
        return singular ? models[0] : models;
      },

      // When you have more items than you want to add or remove individually,
      // you can reset the entire set with a new list of models, without firing
      // any granular `add` or `remove` events. Fires `reset` when finished.
      // Useful for bulk operations and optimizations.
      reset: function(models, options) {
        options = options ? _.clone(options) : {};
        for (var i = 0; i < this.models.length; i++) {
          this._removeReference(this.models[i], options);
        }
        options.previousModels = this.models;
        this._reset();
        models = this.add(models, _.extend({silent: true}, options));
        if (!options.silent) this.trigger('reset', this, options);
        return models;
      },

      // Add a model to the end of the collection.
      push: function(model, options) {
        return this.add(model, _.extend({at: this.length}, options));
      },

      // Remove a model from the end of the collection.
      pop: function(options) {
        var model = this.at(this.length - 1);
        return this.remove(model, options);
      },

      // Add a model to the beginning of the collection.
      unshift: function(model, options) {
        return this.add(model, _.extend({at: 0}, options));
      },

      // Remove a model from the beginning of the collection.
      shift: function(options) {
        var model = this.at(0);
        return this.remove(model, options);
      },

      // Slice out a sub-array of models from the collection.
      slice: function() {
        return slice.apply(this.models, arguments);
      },

      // Get a model from the set by id.
      get: function(obj) {
        if (obj == null) return void 0;
        var id = this.modelId(this._isModel(obj) ? obj.attributes : obj);
        return this._byId[obj] || this._byId[id] || this._byId[obj.cid];
      },

      // Get the model at the given index.
      at: function(index) {
        if (index < 0) index += this.length;
        return this.models[index];
      },

      // Return models with matching attributes. Useful for simple cases of
      // `filter`.
      where: function(attrs, first) {
        return this[first ? 'find' : 'filter'](attrs);
      },

      // Return the first model with matching attributes. Useful for simple cases
      // of `find`.
      findWhere: function(attrs) {
        return this.where(attrs, true);
      },

      // Force the collection to re-sort itself. You don't need to call this under
      // normal circumstances, as the set will maintain sort order as each item
      // is added.
      sort: function(options) {
        var comparator = this.comparator;
        if (!comparator) throw new Error('Cannot sort a set without a comparator');
        options || (options = {});

        var length = comparator.length;
        if (_.isFunction(comparator)) comparator = _.bind(comparator, this);

        // Run sort based on type of `comparator`.
        if (length === 1 || _.isString(comparator)) {
          this.models = this.sortBy(comparator);
        } else {
          this.models.sort(comparator);
        }
        if (!options.silent) this.trigger('sort', this, options);
        return this;
      },

      // Pluck an attribute from each model in the collection.
      pluck: function(attr) {
        return _.invoke(this.models, 'get', attr);
      },

      // Fetch the default set of models for this collection, resetting the
      // collection when they arrive. If `reset: true` is passed, the response
      // data will be passed through the `reset` method instead of `set`.
      fetch: function(options) {
        options = _.extend({parse: true}, options);
        var success = options.success;
        var collection = this;
        options.success = function(resp) {
          var method = options.reset ? 'reset' : 'set';
          collection[method](resp, options);
          if (success) success.call(options.context, collection, resp, options);
          collection.trigger('sync', collection, resp, options);
        };
        wrapError(this, options);
        return this.sync('read', this, options);
      },

      // Create a new instance of a model in this collection. Add the model to the
      // collection immediately, unless `wait: true` is passed, in which case we
      // wait for the server to agree.
      create: function(model, options) {
        options = options ? _.clone(options) : {};
        var wait = options.wait;
        model = this._prepareModel(model, options);
        if (!model) return false;
        if (!wait) this.add(model, options);
        var collection = this;
        var success = options.success;
        options.success = function(model, resp, callbackOpts) {
          if (wait) collection.add(model, callbackOpts);
          if (success) success.call(callbackOpts.context, model, resp, callbackOpts);
        };
        model.save(null, options);
        return model;
      },

      // **parse** converts a response into a list of models to be added to the
      // collection. The default implementation is just to pass it through.
      parse: function(resp, options) {
        return resp;
      },

      // Create a new collection with an identical list of models as this one.
      clone: function() {
        return new this.constructor(this.models, {
          model: this.model,
          comparator: this.comparator
        });
      },

      // Define how to uniquely identify models in the collection.
      modelId: function (attrs) {
        return attrs[this.model.prototype.idAttribute || 'id'];
      },

      // Private method to reset all internal state. Called when the collection
      // is first initialized or reset.
      _reset: function() {
        this.length = 0;
        this.models = [];
        this._byId  = {};
      },

      // Prepare a hash of attributes (or other model) to be added to this
      // collection.
      _prepareModel: function(attrs, options) {
        if (this._isModel(attrs)) {
          if (!attrs.collection) attrs.collection = this;
          return attrs;
        }
        options = options ? _.clone(options) : {};
        options.collection = this;
        var model = new this.model(attrs, options);
        if (!model.validationError) return model;
        this.trigger('invalid', this, model.validationError, options);
        return false;
      },

      // Internal method called by both remove and set.
      _removeModels: function(models, options) {
        var removed = [];
        for (var i = 0; i < models.length; i++) {
          var model = this.get(models[i]);
          if (!model) continue;

          var index = this.indexOf(model);
          this.models.splice(index, 1);
          this.length--;

          if (!options.silent) {
            options.index = index;
            model.trigger('remove', model, this, options);
          }

          removed.push(model);
          this._removeReference(model, options);
        }
        return removed.length ? removed : false;
      },

      // Method for checking whether an object should be considered a model for
      // the purposes of adding to the collection.
      _isModel: function (model) {
        return model instanceof Model;
      },

      // Internal method to create a model's ties to a collection.
      _addReference: function(model, options) {
        this._byId[model.cid] = model;
        var id = this.modelId(model.attributes);
        if (id != null) this._byId[id] = model;
        model.on('all', this._onModelEvent, this);
      },

      // Internal method to sever a model's ties to a collection.
      _removeReference: function(model, options) {
        delete this._byId[model.cid];
        var id = this.modelId(model.attributes);
        if (id != null) delete this._byId[id];
        if (this === model.collection) delete model.collection;
        model.off('all', this._onModelEvent, this);
      },

      // Internal method called every time a model in the set fires an event.
      // Sets need to update their indexes when models change ids. All other
      // events simply proxy through. "add" and "remove" events that originate
      // in other collections are ignored.
      _onModelEvent: function(event, model, collection, options) {
        if ((event === 'add' || event === 'remove') && collection !== this) return;
        if (event === 'destroy') this.remove(model, options);
        if (event === 'change') {
          var prevId = this.modelId(model.previousAttributes());
          var id = this.modelId(model.attributes);
          if (prevId !== id) {
            if (prevId != null) delete this._byId[prevId];
            if (id != null) this._byId[id] = model;
          }
        }
        this.trigger.apply(this, arguments);
      }

    });

    // Underscore methods that we want to implement on the Collection.
    // 90% of the core usefulness of Backbone Collections is actually implemented
    // right here:
    var collectionMethods = { forEach: 3, each: 3, map: 3, collect: 3, reduce: 4,
        foldl: 4, inject: 4, reduceRight: 4, foldr: 4, find: 3, detect: 3, filter: 3,
        select: 3, reject: 3, every: 3, all: 3, some: 3, any: 3, include: 3, includes: 3,
        contains: 3, invoke: 0, max: 3, min: 3, toArray: 1, size: 1, first: 3,
        head: 3, take: 3, initial: 3, rest: 3, tail: 3, drop: 3, last: 3,
        without: 0, difference: 0, indexOf: 3, shuffle: 1, lastIndexOf: 3,
        isEmpty: 1, chain: 1, sample: 3, partition: 3, groupBy: 3, countBy: 3,
        sortBy: 3, indexBy: 3};

    // Mix in each Underscore method as a proxy to `Collection#models`.
    addUnderscoreMethods(Collection, collectionMethods, 'models');

    // Backbone.View
    // -------------

    // Backbone Views are almost more convention than they are actual code. A View
    // is simply a JavaScript object that represents a logical chunk of UI in the
    // DOM. This might be a single item, an entire list, a sidebar or panel, or
    // even the surrounding frame which wraps your whole app. Defining a chunk of
    // UI as a **View** allows you to define your DOM events declaratively, without
    // having to worry about render order ... and makes it easy for the view to
    // react to specific changes in the state of your models.

    // Creating a Backbone.View creates its initial element outside of the DOM,
    // if an existing element is not provided...
    var View = Backbone.View = function(options) {
      this.cid = _.uniqueId('view');
      _.extend(this, _.pick(options, viewOptions));
      this._ensureElement();
      this.initialize.apply(this, arguments);
    };

    // Cached regex to split keys for `delegate`.
    var delegateEventSplitter = /^(\S+)\s*(.*)$/;

    // List of view options to be set as properties.
    var viewOptions = ['model', 'collection', 'el', 'id', 'attributes', 'className', 'tagName', 'events'];

    // Set up all inheritable **Backbone.View** properties and methods.
    _.extend(View.prototype, Events, {

      // The default `tagName` of a View's element is `"div"`.
      tagName: 'div',

      // jQuery delegate for element lookup, scoped to DOM elements within the
      // current view. This should be preferred to global lookups where possible.
      $: function(selector) {
        return this.$el.find(selector);
      },

      // Initialize is an empty function by default. Override it with your own
      // initialization logic.
      initialize: function(){},

      // **render** is the core function that your view should override, in order
      // to populate its element (`this.el`), with the appropriate HTML. The
      // convention is for **render** to always return `this`.
      render: function() {
        return this;
      },

      // Remove this view by taking the element out of the DOM, and removing any
      // applicable Backbone.Events listeners.
      remove: function() {
        this._removeElement();
        this.stopListening();
        return this;
      },

      // Remove this view's element from the document and all event listeners
      // attached to it. Exposed for subclasses using an alternative DOM
      // manipulation API.
      _removeElement: function() {
        this.$el.remove();
      },

      // Change the view's element (`this.el` property) and re-delegate the
      // view's events on the new element.
      setElement: function(element) {
        this.undelegateEvents();
        this._setElement(element);
        this.delegateEvents();
        return this;
      },

      // Creates the `this.el` and `this.$el` references for this view using the
      // given `el`. `el` can be a CSS selector or an HTML string, a jQuery
      // context or an element. Subclasses can override this to utilize an
      // alternative DOM manipulation API and are only required to set the
      // `this.el` property.
      _setElement: function(el) {
        this.$el = el instanceof Backbone.$ ? el : Backbone.$(el);
        this.el = this.$el[0];
      },

      // Set callbacks, where `this.events` is a hash of
      //
      // *{"event selector": "callback"}*
      //
      //     {
      //       'mousedown .title':  'edit',
      //       'click .button':     'save',
      //       'click .open':       function(e) { ... }
      //     }
      //
      // pairs. Callbacks will be bound to the view, with `this` set properly.
      // Uses event delegation for efficiency.
      // Omitting the selector binds the event to `this.el`.
      delegateEvents: function(events) {
        events || (events = _.result(this, 'events'));
        if (!events) return this;
        this.undelegateEvents();
        for (var key in events) {
          var method = events[key];
          if (!_.isFunction(method)) method = this[method];
          if (!method) continue;
          var match = key.match(delegateEventSplitter);
          this.delegate(match[1], match[2], _.bind(method, this));
        }
        return this;
      },

      // Add a single event listener to the view's element (or a child element
      // using `selector`). This only works for delegate-able events: not `focus`,
      // `blur`, and not `change`, `submit`, and `reset` in Internet Explorer.
      delegate: function(eventName, selector, listener) {
        this.$el.on(eventName + '.delegateEvents' + this.cid, selector, listener);
        return this;
      },

      // Clears all callbacks previously bound to the view by `delegateEvents`.
      // You usually don't need to use this, but may wish to if you have multiple
      // Backbone views attached to the same DOM element.
      undelegateEvents: function() {
        if (this.$el) this.$el.off('.delegateEvents' + this.cid);
        return this;
      },

      // A finer-grained `undelegateEvents` for removing a single delegated event.
      // `selector` and `listener` are both optional.
      undelegate: function(eventName, selector, listener) {
        this.$el.off(eventName + '.delegateEvents' + this.cid, selector, listener);
        return this;
      },

      // Produces a DOM element to be assigned to your view. Exposed for
      // subclasses using an alternative DOM manipulation API.
      _createElement: function(tagName) {
        return document.createElement(tagName);
      },

      // Ensure that the View has a DOM element to render into.
      // If `this.el` is a string, pass it through `$()`, take the first
      // matching element, and re-assign it to `el`. Otherwise, create
      // an element from the `id`, `className` and `tagName` properties.
      _ensureElement: function() {
        if (!this.el) {
          var attrs = _.extend({}, _.result(this, 'attributes'));
          if (this.id) attrs.id = _.result(this, 'id');
          if (this.className) attrs['class'] = _.result(this, 'className');
          this.setElement(this._createElement(_.result(this, 'tagName')));
          this._setAttributes(attrs);
        } else {
          this.setElement(_.result(this, 'el'));
        }
      },

      // Set attributes from a hash on this view's element.  Exposed for
      // subclasses using an alternative DOM manipulation API.
      _setAttributes: function(attributes) {
        this.$el.attr(attributes);
      }

    });

    // Backbone.sync
    // -------------

    // Override this function to change the manner in which Backbone persists
    // models to the server. You will be passed the type of request, and the
    // model in question. By default, makes a RESTful Ajax request
    // to the model's `url()`. Some possible customizations could be:
    //
    // * Use `setTimeout` to batch rapid-fire updates into a single request.
    // * Send up the models as XML instead of JSON.
    // * Persist models via WebSockets instead of Ajax.
    //
    // Turn on `Backbone.emulateHTTP` in order to send `PUT` and `DELETE` requests
    // as `POST`, with a `_method` parameter containing the true HTTP method,
    // as well as all requests with the body as `application/x-www-form-urlencoded`
    // instead of `application/json` with the model in a param named `model`.
    // Useful when interfacing with server-side languages like **PHP** that make
    // it difficult to read the body of `PUT` requests.
    Backbone.sync = function(method, model, options) {
      var type = methodMap[method];

      // Default options, unless specified.
      _.defaults(options || (options = {}), {
        emulateHTTP: Backbone.emulateHTTP,
        emulateJSON: Backbone.emulateJSON
      });

      // Default JSON-request options.
      var params = {type: type, dataType: 'json'};

      // Ensure that we have a URL.
      if (!options.url) {
        params.url = _.result(model, 'url') || urlError();
      }

      // Ensure that we have the appropriate request data.
      if (options.data == null && model && (method === 'create' || method === 'update' || method === 'patch')) {
        params.contentType = 'application/json';
        params.data = JSON.stringify(options.attrs || model.toJSON(options));
      }

      // For older servers, emulate JSON by encoding the request into an HTML-form.
      if (options.emulateJSON) {
        params.contentType = 'application/x-www-form-urlencoded';
        params.data = params.data ? {model: params.data} : {};
      }

      // For older servers, emulate HTTP by mimicking the HTTP method with `_method`
      // And an `X-HTTP-Method-Override` header.
      if (options.emulateHTTP && (type === 'PUT' || type === 'DELETE' || type === 'PATCH')) {
        params.type = 'POST';
        if (options.emulateJSON) params.data._method = type;
        var beforeSend = options.beforeSend;
        options.beforeSend = function(xhr) {
          xhr.setRequestHeader('X-HTTP-Method-Override', type);
          if (beforeSend) return beforeSend.apply(this, arguments);
        };
      }

      // Don't process data on a non-GET request.
      if (params.type !== 'GET' && !options.emulateJSON) {
        params.processData = false;
      }

      // Pass along `textStatus` and `errorThrown` from jQuery.
      var error = options.error;
      options.error = function(xhr, textStatus, errorThrown) {
        options.textStatus = textStatus;
        options.errorThrown = errorThrown;
        if (error) error.call(options.context, xhr, textStatus, errorThrown);
      };

      // Make the request, allowing the user to override any Ajax options.
      var xhr = options.xhr = Backbone.ajax(_.extend(params, options));
      model.trigger('request', model, xhr, options);
      return xhr;
    };

    // Map from CRUD to HTTP for our default `Backbone.sync` implementation.
    var methodMap = {
      'create': 'POST',
      'update': 'PUT',
      'patch':  'PATCH',
      'delete': 'DELETE',
      'read':   'GET'
    };

    // Set the default implementation of `Backbone.ajax` to proxy through to `$`.
    // Override this if you'd like to use a different library.
    Backbone.ajax = function() {
      return Backbone.$.ajax.apply(Backbone.$, arguments);
    };

    // Backbone.Router
    // ---------------

    // Routers map faux-URLs to actions, and fire events when routes are
    // matched. Creating a new one sets its `routes` hash, if not set statically.
    var Router = Backbone.Router = function(options) {
      options || (options = {});
      if (options.routes) this.routes = options.routes;
      this._bindRoutes();
      this.initialize.apply(this, arguments);
    };

    // Cached regular expressions for matching named param parts and splatted
    // parts of route strings.
    var optionalParam = /\((.*?)\)/g;
    var namedParam    = /(\(\?)?:\w+/g;
    var splatParam    = /\*\w+/g;
    var escapeRegExp  = /[\-{}\[\]+?.,\\\^$|#\s]/g;

    // Set up all inheritable **Backbone.Router** properties and methods.
    _.extend(Router.prototype, Events, {

      // Initialize is an empty function by default. Override it with your own
      // initialization logic.
      initialize: function(){},

      // Manually bind a single named route to a callback. For example:
      //
      //     this.route('search/:query/p:num', 'search', function(query, num) {
      //       ...
      //     });
      //
      route: function(route, name, callback) {
        if (!_.isRegExp(route)) route = this._routeToRegExp(route);
        if (_.isFunction(name)) {
          callback = name;
          name = '';
        }
        if (!callback) callback = this[name];
        var router = this;
        Backbone.history.route(route, function(fragment) {
          var args = router._extractParameters(route, fragment);
          if (router.execute(callback, args, name) !== false) {
            router.trigger.apply(router, ['route:' + name].concat(args));
            router.trigger('route', name, args);
            Backbone.history.trigger('route', router, name, args);
          }
        });
        return this;
      },

      // Execute a route handler with the provided parameters.  This is an
      // excellent place to do pre-route setup or post-route cleanup.
      execute: function(callback, args, name) {
        if (callback) callback.apply(this, args);
      },

      // Simple proxy to `Backbone.history` to save a fragment into the history.
      navigate: function(fragment, options) {
        Backbone.history.navigate(fragment, options);
        return this;
      },

      // Bind all defined routes to `Backbone.history`. We have to reverse the
      // order of the routes here to support behavior where the most general
      // routes can be defined at the bottom of the route map.
      _bindRoutes: function() {
        if (!this.routes) return;
        this.routes = _.result(this, 'routes');
        var route, routes = _.keys(this.routes);
        while ((route = routes.pop()) != null) {
          this.route(route, this.routes[route]);
        }
      },

      // Convert a route string into a regular expression, suitable for matching
      // against the current location hash.
      _routeToRegExp: function(route) {
        route = route.replace(escapeRegExp, '\\$&')
                     .replace(optionalParam, '(?:$1)?')
                     .replace(namedParam, function(match, optional) {
                       return optional ? match : '([^/?]+)';
                     })
                     .replace(splatParam, '([^?]*?)');
        return new RegExp('^' + route + '(?:\\?([\\s\\S]*))?$');
      },

      // Given a route, and a URL fragment that it matches, return the array of
      // extracted decoded parameters. Empty or unmatched parameters will be
      // treated as `null` to normalize cross-browser behavior.
      _extractParameters: function(route, fragment) {
        var params = route.exec(fragment).slice(1);
        return _.map(params, function(param, i) {
          // Don't decode the search params.
          if (i === params.length - 1) return param || null;
          return param ? decodeURIComponent(param) : null;
        });
      }

    });

    // Backbone.History
    // ----------------

    // Handles cross-browser history management, based on either
    // [pushState](http://diveintohtml5.info/history.html) and real URLs, or
    // [onhashchange](https://developer.mozilla.org/en-US/docs/DOM/window.onhashchange)
    // and URL fragments. If the browser supports neither (old IE, natch),
    // falls back to polling.
    var History = Backbone.History = function() {
      this.handlers = [];
      this.checkUrl = _.bind(this.checkUrl, this);

      // Ensure that `History` can be used outside of the browser.
      if (typeof window !== 'undefined') {
        this.location = window.location;
        this.history = window.history;
      }
    };

    // Cached regex for stripping a leading hash/slash and trailing space.
    var routeStripper = /^[#\/]|\s+$/g;

    // Cached regex for stripping leading and trailing slashes.
    var rootStripper = /^\/+|\/+$/g;

    // Cached regex for stripping urls of hash.
    var pathStripper = /#.*$/;

    // Has the history handling already been started?
    History.started = false;

    // Set up all inheritable **Backbone.History** properties and methods.
    _.extend(History.prototype, Events, {

      // The default interval to poll for hash changes, if necessary, is
      // twenty times a second.
      interval: 50,

      // Are we at the app root?
      atRoot: function() {
        var path = this.location.pathname.replace(/[^\/]$/, '$&/');
        return path === this.root && !this.getSearch();
      },

      // Does the pathname match the root?
      matchRoot: function() {
        var path = this.decodeFragment(this.location.pathname);
        var root = path.slice(0, this.root.length - 1) + '/';
        return root === this.root;
      },

      // Unicode characters in `location.pathname` are percent encoded so they're
      // decoded for comparison. `%25` should not be decoded since it may be part
      // of an encoded parameter.
      decodeFragment: function(fragment) {
        return decodeURI(fragment.replace(/%25/g, '%2525'));
      },

      // In IE6, the hash fragment and search params are incorrect if the
      // fragment contains `?`.
      getSearch: function() {
        var match = this.location.href.replace(/#.*/, '').match(/\?.+/);
        return match ? match[0] : '';
      },

      // Gets the true hash value. Cannot use location.hash directly due to bug
      // in Firefox where location.hash will always be decoded.
      getHash: function(window) {
        var match = (window || this).location.href.match(/#(.*)$/);
        return match ? match[1] : '';
      },

      // Get the pathname and search params, without the root.
      getPath: function() {
        var path = this.decodeFragment(
          this.location.pathname + this.getSearch()
        ).slice(this.root.length - 1);
        return path.charAt(0) === '/' ? path.slice(1) : path;
      },

      // Get the cross-browser normalized URL fragment from the path or hash.
      getFragment: function(fragment) {
        if (fragment == null) {
          if (this._usePushState || !this._wantsHashChange) {
            fragment = this.getPath();
          } else {
            fragment = this.getHash();
          }
        }
        return fragment.replace(routeStripper, '');
      },

      // Start the hash change handling, returning `true` if the current URL matches
      // an existing route, and `false` otherwise.
      start: function(options) {
        if (History.started) throw new Error('Backbone.history has already been started');
        History.started = true;

        // Figure out the initial configuration. Do we need an iframe?
        // Is pushState desired ... is it available?
        this.options          = _.extend({root: '/'}, this.options, options);
        this.root             = this.options.root;
        this._wantsHashChange = this.options.hashChange !== false;
        this._hasHashChange   = 'onhashchange' in window && (document.documentMode === void 0 || document.documentMode > 7);
        this._useHashChange   = this._wantsHashChange && this._hasHashChange;
        this._wantsPushState  = !!this.options.pushState;
        this._hasPushState    = !!(this.history && this.history.pushState);
        this._usePushState    = this._wantsPushState && this._hasPushState;
        this.fragment         = this.getFragment();

        // Normalize root to always include a leading and trailing slash.
        this.root = ('/' + this.root + '/').replace(rootStripper, '/');

        // Transition from hashChange to pushState or vice versa if both are
        // requested.
        if (this._wantsHashChange && this._wantsPushState) {

          // If we've started off with a route from a `pushState`-enabled
          // browser, but we're currently in a browser that doesn't support it...
          if (!this._hasPushState && !this.atRoot()) {
            var root = this.root.slice(0, -1) || '/';
            this.location.replace(root + '#' + this.getPath());
            // Return immediately as browser will do redirect to new url
            return true;

          // Or if we've started out with a hash-based route, but we're currently
          // in a browser where it could be `pushState`-based instead...
          } else if (this._hasPushState && this.atRoot()) {
            this.navigate(this.getHash(), {replace: true});
          }

        }

        // Proxy an iframe to handle location events if the browser doesn't
        // support the `hashchange` event, HTML5 history, or the user wants
        // `hashChange` but not `pushState`.
        if (!this._hasHashChange && this._wantsHashChange && !this._usePushState) {
          this.iframe = document.createElement('iframe');
          this.iframe.src = 'javascript:0';
          this.iframe.style.display = 'none';
          this.iframe.tabIndex = -1;
          var body = document.body;
          // Using `appendChild` will throw on IE < 9 if the document is not ready.
          var iWindow = body.insertBefore(this.iframe, body.firstChild).contentWindow;
          iWindow.document.open();
          iWindow.document.close();
          iWindow.location.hash = '#' + this.fragment;
        }

        // Add a cross-platform `addEventListener` shim for older browsers.
        var addEventListener = window.addEventListener || function (eventName, listener) {
          return attachEvent('on' + eventName, listener);
        };

        // Depending on whether we're using pushState or hashes, and whether
        // 'onhashchange' is supported, determine how we check the URL state.
        if (this._usePushState) {
          addEventListener('popstate', this.checkUrl, false);
        } else if (this._useHashChange && !this.iframe) {
          addEventListener('hashchange', this.checkUrl, false);
        } else if (this._wantsHashChange) {
          this._checkUrlInterval = setInterval(this.checkUrl, this.interval);
        }

        if (!this.options.silent) return this.loadUrl();
      },

      // Disable Backbone.history, perhaps temporarily. Not useful in a real app,
      // but possibly useful for unit testing Routers.
      stop: function() {
        // Add a cross-platform `removeEventListener` shim for older browsers.
        var removeEventListener = window.removeEventListener || function (eventName, listener) {
          return detachEvent('on' + eventName, listener);
        };

        // Remove window listeners.
        if (this._usePushState) {
          removeEventListener('popstate', this.checkUrl, false);
        } else if (this._useHashChange && !this.iframe) {
          removeEventListener('hashchange', this.checkUrl, false);
        }

        // Clean up the iframe if necessary.
        if (this.iframe) {
          document.body.removeChild(this.iframe);
          this.iframe = null;
        }

        // Some environments will throw when clearing an undefined interval.
        if (this._checkUrlInterval) clearInterval(this._checkUrlInterval);
        History.started = false;
      },

      // Add a route to be tested when the fragment changes. Routes added later
      // may override previous routes.
      route: function(route, callback) {
        this.handlers.unshift({route: route, callback: callback});
      },

      // Checks the current URL to see if it has changed, and if it has,
      // calls `loadUrl`, normalizing across the hidden iframe.
      checkUrl: function(e) {
        var current = this.getFragment();

        // If the user pressed the back button, the iframe's hash will have
        // changed and we should use that for comparison.
        if (current === this.fragment && this.iframe) {
          current = this.getHash(this.iframe.contentWindow);
        }

        if (current === this.fragment) return false;
        if (this.iframe) this.navigate(current);
        this.loadUrl();
      },

      // Attempt to load the current URL fragment. If a route succeeds with a
      // match, returns `true`. If no defined routes matches the fragment,
      // returns `false`.
      loadUrl: function(fragment) {
        // If the root doesn't match, no routes can match either.
        if (!this.matchRoot()) return false;
        fragment = this.fragment = this.getFragment(fragment);
        return _.some(this.handlers, function(handler) {
          if (handler.route.test(fragment)) {
            handler.callback(fragment);
            return true;
          }
        });
      },

      // Save a fragment into the hash history, or replace the URL state if the
      // 'replace' option is passed. You are responsible for properly URL-encoding
      // the fragment in advance.
      //
      // The options object can contain `trigger: true` if you wish to have the
      // route callback be fired (not usually desirable), or `replace: true`, if
      // you wish to modify the current URL without adding an entry to the history.
      navigate: function(fragment, options) {
        if (!History.started) return false;
        if (!options || options === true) options = {trigger: !!options};

        // Normalize the fragment.
        fragment = this.getFragment(fragment || '');

        // Don't include a trailing slash on the root.
        var root = this.root;
        if (fragment === '' || fragment.charAt(0) === '?') {
          root = root.slice(0, -1) || '/';
        }
        var url = root + fragment;

        // Strip the hash and decode for matching.
        fragment = this.decodeFragment(fragment.replace(pathStripper, ''));

        if (this.fragment === fragment) return;
        this.fragment = fragment;

        // If pushState is available, we use it to set the fragment as a real URL.
        if (this._usePushState) {
          this.history[options.replace ? 'replaceState' : 'pushState']({}, document.title, url);

        // If hash changes haven't been explicitly disabled, update the hash
        // fragment to store history.
        } else if (this._wantsHashChange) {
          this._updateHash(this.location, fragment, options.replace);
          if (this.iframe && (fragment !== this.getHash(this.iframe.contentWindow))) {
            var iWindow = this.iframe.contentWindow;

            // Opening and closing the iframe tricks IE7 and earlier to push a
            // history entry on hash-tag change.  When replace is true, we don't
            // want this.
            if (!options.replace) {
              iWindow.document.open();
              iWindow.document.close();
            }

            this._updateHash(iWindow.location, fragment, options.replace);
          }

        // If you've told us that you explicitly don't want fallback hashchange-
        // based history, then `navigate` becomes a page refresh.
        } else {
          return this.location.assign(url);
        }
        if (options.trigger) return this.loadUrl(fragment);
      },

      // Update the hash location, either replacing the current entry, or adding
      // a new one to the browser history.
      _updateHash: function(location, fragment, replace) {
        if (replace) {
          var href = location.href.replace(/(javascript:|#).*$/, '');
          location.replace(href + '#' + fragment);
        } else {
          // Some browsers require that `hash` contains a leading #.
          location.hash = '#' + fragment;
        }
      }

    });

    // Create the default Backbone.history.
    Backbone.history = new History;

    // Helpers
    // -------

    // Helper function to correctly set up the prototype chain for subclasses.
    // Similar to `goog.inherits`, but uses a hash of prototype properties and
    // class properties to be extended.
    var extend = function(protoProps, staticProps) {
      var parent = this;
      var child;

      // The constructor function for the new subclass is either defined by you
      // (the "constructor" property in your `extend` definition), or defaulted
      // by us to simply call the parent constructor.
      if (protoProps && _.has(protoProps, 'constructor')) {
        child = protoProps.constructor;
      } else {
        child = function(){ return parent.apply(this, arguments); };
      }

      // Add static properties to the constructor function, if supplied.
      _.extend(child, parent, staticProps);

      // Set the prototype chain to inherit from `parent`, without calling
      // `parent` constructor function.
      var Surrogate = function(){ this.constructor = child; };
      Surrogate.prototype = parent.prototype;
      child.prototype = new Surrogate;

      // Add prototype properties (instance properties) to the subclass,
      // if supplied.
      if (protoProps) _.extend(child.prototype, protoProps);

      // Set a convenience property in case the parent's prototype is needed
      // later.
      child.__super__ = parent.prototype;

      return child;
    };

    // Set up inheritance for the model, collection, router, view and history.
    Model.extend = Collection.extend = Router.extend = View.extend = History.extend = extend;

    // Throw an error when a URL is needed, and none is supplied.
    var urlError = function() {
      throw new Error('A "url" property or function must be specified');
    };

    // Wrap an optional error callback with a fallback error event.
    var wrapError = function(model, options) {
      var error = options.error;
      options.error = function(resp) {
        if (error) error.call(options.context, model, resp, options);
        model.trigger('error', model, resp, options);
      };
    };

    return Backbone;

  }));

  var __export1 = exports.View;
  var __export2 = exports.Model;

  // Copyright (c) Jupyter Development Team.
  // Distributed under the terms of the Modified BSD License.
  var __extends$1 = (undefined && undefined.__extends) || (function () {
      var extendStatics = function (d, b) {
          extendStatics = Object.setPrototypeOf ||
              ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
              function (d, b) { for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p]; };
          return extendStatics(d, b);
      };
      return function (d, b) {
          extendStatics(d, b);
          function __() { this.constructor = d; }
          d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
      };
  })();
  // Caches a local reference to `Element.prototype` for faster access.
  var ElementProto = Element.prototype; // : typeof Element = (typeof Element !== 'undefined' && Element.prototype) || {};
  // Find the right `Element#matches` for IE>=9 and modern browsers.
  var matchesSelector = ElementProto.matches ||
      ElementProto['webkitMatchesSelector'] ||
      ElementProto['mozMatchesSelector'] ||
      ElementProto['msMatchesSelector'] ||
      ElementProto['oMatchesSelector'] ||
      function matches(selector) {
          /* tslint:disable:no-invalid-this */
          var matches = (this.document || this.ownerDocument).querySelectorAll(selector);
          var i = matches.length;
          while (--i >= 0 && matches.item(i) !== this) {
              continue;
          }
          return i > -1;
          /* tslint:enable:no-invalid-this */
      };
  var NativeView = /** @class */ (function (_super) {
      __extends$1(NativeView, _super);
      function NativeView() {
          return _super !== null && _super.apply(this, arguments) || this;
      }
      NativeView.prototype._removeElement = function () {
          this.undelegateEvents();
          if (this.el.parentNode) {
              this.el.parentNode.removeChild(this.el);
          }
      };
      // Apply the `element` to the view.
      NativeView.prototype._setElement = function (element) {
          this.el = element;
      };
      // Set a hash of attributes to the view's `el`. We use the "prop" version
      // if available, falling back to `setAttribute` for the catch-all.
      NativeView.prototype._setAttributes = function (attrs) {
          for (var attr in attrs) {
              attr in this.el ? this.el[attr] = attrs[attr] : this.el.setAttribute(attr, attrs[attr]);
          }
      };
      /**
       * Make an event delegation handler for the given `eventName` and `selector`
       * and attach it to `this.el`.
       * If selector is empty, the listener will be bound to `this.el`. If not, a
       * new handler that will recursively traverse up the event target's DOM
       * hierarchy looking for a node that matches the selector. If one is found,
       * the event's `delegateTarget` property is set to it and the return the
       * result of calling bound `listener` with the parameters given to the
       * handler.
       *
       * This does not properly handle selectors for things like focus and blur (see
       * https://github.com/jquery/jquery/blob/7d21f02b9ec9f655583e898350badf89165ed4d5/src/event.js#L442
       * for some similar exceptional cases).
       */
      NativeView.prototype.delegate = function (eventName, selector, listener) {
          if (typeof selector !== 'string') {
              listener = selector;
              selector = null;
          }
          // We have to initialize this here, instead of in the constructor, because the
          // super constructor eventually calls this method before we get a chance to initialize
          // this._domEvents to an empty list.
          if (this._domEvents === void 0) {
              this._domEvents = [];
          }
          var root = this.el;
          var handler = selector ? function (e) {
              var node = e.target || e.srcElement;
              for (; node && node !== root; node = node.parentNode) {
                  if (matchesSelector.call(node, selector)) {
                      e.delegateTarget = node;
                      if (listener.handleEvent) {
                          return listener.handleEvent(e);
                      }
                      else {
                          return listener(e);
                      }
                  }
              }
          } : listener;
          this.el.addEventListener(eventName, handler, false);
          this._domEvents.push({ eventName: eventName, handler: handler, listener: listener, selector: selector });
          return handler;
      };
      // Remove a single delegated event. Either `eventName` or `selector` must
      // be included, `selector` and `listener` are optional.
      NativeView.prototype.undelegate = function (eventName, selector, listener) {
          if (typeof selector === 'function') {
              listener = selector;
              selector = null;
          }
          if (this.el && this._domEvents) {
              var handlers = this._domEvents.slice();
              var i = handlers.length;
              while (i--) {
                  var item = handlers[i];
                  var match = item.eventName === eventName &&
                      (listener ? item.listener === listener : true) &&
                      (selector ? item.selector === selector : true);
                  if (!match) {
                      continue;
                  }
                  this.el.removeEventListener(item.eventName, item.handler, false);
                  this._domEvents.splice(i, 1);
              }
          }
          return this;
      };
      // Remove all events created with `delegate` from `el`
      NativeView.prototype.undelegateEvents = function () {
          if (this.el && this._domEvents) {
              var len = this._domEvents.length;
              for (var i = 0; i < len; i++) {
                  var item = this._domEvents[i];
                  this.el.removeEventListener(item.eventName, item.handler, false);
              }
              this._domEvents.length = 0;
          }
          return this;
      };
      return NativeView;
  }(__export1));

  // Copyright (c) Jupyter Development Team.
  // Distributed under the terms of the Modified BSD License.
  /*-----------------------------------------------------------------------------
  | Copyright (c) 2014-2017, PhosphorJS Contributors
  |
  | Distributed under the terms of the BSD 3-Clause License.
  |
  | The full license is in the file LICENSE, distributed with this software.
  |----------------------------------------------------------------------------*/
  /**
   * The namespace for array-specific algorithms.
   */
  var ArrayExt;
  (function (ArrayExt) {
      /**
       * Find the index of the first occurrence of a value in an array.
       *
       * @param array - The array-like object to search.
       *
       * @param value - The value to locate in the array. Values are
       *   compared using strict `===` equality.
       *
       * @param start - The index of the first element in the range to be
       *   searched, inclusive. The default value is `0`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @param stop - The index of the last element in the range to be
       *   searched, inclusive. The default value is `-1`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @returns The index of the first occurrence of the value, or `-1`
       *   if the value is not found.
       *
       * #### Notes
       * If `stop < start` the search will wrap at the end of the array.
       *
       * #### Complexity
       * Linear.
       *
       * #### Undefined Behavior
       * A `start` or `stop` which is non-integral.
       *
       * #### Example
       * ```typescript
       * import { ArrayExt } from '@lumino/algorithm';
       *
       * let data = ['one', 'two', 'three', 'four', 'one'];
       * ArrayExt.firstIndexOf(data, 'red');        // -1
       * ArrayExt.firstIndexOf(data, 'one');        // 0
       * ArrayExt.firstIndexOf(data, 'one', 1);     // 4
       * ArrayExt.firstIndexOf(data, 'two', 2);     // -1
       * ArrayExt.firstIndexOf(data, 'two', 2, 1);  // 1
       * ```
       */
      function firstIndexOf(array, value, start, stop) {
          if (start === void 0) { start = 0; }
          if (stop === void 0) { stop = -1; }
          var n = array.length;
          if (n === 0) {
              return -1;
          }
          if (start < 0) {
              start = Math.max(0, start + n);
          }
          else {
              start = Math.min(start, n - 1);
          }
          if (stop < 0) {
              stop = Math.max(0, stop + n);
          }
          else {
              stop = Math.min(stop, n - 1);
          }
          var span;
          if (stop < start) {
              span = (stop + 1) + (n - start);
          }
          else {
              span = stop - start + 1;
          }
          for (var i = 0; i < span; ++i) {
              var j = (start + i) % n;
              if (array[j] === value) {
                  return j;
              }
          }
          return -1;
      }
      ArrayExt.firstIndexOf = firstIndexOf;
      /**
       * Find the index of the last occurrence of a value in an array.
       *
       * @param array - The array-like object to search.
       *
       * @param value - The value to locate in the array. Values are
       *   compared using strict `===` equality.
       *
       * @param start - The index of the first element in the range to be
       *   searched, inclusive. The default value is `-1`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @param stop - The index of the last element in the range to be
       *   searched, inclusive. The default value is `0`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @returns The index of the last occurrence of the value, or `-1`
       *   if the value is not found.
       *
       * #### Notes
       * If `start < stop` the search will wrap at the front of the array.
       *
       * #### Complexity
       * Linear.
       *
       * #### Undefined Behavior
       * A `start` or `stop` which is non-integral.
       *
       * #### Example
       * ```typescript
       * import { ArrayExt } from '@lumino/algorithm';
       *
       * let data = ['one', 'two', 'three', 'four', 'one'];
       * ArrayExt.lastIndexOf(data, 'red');        // -1
       * ArrayExt.lastIndexOf(data, 'one');        // 4
       * ArrayExt.lastIndexOf(data, 'one', 1);     // 0
       * ArrayExt.lastIndexOf(data, 'two', 0);     // -1
       * ArrayExt.lastIndexOf(data, 'two', 0, 1);  // 1
       * ```
       */
      function lastIndexOf(array, value, start, stop) {
          if (start === void 0) { start = -1; }
          if (stop === void 0) { stop = 0; }
          var n = array.length;
          if (n === 0) {
              return -1;
          }
          if (start < 0) {
              start = Math.max(0, start + n);
          }
          else {
              start = Math.min(start, n - 1);
          }
          if (stop < 0) {
              stop = Math.max(0, stop + n);
          }
          else {
              stop = Math.min(stop, n - 1);
          }
          var span;
          if (start < stop) {
              span = (start + 1) + (n - stop);
          }
          else {
              span = start - stop + 1;
          }
          for (var i = 0; i < span; ++i) {
              var j = (start - i + n) % n;
              if (array[j] === value) {
                  return j;
              }
          }
          return -1;
      }
      ArrayExt.lastIndexOf = lastIndexOf;
      /**
       * Find the index of the first value which matches a predicate.
       *
       * @param array - The array-like object to search.
       *
       * @param fn - The predicate function to apply to the values.
       *
       * @param start - The index of the first element in the range to be
       *   searched, inclusive. The default value is `0`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @param stop - The index of the last element in the range to be
       *   searched, inclusive. The default value is `-1`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @returns The index of the first matching value, or `-1` if no
       *   matching value is found.
       *
       * #### Notes
       * If `stop < start` the search will wrap at the end of the array.
       *
       * #### Complexity
       * Linear.
       *
       * #### Undefined Behavior
       * A `start` or `stop` which is non-integral.
       *
       * Modifying the length of the array while searching.
       *
       * #### Example
       * ```typescript
       * import { ArrayExt } from '@lumino/algorithm';
       *
       * function isEven(value: number): boolean {
       *   return value % 2 === 0;
       * }
       *
       * let data = [1, 2, 3, 4, 3, 2, 1];
       * ArrayExt.findFirstIndex(data, isEven);       // 1
       * ArrayExt.findFirstIndex(data, isEven, 4);    // 5
       * ArrayExt.findFirstIndex(data, isEven, 6);    // -1
       * ArrayExt.findFirstIndex(data, isEven, 6, 5); // 1
       * ```
       */
      function findFirstIndex(array, fn, start, stop) {
          if (start === void 0) { start = 0; }
          if (stop === void 0) { stop = -1; }
          var n = array.length;
          if (n === 0) {
              return -1;
          }
          if (start < 0) {
              start = Math.max(0, start + n);
          }
          else {
              start = Math.min(start, n - 1);
          }
          if (stop < 0) {
              stop = Math.max(0, stop + n);
          }
          else {
              stop = Math.min(stop, n - 1);
          }
          var span;
          if (stop < start) {
              span = (stop + 1) + (n - start);
          }
          else {
              span = stop - start + 1;
          }
          for (var i = 0; i < span; ++i) {
              var j = (start + i) % n;
              if (fn(array[j], j)) {
                  return j;
              }
          }
          return -1;
      }
      ArrayExt.findFirstIndex = findFirstIndex;
      /**
       * Find the index of the last value which matches a predicate.
       *
       * @param object - The array-like object to search.
       *
       * @param fn - The predicate function to apply to the values.
       *
       * @param start - The index of the first element in the range to be
       *   searched, inclusive. The default value is `-1`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @param stop - The index of the last element in the range to be
       *   searched, inclusive. The default value is `0`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @returns The index of the last matching value, or `-1` if no
       *   matching value is found.
       *
       * #### Notes
       * If `start < stop` the search will wrap at the front of the array.
       *
       * #### Complexity
       * Linear.
       *
       * #### Undefined Behavior
       * A `start` or `stop` which is non-integral.
       *
       * Modifying the length of the array while searching.
       *
       * #### Example
       * ```typescript
       * import { ArrayExt } from '@lumino/algorithm';
       *
       * function isEven(value: number): boolean {
       *   return value % 2 === 0;
       * }
       *
       * let data = [1, 2, 3, 4, 3, 2, 1];
       * ArrayExt.findLastIndex(data, isEven);        // 5
       * ArrayExt.findLastIndex(data, isEven, 4);     // 3
       * ArrayExt.findLastIndex(data, isEven, 0);     // -1
       * ArrayExt.findLastIndex(data, isEven, 0, 1);  // 5
       * ```
       */
      function findLastIndex(array, fn, start, stop) {
          if (start === void 0) { start = -1; }
          if (stop === void 0) { stop = 0; }
          var n = array.length;
          if (n === 0) {
              return -1;
          }
          if (start < 0) {
              start = Math.max(0, start + n);
          }
          else {
              start = Math.min(start, n - 1);
          }
          if (stop < 0) {
              stop = Math.max(0, stop + n);
          }
          else {
              stop = Math.min(stop, n - 1);
          }
          var d;
          if (start < stop) {
              d = (start + 1) + (n - stop);
          }
          else {
              d = start - stop + 1;
          }
          for (var i = 0; i < d; ++i) {
              var j = (start - i + n) % n;
              if (fn(array[j], j)) {
                  return j;
              }
          }
          return -1;
      }
      ArrayExt.findLastIndex = findLastIndex;
      /**
       * Find the first value which matches a predicate.
       *
       * @param array - The array-like object to search.
       *
       * @param fn - The predicate function to apply to the values.
       *
       * @param start - The index of the first element in the range to be
       *   searched, inclusive. The default value is `0`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @param stop - The index of the last element in the range to be
       *   searched, inclusive. The default value is `-1`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @returns The first matching value, or `undefined` if no matching
       *   value is found.
       *
       * #### Notes
       * If `stop < start` the search will wrap at the end of the array.
       *
       * #### Complexity
       * Linear.
       *
       * #### Undefined Behavior
       * A `start` or `stop` which is non-integral.
       *
       * Modifying the length of the array while searching.
       *
       * #### Example
       * ```typescript
       * import { ArrayExt } from '@lumino/algorithm';
       *
       * function isEven(value: number): boolean {
       *   return value % 2 === 0;
       * }
       *
       * let data = [1, 2, 3, 4, 3, 2, 1];
       * ArrayExt.findFirstValue(data, isEven);       // 2
       * ArrayExt.findFirstValue(data, isEven, 2);    // 4
       * ArrayExt.findFirstValue(data, isEven, 6);    // undefined
       * ArrayExt.findFirstValue(data, isEven, 6, 5); // 2
       * ```
       */
      function findFirstValue(array, fn, start, stop) {
          if (start === void 0) { start = 0; }
          if (stop === void 0) { stop = -1; }
          var index = findFirstIndex(array, fn, start, stop);
          return index !== -1 ? array[index] : undefined;
      }
      ArrayExt.findFirstValue = findFirstValue;
      /**
       * Find the last value which matches a predicate.
       *
       * @param object - The array-like object to search.
       *
       * @param fn - The predicate function to apply to the values.
       *
       * @param start - The index of the first element in the range to be
       *   searched, inclusive. The default value is `-1`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @param stop - The index of the last element in the range to be
       *   searched, inclusive. The default value is `0`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @returns The last matching value, or `undefined` if no matching
       *   value is found.
       *
       * #### Notes
       * If `start < stop` the search will wrap at the front of the array.
       *
       * #### Complexity
       * Linear.
       *
       * #### Undefined Behavior
       * A `start` or `stop` which is non-integral.
       *
       * Modifying the length of the array while searching.
       *
       * #### Example
       * ```typescript
       * import { ArrayExt } from '@lumino/algorithm';
       *
       * function isEven(value: number): boolean {
       *   return value % 2 === 0;
       * }
       *
       * let data = [1, 2, 3, 4, 3, 2, 1];
       * ArrayExt.findLastValue(data, isEven);        // 2
       * ArrayExt.findLastValue(data, isEven, 4);     // 4
       * ArrayExt.findLastValue(data, isEven, 0);     // undefined
       * ArrayExt.findLastValue(data, isEven, 0, 1);  // 2
       * ```
       */
      function findLastValue(array, fn, start, stop) {
          if (start === void 0) { start = -1; }
          if (stop === void 0) { stop = 0; }
          var index = findLastIndex(array, fn, start, stop);
          return index !== -1 ? array[index] : undefined;
      }
      ArrayExt.findLastValue = findLastValue;
      /**
       * Find the index of the first element which compares `>=` to a value.
       *
       * @param array - The sorted array-like object to search.
       *
       * @param value - The value to locate in the array.
       *
       * @param fn - The 3-way comparison function to apply to the values.
       *   It should return `< 0` if an element is less than a value, `0` if
       *   an element is equal to a value, or `> 0` if an element is greater
       *   than a value.
       *
       * @param start - The index of the first element in the range to be
       *   searched, inclusive. The default value is `0`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @param stop - The index of the last element in the range to be
       *   searched, inclusive. The default value is `-1`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @returns The index of the first element which compares `>=` to the
       *   value, or `length` if there is no such element. If the computed
       *   index for `stop` is less than `start`, then the computed index
       *   for `start` is returned.
       *
       * #### Notes
       * The array must already be sorted in ascending order according to
       * the comparison function.
       *
       * #### Complexity
       * Logarithmic.
       *
       * #### Undefined Behavior
       * Searching a range which is not sorted in ascending order.
       *
       * A `start` or `stop` which is non-integral.
       *
       * Modifying the length of the array while searching.
       *
       * #### Example
       * ```typescript
       * import { ArrayExt } from '@lumino/algorithm';
       *
       * function numberCmp(a: number, b: number): number {
       *   return a - b;
       * }
       *
       * let data = [0, 3, 4, 7, 7, 9];
       * ArrayExt.lowerBound(data, 0, numberCmp);   // 0
       * ArrayExt.lowerBound(data, 6, numberCmp);   // 3
       * ArrayExt.lowerBound(data, 7, numberCmp);   // 3
       * ArrayExt.lowerBound(data, -1, numberCmp);  // 0
       * ArrayExt.lowerBound(data, 10, numberCmp);  // 6
       * ```
       */
      function lowerBound(array, value, fn, start, stop) {
          if (start === void 0) { start = 0; }
          if (stop === void 0) { stop = -1; }
          var n = array.length;
          if (n === 0) {
              return 0;
          }
          if (start < 0) {
              start = Math.max(0, start + n);
          }
          else {
              start = Math.min(start, n - 1);
          }
          if (stop < 0) {
              stop = Math.max(0, stop + n);
          }
          else {
              stop = Math.min(stop, n - 1);
          }
          var begin = start;
          var span = stop - start + 1;
          while (span > 0) {
              var half = span >> 1;
              var middle = begin + half;
              if (fn(array[middle], value) < 0) {
                  begin = middle + 1;
                  span -= half + 1;
              }
              else {
                  span = half;
              }
          }
          return begin;
      }
      ArrayExt.lowerBound = lowerBound;
      /**
       * Find the index of the first element which compares `>` than a value.
       *
       * @param array - The sorted array-like object to search.
       *
       * @param value - The value to locate in the array.
       *
       * @param fn - The 3-way comparison function to apply to the values.
       *   It should return `< 0` if an element is less than a value, `0` if
       *   an element is equal to a value, or `> 0` if an element is greater
       *   than a value.
       *
       * @param start - The index of the first element in the range to be
       *   searched, inclusive. The default value is `0`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @param stop - The index of the last element in the range to be
       *   searched, inclusive. The default value is `-1`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @returns The index of the first element which compares `>` than the
       *   value, or `length` if there is no such element. If the computed
       *   index for `stop` is less than `start`, then the computed index
       *   for `start` is returned.
       *
       * #### Notes
       * The array must already be sorted in ascending order according to
       * the comparison function.
       *
       * #### Complexity
       * Logarithmic.
       *
       * #### Undefined Behavior
       * Searching a range which is not sorted in ascending order.
       *
       * A `start` or `stop` which is non-integral.
       *
       * Modifying the length of the array while searching.
       *
       * #### Example
       * ```typescript
       * import { ArrayExt } from '@lumino/algorithm';
       *
       * function numberCmp(a: number, b: number): number {
       *   return a - b;
       * }
       *
       * let data = [0, 3, 4, 7, 7, 9];
       * ArrayExt.upperBound(data, 0, numberCmp);   // 1
       * ArrayExt.upperBound(data, 6, numberCmp);   // 3
       * ArrayExt.upperBound(data, 7, numberCmp);   // 5
       * ArrayExt.upperBound(data, -1, numberCmp);  // 0
       * ArrayExt.upperBound(data, 10, numberCmp);  // 6
       * ```
       */
      function upperBound(array, value, fn, start, stop) {
          if (start === void 0) { start = 0; }
          if (stop === void 0) { stop = -1; }
          var n = array.length;
          if (n === 0) {
              return 0;
          }
          if (start < 0) {
              start = Math.max(0, start + n);
          }
          else {
              start = Math.min(start, n - 1);
          }
          if (stop < 0) {
              stop = Math.max(0, stop + n);
          }
          else {
              stop = Math.min(stop, n - 1);
          }
          var begin = start;
          var span = stop - start + 1;
          while (span > 0) {
              var half = span >> 1;
              var middle = begin + half;
              if (fn(array[middle], value) > 0) {
                  span = half;
              }
              else {
                  begin = middle + 1;
                  span -= half + 1;
              }
          }
          return begin;
      }
      ArrayExt.upperBound = upperBound;
      /**
       * Test whether two arrays are shallowly equal.
       *
       * @param a - The first array-like object to compare.
       *
       * @param b - The second array-like object to compare.
       *
       * @param fn - The comparison function to apply to the elements. It
       *   should return `true` if the elements are "equal". The default
       *   compares elements using strict `===` equality.
       *
       * @returns Whether the two arrays are shallowly equal.
       *
       * #### Complexity
       * Linear.
       *
       * #### Undefined Behavior
       * Modifying the length of the arrays while comparing.
       *
       * #### Example
       * ```typescript
       * import { ArrayExt } from '@lumino/algorithm';
       *
       * let d1 = [0, 3, 4, 7, 7, 9];
       * let d2 = [0, 3, 4, 7, 7, 9];
       * let d3 = [42];
       * ArrayExt.shallowEqual(d1, d2);  // true
       * ArrayExt.shallowEqual(d2, d3);  // false
       * ```
       */
      function shallowEqual(a, b, fn) {
          // Check for object identity first.
          if (a === b) {
              return true;
          }
          // Bail early if the lengths are different.
          if (a.length !== b.length) {
              return false;
          }
          // Compare each element for equality.
          for (var i = 0, n = a.length; i < n; ++i) {
              if (fn ? !fn(a[i], b[i]) : a[i] !== b[i]) {
                  return false;
              }
          }
          // The array are shallowly equal.
          return true;
      }
      ArrayExt.shallowEqual = shallowEqual;
      /**
       * Create a slice of an array subject to an optional step.
       *
       * @param array - The array-like object of interest.
       *
       * @param options - The options for configuring the slice.
       *
       * @returns A new array with the specified values.
       *
       * @throws An exception if the slice `step` is `0`.
       *
       * #### Complexity
       * Linear.
       *
       * #### Undefined Behavior
       * A `start`, `stop`, or `step` which is non-integral.
       *
       * #### Example
       * ```typescript
       * import { ArrayExt } from '@lumino/algorithm';
       *
       * let data = [0, 3, 4, 7, 7, 9];
       * ArrayExt.slice(data);                         // [0, 3, 4, 7, 7, 9]
       * ArrayExt.slice(data, { start: 2 });           // [4, 7, 7, 9]
       * ArrayExt.slice(data, { start: 0, stop: 4 });  // [0, 3, 4, 7]
       * ArrayExt.slice(data, { step: 2 });            // [0, 4, 7]
       * ArrayExt.slice(data, { step: -1 });           // [9, 7, 7, 4, 3, 0]
       * ```
       */
      function slice(array, options) {
          if (options === void 0) { options = {}; }
          // Extract the options.
          var start = options.start, stop = options.stop, step = options.step;
          // Set up the `step` value.
          if (step === undefined) {
              step = 1;
          }
          // Validate the step size.
          if (step === 0) {
              throw new Error('Slice `step` cannot be zero.');
          }
          // Look up the length of the array.
          var n = array.length;
          // Set up the `start` value.
          if (start === undefined) {
              start = step < 0 ? n - 1 : 0;
          }
          else if (start < 0) {
              start = Math.max(start + n, step < 0 ? -1 : 0);
          }
          else if (start >= n) {
              start = step < 0 ? n - 1 : n;
          }
          // Set up the `stop` value.
          if (stop === undefined) {
              stop = step < 0 ? -1 : n;
          }
          else if (stop < 0) {
              stop = Math.max(stop + n, step < 0 ? -1 : 0);
          }
          else if (stop >= n) {
              stop = step < 0 ? n - 1 : n;
          }
          // Compute the slice length.
          var length;
          if ((step < 0 && stop >= start) || (step > 0 && start >= stop)) {
              length = 0;
          }
          else if (step < 0) {
              length = Math.floor((stop - start + 1) / step + 1);
          }
          else {
              length = Math.floor((stop - start - 1) / step + 1);
          }
          // Compute the sliced result.
          var result = [];
          for (var i = 0; i < length; ++i) {
              result[i] = array[start + i * step];
          }
          // Return the result.
          return result;
      }
      ArrayExt.slice = slice;
      /**
       * Move an element in an array from one index to another.
       *
       * @param array - The mutable array-like object of interest.
       *
       * @param fromIndex - The index of the element to move. Negative
       *   values are taken as an offset from the end of the array.
       *
       * @param toIndex - The target index of the element. Negative
       *   values are taken as an offset from the end of the array.
       *
       * #### Complexity
       * Linear.
       *
       * #### Undefined Behavior
       * A `fromIndex` or `toIndex` which is non-integral.
       *
       * #### Example
       * ```typescript
       * import { ArrayExt } from from '@lumino/algorithm';
       *
       * let data = [0, 1, 2, 3, 4];
       * ArrayExt.move(data, 1, 2);  // [0, 2, 1, 3, 4]
       * ArrayExt.move(data, 4, 2);  // [0, 2, 4, 1, 3]
       * ```
       */
      function move(array, fromIndex, toIndex) {
          var n = array.length;
          if (n <= 1) {
              return;
          }
          if (fromIndex < 0) {
              fromIndex = Math.max(0, fromIndex + n);
          }
          else {
              fromIndex = Math.min(fromIndex, n - 1);
          }
          if (toIndex < 0) {
              toIndex = Math.max(0, toIndex + n);
          }
          else {
              toIndex = Math.min(toIndex, n - 1);
          }
          if (fromIndex === toIndex) {
              return;
          }
          var value = array[fromIndex];
          var d = fromIndex < toIndex ? 1 : -1;
          for (var i = fromIndex; i !== toIndex; i += d) {
              array[i] = array[i + d];
          }
          array[toIndex] = value;
      }
      ArrayExt.move = move;
      /**
       * Reverse an array in-place.
       *
       * @param array - The mutable array-like object of interest.
       *
       * @param start - The index of the first element in the range to be
       *   reversed, inclusive. The default value is `0`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @param stop - The index of the last element in the range to be
       *   reversed, inclusive. The default value is `-1`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * #### Complexity
       * Linear.
       *
       * #### Undefined Behavior
       * A `start` or  `stop` index which is non-integral.
       *
       * #### Example
       * ```typescript
       * import { ArrayExt } from '@lumino/algorithm';
       *
       * let data = [0, 1, 2, 3, 4];
       * ArrayExt.reverse(data, 1, 3);  // [0, 3, 2, 1, 4]
       * ArrayExt.reverse(data, 3);     // [0, 3, 2, 4, 1]
       * ArrayExt.reverse(data);        // [1, 4, 2, 3, 0]
       * ```
       */
      function reverse(array, start, stop) {
          if (start === void 0) { start = 0; }
          if (stop === void 0) { stop = -1; }
          var n = array.length;
          if (n <= 1) {
              return;
          }
          if (start < 0) {
              start = Math.max(0, start + n);
          }
          else {
              start = Math.min(start, n - 1);
          }
          if (stop < 0) {
              stop = Math.max(0, stop + n);
          }
          else {
              stop = Math.min(stop, n - 1);
          }
          while (start < stop) {
              var a = array[start];
              var b = array[stop];
              array[start++] = b;
              array[stop--] = a;
          }
      }
      ArrayExt.reverse = reverse;
      /**
       * Rotate the elements of an array in-place.
       *
       * @param array - The mutable array-like object of interest.
       *
       * @param delta - The amount of rotation to apply to the elements. A
       *   positive value will rotate the elements to the left. A negative
       *   value will rotate the elements to the right.
       *
       * @param start - The index of the first element in the range to be
       *   rotated, inclusive. The default value is `0`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @param stop - The index of the last element in the range to be
       *   rotated, inclusive. The default value is `-1`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * #### Complexity
       * Linear.
       *
       * #### Undefined Behavior
       * A `delta`, `start`, or `stop` which is non-integral.
       *
       * #### Example
       * ```typescript
       * import { ArrayExt } from '@lumino/algorithm';
       *
       * let data = [0, 1, 2, 3, 4];
       * ArrayExt.rotate(data, 2);        // [2, 3, 4, 0, 1]
       * ArrayExt.rotate(data, -2);       // [0, 1, 2, 3, 4]
       * ArrayExt.rotate(data, 10);       // [0, 1, 2, 3, 4]
       * ArrayExt.rotate(data, 9);        // [4, 0, 1, 2, 3]
       * ArrayExt.rotate(data, 2, 1, 3);  // [4, 2, 0, 1, 3]
       * ```
       */
      function rotate(array, delta, start, stop) {
          if (start === void 0) { start = 0; }
          if (stop === void 0) { stop = -1; }
          var n = array.length;
          if (n <= 1) {
              return;
          }
          if (start < 0) {
              start = Math.max(0, start + n);
          }
          else {
              start = Math.min(start, n - 1);
          }
          if (stop < 0) {
              stop = Math.max(0, stop + n);
          }
          else {
              stop = Math.min(stop, n - 1);
          }
          if (start >= stop) {
              return;
          }
          var length = stop - start + 1;
          if (delta > 0) {
              delta = delta % length;
          }
          else if (delta < 0) {
              delta = ((delta % length) + length) % length;
          }
          if (delta === 0) {
              return;
          }
          var pivot = start + delta;
          reverse(array, start, pivot - 1);
          reverse(array, pivot, stop);
          reverse(array, start, stop);
      }
      ArrayExt.rotate = rotate;
      /**
       * Fill an array with a static value.
       *
       * @param array - The mutable array-like object to fill.
       *
       * @param value - The static value to use to fill the array.
       *
       * @param start - The index of the first element in the range to be
       *   filled, inclusive. The default value is `0`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @param stop - The index of the last element in the range to be
       *   filled, inclusive. The default value is `-1`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * #### Notes
       * If `stop < start` the fill will wrap at the end of the array.
       *
       * #### Complexity
       * Linear.
       *
       * #### Undefined Behavior
       * A `start` or `stop` which is non-integral.
       *
       * #### Example
       * ```typescript
       * import { ArrayExt } from '@lumino/algorithm';
       *
       * let data = ['one', 'two', 'three', 'four'];
       * ArrayExt.fill(data, 'r');        // ['r', 'r', 'r', 'r']
       * ArrayExt.fill(data, 'g', 1);     // ['r', 'g', 'g', 'g']
       * ArrayExt.fill(data, 'b', 2, 3);  // ['r', 'g', 'b', 'b']
       * ArrayExt.fill(data, 'z', 3, 1);  // ['z', 'z', 'b', 'z']
       * ```
       */
      function fill(array, value, start, stop) {
          if (start === void 0) { start = 0; }
          if (stop === void 0) { stop = -1; }
          var n = array.length;
          if (n === 0) {
              return;
          }
          if (start < 0) {
              start = Math.max(0, start + n);
          }
          else {
              start = Math.min(start, n - 1);
          }
          if (stop < 0) {
              stop = Math.max(0, stop + n);
          }
          else {
              stop = Math.min(stop, n - 1);
          }
          var span;
          if (stop < start) {
              span = (stop + 1) + (n - start);
          }
          else {
              span = stop - start + 1;
          }
          for (var i = 0; i < span; ++i) {
              array[(start + i) % n] = value;
          }
      }
      ArrayExt.fill = fill;
      /**
       * Insert a value into an array at a specific index.
       *
       * @param array - The array of interest.
       *
       * @param index - The index at which to insert the value. Negative
       *   values are taken as an offset from the end of the array.
       *
       * @param value - The value to set at the specified index.
       *
       * #### Complexity
       * Linear.
       *
       * #### Undefined Behavior
       * An `index` which is non-integral.
       *
       * #### Example
       * ```typescript
       * import { ArrayExt } from '@lumino/algorithm';
       *
       * let data = [0, 1, 2];
       * ArrayExt.insert(data, 0, -1);  // [-1, 0, 1, 2]
       * ArrayExt.insert(data, 2, 12);  // [-1, 0, 12, 1, 2]
       * ArrayExt.insert(data, -1, 7);  // [-1, 0, 12, 1, 7, 2]
       * ArrayExt.insert(data, 6, 19);  // [-1, 0, 12, 1, 7, 2, 19]
       * ```
       */
      function insert(array, index, value) {
          var n = array.length;
          if (index < 0) {
              index = Math.max(0, index + n);
          }
          else {
              index = Math.min(index, n);
          }
          for (var i = n; i > index; --i) {
              array[i] = array[i - 1];
          }
          array[index] = value;
      }
      ArrayExt.insert = insert;
      /**
       * Remove and return a value at a specific index in an array.
       *
       * @param array - The array of interest.
       *
       * @param index - The index of the value to remove. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @returns The value at the specified index, or `undefined` if the
       *   index is out of range.
       *
       * #### Complexity
       * Linear.
       *
       * #### Undefined Behavior
       * An `index` which is non-integral.
       *
       * #### Example
       * ```typescript
       * import { ArrayExt } from '@lumino/algorithm';
       *
       * let data = [0, 12, 23, 39, 14, 12, 75];
       * ArrayExt.removeAt(data, 2);   // 23
       * ArrayExt.removeAt(data, -2);  // 12
       * ArrayExt.removeAt(data, 10);  // undefined;
       * ```
       */
      function removeAt(array, index) {
          var n = array.length;
          if (index < 0) {
              index += n;
          }
          if (index < 0 || index >= n) {
              return undefined;
          }
          var value = array[index];
          for (var i = index + 1; i < n; ++i) {
              array[i - 1] = array[i];
          }
          array.length = n - 1;
          return value;
      }
      ArrayExt.removeAt = removeAt;
      /**
       * Remove the first occurrence of a value from an array.
       *
       * @param array - The array of interest.
       *
       * @param value - The value to remove from the array. Values are
       *   compared using strict `===` equality.
       *
       * @param start - The index of the first element in the range to be
       *   searched, inclusive. The default value is `0`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @param stop - The index of the last element in the range to be
       *   searched, inclusive. The default value is `-1`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @returns The index of the removed value, or `-1` if the value
       *   is not contained in the array.
       *
       * #### Notes
       * If `stop < start` the search will wrap at the end of the array.
       *
       * #### Complexity
       * Linear.
       *
       * #### Example
       * ```typescript
       * import { ArrayExt } from '@lumino/algorithm';
       *
       * let data = [0, 12, 23, 39, 14, 12, 75];
       * ArrayExt.removeFirstOf(data, 12);        // 1
       * ArrayExt.removeFirstOf(data, 17);        // -1
       * ArrayExt.removeFirstOf(data, 39, 3);     // -1
       * ArrayExt.removeFirstOf(data, 39, 3, 2);  // 2
       * ```
       */
      function removeFirstOf(array, value, start, stop) {
          if (start === void 0) { start = 0; }
          if (stop === void 0) { stop = -1; }
          var index = firstIndexOf(array, value, start, stop);
          if (index !== -1) {
              removeAt(array, index);
          }
          return index;
      }
      ArrayExt.removeFirstOf = removeFirstOf;
      /**
       * Remove the last occurrence of a value from an array.
       *
       * @param array - The array of interest.
       *
       * @param value - The value to remove from the array. Values are
       *   compared using strict `===` equality.
       *
       * @param start - The index of the first element in the range to be
       *   searched, inclusive. The default value is `-1`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @param stop - The index of the last element in the range to be
       *   searched, inclusive. The default value is `0`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @returns The index of the removed value, or `-1` if the value
       *   is not contained in the array.
       *
       * #### Notes
       * If `start < stop` the search will wrap at the end of the array.
       *
       * #### Complexity
       * Linear.
       *
       * #### Example
       * ```typescript
       * import { ArrayExt } from '@lumino/algorithm';
       *
       * let data = [0, 12, 23, 39, 14, 12, 75];
       * ArrayExt.removeLastOf(data, 12);        // 5
       * ArrayExt.removeLastOf(data, 17);        // -1
       * ArrayExt.removeLastOf(data, 39, 2);     // -1
       * ArrayExt.removeLastOf(data, 39, 2, 3);  // 3
       * ```
       */
      function removeLastOf(array, value, start, stop) {
          if (start === void 0) { start = -1; }
          if (stop === void 0) { stop = 0; }
          var index = lastIndexOf(array, value, start, stop);
          if (index !== -1) {
              removeAt(array, index);
          }
          return index;
      }
      ArrayExt.removeLastOf = removeLastOf;
      /**
       * Remove all occurrences of a value from an array.
       *
       * @param array - The array of interest.
       *
       * @param value - The value to remove from the array. Values are
       *   compared using strict `===` equality.
       *
       * @param start - The index of the first element in the range to be
       *   searched, inclusive. The default value is `0`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @param stop - The index of the last element in the range to be
       *   searched, inclusive. The default value is `-1`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @returns The number of elements removed from the array.
       *
       * #### Notes
       * If `stop < start` the search will conceptually wrap at the end of
       * the array, however the array will be traversed front-to-back.
       *
       * #### Complexity
       * Linear.
       *
       * #### Example
       * ```typescript
       * import { ArrayExt } from '@lumino/algorithm';
       *
       * let data = [14, 12, 23, 39, 14, 12, 19, 14];
       * ArrayExt.removeAllOf(data, 12);        // 2
       * ArrayExt.removeAllOf(data, 17);        // 0
       * ArrayExt.removeAllOf(data, 14, 1, 4);  // 1
       * ```
       */
      function removeAllOf(array, value, start, stop) {
          if (start === void 0) { start = 0; }
          if (stop === void 0) { stop = -1; }
          var n = array.length;
          if (n === 0) {
              return 0;
          }
          if (start < 0) {
              start = Math.max(0, start + n);
          }
          else {
              start = Math.min(start, n - 1);
          }
          if (stop < 0) {
              stop = Math.max(0, stop + n);
          }
          else {
              stop = Math.min(stop, n - 1);
          }
          var count = 0;
          for (var i = 0; i < n; ++i) {
              if (start <= stop && (i >= start && i <= stop) && array[i] === value) {
                  count++;
              }
              else if (stop < start && (i <= stop || i >= start) && array[i] === value) {
                  count++;
              }
              else if (count > 0) {
                  array[i - count] = array[i];
              }
          }
          if (count > 0) {
              array.length = n - count;
          }
          return count;
      }
      ArrayExt.removeAllOf = removeAllOf;
      /**
       * Remove the first occurrence of a value which matches a predicate.
       *
       * @param array - The array of interest.
       *
       * @param fn - The predicate function to apply to the values.
       *
       * @param start - The index of the first element in the range to be
       *   searched, inclusive. The default value is `0`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @param stop - The index of the last element in the range to be
       *   searched, inclusive. The default value is `-1`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @returns The removed `{ index, value }`, which will be `-1` and
       *   `undefined` if the value is not contained in the array.
       *
       * #### Notes
       * If `stop < start` the search will wrap at the end of the array.
       *
       * #### Complexity
       * Linear.
       *
       * #### Example
       * ```typescript
       * import { ArrayExt } from '@lumino/algorithm';
       *
       * function isEven(value: number): boolean {
       *   return value % 2 === 0;
       * }
       *
       * let data = [0, 12, 23, 39, 14, 12, 75];
       * ArrayExt.removeFirstWhere(data, isEven);     // { index: 0, value: 0 }
       * ArrayExt.removeFirstWhere(data, isEven, 2);  // { index: 3, value: 14 }
       * ArrayExt.removeFirstWhere(data, isEven, 4);  // { index: -1, value: undefined }
       * ```
       */
      function removeFirstWhere(array, fn, start, stop) {
          if (start === void 0) { start = 0; }
          if (stop === void 0) { stop = -1; }
          var value;
          var index = findFirstIndex(array, fn, start, stop);
          if (index !== -1) {
              value = removeAt(array, index);
          }
          return { index: index, value: value };
      }
      ArrayExt.removeFirstWhere = removeFirstWhere;
      /**
       * Remove the last occurrence of a value which matches a predicate.
       *
       * @param array - The array of interest.
       *
       * @param fn - The predicate function to apply to the values.
       *
       * @param start - The index of the first element in the range to be
       *   searched, inclusive. The default value is `-1`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @param stop - The index of the last element in the range to be
       *   searched, inclusive. The default value is `0`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @returns The removed `{ index, value }`, which will be `-1` and
       *   `undefined` if the value is not contained in the array.
       *
       * #### Notes
       * If `start < stop` the search will wrap at the end of the array.
       *
       * #### Complexity
       * Linear.
       *
       * #### Example
       * ```typescript
       * import { ArrayExt } from '@lumino/algorithm';
       *
       * function isEven(value: number): boolean {
       *   return value % 2 === 0;
       * }
       *
       * let data = [0, 12, 23, 39, 14, 12, 75];
       * ArrayExt.removeLastWhere(data, isEven);        // { index: 5, value: 12 }
       * ArrayExt.removeLastWhere(data, isEven, 2);     // { index: 1, value: 12 }
       * ArrayExt.removeLastWhere(data, isEven, 2, 1);  // { index: -1, value: undefined }
       * ```
       */
      function removeLastWhere(array, fn, start, stop) {
          if (start === void 0) { start = -1; }
          if (stop === void 0) { stop = 0; }
          var value;
          var index = findLastIndex(array, fn, start, stop);
          if (index !== -1) {
              value = removeAt(array, index);
          }
          return { index: index, value: value };
      }
      ArrayExt.removeLastWhere = removeLastWhere;
      /**
       * Remove all occurrences of values which match a predicate.
       *
       * @param array - The array of interest.
       *
       * @param fn - The predicate function to apply to the values.
       *
       * @param start - The index of the first element in the range to be
       *   searched, inclusive. The default value is `0`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @param stop - The index of the last element in the range to be
       *   searched, inclusive. The default value is `-1`. Negative values
       *   are taken as an offset from the end of the array.
       *
       * @returns The number of elements removed from the array.
       *
       * #### Notes
       * If `stop < start` the search will conceptually wrap at the end of
       * the array, however the array will be traversed front-to-back.
       *
       * #### Complexity
       * Linear.
       *
       * #### Example
       * ```typescript
       * import { ArrayExt } from '@lumino/algorithm';
       *
       * function isEven(value: number): boolean {
       *   return value % 2 === 0;
       * }
       *
       * function isNegative(value: number): boolean {
       *   return value < 0;
       * }
       *
       * let data = [0, 12, -13, -9, 23, 39, 14, -15, 12, 75];
       * ArrayExt.removeAllWhere(data, isEven);            // 4
       * ArrayExt.removeAllWhere(data, isNegative, 0, 3);  // 2
       * ```
       */
      function removeAllWhere(array, fn, start, stop) {
          if (start === void 0) { start = 0; }
          if (stop === void 0) { stop = -1; }
          var n = array.length;
          if (n === 0) {
              return 0;
          }
          if (start < 0) {
              start = Math.max(0, start + n);
          }
          else {
              start = Math.min(start, n - 1);
          }
          if (stop < 0) {
              stop = Math.max(0, stop + n);
          }
          else {
              stop = Math.min(stop, n - 1);
          }
          var count = 0;
          for (var i = 0; i < n; ++i) {
              if (start <= stop && (i >= start && i <= stop) && fn(array[i], i)) {
                  count++;
              }
              else if (stop < start && (i <= stop || i >= start) && fn(array[i], i)) {
                  count++;
              }
              else if (count > 0) {
                  array[i - count] = array[i];
              }
          }
          if (count > 0) {
              array.length = n - count;
          }
          return count;
      }
      ArrayExt.removeAllWhere = removeAllWhere;
  })(ArrayExt || (ArrayExt = {}));

  // Copyright (c) Jupyter Development Team.
  // Distributed under the terms of the Modified BSD License.
  /*-----------------------------------------------------------------------------
  | Copyright (c) 2014-2017, PhosphorJS Contributors
  |
  | Distributed under the terms of the BSD 3-Clause License.
  |
  | The full license is in the file LICENSE, distributed with this software.
  |----------------------------------------------------------------------------*/
  /**
   * Create an iterator for an iterable object.
   *
   * @param object - The iterable or array-like object of interest.
   *
   * @returns A new iterator for the given object.
   *
   * #### Notes
   * This function allows iteration algorithms to operate on user-defined
   * iterable types and builtin array-like objects in a uniform fashion.
   */
  function iter(object) {
      var it;
      if (typeof object.iter === 'function') {
          it = object.iter();
      }
      else {
          it = new ArrayIterator(object);
      }
      return it;
  }
  /**
   * Invoke a function for each value in an iterable.
   *
   * @param object - The iterable or array-like object of interest.
   *
   * @param fn - The callback function to invoke for each value.
   *
   * #### Notes
   * Iteration can be terminated early by returning `false` from the
   * callback function.
   *
   * #### Complexity
   * Linear.
   *
   * #### Example
   * ```typescript
   * import { each } from '@lumino/algorithm';
   *
   * let data = [5, 7, 0, -2, 9];
   *
   * each(data, value => { console.log(value); });
   * ```
   */
  function each$1(object, fn) {
      var index = 0;
      var it = iter(object);
      var value;
      while ((value = it.next()) !== undefined) {
          if (fn(value, index++) === false) {
              return;
          }
      }
  }
  /**
   * Test whether all values in an iterable satisfy a predicate.
   *
   * @param object - The iterable or array-like object of interest.
   *
   * @param fn - The predicate function to invoke for each value.
   *
   * @returns `true` if all values pass the test, `false` otherwise.
   *
   * #### Notes
   * Iteration terminates on the first `false` predicate result.
   *
   * #### Complexity
   * Linear.
   *
   * #### Example
   * ```typescript
   * import { every } from '@lumino/algorithm';
   *
   * let data = [5, 7, 1];
   *
   * every(data, value => value % 2 === 0);  // false
   * every(data, value => value % 2 === 1);  // true
   * ```
   */
  function every$1(object, fn) {
      var index = 0;
      var it = iter(object);
      var value;
      while ((value = it.next()) !== undefined) {
          if (!fn(value, index++)) {
              return false;
          }
      }
      return true;
  }
  /**
   * Test whether any value in an iterable satisfies a predicate.
   *
   * @param object - The iterable or array-like object of interest.
   *
   * @param fn - The predicate function to invoke for each value.
   *
   * @returns `true` if any value passes the test, `false` otherwise.
   *
   * #### Notes
   * Iteration terminates on the first `true` predicate result.
   *
   * #### Complexity
   * Linear.
   *
   * #### Example
   * ```typescript
   * import { some } from '@lumino/algorithm';
   *
   * let data = [5, 7, 1];
   *
   * some(data, value => value === 7);  // true
   * some(data, value => value === 3);  // false
   * ```
   */
  function some$1(object, fn) {
      var index = 0;
      var it = iter(object);
      var value;
      while ((value = it.next()) !== undefined) {
          if (fn(value, index++)) {
              return true;
          }
      }
      return false;
  }
  /**
   * Create an array from an iterable of values.
   *
   * @param object - The iterable or array-like object of interest.
   *
   * @returns A new array of values from the given object.
   *
   * #### Example
   * ```typescript
   * import { iter, toArray } from '@lumino/algorithm';
   *
   * let data = [1, 2, 3, 4, 5, 6];
   *
   * let stream = iter(data);
   *
   * toArray(stream);  // [1, 2, 3, 4, 5, 6];
   * ```
   */
  function toArray$1(object) {
      var index = 0;
      var result = [];
      var it = iter(object);
      var value;
      while ((value = it.next()) !== undefined) {
          result[index++] = value;
      }
      return result;
  }
  /**
   * An iterator for an array-like object.
   *
   * #### Notes
   * This iterator can be used for any builtin JS array-like object.
   */
  var ArrayIterator = /** @class */ (function () {
      /**
       * Construct a new array iterator.
       *
       * @param source - The array-like object of interest.
       */
      function ArrayIterator(source) {
          this._index = 0;
          this._source = source;
      }
      /**
       * Get an iterator over the object's values.
       *
       * @returns An iterator which yields the object's values.
       */
      ArrayIterator.prototype.iter = function () {
          return this;
      };
      /**
       * Create an independent clone of the iterator.
       *
       * @returns A new independent clone of the iterator.
       */
      ArrayIterator.prototype.clone = function () {
          var result = new ArrayIterator(this._source);
          result._index = this._index;
          return result;
      };
      /**
       * Get the next value from the iterator.
       *
       * @returns The next value from the iterator, or `undefined`.
       */
      ArrayIterator.prototype.next = function () {
          if (this._index >= this._source.length) {
              return undefined;
          }
          return this._source[this._index++];
      };
      return ArrayIterator;
  }());

  // Copyright (c) Jupyter Development Team.
  /**
   * Chain together several iterables.
   *
   * @param objects - The iterable or array-like objects of interest.
   *
   * @returns An iterator which yields the values of the iterables
   *   in the order in which they are supplied.
   *
   * #### Example
   * ```typescript
   * import { chain, toArray } from '@lumino/algorithm';
   *
   * let data1 = [1, 2, 3];
   * let data2 = [4, 5, 6];
   *
   * let stream = chain(data1, data2);
   *
   * toArray(stream);  // [1, 2, 3, 4, 5, 6]
   * ```
   */
  function chain$1() {
      var objects = [];
      for (var _i = 0; _i < arguments.length; _i++) {
          objects[_i] = arguments[_i];
      }
      return new ChainIterator(iter(objects.map(iter)));
  }
  /**
   * An iterator which chains together several iterators.
   */
  var ChainIterator = /** @class */ (function () {
      /**
       * Construct a new chain iterator.
       *
       * @param source - The iterator of iterators of interest.
       */
      function ChainIterator(source) {
          this._cloned = false;
          this._source = source;
          this._active = undefined;
      }
      /**
       * Get an iterator over the object's values.
       *
       * @returns An iterator which yields the object's values.
       */
      ChainIterator.prototype.iter = function () {
          return this;
      };
      /**
       * Create an independent clone of the iterator.
       *
       * @returns A new independent clone of the iterator.
       */
      ChainIterator.prototype.clone = function () {
          var result = new ChainIterator(this._source.clone());
          result._active = this._active && this._active.clone();
          result._cloned = true;
          this._cloned = true;
          return result;
      };
      /**
       * Get the next value from the iterator.
       *
       * @returns The next value from the iterator, or `undefined`.
       */
      ChainIterator.prototype.next = function () {
          if (this._active === undefined) {
              var active = this._source.next();
              if (active === undefined) {
                  return undefined;
              }
              this._active = this._cloned ? active.clone() : active;
          }
          var value = this._active.next();
          if (value !== undefined) {
              return value;
          }
          this._active = undefined;
          return this.next();
      };
      return ChainIterator;
  }());

  /**
   * Create an empty iterator.
   *
   * @returns A new iterator which yields nothing.
   *
   * #### Example
   * ```typescript
   * import { empty, toArray } from '@lumino/algorithm';
   *
   * let stream = empty<number>();
   *
   * toArray(stream);  // []
   * ```
   */
  function empty() {
      return new EmptyIterator();
  }
  /**
   * An iterator which is always empty.
   */
  var EmptyIterator = /** @class */ (function () {
      /**
       * Construct a new empty iterator.
       */
      function EmptyIterator() {
      }
      /**
       * Get an iterator over the object's values.
       *
       * @returns An iterator which yields the object's values.
       */
      EmptyIterator.prototype.iter = function () {
          return this;
      };
      /**
       * Create an independent clone of the iterator.
       *
       * @returns A new independent clone of the iterator.
       */
      EmptyIterator.prototype.clone = function () {
          return new EmptyIterator();
      };
      /**
       * Get the next value from the iterator.
       *
       * @returns The next value from the iterator, or `undefined`.
       */
      EmptyIterator.prototype.next = function () {
          return undefined;
      };
      return EmptyIterator;
  }());

  // Copyright (c) Jupyter Development Team.
  /**
   * Filter an iterable for values which pass a test.
   *
   * @param object - The iterable or array-like object of interest.
   *
   * @param fn - The predicate function to invoke for each value.
   *
   * @returns An iterator which yields the values which pass the test.
   *
   * #### Example
   * ```typescript
   * import { filter, toArray } from '@lumino/algorithm';
   *
   * let data = [1, 2, 3, 4, 5, 6];
   *
   * let stream = filter(data, value => value % 2 === 0);
   *
   * toArray(stream);  // [2, 4, 6]
   * ```
   */
  function filter$1(object, fn) {
      return new FilterIterator(iter(object), fn);
  }
  /**
   * An iterator which yields values which pass a test.
   */
  var FilterIterator = /** @class */ (function () {
      /**
       * Construct a new filter iterator.
       *
       * @param source - The iterator of values of interest.
       *
       * @param fn - The predicate function to invoke for each value.
       */
      function FilterIterator(source, fn) {
          this._index = 0;
          this._source = source;
          this._fn = fn;
      }
      /**
       * Get an iterator over the object's values.
       *
       * @returns An iterator which yields the object's values.
       */
      FilterIterator.prototype.iter = function () {
          return this;
      };
      /**
       * Create an independent clone of the iterator.
       *
       * @returns A new independent clone of the iterator.
       */
      FilterIterator.prototype.clone = function () {
          var result = new FilterIterator(this._source.clone(), this._fn);
          result._index = this._index;
          return result;
      };
      /**
       * Get the next value from the iterator.
       *
       * @returns The next value from the iterator, or `undefined`.
       */
      FilterIterator.prototype.next = function () {
          var fn = this._fn;
          var it = this._source;
          var value;
          while ((value = it.next()) !== undefined) {
              if (fn(value, this._index++)) {
                  return value;
              }
          }
          return undefined;
      };
      return FilterIterator;
  }());

  // Copyright (c) Jupyter Development Team.
  /**
   * Find the first value in an iterable which matches a predicate.
   *
   * @param object - The iterable or array-like object to search.
   *
   * @param fn - The predicate function to apply to the values.
   *
   * @returns The first matching value, or `undefined` if no matching
   *   value is found.
   *
   * #### Complexity
   * Linear.
   *
   * #### Example
   * ```typescript
   * import { find } from '@lumino/algorithm';
   *
   * interface IAnimal { species: string, name: string };
   *
   * function isCat(value: IAnimal): boolean {
   *   return value.species === 'cat';
   * }
   *
   * let data: IAnimal[] = [
   *   { species: 'dog', name: 'spot' },
   *   { species: 'cat', name: 'fluffy' },
   *   { species: 'alligator', name: 'pocho' }
   * ];
   *
   * find(data, isCat).name;  // 'fluffy'
   * ```
   */
  function find$1(object, fn) {
      var index = 0;
      var it = iter(object);
      var value;
      while ((value = it.next()) !== undefined) {
          if (fn(value, index++)) {
              return value;
          }
      }
      return undefined;
  }
  /**
   * Find the maximum value in an iterable.
   *
   * @param object - The iterable or array-like object to search.
   *
   * @param fn - The 3-way comparison function to apply to the values.
   *   It should return `< 0` if the first value is less than the second.
   *   `0` if the values are equivalent, or `> 0` if the first value is
   *   greater than the second.
   *
   * @returns The maximum value in the iterable. If multiple values are
   *   equivalent to the maximum, the left-most value is returned. If
   *   the iterable is empty, this returns `undefined`.
   *
   * #### Complexity
   * Linear.
   *
   * #### Example
   * ```typescript
   * import { max } from '@lumino/algorithm';
   *
   * function numberCmp(a: number, b: number): number {
   *   return a - b;
   * }
   *
   * max([7, 4, 0, 3, 9, 4], numberCmp);  // 9
   * ```
   */
  function max$1(object, fn) {
      var it = iter(object);
      var value = it.next();
      if (value === undefined) {
          return undefined;
      }
      var result = value;
      while ((value = it.next()) !== undefined) {
          if (fn(value, result) > 0) {
              result = value;
          }
      }
      return result;
  }

  // Copyright (c) Jupyter Development Team.
  /**
   * Transform the values of an iterable with a mapping function.
   *
   * @param object - The iterable or array-like object of interest.
   *
   * @param fn - The mapping function to invoke for each value.
   *
   * @returns An iterator which yields the transformed values.
   *
   * #### Example
   * ```typescript
   * import { map, toArray } from '@lumino/algorithm';
   *
   * let data = [1, 2, 3];
   *
   * let stream = map(data, value => value * 2);
   *
   * toArray(stream);  // [2, 4, 6]
   * ```
   */
  function map$1(object, fn) {
      return new MapIterator(iter(object), fn);
  }
  /**
   * An iterator which transforms values using a mapping function.
   */
  var MapIterator = /** @class */ (function () {
      /**
       * Construct a new map iterator.
       *
       * @param source - The iterator of values of interest.
       *
       * @param fn - The mapping function to invoke for each value.
       */
      function MapIterator(source, fn) {
          this._index = 0;
          this._source = source;
          this._fn = fn;
      }
      /**
       * Get an iterator over the object's values.
       *
       * @returns An iterator which yields the object's values.
       */
      MapIterator.prototype.iter = function () {
          return this;
      };
      /**
       * Create an independent clone of the iterator.
       *
       * @returns A new independent clone of the iterator.
       */
      MapIterator.prototype.clone = function () {
          var result = new MapIterator(this._source.clone(), this._fn);
          result._index = this._index;
          return result;
      };
      /**
       * Get the next value from the iterator.
       *
       * @returns The next value from the iterator, or `undefined`.
       */
      MapIterator.prototype.next = function () {
          var value = this._source.next();
          if (value === undefined) {
              return undefined;
          }
          return this._fn.call(undefined, value, this._index++);
      };
      return MapIterator;
  }());
  /**
   * The namespace for the module implementation details.
   */
  var Private;
  (function (Private) {
      /**
       * Compute the effective length of a range.
       *
       * @param start - The starting value for the range, inclusive.
       *
       * @param stop - The stopping value for the range, exclusive.
       *
       * @param step - The distance between each value.
       *
       * @returns The number of steps need to traverse the range.
       */
      function rangeLength(start, stop, step) {
          if (step === 0) {
              return Infinity;
          }
          if (start > stop && step > 0) {
              return 0;
          }
          if (start < stop && step < 0) {
              return 0;
          }
          return Math.ceil((stop - start) / step);
      }
      Private.rangeLength = rangeLength;
  })(Private || (Private = {}));

  // Copyright (c) Jupyter Development Team.
  function reduce$1(object, fn, initial) {
      // Setup the iterator and fetch the first value.
      var index = 0;
      var it = iter(object);
      var first = it.next();
      // An empty iterator and no initial value is an error.
      if (first === undefined && initial === undefined) {
          throw new TypeError('Reduce of empty iterable with no initial value.');
      }
      // If the iterator is empty, return the initial value.
      if (first === undefined) {
          return initial;
      }
      // If the iterator has a single item and no initial value, the
      // reducer is not invoked and the first item is the return value.
      var second = it.next();
      if (second === undefined && initial === undefined) {
          return first;
      }
      // If iterator has a single item and an initial value is provided,
      // the reducer is invoked and that result is the return value.
      if (second === undefined) {
          return fn(initial, first, index++);
      }
      // Setup the initial accumlated value.
      var accumulator;
      if (initial === undefined) {
          accumulator = fn(first, second, index++);
      }
      else {
          accumulator = fn(fn(initial, first, index++), second, index++);
      }
      // Iterate the rest of the values, updating the accumulator.
      var next;
      while ((next = it.next()) !== undefined) {
          accumulator = fn(accumulator, next, index++);
      }
      // Return the final accumulated value.
      return accumulator;
  }
  /**
   * Create an iterator which yields a value a single time.
   *
   * @param value - The value to wrap in an iterator.
   *
   * @returns A new iterator which yields the value a single time.
   *
   * #### Example
   * ```typescript
   * import { once, toArray } from '@lumino/algorithm';
   *
   * let stream = once(7);
   *
   * toArray(stream);  // [7]
   * ```
   */
  function once$1(value) {
      return new RepeatIterator(value, 1);
  }
  /**
   * An iterator which repeats a value a specified number of times.
   */
  var RepeatIterator = /** @class */ (function () {
      /**
       * Construct a new repeat iterator.
       *
       * @param value - The value to repeat.
       *
       * @param count - The number of times to repeat the value.
       */
      function RepeatIterator(value, count) {
          this._value = value;
          this._count = count;
      }
      /**
       * Get an iterator over the object's values.
       *
       * @returns An iterator which yields the object's values.
       */
      RepeatIterator.prototype.iter = function () {
          return this;
      };
      /**
       * Create an independent clone of the iterator.
       *
       * @returns A new independent clone of the iterator.
       */
      RepeatIterator.prototype.clone = function () {
          return new RepeatIterator(this._value, this._count);
      };
      /**
       * Get the next value from the iterator.
       *
       * @returns The next value from the iterator, or `undefined`.
       */
      RepeatIterator.prototype.next = function () {
          if (this._count <= 0) {
              return undefined;
          }
          this._count--;
          return this._value;
      };
      return RepeatIterator;
  }());

  /**
   * Create an iterator for a retroable object.
   *
   * @param object - The retroable or array-like object of interest.
   *
   * @returns An iterator which traverses the object's values in reverse.
   *
   * #### Example
   * ```typescript
   * import { retro, toArray } from '@lumino/algorithm';
   *
   * let data = [1, 2, 3, 4, 5, 6];
   *
   * let stream = retro(data);
   *
   * toArray(stream);  // [6, 5, 4, 3, 2, 1]
   * ```
   */
  function retro(object) {
      var it;
      if (typeof object.retro === 'function') {
          it = object.retro();
      }
      else {
          it = new RetroArrayIterator(object);
      }
      return it;
  }
  /**
   * An iterator which traverses an array-like object in reverse.
   *
   * #### Notes
   * This iterator can be used for any builtin JS array-like object.
   */
  var RetroArrayIterator = /** @class */ (function () {
      /**
       * Construct a new retro iterator.
       *
       * @param source - The array-like object of interest.
       */
      function RetroArrayIterator(source) {
          this._source = source;
          this._index = source.length - 1;
      }
      /**
       * Get an iterator over the object's values.
       *
       * @returns An iterator which yields the object's values.
       */
      RetroArrayIterator.prototype.iter = function () {
          return this;
      };
      /**
       * Create an independent clone of the iterator.
       *
       * @returns A new independent clone of the iterator.
       */
      RetroArrayIterator.prototype.clone = function () {
          var result = new RetroArrayIterator(this._source);
          result._index = this._index;
          return result;
      };
      /**
       * Get the next value from the iterator.
       *
       * @returns The next value from the iterator, or `undefined`.
       */
      RetroArrayIterator.prototype.next = function () {
          if (this._index < 0 || this._index >= this._source.length) {
              return undefined;
          }
          return this._source[this._index--];
      };
      return RetroArrayIterator;
  }());

  // Copyright (c) Jupyter Development Team.
  // Distributed under the terms of the Modified BSD License.
  /*-----------------------------------------------------------------------------
  | Copyright (c) 2014-2017, PhosphorJS Contributors
  |
  | Distributed under the terms of the BSD 3-Clause License.
  |
  | The full license is in the file LICENSE, distributed with this software.
  |----------------------------------------------------------------------------*/
  /**
   * The namespace for string-specific algorithms.
   */
  var StringExt;
  (function (StringExt) {
      /**
       * Find the indices of characters in a source text.
       *
       * @param source - The source text which should be searched.
       *
       * @param query - The characters to locate in the source text.
       *
       * @param start - The index to start the search.
       *
       * @returns The matched indices, or `null` if there is no match.
       *
       * #### Complexity
       * Linear on `sourceText`.
       *
       * #### Notes
       * In order for there to be a match, all of the characters in `query`
       * **must** appear in `source` in the order given by `query`.
       *
       * Characters are matched using strict `===` equality.
       */
      function findIndices(source, query, start) {
          if (start === void 0) { start = 0; }
          var indices = new Array(query.length);
          for (var i = 0, j = start, n = query.length; i < n; ++i, ++j) {
              j = source.indexOf(query[i], j);
              if (j === -1) {
                  return null;
              }
              indices[i] = j;
          }
          return indices;
      }
      StringExt.findIndices = findIndices;
      /**
       * A string matcher which uses a sum-of-squares algorithm.
       *
       * @param source - The source text which should be searched.
       *
       * @param query - The characters to locate in the source text.
       *
       * @param start - The index to start the search.
       *
       * @returns The match result, or `null` if there is no match.
       *   A lower `score` represents a stronger match.
       *
       * #### Complexity
       * Linear on `sourceText`.
       *
       * #### Notes
       * This scoring algorithm uses a sum-of-squares approach to determine
       * the score. In order for there to be a match, all of the characters
       * in `query` **must** appear in `source` in order. The index of each
       * matching character is squared and added to the score. This means
       * that early and consecutive character matches are preferred, while
       * late matches are heavily penalized.
       */
      function matchSumOfSquares(source, query, start) {
          if (start === void 0) { start = 0; }
          var indices = findIndices(source, query, start);
          if (!indices) {
              return null;
          }
          var score = 0;
          for (var i = 0, n = indices.length; i < n; ++i) {
              var j = indices[i] - start;
              score += j * j;
          }
          return { score: score, indices: indices };
      }
      StringExt.matchSumOfSquares = matchSumOfSquares;
      /**
       * A string matcher which uses a sum-of-deltas algorithm.
       *
       * @param source - The source text which should be searched.
       *
       * @param query - The characters to locate in the source text.
       *
       * @param start - The index to start the search.
       *
       * @returns The match result, or `null` if there is no match.
       *   A lower `score` represents a stronger match.
       *
       * #### Complexity
       * Linear on `sourceText`.
       *
       * #### Notes
       * This scoring algorithm uses a sum-of-deltas approach to determine
       * the score. In order for there to be a match, all of the characters
       * in `query` **must** appear in `source` in order. The delta between
       * the indices are summed to create the score. This means that groups
       * of matched characters are preferred, while fragmented matches are
       * penalized.
       */
      function matchSumOfDeltas(source, query, start) {
          if (start === void 0) { start = 0; }
          var indices = findIndices(source, query, start);
          if (!indices) {
              return null;
          }
          var score = 0;
          var last = start - 1;
          for (var i = 0, n = indices.length; i < n; ++i) {
              var j = indices[i];
              score += j - last - 1;
              last = j;
          }
          return { score: score, indices: indices };
      }
      StringExt.matchSumOfDeltas = matchSumOfDeltas;
      /**
       * Highlight the matched characters of a source text.
       *
       * @param source - The text which should be highlighted.
       *
       * @param indices - The indices of the matched characters. They must
       *   appear in increasing order and must be in bounds of the source.
       *
       * @param fn - The function to apply to the matched chunks.
       *
       * @returns An array of unmatched and highlighted chunks.
       */
      function highlight(source, indices, fn) {
          // Set up the result array.
          var result = [];
          // Set up the counter variables.
          var k = 0;
          var last = 0;
          var n = indices.length;
          // Iterator over each index.
          while (k < n) {
              // Set up the chunk indices.
              var i = indices[k];
              var j = indices[k];
              // Advance the right chunk index until it's non-contiguous.
              while (++k < n && indices[k] === j + 1) {
                  j++;
              }
              // Extract the unmatched text.
              if (last < i) {
                  result.push(source.slice(last, i));
              }
              // Extract and highlight the matched text.
              if (i < j + 1) {
                  result.push(fn(source.slice(i, j + 1)));
              }
              // Update the last visited index.
              last = j + 1;
          }
          // Extract any remaining unmatched text.
          if (last < source.length) {
              result.push(source.slice(last));
          }
          // Return the highlighted result.
          return result;
      }
      StringExt.highlight = highlight;
      /**
       * A 3-way string comparison function.
       *
       * @param a - The first string of interest.
       *
       * @param b - The second string of interest.
       *
       * @returns `-1` if `a < b`, else `1` if `a > b`, else `0`.
       */
      function cmp(a, b) {
          return a < b ? -1 : a > b ? 1 : 0;
      }
      StringExt.cmp = cmp;
  })(StringExt || (StringExt = {}));

  // Copyright (c) Jupyter Development Team.
  // Distributed under the terms of the Modified BSD License.
  /*-----------------------------------------------------------------------------
  | Copyright (c) 2014-2019, PhosphorJS Contributors
  |
  | Distributed under the terms of the BSD 3-Clause License.
  |
  | The full license is in the file LICENSE, distributed with this software.
  |----------------------------------------------------------------------------*/
  /**
   * The namespace for clipboard related functionality.
   */
  var ClipboardExt;
  (function (ClipboardExt) {
      /**
       * Copy text to the system clipboard.
       *
       * @param text - The text to copy to the clipboard.
       */
      function copyText(text) {
          // Fetch the document body.
          var body = document.body;
          // Set up the clipboard event listener.
          var handler = function (event) {
              // Stop the event propagation.
              event.preventDefault();
              event.stopPropagation();
              // Set the clipboard data.
              event.clipboardData.setData('text', text);
              // Remove the event listener.
              body.removeEventListener('copy', handler, true);
          };
          // Add the event listener.
          body.addEventListener('copy', handler, true);
          // Trigger the event.
          document.execCommand('copy');
      }
      ClipboardExt.copyText = copyText;
  })(ClipboardExt || (ClipboardExt = {}));

  // Copyright (c) Jupyter Development Team.
  // Distributed under the terms of the Modified BSD License.
  /*-----------------------------------------------------------------------------
  | Copyright (c) 2014-2017, PhosphorJS Contributors
  |
  | Distributed under the terms of the BSD 3-Clause License.
  |
  | The full license is in the file LICENSE, distributed with this software.
  |----------------------------------------------------------------------------*/
  /**
   * The namespace for element related utilities.
   */
  var ElementExt;
  (function (ElementExt) {
      /**
       * Compute the box sizing for an element.
       *
       * @param element - The element of interest.
       *
       * @returns The box sizing data for the specified element.
       */
      function boxSizing(element) {
          var style = window.getComputedStyle(element);
          var bt = parseFloat(style.borderTopWidth) || 0;
          var bl = parseFloat(style.borderLeftWidth) || 0;
          var br = parseFloat(style.borderRightWidth) || 0;
          var bb = parseFloat(style.borderBottomWidth) || 0;
          var pt = parseFloat(style.paddingTop) || 0;
          var pl = parseFloat(style.paddingLeft) || 0;
          var pr = parseFloat(style.paddingRight) || 0;
          var pb = parseFloat(style.paddingBottom) || 0;
          var hs = bl + pl + pr + br;
          var vs = bt + pt + pb + bb;
          return {
              borderTop: bt,
              borderLeft: bl,
              borderRight: br,
              borderBottom: bb,
              paddingTop: pt,
              paddingLeft: pl,
              paddingRight: pr,
              paddingBottom: pb,
              horizontalSum: hs,
              verticalSum: vs
          };
      }
      ElementExt.boxSizing = boxSizing;
      /**
       * Compute the size limits for an element.
       *
       * @param element - The element of interest.
       *
       * @returns The size limit data for the specified element.
       */
      function sizeLimits(element) {
          var style = window.getComputedStyle(element);
          var minWidth = parseFloat(style.minWidth) || 0;
          var minHeight = parseFloat(style.minHeight) || 0;
          var maxWidth = parseFloat(style.maxWidth) || Infinity;
          var maxHeight = parseFloat(style.maxHeight) || Infinity;
          maxWidth = Math.max(minWidth, maxWidth);
          maxHeight = Math.max(minHeight, maxHeight);
          return { minWidth: minWidth, minHeight: minHeight, maxWidth: maxWidth, maxHeight: maxHeight };
      }
      ElementExt.sizeLimits = sizeLimits;
      /**
       * Test whether a client position lies within an element.
       *
       * @param element - The DOM element of interest.
       *
       * @param clientX - The client X coordinate of interest.
       *
       * @param clientY - The client Y coordinate of interest.
       *
       * @returns Whether the point is within the given element.
       */
      function hitTest(element, clientX, clientY) {
          var rect = element.getBoundingClientRect();
          return (clientX >= rect.left &&
              clientX < rect.right &&
              clientY >= rect.top &&
              clientY < rect.bottom);
      }
      ElementExt.hitTest = hitTest;
      /**
       * Vertically scroll an element into view if needed.
       *
       * @param area - The scroll area element.
       *
       * @param element - The element of interest.
       *
       * #### Notes
       * This follows the "nearest" behavior of the native `scrollIntoView`
       * method, which is not supported by all browsers.
       * https://drafts.csswg.org/cssom-view/#element-scrolling-members
       *
       * If the element fully covers the visible area or is fully contained
       * within the visible area, no scrolling will take place. Otherwise,
       * the nearest edges of the area and element are aligned.
       */
      function scrollIntoViewIfNeeded(area, element) {
          var ar = area.getBoundingClientRect();
          var er = element.getBoundingClientRect();
          if (er.top <= ar.top && er.bottom >= ar.bottom) {
              return;
          }
          if (er.top < ar.top && er.height <= ar.height) {
              area.scrollTop -= ar.top - er.top;
              return;
          }
          if (er.bottom > ar.bottom && er.height >= ar.height) {
              area.scrollTop -= ar.top - er.top;
              return;
          }
          if (er.top < ar.top && er.height > ar.height) {
              area.scrollTop -= ar.bottom - er.bottom;
              return;
          }
          if (er.bottom > ar.bottom && er.height < ar.height) {
              area.scrollTop -= ar.bottom - er.bottom;
              return;
          }
      }
      ElementExt.scrollIntoViewIfNeeded = scrollIntoViewIfNeeded;
  })(ElementExt || (ElementExt = {}));

  // Copyright (c) Jupyter Development Team.
  // Distributed under the terms of the Modified BSD License.
  /*-----------------------------------------------------------------------------
  | Copyright (c) 2014-2017, PhosphorJS Contributors
  |
  | Distributed under the terms of the BSD 3-Clause License.
  |
  | The full license is in the file LICENSE, distributed with this software.
  |----------------------------------------------------------------------------*/
  /**
   * The namespace for platform related utilities.
   */
  var Platform;
  (function (Platform) {
      /**
       * A flag indicating whether the platform is Mac.
       */
      Platform.IS_MAC = !!navigator.platform.match(/Mac/i);
      /**
       * A flag indicating whether the platform is Windows.
       */
      Platform.IS_WIN = !!navigator.platform.match(/Win/i);
      /**
       * A flag indicating whether the browser is IE.
       */
      Platform.IS_IE = /Trident/.test(navigator.userAgent);
      /**
       * A flag indicating whether the browser is Edge.
       */
      Platform.IS_EDGE = /Edge/.test(navigator.userAgent);
      /**
       * Test whether the `accel` key is pressed.
       *
       * @param event - The keyboard or mouse event of interest.
       *
       * @returns Whether the `accel` key is pressed.
       *
       * #### Notes
       * On Mac the `accel` key is the command key. On all other
       * platforms the `accel` key is the control key.
       */
      function accelKey(event) {
          return Platform.IS_MAC ? event.metaKey : event.ctrlKey;
      }
      Platform.accelKey = accelKey;
  })(Platform || (Platform = {}));

  // Copyright (c) Jupyter Development Team.
  // Distributed under the terms of the Modified BSD License.
  /*-----------------------------------------------------------------------------
  | Copyright (c) 2014-2017, PhosphorJS Contributors
  |
  | Distributed under the terms of the BSD 3-Clause License.
  |
  | The full license is in the file LICENSE, distributed with this software.
  |----------------------------------------------------------------------------*/
  /**
   * The namespace for selector related utilities.
   */
  var Selector;
  (function (Selector) {
      /**
       * Calculate the specificity of a single CSS selector.
       *
       * @param selector - The CSS selector of interest.
       *
       * @returns The specificity of the selector.
       *
       * #### Undefined Behavior
       * The selector is invalid.
       *
       * #### Notes
       * This is based on https://www.w3.org/TR/css3-selectors/#specificity
       *
       * A larger number represents a more specific selector.
       *
       * The smallest possible specificity is `0`.
       *
       * The result is represented as a hex number `0x<aa><bb><cc>` where
       * each component is the count of the respective selector clause.
       *
       * If the selector contains commas, only the first clause is used.
       *
       * The computed result is cached, so subsequent calculations for the
       * same selector are extremely fast.
       */
      function calculateSpecificity(selector) {
          if (selector in Private$1.specificityCache) {
              return Private$1.specificityCache[selector];
          }
          var result = Private$1.calculateSingle(selector);
          return Private$1.specificityCache[selector] = result;
      }
      Selector.calculateSpecificity = calculateSpecificity;
      /**
       * Test whether a selector is a valid CSS selector.
       *
       * @param selector - The CSS selector of interest.
       *
       * @returns `true` if the selector is valid, `false` otherwise.
       *
       * #### Notes
       * The computed result is cached, so subsequent tests for the same
       * selector are extremely fast.
       */
      function isValid(selector) {
          if (selector in Private$1.validityCache) {
              return Private$1.validityCache[selector];
          }
          var result = true;
          try {
              Private$1.testElem.querySelector(selector);
          }
          catch (err) {
              result = false;
          }
          return Private$1.validityCache[selector] = result;
      }
      Selector.isValid = isValid;
      /**
       * Test whether an element matches a CSS selector.
       *
       * @param element - The element of interest.
       *
       * @param selector - The valid CSS selector of interest.
       *
       * @returns `true` if the element is a match, `false` otherwise.
       *
       * #### Notes
       * This function uses the builtin browser capabilities when possible,
       * falling back onto a document query otherwise.
       */
      function matches(element, selector) {
          return Private$1.protoMatchFunc.call(element, selector);
      }
      Selector.matches = matches;
  })(Selector || (Selector = {}));
  /**
   * The namespace for the module implementation details.
   */
  var Private$1;
  (function (Private) {
      /**
       * A cache of computed selector specificity values.
       */
      Private.specificityCache = Object.create(null);
      /**
       * A cache of computed selector validity.
       */
      Private.validityCache = Object.create(null);
      /**
       * An empty element for testing selector validity.
       */
      Private.testElem = document.createElement('div');
      /**
       * A cross-browser CSS selector matching prototype function.
       */
      Private.protoMatchFunc = (function () {
          var proto = Element.prototype;
          return (proto.matches ||
              proto.matchesSelector ||
              proto.mozMatchesSelector ||
              proto.msMatchesSelector ||
              proto.oMatchesSelector ||
              proto.webkitMatchesSelector ||
              (function (selector) {
                  var elem = this;
                  var matches = elem.ownerDocument ? elem.ownerDocument.querySelectorAll(selector) : [];
                  return Array.prototype.indexOf.call(matches, elem) !== -1;
              }));
      })();
      /**
       * Calculate the specificity of a single selector.
       *
       * The behavior is undefined if the selector is invalid.
       */
      function calculateSingle(selector) {
          // Ignore anything after the first comma.
          selector = selector.split(',', 1)[0];
          // Setup the aggregate counters.
          var a = 0;
          var b = 0;
          var c = 0;
          // Apply a regex to the front of the selector. If it succeeds, that
          // portion of the selector is removed. Returns a success/fail flag.
          function match(re) {
              var match = selector.match(re);
              if (match === null) {
                  return false;
              }
              selector = selector.slice(match[0].length);
              return true;
          }
          // Replace the negation pseudo-class (which is ignored),
          // but keep its inner content (which is not ignored).
          selector = selector.replace(NEGATION_RE, ' $1 ');
          // Continue matching until the selector is consumed.
          while (selector.length > 0) {
              // Match an ID selector.
              if (match(ID_RE)) {
                  a++;
                  continue;
              }
              // Match a class selector.
              if (match(CLASS_RE)) {
                  b++;
                  continue;
              }
              // Match an attribute selector.
              if (match(ATTR_RE)) {
                  b++;
                  continue;
              }
              // Match a pseudo-element selector. This is done before matching
              // a pseudo-class since this regex overlaps with that regex.
              if (match(PSEUDO_ELEM_RE)) {
                  c++;
                  continue;
              }
              // Match a pseudo-class selector.
              if (match(PSEDUO_CLASS_RE)) {
                  b++;
                  continue;
              }
              // Match a plain type selector.
              if (match(TYPE_RE)) {
                  c++;
                  continue;
              }
              // Finally, match any ignored characters.
              if (match(IGNORE_RE)) {
                  continue;
              }
              // At this point, the selector is assumed to be invalid.
              return 0;
          }
          // Clamp each component to a reasonable base.
          a = Math.min(a, 0xFF);
          b = Math.min(b, 0xFF);
          c = Math.min(c, 0xFF);
          // Combine the components into a single result.
          return (a << 16) | (b << 8) | c;
      }
      Private.calculateSingle = calculateSingle;
      /**
       * A regex which matches an ID selector at string start.
       */
      var ID_RE = /^#[^\s\+>~#\.\[:]+/;
      /**
       * A regex which matches a class selector at string start.
       */
      var CLASS_RE = /^\.[^\s\+>~#\.\[:]+/;
      /**
       * A regex which matches an attribute selector at string start.
       */
      var ATTR_RE = /^\[[^\]]+\]/;
      /**
       * A regex which matches a type selector at string start.
       */
      var TYPE_RE = /^[^\s\+>~#\.\[:]+/;
      /**
       * A regex which matches a pseudo-element selector at string start.
       */
      var PSEUDO_ELEM_RE = /^(::[^\s\+>~#\.\[:]+|:first-line|:first-letter|:before|:after)/;
      /**
       * A regex which matches a pseudo-class selector at string start.
       */
      var PSEDUO_CLASS_RE = /^:[^\s\+>~#\.\[:]+/;
      /**
       * A regex which matches ignored characters at string start.
       */
      var IGNORE_RE = /^[\s\+>~\*]+/;
      /**
       * A regex which matches the negation pseudo-class globally.
       */
      var NEGATION_RE = /:not\(([^\)]+)\)/g;
  })(Private$1 || (Private$1 = {}));

  // Copyright (c) Jupyter Development Team.
  /**
   * A generic B+ tree.
   *
   * #### Notes
   * Most operations have `O(log32 n)` or better complexity.
   */
  var BPlusTree = /** @class */ (function () {
      /**
       * Construct a new B+ tree.
       *
       * @param cmp - The item comparison function for the tree.
       */
      function BPlusTree(cmp) {
          this._root = new Private$2.LeafNode();
          this.cmp = cmp;
      }
      Object.defineProperty(BPlusTree.prototype, "isEmpty", {
          /**
           * Whether the tree is empty.
           *
           * #### Complexity
           * `O(1)`
           */
          get: function () {
              return this._root.size === 0;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(BPlusTree.prototype, "size", {
          /**
           * The size of the tree.
           *
           * #### Complexity
           * `O(1)`
           */
          get: function () {
              return this._root.size;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(BPlusTree.prototype, "first", {
          /**
           * The first item in the tree.
           *
           * This is `undefined` if the tree is empty.
           *
           * #### Complexity
           * `O(log32 n)`
           */
          get: function () {
              var node = Private$2.firstLeaf(this._root);
              return node.size > 0 ? node.items[0] : undefined;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(BPlusTree.prototype, "last", {
          /**
           * The last item in the tree.
           *
           * This is `undefined` if the tree is empty.
           *
           * #### Complexity
           * `O(log32 n)`
           */
          get: function () {
              var node = Private$2.lastLeaf(this._root);
              return node.size > 0 ? node.items[node.size - 1] : undefined;
          },
          enumerable: true,
          configurable: true
      });
      /**
       * Create an iterator over the items in the tree.
       *
       * @returns A new iterator starting with the first item.
       *
       * #### Complexity
       * `O(log32 n)`
       */
      BPlusTree.prototype.iter = function () {
          return Private$2.iterItems(this._root);
      };
      /**
       * Create a reverse iterator over the items in the tree.
       *
       * @returns A new iterator starting with the last item.
       *
       * #### Complexity
       * `O(log32 n)`
       */
      BPlusTree.prototype.retro = function () {
          return Private$2.retroItems(this._root);
      };
      /**
       * Create an iterator for a slice of items in the tree.
       *
       * @param start - The index of the first item, inclusive. This
       *   should be `< stop`. Negative values are taken as an offset
       *   from the end of the tree. The default is `0`.
       *
       * @param stop - The index of the last item, exclusive. This
       *   should be `> start`. Negative values are taken as an offset
       *   from the end of the tree. The default is `size`.
       *
       * @returns A new iterator starting with the specified item.
       *
       * #### Complexity
       * `O(log32 n)`
       */
      BPlusTree.prototype.slice = function (start, stop) {
          return Private$2.sliceItems(this._root, start, stop);
      };
      /**
       * Create a reverse iterator for a slice of items in the tree.
       *
       * @param start - The index of the first item, inclusive. This
       *   should be `> stop`. Negative values are taken as an offset
       *   from the end of the tree. The default is `size - 1`.
       *
       * @param stop - The index of the last item, exclusive. This
       *   should be `< start`. Negative values are taken as an offset
       *   from the end of the tree. The default is `-size - 1`.
       *
       * @returns A new reverse iterator starting with the specified item.
       *
       * #### Complexity
       * `O(log32 n)`
       */
      BPlusTree.prototype.retroSlice = function (start, stop) {
          return Private$2.retroSliceItems(this._root, start, stop);
      };
      /**
       * Get the item at a particular index.
       *
       * @param index - The index of the item of interest. Negative
       *   values are taken as an offset from the end of the tree.
       *
       * @returns The item at the specified index, or `undefined` if
       *   the index is out of range.
       *
       * #### Complexity
       * `O(log32 n)`
       */
      BPlusTree.prototype.at = function (index) {
          return Private$2.itemAt(this._root, index);
      };
      /**
       * Test whether the tree has an item which matches a key.
       *
       * @param key - The key of interest.
       *
       * @param cmp - A function which compares an item against the key.
       *
       * @returns `true` if the tree has an item which matches the given
       *   key, `false` otherwise.
       *
       * #### Complexity
       * `O(log32 n)`
       */
      BPlusTree.prototype.has = function (key, cmp) {
          return Private$2.hasItem(this._root, key, cmp);
      };
      /**
       * Get the index of an item which matches a key.
       *
       * @param key - The key of interest.
       *
       * @param cmp - A function which compares an item against the key.
       *
       * @returns The index of the item which matches the given key. A
       *   negative value means that a matching item does not exist in
       *   the tree, but if one did it would reside at `-index - 1`.
       *
       * #### Complexity
       * `O(log32 n)`
       */
      BPlusTree.prototype.indexOf = function (key, cmp) {
          return Private$2.indexOf(this._root, key, cmp);
      };
      /**
       * Get the item which matches a key.
       *
       * @param item - The key of interest.
       *
       * @param cmp - A function which compares an item against the key.
       *
       * @returns The item which matches the given key, or `undefined` if
       *   the tree does not have a matching item.
       *
       * #### Complexity
       * `O(log32 n)`
       */
      BPlusTree.prototype.get = function (key, cmp) {
          return Private$2.getItem(this._root, key, cmp);
      };
      /**
       * Assign new items to the tree, replacing all current items.
       *
       * @param items - The items to assign to the tree.
       *
       * #### Complexity
       * `O(n log32 n)`
       */
      BPlusTree.prototype.assign = function (items) {
          this.clear();
          this.update(items);
      };
      /**
       * Insert an item into the tree.
       *
       * @param item - The item of interest.
       *
       * @returns If the given item matches an existing item in the tree,
       *   the given item will replace it, and the existing item will be
       *   returned. Otherwise, this method returns `undefined`.
       *
       * #### Complexity
       * `O(log32 n)`
       */
      BPlusTree.prototype.insert = function (item) {
          var existing = Private$2.insertItem(this._root, item, this.cmp);
          this._root = Private$2.maybeSplitRoot(this._root);
          return existing;
      };
      /**
       * Update the tree with multiple items.
       *
       * @param items - The items to insert into the tree.
       *
       * #### Complexity
       * `O(k log32 n)`
       */
      BPlusTree.prototype.update = function (items) {
          var _this = this;
          each$1(items, function (item) { _this.insert(item); });
      };
      /**
       * Delete an item which matches a particular key.
       *
       * @param key - The key of interest.
       *
       * @param cmp - A function which compares an item against the key.
       *
       * @returns The item removed from the tree, or `undefined` if no
       *   item matched the given key.
       *
       * #### Complexity
       * `O(log32 n)`
       */
      BPlusTree.prototype.delete = function (key, cmp) {
          var item = Private$2.deleteItem(this._root, key, cmp);
          this._root = Private$2.maybeExtractRoot(this._root);
          return item;
      };
      /**
       * Remove an item at a particular index.
       *
       * @param index - The index of the item to remove. Negative
       *   values are taken as an offset from the end of the tree.
       *
       * @returns The item removed from the tree, or `undefined` if
       *   the given index is out of range.
       *
       * #### Complexity
       * `O(log32 n)`
       */
      BPlusTree.prototype.remove = function (index) {
          var item = Private$2.removeItem(this._root, index);
          this._root = Private$2.maybeExtractRoot(this._root);
          return item;
      };
      /**
       * Clear the contents of the tree.
       *
       * #### Complexity
       * `O(n)`
       */
      BPlusTree.prototype.clear = function () {
          Private$2.clear(this._root);
          this._root = new Private$2.LeafNode();
      };
      return BPlusTree;
  }());
  /**
   * The namespace for the `BPlusTree` class statics.
   */
  (function (BPlusTree) {
      /**
       * Create a new B+ tree populated with the given items.
       *
       * @param items - The items to add to the tree.
       *
       * @param cmp - The item comparison function for the tree.
       *
       * @returns A new B+ tree populated with the given items.
       *
       * #### Complexity
       * `O(n log32 n)`
       */
      function from(items, cmp) {
          var tree = new BPlusTree(cmp);
          tree.assign(items);
          return tree;
      }
      BPlusTree.from = from;
  })(BPlusTree || (BPlusTree = {}));
  /**
   * The namespace for the module implementation details.
   */
  var Private$2;
  (function (Private) {
      /**
       * A branch node in a B+ tree.
       */
      var BranchNode = /** @class */ (function () {
          function BranchNode() {
              /**
               * The left-most item of each child subtree.
               */
              this.items = [];
              /**
               * The cumulative sizes of each child subtree.
               */
              this.sizes = [];
              /**
               * The child nodes of this branch node.
               */
              this.children = [];
          }
          Object.defineProperty(BranchNode.prototype, "type", {
              /**
               * The discriminated type of the node.
               */
              get: function () {
                  return 0 /* Branch */;
              },
              enumerable: true,
              configurable: true
          });
          Object.defineProperty(BranchNode.prototype, "size", {
              /**
               * The total number of items in the subtree.
               */
              get: function () {
                  return this.sizes[this.sizes.length - 1];
              },
              enumerable: true,
              configurable: true
          });
          Object.defineProperty(BranchNode.prototype, "width", {
              /**
               * The tree width of the node.
               */
              get: function () {
                  return this.children.length;
              },
              enumerable: true,
              configurable: true
          });
          return BranchNode;
      }());
      Private.BranchNode = BranchNode;
      /**
       * A leaf node in a B+ tree.
       */
      var LeafNode = /** @class */ (function () {
          function LeafNode() {
              /**
               * The next sibling leaf node of this leaf node.
               */
              this.next = null;
              /**
               * The previous sibling leaf node of this leaf node.
               */
              this.prev = null;
              /**
               * The items of the leaf.
               */
              this.items = [];
          }
          Object.defineProperty(LeafNode.prototype, "type", {
              /**
               * The discriminated type of the node.
               */
              get: function () {
                  return 1 /* Leaf */;
              },
              enumerable: true,
              configurable: true
          });
          Object.defineProperty(LeafNode.prototype, "size", {
              /**
               * The total number of items in the leaf.
               */
              get: function () {
                  return this.items.length;
              },
              enumerable: true,
              configurable: true
          });
          Object.defineProperty(LeafNode.prototype, "width", {
              /**
               * The tree width of the node.
               */
              get: function () {
                  return this.items.length;
              },
              enumerable: true,
              configurable: true
          });
          return LeafNode;
      }());
      Private.LeafNode = LeafNode;
      /**
       * Get the first leaf node in the tree.
       *
       * @param node - The root node of interest.
       *
       * @returns The first leaf node in the tree.
       *
       * #### Complexity
       * `O(log32 n)`
       */
      function firstLeaf(node) {
          while (node.type === 0 /* Branch */) {
              node = node.children[0];
          }
          return node;
      }
      Private.firstLeaf = firstLeaf;
      /**
       * Get the last leaf node in the tree.
       *
       * @param node - The root node of interest.
       *
       * @returns The last leaf node in the tree.
       *
       * #### Complexity
       * `O(log32 n)`
       */
      function lastLeaf(node) {
          while (node.type === 0 /* Branch */) {
              node = node.children[node.children.length - 1];
          }
          return node;
      }
      Private.lastLeaf = lastLeaf;
      /**
       * Create a forward iterator for the items in the tree.
       *
       * @param node - The root node of interest.
       *
       * @returns A new forward iterator starting with the first item.
       *
       * #### Complexity
       * `O(log32 n)`
       */
      function iterItems(node) {
          var leaf = firstLeaf(node);
          return new ForwardIterator(leaf, 0, -1);
      }
      Private.iterItems = iterItems;
      /**
       * Create a reverse iterator for the items in the tree.
       *
       * @param node - The root node of interest.
       *
       * @returns A new reverse iterator starting with the last item.
       *
       * #### Complexity
       * `O(log32 n)`
       */
      function retroItems(node) {
          var leaf = lastLeaf(node);
          return new RetroIterator(leaf, leaf.size - 1, -1);
      }
      Private.retroItems = retroItems;
      /**
       * Create an iterator for a slice of items in the tree.
       *
       * @param node - The root node of interest.
       *
       * @param start - The index of the first item, inclusive. This
       *   should be `< stop`. Negative values are taken as an offset
       *   from the end of the tree. The default is `0`.
       *
       * @param stop - The index of the last item, exclusive. This
       *   should be `> start`. Negative values are taken as an offset
       *   from the end of the tree. The default is `size`.
       *
       * @returns A new iterator starting with the specified item.
       *
       * #### Complexity
       * `O(log32 n)`
       */
      function sliceItems(node, start, stop) {
          // Normalize the start index.
          if (start === undefined) {
              start = 0;
          }
          else if (start < 0) {
              start = Math.max(0, start + node.size);
          }
          else {
              start = Math.min(start, node.size);
          }
          // Normalize the stop index.
          if (stop === undefined) {
              stop = node.size;
          }
          else if (stop < 0) {
              stop = Math.max(0, stop + node.size);
          }
          else {
              stop = Math.min(stop, node.size);
          }
          // Compute effective count.
          var count = Math.max(0, stop - start);
          // Bail early if there is nothing to iterate.
          if (count === 0) {
              return empty();
          }
          // Find the starting leaf node and local index.
          while (node.type === 0 /* Branch */) {
              var i = findPivotIndexByIndex(node.sizes, start);
              if (i > 0)
                  start -= node.sizes[i - 1];
              node = node.children[i];
          }
          // Return the forward iterator for the range.
          return new ForwardIterator(node, start, count);
      }
      Private.sliceItems = sliceItems;
      /**
       * Create a reverse iterator for a slice of items in the tree.
       *
       * @param node - The root node of interest.
       *
       * @param start - The index of the first item, inclusive. This
       *   should be `> stop`. Negative values are taken as an offset
       *   from the end of the tree. The default is `size - 1`.
       *
       * @param stop - The index of the last item, exclusive. This
       *   should be `< start`. Negative values are taken as an offset
       *   from the end of the tree. The default is `-size - 1`.
       *
       * @returns A new reverse iterator starting with the specified item.
       *
       * #### Complexity
       * `O(log32 n)`
       */
      function retroSliceItems(node, start, stop) {
          // Normalize the start index.
          if (start === undefined) {
              start = node.size - 1;
          }
          else if (start < 0) {
              start = Math.max(-1, start + node.size);
          }
          else {
              start = Math.min(start, node.size - 1);
          }
          // Normalize the stop index.
          if (stop === undefined) {
              stop = -1;
          }
          else if (stop < 0) {
              stop = Math.max(-1, stop + node.size);
          }
          else {
              stop = Math.min(stop, node.size - 1);
          }
          // Compute the effective count.
          var count = Math.max(0, start - stop);
          // Bail early if there is nothing to iterate.
          if (count === 0) {
              return empty();
          }
          // Find the starting leaf node and local index.
          while (node.type === 0 /* Branch */) {
              var i = findPivotIndexByIndex(node.sizes, start);
              if (i > 0)
                  start -= node.sizes[i - 1];
              node = node.children[i];
          }
          // Return the retro iterator for the range.
          return new RetroIterator(node, start, count);
      }
      Private.retroSliceItems = retroSliceItems;
      /**
       * Get the item at the specified index.
       *
       * @param node - The root node of interest.
       *
       * @param index - The index of the item of interest. Negative
       *   values are taken as an offset from the end of the tree.
       *
       * @returns The item at the specified index, or `undefined` if
       *   the index is out of range.
       *
       * #### Complexity
       * `O(log32 n)`
       */
      function itemAt(node, index) {
          // Wrap negative indices.
          if (index < 0) {
              index += node.size;
          }
          // Bail early if the index is out of range.
          if (index < 0 || index >= node.size) {
              return undefined;
          }
          // Find the containing leaf node and local index.
          while (node.type === 0 /* Branch */) {
              var i = findPivotIndexByIndex(node.sizes, index);
              if (i > 0)
                  index -= node.sizes[i - 1];
              node = node.children[i];
          }
          // Return the item at the specified index.
          return node.items[index];
      }
      Private.itemAt = itemAt;
      /**
       * Test whether the tree contains an item which matches a key.
       *
       * @param node - The root node of interest.
       *
       * @param key - The key of interest.
       *
       * @param cmp - The key comparison function.
       *
       * @returns Whether the tree contains a matching item.
       *
       * #### Complexity
       * `O(log32 n)`
       */
      function hasItem(node, key, cmp) {
          // Find the containing leaf node.
          while (node.type === 0 /* Branch */) {
              var i_1 = findPivotIndexByKey(node.items, key, cmp);
              node = node.children[i_1];
          }
          // Find the key index.
          var i = findKeyIndex(node.items, key, cmp);
          // Return whether or not the node contains a matching item.
          return i >= 0;
      }
      Private.hasItem = hasItem;
      /**
       * Get the index of the item which matches a key.
       *
       * @param node - The node of interest.
       *
       * @param key - The key of interest.
       *
       * @param cmp - The key comparison function.
       *
       * @returns The index of the item which matches the given key. A
       *   negative value means that a matching item does not exist in
       *   the tree, but if one did it would reside at `-index - 1`.
       *
       * #### Complexity
       * `O(log32 n)`
       */
      function indexOf(node, key, cmp) {
          // Set up the global index.
          var index = 0;
          // Find the containing leaf node and global index.
          while (node.type === 0 /* Branch */) {
              var i_2 = findPivotIndexByKey(node.items, key, cmp);
              if (i_2 > 0)
                  index += node.sizes[i_2 - 1];
              node = node.children[i_2];
          }
          // Find the key index.
          var i = findKeyIndex(node.items, key, cmp);
          // Return the final computed index.
          return i >= 0 ? index + i : -index + i;
      }
      Private.indexOf = indexOf;
      /**
       * Get the item for a particular key.
       *
       * @param node - The node of interest.
       *
       * @param key - The key of interest.
       *
       * @param cmp - The key comparison function.
       *
       * @returns The item for the specified key, or `undefined` if
       *   the tree does not have a matching item for the key.
       *
       * #### Complexity
       * `O(log32 n)`
       */
      function getItem(node, key, cmp) {
          // Find the containing leaf node.
          while (node.type === 0 /* Branch */) {
              var i_3 = findPivotIndexByKey(node.items, key, cmp);
              node = node.children[i_3];
          }
          // Find the key index.
          var i = findKeyIndex(node.items, key, cmp);
          // Return the item for the given key.
          return i >= 0 ? node.items[i] : undefined;
      }
      Private.getItem = getItem;
      /**
       * Insert an item into the tree.
       *
       * @param node - The root node of interest.
       *
       * @param item - The item of interest.
       *
       * @param cmp - The item comparison function.
       *
       * @returns If the given item matches an existing item in the tree,
       *   the given item will replace it, and the existing item will be
       *   returned. Otherwise, this function returns `undefined`.
       *
       * #### Complexity
       * `O(log32 n)`
       *
       * #### Notes
       * The root may be overfull after calling this function.
       */
      function insertItem(node, item, cmp) {
          // Handle leaf nodes first.
          if (node.type === 1 /* Leaf */) {
              // Find the index for the given item.
              var i_4 = findKeyIndex(node.items, item, cmp);
              // Fetch the existing item and insert the new item.
              var existing_1;
              if (i_4 >= 0) {
                  existing_1 = node.items[i_4];
                  node.items[i_4] = item;
              }
              else {
                  existing_1 = undefined;
                  ArrayExt.insert(node.items, -i_4 - 1, item);
              }
              // Return the existing item.
              return existing_1;
          }
          // Find the pivot index for the insert.
          var i = findPivotIndexByKey(node.items, item, cmp);
          // Fetch the pivot child.
          var child = node.children[i];
          // Fetch the current size of the child.
          var prevSize = child.size;
          // Recursively insert the item into the child.
          var existing = insertItem(child, item, cmp);
          // Fetch the updated size of the child.
          var currSize = child.size;
          // Update the item state of the branch.
          node.items[i] = child.items[0];
          // Bail early if the child size did not change.
          if (prevSize === currSize) {
              return existing;
          }
          // Split the child if it's overfull.
          if (child.width > MAX_NODE_WIDTH) {
              var next = splitNode(child);
              ArrayExt.insert(node.children, i + 1, next);
              ArrayExt.insert(node.items, i + 1, next.items[0]);
          }
          // Update the dirty sizes of the branch.
          updateSizes(node, i);
          // Return the existing item.
          return existing;
      }
      Private.insertItem = insertItem;
      /**
       * Delete an item in the tree.
       *
       * @param node - The node of interest.
       *
       * @param key - The key of interest.
       *
       * @param cmp - The key comparison function.
       *
       * @returns The deleted item or `undefined`.
       *
       * #### Complexity
       * `O(log32 n)`
       *
       * #### Notes
       * The root may be underfull after calling this function.
       */
      function deleteItem(node, key, cmp) {
          // Handle leaf nodes first.
          if (node.type === 1 /* Leaf */) {
              // Find the index for the given key.
              var i_5 = findKeyIndex(node.items, key, cmp);
              // Bail early if the item does not exist.
              if (i_5 < 0) {
                  return undefined;
              }
              // Remove the item at the computed index.
              return ArrayExt.removeAt(node.items, i_5);
          }
          // Find the pivot index for the delete.
          var i = findPivotIndexByKey(node.items, key, cmp);
          // Fetch the pivot child.
          var child = node.children[i];
          // Fetch the current size of the child.
          var prevSize = child.size;
          // Recursively remove the item from the child.
          var item = deleteItem(child, key, cmp);
          // Fetch the updated size of the child.
          var currSize = child.size;
          // Bail early if the child size did not change.
          if (prevSize === currSize) {
              return item;
          }
          // Update the item state of the branch.
          node.items[i] = child.items[0];
          // Join the child if it's underfull.
          if (child.width < MIN_NODE_WIDTH) {
              i = joinChild(node, i);
          }
          // Update the dirty sizes of the branch.
          updateSizes(node, i);
          // Return the deleted item.
          return item;
      }
      Private.deleteItem = deleteItem;
      /**
       * Remove an item from the tree.
       *
       * @param node - The node of interest.
       *
       * @param index - The index of interest.
       *
       * @returns The removed item or `undefined`.
       *
       * #### Complexity
       * `O(log32 n)`
       *
       * #### Notes
       * The root may be underfull after calling this function.
       */
      function removeItem(node, index) {
          // Wrap negative indices.
          if (index < 0) {
              index += node.size;
          }
          // Bail early if the index is out of range.
          if (index < 0 || index >= node.size) {
              return undefined;
          }
          // Handle leaf nodes first.
          if (node.type === 1 /* Leaf */) {
              return ArrayExt.removeAt(node.items, index);
          }
          // Find the pivot index for the remove.
          var i = findPivotIndexByIndex(node.sizes, index);
          if (i > 0)
              index -= node.sizes[i];
          // Fetch the pivot child.
          var child = node.children[i];
          // Recursively remove the item from the child.
          var item = removeItem(child, index);
          // Update the item state of the branch.
          node.items[i] = child.items[0];
          // Join the child if it's underfull.
          if (child.width < MIN_NODE_WIDTH) {
              i = joinChild(node, i);
          }
          // Update the dirty sizes of the branch.
          updateSizes(node, i);
          // Return the removed item.
          return item;
      }
      Private.removeItem = removeItem;
      /**
       * Recursively clear the contents of a node.
       *
       * @param node - The node of interest.
       *
       * #### Complexity
       * `O(n)`
       */
      function clear(node) {
          if (node.type === 0 /* Branch */) {
              each$1(node.children, clear);
              node.children.length = 0;
              node.sizes.length = 0;
              node.items.length = 0;
          }
          else {
              node.items.length = 0;
              node.next = null;
              node.prev = null;
          }
      }
      Private.clear = clear;
      /**
       * Split a root node and create a new root, if needed.
       *
       * @param node - The root node of interest.
       *
       * @returns The new root node.
       */
      function maybeSplitRoot(node) {
          // Bail early if the current root is not overfull.
          if (node.width <= MAX_NODE_WIDTH) {
              return node;
          }
          // Create a new root branch node.
          var root = new BranchNode();
          // Split the node to the right and create a new sibling.
          var next = splitNode(node);
          // Add the sizes to the root.
          root.sizes[0] = node.size;
          root.sizes[1] = node.size + next.size;
          // Add the children to the root.
          root.children[0] = node;
          root.children[1] = next;
          // Add the items to the root.
          root.items[0] = node.items[0];
          root.items[1] = next.items[0];
          // Return the new root node.
          return root;
      }
      Private.maybeSplitRoot = maybeSplitRoot;
      /**
       * Extract a single node child as a new root, if needed.
       *
       * @param node - The root node of interest.
       *
       * @returns The new root node.
       */
      function maybeExtractRoot(node) {
          // Bail early if the node it already a leaf.
          if (node.type === 1 /* Leaf */) {
              return node;
          }
          // Bail early if the branch has more than one child.
          if (node.children.length > 1) {
              return node;
          }
          // Extract the sole remaining child as the new root.
          var root = node.children.pop();
          // Clear the rest of the node state.
          clear(node);
          // Return the new root.
          return root;
      }
      Private.maybeExtractRoot = maybeExtractRoot;
      /**
       * The maximum width for a node in the tree.
       */
      var MAX_NODE_WIDTH = 32;
      /**
       * The minimum width for a node in the tree.
       */
      var MIN_NODE_WIDTH = MAX_NODE_WIDTH >> 1;
      /**
       * A forward iterator for a B+ tree.
       */
      var ForwardIterator = /** @class */ (function () {
          /**
           * Construct a new forward iterator.
           *
           * @param node - The first leaf node in the chain.
           *
           * @param index - The local index of the first item.
           *
           * @param count - The number of items to iterate. A value `< 0`
           *   will iterate all available items.
           */
          function ForwardIterator(node, index, count) {
              this._node = node;
              this._index = index;
              this._count = count;
          }
          /**
           * Get an iterator over the object's values.
           *
           * @returns An iterator which yields the object's values.
           */
          ForwardIterator.prototype.iter = function () {
              return this;
          };
          /**
           * Create an independent clone of the iterator.
           *
           * @returns A new independent clone of the iterator.
           */
          ForwardIterator.prototype.clone = function () {
              return new ForwardIterator(this._node, this._index, this._count);
          };
          /**
           * Get the next value from the iterator.
           *
           * @returns The next value from the iterator, or `undefined`.
           */
          ForwardIterator.prototype.next = function () {
              if (this._node === null || this._count === 0) {
                  return undefined;
              }
              if (this._index >= this._node.size) {
                  this._node = this._node.next;
                  this._index = 0;
                  return this.next();
              }
              if (this._count > 0) {
                  this._count--;
              }
              return this._node.items[this._index++];
          };
          return ForwardIterator;
      }());
      /**
       * A reverse iterator for a B+ tree.
       */
      var RetroIterator = /** @class */ (function () {
          /**
           * Construct a new retro iterator.
           *
           * @param node - The last leaf node in the chain.
           *
           * @param index - The local index of the last item.
           *
           * @param count - The number of items to iterate. A value `< 0`
           *   will iterate all available items.
           */
          function RetroIterator(node, index, count) {
              this._node = node;
              this._index = index;
              this._count = count;
          }
          /**
           * Get an iterator over the object's values.
           *
           * @returns An iterator which yields the object's values.
           */
          RetroIterator.prototype.iter = function () {
              return this;
          };
          /**
           * Create an independent clone of the iterator.
           *
           * @returns A new independent clone of the iterator.
           */
          RetroIterator.prototype.clone = function () {
              return new RetroIterator(this._node, this._index, this._count);
          };
          /**
           * Get the next value from the iterator.
           *
           * @returns The next value from the iterator, or `undefined`.
           */
          RetroIterator.prototype.next = function () {
              if (this._node === null || this._count === 0) {
                  return undefined;
              }
              if (this._index >= this._node.size) {
                  this._index = this._node.size - 1;
              }
              if (this._index < 0) {
                  this._node = this._node.prev;
                  this._index = this._node ? this._node.size - 1 : -1;
                  return this.next();
              }
              if (this._count > 0) {
                  this._count--;
              }
              return this._node.items[this._index--];
          };
          return RetroIterator;
      }());
      /**
       * Find the pivot index for a particular local index.
       */
      function findPivotIndexByIndex(sizes, index) {
          var n = sizes.length;
          for (var i = 0; i < n; ++i) {
              if (sizes[i] > index) {
                  return i;
              }
          }
          return n - 1;
      }
      /**
       * Find the pivot index for a particular key.
       */
      function findPivotIndexByKey(items, key, cmp) {
          var n = items.length;
          for (var i = 1; i < n; ++i) {
              if (cmp(items[i], key) > 0) {
                  return i - 1;
              }
          }
          return n - 1;
      }
      /**
       * Find the key index for a particular key.
       */
      function findKeyIndex(items, key, cmp) {
          var n = items.length;
          for (var i = 0; i < n; ++i) {
              var c = cmp(items[i], key);
              if (c === 0) {
                  return i;
              }
              if (c > 0) {
                  return -i - 1;
              }
          }
          return -n - 1;
      }
      /**
       * Update the sizes of a branch node starting at the given index.
       */
      function updateSizes(node, i) {
          var sizes = node.sizes, children = node.children;
          var last = i > 0 ? sizes[i - 1] : 0;
          for (var n = children.length; i < n; ++i) {
              last = sizes[i] = last + children[i].size;
          }
          sizes.length = children.length;
      }
      /**
       * Split a node and return its new next sibling.
       *
       * @param node - The node of interest.
       *
       * @returns The new next sibling node.
       */
      function splitNode(node) {
          // Handle leaf nodes first.
          if (node.type === 1 /* Leaf */) {
              // Create the new sibling leaf node.
              var next_1 = new LeafNode();
              // Move the items to the new sibling.
              var v1_1 = node.items;
              var v2_1 = next_1.items;
              for (var i = MIN_NODE_WIDTH, n = v1_1.length; i < n; ++i) {
                  v2_1.push(v1_1[i]);
              }
              v1_1.length = MIN_NODE_WIDTH;
              // Patch up the sibling links.
              if (node.next)
                  node.next.prev = next_1;
              next_1.next = node.next;
              next_1.prev = node;
              node.next = next_1;
              // Return the new next sibling.
              return next_1;
          }
          // Create the new sibling branch node.
          var next = new BranchNode();
          // Move the children to the new sibling.
          var c1 = node.children;
          var c2 = next.children;
          for (var i = MIN_NODE_WIDTH, n = c1.length; i < n; ++i) {
              c2.push(c1[i]);
          }
          c1.length = MIN_NODE_WIDTH;
          // Move the items to the new sibling.
          var v1 = node.items;
          var v2 = next.items;
          for (var i = MIN_NODE_WIDTH, n = v1.length; i < n; ++i) {
              v2.push(v1[i]);
          }
          v1.length = MIN_NODE_WIDTH;
          // Update the dirty sizes of the nodes.
          updateSizes(node, MIN_NODE_WIDTH);
          updateSizes(next, 0);
          // Return the new next sibling.
          return next;
      }
      /**
       * Join a child node of a branch with one of its siblings.
       *
       * @param node - The branch node of interest.
       *
       * @param i - The index of the child node of interest.
       *
       * @returns The first modified index.
       *
       * #### Notes
       * This may cause the branch to become underfull.
       */
      function joinChild(node, i) {
          var _a, _b, _c, _d, _e, _f;
          // Fetch the child to be joined.
          var child = node.children[i];
          // Fetch the relevant sibling.
          var sibling = i === 0 ? node.children[i + 1] : node.children[i - 1];
          // Compute the flags which control the join behavior.
          var hasNext = i === 0;
          var isLeaf = child.type === 1 /* Leaf */;
          var hasExtra = sibling.width > MIN_NODE_WIDTH;
          // Join case #1: steal from next sibling leaf
          if (isLeaf && hasExtra && hasNext) {
              // Cast the children as leaves.
              var c = child;
              var s = sibling;
              // Steal an item.
              c.items.push(s.items.shift());
              // Update the branch items.
              node.items[i + 1] = s.items[0];
              // Return the first modified index.
              return i;
          }
          // Join case #2: steal from previous sibling leaf
          if (isLeaf && hasExtra && !hasNext) {
              // Cast the children as leaves.
              var c = child;
              var s = sibling;
              // Steal an item.
              c.items.unshift(s.items.pop());
              // Update the branch items.
              node.items[i] = c.items[0];
              // Return the first modified index.
              return i - 1;
          }
          // Join case #3: merge with next sibling leaf
          if (isLeaf && !hasExtra && hasNext) {
              // Cast the children as leaves.
              var c = child;
              var s = sibling;
              // Merge items.
              (_a = s.items).unshift.apply(_a, c.items);
              // Remove the old branch child.
              ArrayExt.removeAt(node.children, i);
              // Remove the stale branch item.
              ArrayExt.removeAt(node.items, i + 1);
              // Patch up the sibling links.
              if (c.prev)
                  c.prev.next = s;
              s.prev = c.prev;
              // Clear the original child.
              clear(c);
              // Return the first modified index.
              return i;
          }
          // Join case #4: merge with previous sibling leaf
          if (isLeaf && !hasExtra && !hasNext) {
              // Cast the children as leaves.
              var c = child;
              var s = sibling;
              // Merge items.
              (_b = s.items).push.apply(_b, c.items);
              // Remove the old branch child.
              ArrayExt.removeAt(node.children, i);
              // Remove the stale branch item.
              ArrayExt.removeAt(node.items, i);
              // Patch up the sibling links.
              if (c.next)
                  c.next.prev = s;
              s.next = c.next;
              // Clear the original child.
              clear(c);
              // Return the first modified index.
              return i - 1;
          }
          // Join case #5: steal from next sibling branch
          if (!isLeaf && hasExtra && hasNext) {
              // Cast the children to branches.
              var c = child;
              var s = sibling;
              // Steal a child from the next sibling.
              c.children.push(s.children.shift());
              // Steal an item from the next sibling.
              c.items.push(s.items.shift());
              // Update the branch items.
              node.items[i + 1] = s.items[0];
              // Update the sibling sizes.
              updateSizes(c, c.width - 1);
              updateSizes(s, 0);
              // Return the first modified index.
              return i;
          }
          // Join case #6: steal from previous sibling branch
          if (!isLeaf && hasExtra && !hasNext) {
              // Cast the children to branches.
              var c = child;
              var s = sibling;
              // Steal a child from the previous sibling.
              c.children.unshift(s.children.pop());
              // Steal an item from the previous sibling.
              c.items.unshift(s.items.pop());
              // Update the branch items.
              node.items[i] = c.items[0];
              // Update the sibling sizes.
              updateSizes(c, 0);
              updateSizes(s, s.width - 1);
              // Return the first modified index.
              return i - 1;
          }
          // Join case #7: merge with next sibling branch
          if (!isLeaf && !hasExtra && hasNext) {
              // Cast the children to branches.
              var c = child;
              var s = sibling;
              // Merge the children with the next sibling.
              (_c = s.children).unshift.apply(_c, c.children);
              // Merge the items with the next sibling.
              (_d = s.items).unshift.apply(_d, c.items);
              // Remove the old branch child.
              ArrayExt.removeAt(node.children, i);
              // Remove the stale branch item.
              ArrayExt.removeAt(node.items, i + 1);
              // Update the sibling sizes.
              updateSizes(s, 0);
              // Clear the original child but, not its children.
              c.children.length = 0;
              clear(c);
              // Return the first modified index.
              return i;
          }
          // Join case #8: merge with previous sibling branch
          if (!isLeaf && !hasExtra && !hasNext) {
              // Cast the children to branches.
              var c = child;
              var s = sibling;
              // Merge the children with the previous sibling.
              (_e = s.children).push.apply(_e, c.children);
              // Merge the items with the previous sibling.
              (_f = s.items).push.apply(_f, c.items);
              // Remove the old branch child.
              ArrayExt.removeAt(node.children, i);
              // Remove the stale branch item.
              ArrayExt.removeAt(node.items, i);
              // Update the sibling sizes.
              updateSizes(s, 0);
              // Clear the original child, but not its children.
              c.children.length = 0;
              clear(c);
              // Return the first modified index.
              return i - 1;
          }
          // One of the above cases must match.
          throw 'unreachable';
      }
  })(Private$2 || (Private$2 = {}));

  // Copyright (c) Jupyter Development Team.
  /**
   * A generic doubly-linked list.
   */
  var LinkedList = /** @class */ (function () {
      /**
       * Construct a new linked list.
       */
      function LinkedList() {
          this._first = null;
          this._last = null;
          this._size = 0;
      }
      Object.defineProperty(LinkedList.prototype, "isEmpty", {
          /**
           * Whether the list is empty.
           *
           * #### Complexity
           * Constant.
           */
          get: function () {
              return this._size === 0;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(LinkedList.prototype, "size", {
          /**
           * The size of the list.
           *
           * #### Complexity
           * `O(1)`
           *
           * #### Notes
           * This is equivalent to `length`.
           */
          get: function () {
              return this._size;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(LinkedList.prototype, "length", {
          /**
           * The length of the list.
           *
           * #### Complexity
           * Constant.
           *
           * #### Notes
           * This is equivalent to `size`.
           *
           * This property is deprecated.
           */
          get: function () {
              return this._size;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(LinkedList.prototype, "first", {
          /**
           * The first value in the list.
           *
           * This is `undefined` if the list is empty.
           *
           * #### Complexity
           * Constant.
           */
          get: function () {
              return this._first ? this._first.value : undefined;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(LinkedList.prototype, "last", {
          /**
           * The last value in the list.
           *
           * This is `undefined` if the list is empty.
           *
           * #### Complexity
           * Constant.
           */
          get: function () {
              return this._last ? this._last.value : undefined;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(LinkedList.prototype, "firstNode", {
          /**
           * The first node in the list.
           *
           * This is `null` if the list is empty.
           *
           * #### Complexity
           * Constant.
           */
          get: function () {
              return this._first;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(LinkedList.prototype, "lastNode", {
          /**
           * The last node in the list.
           *
           * This is `null` if the list is empty.
           *
           * #### Complexity
           * Constant.
           */
          get: function () {
              return this._last;
          },
          enumerable: true,
          configurable: true
      });
      /**
       * Create an iterator over the values in the list.
       *
       * @returns A new iterator starting with the first value.
       *
       * #### Complexity
       * Constant.
       */
      LinkedList.prototype.iter = function () {
          return new LinkedList.ForwardValueIterator(this._first);
      };
      /**
       * Create a reverse iterator over the values in the list.
       *
       * @returns A new iterator starting with the last value.
       *
       * #### Complexity
       * Constant.
       */
      LinkedList.prototype.retro = function () {
          return new LinkedList.RetroValueIterator(this._last);
      };
      /**
       * Create an iterator over the nodes in the list.
       *
       * @returns A new iterator starting with the first node.
       *
       * #### Complexity
       * Constant.
       */
      LinkedList.prototype.nodes = function () {
          return new LinkedList.ForwardNodeIterator(this._first);
      };
      /**
       * Create a reverse iterator over the nodes in the list.
       *
       * @returns A new iterator starting with the last node.
       *
       * #### Complexity
       * Constant.
       */
      LinkedList.prototype.retroNodes = function () {
          return new LinkedList.RetroNodeIterator(this._last);
      };
      /**
       * Assign new values to the list, replacing all current values.
       *
       * @param values - The values to assign to the list.
       *
       * #### Complexity
       * Linear.
       */
      LinkedList.prototype.assign = function (values) {
          var _this = this;
          this.clear();
          each$1(values, function (value) { _this.addLast(value); });
      };
      /**
       * Add a value to the end of the list.
       *
       * @param value - The value to add to the end of the list.
       *
       * #### Complexity
       * Constant.
       *
       * #### Notes
       * This is equivalent to `addLast`.
       */
      LinkedList.prototype.push = function (value) {
          this.addLast(value);
      };
      /**
       * Remove and return the value at the end of the list.
       *
       * @returns The removed value, or `undefined` if the list is empty.
       *
       * #### Complexity
       * Constant.
       *
       * #### Notes
       * This is equivalent to `removeLast`.
       */
      LinkedList.prototype.pop = function () {
          return this.removeLast();
      };
      /**
       * Add a value to the beginning of the list.
       *
       * @param value - The value to add to the beginning of the list.
       *
       * #### Complexity
       * Constant.
       *
       * #### Notes
       * This is equivalent to `addFirst`.
       */
      LinkedList.prototype.shift = function (value) {
          this.addFirst(value);
      };
      /**
       * Remove and return the value at the beginning of the list.
       *
       * @returns The removed value, or `undefined` if the list is empty.
       *
       * #### Complexity
       * Constant.
       *
       * #### Notes
       * This is equivalent to `removeFirst`.
       */
      LinkedList.prototype.unshift = function () {
          return this.removeFirst();
      };
      /**
       * Add a value to the beginning of the list.
       *
       * @param value - The value to add to the beginning of the list.
       *
       * @returns The list node which holds the value.
       *
       * #### Complexity
       * Constant.
       */
      LinkedList.prototype.addFirst = function (value) {
          var node = new Private$1$1.LinkedListNode(this, value);
          if (!this._first) {
              this._first = node;
              this._last = node;
          }
          else {
              node.next = this._first;
              this._first.prev = node;
              this._first = node;
          }
          this._size++;
          return node;
      };
      /**
       * Add a value to the end of the list.
       *
       * @param value - The value to add to the end of the list.
       *
       * @returns The list node which holds the value.
       *
       * #### Complexity
       * Constant.
       */
      LinkedList.prototype.addLast = function (value) {
          var node = new Private$1$1.LinkedListNode(this, value);
          if (!this._last) {
              this._first = node;
              this._last = node;
          }
          else {
              node.prev = this._last;
              this._last.next = node;
              this._last = node;
          }
          this._size++;
          return node;
      };
      /**
       * Insert a value before a specific node in the list.
       *
       * @param value - The value to insert before the reference node.
       *
       * @param ref - The reference node of interest. If this is `null`,
       *   the value will be added to the beginning of the list.
       *
       * @returns The list node which holds the value.
       *
       * #### Notes
       * The reference node must be owned by the list.
       *
       * #### Complexity
       * Constant.
       */
      LinkedList.prototype.insertBefore = function (value, ref) {
          if (!ref || ref === this._first) {
              return this.addFirst(value);
          }
          if (!(ref instanceof Private$1$1.LinkedListNode) || ref.list !== this) {
              throw new Error('Reference node is not owned by the list.');
          }
          var node = new Private$1$1.LinkedListNode(this, value);
          var _ref = ref;
          var prev = _ref.prev;
          node.next = _ref;
          node.prev = prev;
          _ref.prev = node;
          prev.next = node;
          this._size++;
          return node;
      };
      /**
       * Insert a value after a specific node in the list.
       *
       * @param value - The value to insert after the reference node.
       *
       * @param ref - The reference node of interest. If this is `null`,
       *   the value will be added to the end of the list.
       *
       * @returns The list node which holds the value.
       *
       * #### Notes
       * The reference node must be owned by the list.
       *
       * #### Complexity
       * Constant.
       */
      LinkedList.prototype.insertAfter = function (value, ref) {
          if (!ref || ref === this._last) {
              return this.addLast(value);
          }
          if (!(ref instanceof Private$1$1.LinkedListNode) || ref.list !== this) {
              throw new Error('Reference node is not owned by the list.');
          }
          var node = new Private$1$1.LinkedListNode(this, value);
          var _ref = ref;
          var next = _ref.next;
          node.next = next;
          node.prev = _ref;
          _ref.next = node;
          next.prev = node;
          this._size++;
          return node;
      };
      /**
       * Remove and return the value at the beginning of the list.
       *
       * @returns The removed value, or `undefined` if the list is empty.
       *
       * #### Complexity
       * Constant.
       */
      LinkedList.prototype.removeFirst = function () {
          var node = this._first;
          if (!node) {
              return undefined;
          }
          if (node === this._last) {
              this._first = null;
              this._last = null;
          }
          else {
              this._first = node.next;
              this._first.prev = null;
          }
          node.list = null;
          node.next = null;
          node.prev = null;
          this._size--;
          return node.value;
      };
      /**
       * Remove and return the value at the end of the list.
       *
       * @returns The removed value, or `undefined` if the list is empty.
       *
       * #### Complexity
       * Constant.
       */
      LinkedList.prototype.removeLast = function () {
          var node = this._last;
          if (!node) {
              return undefined;
          }
          if (node === this._first) {
              this._first = null;
              this._last = null;
          }
          else {
              this._last = node.prev;
              this._last.next = null;
          }
          node.list = null;
          node.next = null;
          node.prev = null;
          this._size--;
          return node.value;
      };
      /**
       * Remove a specific node from the list.
       *
       * @param node - The node to remove from the list.
       *
       * #### Complexity
       * Constant.
       *
       * #### Notes
       * The node must be owned by the list.
       */
      LinkedList.prototype.removeNode = function (node) {
          if (!(node instanceof Private$1$1.LinkedListNode) || node.list !== this) {
              throw new Error('Node is not owned by the list.');
          }
          var _node = node;
          if (_node === this._first && _node === this._last) {
              this._first = null;
              this._last = null;
          }
          else if (_node === this._first) {
              this._first = _node.next;
              this._first.prev = null;
          }
          else if (_node === this._last) {
              this._last = _node.prev;
              this._last.next = null;
          }
          else {
              _node.next.prev = _node.prev;
              _node.prev.next = _node.next;
          }
          _node.list = null;
          _node.next = null;
          _node.prev = null;
          this._size--;
      };
      /**
       * Remove all values from the list.
       *
       * #### Complexity
       * Linear.
       */
      LinkedList.prototype.clear = function () {
          var node = this._first;
          while (node) {
              var next = node.next;
              node.list = null;
              node.prev = null;
              node.next = null;
              node = next;
          }
          this._first = null;
          this._last = null;
          this._size = 0;
      };
      return LinkedList;
  }());
  /**
   * The namespace for the `LinkedList` class statics.
   */
  (function (LinkedList) {
      /**
       * Create a linked list from an iterable of values.
       *
       * @param values - The iterable or array-like object of interest.
       *
       * @returns A new linked list initialized with the given values.
       *
       * #### Complexity
       * Linear.
       */
      function from(values) {
          var list = new LinkedList();
          list.assign(values);
          return list;
      }
      LinkedList.from = from;
      /**
       * A forward iterator for values in a linked list.
       */
      var ForwardValueIterator = /** @class */ (function () {
          /**
           * Construct a forward value iterator.
           *
           * @param node - The first node in the list.
           */
          function ForwardValueIterator(node) {
              this._node = node;
          }
          /**
           * Get an iterator over the object's values.
           *
           * @returns An iterator which yields the object's values.
           */
          ForwardValueIterator.prototype.iter = function () {
              return this;
          };
          /**
           * Create an independent clone of the iterator.
           *
           * @returns A new independent clone of the iterator.
           */
          ForwardValueIterator.prototype.clone = function () {
              return new ForwardValueIterator(this._node);
          };
          /**
           * Get the next value from the iterator.
           *
           * @returns The next value from the iterator, or `undefined`.
           */
          ForwardValueIterator.prototype.next = function () {
              if (!this._node) {
                  return undefined;
              }
              var node = this._node;
              this._node = node.next;
              return node.value;
          };
          return ForwardValueIterator;
      }());
      LinkedList.ForwardValueIterator = ForwardValueIterator;
      /**
       * A reverse iterator for values in a linked list.
       */
      var RetroValueIterator = /** @class */ (function () {
          /**
           * Construct a retro value iterator.
           *
           * @param node - The last node in the list.
           */
          function RetroValueIterator(node) {
              this._node = node;
          }
          /**
           * Get an iterator over the object's values.
           *
           * @returns An iterator which yields the object's values.
           */
          RetroValueIterator.prototype.iter = function () {
              return this;
          };
          /**
           * Create an independent clone of the iterator.
           *
           * @returns A new independent clone of the iterator.
           */
          RetroValueIterator.prototype.clone = function () {
              return new RetroValueIterator(this._node);
          };
          /**
           * Get the next value from the iterator.
           *
           * @returns The next value from the iterator, or `undefined`.
           */
          RetroValueIterator.prototype.next = function () {
              if (!this._node) {
                  return undefined;
              }
              var node = this._node;
              this._node = node.prev;
              return node.value;
          };
          return RetroValueIterator;
      }());
      LinkedList.RetroValueIterator = RetroValueIterator;
      /**
       * A forward iterator for nodes in a linked list.
       */
      var ForwardNodeIterator = /** @class */ (function () {
          /**
           * Construct a forward node iterator.
           *
           * @param node - The first node in the list.
           */
          function ForwardNodeIterator(node) {
              this._node = node;
          }
          /**
           * Get an iterator over the object's values.
           *
           * @returns An iterator which yields the object's values.
           */
          ForwardNodeIterator.prototype.iter = function () {
              return this;
          };
          /**
           * Create an independent clone of the iterator.
           *
           * @returns A new independent clone of the iterator.
           */
          ForwardNodeIterator.prototype.clone = function () {
              return new ForwardNodeIterator(this._node);
          };
          /**
           * Get the next value from the iterator.
           *
           * @returns The next value from the iterator, or `undefined`.
           */
          ForwardNodeIterator.prototype.next = function () {
              if (!this._node) {
                  return undefined;
              }
              var node = this._node;
              this._node = node.next;
              return node;
          };
          return ForwardNodeIterator;
      }());
      LinkedList.ForwardNodeIterator = ForwardNodeIterator;
      /**
       * A reverse iterator for nodes in a linked list.
       */
      var RetroNodeIterator = /** @class */ (function () {
          /**
           * Construct a retro node iterator.
           *
           * @param node - The last node in the list.
           */
          function RetroNodeIterator(node) {
              this._node = node;
          }
          /**
           * Get an iterator over the object's values.
           *
           * @returns An iterator which yields the object's values.
           */
          RetroNodeIterator.prototype.iter = function () {
              return this;
          };
          /**
           * Create an independent clone of the iterator.
           *
           * @returns A new independent clone of the iterator.
           */
          RetroNodeIterator.prototype.clone = function () {
              return new RetroNodeIterator(this._node);
          };
          /**
           * Get the next value from the iterator.
           *
           * @returns The next value from the iterator, or `undefined`.
           */
          RetroNodeIterator.prototype.next = function () {
              if (!this._node) {
                  return undefined;
              }
              var node = this._node;
              this._node = node.prev;
              return node;
          };
          return RetroNodeIterator;
      }());
      LinkedList.RetroNodeIterator = RetroNodeIterator;
  })(LinkedList || (LinkedList = {}));
  /**
   * The namespace for the module implementation details.
   */
  var Private$1$1;
  (function (Private) {
      /**
       * The internal linked list node implementation.
       */
      var LinkedListNode = /** @class */ (function () {
          /**
           * Construct a new linked list node.
           *
           * @param list - The list which owns the node.
           *
           * @param value - The value for the link.
           */
          function LinkedListNode(list, value) {
              /**
               * The linked list which created and owns the node.
               */
              this.list = null;
              /**
               * The next node in the list.
               */
              this.next = null;
              /**
               * The previous node in the list.
               */
              this.prev = null;
              this.list = list;
              this.value = value;
          }
          return LinkedListNode;
      }());
      Private.LinkedListNode = LinkedListNode;
  })(Private$1$1 || (Private$1$1 = {}));

  /*! *****************************************************************************
  Copyright (c) Microsoft Corporation. All rights reserved.
  Licensed under the Apache License, Version 2.0 (the "License"); you may not use
  this file except in compliance with the License. You may obtain a copy of the
  License at http://www.apache.org/licenses/LICENSE-2.0

  THIS CODE IS PROVIDED ON AN *AS IS* BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
  KIND, EITHER EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION ANY IMPLIED
  WARRANTIES OR CONDITIONS OF TITLE, FITNESS FOR A PARTICULAR PURPOSE,
  MERCHANTABLITY OR NON-INFRINGEMENT.

  See the Apache Version 2.0 License for specific language governing permissions
  and limitations under the License.
  ***************************************************************************** */
  /* global Reflect, Promise */

  var extendStatics = function(d, b) {
      extendStatics = Object.setPrototypeOf ||
          ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
          function (d, b) { for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p]; };
      return extendStatics(d, b);
  };

  function __extends$2(d, b) {
      extendStatics(d, b);
      function __() { this.constructor = d; }
      d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
  }

  /**
   * A message which can be delivered to a message handler.
   *
   * #### Notes
   * This class may be subclassed to create complex message types.
   */
  var Message = /** @class */ (function () {
      /**
       * Construct a new message.
       *
       * @param type - The type of the message.
       */
      function Message(type) {
          this.type = type;
      }
      Object.defineProperty(Message.prototype, "isConflatable", {
          /**
           * Test whether the message is conflatable.
           *
           * #### Notes
           * Message conflation is an advanced topic. Most message types will
           * not make use of this feature.
           *
           * If a conflatable message is posted to a handler while another
           * conflatable message of the same `type` has already been posted
           * to the handler, the `conflate()` method of the existing message
           * will be invoked. If that method returns `true`, the new message
           * will not be enqueued. This allows messages to be compressed, so
           * that only a single instance of the message type is processed per
           * cycle, no matter how many times messages of that type are posted.
           *
           * Custom message types may reimplement this property.
           *
           * The default implementation is always `false`.
           */
          get: function () {
              return false;
          },
          enumerable: true,
          configurable: true
      });
      /**
       * Conflate this message with another message of the same `type`.
       *
       * @param other - A conflatable message of the same `type`.
       *
       * @returns `true` if the message was successfully conflated, or
       *   `false` otherwise.
       *
       * #### Notes
       * Message conflation is an advanced topic. Most message types will
       * not make use of this feature.
       *
       * This method is called automatically by the message loop when the
       * given message is posted to the handler paired with this message.
       * This message will already be enqueued and conflatable, and the
       * given message will have the same `type` and also be conflatable.
       *
       * This method should merge the state of the other message into this
       * message as needed so that when this message is finally delivered
       * to the handler, it receives the most up-to-date information.
       *
       * If this method returns `true`, it signals that the other message
       * was successfully conflated and that message will not be enqueued.
       *
       * If this method returns `false`, the other message will be enqueued
       * for normal delivery.
       *
       * Custom message types may reimplement this method.
       *
       * The default implementation always returns `false`.
       */
      Message.prototype.conflate = function (other) {
          return false;
      };
      return Message;
  }());
  /**
   * A convenience message class which conflates automatically.
   *
   * #### Notes
   * Message conflation is an advanced topic. Most user code will not
   * make use of this class.
   *
   * This message class is useful for creating message instances which
   * should be conflated, but which have no state other than `type`.
   *
   * If conflation of stateful messages is required, a custom `Message`
   * subclass should be created.
   */
  var ConflatableMessage = /** @class */ (function (_super) {
      __extends$2(ConflatableMessage, _super);
      function ConflatableMessage() {
          return _super !== null && _super.apply(this, arguments) || this;
      }
      Object.defineProperty(ConflatableMessage.prototype, "isConflatable", {
          /**
           * Test whether the message is conflatable.
           *
           * #### Notes
           * This property is always `true`.
           */
          get: function () {
              return true;
          },
          enumerable: true,
          configurable: true
      });
      /**
       * Conflate this message with another message of the same `type`.
       *
       * #### Notes
       * This method always returns `true`.
       */
      ConflatableMessage.prototype.conflate = function (other) {
          return true;
      };
      return ConflatableMessage;
  }(Message));
  /**
   * The namespace for the global singleton message loop.
   */
  var MessageLoop;
  (function (MessageLoop) {
      /**
       * Send a message to a message handler to process immediately.
       *
       * @param handler - The handler which should process the message.
       *
       * @param msg - The message to deliver to the handler.
       *
       * #### Notes
       * The message will first be sent through any installed message hooks
       * for the handler. If the message passes all hooks, it will then be
       * delivered to the `processMessage` method of the handler.
       *
       * The message will not be conflated with pending posted messages.
       *
       * Exceptions in hooks and handlers will be caught and logged.
       */
      function sendMessage(handler, msg) {
          // Lookup the message hooks for the handler.
          var hooks = messageHooks.get(handler);
          // Handle the common case of no installed hooks.
          if (!hooks || hooks.length === 0) {
              invokeHandler(handler, msg);
              return;
          }
          // Invoke the message hooks starting with the newest first.
          var passed = every$1(retro(hooks), function (hook) {
              return hook ? invokeHook(hook, handler, msg) : true;
          });
          // Invoke the handler if the message passes all hooks.
          if (passed) {
              invokeHandler(handler, msg);
          }
      }
      MessageLoop.sendMessage = sendMessage;
      /**
       * Post a message to a message handler to process in the future.
       *
       * @param handler - The handler which should process the message.
       *
       * @param msg - The message to post to the handler.
       *
       * #### Notes
       * The message will be conflated with the pending posted messages for
       * the handler, if possible. If the message is not conflated, it will
       * be queued for normal delivery on the next cycle of the event loop.
       *
       * Exceptions in hooks and handlers will be caught and logged.
       */
      function postMessage(handler, msg) {
          // Handle the common case of a non-conflatable message.
          if (!msg.isConflatable) {
              enqueueMessage(handler, msg);
              return;
          }
          // Conflate the message with an existing message if possible.
          var conflated = some$1(messageQueue, function (posted) {
              if (posted.handler !== handler) {
                  return false;
              }
              if (!posted.msg) {
                  return false;
              }
              if (posted.msg.type !== msg.type) {
                  return false;
              }
              if (!posted.msg.isConflatable) {
                  return false;
              }
              return posted.msg.conflate(msg);
          });
          // Enqueue the message if it was not conflated.
          if (!conflated) {
              enqueueMessage(handler, msg);
          }
      }
      MessageLoop.postMessage = postMessage;
      /**
       * Install a message hook for a message handler.
       *
       * @param handler - The message handler of interest.
       *
       * @param hook - The message hook to install.
       *
       * #### Notes
       * A message hook is invoked before a message is delivered to the
       * handler. If the hook returns `false`, no other hooks will be
       * invoked and the message will not be delivered to the handler.
       *
       * The most recently installed message hook is executed first.
       *
       * If the hook is already installed, this is a no-op.
       */
      function installMessageHook(handler, hook) {
          // Lookup the hooks for the handler.
          var hooks = messageHooks.get(handler);
          // Bail early if the hook is already installed.
          if (hooks && hooks.indexOf(hook) !== -1) {
              return;
          }
          // Add the hook to the end, so it will be the first to execute.
          if (!hooks) {
              messageHooks.set(handler, [hook]);
          }
          else {
              hooks.push(hook);
          }
      }
      MessageLoop.installMessageHook = installMessageHook;
      /**
       * Remove an installed message hook for a message handler.
       *
       * @param handler - The message handler of interest.
       *
       * @param hook - The message hook to remove.
       *
       * #### Notes
       * It is safe to call this function while the hook is executing.
       *
       * If the hook is not installed, this is a no-op.
       */
      function removeMessageHook(handler, hook) {
          // Lookup the hooks for the handler.
          var hooks = messageHooks.get(handler);
          // Bail early if the hooks do not exist.
          if (!hooks) {
              return;
          }
          // Lookup the index of the hook and bail if not found.
          var i = hooks.indexOf(hook);
          if (i === -1) {
              return;
          }
          // Clear the hook and schedule a cleanup of the array.
          hooks[i] = null;
          scheduleCleanup(hooks);
      }
      MessageLoop.removeMessageHook = removeMessageHook;
      /**
       * Clear all message data associated with a message handler.
       *
       * @param handler - The message handler of interest.
       *
       * #### Notes
       * This will clear all posted messages and hooks for the handler.
       */
      function clearData(handler) {
          // Lookup the hooks for the handler.
          var hooks = messageHooks.get(handler);
          // Clear all messsage hooks for the handler.
          if (hooks && hooks.length > 0) {
              ArrayExt.fill(hooks, null);
              scheduleCleanup(hooks);
          }
          // Clear all posted messages for the handler.
          each$1(messageQueue, function (posted) {
              if (posted.handler === handler) {
                  posted.handler = null;
                  posted.msg = null;
              }
          });
      }
      MessageLoop.clearData = clearData;
      /**
       * Process the pending posted messages in the queue immediately.
       *
       * #### Notes
       * This function is useful when posted messages must be processed
       * immediately, instead of on the next animation frame.
       *
       * This function should normally not be needed, but it may be
       * required to work around certain browser idiosyncrasies.
       *
       * Recursing into this function is a no-op.
       */
      function flush() {
          // Bail if recursion is detected or if there is no pending task.
          if (flushGuard || loopTaskID === 0) {
              return;
          }
          // Unschedule the pending loop task.
          unschedule(loopTaskID);
          // Run the message loop within the recursion guard.
          flushGuard = true;
          runMessageLoop();
          flushGuard = false;
      }
      MessageLoop.flush = flush;
      /**
       * Get the message loop exception handler.
       *
       * @returns The current exception handler.
       *
       * #### Notes
       * The default exception handler is `console.error`.
       */
      function getExceptionHandler() {
          return exceptionHandler;
      }
      MessageLoop.getExceptionHandler = getExceptionHandler;
      /**
       * Set the message loop exception handler.
       *
       * @param handler - The function to use as the exception handler.
       *
       * @returns The old exception handler.
       *
       * #### Notes
       * The exception handler is invoked when a message handler or a
       * message hook throws an exception.
       */
      function setExceptionHandler(handler) {
          var old = exceptionHandler;
          exceptionHandler = handler;
          return old;
      }
      MessageLoop.setExceptionHandler = setExceptionHandler;
      /**
       * The queue of posted message pairs.
       */
      var messageQueue = new LinkedList();
      /**
       * A mapping of handler to array of installed message hooks.
       */
      var messageHooks = new WeakMap();
      /**
       * A set of message hook arrays which are pending cleanup.
       */
      var dirtySet = new Set();
      /**
       * The message loop exception handler.
       */
      var exceptionHandler = function (err) {
          console.error(err);
      };
      /**
       * The id of the pending loop task animation frame.
       */
      var loopTaskID = 0;
      /**
       * A guard flag to prevent flush recursion.
       */
      var flushGuard = false;
      /**
       * A function to schedule an event loop callback.
       */
      var schedule = (function () {
          var ok = typeof requestAnimationFrame === 'function';
          return ok ? requestAnimationFrame : setImmediate;
      })();
      /**
       * A function to unschedule an event loop callback.
       */
      var unschedule = (function () {
          var ok = typeof cancelAnimationFrame === 'function';
          return ok ? cancelAnimationFrame : clearImmediate;
      })();
      /**
       * Invoke a message hook with the specified handler and message.
       *
       * Returns the result of the hook, or `true` if the hook throws.
       *
       * Exceptions in the hook will be caught and logged.
       */
      function invokeHook(hook, handler, msg) {
          var result = true;
          try {
              if (typeof hook === 'function') {
                  result = hook(handler, msg);
              }
              else {
                  result = hook.messageHook(handler, msg);
              }
          }
          catch (err) {
              exceptionHandler(err);
          }
          return result;
      }
      /**
       * Invoke a message handler with the specified message.
       *
       * Exceptions in the handler will be caught and logged.
       */
      function invokeHandler(handler, msg) {
          try {
              handler.processMessage(msg);
          }
          catch (err) {
              exceptionHandler(err);
          }
      }
      /**
       * Add a message to the end of the message queue.
       *
       * This will automatically schedule a run of the message loop.
       */
      function enqueueMessage(handler, msg) {
          // Add the posted message to the queue.
          messageQueue.addLast({ handler: handler, msg: msg });
          // Bail if a loop task is already pending.
          if (loopTaskID !== 0) {
              return;
          }
          // Schedule a run of the message loop.
          loopTaskID = schedule(runMessageLoop);
      }
      /**
       * Run an iteration of the message loop.
       *
       * This will process all pending messages in the queue. If a message
       * is added to the queue while the message loop is running, it will
       * be processed on the next cycle of the loop.
       */
      function runMessageLoop() {
          // Clear the task ID so the next loop can be scheduled.
          loopTaskID = 0;
          // If the message queue is empty, there is nothing else to do.
          if (messageQueue.isEmpty) {
              return;
          }
          // Add a sentinel value to the end of the queue. The queue will
          // only be processed up to the sentinel. Messages posted during
          // this cycle will execute on the next cycle.
          var sentinel = { handler: null, msg: null };
          messageQueue.addLast(sentinel);
          // Enter the message loop.
          while (true) {
              // Remove the first posted message in the queue.
              var posted = messageQueue.removeFirst();
              // If the value is the sentinel, exit the loop.
              if (posted === sentinel) {
                  return;
              }
              // Dispatch the message if it has not been cleared.
              if (posted.handler && posted.msg) {
                  sendMessage(posted.handler, posted.msg);
              }
          }
      }
      /**
       * Schedule a cleanup of a message hooks array.
       *
       * This will add the array to the dirty set and schedule a deferred
       * cleanup of the array contents. On cleanup, any `null` hook will
       * be removed from the array.
       */
      function scheduleCleanup(hooks) {
          if (dirtySet.size === 0) {
              schedule(cleanupDirtySet);
          }
          dirtySet.add(hooks);
      }
      /**
       * Cleanup the message hook arrays in the dirty set.
       *
       * This function should only be invoked asynchronously, when the
       * stack frame is guaranteed to not be on the path of user code.
       */
      function cleanupDirtySet() {
          dirtySet.forEach(cleanupHooks);
          dirtySet.clear();
      }
      /**
       * Cleanup the dirty hooks in a message hooks array.
       *
       * This will remove any `null` hook from the array.
       *
       * This function should only be invoked asynchronously, when the
       * stack frame is guaranteed to not be on the path of user code.
       */
      function cleanupHooks(hooks) {
          ArrayExt.removeAllWhere(hooks, isNull);
      }
      /**
       * Test whether a value is `null`.
       */
      function isNull(value) {
          return value === null;
      }
  })(MessageLoop || (MessageLoop = {}));

  // Copyright (c) Jupyter Development Team.
  // Distributed under the terms of the Modified BSD License.
  /*-----------------------------------------------------------------------------
  | Copyright (c) 2014-2017, PhosphorJS Contributors
  |
  | Distributed under the terms of the BSD 3-Clause License.
  |
  | The full license is in the file LICENSE, distributed with this software.
  |----------------------------------------------------------------------------*/
  /**
   * A class which attaches a value to an external object.
   *
   * #### Notes
   * Attached properties are used to extend the state of an object with
   * semantic data from an unrelated class. They also encapsulate value
   * creation, coercion, and notification.
   *
   * Because attached property values are stored in a hash table, which
   * in turn is stored in a WeakMap keyed on the owner object, there is
   * non-trivial storage overhead involved in their use. The pattern is
   * therefore best used for the storage of rare data.
   */
  var AttachedProperty = /** @class */ (function () {
      /**
       * Construct a new attached property.
       *
       * @param options - The options for initializing the property.
       */
      function AttachedProperty(options) {
          this._pid = Private$3.nextPID();
          this.name = options.name;
          this._create = options.create;
          this._coerce = options.coerce || null;
          this._compare = options.compare || null;
          this._changed = options.changed || null;
      }
      /**
       * Get the current value of the property for a given owner.
       *
       * @param owner - The property owner of interest.
       *
       * @returns The current value of the property.
       *
       * #### Notes
       * If the value has not yet been set, the default value will be
       * computed and assigned as the current value of the property.
       */
      AttachedProperty.prototype.get = function (owner) {
          var value;
          var map = Private$3.ensureMap(owner);
          if (this._pid in map) {
              value = map[this._pid];
          }
          else {
              value = map[this._pid] = this._createValue(owner);
          }
          return value;
      };
      /**
       * Set the current value of the property for a given owner.
       *
       * @param owner - The property owner of interest.
       *
       * @param value - The value for the property.
       *
       * #### Notes
       * If the value has not yet been set, the default value will be
       * computed and used as the previous value for the comparison.
       */
      AttachedProperty.prototype.set = function (owner, value) {
          var oldValue;
          var map = Private$3.ensureMap(owner);
          if (this._pid in map) {
              oldValue = map[this._pid];
          }
          else {
              oldValue = map[this._pid] = this._createValue(owner);
          }
          var newValue = this._coerceValue(owner, value);
          this._maybeNotify(owner, oldValue, map[this._pid] = newValue);
      };
      /**
       * Explicitly coerce the current property value for a given owner.
       *
       * @param owner - The property owner of interest.
       *
       * #### Notes
       * If the value has not yet been set, the default value will be
       * computed and used as the previous value for the comparison.
       */
      AttachedProperty.prototype.coerce = function (owner) {
          var oldValue;
          var map = Private$3.ensureMap(owner);
          if (this._pid in map) {
              oldValue = map[this._pid];
          }
          else {
              oldValue = map[this._pid] = this._createValue(owner);
          }
          var newValue = this._coerceValue(owner, oldValue);
          this._maybeNotify(owner, oldValue, map[this._pid] = newValue);
      };
      /**
       * Get or create the default value for the given owner.
       */
      AttachedProperty.prototype._createValue = function (owner) {
          var create = this._create;
          return create(owner);
      };
      /**
       * Coerce the value for the given owner.
       */
      AttachedProperty.prototype._coerceValue = function (owner, value) {
          var coerce = this._coerce;
          return coerce ? coerce(owner, value) : value;
      };
      /**
       * Compare the old value and new value for equality.
       */
      AttachedProperty.prototype._compareValue = function (oldValue, newValue) {
          var compare = this._compare;
          return compare ? compare(oldValue, newValue) : oldValue === newValue;
      };
      /**
       * Run the change notification if the given values are different.
       */
      AttachedProperty.prototype._maybeNotify = function (owner, oldValue, newValue) {
          var changed = this._changed;
          if (changed && !this._compareValue(oldValue, newValue)) {
              changed(owner, oldValue, newValue);
          }
      };
      return AttachedProperty;
  }());
  /**
   * The namespace for the `AttachedProperty` class statics.
   */
  (function (AttachedProperty) {
      /**
       * Clear the stored property data for the given owner.
       *
       * @param owner - The property owner of interest.
       *
       * #### Notes
       * This will clear all property values for the owner, but it will
       * **not** run the change notification for any of the properties.
       */
      function clearData(owner) {
          Private$3.ownerData.delete(owner);
      }
      AttachedProperty.clearData = clearData;
  })(AttachedProperty || (AttachedProperty = {}));
  /**
   * The namespace for the module implementation details.
   */
  var Private$3;
  (function (Private) {
      /**
       * A weak mapping of property owner to property map.
       */
      Private.ownerData = new WeakMap();
      /**
       * A function which computes successive unique property ids.
       */
      Private.nextPID = (function () {
          var id = 0;
          return function () {
              var rand = Math.random();
              var stem = ("" + rand).slice(2);
              return "pid-" + stem + "-" + id++;
          };
      })();
      /**
       * Lookup the data map for the property owner.
       *
       * This will create the map if one does not already exist.
       */
      function ensureMap(owner) {
          var map = Private.ownerData.get(owner);
          if (map) {
              return map;
          }
          map = Object.create(null);
          Private.ownerData.set(owner, map);
          return map;
      }
      Private.ensureMap = ensureMap;
  })(Private$3 || (Private$3 = {}));

  // Copyright (c) Jupyter Development Team.
  /**
   * A concrete implementation of `ISignal`.
   *
   * #### Example
   * ```typescript
   * import { ISignal, Signal } from '@lumino/signaling';
   *
   * class SomeClass {
   *
   *   constructor(name: string) {
   *     this.name = name;
   *   }
   *
   *   readonly name: string;
   *
   *   get valueChanged: ISignal<this, number> {
   *     return this._valueChanged;
   *   }
   *
   *   get value(): number {
   *     return this._value;
   *   }
   *
   *   set value(value: number) {
   *     if (value === this._value) {
   *       return;
   *     }
   *     this._value = value;
   *     this._valueChanged.emit(value);
   *   }
   *
   *   private _value = 0;
   *   private _valueChanged = new Signal<this, number>(this);
   * }
   *
   * function logger(sender: SomeClass, value: number): void {
   *   console.log(sender.name, value);
   * }
   *
   * let m1 = new SomeClass('foo');
   * let m2 = new SomeClass('bar');
   *
   * m1.valueChanged.connect(logger);
   * m2.valueChanged.connect(logger);
   *
   * m1.value = 42;  // logs: foo 42
   * m2.value = 17;  // logs: bar 17
   * ```
   */
  var Signal = /** @class */ (function () {
      /**
       * Construct a new signal.
       *
       * @param sender - The sender which owns the signal.
       */
      function Signal(sender) {
          this.sender = sender;
      }
      /**
       * Connect a slot to the signal.
       *
       * @param slot - The slot to invoke when the signal is emitted.
       *
       * @param thisArg - The `this` context for the slot. If provided,
       *   this must be a non-primitive object.
       *
       * @returns `true` if the connection succeeds, `false` otherwise.
       */
      Signal.prototype.connect = function (slot, thisArg) {
          return Private$4.connect(this, slot, thisArg);
      };
      /**
       * Disconnect a slot from the signal.
       *
       * @param slot - The slot to disconnect from the signal.
       *
       * @param thisArg - The `this` context for the slot. If provided,
       *   this must be a non-primitive object.
       *
       * @returns `true` if the connection is removed, `false` otherwise.
       */
      Signal.prototype.disconnect = function (slot, thisArg) {
          return Private$4.disconnect(this, slot, thisArg);
      };
      /**
       * Emit the signal and invoke the connected slots.
       *
       * @param args - The args to pass to the connected slots.
       *
       * #### Notes
       * Slots are invoked synchronously in connection order.
       *
       * Exceptions thrown by connected slots will be caught and logged.
       */
      Signal.prototype.emit = function (args) {
          Private$4.emit(this, args);
      };
      return Signal;
  }());
  /**
   * The namespace for the `Signal` class statics.
   */
  (function (Signal) {
      /**
       * Remove all connections between a sender and receiver.
       *
       * @param sender - The sender object of interest.
       *
       * @param receiver - The receiver object of interest.
       *
       * #### Notes
       * If a `thisArg` is provided when connecting a signal, that object
       * is considered the receiver. Otherwise, the `slot` is considered
       * the receiver.
       */
      function disconnectBetween(sender, receiver) {
          Private$4.disconnectBetween(sender, receiver);
      }
      Signal.disconnectBetween = disconnectBetween;
      /**
       * Remove all connections where the given object is the sender.
       *
       * @param sender - The sender object of interest.
       */
      function disconnectSender(sender) {
          Private$4.disconnectSender(sender);
      }
      Signal.disconnectSender = disconnectSender;
      /**
       * Remove all connections where the given object is the receiver.
       *
       * @param receiver - The receiver object of interest.
       *
       * #### Notes
       * If a `thisArg` is provided when connecting a signal, that object
       * is considered the receiver. Otherwise, the `slot` is considered
       * the receiver.
       */
      function disconnectReceiver(receiver) {
          Private$4.disconnectReceiver(receiver);
      }
      Signal.disconnectReceiver = disconnectReceiver;
      /**
       * Remove all connections where an object is the sender or receiver.
       *
       * @param object - The object of interest.
       *
       * #### Notes
       * If a `thisArg` is provided when connecting a signal, that object
       * is considered the receiver. Otherwise, the `slot` is considered
       * the receiver.
       */
      function disconnectAll(object) {
          Private$4.disconnectAll(object);
      }
      Signal.disconnectAll = disconnectAll;
      /**
       * Clear all signal data associated with the given object.
       *
       * @param object - The object for which the data should be cleared.
       *
       * #### Notes
       * This removes all signal connections and any other signal data
       * associated with the object.
       */
      function clearData(object) {
          Private$4.disconnectAll(object);
      }
      Signal.clearData = clearData;
      /**
       * Get the signal exception handler.
       *
       * @returns The current exception handler.
       *
       * #### Notes
       * The default exception handler is `console.error`.
       */
      function getExceptionHandler() {
          return Private$4.exceptionHandler;
      }
      Signal.getExceptionHandler = getExceptionHandler;
      /**
       * Set the signal exception handler.
       *
       * @param handler - The function to use as the exception handler.
       *
       * @returns The old exception handler.
       *
       * #### Notes
       * The exception handler is invoked when a slot throws an exception.
       */
      function setExceptionHandler(handler) {
          var old = Private$4.exceptionHandler;
          Private$4.exceptionHandler = handler;
          return old;
      }
      Signal.setExceptionHandler = setExceptionHandler;
  })(Signal || (Signal = {}));
  /**
   * The namespace for the module implementation details.
   */
  var Private$4;
  (function (Private) {
      /**
       * The signal exception handler function.
       */
      Private.exceptionHandler = function (err) {
          console.error(err);
      };
      /**
       * Connect a slot to a signal.
       *
       * @param signal - The signal of interest.
       *
       * @param slot - The slot to invoke when the signal is emitted.
       *
       * @param thisArg - The `this` context for the slot. If provided,
       *   this must be a non-primitive object.
       *
       * @returns `true` if the connection succeeds, `false` otherwise.
       */
      function connect(signal, slot, thisArg) {
          // Coerce a `null` `thisArg` to `undefined`.
          thisArg = thisArg || undefined;
          // Ensure the sender's array of receivers is created.
          var receivers = receiversForSender.get(signal.sender);
          if (!receivers) {
              receivers = [];
              receiversForSender.set(signal.sender, receivers);
          }
          // Bail if a matching connection already exists.
          if (findConnection(receivers, signal, slot, thisArg)) {
              return false;
          }
          // Choose the best object for the receiver.
          var receiver = thisArg || slot;
          // Ensure the receiver's array of senders is created.
          var senders = sendersForReceiver.get(receiver);
          if (!senders) {
              senders = [];
              sendersForReceiver.set(receiver, senders);
          }
          // Create a new connection and add it to the end of each array.
          var connection = { signal: signal, slot: slot, thisArg: thisArg };
          receivers.push(connection);
          senders.push(connection);
          // Indicate a successful connection.
          return true;
      }
      Private.connect = connect;
      /**
       * Disconnect a slot from a signal.
       *
       * @param signal - The signal of interest.
       *
       * @param slot - The slot to disconnect from the signal.
       *
       * @param thisArg - The `this` context for the slot. If provided,
       *   this must be a non-primitive object.
       *
       * @returns `true` if the connection is removed, `false` otherwise.
       */
      function disconnect(signal, slot, thisArg) {
          // Coerce a `null` `thisArg` to `undefined`.
          thisArg = thisArg || undefined;
          // Lookup the list of receivers, and bail if none exist.
          var receivers = receiversForSender.get(signal.sender);
          if (!receivers || receivers.length === 0) {
              return false;
          }
          // Bail if no matching connection exits.
          var connection = findConnection(receivers, signal, slot, thisArg);
          if (!connection) {
              return false;
          }
          // Choose the best object for the receiver.
          var receiver = thisArg || slot;
          // Lookup the array of senders, which is now known to exist.
          var senders = sendersForReceiver.get(receiver);
          // Clear the connection and schedule cleanup of the arrays.
          connection.signal = null;
          scheduleCleanup(receivers);
          scheduleCleanup(senders);
          // Indicate a successful disconnection.
          return true;
      }
      Private.disconnect = disconnect;
      /**
       * Remove all connections between a sender and receiver.
       *
       * @param sender - The sender object of interest.
       *
       * @param receiver - The receiver object of interest.
       */
      function disconnectBetween(sender, receiver) {
          // If there are no receivers, there is nothing to do.
          var receivers = receiversForSender.get(sender);
          if (!receivers || receivers.length === 0) {
              return;
          }
          // If there are no senders, there is nothing to do.
          var senders = sendersForReceiver.get(receiver);
          if (!senders || senders.length === 0) {
              return;
          }
          // Clear each connection between the sender and receiver.
          each$1(senders, function (connection) {
              // Skip connections which have already been cleared.
              if (!connection.signal) {
                  return;
              }
              // Clear the connection if it matches the sender.
              if (connection.signal.sender === sender) {
                  connection.signal = null;
              }
          });
          // Schedule a cleanup of the senders and receivers.
          scheduleCleanup(receivers);
          scheduleCleanup(senders);
      }
      Private.disconnectBetween = disconnectBetween;
      /**
       * Remove all connections where the given object is the sender.
       *
       * @param sender - The sender object of interest.
       */
      function disconnectSender(sender) {
          // If there are no receivers, there is nothing to do.
          var receivers = receiversForSender.get(sender);
          if (!receivers || receivers.length === 0) {
              return;
          }
          // Clear each receiver connection.
          each$1(receivers, function (connection) {
              // Skip connections which have already been cleared.
              if (!connection.signal) {
                  return;
              }
              // Choose the best object for the receiver.
              var receiver = connection.thisArg || connection.slot;
              // Clear the connection.
              connection.signal = null;
              // Cleanup the array of senders, which is now known to exist.
              scheduleCleanup(sendersForReceiver.get(receiver));
          });
          // Schedule a cleanup of the receivers.
          scheduleCleanup(receivers);
      }
      Private.disconnectSender = disconnectSender;
      /**
       * Remove all connections where the given object is the receiver.
       *
       * @param receiver - The receiver object of interest.
       */
      function disconnectReceiver(receiver) {
          // If there are no senders, there is nothing to do.
          var senders = sendersForReceiver.get(receiver);
          if (!senders || senders.length === 0) {
              return;
          }
          // Clear each sender connection.
          each$1(senders, function (connection) {
              // Skip connections which have already been cleared.
              if (!connection.signal) {
                  return;
              }
              // Lookup the sender for the connection.
              var sender = connection.signal.sender;
              // Clear the connection.
              connection.signal = null;
              // Cleanup the array of receivers, which is now known to exist.
              scheduleCleanup(receiversForSender.get(sender));
          });
          // Schedule a cleanup of the list of senders.
          scheduleCleanup(senders);
      }
      Private.disconnectReceiver = disconnectReceiver;
      /**
       * Remove all connections where an object is the sender or receiver.
       *
       * @param object - The object of interest.
       */
      function disconnectAll(object) {
          // Remove all connections where the given object is the sender.
          disconnectSender(object);
          // Remove all connections where the given object is the receiver.
          disconnectReceiver(object);
      }
      Private.disconnectAll = disconnectAll;
      /**
       * Emit a signal and invoke its connected slots.
       *
       * @param signal - The signal of interest.
       *
       * @param args - The args to pass to the connected slots.
       *
       * #### Notes
       * Slots are invoked synchronously in connection order.
       *
       * Exceptions thrown by connected slots will be caught and logged.
       */
      function emit(signal, args) {
          // If there are no receivers, there is nothing to do.
          var receivers = receiversForSender.get(signal.sender);
          if (!receivers || receivers.length === 0) {
              return;
          }
          // Invoke the slots for connections with a matching signal.
          // Any connections added during emission are not invoked.
          for (var i = 0, n = receivers.length; i < n; ++i) {
              var connection = receivers[i];
              if (connection.signal === signal) {
                  invokeSlot(connection, args);
              }
          }
      }
      Private.emit = emit;
      /**
       * A weak mapping of sender to array of receiver connections.
       */
      var receiversForSender = new WeakMap();
      /**
       * A weak mapping of receiver to array of sender connections.
       */
      var sendersForReceiver = new WeakMap();
      /**
       * A set of connection arrays which are pending cleanup.
       */
      var dirtySet = new Set();
      /**
       * A function to schedule an event loop callback.
       */
      var schedule = (function () {
          var ok = typeof requestAnimationFrame === 'function';
          // @ts-ignore
          return ok ? requestAnimationFrame : setImmediate;
      })();
      /**
       * Find a connection which matches the given parameters.
       */
      function findConnection(connections, signal, slot, thisArg) {
          return find$1(connections, function (connection) { return (connection.signal === signal &&
              connection.slot === slot &&
              connection.thisArg === thisArg); });
      }
      /**
       * Invoke a slot with the given parameters.
       *
       * The connection is assumed to be valid.
       *
       * Exceptions in the slot will be caught and logged.
       */
      function invokeSlot(connection, args) {
          var signal = connection.signal, slot = connection.slot, thisArg = connection.thisArg;
          try {
              slot.call(thisArg, signal.sender, args);
          }
          catch (err) {
              Private.exceptionHandler(err);
          }
      }
      /**
       * Schedule a cleanup of a connection array.
       *
       * This will add the array to the dirty set and schedule a deferred
       * cleanup of the array contents. On cleanup, any connection with a
       * `null` signal will be removed from the array.
       */
      function scheduleCleanup(array) {
          if (dirtySet.size === 0) {
              schedule(cleanupDirtySet);
          }
          dirtySet.add(array);
      }
      /**
       * Cleanup the connection lists in the dirty set.
       *
       * This function should only be invoked asynchronously, when the
       * stack frame is guaranteed to not be on the path of user code.
       */
      function cleanupDirtySet() {
          dirtySet.forEach(cleanupConnections);
          dirtySet.clear();
      }
      /**
       * Cleanup the dirty connections in a connections array.
       *
       * This will remove any connection with a `null` signal.
       *
       * This function should only be invoked asynchronously, when the
       * stack frame is guaranteed to not be on the path of user code.
       */
      function cleanupConnections(connections) {
          ArrayExt.removeAllWhere(connections, isDeadConnection);
      }
      /**
       * Test whether a connection is dead.
       *
       * A dead connection has a `null` signal.
       */
      function isDeadConnection(connection) {
          return connection.signal === null;
      }
  })(Private$4 || (Private$4 = {}));

  /*! *****************************************************************************
  Copyright (c) Microsoft Corporation. All rights reserved.
  Licensed under the Apache License, Version 2.0 (the "License"); you may not use
  this file except in compliance with the License. You may obtain a copy of the
  License at http://www.apache.org/licenses/LICENSE-2.0

  THIS CODE IS PROVIDED ON AN *AS IS* BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
  KIND, EITHER EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION ANY IMPLIED
  WARRANTIES OR CONDITIONS OF TITLE, FITNESS FOR A PARTICULAR PURPOSE,
  MERCHANTABLITY OR NON-INFRINGEMENT.

  See the Apache Version 2.0 License for specific language governing permissions
  and limitations under the License.
  ***************************************************************************** */
  /* global Reflect, Promise */

  var extendStatics$1 = function(d, b) {
      extendStatics$1 = Object.setPrototypeOf ||
          ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
          function (d, b) { for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p]; };
      return extendStatics$1(d, b);
  };

  function __extends$3(d, b) {
      extendStatics$1(d, b);
      function __() { this.constructor = d; }
      d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
  }

  /**
   * A disposable object which delegates to a callback function.
   */
  var DisposableDelegate = /** @class */ (function () {
      /**
       * Construct a new disposable delegate.
       *
       * @param fn - The callback function to invoke on dispose.
       */
      function DisposableDelegate(fn) {
          this._fn = fn;
      }
      Object.defineProperty(DisposableDelegate.prototype, "isDisposed", {
          /**
           * Test whether the delegate has been disposed.
           */
          get: function () {
              return !this._fn;
          },
          enumerable: true,
          configurable: true
      });
      /**
       * Dispose of the delegate and invoke the callback function.
       */
      DisposableDelegate.prototype.dispose = function () {
          if (!this._fn) {
              return;
          }
          var fn = this._fn;
          this._fn = null;
          fn();
      };
      return DisposableDelegate;
  }());
  /**
   * An observable disposable object which delegates to a callback function.
   */
  var ObservableDisposableDelegate = /** @class */ (function (_super) {
      __extends$3(ObservableDisposableDelegate, _super);
      function ObservableDisposableDelegate() {
          var _this = _super !== null && _super.apply(this, arguments) || this;
          _this._disposed = new Signal(_this);
          return _this;
      }
      Object.defineProperty(ObservableDisposableDelegate.prototype, "disposed", {
          /**
           * A signal emitted when the delegate is disposed.
           */
          get: function () {
              return this._disposed;
          },
          enumerable: true,
          configurable: true
      });
      /**
       * Dispose of the delegate and invoke the callback function.
       */
      ObservableDisposableDelegate.prototype.dispose = function () {
          if (this.isDisposed) {
              return;
          }
          _super.prototype.dispose.call(this);
          this._disposed.emit(undefined);
          Signal.clearData(this);
      };
      return ObservableDisposableDelegate;
  }(DisposableDelegate));
  /**
   * An object which manages a collection of disposable items.
   */
  var DisposableSet = /** @class */ (function () {
      /**
       * Construct a new disposable set.
       */
      function DisposableSet() {
          this._isDisposed = false;
          this._items = new Set();
      }
      Object.defineProperty(DisposableSet.prototype, "isDisposed", {
          /**
           * Test whether the set has been disposed.
           */
          get: function () {
              return this._isDisposed;
          },
          enumerable: true,
          configurable: true
      });
      /**
       * Dispose of the set and the items it contains.
       *
       * #### Notes
       * Items are disposed in the order they are added to the set.
       */
      DisposableSet.prototype.dispose = function () {
          if (this._isDisposed) {
              return;
          }
          this._isDisposed = true;
          this._items.forEach(function (item) { item.dispose(); });
          this._items.clear();
      };
      /**
       * Test whether the set contains a specific item.
       *
       * @param item - The item of interest.
       *
       * @returns `true` if the set contains the item, `false` otherwise.
       */
      DisposableSet.prototype.contains = function (item) {
          return this._items.has(item);
      };
      /**
       * Add a disposable item to the set.
       *
       * @param item - The item to add to the set.
       *
       * #### Notes
       * If the item is already contained in the set, this is a no-op.
       */
      DisposableSet.prototype.add = function (item) {
          this._items.add(item);
      };
      /**
       * Remove a disposable item from the set.
       *
       * @param item - The item to remove from the set.
       *
       * #### Notes
       * If the item is not contained in the set, this is a no-op.
       */
      DisposableSet.prototype.remove = function (item) {
          this._items.delete(item);
      };
      /**
       * Remove all items from the set.
       */
      DisposableSet.prototype.clear = function () {
          this._items.clear();
      };
      return DisposableSet;
  }());
  /**
   * The namespace for the `DisposableSet` class statics.
   */
  (function (DisposableSet) {
      /**
       * Create a disposable set from an iterable of items.
       *
       * @param items - The iterable or array-like object of interest.
       *
       * @returns A new disposable initialized with the given items.
       */
      function from(items) {
          var set = new DisposableSet();
          each$1(items, function (item) { set.add(item); });
          return set;
      }
      DisposableSet.from = from;
  })(DisposableSet || (DisposableSet = {}));
  /**
   * An observable object which manages a collection of disposable items.
   */
  var ObservableDisposableSet = /** @class */ (function (_super) {
      __extends$3(ObservableDisposableSet, _super);
      function ObservableDisposableSet() {
          var _this = _super !== null && _super.apply(this, arguments) || this;
          _this._disposed = new Signal(_this);
          return _this;
      }
      Object.defineProperty(ObservableDisposableSet.prototype, "disposed", {
          /**
           * A signal emitted when the set is disposed.
           */
          get: function () {
              return this._disposed;
          },
          enumerable: true,
          configurable: true
      });
      /**
       * Dispose of the set and the items it contains.
       *
       * #### Notes
       * Items are disposed in the order they are added to the set.
       */
      ObservableDisposableSet.prototype.dispose = function () {
          if (this.isDisposed) {
              return;
          }
          _super.prototype.dispose.call(this);
          this._disposed.emit(undefined);
          Signal.clearData(this);
      };
      return ObservableDisposableSet;
  }(DisposableSet));
  /**
   * The namespace for the `ObservableDisposableSet` class statics.
   */
  (function (ObservableDisposableSet) {
      /**
       * Create an observable disposable set from an iterable of items.
       *
       * @param items - The iterable or array-like object of interest.
       *
       * @returns A new disposable initialized with the given items.
       */
      function from(items) {
          var set = new ObservableDisposableSet();
          each$1(items, function (item) { set.add(item); });
          return set;
      }
      ObservableDisposableSet.from = from;
  })(ObservableDisposableSet || (ObservableDisposableSet = {}));

  // Copyright (c) Jupyter Development Team.
  // Distributed under the terms of the Modified BSD License.
  /*-----------------------------------------------------------------------------
  | Copyright (c) 2014-2017, PhosphorJS Contributors
  |
  | Distributed under the terms of the BSD 3-Clause License.
  |
  | The full license is in the file LICENSE, distributed with this software.
  |----------------------------------------------------------------------------*/
  /**
   * Get the global application keyboard layout instance.
   *
   * @returns The keyboard layout for use by the application.
   *
   * #### Notes
   * The default keyboard layout is US-English.
   */
  function getKeyboardLayout() {
      return Private$5.keyboardLayout;
  }
  /**
   * A concrete implementation of [[IKeyboardLayout]] based on keycodes.
   *
   * The `keyCode` property of a `'keydown'` event is a browser and OS
   * specific representation of the physical key (not character) which
   * was pressed on a keyboard. While not the most convenient API, it
   * is currently the only one which works reliably on all browsers.
   *
   * This class accepts a user-defined mapping of keycode to key, which
   * allows for reliable shortcuts tailored to the user's system.
   */
  var KeycodeLayout = /** @class */ (function () {
      /**
       * Construct a new keycode layout.
       *
       * @param name - The human readable name for the layout.
       *
       * @param codes - A mapping of keycode to key value.
       */
      function KeycodeLayout(name, codes) {
          this.name = name;
          this._codes = codes;
          this._keys = KeycodeLayout.extractKeys(codes);
      }
      /**
       * Get an array of the key values supported by the layout.
       *
       * @returns A new array of the supported key values.
       */
      KeycodeLayout.prototype.keys = function () {
          return Object.keys(this._keys);
      };
      /**
       * Test whether the given key is a valid value for the layout.
       *
       * @param key - The user provided key to test for validity.
       *
       * @returns `true` if the key is valid, `false` otherwise.
       */
      KeycodeLayout.prototype.isValidKey = function (key) {
          return key in this._keys;
      };
      /**
       * Get the key for a `'keydown'` event.
       *
       * @param event - The event object for a `'keydown'` event.
       *
       * @returns The associated key value, or an empty string if
       *   the event does not represent a valid primary key.
       */
      KeycodeLayout.prototype.keyForKeydownEvent = function (event) {
          return this._codes[event.keyCode] || '';
      };
      return KeycodeLayout;
  }());
  /**
   * The namespace for the `KeycodeLayout` class statics.
   */
  (function (KeycodeLayout) {
      /**
       * Extract the set of keys from a code map.
       *
       * @param code - The code map of interest.
       *
       * @returns A set of the keys in the code map.
       */
      function extractKeys(codes) {
          var keys = Object.create(null);
          for (var c in codes) {
              keys[codes[c]] = true;
          }
          return keys;
      }
      KeycodeLayout.extractKeys = extractKeys;
  })(KeycodeLayout || (KeycodeLayout = {}));
  /**
   * A keycode-based keyboard layout for US English keyboards.
   *
   * This layout is valid for the following OS/Browser combinations.
   *
   * - Windows
   *   - Chrome
   *   - Firefox
   *   - IE
   *
   * - OSX
   *   - Chrome
   *   - Firefox
   *   - Safari
   *
   * - Linux
   *   - Chrome
   *   - Firefox
   *
   * Other combinations may also work, but are untested.
   */
  var EN_US = new KeycodeLayout('en-us', {
      8: 'Backspace',
      9: 'Tab',
      13: 'Enter',
      19: 'Pause',
      27: 'Escape',
      32: 'Space',
      33: 'PageUp',
      34: 'PageDown',
      35: 'End',
      36: 'Home',
      37: 'ArrowLeft',
      38: 'ArrowUp',
      39: 'ArrowRight',
      40: 'ArrowDown',
      45: 'Insert',
      46: 'Delete',
      48: '0',
      49: '1',
      50: '2',
      51: '3',
      52: '4',
      53: '5',
      54: '6',
      55: '7',
      56: '8',
      57: '9',
      59: ';',
      61: '=',
      65: 'A',
      66: 'B',
      67: 'C',
      68: 'D',
      69: 'E',
      70: 'F',
      71: 'G',
      72: 'H',
      73: 'I',
      74: 'J',
      75: 'K',
      76: 'L',
      77: 'M',
      78: 'N',
      79: 'O',
      80: 'P',
      81: 'Q',
      82: 'R',
      83: 'S',
      84: 'T',
      85: 'U',
      86: 'V',
      87: 'W',
      88: 'X',
      89: 'Y',
      90: 'Z',
      93: 'ContextMenu',
      96: '0',
      97: '1',
      98: '2',
      99: '3',
      100: '4',
      101: '5',
      102: '6',
      103: '7',
      104: '8',
      105: '9',
      106: '*',
      107: '+',
      109: '-',
      110: '.',
      111: '/',
      112: 'F1',
      113: 'F2',
      114: 'F3',
      115: 'F4',
      116: 'F5',
      117: 'F6',
      118: 'F7',
      119: 'F8',
      120: 'F9',
      121: 'F10',
      122: 'F11',
      123: 'F12',
      173: '-',
      186: ';',
      187: '=',
      188: ',',
      189: '-',
      190: '.',
      191: '/',
      192: '`',
      219: '[',
      220: '\\',
      221: ']',
      222: '\''
  });
  /**
   * The namespace for the module implementation details.
   */
  var Private$5;
  (function (Private) {
      /**
       * The global keyboard layout instance.
       */
      Private.keyboardLayout = EN_US;
  })(Private$5 || (Private$5 = {}));

  // Copyright (c) Jupyter Development Team.
  /**
   * An object which manages a collection of commands.
   *
   * #### Notes
   * A command registry can be used to populate a variety of action-based
   * widgets, such as command palettes, menus, and toolbars.
   */
  var CommandRegistry = /** @class */ (function () {
      /**
       * Construct a new command registry.
       */
      function CommandRegistry() {
          this._timerID = 0;
          this._replaying = false;
          this._keystrokes = [];
          this._keydownEvents = [];
          this._keyBindings = [];
          this._exactKeyMatch = null;
          this._commands = Object.create(null);
          this._commandChanged = new Signal(this);
          this._commandExecuted = new Signal(this);
          this._keyBindingChanged = new Signal(this);
      }
      Object.defineProperty(CommandRegistry.prototype, "commandChanged", {
          /**
           * A signal emitted when a command has changed.
           *
           * #### Notes
           * This signal is useful for visual representations of commands which
           * need to refresh when the state of a relevant command has changed.
           */
          get: function () {
              return this._commandChanged;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(CommandRegistry.prototype, "commandExecuted", {
          /**
           * A signal emitted when a command has executed.
           *
           * #### Notes
           * Care should be taken when consuming this signal. It is intended to
           * be used largely for debugging and logging purposes. It should not
           * be (ab)used for general purpose spying on command execution.
           */
          get: function () {
              return this._commandExecuted;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(CommandRegistry.prototype, "keyBindingChanged", {
          /**
           * A signal emitted when a key binding is changed.
           */
          get: function () {
              return this._keyBindingChanged;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(CommandRegistry.prototype, "keyBindings", {
          /**
           * A read-only array of the key bindings in the registry.
           */
          get: function () {
              return this._keyBindings;
          },
          enumerable: true,
          configurable: true
      });
      /**
       * List the ids of the registered commands.
       *
       * @returns A new array of the registered command ids.
       */
      CommandRegistry.prototype.listCommands = function () {
          return Object.keys(this._commands);
      };
      /**
       * Test whether a specific command is registered.
       *
       * @param id - The id of the command of interest.
       *
       * @returns `true` if the command is registered, `false` otherwise.
       */
      CommandRegistry.prototype.hasCommand = function (id) {
          return id in this._commands;
      };
      /**
       * Add a command to the registry.
       *
       * @param id - The unique id of the command.
       *
       * @param options - The options for the command.
       *
       * @returns A disposable which will remove the command.
       *
       * @throws An error if the given `id` is already registered.
       */
      CommandRegistry.prototype.addCommand = function (id, options) {
          var _this = this;
          // Throw an error if the id is already registered.
          if (id in this._commands) {
              throw new Error("Command '" + id + "' already registered.");
          }
          // Add the command to the registry.
          this._commands[id] = Private$6.createCommand(options);
          // Emit the `commandChanged` signal.
          this._commandChanged.emit({ id: id, type: 'added' });
          // Return a disposable which will remove the command.
          return new DisposableDelegate(function () {
              // Remove the command from the registry.
              delete _this._commands[id];
              // Emit the `commandChanged` signal.
              _this._commandChanged.emit({ id: id, type: 'removed' });
          });
      };
      /**
       * Notify listeners that the state of a command has changed.
       *
       * @param id - The id of the command which has changed. If more than
       *   one command has changed, this argument should be omitted.
       *
       * @throws An error if the given `id` is not registered.
       *
       * #### Notes
       * This method should be called by the command author whenever the
       * application state changes such that the results of the command
       * metadata functions may have changed.
       *
       * This will cause the `commandChanged` signal to be emitted.
       */
      CommandRegistry.prototype.notifyCommandChanged = function (id) {
          if (id !== undefined && !(id in this._commands)) {
              throw new Error("Command '" + id + "' is not registered.");
          }
          this._commandChanged.emit({ id: id, type: id ? 'changed' : 'many-changed' });
      };
      /**
       * Get the display label for a specific command.
       *
       * @param id - The id of the command of interest.
       *
       * @param args - The arguments for the command.
       *
       * @returns The display label for the command, or an empty string
       *   if the command is not registered.
       */
      CommandRegistry.prototype.label = function (id, args) {
          if (args === void 0) { args = JSONExt.emptyObject; }
          var cmd = this._commands[id];
          return cmd ? cmd.label.call(undefined, args) : '';
      };
      /**
       * Get the mnemonic index for a specific command.
       *
       * @param id - The id of the command of interest.
       *
       * @param args - The arguments for the command.
       *
       * @returns The mnemonic index for the command, or `-1` if the
       *   command is not registered.
       */
      CommandRegistry.prototype.mnemonic = function (id, args) {
          if (args === void 0) { args = JSONExt.emptyObject; }
          var cmd = this._commands[id];
          return cmd ? cmd.mnemonic.call(undefined, args) : -1;
      };
      /**
       * Get the icon renderer for a specific command.
       *
       * DEPRECATED: if set to a string value, the .icon field will
       * function as an alias for the .iconClass field, for backwards
       * compatibility. In the future when this is removed, the default
       * return type will become undefined.
       *
       * @param id - The id of the command of interest.
       *
       * @param args - The arguments for the command.
       *
       * @returns The icon renderer for the command, or
       *   an empty string if the command is not registered.
       */
      CommandRegistry.prototype.icon = function (id, args) {
          if (args === void 0) { args = JSONExt.emptyObject; }
          var cmd = this._commands[id];
          return cmd ? cmd.icon.call(undefined, args) : /* <DEPRECATED> */ '' /* </DEPRECATED> */ /* <FUTURE> undefined </FUTURE> */;
      };
      /**
       * Get the icon class for a specific command.
       *
       * @param id - The id of the command of interest.
       *
       * @param args - The arguments for the command.
       *
       * @returns The icon class for the command, or an empty string if
       *   the command is not registered.
       */
      CommandRegistry.prototype.iconClass = function (id, args) {
          if (args === void 0) { args = JSONExt.emptyObject; }
          var cmd = this._commands[id];
          return cmd ? cmd.iconClass.call(undefined, args) : '';
      };
      /**
       * Get the icon label for a specific command.
       *
       * @param id - The id of the command of interest.
       *
       * @param args - The arguments for the command.
       *
       * @returns The icon label for the command, or an empty string if
       *   the command is not registered.
       */
      CommandRegistry.prototype.iconLabel = function (id, args) {
          if (args === void 0) { args = JSONExt.emptyObject; }
          var cmd = this._commands[id];
          return cmd ? cmd.iconLabel.call(undefined, args) : '';
      };
      /**
       * Get the short form caption for a specific command.
       *
       * @param id - The id of the command of interest.
       *
       * @param args - The arguments for the command.
       *
       * @returns The caption for the command, or an empty string if the
       *   command is not registered.
       */
      CommandRegistry.prototype.caption = function (id, args) {
          if (args === void 0) { args = JSONExt.emptyObject; }
          var cmd = this._commands[id];
          return cmd ? cmd.caption.call(undefined, args) : '';
      };
      /**
       * Get the usage help text for a specific command.
       *
       * @param id - The id of the command of interest.
       *
       * @param args - The arguments for the command.
       *
       * @returns The usage text for the command, or an empty string if
       *   the command is not registered.
       */
      CommandRegistry.prototype.usage = function (id, args) {
          if (args === void 0) { args = JSONExt.emptyObject; }
          var cmd = this._commands[id];
          return cmd ? cmd.usage.call(undefined, args) : '';
      };
      /**
       * Get the extra class name for a specific command.
       *
       * @param id - The id of the command of interest.
       *
       * @param args - The arguments for the command.
       *
       * @returns The class name for the command, or an empty string if
       *   the command is not registered.
       */
      CommandRegistry.prototype.className = function (id, args) {
          if (args === void 0) { args = JSONExt.emptyObject; }
          var cmd = this._commands[id];
          return cmd ? cmd.className.call(undefined, args) : '';
      };
      /**
       * Get the dataset for a specific command.
       *
       * @param id - The id of the command of interest.
       *
       * @param args - The arguments for the command.
       *
       * @returns The dataset for the command, or an empty dataset if
       *   the command is not registered.
       */
      CommandRegistry.prototype.dataset = function (id, args) {
          if (args === void 0) { args = JSONExt.emptyObject; }
          var cmd = this._commands[id];
          return cmd ? cmd.dataset.call(undefined, args) : {};
      };
      /**
       * Test whether a specific command is enabled.
       *
       * @param id - The id of the command of interest.
       *
       * @param args - The arguments for the command.
       *
       * @returns A boolean indicating whether the command is enabled,
       *   or `false` if the command is not registered.
       */
      CommandRegistry.prototype.isEnabled = function (id, args) {
          if (args === void 0) { args = JSONExt.emptyObject; }
          var cmd = this._commands[id];
          return cmd ? cmd.isEnabled.call(undefined, args) : false;
      };
      /**
       * Test whether a specific command is toggled.
       *
       * @param id - The id of the command of interest.
       *
       * @param args - The arguments for the command.
       *
       * @returns A boolean indicating whether the command is toggled,
       *   or `false` if the command is not registered.
       */
      CommandRegistry.prototype.isToggled = function (id, args) {
          if (args === void 0) { args = JSONExt.emptyObject; }
          var cmd = this._commands[id];
          return cmd ? cmd.isToggled.call(undefined, args) : false;
      };
      /**
       * Test whether a specific command is visible.
       *
       * @param id - The id of the command of interest.
       *
       * @param args - The arguments for the command.
       *
       * @returns A boolean indicating whether the command is visible,
       *   or `false` if the command is not registered.
       */
      CommandRegistry.prototype.isVisible = function (id, args) {
          if (args === void 0) { args = JSONExt.emptyObject; }
          var cmd = this._commands[id];
          return cmd ? cmd.isVisible.call(undefined, args) : false;
      };
      /**
       * Execute a specific command.
       *
       * @param id - The id of the command of interest.
       *
       * @param args - The arguments for the command.
       *
       * @returns A promise which resolves with the result of the command.
       *
       * #### Notes
       * The promise will reject if the command throws an exception,
       * or if the command is not registered.
       */
      CommandRegistry.prototype.execute = function (id, args) {
          if (args === void 0) { args = JSONExt.emptyObject; }
          // Reject if the command is not registered.
          var cmd = this._commands[id];
          if (!cmd) {
              return Promise.reject(new Error("Command '" + id + "' not registered."));
          }
          // Execute the command and reject if an exception is thrown.
          var value;
          try {
              value = cmd.execute.call(undefined, args);
          }
          catch (err) {
              value = Promise.reject(err);
          }
          // Create the return promise which resolves the result.
          var result = Promise.resolve(value);
          // Emit the command executed signal.
          this._commandExecuted.emit({ id: id, args: args, result: result });
          // Return the result promise to the caller.
          return result;
      };
      /**
       * Add a key binding to the registry.
       *
       * @param options - The options for creating the key binding.
       *
       * @returns A disposable which removes the added key binding.
       *
       * #### Notes
       * If multiple key bindings are registered for the same sequence, the
       * binding with the highest selector specificity is executed first. A
       * tie is broken by using the most recently added key binding.
       *
       * Ambiguous key bindings are resolved with a timeout. As an example,
       * suppose two key bindings are registered: one with the key sequence
       * `['Ctrl D']`, and another with `['Ctrl D', 'Ctrl W']`. If the user
       * presses `Ctrl D`, the first binding cannot be immediately executed
       * since the user may intend to complete the chord with `Ctrl W`. For
       * such cases, a timer is used to allow the chord to be completed. If
       * the chord is not completed before the timeout, the first binding
       * is executed.
       */
      CommandRegistry.prototype.addKeyBinding = function (options) {
          var _this = this;
          // Create the binding for the given options.
          var binding = Private$6.createKeyBinding(options);
          // Add the key binding to the bindings array.
          this._keyBindings.push(binding);
          // Emit the `bindingChanged` signal.
          this._keyBindingChanged.emit({ binding: binding, type: 'added' });
          // Return a disposable which will remove the binding.
          return new DisposableDelegate(function () {
              // Remove the binding from the array.
              ArrayExt.removeFirstOf(_this._keyBindings, binding);
              // Emit the `bindingChanged` signal.
              _this._keyBindingChanged.emit({ binding: binding, type: 'removed' });
          });
      };
      /**
       * Process a `'keydown'` event and invoke a matching key binding.
       *
       * @param event - The event object for a `'keydown'` event.
       *
       * #### Notes
       * This should be called in response to a `'keydown'` event in order
       * to invoke the command for the best matching key binding.
       *
       * The registry **does not** install its own listener for `'keydown'`
       * events. This allows the application full control over the nodes
       * and phase for which the registry processes `'keydown'` events.
       *
       * When the keydown event is processed, if the event target or any of its
       * ancestor nodes has a `data-lm-suppress-shortcuts` attribute, its keydown
       * events will not invoke commands.
       */
      CommandRegistry.prototype.processKeydownEvent = function (event) {
          // Bail immediately if playing back keystrokes.
          if (this._replaying) {
              return;
          }
          // Get the normalized keystroke for the event.
          var keystroke = CommandRegistry.keystrokeForKeydownEvent(event);
          // If the keystroke is not valid for the keyboard layout, replay
          // any suppressed events and clear the pending state.
          if (!keystroke) {
              this._replayKeydownEvents();
              this._clearPendingState();
              return;
          }
          // Add the keystroke to the current key sequence.
          this._keystrokes.push(keystroke);
          // Find the exact and partial matches for the key sequence.
          var _a = Private$6.matchKeyBinding(this._keyBindings, this._keystrokes, event), exact = _a.exact, partial = _a.partial;
          // If there is no exact match and no partial match, replay
          // any suppressed events and clear the pending state.
          if (!exact && !partial) {
              this._replayKeydownEvents();
              this._clearPendingState();
              return;
          }
          // Stop propagation of the event. If there is only a partial match,
          // the event will be replayed if a final exact match never occurs.
          event.preventDefault();
          event.stopPropagation();
          // If there is an exact match but no partial match, the exact match
          // can be dispatched immediately. The pending state is cleared so
          // the next key press starts from the default state.
          if (exact && !partial) {
              this._executeKeyBinding(exact);
              this._clearPendingState();
              return;
          }
          // If there is both an exact match and a partial match, the exact
          // match is stored for future dispatch in case the timer expires
          // before a more specific match is triggered.
          if (exact) {
              this._exactKeyMatch = exact;
          }
          // Store the event for possible playback in the future.
          this._keydownEvents.push(event);
          // (Re)start the timer to dispatch the most recent exact match
          // in case the partial match fails to result in an exact match.
          this._startTimer();
      };
      /**
       * Start or restart the pending timeout.
       */
      CommandRegistry.prototype._startTimer = function () {
          var _this = this;
          this._clearTimer();
          this._timerID = window.setTimeout(function () {
              _this._onPendingTimeout();
          }, Private$6.CHORD_TIMEOUT);
      };
      /**
       * Clear the pending timeout.
       */
      CommandRegistry.prototype._clearTimer = function () {
          if (this._timerID !== 0) {
              clearTimeout(this._timerID);
              this._timerID = 0;
          }
      };
      /**
       * Replay the keydown events which were suppressed.
       */
      CommandRegistry.prototype._replayKeydownEvents = function () {
          if (this._keydownEvents.length === 0) {
              return;
          }
          this._replaying = true;
          this._keydownEvents.forEach(Private$6.replayKeyEvent);
          this._replaying = false;
      };
      /**
       * Execute the command for the given key binding.
       *
       * If the command is missing or disabled, a warning will be logged.
       */
      CommandRegistry.prototype._executeKeyBinding = function (binding) {
          var command = binding.command, args = binding.args;
          if (!this.hasCommand(command) || !this.isEnabled(command, args)) {
              var word = this.hasCommand(command) ? 'enabled' : 'registered';
              var keys = binding.keys.join(', ');
              var msg1 = "Cannot execute key binding '" + keys + "':";
              var msg2 = "command '" + command + "' is not " + word + ".";
              console.warn(msg1 + " " + msg2);
              return;
          }
          this.execute(command, args);
      };
      /**
       * Clear the internal pending state.
       */
      CommandRegistry.prototype._clearPendingState = function () {
          this._clearTimer();
          this._exactKeyMatch = null;
          this._keystrokes.length = 0;
          this._keydownEvents.length = 0;
      };
      /**
       * Handle the partial match timeout.
       */
      CommandRegistry.prototype._onPendingTimeout = function () {
          this._timerID = 0;
          if (this._exactKeyMatch) {
              this._executeKeyBinding(this._exactKeyMatch);
          }
          else {
              this._replayKeydownEvents();
          }
          this._clearPendingState();
      };
      return CommandRegistry;
  }());
  /**
   * The namespace for the `CommandRegistry` class statics.
   */
  (function (CommandRegistry) {
      /**
       * Parse a keystroke into its constituent components.
       *
       * @param keystroke - The keystroke of interest.
       *
       * @returns The parsed components of the keystroke.
       *
       * #### Notes
       * The keystroke should be of the form:
       *   `[<modifier 1> [<modifier 2> [<modifier N> ]]]<primary key>`
       *
       * The supported modifiers are: `Accel`, `Alt`, `Cmd`, `Ctrl`, and
       * `Shift`. The `Accel` modifier is translated to `Cmd` on Mac and
       * `Ctrl` on all other platforms.
       *
       * The parsing is tolerant and will not throw exceptions. Notably:
       *   - Duplicate modifiers are ignored.
       *   - Extra primary keys are ignored.
       *   - The order of modifiers and primary key is irrelevant.
       *   - The keystroke parts should be separated by whitespace.
       *   - The keystroke is case sensitive.
       */
      function parseKeystroke(keystroke) {
          var key = '';
          var alt = false;
          var cmd = false;
          var ctrl = false;
          var shift = false;
          for (var _i = 0, _a = keystroke.split(/\s+/); _i < _a.length; _i++) {
              var token = _a[_i];
              if (token === 'Accel') {
                  if (Platform.IS_MAC) {
                      cmd = true;
                  }
                  else {
                      ctrl = true;
                  }
              }
              else if (token === 'Alt') {
                  alt = true;
              }
              else if (token === 'Cmd') {
                  cmd = true;
              }
              else if (token === 'Ctrl') {
                  ctrl = true;
              }
              else if (token === 'Shift') {
                  shift = true;
              }
              else if (token.length > 0) {
                  key = token;
              }
          }
          return { cmd: cmd, ctrl: ctrl, alt: alt, shift: shift, key: key };
      }
      CommandRegistry.parseKeystroke = parseKeystroke;
      /**
       * Normalize a keystroke into a canonical representation.
       *
       * @param keystroke - The keystroke of interest.
       *
       * @returns The normalized representation of the keystroke.
       *
       * #### Notes
       * This normalizes the keystroke by removing duplicate modifiers and
       * extra primary keys, and assembling the parts in a canonical order.
       *
       * The `Cmd` modifier is ignored on non-Mac platforms.
       */
      function normalizeKeystroke(keystroke) {
          var mods = '';
          var parts = parseKeystroke(keystroke);
          if (parts.ctrl) {
              mods += 'Ctrl ';
          }
          if (parts.alt) {
              mods += 'Alt ';
          }
          if (parts.shift) {
              mods += 'Shift ';
          }
          if (parts.cmd && Platform.IS_MAC) {
              mods += 'Cmd ';
          }
          return mods + parts.key;
      }
      CommandRegistry.normalizeKeystroke = normalizeKeystroke;
      /**
       * Get the platform-specific normalized keys for an options object.
       *
       * @param options - The options for the key binding.
       *
       * @returns Array of combined, normalized keys.
       */
      function normalizeKeys(options) {
          var keys;
          if (Platform.IS_WIN) {
              keys = options.winKeys || options.keys;
          }
          else if (Platform.IS_MAC) {
              keys = options.macKeys || options.keys;
          }
          else {
              keys = options.linuxKeys || options.keys;
          }
          return keys.map(normalizeKeystroke);
      }
      CommandRegistry.normalizeKeys = normalizeKeys;
      /**
       * Format a keystroke for display on the local system.
       */
      function formatKeystroke(keystroke) {
          var mods = '';
          var parts = parseKeystroke(keystroke);
          if (Platform.IS_MAC) {
              if (parts.ctrl) {
                  mods += '\u2303 ';
              }
              if (parts.alt) {
                  mods += '\u2325 ';
              }
              if (parts.shift) {
                  mods += '\u21E7 ';
              }
              if (parts.cmd) {
                  mods += '\u2318 ';
              }
          }
          else {
              if (parts.ctrl) {
                  mods += 'Ctrl+';
              }
              if (parts.alt) {
                  mods += 'Alt+';
              }
              if (parts.shift) {
                  mods += 'Shift+';
              }
          }
          return mods + parts.key;
      }
      CommandRegistry.formatKeystroke = formatKeystroke;
      /**
       * Create a normalized keystroke for a `'keydown'` event.
       *
       * @param event - The event object for a `'keydown'` event.
       *
       * @returns A normalized keystroke, or an empty string if the event
       *   does not represent a valid keystroke for the given layout.
       */
      function keystrokeForKeydownEvent(event) {
          var key = getKeyboardLayout().keyForKeydownEvent(event);
          if (!key) {
              return '';
          }
          var mods = '';
          if (event.ctrlKey) {
              mods += 'Ctrl ';
          }
          if (event.altKey) {
              mods += 'Alt ';
          }
          if (event.shiftKey) {
              mods += 'Shift ';
          }
          if (event.metaKey && Platform.IS_MAC) {
              mods += 'Cmd ';
          }
          return mods + key;
      }
      CommandRegistry.keystrokeForKeydownEvent = keystrokeForKeydownEvent;
  })(CommandRegistry || (CommandRegistry = {}));
  /**
   * The namespace for the module implementation details.
   */
  var Private$6;
  (function (Private) {
      /**
       * The timeout in ms for triggering a key binding chord.
       */
      Private.CHORD_TIMEOUT = 1000;
      /**
       * Create a normalized command from an options object.
       */
      function createCommand(options) {
          var icon;
          var iconClass;
          /* <DEPRECATED> */
          if (!(options.icon) || typeof options.icon === 'string') {
              // alias icon to iconClass
              iconClass = asFunc(options.iconClass || options.icon, emptyStringFunc);
              icon = iconClass;
          }
          else {
              /* /<DEPRECATED> */
              iconClass = asFunc(options.iconClass, emptyStringFunc);
              icon = asFunc(options.icon, undefinedFunc);
              /* <DEPRECATED> */
          }
          /* </DEPRECATED> */
          return {
              execute: options.execute,
              label: asFunc(options.label, emptyStringFunc),
              mnemonic: asFunc(options.mnemonic, negativeOneFunc),
              icon: icon,
              iconClass: iconClass,
              iconLabel: asFunc(options.iconLabel, emptyStringFunc),
              caption: asFunc(options.caption, emptyStringFunc),
              usage: asFunc(options.usage, emptyStringFunc),
              className: asFunc(options.className, emptyStringFunc),
              dataset: asFunc(options.dataset, emptyDatasetFunc),
              isEnabled: options.isEnabled || trueFunc,
              isToggled: options.isToggled || falseFunc,
              isVisible: options.isVisible || trueFunc
          };
      }
      Private.createCommand = createCommand;
      /**
       * Create a key binding object from key binding options.
       */
      function createKeyBinding(options) {
          return {
              keys: CommandRegistry.normalizeKeys(options),
              selector: validateSelector(options),
              command: options.command,
              args: options.args || JSONExt.emptyObject
          };
      }
      Private.createKeyBinding = createKeyBinding;
      /**
       * Find the key bindings which match a key sequence.
       *
       * This returns a match result which contains the best exact matching
       * binding, and a flag which indicates if there are partial matches.
       */
      function matchKeyBinding(bindings, keys, event) {
          // The current best exact match.
          var exact = null;
          // Whether a partial match has been found.
          var partial = false;
          // The match distance for the exact match.
          var distance = Infinity;
          // The specificity for the exact match.
          var specificity = 0;
          // Iterate over the bindings and search for the best match.
          for (var i = 0, n = bindings.length; i < n; ++i) {
              // Lookup the current binding.
              var binding = bindings[i];
              // Check whether the key binding sequence is a match.
              var sqm = matchSequence(binding.keys, keys);
              // If there is no match, the binding is ignored.
              if (sqm === 0 /* None */) {
                  continue;
              }
              // If it is a partial match and no other partial match has been
              // found, ensure the selector matches and set the partial flag.
              if (sqm === 2 /* Partial */) {
                  if (!partial && targetDistance(binding.selector, event) !== -1) {
                      partial = true;
                  }
                  continue;
              }
              // Ignore the match if the selector doesn't match, or if the
              // matched node is farther away than the current best match.
              var td = targetDistance(binding.selector, event);
              if (td === -1 || td > distance) {
                  continue;
              }
              // Get the specificity for the selector.
              var sp = Selector.calculateSpecificity(binding.selector);
              // Update the best match if this match is stronger.
              if (!exact || td < distance || sp >= specificity) {
                  exact = binding;
                  distance = td;
                  specificity = sp;
              }
          }
          // Return the match result.
          return { exact: exact, partial: partial };
      }
      Private.matchKeyBinding = matchKeyBinding;
      /**
       * Replay a keyboard event.
       *
       * This synthetically dispatches a clone of the keyboard event.
       */
      function replayKeyEvent(event) {
          event.target.dispatchEvent(cloneKeyboardEvent(event));
      }
      Private.replayKeyEvent = replayKeyEvent;
      /**
       * A singleton empty string function.
       */
      var emptyStringFunc = function () { return ''; };
      /**
       * A singleton `-1` number function
       */
      var negativeOneFunc = function () { return -1; };
      /**
       * A singleton true boolean function.
       */
      var trueFunc = function () { return true; };
      /**
       * A singleton false boolean function.
       */
      var falseFunc = function () { return false; };
      /**
       * A singleton empty dataset function.
       */
      var emptyDatasetFunc = function () { return ({}); };
      /**
       * A singleton undefined function
       */
      var undefinedFunc = function () { return undefined; };
      /**
       * Cast a value or command func to a command func.
       */
      function asFunc(value, dfault) {
          if (value === undefined) {
              return dfault;
          }
          if (typeof value === 'function') {
              return value;
          }
          return function () { return value; };
      }
      /**
       * Validate the selector for an options object.
       *
       * This returns the validated selector, or throws if the selector is
       * invalid or contains commas.
       */
      function validateSelector(options) {
          if (options.selector.indexOf(',') !== -1) {
              throw new Error("Selector cannot contain commas: " + options.selector);
          }
          if (!Selector.isValid(options.selector)) {
              throw new Error("Invalid selector: " + options.selector);
          }
          return options.selector;
      }
      /**
       * Test whether a key binding sequence matches a key sequence.
       *
       * Returns a `SequenceMatch` value indicating the type of match.
       */
      function matchSequence(bindKeys, userKeys) {
          if (bindKeys.length < userKeys.length) {
              return 0 /* None */;
          }
          for (var i = 0, n = userKeys.length; i < n; ++i) {
              if (bindKeys[i] !== userKeys[i]) {
                  return 0 /* None */;
              }
          }
          if (bindKeys.length > userKeys.length) {
              return 2 /* Partial */;
          }
          return 1 /* Exact */;
      }
      /**
       * Find the distance from the target node to the first matching node.
       *
       * This traverses the event path from `target` to `currentTarget` and
       * computes the distance from `target` to the first node which matches
       * the CSS selector. If no match is found, `-1` is returned.
       */
      function targetDistance(selector, event) {
          var targ = event.target;
          var curr = event.currentTarget;
          for (var dist = 0; targ !== null; targ = targ.parentElement, ++dist) {
              if (targ.hasAttribute('data-lm-suppress-shortcuts')) {
                  return -1;
              }
              /* <DEPRECATED> */
              if (targ.hasAttribute('data-p-suppress-shortcuts')) {
                  return -1;
              }
              /* </DEPRECATED> */
              if (Selector.matches(targ, selector)) {
                  return dist;
              }
              if (targ === curr) {
                  return -1;
              }
          }
          return -1;
      }
      /**
       * Clone a keyboard event.
       */
      function cloneKeyboardEvent(event) {
          // A custom event is required because Chrome nulls out the
          // `keyCode` field in user-generated `KeyboardEvent` types.
          var clone = document.createEvent('Event');
          var bubbles = event.bubbles || true;
          var cancelable = event.cancelable || true;
          clone.initEvent(event.type || 'keydown', bubbles, cancelable);
          clone.key = event.key || '';
          clone.keyCode = event.keyCode || 0;
          clone.which = event.keyCode || 0;
          clone.ctrlKey = event.ctrlKey || false;
          clone.altKey = event.altKey || false;
          clone.shiftKey = event.shiftKey || false;
          clone.metaKey = event.metaKey || false;
          clone.view = event.view || window;
          return clone;
      }
  })(Private$6 || (Private$6 = {}));

  /*! *****************************************************************************
  Copyright (c) Microsoft Corporation. All rights reserved.
  Licensed under the Apache License, Version 2.0 (the "License"); you may not use
  this file except in compliance with the License. You may obtain a copy of the
  License at http://www.apache.org/licenses/LICENSE-2.0

  THIS CODE IS PROVIDED ON AN *AS IS* BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
  KIND, EITHER EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION ANY IMPLIED
  WARRANTIES OR CONDITIONS OF TITLE, FITNESS FOR A PARTICULAR PURPOSE,
  MERCHANTABLITY OR NON-INFRINGEMENT.

  See the Apache Version 2.0 License for specific language governing permissions
  and limitations under the License.
  ***************************************************************************** */
  /* global Reflect, Promise */

  var extendStatics$2 = function(d, b) {
      extendStatics$2 = Object.setPrototypeOf ||
          ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
          function (d, b) { for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p]; };
      return extendStatics$2(d, b);
  };

  function __extends$4(d, b) {
      extendStatics$2(d, b);
      function __() { this.constructor = d; }
      d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
  }

  /**
   * A virtual node which represents plain text content.
   *
   * #### Notes
   * User code will not typically create a `VirtualText` node directly.
   * Instead, the `h()` function will be used to create an element tree.
   */
  var VirtualText = /** @class */ (function () {
      /**
       * Construct a new virtual text node.
       *
       * @param content - The text content for the node.
       */
      function VirtualText(content) {
          /**
           * The type of the node.
           *
           * This value can be used as a type guard for discriminating the
           * `VirtualNode` union type.
           */
          this.type = 'text';
          this.content = content;
      }
      return VirtualText;
  }());
  /**
   * A virtual node which represents an HTML element.
   *
   * #### Notes
   * User code will not typically create a `VirtualElement` node directly.
   * Instead, the `h()` function will be used to create an element tree.
   */
  var VirtualElement = /** @class */ (function () {
      /**
       * Construct a new virtual element node.
       *
       * @param tag - The element tag name.
       *
       * @param attrs - The element attributes.
       *
       * @param children - The element children.
       *
       * @param renderer - An optional custom renderer for the element.
       */
      function VirtualElement(tag, attrs, children, renderer) {
          /**
           * The type of the node.
           *
           * This value can be used as a type guard for discriminating the
           * `VirtualNode` union type.
           */
          this.type = 'element';
          this.tag = tag;
          this.attrs = attrs;
          this.children = children;
          this.renderer = renderer;
      }
      return VirtualElement;
  }());
  /**
   * DEPRECATED - use VirtualElement with a defined renderer param instead.
   * This class is provided as a backwards compatibility shim
   *
   * A "pass thru" virtual node whose children are managed by a render and an
   * unrender callback. The intent of this flavor of virtual node is to make
   * it easy to blend other kinds of virtualdom (eg React) into Phosphor's
   * virtualdom.
   *
   * #### Notes
   * User code will not typically create a `VirtualElementPass` node directly.
   * Instead, the `hpass()` function will be used to create an element tree.
   */
  var VirtualElementPass = /** @class */ (function (_super) {
      __extends$4(VirtualElementPass, _super);
      /**
       * DEPRECATED - use VirtualElement with a defined renderer param instead
       *
       * Construct a new virtual element pass thru node.
       *
       * @param tag - the tag of the parent element of this node. Once the parent
       * element is rendered, it will be passed as an argument to
       * renderer.render
       *
       * @param attrs - attributes that will assigned to the
       * parent element
       *
       * @param renderer - an object with render and unrender
       * functions, each of which should take a single argument of type
       * HTMLElement and return nothing. If null, the parent element
       * will be rendered barren without any children.
       */
      function VirtualElementPass(tag, attrs, renderer) {
          return _super.call(this, tag, attrs, [], renderer || undefined) || this;
      }
      return VirtualElementPass;
  }(VirtualElement));
  function h(tag) {
      var attrs = {};
      var renderer;
      var children = [];
      for (var i = 1, n = arguments.length; i < n; ++i) {
          var arg = arguments[i];
          if (typeof arg === 'string') {
              children.push(new VirtualText(arg));
          }
          else if (arg instanceof VirtualText) {
              children.push(arg);
          }
          else if (arg instanceof VirtualElement) {
              children.push(arg);
          }
          else if (arg instanceof Array) {
              extend(children, arg);
          }
          else if ((i === 1 || i === 2) && arg && typeof arg === 'object') {
              if ("render" in arg) {
                  renderer = arg;
              }
              else {
                  attrs = arg;
              }
          }
      }
      return new VirtualElement(tag, attrs, children, renderer);
      function extend(array, values) {
          for (var _i = 0, values_1 = values; _i < values_1.length; _i++) {
              var child = values_1[_i];
              if (typeof child === 'string') {
                  array.push(new VirtualText(child));
              }
              else if (child instanceof VirtualText) {
                  array.push(child);
              }
              else if (child instanceof VirtualElement) {
                  array.push(child);
              }
          }
      }
  }
  /**
   * The namespace for the `h` function statics.
   */
  (function (h) {
      h.a = h.bind(undefined, 'a');
      h.abbr = h.bind(undefined, 'abbr');
      h.address = h.bind(undefined, 'address');
      h.area = h.bind(undefined, 'area');
      h.article = h.bind(undefined, 'article');
      h.aside = h.bind(undefined, 'aside');
      h.audio = h.bind(undefined, 'audio');
      h.b = h.bind(undefined, 'b');
      h.bdi = h.bind(undefined, 'bdi');
      h.bdo = h.bind(undefined, 'bdo');
      h.blockquote = h.bind(undefined, 'blockquote');
      h.br = h.bind(undefined, 'br');
      h.button = h.bind(undefined, 'button');
      h.canvas = h.bind(undefined, 'canvas');
      h.caption = h.bind(undefined, 'caption');
      h.cite = h.bind(undefined, 'cite');
      h.code = h.bind(undefined, 'code');
      h.col = h.bind(undefined, 'col');
      h.colgroup = h.bind(undefined, 'colgroup');
      h.data = h.bind(undefined, 'data');
      h.datalist = h.bind(undefined, 'datalist');
      h.dd = h.bind(undefined, 'dd');
      h.del = h.bind(undefined, 'del');
      h.dfn = h.bind(undefined, 'dfn');
      h.div = h.bind(undefined, 'div');
      h.dl = h.bind(undefined, 'dl');
      h.dt = h.bind(undefined, 'dt');
      h.em = h.bind(undefined, 'em');
      h.embed = h.bind(undefined, 'embed');
      h.fieldset = h.bind(undefined, 'fieldset');
      h.figcaption = h.bind(undefined, 'figcaption');
      h.figure = h.bind(undefined, 'figure');
      h.footer = h.bind(undefined, 'footer');
      h.form = h.bind(undefined, 'form');
      h.h1 = h.bind(undefined, 'h1');
      h.h2 = h.bind(undefined, 'h2');
      h.h3 = h.bind(undefined, 'h3');
      h.h4 = h.bind(undefined, 'h4');
      h.h5 = h.bind(undefined, 'h5');
      h.h6 = h.bind(undefined, 'h6');
      h.header = h.bind(undefined, 'header');
      h.hr = h.bind(undefined, 'hr');
      h.i = h.bind(undefined, 'i');
      h.iframe = h.bind(undefined, 'iframe');
      h.img = h.bind(undefined, 'img');
      h.input = h.bind(undefined, 'input');
      h.ins = h.bind(undefined, 'ins');
      h.kbd = h.bind(undefined, 'kbd');
      h.label = h.bind(undefined, 'label');
      h.legend = h.bind(undefined, 'legend');
      h.li = h.bind(undefined, 'li');
      h.main = h.bind(undefined, 'main');
      h.map = h.bind(undefined, 'map');
      h.mark = h.bind(undefined, 'mark');
      h.meter = h.bind(undefined, 'meter');
      h.nav = h.bind(undefined, 'nav');
      h.noscript = h.bind(undefined, 'noscript');
      h.object = h.bind(undefined, 'object');
      h.ol = h.bind(undefined, 'ol');
      h.optgroup = h.bind(undefined, 'optgroup');
      h.option = h.bind(undefined, 'option');
      h.output = h.bind(undefined, 'output');
      h.p = h.bind(undefined, 'p');
      h.param = h.bind(undefined, 'param');
      h.pre = h.bind(undefined, 'pre');
      h.progress = h.bind(undefined, 'progress');
      h.q = h.bind(undefined, 'q');
      h.rp = h.bind(undefined, 'rp');
      h.rt = h.bind(undefined, 'rt');
      h.ruby = h.bind(undefined, 'ruby');
      h.s = h.bind(undefined, 's');
      h.samp = h.bind(undefined, 'samp');
      h.section = h.bind(undefined, 'section');
      h.select = h.bind(undefined, 'select');
      h.small = h.bind(undefined, 'small');
      h.source = h.bind(undefined, 'source');
      h.span = h.bind(undefined, 'span');
      h.strong = h.bind(undefined, 'strong');
      h.sub = h.bind(undefined, 'sub');
      h.summary = h.bind(undefined, 'summary');
      h.sup = h.bind(undefined, 'sup');
      h.table = h.bind(undefined, 'table');
      h.tbody = h.bind(undefined, 'tbody');
      h.td = h.bind(undefined, 'td');
      h.textarea = h.bind(undefined, 'textarea');
      h.tfoot = h.bind(undefined, 'tfoot');
      h.th = h.bind(undefined, 'th');
      h.thead = h.bind(undefined, 'thead');
      h.time = h.bind(undefined, 'time');
      h.title = h.bind(undefined, 'title');
      h.tr = h.bind(undefined, 'tr');
      h.track = h.bind(undefined, 'track');
      h.u = h.bind(undefined, 'u');
      h.ul = h.bind(undefined, 'ul');
      h.var_ = h.bind(undefined, 'var');
      h.video = h.bind(undefined, 'video');
      h.wbr = h.bind(undefined, 'wbr');
  })(h || (h = {}));
  /**
   * The namespace for the virtual DOM rendering functions.
   */
  var VirtualDOM;
  (function (VirtualDOM) {
      function realize(node) {
          return Private$7.createDOMNode(node);
      }
      VirtualDOM.realize = realize;
      /**
       * Render virtual DOM content into a host element.
       *
       * @param content - The virtual DOM content to render.
       *
       * @param host - The host element for the rendered content.
       *
       * #### Notes
       * This renders the delta from the previous rendering. It assumes that
       * the content of the host element is not manipulated by external code.
       *
       * Providing `null` content will clear the rendering.
       *
       * Externally modifying the provided content or the host element will
       * result in undefined rendering behavior.
       */
      function render(content, host) {
          var oldContent = Private$7.hostMap.get(host) || [];
          var newContent = Private$7.asContentArray(content);
          Private$7.hostMap.set(host, newContent);
          Private$7.updateContent(host, oldContent, newContent);
      }
      VirtualDOM.render = render;
  })(VirtualDOM || (VirtualDOM = {}));
  /**
   * The namespace for the module implementation details.
   */
  var Private$7;
  (function (Private) {
      /**
       * A weak mapping of host element to virtual DOM content.
       */
      Private.hostMap = new WeakMap();
      /**
       * Cast a content value to a content array.
       */
      function asContentArray(value) {
          if (!value) {
              return [];
          }
          if (value instanceof Array) {
              return value;
          }
          return [value];
      }
      Private.asContentArray = asContentArray;
      function createDOMNode(node) {
          var host = arguments[1] || null;
          var before = arguments[2] || null;
          if (host) {
              host.insertBefore(createDOMNode(node), before);
          }
          else {
              // Create a text node for a virtual text node.
              if (node.type === 'text') {
                  return document.createTextNode(node.content);
              }
              // Create the HTML element with the specified tag.
              host = document.createElement(node.tag);
              // Add the attributes for the new element.
              addAttrs(host, node.attrs);
              if (node.renderer) {
                  node.renderer.render(host, { attrs: node.attrs, children: node.children });
                  return host;
              }
              // Recursively populate the element with child content.
              for (var i = 0, n = node.children.length; i < n; ++i) {
                  createDOMNode(node.children[i], host);
              }
          }
          return host;
      }
      Private.createDOMNode = createDOMNode;
      /**
       * Update a host element with the delta of the virtual content.
       *
       * This is the core "diff" algorithm. There is no explicit "patch"
       * phase. The host is patched at each step as the diff progresses.
       */
      function updateContent(host, oldContent, newContent) {
          // Bail early if the content is identical.
          if (oldContent === newContent) {
              return;
          }
          // Collect the old keyed elems into a mapping.
          var oldKeyed = collectKeys(host, oldContent);
          // Create a copy of the old content which can be modified in-place.
          var oldCopy = oldContent.slice();
          // Update the host with the new content. The diff always proceeds
          // forward and never modifies a previously visited index. The old
          // copy array is modified in-place to reflect the changes made to
          // the host children. This causes the stale nodes to be pushed to
          // the end of the host node and removed at the end of the loop.
          var currElem = host.firstChild;
          var newCount = newContent.length;
          for (var i = 0; i < newCount; ++i) {
              // If the old content is exhausted, create a new node.
              if (i >= oldCopy.length) {
                  createDOMNode(newContent[i], host);
                  continue;
              }
              // Lookup the old and new virtual nodes.
              var oldVNode = oldCopy[i];
              var newVNode = newContent[i];
              // If both elements are identical, there is nothing to do.
              if (oldVNode === newVNode) {
                  currElem = currElem.nextSibling;
                  continue;
              }
              // Handle the simplest case of in-place text update first.
              if (oldVNode.type === 'text' && newVNode.type === 'text') {
                  currElem.textContent = newVNode.content;
                  currElem = currElem.nextSibling;
                  continue;
              }
              // If the old or new node is a text node, the other node is now
              // known to be an element node, so create and insert a new node.
              if (oldVNode.type === 'text' || newVNode.type === 'text') {
                  ArrayExt.insert(oldCopy, i, newVNode);
                  createDOMNode(newVNode, host, currElem);
                  continue;
              }
              // If the old XOR new node has a custom renderer,
              // create and insert a new node.
              if (!(oldVNode.renderer) != !(newVNode.renderer)) {
                  ArrayExt.insert(oldCopy, i, newVNode);
                  createDOMNode(newVNode, host, currElem);
                  continue;
              }
              // At this point, both nodes are known to be element nodes.
              // If the new elem is keyed, move an old keyed elem to the proper
              // location before proceeding with the diff. The search can start
              // at the current index, since the unmatched old keyed elems are
              // pushed forward in the old copy array.
              var newKey = newVNode.attrs.key;
              if (newKey && newKey in oldKeyed) {
                  var pair = oldKeyed[newKey];
                  if (pair.vNode !== oldVNode) {
                      ArrayExt.move(oldCopy, oldCopy.indexOf(pair.vNode, i + 1), i);
                      host.insertBefore(pair.element, currElem);
                      oldVNode = pair.vNode;
                      currElem = pair.element;
                  }
              }
              // If both elements are identical, there is nothing to do.
              if (oldVNode === newVNode) {
                  currElem = currElem.nextSibling;
                  continue;
              }
              // If the old elem is keyed and does not match the new elem key,
              // create a new node. This is necessary since the old keyed elem
              // may be matched at a later point in the diff.
              var oldKey = oldVNode.attrs.key;
              if (oldKey && oldKey !== newKey) {
                  ArrayExt.insert(oldCopy, i, newVNode);
                  createDOMNode(newVNode, host, currElem);
                  continue;
              }
              // If the tags are different, create a new node.
              if (oldVNode.tag !== newVNode.tag) {
                  ArrayExt.insert(oldCopy, i, newVNode);
                  createDOMNode(newVNode, host, currElem);
                  continue;
              }
              // At this point, the element can be updated in-place.
              // Update the element attributes.
              updateAttrs(currElem, oldVNode.attrs, newVNode.attrs);
              // Update the element content.
              if (newVNode.renderer) {
                  newVNode.renderer.render(currElem, { attrs: newVNode.attrs, children: newVNode.children });
              }
              else {
                  updateContent(currElem, oldVNode.children, newVNode.children);
              }
              // Step to the next sibling element.
              currElem = currElem.nextSibling;
          }
          // Cleanup stale DOM
          removeContent(host, oldCopy, newCount, true);
      }
      Private.updateContent = updateContent;
      /**
       * Handle cleanup of stale vdom and its associated DOM. The host node is
       * traversed recursively (in depth-first order), and any explicit cleanup
       * required by a child node is carried out when it is visited (eg if a node
       * has a custom renderer, the renderer.unrender function will be called).
       * Once the subtree beneath each child of host has been completely visited,
       * that child will be removed via a call to host.removeChild.
       */
      function removeContent(host, oldContent, newCount, _sentinel) {
          // Dispose of the old nodes pushed to the end of the host.
          for (var i = oldContent.length - 1; i >= newCount; --i) {
              var oldNode = oldContent[i];
              var child = (_sentinel ? host.lastChild : host.childNodes[i]);
              // recursively clean up host children
              if (oldNode.type === 'text') ;
              else if (oldNode.renderer && oldNode.renderer.unrender) {
                  oldNode.renderer.unrender(child, { attrs: oldNode.attrs, children: oldNode.children });
              }
              else {
                  removeContent(child, oldNode.children, 0, false);
              }
              if (_sentinel) {
                  host.removeChild(child);
              }
          }
      }
      /**
       * A set of special-cased attribute names.
       */
      var specialAttrs = {
          'key': true,
          'className': true,
          'htmlFor': true,
          'dataset': true,
          'style': true,
      };
      /**
       * Add element attributes to a newly created HTML element.
       */
      function addAttrs(element, attrs) {
          // Add the inline event listeners and node attributes.
          for (var name_1 in attrs) {
              if (name_1 in specialAttrs) {
                  continue;
              }
              if (name_1.substr(0, 2) === 'on') {
                  element[name_1] = attrs[name_1];
              }
              else {
                  element.setAttribute(name_1, attrs[name_1]);
              }
          }
          // Add the element `class` attribute.
          if (attrs.className !== undefined) {
              element.setAttribute('class', attrs.className);
          }
          // Add the element `for` attribute.
          if (attrs.htmlFor !== undefined) {
              element.setAttribute('for', attrs.htmlFor);
          }
          // Add the dataset values.
          if (attrs.dataset) {
              addDataset(element, attrs.dataset);
          }
          // Add the inline styles.
          if (attrs.style) {
              addStyle(element, attrs.style);
          }
      }
      /**
       * Update the element attributes of an HTML element.
       */
      function updateAttrs(element, oldAttrs, newAttrs) {
          // Do nothing if the attrs are the same object.
          if (oldAttrs === newAttrs) {
              return;
          }
          // Setup the strongly typed loop variable.
          var name;
          // Remove attributes and listeners which no longer exist.
          for (name in oldAttrs) {
              if (name in specialAttrs || name in newAttrs) {
                  continue;
              }
              if (name.substr(0, 2) === 'on') {
                  element[name] = null;
              }
              else {
                  element.removeAttribute(name);
              }
          }
          // Add and update new and existing attributes and listeners.
          for (name in newAttrs) {
              if (name in specialAttrs || oldAttrs[name] === newAttrs[name]) {
                  continue;
              }
              if (name.substr(0, 2) === 'on') {
                  element[name] = newAttrs[name];
              }
              else {
                  element.setAttribute(name, newAttrs[name]);
              }
          }
          // Update the element `class` attribute.
          if (oldAttrs.className !== newAttrs.className) {
              if (newAttrs.className !== undefined) {
                  element.setAttribute('class', newAttrs.className);
              }
              else {
                  element.removeAttribute('class');
              }
          }
          // Add the element `for` attribute.
          if (oldAttrs.htmlFor !== newAttrs.htmlFor) {
              if (newAttrs.htmlFor !== undefined) {
                  element.setAttribute('for', newAttrs.htmlFor);
              }
              else {
                  element.removeAttribute('for');
              }
          }
          // Update the dataset values.
          if (oldAttrs.dataset !== newAttrs.dataset) {
              updateDataset(element, oldAttrs.dataset || {}, newAttrs.dataset || {});
          }
          // Update the inline styles.
          if (oldAttrs.style !== newAttrs.style) {
              updateStyle(element, oldAttrs.style || {}, newAttrs.style || {});
          }
      }
      /**
       * Add dataset values to a newly created HTML element.
       */
      function addDataset(element, dataset) {
          for (var name_2 in dataset) {
              element.setAttribute("data-" + name_2, dataset[name_2]);
          }
      }
      /**
       * Update the dataset values of an HTML element.
       */
      function updateDataset(element, oldDataset, newDataset) {
          for (var name_3 in oldDataset) {
              if (!(name_3 in newDataset)) {
                  element.removeAttribute("data-" + name_3);
              }
          }
          for (var name_4 in newDataset) {
              if (oldDataset[name_4] !== newDataset[name_4]) {
                  element.setAttribute("data-" + name_4, newDataset[name_4]);
              }
          }
      }
      /**
       * Add inline style values to a newly created HTML element.
       */
      function addStyle(element, style) {
          var elemStyle = element.style;
          var name;
          for (name in style) {
              elemStyle[name] = style[name];
          }
      }
      /**
       * Update the inline style values of an HTML element.
       */
      function updateStyle(element, oldStyle, newStyle) {
          var elemStyle = element.style;
          var name;
          for (name in oldStyle) {
              if (!(name in newStyle)) {
                  elemStyle[name] = '';
              }
          }
          for (name in newStyle) {
              if (oldStyle[name] !== newStyle[name]) {
                  elemStyle[name] = newStyle[name];
              }
          }
      }
      /**
       * Collect a mapping of keyed elements for the host content.
       */
      function collectKeys(host, content) {
          var node = host.firstChild;
          var keyMap = Object.create(null);
          for (var _i = 0, content_1 = content; _i < content_1.length; _i++) {
              var vNode = content_1[_i];
              if (vNode.type === 'element' && vNode.attrs.key) {
                  keyMap[vNode.attrs.key] = { vNode: vNode, element: node };
              }
              node = node.nextSibling;
          }
          return keyMap;
      }
  })(Private$7 || (Private$7 = {}));

  /**
   * An object which manages a drag-drop operation.
   *
   * A drag object dispatches four different events to drop targets:
   *
   * - `'lm-dragenter'` - Dispatched when the mouse enters the target
   *   element. This event must be canceled in order to receive any
   *   of the other events.
   *
   * - `'lm-dragover'` - Dispatched when the mouse moves over the drop
   *   target. It must cancel the event and set the `dropAction` to one
   *   of the supported actions in order to receive drop events.
   *
   * - `'lm-dragleave'` - Dispatched when the mouse leaves the target
   *   element. This includes moving the mouse into child elements.
   *
   * - `'lm-drop'`- Dispatched when the mouse is released over the target
   *   element when the target indicates an appropriate drop action. If
   *   the event is canceled, the indicated drop action is returned to
   *   the initiator through the resolved promise.
   *
   * A drag operation can be terminated at any time by pressing `Escape`
   * or by disposing the drag object.
   *
   * A drag object has the ability to automatically scroll a scrollable
   * element when the mouse is hovered near one of its edges. To enable
   * this, add the `data-lm-dragscroll` attribute to any element which
   * the drag object should consider for scrolling.
   *
   * #### Notes
   * This class is designed to be used when dragging and dropping custom
   * data *within* a single application. It is *not* a replacement for
   * the native drag-drop API. Instead, it provides an API which allows
   * drag operations to be initiated programmatically and enables the
   * transfer of arbitrary non-string objects; features which are not
   * possible with the native drag-drop API.
   */
  var Drag = /** @class */ (function () {
      /**
       * Construct a new drag object.
       *
       * @param options - The options for initializing the drag.
       */
      function Drag(options) {
          var _this = this;
          /**
           * The scroll loop handler function.
           */
          this._onScrollFrame = function () {
              // Bail early if there is no scroll target.
              if (!_this._scrollTarget) {
                  return;
              }
              // Unpack the scroll target.
              var _a = _this._scrollTarget, element = _a.element, edge = _a.edge, distance = _a.distance;
              // Calculate the scroll delta using nonlinear acceleration.
              var d = Private$8.SCROLL_EDGE_SIZE - distance;
              var f = Math.pow(d / Private$8.SCROLL_EDGE_SIZE, 2);
              var s = Math.max(1, Math.round(f * Private$8.SCROLL_EDGE_SIZE));
              // Scroll the element in the specified direction.
              switch (edge) {
                  case 'top':
                      element.scrollTop -= s;
                      break;
                  case 'left':
                      element.scrollLeft -= s;
                      break;
                  case 'right':
                      element.scrollLeft += s;
                      break;
                  case 'bottom':
                      element.scrollTop += s;
                      break;
              }
              // Request the next cycle of the scroll loop.
              requestAnimationFrame(_this._onScrollFrame);
          };
          this._disposed = false;
          this._dropAction = 'none';
          this._override = null;
          this._currentTarget = null;
          this._currentElement = null;
          this._promise = null;
          this._scrollTarget = null;
          this._resolve = null;
          this.mimeData = options.mimeData;
          this.dragImage = options.dragImage || null;
          this.proposedAction = options.proposedAction || 'copy';
          this.supportedActions = options.supportedActions || 'all';
          this.source = options.source || null;
      }
      /**
       * Dispose of the resources held by the drag object.
       *
       * #### Notes
       * This will cancel the drag operation if it is active.
       */
      Drag.prototype.dispose = function () {
          // Do nothing if the drag object is already disposed.
          if (this._disposed) {
              return;
          }
          this._disposed = true;
          // If there is a current target, dispatch a drag leave event.
          if (this._currentTarget) {
              var event_1 = Private$8.createMouseEvent('mouseup', -1, -1);
              Private$8.dispatchDragLeave(this, this._currentTarget, null, event_1);
          }
          // Finalize the drag object with `'none'`.
          this._finalize('none');
      };
      Object.defineProperty(Drag.prototype, "isDisposed", {
          /**
           * Test whether the drag object is disposed.
           */
          get: function () {
              return this._disposed;
          },
          enumerable: true,
          configurable: true
      });
      /**
       * Start the drag operation at the specified client position.
       *
       * @param clientX - The client X position for the drag start.
       *
       * @param clientY - The client Y position for the drag start.
       *
       * @returns A promise which resolves to the result of the drag.
       *
       * #### Notes
       * If the drag has already been started, the promise created by the
       * first call to `start` is returned.
       *
       * If the drag operation has ended, or if the drag object has been
       * disposed, the returned promise will resolve to `'none'`.
       *
       * The drag object will be automatically disposed when drag operation
       * completes. This means `Drag` objects are for single-use only.
       *
       * This method assumes the left mouse button is already held down.
       */
      Drag.prototype.start = function (clientX, clientY) {
          var _this = this;
          // If the drag object is already disposed, resolve to `None`.
          if (this._disposed) {
              return Promise.resolve('none');
          }
          // If the drag has already been started, return the promise.
          if (this._promise) {
              return this._promise;
          }
          // Install the document listeners for the drag object.
          this._addListeners();
          // Attach the drag image at the specified client position.
          this._attachDragImage(clientX, clientY);
          // Create the promise which will be resolved on completion.
          this._promise = new Promise(function (resolve, reject) {
              _this._resolve = resolve;
          });
          // Trigger a fake move event to kick off the drag operation.
          var event = Private$8.createMouseEvent('mousemove', clientX, clientY);
          document.dispatchEvent(event);
          // Return the pending promise for the drag operation.
          return this._promise;
      };
      /**
       * Handle the DOM events for the drag operation.
       *
       * @param event - The DOM event sent to the drag object.
       *
       * #### Notes
       * This method implements the DOM `EventListener` interface and is
       * called in response to events on the document. It should not be
       * called directly by user code.
       */
      Drag.prototype.handleEvent = function (event) {
          switch (event.type) {
              case 'mousemove':
                  this._evtMouseMove(event);
                  break;
              case 'mouseup':
                  this._evtMouseUp(event);
                  break;
              case 'keydown':
                  this._evtKeyDown(event);
                  break;
              default:
                  // Stop all other events during drag-drop.
                  event.preventDefault();
                  event.stopPropagation();
                  break;
          }
      };
      /**
       * Handle the `'mousemove'` event for the drag object.
       */
      Drag.prototype._evtMouseMove = function (event) {
          // Stop all input events during drag-drop.
          event.preventDefault();
          event.stopPropagation();
          // Update the current target node and dispatch enter/leave events.
          this._updateCurrentTarget(event);
          // Update the drag scroll element.
          this._updateDragScroll(event);
          // Move the drag image to the specified client position. This is
          // performed *after* dispatching to prevent unnecessary reflows.
          this._moveDragImage(event.clientX, event.clientY);
      };
      /**
       * Handle the `'mouseup'` event for the drag object.
       */
      Drag.prototype._evtMouseUp = function (event) {
          // Stop all input events during drag-drop.
          event.preventDefault();
          event.stopPropagation();
          // Do nothing if the left button is not released.
          if (event.button !== 0) {
              return;
          }
          // Update the current target node and dispatch enter/leave events.
          // This prevents a subtle issue where the DOM mutates under the
          // cursor after the last move event but before the drop event.
          this._updateCurrentTarget(event);
          // If there is no current target, finalize with `'none'`.
          if (!this._currentTarget) {
              this._finalize('none');
              return;
          }
          // If the last drop action was `'none'`, dispatch a leave event
          // to the current target and finalize the drag with `'none'`.
          if (this._dropAction === 'none') {
              Private$8.dispatchDragLeave(this, this._currentTarget, null, event);
              this._finalize('none');
              return;
          }
          // Dispatch the drop event at the current target and finalize
          // with the resulting drop action.
          var action = Private$8.dispatchDrop(this, this._currentTarget, event);
          this._finalize(action);
      };
      /**
       * Handle the `'keydown'` event for the drag object.
       */
      Drag.prototype._evtKeyDown = function (event) {
          // Stop all input events during drag-drop.
          event.preventDefault();
          event.stopPropagation();
          // Cancel the drag if `Escape` is pressed.
          if (event.keyCode === 27) {
              this.dispose();
          }
      };
      /**
       * Add the document event listeners for the drag object.
       */
      Drag.prototype._addListeners = function () {
          document.addEventListener('mousedown', this, true);
          document.addEventListener('mousemove', this, true);
          document.addEventListener('mouseup', this, true);
          document.addEventListener('mouseenter', this, true);
          document.addEventListener('mouseleave', this, true);
          document.addEventListener('mouseover', this, true);
          document.addEventListener('mouseout', this, true);
          document.addEventListener('keydown', this, true);
          document.addEventListener('keyup', this, true);
          document.addEventListener('keypress', this, true);
          document.addEventListener('contextmenu', this, true);
      };
      /**
       * Remove the document event listeners for the drag object.
       */
      Drag.prototype._removeListeners = function () {
          document.removeEventListener('mousedown', this, true);
          document.removeEventListener('mousemove', this, true);
          document.removeEventListener('mouseup', this, true);
          document.removeEventListener('mouseenter', this, true);
          document.removeEventListener('mouseleave', this, true);
          document.removeEventListener('mouseover', this, true);
          document.removeEventListener('mouseout', this, true);
          document.removeEventListener('keydown', this, true);
          document.removeEventListener('keyup', this, true);
          document.removeEventListener('keypress', this, true);
          document.removeEventListener('contextmenu', this, true);
      };
      /**
       * Update the drag scroll element under the mouse.
       */
      Drag.prototype._updateDragScroll = function (event) {
          // Find the scroll target under the mouse.
          var target = Private$8.findScrollTarget(event);
          // Bail if there is nothing to scroll.
          if (!this._scrollTarget && !target) {
              return;
          }
          // Start the scroll loop if needed.
          if (!this._scrollTarget) {
              setTimeout(this._onScrollFrame, 500);
          }
          // Update the scroll target.
          this._scrollTarget = target;
      };
      /**
       * Update the current target node using the given mouse event.
       */
      Drag.prototype._updateCurrentTarget = function (event) {
          // Fetch common local state.
          var prevTarget = this._currentTarget;
          var currTarget = this._currentTarget;
          var prevElem = this._currentElement;
          // Find the current indicated element at the given position.
          var currElem = document.elementFromPoint(event.clientX, event.clientY);
          // Update the current element reference.
          this._currentElement = currElem;
          // If the indicated element changes from the previous iteration,
          // and is different from the current target, dispatch the exit
          // event to the target.
          if (currElem !== prevElem && currElem !== currTarget) {
              Private$8.dispatchDragExit(this, currTarget, currElem, event);
          }
          // If the indicated element changes from the previous iteration,
          // and is different from the current target, dispatch the enter
          // event and compute the new target element.
          if (currElem !== prevElem && currElem !== currTarget) {
              currTarget = Private$8.dispatchDragEnter(this, currElem, currTarget, event);
          }
          // If the current target element has changed, update the current
          // target reference and dispatch the leave event to the old target.
          if (currTarget !== prevTarget) {
              this._currentTarget = currTarget;
              Private$8.dispatchDragLeave(this, prevTarget, currTarget, event);
          }
          // Dispatch the drag over event and update the drop action.
          var action = Private$8.dispatchDragOver(this, currTarget, event);
          this._setDropAction(action);
      };
      /**
       * Attach the drag image element at the specified location.
       *
       * This is a no-op if there is no drag image element.
       */
      Drag.prototype._attachDragImage = function (clientX, clientY) {
          if (!this.dragImage) {
              return;
          }
          this.dragImage.classList.add('lm-mod-drag-image');
          /* <DEPRECATED> */
          this.dragImage.classList.add('p-mod-drag-image');
          /* </DEPRECATED> */
          var style = this.dragImage.style;
          style.pointerEvents = 'none';
          style.position = 'fixed';
          style.top = clientY + "px";
          style.left = clientX + "px";
          document.body.appendChild(this.dragImage);
      };
      /**
       * Move the drag image element to the specified location.
       *
       * This is a no-op if there is no drag image element.
       */
      Drag.prototype._moveDragImage = function (clientX, clientY) {
          if (!this.dragImage) {
              return;
          }
          var style = this.dragImage.style;
          style.top = clientY + "px";
          style.left = clientX + "px";
      };
      /**
       * Detach the drag image element from the DOM.
       *
       * This is a no-op if there is no drag image element.
       */
      Drag.prototype._detachDragImage = function () {
          if (!this.dragImage) {
              return;
          }
          var parent = this.dragImage.parentNode;
          if (!parent) {
              return;
          }
          parent.removeChild(this.dragImage);
      };
      /**
       * Set the internal drop action state and update the drag cursor.
       */
      Drag.prototype._setDropAction = function (action) {
          action = Private$8.validateAction(action, this.supportedActions);
          if (this._override && this._dropAction === action) {
              return;
          }
          switch (action) {
              case 'none':
                  this._dropAction = action;
                  this._override = Drag.overrideCursor('no-drop');
                  break;
              case 'copy':
                  this._dropAction = action;
                  this._override = Drag.overrideCursor('copy');
                  break;
              case 'link':
                  this._dropAction = action;
                  this._override = Drag.overrideCursor('alias');
                  break;
              case 'move':
                  this._dropAction = action;
                  this._override = Drag.overrideCursor('move');
                  break;
          }
      };
      /**
       * Finalize the drag operation and resolve the drag promise.
       */
      Drag.prototype._finalize = function (action) {
          // Store the resolve function as a temp variable.
          var resolve = this._resolve;
          // Remove the document event listeners.
          this._removeListeners();
          // Detach the drag image.
          this._detachDragImage();
          // Dispose of the cursor override.
          if (this._override) {
              this._override.dispose();
              this._override = null;
          }
          // Clear the mime data.
          this.mimeData.clear();
          // Clear the rest of the internal drag state.
          this._disposed = true;
          this._dropAction = 'none';
          this._currentTarget = null;
          this._currentElement = null;
          this._scrollTarget = null;
          this._promise = null;
          this._resolve = null;
          // Finally, resolve the promise to the given drop action.
          if (resolve) {
              resolve(action);
          }
      };
      return Drag;
  }());
  /**
   * The namespace for the `Drag` class statics.
   */
  (function (Drag) {
      /**
       * Override the cursor icon for the entire document.
       *
       * @param cursor - The string representing the cursor style.
       *
       * @returns A disposable which will clear the override when disposed.
       *
       * #### Notes
       * The most recent call to `overrideCursor` takes precedence.
       * Disposing an old override has no effect on the current override.
       *
       * This utility function is used by the `Drag` class to override the
       * mouse cursor during a drag-drop operation, but it can also be used
       * by other classes to fix the cursor icon during normal mouse drags.
       *
       * #### Example
       * ```typescript
       * import { Drag } from '@lumino/dragdrop';
       *
       * // Force the cursor to be 'wait' for the entire document.
       * let override = Drag.overrideCursor('wait');
       *
       * // Clear the override by disposing the return value.
       * override.dispose();
       * ```
       */
      function overrideCursor(cursor) {
          var id = ++overrideCursorID;
          document.body.style.cursor = cursor;
          document.body.classList.add('lm-mod-override-cursor');
          /* <DEPRECATED> */
          document.body.classList.add('p-mod-override-cursor');
          /* </DEPRECATED> */
          return new DisposableDelegate(function () {
              if (id === overrideCursorID) {
                  document.body.style.cursor = '';
                  document.body.classList.remove('lm-mod-override-cursor');
                  /* <DEPRECATED> */
                  document.body.classList.remove('p-mod-override-cursor');
                  /* </DEPRECATED> */
              }
          });
      }
      Drag.overrideCursor = overrideCursor;
      /**
       * The internal id for the active cursor override.
       */
      var overrideCursorID = 0;
  })(Drag || (Drag = {}));
  /**
   * The namespace for the module implementation details.
   */
  var Private$8;
  (function (Private) {
      /**
       * The size of a drag scroll edge, in pixels.
       */
      Private.SCROLL_EDGE_SIZE = 20;
      /**
       * Validate the given action is one of the supported actions.
       *
       * Returns the given action or `'none'` if the action is unsupported.
       */
      function validateAction(action, supported) {
          return (actionTable[action] & supportedTable[supported]) ? action : 'none';
      }
      Private.validateAction = validateAction;
      /**
       * Create a left mouse event at the given position.
       *
       * @param type - The event type for the mouse event.
       *
       * @param clientX - The client X position.
       *
       * @param clientY - The client Y position.
       *
       * @returns A newly created and initialized mouse event.
       */
      function createMouseEvent(type, clientX, clientY) {
          var event = document.createEvent('MouseEvent');
          event.initMouseEvent(type, true, true, window, 0, 0, 0, clientX, clientY, false, false, false, false, 0, null);
          return event;
      }
      Private.createMouseEvent = createMouseEvent;
      /**
       * Find the drag scroll target under the mouse, if any.
       */
      function findScrollTarget(event) {
          // Look up the client mouse position.
          var x = event.clientX;
          var y = event.clientY;
          // Get the element under the mouse.
          var element = document.elementFromPoint(x, y);
          // Search for a scrollable target based on the mouse position.
          // The null assert in third clause of for-loop is required due to:
          // https://github.com/Microsoft/TypeScript/issues/14143
          for (; element; element = element.parentElement) {
              // Ignore elements which are not marked as scrollable.
              var scrollable = element.hasAttribute('data-lm-dragscroll');
              /* <DEPRECATED> */
              scrollable = scrollable || element.hasAttribute('data-p-dragscroll');
              /* </DEPRECATED> */
              if (!scrollable) {
                  continue;
              }
              // Set up the coordinate offsets for the element.
              var offsetX = 0;
              var offsetY = 0;
              if (element === document.body) {
                  offsetX = window.pageXOffset;
                  offsetY = window.pageYOffset;
              }
              // Get the element bounds in viewport coordinates.
              var r = element.getBoundingClientRect();
              var top_1 = r.top + offsetY;
              var left = r.left + offsetX;
              var right = left + r.width;
              var bottom = top_1 + r.height;
              // Skip the element if it's not under the mouse.
              if (x < left || x >= right || y < top_1 || y >= bottom) {
                  continue;
              }
              // Compute the distance to each edge.
              var dl = x - left + 1;
              var dt = y - top_1 + 1;
              var dr = right - x;
              var db = bottom - y;
              // Find the smallest of the edge distances.
              var distance = Math.min(dl, dt, dr, db);
              // Skip the element if the mouse is not within a scroll edge.
              if (distance > Private.SCROLL_EDGE_SIZE) {
                  continue;
              }
              // Set up the edge result variable.
              var edge = void 0;
              // Find the edge for the computed distance.
              switch (distance) {
                  case db:
                      edge = 'bottom';
                      break;
                  case dt:
                      edge = 'top';
                      break;
                  case dr:
                      edge = 'right';
                      break;
                  case dl:
                      edge = 'left';
                      break;
                  default:
                      throw 'unreachable';
              }
              // Compute how much the element can scroll in width and height.
              var dsw = element.scrollWidth - element.clientWidth;
              var dsh = element.scrollHeight - element.clientHeight;
              // Determine if the element should be scrolled for the edge.
              var shouldScroll = void 0;
              switch (edge) {
                  case 'top':
                      shouldScroll = dsh > 0 && element.scrollTop > 0;
                      break;
                  case 'left':
                      shouldScroll = dsw > 0 && element.scrollLeft > 0;
                      break;
                  case 'right':
                      shouldScroll = dsw > 0 && element.scrollLeft < dsw;
                      break;
                  case 'bottom':
                      shouldScroll = dsh > 0 && element.scrollTop < dsh;
                      break;
                  default:
                      throw 'unreachable';
              }
              // Skip the element if it should not be scrolled.
              if (!shouldScroll) {
                  continue;
              }
              // Return the drag scroll target.
              return { element: element, edge: edge, distance: distance };
          }
          // No drag scroll target was found.
          return null;
      }
      Private.findScrollTarget = findScrollTarget;
      /**
       * Dispatch a drag enter event to the indicated element.
       *
       * @param drag - The drag object associated with the action.
       *
       * @param currElem - The currently indicated element, or `null`. This
       *   is the "immediate user selection" from the whatwg spec.
       *
       * @param currTarget - The current drag target element, or `null`. This
       *   is the "current target element" from the whatwg spec.
       *
       * @param event - The mouse event related to the action.
       *
       * @returns The element to use as the current drag target. This is the
       *   "current target element" from the whatwg spec, and may be `null`.
       *
       * #### Notes
       * This largely implements the drag enter portion of the whatwg spec:
       * https://html.spec.whatwg.org/multipage/interaction.html#drag-and-drop-processing-model
       */
      function dispatchDragEnter(drag, currElem, currTarget, event) {
          // If the current element is null, return null as the new target.
          if (!currElem) {
              return null;
          }
          // Dispatch a drag enter event to the current element.
          var dragEvent = createDragEvent('lm-dragenter', drag, event, currTarget);
          var canceled = !currElem.dispatchEvent(dragEvent);
          // If the event was canceled, use the current element as the new target.
          if (canceled) {
              return currElem;
          }
          /* <DEPRECATED> */
          dragEvent = createDragEvent('p-dragenter', drag, event, currTarget);
          canceled = !currElem.dispatchEvent(dragEvent);
          if (canceled) {
              return currElem;
          }
          /* </DEPRECATED> */
          // If the current element is the document body, keep the original target.
          if (currElem === document.body) {
              return currTarget;
          }
          // Dispatch a drag enter event on the document body.
          dragEvent = createDragEvent('lm-dragenter', drag, event, currTarget);
          document.body.dispatchEvent(dragEvent);
          /* <DEPRECATED> */
          dragEvent = createDragEvent('p-dragenter', drag, event, currTarget);
          document.body.dispatchEvent(dragEvent);
          /* </DEPRECATED> */
          // Ignore the event cancellation, and use the body as the new target.
          return document.body;
      }
      Private.dispatchDragEnter = dispatchDragEnter;
      /**
       * Dispatch a drag exit event to the indicated element.
       *
       * @param drag - The drag object associated with the action.
       *
       * @param prevTarget - The previous target element, or `null`. This
       *   is the previous "current target element" from the whatwg spec.
       *
       * @param currTarget - The current drag target element, or `null`. This
       *   is the "current target element" from the whatwg spec.
       *
       * @param event - The mouse event related to the action.
       *
       * #### Notes
       * This largely implements the drag exit portion of the whatwg spec:
       * https://html.spec.whatwg.org/multipage/interaction.html#drag-and-drop-processing-model
       */
      function dispatchDragExit(drag, prevTarget, currTarget, event) {
          // If the previous target is null, do nothing.
          if (!prevTarget) {
              return;
          }
          // Dispatch the drag exit event to the previous target.
          var dragEvent = createDragEvent('lm-dragexit', drag, event, currTarget);
          prevTarget.dispatchEvent(dragEvent);
          /* <DEPRECATED> */
          dragEvent = createDragEvent('p-dragexit', drag, event, currTarget);
          prevTarget.dispatchEvent(dragEvent);
          /* </DEPRECATED> */
      }
      Private.dispatchDragExit = dispatchDragExit;
      /**
       * Dispatch a drag leave event to the indicated element.
       *
       * @param drag - The drag object associated with the action.
       *
       * @param prevTarget - The previous target element, or `null`. This
       *   is the previous "current target element" from the whatwg spec.
       *
       * @param currTarget - The current drag target element, or `null`. This
       *   is the "current target element" from the whatwg spec.
       *
       * @param event - The mouse event related to the action.
       *
       * #### Notes
       * This largely implements the drag leave portion of the whatwg spec:
       * https://html.spec.whatwg.org/multipage/interaction.html#drag-and-drop-processing-model
       */
      function dispatchDragLeave(drag, prevTarget, currTarget, event) {
          // If the previous target is null, do nothing.
          if (!prevTarget) {
              return;
          }
          // Dispatch the drag leave event to the previous target.
          var dragEvent = createDragEvent('lm-dragleave', drag, event, currTarget);
          prevTarget.dispatchEvent(dragEvent);
          /* <DEPRECATED> */
          dragEvent = createDragEvent('p-dragleave', drag, event, currTarget);
          prevTarget.dispatchEvent(dragEvent);
          /* </DEPRECATED> */
      }
      Private.dispatchDragLeave = dispatchDragLeave;
      /**
       * Dispatch a drag over event to the indicated element.
       *
       * @param drag - The drag object associated with the action.
       *
       * @param currTarget - The current drag target element, or `null`. This
       *   is the "current target element" from the whatwg spec.
       *
       * @param event - The mouse event related to the action.
       *
       * @returns The `DropAction` result of the drag over event.
       *
       * #### Notes
       * This largely implements the drag over portion of the whatwg spec:
       * https://html.spec.whatwg.org/multipage/interaction.html#drag-and-drop-processing-model
       */
      function dispatchDragOver(drag, currTarget, event) {
          // If there is no current target, the drop action is none.
          if (!currTarget) {
              return 'none';
          }
          // Dispatch the drag over event to the current target.
          var dragEvent = createDragEvent('lm-dragover', drag, event, null);
          var canceled = !currTarget.dispatchEvent(dragEvent);
          // If the event was canceled, return the drop action result.
          if (canceled) {
              return dragEvent.dropAction;
          }
          /* <DEPRECATED> */
          dragEvent = createDragEvent('p-dragover', drag, event, null);
          canceled = !currTarget.dispatchEvent(dragEvent);
          if (canceled) {
              return dragEvent.dropAction;
          }
          /* </DEPRECATED> */
          // Otherwise, the effective drop action is none.
          return 'none';
      }
      Private.dispatchDragOver = dispatchDragOver;
      /**
       * Dispatch a drop event to the indicated element.
       *
       * @param drag - The drag object associated with the action.
       *
       * @param currTarget - The current drag target element, or `null`. This
       *   is the "current target element" from the whatwg spec.
       *
       * @param event - The mouse event related to the action.
       *
       * @returns The `DropAction` result of the drop event.
       *
       * #### Notes
       * This largely implements the drag over portion of the whatwg spec:
       * https://html.spec.whatwg.org/multipage/interaction.html#drag-and-drop-processing-model
       */
      function dispatchDrop(drag, currTarget, event) {
          // If there is no current target, the drop action is none.
          if (!currTarget) {
              return 'none';
          }
          // Dispatch the drop event to the current target.
          var dragEvent = createDragEvent('lm-drop', drag, event, null);
          var canceled = !currTarget.dispatchEvent(dragEvent);
          // If the event was canceled, return the drop action result.
          if (canceled) {
              return dragEvent.dropAction;
          }
          /* <DEPRECATED> */
          dragEvent = createDragEvent('p-drop', drag, event, null);
          canceled = !currTarget.dispatchEvent(dragEvent);
          if (canceled) {
              return dragEvent.dropAction;
          }
          /* </DEPRECATED> */
          // Otherwise, the effective drop action is none.
          return 'none';
      }
      Private.dispatchDrop = dispatchDrop;
      /**
       * A lookup table from drop action to bit value.
       */
      var actionTable = {
          'none': 0x0,
          'copy': 0x1,
          'link': 0x2,
          'move': 0x4
      };
      /**
       * A lookup table from supported action to drop action bit mask.
       */
      var supportedTable = {
          'none': actionTable['none'],
          'copy': actionTable['copy'],
          'link': actionTable['link'],
          'move': actionTable['move'],
          'copy-link': actionTable['copy'] | actionTable['link'],
          'copy-move': actionTable['copy'] | actionTable['move'],
          'link-move': actionTable['link'] | actionTable['move'],
          'all': actionTable['copy'] | actionTable['link'] | actionTable['move']
      };
      /**
       * Create a new initialized `IDragEvent` from the given data.
       *
       * @param type - The event type for the drag event.
       *
       * @param drag - The drag object to use for seeding the drag data.
       *
       * @param event - The mouse event to use for seeding the mouse data.
       *
       * @param related - The related target for the event, or `null`.
       *
       * @returns A new object which implements `IDragEvent`.
       */
      function createDragEvent(type, drag, event, related) {
          // Create a new mouse event to use as the drag event. Currently,
          // JS engines do now allow user-defined Event subclasses.
          var dragEvent = document.createEvent('MouseEvent');
          // Initialize the mouse event data.
          dragEvent.initMouseEvent(type, true, true, window, 0, event.screenX, event.screenY, event.clientX, event.clientY, event.ctrlKey, event.altKey, event.shiftKey, event.metaKey, event.button, related);
          // Forcefully add the custom drag event properties.
          dragEvent.dropAction = 'none';
          dragEvent.mimeData = drag.mimeData;
          dragEvent.proposedAction = drag.proposedAction;
          dragEvent.supportedActions = drag.supportedActions;
          dragEvent.source = drag.source;
          // Return the fully initialized drag event.
          return dragEvent;
      }
  })(Private$8 || (Private$8 = {}));

  // Copyright (c) Jupyter Development Team.
  // Distributed under the terms of the Modified BSD License.
  /*-----------------------------------------------------------------------------
  | Copyright (c) 2014-2017, PhosphorJS Contributors
  |
  | Distributed under the terms of the BSD 3-Clause License.
  |
  | The full license is in the file LICENSE, distributed with this software.
  |----------------------------------------------------------------------------*/
  /**
   * A sizer object for use with the box engine layout functions.
   *
   * #### Notes
   * A box sizer holds the geometry information for an object along an
   * arbitrary layout orientation.
   *
   * For best performance, this class should be treated as a raw data
   * struct. It should not typically be subclassed.
   */
  var BoxSizer = /** @class */ (function () {
      function BoxSizer() {
          /**
           * The preferred size for the sizer.
           *
           * #### Notes
           * The sizer will be given this initial size subject to its size
           * bounds. The sizer will not deviate from this size unless such
           * deviation is required to fit into the available layout space.
           *
           * There is no limit to this value, but it will be clamped to the
           * bounds defined by [[minSize]] and [[maxSize]].
           *
           * The default value is `0`.
           */
          this.sizeHint = 0;
          /**
           * The minimum size of the sizer.
           *
           * #### Notes
           * The sizer will never be sized less than this value, even if
           * it means the sizer will overflow the available layout space.
           *
           * It is assumed that this value lies in the range `[0, Infinity)`
           * and that it is `<=` to [[maxSize]]. Failure to adhere to this
           * constraint will yield undefined results.
           *
           * The default value is `0`.
           */
          this.minSize = 0;
          /**
           * The maximum size of the sizer.
           *
           * #### Notes
           * The sizer will never be sized greater than this value, even if
           * it means the sizer will underflow the available layout space.
           *
           * It is assumed that this value lies in the range `[0, Infinity]`
           * and that it is `>=` to [[minSize]]. Failure to adhere to this
           * constraint will yield undefined results.
           *
           * The default value is `Infinity`.
           */
          this.maxSize = Infinity;
          /**
           * The stretch factor for the sizer.
           *
           * #### Notes
           * This controls how much the sizer stretches relative to its sibling
           * sizers when layout space is distributed. A stretch factor of zero
           * is special and will cause the sizer to only be resized after all
           * other sizers with a stretch factor greater than zero have been
           * resized to their limits.
           *
           * It is assumed that this value is an integer that lies in the range
           * `[0, Infinity)`. Failure to adhere to this constraint will yield
           * undefined results.
           *
           * The default value is `1`.
           */
          this.stretch = 1;
          /**
           * The computed size of the sizer.
           *
           * #### Notes
           * This value is the output of a call to [[boxCalc]]. It represents
           * the computed size for the object along the layout orientation,
           * and will always lie in the range `[minSize, maxSize]`.
           *
           * This value is output only.
           *
           * Changing this value will have no effect.
           */
          this.size = 0;
          /**
           * An internal storage property for the layout algorithm.
           *
           * #### Notes
           * This value is used as temporary storage by the layout algorithm.
           *
           * Changing this value will have no effect.
           */
          this.done = false;
      }
      return BoxSizer;
  }());
  /**
   * The namespace for the box engine layout functions.
   */
  var BoxEngine;
  (function (BoxEngine) {
      /**
       * Calculate the optimal layout sizes for a sequence of box sizers.
       *
       * This distributes the available layout space among the box sizers
       * according to the following algorithm:
       *
       * 1. Initialize the sizers's size to its size hint and compute the
       *    sums for each of size hint, min size, and max size.
       *
       * 2. If the total size hint equals the available space, return.
       *
       * 3. If the available space is less than the total min size, set all
       *    sizers to their min size and return.
       *
       * 4. If the available space is greater than the total max size, set
       *    all sizers to their max size and return.
       *
       * 5. If the layout space is less than the total size hint, distribute
       *    the negative delta as follows:
       *
       *    a. Shrink each sizer with a stretch factor greater than zero by
       *       an amount proportional to the negative space and the sum of
       *       stretch factors. If the sizer reaches its min size, remove
       *       it and its stretch factor from the computation.
       *
       *    b. If after adjusting all stretch sizers there remains negative
       *       space, distribute the space equally among the sizers with a
       *       stretch factor of zero. If a sizer reaches its min size,
       *       remove it from the computation.
       *
       * 6. If the layout space is greater than the total size hint,
       *    distribute the positive delta as follows:
       *
       *    a. Expand each sizer with a stretch factor greater than zero by
       *       an amount proportional to the postive space and the sum of
       *       stretch factors. If the sizer reaches its max size, remove
       *       it and its stretch factor from the computation.
       *
       *    b. If after adjusting all stretch sizers there remains positive
       *       space, distribute the space equally among the sizers with a
       *       stretch factor of zero. If a sizer reaches its max size,
       *       remove it from the computation.
       *
       * 7. return
       *
       * @param sizers - The sizers for a particular layout line.
       *
       * @param space - The available layout space for the sizers.
       *
       * @returns The delta between the provided available space and the
       *   actual consumed space. This value will be zero if the sizers
       *   can be adjusted to fit, negative if the available space is too
       *   small, and positive if the available space is too large.
       *
       * #### Notes
       * The [[size]] of each sizer is updated with the computed size.
       *
       * This function can be called at any time to recompute the layout for
       * an existing sequence of sizers. The previously computed results will
       * have no effect on the new output. It is therefore not necessary to
       * create new sizer objects on each resize event.
       */
      function calc(sizers, space) {
          // Bail early if there is nothing to do.
          var count = sizers.length;
          if (count === 0) {
              return space;
          }
          // Setup the size and stretch counters.
          var totalMin = 0;
          var totalMax = 0;
          var totalSize = 0;
          var totalStretch = 0;
          var stretchCount = 0;
          // Setup the sizers and compute the totals.
          for (var i = 0; i < count; ++i) {
              var sizer = sizers[i];
              var min = sizer.minSize;
              var max = sizer.maxSize;
              var hint = sizer.sizeHint;
              sizer.done = false;
              sizer.size = Math.max(min, Math.min(hint, max));
              totalSize += sizer.size;
              totalMin += min;
              totalMax += max;
              if (sizer.stretch > 0) {
                  totalStretch += sizer.stretch;
                  stretchCount++;
              }
          }
          // If the space is equal to the total size, return early.
          if (space === totalSize) {
              return 0;
          }
          // If the space is less than the total min, minimize each sizer.
          if (space <= totalMin) {
              for (var i = 0; i < count; ++i) {
                  var sizer = sizers[i];
                  sizer.size = sizer.minSize;
              }
              return space - totalMin;
          }
          // If the space is greater than the total max, maximize each sizer.
          if (space >= totalMax) {
              for (var i = 0; i < count; ++i) {
                  var sizer = sizers[i];
                  sizer.size = sizer.maxSize;
              }
              return space - totalMax;
          }
          // The loops below perform sub-pixel precision sizing. A near zero
          // value is used for compares instead of zero to ensure that the
          // loop terminates when the subdivided space is reasonably small.
          var nearZero = 0.01;
          // A counter which is decremented each time a sizer is resized to
          // its limit. This ensures the loops terminate even if there is
          // space remaining to distribute.
          var notDoneCount = count;
          // Distribute negative delta space.
          if (space < totalSize) {
              // Shrink each stretchable sizer by an amount proportional to its
              // stretch factor. If a sizer reaches its min size it's marked as
              // done. The loop progresses in phases where each sizer is given
              // a chance to consume its fair share for the pass, regardless of
              // whether a sizer before it reached its limit. This continues
              // until the stretchable sizers or the free space is exhausted.
              var freeSpace = totalSize - space;
              while (stretchCount > 0 && freeSpace > nearZero) {
                  var distSpace = freeSpace;
                  var distStretch = totalStretch;
                  for (var i = 0; i < count; ++i) {
                      var sizer = sizers[i];
                      if (sizer.done || sizer.stretch === 0) {
                          continue;
                      }
                      var amt = sizer.stretch * distSpace / distStretch;
                      if (sizer.size - amt <= sizer.minSize) {
                          freeSpace -= sizer.size - sizer.minSize;
                          totalStretch -= sizer.stretch;
                          sizer.size = sizer.minSize;
                          sizer.done = true;
                          notDoneCount--;
                          stretchCount--;
                      }
                      else {
                          freeSpace -= amt;
                          sizer.size -= amt;
                      }
                  }
              }
              // Distribute any remaining space evenly among the non-stretchable
              // sizers. This progresses in phases in the same manner as above.
              while (notDoneCount > 0 && freeSpace > nearZero) {
                  var amt = freeSpace / notDoneCount;
                  for (var i = 0; i < count; ++i) {
                      var sizer = sizers[i];
                      if (sizer.done) {
                          continue;
                      }
                      if (sizer.size - amt <= sizer.minSize) {
                          freeSpace -= sizer.size - sizer.minSize;
                          sizer.size = sizer.minSize;
                          sizer.done = true;
                          notDoneCount--;
                      }
                      else {
                          freeSpace -= amt;
                          sizer.size -= amt;
                      }
                  }
              }
          }
          // Distribute positive delta space.
          else {
              // Expand each stretchable sizer by an amount proportional to its
              // stretch factor. If a sizer reaches its max size it's marked as
              // done. The loop progresses in phases where each sizer is given
              // a chance to consume its fair share for the pass, regardless of
              // whether a sizer before it reached its limit. This continues
              // until the stretchable sizers or the free space is exhausted.
              var freeSpace = space - totalSize;
              while (stretchCount > 0 && freeSpace > nearZero) {
                  var distSpace = freeSpace;
                  var distStretch = totalStretch;
                  for (var i = 0; i < count; ++i) {
                      var sizer = sizers[i];
                      if (sizer.done || sizer.stretch === 0) {
                          continue;
                      }
                      var amt = sizer.stretch * distSpace / distStretch;
                      if (sizer.size + amt >= sizer.maxSize) {
                          freeSpace -= sizer.maxSize - sizer.size;
                          totalStretch -= sizer.stretch;
                          sizer.size = sizer.maxSize;
                          sizer.done = true;
                          notDoneCount--;
                          stretchCount--;
                      }
                      else {
                          freeSpace -= amt;
                          sizer.size += amt;
                      }
                  }
              }
              // Distribute any remaining space evenly among the non-stretchable
              // sizers. This progresses in phases in the same manner as above.
              while (notDoneCount > 0 && freeSpace > nearZero) {
                  var amt = freeSpace / notDoneCount;
                  for (var i = 0; i < count; ++i) {
                      var sizer = sizers[i];
                      if (sizer.done) {
                          continue;
                      }
                      if (sizer.size + amt >= sizer.maxSize) {
                          freeSpace -= sizer.maxSize - sizer.size;
                          sizer.size = sizer.maxSize;
                          sizer.done = true;
                          notDoneCount--;
                      }
                      else {
                          freeSpace -= amt;
                          sizer.size += amt;
                      }
                  }
              }
          }
          // Indicate that the consumed space equals the available space.
          return 0;
      }
      BoxEngine.calc = calc;
      /**
       * Adjust a sizer by a delta and update its neighbors accordingly.
       *
       * @param sizers - The sizers which should be adjusted.
       *
       * @param index - The index of the sizer to grow.
       *
       * @param delta - The amount to adjust the sizer, positive or negative.
       *
       * #### Notes
       * This will adjust the indicated sizer by the specified amount, along
       * with the sizes of the appropriate neighbors, subject to the limits
       * specified by each of the sizers.
       *
       * This is useful when implementing box layouts where the boundaries
       * between the sizers are interactively adjustable by the user.
       */
      function adjust(sizers, index, delta) {
          // Bail early when there is nothing to do.
          if (sizers.length === 0 || delta === 0) {
              return;
          }
          // Dispatch to the proper implementation.
          if (delta > 0) {
              growSizer(sizers, index, delta);
          }
          else {
              shrinkSizer(sizers, index, -delta);
          }
      }
      BoxEngine.adjust = adjust;
      /**
       * Grow a sizer by a positive delta and adjust neighbors.
       */
      function growSizer(sizers, index, delta) {
          // Compute how much the items to the left can expand.
          var growLimit = 0;
          for (var i = 0; i <= index; ++i) {
              var sizer = sizers[i];
              growLimit += sizer.maxSize - sizer.size;
          }
          // Compute how much the items to the right can shrink.
          var shrinkLimit = 0;
          for (var i = index + 1, n = sizers.length; i < n; ++i) {
              var sizer = sizers[i];
              shrinkLimit += sizer.size - sizer.minSize;
          }
          // Clamp the delta adjustment to the limits.
          delta = Math.min(delta, growLimit, shrinkLimit);
          // Grow the sizers to the left by the delta.
          var grow = delta;
          for (var i = index; i >= 0 && grow > 0; --i) {
              var sizer = sizers[i];
              var limit = sizer.maxSize - sizer.size;
              if (limit >= grow) {
                  sizer.sizeHint = sizer.size + grow;
                  grow = 0;
              }
              else {
                  sizer.sizeHint = sizer.size + limit;
                  grow -= limit;
              }
          }
          // Shrink the sizers to the right by the delta.
          var shrink = delta;
          for (var i = index + 1, n = sizers.length; i < n && shrink > 0; ++i) {
              var sizer = sizers[i];
              var limit = sizer.size - sizer.minSize;
              if (limit >= shrink) {
                  sizer.sizeHint = sizer.size - shrink;
                  shrink = 0;
              }
              else {
                  sizer.sizeHint = sizer.size - limit;
                  shrink -= limit;
              }
          }
      }
      /**
       * Shrink a sizer by a positive delta and adjust neighbors.
       */
      function shrinkSizer(sizers, index, delta) {
          // Compute how much the items to the right can expand.
          var growLimit = 0;
          for (var i = index + 1, n = sizers.length; i < n; ++i) {
              var sizer = sizers[i];
              growLimit += sizer.maxSize - sizer.size;
          }
          // Compute how much the items to the left can shrink.
          var shrinkLimit = 0;
          for (var i = 0; i <= index; ++i) {
              var sizer = sizers[i];
              shrinkLimit += sizer.size - sizer.minSize;
          }
          // Clamp the delta adjustment to the limits.
          delta = Math.min(delta, growLimit, shrinkLimit);
          // Grow the sizers to the right by the delta.
          var grow = delta;
          for (var i = index + 1, n = sizers.length; i < n && grow > 0; ++i) {
              var sizer = sizers[i];
              var limit = sizer.maxSize - sizer.size;
              if (limit >= grow) {
                  sizer.sizeHint = sizer.size + grow;
                  grow = 0;
              }
              else {
                  sizer.sizeHint = sizer.size + limit;
                  grow -= limit;
              }
          }
          // Shrink the sizers to the left by the delta.
          var shrink = delta;
          for (var i = index; i >= 0 && shrink > 0; --i) {
              var sizer = sizers[i];
              var limit = sizer.size - sizer.minSize;
              if (limit >= shrink) {
                  sizer.sizeHint = sizer.size - shrink;
                  shrink = 0;
              }
              else {
                  sizer.sizeHint = sizer.size - limit;
                  shrink -= limit;
              }
          }
      }
  })(BoxEngine || (BoxEngine = {}));

  /*! *****************************************************************************
  Copyright (c) Microsoft Corporation. All rights reserved.
  Licensed under the Apache License, Version 2.0 (the "License"); you may not use
  this file except in compliance with the License. You may obtain a copy of the
  License at http://www.apache.org/licenses/LICENSE-2.0

  THIS CODE IS PROVIDED ON AN *AS IS* BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
  KIND, EITHER EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION ANY IMPLIED
  WARRANTIES OR CONDITIONS OF TITLE, FITNESS FOR A PARTICULAR PURPOSE,
  MERCHANTABLITY OR NON-INFRINGEMENT.

  See the Apache Version 2.0 License for specific language governing permissions
  and limitations under the License.
  ***************************************************************************** */
  /* global Reflect, Promise */

  var extendStatics$3 = function(d, b) {
      extendStatics$3 = Object.setPrototypeOf ||
          ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
          function (d, b) { for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p]; };
      return extendStatics$3(d, b);
  };

  function __extends$5(d, b) {
      extendStatics$3(d, b);
      function __() { this.constructor = d; }
      d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
  }

  var __assign$3 = function() {
      __assign$3 = Object.assign || function __assign(t) {
          for (var s, i = 1, n = arguments.length; i < n; i++) {
              s = arguments[i];
              for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p)) t[p] = s[p];
          }
          return t;
      };
      return __assign$3.apply(this, arguments);
  };

  // Copyright (c) Jupyter Development Team.
  /**
   * An object which holds data related to an object's title.
   *
   * #### Notes
   * A title object is intended to hold the data necessary to display a
   * header for a particular object. A common example is the `TabPanel`,
   * which uses the widget title to populate the tab for a child widget.
   */
  var Title = /** @class */ (function () {
      /**
       * Construct a new title.
       *
       * @param options - The options for initializing the title.
       */
      function Title(options) {
          this._label = '';
          this._caption = '';
          this._mnemonic = -1;
          this._iconClass = '';
          this._iconLabel = '';
          this._className = '';
          this._closable = false;
          this._changed = new Signal(this);
          this.owner = options.owner;
          if (options.label !== undefined) {
              this._label = options.label;
          }
          if (options.mnemonic !== undefined) {
              this._mnemonic = options.mnemonic;
          }
          if (options.icon !== undefined) {
              /* <DEPRECATED> */
              if (typeof options.icon === "string") {
                  // when ._icon is null, the .icon getter will alias .iconClass
                  this._icon = null;
                  this._iconClass = options.icon;
              }
              else {
                  /* </DEPRECATED> */
                  this._icon = options.icon;
                  /* <DEPRECATED> */
              }
              /* </DEPRECATED> */
          }
          /* <DEPRECATED> */
          else {
              // if unset, default to aliasing .iconClass
              this._icon = null;
          }
          /* </DEPRECATED> */
          if (options.iconClass !== undefined) {
              this._iconClass = options.iconClass;
          }
          if (options.iconLabel !== undefined) {
              this._iconLabel = options.iconLabel;
          }
          if (options.iconRenderer !== undefined) {
              this._icon = options.iconRenderer;
          }
          if (options.caption !== undefined) {
              this._caption = options.caption;
          }
          if (options.className !== undefined) {
              this._className = options.className;
          }
          if (options.closable !== undefined) {
              this._closable = options.closable;
          }
          this._dataset = options.dataset || {};
      }
      Object.defineProperty(Title.prototype, "changed", {
          /**
           * A signal emitted when the state of the title changes.
           */
          get: function () {
              return this._changed;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(Title.prototype, "label", {
          /**
           * Get the label for the title.
           *
           * #### Notes
           * The default value is an empty string.
           */
          get: function () {
              return this._label;
          },
          /**
           * Set the label for the title.
           */
          set: function (value) {
              if (this._label === value) {
                  return;
              }
              this._label = value;
              this._changed.emit(undefined);
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(Title.prototype, "mnemonic", {
          /**
           * Get the mnemonic index for the title.
           *
           * #### Notes
           * The default value is `-1`.
           */
          get: function () {
              return this._mnemonic;
          },
          /**
           * Set the mnemonic index for the title.
           */
          set: function (value) {
              if (this._mnemonic === value) {
                  return;
              }
              this._mnemonic = value;
              this._changed.emit(undefined);
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(Title.prototype, "icon", {
          /**
           * Get the icon renderer for the title.
           *
           * #### Notes
           * The default value is undefined.
           *
           * DEPRECATED: if set to a string value, the .icon field will function as
           * an alias for the .iconClass field, for backwards compatibility
           */
          get: function () {
              /* <DEPRECATED> */
              if (this._icon === null) {
                  // only alias .iconClass if ._icon has been explicitly nulled
                  return this.iconClass;
              }
              /* </DEPRECATED> */
              return this._icon;
          },
          /**
           * Set the icon renderer for the title.
           *
           * #### Notes
           * A renderer is an object that supplies a render and unrender function.
           *
           * DEPRECATED: if set to a string value, the .icon field will function as
           * an alias for the .iconClass field, for backwards compatibility
           */
          set: function (value /* </DEPRECATED> */) {
              /* <DEPRECATED> */
              if (typeof value === "string") {
                  // when ._icon is null, the .icon getter will alias .iconClass
                  this._icon = null;
                  this.iconClass = value;
              }
              else {
                  /* </DEPRECATED> */
                  if (this._icon === value) {
                      return;
                  }
                  this._icon = value;
                  this._changed.emit(undefined);
                  /* <DEPRECATED> */
              }
              /* </DEPRECATED> */
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(Title.prototype, "iconClass", {
          /**
           * Get the icon class name for the title.
           *
           * #### Notes
           * The default value is an empty string.
           */
          get: function () {
              return this._iconClass;
          },
          /**
           * Set the icon class name for the title.
           *
           * #### Notes
           * Multiple class names can be separated with whitespace.
           */
          set: function (value) {
              if (this._iconClass === value) {
                  return;
              }
              this._iconClass = value;
              this._changed.emit(undefined);
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(Title.prototype, "iconLabel", {
          /**
           * Get the icon label for the title.
           *
           * #### Notes
           * The default value is an empty string.
           */
          get: function () {
              return this._iconLabel;
          },
          /**
           * Set the icon label for the title.
           *
           * #### Notes
           * Multiple class names can be separated with whitespace.
           */
          set: function (value) {
              if (this._iconLabel === value) {
                  return;
              }
              this._iconLabel = value;
              this._changed.emit(undefined);
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(Title.prototype, "iconRenderer", {
          /**
           * @deprecated Use `icon` instead.
           */
          get: function () {
              return this._icon || undefined;
          },
          /**
           * @deprecated Use `icon` instead.
           */
          set: function (value) {
              this.icon = value;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(Title.prototype, "caption", {
          /**
           * Get the caption for the title.
           *
           * #### Notes
           * The default value is an empty string.
           */
          get: function () {
              return this._caption;
          },
          /**
           * Set the caption for the title.
           */
          set: function (value) {
              if (this._caption === value) {
                  return;
              }
              this._caption = value;
              this._changed.emit(undefined);
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(Title.prototype, "className", {
          /**
           * Get the extra class name for the title.
           *
           * #### Notes
           * The default value is an empty string.
           */
          get: function () {
              return this._className;
          },
          /**
           * Set the extra class name for the title.
           *
           * #### Notes
           * Multiple class names can be separated with whitespace.
           */
          set: function (value) {
              if (this._className === value) {
                  return;
              }
              this._className = value;
              this._changed.emit(undefined);
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(Title.prototype, "closable", {
          /**
           * Get the closable state for the title.
           *
           * #### Notes
           * The default value is `false`.
           */
          get: function () {
              return this._closable;
          },
          /**
           * Set the closable state for the title.
           *
           * #### Notes
           * This controls the presence of a close icon when applicable.
           */
          set: function (value) {
              if (this._closable === value) {
                  return;
              }
              this._closable = value;
              this._changed.emit(undefined);
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(Title.prototype, "dataset", {
          /**
           * Get the dataset for the title.
           *
           * #### Notes
           * The default value is an empty dataset.
           */
          get: function () {
              return this._dataset;
          },
          /**
           * Set the dataset for the title.
           *
           * #### Notes
           * This controls the data attributes when applicable.
           */
          set: function (value) {
              if (this._dataset === value) {
                  return;
              }
              this._dataset = value;
              this._changed.emit(undefined);
          },
          enumerable: true,
          configurable: true
      });
      return Title;
  }());

  /**
   * The base class of the lumino widget hierarchy.
   *
   * #### Notes
   * This class will typically be subclassed in order to create a useful
   * widget. However, it can be used directly to host externally created
   * content.
   */
  var Widget = /** @class */ (function () {
      /**
       * Construct a new widget.
       *
       * @param options - The options for initializing the widget.
       */
      function Widget(options) {
          if (options === void 0) { options = {}; }
          this._flags = 0;
          this._layout = null;
          this._parent = null;
          this._disposed = new Signal(this);
          this.node = Private$9.createNode(options);
          this.addClass('lm-Widget');
          /* <DEPRECATED> */
          this.addClass('p-Widget');
          /* </DEPRECATED> */
      }
      /**
       * Dispose of the widget and its descendant widgets.
       *
       * #### Notes
       * It is unsafe to use the widget after it has been disposed.
       *
       * All calls made to this method after the first are a no-op.
       */
      Widget.prototype.dispose = function () {
          // Do nothing if the widget is already disposed.
          if (this.isDisposed) {
              return;
          }
          // Set the disposed flag and emit the disposed signal.
          this.setFlag(Widget.Flag.IsDisposed);
          this._disposed.emit(undefined);
          // Remove or detach the widget if necessary.
          if (this.parent) {
              this.parent = null;
          }
          else if (this.isAttached) {
              Widget.detach(this);
          }
          // Dispose of the widget layout.
          if (this._layout) {
              this._layout.dispose();
              this._layout = null;
          }
          // Clear the extra data associated with the widget.
          Signal.clearData(this);
          MessageLoop.clearData(this);
          AttachedProperty.clearData(this);
      };
      Object.defineProperty(Widget.prototype, "disposed", {
          /**
           * A signal emitted when the widget is disposed.
           */
          get: function () {
              return this._disposed;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(Widget.prototype, "isDisposed", {
          /**
           * Test whether the widget has been disposed.
           */
          get: function () {
              return this.testFlag(Widget.Flag.IsDisposed);
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(Widget.prototype, "isAttached", {
          /**
           * Test whether the widget's node is attached to the DOM.
           */
          get: function () {
              return this.testFlag(Widget.Flag.IsAttached);
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(Widget.prototype, "isHidden", {
          /**
           * Test whether the widget is explicitly hidden.
           */
          get: function () {
              return this.testFlag(Widget.Flag.IsHidden);
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(Widget.prototype, "isVisible", {
          /**
           * Test whether the widget is visible.
           *
           * #### Notes
           * A widget is visible when it is attached to the DOM, is not
           * explicitly hidden, and has no explicitly hidden ancestors.
           */
          get: function () {
              return this.testFlag(Widget.Flag.IsVisible);
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(Widget.prototype, "title", {
          /**
           * The title object for the widget.
           *
           * #### Notes
           * The title object is used by some container widgets when displaying
           * the widget alongside some title, such as a tab panel or side bar.
           *
           * Since not all widgets will use the title, it is created on demand.
           *
           * The `owner` property of the title is set to this widget.
           */
          get: function () {
              return Private$9.titleProperty.get(this);
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(Widget.prototype, "id", {
          /**
           * Get the id of the widget's DOM node.
           */
          get: function () {
              return this.node.id;
          },
          /**
           * Set the id of the widget's DOM node.
           */
          set: function (value) {
              this.node.id = value;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(Widget.prototype, "dataset", {
          /**
           * The dataset for the widget's DOM node.
           */
          get: function () {
              return this.node.dataset;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(Widget.prototype, "parent", {
          /**
           * Get the parent of the widget.
           */
          get: function () {
              return this._parent;
          },
          /**
           * Set the parent of the widget.
           *
           * #### Notes
           * Children are typically added to a widget by using a layout, which
           * means user code will not normally set the parent widget directly.
           *
           * The widget will be automatically removed from its old parent.
           *
           * This is a no-op if there is no effective parent change.
           */
          set: function (value) {
              if (this._parent === value) {
                  return;
              }
              if (value && this.contains(value)) {
                  throw new Error('Invalid parent widget.');
              }
              if (this._parent && !this._parent.isDisposed) {
                  var msg = new Widget.ChildMessage('child-removed', this);
                  MessageLoop.sendMessage(this._parent, msg);
              }
              this._parent = value;
              if (this._parent && !this._parent.isDisposed) {
                  var msg = new Widget.ChildMessage('child-added', this);
                  MessageLoop.sendMessage(this._parent, msg);
              }
              if (!this.isDisposed) {
                  MessageLoop.sendMessage(this, Widget.Msg.ParentChanged);
              }
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(Widget.prototype, "layout", {
          /**
           * Get the layout for the widget.
           */
          get: function () {
              return this._layout;
          },
          /**
           * Set the layout for the widget.
           *
           * #### Notes
           * The layout is single-use only. It cannot be changed after the
           * first assignment.
           *
           * The layout is disposed automatically when the widget is disposed.
           */
          set: function (value) {
              if (this._layout === value) {
                  return;
              }
              if (this.testFlag(Widget.Flag.DisallowLayout)) {
                  throw new Error('Cannot set widget layout.');
              }
              if (this._layout) {
                  throw new Error('Cannot change widget layout.');
              }
              if (value.parent) {
                  throw new Error('Cannot change layout parent.');
              }
              this._layout = value;
              value.parent = this;
          },
          enumerable: true,
          configurable: true
      });
      /**
       * Create an iterator over the widget's children.
       *
       * @returns A new iterator over the children of the widget.
       *
       * #### Notes
       * The widget must have a populated layout in order to have children.
       *
       * If a layout is not installed, the returned iterator will be empty.
       */
      Widget.prototype.children = function () {
          return this._layout ? this._layout.iter() : empty();
      };
      /**
       * Test whether a widget is a descendant of this widget.
       *
       * @param widget - The descendant widget of interest.
       *
       * @returns `true` if the widget is a descendant, `false` otherwise.
       */
      Widget.prototype.contains = function (widget) {
          for (var value = widget; value; value = value._parent) {
              if (value === this) {
                  return true;
              }
          }
          return false;
      };
      /**
       * Test whether the widget's DOM node has the given class name.
       *
       * @param name - The class name of interest.
       *
       * @returns `true` if the node has the class, `false` otherwise.
       */
      Widget.prototype.hasClass = function (name) {
          return this.node.classList.contains(name);
      };
      /**
       * Add a class name to the widget's DOM node.
       *
       * @param name - The class name to add to the node.
       *
       * #### Notes
       * If the class name is already added to the node, this is a no-op.
       *
       * The class name must not contain whitespace.
       */
      Widget.prototype.addClass = function (name) {
          this.node.classList.add(name);
      };
      /**
       * Remove a class name from the widget's DOM node.
       *
       * @param name - The class name to remove from the node.
       *
       * #### Notes
       * If the class name is not yet added to the node, this is a no-op.
       *
       * The class name must not contain whitespace.
       */
      Widget.prototype.removeClass = function (name) {
          this.node.classList.remove(name);
      };
      /**
       * Toggle a class name on the widget's DOM node.
       *
       * @param name - The class name to toggle on the node.
       *
       * @param force - Whether to force add the class (`true`) or force
       *   remove the class (`false`). If not provided, the presence of
       *   the class will be toggled from its current state.
       *
       * @returns `true` if the class is now present, `false` otherwise.
       *
       * #### Notes
       * The class name must not contain whitespace.
       */
      Widget.prototype.toggleClass = function (name, force) {
          if (force === true) {
              this.node.classList.add(name);
              return true;
          }
          if (force === false) {
              this.node.classList.remove(name);
              return false;
          }
          return this.node.classList.toggle(name);
      };
      /**
       * Post an `'update-request'` message to the widget.
       *
       * #### Notes
       * This is a simple convenience method for posting the message.
       */
      Widget.prototype.update = function () {
          MessageLoop.postMessage(this, Widget.Msg.UpdateRequest);
      };
      /**
       * Post a `'fit-request'` message to the widget.
       *
       * #### Notes
       * This is a simple convenience method for posting the message.
       */
      Widget.prototype.fit = function () {
          MessageLoop.postMessage(this, Widget.Msg.FitRequest);
      };
      /**
       * Post an `'activate-request'` message to the widget.
       *
       * #### Notes
       * This is a simple convenience method for posting the message.
       */
      Widget.prototype.activate = function () {
          MessageLoop.postMessage(this, Widget.Msg.ActivateRequest);
      };
      /**
       * Send a `'close-request'` message to the widget.
       *
       * #### Notes
       * This is a simple convenience method for sending the message.
       */
      Widget.prototype.close = function () {
          MessageLoop.sendMessage(this, Widget.Msg.CloseRequest);
      };
      /**
       * Show the widget and make it visible to its parent widget.
       *
       * #### Notes
       * This causes the [[isHidden]] property to be `false`.
       *
       * If the widget is not explicitly hidden, this is a no-op.
       */
      Widget.prototype.show = function () {
          if (!this.testFlag(Widget.Flag.IsHidden)) {
              return;
          }
          if (this.isAttached && (!this.parent || this.parent.isVisible)) {
              MessageLoop.sendMessage(this, Widget.Msg.BeforeShow);
          }
          this.clearFlag(Widget.Flag.IsHidden);
          this.removeClass('lm-mod-hidden');
          /* <DEPRECATED> */
          this.removeClass('p-mod-hidden');
          /* </DEPRECATED> */
          if (this.isAttached && (!this.parent || this.parent.isVisible)) {
              MessageLoop.sendMessage(this, Widget.Msg.AfterShow);
          }
          if (this.parent) {
              var msg = new Widget.ChildMessage('child-shown', this);
              MessageLoop.sendMessage(this.parent, msg);
          }
      };
      /**
       * Hide the widget and make it hidden to its parent widget.
       *
       * #### Notes
       * This causes the [[isHidden]] property to be `true`.
       *
       * If the widget is explicitly hidden, this is a no-op.
       */
      Widget.prototype.hide = function () {
          if (this.testFlag(Widget.Flag.IsHidden)) {
              return;
          }
          if (this.isAttached && (!this.parent || this.parent.isVisible)) {
              MessageLoop.sendMessage(this, Widget.Msg.BeforeHide);
          }
          this.setFlag(Widget.Flag.IsHidden);
          this.addClass('lm-mod-hidden');
          /* <DEPRECATED> */
          this.addClass('p-mod-hidden');
          /* </DEPRECATED> */
          if (this.isAttached && (!this.parent || this.parent.isVisible)) {
              MessageLoop.sendMessage(this, Widget.Msg.AfterHide);
          }
          if (this.parent) {
              var msg = new Widget.ChildMessage('child-hidden', this);
              MessageLoop.sendMessage(this.parent, msg);
          }
      };
      /**
       * Show or hide the widget according to a boolean value.
       *
       * @param hidden - `true` to hide the widget, or `false` to show it.
       *
       * #### Notes
       * This is a convenience method for `hide()` and `show()`.
       */
      Widget.prototype.setHidden = function (hidden) {
          if (hidden) {
              this.hide();
          }
          else {
              this.show();
          }
      };
      /**
       * Test whether the given widget flag is set.
       *
       * #### Notes
       * This will not typically be called directly by user code.
       */
      Widget.prototype.testFlag = function (flag) {
          return (this._flags & flag) !== 0;
      };
      /**
       * Set the given widget flag.
       *
       * #### Notes
       * This will not typically be called directly by user code.
       */
      Widget.prototype.setFlag = function (flag) {
          this._flags |= flag;
      };
      /**
       * Clear the given widget flag.
       *
       * #### Notes
       * This will not typically be called directly by user code.
       */
      Widget.prototype.clearFlag = function (flag) {
          this._flags &= ~flag;
      };
      /**
       * Process a message sent to the widget.
       *
       * @param msg - The message sent to the widget.
       *
       * #### Notes
       * Subclasses may reimplement this method as needed.
       */
      Widget.prototype.processMessage = function (msg) {
          switch (msg.type) {
              case 'resize':
                  this.notifyLayout(msg);
                  this.onResize(msg);
                  break;
              case 'update-request':
                  this.notifyLayout(msg);
                  this.onUpdateRequest(msg);
                  break;
              case 'fit-request':
                  this.notifyLayout(msg);
                  this.onFitRequest(msg);
                  break;
              case 'before-show':
                  this.notifyLayout(msg);
                  this.onBeforeShow(msg);
                  break;
              case 'after-show':
                  this.setFlag(Widget.Flag.IsVisible);
                  this.notifyLayout(msg);
                  this.onAfterShow(msg);
                  break;
              case 'before-hide':
                  this.notifyLayout(msg);
                  this.onBeforeHide(msg);
                  break;
              case 'after-hide':
                  this.clearFlag(Widget.Flag.IsVisible);
                  this.notifyLayout(msg);
                  this.onAfterHide(msg);
                  break;
              case 'before-attach':
                  this.notifyLayout(msg);
                  this.onBeforeAttach(msg);
                  break;
              case 'after-attach':
                  if (!this.isHidden && (!this.parent || this.parent.isVisible)) {
                      this.setFlag(Widget.Flag.IsVisible);
                  }
                  this.setFlag(Widget.Flag.IsAttached);
                  this.notifyLayout(msg);
                  this.onAfterAttach(msg);
                  break;
              case 'before-detach':
                  this.notifyLayout(msg);
                  this.onBeforeDetach(msg);
                  break;
              case 'after-detach':
                  this.clearFlag(Widget.Flag.IsVisible);
                  this.clearFlag(Widget.Flag.IsAttached);
                  this.notifyLayout(msg);
                  this.onAfterDetach(msg);
                  break;
              case 'activate-request':
                  this.notifyLayout(msg);
                  this.onActivateRequest(msg);
                  break;
              case 'close-request':
                  this.notifyLayout(msg);
                  this.onCloseRequest(msg);
                  break;
              case 'child-added':
                  this.notifyLayout(msg);
                  this.onChildAdded(msg);
                  break;
              case 'child-removed':
                  this.notifyLayout(msg);
                  this.onChildRemoved(msg);
                  break;
              default:
                  this.notifyLayout(msg);
                  break;
          }
      };
      /**
       * Invoke the message processing routine of the widget's layout.
       *
       * @param msg - The message to dispatch to the layout.
       *
       * #### Notes
       * This is a no-op if the widget does not have a layout.
       *
       * This will not typically be called directly by user code.
       */
      Widget.prototype.notifyLayout = function (msg) {
          if (this._layout) {
              this._layout.processParentMessage(msg);
          }
      };
      /**
       * A message handler invoked on a `'close-request'` message.
       *
       * #### Notes
       * The default implementation unparents or detaches the widget.
       */
      Widget.prototype.onCloseRequest = function (msg) {
          if (this.parent) {
              this.parent = null;
          }
          else if (this.isAttached) {
              Widget.detach(this);
          }
      };
      /**
       * A message handler invoked on a `'resize'` message.
       *
       * #### Notes
       * The default implementation of this handler is a no-op.
       */
      Widget.prototype.onResize = function (msg) { };
      /**
       * A message handler invoked on an `'update-request'` message.
       *
       * #### Notes
       * The default implementation of this handler is a no-op.
       */
      Widget.prototype.onUpdateRequest = function (msg) { };
      /**
       * A message handler invoked on a `'fit-request'` message.
       *
       * #### Notes
       * The default implementation of this handler is a no-op.
       */
      Widget.prototype.onFitRequest = function (msg) { };
      /**
       * A message handler invoked on an `'activate-request'` message.
       *
       * #### Notes
       * The default implementation of this handler is a no-op.
       */
      Widget.prototype.onActivateRequest = function (msg) { };
      /**
       * A message handler invoked on a `'before-show'` message.
       *
       * #### Notes
       * The default implementation of this handler is a no-op.
       */
      Widget.prototype.onBeforeShow = function (msg) { };
      /**
       * A message handler invoked on an `'after-show'` message.
       *
       * #### Notes
       * The default implementation of this handler is a no-op.
       */
      Widget.prototype.onAfterShow = function (msg) { };
      /**
       * A message handler invoked on a `'before-hide'` message.
       *
       * #### Notes
       * The default implementation of this handler is a no-op.
       */
      Widget.prototype.onBeforeHide = function (msg) { };
      /**
       * A message handler invoked on an `'after-hide'` message.
       *
       * #### Notes
       * The default implementation of this handler is a no-op.
       */
      Widget.prototype.onAfterHide = function (msg) { };
      /**
       * A message handler invoked on a `'before-attach'` message.
       *
       * #### Notes
       * The default implementation of this handler is a no-op.
       */
      Widget.prototype.onBeforeAttach = function (msg) { };
      /**
       * A message handler invoked on an `'after-attach'` message.
       *
       * #### Notes
       * The default implementation of this handler is a no-op.
       */
      Widget.prototype.onAfterAttach = function (msg) { };
      /**
       * A message handler invoked on a `'before-detach'` message.
       *
       * #### Notes
       * The default implementation of this handler is a no-op.
       */
      Widget.prototype.onBeforeDetach = function (msg) { };
      /**
       * A message handler invoked on an `'after-detach'` message.
       *
       * #### Notes
       * The default implementation of this handler is a no-op.
       */
      Widget.prototype.onAfterDetach = function (msg) { };
      /**
       * A message handler invoked on a `'child-added'` message.
       *
       * #### Notes
       * The default implementation of this handler is a no-op.
       */
      Widget.prototype.onChildAdded = function (msg) { };
      /**
       * A message handler invoked on a `'child-removed'` message.
       *
       * #### Notes
       * The default implementation of this handler is a no-op.
       */
      Widget.prototype.onChildRemoved = function (msg) { };
      return Widget;
  }());
  /**
   * The namespace for the `Widget` class statics.
   */
  (function (Widget) {
      /**
       * An enum of widget bit flags.
       */
      var Flag;
      (function (Flag) {
          /**
           * The widget has been disposed.
           */
          Flag[Flag["IsDisposed"] = 1] = "IsDisposed";
          /**
           * The widget is attached to the DOM.
           */
          Flag[Flag["IsAttached"] = 2] = "IsAttached";
          /**
           * The widget is hidden.
           */
          Flag[Flag["IsHidden"] = 4] = "IsHidden";
          /**
           * The widget is visible.
           */
          Flag[Flag["IsVisible"] = 8] = "IsVisible";
          /**
           * A layout cannot be set on the widget.
           */
          Flag[Flag["DisallowLayout"] = 16] = "DisallowLayout";
      })(Flag = Widget.Flag || (Widget.Flag = {}));
      /**
       * A collection of stateless messages related to widgets.
       */
      var Msg;
      (function (Msg) {
          /**
           * A singleton `'before-show'` message.
           *
           * #### Notes
           * This message is sent to a widget before it becomes visible.
           *
           * This message is **not** sent when the widget is being attached.
           */
          Msg.BeforeShow = new Message('before-show');
          /**
           * A singleton `'after-show'` message.
           *
           * #### Notes
           * This message is sent to a widget after it becomes visible.
           *
           * This message is **not** sent when the widget is being attached.
           */
          Msg.AfterShow = new Message('after-show');
          /**
           * A singleton `'before-hide'` message.
           *
           * #### Notes
           * This message is sent to a widget before it becomes not-visible.
           *
           * This message is **not** sent when the widget is being detached.
           */
          Msg.BeforeHide = new Message('before-hide');
          /**
           * A singleton `'after-hide'` message.
           *
           * #### Notes
           * This message is sent to a widget after it becomes not-visible.
           *
           * This message is **not** sent when the widget is being detached.
           */
          Msg.AfterHide = new Message('after-hide');
          /**
           * A singleton `'before-attach'` message.
           *
           * #### Notes
           * This message is sent to a widget before it is attached.
           */
          Msg.BeforeAttach = new Message('before-attach');
          /**
           * A singleton `'after-attach'` message.
           *
           * #### Notes
           * This message is sent to a widget after it is attached.
           */
          Msg.AfterAttach = new Message('after-attach');
          /**
           * A singleton `'before-detach'` message.
           *
           * #### Notes
           * This message is sent to a widget before it is detached.
           */
          Msg.BeforeDetach = new Message('before-detach');
          /**
           * A singleton `'after-detach'` message.
           *
           * #### Notes
           * This message is sent to a widget after it is detached.
           */
          Msg.AfterDetach = new Message('after-detach');
          /**
           * A singleton `'parent-changed'` message.
           *
           * #### Notes
           * This message is sent to a widget when its parent has changed.
           */
          Msg.ParentChanged = new Message('parent-changed');
          /**
           * A singleton conflatable `'update-request'` message.
           *
           * #### Notes
           * This message can be dispatched to supporting widgets in order to
           * update their content based on the current widget state. Not all
           * widgets will respond to messages of this type.
           *
           * For widgets with a layout, this message will inform the layout to
           * update the position and size of its child widgets.
           */
          Msg.UpdateRequest = new ConflatableMessage('update-request');
          /**
           * A singleton conflatable `'fit-request'` message.
           *
           * #### Notes
           * For widgets with a layout, this message will inform the layout to
           * recalculate its size constraints to fit the space requirements of
           * its child widgets, and to update their position and size. Not all
           * layouts will respond to messages of this type.
           */
          Msg.FitRequest = new ConflatableMessage('fit-request');
          /**
           * A singleton conflatable `'activate-request'` message.
           *
           * #### Notes
           * This message should be dispatched to a widget when it should
           * perform the actions necessary to activate the widget, which
           * may include focusing its node or descendant node.
           */
          Msg.ActivateRequest = new ConflatableMessage('activate-request');
          /**
           * A singleton conflatable `'close-request'` message.
           *
           * #### Notes
           * This message should be dispatched to a widget when it should close
           * and remove itself from the widget hierarchy.
           */
          Msg.CloseRequest = new ConflatableMessage('close-request');
      })(Msg = Widget.Msg || (Widget.Msg = {}));
      /**
       * A message class for child related messages.
       */
      var ChildMessage = /** @class */ (function (_super) {
          __extends$5(ChildMessage, _super);
          /**
           * Construct a new child message.
           *
           * @param type - The message type.
           *
           * @param child - The child widget for the message.
           */
          function ChildMessage(type, child) {
              var _this = _super.call(this, type) || this;
              _this.child = child;
              return _this;
          }
          return ChildMessage;
      }(Message));
      Widget.ChildMessage = ChildMessage;
      /**
       * A message class for `'resize'` messages.
       */
      var ResizeMessage = /** @class */ (function (_super) {
          __extends$5(ResizeMessage, _super);
          /**
           * Construct a new resize message.
           *
           * @param width - The **offset width** of the widget, or `-1` if
           *   the width is not known.
           *
           * @param height - The **offset height** of the widget, or `-1` if
           *   the height is not known.
           */
          function ResizeMessage(width, height) {
              var _this = _super.call(this, 'resize') || this;
              _this.width = width;
              _this.height = height;
              return _this;
          }
          return ResizeMessage;
      }(Message));
      Widget.ResizeMessage = ResizeMessage;
      /**
       * The namespace for the `ResizeMessage` class statics.
       */
      (function (ResizeMessage) {
          /**
           * A singleton `'resize'` message with an unknown size.
           */
          ResizeMessage.UnknownSize = new ResizeMessage(-1, -1);
      })(ResizeMessage = Widget.ResizeMessage || (Widget.ResizeMessage = {}));
      /**
       * Attach a widget to a host DOM node.
       *
       * @param widget - The widget of interest.
       *
       * @param host - The DOM node to use as the widget's host.
       *
       * @param ref - The child of `host` to use as the reference element.
       *   If this is provided, the widget will be inserted before this
       *   node in the host. The default is `null`, which will cause the
       *   widget to be added as the last child of the host.
       *
       * #### Notes
       * This will throw an error if the widget is not a root widget, if
       * the widget is already attached, or if the host is not attached
       * to the DOM.
       */
      function attach(widget, host, ref) {
          if (ref === void 0) { ref = null; }
          if (widget.parent) {
              throw new Error('Cannot attach a child widget.');
          }
          if (widget.isAttached || document.body.contains(widget.node)) {
              throw new Error('Widget is already attached.');
          }
          if (!document.body.contains(host)) {
              throw new Error('Host is not attached.');
          }
          MessageLoop.sendMessage(widget, Widget.Msg.BeforeAttach);
          host.insertBefore(widget.node, ref);
          MessageLoop.sendMessage(widget, Widget.Msg.AfterAttach);
      }
      Widget.attach = attach;
      /**
       * Detach the widget from its host DOM node.
       *
       * @param widget - The widget of interest.
       *
       * #### Notes
       * This will throw an error if the widget is not a root widget,
       * or if the widget is not attached to the DOM.
       */
      function detach(widget) {
          if (widget.parent) {
              throw new Error('Cannot detach a child widget.');
          }
          if (!widget.isAttached || !document.body.contains(widget.node)) {
              throw new Error('Widget is not attached.');
          }
          MessageLoop.sendMessage(widget, Widget.Msg.BeforeDetach);
          widget.node.parentNode.removeChild(widget.node);
          MessageLoop.sendMessage(widget, Widget.Msg.AfterDetach);
      }
      Widget.detach = detach;
  })(Widget || (Widget = {}));
  /**
   * The namespace for the module implementation details.
   */
  var Private$9;
  (function (Private) {
      /**
       * An attached property for the widget title object.
       */
      Private.titleProperty = new AttachedProperty({
          name: 'title',
          create: function (owner) { return new Title({ owner: owner }); },
      });
      /**
       * Create a DOM node for the given widget options.
       */
      function createNode(options) {
          return options.node || document.createElement('div');
      }
      Private.createNode = createNode;
  })(Private$9 || (Private$9 = {}));

  // Copyright (c) Jupyter Development Team.
  /**
   * An abstract base class for creating lumino layouts.
   *
   * #### Notes
   * A layout is used to add widgets to a parent and to arrange those
   * widgets within the parent's DOM node.
   *
   * This class implements the base functionality which is required of
   * nearly all layouts. It must be subclassed in order to be useful.
   *
   * Notably, this class does not define a uniform interface for adding
   * widgets to the layout. A subclass should define that API in a way
   * which is meaningful for its intended use.
   */
  var Layout = /** @class */ (function () {
      /**
       * Construct a new layout.
       *
       * @param options - The options for initializing the layout.
       */
      function Layout(options) {
          if (options === void 0) { options = {}; }
          this._disposed = false;
          this._parent = null;
          this._fitPolicy = options.fitPolicy || 'set-min-size';
      }
      /**
       * Dispose of the resources held by the layout.
       *
       * #### Notes
       * This should be reimplemented to clear and dispose of the widgets.
       *
       * All reimplementations should call the superclass method.
       *
       * This method is called automatically when the parent is disposed.
       */
      Layout.prototype.dispose = function () {
          this._parent = null;
          this._disposed = true;
          Signal.clearData(this);
          AttachedProperty.clearData(this);
      };
      Object.defineProperty(Layout.prototype, "isDisposed", {
          /**
           * Test whether the layout is disposed.
           */
          get: function () {
              return this._disposed;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(Layout.prototype, "parent", {
          /**
           * Get the parent widget of the layout.
           */
          get: function () {
              return this._parent;
          },
          /**
           * Set the parent widget of the layout.
           *
           * #### Notes
           * This is set automatically when installing the layout on the parent
           * widget. The parent widget should not be set directly by user code.
           */
          set: function (value) {
              if (this._parent === value) {
                  return;
              }
              if (this._parent) {
                  throw new Error('Cannot change parent widget.');
              }
              if (value.layout !== this) {
                  throw new Error('Invalid parent widget.');
              }
              this._parent = value;
              this.init();
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(Layout.prototype, "fitPolicy", {
          /**
           * Get the fit policy for the layout.
           *
           * #### Notes
           * The fit policy controls the computed size constraints which are
           * applied to the parent widget by the layout.
           *
           * Some layout implementations may ignore the fit policy.
           */
          get: function () {
              return this._fitPolicy;
          },
          /**
           * Set the fit policy for the layout.
           *
           * #### Notes
           * The fit policy controls the computed size constraints which are
           * applied to the parent widget by the layout.
           *
           * Some layout implementations may ignore the fit policy.
           *
           * Changing the fit policy will clear the current size constraint
           * for the parent widget and then re-fit the parent.
           */
          set: function (value) {
              // Bail if the policy does not change
              if (this._fitPolicy === value) {
                  return;
              }
              // Update the internal policy.
              this._fitPolicy = value;
              // Clear the size constraints and schedule a fit of the parent.
              if (this._parent) {
                  var style = this._parent.node.style;
                  style.minWidth = '';
                  style.minHeight = '';
                  style.maxWidth = '';
                  style.maxHeight = '';
                  this._parent.fit();
              }
          },
          enumerable: true,
          configurable: true
      });
      /**
       * Process a message sent to the parent widget.
       *
       * @param msg - The message sent to the parent widget.
       *
       * #### Notes
       * This method is called by the parent widget to process a message.
       *
       * Subclasses may reimplement this method as needed.
       */
      Layout.prototype.processParentMessage = function (msg) {
          switch (msg.type) {
              case 'resize':
                  this.onResize(msg);
                  break;
              case 'update-request':
                  this.onUpdateRequest(msg);
                  break;
              case 'fit-request':
                  this.onFitRequest(msg);
                  break;
              case 'before-show':
                  this.onBeforeShow(msg);
                  break;
              case 'after-show':
                  this.onAfterShow(msg);
                  break;
              case 'before-hide':
                  this.onBeforeHide(msg);
                  break;
              case 'after-hide':
                  this.onAfterHide(msg);
                  break;
              case 'before-attach':
                  this.onBeforeAttach(msg);
                  break;
              case 'after-attach':
                  this.onAfterAttach(msg);
                  break;
              case 'before-detach':
                  this.onBeforeDetach(msg);
                  break;
              case 'after-detach':
                  this.onAfterDetach(msg);
                  break;
              case 'child-removed':
                  this.onChildRemoved(msg);
                  break;
              case 'child-shown':
                  this.onChildShown(msg);
                  break;
              case 'child-hidden':
                  this.onChildHidden(msg);
                  break;
          }
      };
      /**
       * Perform layout initialization which requires the parent widget.
       *
       * #### Notes
       * This method is invoked immediately after the layout is installed
       * on the parent widget.
       *
       * The default implementation reparents all of the widgets to the
       * layout parent widget.
       *
       * Subclasses should reimplement this method and attach the child
       * widget nodes to the parent widget's node.
       */
      Layout.prototype.init = function () {
          var _this = this;
          each$1(this, function (widget) {
              widget.parent = _this.parent;
          });
      };
      /**
       * A message handler invoked on a `'resize'` message.
       *
       * #### Notes
       * The layout should ensure that its widgets are resized according
       * to the specified layout space, and that they are sent a `'resize'`
       * message if appropriate.
       *
       * The default implementation of this method sends an `UnknownSize`
       * resize message to all widgets.
       *
       * This may be reimplemented by subclasses as needed.
       */
      Layout.prototype.onResize = function (msg) {
          each$1(this, function (widget) {
              MessageLoop.sendMessage(widget, Widget.ResizeMessage.UnknownSize);
          });
      };
      /**
       * A message handler invoked on an `'update-request'` message.
       *
       * #### Notes
       * The layout should ensure that its widgets are resized according
       * to the available layout space, and that they are sent a `'resize'`
       * message if appropriate.
       *
       * The default implementation of this method sends an `UnknownSize`
       * resize message to all widgets.
       *
       * This may be reimplemented by subclasses as needed.
       */
      Layout.prototype.onUpdateRequest = function (msg) {
          each$1(this, function (widget) {
              MessageLoop.sendMessage(widget, Widget.ResizeMessage.UnknownSize);
          });
      };
      /**
       * A message handler invoked on a `'before-attach'` message.
       *
       * #### Notes
       * The default implementation of this method forwards the message
       * to all widgets. It assumes all widget nodes are attached to the
       * parent widget node.
       *
       * This may be reimplemented by subclasses as needed.
       */
      Layout.prototype.onBeforeAttach = function (msg) {
          each$1(this, function (widget) {
              MessageLoop.sendMessage(widget, msg);
          });
      };
      /**
       * A message handler invoked on an `'after-attach'` message.
       *
       * #### Notes
       * The default implementation of this method forwards the message
       * to all widgets. It assumes all widget nodes are attached to the
       * parent widget node.
       *
       * This may be reimplemented by subclasses as needed.
       */
      Layout.prototype.onAfterAttach = function (msg) {
          each$1(this, function (widget) {
              MessageLoop.sendMessage(widget, msg);
          });
      };
      /**
       * A message handler invoked on a `'before-detach'` message.
       *
       * #### Notes
       * The default implementation of this method forwards the message
       * to all widgets. It assumes all widget nodes are attached to the
       * parent widget node.
       *
       * This may be reimplemented by subclasses as needed.
       */
      Layout.prototype.onBeforeDetach = function (msg) {
          each$1(this, function (widget) {
              MessageLoop.sendMessage(widget, msg);
          });
      };
      /**
       * A message handler invoked on an `'after-detach'` message.
       *
       * #### Notes
       * The default implementation of this method forwards the message
       * to all widgets. It assumes all widget nodes are attached to the
       * parent widget node.
       *
       * This may be reimplemented by subclasses as needed.
       */
      Layout.prototype.onAfterDetach = function (msg) {
          each$1(this, function (widget) {
              MessageLoop.sendMessage(widget, msg);
          });
      };
      /**
       * A message handler invoked on a `'before-show'` message.
       *
       * #### Notes
       * The default implementation of this method forwards the message to
       * all non-hidden widgets. It assumes all widget nodes are attached
       * to the parent widget node.
       *
       * This may be reimplemented by subclasses as needed.
       */
      Layout.prototype.onBeforeShow = function (msg) {
          each$1(this, function (widget) {
              if (!widget.isHidden) {
                  MessageLoop.sendMessage(widget, msg);
              }
          });
      };
      /**
       * A message handler invoked on an `'after-show'` message.
       *
       * #### Notes
       * The default implementation of this method forwards the message to
       * all non-hidden widgets. It assumes all widget nodes are attached
       * to the parent widget node.
       *
       * This may be reimplemented by subclasses as needed.
       */
      Layout.prototype.onAfterShow = function (msg) {
          each$1(this, function (widget) {
              if (!widget.isHidden) {
                  MessageLoop.sendMessage(widget, msg);
              }
          });
      };
      /**
       * A message handler invoked on a `'before-hide'` message.
       *
       * #### Notes
       * The default implementation of this method forwards the message to
       * all non-hidden widgets. It assumes all widget nodes are attached
       * to the parent widget node.
       *
       * This may be reimplemented by subclasses as needed.
       */
      Layout.prototype.onBeforeHide = function (msg) {
          each$1(this, function (widget) {
              if (!widget.isHidden) {
                  MessageLoop.sendMessage(widget, msg);
              }
          });
      };
      /**
       * A message handler invoked on an `'after-hide'` message.
       *
       * #### Notes
       * The default implementation of this method forwards the message to
       * all non-hidden widgets. It assumes all widget nodes are attached
       * to the parent widget node.
       *
       * This may be reimplemented by subclasses as needed.
       */
      Layout.prototype.onAfterHide = function (msg) {
          each$1(this, function (widget) {
              if (!widget.isHidden) {
                  MessageLoop.sendMessage(widget, msg);
              }
          });
      };
      /**
       * A message handler invoked on a `'child-removed'` message.
       *
       * #### Notes
       * This will remove the child widget from the layout.
       *
       * Subclasses should **not** typically reimplement this method.
       */
      Layout.prototype.onChildRemoved = function (msg) {
          this.removeWidget(msg.child);
      };
      /**
       * A message handler invoked on a `'fit-request'` message.
       *
       * #### Notes
       * The default implementation of this handler is a no-op.
       */
      Layout.prototype.onFitRequest = function (msg) { };
      /**
       * A message handler invoked on a `'child-shown'` message.
       *
       * #### Notes
       * The default implementation of this handler is a no-op.
       */
      Layout.prototype.onChildShown = function (msg) { };
      /**
       * A message handler invoked on a `'child-hidden'` message.
       *
       * #### Notes
       * The default implementation of this handler is a no-op.
       */
      Layout.prototype.onChildHidden = function (msg) { };
      return Layout;
  }());
  /**
   * The namespace for the `Layout` class statics.
   */
  (function (Layout) {
      /**
       * Get the horizontal alignment for a widget.
       *
       * @param widget - The widget of interest.
       *
       * @returns The horizontal alignment for the widget.
       *
       * #### Notes
       * If the layout width allocated to a widget is larger than its max
       * width, the horizontal alignment controls how the widget is placed
       * within the extra horizontal space.
       *
       * If the allocated width is less than the widget's max width, the
       * horizontal alignment has no effect.
       *
       * Some layout implementations may ignore horizontal alignment.
       */
      function getHorizontalAlignment(widget) {
          return Private$1$2.horizontalAlignmentProperty.get(widget);
      }
      Layout.getHorizontalAlignment = getHorizontalAlignment;
      /**
       * Set the horizontal alignment for a widget.
       *
       * @param widget - The widget of interest.
       *
       * @param value - The value for the horizontal alignment.
       *
       * #### Notes
       * If the layout width allocated to a widget is larger than its max
       * width, the horizontal alignment controls how the widget is placed
       * within the extra horizontal space.
       *
       * If the allocated width is less than the widget's max width, the
       * horizontal alignment has no effect.
       *
       * Some layout implementations may ignore horizontal alignment.
       *
       * Changing the horizontal alignment will post an `update-request`
       * message to widget's parent, provided the parent has a layout
       * installed.
       */
      function setHorizontalAlignment(widget, value) {
          Private$1$2.horizontalAlignmentProperty.set(widget, value);
      }
      Layout.setHorizontalAlignment = setHorizontalAlignment;
      /**
       * Get the vertical alignment for a widget.
       *
       * @param widget - The widget of interest.
       *
       * @returns The vertical alignment for the widget.
       *
       * #### Notes
       * If the layout height allocated to a widget is larger than its max
       * height, the vertical alignment controls how the widget is placed
       * within the extra vertical space.
       *
       * If the allocated height is less than the widget's max height, the
       * vertical alignment has no effect.
       *
       * Some layout implementations may ignore vertical alignment.
       */
      function getVerticalAlignment(widget) {
          return Private$1$2.verticalAlignmentProperty.get(widget);
      }
      Layout.getVerticalAlignment = getVerticalAlignment;
      /**
       * Set the vertical alignment for a widget.
       *
       * @param widget - The widget of interest.
       *
       * @param value - The value for the vertical alignment.
       *
       * #### Notes
       * If the layout height allocated to a widget is larger than its max
       * height, the vertical alignment controls how the widget is placed
       * within the extra vertical space.
       *
       * If the allocated height is less than the widget's max height, the
       * vertical alignment has no effect.
       *
       * Some layout implementations may ignore vertical alignment.
       *
       * Changing the horizontal alignment will post an `update-request`
       * message to widget's parent, provided the parent has a layout
       * installed.
       */
      function setVerticalAlignment(widget, value) {
          Private$1$2.verticalAlignmentProperty.set(widget, value);
      }
      Layout.setVerticalAlignment = setVerticalAlignment;
  })(Layout || (Layout = {}));
  /**
   * An object which assists in the absolute layout of widgets.
   *
   * #### Notes
   * This class is useful when implementing a layout which arranges its
   * widgets using absolute positioning.
   *
   * This class is used by nearly all of the built-in lumino layouts.
   */
  var LayoutItem = /** @class */ (function () {
      /**
       * Construct a new layout item.
       *
       * @param widget - The widget to be managed by the item.
       *
       * #### Notes
       * The widget will be set to absolute positioning.
       */
      function LayoutItem(widget) {
          this._top = NaN;
          this._left = NaN;
          this._width = NaN;
          this._height = NaN;
          this._minWidth = 0;
          this._minHeight = 0;
          this._maxWidth = Infinity;
          this._maxHeight = Infinity;
          this._disposed = false;
          this.widget = widget;
          this.widget.node.style.position = 'absolute';
      }
      /**
       * Dispose of the the layout item.
       *
       * #### Notes
       * This will reset the positioning of the widget.
       */
      LayoutItem.prototype.dispose = function () {
          // Do nothing if the item is already disposed.
          if (this._disposed) {
              return;
          }
          // Mark the item as disposed.
          this._disposed = true;
          // Reset the widget style.
          var style = this.widget.node.style;
          style.position = '';
          style.top = '';
          style.left = '';
          style.width = '';
          style.height = '';
      };
      Object.defineProperty(LayoutItem.prototype, "minWidth", {
          /**
           * The computed minimum width of the widget.
           *
           * #### Notes
           * This value can be updated by calling the `fit` method.
           */
          get: function () {
              return this._minWidth;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(LayoutItem.prototype, "minHeight", {
          /**
           * The computed minimum height of the widget.
           *
           * #### Notes
           * This value can be updated by calling the `fit` method.
           */
          get: function () {
              return this._minHeight;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(LayoutItem.prototype, "maxWidth", {
          /**
           * The computed maximum width of the widget.
           *
           * #### Notes
           * This value can be updated by calling the `fit` method.
           */
          get: function () {
              return this._maxWidth;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(LayoutItem.prototype, "maxHeight", {
          /**
           * The computed maximum height of the widget.
           *
           * #### Notes
           * This value can be updated by calling the `fit` method.
           */
          get: function () {
              return this._maxHeight;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(LayoutItem.prototype, "isDisposed", {
          /**
           * Whether the layout item is disposed.
           */
          get: function () {
              return this._disposed;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(LayoutItem.prototype, "isHidden", {
          /**
           * Whether the managed widget is hidden.
           */
          get: function () {
              return this.widget.isHidden;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(LayoutItem.prototype, "isVisible", {
          /**
           * Whether the managed widget is visible.
           */
          get: function () {
              return this.widget.isVisible;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(LayoutItem.prototype, "isAttached", {
          /**
           * Whether the managed widget is attached.
           */
          get: function () {
              return this.widget.isAttached;
          },
          enumerable: true,
          configurable: true
      });
      /**
       * Update the computed size limits of the managed widget.
       */
      LayoutItem.prototype.fit = function () {
          var limits = ElementExt.sizeLimits(this.widget.node);
          this._minWidth = limits.minWidth;
          this._minHeight = limits.minHeight;
          this._maxWidth = limits.maxWidth;
          this._maxHeight = limits.maxHeight;
      };
      /**
       * Update the position and size of the managed widget.
       *
       * @param left - The left edge position of the layout box.
       *
       * @param top - The top edge position of the layout box.
       *
       * @param width - The width of the layout box.
       *
       * @param height - The height of the layout box.
       */
      LayoutItem.prototype.update = function (left, top, width, height) {
          // Clamp the size to the computed size limits.
          var clampW = Math.max(this._minWidth, Math.min(width, this._maxWidth));
          var clampH = Math.max(this._minHeight, Math.min(height, this._maxHeight));
          // Adjust the left edge for the horizontal alignment, if needed.
          if (clampW < width) {
              switch (Layout.getHorizontalAlignment(this.widget)) {
                  case 'left':
                      break;
                  case 'center':
                      left += (width - clampW) / 2;
                      break;
                  case 'right':
                      left += width - clampW;
                      break;
                  default:
                      throw 'unreachable';
              }
          }
          // Adjust the top edge for the vertical alignment, if needed.
          if (clampH < height) {
              switch (Layout.getVerticalAlignment(this.widget)) {
                  case 'top':
                      break;
                  case 'center':
                      top += (height - clampH) / 2;
                      break;
                  case 'bottom':
                      top += height - clampH;
                      break;
                  default:
                      throw 'unreachable';
              }
          }
          // Set up the resize variables.
          var resized = false;
          var style = this.widget.node.style;
          // Update the top edge of the widget if needed.
          if (this._top !== top) {
              this._top = top;
              style.top = top + "px";
          }
          // Update the left edge of the widget if needed.
          if (this._left !== left) {
              this._left = left;
              style.left = left + "px";
          }
          // Update the width of the widget if needed.
          if (this._width !== clampW) {
              resized = true;
              this._width = clampW;
              style.width = clampW + "px";
          }
          // Update the height of the widget if needed.
          if (this._height !== clampH) {
              resized = true;
              this._height = clampH;
              style.height = clampH + "px";
          }
          // Send a resize message to the widget if needed.
          if (resized) {
              var msg = new Widget.ResizeMessage(clampW, clampH);
              MessageLoop.sendMessage(this.widget, msg);
          }
      };
      return LayoutItem;
  }());
  /**
   * The namespace for the module implementation details.
   */
  var Private$1$2;
  (function (Private) {
      /**
       * The attached property for a widget horizontal alignment.
       */
      Private.horizontalAlignmentProperty = new AttachedProperty({
          name: 'horizontalAlignment',
          create: function () { return 'center'; },
          changed: onAlignmentChanged
      });
      /**
       * The attached property for a widget vertical alignment.
       */
      Private.verticalAlignmentProperty = new AttachedProperty({
          name: 'verticalAlignment',
          create: function () { return 'top'; },
          changed: onAlignmentChanged
      });
      /**
       * The change handler for the attached alignment properties.
       */
      function onAlignmentChanged(child) {
          if (child.parent && child.parent.layout) {
              child.parent.update();
          }
      }
  })(Private$1$2 || (Private$1$2 = {}));

  /**
   * A concrete layout implementation suitable for many use cases.
   *
   * #### Notes
   * This class is suitable as a base class for implementing a variety of
   * layouts, but can also be used directly with standard CSS to layout a
   * collection of widgets.
   */
  var PanelLayout = /** @class */ (function (_super) {
      __extends$5(PanelLayout, _super);
      function PanelLayout() {
          var _this = _super !== null && _super.apply(this, arguments) || this;
          _this._widgets = [];
          return _this;
      }
      /**
       * Dispose of the resources held by the layout.
       *
       * #### Notes
       * This will clear and dispose all widgets in the layout.
       *
       * All reimplementations should call the superclass method.
       *
       * This method is called automatically when the parent is disposed.
       */
      PanelLayout.prototype.dispose = function () {
          while (this._widgets.length > 0) {
              this._widgets.pop().dispose();
          }
          _super.prototype.dispose.call(this);
      };
      Object.defineProperty(PanelLayout.prototype, "widgets", {
          /**
           * A read-only array of the widgets in the layout.
           */
          get: function () {
              return this._widgets;
          },
          enumerable: true,
          configurable: true
      });
      /**
       * Create an iterator over the widgets in the layout.
       *
       * @returns A new iterator over the widgets in the layout.
       */
      PanelLayout.prototype.iter = function () {
          return iter(this._widgets);
      };
      /**
       * Add a widget to the end of the layout.
       *
       * @param widget - The widget to add to the layout.
       *
       * #### Notes
       * If the widget is already contained in the layout, it will be moved.
       */
      PanelLayout.prototype.addWidget = function (widget) {
          this.insertWidget(this._widgets.length, widget);
      };
      /**
       * Insert a widget into the layout at the specified index.
       *
       * @param index - The index at which to insert the widget.
       *
       * @param widget - The widget to insert into the layout.
       *
       * #### Notes
       * The index will be clamped to the bounds of the widgets.
       *
       * If the widget is already added to the layout, it will be moved.
       *
       * #### Undefined Behavior
       * An `index` which is non-integral.
       */
      PanelLayout.prototype.insertWidget = function (index, widget) {
          // Remove the widget from its current parent. This is a no-op
          // if the widget's parent is already the layout parent widget.
          widget.parent = this.parent;
          // Look up the current index of the widget.
          var i = this._widgets.indexOf(widget);
          // Clamp the insert index to the array bounds.
          var j = Math.max(0, Math.min(index, this._widgets.length));
          // If the widget is not in the array, insert it.
          if (i === -1) {
              // Insert the widget into the array.
              ArrayExt.insert(this._widgets, j, widget);
              // If the layout is parented, attach the widget to the DOM.
              if (this.parent) {
                  this.attachWidget(j, widget);
              }
              // There is nothing more to do.
              return;
          }
          // Otherwise, the widget exists in the array and should be moved.
          // Adjust the index if the location is at the end of the array.
          if (j === this._widgets.length) {
              j--;
          }
          // Bail if there is no effective move.
          if (i === j) {
              return;
          }
          // Move the widget to the new location.
          ArrayExt.move(this._widgets, i, j);
          // If the layout is parented, move the widget in the DOM.
          if (this.parent) {
              this.moveWidget(i, j, widget);
          }
      };
      /**
       * Remove a widget from the layout.
       *
       * @param widget - The widget to remove from the layout.
       *
       * #### Notes
       * A widget is automatically removed from the layout when its `parent`
       * is set to `null`. This method should only be invoked directly when
       * removing a widget from a layout which has yet to be installed on a
       * parent widget.
       *
       * This method does *not* modify the widget's `parent`.
       */
      PanelLayout.prototype.removeWidget = function (widget) {
          this.removeWidgetAt(this._widgets.indexOf(widget));
      };
      /**
       * Remove the widget at a given index from the layout.
       *
       * @param index - The index of the widget to remove.
       *
       * #### Notes
       * A widget is automatically removed from the layout when its `parent`
       * is set to `null`. This method should only be invoked directly when
       * removing a widget from a layout which has yet to be installed on a
       * parent widget.
       *
       * This method does *not* modify the widget's `parent`.
       *
       * #### Undefined Behavior
       * An `index` which is non-integral.
       */
      PanelLayout.prototype.removeWidgetAt = function (index) {
          // Remove the widget from the array.
          var widget = ArrayExt.removeAt(this._widgets, index);
          // If the layout is parented, detach the widget from the DOM.
          if (widget && this.parent) {
              this.detachWidget(index, widget);
          }
      };
      /**
       * Perform layout initialization which requires the parent widget.
       */
      PanelLayout.prototype.init = function () {
          var _this = this;
          _super.prototype.init.call(this);
          each$1(this, function (widget, index) {
              _this.attachWidget(index, widget);
          });
      };
      /**
       * Attach a widget to the parent's DOM node.
       *
       * @param index - The current index of the widget in the layout.
       *
       * @param widget - The widget to attach to the parent.
       *
       * #### Notes
       * This method is called automatically by the panel layout at the
       * appropriate time. It should not be called directly by user code.
       *
       * The default implementation adds the widgets's node to the parent's
       * node at the proper location, and sends the appropriate attach
       * messages to the widget if the parent is attached to the DOM.
       *
       * Subclasses may reimplement this method to control how the widget's
       * node is added to the parent's node.
       */
      PanelLayout.prototype.attachWidget = function (index, widget) {
          // Look up the next sibling reference node.
          var ref = this.parent.node.children[index];
          // Send a `'before-attach'` message if the parent is attached.
          if (this.parent.isAttached) {
              MessageLoop.sendMessage(widget, Widget.Msg.BeforeAttach);
          }
          // Insert the widget's node before the sibling.
          this.parent.node.insertBefore(widget.node, ref);
          // Send an `'after-attach'` message if the parent is attached.
          if (this.parent.isAttached) {
              MessageLoop.sendMessage(widget, Widget.Msg.AfterAttach);
          }
      };
      /**
       * Move a widget in the parent's DOM node.
       *
       * @param fromIndex - The previous index of the widget in the layout.
       *
       * @param toIndex - The current index of the widget in the layout.
       *
       * @param widget - The widget to move in the parent.
       *
       * #### Notes
       * This method is called automatically by the panel layout at the
       * appropriate time. It should not be called directly by user code.
       *
       * The default implementation moves the widget's node to the proper
       * location in the parent's node and sends the appropriate attach and
       * detach messages to the widget if the parent is attached to the DOM.
       *
       * Subclasses may reimplement this method to control how the widget's
       * node is moved in the parent's node.
       */
      PanelLayout.prototype.moveWidget = function (fromIndex, toIndex, widget) {
          // Send a `'before-detach'` message if the parent is attached.
          if (this.parent.isAttached) {
              MessageLoop.sendMessage(widget, Widget.Msg.BeforeDetach);
          }
          // Remove the widget's node from the parent.
          this.parent.node.removeChild(widget.node);
          // Send an `'after-detach'` and  message if the parent is attached.
          if (this.parent.isAttached) {
              MessageLoop.sendMessage(widget, Widget.Msg.AfterDetach);
          }
          // Look up the next sibling reference node.
          var ref = this.parent.node.children[toIndex];
          // Send a `'before-attach'` message if the parent is attached.
          if (this.parent.isAttached) {
              MessageLoop.sendMessage(widget, Widget.Msg.BeforeAttach);
          }
          // Insert the widget's node before the sibling.
          this.parent.node.insertBefore(widget.node, ref);
          // Send an `'after-attach'` message if the parent is attached.
          if (this.parent.isAttached) {
              MessageLoop.sendMessage(widget, Widget.Msg.AfterAttach);
          }
      };
      /**
       * Detach a widget from the parent's DOM node.
       *
       * @param index - The previous index of the widget in the layout.
       *
       * @param widget - The widget to detach from the parent.
       *
       * #### Notes
       * This method is called automatically by the panel layout at the
       * appropriate time. It should not be called directly by user code.
       *
       * The default implementation removes the widget's node from the
       * parent's node, and sends the appropriate detach messages to the
       * widget if the parent is attached to the DOM.
       *
       * Subclasses may reimplement this method to control how the widget's
       * node is removed from the parent's node.
       */
      PanelLayout.prototype.detachWidget = function (index, widget) {
          // Send a `'before-detach'` message if the parent is attached.
          if (this.parent.isAttached) {
              MessageLoop.sendMessage(widget, Widget.Msg.BeforeDetach);
          }
          // Remove the widget's node from the parent.
          this.parent.node.removeChild(widget.node);
          // Send an `'after-detach'` message if the parent is attached.
          if (this.parent.isAttached) {
              MessageLoop.sendMessage(widget, Widget.Msg.AfterDetach);
          }
      };
      return PanelLayout;
  }(Layout));

  /**
   * A layout which arranges its widgets in a single row or column.
   */
  var BoxLayout = /** @class */ (function (_super) {
      __extends$5(BoxLayout, _super);
      /**
       * Construct a new box layout.
       *
       * @param options - The options for initializing the layout.
       */
      function BoxLayout(options) {
          if (options === void 0) { options = {}; }
          var _this = _super.call(this) || this;
          _this._fixed = 0;
          _this._spacing = 4;
          _this._dirty = false;
          _this._sizers = [];
          _this._items = [];
          _this._box = null;
          _this._alignment = 'start';
          _this._direction = 'top-to-bottom';
          if (options.direction !== undefined) {
              _this._direction = options.direction;
          }
          if (options.alignment !== undefined) {
              _this._alignment = options.alignment;
          }
          if (options.spacing !== undefined) {
              _this._spacing = Private$2$1.clampSpacing(options.spacing);
          }
          return _this;
      }
      /**
       * Dispose of the resources held by the layout.
       */
      BoxLayout.prototype.dispose = function () {
          // Dispose of the layout items.
          each$1(this._items, function (item) { item.dispose(); });
          // Clear the layout state.
          this._box = null;
          this._items.length = 0;
          this._sizers.length = 0;
          // Dispose of the rest of the layout.
          _super.prototype.dispose.call(this);
      };
      Object.defineProperty(BoxLayout.prototype, "direction", {
          /**
           * Get the layout direction for the box layout.
           */
          get: function () {
              return this._direction;
          },
          /**
           * Set the layout direction for the box layout.
           */
          set: function (value) {
              if (this._direction === value) {
                  return;
              }
              this._direction = value;
              if (!this.parent) {
                  return;
              }
              this.parent.dataset['direction'] = value;
              this.parent.fit();
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(BoxLayout.prototype, "alignment", {
          /**
           * Get the content alignment for the box layout.
           *
           * #### Notes
           * This is the alignment of the widgets in the layout direction.
           *
           * The alignment has no effect if the widgets can expand to fill the
           * entire box layout.
           */
          get: function () {
              return this._alignment;
          },
          /**
           * Set the content alignment for the box layout.
           *
           * #### Notes
           * This is the alignment of the widgets in the layout direction.
           *
           * The alignment has no effect if the widgets can expand to fill the
           * entire box layout.
           */
          set: function (value) {
              if (this._alignment === value) {
                  return;
              }
              this._alignment = value;
              if (!this.parent) {
                  return;
              }
              this.parent.dataset['alignment'] = value;
              this.parent.update();
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(BoxLayout.prototype, "spacing", {
          /**
           * Get the inter-element spacing for the box layout.
           */
          get: function () {
              return this._spacing;
          },
          /**
           * Set the inter-element spacing for the box layout.
           */
          set: function (value) {
              value = Private$2$1.clampSpacing(value);
              if (this._spacing === value) {
                  return;
              }
              this._spacing = value;
              if (!this.parent) {
                  return;
              }
              this.parent.fit();
          },
          enumerable: true,
          configurable: true
      });
      /**
       * Perform layout initialization which requires the parent widget.
       */
      BoxLayout.prototype.init = function () {
          this.parent.dataset['direction'] = this.direction;
          this.parent.dataset['alignment'] = this.alignment;
          _super.prototype.init.call(this);
      };
      /**
       * Attach a widget to the parent's DOM node.
       *
       * @param index - The current index of the widget in the layout.
       *
       * @param widget - The widget to attach to the parent.
       *
       * #### Notes
       * This is a reimplementation of the superclass method.
       */
      BoxLayout.prototype.attachWidget = function (index, widget) {
          // Create and add a new layout item for the widget.
          ArrayExt.insert(this._items, index, new LayoutItem(widget));
          // Create and add a new sizer for the widget.
          ArrayExt.insert(this._sizers, index, new BoxSizer());
          // Send a `'before-attach'` message if the parent is attached.
          if (this.parent.isAttached) {
              MessageLoop.sendMessage(widget, Widget.Msg.BeforeAttach);
          }
          // Add the widget's node to the parent.
          this.parent.node.appendChild(widget.node);
          // Send an `'after-attach'` message if the parent is attached.
          if (this.parent.isAttached) {
              MessageLoop.sendMessage(widget, Widget.Msg.AfterAttach);
          }
          // Post a fit request for the parent widget.
          this.parent.fit();
      };
      /**
       * Move a widget in the parent's DOM node.
       *
       * @param fromIndex - The previous index of the widget in the layout.
       *
       * @param toIndex - The current index of the widget in the layout.
       *
       * @param widget - The widget to move in the parent.
       *
       * #### Notes
       * This is a reimplementation of the superclass method.
       */
      BoxLayout.prototype.moveWidget = function (fromIndex, toIndex, widget) {
          // Move the layout item for the widget.
          ArrayExt.move(this._items, fromIndex, toIndex);
          // Move the sizer for the widget.
          ArrayExt.move(this._sizers, fromIndex, toIndex);
          // Post an update request for the parent widget.
          this.parent.update();
      };
      /**
       * Detach a widget from the parent's DOM node.
       *
       * @param index - The previous index of the widget in the layout.
       *
       * @param widget - The widget to detach from the parent.
       *
       * #### Notes
       * This is a reimplementation of the superclass method.
       */
      BoxLayout.prototype.detachWidget = function (index, widget) {
          // Remove the layout item for the widget.
          var item = ArrayExt.removeAt(this._items, index);
          // Remove the sizer for the widget.
          ArrayExt.removeAt(this._sizers, index);
          // Send a `'before-detach'` message if the parent is attached.
          if (this.parent.isAttached) {
              MessageLoop.sendMessage(widget, Widget.Msg.BeforeDetach);
          }
          // Remove the widget's node from the parent.
          this.parent.node.removeChild(widget.node);
          // Send an `'after-detach'` message if the parent is attached.
          if (this.parent.isAttached) {
              MessageLoop.sendMessage(widget, Widget.Msg.AfterDetach);
          }
          // Dispose of the layout item.
          item.dispose();
          // Post a fit request for the parent widget.
          this.parent.fit();
      };
      /**
       * A message handler invoked on a `'before-show'` message.
       */
      BoxLayout.prototype.onBeforeShow = function (msg) {
          _super.prototype.onBeforeShow.call(this, msg);
          this.parent.update();
      };
      /**
       * A message handler invoked on a `'before-attach'` message.
       */
      BoxLayout.prototype.onBeforeAttach = function (msg) {
          _super.prototype.onBeforeAttach.call(this, msg);
          this.parent.fit();
      };
      /**
       * A message handler invoked on a `'child-shown'` message.
       */
      BoxLayout.prototype.onChildShown = function (msg) {
          this.parent.fit();
      };
      /**
       * A message handler invoked on a `'child-hidden'` message.
       */
      BoxLayout.prototype.onChildHidden = function (msg) {
          this.parent.fit();
      };
      /**
       * A message handler invoked on a `'resize'` message.
       */
      BoxLayout.prototype.onResize = function (msg) {
          if (this.parent.isVisible) {
              this._update(msg.width, msg.height);
          }
      };
      /**
       * A message handler invoked on an `'update-request'` message.
       */
      BoxLayout.prototype.onUpdateRequest = function (msg) {
          if (this.parent.isVisible) {
              this._update(-1, -1);
          }
      };
      /**
       * A message handler invoked on a `'fit-request'` message.
       */
      BoxLayout.prototype.onFitRequest = function (msg) {
          if (this.parent.isAttached) {
              this._fit();
          }
      };
      /**
       * Fit the layout to the total size required by the widgets.
       */
      BoxLayout.prototype._fit = function () {
          // Compute the visible item count.
          var nVisible = 0;
          for (var i = 0, n = this._items.length; i < n; ++i) {
              nVisible += +!this._items[i].isHidden;
          }
          // Update the fixed space for the visible items.
          this._fixed = this._spacing * Math.max(0, nVisible - 1);
          // Setup the computed minimum size.
          var horz = Private$2$1.isHorizontal(this._direction);
          var minW = horz ? this._fixed : 0;
          var minH = horz ? 0 : this._fixed;
          // Update the sizers and computed minimum size.
          for (var i = 0, n = this._items.length; i < n; ++i) {
              // Fetch the item and corresponding box sizer.
              var item = this._items[i];
              var sizer = this._sizers[i];
              // If the item is hidden, it should consume zero size.
              if (item.isHidden) {
                  sizer.minSize = 0;
                  sizer.maxSize = 0;
                  continue;
              }
              // Update the size limits for the item.
              item.fit();
              // Update the size basis and stretch factor.
              sizer.sizeHint = BoxLayout.getSizeBasis(item.widget);
              sizer.stretch = BoxLayout.getStretch(item.widget);
              // Update the sizer limits and computed min size.
              if (horz) {
                  sizer.minSize = item.minWidth;
                  sizer.maxSize = item.maxWidth;
                  minW += item.minWidth;
                  minH = Math.max(minH, item.minHeight);
              }
              else {
                  sizer.minSize = item.minHeight;
                  sizer.maxSize = item.maxHeight;
                  minH += item.minHeight;
                  minW = Math.max(minW, item.minWidth);
              }
          }
          // Update the box sizing and add it to the computed min size.
          var box = this._box = ElementExt.boxSizing(this.parent.node);
          minW += box.horizontalSum;
          minH += box.verticalSum;
          // Update the parent's min size constraints.
          var style = this.parent.node.style;
          style.minWidth = minW + "px";
          style.minHeight = minH + "px";
          // Set the dirty flag to ensure only a single update occurs.
          this._dirty = true;
          // Notify the ancestor that it should fit immediately. This may
          // cause a resize of the parent, fulfilling the required update.
          if (this.parent.parent) {
              MessageLoop.sendMessage(this.parent.parent, Widget.Msg.FitRequest);
          }
          // If the dirty flag is still set, the parent was not resized.
          // Trigger the required update on the parent widget immediately.
          if (this._dirty) {
              MessageLoop.sendMessage(this.parent, Widget.Msg.UpdateRequest);
          }
      };
      /**
       * Update the layout position and size of the widgets.
       *
       * The parent offset dimensions should be `-1` if unknown.
       */
      BoxLayout.prototype._update = function (offsetWidth, offsetHeight) {
          // Clear the dirty flag to indicate the update occurred.
          this._dirty = false;
          // Compute the visible item count.
          var nVisible = 0;
          for (var i = 0, n = this._items.length; i < n; ++i) {
              nVisible += +!this._items[i].isHidden;
          }
          // Bail early if there are no visible items to layout.
          if (nVisible === 0) {
              return;
          }
          // Measure the parent if the offset dimensions are unknown.
          if (offsetWidth < 0) {
              offsetWidth = this.parent.node.offsetWidth;
          }
          if (offsetHeight < 0) {
              offsetHeight = this.parent.node.offsetHeight;
          }
          // Ensure the parent box sizing data is computed.
          if (!this._box) {
              this._box = ElementExt.boxSizing(this.parent.node);
          }
          // Compute the layout area adjusted for border and padding.
          var top = this._box.paddingTop;
          var left = this._box.paddingLeft;
          var width = offsetWidth - this._box.horizontalSum;
          var height = offsetHeight - this._box.verticalSum;
          // Distribute the layout space and adjust the start position.
          var delta;
          switch (this._direction) {
              case 'left-to-right':
                  delta = BoxEngine.calc(this._sizers, Math.max(0, width - this._fixed));
                  break;
              case 'top-to-bottom':
                  delta = BoxEngine.calc(this._sizers, Math.max(0, height - this._fixed));
                  break;
              case 'right-to-left':
                  delta = BoxEngine.calc(this._sizers, Math.max(0, width - this._fixed));
                  left += width;
                  break;
              case 'bottom-to-top':
                  delta = BoxEngine.calc(this._sizers, Math.max(0, height - this._fixed));
                  top += height;
                  break;
              default:
                  throw 'unreachable';
          }
          // Setup the variables for justification and alignment offset.
          var extra = 0;
          var offset = 0;
          // Account for alignment if there is extra layout space.
          if (delta > 0) {
              switch (this._alignment) {
                  case 'start':
                      break;
                  case 'center':
                      extra = 0;
                      offset = delta / 2;
                      break;
                  case 'end':
                      extra = 0;
                      offset = delta;
                      break;
                  case 'justify':
                      extra = delta / nVisible;
                      offset = 0;
                      break;
                  default:
                      throw 'unreachable';
              }
          }
          // Layout the items using the computed box sizes.
          for (var i = 0, n = this._items.length; i < n; ++i) {
              // Fetch the item.
              var item = this._items[i];
              // Ignore hidden items.
              if (item.isHidden) {
                  continue;
              }
              // Fetch the computed size for the widget.
              var size = this._sizers[i].size;
              // Update the widget geometry and advance the relevant edge.
              switch (this._direction) {
                  case 'left-to-right':
                      item.update(left + offset, top, size + extra, height);
                      left += size + extra + this._spacing;
                      break;
                  case 'top-to-bottom':
                      item.update(left, top + offset, width, size + extra);
                      top += size + extra + this._spacing;
                      break;
                  case 'right-to-left':
                      item.update(left - offset - size - extra, top, size + extra, height);
                      left -= size + extra + this._spacing;
                      break;
                  case 'bottom-to-top':
                      item.update(left, top - offset - size - extra, width, size + extra);
                      top -= size + extra + this._spacing;
                      break;
                  default:
                      throw 'unreachable';
              }
          }
      };
      return BoxLayout;
  }(PanelLayout));
  /**
   * The namespace for the `BoxLayout` class statics.
   */
  (function (BoxLayout) {
      /**
       * Get the box layout stretch factor for the given widget.
       *
       * @param widget - The widget of interest.
       *
       * @returns The box layout stretch factor for the widget.
       */
      function getStretch(widget) {
          return Private$2$1.stretchProperty.get(widget);
      }
      BoxLayout.getStretch = getStretch;
      /**
       * Set the box layout stretch factor for the given widget.
       *
       * @param widget - The widget of interest.
       *
       * @param value - The value for the stretch factor.
       */
      function setStretch(widget, value) {
          Private$2$1.stretchProperty.set(widget, value);
      }
      BoxLayout.setStretch = setStretch;
      /**
       * Get the box layout size basis for the given widget.
       *
       * @param widget - The widget of interest.
       *
       * @returns The box layout size basis for the widget.
       */
      function getSizeBasis(widget) {
          return Private$2$1.sizeBasisProperty.get(widget);
      }
      BoxLayout.getSizeBasis = getSizeBasis;
      /**
       * Set the box layout size basis for the given widget.
       *
       * @param widget - The widget of interest.
       *
       * @param value - The value for the size basis.
       */
      function setSizeBasis(widget, value) {
          Private$2$1.sizeBasisProperty.set(widget, value);
      }
      BoxLayout.setSizeBasis = setSizeBasis;
  })(BoxLayout || (BoxLayout = {}));
  /**
   * The namespace for the module implementation details.
   */
  var Private$2$1;
  (function (Private) {
      /**
       * The property descriptor for a widget stretch factor.
       */
      Private.stretchProperty = new AttachedProperty({
          name: 'stretch',
          create: function () { return 0; },
          coerce: function (owner, value) { return Math.max(0, Math.floor(value)); },
          changed: onChildSizingChanged
      });
      /**
       * The property descriptor for a widget size basis.
       */
      Private.sizeBasisProperty = new AttachedProperty({
          name: 'sizeBasis',
          create: function () { return 0; },
          coerce: function (owner, value) { return Math.max(0, Math.floor(value)); },
          changed: onChildSizingChanged
      });
      /**
       * Test whether a direction has horizontal orientation.
       */
      function isHorizontal(dir) {
          return dir === 'left-to-right' || dir === 'right-to-left';
      }
      Private.isHorizontal = isHorizontal;
      /**
       * Clamp a spacing value to an integer >= 0.
       */
      function clampSpacing(value) {
          return Math.max(0, Math.floor(value));
      }
      Private.clampSpacing = clampSpacing;
      /**
       * The change handler for the attached sizing properties.
       */
      function onChildSizingChanged(child) {
          if (child.parent && child.parent.layout instanceof BoxLayout) {
              child.parent.fit();
          }
      }
  })(Private$2$1 || (Private$2$1 = {}));

  /**
   * A simple and convenient panel widget class.
   *
   * #### Notes
   * This class is suitable as a base class for implementing a variety of
   * convenience panel widgets, but can also be used directly with CSS to
   * arrange a collection of widgets.
   *
   * This class provides a convenience wrapper around a [[PanelLayout]].
   */
  var Panel = /** @class */ (function (_super) {
      __extends$5(Panel, _super);
      /**
       * Construct a new panel.
       *
       * @param options - The options for initializing the panel.
       */
      function Panel(options) {
          if (options === void 0) { options = {}; }
          var _this = _super.call(this) || this;
          _this.addClass('lm-Panel');
          /* <DEPRECATED> */
          _this.addClass('p-Panel');
          /* </DEPRECATED> */
          _this.layout = Private$3$1.createLayout(options);
          return _this;
      }
      Object.defineProperty(Panel.prototype, "widgets", {
          /**
           * A read-only array of the widgets in the panel.
           */
          get: function () {
              return this.layout.widgets;
          },
          enumerable: true,
          configurable: true
      });
      /**
       * Add a widget to the end of the panel.
       *
       * @param widget - The widget to add to the panel.
       *
       * #### Notes
       * If the widget is already contained in the panel, it will be moved.
       */
      Panel.prototype.addWidget = function (widget) {
          this.layout.addWidget(widget);
      };
      /**
       * Insert a widget at the specified index.
       *
       * @param index - The index at which to insert the widget.
       *
       * @param widget - The widget to insert into to the panel.
       *
       * #### Notes
       * If the widget is already contained in the panel, it will be moved.
       */
      Panel.prototype.insertWidget = function (index, widget) {
          this.layout.insertWidget(index, widget);
      };
      return Panel;
  }(Widget));
  /**
   * The namespace for the module implementation details.
   */
  var Private$3$1;
  (function (Private) {
      /**
       * Create a panel layout for the given panel options.
       */
      function createLayout(options) {
          return options.layout || new PanelLayout();
      }
      Private.createLayout = createLayout;
  })(Private$3$1 || (Private$3$1 = {}));

  /**
   * A panel which arranges its widgets in a single row or column.
   *
   * #### Notes
   * This class provides a convenience wrapper around a [[BoxLayout]].
   */
  var BoxPanel = /** @class */ (function (_super) {
      __extends$5(BoxPanel, _super);
      /**
       * Construct a new box panel.
       *
       * @param options - The options for initializing the box panel.
       */
      function BoxPanel(options) {
          if (options === void 0) { options = {}; }
          var _this = _super.call(this, { layout: Private$4$1.createLayout(options) }) || this;
          _this.addClass('lm-BoxPanel');
          /* <DEPRECATED> */
          _this.addClass('p-BoxPanel');
          return _this;
          /* </DEPRECATED> */
      }
      Object.defineProperty(BoxPanel.prototype, "direction", {
          /**
           * Get the layout direction for the box panel.
           */
          get: function () {
              return this.layout.direction;
          },
          /**
           * Set the layout direction for the box panel.
           */
          set: function (value) {
              this.layout.direction = value;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(BoxPanel.prototype, "alignment", {
          /**
           * Get the content alignment for the box panel.
           *
           * #### Notes
           * This is the alignment of the widgets in the layout direction.
           *
           * The alignment has no effect if the widgets can expand to fill the
           * entire box layout.
           */
          get: function () {
              return this.layout.alignment;
          },
          /**
           * Set the content alignment for the box panel.
           *
           * #### Notes
           * This is the alignment of the widgets in the layout direction.
           *
           * The alignment has no effect if the widgets can expand to fill the
           * entire box layout.
           */
          set: function (value) {
              this.layout.alignment = value;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(BoxPanel.prototype, "spacing", {
          /**
           * Get the inter-element spacing for the box panel.
           */
          get: function () {
              return this.layout.spacing;
          },
          /**
           * Set the inter-element spacing for the box panel.
           */
          set: function (value) {
              this.layout.spacing = value;
          },
          enumerable: true,
          configurable: true
      });
      /**
       * A message handler invoked on a `'child-added'` message.
       */
      BoxPanel.prototype.onChildAdded = function (msg) {
          msg.child.addClass('lm-BoxPanel-child');
          /* <DEPRECATED> */
          msg.child.addClass('p-BoxPanel-child');
          /* </DEPRECATED> */
      };
      /**
       * A message handler invoked on a `'child-removed'` message.
       */
      BoxPanel.prototype.onChildRemoved = function (msg) {
          msg.child.removeClass('lm-BoxPanel-child');
          /* <DEPRECATED> */
          msg.child.removeClass('p-BoxPanel-child');
          /* </DEPRECATED> */
      };
      return BoxPanel;
  }(Panel));
  /**
   * The namespace for the `BoxPanel` class statics.
   */
  (function (BoxPanel) {
      /**
       * Get the box panel stretch factor for the given widget.
       *
       * @param widget - The widget of interest.
       *
       * @returns The box panel stretch factor for the widget.
       */
      function getStretch(widget) {
          return BoxLayout.getStretch(widget);
      }
      BoxPanel.getStretch = getStretch;
      /**
       * Set the box panel stretch factor for the given widget.
       *
       * @param widget - The widget of interest.
       *
       * @param value - The value for the stretch factor.
       */
      function setStretch(widget, value) {
          BoxLayout.setStretch(widget, value);
      }
      BoxPanel.setStretch = setStretch;
      /**
       * Get the box panel size basis for the given widget.
       *
       * @param widget - The widget of interest.
       *
       * @returns The box panel size basis for the widget.
       */
      function getSizeBasis(widget) {
          return BoxLayout.getSizeBasis(widget);
      }
      BoxPanel.getSizeBasis = getSizeBasis;
      /**
       * Set the box panel size basis for the given widget.
       *
       * @param widget - The widget of interest.
       *
       * @param value - The value for the size basis.
       */
      function setSizeBasis(widget, value) {
          BoxLayout.setSizeBasis(widget, value);
      }
      BoxPanel.setSizeBasis = setSizeBasis;
  })(BoxPanel || (BoxPanel = {}));
  /**
   * The namespace for the module implementation details.
   */
  var Private$4$1;
  (function (Private) {
      /**
       * Create a box layout for the given panel options.
       */
      function createLayout(options) {
          return options.layout || new BoxLayout(options);
      }
      Private.createLayout = createLayout;
  })(Private$4$1 || (Private$4$1 = {}));

  /**
   * A widget which displays command items as a searchable palette.
   */
  var CommandPalette = /** @class */ (function (_super) {
      __extends$5(CommandPalette, _super);
      /**
       * Construct a new command palette.
       *
       * @param options - The options for initializing the palette.
       */
      function CommandPalette(options) {
          var _this = _super.call(this, { node: Private$5$1.createNode() }) || this;
          _this._activeIndex = -1;
          _this._items = [];
          _this._results = null;
          _this.addClass('lm-CommandPalette');
          /* <DEPRECATED> */
          _this.addClass('p-CommandPalette');
          /* </DEPRECATED> */
          _this.setFlag(Widget.Flag.DisallowLayout);
          _this.commands = options.commands;
          _this.renderer = options.renderer || CommandPalette.defaultRenderer;
          _this.commands.commandChanged.connect(_this._onGenericChange, _this);
          _this.commands.keyBindingChanged.connect(_this._onGenericChange, _this);
          return _this;
      }
      /**
       * Dispose of the resources held by the widget.
       */
      CommandPalette.prototype.dispose = function () {
          this._items.length = 0;
          this._results = null;
          _super.prototype.dispose.call(this);
      };
      Object.defineProperty(CommandPalette.prototype, "searchNode", {
          /**
           * The command palette search node.
           *
           * #### Notes
           * This is the node which contains the search-related elements.
           */
          get: function () {
              return this.node.getElementsByClassName('lm-CommandPalette-search')[0];
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(CommandPalette.prototype, "inputNode", {
          /**
           * The command palette input node.
           *
           * #### Notes
           * This is the actual input node for the search area.
           */
          get: function () {
              return this.node.getElementsByClassName('lm-CommandPalette-input')[0];
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(CommandPalette.prototype, "contentNode", {
          /**
           * The command palette content node.
           *
           * #### Notes
           * This is the node which holds the command item nodes.
           *
           * Modifying this node directly can lead to undefined behavior.
           */
          get: function () {
              return this.node.getElementsByClassName('lm-CommandPalette-content')[0];
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(CommandPalette.prototype, "items", {
          /**
           * A read-only array of the command items in the palette.
           */
          get: function () {
              return this._items;
          },
          enumerable: true,
          configurable: true
      });
      /**
       * Add a command item to the command palette.
       *
       * @param options - The options for creating the command item.
       *
       * @returns The command item added to the palette.
       */
      CommandPalette.prototype.addItem = function (options) {
          // Create a new command item for the options.
          var item = Private$5$1.createItem(this.commands, options);
          // Add the item to the array.
          this._items.push(item);
          // Refresh the search results.
          this.refresh();
          // Return the item added to the palette.
          return item;
      };
      /**
       * Remove an item from the command palette.
       *
       * @param item - The item to remove from the palette.
       *
       * #### Notes
       * This is a no-op if the item is not in the palette.
       */
      CommandPalette.prototype.removeItem = function (item) {
          this.removeItemAt(this._items.indexOf(item));
      };
      /**
       * Remove the item at a given index from the command palette.
       *
       * @param index - The index of the item to remove.
       *
       * #### Notes
       * This is a no-op if the index is out of range.
       */
      CommandPalette.prototype.removeItemAt = function (index) {
          // Remove the item from the array.
          var item = ArrayExt.removeAt(this._items, index);
          // Bail if the index is out of range.
          if (!item) {
              return;
          }
          // Refresh the search results.
          this.refresh();
      };
      /**
       * Remove all items from the command palette.
       */
      CommandPalette.prototype.clearItems = function () {
          // Bail if there is nothing to remove.
          if (this._items.length === 0) {
              return;
          }
          // Clear the array of items.
          this._items.length = 0;
          // Refresh the search results.
          this.refresh();
      };
      /**
       * Clear the search results and schedule an update.
       *
       * #### Notes
       * This should be called whenever the search results of the palette
       * should be updated.
       *
       * This is typically called automatically by the palette as needed,
       * but can be called manually if the input text is programatically
       * changed.
       *
       * The rendered results are updated asynchronously.
       */
      CommandPalette.prototype.refresh = function () {
          this._results = null;
          if (this.inputNode.value !== '') {
              var clear = this.node.getElementsByClassName('lm-close-icon')[0];
              clear.style.display = 'inherit';
          }
          else {
              var clear = this.node.getElementsByClassName('lm-close-icon')[0];
              clear.style.display = 'none';
          }
          this.update();
      };
      /**
       * Handle the DOM events for the command palette.
       *
       * @param event - The DOM event sent to the command palette.
       *
       * #### Notes
       * This method implements the DOM `EventListener` interface and is
       * called in response to events on the command palette's DOM node.
       * It should not be called directly by user code.
       */
      CommandPalette.prototype.handleEvent = function (event) {
          switch (event.type) {
              case 'click':
                  this._evtClick(event);
                  break;
              case 'keydown':
                  this._evtKeyDown(event);
                  break;
              case 'input':
                  this.refresh();
                  break;
              case 'focus':
              case 'blur':
                  this._toggleFocused();
                  break;
          }
      };
      /**
       * A message handler invoked on a `'before-attach'` message.
       */
      CommandPalette.prototype.onBeforeAttach = function (msg) {
          this.node.addEventListener('click', this);
          this.node.addEventListener('keydown', this);
          this.node.addEventListener('input', this);
          this.node.addEventListener('focus', this, true);
          this.node.addEventListener('blur', this, true);
      };
      /**
       * A message handler invoked on an `'after-detach'` message.
       */
      CommandPalette.prototype.onAfterDetach = function (msg) {
          this.node.removeEventListener('click', this);
          this.node.removeEventListener('keydown', this);
          this.node.removeEventListener('input', this);
          this.node.removeEventListener('focus', this, true);
          this.node.removeEventListener('blur', this, true);
      };
      /**
       * A message handler invoked on an `'activate-request'` message.
       */
      CommandPalette.prototype.onActivateRequest = function (msg) {
          if (this.isAttached) {
              var input = this.inputNode;
              input.focus();
              input.select();
          }
      };
      /**
       * A message handler invoked on an `'update-request'` message.
       */
      CommandPalette.prototype.onUpdateRequest = function (msg) {
          // Fetch the current query text and content node.
          var query = this.inputNode.value;
          var contentNode = this.contentNode;
          // Ensure the search results are generated.
          var results = this._results;
          if (!results) {
              // Generate and store the new search results.
              results = this._results = Private$5$1.search(this._items, query);
              // Reset the active index.
              this._activeIndex = (query ? ArrayExt.findFirstIndex(results, Private$5$1.canActivate) : -1);
          }
          // If there is no query and no results, clear the content.
          if (!query && results.length === 0) {
              VirtualDOM.render(null, contentNode);
              return;
          }
          // If the is a query but no results, render the empty message.
          if (query && results.length === 0) {
              var content_1 = this.renderer.renderEmptyMessage({ query: query });
              VirtualDOM.render(content_1, contentNode);
              return;
          }
          // Create the render content for the search results.
          var renderer = this.renderer;
          var activeIndex = this._activeIndex;
          var content = new Array(results.length);
          for (var i = 0, n = results.length; i < n; ++i) {
              var result = results[i];
              if (result.type === 'header') {
                  var indices = result.indices;
                  var category = result.category;
                  content[i] = renderer.renderHeader({ category: category, indices: indices });
              }
              else {
                  var item = result.item;
                  var indices = result.indices;
                  var active = i === activeIndex;
                  content[i] = renderer.renderItem({ item: item, indices: indices, active: active });
              }
          }
          // Render the search result content.
          VirtualDOM.render(content, contentNode);
          // Adjust the scroll position as needed.
          if (activeIndex < 0 || activeIndex >= results.length) {
              contentNode.scrollTop = 0;
          }
          else {
              var element = contentNode.children[activeIndex];
              ElementExt.scrollIntoViewIfNeeded(contentNode, element);
          }
      };
      /**
       * Handle the `'click'` event for the command palette.
       */
      CommandPalette.prototype._evtClick = function (event) {
          // Bail if the click is not the left button.
          if (event.button !== 0) {
              return;
          }
          // Clear input if the target is clear button
          if (event.target.classList.contains("lm-close-icon")) {
              this.inputNode.value = '';
              this.refresh();
              return;
          }
          // Find the index of the item which was clicked.
          var index = ArrayExt.findFirstIndex(this.contentNode.children, function (node) {
              return node.contains(event.target);
          });
          // Bail if the click was not on an item.
          if (index === -1) {
              return;
          }
          // Kill the event when a content item is clicked.
          event.preventDefault();
          event.stopPropagation();
          // Execute the item if possible.
          this._execute(index);
      };
      /**
       * Handle the `'keydown'` event for the command palette.
       */
      CommandPalette.prototype._evtKeyDown = function (event) {
          if (event.altKey || event.ctrlKey || event.metaKey || event.shiftKey) {
              return;
          }
          switch (event.keyCode) {
              case 13: // Enter
                  event.preventDefault();
                  event.stopPropagation();
                  this._execute(this._activeIndex);
                  break;
              case 38: // Up Arrow
                  event.preventDefault();
                  event.stopPropagation();
                  this._activatePreviousItem();
                  break;
              case 40: // Down Arrow
                  event.preventDefault();
                  event.stopPropagation();
                  this._activateNextItem();
                  break;
          }
      };
      /**
       * Activate the next enabled command item.
       */
      CommandPalette.prototype._activateNextItem = function () {
          // Bail if there are no search results.
          if (!this._results || this._results.length === 0) {
              return;
          }
          // Find the next enabled item index.
          var ai = this._activeIndex;
          var n = this._results.length;
          var start = ai < n - 1 ? ai + 1 : 0;
          var stop = start === 0 ? n - 1 : start - 1;
          this._activeIndex = ArrayExt.findFirstIndex(this._results, Private$5$1.canActivate, start, stop);
          // Schedule an update of the items.
          this.update();
      };
      /**
       * Activate the previous enabled command item.
       */
      CommandPalette.prototype._activatePreviousItem = function () {
          // Bail if there are no search results.
          if (!this._results || this._results.length === 0) {
              return;
          }
          // Find the previous enabled item index.
          var ai = this._activeIndex;
          var n = this._results.length;
          var start = ai <= 0 ? n - 1 : ai - 1;
          var stop = start === n - 1 ? 0 : start + 1;
          this._activeIndex = ArrayExt.findLastIndex(this._results, Private$5$1.canActivate, start, stop);
          // Schedule an update of the items.
          this.update();
      };
      /**
       * Execute the command item at the given index, if possible.
       */
      CommandPalette.prototype._execute = function (index) {
          // Bail if there are no search results.
          if (!this._results) {
              return;
          }
          // Bail if the index is out of range.
          var part = this._results[index];
          if (!part) {
              return;
          }
          // Update the search text if the item is a header.
          if (part.type === 'header') {
              var input = this.inputNode;
              input.value = part.category.toLowerCase() + " ";
              input.focus();
              this.refresh();
              return;
          }
          // Bail if item is not enabled.
          if (!part.item.isEnabled) {
              return;
          }
          // Execute the item.
          this.commands.execute(part.item.command, part.item.args);
          // Clear the query text.
          this.inputNode.value = '';
          // Refresh the search results.
          this.refresh();
      };
      /**
       * Toggle the focused modifier based on the input node focus state.
       */
      CommandPalette.prototype._toggleFocused = function () {
          var focused = document.activeElement === this.inputNode;
          this.toggleClass('lm-mod-focused', focused);
          /* <DEPRECATED> */
          this.toggleClass('p-mod-focused', focused);
          /* </DEPRECATED> */
      };
      /**
       * A signal handler for generic command changes.
       */
      CommandPalette.prototype._onGenericChange = function () {
          this.refresh();
      };
      return CommandPalette;
  }(Widget));
  /**
   * The namespace for the `CommandPalette` class statics.
   */
  (function (CommandPalette) {
      /**
       * The default implementation of `IRenderer`.
       */
      var Renderer = /** @class */ (function () {
          function Renderer() {
          }
          /**
           * Render the virtual element for a command palette header.
           *
           * @param data - The data to use for rendering the header.
           *
           * @returns A virtual element representing the header.
           */
          Renderer.prototype.renderHeader = function (data) {
              var content = this.formatHeader(data);
              return h.li({ className: 'lm-CommandPalette-header'
                      /* <DEPRECATED> */
                      + ' p-CommandPalette-header'
                  /* </DEPRECATED> */
              }, content);
          };
          /**
           * Render the virtual element for a command palette item.
           *
           * @param data - The data to use for rendering the item.
           *
           * @returns A virtual element representing the item.
           */
          Renderer.prototype.renderItem = function (data) {
              var className = this.createItemClass(data);
              var dataset = this.createItemDataset(data);
              return (h.li({ className: className, dataset: dataset }, this.renderItemIcon(data), this.renderItemContent(data), this.renderItemShortcut(data)));
          };
          /**
           * Render the empty results message for a command palette.
           *
           * @param data - The data to use for rendering the message.
           *
           * @returns A virtual element representing the message.
           */
          Renderer.prototype.renderEmptyMessage = function (data) {
              var content = this.formatEmptyMessage(data);
              return h.li({
                  className: 'lm-CommandPalette-emptyMessage'
                      /* <DEPRECATED> */
                      + ' p-CommandPalette-emptyMessage'
                  /* </DEPRECATED> */
              }, content);
          };
          /**
           * Render the icon for a command palette item.
           *
           * @param data - The data to use for rendering the icon.
           *
           * @returns A virtual element representing the icon.
           */
          Renderer.prototype.renderItemIcon = function (data) {
              var className = this.createIconClass(data);
              /* <DEPRECATED> */
              if (typeof data.item.icon === 'string') {
                  return h.div({ className: className }, data.item.iconLabel);
              }
              /* </DEPRECATED> */
              // if data.item.icon is undefined, it will be ignored
              return h.div({ className: className }, data.item.icon, data.item.iconLabel);
          };
          /**
           * Render the content for a command palette item.
           *
           * @param data - The data to use for rendering the content.
           *
           * @returns A virtual element representing the content.
           */
          Renderer.prototype.renderItemContent = function (data) {
              return (h.div({
                  className: 'lm-CommandPalette-itemContent'
                      /* <DEPRECATED> */
                      + ' p-CommandPalette-itemContent'
                  /* </DEPRECATED> */
              }, this.renderItemLabel(data), this.renderItemCaption(data)));
          };
          /**
           * Render the label for a command palette item.
           *
           * @param data - The data to use for rendering the label.
           *
           * @returns A virtual element representing the label.
           */
          Renderer.prototype.renderItemLabel = function (data) {
              var content = this.formatItemLabel(data);
              return h.div({
                  className: 'lm-CommandPalette-itemLabel'
                      /* <DEPRECATED> */
                      + ' p-CommandPalette-itemLabel'
                  /* </DEPRECATED> */
              }, content);
          };
          /**
           * Render the caption for a command palette item.
           *
           * @param data - The data to use for rendering the caption.
           *
           * @returns A virtual element representing the caption.
           */
          Renderer.prototype.renderItemCaption = function (data) {
              var content = this.formatItemCaption(data);
              return h.div({
                  className: 'lm-CommandPalette-itemCaption'
                      /* <DEPRECATED> */
                      + ' p-CommandPalette-itemCaption'
                  /* </DEPRECATED> */
              }, content);
          };
          /**
           * Render the shortcut for a command palette item.
           *
           * @param data - The data to use for rendering the shortcut.
           *
           * @returns A virtual element representing the shortcut.
           */
          Renderer.prototype.renderItemShortcut = function (data) {
              var content = this.formatItemShortcut(data);
              return h.div({
                  className: 'lm-CommandPalette-itemShortcut'
                      /* <DEPRECATED> */
                      + ' p-CommandPalette-itemShortcut'
                  /* </DEPRECATED> */
              }, content);
          };
          /**
           * Create the class name for the command palette item.
           *
           * @param data - The data to use for the class name.
           *
           * @returns The full class name for the command palette item.
           */
          Renderer.prototype.createItemClass = function (data) {
              // Set up the initial class name.
              var name = 'lm-CommandPalette-item';
              /* <DEPRECATED> */
              name += ' p-CommandPalette-item';
              /* </DEPRECATED> */
              // Add the boolean state classes.
              if (!data.item.isEnabled) {
                  name += ' lm-mod-disabled';
                  /* <DEPRECATED> */
                  name += ' p-mod-disabled';
                  /* </DEPRECATED> */
              }
              if (data.item.isToggled) {
                  name += ' lm-mod-toggled';
                  /* <DEPRECATED> */
                  name += ' p-mod-toggled';
                  /* </DEPRECATED> */
              }
              if (data.active) {
                  name += ' lm-mod-active';
                  /* <DEPRECATED> */
                  name += ' p-mod-active';
                  /* </DEPRECATED> */
              }
              // Add the extra class.
              var extra = data.item.className;
              if (extra) {
                  name += " " + extra;
              }
              // Return the complete class name.
              return name;
          };
          /**
           * Create the dataset for the command palette item.
           *
           * @param data - The data to use for creating the dataset.
           *
           * @returns The dataset for the command palette item.
           */
          Renderer.prototype.createItemDataset = function (data) {
              return __assign$3(__assign$3({}, data.item.dataset), { command: data.item.command });
          };
          /**
           * Create the class name for the command item icon.
           *
           * @param data - The data to use for the class name.
           *
           * @returns The full class name for the item icon.
           */
          Renderer.prototype.createIconClass = function (data) {
              var name = 'lm-CommandPalette-itemIcon';
              /* <DEPRECATED> */
              name += ' p-CommandPalette-itemIcon';
              /* </DEPRECATED> */
              var extra = data.item.iconClass;
              return extra ? name + " " + extra : name;
          };
          /**
           * Create the render content for the header node.
           *
           * @param data - The data to use for the header content.
           *
           * @returns The content to add to the header node.
           */
          Renderer.prototype.formatHeader = function (data) {
              if (!data.indices || data.indices.length === 0) {
                  return data.category;
              }
              return StringExt.highlight(data.category, data.indices, h.mark);
          };
          /**
           * Create the render content for the empty message node.
           *
           * @param data - The data to use for the empty message content.
           *
           * @returns The content to add to the empty message node.
           */
          Renderer.prototype.formatEmptyMessage = function (data) {
              return "No commands found that match '" + data.query + "'";
          };
          /**
           * Create the render content for the item shortcut node.
           *
           * @param data - The data to use for the shortcut content.
           *
           * @returns The content to add to the shortcut node.
           */
          Renderer.prototype.formatItemShortcut = function (data) {
              var kb = data.item.keyBinding;
              return kb ? kb.keys.map(CommandRegistry.formatKeystroke).join(', ') : null;
          };
          /**
           * Create the render content for the item label node.
           *
           * @param data - The data to use for the label content.
           *
           * @returns The content to add to the label node.
           */
          Renderer.prototype.formatItemLabel = function (data) {
              if (!data.indices || data.indices.length === 0) {
                  return data.item.label;
              }
              return StringExt.highlight(data.item.label, data.indices, h.mark);
          };
          /**
           * Create the render content for the item caption node.
           *
           * @param data - The data to use for the caption content.
           *
           * @returns The content to add to the caption node.
           */
          Renderer.prototype.formatItemCaption = function (data) {
              return data.item.caption;
          };
          return Renderer;
      }());
      CommandPalette.Renderer = Renderer;
      /**
       * The default `Renderer` instance.
       */
      CommandPalette.defaultRenderer = new Renderer();
  })(CommandPalette || (CommandPalette = {}));
  /**
   * The namespace for the module implementation details.
   */
  var Private$5$1;
  (function (Private) {
      /**
       * Create the DOM node for a command palette.
       */
      function createNode() {
          var node = document.createElement('div');
          var search = document.createElement('div');
          var wrapper = document.createElement('div');
          var input = document.createElement('input');
          var content = document.createElement('ul');
          var clear = document.createElement('button');
          search.className = 'lm-CommandPalette-search';
          wrapper.className = 'lm-CommandPalette-wrapper';
          input.className = 'lm-CommandPalette-input';
          clear.className = 'lm-close-icon';
          content.className = 'lm-CommandPalette-content';
          /* <DEPRECATED> */
          search.classList.add('p-CommandPalette-search');
          wrapper.classList.add('p-CommandPalette-wrapper');
          input.classList.add('p-CommandPalette-input');
          content.classList.add('p-CommandPalette-content');
          /* </DEPRECATED> */
          input.spellcheck = false;
          wrapper.appendChild(input);
          wrapper.appendChild(clear);
          search.appendChild(wrapper);
          node.appendChild(search);
          node.appendChild(content);
          return node;
      }
      Private.createNode = createNode;
      /**
       * Create a new command item from a command registry and options.
       */
      function createItem(commands, options) {
          return new CommandItem(commands, options);
      }
      Private.createItem = createItem;
      /**
       * Search an array of command items for fuzzy matches.
       */
      function search(items, query) {
          // Fuzzy match the items for the query.
          var scores = matchItems(items, query);
          // Sort the items based on their score.
          scores.sort(scoreCmp);
          // Create the results for the search.
          return createResults(scores);
      }
      Private.search = search;
      /**
       * Test whether a result item can be activated.
       */
      function canActivate(result) {
          return result.type === 'item' && result.item.isEnabled;
      }
      Private.canActivate = canActivate;
      /**
       * Normalize a category for a command item.
       */
      function normalizeCategory(category) {
          return category.trim().replace(/\s+/g, ' ');
      }
      /**
       * Normalize the query text for a fuzzy search.
       */
      function normalizeQuery(text) {
          return text.replace(/\s+/g, '').toLowerCase();
      }
      /**
       * Perform a fuzzy match on an array of command items.
       */
      function matchItems(items, query) {
          // Normalize the query text to lower case with no whitespace.
          query = normalizeQuery(query);
          // Create the array to hold the scores.
          var scores = [];
          // Iterate over the items and match against the query.
          for (var i = 0, n = items.length; i < n; ++i) {
              // Ignore items which are not visible.
              var item = items[i];
              if (!item.isVisible) {
                  continue;
              }
              // If the query is empty, all items are matched by default.
              if (!query) {
                  scores.push({
                      matchType: 3 /* Default */,
                      categoryIndices: null,
                      labelIndices: null,
                      score: 0, item: item
                  });
                  continue;
              }
              // Run the fuzzy search for the item and query.
              var score = fuzzySearch(item, query);
              // Ignore the item if it is not a match.
              if (!score) {
                  continue;
              }
              // Penalize disabled items.
              // TODO - push disabled items all the way down in sort cmp?
              if (!item.isEnabled) {
                  score.score += 1000;
              }
              // Add the score to the results.
              scores.push(score);
          }
          // Return the final array of scores.
          return scores;
      }
      /**
       * Perform a fuzzy search on a single command item.
       */
      function fuzzySearch(item, query) {
          // Create the source text to be searched.
          var category = item.category.toLowerCase();
          var label = item.label.toLowerCase();
          var source = category + " " + label;
          // Set up the match score and indices array.
          var score = Infinity;
          var indices = null;
          // The regex for search word boundaries
          var rgx = /\b\w/g;
          // Search the source by word boundary.
          while (true) {
              // Find the next word boundary in the source.
              var rgxMatch = rgx.exec(source);
              // Break if there is no more source context.
              if (!rgxMatch) {
                  break;
              }
              // Run the string match on the relevant substring.
              var match = StringExt.matchSumOfDeltas(source, query, rgxMatch.index);
              // Break if there is no match.
              if (!match) {
                  break;
              }
              // Update the match if the score is better.
              if (match && match.score <= score) {
                  score = match.score;
                  indices = match.indices;
              }
          }
          // Bail if there was no match.
          if (!indices || score === Infinity) {
              return null;
          }
          // Compute the pivot index between category and label text.
          var pivot = category.length + 1;
          // Find the slice index to separate matched indices.
          var j = ArrayExt.lowerBound(indices, pivot, function (a, b) { return a - b; });
          // Extract the matched category and label indices.
          var categoryIndices = indices.slice(0, j);
          var labelIndices = indices.slice(j);
          // Adjust the label indices for the pivot offset.
          for (var i = 0, n = labelIndices.length; i < n; ++i) {
              labelIndices[i] -= pivot;
          }
          // Handle a pure label match.
          if (categoryIndices.length === 0) {
              return {
                  matchType: 0 /* Label */,
                  categoryIndices: null,
                  labelIndices: labelIndices,
                  score: score, item: item
              };
          }
          // Handle a pure category match.
          if (labelIndices.length === 0) {
              return {
                  matchType: 1 /* Category */,
                  categoryIndices: categoryIndices,
                  labelIndices: null,
                  score: score, item: item
              };
          }
          // Handle a split match.
          return {
              matchType: 2 /* Split */,
              categoryIndices: categoryIndices,
              labelIndices: labelIndices,
              score: score, item: item
          };
      }
      /**
       * A sort comparison function for a match score.
       */
      function scoreCmp(a, b) {
          // First compare based on the match type
          var m1 = a.matchType - b.matchType;
          if (m1 !== 0) {
              return m1;
          }
          // Otherwise, compare based on the match score.
          var d1 = a.score - b.score;
          if (d1 !== 0) {
              return d1;
          }
          // Find the match index based on the match type.
          var i1 = 0;
          var i2 = 0;
          switch (a.matchType) {
              case 0 /* Label */:
                  i1 = a.labelIndices[0];
                  i2 = b.labelIndices[0];
                  break;
              case 1 /* Category */:
              case 2 /* Split */:
                  i1 = a.categoryIndices[0];
                  i2 = b.categoryIndices[0];
                  break;
          }
          // Compare based on the match index.
          if (i1 !== i2) {
              return i1 - i2;
          }
          // Otherwise, compare by category.
          var d2 = a.item.category.localeCompare(b.item.category);
          if (d2 !== 0) {
              return d2;
          }
          // Otherwise, compare by rank.
          var r1 = a.item.rank;
          var r2 = b.item.rank;
          if (r1 !== r2) {
              return r1 < r2 ? -1 : 1; // Infinity safe
          }
          // Finally, compare by label.
          return a.item.label.localeCompare(b.item.label);
      }
      /**
       * Create the results from an array of sorted scores.
       */
      function createResults(scores) {
          // Set up an array to track which scores have been visited.
          var visited = new Array(scores.length);
          ArrayExt.fill(visited, false);
          // Set up the search results array.
          var results = [];
          // Iterate over each score in the array.
          for (var i = 0, n = scores.length; i < n; ++i) {
              // Ignore a score which has already been processed.
              if (visited[i]) {
                  continue;
              }
              // Extract the current item and indices.
              var _a = scores[i], item = _a.item, categoryIndices = _a.categoryIndices;
              // Extract the category for the current item.
              var category = item.category;
              // Add the header result for the category.
              results.push({ type: 'header', category: category, indices: categoryIndices });
              // Find the rest of the scores with the same category.
              for (var j = i; j < n; ++j) {
                  // Ignore a score which has already been processed.
                  if (visited[j]) {
                      continue;
                  }
                  // Extract the data for the current score.
                  var _b = scores[j], item_1 = _b.item, labelIndices = _b.labelIndices;
                  // Ignore an item with a different category.
                  if (item_1.category !== category) {
                      continue;
                  }
                  // Create the item result for the score.
                  results.push({ type: 'item', item: item_1, indices: labelIndices });
                  // Mark the score as processed.
                  visited[j] = true;
              }
          }
          // Return the final results.
          return results;
      }
      /**
       * A concrete implementation of `CommandPalette.IItem`.
       */
      var CommandItem = /** @class */ (function () {
          /**
           * Construct a new command item.
           */
          function CommandItem(commands, options) {
              this._commands = commands;
              this.category = normalizeCategory(options.category);
              this.command = options.command;
              this.args = options.args || JSONExt.emptyObject;
              this.rank = options.rank !== undefined ? options.rank : Infinity;
          }
          Object.defineProperty(CommandItem.prototype, "label", {
              /**
               * The display label for the command item.
               */
              get: function () {
                  return this._commands.label(this.command, this.args);
              },
              enumerable: true,
              configurable: true
          });
          Object.defineProperty(CommandItem.prototype, "icon", {
              /**
               * The icon renderer for the command item.
               */
              get: function () {
                  return this._commands.icon(this.command, this.args);
              },
              enumerable: true,
              configurable: true
          });
          Object.defineProperty(CommandItem.prototype, "iconClass", {
              /**
               * The icon class for the command item.
               */
              get: function () {
                  return this._commands.iconClass(this.command, this.args);
              },
              enumerable: true,
              configurable: true
          });
          Object.defineProperty(CommandItem.prototype, "iconLabel", {
              /**
               * The icon label for the command item.
               */
              get: function () {
                  return this._commands.iconLabel(this.command, this.args);
              },
              enumerable: true,
              configurable: true
          });
          Object.defineProperty(CommandItem.prototype, "caption", {
              /**
               * The display caption for the command item.
               */
              get: function () {
                  return this._commands.caption(this.command, this.args);
              },
              enumerable: true,
              configurable: true
          });
          Object.defineProperty(CommandItem.prototype, "className", {
              /**
               * The extra class name for the command item.
               */
              get: function () {
                  return this._commands.className(this.command, this.args);
              },
              enumerable: true,
              configurable: true
          });
          Object.defineProperty(CommandItem.prototype, "dataset", {
              /**
               * The dataset for the command item.
               */
              get: function () {
                  return this._commands.dataset(this.command, this.args);
              },
              enumerable: true,
              configurable: true
          });
          Object.defineProperty(CommandItem.prototype, "isEnabled", {
              /**
               * Whether the command item is enabled.
               */
              get: function () {
                  return this._commands.isEnabled(this.command, this.args);
              },
              enumerable: true,
              configurable: true
          });
          Object.defineProperty(CommandItem.prototype, "isToggled", {
              /**
               * Whether the command item is toggled.
               */
              get: function () {
                  return this._commands.isToggled(this.command, this.args);
              },
              enumerable: true,
              configurable: true
          });
          Object.defineProperty(CommandItem.prototype, "isVisible", {
              /**
               * Whether the command item is visible.
               */
              get: function () {
                  return this._commands.isVisible(this.command, this.args);
              },
              enumerable: true,
              configurable: true
          });
          Object.defineProperty(CommandItem.prototype, "keyBinding", {
              /**
               * The key binding for the command item.
               */
              get: function () {
                  var _a = this, command = _a.command, args = _a.args;
                  return ArrayExt.findLastValue(this._commands.keyBindings, function (kb) {
                      return kb.command === command && JSONExt.deepEqual(kb.args, args);
                  }) || null;
              },
              enumerable: true,
              configurable: true
          });
          return CommandItem;
      }());
  })(Private$5$1 || (Private$5$1 = {}));

  /**
   * A widget which displays items as a canonical menu.
   */
  var Menu = /** @class */ (function (_super) {
      __extends$5(Menu, _super);
      /**
       * Construct a new menu.
       *
       * @param options - The options for initializing the menu.
       */
      function Menu(options) {
          var _this = _super.call(this, { node: Private$6$1.createNode() }) || this;
          _this._childIndex = -1;
          _this._activeIndex = -1;
          _this._openTimerID = 0;
          _this._closeTimerID = 0;
          _this._items = [];
          _this._childMenu = null;
          _this._parentMenu = null;
          _this._aboutToClose = new Signal(_this);
          _this._menuRequested = new Signal(_this);
          _this.addClass('lm-Menu');
          /* <DEPRECATED> */
          _this.addClass('p-Menu');
          /* </DEPRECATED> */
          _this.setFlag(Widget.Flag.DisallowLayout);
          _this.commands = options.commands;
          _this.renderer = options.renderer || Menu.defaultRenderer;
          return _this;
      }
      /**
       * Dispose of the resources held by the menu.
       */
      Menu.prototype.dispose = function () {
          this.close();
          this._items.length = 0;
          _super.prototype.dispose.call(this);
      };
      Object.defineProperty(Menu.prototype, "aboutToClose", {
          /**
           * A signal emitted just before the menu is closed.
           *
           * #### Notes
           * This signal is emitted when the menu receives a `'close-request'`
           * message, just before it removes itself from the DOM.
           *
           * This signal is not emitted if the menu is already detached from
           * the DOM when it receives the `'close-request'` message.
           */
          get: function () {
              return this._aboutToClose;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(Menu.prototype, "menuRequested", {
          /**
           * A signal emitted when a new menu is requested by the user.
           *
           * #### Notes
           * This signal is emitted whenever the user presses the right or left
           * arrow keys, and a submenu cannot be opened or closed in response.
           *
           * This signal is useful when implementing menu bars in order to open
           * the next or previous menu in response to a user key press.
           *
           * This signal is only emitted for the root menu in a hierarchy.
           */
          get: function () {
              return this._menuRequested;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(Menu.prototype, "parentMenu", {
          /**
           * The parent menu of the menu.
           *
           * #### Notes
           * This is `null` unless the menu is an open submenu.
           */
          get: function () {
              return this._parentMenu;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(Menu.prototype, "childMenu", {
          /**
           * The child menu of the menu.
           *
           * #### Notes
           * This is `null` unless the menu has an open submenu.
           */
          get: function () {
              return this._childMenu;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(Menu.prototype, "rootMenu", {
          /**
           * The root menu of the menu hierarchy.
           */
          get: function () {
              var menu = this;
              while (menu._parentMenu) {
                  menu = menu._parentMenu;
              }
              return menu;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(Menu.prototype, "leafMenu", {
          /**
           * The leaf menu of the menu hierarchy.
           */
          get: function () {
              var menu = this;
              while (menu._childMenu) {
                  menu = menu._childMenu;
              }
              return menu;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(Menu.prototype, "contentNode", {
          /**
           * The menu content node.
           *
           * #### Notes
           * This is the node which holds the menu item nodes.
           *
           * Modifying this node directly can lead to undefined behavior.
           */
          get: function () {
              return this.node.getElementsByClassName('lm-Menu-content')[0];
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(Menu.prototype, "activeItem", {
          /**
           * Get the currently active menu item.
           */
          get: function () {
              return this._items[this._activeIndex] || null;
          },
          /**
           * Set the currently active menu item.
           *
           * #### Notes
           * If the item cannot be activated, the item will be set to `null`.
           */
          set: function (value) {
              this.activeIndex = value ? this._items.indexOf(value) : -1;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(Menu.prototype, "activeIndex", {
          /**
           * Get the index of the currently active menu item.
           *
           * #### Notes
           * This will be `-1` if no menu item is active.
           */
          get: function () {
              return this._activeIndex;
          },
          /**
           * Set the index of the currently active menu item.
           *
           * #### Notes
           * If the item cannot be activated, the index will be set to `-1`.
           */
          set: function (value) {
              // Adjust the value for an out of range index.
              if (value < 0 || value >= this._items.length) {
                  value = -1;
              }
              // Ensure the item can be activated.
              if (value !== -1 && !Private$6$1.canActivate(this._items[value])) {
                  value = -1;
              }
              // Bail if the index will not change.
              if (this._activeIndex === value) {
                  return;
              }
              // Update the active index.
              this._activeIndex = value;
              // schedule an update of the items.
              this.update();
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(Menu.prototype, "items", {
          /**
           * A read-only array of the menu items in the menu.
           */
          get: function () {
              return this._items;
          },
          enumerable: true,
          configurable: true
      });
      /**
       * Activate the next selectable item in the menu.
       *
       * #### Notes
       * If no item is selectable, the index will be set to `-1`.
       */
      Menu.prototype.activateNextItem = function () {
          var n = this._items.length;
          var ai = this._activeIndex;
          var start = ai < n - 1 ? ai + 1 : 0;
          var stop = start === 0 ? n - 1 : start - 1;
          this.activeIndex = ArrayExt.findFirstIndex(this._items, Private$6$1.canActivate, start, stop);
      };
      /**
       * Activate the previous selectable item in the menu.
       *
       * #### Notes
       * If no item is selectable, the index will be set to `-1`.
       */
      Menu.prototype.activatePreviousItem = function () {
          var n = this._items.length;
          var ai = this._activeIndex;
          var start = ai <= 0 ? n - 1 : ai - 1;
          var stop = start === n - 1 ? 0 : start + 1;
          this.activeIndex = ArrayExt.findLastIndex(this._items, Private$6$1.canActivate, start, stop);
      };
      /**
       * Trigger the active menu item.
       *
       * #### Notes
       * If the active item is a submenu, it will be opened and the first
       * item will be activated.
       *
       * If the active item is a command, the command will be executed.
       *
       * If the menu is not attached, this is a no-op.
       *
       * If there is no active item, this is a no-op.
       */
      Menu.prototype.triggerActiveItem = function () {
          // Bail if the menu is not attached.
          if (!this.isAttached) {
              return;
          }
          // Bail if there is no active item.
          var item = this.activeItem;
          if (!item) {
              return;
          }
          // Cancel the pending timers.
          this._cancelOpenTimer();
          this._cancelCloseTimer();
          // If the item is a submenu, open it.
          if (item.type === 'submenu') {
              this._openChildMenu(true);
              return;
          }
          // Close the root menu before executing the command.
          this.rootMenu.close();
          // Execute the command for the item.
          var command = item.command, args = item.args;
          if (this.commands.isEnabled(command, args)) {
              this.commands.execute(command, args);
          }
          else {
              console.log("Command '" + command + "' is disabled.");
          }
      };
      /**
       * Add a menu item to the end of the menu.
       *
       * @param options - The options for creating the menu item.
       *
       * @returns The menu item added to the menu.
       */
      Menu.prototype.addItem = function (options) {
          return this.insertItem(this._items.length, options);
      };
      /**
       * Insert a menu item into the menu at the specified index.
       *
       * @param index - The index at which to insert the item.
       *
       * @param options - The options for creating the menu item.
       *
       * @returns The menu item added to the menu.
       *
       * #### Notes
       * The index will be clamped to the bounds of the items.
       */
      Menu.prototype.insertItem = function (index, options) {
          // Close the menu if it's attached.
          if (this.isAttached) {
              this.close();
          }
          // Reset the active index.
          this.activeIndex = -1;
          // Clamp the insert index to the array bounds.
          var i = Math.max(0, Math.min(index, this._items.length));
          // Create the item for the options.
          var item = Private$6$1.createItem(this, options);
          // Insert the item into the array.
          ArrayExt.insert(this._items, i, item);
          // Schedule an update of the items.
          this.update();
          // Return the item added to the menu.
          return item;
      };
      /**
       * Remove an item from the menu.
       *
       * @param item - The item to remove from the menu.
       *
       * #### Notes
       * This is a no-op if the item is not in the menu.
       */
      Menu.prototype.removeItem = function (item) {
          this.removeItemAt(this._items.indexOf(item));
      };
      /**
       * Remove the item at a given index from the menu.
       *
       * @param index - The index of the item to remove.
       *
       * #### Notes
       * This is a no-op if the index is out of range.
       */
      Menu.prototype.removeItemAt = function (index) {
          // Close the menu if it's attached.
          if (this.isAttached) {
              this.close();
          }
          // Reset the active index.
          this.activeIndex = -1;
          // Remove the item from the array.
          var item = ArrayExt.removeAt(this._items, index);
          // Bail if the index is out of range.
          if (!item) {
              return;
          }
          // Schedule an update of the items.
          this.update();
      };
      /**
       * Remove all menu items from the menu.
       */
      Menu.prototype.clearItems = function () {
          // Close the menu if it's attached.
          if (this.isAttached) {
              this.close();
          }
          // Reset the active index.
          this.activeIndex = -1;
          // Bail if there is nothing to remove.
          if (this._items.length === 0) {
              return;
          }
          // Clear the items.
          this._items.length = 0;
          // Schedule an update of the items.
          this.update();
      };
      /**
       * Open the menu at the specified location.
       *
       * @param x - The client X coordinate of the menu location.
       *
       * @param y - The client Y coordinate of the menu location.
       *
       * @param options - The additional options for opening the menu.
       *
       * #### Notes
       * The menu will be opened at the given location unless it will not
       * fully fit on the screen. If it will not fit, it will be adjusted
       * to fit naturally on the screen.
       *
       * This is a no-op if the menu is already attached to the DOM.
       */
      Menu.prototype.open = function (x, y, options) {
          if (options === void 0) { options = {}; }
          // Bail early if the menu is already attached.
          if (this.isAttached) {
              return;
          }
          // Extract the position options.
          var forceX = options.forceX || false;
          var forceY = options.forceY || false;
          // Open the menu as a root menu.
          Private$6$1.openRootMenu(this, x, y, forceX, forceY);
          // Activate the menu to accept keyboard input.
          this.activate();
      };
      /**
       * Handle the DOM events for the menu.
       *
       * @param event - The DOM event sent to the menu.
       *
       * #### Notes
       * This method implements the DOM `EventListener` interface and is
       * called in response to events on the menu's DOM nodes. It should
       * not be called directly by user code.
       */
      Menu.prototype.handleEvent = function (event) {
          switch (event.type) {
              case 'keydown':
                  this._evtKeyDown(event);
                  break;
              case 'mouseup':
                  this._evtMouseUp(event);
                  break;
              case 'mousemove':
                  this._evtMouseMove(event);
                  break;
              case 'mouseenter':
                  this._evtMouseEnter(event);
                  break;
              case 'mouseleave':
                  this._evtMouseLeave(event);
                  break;
              case 'mousedown':
                  this._evtMouseDown(event);
                  break;
              case 'contextmenu':
                  event.preventDefault();
                  event.stopPropagation();
                  break;
          }
      };
      /**
       * A message handler invoked on a `'before-attach'` message.
       */
      Menu.prototype.onBeforeAttach = function (msg) {
          this.node.addEventListener('keydown', this);
          this.node.addEventListener('mouseup', this);
          this.node.addEventListener('mousemove', this);
          this.node.addEventListener('mouseenter', this);
          this.node.addEventListener('mouseleave', this);
          this.node.addEventListener('contextmenu', this);
          document.addEventListener('mousedown', this, true);
      };
      /**
       * A message handler invoked on an `'after-detach'` message.
       */
      Menu.prototype.onAfterDetach = function (msg) {
          this.node.removeEventListener('keydown', this);
          this.node.removeEventListener('mouseup', this);
          this.node.removeEventListener('mousemove', this);
          this.node.removeEventListener('mouseenter', this);
          this.node.removeEventListener('mouseleave', this);
          this.node.removeEventListener('contextmenu', this);
          document.removeEventListener('mousedown', this, true);
      };
      /**
       * A message handler invoked on an `'activate-request'` message.
       */
      Menu.prototype.onActivateRequest = function (msg) {
          if (this.isAttached) {
              this.node.focus();
          }
      };
      /**
       * A message handler invoked on an `'update-request'` message.
       */
      Menu.prototype.onUpdateRequest = function (msg) {
          var items = this._items;
          var renderer = this.renderer;
          var activeIndex = this._activeIndex;
          var collapsedFlags = Private$6$1.computeCollapsed(items);
          var content = new Array(items.length);
          for (var i = 0, n = items.length; i < n; ++i) {
              var item = items[i];
              var active = i === activeIndex;
              var collapsed = collapsedFlags[i];
              content[i] = renderer.renderItem({ item: item, active: active, collapsed: collapsed });
          }
          VirtualDOM.render(content, this.contentNode);
      };
      /**
       * A message handler invoked on a `'close-request'` message.
       */
      Menu.prototype.onCloseRequest = function (msg) {
          // Cancel the pending timers.
          this._cancelOpenTimer();
          this._cancelCloseTimer();
          // Reset the active index.
          this.activeIndex = -1;
          // Close any open child menu.
          var childMenu = this._childMenu;
          if (childMenu) {
              this._childIndex = -1;
              this._childMenu = null;
              childMenu._parentMenu = null;
              childMenu.close();
          }
          // Remove this menu from its parent and activate the parent.
          var parentMenu = this._parentMenu;
          if (parentMenu) {
              this._parentMenu = null;
              parentMenu._childIndex = -1;
              parentMenu._childMenu = null;
              parentMenu.activate();
          }
          // Emit the `aboutToClose` signal if the menu is attached.
          if (this.isAttached) {
              this._aboutToClose.emit(undefined);
          }
          // Finish closing the menu.
          _super.prototype.onCloseRequest.call(this, msg);
      };
      /**
       * Handle the `'keydown'` event for the menu.
       *
       * #### Notes
       * This listener is attached to the menu node.
       */
      Menu.prototype._evtKeyDown = function (event) {
          // A menu handles all keydown events.
          event.preventDefault();
          event.stopPropagation();
          // Fetch the key code for the event.
          var kc = event.keyCode;
          // Enter
          if (kc === 13) {
              this.triggerActiveItem();
              return;
          }
          // Escape
          if (kc === 27) {
              this.close();
              return;
          }
          // Left Arrow
          if (kc === 37) {
              if (this._parentMenu) {
                  this.close();
              }
              else {
                  this._menuRequested.emit('previous');
              }
              return;
          }
          // Up Arrow
          if (kc === 38) {
              this.activatePreviousItem();
              return;
          }
          // Right Arrow
          if (kc === 39) {
              var item = this.activeItem;
              if (item && item.type === 'submenu') {
                  this.triggerActiveItem();
              }
              else {
                  this.rootMenu._menuRequested.emit('next');
              }
              return;
          }
          // Down Arrow
          if (kc === 40) {
              this.activateNextItem();
              return;
          }
          // Get the pressed key character.
          var key = getKeyboardLayout().keyForKeydownEvent(event);
          // Bail if the key is not valid.
          if (!key) {
              return;
          }
          // Search for the next best matching mnemonic item.
          var start = this._activeIndex + 1;
          var result = Private$6$1.findMnemonic(this._items, key, start);
          // Handle the requested mnemonic based on the search results.
          // If exactly one mnemonic is matched, that item is triggered.
          // Otherwise, the next mnemonic is activated if available,
          // followed by the auto mnemonic if available.
          if (result.index !== -1 && !result.multiple) {
              this.activeIndex = result.index;
              this.triggerActiveItem();
          }
          else if (result.index !== -1) {
              this.activeIndex = result.index;
          }
          else if (result.auto !== -1) {
              this.activeIndex = result.auto;
          }
      };
      /**
       * Handle the `'mouseup'` event for the menu.
       *
       * #### Notes
       * This listener is attached to the menu node.
       */
      Menu.prototype._evtMouseUp = function (event) {
          if (event.button !== 0) {
              return;
          }
          event.preventDefault();
          event.stopPropagation();
          this.triggerActiveItem();
      };
      /**
       * Handle the `'mousemove'` event for the menu.
       *
       * #### Notes
       * This listener is attached to the menu node.
       */
      Menu.prototype._evtMouseMove = function (event) {
          // Hit test the item nodes for the item under the mouse.
          var index = ArrayExt.findFirstIndex(this.contentNode.children, function (node) {
              return ElementExt.hitTest(node, event.clientX, event.clientY);
          });
          // Bail early if the mouse is already over the active index.
          if (index === this._activeIndex) {
              return;
          }
          // Update and coerce the active index.
          this.activeIndex = index;
          index = this.activeIndex;
          // If the index is the current child index, cancel the timers.
          if (index === this._childIndex) {
              this._cancelOpenTimer();
              this._cancelCloseTimer();
              return;
          }
          // If a child menu is currently open, start the close timer.
          if (this._childIndex !== -1) {
              this._startCloseTimer();
          }
          // Cancel the open timer to give a full delay for opening.
          this._cancelOpenTimer();
          // Bail if the active item is not a valid submenu item.
          var item = this.activeItem;
          if (!item || item.type !== 'submenu' || !item.submenu) {
              return;
          }
          // Start the open timer to open the active item submenu.
          this._startOpenTimer();
      };
      /**
       * Handle the `'mouseenter'` event for the menu.
       *
       * #### Notes
       * This listener is attached to the menu node.
       */
      Menu.prototype._evtMouseEnter = function (event) {
          // Synchronize the active ancestor items.
          for (var menu = this._parentMenu; menu; menu = menu._parentMenu) {
              menu._cancelOpenTimer();
              menu._cancelCloseTimer();
              menu.activeIndex = menu._childIndex;
          }
      };
      /**
       * Handle the `'mouseleave'` event for the menu.
       *
       * #### Notes
       * This listener is attached to the menu node.
       */
      Menu.prototype._evtMouseLeave = function (event) {
          // Cancel any pending submenu opening.
          this._cancelOpenTimer();
          // If there is no open child menu, just reset the active index.
          if (!this._childMenu) {
              this.activeIndex = -1;
              return;
          }
          // If the mouse is over the child menu, cancel the close timer.
          var clientX = event.clientX, clientY = event.clientY;
          if (ElementExt.hitTest(this._childMenu.node, clientX, clientY)) {
              this._cancelCloseTimer();
              return;
          }
          // Otherwise, reset the active index and start the close timer.
          this.activeIndex = -1;
          this._startCloseTimer();
      };
      /**
       * Handle the `'mousedown'` event for the menu.
       *
       * #### Notes
       * This listener is attached to the document node.
       */
      Menu.prototype._evtMouseDown = function (event) {
          // Bail if the menu is not a root menu.
          if (this._parentMenu) {
              return;
          }
          // The mouse button which is pressed is irrelevant. If the press
          // is not on a menu, the entire hierarchy is closed and the event
          // is allowed to propagate. This allows other code to act on the
          // event, such as focusing the clicked element.
          if (Private$6$1.hitTestMenus(this, event.clientX, event.clientY)) {
              event.preventDefault();
              event.stopPropagation();
          }
          else {
              this.close();
          }
      };
      /**
       * Open the child menu at the active index immediately.
       *
       * If a different child menu is already open, it will be closed,
       * even if the active item is not a valid submenu.
       */
      Menu.prototype._openChildMenu = function (activateFirst) {
          if (activateFirst === void 0) { activateFirst = false; }
          // If the item is not a valid submenu, close the child menu.
          var item = this.activeItem;
          if (!item || item.type !== 'submenu' || !item.submenu) {
              this._closeChildMenu();
              return;
          }
          // Do nothing if the child menu will not change.
          var submenu = item.submenu;
          if (submenu === this._childMenu) {
              return;
          }
          // Ensure the current child menu is closed.
          this._closeChildMenu();
          // Update the private child state.
          this._childMenu = submenu;
          this._childIndex = this._activeIndex;
          // Set the parent menu reference for the child.
          submenu._parentMenu = this;
          // Ensure the menu is updated and lookup the item node.
          MessageLoop.sendMessage(this, Widget.Msg.UpdateRequest);
          var itemNode = this.contentNode.children[this._activeIndex];
          // Open the submenu at the active node.
          Private$6$1.openSubmenu(submenu, itemNode);
          // Activate the first item if desired.
          if (activateFirst) {
              submenu.activeIndex = -1;
              submenu.activateNextItem();
          }
          // Activate the child menu.
          submenu.activate();
      };
      /**
       * Close the child menu immediately.
       *
       * This is a no-op if a child menu is not open.
       */
      Menu.prototype._closeChildMenu = function () {
          if (this._childMenu) {
              this._childMenu.close();
          }
      };
      /**
       * Start the open timer, unless it is already pending.
       */
      Menu.prototype._startOpenTimer = function () {
          var _this = this;
          if (this._openTimerID === 0) {
              this._openTimerID = window.setTimeout(function () {
                  _this._openTimerID = 0;
                  _this._openChildMenu();
              }, Private$6$1.TIMER_DELAY);
          }
      };
      /**
       * Start the close timer, unless it is already pending.
       */
      Menu.prototype._startCloseTimer = function () {
          var _this = this;
          if (this._closeTimerID === 0) {
              this._closeTimerID = window.setTimeout(function () {
                  _this._closeTimerID = 0;
                  _this._closeChildMenu();
              }, Private$6$1.TIMER_DELAY);
          }
      };
      /**
       * Cancel the open timer, if the timer is pending.
       */
      Menu.prototype._cancelOpenTimer = function () {
          if (this._openTimerID !== 0) {
              clearTimeout(this._openTimerID);
              this._openTimerID = 0;
          }
      };
      /**
       * Cancel the close timer, if the timer is pending.
       */
      Menu.prototype._cancelCloseTimer = function () {
          if (this._closeTimerID !== 0) {
              clearTimeout(this._closeTimerID);
              this._closeTimerID = 0;
          }
      };
      return Menu;
  }(Widget));
  /**
   * The namespace for the `Menu` class statics.
   */
  (function (Menu) {
      /**
       * The default implementation of `IRenderer`.
       *
       * #### Notes
       * Subclasses are free to reimplement rendering methods as needed.
       */
      var Renderer = /** @class */ (function () {
          /**
           * Construct a new renderer.
           */
          function Renderer() {
          }
          /**
           * Render the virtual element for a menu item.
           *
           * @param data - The data to use for rendering the item.
           *
           * @returns A virtual element representing the item.
           */
          Renderer.prototype.renderItem = function (data) {
              var className = this.createItemClass(data);
              var dataset = this.createItemDataset(data);
              return (h.li({ className: className, dataset: dataset }, this.renderIcon(data), this.renderLabel(data), this.renderShortcut(data), this.renderSubmenu(data)));
          };
          /**
           * Render the icon element for a menu item.
           *
           * @param data - The data to use for rendering the icon.
           *
           * @returns A virtual element representing the item icon.
           */
          Renderer.prototype.renderIcon = function (data) {
              var className = this.createIconClass(data);
              /* <DEPRECATED> */
              if (typeof data.item.icon === 'string') {
                  return h.div({ className: className }, data.item.iconLabel);
              }
              /* </DEPRECATED> */
              // if data.item.icon is undefined, it will be ignored
              return h.div({ className: className }, data.item.icon, data.item.iconLabel);
          };
          /**
           * Render the label element for a menu item.
           *
           * @param data - The data to use for rendering the label.
           *
           * @returns A virtual element representing the item label.
           */
          Renderer.prototype.renderLabel = function (data) {
              var content = this.formatLabel(data);
              return h.div({
                  className: 'lm-Menu-itemLabel'
                      /* <DEPRECATED> */
                      + ' p-Menu-itemLabel'
                  /* </DEPRECATED> */
              }, content);
          };
          /**
           * Render the shortcut element for a menu item.
           *
           * @param data - The data to use for rendering the shortcut.
           *
           * @returns A virtual element representing the item shortcut.
           */
          Renderer.prototype.renderShortcut = function (data) {
              var content = this.formatShortcut(data);
              return h.div({
                  className: 'lm-Menu-itemShortcut'
                      /* <DEPRECATED> */
                      + ' p-Menu-itemShortcut'
                  /* </DEPRECATED> */
              }, content);
          };
          /**
           * Render the submenu icon element for a menu item.
           *
           * @param data - The data to use for rendering the submenu icon.
           *
           * @returns A virtual element representing the submenu icon.
           */
          Renderer.prototype.renderSubmenu = function (data) {
              return h.div({
                  className: 'lm-Menu-itemSubmenuIcon'
                      /* <DEPRECATED> */
                      + ' p-Menu-itemSubmenuIcon'
                  /* </DEPRECATED> */
              });
          };
          /**
           * Create the class name for the menu item.
           *
           * @param data - The data to use for the class name.
           *
           * @returns The full class name for the menu item.
           */
          Renderer.prototype.createItemClass = function (data) {
              // Setup the initial class name.
              var name = 'lm-Menu-item';
              /* <DEPRECATED> */
              name += ' p-Menu-item';
              /* </DEPRECATED> */
              // Add the boolean state classes.
              if (!data.item.isEnabled) {
                  name += ' lm-mod-disabled';
                  /* <DEPRECATED> */
                  name += ' p-mod-disabled';
                  /* </DEPRECATED> */
              }
              if (data.item.isToggled) {
                  name += ' lm-mod-toggled';
                  /* <DEPRECATED> */
                  name += ' p-mod-toggled';
                  /* </DEPRECATED> */
              }
              if (!data.item.isVisible) {
                  name += ' lm-mod-hidden';
                  /* <DEPRECATED> */
                  name += ' p-mod-hidden';
                  /* </DEPRECATED> */
              }
              if (data.active) {
                  name += ' lm-mod-active';
                  /* <DEPRECATED> */
                  name += ' p-mod-active';
                  /* </DEPRECATED> */
              }
              if (data.collapsed) {
                  name += ' lm-mod-collapsed';
                  /* <DEPRECATED> */
                  name += ' p-mod-collapsed';
                  /* </DEPRECATED> */
              }
              // Add the extra class.
              var extra = data.item.className;
              if (extra) {
                  name += " " + extra;
              }
              // Return the complete class name.
              return name;
          };
          /**
           * Create the dataset for the menu item.
           *
           * @param data - The data to use for creating the dataset.
           *
           * @returns The dataset for the menu item.
           */
          Renderer.prototype.createItemDataset = function (data) {
              var result;
              var _a = data.item, type = _a.type, command = _a.command, dataset = _a.dataset;
              if (type === 'command') {
                  result = __assign$3(__assign$3({}, dataset), { type: type, command: command });
              }
              else {
                  result = __assign$3(__assign$3({}, dataset), { type: type });
              }
              return result;
          };
          /**
           * Create the class name for the menu item icon.
           *
           * @param data - The data to use for the class name.
           *
           * @returns The full class name for the item icon.
           */
          Renderer.prototype.createIconClass = function (data) {
              var name = 'lm-Menu-itemIcon';
              /* <DEPRECATED> */
              name += ' p-Menu-itemIcon';
              /* </DEPRECATED> */
              var extra = data.item.iconClass;
              return extra ? name + " " + extra : name;
          };
          /**
           * Create the render content for the label node.
           *
           * @param data - The data to use for the label content.
           *
           * @returns The content to add to the label node.
           */
          Renderer.prototype.formatLabel = function (data) {
              // Fetch the label text and mnemonic index.
              var _a = data.item, label = _a.label, mnemonic = _a.mnemonic;
              // If the index is out of range, do not modify the label.
              if (mnemonic < 0 || mnemonic >= label.length) {
                  return label;
              }
              // Split the label into parts.
              var prefix = label.slice(0, mnemonic);
              var suffix = label.slice(mnemonic + 1);
              var char = label[mnemonic];
              // Wrap the mnemonic character in a span.
              var span = h.span({
                  className: 'lm-Menu-itemMnemonic'
                      /* <DEPRECATED> */
                      + ' p-Menu-itemMnemonic'
                  /* </DEPRECATED> */
              }, char);
              // Return the content parts.
              return [prefix, span, suffix];
          };
          /**
           * Create the render content for the shortcut node.
           *
           * @param data - The data to use for the shortcut content.
           *
           * @returns The content to add to the shortcut node.
           */
          Renderer.prototype.formatShortcut = function (data) {
              var kb = data.item.keyBinding;
              return kb ? kb.keys.map(CommandRegistry.formatKeystroke).join(', ') : null;
          };
          return Renderer;
      }());
      Menu.Renderer = Renderer;
      /**
       * The default `Renderer` instance.
       */
      Menu.defaultRenderer = new Renderer();
  })(Menu || (Menu = {}));
  /**
   * The namespace for the module implementation details.
   */
  var Private$6$1;
  (function (Private) {
      /**
       * The ms delay for opening and closing a submenu.
       */
      Private.TIMER_DELAY = 300;
      /**
       * The horizontal pixel overlap for an open submenu.
       */
      Private.SUBMENU_OVERLAP = 3;
      /**
       * Create the DOM node for a menu.
       */
      function createNode() {
          var node = document.createElement('div');
          var content = document.createElement('ul');
          content.className = 'lm-Menu-content';
          /* <DEPRECATED> */
          content.classList.add('p-Menu-content');
          /* </DEPRECATED> */
          node.appendChild(content);
          node.tabIndex = -1;
          return node;
      }
      Private.createNode = createNode;
      /**
       * Test whether a menu item can be activated.
       */
      function canActivate(item) {
          return item.type !== 'separator' && item.isEnabled && item.isVisible;
      }
      Private.canActivate = canActivate;
      /**
       * Create a new menu item for an owner menu.
       */
      function createItem(owner, options) {
          return new MenuItem(owner.commands, options);
      }
      Private.createItem = createItem;
      /**
       * Hit test a menu hierarchy starting at the given root.
       */
      function hitTestMenus(menu, x, y) {
          for (var temp = menu; temp; temp = temp.childMenu) {
              if (ElementExt.hitTest(temp.node, x, y)) {
                  return true;
              }
          }
          return false;
      }
      Private.hitTestMenus = hitTestMenus;
      /**
       * Compute which extra separator items should be collapsed.
       */
      function computeCollapsed(items) {
          // Allocate the return array and fill it with `false`.
          var result = new Array(items.length);
          ArrayExt.fill(result, false);
          // Collapse the leading separators.
          var k1 = 0;
          var n = items.length;
          for (; k1 < n; ++k1) {
              var item = items[k1];
              if (!item.isVisible) {
                  continue;
              }
              if (item.type !== 'separator') {
                  break;
              }
              result[k1] = true;
          }
          // Hide the trailing separators.
          var k2 = n - 1;
          for (; k2 >= 0; --k2) {
              var item = items[k2];
              if (!item.isVisible) {
                  continue;
              }
              if (item.type !== 'separator') {
                  break;
              }
              result[k2] = true;
          }
          // Hide the remaining consecutive separators.
          var hide = false;
          while (++k1 < k2) {
              var item = items[k1];
              if (!item.isVisible) {
                  continue;
              }
              if (item.type !== 'separator') {
                  hide = false;
              }
              else if (hide) {
                  result[k1] = true;
              }
              else {
                  hide = true;
              }
          }
          // Return the resulting flags.
          return result;
      }
      Private.computeCollapsed = computeCollapsed;
      /**
       * Open a menu as a root menu at the target location.
       */
      function openRootMenu(menu, x, y, forceX, forceY) {
          // Ensure the menu is updated before attaching and measuring.
          MessageLoop.sendMessage(menu, Widget.Msg.UpdateRequest);
          // Get the current position and size of the main viewport.
          var px = window.pageXOffset;
          var py = window.pageYOffset;
          var cw = document.documentElement.clientWidth;
          var ch = document.documentElement.clientHeight;
          // Compute the maximum allowed height for the menu.
          var maxHeight = ch - (forceY ? y : 0);
          // Fetch common variables.
          var node = menu.node;
          var style = node.style;
          // Clear the menu geometry and prepare it for measuring.
          style.top = '';
          style.left = '';
          style.width = '';
          style.height = '';
          style.visibility = 'hidden';
          style.maxHeight = maxHeight + "px";
          // Attach the menu to the document.
          Widget.attach(menu, document.body);
          // Measure the size of the menu.
          var _a = node.getBoundingClientRect(), width = _a.width, height = _a.height;
          // Adjust the X position of the menu to fit on-screen.
          if (!forceX && (x + width > px + cw)) {
              x = px + cw - width;
          }
          // Adjust the Y position of the menu to fit on-screen.
          if (!forceY && (y + height > py + ch)) {
              if (y > py + ch) {
                  y = py + ch - height;
              }
              else {
                  y = y - height;
              }
          }
          // Update the position of the menu to the computed position.
          style.top = Math.max(0, y) + "px";
          style.left = Math.max(0, x) + "px";
          // Finally, make the menu visible on the screen.
          style.visibility = '';
      }
      Private.openRootMenu = openRootMenu;
      /**
       * Open a menu as a submenu using an item node for positioning.
       */
      function openSubmenu(submenu, itemNode) {
          // Ensure the menu is updated before opening.
          MessageLoop.sendMessage(submenu, Widget.Msg.UpdateRequest);
          // Get the current position and size of the main viewport.
          var px = window.pageXOffset;
          var py = window.pageYOffset;
          var cw = document.documentElement.clientWidth;
          var ch = document.documentElement.clientHeight;
          // Compute the maximum allowed height for the menu.
          var maxHeight = ch;
          // Fetch common variables.
          var node = submenu.node;
          var style = node.style;
          // Clear the menu geometry and prepare it for measuring.
          style.top = '';
          style.left = '';
          style.width = '';
          style.height = '';
          style.visibility = 'hidden';
          style.maxHeight = maxHeight + "px";
          // Attach the menu to the document.
          Widget.attach(submenu, document.body);
          // Measure the size of the menu.
          var _a = node.getBoundingClientRect(), width = _a.width, height = _a.height;
          // Compute the box sizing for the menu.
          var box = ElementExt.boxSizing(submenu.node);
          // Get the bounding rect for the target item node.
          var itemRect = itemNode.getBoundingClientRect();
          // Compute the target X position.
          var x = itemRect.right - Private.SUBMENU_OVERLAP;
          // Adjust the X position to fit on the screen.
          if (x + width > px + cw) {
              x = itemRect.left + Private.SUBMENU_OVERLAP - width;
          }
          // Compute the target Y position.
          var y = itemRect.top - box.borderTop - box.paddingTop;
          // Adjust the Y position to fit on the screen.
          if (y + height > py + ch) {
              y = itemRect.bottom + box.borderBottom + box.paddingBottom - height;
          }
          // Update the position of the menu to the computed position.
          style.top = Math.max(0, y) + "px";
          style.left = Math.max(0, x) + "px";
          // Finally, make the menu visible on the screen.
          style.visibility = '';
      }
      Private.openSubmenu = openSubmenu;
      /**
       * Find the best matching mnemonic item.
       *
       * The search starts at the given index and wraps around.
       */
      function findMnemonic(items, key, start) {
          // Setup the result variables.
          var index = -1;
          var auto = -1;
          var multiple = false;
          // Normalize the key to upper case.
          var upperKey = key.toUpperCase();
          // Search the items from the given start index.
          for (var i = 0, n = items.length; i < n; ++i) {
              // Compute the wrapped index.
              var k = (i + start) % n;
              // Lookup the item
              var item = items[k];
              // Ignore items which cannot be activated.
              if (!canActivate(item)) {
                  continue;
              }
              // Ignore items with an empty label.
              var label = item.label;
              if (label.length === 0) {
                  continue;
              }
              // Lookup the mnemonic index for the label.
              var mn = item.mnemonic;
              // Handle a valid mnemonic index.
              if (mn >= 0 && mn < label.length) {
                  if (label[mn].toUpperCase() === upperKey) {
                      if (index === -1) {
                          index = k;
                      }
                      else {
                          multiple = true;
                      }
                  }
                  continue;
              }
              // Finally, handle the auto index if possible.
              if (auto === -1 && label[0].toUpperCase() === upperKey) {
                  auto = k;
              }
          }
          // Return the search results.
          return { index: index, multiple: multiple, auto: auto };
      }
      Private.findMnemonic = findMnemonic;
      /**
       * A concrete implementation of `Menu.IItem`.
       */
      var MenuItem = /** @class */ (function () {
          /**
           * Construct a new menu item.
           */
          function MenuItem(commands, options) {
              this._commands = commands;
              this.type = options.type || 'command';
              this.command = options.command || '';
              this.args = options.args || JSONExt.emptyObject;
              this.submenu = options.submenu || null;
          }
          Object.defineProperty(MenuItem.prototype, "label", {
              /**
               * The display label for the menu item.
               */
              get: function () {
                  if (this.type === 'command') {
                      return this._commands.label(this.command, this.args);
                  }
                  if (this.type === 'submenu' && this.submenu) {
                      return this.submenu.title.label;
                  }
                  return '';
              },
              enumerable: true,
              configurable: true
          });
          Object.defineProperty(MenuItem.prototype, "mnemonic", {
              /**
               * The mnemonic index for the menu item.
               */
              get: function () {
                  if (this.type === 'command') {
                      return this._commands.mnemonic(this.command, this.args);
                  }
                  if (this.type === 'submenu' && this.submenu) {
                      return this.submenu.title.mnemonic;
                  }
                  return -1;
              },
              enumerable: true,
              configurable: true
          });
          Object.defineProperty(MenuItem.prototype, "icon", {
              /**
               * The icon renderer for the menu item.
               */
              get: function () {
                  if (this.type === 'command') {
                      return this._commands.icon(this.command, this.args);
                  }
                  if (this.type === 'submenu' && this.submenu) {
                      return this.submenu.title.icon;
                  }
                  /* <DEPRECATED> */
                  // alias to icon class if not otherwise defined
                  return this.iconClass;
                  /* </DEPRECATED> */
                  /* <FUTURE>
                  return undefined;
                  </FUTURE> */
              },
              enumerable: true,
              configurable: true
          });
          Object.defineProperty(MenuItem.prototype, "iconClass", {
              /**
               * The icon class for the menu item.
               */
              get: function () {
                  if (this.type === 'command') {
                      return this._commands.iconClass(this.command, this.args);
                  }
                  if (this.type === 'submenu' && this.submenu) {
                      return this.submenu.title.iconClass;
                  }
                  return '';
              },
              enumerable: true,
              configurable: true
          });
          Object.defineProperty(MenuItem.prototype, "iconLabel", {
              /**
               * The icon label for the menu item.
               */
              get: function () {
                  if (this.type === 'command') {
                      return this._commands.iconLabel(this.command, this.args);
                  }
                  if (this.type === 'submenu' && this.submenu) {
                      return this.submenu.title.iconLabel;
                  }
                  return '';
              },
              enumerable: true,
              configurable: true
          });
          Object.defineProperty(MenuItem.prototype, "caption", {
              /**
               * The display caption for the menu item.
               */
              get: function () {
                  if (this.type === 'command') {
                      return this._commands.caption(this.command, this.args);
                  }
                  if (this.type === 'submenu' && this.submenu) {
                      return this.submenu.title.caption;
                  }
                  return '';
              },
              enumerable: true,
              configurable: true
          });
          Object.defineProperty(MenuItem.prototype, "className", {
              /**
               * The extra class name for the menu item.
               */
              get: function () {
                  if (this.type === 'command') {
                      return this._commands.className(this.command, this.args);
                  }
                  if (this.type === 'submenu' && this.submenu) {
                      return this.submenu.title.className;
                  }
                  return '';
              },
              enumerable: true,
              configurable: true
          });
          Object.defineProperty(MenuItem.prototype, "dataset", {
              /**
               * The dataset for the menu item.
               */
              get: function () {
                  if (this.type === 'command') {
                      return this._commands.dataset(this.command, this.args);
                  }
                  if (this.type === 'submenu' && this.submenu) {
                      return this.submenu.title.dataset;
                  }
                  return {};
              },
              enumerable: true,
              configurable: true
          });
          Object.defineProperty(MenuItem.prototype, "isEnabled", {
              /**
               * Whether the menu item is enabled.
               */
              get: function () {
                  if (this.type === 'command') {
                      return this._commands.isEnabled(this.command, this.args);
                  }
                  if (this.type === 'submenu') {
                      return this.submenu !== null;
                  }
                  return true;
              },
              enumerable: true,
              configurable: true
          });
          Object.defineProperty(MenuItem.prototype, "isToggled", {
              /**
               * Whether the menu item is toggled.
               */
              get: function () {
                  if (this.type === 'command') {
                      return this._commands.isToggled(this.command, this.args);
                  }
                  return false;
              },
              enumerable: true,
              configurable: true
          });
          Object.defineProperty(MenuItem.prototype, "isVisible", {
              /**
               * Whether the menu item is visible.
               */
              get: function () {
                  if (this.type === 'command') {
                      return this._commands.isVisible(this.command, this.args);
                  }
                  if (this.type === 'submenu') {
                      return this.submenu !== null;
                  }
                  return true;
              },
              enumerable: true,
              configurable: true
          });
          Object.defineProperty(MenuItem.prototype, "keyBinding", {
              /**
               * The key binding for the menu item.
               */
              get: function () {
                  if (this.type === 'command') {
                      var _a = this, command_1 = _a.command, args_1 = _a.args;
                      return ArrayExt.findLastValue(this._commands.keyBindings, function (kb) {
                          return kb.command === command_1 && JSONExt.deepEqual(kb.args, args_1);
                      }) || null;
                  }
                  return null;
              },
              enumerable: true,
              configurable: true
          });
          return MenuItem;
      }());
  })(Private$6$1 || (Private$6$1 = {}));

  /**
   * An object which implements a universal context menu.
   *
   * #### Notes
   * The items shown in the context menu are determined by CSS selector
   * matching against the DOM hierarchy at the site of the mouse click.
   * This is similar in concept to how keyboard shortcuts are matched
   * in the command registry.
   */
  var ContextMenu = /** @class */ (function () {
      /**
       * Construct a new context menu.
       *
       * @param options - The options for initializing the menu.
       */
      function ContextMenu(options) {
          this._idTick = 0;
          this._items = [];
          this.menu = new Menu(options);
      }
      /**
       * Add an item to the context menu.
       *
       * @param options - The options for creating the item.
       *
       * @returns A disposable which will remove the item from the menu.
       */
      ContextMenu.prototype.addItem = function (options) {
          var _this = this;
          // Create an item from the given options.
          var item = Private$7$1.createItem(options, this._idTick++);
          // Add the item to the internal array.
          this._items.push(item);
          // Return a disposable which will remove the item.
          return new DisposableDelegate(function () {
              ArrayExt.removeFirstOf(_this._items, item);
          });
      };
      /**
       * Open the context menu in response to a `'contextmenu'` event.
       *
       * @param event - The `'contextmenu'` event of interest.
       *
       * @returns `true` if the menu was opened, or `false` if no items
       *   matched the event and the menu was not opened.
       *
       * #### Notes
       * This method will populate the context menu with items which match
       * the propagation path of the event, then open the menu at the mouse
       * position indicated by the event.
       */
      ContextMenu.prototype.open = function (event) {
          var _this = this;
          // Clear the current contents of the context menu.
          this.menu.clearItems();
          // Bail early if there are no items to match.
          if (this._items.length === 0) {
              return false;
          }
          // Find the matching items for the event.
          var items = Private$7$1.matchItems(this._items, event);
          // Bail if there are no matching items.
          if (!items || items.length === 0) {
              return false;
          }
          // Add the filtered items to the menu.
          each$1(items, function (item) { _this.menu.addItem(item); });
          // Open the context menu at the current mouse position.
          this.menu.open(event.clientX, event.clientY);
          // Indicate success.
          return true;
      };
      return ContextMenu;
  }());
  /**
   * The namespace for the module implementation details.
   */
  var Private$7$1;
  (function (Private) {
      /**
       * Create a normalized context menu item from an options object.
       */
      function createItem(options, id) {
          var selector = validateSelector(options.selector);
          var rank = options.rank !== undefined ? options.rank : Infinity;
          return __assign$3(__assign$3({}, options), { selector: selector, rank: rank, id: id });
      }
      Private.createItem = createItem;
      /**
       * Find the items which match a context menu event.
       *
       * The results are sorted by DOM level, specificity, and rank.
       */
      function matchItems(items, event) {
          // Look up the target of the event.
          var target = event.target;
          // Bail if there is no target.
          if (!target) {
              return null;
          }
          // Look up the current target of the event.
          var currentTarget = event.currentTarget;
          // Bail if there is no current target.
          if (!currentTarget) {
              return null;
          }
          // There are some third party libraries that cause the `target` to
          // be detached from the DOM before lumino can process the event.
          // If that happens, search for a new target node by point. If that
          // node is still dangling, bail.
          if (!currentTarget.contains(target)) {
              target = document.elementFromPoint(event.clientX, event.clientY);
              if (!target || !currentTarget.contains(target)) {
                  return null;
              }
          }
          // Set up the result array.
          var result = [];
          // Copy the items array to allow in-place modification.
          var availableItems = items.slice();
          // Walk up the DOM hierarchy searching for matches.
          while (target !== null) {
              // Set up the match array for this DOM level.
              var matches = [];
              // Search the remaining items for matches.
              for (var i = 0, n = availableItems.length; i < n; ++i) {
                  // Fetch the item.
                  var item = availableItems[i];
                  // Skip items which are already consumed.
                  if (!item) {
                      continue;
                  }
                  // Skip items which do not match the element.
                  if (!Selector.matches(target, item.selector)) {
                      continue;
                  }
                  // Add the matched item to the result for this DOM level.
                  matches.push(item);
                  // Mark the item as consumed.
                  availableItems[i] = null;
              }
              // Sort the matches for this level and add them to the results.
              if (matches.length !== 0) {
                  matches.sort(itemCmp);
                  result.push.apply(result, matches);
              }
              // Stop searching at the limits of the DOM range.
              if (target === currentTarget) {
                  break;
              }
              // Step to the parent DOM level.
              target = target.parentElement;
          }
          // Return the matched and sorted results.
          return result;
      }
      Private.matchItems = matchItems;
      /**
       * Validate the selector for a menu item.
       *
       * This returns the validated selector, or throws if the selector is
       * invalid or contains commas.
       */
      function validateSelector(selector) {
          if (selector.indexOf(',') !== -1) {
              throw new Error("Selector cannot contain commas: " + selector);
          }
          if (!Selector.isValid(selector)) {
              throw new Error("Invalid selector: " + selector);
          }
          return selector;
      }
      /**
       * A sort comparison function for a context menu item.
       */
      function itemCmp(a, b) {
          // Sort first based on selector specificity.
          var s1 = Selector.calculateSpecificity(a.selector);
          var s2 = Selector.calculateSpecificity(b.selector);
          if (s1 !== s2) {
              return s2 - s1;
          }
          // If specificities are equal, sort based on rank.
          var r1 = a.rank;
          var r2 = b.rank;
          if (r1 !== r2) {
              return r1 < r2 ? -1 : 1; // Infinity-safe
          }
          // When all else fails, sort by item id.
          return a.id - b.id;
      }
  })(Private$7$1 || (Private$7$1 = {}));

  /**
   * A layout which provides a flexible docking arrangement.
   *
   * #### Notes
   * The consumer of this layout is responsible for handling all signals
   * from the generated tab bars and managing the visibility of widgets
   * and tab bars as needed.
   */
  var DockLayout = /** @class */ (function (_super) {
      __extends$5(DockLayout, _super);
      /**
       * Construct a new dock layout.
       *
       * @param options - The options for initializing the layout.
       */
      function DockLayout(options) {
          var _this = _super.call(this) || this;
          _this._spacing = 4;
          _this._dirty = false;
          _this._root = null;
          _this._box = null;
          _this._items = new Map();
          _this.renderer = options.renderer;
          if (options.spacing !== undefined) {
              _this._spacing = Private$8$1.clampSpacing(options.spacing);
          }
          return _this;
      }
      /**
       * Dispose of the resources held by the layout.
       *
       * #### Notes
       * This will clear and dispose all widgets in the layout.
       */
      DockLayout.prototype.dispose = function () {
          // Get an iterator over the widgets in the layout.
          var widgets = this.iter();
          // Dispose of the layout items.
          this._items.forEach(function (item) { item.dispose(); });
          // Clear the layout state before disposing the widgets.
          this._box = null;
          this._root = null;
          this._items.clear();
          // Dispose of the widgets contained in the old layout root.
          each$1(widgets, function (widget) { widget.dispose(); });
          // Dispose of the base class.
          _super.prototype.dispose.call(this);
      };
      Object.defineProperty(DockLayout.prototype, "spacing", {
          /**
           * Get the inter-element spacing for the dock layout.
           */
          get: function () {
              return this._spacing;
          },
          /**
           * Set the inter-element spacing for the dock layout.
           */
          set: function (value) {
              value = Private$8$1.clampSpacing(value);
              if (this._spacing === value) {
                  return;
              }
              this._spacing = value;
              if (!this.parent) {
                  return;
              }
              this.parent.fit();
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(DockLayout.prototype, "isEmpty", {
          /**
           * Whether the dock layout is empty.
           */
          get: function () {
              return this._root === null;
          },
          enumerable: true,
          configurable: true
      });
      /**
       * Create an iterator over all widgets in the layout.
       *
       * @returns A new iterator over the widgets in the layout.
       *
       * #### Notes
       * This iterator includes the generated tab bars.
       */
      DockLayout.prototype.iter = function () {
          return this._root ? this._root.iterAllWidgets() : empty();
      };
      /**
       * Create an iterator over the user widgets in the layout.
       *
       * @returns A new iterator over the user widgets in the layout.
       *
       * #### Notes
       * This iterator does not include the generated tab bars.
       */
      DockLayout.prototype.widgets = function () {
          return this._root ? this._root.iterUserWidgets() : empty();
      };
      /**
       * Create an iterator over the selected widgets in the layout.
       *
       * @returns A new iterator over the selected user widgets.
       *
       * #### Notes
       * This iterator yields the widgets corresponding to the current tab
       * of each tab bar in the layout.
       */
      DockLayout.prototype.selectedWidgets = function () {
          return this._root ? this._root.iterSelectedWidgets() : empty();
      };
      /**
       * Create an iterator over the tab bars in the layout.
       *
       * @returns A new iterator over the tab bars in the layout.
       *
       * #### Notes
       * This iterator does not include the user widgets.
       */
      DockLayout.prototype.tabBars = function () {
          return this._root ? this._root.iterTabBars() : empty();
      };
      /**
       * Create an iterator over the handles in the layout.
       *
       * @returns A new iterator over the handles in the layout.
       */
      DockLayout.prototype.handles = function () {
          return this._root ? this._root.iterHandles() : empty();
      };
      /**
       * Move a handle to the given offset position.
       *
       * @param handle - The handle to move.
       *
       * @param offsetX - The desired offset X position of the handle.
       *
       * @param offsetY - The desired offset Y position of the handle.
       *
       * #### Notes
       * If the given handle is not contained in the layout, this is no-op.
       *
       * The handle will be moved as close as possible to the desired
       * position without violating any of the layout constraints.
       *
       * Only one of the coordinates is used depending on the orientation
       * of the handle. This method accepts both coordinates to make it
       * easy to invoke from a mouse move event without needing to know
       * the handle orientation.
       */
      DockLayout.prototype.moveHandle = function (handle, offsetX, offsetY) {
          // Bail early if there is no root or if the handle is hidden.
          var hidden = handle.classList.contains('lm-mod-hidden');
          /* <DEPRECATED> */
          hidden = hidden || handle.classList.contains('p-mod-hidden');
          /* </DEPRECATED> */
          if (!this._root || hidden) {
              return;
          }
          // Lookup the split node for the handle.
          var data = this._root.findSplitNode(handle);
          if (!data) {
              return;
          }
          // Compute the desired delta movement for the handle.
          var delta;
          if (data.node.orientation === 'horizontal') {
              delta = offsetX - handle.offsetLeft;
          }
          else {
              delta = offsetY - handle.offsetTop;
          }
          // Bail if there is no handle movement.
          if (delta === 0) {
              return;
          }
          // Prevent sibling resizing unless needed.
          data.node.holdSizes();
          // Adjust the sizers to reflect the handle movement.
          BoxEngine.adjust(data.node.sizers, data.index, delta);
          // Update the layout of the widgets.
          if (this.parent) {
              this.parent.update();
          }
      };
      /**
       * Save the current configuration of the dock layout.
       *
       * @returns A new config object for the current layout state.
       *
       * #### Notes
       * The return value can be provided to the `restoreLayout` method
       * in order to restore the layout to its current configuration.
       */
      DockLayout.prototype.saveLayout = function () {
          // Bail early if there is no root.
          if (!this._root) {
              return { main: null };
          }
          // Hold the current sizes in the layout tree.
          this._root.holdAllSizes();
          // Return the layout config.
          return { main: this._root.createConfig() };
      };
      /**
       * Restore the layout to a previously saved configuration.
       *
       * @param config - The layout configuration to restore.
       *
       * #### Notes
       * Widgets which currently belong to the layout but which are not
       * contained in the config will be unparented.
       */
      DockLayout.prototype.restoreLayout = function (config) {
          var _this = this;
          // Create the widget set for validating the config.
          var widgetSet = new Set();
          // Normalize the main area config and collect the widgets.
          var mainConfig;
          if (config.main) {
              mainConfig = Private$8$1.normalizeAreaConfig(config.main, widgetSet);
          }
          else {
              mainConfig = null;
          }
          // Create iterators over the old content.
          var oldWidgets = this.widgets();
          var oldTabBars = this.tabBars();
          var oldHandles = this.handles();
          // Clear the root before removing the old content.
          this._root = null;
          // Unparent the old widgets which are not in the new config.
          each$1(oldWidgets, function (widget) {
              if (!widgetSet.has(widget)) {
                  widget.parent = null;
              }
          });
          // Dispose of the old tab bars.
          each$1(oldTabBars, function (tabBar) {
              tabBar.dispose();
          });
          // Remove the old handles.
          each$1(oldHandles, function (handle) {
              if (handle.parentNode) {
                  handle.parentNode.removeChild(handle);
              }
          });
          // Reparent the new widgets to the current parent.
          widgetSet.forEach(function (widget) {
              widget.parent = _this.parent;
          });
          // Create the root node for the new config.
          if (mainConfig) {
              this._root = Private$8$1.realizeAreaConfig(mainConfig, {
                  createTabBar: function () { return _this._createTabBar(); },
                  createHandle: function () { return _this._createHandle(); }
              });
          }
          else {
              this._root = null;
          }
          // If there is no parent, there is nothing more to do.
          if (!this.parent) {
              return;
          }
          // Attach the new widgets to the parent.
          widgetSet.forEach(function (widget) {
              _this.attachWidget(widget);
          });
          // Post a fit request to the parent.
          this.parent.fit();
      };
      /**
       * Add a widget to the dock layout.
       *
       * @param widget - The widget to add to the dock layout.
       *
       * @param options - The additional options for adding the widget.
       *
       * #### Notes
       * The widget will be moved if it is already contained in the layout.
       *
       * An error will be thrown if the reference widget is invalid.
       */
      DockLayout.prototype.addWidget = function (widget, options) {
          if (options === void 0) { options = {}; }
          // Parse the options.
          var ref = options.ref || null;
          var mode = options.mode || 'tab-after';
          // Find the tab node which holds the reference widget.
          var refNode = null;
          if (this._root && ref) {
              refNode = this._root.findTabNode(ref);
          }
          // Throw an error if the reference widget is invalid.
          if (ref && !refNode) {
              throw new Error('Reference widget is not in the layout.');
          }
          // Reparent the widget to the current layout parent.
          widget.parent = this.parent;
          // Insert the widget according to the insert mode.
          switch (mode) {
              case 'tab-after':
                  this._insertTab(widget, ref, refNode, true);
                  break;
              case 'tab-before':
                  this._insertTab(widget, ref, refNode, false);
                  break;
              case 'split-top':
                  this._insertSplit(widget, ref, refNode, 'vertical', false);
                  break;
              case 'split-left':
                  this._insertSplit(widget, ref, refNode, 'horizontal', false);
                  break;
              case 'split-right':
                  this._insertSplit(widget, ref, refNode, 'horizontal', true);
                  break;
              case 'split-bottom':
                  this._insertSplit(widget, ref, refNode, 'vertical', true);
                  break;
          }
          // Do nothing else if there is no parent widget.
          if (!this.parent) {
              return;
          }
          // Ensure the widget is attached to the parent widget.
          this.attachWidget(widget);
          // Post a fit request for the parent widget.
          this.parent.fit();
      };
      /**
       * Remove a widget from the layout.
       *
       * @param widget - The widget to remove from the layout.
       *
       * #### Notes
       * A widget is automatically removed from the layout when its `parent`
       * is set to `null`. This method should only be invoked directly when
       * removing a widget from a layout which has yet to be installed on a
       * parent widget.
       *
       * This method does *not* modify the widget's `parent`.
       */
      DockLayout.prototype.removeWidget = function (widget) {
          // Remove the widget from its current layout location.
          this._removeWidget(widget);
          // Do nothing else if there is no parent widget.
          if (!this.parent) {
              return;
          }
          // Detach the widget from the parent widget.
          this.detachWidget(widget);
          // Post a fit request for the parent widget.
          this.parent.fit();
      };
      /**
       * Find the tab area which contains the given client position.
       *
       * @param clientX - The client X position of interest.
       *
       * @param clientY - The client Y position of interest.
       *
       * @returns The geometry of the tab area at the given position, or
       *   `null` if there is no tab area at the given position.
       */
      DockLayout.prototype.hitTestTabAreas = function (clientX, clientY) {
          // Bail early if hit testing cannot produce valid results.
          if (!this._root || !this.parent || !this.parent.isVisible) {
              return null;
          }
          // Ensure the parent box sizing data is computed.
          if (!this._box) {
              this._box = ElementExt.boxSizing(this.parent.node);
          }
          // Convert from client to local coordinates.
          var rect = this.parent.node.getBoundingClientRect();
          var x = clientX - rect.left - this._box.borderLeft;
          var y = clientY - rect.top - this._box.borderTop;
          // Find the tab layout node at the local position.
          var tabNode = this._root.hitTestTabNodes(x, y);
          // Bail if a tab layout node was not found.
          if (!tabNode) {
              return null;
          }
          // Extract the data from the tab node.
          var tabBar = tabNode.tabBar, top = tabNode.top, left = tabNode.left, width = tabNode.width, height = tabNode.height;
          // Compute the right and bottom edges of the tab area.
          var borderWidth = this._box.borderLeft + this._box.borderRight;
          var borderHeight = this._box.borderTop + this._box.borderBottom;
          var right = rect.width - borderWidth - (left + width);
          var bottom = rect.height - borderHeight - (top + height);
          // Return the hit test results.
          return { tabBar: tabBar, x: x, y: y, top: top, left: left, right: right, bottom: bottom, width: width, height: height };
      };
      /**
       * Perform layout initialization which requires the parent widget.
       */
      DockLayout.prototype.init = function () {
          var _this = this;
          // Perform superclass initialization.
          _super.prototype.init.call(this);
          // Attach each widget to the parent.
          each$1(this, function (widget) { _this.attachWidget(widget); });
          // Attach each handle to the parent.
          each$1(this.handles(), function (handle) { _this.parent.node.appendChild(handle); });
          // Post a fit request for the parent widget.
          this.parent.fit();
      };
      /**
       * Attach the widget to the layout parent widget.
       *
       * @param widget - The widget to attach to the parent.
       *
       * #### Notes
       * This is a no-op if the widget is already attached.
       */
      DockLayout.prototype.attachWidget = function (widget) {
          // Do nothing if the widget is already attached.
          if (this.parent.node === widget.node.parentNode) {
              return;
          }
          // Create the layout item for the widget.
          this._items.set(widget, new LayoutItem(widget));
          // Send a `'before-attach'` message if the parent is attached.
          if (this.parent.isAttached) {
              MessageLoop.sendMessage(widget, Widget.Msg.BeforeAttach);
          }
          // Add the widget's node to the parent.
          this.parent.node.appendChild(widget.node);
          // Send an `'after-attach'` message if the parent is attached.
          if (this.parent.isAttached) {
              MessageLoop.sendMessage(widget, Widget.Msg.AfterAttach);
          }
      };
      /**
       * Detach the widget from the layout parent widget.
       *
       * @param widget - The widget to detach from the parent.
       *
       * #### Notes
       * This is a no-op if the widget is not attached.
       */
      DockLayout.prototype.detachWidget = function (widget) {
          // Do nothing if the widget is not attached.
          if (this.parent.node !== widget.node.parentNode) {
              return;
          }
          // Send a `'before-detach'` message if the parent is attached.
          if (this.parent.isAttached) {
              MessageLoop.sendMessage(widget, Widget.Msg.BeforeDetach);
          }
          // Remove the widget's node from the parent.
          this.parent.node.removeChild(widget.node);
          // Send an `'after-detach'` message if the parent is attached.
          if (this.parent.isAttached) {
              MessageLoop.sendMessage(widget, Widget.Msg.AfterDetach);
          }
          // Delete the layout item for the widget.
          var item = this._items.get(widget);
          if (item) {
              this._items.delete(widget);
              item.dispose();
          }
      };
      /**
       * A message handler invoked on a `'before-show'` message.
       */
      DockLayout.prototype.onBeforeShow = function (msg) {
          _super.prototype.onBeforeShow.call(this, msg);
          this.parent.update();
      };
      /**
       * A message handler invoked on a `'before-attach'` message.
       */
      DockLayout.prototype.onBeforeAttach = function (msg) {
          _super.prototype.onBeforeAttach.call(this, msg);
          this.parent.fit();
      };
      /**
       * A message handler invoked on a `'child-shown'` message.
       */
      DockLayout.prototype.onChildShown = function (msg) {
          this.parent.fit();
      };
      /**
       * A message handler invoked on a `'child-hidden'` message.
       */
      DockLayout.prototype.onChildHidden = function (msg) {
          this.parent.fit();
      };
      /**
       * A message handler invoked on a `'resize'` message.
       */
      DockLayout.prototype.onResize = function (msg) {
          if (this.parent.isVisible) {
              this._update(msg.width, msg.height);
          }
      };
      /**
       * A message handler invoked on an `'update-request'` message.
       */
      DockLayout.prototype.onUpdateRequest = function (msg) {
          if (this.parent.isVisible) {
              this._update(-1, -1);
          }
      };
      /**
       * A message handler invoked on a `'fit-request'` message.
       */
      DockLayout.prototype.onFitRequest = function (msg) {
          if (this.parent.isAttached) {
              this._fit();
          }
      };
      /**
       * Remove the specified widget from the layout structure.
       *
       * #### Notes
       * This is a no-op if the widget is not in the layout tree.
       *
       * This does not detach the widget from the parent node.
       */
      DockLayout.prototype._removeWidget = function (widget) {
          // Bail early if there is no layout root.
          if (!this._root) {
              return;
          }
          // Find the tab node which contains the given widget.
          var tabNode = this._root.findTabNode(widget);
          // Bail early if the tab node is not found.
          if (!tabNode) {
              return;
          }
          // If there are multiple tabs, just remove the widget's tab.
          if (tabNode.tabBar.titles.length > 1) {
              tabNode.tabBar.removeTab(widget.title);
              return;
          }
          // Otherwise, the tab node needs to be removed...
          // Dispose the tab bar.
          tabNode.tabBar.dispose();
          // Handle the case where the tab node is the root.
          if (this._root === tabNode) {
              this._root = null;
              return;
          }
          // Otherwise, remove the tab node from its parent...
          // Prevent widget resizing unless needed.
          this._root.holdAllSizes();
          // Clear the parent reference on the tab node.
          var splitNode = tabNode.parent;
          tabNode.parent = null;
          // Remove the tab node from its parent split node.
          var i = ArrayExt.removeFirstOf(splitNode.children, tabNode);
          var handle = ArrayExt.removeAt(splitNode.handles, i);
          ArrayExt.removeAt(splitNode.sizers, i);
          // Remove the handle from its parent DOM node.
          if (handle.parentNode) {
              handle.parentNode.removeChild(handle);
          }
          // If there are multiple children, just update the handles.
          if (splitNode.children.length > 1) {
              splitNode.syncHandles();
              return;
          }
          // Otherwise, the split node also needs to be removed...
          // Clear the parent reference on the split node.
          var maybeParent = splitNode.parent;
          splitNode.parent = null;
          // Lookup the remaining child node and handle.
          var childNode = splitNode.children[0];
          var childHandle = splitNode.handles[0];
          // Clear the split node data.
          splitNode.children.length = 0;
          splitNode.handles.length = 0;
          splitNode.sizers.length = 0;
          // Remove the child handle from its parent node.
          if (childHandle.parentNode) {
              childHandle.parentNode.removeChild(childHandle);
          }
          // Handle the case where the split node is the root.
          if (this._root === splitNode) {
              childNode.parent = null;
              this._root = childNode;
              return;
          }
          // Otherwise, move the child node to the parent node...
          var parentNode = maybeParent;
          // Lookup the index of the split node.
          var j = parentNode.children.indexOf(splitNode);
          // Handle the case where the child node is a tab node.
          if (childNode instanceof Private$8$1.TabLayoutNode) {
              childNode.parent = parentNode;
              parentNode.children[j] = childNode;
              return;
          }
          // Remove the split data from the parent.
          var splitHandle = ArrayExt.removeAt(parentNode.handles, j);
          ArrayExt.removeAt(parentNode.children, j);
          ArrayExt.removeAt(parentNode.sizers, j);
          // Remove the handle from its parent node.
          if (splitHandle.parentNode) {
              splitHandle.parentNode.removeChild(splitHandle);
          }
          // The child node and the split parent node will have the same
          // orientation. Merge the grand-children with the parent node.
          for (var i_1 = 0, n = childNode.children.length; i_1 < n; ++i_1) {
              var gChild = childNode.children[i_1];
              var gHandle = childNode.handles[i_1];
              var gSizer = childNode.sizers[i_1];
              ArrayExt.insert(parentNode.children, j + i_1, gChild);
              ArrayExt.insert(parentNode.handles, j + i_1, gHandle);
              ArrayExt.insert(parentNode.sizers, j + i_1, gSizer);
              gChild.parent = parentNode;
          }
          // Clear the child node.
          childNode.children.length = 0;
          childNode.handles.length = 0;
          childNode.sizers.length = 0;
          childNode.parent = null;
          // Sync the handles on the parent node.
          parentNode.syncHandles();
      };
      /**
       * Insert a widget next to an existing tab.
       *
       * #### Notes
       * This does not attach the widget to the parent widget.
       */
      DockLayout.prototype._insertTab = function (widget, ref, refNode, after) {
          // Do nothing if the tab is inserted next to itself.
          if (widget === ref) {
              return;
          }
          // Create the root if it does not exist.
          if (!this._root) {
              var tabNode = new Private$8$1.TabLayoutNode(this._createTabBar());
              tabNode.tabBar.addTab(widget.title);
              this._root = tabNode;
              return;
          }
          // Use the first tab node as the ref node if needed.
          if (!refNode) {
              refNode = this._root.findFirstTabNode();
          }
          // If the widget is not contained in the ref node, ensure it is
          // removed from the layout and hidden before being added again.
          if (refNode.tabBar.titles.indexOf(widget.title) === -1) {
              this._removeWidget(widget);
              widget.hide();
          }
          // Lookup the target index for inserting the tab.
          var index;
          if (ref) {
              index = refNode.tabBar.titles.indexOf(ref.title);
          }
          else {
              index = refNode.tabBar.currentIndex;
          }
          // Insert the widget's tab relative to the target index.
          refNode.tabBar.insertTab(index + (after ? 1 : 0), widget.title);
      };
      /**
       * Insert a widget as a new split area.
       *
       * #### Notes
       * This does not attach the widget to the parent widget.
       */
      DockLayout.prototype._insertSplit = function (widget, ref, refNode, orientation, after) {
          // Do nothing if there is no effective split.
          if (widget === ref && refNode && refNode.tabBar.titles.length === 1) {
              return;
          }
          // Ensure the widget is removed from the current layout.
          this._removeWidget(widget);
          // Create the tab layout node to hold the widget.
          var tabNode = new Private$8$1.TabLayoutNode(this._createTabBar());
          tabNode.tabBar.addTab(widget.title);
          // Set the root if it does not exist.
          if (!this._root) {
              this._root = tabNode;
              return;
          }
          // If the ref node parent is null, split the root.
          if (!refNode || !refNode.parent) {
              // Ensure the root is split with the correct orientation.
              var root = this._splitRoot(orientation);
              // Determine the insert index for the new tab node.
              var i_2 = after ? root.children.length : 0;
              // Normalize the split node.
              root.normalizeSizes();
              // Create the sizer for new tab node.
              var sizer = Private$8$1.createSizer(refNode ? 1 : Private$8$1.GOLDEN_RATIO);
              // Insert the tab node sized to the golden ratio.
              ArrayExt.insert(root.children, i_2, tabNode);
              ArrayExt.insert(root.sizers, i_2, sizer);
              ArrayExt.insert(root.handles, i_2, this._createHandle());
              tabNode.parent = root;
              // Re-normalize the split node to maintain the ratios.
              root.normalizeSizes();
              // Finally, synchronize the visibility of the handles.
              root.syncHandles();
              return;
          }
          // Lookup the split node for the ref widget.
          var splitNode = refNode.parent;
          // If the split node already had the correct orientation,
          // the widget can be inserted into the split node directly.
          if (splitNode.orientation === orientation) {
              // Find the index of the ref node.
              var i_3 = splitNode.children.indexOf(refNode);
              // Normalize the split node.
              splitNode.normalizeSizes();
              // Consume half the space for the insert location.
              var s = splitNode.sizers[i_3].sizeHint /= 2;
              // Insert the tab node sized to the other half.
              var j_1 = i_3 + (after ? 1 : 0);
              ArrayExt.insert(splitNode.children, j_1, tabNode);
              ArrayExt.insert(splitNode.sizers, j_1, Private$8$1.createSizer(s));
              ArrayExt.insert(splitNode.handles, j_1, this._createHandle());
              tabNode.parent = splitNode;
              // Finally, synchronize the visibility of the handles.
              splitNode.syncHandles();
              return;
          }
          // Remove the ref node from the split node.
          var i = ArrayExt.removeFirstOf(splitNode.children, refNode);
          // Create a new normalized split node for the children.
          var childNode = new Private$8$1.SplitLayoutNode(orientation);
          childNode.normalized = true;
          // Add the ref node sized to half the space.
          childNode.children.push(refNode);
          childNode.sizers.push(Private$8$1.createSizer(0.5));
          childNode.handles.push(this._createHandle());
          refNode.parent = childNode;
          // Add the tab node sized to the other half.
          var j = after ? 1 : 0;
          ArrayExt.insert(childNode.children, j, tabNode);
          ArrayExt.insert(childNode.sizers, j, Private$8$1.createSizer(0.5));
          ArrayExt.insert(childNode.handles, j, this._createHandle());
          tabNode.parent = childNode;
          // Synchronize the visibility of the handles.
          childNode.syncHandles();
          // Finally, add the new child node to the original split node.
          ArrayExt.insert(splitNode.children, i, childNode);
          childNode.parent = splitNode;
      };
      /**
       * Ensure the root is a split node with the given orientation.
       */
      DockLayout.prototype._splitRoot = function (orientation) {
          // Bail early if the root already meets the requirements.
          var oldRoot = this._root;
          if (oldRoot instanceof Private$8$1.SplitLayoutNode) {
              if (oldRoot.orientation === orientation) {
                  return oldRoot;
              }
          }
          // Create a new root node with the specified orientation.
          var newRoot = this._root = new Private$8$1.SplitLayoutNode(orientation);
          // Add the old root to the new root.
          if (oldRoot) {
              newRoot.children.push(oldRoot);
              newRoot.sizers.push(Private$8$1.createSizer(0));
              newRoot.handles.push(this._createHandle());
              oldRoot.parent = newRoot;
          }
          // Return the new root as a convenience.
          return newRoot;
      };
      /**
       * Fit the layout to the total size required by the widgets.
       */
      DockLayout.prototype._fit = function () {
          // Set up the computed minimum size.
          var minW = 0;
          var minH = 0;
          // Update the size limits for the layout tree.
          if (this._root) {
              var limits = this._root.fit(this._spacing, this._items);
              minW = limits.minWidth;
              minH = limits.minHeight;
          }
          // Update the box sizing and add it to the computed min size.
          var box = this._box = ElementExt.boxSizing(this.parent.node);
          minW += box.horizontalSum;
          minH += box.verticalSum;
          // Update the parent's min size constraints.
          var style = this.parent.node.style;
          style.minWidth = minW + "px";
          style.minHeight = minH + "px";
          // Set the dirty flag to ensure only a single update occurs.
          this._dirty = true;
          // Notify the ancestor that it should fit immediately. This may
          // cause a resize of the parent, fulfilling the required update.
          if (this.parent.parent) {
              MessageLoop.sendMessage(this.parent.parent, Widget.Msg.FitRequest);
          }
          // If the dirty flag is still set, the parent was not resized.
          // Trigger the required update on the parent widget immediately.
          if (this._dirty) {
              MessageLoop.sendMessage(this.parent, Widget.Msg.UpdateRequest);
          }
      };
      /**
       * Update the layout position and size of the widgets.
       *
       * The parent offset dimensions should be `-1` if unknown.
       */
      DockLayout.prototype._update = function (offsetWidth, offsetHeight) {
          // Clear the dirty flag to indicate the update occurred.
          this._dirty = false;
          // Bail early if there is no root layout node.
          if (!this._root) {
              return;
          }
          // Measure the parent if the offset dimensions are unknown.
          if (offsetWidth < 0) {
              offsetWidth = this.parent.node.offsetWidth;
          }
          if (offsetHeight < 0) {
              offsetHeight = this.parent.node.offsetHeight;
          }
          // Ensure the parent box sizing data is computed.
          if (!this._box) {
              this._box = ElementExt.boxSizing(this.parent.node);
          }
          // Compute the actual layout bounds adjusted for border and padding.
          var x = this._box.paddingTop;
          var y = this._box.paddingLeft;
          var width = offsetWidth - this._box.horizontalSum;
          var height = offsetHeight - this._box.verticalSum;
          // Update the geometry of the layout tree.
          this._root.update(x, y, width, height, this._spacing, this._items);
      };
      /**
       * Create a new tab bar for use by the dock layout.
       *
       * #### Notes
       * The tab bar will be attached to the parent if it exists.
       */
      DockLayout.prototype._createTabBar = function () {
          // Create the tab bar using the renderer.
          var tabBar = this.renderer.createTabBar();
          // Enforce necessary tab bar behavior.
          tabBar.orientation = 'horizontal';
          // Reparent and attach the tab bar to the parent if possible.
          if (this.parent) {
              tabBar.parent = this.parent;
              this.attachWidget(tabBar);
          }
          // Return the initialized tab bar.
          return tabBar;
      };
      /**
       * Create a new handle for the dock layout.
       *
       * #### Notes
       * The handle will be attached to the parent if it exists.
       */
      DockLayout.prototype._createHandle = function () {
          // Create the handle using the renderer.
          var handle = this.renderer.createHandle();
          // Initialize the handle layout behavior.
          var style = handle.style;
          style.position = 'absolute';
          style.top = '0';
          style.left = '0';
          style.width = '0';
          style.height = '0';
          // Attach the handle to the parent if it exists.
          if (this.parent) {
              this.parent.node.appendChild(handle);
          }
          // Return the initialized handle.
          return handle;
      };
      return DockLayout;
  }(Layout));
  /**
   * The namespace for the module implementation details.
   */
  var Private$8$1;
  (function (Private) {
      /**
       * A fraction used for sizing root panels; ~= `1 / golden_ratio`.
       */
      Private.GOLDEN_RATIO = 0.618;
      /**
       * Clamp a spacing value to an integer >= 0.
       */
      function clampSpacing(value) {
          return Math.max(0, Math.floor(value));
      }
      Private.clampSpacing = clampSpacing;
      /**
       * Create a box sizer with an initial size hint.
       */
      function createSizer(hint) {
          var sizer = new BoxSizer();
          sizer.sizeHint = hint;
          sizer.size = hint;
          return sizer;
      }
      Private.createSizer = createSizer;
      /**
       * Normalize an area config object and collect the visited widgets.
       */
      function normalizeAreaConfig(config, widgetSet) {
          var result;
          if (config.type === 'tab-area') {
              result = normalizeTabAreaConfig(config, widgetSet);
          }
          else {
              result = normalizeSplitAreaConfig(config, widgetSet);
          }
          return result;
      }
      Private.normalizeAreaConfig = normalizeAreaConfig;
      /**
       * Convert a normalized area config into a layout tree.
       */
      function realizeAreaConfig(config, renderer) {
          var node;
          if (config.type === 'tab-area') {
              node = realizeTabAreaConfig(config, renderer);
          }
          else {
              node = realizeSplitAreaConfig(config, renderer);
          }
          return node;
      }
      Private.realizeAreaConfig = realizeAreaConfig;
      /**
       * A layout node which holds the data for a tabbed area.
       */
      var TabLayoutNode = /** @class */ (function () {
          /**
           * Construct a new tab layout node.
           *
           * @param tabBar - The tab bar to use for the layout node.
           */
          function TabLayoutNode(tabBar) {
              /**
               * The parent of the layout node.
               */
              this.parent = null;
              this._top = 0;
              this._left = 0;
              this._width = 0;
              this._height = 0;
              var tabSizer = new BoxSizer();
              var widgetSizer = new BoxSizer();
              tabSizer.stretch = 0;
              widgetSizer.stretch = 1;
              this.tabBar = tabBar;
              this.sizers = [tabSizer, widgetSizer];
          }
          Object.defineProperty(TabLayoutNode.prototype, "top", {
              /**
               * The most recent value for the `top` edge of the layout box.
               */
              get: function () {
                  return this._top;
              },
              enumerable: true,
              configurable: true
          });
          Object.defineProperty(TabLayoutNode.prototype, "left", {
              /**
               * The most recent value for the `left` edge of the layout box.
               */
              get: function () {
                  return this._left;
              },
              enumerable: true,
              configurable: true
          });
          Object.defineProperty(TabLayoutNode.prototype, "width", {
              /**
               * The most recent value for the `width` of the layout box.
               */
              get: function () {
                  return this._width;
              },
              enumerable: true,
              configurable: true
          });
          Object.defineProperty(TabLayoutNode.prototype, "height", {
              /**
               * The most recent value for the `height` of the layout box.
               */
              get: function () {
                  return this._height;
              },
              enumerable: true,
              configurable: true
          });
          /**
           * Create an iterator for all widgets in the layout tree.
           */
          TabLayoutNode.prototype.iterAllWidgets = function () {
              return chain$1(once$1(this.tabBar), this.iterUserWidgets());
          };
          /**
           * Create an iterator for the user widgets in the layout tree.
           */
          TabLayoutNode.prototype.iterUserWidgets = function () {
              return map$1(this.tabBar.titles, function (title) { return title.owner; });
          };
          /**
           * Create an iterator for the selected widgets in the layout tree.
           */
          TabLayoutNode.prototype.iterSelectedWidgets = function () {
              var title = this.tabBar.currentTitle;
              return title ? once$1(title.owner) : empty();
          };
          /**
           * Create an iterator for the tab bars in the layout tree.
           */
          TabLayoutNode.prototype.iterTabBars = function () {
              return once$1(this.tabBar);
          };
          /**
           * Create an iterator for the handles in the layout tree.
           */
          TabLayoutNode.prototype.iterHandles = function () {
              return empty();
          };
          /**
           * Find the tab layout node which contains the given widget.
           */
          TabLayoutNode.prototype.findTabNode = function (widget) {
              return this.tabBar.titles.indexOf(widget.title) !== -1 ? this : null;
          };
          /**
           * Find the split layout node which contains the given handle.
           */
          TabLayoutNode.prototype.findSplitNode = function (handle) {
              return null;
          };
          /**
           * Find the first tab layout node in a layout tree.
           */
          TabLayoutNode.prototype.findFirstTabNode = function () {
              return this;
          };
          /**
           * Find the tab layout node which contains the local point.
           */
          TabLayoutNode.prototype.hitTestTabNodes = function (x, y) {
              if (x < this._left || x >= this._left + this._width) {
                  return null;
              }
              if (y < this._top || y >= this._top + this._height) {
                  return null;
              }
              return this;
          };
          /**
           * Create a configuration object for the layout tree.
           */
          TabLayoutNode.prototype.createConfig = function () {
              var widgets = this.tabBar.titles.map(function (title) { return title.owner; });
              var currentIndex = this.tabBar.currentIndex;
              return { type: 'tab-area', widgets: widgets, currentIndex: currentIndex };
          };
          /**
           * Recursively hold all of the sizes in the layout tree.
           *
           * This ignores the sizers of tab layout nodes.
           */
          TabLayoutNode.prototype.holdAllSizes = function () {
              return;
          };
          /**
           * Fit the layout tree.
           */
          TabLayoutNode.prototype.fit = function (spacing, items) {
              // Set up the limit variables.
              var minWidth = 0;
              var minHeight = 0;
              var maxWidth = Infinity;
              var maxHeight = Infinity;
              // Lookup the tab bar layout item.
              var tabBarItem = items.get(this.tabBar);
              // Lookup the widget layout item.
              var current = this.tabBar.currentTitle;
              var widgetItem = current ? items.get(current.owner) : undefined;
              // Lookup the tab bar and widget sizers.
              var _a = this.sizers, tabBarSizer = _a[0], widgetSizer = _a[1];
              // Update the tab bar limits.
              if (tabBarItem) {
                  tabBarItem.fit();
              }
              // Update the widget limits.
              if (widgetItem) {
                  widgetItem.fit();
              }
              // Update the results and sizer for the tab bar.
              if (tabBarItem && !tabBarItem.isHidden) {
                  minWidth = Math.max(minWidth, tabBarItem.minWidth);
                  minHeight += tabBarItem.minHeight;
                  tabBarSizer.minSize = tabBarItem.minHeight;
                  tabBarSizer.maxSize = tabBarItem.maxHeight;
              }
              else {
                  tabBarSizer.minSize = 0;
                  tabBarSizer.maxSize = 0;
              }
              // Update the results and sizer for the current widget.
              if (widgetItem && !widgetItem.isHidden) {
                  minWidth = Math.max(minWidth, widgetItem.minWidth);
                  minHeight += widgetItem.minHeight;
                  widgetSizer.minSize = widgetItem.minHeight;
                  widgetSizer.maxSize = Infinity;
              }
              else {
                  widgetSizer.minSize = 0;
                  widgetSizer.maxSize = Infinity;
              }
              // Return the computed size limits for the layout node.
              return { minWidth: minWidth, minHeight: minHeight, maxWidth: maxWidth, maxHeight: maxHeight };
          };
          /**
           * Update the layout tree.
           */
          TabLayoutNode.prototype.update = function (left, top, width, height, spacing, items) {
              // Update the layout box values.
              this._top = top;
              this._left = left;
              this._width = width;
              this._height = height;
              // Lookup the tab bar layout item.
              var tabBarItem = items.get(this.tabBar);
              // Lookup the widget layout item.
              var current = this.tabBar.currentTitle;
              var widgetItem = current ? items.get(current.owner) : undefined;
              // Distribute the layout space to the sizers.
              BoxEngine.calc(this.sizers, height);
              // Update the tab bar item using the computed size.
              if (tabBarItem && !tabBarItem.isHidden) {
                  var size = this.sizers[0].size;
                  tabBarItem.update(left, top, width, size);
                  top += size;
              }
              // Layout the widget using the computed size.
              if (widgetItem && !widgetItem.isHidden) {
                  var size = this.sizers[1].size;
                  widgetItem.update(left, top, width, size);
              }
          };
          return TabLayoutNode;
      }());
      Private.TabLayoutNode = TabLayoutNode;
      /**
       * A layout node which holds the data for a split area.
       */
      var SplitLayoutNode = /** @class */ (function () {
          /**
           * Construct a new split layout node.
           *
           * @param orientation - The orientation of the node.
           */
          function SplitLayoutNode(orientation) {
              /**
               * The parent of the layout node.
               */
              this.parent = null;
              /**
               * Whether the sizers have been normalized.
               */
              this.normalized = false;
              /**
               * The child nodes for the split node.
               */
              this.children = [];
              /**
               * The box sizers for the layout children.
               */
              this.sizers = [];
              /**
               * The handles for the layout children.
               */
              this.handles = [];
              this.orientation = orientation;
          }
          /**
           * Create an iterator for all widgets in the layout tree.
           */
          SplitLayoutNode.prototype.iterAllWidgets = function () {
              var children = map$1(this.children, function (child) { return child.iterAllWidgets(); });
              return new ChainIterator(children);
          };
          /**
           * Create an iterator for the user widgets in the layout tree.
           */
          SplitLayoutNode.prototype.iterUserWidgets = function () {
              var children = map$1(this.children, function (child) { return child.iterUserWidgets(); });
              return new ChainIterator(children);
          };
          /**
           * Create an iterator for the selected widgets in the layout tree.
           */
          SplitLayoutNode.prototype.iterSelectedWidgets = function () {
              var children = map$1(this.children, function (child) { return child.iterSelectedWidgets(); });
              return new ChainIterator(children);
          };
          /**
           * Create an iterator for the tab bars in the layout tree.
           */
          SplitLayoutNode.prototype.iterTabBars = function () {
              var children = map$1(this.children, function (child) { return child.iterTabBars(); });
              return new ChainIterator(children);
          };
          /**
           * Create an iterator for the handles in the layout tree.
           */
          SplitLayoutNode.prototype.iterHandles = function () {
              var children = map$1(this.children, function (child) { return child.iterHandles(); });
              return chain$1(this.handles, new ChainIterator(children));
          };
          /**
           * Find the tab layout node which contains the given widget.
           */
          SplitLayoutNode.prototype.findTabNode = function (widget) {
              for (var i = 0, n = this.children.length; i < n; ++i) {
                  var result = this.children[i].findTabNode(widget);
                  if (result) {
                      return result;
                  }
              }
              return null;
          };
          /**
           * Find the split layout node which contains the given handle.
           */
          SplitLayoutNode.prototype.findSplitNode = function (handle) {
              var index = this.handles.indexOf(handle);
              if (index !== -1) {
                  return { index: index, node: this };
              }
              for (var i = 0, n = this.children.length; i < n; ++i) {
                  var result = this.children[i].findSplitNode(handle);
                  if (result) {
                      return result;
                  }
              }
              return null;
          };
          /**
           * Find the first tab layout node in a layout tree.
           */
          SplitLayoutNode.prototype.findFirstTabNode = function () {
              if (this.children.length === 0) {
                  return null;
              }
              return this.children[0].findFirstTabNode();
          };
          /**
           * Find the tab layout node which contains the local point.
           */
          SplitLayoutNode.prototype.hitTestTabNodes = function (x, y) {
              for (var i = 0, n = this.children.length; i < n; ++i) {
                  var result = this.children[i].hitTestTabNodes(x, y);
                  if (result) {
                      return result;
                  }
              }
              return null;
          };
          /**
           * Create a configuration object for the layout tree.
           */
          SplitLayoutNode.prototype.createConfig = function () {
              var orientation = this.orientation;
              var sizes = this.createNormalizedSizes();
              var children = this.children.map(function (child) { return child.createConfig(); });
              return { type: 'split-area', orientation: orientation, children: children, sizes: sizes };
          };
          /**
           * Sync the visibility and orientation of the handles.
           */
          SplitLayoutNode.prototype.syncHandles = function () {
              var _this = this;
              each$1(this.handles, function (handle, i) {
                  handle.setAttribute('data-orientation', _this.orientation);
                  if (i === _this.handles.length - 1) {
                      handle.classList.add('lm-mod-hidden');
                      /* <DEPRECATED> */
                      handle.classList.add('p-mod-hidden');
                      /* </DEPRECATED> */
                  }
                  else {
                      handle.classList.remove('lm-mod-hidden');
                      /* <DEPRECATED> */
                      handle.classList.remove('p-mod-hidden');
                      /* </DEPRECATED> */
                  }
              });
          };
          /**
           * Hold the current sizes of the box sizers.
           *
           * This sets the size hint of each sizer to its current size.
           */
          SplitLayoutNode.prototype.holdSizes = function () {
              each$1(this.sizers, function (sizer) { sizer.sizeHint = sizer.size; });
          };
          /**
           * Recursively hold all of the sizes in the layout tree.
           *
           * This ignores the sizers of tab layout nodes.
           */
          SplitLayoutNode.prototype.holdAllSizes = function () {
              each$1(this.children, function (child) { return child.holdAllSizes(); });
              this.holdSizes();
          };
          /**
           * Normalize the sizes of the split layout node.
           */
          SplitLayoutNode.prototype.normalizeSizes = function () {
              // Bail early if the sizers are empty.
              var n = this.sizers.length;
              if (n === 0) {
                  return;
              }
              // Hold the current sizes of the sizers.
              this.holdSizes();
              // Compute the sum of the sizes.
              var sum = reduce$1(this.sizers, function (v, sizer) { return v + sizer.sizeHint; }, 0);
              // Normalize the sizes based on the sum.
              if (sum === 0) {
                  each$1(this.sizers, function (sizer) {
                      sizer.size = sizer.sizeHint = 1 / n;
                  });
              }
              else {
                  each$1(this.sizers, function (sizer) {
                      sizer.size = sizer.sizeHint /= sum;
                  });
              }
              // Mark the sizes as normalized.
              this.normalized = true;
          };
          /**
           * Snap the normalized sizes of the split layout node.
           */
          SplitLayoutNode.prototype.createNormalizedSizes = function () {
              // Bail early if the sizers are empty.
              var n = this.sizers.length;
              if (n === 0) {
                  return [];
              }
              // Grab the current sizes of the sizers.
              var sizes = this.sizers.map(function (sizer) { return sizer.size; });
              // Compute the sum of the sizes.
              var sum = reduce$1(sizes, function (v, size) { return v + size; }, 0);
              // Normalize the sizes based on the sum.
              if (sum === 0) {
                  each$1(sizes, function (size, i) { sizes[i] = 1 / n; });
              }
              else {
                  each$1(sizes, function (size, i) { sizes[i] = size / sum; });
              }
              // Return the normalized sizes.
              return sizes;
          };
          /**
           * Fit the layout tree.
           */
          SplitLayoutNode.prototype.fit = function (spacing, items) {
              // Compute the required fixed space.
              var horizontal = this.orientation === 'horizontal';
              var fixed = Math.max(0, this.children.length - 1) * spacing;
              // Set up the limit variables.
              var minWidth = horizontal ? fixed : 0;
              var minHeight = horizontal ? 0 : fixed;
              var maxWidth = Infinity;
              var maxHeight = Infinity;
              // Fit the children and update the limits.
              for (var i = 0, n = this.children.length; i < n; ++i) {
                  var limits = this.children[i].fit(spacing, items);
                  if (horizontal) {
                      minHeight = Math.max(minHeight, limits.minHeight);
                      minWidth += limits.minWidth;
                      this.sizers[i].minSize = limits.minWidth;
                  }
                  else {
                      minWidth = Math.max(minWidth, limits.minWidth);
                      minHeight += limits.minHeight;
                      this.sizers[i].minSize = limits.minHeight;
                  }
              }
              // Return the computed limits for the layout node.
              return { minWidth: minWidth, minHeight: minHeight, maxWidth: maxWidth, maxHeight: maxHeight };
          };
          /**
           * Update the layout tree.
           */
          SplitLayoutNode.prototype.update = function (left, top, width, height, spacing, items) {
              // Compute the available layout space.
              var horizontal = this.orientation === 'horizontal';
              var fixed = Math.max(0, this.children.length - 1) * spacing;
              var space = Math.max(0, (horizontal ? width : height) - fixed);
              // De-normalize the sizes if needed.
              if (this.normalized) {
                  each$1(this.sizers, function (sizer) { sizer.sizeHint *= space; });
                  this.normalized = false;
              }
              // Distribute the layout space to the sizers.
              BoxEngine.calc(this.sizers, space);
              // Update the geometry of the child nodes and handles.
              for (var i = 0, n = this.children.length; i < n; ++i) {
                  var child = this.children[i];
                  var size = this.sizers[i].size;
                  var handleStyle = this.handles[i].style;
                  if (horizontal) {
                      child.update(left, top, size, height, spacing, items);
                      left += size;
                      handleStyle.top = top + "px";
                      handleStyle.left = left + "px";
                      handleStyle.width = spacing + "px";
                      handleStyle.height = height + "px";
                      left += spacing;
                  }
                  else {
                      child.update(left, top, width, size, spacing, items);
                      top += size;
                      handleStyle.top = top + "px";
                      handleStyle.left = left + "px";
                      handleStyle.width = width + "px";
                      handleStyle.height = spacing + "px";
                      top += spacing;
                  }
              }
          };
          return SplitLayoutNode;
      }());
      Private.SplitLayoutNode = SplitLayoutNode;
      /**
       * Normalize a tab area config and collect the visited widgets.
       */
      function normalizeTabAreaConfig(config, widgetSet) {
          // Bail early if there is no content.
          if (config.widgets.length === 0) {
              return null;
          }
          // Setup the filtered widgets array.
          var widgets = [];
          // Filter the config for unique widgets.
          each$1(config.widgets, function (widget) {
              if (!widgetSet.has(widget)) {
                  widgetSet.add(widget);
                  widgets.push(widget);
              }
          });
          // Bail if there are no effective widgets.
          if (widgets.length === 0) {
              return null;
          }
          // Normalize the current index.
          var index = config.currentIndex;
          if (index !== -1 && (index < 0 || index >= widgets.length)) {
              index = 0;
          }
          // Return a normalized config object.
          return { type: 'tab-area', widgets: widgets, currentIndex: index };
      }
      /**
       * Normalize a split area config and collect the visited widgets.
       */
      function normalizeSplitAreaConfig(config, widgetSet) {
          // Set up the result variables.
          var orientation = config.orientation;
          var children = [];
          var sizes = [];
          // Normalize the config children.
          for (var i = 0, n = config.children.length; i < n; ++i) {
              // Normalize the child config.
              var child = normalizeAreaConfig(config.children[i], widgetSet);
              // Ignore an empty child.
              if (!child) {
                  continue;
              }
              // Add the child or hoist its content as appropriate.
              if (child.type === 'tab-area' || child.orientation !== orientation) {
                  children.push(child);
                  sizes.push(Math.abs(config.sizes[i] || 0));
              }
              else {
                  children.push.apply(children, child.children);
                  sizes.push.apply(sizes, child.sizes);
              }
          }
          // Bail if there are no effective children.
          if (children.length === 0) {
              return null;
          }
          // If there is only one effective child, return that child.
          if (children.length === 1) {
              return children[0];
          }
          // Return a normalized config object.
          return { type: 'split-area', orientation: orientation, children: children, sizes: sizes };
      }
      /**
       * Convert a normalized tab area config into a layout tree.
       */
      function realizeTabAreaConfig(config, renderer) {
          // Create the tab bar for the layout node.
          var tabBar = renderer.createTabBar();
          // Hide each widget and add it to the tab bar.
          each$1(config.widgets, function (widget) {
              widget.hide();
              tabBar.addTab(widget.title);
          });
          // Set the current index of the tab bar.
          tabBar.currentIndex = config.currentIndex;
          // Return the new tab layout node.
          return new TabLayoutNode(tabBar);
      }
      /**
       * Convert a normalized split area config into a layout tree.
       */
      function realizeSplitAreaConfig(config, renderer) {
          // Create the split layout node.
          var node = new SplitLayoutNode(config.orientation);
          // Add each child to the layout node.
          each$1(config.children, function (child, i) {
              // Create the child data for the layout node.
              var childNode = realizeAreaConfig(child, renderer);
              var sizer = createSizer(config.sizes[i]);
              var handle = renderer.createHandle();
              // Add the child data to the layout node.
              node.children.push(childNode);
              node.handles.push(handle);
              node.sizers.push(sizer);
              // Update the parent for the child node.
              childNode.parent = node;
          });
          // Synchronize the handle state for the layout node.
          node.syncHandles();
          // Normalize the sizes for the layout node.
          node.normalizeSizes();
          // Return the new layout node.
          return node;
      }
  })(Private$8$1 || (Private$8$1 = {}));

  /**
   * A widget which displays titles as a single row or column of tabs.
   *
   * #### Notes
   * If CSS transforms are used to rotate nodes for vertically oriented
   * text, then tab dragging will not work correctly. The `tabsMovable`
   * property should be set to `false` when rotating nodes from CSS.
   */
  var TabBar = /** @class */ (function (_super) {
      __extends$5(TabBar, _super);
      /**
       * Construct a new tab bar.
       *
       * @param options - The options for initializing the tab bar.
       */
      function TabBar(options) {
          if (options === void 0) { options = {}; }
          var _this = _super.call(this, { node: Private$9$1.createNode() }) || this;
          _this._currentIndex = -1;
          _this._titles = [];
          _this._previousTitle = null;
          _this._dragData = null;
          _this._tabMoved = new Signal(_this);
          _this._currentChanged = new Signal(_this);
          _this._tabCloseRequested = new Signal(_this);
          _this._tabDetachRequested = new Signal(_this);
          _this._tabActivateRequested = new Signal(_this);
          _this.addClass('lm-TabBar');
          /* <DEPRECATED> */
          _this.addClass('p-TabBar');
          /* </DEPRECATED> */
          _this.setFlag(Widget.Flag.DisallowLayout);
          _this.tabsMovable = options.tabsMovable || false;
          _this.allowDeselect = options.allowDeselect || false;
          _this.insertBehavior = options.insertBehavior || 'select-tab-if-needed';
          _this.removeBehavior = options.removeBehavior || 'select-tab-after';
          _this.renderer = options.renderer || TabBar.defaultRenderer;
          _this._orientation = options.orientation || 'horizontal';
          _this.dataset['orientation'] = _this._orientation;
          return _this;
      }
      /**
       * Dispose of the resources held by the widget.
       */
      TabBar.prototype.dispose = function () {
          this._releaseMouse();
          this._titles.length = 0;
          this._previousTitle = null;
          _super.prototype.dispose.call(this);
      };
      Object.defineProperty(TabBar.prototype, "currentChanged", {
          /**
           * A signal emitted when the current tab is changed.
           *
           * #### Notes
           * This signal is emitted when the currently selected tab is changed
           * either through user or programmatic interaction.
           *
           * Notably, this signal is not emitted when the index of the current
           * tab changes due to tabs being inserted, removed, or moved. It is
           * only emitted when the actual current tab node is changed.
           */
          get: function () {
              return this._currentChanged;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(TabBar.prototype, "tabMoved", {
          /**
           * A signal emitted when a tab is moved by the user.
           *
           * #### Notes
           * This signal is emitted when a tab is moved by user interaction.
           *
           * This signal is not emitted when a tab is moved programmatically.
           */
          get: function () {
              return this._tabMoved;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(TabBar.prototype, "tabActivateRequested", {
          /**
           * A signal emitted when a tab is clicked by the user.
           *
           * #### Notes
           * If the clicked tab is not the current tab, the clicked tab will be
           * made current and the `currentChanged` signal will be emitted first.
           *
           * This signal is emitted even if the clicked tab is the current tab.
           */
          get: function () {
              return this._tabActivateRequested;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(TabBar.prototype, "tabCloseRequested", {
          /**
           * A signal emitted when a tab close icon is clicked.
           *
           * #### Notes
           * This signal is not emitted unless the tab title is `closable`.
           */
          get: function () {
              return this._tabCloseRequested;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(TabBar.prototype, "tabDetachRequested", {
          /**
           * A signal emitted when a tab is dragged beyond the detach threshold.
           *
           * #### Notes
           * This signal is emitted when the user drags a tab with the mouse,
           * and mouse is dragged beyond the detach threshold.
           *
           * The consumer of the signal should call `releaseMouse` and remove
           * the tab in order to complete the detach.
           *
           * This signal is only emitted once per drag cycle.
           */
          get: function () {
              return this._tabDetachRequested;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(TabBar.prototype, "currentTitle", {
          /**
           * Get the currently selected title.
           *
           * #### Notes
           * This will be `null` if no tab is selected.
           */
          get: function () {
              return this._titles[this._currentIndex] || null;
          },
          /**
           * Set the currently selected title.
           *
           * #### Notes
           * If the title does not exist, the title will be set to `null`.
           */
          set: function (value) {
              this.currentIndex = value ? this._titles.indexOf(value) : -1;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(TabBar.prototype, "currentIndex", {
          /**
           * Get the index of the currently selected tab.
           *
           * #### Notes
           * This will be `-1` if no tab is selected.
           */
          get: function () {
              return this._currentIndex;
          },
          /**
           * Set the index of the currently selected tab.
           *
           * #### Notes
           * If the value is out of range, the index will be set to `-1`.
           */
          set: function (value) {
              // Adjust for an out of range index.
              if (value < 0 || value >= this._titles.length) {
                  value = -1;
              }
              // Bail early if the index will not change.
              if (this._currentIndex === value) {
                  return;
              }
              // Look up the previous index and title.
              var pi = this._currentIndex;
              var pt = this._titles[pi] || null;
              // Look up the current index and title.
              var ci = value;
              var ct = this._titles[ci] || null;
              // Update the current index and previous title.
              this._currentIndex = ci;
              this._previousTitle = pt;
              // Schedule an update of the tabs.
              this.update();
              // Emit the current changed signal.
              this._currentChanged.emit({
                  previousIndex: pi, previousTitle: pt,
                  currentIndex: ci, currentTitle: ct
              });
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(TabBar.prototype, "orientation", {
          /**
           * Get the orientation of the tab bar.
           *
           * #### Notes
           * This controls whether the tabs are arranged in a row or column.
           */
          get: function () {
              return this._orientation;
          },
          /**
           * Set the orientation of the tab bar.
           *
           * #### Notes
           * This controls whether the tabs are arranged in a row or column.
           */
          set: function (value) {
              // Do nothing if the orientation does not change.
              if (this._orientation === value) {
                  return;
              }
              // Release the mouse before making any changes.
              this._releaseMouse();
              // Toggle the orientation values.
              this._orientation = value;
              this.dataset['orientation'] = value;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(TabBar.prototype, "titles", {
          /**
           * A read-only array of the titles in the tab bar.
           */
          get: function () {
              return this._titles;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(TabBar.prototype, "contentNode", {
          /**
           * The tab bar content node.
           *
           * #### Notes
           * This is the node which holds the tab nodes.
           *
           * Modifying this node directly can lead to undefined behavior.
           */
          get: function () {
              return this.node.getElementsByClassName('lm-TabBar-content')[0];
          },
          enumerable: true,
          configurable: true
      });
      /**
       * Add a tab to the end of the tab bar.
       *
       * @param value - The title which holds the data for the tab,
       *   or an options object to convert to a title.
       *
       * @returns The title object added to the tab bar.
       *
       * #### Notes
       * If the title is already added to the tab bar, it will be moved.
       */
      TabBar.prototype.addTab = function (value) {
          return this.insertTab(this._titles.length, value);
      };
      /**
       * Insert a tab into the tab bar at the specified index.
       *
       * @param index - The index at which to insert the tab.
       *
       * @param value - The title which holds the data for the tab,
       *   or an options object to convert to a title.
       *
       * @returns The title object added to the tab bar.
       *
       * #### Notes
       * The index will be clamped to the bounds of the tabs.
       *
       * If the title is already added to the tab bar, it will be moved.
       */
      TabBar.prototype.insertTab = function (index, value) {
          // Release the mouse before making any changes.
          this._releaseMouse();
          // Coerce the value to a title.
          var title = Private$9$1.asTitle(value);
          // Look up the index of the title.
          var i = this._titles.indexOf(title);
          // Clamp the insert index to the array bounds.
          var j = Math.max(0, Math.min(index, this._titles.length));
          // If the title is not in the array, insert it.
          if (i === -1) {
              // Insert the title into the array.
              ArrayExt.insert(this._titles, j, title);
              // Connect to the title changed signal.
              title.changed.connect(this._onTitleChanged, this);
              // Schedule an update of the tabs.
              this.update();
              // Adjust the current index for the insert.
              this._adjustCurrentForInsert(j, title);
              // Return the title added to the tab bar.
              return title;
          }
          // Otherwise, the title exists in the array and should be moved.
          // Adjust the index if the location is at the end of the array.
          if (j === this._titles.length) {
              j--;
          }
          // Bail if there is no effective move.
          if (i === j) {
              return title;
          }
          // Move the title to the new location.
          ArrayExt.move(this._titles, i, j);
          // Schedule an update of the tabs.
          this.update();
          // Adjust the current index for the move.
          this._adjustCurrentForMove(i, j);
          // Return the title added to the tab bar.
          return title;
      };
      /**
       * Remove a tab from the tab bar.
       *
       * @param title - The title for the tab to remove.
       *
       * #### Notes
       * This is a no-op if the title is not in the tab bar.
       */
      TabBar.prototype.removeTab = function (title) {
          this.removeTabAt(this._titles.indexOf(title));
      };
      /**
       * Remove the tab at a given index from the tab bar.
       *
       * @param index - The index of the tab to remove.
       *
       * #### Notes
       * This is a no-op if the index is out of range.
       */
      TabBar.prototype.removeTabAt = function (index) {
          // Release the mouse before making any changes.
          this._releaseMouse();
          // Remove the title from the array.
          var title = ArrayExt.removeAt(this._titles, index);
          // Bail if the index is out of range.
          if (!title) {
              return;
          }
          // Disconnect from the title changed signal.
          title.changed.disconnect(this._onTitleChanged, this);
          // Clear the previous title if it's being removed.
          if (title === this._previousTitle) {
              this._previousTitle = null;
          }
          // Schedule an update of the tabs.
          this.update();
          // Adjust the current index for the remove.
          this._adjustCurrentForRemove(index, title);
      };
      /**
       * Remove all tabs from the tab bar.
       */
      TabBar.prototype.clearTabs = function () {
          // Bail if there is nothing to remove.
          if (this._titles.length === 0) {
              return;
          }
          // Release the mouse before making any changes.
          this._releaseMouse();
          // Disconnect from the title changed signals.
          for (var _i = 0, _a = this._titles; _i < _a.length; _i++) {
              var title = _a[_i];
              title.changed.disconnect(this._onTitleChanged, this);
          }
          // Get the current index and title.
          var pi = this.currentIndex;
          var pt = this.currentTitle;
          // Reset the current index and previous title.
          this._currentIndex = -1;
          this._previousTitle = null;
          // Clear the title array.
          this._titles.length = 0;
          // Schedule an update of the tabs.
          this.update();
          // If no tab was selected, there's nothing else to do.
          if (pi === -1) {
              return;
          }
          // Emit the current changed signal.
          this._currentChanged.emit({
              previousIndex: pi, previousTitle: pt,
              currentIndex: -1, currentTitle: null
          });
      };
      /**
       * Release the mouse and restore the non-dragged tab positions.
       *
       * #### Notes
       * This will cause the tab bar to stop handling mouse events and to
       * restore the tabs to their non-dragged positions.
       */
      TabBar.prototype.releaseMouse = function () {
          this._releaseMouse();
      };
      /**
       * Handle the DOM events for the tab bar.
       *
       * @param event - The DOM event sent to the tab bar.
       *
       * #### Notes
       * This method implements the DOM `EventListener` interface and is
       * called in response to events on the tab bar's DOM node.
       *
       * This should not be called directly by user code.
       */
      TabBar.prototype.handleEvent = function (event) {
          switch (event.type) {
              case 'mousedown':
                  this._evtMouseDown(event);
                  break;
              case 'mousemove':
                  this._evtMouseMove(event);
                  break;
              case 'mouseup':
                  this._evtMouseUp(event);
                  break;
              case 'keydown':
                  this._evtKeyDown(event);
                  break;
              case 'contextmenu':
                  event.preventDefault();
                  event.stopPropagation();
                  break;
          }
      };
      /**
       * A message handler invoked on a `'before-attach'` message.
       */
      TabBar.prototype.onBeforeAttach = function (msg) {
          this.node.addEventListener('mousedown', this);
      };
      /**
       * A message handler invoked on an `'after-detach'` message.
       */
      TabBar.prototype.onAfterDetach = function (msg) {
          this.node.removeEventListener('mousedown', this);
          this._releaseMouse();
      };
      /**
       * A message handler invoked on an `'update-request'` message.
       */
      TabBar.prototype.onUpdateRequest = function (msg) {
          var titles = this._titles;
          var renderer = this.renderer;
          var currentTitle = this.currentTitle;
          var content = new Array(titles.length);
          for (var i = 0, n = titles.length; i < n; ++i) {
              var title = titles[i];
              var current = title === currentTitle;
              var zIndex = current ? n : n - i - 1;
              content[i] = renderer.renderTab({ title: title, current: current, zIndex: zIndex });
          }
          VirtualDOM.render(content, this.contentNode);
      };
      /**
       * Handle the `'keydown'` event for the tab bar.
       */
      TabBar.prototype._evtKeyDown = function (event) {
          // Stop all input events during drag.
          event.preventDefault();
          event.stopPropagation();
          // Release the mouse if `Escape` is pressed.
          if (event.keyCode === 27) {
              this._releaseMouse();
          }
      };
      /**
       * Handle the `'mousedown'` event for the tab bar.
       */
      TabBar.prototype._evtMouseDown = function (event) {
          // Do nothing if it's not a left or middle mouse press.
          if (event.button !== 0 && event.button !== 1) {
              return;
          }
          // Do nothing if a drag is in progress.
          if (this._dragData) {
              return;
          }
          // Lookup the tab nodes.
          var tabs = this.contentNode.children;
          // Find the index of the pressed tab.
          var index = ArrayExt.findFirstIndex(tabs, function (tab) {
              return ElementExt.hitTest(tab, event.clientX, event.clientY);
          });
          // Do nothing if the press is not on a tab.
          if (index === -1) {
              return;
          }
          // Pressing on a tab stops the event propagation.
          event.preventDefault();
          event.stopPropagation();
          // Initialize the non-measured parts of the drag data.
          this._dragData = {
              tab: tabs[index],
              index: index,
              pressX: event.clientX,
              pressY: event.clientY,
              tabPos: -1,
              tabSize: -1,
              tabPressPos: -1,
              targetIndex: -1,
              tabLayout: null,
              contentRect: null,
              override: null,
              dragActive: false,
              dragAborted: false,
              detachRequested: false
          };
          // Add the document mouse up listener.
          document.addEventListener('mouseup', this, true);
          // Do nothing else if the middle button is clicked.
          if (event.button === 1) {
              return;
          }
          // Do nothing else if the close icon is clicked.
          var icon = tabs[index].querySelector(this.renderer.closeIconSelector);
          if (icon && icon.contains(event.target)) {
              return;
          }
          // Add the extra listeners if the tabs are movable.
          if (this.tabsMovable) {
              document.addEventListener('mousemove', this, true);
              document.addEventListener('keydown', this, true);
              document.addEventListener('contextmenu', this, true);
          }
          // Update the current index as appropriate.
          if (this.allowDeselect && this.currentIndex === index) {
              this.currentIndex = -1;
          }
          else {
              this.currentIndex = index;
          }
          // Do nothing else if there is no current tab.
          if (this.currentIndex === -1) {
              return;
          }
          // Emit the tab activate request signal.
          this._tabActivateRequested.emit({
              index: this.currentIndex, title: this.currentTitle
          });
      };
      /**
       * Handle the `'mousemove'` event for the tab bar.
       */
      TabBar.prototype._evtMouseMove = function (event) {
          // Do nothing if no drag is in progress.
          var data = this._dragData;
          if (!data) {
              return;
          }
          // Suppress the event during a drag.
          event.preventDefault();
          event.stopPropagation();
          // Lookup the tab nodes.
          var tabs = this.contentNode.children;
          // Bail early if the drag threshold has not been met.
          if (!data.dragActive && !Private$9$1.dragExceeded(data, event)) {
              return;
          }
          // Activate the drag if necessary.
          if (!data.dragActive) {
              // Fill in the rest of the drag data measurements.
              var tabRect = data.tab.getBoundingClientRect();
              if (this._orientation === 'horizontal') {
                  data.tabPos = data.tab.offsetLeft;
                  data.tabSize = tabRect.width;
                  data.tabPressPos = data.pressX - tabRect.left;
              }
              else {
                  data.tabPos = data.tab.offsetTop;
                  data.tabSize = tabRect.height;
                  data.tabPressPos = data.pressY - tabRect.top;
              }
              data.tabLayout = Private$9$1.snapTabLayout(tabs, this._orientation);
              data.contentRect = this.contentNode.getBoundingClientRect();
              data.override = Drag.overrideCursor('default');
              // Add the dragging style classes.
              data.tab.classList.add('lm-mod-dragging');
              this.addClass('lm-mod-dragging');
              /* <DEPRECATED> */
              data.tab.classList.add('p-mod-dragging');
              this.addClass('p-mod-dragging');
              /* </DEPRECATED> */
              // Mark the drag as active.
              data.dragActive = true;
          }
          // Emit the detach requested signal if the threshold is exceeded.
          if (!data.detachRequested && Private$9$1.detachExceeded(data, event)) {
              // Only emit the signal once per drag cycle.
              data.detachRequested = true;
              // Setup the arguments for the signal.
              var index = data.index;
              var clientX = event.clientX;
              var clientY = event.clientY;
              var tab = tabs[index];
              var title = this._titles[index];
              // Emit the tab detach requested signal.
              this._tabDetachRequested.emit({ index: index, title: title, tab: tab, clientX: clientX, clientY: clientY });
              // Bail if the signal handler aborted the drag.
              if (data.dragAborted) {
                  return;
              }
          }
          // Update the positions of the tabs.
          Private$9$1.layoutTabs(tabs, data, event, this._orientation);
      };
      /**
       * Handle the `'mouseup'` event for the document.
       */
      TabBar.prototype._evtMouseUp = function (event) {
          var _this = this;
          // Do nothing if it's not a left or middle mouse release.
          if (event.button !== 0 && event.button !== 1) {
              return;
          }
          // Do nothing if no drag is in progress.
          var data = this._dragData;
          if (!data) {
              return;
          }
          // Stop the event propagation.
          event.preventDefault();
          event.stopPropagation();
          // Remove the extra mouse event listeners.
          document.removeEventListener('mousemove', this, true);
          document.removeEventListener('mouseup', this, true);
          document.removeEventListener('keydown', this, true);
          document.removeEventListener('contextmenu', this, true);
          // Handle a release when the drag is not active.
          if (!data.dragActive) {
              // Clear the drag data.
              this._dragData = null;
              // Lookup the tab nodes.
              var tabs = this.contentNode.children;
              // Find the index of the released tab.
              var index = ArrayExt.findFirstIndex(tabs, function (tab) {
                  return ElementExt.hitTest(tab, event.clientX, event.clientY);
              });
              // Do nothing if the release is not on the original pressed tab.
              if (index !== data.index) {
                  return;
              }
              // Ignore the release if the title is not closable.
              var title = this._titles[index];
              if (!title.closable) {
                  return;
              }
              // Emit the close requested signal if the middle button is released.
              if (event.button === 1) {
                  this._tabCloseRequested.emit({ index: index, title: title });
                  return;
              }
              // Emit the close requested signal if the close icon was released.
              var icon = tabs[index].querySelector(this.renderer.closeIconSelector);
              if (icon && icon.contains(event.target)) {
                  this._tabCloseRequested.emit({ index: index, title: title });
                  return;
              }
              // Otherwise, there is nothing left to do.
              return;
          }
          // Do nothing if the left button is not released.
          if (event.button !== 0) {
              return;
          }
          // Position the tab at its final resting position.
          Private$9$1.finalizeTabPosition(data, this._orientation);
          // Remove the dragging class from the tab so it can be transitioned.
          data.tab.classList.remove('lm-mod-dragging');
          /* <DEPRECATED> */
          data.tab.classList.remove('p-mod-dragging');
          /* </DEPRECATED> */
          // Parse the transition duration for releasing the tab.
          var duration = Private$9$1.parseTransitionDuration(data.tab);
          // Complete the release on a timer to allow the tab to transition.
          setTimeout(function () {
              // Do nothing if the drag has been aborted.
              if (data.dragAborted) {
                  return;
              }
              // Clear the drag data reference.
              _this._dragData = null;
              // Reset the positions of the tabs.
              Private$9$1.resetTabPositions(_this.contentNode.children, _this._orientation);
              // Clear the cursor grab.
              data.override.dispose();
              // Remove the remaining dragging style.
              _this.removeClass('lm-mod-dragging');
              /* <DEPRECATED> */
              _this.removeClass('p-mod-dragging');
              /* </DEPRECATED> */
              // If the tab was not moved, there is nothing else to do.
              var i = data.index;
              var j = data.targetIndex;
              if (j === -1 || i === j) {
                  return;
              }
              // Move the title to the new locations.
              ArrayExt.move(_this._titles, i, j);
              // Adjust the current index for the move.
              _this._adjustCurrentForMove(i, j);
              // Emit the tab moved signal.
              _this._tabMoved.emit({
                  fromIndex: i, toIndex: j, title: _this._titles[j]
              });
              // Update the tabs immediately to prevent flicker.
              MessageLoop.sendMessage(_this, Widget.Msg.UpdateRequest);
          }, duration);
      };
      /**
       * Release the mouse and restore the non-dragged tab positions.
       */
      TabBar.prototype._releaseMouse = function () {
          // Do nothing if no drag is in progress.
          var data = this._dragData;
          if (!data) {
              return;
          }
          // Clear the drag data reference.
          this._dragData = null;
          // Remove the extra mouse listeners.
          document.removeEventListener('mousemove', this, true);
          document.removeEventListener('mouseup', this, true);
          document.removeEventListener('keydown', this, true);
          document.removeEventListener('contextmenu', this, true);
          // Indicate the drag has been aborted. This allows the mouse
          // event handlers to return early when the drag is canceled.
          data.dragAborted = true;
          // If the drag is not active, there's nothing more to do.
          if (!data.dragActive) {
              return;
          }
          // Reset the tabs to their non-dragged positions.
          Private$9$1.resetTabPositions(this.contentNode.children, this._orientation);
          // Clear the cursor override.
          data.override.dispose();
          // Clear the dragging style classes.
          data.tab.classList.remove('lm-mod-dragging');
          this.removeClass('lm-mod-dragging');
          /* <DEPRECATED> */
          data.tab.classList.remove('p-mod-dragging');
          this.removeClass('p-mod-dragging');
          /* </DEPRECATED> */
      };
      /**
       * Adjust the current index for a tab insert operation.
       *
       * This method accounts for the tab bar's insertion behavior when
       * adjusting the current index and emitting the changed signal.
       */
      TabBar.prototype._adjustCurrentForInsert = function (i, title) {
          // Lookup commonly used variables.
          var ct = this.currentTitle;
          var ci = this._currentIndex;
          var bh = this.insertBehavior;
          // Handle the behavior where the new tab is always selected,
          // or the behavior where the new tab is selected if needed.
          if (bh === 'select-tab' || (bh === 'select-tab-if-needed' && ci === -1)) {
              this._currentIndex = i;
              this._previousTitle = ct;
              this._currentChanged.emit({
                  previousIndex: ci, previousTitle: ct,
                  currentIndex: i, currentTitle: title
              });
              return;
          }
          // Otherwise, silently adjust the current index if needed.
          if (ci >= i) {
              this._currentIndex++;
          }
      };
      /**
       * Adjust the current index for a tab move operation.
       *
       * This method will not cause the actual current tab to change.
       * It silently adjusts the index to account for the given move.
       */
      TabBar.prototype._adjustCurrentForMove = function (i, j) {
          if (this._currentIndex === i) {
              this._currentIndex = j;
          }
          else if (this._currentIndex < i && this._currentIndex >= j) {
              this._currentIndex++;
          }
          else if (this._currentIndex > i && this._currentIndex <= j) {
              this._currentIndex--;
          }
      };
      /**
       * Adjust the current index for a tab remove operation.
       *
       * This method accounts for the tab bar's remove behavior when
       * adjusting the current index and emitting the changed signal.
       */
      TabBar.prototype._adjustCurrentForRemove = function (i, title) {
          // Lookup commonly used variables.
          var ci = this._currentIndex;
          var bh = this.removeBehavior;
          // Silently adjust the index if the current tab is not removed.
          if (ci !== i) {
              if (ci > i) {
                  this._currentIndex--;
              }
              return;
          }
          // No tab gets selected if the tab bar is empty.
          if (this._titles.length === 0) {
              this._currentIndex = -1;
              this._currentChanged.emit({
                  previousIndex: i, previousTitle: title,
                  currentIndex: -1, currentTitle: null
              });
              return;
          }
          // Handle behavior where the next sibling tab is selected.
          if (bh === 'select-tab-after') {
              this._currentIndex = Math.min(i, this._titles.length - 1);
              this._currentChanged.emit({
                  previousIndex: i, previousTitle: title,
                  currentIndex: this._currentIndex, currentTitle: this.currentTitle
              });
              return;
          }
          // Handle behavior where the previous sibling tab is selected.
          if (bh === 'select-tab-before') {
              this._currentIndex = Math.max(0, i - 1);
              this._currentChanged.emit({
                  previousIndex: i, previousTitle: title,
                  currentIndex: this._currentIndex, currentTitle: this.currentTitle
              });
              return;
          }
          // Handle behavior where the previous history tab is selected.
          if (bh === 'select-previous-tab') {
              if (this._previousTitle) {
                  this._currentIndex = this._titles.indexOf(this._previousTitle);
                  this._previousTitle = null;
              }
              else {
                  this._currentIndex = Math.min(i, this._titles.length - 1);
              }
              this._currentChanged.emit({
                  previousIndex: i, previousTitle: title,
                  currentIndex: this._currentIndex, currentTitle: this.currentTitle
              });
              return;
          }
          // Otherwise, no tab gets selected.
          this._currentIndex = -1;
          this._currentChanged.emit({
              previousIndex: i, previousTitle: title,
              currentIndex: -1, currentTitle: null
          });
      };
      /**
       * Handle the `changed` signal of a title object.
       */
      TabBar.prototype._onTitleChanged = function (sender) {
          this.update();
      };
      return TabBar;
  }(Widget));
  /**
   * The namespace for the `TabBar` class statics.
   */
  (function (TabBar) {
      /**
       * The default implementation of `IRenderer`.
       *
       * #### Notes
       * Subclasses are free to reimplement rendering methods as needed.
       */
      var Renderer = /** @class */ (function () {
          /**
           * Construct a new renderer.
           */
          function Renderer() {
              /**
               * A selector which matches the close icon node in a tab.
               */
              this.closeIconSelector = '.lm-TabBar-tabCloseIcon';
              this._tabID = 0;
              this._tabKeys = new WeakMap();
          }
          /**
           * Render the virtual element for a tab.
           *
           * @param data - The data to use for rendering the tab.
           *
           * @returns A virtual element representing the tab.
           */
          Renderer.prototype.renderTab = function (data) {
              var title = data.title.caption;
              var key = this.createTabKey(data);
              var style = this.createTabStyle(data);
              var className = this.createTabClass(data);
              var dataset = this.createTabDataset(data);
              return (h.li({ key: key, className: className, title: title, style: style, dataset: dataset }, this.renderIcon(data), this.renderLabel(data), this.renderCloseIcon(data)));
          };
          /**
           * Render the icon element for a tab.
           *
           * @param data - The data to use for rendering the tab.
           *
           * @returns A virtual element representing the tab icon.
           */
          Renderer.prototype.renderIcon = function (data) {
              var title = data.title;
              var className = this.createIconClass(data);
              /* <DEPRECATED> */
              if (typeof title.icon === 'string') {
                  return h.div({ className: className }, title.iconLabel);
              }
              /* </DEPRECATED> */
              // if title.icon is undefined, it will be ignored
              return h.div({ className: className }, title.icon, title.iconLabel);
          };
          /**
           * Render the label element for a tab.
           *
           * @param data - The data to use for rendering the tab.
           *
           * @returns A virtual element representing the tab label.
           */
          Renderer.prototype.renderLabel = function (data) {
              return h.div({
                  className: 'lm-TabBar-tabLabel'
                      /* <DEPRECATED> */
                      + ' p-TabBar-tabLabel'
                  /* </DEPRECATED> */
              }, data.title.label);
          };
          /**
           * Render the close icon element for a tab.
           *
           * @param data - The data to use for rendering the tab.
           *
           * @returns A virtual element representing the tab close icon.
           */
          Renderer.prototype.renderCloseIcon = function (data) {
              return h.div({
                  className: 'lm-TabBar-tabCloseIcon'
                      /* <DEPRECATED> */
                      + ' p-TabBar-tabCloseIcon'
                  /* </DEPRECATED> */
              });
          };
          /**
           * Create a unique render key for the tab.
           *
           * @param data - The data to use for the tab.
           *
           * @returns The unique render key for the tab.
           *
           * #### Notes
           * This method caches the key against the tab title the first time
           * the key is generated. This enables efficient rendering of moved
           * tabs and avoids subtle hover style artifacts.
           */
          Renderer.prototype.createTabKey = function (data) {
              var key = this._tabKeys.get(data.title);
              if (key === undefined) {
                  key = "tab-key-" + this._tabID++;
                  this._tabKeys.set(data.title, key);
              }
              return key;
          };
          /**
           * Create the inline style object for a tab.
           *
           * @param data - The data to use for the tab.
           *
           * @returns The inline style data for the tab.
           */
          Renderer.prototype.createTabStyle = function (data) {
              return { zIndex: "" + data.zIndex };
          };
          /**
           * Create the class name for the tab.
           *
           * @param data - The data to use for the tab.
           *
           * @returns The full class name for the tab.
           */
          Renderer.prototype.createTabClass = function (data) {
              var name = 'lm-TabBar-tab';
              /* <DEPRECATED> */
              name += ' p-TabBar-tab';
              /* </DEPRECATED> */
              if (data.title.className) {
                  name += " " + data.title.className;
              }
              if (data.title.closable) {
                  name += ' lm-mod-closable';
                  /* <DEPRECATED> */
                  name += ' p-mod-closable';
                  /* </DEPRECATED> */
              }
              if (data.current) {
                  name += ' lm-mod-current';
                  /* <DEPRECATED> */
                  name += ' p-mod-current';
                  /* </DEPRECATED> */
              }
              return name;
          };
          /**
           * Create the dataset for a tab.
           *
           * @param data - The data to use for the tab.
           *
           * @returns The dataset for the tab.
           */
          Renderer.prototype.createTabDataset = function (data) {
              return data.title.dataset;
          };
          /**
           * Create the class name for the tab icon.
           *
           * @param data - The data to use for the tab.
           *
           * @returns The full class name for the tab icon.
           */
          Renderer.prototype.createIconClass = function (data) {
              var name = 'lm-TabBar-tabIcon';
              /* <DEPRECATED> */
              name += ' p-TabBar-tabIcon';
              /* </DEPRECATED> */
              var extra = data.title.iconClass;
              return extra ? name + " " + extra : name;
          };
          return Renderer;
      }());
      TabBar.Renderer = Renderer;
      /**
       * The default `Renderer` instance.
       */
      TabBar.defaultRenderer = new Renderer();
  })(TabBar || (TabBar = {}));
  /**
   * The namespace for the module implementation details.
   */
  var Private$9$1;
  (function (Private) {
      /**
       * The start drag distance threshold.
       */
      Private.DRAG_THRESHOLD = 5;
      /**
       * The detach distance threshold.
       */
      Private.DETACH_THRESHOLD = 20;
      /**
       * Create the DOM node for a tab bar.
       */
      function createNode() {
          var node = document.createElement('div');
          var content = document.createElement('ul');
          content.className = 'lm-TabBar-content';
          /* <DEPRECATED> */
          content.classList.add('p-TabBar-content');
          /* </DEPRECATED> */
          node.appendChild(content);
          return node;
      }
      Private.createNode = createNode;
      /**
       * Coerce a title or options into a real title.
       */
      function asTitle(value) {
          return value instanceof Title ? value : new Title(value);
      }
      Private.asTitle = asTitle;
      /**
       * Parse the transition duration for a tab node.
       */
      function parseTransitionDuration(tab) {
          var style = window.getComputedStyle(tab);
          return 1000 * (parseFloat(style.transitionDuration) || 0);
      }
      Private.parseTransitionDuration = parseTransitionDuration;
      /**
       * Get a snapshot of the current tab layout values.
       */
      function snapTabLayout(tabs, orientation) {
          var layout = new Array(tabs.length);
          for (var i = 0, n = tabs.length; i < n; ++i) {
              var node = tabs[i];
              var style = window.getComputedStyle(node);
              if (orientation === 'horizontal') {
                  layout[i] = {
                      pos: node.offsetLeft,
                      size: node.offsetWidth,
                      margin: parseFloat(style.marginLeft) || 0
                  };
              }
              else {
                  layout[i] = {
                      pos: node.offsetTop,
                      size: node.offsetHeight,
                      margin: parseFloat(style.marginTop) || 0
                  };
              }
          }
          return layout;
      }
      Private.snapTabLayout = snapTabLayout;
      /**
       * Test if the event exceeds the drag threshold.
       */
      function dragExceeded(data, event) {
          var dx = Math.abs(event.clientX - data.pressX);
          var dy = Math.abs(event.clientY - data.pressY);
          return dx >= Private.DRAG_THRESHOLD || dy >= Private.DRAG_THRESHOLD;
      }
      Private.dragExceeded = dragExceeded;
      /**
       * Test if the event exceeds the drag detach threshold.
       */
      function detachExceeded(data, event) {
          var rect = data.contentRect;
          return ((event.clientX < rect.left - Private.DETACH_THRESHOLD) ||
              (event.clientX >= rect.right + Private.DETACH_THRESHOLD) ||
              (event.clientY < rect.top - Private.DETACH_THRESHOLD) ||
              (event.clientY >= rect.bottom + Private.DETACH_THRESHOLD));
      }
      Private.detachExceeded = detachExceeded;
      /**
       * Update the relative tab positions and computed target index.
       */
      function layoutTabs(tabs, data, event, orientation) {
          // Compute the orientation-sensitive values.
          var pressPos;
          var localPos;
          var clientPos;
          var clientSize;
          if (orientation === 'horizontal') {
              pressPos = data.pressX;
              localPos = event.clientX - data.contentRect.left;
              clientPos = event.clientX;
              clientSize = data.contentRect.width;
          }
          else {
              pressPos = data.pressY;
              localPos = event.clientY - data.contentRect.top;
              clientPos = event.clientY;
              clientSize = data.contentRect.height;
          }
          // Compute the target data.
          var targetIndex = data.index;
          var targetPos = localPos - data.tabPressPos;
          var targetEnd = targetPos + data.tabSize;
          // Update the relative tab positions.
          for (var i = 0, n = tabs.length; i < n; ++i) {
              var pxPos = void 0;
              var layout = data.tabLayout[i];
              var threshold = layout.pos + (layout.size >> 1);
              if (i < data.index && targetPos < threshold) {
                  pxPos = data.tabSize + data.tabLayout[i + 1].margin + "px";
                  targetIndex = Math.min(targetIndex, i);
              }
              else if (i > data.index && targetEnd > threshold) {
                  pxPos = -data.tabSize - layout.margin + "px";
                  targetIndex = Math.max(targetIndex, i);
              }
              else if (i === data.index) {
                  var ideal = clientPos - pressPos;
                  var limit = clientSize - (data.tabPos + data.tabSize);
                  pxPos = Math.max(-data.tabPos, Math.min(ideal, limit)) + "px";
              }
              else {
                  pxPos = '';
              }
              if (orientation === 'horizontal') {
                  tabs[i].style.left = pxPos;
              }
              else {
                  tabs[i].style.top = pxPos;
              }
          }
          // Update the computed target index.
          data.targetIndex = targetIndex;
      }
      Private.layoutTabs = layoutTabs;
      /**
       * Position the drag tab at its final resting relative position.
       */
      function finalizeTabPosition(data, orientation) {
          // Compute the orientation-sensitive client size.
          var clientSize;
          if (orientation === 'horizontal') {
              clientSize = data.contentRect.width;
          }
          else {
              clientSize = data.contentRect.height;
          }
          // Compute the ideal final tab position.
          var ideal;
          if (data.targetIndex === data.index) {
              ideal = 0;
          }
          else if (data.targetIndex > data.index) {
              var tgt = data.tabLayout[data.targetIndex];
              ideal = tgt.pos + tgt.size - data.tabSize - data.tabPos;
          }
          else {
              var tgt = data.tabLayout[data.targetIndex];
              ideal = tgt.pos - data.tabPos;
          }
          // Compute the tab position limit.
          var limit = clientSize - (data.tabPos + data.tabSize);
          var final = Math.max(-data.tabPos, Math.min(ideal, limit));
          // Set the final orientation-sensitive position.
          if (orientation === 'horizontal') {
              data.tab.style.left = final + "px";
          }
          else {
              data.tab.style.top = final + "px";
          }
      }
      Private.finalizeTabPosition = finalizeTabPosition;
      /**
       * Reset the relative positions of the given tabs.
       */
      function resetTabPositions(tabs, orientation) {
          each$1(tabs, function (tab) {
              if (orientation === 'horizontal') {
                  tab.style.left = '';
              }
              else {
                  tab.style.top = '';
              }
          });
      }
      Private.resetTabPositions = resetTabPositions;
  })(Private$9$1 || (Private$9$1 = {}));

  /**
   * A widget which provides a flexible docking area for widgets.
   */
  var DockPanel = /** @class */ (function (_super) {
      __extends$5(DockPanel, _super);
      /**
       * Construct a new dock panel.
       *
       * @param options - The options for initializing the panel.
       */
      function DockPanel(options) {
          if (options === void 0) { options = {}; }
          var _this = _super.call(this) || this;
          _this._drag = null;
          _this._tabsMovable = true;
          _this._pressData = null;
          _this._layoutModified = new Signal(_this);
          _this.addClass('lm-DockPanel');
          /* <DEPRECATED> */
          _this.addClass('p-DockPanel');
          /* </DEPRECATED> */
          _this._mode = options.mode || 'multiple-document';
          _this._renderer = options.renderer || DockPanel.defaultRenderer;
          _this._edges = options.edges || Private$a.DEFAULT_EDGES;
          if (options.tabsMovable !== undefined) {
              _this._tabsMovable = options.tabsMovable;
          }
          // Toggle the CSS mode attribute.
          _this.dataset['mode'] = _this._mode;
          // Create the delegate renderer for the layout.
          var renderer = {
              createTabBar: function () { return _this._createTabBar(); },
              createHandle: function () { return _this._createHandle(); }
          };
          // Set up the dock layout for the panel.
          _this.layout = new DockLayout({ renderer: renderer, spacing: options.spacing });
          // Set up the overlay drop indicator.
          _this.overlay = options.overlay || new DockPanel.Overlay();
          _this.node.appendChild(_this.overlay.node);
          return _this;
      }
      /**
       * Dispose of the resources held by the panel.
       */
      DockPanel.prototype.dispose = function () {
          // Ensure the mouse is released.
          this._releaseMouse();
          // Hide the overlay.
          this.overlay.hide(0);
          // Cancel a drag if one is in progress.
          if (this._drag) {
              this._drag.dispose();
          }
          // Dispose of the base class.
          _super.prototype.dispose.call(this);
      };
      Object.defineProperty(DockPanel.prototype, "layoutModified", {
          /**
           * A signal emitted when the layout configuration is modified.
           *
           * #### Notes
           * This signal is emitted whenever the current layout configuration
           * may have changed.
           *
           * This signal is emitted asynchronously in a collapsed fashion, so
           * that multiple synchronous modifications results in only a single
           * emit of the signal.
           */
          get: function () {
              return this._layoutModified;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(DockPanel.prototype, "renderer", {
          /**
           * The renderer used by the dock panel.
           */
          get: function () {
              return this.layout.renderer;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(DockPanel.prototype, "spacing", {
          /**
           * Get the spacing between the widgets.
           */
          get: function () {
              return this.layout.spacing;
          },
          /**
           * Set the spacing between the widgets.
           */
          set: function (value) {
              this.layout.spacing = value;
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(DockPanel.prototype, "mode", {
          /**
           * Get the mode for the dock panel.
           */
          get: function () {
              return this._mode;
          },
          /**
           * Set the mode for the dock panel.
           *
           * #### Notes
           * Changing the mode is a destructive operation with respect to the
           * panel's layout configuration. If layout state must be preserved,
           * save the current layout config before changing the mode.
           */
          set: function (value) {
              // Bail early if the mode does not change.
              if (this._mode === value) {
                  return;
              }
              // Update the internal mode.
              this._mode = value;
              // Toggle the CSS mode attribute.
              this.dataset['mode'] = value;
              // Get the layout for the panel.
              var layout = this.layout;
              // Configure the layout for the specified mode.
              switch (value) {
                  case 'multiple-document':
                      each$1(layout.tabBars(), function (tabBar) { tabBar.show(); });
                      break;
                  case 'single-document':
                      layout.restoreLayout(Private$a.createSingleDocumentConfig(this));
                      break;
                  default:
                      throw 'unreachable';
              }
              // Schedule an emit of the layout modified signal.
              MessageLoop.postMessage(this, Private$a.LayoutModified);
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(DockPanel.prototype, "tabsMovable", {
          /**
           * Whether the tabs can be dragged / moved at runtime.
           */
          get: function () {
              return this._tabsMovable;
          },
          /**
           * Enable / Disable draggable / movable tabs.
           */
          set: function (value) {
              this._tabsMovable = value;
              each$1(this.tabBars(), function (tabbar) { return tabbar.tabsMovable = value; });
          },
          enumerable: true,
          configurable: true
      });
      Object.defineProperty(DockPanel.prototype, "isEmpty", {
          /**
           * Whether the dock panel is empty.
           */
          get: function () {
              return this.layout.isEmpty;
          },
          enumerable: true,
          configurable: true
      });
      /**
       * Create an iterator over the user widgets in the panel.
       *
       * @returns A new iterator over the user widgets in the panel.
       *
       * #### Notes
       * This iterator does not include the generated tab bars.
       */
      DockPanel.prototype.widgets = function () {
          return this.layout.widgets();
      };
      /**
       * Create an iterator over the selected widgets in the panel.
       *
       * @returns A new iterator over the selected user widgets.
       *
       * #### Notes
       * This iterator yields the widgets corresponding to the current tab
       * of each tab bar in the panel.
       */
      DockPanel.prototype.selectedWidgets = function () {
          return this.layout.selectedWidgets();
      };
      /**
       * Create an iterator over the tab bars in the panel.
       *
       * @returns A new iterator over the tab bars in the panel.
       *
       * #### Notes
       * This iterator does not include the user widgets.
       */
      DockPanel.prototype.tabBars = function () {
          return this.layout.tabBars();
      };
      /**
       * Create an iterator over the handles in the panel.
       *
       * @returns A new iterator over the handles in the panel.
       */
      DockPanel.prototype.handles = function () {
          return this.layout.handles();
      };
      /**
       * Select a specific widget in the dock panel.
       *
       * @param widget - The widget of interest.
       *
       * #### Notes
       * This will make the widget the current widget in its tab area.
       */
      DockPanel.prototype.selectWidget = function (widget) {
          // Find the tab bar which contains the widget.
          var tabBar = find$1(this.tabBars(), function (bar) {
              return bar.titles.indexOf(widget.title) !== -1;
          });
          // Throw an error if no tab bar is found.
          if (!tabBar) {
              throw new Error('Widget is not contained in the dock panel.');
          }
          // Ensure the widget is the current widget.
          tabBar.currentTitle = widget.title;
      };
      /**
       * Activate a specified widget in the dock panel.
       *
       * @param widget - The widget of interest.
       *
       * #### Notes
       * This will select and activate the given widget.
       */
      DockPanel.prototype.activateWidget = function (widget) {
          this.selectWidget(widget);
          widget.activate();
      };
      /**
       * Save the current layout configuration of the dock panel.
       *
       * @returns A new config object for the current layout state.
       *
       * #### Notes
       * The return value can be provided to the `restoreLayout` method
       * in order to restore the layout to its current configuration.
       */
      DockPanel.prototype.saveLayout = function () {
          return this.layout.saveLayout();
      };
      /**
       * Restore the layout to a previously saved configuration.
       *
       * @param config - The layout configuration to restore.
       *
       * #### Notes
       * Widgets which currently belong to the layout but which are not
       * contained in the config will be unparented.
       *
       * The dock panel automatically reverts to `'multiple-document'`
       * mode when a layout config is restored.
       */
      DockPanel.prototype.restoreLayout = function (config) {
          // Reset the mode.
          this._mode = 'multiple-document';
          // Restore the layout.
          this.layout.restoreLayout(config);
          // Flush the message loop on IE and Edge to prevent flicker.
          if (Platform.IS_EDGE || Platform.IS_IE) {
              MessageLoop.flush();
          }
          // Schedule an emit of the layout modified signal.
          MessageLoop.postMessage(this, Private$a.LayoutModified);
      };
      /**
       * Add a widget to the dock panel.
       *
       * @param widget - The widget to add to the dock panel.
       *
       * @param options - The additional options for adding the widget.
       *
       * #### Notes
       * If the panel is in single document mode, the options are ignored
       * and the widget is always added as tab in the hidden tab bar.
       */
      DockPanel.prototype.addWidget = function (widget, options) {
          if (options === void 0) { options = {}; }
          // Add the widget to the layout.
          if (this._mode === 'single-document') {
      