/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */
#include <aws/checksums/private/crc32_priv.h>
#include <aws/checksums/private/crc_util.h>
#include <stddef.h>

/* The Ethernet, gzip, et.al CRC32 polynomial (reverse of 0x04C11DB7) */
#define CRC32_POLYNOMIAL 0xEDB88320

/* The Castagnoli, iSCSI CRC32c polynomial (reverse of 0x1EDC6F41) */
#define CRC32C_POLYNOMIAL 0x82F63B78

/** CRC32 (Ethernet, gzip) lookup table for slice-by-4/8/16 */
const uint32_t CRC32_TABLE[16][256] = {
    {
        0x00000000, 0x77073096, 0xEE0E612C, 0x990951BA, 0x076DC419, 0x706AF48F, 0xE963A535, 0x9E6495A3, /* [0][0x08]*/
        0x0EDB8832, 0x79DCB8A4, 0xE0D5E91E, 0x97D2D988, 0x09B64C2B, 0x7EB17CBD, 0xE7B82D07, 0x90BF1D91, /* [0][0x10]*/
        0x1DB71064, 0x6AB020F2, 0xF3B97148, 0x84BE41DE, 0x1ADAD47D, 0x6DDDE4EB, 0xF4D4B551, 0x83D385C7, /* [0][0x18]*/
        0x136C9856, 0x646BA8C0, 0xFD62F97A, 0x8A65C9EC, 0x14015C4F, 0x63066CD9, 0xFA0F3D63, 0x8D080DF5, /* [0][0x20]*/
        0x3B6E20C8, 0x4C69105E, 0xD56041E4, 0xA2677172, 0x3C03E4D1, 0x4B04D447, 0xD20D85FD, 0xA50AB56B, /* [0][0x28]*/
        0x35B5A8FA, 0x42B2986C, 0xDBBBC9D6, 0xACBCF940, 0x32D86CE3, 0x45DF5C75, 0xDCD60DCF, 0xABD13D59, /* [0][0x30]*/
        0x26D930AC, 0x51DE003A, 0xC8D75180, 0xBFD06116, 0x21B4F4B5, 0x56B3C423, 0xCFBA9599, 0xB8BDA50F, /* [0][0x38]*/
        0x2802B89E, 0x5F058808, 0xC60CD9B2, 0xB10BE924, 0x2F6F7C87, 0x58684C11, 0xC1611DAB, 0xB6662D3D, /* [0][0x40]*/
        0x76DC4190, 0x01DB7106, 0x98D220BC, 0xEFD5102A, 0x71B18589, 0x06B6B51F, 0x9FBFE4A5, 0xE8B8D433, /* [0][0x48]*/
        0x7807C9A2, 0x0F00F934, 0x9609A88E, 0xE10E9818, 0x7F6A0DBB, 0x086D3D2D, 0x91646C97, 0xE6635C01, /* [0][0x50]*/
        0x6B6B51F4, 0x1C6C6162, 0x856530D8, 0xF262004E, 0x6C0695ED, 0x1B01A57B, 0x8208F4C1, 0xF50FC457, /* [0][0x58]*/
        0x65B0D9C6, 0x12B7E950, 0x8BBEB8EA, 0xFCB9887C, 0x62DD1DDF, 0x15DA2D49, 0x8CD37CF3, 0xFBD44C65, /* [0][0x60]*/
        0x4DB26158, 0x3AB551CE, 0xA3BC0074, 0xD4BB30E2, 0x4ADFA541, 0x3DD895D7, 0xA4D1C46D, 0xD3D6F4FB, /* [0][0x68]*/
        0x4369E96A, 0x346ED9FC, 0xAD678846, 0xDA60B8D0, 0x44042D73, 0x33031DE5, 0xAA0A4C5F, 0xDD0D7CC9, /* [0][0x70]*/
        0x5005713C, 0x270241AA, 0xBE0B1010, 0xC90C2086, 0x5768B525, 0x206F85B3, 0xB966D409, 0xCE61E49F, /* [0][0x78]*/
        0x5EDEF90E, 0x29D9C998, 0xB0D09822, 0xC7D7A8B4, 0x59B33D17, 0x2EB40D81, 0xB7BD5C3B, 0xC0BA6CAD, /* [0][0x80]*/
        0xEDB88320, 0x9ABFB3B6, 0x03B6E20C, 0x74B1D29A, 0xEAD54739, 0x9DD277AF, 0x04DB2615, 0x73DC1683, /* [0][0x88]*/
        0xE3630B12, 0x94643B84, 0x0D6D6A3E, 0x7A6A5AA8, 0xE40ECF0B, 0x9309FF9D, 0x0A00AE27, 0x7D079EB1, /* [0][0x90]*/
        0xF00F9344, 0x8708A3D2, 0x1E01F268, 0x6906C2FE, 0xF762575D, 0x806567CB, 0x196C3671, 0x6E6B06E7, /* [0][0x98]*/
        0xFED41B76, 0x89D32BE0, 0x10DA7A5A, 0x67DD4ACC, 0xF9B9DF6F, 0x8EBEEFF9, 0x17B7BE43, 0x60B08ED5, /* [0][0xa0]*/
        0xD6D6A3E8, 0xA1D1937E, 0x38D8C2C4, 0x4FDFF252, 0xD1BB67F1, 0xA6BC5767, 0x3FB506DD, 0x48B2364B, /* [0][0xa8]*/
        0xD80D2BDA, 0xAF0A1B4C, 0x36034AF6, 0x41047A60, 0xDF60EFC3, 0xA867DF55, 0x316E8EEF, 0x4669BE79, /* [0][0xb0]*/
        0xCB61B38C, 0xBC66831A, 0x256FD2A0, 0x5268E236, 0xCC0C7795, 0xBB0B4703, 0x220216B9, 0x5505262F, /* [0][0xb8]*/
        0xC5BA3BBE, 0xB2BD0B28, 0x2BB45A92, 0x5CB36A04, 0xC2D7FFA7, 0xB5D0CF31, 0x2CD99E8B, 0x5BDEAE1D, /* [0][0xc0]*/
        0x9B64C2B0, 0xEC63F226, 0x756AA39C, 0x026D930A, 0x9C0906A9, 0xEB0E363F, 0x72076785, 0x05005713, /* [0][0xc8]*/
        0x95BF4A82, 0xE2B87A14, 0x7BB12BAE, 0x0CB61B38, 0x92D28E9B, 0xE5D5BE0D, 0x7CDCEFB7, 0x0BDBDF21, /* [0][0xd0]*/
        0x86D3D2D4, 0xF1D4E242, 0x68DDB3F8, 0x1FDA836E, 0x81BE16CD, 0xF6B9265B, 0x6FB077E1, 0x18B74777, /* [0][0xd8]*/
        0x88085AE6, 0xFF0F6A70, 0x66063BCA, 0x11010B5C, 0x8F659EFF, 0xF862AE69, 0x616BFFD3, 0x166CCF45, /* [0][0xe0]*/
        0xA00AE278, 0xD70DD2EE, 0x4E048354, 0x3903B3C2, 0xA7672661, 0xD06016F7, 0x4969474D, 0x3E6E77DB, /* [0][0xe8]*/
        0xAED16A4A, 0xD9D65ADC, 0x40DF0B66, 0x37D83BF0, 0xA9BCAE53, 0xDEBB9EC5, 0x47B2CF7F, 0x30B5FFE9, /* [0][0xf0]*/
        0xBDBDF21C, 0xCABAC28A, 0x53B39330, 0x24B4A3A6, 0xBAD03605, 0xCDD70693, 0x54DE5729, 0x23D967BF, /* [0][0xf8]*/
        0xB3667A2E, 0xC4614AB8, 0x5D681B02, 0x2A6F2B94, 0xB40BBE37, 0xC30C8EA1, 0x5A05DF1B, 0x2D02EF8D  /* [0][0x100]*/
    },
    {
        0x00000000, 0x191B3141, 0x32366282, 0x2B2D53C3, 0x646CC504, 0x7D77F445, 0x565AA786, 0x4F4196C7, /* [1][0x08]*/
        0xC8D98A08, 0xD1C2BB49, 0xFAEFE88A, 0xE3F4D9CB, 0xACB54F0C, 0xB5AE7E4D, 0x9E832D8E, 0x87981CCF, /* [1][0x10]*/
        0x4AC21251, 0x53D92310, 0x78F470D3, 0x61EF4192, 0x2EAED755, 0x37B5E614, 0x1C98B5D7, 0x05838496, /* [1][0x18]*/
        0x821B9859, 0x9B00A918, 0xB02DFADB, 0xA936CB9A, 0xE6775D5D, 0xFF6C6C1C, 0xD4413FDF, 0xCD5A0E9E, /* [1][0x20]*/
        0x958424A2, 0x8C9F15E3, 0xA7B24620, 0xBEA97761, 0xF1E8E1A6, 0xE8F3D0E7, 0xC3DE8324, 0xDAC5B265, /* [1][0x28]*/
        0x5D5DAEAA, 0x44469FEB, 0x6F6BCC28, 0x7670FD69, 0x39316BAE, 0x202A5AEF, 0x0B07092C, 0x121C386D, /* [1][0x30]*/
        0xDF4636F3, 0xC65D07B2, 0xED705471, 0xF46B6530, 0xBB2AF3F7, 0xA231C2B6, 0x891C9175, 0x9007A034, /* [1][0x38]*/
        0x179FBCFB, 0x0E848DBA, 0x25A9DE79, 0x3CB2EF38, 0x73F379FF, 0x6AE848BE, 0x41C51B7D, 0x58DE2A3C, /* [1][0x40]*/
        0xF0794F05, 0xE9627E44, 0xC24F2D87, 0xDB541CC6, 0x94158A01, 0x8D0EBB40, 0xA623E883, 0xBF38D9C2, /* [1][0x48]*/
        0x38A0C50D, 0x21BBF44C, 0x0A96A78F, 0x138D96CE, 0x5CCC0009, 0x45D73148, 0x6EFA628B, 0x77E153CA, /* [1][0x50]*/
        0xBABB5D54, 0xA3A06C15, 0x888D3FD6, 0x91960E97, 0xDED79850, 0xC7CCA911, 0xECE1FAD2, 0xF5FACB93, /* [1][0x58]*/
        0x7262D75C, 0x6B79E61D, 0x4054B5DE, 0x594F849F, 0x160E1258, 0x0F152319, 0x243870DA, 0x3D23419B, /* [1][0x60]*/
        0x65FD6BA7, 0x7CE65AE6, 0x57CB0925, 0x4ED03864, 0x0191AEA3, 0x188A9FE2, 0x33A7CC21, 0x2ABCFD60, /* [1][0x68]*/
        0xAD24E1AF, 0xB43FD0EE, 0x9F12832D, 0x8609B26C, 0xC94824AB, 0xD05315EA, 0xFB7E4629, 0xE2657768, /* [1][0x70]*/
        0x2F3F79F6, 0x362448B7, 0x1D091B74, 0x04122A35, 0x4B53BCF2, 0x52488DB3, 0x7965DE70, 0x607EEF31, /* [1][0x78]*/
        0xE7E6F3FE, 0xFEFDC2BF, 0xD5D0917C, 0xCCCBA03D, 0x838A36FA, 0x9A9107BB, 0xB1BC5478, 0xA8A76539, /* [1][0x80]*/
        0x3B83984B, 0x2298A90A, 0x09B5FAC9, 0x10AECB88, 0x5FEF5D4F, 0x46F46C0E, 0x6DD93FCD, 0x74C20E8C, /* [1][0x88]*/
        0xF35A1243, 0xEA412302, 0xC16C70C1, 0xD8774180, 0x9736D747, 0x8E2DE606, 0xA500B5C5, 0xBC1B8484, /* [1][0x90]*/
        0x71418A1A, 0x685ABB5B, 0x4377E898, 0x5A6CD9D9, 0x152D4F1E, 0x0C367E5F, 0x271B2D9C, 0x3E001CDD, /* [1][0x98]*/
        0xB9980012, 0xA0833153, 0x8BAE6290, 0x92B553D1, 0xDDF4C516, 0xC4EFF457, 0xEFC2A794, 0xF6D996D5, /* [1][0xa0]*/
        0xAE07BCE9, 0xB71C8DA8, 0x9C31DE6B, 0x852AEF2A, 0xCA6B79ED, 0xD37048AC, 0xF85D1B6F, 0xE1462A2E, /* [1][0xa8]*/
        0x66DE36E1, 0x7FC507A0, 0x54E85463, 0x4DF36522, 0x02B2F3E5, 0x1BA9C2A4, 0x30849167, 0x299FA026, /* [1][0xb0]*/
        0xE4C5AEB8, 0xFDDE9FF9, 0xD6F3CC3A, 0xCFE8FD7B, 0x80A96BBC, 0x99B25AFD, 0xB29F093E, 0xAB84387F, /* [1][0xb8]*/
        0x2C1C24B0, 0x350715F1, 0x1E2A4632, 0x07317773, 0x4870E1B4, 0x516BD0F5, 0x7A468336, 0x635DB277, /* [1][0xc0]*/
        0xCBFAD74E, 0xD2E1E60F, 0xF9CCB5CC, 0xE0D7848D, 0xAF96124A, 0xB68D230B, 0x9DA070C8, 0x84BB4189, /* [1][0xc8]*/
        0x03235D46, 0x1A386C07, 0x31153FC4, 0x280E0E85, 0x674F9842, 0x7E54A903, 0x5579FAC0, 0x4C62CB81, /* [1][0xd0]*/
        0x8138C51F, 0x9823F45E, 0xB30EA79D, 0xAA1596DC, 0xE554001B, 0xFC4F315A, 0xD7626299, 0xCE7953D8, /* [1][0xd8]*/
        0x49E14F17, 0x50FA7E56, 0x7BD72D95, 0x62CC1CD4, 0x2D8D8A13, 0x3496BB52, 0x1FBBE891, 0x06A0D9D0, /* [1][0xe0]*/
        0x5E7EF3EC, 0x4765C2AD, 0x6C48916E, 0x7553A02F, 0x3A1236E8, 0x230907A9, 0x0824546A, 0x113F652B, /* [1][0xe8]*/
        0x96A779E4, 0x8FBC48A5, 0xA4911B66, 0xBD8A2A27, 0xF2CBBCE0, 0xEBD08DA1, 0xC0FDDE62, 0xD9E6EF23, /* [1][0xf0]*/
        0x14BCE1BD, 0x0DA7D0FC, 0x268A833F, 0x3F91B27E, 0x70D024B9, 0x69CB15F8, 0x42E6463B, 0x5BFD777A, /* [1][0xf8]*/
        0xDC656BB5, 0xC57E5AF4, 0xEE530937, 0xF7483876, 0xB809AEB1, 0xA1129FF0, 0x8A3FCC33, 0x9324FD72  /* [1][0x100]*/
    },
    {
        0x00000000, 0x01C26A37, 0x0384D46E, 0x0246BE59, 0x0709A8DC, 0x06CBC2EB, 0x048D7CB2, 0x054F1685, /* [2][0x08]*/
        0x0E1351B8, 0x0FD13B8F, 0x0D9785D6, 0x0C55EFE1, 0x091AF964, 0x08D89353, 0x0A9E2D0A, 0x0B5C473D, /* [2][0x10]*/
        0x1C26A370, 0x1DE4C947, 0x1FA2771E, 0x1E601D29, 0x1B2F0BAC, 0x1AED619B, 0x18ABDFC2, 0x1969B5F5, /* [2][0x18]*/
        0x1235F2C8, 0x13F798FF, 0x11B126A6, 0x10734C91, 0x153C5A14, 0x14FE3023, 0x16B88E7A, 0x177AE44D, /* [2][0x20]*/
        0x384D46E0, 0x398F2CD7, 0x3BC9928E, 0x3A0BF8B9, 0x3F44EE3C, 0x3E86840B, 0x3CC03A52, 0x3D025065, /* [2][0x28]*/
        0x365E1758, 0x379C7D6F, 0x35DAC336, 0x3418A901, 0x3157BF84, 0x3095D5B3, 0x32D36BEA, 0x331101DD, /* [2][0x30]*/
        0x246BE590, 0x25A98FA7, 0x27EF31FE, 0x262D5BC9, 0x23624D4C, 0x22A0277B, 0x20E69922, 0x2124F315, /* [2][0x38]*/
        0x2A78B428, 0x2BBADE1F, 0x29FC6046, 0x283E0A71, 0x2D711CF4, 0x2CB376C3, 0x2EF5C89A, 0x2F37A2AD, /* [2][0x40]*/
        0x709A8DC0, 0x7158E7F7, 0x731E59AE, 0x72DC3399, 0x7793251C, 0x76514F2B, 0x7417F172, 0x75D59B45, /* [2][0x48]*/
        0x7E89DC78, 0x7F4BB64F, 0x7D0D0816, 0x7CCF6221, 0x798074A4, 0x78421E93, 0x7A04A0CA, 0x7BC6CAFD, /* [2][0x50]*/
        0x6CBC2EB0, 0x6D7E4487, 0x6F38FADE, 0x6EFA90E9, 0x6BB5866C, 0x6A77EC5B, 0x68315202, 0x69F33835, /* [2][0x58]*/
        0x62AF7F08, 0x636D153F, 0x612BAB66, 0x60E9C151, 0x65A6D7D4, 0x6464BDE3, 0x662203BA, 0x67E0698D, /* [2][0x60]*/
        0x48D7CB20, 0x4915A117, 0x4B531F4E, 0x4A917579, 0x4FDE63FC, 0x4E1C09CB, 0x4C5AB792, 0x4D98DDA5, /* [2][0x68]*/
        0x46C49A98, 0x4706F0AF, 0x45404EF6, 0x448224C1, 0x41CD3244, 0x400F5873, 0x4249E62A, 0x438B8C1D, /* [2][0x70]*/
        0x54F16850, 0x55330267, 0x5775BC3E, 0x56B7D609, 0x53F8C08C, 0x523AAABB, 0x507C14E2, 0x51BE7ED5, /* [2][0x78]*/
        0x5AE239E8, 0x5B2053DF, 0x5966ED86, 0x58A487B1, 0x5DEB9134, 0x5C29FB03, 0x5E6F455A, 0x5FAD2F6D, /* [2][0x80]*/
        0xE1351B80, 0xE0F771B7, 0xE2B1CFEE, 0xE373A5D9, 0xE63CB35C, 0xE7FED96B, 0xE5B86732, 0xE47A0D05, /* [2][0x88]*/
        0xEF264A38, 0xEEE4200F, 0xECA29E56, 0xED60F461, 0xE82FE2E4, 0xE9ED88D3, 0xEBAB368A, 0xEA695CBD, /* [2][0x90]*/
        0xFD13B8F0, 0xFCD1D2C7, 0xFE976C9E, 0xFF5506A9, 0xFA1A102C, 0xFBD87A1B, 0xF99EC442, 0xF85CAE75, /* [2][0x98]*/
        0xF300E948, 0xF2C2837F, 0xF0843D26, 0xF1465711, 0xF4094194, 0xF5CB2BA3, 0xF78D95FA, 0xF64FFFCD, /* [2][0xa0]*/
        0xD9785D60, 0xD8BA3757, 0xDAFC890E, 0xDB3EE339, 0xDE71F5BC, 0xDFB39F8B, 0xDDF521D2, 0xDC374BE5, /* [2][0xa8]*/
        0xD76B0CD8, 0xD6A966EF, 0xD4EFD8B6, 0xD52DB281, 0xD062A404, 0xD1A0CE33, 0xD3E6706A, 0xD2241A5D, /* [2][0xb0]*/
        0xC55EFE10, 0xC49C9427, 0xC6DA2A7E, 0xC7184049, 0xC25756CC, 0xC3953CFB, 0xC1D382A2, 0xC011E895, /* [2][0xb8]*/
        0xCB4DAFA8, 0xCA8FC59F, 0xC8C97BC6, 0xC90B11F1, 0xCC440774, 0xCD866D43, 0xCFC0D31A, 0xCE02B92D, /* [2][0xc0]*/
        0x91AF9640, 0x906DFC77, 0x922B422E, 0x93E92819, 0x96A63E9C, 0x976454AB, 0x9522EAF2, 0x94E080C5, /* [2][0xc8]*/
        0x9FBCC7F8, 0x9E7EADCF, 0x9C381396, 0x9DFA79A1, 0x98B56F24, 0x99770513, 0x9B31BB4A, 0x9AF3D17D, /* [2][0xd0]*/
        0x8D893530, 0x8C4B5F07, 0x8E0DE15E, 0x8FCF8B69, 0x8A809DEC, 0x8B42F7DB, 0x89044982, 0x88C623B5, /* [2][0xd8]*/
        0x839A6488, 0x82580EBF, 0x801EB0E6, 0x81DCDAD1, 0x8493CC54, 0x8551A663, 0x8717183A, 0x86D5720D, /* [2][0xe0]*/
        0xA9E2D0A0, 0xA820BA97, 0xAA6604CE, 0xABA46EF9, 0xAEEB787C, 0xAF29124B, 0xAD6FAC12, 0xACADC625, /* [2][0xe8]*/
        0xA7F18118, 0xA633EB2F, 0xA4755576, 0xA5B73F41, 0xA0F829C4, 0xA13A43F3, 0xA37CFDAA, 0xA2BE979D, /* [2][0xf0]*/
        0xB5C473D0, 0xB40619E7, 0xB640A7BE, 0xB782CD89, 0xB2CDDB0C, 0xB30FB13B, 0xB1490F62, 0xB08B6555, /* [2][0xf8]*/
        0xBBD72268, 0xBA15485F, 0xB853F606, 0xB9919C31, 0xBCDE8AB4, 0xBD1CE083, 0xBF5A5EDA, 0xBE9834ED  /* [2][0x100]*/
    },
    {
        0x00000000, 0xB8BC6765, 0xAA09C88B, 0x12B5AFEE, 0x8F629757, 0x37DEF032, 0x256B5FDC, 0x9DD738B9, /* [3][0x08]*/
        0xC5B428EF, 0x7D084F8A, 0x6FBDE064, 0xD7018701, 0x4AD6BFB8, 0xF26AD8DD, 0xE0DF7733, 0x58631056, /* [3][0x10]*/
        0x5019579F, 0xE8A530FA, 0xFA109F14, 0x42ACF871, 0xDF7BC0C8, 0x67C7A7AD, 0x75720843, 0xCDCE6F26, /* [3][0x18]*/
        0x95AD7F70, 0x2D111815, 0x3FA4B7FB, 0x8718D09E, 0x1ACFE827, 0xA2738F42, 0xB0C620AC, 0x087A47C9, /* [3][0x20]*/
        0xA032AF3E, 0x188EC85B, 0x0A3B67B5, 0xB28700D0, 0x2F503869, 0x97EC5F0C, 0x8559F0E2, 0x3DE59787, /* [3][0x28]*/
        0x658687D1, 0xDD3AE0B4, 0xCF8F4F5A, 0x7733283F, 0xEAE41086, 0x525877E3, 0x40EDD80D, 0xF851BF68, /* [3][0x30]*/
        0xF02BF8A1, 0x48979FC4, 0x5A22302A, 0xE29E574F, 0x7F496FF6, 0xC7F50893, 0xD540A77D, 0x6DFCC018, /* [3][0x38]*/
        0x359FD04E, 0x8D23B72B, 0x9F9618C5, 0x272A7FA0, 0xBAFD4719, 0x0241207C, 0x10F48F92, 0xA848E8F7, /* [3][0x40]*/
        0x9B14583D, 0x23A83F58, 0x311D90B6, 0x89A1F7D3, 0x1476CF6A, 0xACCAA80F, 0xBE7F07E1, 0x06C36084, /* [3][0x48]*/
        0x5EA070D2, 0xE61C17B7, 0xF4A9B859, 0x4C15DF3C, 0xD1C2E785, 0x697E80E0, 0x7BCB2F0E, 0xC377486B, /* [3][0x50]*/
        0xCB0D0FA2, 0x73B168C7, 0x6104C729, 0xD9B8A04C, 0x446F98F5, 0xFCD3FF90, 0xEE66507E, 0x56DA371B, /* [3][0x58]*/
        0x0EB9274D, 0xB6054028, 0xA4B0EFC6, 0x1C0C88A3, 0x81DBB01A, 0x3967D77F, 0x2BD27891, 0x936E1FF4, /* [3][0x60]*/
        0x3B26F703, 0x839A9066, 0x912F3F88, 0x299358ED, 0xB4446054, 0x0CF80731, 0x1E4DA8DF, 0xA6F1CFBA, /* [3][0x68]*/
        0xFE92DFEC, 0x462EB889, 0x549B1767, 0xEC277002, 0x71F048BB, 0xC94C2FDE, 0xDBF98030, 0x6345E755, /* [3][0x70]*/
        0x6B3FA09C, 0xD383C7F9, 0xC1366817, 0x798A0F72, 0xE45D37CB, 0x5CE150AE, 0x4E54FF40, 0xF6E89825, /* [3][0x78]*/
        0xAE8B8873, 0x1637EF16, 0x048240F8, 0xBC3E279D, 0x21E91F24, 0x99557841, 0x8BE0D7AF, 0x335CB0CA, /* [3][0x80]*/
        0xED59B63B, 0x55E5D15E, 0x47507EB0, 0xFFEC19D5, 0x623B216C, 0xDA874609, 0xC832E9E7, 0x708E8E82, /* [3][0x88]*/
        0x28ED9ED4, 0x9051F9B1, 0x82E4565F, 0x3A58313A, 0xA78F0983, 0x1F336EE6, 0x0D86C108, 0xB53AA66D, /* [3][0x90]*/
        0xBD40E1A4, 0x05FC86C1, 0x1749292F, 0xAFF54E4A, 0x322276F3, 0x8A9E1196, 0x982BBE78, 0x2097D91D, /* [3][0x98]*/
        0x78F4C94B, 0xC048AE2E, 0xD2FD01C0, 0x6A4166A5, 0xF7965E1C, 0x4F2A3979, 0x5D9F9697, 0xE523F1F2, /* [3][0xa0]*/
        0x4D6B1905, 0xF5D77E60, 0xE762D18E, 0x5FDEB6EB, 0xC2098E52, 0x7AB5E937, 0x680046D9, 0xD0BC21BC, /* [3][0xa8]*/
        0x88DF31EA, 0x3063568F, 0x22D6F961, 0x9A6A9E04, 0x07BDA6BD, 0xBF01C1D8, 0xADB46E36, 0x15080953, /* [3][0xb0]*/
        0x1D724E9A, 0xA5CE29FF, 0xB77B8611, 0x0FC7E174, 0x9210D9CD, 0x2AACBEA8, 0x38191146, 0x80A57623, /* [3][0xb8]*/
        0xD8C66675, 0x607A0110, 0x72CFAEFE, 0xCA73C99B, 0x57A4F122, 0xEF189647, 0xFDAD39A9, 0x45115ECC, /* [3][0xc0]*/
        0x764DEE06, 0xCEF18963, 0xDC44268D, 0x64F841E8, 0xF92F7951, 0x41931E34, 0x5326B1DA, 0xEB9AD6BF, /* [3][0xc8]*/
        0xB3F9C6E9, 0x0B45A18C, 0x19F00E62, 0xA14C6907, 0x3C9B51BE, 0x842736DB, 0x96929935, 0x2E2EFE50, /* [3][0xd0]*/
        0x2654B999, 0x9EE8DEFC, 0x8C5D7112, 0x34E11677, 0xA9362ECE, 0x118A49AB, 0x033FE645, 0xBB838120, /* [3][0xd8]*/
        0xE3E09176, 0x5B5CF613, 0x49E959FD, 0xF1553E98, 0x6C820621, 0xD43E6144, 0xC68BCEAA, 0x7E37A9CF, /* [3][0xe0]*/
        0xD67F4138, 0x6EC3265D, 0x7C7689B3, 0xC4CAEED6, 0x591DD66F, 0xE1A1B10A, 0xF3141EE4, 0x4BA87981, /* [3][0xe8]*/
        0x13CB69D7, 0xAB770EB2, 0xB9C2A15C, 0x017EC639, 0x9CA9FE80, 0x241599E5, 0x36A0360B, 0x8E1C516E, /* [3][0xf0]*/
        0x866616A7, 0x3EDA71C2, 0x2C6FDE2C, 0x94D3B949, 0x090481F0, 0xB1B8E695, 0xA30D497B, 0x1BB12E1E, /* [3][0xf8]*/
        0x43D23E48, 0xFB6E592D, 0xE9DBF6C3, 0x516791A6, 0xCCB0A91F, 0x740CCE7A, 0x66B96194, 0xDE0506F1  /* [3][0x100]*/
    },
    {
        0x00000000, 0x3D6029B0, 0x7AC05360, 0x47A07AD0, 0xF580A6C0, 0xC8E08F70, 0x8F40F5A0, 0xB220DC10, /* [4][0x08]*/
        0x30704BC1, 0x0D106271, 0x4AB018A1, 0x77D03111, 0xC5F0ED01, 0xF890C4B1, 0xBF30BE61, 0x825097D1, /* [4][0x10]*/
        0x60E09782, 0x5D80BE32, 0x1A20C4E2, 0x2740ED52, 0x95603142, 0xA80018F2, 0xEFA06222, 0xD2C04B92, /* [4][0x18]*/
        0x5090DC43, 0x6DF0F5F3, 0x2A508F23, 0x1730A693, 0xA5107A83, 0x98705333, 0xDFD029E3, 0xE2B00053, /* [4][0x20]*/
        0xC1C12F04, 0xFCA106B4, 0xBB017C64, 0x866155D4, 0x344189C4, 0x0921A074, 0x4E81DAA4, 0x73E1F314, /* [4][0x28]*/
        0xF1B164C5, 0xCCD14D75, 0x8B7137A5, 0xB6111E15, 0x0431C205, 0x3951EBB5, 0x7EF19165, 0x4391B8D5, /* [4][0x30]*/
        0xA121B886, 0x9C419136, 0xDBE1EBE6, 0xE681C256, 0x54A11E46, 0x69C137F6, 0x2E614D26, 0x13016496, /* [4][0x38]*/
        0x9151F347, 0xAC31DAF7, 0xEB91A027, 0xD6F18997, 0x64D15587, 0x59B17C37, 0x1E1106E7, 0x23712F57, /* [4][0x40]*/
        0x58F35849, 0x659371F9, 0x22330B29, 0x1F532299, 0xAD73FE89, 0x9013D739, 0xD7B3ADE9, 0xEAD38459, /* [4][0x48]*/
        0x68831388, 0x55E33A38, 0x124340E8, 0x2F236958, 0x9D03B548, 0xA0639CF8, 0xE7C3E628, 0xDAA3CF98, /* [4][0x50]*/
        0x3813CFCB, 0x0573E67B, 0x42D39CAB, 0x7FB3B51B, 0xCD93690B, 0xF0F340BB, 0xB7533A6B, 0x8A3313DB, /* [4][0x58]*/
        0x0863840A, 0x3503ADBA, 0x72A3D76A, 0x4FC3FEDA, 0xFDE322CA, 0xC0830B7A, 0x872371AA, 0xBA43581A, /* [4][0x60]*/
        0x9932774D, 0xA4525EFD, 0xE3F2242D, 0xDE920D9D, 0x6CB2D18D, 0x51D2F83D, 0x167282ED, 0x2B12AB5D, /* [4][0x68]*/
        0xA9423C8C, 0x9422153C, 0xD3826FEC, 0xEEE2465C, 0x5CC29A4C, 0x61A2B3FC, 0x2602C92C, 0x1B62E09C, /* [4][0x70]*/
        0xF9D2E0CF, 0xC4B2C97F, 0x8312B3AF, 0xBE729A1F, 0x0C52460F, 0x31326FBF, 0x7692156F, 0x4BF23CDF, /* [4][0x78]*/
        0xC9A2AB0E, 0xF4C282BE, 0xB362F86E, 0x8E02D1DE, 0x3C220DCE, 0x0142247E, 0x46E25EAE, 0x7B82771E, /* [4][0x80]*/
        0xB1E6B092, 0x8C869922, 0xCB26E3F2, 0xF646CA42, 0x44661652, 0x79063FE2, 0x3EA64532, 0x03C66C82, /* [4][0x88]*/
        0x8196FB53, 0xBCF6D2E3, 0xFB56A833, 0xC6368183, 0x74165D93, 0x49767423, 0x0ED60EF3, 0x33B62743, /* [4][0x90]*/
        0xD1062710, 0xEC660EA0, 0xABC67470, 0x96A65DC0, 0x248681D0, 0x19E6A860, 0x5E46D2B0, 0x6326FB00, /* [4][0x98]*/
        0xE1766CD1, 0xDC164561, 0x9BB63FB1, 0xA6D61601, 0x14F6CA11, 0x2996E3A1, 0x6E369971, 0x5356B0C1, /* [4][0xa0]*/
        0x70279F96, 0x4D47B626, 0x0AE7CCF6, 0x3787E546, 0x85A73956, 0xB8C710E6, 0xFF676A36, 0xC2074386, /* [4][0xa8]*/
        0x4057D457, 0x7D37FDE7, 0x3A978737, 0x07F7AE87, 0xB5D77297, 0x88B75B27, 0xCF1721F7, 0xF2770847, /* [4][0xb0]*/
        0x10C70814, 0x2DA721A4, 0x6A075B74, 0x576772C4, 0xE547AED4, 0xD8278764, 0x9F87FDB4, 0xA2E7D404, /* [4][0xb8]*/
        0x20B743D5, 0x1DD76A65, 0x5A7710B5, 0x67173905, 0xD537E515, 0xE857CCA5, 0xAFF7B675, 0x92979FC5, /* [4][0xc0]*/
        0xE915E8DB, 0xD475C16B, 0x93D5BBBB, 0xAEB5920B, 0x1C954E1B, 0x21F567AB, 0x66551D7B, 0x5B3534CB, /* [4][0xc8]*/
        0xD965A31A, 0xE4058AAA, 0xA3A5F07A, 0x9EC5D9CA, 0x2CE505DA, 0x11852C6A, 0x562556BA, 0x6B457F0A, /* [4][0xd0]*/
        0x89F57F59, 0xB49556E9, 0xF3352C39, 0xCE550589, 0x7C75D999, 0x4115F029, 0x06B58AF9, 0x3BD5A349, /* [4][0xd8]*/
        0xB9853498, 0x84E51D28, 0xC34567F8, 0xFE254E48, 0x4C059258, 0x7165BBE8, 0x36C5C138, 0x0BA5E888, /* [4][0xe0]*/
        0x28D4C7DF, 0x15B4EE6F, 0x521494BF, 0x6F74BD0F, 0xDD54611F, 0xE03448AF, 0xA794327F, 0x9AF41BCF, /* [4][0xe8]*/
        0x18A48C1E, 0x25C4A5AE, 0x6264DF7E, 0x5F04F6CE, 0xED242ADE, 0xD044036E, 0x97E479BE, 0xAA84500E, /* [4][0xf0]*/
        0x4834505D, 0x755479ED, 0x32F4033D, 0x0F942A8D, 0xBDB4F69D, 0x80D4DF2D, 0xC774A5FD, 0xFA148C4D, /* [4][0xf8]*/
        0x78441B9C, 0x4524322C, 0x028448FC, 0x3FE4614C, 0x8DC4BD5C, 0xB0A494EC, 0xF704EE3C, 0xCA64C78C  /* [4][0x100]*/
    },
    {
        0x00000000, 0xCB5CD3A5, 0x4DC8A10B, 0x869472AE, 0x9B914216, 0x50CD91B3, 0xD659E31D, 0x1D0530B8, /* [5][0x08]*/
        0xEC53826D, 0x270F51C8, 0xA19B2366, 0x6AC7F0C3, 0x77C2C07B, 0xBC9E13DE, 0x3A0A6170, 0xF156B2D5, /* [5][0x10]*/
        0x03D6029B, 0xC88AD13E, 0x4E1EA390, 0x85427035, 0x9847408D, 0x531B9328, 0xD58FE186, 0x1ED33223, /* [5][0x18]*/
        0xEF8580F6, 0x24D95353, 0xA24D21FD, 0x6911F258, 0x7414C2E0, 0xBF481145, 0x39DC63EB, 0xF280B04E, /* [5][0x20]*/
        0x07AC0536, 0xCCF0D693, 0x4A64A43D, 0x81387798, 0x9C3D4720, 0x57619485, 0xD1F5E62B, 0x1AA9358E, /* [5][0x28]*/
        0xEBFF875B, 0x20A354FE, 0xA6372650, 0x6D6BF5F5, 0x706EC54D, 0xBB3216E8, 0x3DA66446, 0xF6FAB7E3, /* [5][0x30]*/
        0x047A07AD, 0xCF26D408, 0x49B2A6A6, 0x82EE7503, 0x9FEB45BB, 0x54B7961E, 0xD223E4B0, 0x197F3715, /* [5][0x38]*/
        0xE82985C0, 0x23755665, 0xA5E124CB, 0x6EBDF76E, 0x73B8C7D6, 0xB8E41473, 0x3E7066DD, 0xF52CB578, /* [5][0x40]*/
        0x0F580A6C, 0xC404D9C9, 0x4290AB67, 0x89CC78C2, 0x94C9487A, 0x5F959BDF, 0xD901E971, 0x125D3AD4, /* [5][0x48]*/
        0xE30B8801, 0x28575BA4, 0xAEC3290A, 0x659FFAAF, 0x789ACA17, 0xB3C619B2, 0x35526B1C, 0xFE0EB8B9, /* [5][0x50]*/
        0x0C8E08F7, 0xC7D2DB52, 0x4146A9FC, 0x8A1A7A59, 0x971F4AE1, 0x5C439944, 0xDAD7EBEA, 0x118B384F, /* [5][0x58]*/
        0xE0DD8A9A, 0x2B81593F, 0xAD152B91, 0x6649F834, 0x7B4CC88C, 0xB0101B29, 0x36846987, 0xFDD8BA22, /* [5][0x60]*/
        0x08F40F5A, 0xC3A8DCFF, 0x453CAE51, 0x8E607DF4, 0x93654D4C, 0x58399EE9, 0xDEADEC47, 0x15F13FE2, /* [5][0x68]*/
        0xE4A78D37, 0x2FFB5E92, 0xA96F2C3C, 0x6233FF99, 0x7F36CF21, 0xB46A1C84, 0x32FE6E2A, 0xF9A2BD8F, /* [5][0x70]*/
        0x0B220DC1, 0xC07EDE64, 0x46EAACCA, 0x8DB67F6F, 0x90B34FD7, 0x5BEF9C72, 0xDD7BEEDC, 0x16273D79, /* [5][0x78]*/
        0xE7718FAC, 0x2C2D5C09, 0xAAB92EA7, 0x61E5FD02, 0x7CE0CDBA, 0xB7BC1E1F, 0x31286CB1, 0xFA74BF14, /* [5][0x80]*/
        0x1EB014D8, 0xD5ECC77D, 0x5378B5D3, 0x98246676, 0x852156CE, 0x4E7D856B, 0xC8E9F7C5, 0x03B52460, /* [5][0x88]*/
        0xF2E396B5, 0x39BF4510, 0xBF2B37BE, 0x7477E41B, 0x6972D4A3, 0xA22E0706, 0x24BA75A8, 0xEFE6A60D, /* [5][0x90]*/
        0x1D661643, 0xD63AC5E6, 0x50AEB748, 0x9BF264ED, 0x86F75455, 0x4DAB87F0, 0xCB3FF55E, 0x006326FB, /* [5][0x98]*/
        0xF135942E, 0x3A69478B, 0xBCFD3525, 0x77A1E680, 0x6AA4D638, 0xA1F8059D, 0x276C7733, 0xEC30A496, /* [5][0xa0]*/
        0x191C11EE, 0xD240C24B, 0x54D4B0E5, 0x9F886340, 0x828D53F8, 0x49D1805D, 0xCF45F2F3, 0x04192156, /* [5][0xa8]*/
        0xF54F9383, 0x3E134026, 0xB8873288, 0x73DBE12D, 0x6EDED195, 0xA5820230, 0x2316709E, 0xE84AA33B, /* [5][0xb0]*/
        0x1ACA1375, 0xD196C0D0, 0x5702B27E, 0x9C5E61DB, 0x815B5163, 0x4A0782C6, 0xCC93F068, 0x07CF23CD, /* [5][0xb8]*/
        0xF6999118, 0x3DC542BD, 0xBB513013, 0x700DE3B6, 0x6D08D30E, 0xA65400AB, 0x20C07205, 0xEB9CA1A0, /* [5][0xc0]*/
        0x11E81EB4, 0xDAB4CD11, 0x5C20BFBF, 0x977C6C1A, 0x8A795CA2, 0x41258F07, 0xC7B1FDA9, 0x0CED2E0C, /* [5][0xc8]*/
        0xFDBB9CD9, 0x36E74F7C, 0xB0733DD2, 0x7B2FEE77, 0x662ADECF, 0xAD760D6A, 0x2BE27FC4, 0xE0BEAC61, /* [5][0xd0]*/
        0x123E1C2F, 0xD962CF8A, 0x5FF6BD24, 0x94AA6E81, 0x89AF5E39, 0x42F38D9C, 0xC467FF32, 0x0F3B2C97, /* [5][0xd8]*/
        0xFE6D9E42, 0x35314DE7, 0xB3A53F49, 0x78F9ECEC, 0x65FCDC54, 0xAEA00FF1, 0x28347D5F, 0xE368AEFA, /* [5][0xe0]*/
        0x16441B82, 0xDD18C827, 0x5B8CBA89, 0x90D0692C, 0x8DD55994, 0x46898A31, 0xC01DF89F, 0x0B412B3A, /* [5][0xe8]*/
        0xFA1799EF, 0x314B4A4A, 0xB7DF38E4, 0x7C83EB41, 0x6186DBF9, 0xAADA085C, 0x2C4E7AF2, 0xE712A957, /* [5][0xf0]*/
        0x15921919, 0xDECECABC, 0x585AB812, 0x93066BB7, 0x8E035B0F, 0x455F88AA, 0xC3CBFA04, 0x089729A1, /* [5][0xf8]*/
        0xF9C19B74, 0x329D48D1, 0xB4093A7F, 0x7F55E9DA, 0x6250D962, 0xA90C0AC7, 0x2F987869, 0xE4C4ABCC  /* [5][0x100]*/
    },
    {
        0x00000000, 0xA6770BB4, 0x979F1129, 0x31E81A9D, 0xF44F2413, 0x52382FA7, 0x63D0353A, 0xC5A73E8E, /* [6][0x08]*/
        0x33EF4E67, 0x959845D3, 0xA4705F4E, 0x020754FA, 0xC7A06A74, 0x61D761C0, 0x503F7B5D, 0xF64870E9, /* [6][0x10]*/
        0x67DE9CCE, 0xC1A9977A, 0xF0418DE7, 0x56368653, 0x9391B8DD, 0x35E6B369, 0x040EA9F4, 0xA279A240, /* [6][0x18]*/
        0x5431D2A9, 0xF246D91D, 0xC3AEC380, 0x65D9C834, 0xA07EF6BA, 0x0609FD0E, 0x37E1E793, 0x9196EC27, /* [6][0x20]*/
        0xCFBD399C, 0x69CA3228, 0x582228B5, 0xFE552301, 0x3BF21D8F, 0x9D85163B, 0xAC6D0CA6, 0x0A1A0712, /* [6][0x28]*/
        0xFC5277FB, 0x5A257C4F, 0x6BCD66D2, 0xCDBA6D66, 0x081D53E8, 0xAE6A585C, 0x9F8242C1, 0x39F54975, /* [6][0x30]*/
        0xA863A552, 0x0E14AEE6, 0x3FFCB47B, 0x998BBFCF, 0x5C2C8141, 0xFA5B8AF5, 0xCBB39068, 0x6DC49BDC, /* [6][0x38]*/
        0x9B8CEB35, 0x3DFBE081, 0x0C13FA1C, 0xAA64F1A8, 0x6FC3CF26, 0xC9B4C492, 0xF85CDE0F, 0x5E2BD5BB, /* [6][0x40]*/
        0x440B7579, 0xE27C7ECD, 0xD3946450, 0x75E36FE4, 0xB044516A, 0x16335ADE, 0x27DB4043, 0x81AC4BF7, /* [6][0x48]*/
        0x77E43B1E, 0xD19330AA, 0xE07B2A37, 0x460C2183, 0x83AB1F0D, 0x25DC14B9, 0x14340E24, 0xB2430590, /* [6][0x50]*/
        0x23D5E9B7, 0x85A2E203, 0xB44AF89E, 0x123DF32A, 0xD79ACDA4, 0x71EDC610, 0x4005DC8D, 0xE672D739, /* [6][0x58]*/
        0x103AA7D0, 0xB64DAC64, 0x87A5B6F9, 0x21D2BD4D, 0xE47583C3, 0x42028877, 0x73EA92EA, 0xD59D995E, /* [6][0x60]*/
        0x8BB64CE5, 0x2DC14751, 0x1C295DCC, 0xBA5E5678, 0x7FF968F6, 0xD98E6342, 0xE86679DF, 0x4E11726B, /* [6][0x68]*/
        0xB8590282, 0x1E2E0936, 0x2FC613AB, 0x89B1181F, 0x4C162691, 0xEA612D25, 0xDB8937B8, 0x7DFE3C0C, /* [6][0x70]*/
        0xEC68D02B, 0x4A1FDB9F, 0x7BF7C102, 0xDD80CAB6, 0x1827F438, 0xBE50FF8C, 0x8FB8E511, 0x29CFEEA5, /* [6][0x78]*/
        0xDF879E4C, 0x79F095F8, 0x48188F65, 0xEE6F84D1, 0x2BC8BA5F, 0x8DBFB1EB, 0xBC57AB76, 0x1A20A0C2, /* [6][0x80]*/
        0x8816EAF2, 0x2E61E146, 0x1F89FBDB, 0xB9FEF06F, 0x7C59CEE1, 0xDA2EC555, 0xEBC6DFC8, 0x4DB1D47C, /* [6][0x88]*/
        0xBBF9A495, 0x1D8EAF21, 0x2C66B5BC, 0x8A11BE08, 0x4FB68086, 0xE9C18B32, 0xD82991AF, 0x7E5E9A1B, /* [6][0x90]*/
        0xEFC8763C, 0x49BF7D88, 0x78576715, 0xDE206CA1, 0x1B87522F, 0xBDF0599B, 0x8C184306, 0x2A6F48B2, /* [6][0x98]*/
        0xDC27385B, 0x7A5033EF, 0x4BB82972, 0xEDCF22C6, 0x28681C48, 0x8E1F17FC, 0xBFF70D61, 0x198006D5, /* [6][0xa0]*/
        0x47ABD36E, 0xE1DCD8DA, 0xD034C247, 0x7643C9F3, 0xB3E4F77D, 0x1593FCC9, 0x247BE654, 0x820CEDE0, /* [6][0xa8]*/
        0x74449D09, 0xD23396BD, 0xE3DB8C20, 0x45AC8794, 0x800BB91A, 0x267CB2AE, 0x1794A833, 0xB1E3A387, /* [6][0xb0]*/
        0x20754FA0, 0x86024414, 0xB7EA5E89, 0x119D553D, 0xD43A6BB3, 0x724D6007, 0x43A57A9A, 0xE5D2712E, /* [6][0xb8]*/
        0x139A01C7, 0xB5ED0A73, 0x840510EE, 0x22721B5A, 0xE7D525D4, 0x41A22E60, 0x704A34FD, 0xD63D3F49, /* [6][0xc0]*/
        0xCC1D9F8B, 0x6A6A943F, 0x5B828EA2, 0xFDF58516, 0x3852BB98, 0x9E25B02C, 0xAFCDAAB1, 0x09BAA105, /* [6][0xc8]*/
        0xFFF2D1EC, 0x5985DA58, 0x686DC0C5, 0xCE1ACB71, 0x0BBDF5FF, 0xADCAFE4B, 0x9C22E4D6, 0x3A55EF62, /* [6][0xd0]*/
        0xABC30345, 0x0DB408F1, 0x3C5C126C, 0x9A2B19D8, 0x5F8C2756, 0xF9FB2CE2, 0xC813367F, 0x6E643DCB, /* [6][0xd8]*/
        0x982C4D22, 0x3E5B4696, 0x0FB35C0B, 0xA9C457BF, 0x6C636931, 0xCA146285, 0xFBFC7818, 0x5D8B73AC, /* [6][0xe0]*/
        0x03A0A617, 0xA5D7ADA3, 0x943FB73E, 0x3248BC8A, 0xF7EF8204, 0x519889B0, 0x6070932D, 0xC6079899, /* [6][0xe8]*/
        0x304FE870, 0x9638E3C4, 0xA7D0F959, 0x01A7F2ED, 0xC400CC63, 0x6277C7D7, 0x539FDD4A, 0xF5E8D6FE, /* [6][0xf0]*/
        0x647E3AD9, 0xC209316D, 0xF3E12BF0, 0x55962044, 0x90311ECA, 0x3646157E, 0x07AE0FE3, 0xA1D90457, /* [6][0xf8]*/
        0x579174BE, 0xF1E67F0A, 0xC00E6597, 0x66796E23, 0xA3DE50AD, 0x05A95B19, 0x34414184, 0x92364A30  /* [6][0x100]*/
    },
    {
        0x00000000, 0xCCAA009E, 0x4225077D, 0x8E8F07E3, 0x844A0EFA, 0x48E00E64, 0xC66F0987, 0x0AC50919, /* [7][0x08]*/
        0xD3E51BB5, 0x1F4F1B2B, 0x91C01CC8, 0x5D6A1C56, 0x57AF154F, 0x9B0515D1, 0x158A1232, 0xD92012AC, /* [7][0x10]*/
        0x7CBB312B, 0xB01131B5, 0x3E9E3656, 0xF23436C8, 0xF8F13FD1, 0x345B3F4F, 0xBAD438AC, 0x767E3832, /* [7][0x18]*/
        0xAF5E2A9E, 0x63F42A00, 0xED7B2DE3, 0x21D12D7D, 0x2B142464, 0xE7BE24FA, 0x69312319, 0xA59B2387, /* [7][0x20]*/
        0xF9766256, 0x35DC62C8, 0xBB53652B, 0x77F965B5, 0x7D3C6CAC, 0xB1966C32, 0x3F196BD1, 0xF3B36B4F, /* [7][0x28]*/
        0x2A9379E3, 0xE639797D, 0x68B67E9E, 0xA41C7E00, 0xAED97719, 0x62737787, 0xECFC7064, 0x205670FA, /* [7][0x30]*/
        0x85CD537D, 0x496753E3, 0xC7E85400, 0x0B42549E, 0x01875D87, 0xCD2D5D19, 0x43A25AFA, 0x8F085A64, /* [7][0x38]*/
        0x562848C8, 0x9A824856, 0x140D4FB5, 0xD8A74F2B, 0xD2624632, 0x1EC846AC, 0x9047414F, 0x5CED41D1, /* [7][0x40]*/
        0x299DC2ED, 0xE537C273, 0x6BB8C590, 0xA712C50E, 0xADD7CC17, 0x617DCC89, 0xEFF2CB6A, 0x2358CBF4, /* [7][0x48]*/
        0xFA78D958, 0x36D2D9C6, 0xB85DDE25, 0x74F7DEBB, 0x7E32D7A2, 0xB298D73C, 0x3C17D0DF, 0xF0BDD041, /* [7][0x50]*/
        0x5526F3C6, 0x998CF358, 0x1703F4BB, 0xDBA9F425, 0xD16CFD3C, 0x1DC6FDA2, 0x9349FA41, 0x5FE3FADF, /* [7][0x58]*/
        0x86C3E873, 0x4A69E8ED, 0xC4E6EF0E, 0x084CEF90, 0x0289E689, 0xCE23E617, 0x40ACE1F4, 0x8C06E16A, /* [7][0x60]*/
        0xD0EBA0BB, 0x1C41A025, 0x92CEA7C6, 0x5E64A758, 0x54A1AE41, 0x980BAEDF, 0x1684A93C, 0xDA2EA9A2, /* [7][0x68]*/
        0x030EBB0E, 0xCFA4BB90, 0x412BBC73, 0x8D81BCED, 0x8744B5F4, 0x4BEEB56A, 0xC561B289, 0x09CBB217, /* [7][0x70]*/
        0xAC509190, 0x60FA910E, 0xEE7596ED, 0x22DF9673, 0x281A9F6A, 0xE4B09FF4, 0x6A3F9817, 0xA6959889, /* [7][0x78]*/
        0x7FB58A25, 0xB31F8ABB, 0x3D908D58, 0xF13A8DC6, 0xFBFF84DF, 0x37558441, 0xB9DA83A2, 0x7570833C, /* [7][0x80]*/
        0x533B85DA, 0x9F918544, 0x111E82A7, 0xDDB48239, 0xD7718B20, 0x1BDB8BBE, 0x95548C5D, 0x59FE8CC3, /* [7][0x88]*/
        0x80DE9E6F, 0x4C749EF1, 0xC2FB9912, 0x0E51998C, 0x04949095, 0xC83E900B, 0x46B197E8, 0x8A1B9776, /* [7][0x90]*/
        0x2F80B4F1, 0xE32AB46F, 0x6DA5B38C, 0xA10FB312, 0xABCABA0B, 0x6760BA95, 0xE9EFBD76, 0x2545BDE8, /* [7][0x98]*/
        0xFC65AF44, 0x30CFAFDA, 0xBE40A839, 0x72EAA8A7, 0x782FA1BE, 0xB485A120, 0x3A0AA6C3, 0xF6A0A65D, /* [7][0xa0]*/
        0xAA4DE78C, 0x66E7E712, 0xE868E0F1, 0x24C2E06F, 0x2E07E976, 0xE2ADE9E8, 0x6C22EE0B, 0xA088EE95, /* [7][0xa8]*/
        0x79A8FC39, 0xB502FCA7, 0x3B8DFB44, 0xF727FBDA, 0xFDE2F2C3, 0x3148F25D, 0xBFC7F5BE, 0x736DF520, /* [7][0xb0]*/
        0xD6F6D6A7, 0x1A5CD639, 0x94D3D1DA, 0x5879D144, 0x52BCD85D, 0x9E16D8C3, 0x1099DF20, 0xDC33DFBE, /* [7][0xb8]*/
        0x0513CD12, 0xC9B9CD8C, 0x4736CA6F, 0x8B9CCAF1, 0x8159C3E8, 0x4DF3C376, 0xC37CC495, 0x0FD6C40B, /* [7][0xc0]*/
        0x7AA64737, 0xB60C47A9, 0x3883404A, 0xF42940D4, 0xFEEC49CD, 0x32464953, 0xBCC94EB0, 0x70634E2E, /* [7][0xc8]*/
        0xA9435C82, 0x65E95C1C, 0xEB665BFF, 0x27CC5B61, 0x2D095278, 0xE1A352E6, 0x6F2C5505, 0xA386559B, /* [7][0xd0]*/
        0x061D761C, 0xCAB77682, 0x44387161, 0x889271FF, 0x825778E6, 0x4EFD7878, 0xC0727F9B, 0x0CD87F05, /* [7][0xd8]*/
        0xD5F86DA9, 0x19526D37, 0x97DD6AD4, 0x5B776A4A, 0x51B26353, 0x9D1863CD, 0x1397642E, 0xDF3D64B0, /* [7][0xe0]*/
        0x83D02561, 0x4F7A25FF, 0xC1F5221C, 0x0D5F2282, 0x079A2B9B, 0xCB302B05, 0x45BF2CE6, 0x89152C78, /* [7][0xe8]*/
        0x50353ED4, 0x9C9F3E4A, 0x121039A9, 0xDEBA3937, 0xD47F302E, 0x18D530B0, 0x965A3753, 0x5AF037CD, /* [7][0xf0]*/
        0xFF6B144A, 0x33C114D4, 0xBD4E1337, 0x71E413A9, 0x7B211AB0, 0xB78B1A2E, 0x39041DCD, 0xF5AE1D53, /* [7][0xf8]*/
        0x2C8E0FFF, 0xE0240F61, 0x6EAB0882, 0xA201081C, 0xA8C40105, 0x646E019B, 0xEAE10678, 0x264B06E6  /* [7][0x100]*/
    },
    {
        0x00000000, 0x177B1443, 0x2EF62886, 0x398D3CC5, 0x5DEC510C, 0x4A97454F, 0x731A798A, 0x64616DC9, /* [8][0x08]*/
        0xBBD8A218, 0xACA3B65B, 0x952E8A9E, 0x82559EDD, 0xE634F314, 0xF14FE757, 0xC8C2DB92, 0xDFB9CFD1, /* [8][0x10]*/
        0xACC04271, 0xBBBB5632, 0x82366AF7, 0x954D7EB4, 0xF12C137D, 0xE657073E, 0xDFDA3BFB, 0xC8A12FB8, /* [8][0x18]*/
        0x1718E069, 0x0063F42A, 0x39EEC8EF, 0x2E95DCAC, 0x4AF4B165, 0x5D8FA526, 0x640299E3, 0x73798DA0, /* [8][0x20]*/
        0x82F182A3, 0x958A96E0, 0xAC07AA25, 0xBB7CBE66, 0xDF1DD3AF, 0xC866C7EC, 0xF1EBFB29, 0xE690EF6A, /* [8][0x28]*/
        0x392920BB, 0x2E5234F8, 0x17DF083D, 0x00A41C7E, 0x64C571B7, 0x73BE65F4, 0x4A335931, 0x5D484D72, /* [8][0x30]*/
        0x2E31C0D2, 0x394AD491, 0x00C7E854, 0x17BCFC17, 0x73DD91DE, 0x64A6859D, 0x5D2BB958, 0x4A50AD1B, /* [8][0x38]*/
        0x95E962CA, 0x82927689, 0xBB1F4A4C, 0xAC645E0F, 0xC80533C6, 0xDF7E2785, 0xE6F31B40, 0xF1880F03, /* [8][0x40]*/
        0xDE920307, 0xC9E91744, 0xF0642B81, 0xE71F3FC2, 0x837E520B, 0x94054648, 0xAD887A8D, 0xBAF36ECE, /* [8][0x48]*/
        0x654AA11F, 0x7231B55C, 0x4BBC8999, 0x5CC79DDA, 0x38A6F013, 0x2FDDE450, 0x1650D895, 0x012BCCD6, /* [8][0x50]*/
        0x72524176, 0x65295535, 0x5CA469F0, 0x4BDF7DB3, 0x2FBE107A, 0x38C50439, 0x014838FC, 0x16332CBF, /* [8][0x58]*/
        0xC98AE36E, 0xDEF1F72D, 0xE77CCBE8, 0xF007DFAB, 0x9466B262, 0x831DA621, 0xBA909AE4, 0xADEB8EA7, /* [8][0x60]*/
        0x5C6381A4, 0x4B1895E7, 0x7295A922, 0x65EEBD61, 0x018FD0A8, 0x16F4C4EB, 0x2F79F82E, 0x3802EC6D, /* [8][0x68]*/
        0xE7BB23BC, 0xF0C037FF, 0xC94D0B3A, 0xDE361F79, 0xBA5772B0, 0xAD2C66F3, 0x94A15A36, 0x83DA4E75, /* [8][0x70]*/
        0xF0A3C3D5, 0xE7D8D796, 0xDE55EB53, 0xC92EFF10, 0xAD4F92D9, 0xBA34869A, 0x83B9BA5F, 0x94C2AE1C, /* [8][0x78]*/
        0x4B7B61CD, 0x5C00758E, 0x658D494B, 0x72F65D08, 0x169730C1, 0x01EC2482, 0x38611847, 0x2F1A0C04, /* [8][0x80]*/
        0x6655004F, 0x712E140C, 0x48A328C9, 0x5FD83C8A, 0x3BB95143, 0x2CC24500, 0x154F79C5, 0x02346D86, /* [8][0x88]*/
        0xDD8DA257, 0xCAF6B614, 0xF37B8AD1, 0xE4009E92, 0x8061F35B, 0x971AE718, 0xAE97DBDD, 0xB9ECCF9E, /* [8][0x90]*/
        0xCA95423E, 0xDDEE567D, 0xE4636AB8, 0xF3187EFB, 0x97791332, 0x80020771, 0xB98F3BB4, 0xAEF42FF7, /* [8][0x98]*/
        0x714DE026, 0x6636F465, 0x5FBBC8A0, 0x48C0DCE3, 0x2CA1B12A, 0x3BDAA569, 0x025799AC, 0x152C8DEF, /* [8][0xa0]*/
        0xE4A482EC, 0xF3DF96AF, 0xCA52AA6A, 0xDD29BE29, 0xB948D3E0, 0xAE33C7A3, 0x97BEFB66, 0x80C5EF25, /* [8][0xa8]*/
        0x5F7C20F4, 0x480734B7, 0x718A0872, 0x66F11C31, 0x029071F8, 0x15EB65BB, 0x2C66597E, 0x3B1D4D3D, /* [8][0xb0]*/
        0x4864C09D, 0x5F1FD4DE, 0x6692E81B, 0x71E9FC58, 0x15889191, 0x02F385D2, 0x3B7EB917, 0x2C05AD54, /* [8][0xb8]*/
        0xF3BC6285, 0xE4C776C6, 0xDD4A4A03, 0xCA315E40, 0xAE503389, 0xB92B27CA, 0x80A61B0F, 0x97DD0F4C, /* [8][0xc0]*/
        0xB8C70348, 0xAFBC170B, 0x96312BCE, 0x814A3F8D, 0xE52B5244, 0xF2504607, 0xCBDD7AC2, 0xDCA66E81, /* [8][0xc8]*/
        0x031FA150, 0x1464B513, 0x2DE989D6, 0x3A929D95, 0x5EF3F05C, 0x4988E41F, 0x7005D8DA, 0x677ECC99, /* [8][0xd0]*/
        0x14074139, 0x037C557A, 0x3AF169BF, 0x2D8A7DFC, 0x49EB1035, 0x5E900476, 0x671D38B3, 0x70662CF0, /* [8][0xd8]*/
        0xAFDFE321, 0xB8A4F762, 0x8129CBA7, 0x9652DFE4, 0xF233B22D, 0xE548A66E, 0xDCC59AAB, 0xCBBE8EE8, /* [8][0xe0]*/
        0x3A3681EB, 0x2D4D95A8, 0x14C0A96D, 0x03BBBD2E, 0x67DAD0E7, 0x70A1C4A4, 0x492CF861, 0x5E57EC22, /* [8][0xe8]*/
        0x81EE23F3, 0x969537B0, 0xAF180B75, 0xB8631F36, 0xDC0272FF, 0xCB7966BC, 0xF2F45A79, 0xE58F4E3A, /* [8][0xf0]*/
        0x96F6C39A, 0x818DD7D9, 0xB800EB1C, 0xAF7BFF5F, 0xCB1A9296, 0xDC6186D5, 0xE5ECBA10, 0xF297AE53, /* [8][0xf8]*/
        0x2D2E6182, 0x3A5575C1, 0x03D84904, 0x14A35D47, 0x70C2308E, 0x67B924CD, 0x5E341808, 0x494F0C4B  /* [8][0x100]*/
    },
    {
        0x00000000, 0xEFC26B3E, 0x04F5D03D, 0xEB37BB03, 0x09EBA07A, 0xE629CB44, 0x0D1E7047, 0xE2DC1B79, /* [9][0x08]*/
        0x13D740F4, 0xFC152BCA, 0x172290C9, 0xF8E0FBF7, 0x1A3CE08E, 0xF5FE8BB0, 0x1EC930B3, 0xF10B5B8D, /* [9][0x10]*/
        0x27AE81E8, 0xC86CEAD6, 0x235B51D5, 0xCC993AEB, 0x2E452192, 0xC1874AAC, 0x2AB0F1AF, 0xC5729A91, /* [9][0x18]*/
        0x3479C11C, 0xDBBBAA22, 0x308C1121, 0xDF4E7A1F, 0x3D926166, 0xD2500A58, 0x3967B15B, 0xD6A5DA65, /* [9][0x20]*/
        0x4F5D03D0, 0xA09F68EE, 0x4BA8D3ED, 0xA46AB8D3, 0x46B6A3AA, 0xA974C894, 0x42437397, 0xAD8118A9, /* [9][0x28]*/
        0x5C8A4324, 0xB348281A, 0x587F9319, 0xB7BDF827, 0x5561E35E, 0xBAA38860, 0x51943363, 0xBE56585D, /* [9][0x30]*/
        0x68F38238, 0x8731E906, 0x6C065205, 0x83C4393B, 0x61182242, 0x8EDA497C, 0x65EDF27F, 0x8A2F9941, /* [9][0x38]*/
        0x7B24C2CC, 0x94E6A9F2, 0x7FD112F1, 0x901379CF, 0x72CF62B6, 0x9D0D0988, 0x763AB28B, 0x99F8D9B5, /* [9][0x40]*/
        0x9EBA07A0, 0x71786C9E, 0x9A4FD79D, 0x758DBCA3, 0x9751A7DA, 0x7893CCE4, 0x93A477E7, 0x7C661CD9, /* [9][0x48]*/
        0x8D6D4754, 0x62AF2C6A, 0x89989769, 0x665AFC57, 0x8486E72E, 0x6B448C10, 0x80733713, 0x6FB15C2D, /* [9][0x50]*/
        0xB9148648, 0x56D6ED76, 0xBDE15675, 0x52233D4B, 0xB0FF2632, 0x5F3D4D0C, 0xB40AF60F, 0x5BC89D31, /* [9][0x58]*/
        0xAAC3C6BC, 0x4501AD82, 0xAE361681, 0x41F47DBF, 0xA32866C6, 0x4CEA0DF8, 0xA7DDB6FB, 0x481FDDC5, /* [9][0x60]*/
        0xD1E70470, 0x3E256F4E, 0xD512D44D, 0x3AD0BF73, 0xD80CA40A, 0x37CECF34, 0xDCF97437, 0x333B1F09, /* [9][0x68]*/
        0xC2304484, 0x2DF22FBA, 0xC6C594B9, 0x2907FF87, 0xCBDBE4FE, 0x24198FC0, 0xCF2E34C3, 0x20EC5FFD, /* [9][0x70]*/
        0xF6498598, 0x198BEEA6, 0xF2BC55A5, 0x1D7E3E9B, 0xFFA225E2, 0x10604EDC, 0xFB57F5DF, 0x14959EE1, /* [9][0x78]*/
        0xE59EC56C, 0x0A5CAE52, 0xE16B1551, 0x0EA97E6F, 0xEC756516, 0x03B70E28, 0xE880B52B, 0x0742DE15, /* [9][0x80]*/
        0xE6050901, 0x09C7623F, 0xE2F0D93C, 0x0D32B202, 0xEFEEA97B, 0x002CC245, 0xEB1B7946, 0x04D91278, /* [9][0x88]*/
        0xF5D249F5, 0x1A1022CB, 0xF12799C8, 0x1EE5F2F6, 0xFC39E98F, 0x13FB82B1, 0xF8CC39B2, 0x170E528C, /* [9][0x90]*/
        0xC1AB88E9, 0x2E69E3D7, 0xC55E58D4, 0x2A9C33EA, 0xC8402893, 0x278243AD, 0xCCB5F8AE, 0x23779390, /* [9][0x98]*/
        0xD27CC81D, 0x3DBEA323, 0xD6891820, 0x394B731E, 0xDB976867, 0x34550359, 0xDF62B85A, 0x30A0D364, /* [9][0xa0]*/
        0xA9580AD1, 0x469A61EF, 0xADADDAEC, 0x426FB1D2, 0xA0B3AAAB, 0x4F71C195, 0xA4467A96, 0x4B8411A8, /* [9][0xa8]*/
        0xBA8F4A25, 0x554D211B, 0xBE7A9A18, 0x51B8F126, 0xB364EA5F, 0x5CA68161, 0xB7913A62, 0x5853515C, /* [9][0xb0]*/
        0x8EF68B39, 0x6134E007, 0x8A035B04, 0x65C1303A, 0x871D2B43, 0x68DF407D, 0x83E8FB7E, 0x6C2A9040, /* [9][0xb8]*/
        0x9D21CBCD, 0x72E3A0F3, 0x99D41BF0, 0x761670CE, 0x94CA6BB7, 0x7B080089, 0x903FBB8A, 0x7FFDD0B4, /* [9][0xc0]*/
        0x78BF0EA1, 0x977D659F, 0x7C4ADE9C, 0x9388B5A2, 0x7154AEDB, 0x9E96C5E5, 0x75A17EE6, 0x9A6315D8, /* [9][0xc8]*/
        0x6B684E55, 0x84AA256B, 0x6F9D9E68, 0x805FF556, 0x6283EE2F, 0x8D418511, 0x66763E12, 0x89B4552C, /* [9][0xd0]*/
        0x5F118F49, 0xB0D3E477, 0x5BE45F74, 0xB426344A, 0x56FA2F33, 0xB938440D, 0x520FFF0E, 0xBDCD9430, /* [9][0xd8]*/
        0x4CC6CFBD, 0xA304A483, 0x48331F80, 0xA7F174BE, 0x452D6FC7, 0xAAEF04F9, 0x41D8BFFA, 0xAE1AD4C4, /* [9][0xe0]*/
        0x37E20D71, 0xD820664F, 0x3317DD4C, 0xDCD5B672, 0x3E09AD0B, 0xD1CBC635, 0x3AFC7D36, 0xD53E1608, /* [9][0xe8]*/
        0x24354D85, 0xCBF726BB, 0x20C09DB8, 0xCF02F686, 0x2DDEEDFF, 0xC21C86C1, 0x292B3DC2, 0xC6E956FC, /* [9][0xf0]*/
        0x104C8C99, 0xFF8EE7A7, 0x14B95CA4, 0xFB7B379A, 0x19A72CE3, 0xF66547DD, 0x1D52FCDE, 0xF29097E0, /* [9][0xf8]*/
        0x039BCC6D, 0xEC59A753, 0x076E1C50, 0xE8AC776E, 0x0A706C17, 0xE5B20729, 0x0E85BC2A, 0xE147D714  /* [9][0x100]*/
    },
    {
        0x00000000, 0xC18EDFC0, 0x586CB9C1, 0x99E26601, 0xB0D97382, 0x7157AC42, 0xE8B5CA43, 0x293B1583, /* [10][0x08]*/
        0xBAC3E145, 0x7B4D3E85, 0xE2AF5884, 0x23218744, 0x0A1A92C7, 0xCB944D07, 0x52762B06, 0x93F8F4C6, /* [10][0x10]*/
        0xAEF6C4CB, 0x6F781B0B, 0xF69A7D0A, 0x3714A2CA, 0x1E2FB749, 0xDFA16889, 0x46430E88, 0x87CDD148, /* [10][0x18]*/
        0x1435258E, 0xD5BBFA4E, 0x4C599C4F, 0x8DD7438F, 0xA4EC560C, 0x656289CC, 0xFC80EFCD, 0x3D0E300D, /* [10][0x20]*/
        0x869C8FD7, 0x47125017, 0xDEF03616, 0x1F7EE9D6, 0x3645FC55, 0xF7CB2395, 0x6E294594, 0xAFA79A54, /* [10][0x28]*/
        0x3C5F6E92, 0xFDD1B152, 0x6433D753, 0xA5BD0893, 0x8C861D10, 0x4D08C2D0, 0xD4EAA4D1, 0x15647B11, /* [10][0x30]*/
        0x286A4B1C, 0xE9E494DC, 0x7006F2DD, 0xB1882D1D, 0x98B3389E, 0x593DE75E, 0xC0DF815F, 0x01515E9F, /* [10][0x38]*/
        0x92A9AA59, 0x53277599, 0xCAC51398, 0x0B4BCC58, 0x2270D9DB, 0xE3FE061B, 0x7A1C601A, 0xBB92BFDA, /* [10][0x40]*/
        0xD64819EF, 0x17C6C62F, 0x8E24A02E, 0x4FAA7FEE, 0x66916A6D, 0xA71FB5AD, 0x3EFDD3AC, 0xFF730C6C, /* [10][0x48]*/
        0x6C8BF8AA, 0xAD05276A, 0x34E7416B, 0xF5699EAB, 0xDC528B28, 0x1DDC54E8, 0x843E32E9, 0x45B0ED29, /* [10][0x50]*/
        0x78BEDD24, 0xB93002E4, 0x20D264E5, 0xE15CBB25, 0xC867AEA6, 0x09E97166, 0x900B1767, 0x5185C8A7, /* [10][0x58]*/
        0xC27D3C61, 0x03F3E3A1, 0x9A1185A0, 0x5B9F5A60, 0x72A44FE3, 0xB32A9023, 0x2AC8F622, 0xEB4629E2, /* [10][0x60]*/
        0x50D49638, 0x915A49F8, 0x08B82FF9, 0xC936F039, 0xE00DE5BA, 0x21833A7A, 0xB8615C7B, 0x79EF83BB, /* [10][0x68]*/
        0xEA17777D, 0x2B99A8BD, 0xB27BCEBC, 0x73F5117C, 0x5ACE04FF, 0x9B40DB3F, 0x02A2BD3E, 0xC32C62FE, /* [10][0x70]*/
        0xFE2252F3, 0x3FAC8D33, 0xA64EEB32, 0x67C034F2, 0x4EFB2171, 0x8F75FEB1, 0x169798B0, 0xD7194770, /* [10][0x78]*/
        0x44E1B3B6, 0x856F6C76, 0x1C8D0A77, 0xDD03D5B7, 0xF438C034, 0x35B61FF4, 0xAC5479F5, 0x6DDAA635, /* [10][0x80]*/
        0x77E1359F, 0xB66FEA5F, 0x2F8D8C5E, 0xEE03539E, 0xC738461D, 0x06B699DD, 0x9F54FFDC, 0x5EDA201C, /* [10][0x88]*/
        0xCD22D4DA, 0x0CAC0B1A, 0x954E6D1B, 0x54C0B2DB, 0x7DFBA758, 0xBC757898, 0x25971E99, 0xE419C159, /* [10][0x90]*/
        0xD917F154, 0x18992E94, 0x817B4895, 0x40F59755, 0x69CE82D6, 0xA8405D16, 0x31A23B17, 0xF02CE4D7, /* [10][0x98]*/
        0x63D41011, 0xA25ACFD1, 0x3BB8A9D0, 0xFA367610, 0xD30D6393, 0x1283BC53, 0x8B61DA52, 0x4AEF0592, /* [10][0xa0]*/
        0xF17DBA48, 0x30F36588, 0xA9110389, 0x689FDC49, 0x41A4C9CA, 0x802A160A, 0x19C8700B, 0xD846AFCB, /* [10][0xa8]*/
        0x4BBE5B0D, 0x8A3084CD, 0x13D2E2CC, 0xD25C3D0C, 0xFB67288F, 0x3AE9F74F, 0xA30B914E, 0x62854E8E, /* [10][0xb0]*/
        0x5F8B7E83, 0x9E05A143, 0x07E7C742, 0xC6691882, 0xEF520D01, 0x2EDCD2C1, 0xB73EB4C0, 0x76B06B00, /* [10][0xb8]*/
        0xE5489FC6, 0x24C64006, 0xBD242607, 0x7CAAF9C7, 0x5591EC44, 0x941F3384, 0x0DFD5585, 0xCC738A45, /* [10][0xc0]*/
        0xA1A92C70, 0x6027F3B0, 0xF9C595B1, 0x384B4A71, 0x11705FF2, 0xD0FE8032, 0x491CE633, 0x889239F3, /* [10][0xc8]*/
        0x1B6ACD35, 0xDAE412F5, 0x430674F4, 0x8288AB34, 0xABB3BEB7, 0x6A3D6177, 0xF3DF0776, 0x3251D8B6, /* [10][0xd0]*/
        0x0F5FE8BB, 0xCED1377B, 0x5733517A, 0x96BD8EBA, 0xBF869B39, 0x7E0844F9, 0xE7EA22F8, 0x2664FD38, /* [10][0xd8]*/
        0xB59C09FE, 0x7412D63E, 0xEDF0B03F, 0x2C7E6FFF, 0x05457A7C, 0xC4CBA5BC, 0x5D29C3BD, 0x9CA71C7D, /* [10][0xe0]*/
        0x2735A3A7, 0xE6BB7C67, 0x7F591A66, 0xBED7C5A6, 0x97ECD025, 0x56620FE5, 0xCF8069E4, 0x0E0EB624, /* [10][0xe8]*/
        0x9DF642E2, 0x5C789D22, 0xC59AFB23, 0x041424E3, 0x2D2F3160, 0xECA1EEA0, 0x754388A1, 0xB4CD5761, /* [10][0xf0]*/
        0x89C3676C, 0x484DB8AC, 0xD1AFDEAD, 0x1021016D, 0x391A14EE, 0xF894CB2E, 0x6176AD2F, 0xA0F872EF, /* [10][0xf8]*/
        0x33008629, 0xF28E59E9, 0x6B6C3FE8, 0xAAE2E028, 0x83D9F5AB, 0x42572A6B, 0xDBB54C6A, 0x1A3B93AA  /* [10][0x100]*/
    },
    {
        0x00000000, 0x9BA54C6F, 0xEC3B9E9F, 0x779ED2F0, 0x03063B7F, 0x98A37710, 0xEF3DA5E0, 0x7498E98F, /* [11][0x08]*/
        0x060C76FE, 0x9DA93A91, 0xEA37E861, 0x7192A40E, 0x050A4D81, 0x9EAF01EE, 0xE931D31E, 0x72949F71, /* [11][0x10]*/
        0x0C18EDFC, 0x97BDA193, 0xE0237363, 0x7B863F0C, 0x0F1ED683, 0x94BB9AEC, 0xE325481C, 0x78800473, /* [11][0x18]*/
        0x0A149B02, 0x91B1D76D, 0xE62F059D, 0x7D8A49F2, 0x0912A07D, 0x92B7EC12, 0xE5293EE2, 0x7E8C728D, /* [11][0x20]*/
        0x1831DBF8, 0x83949797, 0xF40A4567, 0x6FAF0908, 0x1B37E087, 0x8092ACE8, 0xF70C7E18, 0x6CA93277, /* [11][0x28]*/
        0x1E3DAD06, 0x8598E169, 0xF2063399, 0x69A37FF6, 0x1D3B9679, 0x869EDA16, 0xF10008E6, 0x6AA54489, /* [11][0x30]*/
        0x14293604, 0x8F8C7A6B, 0xF812A89B, 0x63B7E4F4, 0x172F0D7B, 0x8C8A4114, 0xFB1493E4, 0x60B1DF8B, /* [11][0x38]*/
        0x122540FA, 0x89800C95, 0xFE1EDE65, 0x65BB920A, 0x11237B85, 0x8A8637EA, 0xFD18E51A, 0x66BDA975, /* [11][0x40]*/
        0x3063B7F0, 0xABC6FB9F, 0xDC58296F, 0x47FD6500, 0x33658C8F, 0xA8C0C0E0, 0xDF5E1210, 0x44FB5E7F, /* [11][0x48]*/
        0x366FC10E, 0xADCA8D61, 0xDA545F91, 0x41F113FE, 0x3569FA71, 0xAECCB61E, 0xD95264EE, 0x42F72881, /* [11][0x50]*/
        0x3C7B5A0C, 0xA7DE1663, 0xD040C493, 0x4BE588FC, 0x3F7D6173, 0xA4D82D1C, 0xD346FFEC, 0x48E3B383, /* [11][0x58]*/
        0x3A772CF2, 0xA1D2609D, 0xD64CB26D, 0x4DE9FE02, 0x3971178D, 0xA2D45BE2, 0xD54A8912, 0x4EEFC57D, /* [11][0x60]*/
        0x28526C08, 0xB3F72067, 0xC469F297, 0x5FCCBEF8, 0x2B545777, 0xB0F11B18, 0xC76FC9E8, 0x5CCA8587, /* [11][0x68]*/
        0x2E5E1AF6, 0xB5FB5699, 0xC2658469, 0x59C0C806, 0x2D582189, 0xB6FD6DE6, 0xC163BF16, 0x5AC6F379, /* [11][0x70]*/
        0x244A81F4, 0xBFEFCD9B, 0xC8711F6B, 0x53D45304, 0x274CBA8B, 0xBCE9F6E4, 0xCB772414, 0x50D2687B, /* [11][0x78]*/
        0x2246F70A, 0xB9E3BB65, 0xCE7D6995, 0x55D825FA, 0x2140CC75, 0xBAE5801A, 0xCD7B52EA, 0x56DE1E85, /* [11][0x80]*/
        0x60C76FE0, 0xFB62238F, 0x8CFCF17F, 0x1759BD10, 0x63C1549F, 0xF86418F0, 0x8FFACA00, 0x145F866F, /* [11][0x88]*/
        0x66CB191E, 0xFD6E5571, 0x8AF08781, 0x1155CBEE, 0x65CD2261, 0xFE686E0E, 0x89F6BCFE, 0x1253F091, /* [11][0x90]*/
        0x6CDF821C, 0xF77ACE73, 0x80E41C83, 0x1B4150EC, 0x6FD9B963, 0xF47CF50C, 0x83E227FC, 0x18476B93, /* [11][0x98]*/
        0x6AD3F4E2, 0xF176B88D, 0x86E86A7D, 0x1D4D2612, 0x69D5CF9D, 0xF27083F2, 0x85EE5102, 0x1E4B1D6D, /* [11][0xa0]*/
        0x78F6B418, 0xE353F877, 0x94CD2A87, 0x0F6866E8, 0x7BF08F67, 0xE055C308, 0x97CB11F8, 0x0C6E5D97, /* [11][0xa8]*/
        0x7EFAC2E6, 0xE55F8E89, 0x92C15C79, 0x09641016, 0x7DFCF999, 0xE659B5F6, 0x91C76706, 0x0A622B69, /* [11][0xb0]*/
        0x74EE59E4, 0xEF4B158B, 0x98D5C77B, 0x03708B14, 0x77E8629B, 0xEC4D2EF4, 0x9BD3FC04, 0x0076B06B, /* [11][0xb8]*/
        0x72E22F1A, 0xE9476375, 0x9ED9B185, 0x057CFDEA, 0x71E41465, 0xEA41580A, 0x9DDF8AFA, 0x067AC695, /* [11][0xc0]*/
        0x50A4D810, 0xCB01947F, 0xBC9F468F, 0x273A0AE0, 0x53A2E36F, 0xC807AF00, 0xBF997DF0, 0x243C319F, /* [11][0xc8]*/
        0x56A8AEEE, 0xCD0DE281, 0xBA933071, 0x21367C1E, 0x55AE9591, 0xCE0BD9FE, 0xB9950B0E, 0x22304761, /* [11][0xd0]*/
        0x5CBC35EC, 0xC7197983, 0xB087AB73, 0x2B22E71C, 0x5FBA0E93, 0xC41F42FC, 0xB381900C, 0x2824DC63, /* [11][0xd8]*/
        0x5AB04312, 0xC1150F7D, 0xB68BDD8D, 0x2D2E91E2, 0x59B6786D, 0xC2133402, 0xB58DE6F2, 0x2E28AA9D, /* [11][0xe0]*/
        0x489503E8, 0xD3304F87, 0xA4AE9D77, 0x3F0BD118, 0x4B933897, 0xD03674F8, 0xA7A8A608, 0x3C0DEA67, /* [11][0xe8]*/
        0x4E997516, 0xD53C3979, 0xA2A2EB89, 0x3907A7E6, 0x4D9F4E69, 0xD63A0206, 0xA1A4D0F6, 0x3A019C99, /* [11][0xf0]*/
        0x448DEE14, 0xDF28A27B, 0xA8B6708B, 0x33133CE4, 0x478BD56B, 0xDC2E9904, 0xABB04BF4, 0x3015079B, /* [11][0xf8]*/
        0x428198EA, 0xD924D485, 0xAEBA0675, 0x351F4A1A, 0x4187A395, 0xDA22EFFA, 0xADBC3D0A, 0x36197165  /* [11][0x100]*/
    },
    {
        0x00000000, 0xDD96D985, 0x605CB54B, 0xBDCA6CCE, 0xC0B96A96, 0x1D2FB313, 0xA0E5DFDD, 0x7D730658, /* [12][0x08]*/
        0x5A03D36D, 0x87950AE8, 0x3A5F6626, 0xE7C9BFA3, 0x9ABAB9FB, 0x472C607E, 0xFAE60CB0, 0x2770D535, /* [12][0x10]*/
        0xB407A6DA, 0x69917F5F, 0xD45B1391, 0x09CDCA14, 0x74BECC4C, 0xA92815C9, 0x14E27907, 0xC974A082, /* [12][0x18]*/
        0xEE0475B7, 0x3392AC32, 0x8E58C0FC, 0x53CE1979, 0x2EBD1F21, 0xF32BC6A4, 0x4EE1AA6A, 0x937773EF, /* [12][0x20]*/
        0xB37E4BF5, 0x6EE89270, 0xD322FEBE, 0x0EB4273B, 0x73C72163, 0xAE51F8E6, 0x139B9428, 0xCE0D4DAD, /* [12][0x28]*/
        0xE97D9898, 0x34EB411D, 0x89212DD3, 0x54B7F456, 0x29C4F20E, 0xF4522B8B, 0x49984745, 0x940E9EC0, /* [12][0x30]*/
        0x0779ED2F, 0xDAEF34AA, 0x67255864, 0xBAB381E1, 0xC7C087B9, 0x1A565E3C, 0xA79C32F2, 0x7A0AEB77, /* [12][0x38]*/
        0x5D7A3E42, 0x80ECE7C7, 0x3D268B09, 0xE0B0528C, 0x9DC354D4, 0x40558D51, 0xFD9FE19F, 0x2009381A, /* [12][0x40]*/
        0xBD8D91AB, 0x601B482E, 0xDDD124E0, 0x0047FD65, 0x7D34FB3D, 0xA0A222B8, 0x1D684E76, 0xC0FE97F3, /* [12][0x48]*/
        0xE78E42C6, 0x3A189B43, 0x87D2F78D, 0x5A442E08, 0x27372850, 0xFAA1F1D5, 0x476B9D1B, 0x9AFD449E, /* [12][0x50]*/
        0x098A3771, 0xD41CEEF4, 0x69D6823A, 0xB4405BBF, 0xC9335DE7, 0x14A58462, 0xA96FE8AC, 0x74F93129, /* [12][0x58]*/
        0x5389E41C, 0x8E1F3D99, 0x33D55157, 0xEE4388D2, 0x93308E8A, 0x4EA6570F, 0xF36C3BC1, 0x2EFAE244, /* [12][0x60]*/
        0x0EF3DA5E, 0xD36503DB, 0x6EAF6F15, 0xB339B690, 0xCE4AB0C8, 0x13DC694D, 0xAE160583, 0x7380DC06, /* [12][0x68]*/
        0x54F00933, 0x8966D0B6, 0x34ACBC78, 0xE93A65FD, 0x944963A5, 0x49DFBA20, 0xF415D6EE, 0x29830F6B, /* [12][0x70]*/
        0xBAF47C84, 0x6762A501, 0xDAA8C9CF, 0x073E104A, 0x7A4D1612, 0xA7DBCF97, 0x1A11A359, 0xC7877ADC, /* [12][0x78]*/
        0xE0F7AFE9, 0x3D61766C, 0x80AB1AA2, 0x5D3DC327, 0x204EC57F, 0xFDD81CFA, 0x40127034, 0x9D84A9B1, /* [12][0x80]*/
        0xA06A2517, 0x7DFCFC92, 0xC036905C, 0x1DA049D9, 0x60D34F81, 0xBD459604, 0x008FFACA, 0xDD19234F, /* [12][0x88]*/
        0xFA69F67A, 0x27FF2FFF, 0x9A354331, 0x47A39AB4, 0x3AD09CEC, 0xE7464569, 0x5A8C29A7, 0x871AF022, /* [12][0x90]*/
        0x146D83CD, 0xC9FB5A48, 0x74313686, 0xA9A7EF03, 0xD4D4E95B, 0x094230DE, 0xB4885C10, 0x691E8595, /* [12][0x98]*/
        0x4E6E50A0, 0x93F88925, 0x2E32E5EB, 0xF3A43C6E, 0x8ED73A36, 0x5341E3B3, 0xEE8B8F7D, 0x331D56F8, /* [12][0xa0]*/
        0x13146EE2, 0xCE82B767, 0x7348DBA9, 0xAEDE022C, 0xD3AD0474, 0x0E3BDDF1, 0xB3F1B13F, 0x6E6768BA, /* [12][0xa8]*/
        0x4917BD8F, 0x9481640A, 0x294B08C4, 0xF4DDD141, 0x89AED719, 0x54380E9C, 0xE9F26252, 0x3464BBD7, /* [12][0xb0]*/
        0xA713C838, 0x7A8511BD, 0xC74F7D73, 0x1AD9A4F6, 0x67AAA2AE, 0xBA3C7B2B, 0x07F617E5, 0xDA60CE60, /* [12][0xb8]*/
        0xFD101B55, 0x2086C2D0, 0x9D4CAE1E, 0x40DA779B, 0x3DA971C3, 0xE03FA846, 0x5DF5C488, 0x80631D0D, /* [12][0xc0]*/
        0x1DE7B4BC, 0xC0716D39, 0x7DBB01F7, 0xA02DD872, 0xDD5EDE2A, 0x00C807AF, 0xBD026B61, 0x6094B2E4, /* [12][0xc8]*/
        0x47E467D1, 0x9A72BE54, 0x27B8D29A, 0xFA2E0B1F, 0x875D0D47, 0x5ACBD4C2, 0xE701B80C, 0x3A976189, /* [12][0xd0]*/
        0xA9E01266, 0x7476CBE3, 0xC9BCA72D, 0x142A7EA8, 0x695978F0, 0xB4CFA175, 0x0905CDBB, 0xD493143E, /* [12][0xd8]*/
        0xF3E3C10B, 0x2E75188E, 0x93BF7440, 0x4E29ADC5, 0x335AAB9D, 0xEECC7218, 0x53061ED6, 0x8E90C753, /* [12][0xe0]*/
        0xAE99FF49, 0x730F26CC, 0xCEC54A02, 0x13539387, 0x6E2095DF, 0xB3B64C5A, 0x0E7C2094, 0xD3EAF911, /* [12][0xe8]*/
        0xF49A2C24, 0x290CF5A1, 0x94C6996F, 0x495040EA, 0x342346B2, 0xE9B59F37, 0x547FF3F9, 0x89E92A7C, /* [12][0xf0]*/
        0x1A9E5993, 0xC7088016, 0x7AC2ECD8, 0xA754355D, 0xDA273305, 0x07B1EA80, 0xBA7B864E, 0x67ED5FCB, /* [12][0xf8]*/
        0x409D8AFE, 0x9D0B537B, 0x20C13FB5, 0xFD57E630, 0x8024E068, 0x5DB239ED, 0xE0785523, 0x3DEE8CA6  /* [12][0x100]*/
    },
    {
        0x00000000, 0x9D0FE176, 0xE16EC4AD, 0x7C6125DB, 0x19AC8F1B, 0x84A36E6D, 0xF8C24BB6, 0x65CDAAC0, /* [13][0x08]*/
        0x33591E36, 0xAE56FF40, 0xD237DA9B, 0x4F383BED, 0x2AF5912D, 0xB7FA705B, 0xCB9B5580, 0x5694B4F6, /* [13][0x10]*/
        0x66B23C6C, 0xFBBDDD1A, 0x87DCF8C1, 0x1AD319B7, 0x7F1EB377, 0xE2115201, 0x9E7077DA, 0x037F96AC, /* [13][0x18]*/
        0x55EB225A, 0xC8E4C32C, 0xB485E6F7, 0x298A0781, 0x4C47AD41, 0xD1484C37, 0xAD2969EC, 0x3026889A, /* [13][0x20]*/
        0xCD6478D8, 0x506B99AE, 0x2C0ABC75, 0xB1055D03, 0xD4C8F7C3, 0x49C716B5, 0x35A6336E, 0xA8A9D218, /* [13][0x28]*/
        0xFE3D66EE, 0x63328798, 0x1F53A243, 0x825C4335, 0xE791E9F5, 0x7A9E0883, 0x06FF2D58, 0x9BF0CC2E, /* [13][0x30]*/
        0xABD644B4, 0x36D9A5C2, 0x4AB88019, 0xD7B7616F, 0xB27ACBAF, 0x2F752AD9, 0x53140F02, 0xCE1BEE74, /* [13][0x38]*/
        0x988F5A82, 0x0580BBF4, 0x79E19E2F, 0xE4EE7F59, 0x8123D599, 0x1C2C34EF, 0x604D1134, 0xFD42F042, /* [13][0x40]*/
        0x41B9F7F1, 0xDCB61687, 0xA0D7335C, 0x3DD8D22A, 0x581578EA, 0xC51A999C, 0xB97BBC47, 0x24745D31, /* [13][0x48]*/
        0x72E0E9C7, 0xEFEF08B1, 0x938E2D6A, 0x0E81CC1C, 0x6B4C66DC, 0xF64387AA, 0x8A22A271, 0x172D4307, /* [13][0x50]*/
        0x270BCB9D, 0xBA042AEB, 0xC6650F30, 0x5B6AEE46, 0x3EA74486, 0xA3A8A5F0, 0xDFC9802B, 0x42C6615D, /* [13][0x58]*/
        0x1452D5AB, 0x895D34DD, 0xF53C1106, 0x6833F070, 0x0DFE5AB0, 0x90F1BBC6, 0xEC909E1D, 0x719F7F6B, /* [13][0x60]*/
        0x8CDD8F29, 0x11D26E5F, 0x6DB34B84, 0xF0BCAAF2, 0x95710032, 0x087EE144, 0x741FC49F, 0xE91025E9, /* [13][0x68]*/
        0xBF84911F, 0x228B7069, 0x5EEA55B2, 0xC3E5B4C4, 0xA6281E04, 0x3B27FF72, 0x4746DAA9, 0xDA493BDF, /* [13][0x70]*/
        0xEA6FB345, 0x77605233, 0x0B0177E8, 0x960E969E, 0xF3C33C5E, 0x6ECCDD28, 0x12ADF8F3, 0x8FA21985, /* [13][0x78]*/
        0xD936AD73, 0x44394C05, 0x385869DE, 0xA55788A8, 0xC09A2268, 0x5D95C31E, 0x21F4E6C5, 0xBCFB07B3, /* [13][0x80]*/
        0x8373EFE2, 0x1E7C0E94, 0x621D2B4F, 0xFF12CA39, 0x9ADF60F9, 0x07D0818F, 0x7BB1A454, 0xE6BE4522, /* [13][0x88]*/
        0xB02AF1D4, 0x2D2510A2, 0x51443579, 0xCC4BD40F, 0xA9867ECF, 0x34899FB9, 0x48E8BA62, 0xD5E75B14, /* [13][0x90]*/
        0xE5C1D38E, 0x78CE32F8, 0x04AF1723, 0x99A0F655, 0xFC6D5C95, 0x6162BDE3, 0x1D039838, 0x800C794E, /* [13][0x98]*/
        0xD698CDB8, 0x4B972CCE, 0x37F60915, 0xAAF9E863, 0xCF3442A3, 0x523BA3D5, 0x2E5A860E, 0xB3556778, /* [13][0xa0]*/
        0x4E17973A, 0xD318764C, 0xAF795397, 0x3276B2E1, 0x57BB1821, 0xCAB4F957, 0xB6D5DC8C, 0x2BDA3DFA, /* [13][0xa8]*/
        0x7D4E890C, 0xE041687A, 0x9C204DA1, 0x012FACD7, 0x64E20617, 0xF9EDE761, 0x858CC2BA, 0x188323CC, /* [13][0xb0]*/
        0x28A5AB56, 0xB5AA4A20, 0xC9CB6FFB, 0x54C48E8D, 0x3109244D, 0xAC06C53B, 0xD067E0E0, 0x4D680196, /* [13][0xb8]*/
        0x1BFCB560, 0x86F35416, 0xFA9271CD, 0x679D90BB, 0x02503A7B, 0x9F5FDB0D, 0xE33EFED6, 0x7E311FA0, /* [13][0xc0]*/
        0xC2CA1813, 0x5FC5F965, 0x23A4DCBE, 0xBEAB3DC8, 0xDB669708, 0x4669767E, 0x3A0853A5, 0xA707B2D3, /* [13][0xc8]*/
        0xF1930625, 0x6C9CE753, 0x10FDC288, 0x8DF223FE, 0xE83F893E, 0x75306848, 0x09514D93, 0x945EACE5, /* [13][0xd0]*/
        0xA478247F, 0x3977C509, 0x4516E0D2, 0xD81901A4, 0xBDD4AB64, 0x20DB4A12, 0x5CBA6FC9, 0xC1B58EBF, /* [13][0xd8]*/
        0x97213A49, 0x0A2EDB3F, 0x764FFEE4, 0xEB401F92, 0x8E8DB552, 0x13825424, 0x6FE371FF, 0xF2EC9089, /* [13][0xe0]*/
        0x0FAE60CB, 0x92A181BD, 0xEEC0A466, 0x73CF4510, 0x1602EFD0, 0x8B0D0EA6, 0xF76C2B7D, 0x6A63CA0B, /* [13][0xe8]*/
        0x3CF77EFD, 0xA1F89F8B, 0xDD99BA50, 0x40965B26, 0x255BF1E6, 0xB8541090, 0xC435354B, 0x593AD43D, /* [13][0xf0]*/
        0x691C5CA7, 0xF413BDD1, 0x8872980A, 0x157D797C, 0x70B0D3BC, 0xEDBF32CA, 0x91DE1711, 0x0CD1F667, /* [13][0xf8]*/
        0x5A454291, 0xC74AA3E7, 0xBB2B863C, 0x2624674A, 0x43E9CD8A, 0xDEE62CFC, 0xA2870927, 0x3F88E851  /* [13][0x100]*/
    },
    {
        0x00000000, 0xB9FBDBE8, 0xA886B191, 0x117D6A79, 0x8A7C6563, 0x3387BE8B, 0x22FAD4F2, 0x9B010F1A, /* [14][0x08]*/
        0xCF89CC87, 0x7672176F, 0x670F7D16, 0xDEF4A6FE, 0x45F5A9E4, 0xFC0E720C, 0xED731875, 0x5488C39D, /* [14][0x10]*/
        0x44629F4F, 0xFD9944A7, 0xECE42EDE, 0x551FF536, 0xCE1EFA2C, 0x77E521C4, 0x66984BBD, 0xDF639055, /* [14][0x18]*/
        0x8BEB53C8, 0x32108820, 0x236DE259, 0x9A9639B1, 0x019736AB, 0xB86CED43, 0xA911873A, 0x10EA5CD2, /* [14][0x20]*/
        0x88C53E9E, 0x313EE576, 0x20438F0F, 0x99B854E7, 0x02B95BFD, 0xBB428015, 0xAA3FEA6C, 0x13C43184, /* [14][0x28]*/
        0x474CF219, 0xFEB729F1, 0xEFCA4388, 0x56319860, 0xCD30977A, 0x74CB4C92, 0x65B626EB, 0xDC4DFD03, /* [14][0x30]*/
        0xCCA7A1D1, 0x755C7A39, 0x64211040, 0xDDDACBA8, 0x46DBC4B2, 0xFF201F5A, 0xEE5D7523, 0x57A6AECB, /* [14][0x38]*/
        0x032E6D56, 0xBAD5B6BE, 0xABA8DCC7, 0x1253072F, 0x89520835, 0x30A9D3DD, 0x21D4B9A4, 0x982F624C, /* [14][0x40]*/
        0xCAFB7B7D, 0x7300A095, 0x627DCAEC, 0xDB861104, 0x40871E1E, 0xF97CC5F6, 0xE801AF8F, 0x51FA7467, /* [14][0x48]*/
        0x0572B7FA, 0xBC896C12, 0xADF4066B, 0x140FDD83, 0x8F0ED299, 0x36F50971, 0x27886308, 0x9E73B8E0, /* [14][0x50]*/
        0x8E99E432, 0x37623FDA, 0x261F55A3, 0x9FE48E4B, 0x04E58151, 0xBD1E5AB9, 0xAC6330C0, 0x1598EB28, /* [14][0x58]*/
        0x411028B5, 0xF8EBF35D, 0xE9969924, 0x506D42CC, 0xCB6C4DD6, 0x7297963E, 0x63EAFC47, 0xDA1127AF, /* [14][0x60]*/
        0x423E45E3, 0xFBC59E0B, 0xEAB8F472, 0x53432F9A, 0xC8422080, 0x71B9FB68, 0x60C49111, 0xD93F4AF9, /* [14][0x68]*/
        0x8DB78964, 0x344C528C, 0x253138F5, 0x9CCAE31D, 0x07CBEC07, 0xBE3037EF, 0xAF4D5D96, 0x16B6867E, /* [14][0x70]*/
        0x065CDAAC, 0xBFA70144, 0xAEDA6B3D, 0x1721B0D5, 0x8C20BFCF, 0x35DB6427, 0x24A60E5E, 0x9D5DD5B6, /* [14][0x78]*/
        0xC9D5162B, 0x702ECDC3, 0x6153A7BA, 0xD8A87C52, 0x43A97348, 0xFA52A8A0, 0xEB2FC2D9, 0x52D41931, /* [14][0x80]*/
        0x4E87F0BB, 0xF77C2B53, 0xE601412A, 0x5FFA9AC2, 0xC4FB95D8, 0x7D004E30, 0x6C7D2449, 0xD586FFA1, /* [14][0x88]*/
        0x810E3C3C, 0x38F5E7D4, 0x29888DAD, 0x90735645, 0x0B72595F, 0xB28982B7, 0xA3F4E8CE, 0x1A0F3326, /* [14][0x90]*/
        0x0AE56FF4, 0xB31EB41C, 0xA263DE65, 0x1B98058D, 0x80990A97, 0x3962D17F, 0x281FBB06, 0x91E460EE, /* [14][0x98]*/
        0xC56CA373, 0x7C97789B, 0x6DEA12E2, 0xD411C90A, 0x4F10C610, 0xF6EB1DF8, 0xE7967781, 0x5E6DAC69, /* [14][0xa0]*/
        0xC642CE25, 0x7FB915CD, 0x6EC47FB4, 0xD73FA45C, 0x4C3EAB46, 0xF5C570AE, 0xE4B81AD7, 0x5D43C13F, /* [14][0xa8]*/
        0x09CB02A2, 0xB030D94A, 0xA14DB333, 0x18B668DB, 0x83B767C1, 0x3A4CBC29, 0x2B31D650, 0x92CA0DB8, /* [14][0xb0]*/
        0x8220516A, 0x3BDB8A82, 0x2AA6E0FB, 0x935D3B13, 0x085C3409, 0xB1A7EFE1, 0xA0DA8598, 0x19215E70, /* [14][0xb8]*/
        0x4DA99DED, 0xF4524605, 0xE52F2C7C, 0x5CD4F794, 0xC7D5F88E, 0x7E2E2366, 0x6F53491F, 0xD6A892F7, /* [14][0xc0]*/
        0x847C8BC6, 0x3D87502E, 0x2CFA3A57, 0x9501E1BF, 0x0E00EEA5, 0xB7FB354D, 0xA6865F34, 0x1F7D84DC, /* [14][0xc8]*/
        0x4BF54741, 0xF20E9CA9, 0xE373F6D0, 0x5A882D38, 0xC1892222, 0x7872F9CA, 0x690F93B3, 0xD0F4485B, /* [14][0xd0]*/
        0xC01E1489, 0x79E5CF61, 0x6898A518, 0xD1637EF0, 0x4A6271EA, 0xF399AA02, 0xE2E4C07B, 0x5B1F1B93, /* [14][0xd8]*/
        0x0F97D80E, 0xB66C03E6, 0xA711699F, 0x1EEAB277, 0x85EBBD6D, 0x3C106685, 0x2D6D0CFC, 0x9496D714, /* [14][0xe0]*/
        0x0CB9B558, 0xB5426EB0, 0xA43F04C9, 0x1DC4DF21, 0x86C5D03B, 0x3F3E0BD3, 0x2E4361AA, 0x97B8BA42, /* [14][0xe8]*/
        0xC33079DF, 0x7ACBA237, 0x6BB6C84E, 0xD24D13A6, 0x494C1CBC, 0xF0B7C754, 0xE1CAAD2D, 0x583176C5, /* [14][0xf0]*/
        0x48DB2A17, 0xF120F1FF, 0xE05D9B86, 0x59A6406E, 0xC2A74F74, 0x7B5C949C, 0x6A21FEE5, 0xD3DA250D, /* [14][0xf8]*/
        0x8752E690, 0x3EA93D78, 0x2FD45701, 0x962F8CE9, 0x0D2E83F3, 0xB4D5581B, 0xA5A83262, 0x1C53E98A  /* [14][0x100]*/
    },
    {
        0x00000000, 0xAE689191, 0x87A02563, 0x29C8B4F2, 0xD4314C87, 0x7A59DD16, 0x539169E4, 0xFDF9F875, /* [15][0x08]*/
        0x73139F4F, 0xDD7B0EDE, 0xF4B3BA2C, 0x5ADB2BBD, 0xA722D3C8, 0x094A4259, 0x2082F6AB, 0x8EEA673A, /* [15][0x10]*/
        0xE6273E9E, 0x484FAF0F, 0x61871BFD, 0xCFEF8A6C, 0x32167219, 0x9C7EE388, 0xB5B6577A, 0x1BDEC6EB, /* [15][0x18]*/
        0x9534A1D1, 0x3B5C3040, 0x129484B2, 0xBCFC1523, 0x4105ED56, 0xEF6D7CC7, 0xC6A5C835, 0x68CD59A4, /* [15][0x20]*/
        0x173F7B7D, 0xB957EAEC, 0x909F5E1E, 0x3EF7CF8F, 0xC30E37FA, 0x6D66A66B, 0x44AE1299, 0xEAC68308, /* [15][0x28]*/
        0x642CE432, 0xCA4475A3, 0xE38CC151, 0x4DE450C0, 0xB01DA8B5, 0x1E753924, 0x37BD8DD6, 0x99D51C47, /* [15][0x30]*/
        0xF11845E3, 0x5F70D472, 0x76B86080, 0xD8D0F111, 0x25290964, 0x8B4198F5, 0xA2892C07, 0x0CE1BD96, /* [15][0x38]*/
        0x820BDAAC, 0x2C634B3D, 0x05ABFFCF, 0xABC36E5E, 0x563A962B, 0xF85207BA, 0xD19AB348, 0x7FF222D9, /* [15][0x40]*/
        0x2E7EF6FA, 0x8016676B, 0xA9DED399, 0x07B64208, 0xFA4FBA7D, 0x54272BEC, 0x7DEF9F1E, 0xD3870E8F, /* [15][0x48]*/
        0x5D6D69B5, 0xF305F824, 0xDACD4CD6, 0x74A5DD47, 0x895C2532, 0x2734B4A3, 0x0EFC0051, 0xA09491C0, /* [15][0x50]*/
        0xC859C864, 0x663159F5, 0x4FF9ED07, 0xE1917C96, 0x1C6884E3, 0xB2001572, 0x9BC8A180, 0x35A03011, /* [15][0x58]*/
        0xBB4A572B, 0x1522C6BA, 0x3CEA7248, 0x9282E3D9, 0x6F7B1BAC, 0xC1138A3D, 0xE8DB3ECF, 0x46B3AF5E, /* [15][0x60]*/
        0x39418D87, 0x97291C16, 0xBEE1A8E4, 0x10893975, 0xED70C100, 0x43185091, 0x6AD0E463, 0xC4B875F2, /* [15][0x68]*/
        0x4A5212C8, 0xE43A8359, 0xCDF237AB, 0x639AA63A, 0x9E635E4F, 0x300BCFDE, 0x19C37B2C, 0xB7ABEABD, /* [15][0x70]*/
        0xDF66B319, 0x710E2288, 0x58C6967A, 0xF6AE07EB, 0x0B57FF9E, 0xA53F6E0F, 0x8CF7DAFD, 0x229F4B6C, /* [15][0x78]*/
        0xAC752C56, 0x021DBDC7, 0x2BD50935, 0x85BD98A4, 0x784460D1, 0xD62CF140, 0xFFE445B2, 0x518CD423, /* [15][0x80]*/
        0x5CFDEDF4, 0xF2957C65, 0xDB5DC897, 0x75355906, 0x88CCA173, 0x26A430E2, 0x0F6C8410, 0xA1041581, /* [15][0x88]*/
        0x2FEE72BB, 0x8186E32A, 0xA84E57D8, 0x0626C649, 0xFBDF3E3C, 0x55B7AFAD, 0x7C7F1B5F, 0xD2178ACE, /* [15][0x90]*/
        0xBADAD36A, 0x14B242FB, 0x3D7AF609, 0x93126798, 0x6EEB9FED, 0xC0830E7C, 0xE94BBA8E, 0x47232B1F, /* [15][0x98]*/
        0xC9C94C25, 0x67A1DDB4, 0x4E696946, 0xE001F8D7, 0x1DF800A2, 0xB3909133, 0x9A5825C1, 0x3430B450, /* [15][0xa0]*/
        0x4BC29689, 0xE5AA0718, 0xCC62B3EA, 0x620A227B, 0x9FF3DA0E, 0x319B4B9F, 0x1853FF6D, 0xB63B6EFC, /* [15][0xa8]*/
        0x38D109C6, 0x96B99857, 0xBF712CA5, 0x1119BD34, 0xECE04541, 0x4288D4D0, 0x6B406022, 0xC528F1B3, /* [15][0xb0]*/
        0xADE5A817, 0x038D3986, 0x2A458D74, 0x842D1CE5, 0x79D4E490, 0xD7BC7501, 0xFE74C1F3, 0x501C5062, /* [15][0xb8]*/
        0xDEF63758, 0x709EA6C9, 0x5956123B, 0xF73E83AA, 0x0AC77BDF, 0xA4AFEA4E, 0x8D675EBC, 0x230FCF2D, /* [15][0xc0]*/
        0x72831B0E, 0xDCEB8A9F, 0xF5233E6D, 0x5B4BAFFC, 0xA6B25789, 0x08DAC618, 0x211272EA, 0x8F7AE37B, /* [15][0xc8]*/
        0x01908441, 0xAFF815D0, 0x8630A122, 0x285830B3, 0xD5A1C8C6, 0x7BC95957, 0x5201EDA5, 0xFC697C34, /* [15][0xd0]*/
        0x94A42590, 0x3ACCB401, 0x130400F3, 0xBD6C9162, 0x40956917, 0xEEFDF886, 0xC7354C74, 0x695DDDE5, /* [15][0xd8]*/
        0xE7B7BADF, 0x49DF2B4E, 0x60179FBC, 0xCE7F0E2D, 0x3386F658, 0x9DEE67C9, 0xB426D33B, 0x1A4E42AA, /* [15][0xe0]*/
        0x65BC6073, 0xCBD4F1E2, 0xE21C4510, 0x4C74D481, 0xB18D2CF4, 0x1FE5BD65, 0x362D0997, 0x98459806, /* [15][0xe8]*/
        0x16AFFF3C, 0xB8C76EAD, 0x910FDA5F, 0x3F674BCE, 0xC29EB3BB, 0x6CF6222A, 0x453E96D8, 0xEB560749, /* [15][0xf0]*/
        0x839B5EED, 0x2DF3CF7C, 0x043B7B8E, 0xAA53EA1F, 0x57AA126A, 0xF9C283FB, 0xD00A3709, 0x7E62A698, /* [15][0xf8]*/
        0xF088C1A2, 0x5EE05033, 0x7728E4C1, 0xD9407550, 0x24B98D25, 0x8AD11CB4, 0xA319A846, 0x0D7139D7  /* [15][0x100]*/
    }};

/** Castagnoli CRC32c (iSCSI) lookup table for slice-by-4/8/16 */
const uint32_t CRC32C_TABLE[16][256] = {
    {
        0x00000000, 0xF26B8303, 0xE13B70F7, 0x1350F3F4, 0xC79A971F, 0x35F1141C, 0x26A1E7E8, 0xD4CA64EB, /* [0][0x08]*/
        0x8AD958CF, 0x78B2DBCC, 0x6BE22838, 0x9989AB3B, 0x4D43CFD0, 0xBF284CD3, 0xAC78BF27, 0x5E133C24, /* [0][0x10]*/
        0x105EC76F, 0xE235446C, 0xF165B798, 0x030E349B, 0xD7C45070, 0x25AFD373, 0x36FF2087, 0xC494A384, /* [0][0x18]*/
        0x9A879FA0, 0x68EC1CA3, 0x7BBCEF57, 0x89D76C54, 0x5D1D08BF, 0xAF768BBC, 0xBC267848, 0x4E4DFB4B, /* [0][0x20]*/
        0x20BD8EDE, 0xD2D60DDD, 0xC186FE29, 0x33ED7D2A, 0xE72719C1, 0x154C9AC2, 0x061C6936, 0xF477EA35, /* [0][0x28]*/
        0xAA64D611, 0x580F5512, 0x4B5FA6E6, 0xB93425E5, 0x6DFE410E, 0x9F95C20D, 0x8CC531F9, 0x7EAEB2FA, /* [0][0x30]*/
        0x30E349B1, 0xC288CAB2, 0xD1D83946, 0x23B3BA45, 0xF779DEAE, 0x05125DAD, 0x1642AE59, 0xE4292D5A, /* [0][0x38]*/
        0xBA3A117E, 0x4851927D, 0x5B016189, 0xA96AE28A, 0x7DA08661, 0x8FCB0562, 0x9C9BF696, 0x6EF07595, /* [0][0x40]*/
        0x417B1DBC, 0xB3109EBF, 0xA0406D4B, 0x522BEE48, 0x86E18AA3, 0x748A09A0, 0x67DAFA54, 0x95B17957, /* [0][0x48]*/
        0xCBA24573, 0x39C9C670, 0x2A993584, 0xD8F2B687, 0x0C38D26C, 0xFE53516F, 0xED03A29B, 0x1F682198, /* [0][0x50]*/
        0x5125DAD3, 0xA34E59D0, 0xB01EAA24, 0x42752927, 0x96BF4DCC, 0x64D4CECF, 0x77843D3B, 0x85EFBE38, /* [0][0x58]*/
        0xDBFC821C, 0x2997011F, 0x3AC7F2EB, 0xC8AC71E8, 0x1C661503, 0xEE0D9600, 0xFD5D65F4, 0x0F36E6F7, /* [0][0x60]*/
        0x61C69362, 0x93AD1061, 0x80FDE395, 0x72966096, 0xA65C047D, 0x5437877E, 0x4767748A, 0xB50CF789, /* [0][0x68]*/
        0xEB1FCBAD, 0x197448AE, 0x0A24BB5A, 0xF84F3859, 0x2C855CB2, 0xDEEEDFB1, 0xCDBE2C45, 0x3FD5AF46, /* [0][0x70]*/
        0x7198540D, 0x83F3D70E, 0x90A324FA, 0x62C8A7F9, 0xB602C312, 0x44694011, 0x5739B3E5, 0xA55230E6, /* [0][0x78]*/
        0xFB410CC2, 0x092A8FC1, 0x1A7A7C35, 0xE811FF36, 0x3CDB9BDD, 0xCEB018DE, 0xDDE0EB2A, 0x2F8B6829, /* [0][0x80]*/
        0x82F63B78, 0x709DB87B, 0x63CD4B8F, 0x91A6C88C, 0x456CAC67, 0xB7072F64, 0xA457DC90, 0x563C5F93, /* [0][0x88]*/
        0x082F63B7, 0xFA44E0B4, 0xE9141340, 0x1B7F9043, 0xCFB5F4A8, 0x3DDE77AB, 0x2E8E845F, 0xDCE5075C, /* [0][0x90]*/
        0x92A8FC17, 0x60C37F14, 0x73938CE0, 0x81F80FE3, 0x55326B08, 0xA759E80B, 0xB4091BFF, 0x466298FC, /* [0][0x98]*/
        0x1871A4D8, 0xEA1A27DB, 0xF94AD42F, 0x0B21572C, 0xDFEB33C7, 0x2D80B0C4, 0x3ED04330, 0xCCBBC033, /* [0][0xa0]*/
        0xA24BB5A6, 0x502036A5, 0x4370C551, 0xB11B4652, 0x65D122B9, 0x97BAA1BA, 0x84EA524E, 0x7681D14D, /* [0][0xa8]*/
        0x2892ED69, 0xDAF96E6A, 0xC9A99D9E, 0x3BC21E9D, 0xEF087A76, 0x1D63F975, 0x0E330A81, 0xFC588982, /* [0][0xb0]*/
        0xB21572C9, 0x407EF1CA, 0x532E023E, 0xA145813D, 0x758FE5D6, 0x87E466D5, 0x94B49521, 0x66DF1622, /* [0][0xb8]*/
        0x38CC2A06, 0xCAA7A905, 0xD9F75AF1, 0x2B9CD9F2, 0xFF56BD19, 0x0D3D3E1A, 0x1E6DCDEE, 0xEC064EED, /* [0][0xc0]*/
        0xC38D26C4, 0x31E6A5C7, 0x22B65633, 0xD0DDD530, 0x0417B1DB, 0xF67C32D8, 0xE52CC12C, 0x1747422F, /* [0][0xc8]*/
        0x49547E0B, 0xBB3FFD08, 0xA86F0EFC, 0x5A048DFF, 0x8ECEE914, 0x7CA56A17, 0x6FF599E3, 0x9D9E1AE0, /* [0][0xd0]*/
        0xD3D3E1AB, 0x21B862A8, 0x32E8915C, 0xC083125F, 0x144976B4, 0xE622F5B7, 0xF5720643, 0x07198540, /* [0][0xd8]*/
        0x590AB964, 0xAB613A67, 0xB831C993, 0x4A5A4A90, 0x9E902E7B, 0x6CFBAD78, 0x7FAB5E8C, 0x8DC0DD8F, /* [0][0xe0]*/
        0xE330A81A, 0x115B2B19, 0x020BD8ED, 0xF0605BEE, 0x24AA3F05, 0xD6C1BC06, 0xC5914FF2, 0x37FACCF1, /* [0][0xe8]*/
        0x69E9F0D5, 0x9B8273D6, 0x88D28022, 0x7AB90321, 0xAE7367CA, 0x5C18E4C9, 0x4F48173D, 0xBD23943E, /* [0][0xf0]*/
        0xF36E6F75, 0x0105EC76, 0x12551F82, 0xE03E9C81, 0x34F4F86A, 0xC69F7B69, 0xD5CF889D, 0x27A40B9E, /* [0][0xf8]*/
        0x79B737BA, 0x8BDCB4B9, 0x988C474D, 0x6AE7C44E, 0xBE2DA0A5, 0x4C4623A6, 0x5F16D052, 0xAD7D5351  /* [0][0x100]*/
    },
    {
        0x00000000, 0x13A29877, 0x274530EE, 0x34E7A899, 0x4E8A61DC, 0x5D28F9AB, 0x69CF5132, 0x7A6DC945, /* [1][0x08]*/
        0x9D14C3B8, 0x8EB65BCF, 0xBA51F356, 0xA9F36B21, 0xD39EA264, 0xC03C3A13, 0xF4DB928A, 0xE7790AFD, /* [1][0x10]*/
        0x3FC5F181, 0x2C6769F6, 0x1880C16F, 0x0B225918, 0x714F905D, 0x62ED082A, 0x560AA0B3, 0x45A838C4, /* [1][0x18]*/
        0xA2D13239, 0xB173AA4E, 0x859402D7, 0x96369AA0, 0xEC5B53E5, 0xFFF9CB92, 0xCB1E630B, 0xD8BCFB7C, /* [1][0x20]*/
        0x7F8BE302, 0x6C297B75, 0x58CED3EC, 0x4B6C4B9B, 0x310182DE, 0x22A31AA9, 0x1644B230, 0x05E62A47, /* [1][0x28]*/
        0xE29F20BA, 0xF13DB8CD, 0xC5DA1054, 0xD6788823, 0xAC154166, 0xBFB7D911, 0x8B507188, 0x98F2E9FF, /* [1][0x30]*/
        0x404E1283, 0x53EC8AF4, 0x670B226D, 0x74A9BA1A, 0x0EC4735F, 0x1D66EB28, 0x298143B1, 0x3A23DBC6, /* [1][0x38]*/
        0xDD5AD13B, 0xCEF8494C, 0xFA1FE1D5, 0xE9BD79A2, 0x93D0B0E7, 0x80722890, 0xB4958009, 0xA737187E, /* [1][0x40]*/
        0xFF17C604, 0xECB55E73, 0xD852F6EA, 0xCBF06E9D, 0xB19DA7D8, 0xA23F3FAF, 0x96D89736, 0x857A0F41, /* [1][0x48]*/
        0x620305BC, 0x71A19DCB, 0x45463552, 0x56E4AD25, 0x2C896460, 0x3F2BFC17, 0x0BCC548E, 0x186ECCF9, /* [1][0x50]*/
        0xC0D23785, 0xD370AFF2, 0xE797076B, 0xF4359F1C, 0x8E585659, 0x9DFACE2E, 0xA91D66B7, 0xBABFFEC0, /* [1][0x58]*/
        0x5DC6F43D, 0x4E646C4A, 0x7A83C4D3, 0x69215CA4, 0x134C95E1, 0x00EE0D96, 0x3409A50F, 0x27AB3D78, /* [1][0x60]*/
        0x809C2506, 0x933EBD71, 0xA7D915E8, 0xB47B8D9F, 0xCE1644DA, 0xDDB4DCAD, 0xE9537434, 0xFAF1EC43, /* [1][0x68]*/
        0x1D88E6BE, 0x0E2A7EC9, 0x3ACDD650, 0x296F4E27, 0x53028762, 0x40A01F15, 0x7447B78C, 0x67E52FFB, /* [1][0x70]*/
        0xBF59D487, 0xACFB4CF0, 0x981CE469, 0x8BBE7C1E, 0xF1D3B55B, 0xE2712D2C, 0xD69685B5, 0xC5341DC2, /* [1][0x78]*/
        0x224D173F, 0x31EF8F48, 0x050827D1, 0x16AABFA6, 0x6CC776E3, 0x7F65EE94, 0x4B82460D, 0x5820DE7A, /* [1][0x80]*/
        0xFBC3FAF9, 0xE861628E, 0xDC86CA17, 0xCF245260, 0xB5499B25, 0xA6EB0352, 0x920CABCB, 0x81AE33BC, /* [1][0x88]*/
        0x66D73941, 0x7575A136, 0x419209AF, 0x523091D8, 0x285D589D, 0x3BFFC0EA, 0x0F186873, 0x1CBAF004, /* [1][0x90]*/
        0xC4060B78, 0xD7A4930F, 0xE3433B96, 0xF0E1A3E1, 0x8A8C6AA4, 0x992EF2D3, 0xADC95A4A, 0xBE6BC23D, /* [1][0x98]*/
        0x5912C8C0, 0x4AB050B7, 0x7E57F82E, 0x6DF56059, 0x1798A91C, 0x043A316B, 0x30DD99F2, 0x237F0185, /* [1][0xa0]*/
        0x844819FB, 0x97EA818C, 0xA30D2915, 0xB0AFB162, 0xCAC27827, 0xD960E050, 0xED8748C9, 0xFE25D0BE, /* [1][0xa8]*/
        0x195CDA43, 0x0AFE4234, 0x3E19EAAD, 0x2DBB72DA, 0x57D6BB9F, 0x447423E8, 0x70938B71, 0x63311306, /* [1][0xb0]*/
        0xBB8DE87A, 0xA82F700D, 0x9CC8D894, 0x8F6A40E3, 0xF50789A6, 0xE6A511D1, 0xD242B948, 0xC1E0213F, /* [1][0xb8]*/
        0x26992BC2, 0x353BB3B5, 0x01DC1B2C, 0x127E835B, 0x68134A1E, 0x7BB1D269, 0x4F567AF0, 0x5CF4E287, /* [1][0xc0]*/
        0x04D43CFD, 0x1776A48A, 0x23910C13, 0x30339464, 0x4A5E5D21, 0x59FCC556, 0x6D1B6DCF, 0x7EB9F5B8, /* [1][0xc8]*/
        0x99C0FF45, 0x8A626732, 0xBE85CFAB, 0xAD2757DC, 0xD74A9E99, 0xC4E806EE, 0xF00FAE77, 0xE3AD3600, /* [1][0xd0]*/
        0x3B11CD7C, 0x28B3550B, 0x1C54FD92, 0x0FF665E5, 0x759BACA0, 0x663934D7, 0x52DE9C4E, 0x417C0439, /* [1][0xd8]*/
        0xA6050EC4, 0xB5A796B3, 0x81403E2A, 0x92E2A65D, 0xE88F6F18, 0xFB2DF76F, 0xCFCA5FF6, 0xDC68C781, /* [1][0xe0]*/
        0x7B5FDFFF, 0x68FD4788, 0x5C1AEF11, 0x4FB87766, 0x35D5BE23, 0x26772654, 0x12908ECD, 0x013216BA, /* [1][0xe8]*/
        0xE64B1C47, 0xF5E98430, 0xC10E2CA9, 0xD2ACB4DE, 0xA8C17D9B, 0xBB63E5EC, 0x8F844D75, 0x9C26D502, /* [1][0xf0]*/
        0x449A2E7E, 0x5738B609, 0x63DF1E90, 0x707D86E7, 0x0A104FA2, 0x19B2D7D5, 0x2D557F4C, 0x3EF7E73B, /* [1][0xf8]*/
        0xD98EEDC6, 0xCA2C75B1, 0xFECBDD28, 0xED69455F, 0x97048C1A, 0x84A6146D, 0xB041BCF4, 0xA3E32483  /* [1][0x100]*/
    },
    {
        0x00000000, 0xA541927E, 0x4F6F520D, 0xEA2EC073, 0x9EDEA41A, 0x3B9F3664, 0xD1B1F617, 0x74F06469, /* [2][0x08]*/
        0x38513EC5, 0x9D10ACBB, 0x773E6CC8, 0xD27FFEB6, 0xA68F9ADF, 0x03CE08A1, 0xE9E0C8D2, 0x4CA15AAC, /* [2][0x10]*/
        0x70A27D8A, 0xD5E3EFF4, 0x3FCD2F87, 0x9A8CBDF9, 0xEE7CD990, 0x4B3D4BEE, 0xA1138B9D, 0x045219E3, /* [2][0x18]*/
        0x48F3434F, 0xEDB2D131, 0x079C1142, 0xA2DD833C, 0xD62DE755, 0x736C752B, 0x9942B558, 0x3C032726, /* [2][0x20]*/
        0xE144FB14, 0x4405696A, 0xAE2BA919, 0x0B6A3B67, 0x7F9A5F0E, 0xDADBCD70, 0x30F50D03, 0x95B49F7D, /* [2][0x28]*/
        0xD915C5D1, 0x7C5457AF, 0x967A97DC, 0x333B05A2, 0x47CB61CB, 0xE28AF3B5, 0x08A433C6, 0xADE5A1B8, /* [2][0x30]*/
        0x91E6869E, 0x34A714E0, 0xDE89D493, 0x7BC846ED, 0x0F382284, 0xAA79B0FA, 0x40577089, 0xE516E2F7, /* [2][0x38]*/
        0xA9B7B85B, 0x0CF62A25, 0xE6D8EA56, 0x43997828, 0x37691C41, 0x92288E3F, 0x78064E4C, 0xDD47DC32, /* [2][0x40]*/
        0xC76580D9, 0x622412A7, 0x880AD2D4, 0x2D4B40AA, 0x59BB24C3, 0xFCFAB6BD, 0x16D476CE, 0xB395E4B0, /* [2][0x48]*/
        0xFF34BE1C, 0x5A752C62, 0xB05BEC11, 0x151A7E6F, 0x61EA1A06, 0xC4AB8878, 0x2E85480B, 0x8BC4DA75, /* [2][0x50]*/
        0xB7C7FD53, 0x12866F2D, 0xF8A8AF5E, 0x5DE93D20, 0x29195949, 0x8C58CB37, 0x66760B44, 0xC337993A, /* [2][0x58]*/
        0x8F96C396, 0x2AD751E8, 0xC0F9919B, 0x65B803E5, 0x1148678C, 0xB409F5F2, 0x5E273581, 0xFB66A7FF, /* [2][0x60]*/
        0x26217BCD, 0x8360E9B3, 0x694E29C0, 0xCC0FBBBE, 0xB8FFDFD7, 0x1DBE4DA9, 0xF7908DDA, 0x52D11FA4, /* [2][0x68]*/
        0x1E704508, 0xBB31D776, 0x511F1705, 0xF45E857B, 0x80AEE112, 0x25EF736C, 0xCFC1B31F, 0x6A802161, /* [2][0x70]*/
        0x56830647, 0xF3C29439, 0x19EC544A, 0xBCADC634, 0xC85DA25D, 0x6D1C3023, 0x8732F050, 0x2273622E, /* [2][0x78]*/
        0x6ED23882, 0xCB93AAFC, 0x21BD6A8F, 0x84FCF8F1, 0xF00C9C98, 0x554D0EE6, 0xBF63CE95, 0x1A225CEB, /* [2][0x80]*/
        0x8B277743, 0x2E66E53D, 0xC448254E, 0x6109B730, 0x15F9D359, 0xB0B84127, 0x5A968154, 0xFFD7132A, /* [2][0x88]*/
        0xB3764986, 0x1637DBF8, 0xFC191B8B, 0x595889F5, 0x2DA8ED9C, 0x88E97FE2, 0x62C7BF91, 0xC7862DEF, /* [2][0x90]*/
        0xFB850AC9, 0x5EC498B7, 0xB4EA58C4, 0x11ABCABA, 0x655BAED3, 0xC01A3CAD, 0x2A34FCDE, 0x8F756EA0, /* [2][0x98]*/
        0xC3D4340C, 0x6695A672, 0x8CBB6601, 0x29FAF47F, 0x5D0A9016, 0xF84B0268, 0x1265C21B, 0xB7245065, /* [2][0xa0]*/
        0x6A638C57, 0xCF221E29, 0x250CDE5A, 0x804D4C24, 0xF4BD284D, 0x51FCBA33, 0xBBD27A40, 0x1E93E83E, /* [2][0xa8]*/
        0x5232B292, 0xF77320EC, 0x1D5DE09F, 0xB81C72E1, 0xCCEC1688, 0x69AD84F6, 0x83834485, 0x26C2D6FB, /* [2][0xb0]*/
        0x1AC1F1DD, 0xBF8063A3, 0x55AEA3D0, 0xF0EF31AE, 0x841F55C7, 0x215EC7B9, 0xCB7007CA, 0x6E3195B4, /* [2][0xb8]*/
        0x2290CF18, 0x87D15D66, 0x6DFF9D15, 0xC8BE0F6B, 0xBC4E6B02, 0x190FF97C, 0xF321390F, 0x5660AB71, /* [2][0xc0]*/
        0x4C42F79A, 0xE90365E4, 0x032DA597, 0xA66C37E9, 0xD29C5380, 0x77DDC1FE, 0x9DF3018D, 0x38B293F3, /* [2][0xc8]*/
        0x7413C95F, 0xD1525B21, 0x3B7C9B52, 0x9E3D092C, 0xEACD6D45, 0x4F8CFF3B, 0xA5A23F48, 0x00E3AD36, /* [2][0xd0]*/
        0x3CE08A10, 0x99A1186E, 0x738FD81D, 0xD6CE4A63, 0xA23E2E0A, 0x077FBC74, 0xED517C07, 0x4810EE79, /* [2][0xd8]*/
        0x04B1B4D5, 0xA1F026AB, 0x4BDEE6D8, 0xEE9F74A6, 0x9A6F10CF, 0x3F2E82B1, 0xD50042C2, 0x7041D0BC, /* [2][0xe0]*/
        0xAD060C8E, 0x08479EF0, 0xE2695E83, 0x4728CCFD, 0x33D8A894, 0x96993AEA, 0x7CB7FA99, 0xD9F668E7, /* [2][0xe8]*/
        0x9557324B, 0x3016A035, 0xDA386046, 0x7F79F238, 0x0B899651, 0xAEC8042F, 0x44E6C45C, 0xE1A75622, /* [2][0xf0]*/
        0xDDA47104, 0x78E5E37A, 0x92CB2309, 0x378AB177, 0x437AD51E, 0xE63B4760, 0x0C158713, 0xA954156D, /* [2][0xf8]*/
        0xE5F54FC1, 0x40B4DDBF, 0xAA9A1DCC, 0x0FDB8FB2, 0x7B2BEBDB, 0xDE6A79A5, 0x3444B9D6, 0x91052BA8  /* [2][0x100]*/
    },
    {
        0x00000000, 0xDD45AAB8, 0xBF672381, 0x62228939, 0x7B2231F3, 0xA6679B4B, 0xC4451272, 0x1900B8CA, /* [3][0x08]*/
        0xF64463E6, 0x2B01C95E, 0x49234067, 0x9466EADF, 0x8D665215, 0x5023F8AD, 0x32017194, 0xEF44DB2C, /* [3][0x10]*/
        0xE964B13D, 0x34211B85, 0x560392BC, 0x8B463804, 0x924680CE, 0x4F032A76, 0x2D21A34F, 0xF06409F7, /* [3][0x18]*/
        0x1F20D2DB, 0xC2657863, 0xA047F15A, 0x7D025BE2, 0x6402E328, 0xB9474990, 0xDB65C0A9, 0x06206A11, /* [3][0x20]*/
        0xD725148B, 0x0A60BE33, 0x6842370A, 0xB5079DB2, 0xAC072578, 0x71428FC0, 0x136006F9, 0xCE25AC41, /* [3][0x28]*/
        0x2161776D, 0xFC24DDD5, 0x9E0654EC, 0x4343FE54, 0x5A43469E, 0x8706EC26, 0xE524651F, 0x3861CFA7, /* [3][0x30]*/
        0x3E41A5B6, 0xE3040F0E, 0x81268637, 0x5C632C8F, 0x45639445, 0x98263EFD, 0xFA04B7C4, 0x27411D7C, /* [3][0x38]*/
        0xC805C650, 0x15406CE8, 0x7762E5D1, 0xAA274F69, 0xB327F7A3, 0x6E625D1B, 0x0C40D422, 0xD1057E9A, /* [3][0x40]*/
        0xABA65FE7, 0x76E3F55F, 0x14C17C66, 0xC984D6DE, 0xD0846E14, 0x0DC1C4AC, 0x6FE34D95, 0xB2A6E72D, /* [3][0x48]*/
        0x5DE23C01, 0x80A796B9, 0xE2851F80, 0x3FC0B538, 0x26C00DF2, 0xFB85A74A, 0x99A72E73, 0x44E284CB, /* [3][0x50]*/
        0x42C2EEDA, 0x9F874462, 0xFDA5CD5B, 0x20E067E3, 0x39E0DF29, 0xE4A57591, 0x8687FCA8, 0x5BC25610, /* [3][0x58]*/
        0xB4868D3C, 0x69C32784, 0x0BE1AEBD, 0xD6A40405, 0xCFA4BCCF, 0x12E11677, 0x70C39F4E, 0xAD8635F6, /* [3][0x60]*/
        0x7C834B6C, 0xA1C6E1D4, 0xC3E468ED, 0x1EA1C255, 0x07A17A9F, 0xDAE4D027, 0xB8C6591E, 0x6583F3A6, /* [3][0x68]*/
        0x8AC7288A, 0x57828232, 0x35A00B0B, 0xE8E5A1B3, 0xF1E51979, 0x2CA0B3C1, 0x4E823AF8, 0x93C79040, /* [3][0x70]*/
        0x95E7FA51, 0x48A250E9, 0x2A80D9D0, 0xF7C57368, 0xEEC5CBA2, 0x3380611A, 0x51A2E823, 0x8CE7429B, /* [3][0x78]*/
        0x63A399B7, 0xBEE6330F, 0xDCC4BA36, 0x0181108E, 0x1881A844, 0xC5C402FC, 0xA7E68BC5, 0x7AA3217D, /* [3][0x80]*/
        0x52A0C93F, 0x8FE56387, 0xEDC7EABE, 0x30824006, 0x2982F8CC, 0xF4C75274, 0x96E5DB4D, 0x4BA071F5, /* [3][0x88]*/
        0xA4E4AAD9, 0x79A10061, 0x1B838958, 0xC6C623E0, 0xDFC69B2A, 0x02833192, 0x60A1B8AB, 0xBDE41213, /* [3][0x90]*/
        0xBBC47802, 0x6681D2BA, 0x04A35B83, 0xD9E6F13B, 0xC0E649F1, 0x1DA3E349, 0x7F816A70, 0xA2C4C0C8, /* [3][0x98]*/
        0x4D801BE4, 0x90C5B15C, 0xF2E73865, 0x2FA292DD, 0x36A22A17, 0xEBE780AF, 0x89C50996, 0x5480A32E, /* [3][0xa0]*/
        0x8585DDB4, 0x58C0770C, 0x3AE2FE35, 0xE7A7548D, 0xFEA7EC47, 0x23E246FF, 0x41C0CFC6, 0x9C85657E, /* [3][0xa8]*/
        0x73C1BE52, 0xAE8414EA, 0xCCA69DD3, 0x11E3376B, 0x08E38FA1, 0xD5A62519, 0xB784AC20, 0x6AC10698, /* [3][0xb0]*/
        0x6CE16C89, 0xB1A4C631, 0xD3864F08, 0x0EC3E5B0, 0x17C35D7A, 0xCA86F7C2, 0xA8A47EFB, 0x75E1D443, /* [3][0xb8]*/
        0x9AA50F6F, 0x47E0A5D7, 0x25C22CEE, 0xF8878656, 0xE1873E9C, 0x3CC29424, 0x5EE01D1D, 0x83A5B7A5, /* [3][0xc0]*/
        0xF90696D8, 0x24433C60, 0x4661B559, 0x9B241FE1, 0x8224A72B, 0x5F610D93, 0x3D4384AA, 0xE0062E12, /* [3][0xc8]*/
        0x0F42F53E, 0xD2075F86, 0xB025D6BF, 0x6D607C07, 0x7460C4CD, 0xA9256E75, 0xCB07E74C, 0x16424DF4, /* [3][0xd0]*/
        0x106227E5, 0xCD278D5D, 0xAF050464, 0x7240AEDC, 0x6B401616, 0xB605BCAE, 0xD4273597, 0x09629F2F, /* [3][0xd8]*/
        0xE6264403, 0x3B63EEBB, 0x59416782, 0x8404CD3A, 0x9D0475F0, 0x4041DF48, 0x22635671, 0xFF26FCC9, /* [3][0xe0]*/
        0x2E238253, 0xF36628EB, 0x9144A1D2, 0x4C010B6A, 0x5501B3A0, 0x88441918, 0xEA669021, 0x37233A99, /* [3][0xe8]*/
        0xD867E1B5, 0x05224B0D, 0x6700C234, 0xBA45688C, 0xA345D046, 0x7E007AFE, 0x1C22F3C7, 0xC167597F, /* [3][0xf0]*/
        0xC747336E, 0x1A0299D6, 0x782010EF, 0xA565BA57, 0xBC65029D, 0x6120A825, 0x0302211C, 0xDE478BA4, /* [3][0xf8]*/
        0x31035088, 0xEC46FA30, 0x8E647309, 0x5321D9B1, 0x4A21617B, 0x9764CBC3, 0xF54642FA, 0x2803E842  /* [3][0x100]*/
    },
    {
        0x00000000, 0x38116FAC, 0x7022DF58, 0x4833B0F4, 0xE045BEB0, 0xD854D11C, 0x906761E8, 0xA8760E44, /* [4][0x08]*/
        0xC5670B91, 0xFD76643D, 0xB545D4C9, 0x8D54BB65, 0x2522B521, 0x1D33DA8D, 0x55006A79, 0x6D1105D5, /* [4][0x10]*/
        0x8F2261D3, 0xB7330E7F, 0xFF00BE8B, 0xC711D127, 0x6F67DF63, 0x5776B0CF, 0x1F45003B, 0x27546F97, /* [4][0x18]*/
        0x4A456A42, 0x725405EE, 0x3A67B51A, 0x0276DAB6, 0xAA00D4F2, 0x9211BB5E, 0xDA220BAA, 0xE2336406, /* [4][0x20]*/
        0x1BA8B557, 0x23B9DAFB, 0x6B8A6A0F, 0x539B05A3, 0xFBED0BE7, 0xC3FC644B, 0x8BCFD4BF, 0xB3DEBB13, /* [4][0x28]*/
        0xDECFBEC6, 0xE6DED16A, 0xAEED619E, 0x96FC0E32, 0x3E8A0076, 0x069B6FDA, 0x4EA8DF2E, 0x76B9B082, /* [4][0x30]*/
        0x948AD484, 0xAC9BBB28, 0xE4A80BDC, 0xDCB96470, 0x74CF6A34, 0x4CDE0598, 0x04EDB56C, 0x3CFCDAC0, /* [4][0x38]*/
        0x51EDDF15, 0x69FCB0B9, 0x21CF004D, 0x19DE6FE1, 0xB1A861A5, 0x89B90E09, 0xC18ABEFD, 0xF99BD151, /* [4][0x40]*/
        0x37516AAE, 0x0F400502, 0x4773B5F6, 0x7F62DA5A, 0xD714D41E, 0xEF05BBB2, 0xA7360B46, 0x9F2764EA, /* [4][0x48]*/
        0xF236613F, 0xCA270E93, 0x8214BE67, 0xBA05D1CB, 0x1273DF8F, 0x2A62B023, 0x625100D7, 0x5A406F7B, /* [4][0x50]*/
        0xB8730B7D, 0x806264D1, 0xC851D425, 0xF040BB89, 0x5836B5CD, 0x6027DA61, 0x28146A95, 0x10050539, /* [4][0x58]*/
        0x7D1400EC, 0x45056F40, 0x0D36DFB4, 0x3527B018, 0x9D51BE5C, 0xA540D1F0, 0xED736104, 0xD5620EA8, /* [4][0x60]*/
        0x2CF9DFF9, 0x14E8B055, 0x5CDB00A1, 0x64CA6F0D, 0xCCBC6149, 0xF4AD0EE5, 0xBC9EBE11, 0x848FD1BD, /* [4][0x68]*/
        0xE99ED468, 0xD18FBBC4, 0x99BC0B30, 0xA1AD649C, 0x09DB6AD8, 0x31CA0574, 0x79F9B580, 0x41E8DA2C, /* [4][0x70]*/
        0xA3DBBE2A, 0x9BCAD186, 0xD3F96172, 0xEBE80EDE, 0x439E009A, 0x7B8F6F36, 0x33BCDFC2, 0x0BADB06E, /* [4][0x78]*/
        0x66BCB5BB, 0x5EADDA17, 0x169E6AE3, 0x2E8F054F, 0x86F90B0B, 0xBEE864A7, 0xF6DBD453, 0xCECABBFF, /* [4][0x80]*/
        0x6EA2D55C, 0x56B3BAF0, 0x1E800A04, 0x269165A8, 0x8EE76BEC, 0xB6F60440, 0xFEC5B4B4, 0xC6D4DB18, /* [4][0x88]*/
        0xABC5DECD, 0x93D4B161, 0xDBE70195, 0xE3F66E39, 0x4B80607D, 0x73910FD1, 0x3BA2BF25, 0x03B3D089, /* [4][0x90]*/
        0xE180B48F, 0xD991DB23, 0x91A26BD7, 0xA9B3047B, 0x01C50A3F, 0x39D46593, 0x71E7D567, 0x49F6BACB, /* [4][0x98]*/
        0x24E7BF1E, 0x1CF6D0B2, 0x54C56046, 0x6CD40FEA, 0xC4A201AE, 0xFCB36E02, 0xB480DEF6, 0x8C91B15A, /* [4][0xa0]*/
        0x750A600B, 0x4D1B0FA7, 0x0528BF53, 0x3D39D0FF, 0x954FDEBB, 0xAD5EB117, 0xE56D01E3, 0xDD7C6E4F, /* [4][0xa8]*/
        0xB06D6B9A, 0x887C0436, 0xC04FB4C2, 0xF85EDB6E, 0x5028D52A, 0x6839BA86, 0x200A0A72, 0x181B65DE, /* [4][0xb0]*/
        0xFA2801D8, 0xC2396E74, 0x8A0ADE80, 0xB21BB12C, 0x1A6DBF68, 0x227CD0C4, 0x6A4F6030, 0x525E0F9C, /* [4][0xb8]*/
        0x3F4F0A49, 0x075E65E5, 0x4F6DD511, 0x777CBABD, 0xDF0AB4F9, 0xE71BDB55, 0xAF286BA1, 0x9739040D, /* [4][0xc0]*/
        0x59F3BFF2, 0x61E2D05E, 0x29D160AA, 0x11C00F06, 0xB9B60142, 0x81A76EEE, 0xC994DE1A, 0xF185B1B6, /* [4][0xc8]*/
        0x9C94B463, 0xA485DBCF, 0xECB66B3B, 0xD4A70497, 0x7CD10AD3, 0x44C0657F, 0x0CF3D58B, 0x34E2BA27, /* [4][0xd0]*/
        0xD6D1DE21, 0xEEC0B18D, 0xA6F30179, 0x9EE26ED5, 0x36946091, 0x0E850F3D, 0x46B6BFC9, 0x7EA7D065, /* [4][0xd8]*/
        0x13B6D5B0, 0x2BA7BA1C, 0x63940AE8, 0x5B856544, 0xF3F36B00, 0xCBE204AC, 0x83D1B458, 0xBBC0DBF4, /* [4][0xe0]*/
        0x425B0AA5, 0x7A4A6509, 0x3279D5FD, 0x0A68BA51, 0xA21EB415, 0x9A0FDBB9, 0xD23C6B4D, 0xEA2D04E1, /* [4][0xe8]*/
        0x873C0134, 0xBF2D6E98, 0xF71EDE6C, 0xCF0FB1C0, 0x6779BF84, 0x5F68D028, 0x175B60DC, 0x2F4A0F70, /* [4][0xf0]*/
        0xCD796B76, 0xF56804DA, 0xBD5BB42E, 0x854ADB82, 0x2D3CD5C6, 0x152DBA6A, 0x5D1E0A9E, 0x650F6532, /* [4][0xf8]*/
        0x081E60E7, 0x300F0F4B, 0x783CBFBF, 0x402DD013, 0xE85BDE57, 0xD04AB1FB, 0x9879010F, 0xA0686EA3  /* [4][0x100]*/
    },
    {
        0x00000000, 0xEF306B19, 0xDB8CA0C3, 0x34BCCBDA, 0xB2F53777, 0x5DC55C6E, 0x697997B4, 0x8649FCAD, /* [5][0x08]*/
        0x6006181F, 0x8F367306, 0xBB8AB8DC, 0x54BAD3C5, 0xD2F32F68, 0x3DC34471, 0x097F8FAB, 0xE64FE4B2, /* [5][0x10]*/
        0xC00C303E, 0x2F3C5B27, 0x1B8090FD, 0xF4B0FBE4, 0x72F90749, 0x9DC96C50, 0xA975A78A, 0x4645CC93, /* [5][0x18]*/
        0xA00A2821, 0x4F3A4338, 0x7B8688E2, 0x94B6E3FB, 0x12FF1F56, 0xFDCF744F, 0xC973BF95, 0x2643D48C, /* [5][0x20]*/
        0x85F4168D, 0x6AC47D94, 0x5E78B64E, 0xB148DD57, 0x370121FA, 0xD8314AE3, 0xEC8D8139, 0x03BDEA20, /* [5][0x28]*/
        0xE5F20E92, 0x0AC2658B, 0x3E7EAE51, 0xD14EC548, 0x570739E5, 0xB83752FC, 0x8C8B9926, 0x63BBF23F, /* [5][0x30]*/
        0x45F826B3, 0xAAC84DAA, 0x9E748670, 0x7144ED69, 0xF70D11C4, 0x183D7ADD, 0x2C81B107, 0xC3B1DA1E, /* [5][0x38]*/
        0x25FE3EAC, 0xCACE55B5, 0xFE729E6F, 0x1142F576, 0x970B09DB, 0x783B62C2, 0x4C87A918, 0xA3B7C201, /* [5][0x40]*/
        0x0E045BEB, 0xE13430F2, 0xD588FB28, 0x3AB89031, 0xBCF16C9C, 0x53C10785, 0x677DCC5F, 0x884DA746, /* [5][0x48]*/
        0x6E0243F4, 0x813228ED, 0xB58EE337, 0x5ABE882E, 0xDCF77483, 0x33C71F9A, 0x077BD440, 0xE84BBF59, /* [5][0x50]*/
        0xCE086BD5, 0x213800CC, 0x1584CB16, 0xFAB4A00F, 0x7CFD5CA2, 0x93CD37BB, 0xA771FC61, 0x48419778, /* [5][0x58]*/
        0xAE0E73CA, 0x413E18D3, 0x7582D309, 0x9AB2B810, 0x1CFB44BD, 0xF3CB2FA4, 0xC777E47E, 0x28478F67, /* [5][0x60]*/
        0x8BF04D66, 0x64C0267F, 0x507CEDA5, 0xBF4C86BC, 0x39057A11, 0xD6351108, 0xE289DAD2, 0x0DB9B1CB, /* [5][0x68]*/
        0xEBF65579, 0x04C63E60, 0x307AF5BA, 0xDF4A9EA3, 0x5903620E, 0xB6330917, 0x828FC2CD, 0x6DBFA9D4, /* [5][0x70]*/
        0x4BFC7D58, 0xA4CC1641, 0x9070DD9B, 0x7F40B682, 0xF9094A2F, 0x16392136, 0x2285EAEC, 0xCDB581F5, /* [5][0x78]*/
        0x2BFA6547, 0xC4CA0E5E, 0xF076C584, 0x1F46AE9D, 0x990F5230, 0x763F3929, 0x4283F2F3, 0xADB399EA, /* [5][0x80]*/
        0x1C08B7D6, 0xF338DCCF, 0xC7841715, 0x28B47C0C, 0xAEFD80A1, 0x41CDEBB8, 0x75712062, 0x9A414B7B, /* [5][0x88]*/
        0x7C0EAFC9, 0x933EC4D0, 0xA7820F0A, 0x48B26413, 0xCEFB98BE, 0x21CBF3A7, 0x1577387D, 0xFA475364, /* [5][0x90]*/
        0xDC0487E8, 0x3334ECF1, 0x0788272B, 0xE8B84C32, 0x6EF1B09F, 0x81C1DB86, 0xB57D105C, 0x5A4D7B45, /* [5][0x98]*/
        0xBC029FF7, 0x5332F4EE, 0x678E3F34, 0x88BE542D, 0x0EF7A880, 0xE1C7C399, 0xD57B0843, 0x3A4B635A, /* [5][0xa0]*/
        0x99FCA15B, 0x76CCCA42, 0x42700198, 0xAD406A81, 0x2B09962C, 0xC439FD35, 0xF08536EF, 0x1FB55DF6, /* [5][0xa8]*/
        0xF9FAB944, 0x16CAD25D, 0x22761987, 0xCD46729E, 0x4B0F8E33, 0xA43FE52A, 0x90832EF0, 0x7FB345E9, /* [5][0xb0]*/
        0x59F09165, 0xB6C0FA7C, 0x827C31A6, 0x6D4C5ABF, 0xEB05A612, 0x0435CD0B, 0x308906D1, 0xDFB96DC8, /* [5][0xb8]*/
        0x39F6897A, 0xD6C6E263, 0xE27A29B9, 0x0D4A42A0, 0x8B03BE0D, 0x6433D514, 0x508F1ECE, 0xBFBF75D7, /* [5][0xc0]*/
        0x120CEC3D, 0xFD3C8724, 0xC9804CFE, 0x26B027E7, 0xA0F9DB4A, 0x4FC9B053, 0x7B757B89, 0x94451090, /* [5][0xc8]*/
        0x720AF422, 0x9D3A9F3B, 0xA98654E1, 0x46B63FF8, 0xC0FFC355, 0x2FCFA84C, 0x1B736396, 0xF443088F, /* [5][0xd0]*/
        0xD200DC03, 0x3D30B71A, 0x098C7CC0, 0xE6BC17D9, 0x60F5EB74, 0x8FC5806D, 0xBB794BB7, 0x544920AE, /* [5][0xd8]*/
        0xB206C41C, 0x5D36AF05, 0x698A64DF, 0x86BA0FC6, 0x00F3F36B, 0xEFC39872, 0xDB7F53A8, 0x344F38B1, /* [5][0xe0]*/
        0x97F8FAB0, 0x78C891A9, 0x4C745A73, 0xA344316A, 0x250DCDC7, 0xCA3DA6DE, 0xFE816D04, 0x11B1061D, /* [5][0xe8]*/
        0xF7FEE2AF, 0x18CE89B6, 0x2C72426C, 0xC3422975, 0x450BD5D8, 0xAA3BBEC1, 0x9E87751B, 0x71B71E02, /* [5][0xf0]*/
        0x57F4CA8E, 0xB8C4A197, 0x8C786A4D, 0x63480154, 0xE501FDF9, 0x0A3196E0, 0x3E8D5D3A, 0xD1BD3623, /* [5][0xf8]*/
        0x37F2D291, 0xD8C2B988, 0xEC7E7252, 0x034E194B, 0x8507E5E6, 0x6A378EFF, 0x5E8B4525, 0xB1BB2E3C  /* [5][0x100]*/
    },
    {
        0x00000000, 0x68032CC8, 0xD0065990, 0xB8057558, 0xA5E0C5D1, 0xCDE3E919, 0x75E69C41, 0x1DE5B089, /* [6][0x08]*/
        0x4E2DFD53, 0x262ED19B, 0x9E2BA4C3, 0xF628880B, 0xEBCD3882, 0x83CE144A, 0x3BCB6112, 0x53C84DDA, /* [6][0x10]*/
        0x9C5BFAA6, 0xF458D66E, 0x4C5DA336, 0x245E8FFE, 0x39BB3F77, 0x51B813BF, 0xE9BD66E7, 0x81BE4A2F, /* [6][0x18]*/
        0xD27607F5, 0xBA752B3D, 0x02705E65, 0x6A7372AD, 0x7796C224, 0x1F95EEEC, 0xA7909BB4, 0xCF93B77C, /* [6][0x20]*/
        0x3D5B83BD, 0x5558AF75, 0xED5DDA2D, 0x855EF6E5, 0x98BB466C, 0xF0B86AA4, 0x48BD1FFC, 0x20BE3334, /* [6][0x28]*/
        0x73767EEE, 0x1B755226, 0xA370277E, 0xCB730BB6, 0xD696BB3F, 0xBE9597F7, 0x0690E2AF, 0x6E93CE67, /* [6][0x30]*/
        0xA100791B, 0xC90355D3, 0x7106208B, 0x19050C43, 0x04E0BCCA, 0x6CE39002, 0xD4E6E55A, 0xBCE5C992, /* [6][0x38]*/
        0xEF2D8448, 0x872EA880, 0x3F2BDDD8, 0x5728F110, 0x4ACD4199, 0x22CE6D51, 0x9ACB1809, 0xF2C834C1, /* [6][0x40]*/
        0x7AB7077A, 0x12B42BB2, 0xAAB15EEA, 0xC2B27222, 0xDF57C2AB, 0xB754EE63, 0x0F519B3B, 0x6752B7F3, /* [6][0x48]*/
        0x349AFA29, 0x5C99D6E1, 0xE49CA3B9, 0x8C9F8F71, 0x917A3FF8, 0xF9791330, 0x417C6668, 0x297F4AA0, /* [6][0x50]*/
        0xE6ECFDDC, 0x8EEFD114, 0x36EAA44C, 0x5EE98884, 0x430C380D, 0x2B0F14C5, 0x930A619D, 0xFB094D55, /* [6][0x58]*/
        0xA8C1008F, 0xC0C22C47, 0x78C7591F, 0x10C475D7, 0x0D21C55E, 0x6522E996, 0xDD279CCE, 0xB524B006, /* [6][0x60]*/
        0x47EC84C7, 0x2FEFA80F, 0x97EADD57, 0xFFE9F19F, 0xE20C4116, 0x8A0F6DDE, 0x320A1886, 0x5A09344E, /* [6][0x68]*/
        0x09C17994, 0x61C2555C, 0xD9C72004, 0xB1C40CCC, 0xAC21BC45, 0xC422908D, 0x7C27E5D5, 0x1424C91D, /* [6][0x70]*/
        0xDBB77E61, 0xB3B452A9, 0x0BB127F1, 0x63B20B39, 0x7E57BBB0, 0x16549778, 0xAE51E220, 0xC652CEE8, /* [6][0x78]*/
        0x959A8332, 0xFD99AFFA, 0x459CDAA2, 0x2D9FF66A, 0x307A46E3, 0x58796A2B, 0xE07C1F73, 0x887F33BB, /* [6][0x80]*/
        0xF56E0EF4, 0x9D6D223C, 0x25685764, 0x4D6B7BAC, 0x508ECB25, 0x388DE7ED, 0x808892B5, 0xE88BBE7D, /* [6][0x88]*/
        0xBB43F3A7, 0xD340DF6F, 0x6B45AA37, 0x034686FF, 0x1EA33676, 0x76A01ABE, 0xCEA56FE6, 0xA6A6432E, /* [6][0x90]*/
        0x6935F452, 0x0136D89A, 0xB933ADC2, 0xD130810A, 0xCCD53183, 0xA4D61D4B, 0x1CD36813, 0x74D044DB, /* [6][0x98]*/
        0x27180901, 0x4F1B25C9, 0xF71E5091, 0x9F1D7C59, 0x82F8CCD0, 0xEAFBE018, 0x52FE9540, 0x3AFDB988, /* [6][0xa0]*/
        0xC8358D49, 0xA036A181, 0x1833D4D9, 0x7030F811, 0x6DD54898, 0x05D66450, 0xBDD31108, 0xD5D03DC0, /* [6][0xa8]*/
        0x8618701A, 0xEE1B5CD2, 0x561E298A, 0x3E1D0542, 0x23F8B5CB, 0x4BFB9903, 0xF3FEEC5B, 0x9BFDC093, /* [6][0xb0]*/
        0x546E77EF, 0x3C6D5B27, 0x84682E7F, 0xEC6B02B7, 0xF18EB23E, 0x998D9EF6, 0x2188EBAE, 0x498BC766, /* [6][0xb8]*/
        0x1A438ABC, 0x7240A674, 0xCA45D32C, 0xA246FFE4, 0xBFA34F6D, 0xD7A063A5, 0x6FA516FD, 0x07A63A35, /* [6][0xc0]*/
        0x8FD9098E, 0xE7DA2546, 0x5FDF501E, 0x37DC7CD6, 0x2A39CC5F, 0x423AE097, 0xFA3F95CF, 0x923CB907, /* [6][0xc8]*/
        0xC1F4F4DD, 0xA9F7D815, 0x11F2AD4D, 0x79F18185, 0x6414310C, 0x0C171DC4, 0xB412689C, 0xDC114454, /* [6][0xd0]*/
        0x1382F328, 0x7B81DFE0, 0xC384AAB8, 0xAB878670, 0xB66236F9, 0xDE611A31, 0x66646F69, 0x0E6743A1, /* [6][0xd8]*/
        0x5DAF0E7B, 0x35AC22B3, 0x8DA957EB, 0xE5AA7B23, 0xF84FCBAA, 0x904CE762, 0x2849923A, 0x404ABEF2, /* [6][0xe0]*/
        0xB2828A33, 0xDA81A6FB, 0x6284D3A3, 0x0A87FF6B, 0x17624FE2, 0x7F61632A, 0xC7641672, 0xAF673ABA, /* [6][0xe8]*/
        0xFCAF7760, 0x94AC5BA8, 0x2CA92EF0, 0x44AA0238, 0x594FB2B1, 0x314C9E79, 0x8949EB21, 0xE14AC7E9, /* [6][0xf0]*/
        0x2ED97095, 0x46DA5C5D, 0xFEDF2905, 0x96DC05CD, 0x8B39B544, 0xE33A998C, 0x5B3FECD4, 0x333CC01C, /* [6][0xf8]*/
        0x60F48DC6, 0x08F7A10E, 0xB0F2D456, 0xD8F1F89E, 0xC5144817, 0xAD1764DF, 0x15121187, 0x7D113D4F  /* [6][0x100]*/
    },
    {
        0x00000000, 0x493C7D27, 0x9278FA4E, 0xDB448769, 0x211D826D, 0x6821FF4A, 0xB3657823, 0xFA590504, /* [7][0x08]*/
        0x423B04DA, 0x0B0779FD, 0xD043FE94, 0x997F83B3, 0x632686B7, 0x2A1AFB90, 0xF15E7CF9, 0xB86201DE, /* [7][0x10]*/
        0x847609B4, 0xCD4A7493, 0x160EF3FA, 0x5F328EDD, 0xA56B8BD9, 0xEC57F6FE, 0x37137197, 0x7E2F0CB0, /* [7][0x18]*/
        0xC64D0D6E, 0x8F717049, 0x5435F720, 0x1D098A07, 0xE7508F03, 0xAE6CF224, 0x7528754D, 0x3C14086A, /* [7][0x20]*/
        0x0D006599, 0x443C18BE, 0x9F789FD7, 0xD644E2F0, 0x2C1DE7F4, 0x65219AD3, 0xBE651DBA, 0xF759609D, /* [7][0x28]*/
        0x4F3B6143, 0x06071C64, 0xDD439B0D, 0x947FE62A, 0x6E26E32E, 0x271A9E09, 0xFC5E1960, 0xB5626447, /* [7][0x30]*/
        0x89766C2D, 0xC04A110A, 0x1B0E9663, 0x5232EB44, 0xA86BEE40, 0xE1579367, 0x3A13140E, 0x732F6929, /* [7][0x38]*/
        0xCB4D68F7, 0x827115D0, 0x593592B9, 0x1009EF9E, 0xEA50EA9A, 0xA36C97BD, 0x782810D4, 0x31146DF3, /* [7][0x40]*/
        0x1A00CB32, 0x533CB615, 0x8878317C, 0xC1444C5B, 0x3B1D495F, 0x72213478, 0xA965B311, 0xE059CE36, /* [7][0x48]*/
        0x583BCFE8, 0x1107B2CF, 0xCA4335A6, 0x837F4881, 0x79264D85, 0x301A30A2, 0xEB5EB7CB, 0xA262CAEC, /* [7][0x50]*/
        0x9E76C286, 0xD74ABFA1, 0x0C0E38C8, 0x453245EF, 0xBF6B40EB, 0xF6573DCC, 0x2D13BAA5, 0x642FC782, /* [7][0x58]*/
        0xDC4DC65C, 0x9571BB7B, 0x4E353C12, 0x07094135, 0xFD504431, 0xB46C3916, 0x6F28BE7F, 0x2614C358, /* [7][0x60]*/
        0x1700AEAB, 0x5E3CD38C, 0x857854E5, 0xCC4429C2, 0x361D2CC6, 0x7F2151E1, 0xA465D688, 0xED59ABAF, /* [7][0x68]*/
        0x553BAA71, 0x1C07D756, 0xC743503F, 0x8E7F2D18, 0x7426281C, 0x3D1A553B, 0xE65ED252, 0xAF62AF75, /* [7][0x70]*/
        0x9376A71F, 0xDA4ADA38, 0x010E5D51, 0x48322076, 0xB26B2572, 0xFB575855, 0x2013DF3C, 0x692FA21B, /* [7][0x78]*/
        0xD14DA3C5, 0x9871DEE2, 0x4335598B, 0x0A0924AC, 0xF05021A8, 0xB96C5C8F, 0x6228DBE6, 0x2B14A6C1, /* [7][0x80]*/
        0x34019664, 0x7D3DEB43, 0xA6796C2A, 0xEF45110D, 0x151C1409, 0x5C20692E, 0x8764EE47, 0xCE589360, /* [7][0x88]*/
        0x763A92BE, 0x3F06EF99, 0xE44268F0, 0xAD7E15D7, 0x572710D3, 0x1E1B6DF4, 0xC55FEA9D, 0x8C6397BA, /* [7][0x90]*/
        0xB0779FD0, 0xF94BE2F7, 0x220F659E, 0x6B3318B9, 0x916A1DBD, 0xD856609A, 0x0312E7F3, 0x4A2E9AD4, /* [7][0x98]*/
        0xF24C9B0A, 0xBB70E62D, 0x60346144, 0x29081C63, 0xD3511967, 0x9A6D6440, 0x4129E329, 0x08159E0E, /* [7][0xa0]*/
        0x3901F3FD, 0x703D8EDA, 0xAB7909B3, 0xE2457494, 0x181C7190, 0x51200CB7, 0x8A648BDE, 0xC358F6F9, /* [7][0xa8]*/
        0x7B3AF727, 0x32068A00, 0xE9420D69, 0xA07E704E, 0x5A27754A, 0x131B086D, 0xC85F8F04, 0x8163F223, /* [7][0xb0]*/
        0xBD77FA49, 0xF44B876E, 0x2F0F0007, 0x66337D20, 0x9C6A7824, 0xD5560503, 0x0E12826A, 0x472EFF4D, /* [7][0xb8]*/
        0xFF4CFE93, 0xB67083B4, 0x6D3404DD, 0x240879FA, 0xDE517CFE, 0x976D01D9, 0x4C2986B0, 0x0515FB97, /* [7][0xc0]*/
        0x2E015D56, 0x673D2071, 0xBC79A718, 0xF545DA3F, 0x0F1CDF3B, 0x4620A21C, 0x9D642575, 0xD4585852, /* [7][0xc8]*/
        0x6C3A598C, 0x250624AB, 0xFE42A3C2, 0xB77EDEE5, 0x4D27DBE1, 0x041BA6C6, 0xDF5F21AF, 0x96635C88, /* [7][0xd0]*/
        0xAA7754E2, 0xE34B29C5, 0x380FAEAC, 0x7133D38B, 0x8B6AD68F, 0xC256ABA8, 0x19122CC1, 0x502E51E6, /* [7][0xd8]*/
        0xE84C5038, 0xA1702D1F, 0x7A34AA76, 0x3308D751, 0xC951D255, 0x806DAF72, 0x5B29281B, 0x1215553C, /* [7][0xe0]*/
        0x230138CF, 0x6A3D45E8, 0xB179C281, 0xF845BFA6, 0x021CBAA2, 0x4B20C785, 0x906440EC, 0xD9583DCB, /* [7][0xe8]*/
        0x613A3C15, 0x28064132, 0xF342C65B, 0xBA7EBB7C, 0x4027BE78, 0x091BC35F, 0xD25F4436, 0x9B633911, /* [7][0xf0]*/
        0xA777317B, 0xEE4B4C5C, 0x350FCB35, 0x7C33B612, 0x866AB316, 0xCF56CE31, 0x14124958, 0x5D2E347F, /* [7][0xf8]*/
        0xE54C35A1, 0xAC704886, 0x7734CFEF, 0x3E08B2C8, 0xC451B7CC, 0x8D6DCAEB, 0x56294D82, 0x1F1530A5  /* [7][0x100]*/
    },
    {
        0x00000000, 0xF43ED648, 0xED91DA61, 0x19AF0C29, 0xDECFC233, 0x2AF1147B, 0x335E1852, 0xC760CE1A, /* [8][0x08]*/
        0xB873F297, 0x4C4D24DF, 0x55E228F6, 0xA1DCFEBE, 0x66BC30A4, 0x9282E6EC, 0x8B2DEAC5, 0x7F133C8D, /* [8][0x10]*/
        0x750B93DF, 0x81354597, 0x989A49BE, 0x6CA49FF6, 0xABC451EC, 0x5FFA87A4, 0x46558B8D, 0xB26B5DC5, /* [8][0x18]*/
        0xCD786148, 0x3946B700, 0x20E9BB29, 0xD4D76D61, 0x13B7A37B, 0xE7897533, 0xFE26791A, 0x0A18AF52, /* [8][0x20]*/
        0xEA1727BE, 0x1E29F1F6, 0x0786FDDF, 0xF3B82B97, 0x34D8E58D, 0xC0E633C5, 0xD9493FEC, 0x2D77E9A4, /* [8][0x28]*/
        0x5264D529, 0xA65A0361, 0xBFF50F48, 0x4BCBD900, 0x8CAB171A, 0x7895C152, 0x613ACD7B, 0x95041B33, /* [8][0x30]*/
        0x9F1CB461, 0x6B226229, 0x728D6E00, 0x86B3B848, 0x41D37652, 0xB5EDA01A, 0xAC42AC33, 0x587C7A7B, /* [8][0x38]*/
        0x276F46F6, 0xD35190BE, 0xCAFE9C97, 0x3EC04ADF, 0xF9A084C5, 0x0D9E528D, 0x14315EA4, 0xE00F88EC, /* [8][0x40]*/
        0xD1C2398D, 0x25FCEFC5, 0x3C53E3EC, 0xC86D35A4, 0x0F0DFBBE, 0xFB332DF6, 0xE29C21DF, 0x16A2F797, /* [8][0x48]*/
        0x69B1CB1A, 0x9D8F1D52, 0x8420117B, 0x701EC733, 0xB77E0929, 0x4340DF61, 0x5AEFD348, 0xAED10500, /* [8][0x50]*/
        0xA4C9AA52, 0x50F77C1A, 0x49587033, 0xBD66A67B, 0x7A066861, 0x8E38BE29, 0x9797B200, 0x63A96448, /* [8][0x58]*/
        0x1CBA58C5, 0xE8848E8D, 0xF12B82A4, 0x051554EC, 0xC2759AF6, 0x364B4CBE, 0x2FE44097, 0xDBDA96DF, /* [8][0x60]*/
        0x3BD51E33, 0xCFEBC87B, 0xD644C452, 0x227A121A, 0xE51ADC00, 0x11240A48, 0x088B0661, 0xFCB5D029, /* [8][0x68]*/
        0x83A6ECA4, 0x77983AEC, 0x6E3736C5, 0x9A09E08D, 0x5D692E97, 0xA957F8DF, 0xB0F8F4F6, 0x44C622BE, /* [8][0x70]*/
        0x4EDE8DEC, 0xBAE05BA4, 0xA34F578D, 0x577181C5, 0x90114FDF, 0x642F9997, 0x7D8095BE, 0x89BE43F6, /* [8][0x78]*/
        0xF6AD7F7B, 0x0293A933, 0x1B3CA51A, 0xEF027352, 0x2862BD48, 0xDC5C6B00, 0xC5F36729, 0x31CDB161, /* [8][0x80]*/
        0xA66805EB, 0x5256D3A3, 0x4BF9DF8A, 0xBFC709C2, 0x78A7C7D8, 0x8C991190, 0x95361DB9, 0x6108CBF1, /* [8][0x88]*/
        0x1E1BF77C, 0xEA252134, 0xF38A2D1D, 0x07B4FB55, 0xC0D4354F, 0x34EAE307, 0x2D45EF2E, 0xD97B3966, /* [8][0x90]*/
        0xD3639634, 0x275D407C, 0x3EF24C55, 0xCACC9A1D, 0x0DAC5407, 0xF992824F, 0xE03D8E66, 0x1403582E, /* [8][0x98]*/
        0x6B1064A3, 0x9F2EB2EB, 0x8681BEC2, 0x72BF688A, 0xB5DFA690, 0x41E170D8, 0x584E7CF1, 0xAC70AAB9, /* [8][0xa0]*/
        0x4C7F2255, 0xB841F41D, 0xA1EEF834, 0x55D02E7C, 0x92B0E066, 0x668E362E, 0x7F213A07, 0x8B1FEC4F, /* [8][0xa8]*/
        0xF40CD0C2, 0x0032068A, 0x199D0AA3, 0xEDA3DCEB, 0x2AC312F1, 0xDEFDC4B9, 0xC752C890, 0x336C1ED8, /* [8][0xb0]*/
        0x3974B18A, 0xCD4A67C2, 0xD4E56BEB, 0x20DBBDA3, 0xE7BB73B9, 0x1385A5F1, 0x0A2AA9D8, 0xFE147F90, /* [8][0xb8]*/
        0x8107431D, 0x75399555, 0x6C96997C, 0x98A84F34, 0x5FC8812E, 0xABF65766, 0xB2595B4F, 0x46678D07, /* [8][0xc0]*/
        0x77AA3C66, 0x8394EA2E, 0x9A3BE607, 0x6E05304F, 0xA965FE55, 0x5D5B281D, 0x44F42434, 0xB0CAF27C, /* [8][0xc8]*/
        0xCFD9CEF1, 0x3BE718B9, 0x22481490, 0xD676C2D8, 0x11160CC2, 0xE528DA8A, 0xFC87D6A3, 0x08B900EB, /* [8][0xd0]*/
        0x02A1AFB9, 0xF69F79F1, 0xEF3075D8, 0x1B0EA390, 0xDC6E6D8A, 0x2850BBC2, 0x31FFB7EB, 0xC5C161A3, /* [8][0xd8]*/
        0xBAD25D2E, 0x4EEC8B66, 0x5743874F, 0xA37D5107, 0x641D9F1D, 0x90234955, 0x898C457C, 0x7DB29334, /* [8][0xe0]*/
        0x9DBD1BD8, 0x6983CD90, 0x702CC1B9, 0x841217F1, 0x4372D9EB, 0xB74C0FA3, 0xAEE3038A, 0x5ADDD5C2, /* [8][0xe8]*/
        0x25CEE94F, 0xD1F03F07, 0xC85F332E, 0x3C61E566, 0xFB012B7C, 0x0F3FFD34, 0x1690F11D, 0xE2AE2755, /* [8][0xf0]*/
        0xE8B68807, 0x1C885E4F, 0x05275266, 0xF119842E, 0x36794A34, 0xC2479C7C, 0xDBE89055, 0x2FD6461D, /* [8][0xf8]*/
        0x50C57A90, 0xA4FBACD8, 0xBD54A0F1, 0x496A76B9, 0x8E0AB8A3, 0x7A346EEB, 0x639B62C2, 0x97A5B48A  /* [8][0x100]*/
    },
    {
        0x00000000, 0xCB567BA5, 0x934081BB, 0x5816FA1E, 0x236D7587, 0xE83B0E22, 0xB02DF43C, 0x7B7B8F99, /* [9][0x08]*/
        0x46DAEB0E, 0x8D8C90AB, 0xD59A6AB5, 0x1ECC1110, 0x65B79E89, 0xAEE1E52C, 0xF6F71F32, 0x3DA16497, /* [9][0x10]*/
        0x8DB5D61C, 0x46E3ADB9, 0x1EF557A7, 0xD5A32C02, 0xAED8A39B, 0x658ED83E, 0x3D982220, 0xF6CE5985, /* [9][0x18]*/
        0xCB6F3D12, 0x003946B7, 0x582FBCA9, 0x9379C70C, 0xE8024895, 0x23543330, 0x7B42C92E, 0xB014B28B, /* [9][0x20]*/
        0x1E87DAC9, 0xD5D1A16C, 0x8DC75B72, 0x469120D7, 0x3DEAAF4E, 0xF6BCD4EB, 0xAEAA2EF5, 0x65FC5550, /* [9][0x28]*/
        0x585D31C7, 0x930B4A62, 0xCB1DB07C, 0x004BCBD9, 0x7B304440, 0xB0663FE5, 0xE870C5FB, 0x2326BE5E, /* [9][0x30]*/
        0x93320CD5, 0x58647770, 0x00728D6E, 0xCB24F6CB, 0xB05F7952, 0x7B0902F7, 0x231FF8E9, 0xE849834C, /* [9][0x38]*/
        0xD5E8E7DB, 0x1EBE9C7E, 0x46A86660, 0x8DFE1DC5, 0xF685925C, 0x3DD3E9F9, 0x65C513E7, 0xAE936842, /* [9][0x40]*/
        0x3D0FB592, 0xF659CE37, 0xAE4F3429, 0x65194F8C, 0x1E62C015, 0xD534BBB0, 0x8D2241AE, 0x46743A0B, /* [9][0x48]*/
        0x7BD55E9C, 0xB0832539, 0xE895DF27, 0x23C3A482, 0x58B82B1B, 0x93EE50BE, 0xCBF8AAA0, 0x00AED105, /* [9][0x50]*/
        0xB0BA638E, 0x7BEC182B, 0x23FAE235, 0xE8AC9990, 0x93D71609, 0x58816DAC, 0x009797B2, 0xCBC1EC17, /* [9][0x58]*/
        0xF6608880, 0x3D36F325, 0x6520093B, 0xAE76729E, 0xD50DFD07, 0x1E5B86A2, 0x464D7CBC, 0x8D1B0719, /* [9][0x60]*/
        0x23886F5B, 0xE8DE14FE, 0xB0C8EEE0, 0x7B9E9545, 0x00E51ADC, 0xCBB36179, 0x93A59B67, 0x58F3E0C2, /* [9][0x68]*/
        0x65528455, 0xAE04FFF0, 0xF61205EE, 0x3D447E4B, 0x463FF1D2, 0x8D698A77, 0xD57F7069, 0x1E290BCC, /* [9][0x70]*/
        0xAE3DB947, 0x656BC2E2, 0x3D7D38FC, 0xF62B4359, 0x8D50CCC0, 0x4606B765, 0x1E104D7B, 0xD54636DE, /* [9][0x78]*/
        0xE8E75249, 0x23B129EC, 0x7BA7D3F2, 0xB0F1A857, 0xCB8A27CE, 0x00DC5C6B, 0x58CAA675, 0x939CDDD0, /* [9][0x80]*/
        0x7A1F6B24, 0xB1491081, 0xE95FEA9F, 0x2209913A, 0x59721EA3, 0x92246506, 0xCA329F18, 0x0164E4BD, /* [9][0x88]*/
        0x3CC5802A, 0xF793FB8F, 0xAF850191, 0x64D37A34, 0x1FA8F5AD, 0xD4FE8E08, 0x8CE87416, 0x47BE0FB3, /* [9][0x90]*/
        0xF7AABD38, 0x3CFCC69D, 0x64EA3C83, 0xAFBC4726, 0xD4C7C8BF, 0x1F91B31A, 0x47874904, 0x8CD132A1, /* [9][0x98]*/
        0xB1705636, 0x7A262D93, 0x2230D78D, 0xE966AC28, 0x921D23B1, 0x594B5814, 0x015DA20A, 0xCA0BD9AF, /* [9][0xa0]*/
        0x6498B1ED, 0xAFCECA48, 0xF7D83056, 0x3C8E4BF3, 0x47F5C46A, 0x8CA3BFCF, 0xD4B545D1, 0x1FE33E74, /* [9][0xa8]*/
        0x22425AE3, 0xE9142146, 0xB102DB58, 0x7A54A0FD, 0x012F2F64, 0xCA7954C1, 0x926FAEDF, 0x5939D57A, /* [9][0xb0]*/
        0xE92D67F1, 0x227B1C54, 0x7A6DE64A, 0xB13B9DEF, 0xCA401276, 0x011669D3, 0x590093CD, 0x9256E868, /* [9][0xb8]*/
        0xAFF78CFF, 0x64A1F75A, 0x3CB70D44, 0xF7E176E1, 0x8C9AF978, 0x47CC82DD, 0x1FDA78C3, 0xD48C0366, /* [9][0xc0]*/
        0x4710DEB6, 0x8C46A513, 0xD4505F0D, 0x1F0624A8, 0x647DAB31, 0xAF2BD094, 0xF73D2A8A, 0x3C6B512F, /* [9][0xc8]*/
        0x01CA35B8, 0xCA9C4E1D, 0x928AB403, 0x59DCCFA6, 0x22A7403F, 0xE9F13B9A, 0xB1E7C184, 0x7AB1BA21, /* [9][0xd0]*/
        0xCAA508AA, 0x01F3730F, 0x59E58911, 0x92B3F2B4, 0xE9C87D2D, 0x229E0688, 0x7A88FC96, 0xB1DE8733, /* [9][0xd8]*/
        0x8C7FE3A4, 0x47299801, 0x1F3F621F, 0xD46919BA, 0xAF129623, 0x6444ED86, 0x3C521798, 0xF7046C3D, /* [9][0xe0]*/
        0x5997047F, 0x92C17FDA, 0xCAD785C4, 0x0181FE61, 0x7AFA71F8, 0xB1AC0A5D, 0xE9BAF043, 0x22EC8BE6, /* [9][0xe8]*/
        0x1F4DEF71, 0xD41B94D4, 0x8C0D6ECA, 0x475B156F, 0x3C209AF6, 0xF776E153, 0xAF601B4D, 0x643660E8, /* [9][0xf0]*/
        0xD422D263, 0x1F74A9C6, 0x476253D8, 0x8C34287D, 0xF74FA7E4, 0x3C19DC41, 0x640F265F, 0xAF595DFA, /* [9][0xf8]*/
        0x92F8396D, 0x59AE42C8, 0x01B8B8D6, 0xCAEEC373, 0xB1954CEA, 0x7AC3374F, 0x22D5CD51, 0xE983B6F4  /* [9][0x100]*/
    },
    {
        0x00000000, 0x9771F7C1, 0x2B0F9973, 0xBC7E6EB2, 0x561F32E6, 0xC16EC527, 0x7D10AB95, 0xEA615C54, /* [10][0x08]*/
        0xAC3E65CC, 0x3B4F920D, 0x8731FCBF, 0x10400B7E, 0xFA21572A, 0x6D50A0EB, 0xD12ECE59, 0x465F3998, /* [10][0x10]*/
        0x5D90BD69, 0xCAE14AA8, 0x769F241A, 0xE1EED3DB, 0x0B8F8F8F, 0x9CFE784E, 0x208016FC, 0xB7F1E13D, /* [10][0x18]*/
        0xF1AED8A5, 0x66DF2F64, 0xDAA141D6, 0x4DD0B617, 0xA7B1EA43, 0x30C01D82, 0x8CBE7330, 0x1BCF84F1, /* [10][0x20]*/
        0xBB217AD2, 0x2C508D13, 0x902EE3A1, 0x075F1460, 0xED3E4834, 0x7A4FBFF5, 0xC631D147, 0x51402686, /* [10][0x28]*/
        0x171F1F1E, 0x806EE8DF, 0x3C10866D, 0xAB6171AC, 0x41002DF8, 0xD671DA39, 0x6A0FB48B, 0xFD7E434A, /* [10][0x30]*/
        0xE6B1C7BB, 0x71C0307A, 0xCDBE5EC8, 0x5ACFA909, 0xB0AEF55D, 0x27DF029C, 0x9BA16C2E, 0x0CD09BEF, /* [10][0x38]*/
        0x4A8FA277, 0xDDFE55B6, 0x61803B04, 0xF6F1CCC5, 0x1C909091, 0x8BE16750, 0x379F09E2, 0xA0EEFE23, /* [10][0x40]*/
        0x73AE8355, 0xE4DF7494, 0x58A11A26, 0xCFD0EDE7, 0x25B1B1B3, 0xB2C04672, 0x0EBE28C0, 0x99CFDF01, /* [10][0x48]*/
        0xDF90E699, 0x48E11158, 0xF49F7FEA, 0x63EE882B, 0x898FD47F, 0x1EFE23BE, 0xA2804D0C, 0x35F1BACD, /* [10][0x50]*/
        0x2E3E3E3C, 0xB94FC9FD, 0x0531A74F, 0x9240508E, 0x78210CDA, 0xEF50FB1B, 0x532E95A9, 0xC45F6268, /* [10][0x58]*/
        0x82005BF0, 0x1571AC31, 0xA90FC283, 0x3E7E3542, 0xD41F6916, 0x436E9ED7, 0xFF10F065, 0x686107A4, /* [10][0x60]*/
        0xC88FF987, 0x5FFE0E46, 0xE38060F4, 0x74F19735, 0x9E90CB61, 0x09E13CA0, 0xB59F5212, 0x22EEA5D3, /* [10][0x68]*/
        0x64B19C4B, 0xF3C06B8A, 0x4FBE0538, 0xD8CFF2F9, 0x32AEAEAD, 0xA5DF596C, 0x19A137DE, 0x8ED0C01F, /* [10][0x70]*/
        0x951F44EE, 0x026EB32F, 0xBE10DD9D, 0x29612A5C, 0xC3007608, 0x547181C9, 0xE80FEF7B, 0x7F7E18BA, /* [10][0x78]*/
        0x39212122, 0xAE50D6E3, 0x122EB851, 0x855F4F90, 0x6F3E13C4, 0xF84FE405, 0x44318AB7, 0xD3407D76, /* [10][0x80]*/
        0xE75D06AA, 0x702CF16B, 0xCC529FD9, 0x5B236818, 0xB142344C, 0x2633C38D, 0x9A4DAD3F, 0x0D3C5AFE, /* [10][0x88]*/
        0x4B636366, 0xDC1294A7, 0x606CFA15, 0xF71D0DD4, 0x1D7C5180, 0x8A0DA641, 0x3673C8F3, 0xA1023F32, /* [10][0x90]*/
        0xBACDBBC3, 0x2DBC4C02, 0x91C222B0, 0x06B3D571, 0xECD28925, 0x7BA37EE4, 0xC7DD1056, 0x50ACE797, /* [10][0x98]*/
        0x16F3DE0F, 0x818229CE, 0x3DFC477C, 0xAA8DB0BD, 0x40ECECE9, 0xD79D1B28, 0x6BE3759A, 0xFC92825B, /* [10][0xa0]*/
        0x5C7C7C78, 0xCB0D8BB9, 0x7773E50B, 0xE00212CA, 0x0A634E9E, 0x9D12B95F, 0x216CD7ED, 0xB61D202C, /* [10][0xa8]*/
        0xF04219B4, 0x6733EE75, 0xDB4D80C7, 0x4C3C7706, 0xA65D2B52, 0x312CDC93, 0x8D52B221, 0x1A2345E0, /* [10][0xb0]*/
        0x01ECC111, 0x969D36D0, 0x2AE35862, 0xBD92AFA3, 0x57F3F3F7, 0xC0820436, 0x7CFC6A84, 0xEB8D9D45, /* [10][0xb8]*/
        0xADD2A4DD, 0x3AA3531C, 0x86DD3DAE, 0x11ACCA6F, 0xFBCD963B, 0x6CBC61FA, 0xD0C20F48, 0x47B3F889, /* [10][0xc0]*/
        0x94F385FF, 0x0382723E, 0xBFFC1C8C, 0x288DEB4D, 0xC2ECB719, 0x559D40D8, 0xE9E32E6A, 0x7E92D9AB, /* [10][0xc8]*/
        0x38CDE033, 0xAFBC17F2, 0x13C27940, 0x84B38E81, 0x6ED2D2D5, 0xF9A32514, 0x45DD4BA6, 0xD2ACBC67, /* [10][0xd0]*/
        0xC9633896, 0x5E12CF57, 0xE26CA1E5, 0x751D5624, 0x9F7C0A70, 0x080DFDB1, 0xB4739303, 0x230264C2, /* [10][0xd8]*/
        0x655D5D5A, 0xF22CAA9B, 0x4E52C429, 0xD92333E8, 0x33426FBC, 0xA433987D, 0x184DF6CF, 0x8F3C010E, /* [10][0xe0]*/
        0x2FD2FF2D, 0xB8A308EC, 0x04DD665E, 0x93AC919F, 0x79CDCDCB, 0xEEBC3A0A, 0x52C254B8, 0xC5B3A379, /* [10][0xe8]*/
        0x83EC9AE1, 0x149D6D20, 0xA8E30392, 0x3F92F453, 0xD5F3A807, 0x42825FC6, 0xFEFC3174, 0x698DC6B5, /* [10][0xf0]*/
        0x72424244, 0xE533B585, 0x594DDB37, 0xCE3C2CF6, 0x245D70A2, 0xB32C8763, 0x0F52E9D1, 0x98231E10, /* [10][0xf8]*/
        0xDE7C2788, 0x490DD049, 0xF573BEFB, 0x6202493A, 0x8863156E, 0x1F12E2AF, 0xA36C8C1D, 0x341D7BDC  /* [10][0x100]*/
    },
    {
        0x00000000, 0x3171D430, 0x62E3A860, 0x53927C50, 0xC5C750C0, 0xF4B684F0, 0xA724F8A0, 0x96552C90, /* [11][0x08]*/
        0x8E62D771, 0xBF130341, 0xEC817F11, 0xDDF0AB21, 0x4BA587B1, 0x7AD45381, 0x29462FD1, 0x1837FBE1, /* [11][0x10]*/
        0x1929D813, 0x28580C23, 0x7BCA7073, 0x4ABBA443, 0xDCEE88D3, 0xED9F5CE3, 0xBE0D20B3, 0x8F7CF483, /* [11][0x18]*/
        0x974B0F62, 0xA63ADB52, 0xF5A8A702, 0xC4D97332, 0x528C5FA2, 0x63FD8B92, 0x306FF7C2, 0x011E23F2, /* [11][0x20]*/
        0x3253B026, 0x03226416, 0x50B01846, 0x61C1CC76, 0xF794E0E6, 0xC6E534D6, 0x95774886, 0xA4069CB6, /* [11][0x28]*/
        0xBC316757, 0x8D40B367, 0xDED2CF37, 0xEFA31B07, 0x79F63797, 0x4887E3A7, 0x1B159FF7, 0x2A644BC7, /* [11][0x30]*/
        0x2B7A6835, 0x1A0BBC05, 0x4999C055, 0x78E81465, 0xEEBD38F5, 0xDFCCECC5, 0x8C5E9095, 0xBD2F44A5, /* [11][0x38]*/
        0xA518BF44, 0x94696B74, 0xC7FB1724, 0xF68AC314, 0x60DFEF84, 0x51AE3BB4, 0x023C47E4, 0x334D93D4, /* [11][0x40]*/
        0x64A7604C, 0x55D6B47C, 0x0644C82C, 0x37351C1C, 0xA160308C, 0x9011E4BC, 0xC38398EC, 0xF2F24CDC, /* [11][0x48]*/
        0xEAC5B73D, 0xDBB4630D, 0x88261F5D, 0xB957CB6D, 0x2F02E7FD, 0x1E7333CD, 0x4DE14F9D, 0x7C909BAD, /* [11][0x50]*/
        0x7D8EB85F, 0x4CFF6C6F, 0x1F6D103F, 0x2E1CC40F, 0xB849E89F, 0x89383CAF, 0xDAAA40FF, 0xEBDB94CF, /* [11][0x58]*/
        0xF3EC6F2E, 0xC29DBB1E, 0x910FC74E, 0xA07E137E, 0x362B3FEE, 0x075AEBDE, 0x54C8978E, 0x65B943BE, /* [11][0x60]*/
        0x56F4D06A, 0x6785045A, 0x3417780A, 0x0566AC3A, 0x933380AA, 0xA242549A, 0xF1D028CA, 0xC0A1FCFA, /* [11][0x68]*/
        0xD896071B, 0xE9E7D32B, 0xBA75AF7B, 0x8B047B4B, 0x1D5157DB, 0x2C2083EB, 0x7FB2FFBB, 0x4EC32B8B, /* [11][0x70]*/
        0x4FDD0879, 0x7EACDC49, 0x2D3EA019, 0x1C4F7429, 0x8A1A58B9, 0xBB6B8C89, 0xE8F9F0D9, 0xD98824E9, /* [11][0x78]*/
        0xC1BFDF08, 0xF0CE0B38, 0xA35C7768, 0x922DA358, 0x04788FC8, 0x35095BF8, 0x669B27A8, 0x57EAF398, /* [11][0x80]*/
        0xC94EC098, 0xF83F14A8, 0xABAD68F8, 0x9ADCBCC8, 0x0C899058, 0x3DF84468, 0x6E6A3838, 0x5F1BEC08, /* [11][0x88]*/
        0x472C17E9, 0x765DC3D9, 0x25CFBF89, 0x14BE6BB9, 0x82EB4729, 0xB39A9319, 0xE008EF49, 0xD1793B79, /* [11][0x90]*/
        0xD067188B, 0xE116CCBB, 0xB284B0EB, 0x83F564DB, 0x15A0484B, 0x24D19C7B, 0x7743E02B, 0x4632341B, /* [11][0x98]*/
        0x5E05CFFA, 0x6F741BCA, 0x3CE6679A, 0x0D97B3AA, 0x9BC29F3A, 0xAAB34B0A, 0xF921375A, 0xC850E36A, /* [11][0xa0]*/
        0xFB1D70BE, 0xCA6CA48E, 0x99FED8DE, 0xA88F0CEE, 0x3EDA207E, 0x0FABF44E, 0x5C39881E, 0x6D485C2E, /* [11][0xa8]*/
        0x757FA7CF, 0x440E73FF, 0x179C0FAF, 0x26EDDB9F, 0xB0B8F70F, 0x81C9233F, 0xD25B5F6F, 0xE32A8B5F, /* [11][0xb0]*/
        0xE234A8AD, 0xD3457C9D, 0x80D700CD, 0xB1A6D4FD, 0x27F3F86D, 0x16822C5D, 0x4510500D, 0x7461843D, /* [11][0xb8]*/
        0x6C567FDC, 0x5D27ABEC, 0x0EB5D7BC, 0x3FC4038C, 0xA9912F1C, 0x98E0FB2C, 0xCB72877C, 0xFA03534C, /* [11][0xc0]*/
        0xADE9A0D4, 0x9C9874E4, 0xCF0A08B4, 0xFE7BDC84, 0x682EF014, 0x595F2424, 0x0ACD5874, 0x3BBC8C44, /* [11][0xc8]*/
        0x238B77A5, 0x12FAA395, 0x4168DFC5, 0x70190BF5, 0xE64C2765, 0xD73DF355, 0x84AF8F05, 0xB5DE5B35, /* [11][0xd0]*/
        0xB4C078C7, 0x85B1ACF7, 0xD623D0A7, 0xE7520497, 0x71072807, 0x4076FC37, 0x13E48067, 0x22955457, /* [11][0xd8]*/
        0x3AA2AFB6, 0x0BD37B86, 0x584107D6, 0x6930D3E6, 0xFF65FF76, 0xCE142B46, 0x9D865716, 0xACF78326, /* [11][0xe0]*/
        0x9FBA10F2, 0xAECBC4C2, 0xFD59B892, 0xCC286CA2, 0x5A7D4032, 0x6B0C9402, 0x389EE852, 0x09EF3C62, /* [11][0xe8]*/
        0x11D8C783, 0x20A913B3, 0x733B6FE3, 0x424ABBD3, 0xD41F9743, 0xE56E4373, 0xB6FC3F23, 0x878DEB13, /* [11][0xf0]*/
        0x8693C8E1, 0xB7E21CD1, 0xE4706081, 0xD501B4B1, 0x43549821, 0x72254C11, 0x21B73041, 0x10C6E471, /* [11][0xf8]*/
        0x08F11F90, 0x3980CBA0, 0x6A12B7F0, 0x5B6363C0, 0xCD364F50, 0xFC479B60, 0xAFD5E730, 0x9EA43300  /* [11][0x100]*/
    },
    {
        0x00000000, 0x30D23865, 0x61A470CA, 0x517648AF, 0xC348E194, 0xF39AD9F1, 0xA2EC915E, 0x923EA93B, /* [12][0x08]*/
        0x837DB5D9, 0xB3AF8DBC, 0xE2D9C513, 0xD20BFD76, 0x4035544D, 0x70E76C28, 0x21912487, 0x11431CE2, /* [12][0x10]*/
        0x03171D43, 0x33C52526, 0x62B36D89, 0x526155EC, 0xC05FFCD7, 0xF08DC4B2, 0xA1FB8C1D, 0x9129B478, /* [12][0x18]*/
        0x806AA89A, 0xB0B890FF, 0xE1CED850, 0xD11CE035, 0x4322490E, 0x73F0716B, 0x228639C4, 0x125401A1, /* [12][0x20]*/
        0x062E3A86, 0x36FC02E3, 0x678A4A4C, 0x57587229, 0xC566DB12, 0xF5B4E377, 0xA4C2ABD8, 0x941093BD, /* [12][0x28]*/
        0x85538F5F, 0xB581B73A, 0xE4F7FF95, 0xD425C7F0, 0x461B6ECB, 0x76C956AE, 0x27BF1E01, 0x176D2664, /* [12][0x30]*/
        0x053927C5, 0x35EB1FA0, 0x649D570F, 0x544F6F6A, 0xC671C651, 0xF6A3FE34, 0xA7D5B69B, 0x97078EFE, /* [12][0x38]*/
        0x8644921C, 0xB696AA79, 0xE7E0E2D6, 0xD732DAB3, 0x450C7388, 0x75DE4BED, 0x24A80342, 0x147A3B27, /* [12][0x40]*/
        0x0C5C750C, 0x3C8E4D69, 0x6DF805C6, 0x5D2A3DA3, 0xCF149498, 0xFFC6ACFD, 0xAEB0E452, 0x9E62DC37, /* [12][0x48]*/
        0x8F21C0D5, 0xBFF3F8B0, 0xEE85B01F, 0xDE57887A, 0x4C692141, 0x7CBB1924, 0x2DCD518B, 0x1D1F69EE, /* [12][0x50]*/
        0x0F4B684F, 0x3F99502A, 0x6EEF1885, 0x5E3D20E0, 0xCC0389DB, 0xFCD1B1BE, 0xADA7F911, 0x9D75C174, /* [12][0x58]*/
        0x8C36DD96, 0xBCE4E5F3, 0xED92AD5C, 0xDD409539, 0x4F7E3C02, 0x7FAC0467, 0x2EDA4CC8, 0x1E0874AD, /* [12][0x60]*/
        0x0A724F8A, 0x3AA077EF, 0x6BD63F40, 0x5B040725, 0xC93AAE1E, 0xF9E8967B, 0xA89EDED4, 0x984CE6B1, /* [12][0x68]*/
        0x890FFA53, 0xB9DDC236, 0xE8AB8A99, 0xD879B2FC, 0x4A471BC7, 0x7A9523A2, 0x2BE36B0D, 0x1B315368, /* [12][0x70]*/
        0x096552C9, 0x39B76AAC, 0x68C12203, 0x58131A66, 0xCA2DB35D, 0xFAFF8B38, 0xAB89C397, 0x9B5BFBF2, /* [12][0x78]*/
        0x8A18E710, 0xBACADF75, 0xEBBC97DA, 0xDB6EAFBF, 0x49500684, 0x79823EE1, 0x28F4764E, 0x18264E2B, /* [12][0x80]*/
        0x18B8EA18, 0x286AD27D, 0x791C9AD2, 0x49CEA2B7, 0xDBF00B8C, 0xEB2233E9, 0xBA547B46, 0x8A864323, /* [12][0x88]*/
        0x9BC55FC1, 0xAB1767A4, 0xFA612F0B, 0xCAB3176E, 0x588DBE55, 0x685F8630, 0x3929CE9F, 0x09FBF6FA, /* [12][0x90]*/
        0x1BAFF75B, 0x2B7DCF3E, 0x7A0B8791, 0x4AD9BFF4, 0xD8E716CF, 0xE8352EAA, 0xB9436605, 0x89915E60, /* [12][0x98]*/
        0x98D24282, 0xA8007AE7, 0xF9763248, 0xC9A40A2D, 0x5B9AA316, 0x6B489B73, 0x3A3ED3DC, 0x0AECEBB9, /* [12][0xa0]*/
        0x1E96D09E, 0x2E44E8FB, 0x7F32A054, 0x4FE09831, 0xDDDE310A, 0xED0C096F, 0xBC7A41C0, 0x8CA879A5, /* [12][0xa8]*/
        0x9DEB6547, 0xAD395D22, 0xFC4F158D, 0xCC9D2DE8, 0x5EA384D3, 0x6E71BCB6, 0x3F07F419, 0x0FD5CC7C, /* [12][0xb0]*/
        0x1D81CDDD, 0x2D53F5B8, 0x7C25BD17, 0x4CF78572, 0xDEC92C49, 0xEE1B142C, 0xBF6D5C83, 0x8FBF64E6, /* [12][0xb8]*/
        0x9EFC7804, 0xAE2E4061, 0xFF5808CE, 0xCF8A30AB, 0x5DB49990, 0x6D66A1F5, 0x3C10E95A, 0x0CC2D13F, /* [12][0xc0]*/
        0x14E49F14, 0x2436A771, 0x7540EFDE, 0x4592D7BB, 0xD7AC7E80, 0xE77E46E5, 0xB6080E4A, 0x86DA362F, /* [12][0xc8]*/
        0x97992ACD, 0xA74B12A8, 0xF63D5A07, 0xC6EF6262, 0x54D1CB59, 0x6403F33C, 0x3575BB93, 0x05A783F6, /* [12][0xd0]*/
        0x17F38257, 0x2721BA32, 0x7657F29D, 0x4685CAF8, 0xD4BB63C3, 0xE4695BA6, 0xB51F1309, 0x85CD2B6C, /* [12][0xd8]*/
        0x948E378E, 0xA45C0FEB, 0xF52A4744, 0xC5F87F21, 0x57C6D61A, 0x6714EE7F, 0x3662A6D0, 0x06B09EB5, /* [12][0xe0]*/
        0x12CAA592, 0x22189DF7, 0x736ED558, 0x43BCED3D, 0xD1824406, 0xE1507C63, 0xB02634CC, 0x80F40CA9, /* [12][0xe8]*/
        0x91B7104B, 0xA165282E, 0xF0136081, 0xC0C158E4, 0x52FFF1DF, 0x622DC9BA, 0x335B8115, 0x0389B970, /* [12][0xf0]*/
        0x11DDB8D1, 0x210F80B4, 0x7079C81B, 0x40ABF07E, 0xD2955945, 0xE2476120, 0xB331298F, 0x83E311EA, /* [12][0xf8]*/
        0x92A00D08, 0xA272356D, 0xF3047DC2, 0xC3D645A7, 0x51E8EC9C, 0x613AD4F9, 0x304C9C56, 0x009EA433  /* [12][0x100]*/
    },
    {
        0x00000000, 0x54075546, 0xA80EAA8C, 0xFC09FFCA, 0x55F123E9, 0x01F676AF, 0xFDFF8965, 0xA9F8DC23, /* [13][0x08]*/
        0xABE247D2, 0xFFE51294, 0x03ECED5E, 0x57EBB818, 0xFE13643B, 0xAA14317D, 0x561DCEB7, 0x021A9BF1, /* [13][0x10]*/
        0x5228F955, 0x062FAC13, 0xFA2653D9, 0xAE21069F, 0x07D9DABC, 0x53DE8FFA, 0xAFD77030, 0xFBD02576, /* [13][0x18]*/
        0xF9CABE87, 0xADCDEBC1, 0x51C4140B, 0x05C3414D, 0xAC3B9D6E, 0xF83CC828, 0x043537E2, 0x503262A4, /* [13][0x20]*/
        0xA451F2AA, 0xF056A7EC, 0x0C5F5826, 0x58580D60, 0xF1A0D143, 0xA5A78405, 0x59AE7BCF, 0x0DA92E89, /* [13][0x28]*/
        0x0FB3B578, 0x5BB4E03E, 0xA7BD1FF4, 0xF3BA4AB2, 0x5A429691, 0x0E45C3D7, 0xF24C3C1D, 0xA64B695B, /* [13][0x30]*/
        0xF6790BFF, 0xA27E5EB9, 0x5E77A173, 0x0A70F435, 0xA3882816, 0xF78F7D50, 0x0B86829A, 0x5F81D7DC, /* [13][0x38]*/
        0x5D9B4C2D, 0x099C196B, 0xF595E6A1, 0xA192B3E7, 0x086A6FC4, 0x5C6D3A82, 0xA064C548, 0xF463900E, /* [13][0x40]*/
        0x4D4F93A5, 0x1948C6E3, 0xE5413929, 0xB1466C6F, 0x18BEB04C, 0x4CB9E50A, 0xB0B01AC0, 0xE4B74F86, /* [13][0x48]*/
        0xE6ADD477, 0xB2AA8131, 0x4EA37EFB, 0x1AA42BBD, 0xB35CF79E, 0xE75BA2D8, 0x1B525D12, 0x4F550854, /* [13][0x50]*/
        0x1F676AF0, 0x4B603FB6, 0xB769C07C, 0xE36E953A, 0x4A964919, 0x1E911C5F, 0xE298E395, 0xB69FB6D3, /* [13][0x58]*/
        0xB4852D22, 0xE0827864, 0x1C8B87AE, 0x488CD2E8, 0xE1740ECB, 0xB5735B8D, 0x497AA447, 0x1D7DF101, /* [13][0x60]*/
        0xE91E610F, 0xBD193449, 0x4110CB83, 0x15179EC5, 0xBCEF42E6, 0xE8E817A0, 0x14E1E86A, 0x40E6BD2C, /* [13][0x68]*/
        0x42FC26DD, 0x16FB739B, 0xEAF28C51, 0xBEF5D917, 0x170D0534, 0x430A5072, 0xBF03AFB8, 0xEB04FAFE, /* [13][0x70]*/
        0xBB36985A, 0xEF31CD1C, 0x133832D6, 0x473F6790, 0xEEC7BBB3, 0xBAC0EEF5, 0x46C9113F, 0x12CE4479, /* [13][0x78]*/
        0x10D4DF88, 0x44D38ACE, 0xB8DA7504, 0xECDD2042, 0x4525FC61, 0x1122A927, 0xED2B56ED, 0xB92C03AB, /* [13][0x80]*/
        0x9A9F274A, 0xCE98720C, 0x32918DC6, 0x6696D880, 0xCF6E04A3, 0x9B6951E5, 0x6760AE2F, 0x3367FB69, /* [13][0x88]*/
        0x317D6098, 0x657A35DE, 0x9973CA14, 0xCD749F52, 0x648C4371, 0x308B1637, 0xCC82E9FD, 0x9885BCBB, /* [13][0x90]*/
        0xC8B7DE1F, 0x9CB08B59, 0x60B97493, 0x34BE21D5, 0x9D46FDF6, 0xC941A8B0, 0x3548577A, 0x614F023C, /* [13][0x98]*/
        0x635599CD, 0x3752CC8B, 0xCB5B3341, 0x9F5C6607, 0x36A4BA24, 0x62A3EF62, 0x9EAA10A8, 0xCAAD45EE, /* [13][0xa0]*/
        0x3ECED5E0, 0x6AC980A6, 0x96C07F6C, 0xC2C72A2A, 0x6B3FF609, 0x3F38A34F, 0xC3315C85, 0x973609C3, /* [13][0xa8]*/
        0x952C9232, 0xC12BC774, 0x3D2238BE, 0x69256DF8, 0xC0DDB1DB, 0x94DAE49D, 0x68D31B57, 0x3CD44E11, /* [13][0xb0]*/
        0x6CE62CB5, 0x38E179F3, 0xC4E88639, 0x90EFD37F, 0x39170F5C, 0x6D105A1A, 0x9119A5D0, 0xC51EF096, /* [13][0xb8]*/
        0xC7046B67, 0x93033E21, 0x6F0AC1EB, 0x3B0D94AD, 0x92F5488E, 0xC6F21DC8, 0x3AFBE202, 0x6EFCB744, /* [13][0xc0]*/
        0xD7D0B4EF, 0x83D7E1A9, 0x7FDE1E63, 0x2BD94B25, 0x82219706, 0xD626C240, 0x2A2F3D8A, 0x7E2868CC, /* [13][0xc8]*/
        0x7C32F33D, 0x2835A67B, 0xD43C59B1, 0x803B0CF7, 0x29C3D0D4, 0x7DC48592, 0x81CD7A58, 0xD5CA2F1E, /* [13][0xd0]*/
        0x85F84DBA, 0xD1FF18FC, 0x2DF6E736, 0x79F1B270, 0xD0096E53, 0x840E3B15, 0x7807C4DF, 0x2C009199, /* [13][0xd8]*/
        0x2E1A0A68, 0x7A1D5F2E, 0x8614A0E4, 0xD213F5A2, 0x7BEB2981, 0x2FEC7CC7, 0xD3E5830D, 0x87E2D64B, /* [13][0xe0]*/
        0x73814645, 0x27861303, 0xDB8FECC9, 0x8F88B98F, 0x267065AC, 0x727730EA, 0x8E7ECF20, 0xDA799A66, /* [13][0xe8]*/
        0xD8630197, 0x8C6454D1, 0x706DAB1B, 0x246AFE5D, 0x8D92227E, 0xD9957738, 0x259C88F2, 0x719BDDB4, /* [13][0xf0]*/
        0x21A9BF10, 0x75AEEA56, 0x89A7159C, 0xDDA040DA, 0x74589CF9, 0x205FC9BF, 0xDC563675, 0x88516333, /* [13][0xf8]*/
        0x8A4BF8C2, 0xDE4CAD84, 0x2245524E, 0x76420708, 0xDFBADB2B, 0x8BBD8E6D, 0x77B471A7, 0x23B324E1  /* [13][0x100]*/
    },
    {
        0x00000000, 0x678EFD01, 0xCF1DFA02, 0xA8930703, 0x9BD782F5, 0xFC597FF4, 0x54CA78F7, 0x334485F6, /* [14][0x08]*/
        0x3243731B, 0x55CD8E1A, 0xFD5E8919, 0x9AD07418, 0xA994F1EE, 0xCE1A0CEF, 0x66890BEC, 0x0107F6ED, /* [14][0x10]*/
        0x6486E636, 0x03081B37, 0xAB9B1C34, 0xCC15E135, 0xFF5164C3, 0x98DF99C2, 0x304C9EC1, 0x57C263C0, /* [14][0x18]*/
        0x56C5952D, 0x314B682C, 0x99D86F2F, 0xFE56922E, 0xCD1217D8, 0xAA9CEAD9, 0x020FEDDA, 0x658110DB, /* [14][0x20]*/
        0xC90DCC6C, 0xAE83316D, 0x0610366E, 0x619ECB6F, 0x52DA4E99, 0x3554B398, 0x9DC7B49B, 0xFA49499A, /* [14][0x28]*/
        0xFB4EBF77, 0x9CC04276, 0x34534575, 0x53DDB874, 0x60993D82, 0x0717C083, 0xAF84C780, 0xC80A3A81, /* [14][0x30]*/
        0xAD8B2A5A, 0xCA05D75B, 0x6296D058, 0x05182D59, 0x365CA8AF, 0x51D255AE, 0xF94152AD, 0x9ECFAFAC, /* [14][0x38]*/
        0x9FC85941, 0xF846A440, 0x50D5A343, 0x375B5E42, 0x041FDBB4, 0x639126B5, 0xCB0221B6, 0xAC8CDCB7, /* [14][0x40]*/
        0x97F7EE29, 0xF0791328, 0x58EA142B, 0x3F64E92A, 0x0C206CDC, 0x6BAE91DD, 0xC33D96DE, 0xA4B36BDF, /* [14][0x48]*/
        0xA5B49D32, 0xC23A6033, 0x6AA96730, 0x0D279A31, 0x3E631FC7, 0x59EDE2C6, 0xF17EE5C5, 0x96F018C4, /* [14][0x50]*/
        0xF371081F, 0x94FFF51E, 0x3C6CF21D, 0x5BE20F1C, 0x68A68AEA, 0x0F2877EB, 0xA7BB70E8, 0xC0358DE9, /* [14][0x58]*/
        0xC1327B04, 0xA6BC8605, 0x0E2F8106, 0x69A17C07, 0x5AE5F9F1, 0x3D6B04F0, 0x95F803F3, 0xF276FEF2, /* [14][0x60]*/
        0x5EFA2245, 0x3974DF44, 0x91E7D847, 0xF6692546, 0xC52DA0B0, 0xA2A35DB1, 0x0A305AB2, 0x6DBEA7B3, /* [14][0x68]*/
        0x6CB9515E, 0x0B37AC5F, 0xA3A4AB5C, 0xC42A565D, 0xF76ED3AB, 0x90E02EAA, 0x387329A9, 0x5FFDD4A8, /* [14][0x70]*/
        0x3A7CC473, 0x5DF23972, 0xF5613E71, 0x92EFC370, 0xA1AB4686, 0xC625BB87, 0x6EB6BC84, 0x09384185, /* [14][0x78]*/
        0x083FB768, 0x6FB14A69, 0xC7224D6A, 0xA0ACB06B, 0x93E8359D, 0xF466C89C, 0x5CF5CF9F, 0x3B7B329E, /* [14][0x80]*/
        0x2A03AAA3, 0x4D8D57A2, 0xE51E50A1, 0x8290ADA0, 0xB1D42856, 0xD65AD557, 0x7EC9D254, 0x19472F55, /* [14][0x88]*/
        0x1840D9B8, 0x7FCE24B9, 0xD75D23BA, 0xB0D3DEBB, 0x83975B4D, 0xE419A64C, 0x4C8AA14F, 0x2B045C4E, /* [14][0x90]*/
        0x4E854C95, 0x290BB194, 0x8198B697, 0xE6164B96, 0xD552CE60, 0xB2DC3361, 0x1A4F3462, 0x7DC1C963, /* [14][0x98]*/
        0x7CC63F8E, 0x1B48C28F, 0xB3DBC58C, 0xD455388D, 0xE711BD7B, 0x809F407A, 0x280C4779, 0x4F82BA78, /* [14][0xa0]*/
        0xE30E66CF, 0x84809BCE, 0x2C139CCD, 0x4B9D61CC, 0x78D9E43A, 0x1F57193B, 0xB7C41E38, 0xD04AE339, /* [14][0xa8]*/
        0xD14D15D4, 0xB6C3E8D5, 0x1E50EFD6, 0x79DE12D7, 0x4A9A9721, 0x2D146A20, 0x85876D23, 0xE2099022, /* [14][0xb0]*/
        0x878880F9, 0xE0067DF8, 0x48957AFB, 0x2F1B87FA, 0x1C5F020C, 0x7BD1FF0D, 0xD342F80E, 0xB4CC050F, /* [14][0xb8]*/
        0xB5CBF3E2, 0xD2450EE3, 0x7AD609E0, 0x1D58F4E1, 0x2E1C7117, 0x49928C16, 0xE1018B15, 0x868F7614, /* [14][0xc0]*/
        0xBDF4448A, 0xDA7AB98B, 0x72E9BE88, 0x15674389, 0x2623C67F, 0x41AD3B7E, 0xE93E3C7D, 0x8EB0C17C, /* [14][0xc8]*/
        0x8FB73791, 0xE839CA90, 0x40AACD93, 0x27243092, 0x1460B564, 0x73EE4865, 0xDB7D4F66, 0xBCF3B267, /* [14][0xd0]*/
        0xD972A2BC, 0xBEFC5FBD, 0x166F58BE, 0x71E1A5BF, 0x42A52049, 0x252BDD48, 0x8DB8DA4B, 0xEA36274A, /* [14][0xd8]*/
        0xEB31D1A7, 0x8CBF2CA6, 0x242C2BA5, 0x43A2D6A4, 0x70E65352, 0x1768AE53, 0xBFFBA950, 0xD8755451, /* [14][0xe0]*/
        0x74F988E6, 0x137775E7, 0xBBE472E4, 0xDC6A8FE5, 0xEF2E0A13, 0x88A0F712, 0x2033F011, 0x47BD0D10, /* [14][0xe8]*/
        0x46BAFBFD, 0x213406FC, 0x89A701FF, 0xEE29FCFE, 0xDD6D7908, 0xBAE38409, 0x1270830A, 0x75FE7E0B, /* [14][0xf0]*/
        0x107F6ED0, 0x77F193D1, 0xDF6294D2, 0xB8EC69D3, 0x8BA8EC25, 0xEC261124, 0x44B51627, 0x233BEB26, /* [14][0xf8]*/
        0x223C1DCB, 0x45B2E0CA, 0xED21E7C9, 0x8AAF1AC8, 0xB9EB9F3E, 0xDE65623F, 0x76F6653C, 0x1178983D  /* [14][0x100]*/
    },
    {
        0x00000000, 0xF20C0DFE, 0xE1F46D0D, 0x13F860F3, 0xC604ACEB, 0x3408A115, 0x27F0C1E6, 0xD5FCCC18, /* [15][0x08]*/
        0x89E52F27, 0x7BE922D9, 0x6811422A, 0x9A1D4FD4, 0x4FE183CC, 0xBDED8E32, 0xAE15EEC1, 0x5C19E33F, /* [15][0x10]*/
        0x162628BF, 0xE42A2541, 0xF7D245B2, 0x05DE484C, 0xD0228454, 0x222E89AA, 0x31D6E959, 0xC3DAE4A7, /* [15][0x18]*/
        0x9FC30798, 0x6DCF0A66, 0x7E376A95, 0x8C3B676B, 0x59C7AB73, 0xABCBA68D, 0xB833C67E, 0x4A3FCB80, /* [15][0x20]*/
        0x2C4C517E, 0xDE405C80, 0xCDB83C73, 0x3FB4318D, 0xEA48FD95, 0x1844F06B, 0x0BBC9098, 0xF9B09D66, /* [15][0x28]*/
        0xA5A97E59, 0x57A573A7, 0x445D1354, 0xB6511EAA, 0x63ADD2B2, 0x91A1DF4C, 0x8259BFBF, 0x7055B241, /* [15][0x30]*/
        0x3A6A79C1, 0xC866743F, 0xDB9E14CC, 0x29921932, 0xFC6ED52A, 0x0E62D8D4, 0x1D9AB827, 0xEF96B5D9, /* [15][0x38]*/
        0xB38F56E6, 0x41835B18, 0x527B3BEB, 0xA0773615, 0x758BFA0D, 0x8787F7F3, 0x947F9700, 0x66739AFE, /* [15][0x40]*/
        0x5898A2FC, 0xAA94AF02, 0xB96CCFF1, 0x4B60C20F, 0x9E9C0E17, 0x6C9003E9, 0x7F68631A, 0x8D646EE4, /* [15][0x48]*/
        0xD17D8DDB, 0x23718025, 0x3089E0D6, 0xC285ED28, 0x17792130, 0xE5752CCE, 0xF68D4C3D, 0x048141C3, /* [15][0x50]*/
        0x4EBE8A43, 0xBCB287BD, 0xAF4AE74E, 0x5D46EAB0, 0x88BA26A8, 0x7AB62B56, 0x694E4BA5, 0x9B42465B, /* [15][0x58]*/
        0xC75BA564, 0x3557A89A, 0x26AFC869, 0xD4A3C597, 0x015F098F, 0xF3530471, 0xE0AB6482, 0x12A7697C, /* [15][0x60]*/
        0x74D4F382, 0x86D8FE7C, 0x95209E8F, 0x672C9371, 0xB2D05F69, 0x40DC5297, 0x53243264, 0xA1283F9A, /* [15][0x68]*/
        0xFD31DCA5, 0x0F3DD15B, 0x1CC5B1A8, 0xEEC9BC56, 0x3B35704E, 0xC9397DB0, 0xDAC11D43, 0x28CD10BD, /* [15][0x70]*/
        0x62F2DB3D, 0x90FED6C3, 0x8306B630, 0x710ABBCE, 0xA4F677D6, 0x56FA7A28, 0x45021ADB, 0xB70E1725, /* [15][0x78]*/
        0xEB17F41A, 0x191BF9E4, 0x0AE39917, 0xF8EF94E9, 0x2D1358F1, 0xDF1F550F, 0xCCE735FC, 0x3EEB3802, /* [15][0x80]*/
        0xB13145F8, 0x433D4806, 0x50C528F5, 0xA2C9250B, 0x7735E913, 0x8539E4ED, 0x96C1841E, 0x64CD89E0, /* [15][0x88]*/
        0x38D46ADF, 0xCAD86721, 0xD92007D2, 0x2B2C0A2C, 0xFED0C634, 0x0CDCCBCA, 0x1F24AB39, 0xED28A6C7, /* [15][0x90]*/
        0xA7176D47, 0x551B60B9, 0x46E3004A, 0xB4EF0DB4, 0x6113C1AC, 0x931FCC52, 0x80E7ACA1, 0x72EBA15F, /* [15][0x98]*/
        0x2EF24260, 0xDCFE4F9E, 0xCF062F6D, 0x3D0A2293, 0xE8F6EE8B, 0x1AFAE375, 0x09028386, 0xFB0E8E78, /* [15][0xa0]*/
        0x9D7D1486, 0x6F711978, 0x7C89798B, 0x8E857475, 0x5B79B86D, 0xA975B593, 0xBA8DD560, 0x4881D89E, /* [15][0xa8]*/
        0x14983BA1, 0xE694365F, 0xF56C56AC, 0x07605B52, 0xD29C974A, 0x20909AB4, 0x3368FA47, 0xC164F7B9, /* [15][0xb0]*/
        0x8B5B3C39, 0x795731C7, 0x6AAF5134, 0x98A35CCA, 0x4D5F90D2, 0xBF539D2C, 0xACABFDDF, 0x5EA7F021, /* [15][0xb8]*/
        0x02BE131E, 0xF0B21EE0, 0xE34A7E13, 0x114673ED, 0xC4BABFF5, 0x36B6B20B, 0x254ED2F8, 0xD742DF06, /* [15][0xc0]*/
        0xE9A9E704, 0x1BA5EAFA, 0x085D8A09, 0xFA5187F7, 0x2FAD4BEF, 0xDDA14611, 0xCE5926E2, 0x3C552B1C, /* [15][0xc8]*/
        0x604CC823, 0x9240C5DD, 0x81B8A52E, 0x73B4A8D0, 0xA64864C8, 0x54446936, 0x47BC09C5, 0xB5B0043B, /* [15][0xd0]*/
        0xFF8FCFBB, 0x0D83C245, 0x1E7BA2B6, 0xEC77AF48, 0x398B6350, 0xCB876EAE, 0xD87F0E5D, 0x2A7303A3, /* [15][0xd8]*/
        0x766AE09C, 0x8466ED62, 0x979E8D91, 0x6592806F, 0xB06E4C77, 0x42624189, 0x519A217A, 0xA3962C84, /* [15][0xe0]*/
        0xC5E5B67A, 0x37E9BB84, 0x2411DB77, 0xD61DD689, 0x03E11A91, 0xF1ED176F, 0xE215779C, 0x10197A62, /* [15][0xe8]*/
        0x4C00995D, 0xBE0C94A3, 0xADF4F450, 0x5FF8F9AE, 0x8A0435B6, 0x78083848, 0x6BF058BB, 0x99FC5545, /* [15][0xf0]*/
        0xD3C39EC5, 0x21CF933B, 0x3237F3C8, 0xC03BFE36, 0x15C7322E, 0xE7CB3FD0, 0xF4335F23, 0x063F52DD, /* [15][0xf8]*/
        0x5A26B1E2, 0xA82ABC1C, 0xBBD2DCEF, 0x49DED111, 0x9C221D09, 0x6E2E10F7, 0x7DD67004, 0x8FDA7DFA  /* [15][0x100]*/
    }};

/* private (static) function factoring out byte-by-byte CRC computation using just one slice of the lookup table*/
static uint32_t s_crc_generic_sb1(const uint8_t *input, int length, uint32_t crc, const uint32_t *table_ptr) {
    uint32_t(*table)[16][256] = (uint32_t(*)[16][256])table_ptr;
    while (length-- > 0) {
        crc = (crc >> 8) ^ (*table)[0][(crc & 0xff) ^ *input++];
    }
    return crc;
}

/* The inner loops of the CRC functions that process large blocks of data work best when input is aligned*/
/* This function begins processing input data one byte at a time until the input pointer is 4-byte aligned*/
/* Advances the input pointer and reduces the length (both passed by reference)*/
static inline uint32_t s_crc_generic_align(
    const uint8_t **input,
    int *length,
    uint32_t crc,
    const uint32_t *table_ptr) {

    /* Get the 4-byte memory alignment of our input buffer by looking at the least significant 2 bits*/
    size_t input_alignment = ((size_t)*input) & 0x3;

    /* Compute the number of input bytes that precede the first 4-byte aligned block (will be in range 0-3)*/
    size_t leading = (4 - input_alignment) & 0x3;

    /* Determine what's left without the leading input bytes (might be negative)*/
    size_t remaining = *length - leading;

    /* Process unaligned leading input bytes one at a time*/
    if (leading && remaining > 0) {
        crc = s_crc_generic_sb1(*input, (uint32_t)leading, crc, table_ptr);
        *input += leading;
        *length -= (int)leading;
    }

    return crc;
}

/* private (static) function to compute a generic slice-by-4 CRC using the specified lookup table (4 table slices)*/
static uint32_t s_crc_generic_sb4(const uint8_t *input, int length, uint32_t crc, const uint32_t *table_ptr) {
    const uint32_t *current = (const uint32_t *)input;
    int remaining = length;
    uint32_t(*table)[16][256] = (uint32_t(*)[16][256])table_ptr;

    while (remaining >= 4) {
        crc ^= aws_bswap32_if_be(*current++);
        crc = (*table)[3][crc & 0xff] ^ (*table)[2][(crc >> 8) & 0xff] ^ (*table)[1][(crc >> 16) & 0xff] ^
              (*table)[0][crc >> 24];
        remaining -= 4;
    }

    return s_crc_generic_sb1(&input[length - remaining], remaining, crc, table_ptr);
}

/* private (static) function to compute a generic slice-by-8 CRC using the specified lookup table (8 table slices)*/
static uint32_t s_crc_generic_sb8(const uint8_t *input, int length, uint32_t crc, const uint32_t *table_ptr) {
    const uint32_t *current = (const uint32_t *)input;
    int remaining = length;
    uint32_t(*table)[16][256] = (uint32_t(*)[16][256])table_ptr;

    while (remaining >= 8) {
        uint32_t c1 = aws_bswap32_if_be(*current++) ^ crc;
        uint32_t c2 = aws_bswap32_if_be(*current++);
        uint32_t t1 = (*table)[7][c1 & 0xff] ^ (*table)[6][(c1 >> 8) & 0xff] ^ (*table)[5][(c1 >> 16) & 0xff] ^
                      (*table)[4][(c1 >> 24) & 0xff];
        uint32_t t2 = (*table)[3][c2 & 0xff] ^ (*table)[2][(c2 >> 8) & 0xff] ^ (*table)[1][(c2 >> 16) & 0xff] ^
                      (*table)[0][(c2 >> 24) & 0xff];
        crc = t1 ^ t2;
        remaining -= 8;
    }
    return s_crc_generic_sb4(&input[length - remaining], remaining, crc, table_ptr);
}

/* private (static) function to compute a generic slice-by-16 CRC using the specified lookup table (all 16 table
 * slices)*/
static uint32_t s_crc_generic_sb16(const uint8_t *input, int length, uint32_t crc, const uint32_t *table_ptr) {
    const uint32_t *current = (const uint32_t *)input;
    int remaining = length;
    uint32_t(*table)[16][256] = (uint32_t(*)[16][256])table_ptr;

    while (remaining >= 16) {
        uint32_t c1 = aws_bswap32_if_be(*current++) ^ crc;
        uint32_t c2 = aws_bswap32_if_be(*current++);
        uint32_t c3 = aws_bswap32_if_be(*current++);
        uint32_t c4 = aws_bswap32_if_be(*current++);
        uint32_t t1 = (*table)[15][c1 & 0xff] ^ (*table)[14][(c1 >> 8) & 0xff] ^ (*table)[13][(c1 >> 16) & 0xff] ^
                      (*table)[12][(c1 >> 24) & 0xff];
        uint32_t t2 = (*table)[11][c2 & 0xff] ^ (*table)[10][(c2 >> 8) & 0xff] ^ (*table)[9][(c2 >> 16) & 0xff] ^
                      (*table)[8][(c2 >> 24) & 0xff];
        uint32_t t3 = (*table)[7][c3 & 0xff] ^ (*table)[6][(c3 >> 8) & 0xff] ^ (*table)[5][(c3 >> 16) & 0xff] ^
                      (*table)[4][(c3 >> 24) & 0xff];
        uint32_t t4 = (*table)[3][c4 & 0xff] ^ (*table)[2][(c4 >> 8) & 0xff] ^ (*table)[1][(c4 >> 16) & 0xff] ^
                      (*table)[0][(c4 >> 24) & 0xff];
        crc = t1 ^ t2 ^ t3 ^ t4;
        remaining -= 16;
    }
    return s_crc_generic_sb4(&input[length - remaining], remaining, crc, table_ptr);
}

static uint32_t s_crc32_no_slice(const uint8_t *input, int length, uint32_t previousCrc32) {
    return ~s_crc_generic_sb1(input, length, ~previousCrc32, &CRC32_TABLE[0][0]);
}

/* Computes CRC32 (Ethernet, gzip, et. al.) using slice-by-4. */
static uint32_t s_crc32_sb4(const uint8_t *input, int length, uint32_t previousCrc32) {
    uint32_t crc = s_crc_generic_align(&input, &length, ~previousCrc32, &CRC32_TABLE[0][0]);
    return ~s_crc_generic_sb4(input, length, crc, &CRC32_TABLE[0][0]);
}

/* Computes CRC32 (Ethernet, gzip, et. al.) using slice-by-8. */
static uint32_t s_crc32_sb8(const uint8_t *input, int length, uint32_t previousCrc32) {
    uint32_t crc = s_crc_generic_align(&input, &length, ~previousCrc32, &CRC32_TABLE[0][0]);
    return ~s_crc_generic_sb8(input, length, crc, &CRC32_TABLE[0][0]);
}

/* Computes CRC32 (Ethernet, gzip, et. al.) using slice-by-16. */
static uint32_t s_crc32_sb16(const uint8_t *input, int length, uint32_t previousCrc32) {
    uint32_t crc = s_crc_generic_align(&input, &length, ~previousCrc32, &CRC32_TABLE[0][0]);
    return ~s_crc_generic_sb16(input, length, crc, &CRC32_TABLE[0][0]);
}

static uint32_t s_crc32c_no_slice(const uint8_t *input, int length, uint32_t previousCrc32c) {
    return ~s_crc_generic_sb1(input, length, ~previousCrc32c, &CRC32C_TABLE[0][0]);
}

/* Computes the Castagnoli CRC32c (iSCSI) using slice-by-4. */
static uint32_t s_crc32c_sb4(const uint8_t *input, int length, uint32_t previousCrc32) {
    uint32_t crc = s_crc_generic_align(&input, &length, ~previousCrc32, &CRC32C_TABLE[0][0]);
    return ~s_crc_generic_sb4(input, length, crc, &CRC32C_TABLE[0][0]);
}

/* Computes the Castagnoli CRC32c (iSCSI) using slice-by-8. */
static uint32_t s_crc32c_sb8(const uint8_t *input, int length, uint32_t previousCrc32) {
    uint32_t crc = s_crc_generic_align(&input, &length, ~previousCrc32, &CRC32C_TABLE[0][0]);
    return ~s_crc_generic_sb8(input, length, crc, &CRC32C_TABLE[0][0]);
}

/* Computes the Castagnoli CRC32c (iSCSI) using slice-by-16. */
static uint32_t s_crc32c_sb16(const uint8_t *input, int length, uint32_t previousCrc32) {
    uint32_t crc = s_crc_generic_align(&input, &length, ~previousCrc32, &CRC32C_TABLE[0][0]);
    return ~s_crc_generic_sb16(input, length, crc, &CRC32C_TABLE[0][0]);
}

/**
 * Computes the Ethernet, gzip CRC32 of the specified data buffer.
 * Pass 0 in the previousCrc32 parameter as an initial value unless continuing to update a running crc in a subsequent
 * call
 */
uint32_t aws_checksums_crc32_sw(const uint8_t *input, int length, uint32_t previousCrc32) {
    if (length >= 16) {
        return s_crc32_sb16(input, length, previousCrc32);
    }

    if (length >= 8) {
        return s_crc32_sb8(input, length, previousCrc32);
    }

    if (length >= 4) {
        return s_crc32_sb4(input, length, previousCrc32);
    }

    return s_crc32_no_slice(input, length, previousCrc32);
}

/**
 * Computes the Castagnoli iSCSI CRC32c of the specified data buffer.
 * Pass 0 in the previousCrc32c parameter as an initial value unless continuing to update a running crc in a subsequent
 * call
 */
uint32_t aws_checksums_crc32c_sw(const uint8_t *input, int length, uint32_t previousCrc32c) {
    if (length >= 16) {
        return s_crc32c_sb16(input, length, previousCrc32c);
    }

    if (length >= 8) {
        return s_crc32c_sb8(input, length, previousCrc32c);
    }

    if (length >= 4) {
        return s_crc32c_sb4(input, length, previousCrc32c);
    }

    return s_crc32c_no_slice(input, length, previousCrc32c);
}
