/*
 * Copyright (c) 2024
 *      Tim Woodall. All rights reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * SPDX short identifier: BSD-2-Clause
 */

#pragma once

#include <inttypes.h>
#include <memory>
#include <thread>
#include <vector>
#include <condition_variable>
#include <mutex>
#include <cstddef>
#include <chrono>
#include <unistd.h>
#include <iostream>
#include <ostream>

#define DEBUG(tape, f) do { if (tape.debug) *tape.debug << f << std::endl; } while(0)

/*
 * Data is stored on disk as a single file "tape" of repeated 4K BlockTable containing record indices followed by the record data for each index.
 *
 * The first BlockTable has 1011 index entries, it also holds the tape position. Re-opening a file behaves like a non-rewinding tape.
 * Subsequent BlockTables have 1019 index entries.
 * Each index entry is a 32 bits value meaning as follows:
 *     00000000 00...00 (All bits zero) indicates an EOF marker (a read of this block returns a size zero to the reader). The data is a 64bit blockcount of the file.
 *     0000000x xx...xx 1 to 2^25-1 indicates records of 1-2^25-1 (inclusive) bytes of data. (Data is zlib compressed even if the compressed data is bigger than the original)
 *     00000010 xx...xx 2^25 to 2^25+2^24-1 indicates a block of all bytes zero of size 1 to 2^24. No data is written.
 *
 * All other values are currently unused and assumed unused.
 */
struct FakeTapeTimer;
class FakeTape {
	public:
		/* Exposed for tests - there is no reason to use these */
		static constexpr int recsPerBlock = 1019;
		static constexpr int recsInFirstBlock = 1011;
		void trace(bool s) { trace_ = s; }
		bool trace() { return trace_; }
		std::unique_ptr<std::ostream> debug;

		enum class Response : char {
			OK = 'A',
			ERROR = 'E',
			FATAL = 'F'
		};

	private:
		bool trace_ = false;
#pragma pack(push,1)
		struct BlockTable {
			public:
				static constexpr size_t max_block_size = 1<<24; /* 16 MB */

				uint32_t crc = 0;
				uint32_t count = 0;
				uint64_t prevSize = 0;	/* MSB of this indicates whether the previous block contains an EOF marker */
				uint32_t prevCount = 0;
				bool zeroblock(uint32_t idx) const {
					return recdata.size_[idx] & zeroBlockFlag;
				}
				uint32_t size(uint32_t idx) const {
					uint32_t s = recdata.size_[idx];
					if (zeroblock(idx))
						return (s&zeroSizeMask)+1;
					else
						return s;
				}
				uint32_t disksize(uint32_t idx) const {
					if (zeroblock(idx))
						return 0;
					if (size(idx))
						return size(idx);
					return sizeof(uint64_t);
				}
				void setsize(uint32_t idx, uint32_t sz, bool zeroblock) {
					/* sz=0 is invalid when zeroblock is true. We do not check for this case */
					if (zeroblock) {
						sz -= 1;
						sz |= zeroBlockFlag;
					}
					recdata.size_[idx] = sz;
				}
				uint32_t maxblocks() { return (prevSize?sizeof(recdata.size_):sizeof(recdata.firstblock.size_))/sizeof(uint32_t); }
			private:
				static constexpr int zeroSizeMask = (1<<24)-1;
				static constexpr int zeroBlockFlag = 1<<25;
			public:
				union {
					friend struct BlockTable;
					private:
						uint32_t size_[recsPerBlock] = {0};
					public:
						struct {
							friend struct BlockTable;
							private:
								uint32_t size_[recsInFirstBlock];
							public:
								uint32_t cbt_index;
								uint32_t filenum;
								uint64_t fileblock;
								uint64_t blockno;
								int64_t blockTablePos;
						} firstblock;
				} recdata;
				static_assert(sizeof(recdata.size_) == sizeof(recdata.firstblock));
		};
#pragma pack(pop)
		static_assert(sizeof(BlockTable)==4096);

		BlockTable cbt;	/* The current block table we are accessing the tape in */
		uint32_t cbt_index;	/* index into size - a read from the tape reads this block */
		uint64_t block = 0;
		uint32_t filenum = 0;
		uint64_t fileblock = 0;
		int fd = -1;
		bool writing = false;
		bool written = false;

		static size_t max_block_size;

		/*
		 * A tape consists of a BlockTable followed by count blocks
		 * each block is of size[n]
		 * A new blocktable is found after each 1023 blocks.
		 * Zero sized blocks are allowed
		 * prevSize is the size of the previous block including the BlockTable.
		 */

		struct worker_context {
			std::thread t;
			std::vector<uint8_t> in;
			std::vector<uint8_t> out;
			Response response;
			std::condition_variable block_written;
			size_t size;
		};
		static constexpr size_t maxWorkers = 3;
		worker_context context[maxWorkers];

		/* note that these do not wrap and need to be used modulo maxworkers */
		size_t inBufIdx = 0;	/* place to queue jobs in context */
		size_t joinIdx = 0;		/* Threads that need joining */
		size_t writeBufIdx = 0;	/* next job to write to disk */
		void worker(size_t);
		std::mutex filemutex;
		std::condition_variable block_written;

		std::unique_ptr<FakeTapeTimer> writeDataImpl_time;
		std::unique_ptr<FakeTapeTimer> writeDataImpl_wait_time;

		static uint32_t crc32(const BlockTable*);
		/* these should never fail - only a corrupted file or a bug */
		bool readBlockTable();
		bool seekBlockTable();
		bool writeBlockTable();
		bool writeNewBlockTable();
		/* These can fail, for example seeking past the end of the file */
		Response advanceOneBlock();
		Response reverseOneBlock();
		Response blockSeekFwd(size_t);
		Response blockSeekBwd(size_t);
		Response closefd(const char* wmsg);
		Response writeDataImpl(const uint8_t* data, size_t size, bool async);
		Response join();
	public:
		~FakeTape() { close(); }
		Response open(const std::string& filename, int flags);
		void close();
		Response rewind();
		Response erase();
		size_t getPos() const { return block; }
		uint32_t getFileNo() const { return filenum; }
		size_t getFileBlock() const { return fileblock; }
		Response blockSeek(size_t offset);
		Response writeData(const uint8_t* data, size_t size);
		Response writeDataAsync(const uint8_t* data, size_t size);
		Response sync();
		Response readData(uint8_t* data, size_t maxlen, size_t* readlen);
		Response fsr(size_t count);
		Response bsr(size_t count);

		static Response setMaxBlockSize(size_t v) {
			if (v == (size_t)-1) v = BlockTable::max_block_size;
			if (v > BlockTable::max_block_size)
				return Response::ERROR;
			max_block_size = v;
			return Response::OK;
		}
		static size_t getMaxBlockSize() { return max_block_size; }

		off_t getDiskPosForTest() const { return lseek(fd, 0, SEEK_CUR); }
};

struct FakeTapeTimer {
	long long us = 0;
	long long operator+=(long long e) { return us += e; }
	FakeTape& tape;
	const std::string name;
	FakeTapeTimer(FakeTape& tape_, const std::string n) : tape(tape_), name(n) {}
	~FakeTapeTimer() {
		if(tape.trace()) DEBUG(tape, name << " " << us / 1000000.0);
	}
};

struct CalcTime {
	FakeTapeTimer& t_;
	std::chrono::time_point<std::chrono::high_resolution_clock> s_;

	CalcTime(FakeTapeTimer& timer) : t_(timer) { s_ = std::chrono::high_resolution_clock::now(); }
	~CalcTime() {
		auto e = std::chrono::high_resolution_clock::now() - s_;
		t_ += std::chrono::duration_cast<std::chrono::microseconds>(e).count();
	}
};

/* vim: set sw=8 sts=8 ts=8 noexpandtab: */
