      SUBROUTINE slEPV ( DATE, PH, VH, PB, VB )
*+
*  - - - -
*   E P V
*  - - - -
*
*  Earth position and velocity, heliocentric and barycentric, with
*  respect to the Barycentric Celestial Reference System.
*
*  Given:
*     DATE      d         date, TDB Modified Julian Date (Note 1)
*
*  Returned:
*     PH        d(3)      heliocentric Earth position (AU)
*     VH        d(3)      heliocentric Earth velocity (AU,AU/day)
*     PB        d(3)      barycentric Earth position (AU)
*     VB        d(3)      barycentric Earth velocity (AU/day)
*
*  Notes:
*
*  1) The date is TDB as an MJD (=JD-2400000.5).  TT can be used instead
*     of TDB in most applications.
*
*  2) On return, the arrays PH, VH, PV, PB contain the following:
*
*        PH(1)    x       }
*        PH(2)    y       } heliocentric position, AU
*        PH(3)    z       }
*
*        VH(1)    xdot    }
*        VH(2)    ydot    } heliocentric velocity, AU/d
*        VH(3)    zdot    }
*
*        PB(1)    x       }
*        PB(2)    y       } barycentric position, AU
*        PB(3)    z       }
*
*        VB(1)    xdot    }
*        VB(2)    ydot    } barycentric velocity, AU/d
*        VB(3)    zdot    }
*
*     The vectors are with respect to the Barycentric Celestial
*     Reference System (BCRS); velocities are in AU per TDB day.
*
*  3) The routine is a SIMPLIFIED SOLUTION from the planetary theory
*     VSOP2000 (X. Moisson, P. Bretagnon, 2001, Celes. Mechanics &
*     Dyn. Astron., 80, 3/4, 205-213) and is an adaptation of original
*     Fortran code supplied by P. Bretagnon (private comm., 2000).
*
*  4) Comparisons over the time span 1900-2100 with this simplified
*     solution and the JPL DE405 ephemeris give the following results:
*
*                                RMS    max
*           Heliocentric:
*              position error    3.7   11.2   km
*              velocity error    1.4    5.0   mm/s
*
*           Barycentric:
*              position error    4.6   13.4   km
*              velocity error    1.4    4.9   mm/s
*
*     The results deteriorate outside this time span.
*
*  5) The routine slEVP is faster but less accurate.  The present
*     routine targets the case where high accuracy is more important
*     than CPU time, yet the extra complication of reading a pre-
*     computed ephemeris is not justified.
*
*  Last revision:   7 April 2005
*
*  Copyright P.T.Wallace.  All rights reserved.
*
*
*  License:
*    This program is free software; you can redistribute it and/or modify
*    it under the terms of the GNU General Public License as published by
*    the Free Software Foundation; either version 2 of the License, or
*    (at your option) any later version.
*
*    This program is distributed in the hope that it will be useful,
*    but WITHOUT ANY WARRANTY; without even the implied warranty of
*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*    GNU General Public License for more details.
*
*    You should have received a copy of the GNU General Public License
*    along with this program (see SLA_CONDITIONS); if not, write to the
*    Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
*    Boston, MA  02110-1301  USA
*
*  Copyright (C) 1995 Association of Universities for Research in Astronomy Inc.
*-----------------------------------------------------------------------

      IMPLICIT NONE

      DOUBLE PRECISION DATE, PH(3), VH(3), PB(3), VB(3)

      DOUBLE PRECISION T, T2, XYZ, XYZD, A, B, C, CT, P, CP,
     :                 HP(3), HV(3), BP(3), BV(3), X, Y, Z

      INTEGER I, J, K

*  Days per Julian year
      DOUBLE PRECISION DJY
      PARAMETER ( DJY = 365.25D0 )

*  Reference epoch (J2000), MJD
      DOUBLE PRECISION DJM0
      PARAMETER ( DJM0 = 51544.5D0 )
*
*  Matrix elements for orienting the analytical model to DE405/ICRF.
*
*  The corresponding Euler angles are:
*
*                        d  '  "
*    1st rotation    -  23 26 21.4091 about the x-axis  (obliquity)
*    2nd rotation    +         0.0475 about the z-axis  (RA offset)
*
*  These were obtained empirically, by comparisons with DE405 over
*  1900-2100.
*
      DOUBLE PRECISION AM12, AM13, AM21, AM22, AM23, AM32, AM33
      PARAMETER ( AM12 = +0.000000211284D0,
     :            AM13 = -0.000000091603D0,
     :            AM21 = -0.000000230286D0,
     :            AM22 = +0.917482137087D0,
     :            AM23 = -0.397776982902D0,
     :            AM32 = +0.397776982902D0,
     :            AM33 = +0.917482137087D0 )

*  ----------------------
*  Ephemeris Coefficients
*  ----------------------
*
*  The coefficients are stored in arrays of dimension (3,n,3).  There
*  are separate sets of arrays for (i) the Sun to Earth vector and
*  (ii) the Solar-System barycenter to Sun vector.  Each of these two
*  sets contains separate arrays for the terms (n in number) in each
*  power of time (in Julian years since J2000): T^0, T^1 and T^2.
*  Within each array, all the Cartesian x-components, elements (i,j,1),
*  appear first, followed by all the y-components, elements (i,j,2) and
*  finally all the z-components, elements (i,j,3).  At the lowest level
*  are groups of three coefficients.  The first coefficient in each
*  group, element (1,j,k), is the amplitude of the term, the second,
*  element (2,j,k), is the phase and the third, element (3,j,k), is the
*  frequency.
*
*  The naming scheme is such that a block
*
*     DOUBLE PRECISION bn(3,Mbn,3)
*
*  applies to body b and time exponent n:
*
*    . b can be either E (Earth with respect to Sun) or S (Sun with
*      respect to Solar-System Barycenter)
*
*    . n can be 0, 1 or 2, for T^0, T^1 or T^2
*
*  For example, array E2(3,ME2,3) contains the coefficients for
*  the T^2 terms for the Sun-to-Earth vector.
*
*  There is no requirement for the X, Y and Z models for a particular
*  block to use the same number of coefficients.  The number actually
*  used is parameterized, the number of terms being used called NbnX,
*  NbnY, and NbnZ respectively.  The parameter Mbn is the biggest of
*  the three, and defines the array size.  Unused elements are not
*  initialized and are never accessed.
*

      INTEGER NE0(3), NE0X, NE0Y, NE0Z, ME0,
     :        NE1(3), NE1X, NE1Y, NE1Z, ME1,
     :        NE2(3), NE2X, NE2Y, NE2Z, ME2,
     :        NS0(3), NS0X, NS0Y, NS0Z, MS0,
     :        NS1(3), NS1X, NS1Y, NS1Z, MS1,
     :        NS2(3), NS2X, NS2Y, NS2Z, MS2

      PARAMETER ( NE0X = 501, NE0Y = 501, NE0Z = 137, ME0 = NE0X,
     :            NE1X =  79, NE1Y =  80, NE1Z =  12, ME1 = NE1Y,
     :            NE2X =   5, NE2Y =   5, NE2Z =   3, ME2 = NE2X,
     :            NS0X = 212, NS0Y = 213, NS0Z =  69, MS0 = NS0Y,
     :            NS1X =  50, NS1Y =  50, NS1Z =  14, MS1 = NS1X,
     :            NS2X =   9, NS2Y =   9, NS2Z =   2, MS2 = NS2X )

      DOUBLE PRECISION E0(3,ME0,3), E1(3,ME1,3), E2(3,ME2,3),
     :                 S0(3,MS0,3), S1(3,MS1,3), S2(3,MS2,3)

      DATA NE0 / NE0X, NE0Y, NE0Z /
      DATA NE1 / NE1X, NE1Y, NE1Z /
      DATA NE2 / NE2X, NE2Y, NE2Z /
      DATA NS0 / NS0X, NS0Y, NS0Z /
      DATA NS1 / NS1X, NS1Y, NS1Z /
      DATA NS2 / NS2X, NS2Y, NS2Z /

*  Sun-to-Earth, T^0, X
      DATA ((E0(I,J,1),I=1,3),J=  1, 10) /
     :  0.9998292878132D+00, 0.1753485171504D+01, 0.6283075850446D+01,
     :  0.8352579567414D-02, 0.1710344404582D+01, 0.1256615170089D+02,
     :  0.5611445335148D-02, 0.0000000000000D+00, 0.0000000000000D+00,
     :  0.1046664295572D-03, 0.1667225416770D+01, 0.1884922755134D+02,
     :  0.3110842534677D-04, 0.6687513390251D+00, 0.8399684731857D+02,
     :  0.2552413503550D-04, 0.5830637358413D+00, 0.5296909721118D+00,
     :  0.2137207845781D-04, 0.1092330954011D+01, 0.1577343543434D+01,
     :  0.1680240182951D-04, 0.4955366134987D+00, 0.6279552690824D+01,
     :  0.1679012370795D-04, 0.6153014091901D+01, 0.6286599010068D+01,
     :  0.1445526946777D-04, 0.3472744100492D+01, 0.2352866153506D+01 /
      DATA ((E0(I,J,1),I=1,3),J= 11, 20) /
     :  0.1091038246184D-04, 0.3689845786119D+01, 0.5223693906222D+01,
     :  0.9344399733932D-05, 0.6073934645672D+01, 0.1203646072878D+02,
     :  0.8993182910652D-05, 0.3175705249069D+01, 0.1021328554739D+02,
     :  0.5665546034116D-05, 0.2152484672246D+01, 0.1059381944224D+01,
     :  0.6844146703035D-05, 0.1306964099750D+01, 0.5753384878334D+01,
     :  0.7346610905565D-05, 0.4354980070466D+01, 0.3981490189893D+00,
     :  0.6815396474414D-05, 0.2218229211267D+01, 0.4705732307012D+01,
     :  0.6112787253053D-05, 0.5384788425458D+01, 0.6812766822558D+01,
     :  0.4518120711239D-05, 0.6087604012291D+01, 0.5884926831456D+01,
     :  0.4521963430706D-05, 0.1279424524906D+01, 0.6256777527156D+01 /
      DATA ((E0(I,J,1),I=1,3),J= 21, 30) /
     :  0.4497426764085D-05, 0.5369129144266D+01, 0.6309374173736D+01,
     :  0.4062190566959D-05, 0.5436473303367D+00, 0.6681224869435D+01,
     :  0.5412193480192D-05, 0.7867838528395D+00, 0.7755226100720D+00,
     :  0.5469839049386D-05, 0.1461440311134D+01, 0.1414349524433D+02,
     :  0.5205264083477D-05, 0.4432944696116D+01, 0.7860419393880D+01,
     :  0.2149759935455D-05, 0.4502237496846D+01, 0.1150676975667D+02,
     :  0.2279109618501D-05, 0.1239441308815D+01, 0.7058598460518D+01,
     :  0.2259282939683D-05, 0.3272430985331D+01, 0.4694002934110D+01,
     :  0.2558950271319D-05, 0.2265471086404D+01, 0.1216800268190D+02,
     :  0.2561581447555D-05, 0.1454740653245D+01, 0.7099330490126D+00 /
      DATA ((E0(I,J,1),I=1,3),J= 31, 40) /
     :  0.1781441115440D-05, 0.2962068630206D+01, 0.7962980379786D+00,
     :  0.1612005874644D-05, 0.1473255041006D+01, 0.5486777812467D+01,
     :  0.1818630667105D-05, 0.3743903293447D+00, 0.6283008715021D+01,
     :  0.1818601377529D-05, 0.6274174354554D+01, 0.6283142985870D+01,
     :  0.1554475925257D-05, 0.1624110906816D+01, 0.2513230340178D+02,
     :  0.2090948029241D-05, 0.5852052276256D+01, 0.1179062909082D+02,
     :  0.2000176345460D-05, 0.4072093298513D+01, 0.1778984560711D+02,
     :  0.1289535917759D-05, 0.5217019331069D+01, 0.7079373888424D+01,
     :  0.1281135307881D-05, 0.4802054538934D+01, 0.3738761453707D+01,
     :  0.1518229005692D-05, 0.8691914742502D+00, 0.2132990797783D+00 /
      DATA ((E0(I,J,1),I=1,3),J= 41, 50) /
     :  0.9450128579027D-06, 0.4601859529950D+01, 0.1097707878456D+02,
     :  0.7781119494996D-06, 0.1844352816694D+01, 0.8827390247185D+01,
     :  0.7733407759912D-06, 0.3582790154750D+01, 0.5507553240374D+01,
     :  0.7350644318120D-06, 0.2695277788230D+01, 0.1589072916335D+01,
     :  0.6535928827023D-06, 0.3651327986142D+01, 0.1176985366291D+02,
     :  0.6324624183656D-06, 0.2241302375862D+01, 0.6262300422539D+01,
     :  0.6298565300557D-06, 0.4407122406081D+01, 0.6303851278352D+01,
     :  0.8587037089179D-06, 0.3024307223119D+01, 0.1672837615881D+03,
     :  0.8299954491035D-06, 0.6192539428237D+01, 0.3340612434717D+01,
     :  0.6311263503401D-06, 0.2014758795416D+01, 0.7113454667900D-02 /
      DATA ((E0(I,J,1),I=1,3),J= 51, 60) /
     :  0.6005646745452D-06, 0.3399500503397D+01, 0.4136910472696D+01,
     :  0.7917715109929D-06, 0.2493386877837D+01, 0.6069776770667D+01,
     :  0.7556958099685D-06, 0.4159491740143D+01, 0.6496374930224D+01,
     :  0.6773228244949D-06, 0.4034162934230D+01, 0.9437762937313D+01,
     :  0.5370708577847D-06, 0.1562219163734D+01, 0.1194447056968D+01,
     :  0.5710804266203D-06, 0.2662730803386D+01, 0.6282095334605D+01,
     :  0.5709824583726D-06, 0.3985828430833D+01, 0.6284056366286D+01,
     :  0.5143950896447D-06, 0.1308144688689D+01, 0.6290189305114D+01,
     :  0.5088010604546D-06, 0.5352817214804D+01, 0.6275962395778D+01,
     :  0.4960369085172D-06, 0.2644267922349D+01, 0.6127655567643D+01 /
      DATA ((E0(I,J,1),I=1,3),J= 61, 70) /
     :  0.4803137891183D-06, 0.4008844192080D+01, 0.6438496133249D+01,
     :  0.5731747768225D-06, 0.3794550174597D+01, 0.3154687086868D+01,
     :  0.4735947960579D-06, 0.6107118308982D+01, 0.3128388763578D+01,
     :  0.4808348796625D-06, 0.4771458618163D+01, 0.8018209333619D+00,
     :  0.4115073743137D-06, 0.3327111335159D+01, 0.8429241228195D+01,
     :  0.5230575889287D-06, 0.5305708551694D+01, 0.1336797263425D+02,
     :  0.5133977889215D-06, 0.5784230738814D+01, 0.1235285262111D+02,
     :  0.5065815825327D-06, 0.2052064793679D+01, 0.1185621865188D+02,
     :  0.4339831593868D-06, 0.3644994195830D+01, 0.1726015463500D+02,
     :  0.3952928638953D-06, 0.4930376436758D+01, 0.5481254917084D+01 /
      DATA ((E0(I,J,1),I=1,3),J= 71, 80) /
     :  0.4898498111942D-06, 0.4542084219731D+00, 0.9225539266174D+01,
     :  0.4757490209328D-06, 0.3161126388878D+01, 0.5856477690889D+01,
     :  0.4727701669749D-06, 0.6214993845446D+00, 0.2544314396739D+01,
     :  0.3800966681863D-06, 0.3040132339297D+01, 0.4265981595566D+00,
     :  0.3257301077939D-06, 0.8064977360087D+00, 0.3930209696940D+01,
     :  0.3255810528674D-06, 0.1974147981034D+01, 0.2146165377750D+01,
     :  0.3252029748187D-06, 0.2845924913135D+01, 0.4164311961999D+01,
     :  0.3255505635308D-06, 0.3017900824120D+01, 0.5088628793478D+01,
     :  0.2801345211990D-06, 0.6109717793179D+01, 0.1256967486051D+02,
     :  0.3688987740970D-06, 0.2911550235289D+01, 0.1807370494127D+02 /
      DATA ((E0(I,J,1),I=1,3),J= 81, 90) /
     :  0.2475153429458D-06, 0.2179146025856D+01, 0.2629832328990D-01,
     :  0.3033457749150D-06, 0.1994161050744D+01, 0.4535059491685D+01,
     :  0.2186743763110D-06, 0.5125687237936D+01, 0.1137170464392D+02,
     :  0.2764777032774D-06, 0.4822646860252D+00, 0.1256262854127D+02,
     :  0.2199028768592D-06, 0.4637633293831D+01, 0.1255903824622D+02,
     :  0.2046482824760D-06, 0.1467038733093D+01, 0.7084896783808D+01,
     :  0.2611209147507D-06, 0.3044718783485D+00, 0.7143069561767D+02,
     :  0.2286079656818D-06, 0.4764220356805D+01, 0.8031092209206D+01,
     :  0.1855071202587D-06, 0.3383637774428D+01, 0.1748016358760D+01,
     :  0.2324669506784D-06, 0.6189088449251D+01, 0.1831953657923D+02 /
      DATA ((E0(I,J,1),I=1,3),J= 91,100) /
     :  0.1709528015688D-06, 0.5874966729774D+00, 0.4933208510675D+01,
     :  0.2168156875828D-06, 0.4302994009132D+01, 0.1044738781244D+02,
     :  0.2106675556535D-06, 0.3800475419891D+01, 0.7477522907414D+01,
     :  0.1430213830465D-06, 0.1294660846502D+01, 0.2942463415728D+01,
     :  0.1388396901944D-06, 0.4594797202114D+01, 0.8635942003952D+01,
     :  0.1922258844190D-06, 0.4943044543591D+00, 0.1729818233119D+02,
     :  0.1888460058292D-06, 0.2426943912028D+01, 0.1561374759853D+03,
     :  0.1789449386107D-06, 0.1582973303499D+00, 0.1592596075957D+01,
     :  0.1360803685374D-06, 0.5197240440504D+01, 0.1309584267300D+02,
     :  0.1504038014709D-06, 0.3120360916217D+01, 0.1649636139783D+02 /
      DATA ((E0(I,J,1),I=1,3),J=101,110) /
     :  0.1382769533389D-06, 0.6164702888205D+01, 0.7632943190217D+01,
     :  0.1438059769079D-06, 0.1437423770979D+01, 0.2042657109477D+02,
     :  0.1326303260037D-06, 0.3609688799679D+01, 0.1213955354133D+02,
     :  0.1159244950540D-06, 0.5463018167225D+01, 0.5331357529664D+01,
     :  0.1433118149136D-06, 0.6028909912097D+01, 0.7342457794669D+01,
     :  0.1234623148594D-06, 0.3109645574997D+01, 0.6279485555400D+01,
     :  0.1233949875344D-06, 0.3539359332866D+01, 0.6286666145492D+01,
     :  0.9927196061299D-07, 0.1259321569772D+01, 0.7234794171227D+01,
     :  0.1242302191316D-06, 0.1065949392609D+01, 0.1511046609763D+02,
     :  0.1098402195201D-06, 0.2192508743837D+01, 0.1098880815746D+02 /
      DATA ((E0(I,J,1),I=1,3),J=111,120) /
     :  0.1158191395315D-06, 0.4054411278650D+01, 0.5729506548653D+01,
     :  0.9048475596241D-07, 0.5429764748518D+01, 0.9623688285163D+01,
     :  0.8889853269023D-07, 0.5046586206575D+01, 0.6148010737701D+01,
     :  0.1048694242164D-06, 0.2628858030806D+01, 0.6836645152238D+01,
     :  0.1112308378646D-06, 0.4177292719907D+01, 0.1572083878776D+02,
     :  0.8631729709901D-07, 0.1601345232557D+01, 0.6418140963190D+01,
     :  0.8527816951664D-07, 0.2463888997513D+01, 0.1471231707864D+02,
     :  0.7892139456991D-07, 0.3154022088718D+01, 0.2118763888447D+01,
     :  0.1051782905236D-06, 0.4795035816088D+01, 0.1349867339771D+01,
     :  0.1048219943164D-06, 0.2952983395230D+01, 0.5999216516294D+01 /
      DATA ((E0(I,J,1),I=1,3),J=121,130) /
     :  0.7435760775143D-07, 0.5420547991464D+01, 0.6040347114260D+01,
     :  0.9869574106949D-07, 0.3695646753667D+01, 0.6566935184597D+01,
     :  0.9156886364226D-07, 0.3922675306609D+01, 0.5643178611111D+01,
     :  0.7006834356188D-07, 0.1233968624861D+01, 0.6525804586632D+01,
     :  0.9806170182601D-07, 0.1919542280684D+01, 0.2122839202813D+02,
     :  0.9052289673607D-07, 0.4615902724369D+01, 0.4690479774488D+01,
     :  0.7554200867893D-07, 0.1236863719072D+01, 0.1253985337760D+02,
     :  0.8215741286498D-07, 0.3286800101559D+00, 0.1097355562493D+02,
     :  0.7185178575397D-07, 0.5880942158367D+01, 0.6245048154254D+01,
     :  0.7130726476180D-07, 0.7674871987661D+00, 0.6321103546637D+01 /
      DATA ((E0(I,J,1),I=1,3),J=131,140) /
     :  0.6650894461162D-07, 0.6987129150116D+00, 0.5327476111629D+01,
     :  0.7396888823688D-07, 0.3576824794443D+01, 0.5368044267797D+00,
     :  0.7420588884775D-07, 0.5033615245369D+01, 0.2354323048545D+02,
     :  0.6141181642908D-07, 0.9449927045673D+00, 0.1296430071988D+02,
     :  0.6373557924058D-07, 0.6206342280341D+01, 0.9517183207817D+00,
     :  0.6359474329261D-07, 0.5036079095757D+01, 0.1990745094947D+01,
     :  0.5740173582646D-07, 0.6105106371350D+01, 0.9555997388169D+00,
     :  0.7019864084602D-07, 0.7237747359018D+00, 0.5225775174439D+00,
     :  0.6398054487042D-07, 0.3976367969666D+01, 0.2407292145756D+02,
     :  0.7797092650498D-07, 0.4305423910623D+01, 0.2200391463820D+02 /
      DATA ((E0(I,J,1),I=1,3),J=141,150) /
     :  0.6466760000900D-07, 0.3500136825200D+01, 0.5230807360890D+01,
     :  0.7529417043890D-07, 0.3514779246100D+01, 0.1842262939178D+02,
     :  0.6924571140892D-07, 0.2743457928679D+01, 0.1554202828031D+00,
     :  0.6220798650222D-07, 0.2242598118209D+01, 0.1845107853235D+02,
     :  0.5870209391853D-07, 0.2332832707527D+01, 0.6398972393349D+00,
     :  0.6263953473888D-07, 0.2191105358956D+01, 0.6277552955062D+01,
     :  0.6257781390012D-07, 0.4457559396698D+01, 0.6288598745829D+01,
     :  0.5697304945123D-07, 0.3499234761404D+01, 0.1551045220144D+01,
     :  0.6335438746791D-07, 0.6441691079251D+00, 0.5216580451554D+01,
     :  0.6377258441152D-07, 0.2252599151092D+01, 0.5650292065779D+01 /
      DATA ((E0(I,J,1),I=1,3),J=151,160) /
     :  0.6484841818165D-07, 0.1992812417646D+01, 0.1030928125552D+00,
     :  0.4735551485250D-07, 0.3744672082942D+01, 0.1431416805965D+02,
     :  0.4628595996170D-07, 0.1334226211745D+01, 0.5535693017924D+00,
     :  0.6258152336933D-07, 0.4395836159154D+01, 0.2608790314060D+02,
     :  0.6196171366594D-07, 0.2587043007997D+01, 0.8467247584405D+02,
     :  0.6159556952126D-07, 0.4782499769128D+01, 0.2394243902548D+03,
     :  0.4987741172394D-07, 0.7312257619924D+00, 0.7771377146812D+02,
     :  0.5459280703142D-07, 0.3001376372532D+01, 0.6179983037890D+01,
     :  0.4863461189999D-07, 0.3767222128541D+01, 0.9027992316901D+02,
     :  0.5349912093158D-07, 0.3663594450273D+01, 0.6386168663001D+01 /
      DATA ((E0(I,J,1),I=1,3),J=161,170) /
     :  0.5673725607806D-07, 0.4331187919049D+01, 0.6915859635113D+01,
     :  0.4745485060512D-07, 0.5816195745518D+01, 0.6282970628506D+01,
     :  0.4745379005326D-07, 0.8323672435672D+00, 0.6283181072386D+01,
     :  0.4049002796321D-07, 0.3785023976293D+01, 0.6254626709878D+01,
     :  0.4247084014515D-07, 0.2378220728783D+01, 0.7875671926403D+01,
     :  0.4026912363055D-07, 0.2864103423269D+01, 0.6311524991013D+01,
     :  0.4062935011774D-07, 0.2415408595975D+01, 0.3634620989887D+01,
     :  0.5347771048509D-07, 0.3343479309801D+01, 0.2515860172507D+02,
     :  0.4829494136505D-07, 0.2821742398262D+01, 0.5760498333002D+01,
     :  0.4342554404599D-07, 0.5624662458712D+01, 0.7238675589263D+01 /
      DATA ((E0(I,J,1),I=1,3),J=171,180) /
     :  0.4021599184361D-07, 0.5557250275009D+00, 0.1101510648075D+02,
     :  0.4104900474558D-07, 0.3296691780005D+01, 0.6709674010002D+01,
     :  0.4376532905131D-07, 0.3814443999443D+01, 0.6805653367890D+01,
     :  0.3314590480650D-07, 0.3560229189250D+01, 0.1259245002418D+02,
     :  0.3232421839643D-07, 0.5185389180568D+01, 0.1066495398892D+01,
     :  0.3541176318876D-07, 0.3921381909679D+01, 0.9917696840332D+01,
     :  0.3689831242681D-07, 0.4190658955386D+01, 0.1192625446156D+02,
     :  0.3890605376774D-07, 0.5546023371097D+01, 0.7478166569050D-01,
     :  0.3038559339780D-07, 0.6231032794494D+01, 0.1256621883632D+02,
     :  0.3137083969782D-07, 0.6207063419190D+01, 0.4292330755499D+01 /
      DATA ((E0(I,J,1),I=1,3),J=181,190) /
     :  0.4024004081854D-07, 0.1195257375713D+01, 0.1334167431096D+02,
     :  0.3300234879283D-07, 0.1804694240998D+01, 0.1057540660594D+02,
     :  0.3635399155575D-07, 0.5597811343500D+01, 0.6208294184755D+01,
     :  0.3032668691356D-07, 0.3191059366530D+01, 0.1805292951336D+02,
     :  0.2809652069058D-07, 0.4094348032570D+01, 0.3523159621801D-02,
     :  0.3696955383823D-07, 0.5219282738794D+01, 0.5966683958112D+01,
     :  0.3562894142503D-07, 0.1037247544554D+01, 0.6357857516136D+01,
     :  0.3510598524148D-07, 0.1430020816116D+01, 0.6599467742779D+01,
     :  0.3617736142953D-07, 0.3002911403677D+01, 0.6019991944201D+01,
     :  0.2624524910730D-07, 0.2437046757292D+01, 0.6702560555334D+01 /
      DATA ((E0(I,J,1),I=1,3),J=191,200) /
     :  0.2535824204490D-07, 0.1581594689647D+01, 0.3141537925223D+02,
     :  0.3519787226257D-07, 0.5379863121521D+01, 0.2505706758577D+03,
     :  0.2578406709982D-07, 0.4904222639329D+01, 0.1673046366289D+02,
     :  0.3423887981473D-07, 0.3646448997315D+01, 0.6546159756691D+01,
     :  0.2776083886467D-07, 0.3307829300144D+01, 0.1272157198369D+02,
     :  0.3379592818379D-07, 0.1747541251125D+01, 0.1494531617769D+02,
     :  0.3050255426284D-07, 0.1784689432607D-01, 0.4732030630302D+01,
     :  0.2652378350236D-07, 0.4420055276260D+01, 0.5863591145557D+01,
     :  0.2374498173768D-07, 0.3629773929208D+01, 0.2388894113936D+01,
     :  0.2716451255140D-07, 0.3079623706780D+01, 0.1202934727411D+02 /
      DATA ((E0(I,J,1),I=1,3),J=201,210) /
     :  0.3038583699229D-07, 0.3312487903507D+00, 0.1256608456547D+02,
     :  0.2220681228760D-07, 0.5265520401774D+01, 0.1336244973887D+02,
     :  0.3044156540912D-07, 0.4766664081250D+01, 0.2908881142201D+02,
     :  0.2731859923561D-07, 0.5069146530691D+01, 0.1391601904066D+02,
     :  0.2285603018171D-07, 0.5954935112271D+01, 0.6076890225335D+01,
     :  0.2025006454555D-07, 0.4061789589267D+01, 0.4701116388778D+01,
     :  0.2012597519804D-07, 0.2485047705241D+01, 0.6262720680387D+01,
     :  0.2003406962258D-07, 0.4163779209320D+01, 0.6303431020504D+01,
     :  0.2207863441371D-07, 0.6923839133828D+00, 0.6489261475556D+01,
     :  0.2481374305624D-07, 0.5944173595676D+01, 0.1204357418345D+02 /
      DATA ((E0(I,J,1),I=1,3),J=211,220) /
     :  0.2130923288870D-07, 0.4641013671967D+01, 0.5746271423666D+01,
     :  0.2446370543391D-07, 0.6125796518757D+01, 0.1495633313810D+00,
     :  0.1932492759052D-07, 0.2234572324504D+00, 0.1352175143971D+02,
     :  0.2600122568049D-07, 0.4281012405440D+01, 0.4590910121555D+01,
     :  0.2431754047488D-07, 0.1429943874870D+00, 0.1162474756779D+01,
     :  0.1875902869209D-07, 0.9781803816948D+00, 0.6279194432410D+01,
     :  0.1874381139426D-07, 0.5670368130173D+01, 0.6286957268481D+01,
     :  0.2156696047173D-07, 0.2008985006833D+01, 0.1813929450232D+02,
     :  0.1965076182484D-07, 0.2566186202453D+00, 0.4686889479442D+01,
     :  0.2334816372359D-07, 0.4408121891493D+01, 0.1002183730415D+02 /
      DATA ((E0(I,J,1),I=1,3),J=221,230) /
     :  0.1869937408802D-07, 0.5272745038656D+01, 0.2427287361862D+00,
     :  0.2436236460883D-07, 0.4407720479029D+01, 0.9514313292143D+02,
     :  0.1761365216611D-07, 0.1943892315074D+00, 0.1351787002167D+02,
     :  0.2156289480503D-07, 0.1418570924545D+01, 0.6037244212485D+01,
     :  0.2164748979255D-07, 0.4724603439430D+01, 0.2301353951334D+02,
     :  0.2222286670853D-07, 0.2400266874598D+01, 0.1266924451345D+02,
     :  0.2070901414929D-07, 0.5230348028732D+01, 0.6528907488406D+01,
     :  0.1792745177020D-07, 0.2099190328945D+01, 0.6819880277225D+01,
     :  0.1841802068445D-07, 0.3467527844848D+00, 0.6514761976723D+02,
     :  0.1578401631718D-07, 0.7098642356340D+00, 0.2077542790660D-01 /
      DATA ((E0(I,J,1),I=1,3),J=231,240) /
     :  0.1561690152531D-07, 0.5943349620372D+01, 0.6272439236156D+01,
     :  0.1558591045463D-07, 0.7040653478980D+00, 0.6293712464735D+01,
     :  0.1737356469576D-07, 0.4487064760345D+01, 0.1765478049437D+02,
     :  0.1434755619991D-07, 0.2993391570995D+01, 0.1102062672231D+00,
     :  0.1482187806654D-07, 0.2278049198251D+01, 0.1052268489556D+01,
     :  0.1424812827089D-07, 0.1682114725827D+01, 0.1311972100268D+02,
     :  0.1380282448623D-07, 0.3262668602579D+01, 0.1017725758696D+02,
     :  0.1811481244566D-07, 0.3187771221777D+01, 0.1887552587463D+02,
     :  0.1504446185696D-07, 0.5650162308647D+01, 0.7626583626240D-01,
     :  0.1740776154137D-07, 0.5487068607507D+01, 0.1965104848470D+02 /
      DATA ((E0(I,J,1),I=1,3),J=241,250) /
     :  0.1374339536251D-07, 0.5745688172201D+01, 0.6016468784579D+01,
     :  0.1761377477704D-07, 0.5748060203659D+01, 0.2593412433514D+02,
     :  0.1535138225795D-07, 0.6226848505790D+01, 0.9411464614024D+01,
     :  0.1788140543676D-07, 0.6189318878563D+01, 0.3301902111895D+02,
     :  0.1375002807996D-07, 0.5371812884394D+01, 0.6327837846670D+00,
     :  0.1242115758632D-07, 0.1471687569712D+01, 0.3894181736510D+01,
     :  0.1450977333938D-07, 0.4143836662127D+01, 0.1277945078067D+02,
     :  0.1297579575023D-07, 0.9003477661957D+00, 0.6549682916313D+01,
     :  0.1462667934821D-07, 0.5760505536428D+01, 0.1863592847156D+02,
     :  0.1381774374799D-07, 0.1085471729463D+01, 0.2379164476796D+01 /
      DATA ((E0(I,J,1),I=1,3),J=251,260) /
     :  0.1682333169307D-07, 0.5409870870133D+01, 0.1620077269078D+02,
     :  0.1190812918837D-07, 0.1397205174601D+01, 0.1149965630200D+02,
     :  0.1221434762106D-07, 0.9001804809095D+00, 0.1257326515556D+02,
     :  0.1549934644860D-07, 0.4262528275544D+01, 0.1820933031200D+02,
     :  0.1252138953050D-07, 0.1411642012027D+01, 0.6993008899458D+01,
     :  0.1237078905387D-07, 0.2844472403615D+01, 0.2435678079171D+02,
     :  0.1446953389615D-07, 0.5295835522223D+01, 0.3813291813120D-01,
     :  0.1388446457170D-07, 0.4969428135497D+01, 0.2458316379602D+00,
     :  0.1019339179228D-07, 0.2491369561806D+01, 0.6112403035119D+01,
     :  0.1258880815343D-07, 0.4679426248976D+01, 0.5429879531333D+01 /
      DATA ((E0(I,J,1),I=1,3),J=261,270) /
     :  0.1297768238261D-07, 0.1074509953328D+01, 0.1249137003520D+02,
     :  0.9913505718094D-08, 0.4735097918224D+01, 0.6247047890016D+01,
     :  0.9830453155969D-08, 0.4158649187338D+01, 0.6453748665772D+01,
     :  0.1192615865309D-07, 0.3438208613699D+01, 0.6290122169689D+01,
     :  0.9835874798277D-08, 0.1913300781229D+01, 0.6319103810876D+01,
     :  0.9639087569277D-08, 0.9487683644125D+00, 0.8273820945392D+01,
     :  0.1175716107001D-07, 0.3228141664287D+01, 0.6276029531202D+01,
     :  0.1018926508678D-07, 0.2216607854300D+01, 0.1254537627298D+02,
     :  0.9500087869225D-08, 0.2625116459733D+01, 0.1256517118505D+02,
     :  0.9664192916575D-08, 0.5860562449214D+01, 0.6259197520765D+01 /
      DATA ((E0(I,J,1),I=1,3),J=271,280) /
     :  0.9612858712203D-08, 0.7885682917381D+00, 0.6306954180126D+01,
     :  0.1117645675413D-07, 0.3932148831189D+01, 0.1779695906178D+02,
     :  0.1158864052160D-07, 0.9995605521691D+00, 0.1778273215245D+02,
     :  0.9021043467028D-08, 0.5263769742673D+01, 0.6172869583223D+01,
     :  0.8836134773563D-08, 0.1496843220365D+01, 0.1692165728891D+01,
     :  0.1045872200691D-07, 0.7009039517214D+00, 0.2204125344462D+00,
     :  0.1211463487798D-07, 0.4041544938511D+01, 0.8257698122054D+02,
     :  0.8541990804094D-08, 0.1447586692316D+01, 0.6393282117669D+01,
     :  0.1038720703636D-07, 0.4594249718112D+00, 0.1550861511662D+02,
     :  0.1126722351445D-07, 0.3925550579036D+01, 0.2061856251104D+00 /
      DATA ((E0(I,J,1),I=1,3),J=281,290) /
     :  0.8697373859631D-08, 0.4411341856037D+01, 0.9491756770005D+00,
     :  0.8869380028441D-08, 0.2402659724813D+01, 0.3903911373650D+01,
     :  0.9247014693258D-08, 0.1401579743423D+01, 0.6267823317922D+01,
     :  0.9205062930950D-08, 0.5245978000814D+01, 0.6298328382969D+01,
     :  0.8000745038049D-08, 0.3590803356945D+01, 0.2648454860559D+01,
     :  0.9168973650819D-08, 0.2470150501679D+01, 0.1498544001348D+03,
     :  0.1075444949238D-07, 0.1328606161230D+01, 0.3694923081589D+02,
     :  0.7817298525817D-08, 0.6162256225998D+01, 0.4804209201333D+01,
     :  0.9541469226356D-08, 0.3942568967039D+01, 0.1256713221673D+02,
     :  0.9821910122027D-08, 0.2360246287233D+00, 0.1140367694411D+02 /
      DATA ((E0(I,J,1),I=1,3),J=291,300) /
     :  0.9897822023777D-08, 0.4619805634280D+01, 0.2280573557157D+02,
     :  0.7737289283765D-08, 0.3784727847451D+01, 0.7834121070590D+01,
     :  0.9260204034710D-08, 0.2223352487601D+01, 0.2787043132925D+01,
     :  0.7320252888486D-08, 0.1288694636874D+01, 0.6282655592598D+01,
     :  0.7319785780946D-08, 0.5359869567774D+01, 0.6283496108294D+01,
     :  0.7147219933778D-08, 0.5516616675856D+01, 0.1725663147538D+02,
     :  0.7946502829878D-08, 0.2630459984567D+01, 0.1241073141809D+02,
     :  0.9001711808932D-08, 0.2849815827227D+01, 0.6281591679874D+01,
     :  0.8994041507257D-08, 0.3795244450750D+01, 0.6284560021018D+01,
     :  0.8298582787358D-08, 0.5236413127363D+00, 0.1241658836951D+02 /
      DATA ((E0(I,J,1),I=1,3),J=301,310) /
     :  0.8526596520710D-08, 0.4794605424426D+01, 0.1098419223922D+02,
     :  0.8209822103197D-08, 0.1578752370328D+01, 0.1096996532989D+02,
     :  0.6357049861094D-08, 0.5708926113761D+01, 0.1596186371003D+01,
     :  0.7370473179049D-08, 0.3842402530241D+01, 0.4061219149443D+01,
     :  0.7232154664726D-08, 0.3067548981535D+01, 0.1610006857377D+03,
     :  0.6328765494903D-08, 0.1313930030069D+01, 0.1193336791622D+02,
     :  0.8030064908595D-08, 0.3488500408886D+01, 0.8460828644453D+00,
     :  0.6275464259232D-08, 0.1532061626198D+01, 0.8531963191132D+00,
     :  0.7051897446325D-08, 0.3285859929993D+01, 0.5849364236221D+01,
     :  0.6161593705428D-08, 0.1477341999464D+01, 0.5573142801433D+01 /
      DATA ((E0(I,J,1),I=1,3),J=311,320) /
     :  0.7754683957278D-08, 0.1586118663096D+01, 0.8662240327241D+01,
     :  0.5889928990701D-08, 0.1304887868803D+01, 0.1232342296471D+02,
     :  0.5705756047075D-08, 0.4555333589350D+01, 0.1258692712880D+02,
     :  0.5964178808332D-08, 0.3001762842062D+01, 0.5333900173445D+01,
     :  0.6712446027467D-08, 0.4886780007595D+01, 0.1171295538178D+02,
     :  0.5941809275464D-08, 0.4701509603824D+01, 0.9779108567966D+01,
     :  0.5466993627395D-08, 0.4588357817278D+01, 0.1884211409667D+02,
     :  0.6340512090980D-08, 0.1164543038893D+01, 0.5217580628120D+02,
     :  0.6325505710045D-08, 0.3919171259645D+01, 0.1041998632314D+02,
     :  0.6164789509685D-08, 0.2143828253542D+01, 0.6151533897323D+01 /
      DATA ((E0(I,J,1),I=1,3),J=321,330) /
     :  0.5263330812430D-08, 0.6066564434241D+01, 0.1885275071096D+02,
     :  0.5597087780221D-08, 0.2926316429472D+01, 0.4337116142245D+00,
     :  0.5396556236817D-08, 0.3244303591505D+01, 0.6286362197481D+01,
     :  0.5396615148223D-08, 0.3404304703662D+01, 0.6279789503410D+01,
     :  0.7091832443341D-08, 0.8532377803192D+00, 0.4907302013889D+01,
     :  0.6572352589782D-08, 0.4901966774419D+01, 0.1176433076753D+02,
     :  0.5960236060795D-08, 0.1874672315797D+01, 0.1422690933580D-01,
     :  0.5125480043511D-08, 0.3735726064334D+01, 0.1245594543367D+02,
     :  0.5928241866410D-08, 0.4502033899935D+01, 0.6414617803568D+01,
     :  0.5249600357424D-08, 0.4372334799878D+01, 0.1151388321134D+02 /
      DATA ((E0(I,J,1),I=1,3),J=331,340) /
     :  0.6059171276087D-08, 0.2581617302908D+01, 0.6062663316000D+01,
     :  0.5295235081662D-08, 0.2974811513158D+01, 0.3496032717521D+01,
     :  0.5820561875933D-08, 0.1796073748244D+00, 0.2838593341516D+00,
     :  0.4754696606440D-08, 0.1981998136973D+01, 0.3104930017775D+01,
     :  0.6385053548955D-08, 0.2559174171605D+00, 0.6133512519065D+01,
     :  0.6589828273941D-08, 0.2750967106776D+01, 0.4087944051283D+02,
     :  0.5383376567189D-08, 0.6325947523578D+00, 0.2248384854122D+02,
     :  0.5928941683538D-08, 0.1672304519067D+01, 0.1581959461667D+01,
     :  0.4816060709794D-08, 0.3512566172575D+01, 0.9388005868221D+01,
     :  0.6003381586512D-08, 0.5610932219189D+01, 0.5326786718777D+01 /
      DATA ((E0(I,J,1),I=1,3),J=341,350) /
     :  0.5504225393105D-08, 0.4037501131256D+01, 0.6503488384892D+01,
     :  0.5353772620129D-08, 0.6122774968240D+01, 0.1735668374386D+03,
     :  0.5786253768544D-08, 0.5527984999515D+01, 0.1350651127443D+00,
     :  0.5065706702002D-08, 0.9980765573624D+00, 0.1248988586463D+02,
     :  0.5972838885276D-08, 0.6044489493203D+01, 0.2673594526851D+02,
     :  0.5323585877961D-08, 0.3924265998147D+01, 0.4171425416666D+01,
     :  0.5210772682858D-08, 0.6220111376901D+01, 0.2460261242967D+02,
     :  0.4726549040535D-08, 0.3716043206862D+01, 0.7232251527446D+01,
     :  0.6029425105059D-08, 0.8548704071116D+00, 0.3227113045244D+03,
     :  0.4481542826513D-08, 0.1426925072829D+01, 0.5547199253223D+01 /
      DATA ((E0(I,J,1),I=1,3),J=351,360) /
     :  0.5836024505068D-08, 0.7135651752625D-01, 0.7285056171570D+02,
     :  0.4137046613272D-08, 0.5330767643283D+01, 0.1087398597200D+02,
     :  0.5171977473924D-08, 0.4494262335353D+00, 0.1884570439172D+02,
     :  0.5694429833732D-08, 0.2952369582215D+01, 0.9723862754494D+02,
     :  0.4009158925298D-08, 0.3500003416535D+01, 0.6244942932314D+01,
     :  0.4784939596873D-08, 0.6196709413181D+01, 0.2929661536378D+02,
     :  0.3983725022610D-08, 0.5103690031897D+01, 0.4274518229222D+01,
     :  0.3870535232462D-08, 0.3187569587401D+01, 0.6321208768577D+01,
     :  0.5140501213951D-08, 0.1668924357457D+01, 0.1232032006293D+02,
     :  0.3849034819355D-08, 0.4445722510309D+01, 0.1726726808967D+02 /
      DATA ((E0(I,J,1),I=1,3),J=361,370) /
     :  0.4002383075060D-08, 0.5226224152423D+01, 0.7018952447668D+01,
     :  0.3890719543549D-08, 0.4371166550274D+01, 0.1491901785440D+02,
     :  0.4887084607881D-08, 0.5973556689693D+01, 0.1478866649112D+01,
     :  0.3739939287592D-08, 0.2089084714600D+01, 0.6922973089781D+01,
     :  0.5031925918209D-08, 0.4658371936827D+01, 0.1715706182245D+02,
     :  0.4387748764954D-08, 0.4825580552819D+01, 0.2331413144044D+03,
     :  0.4147398098865D-08, 0.3739003524998D+01, 0.1376059875786D+02,
     :  0.3719089993586D-08, 0.1148941386536D+01, 0.6297302759782D+01,
     :  0.3934238461056D-08, 0.1559893008343D+01, 0.7872148766781D+01,
     :  0.3672471375622D-08, 0.5516145383612D+01, 0.6268848941110D+01 /
      DATA ((E0(I,J,1),I=1,3),J=371,380) /
     :  0.3768911277583D-08, 0.6116053700563D+01, 0.4157198507331D+01,
     :  0.4033388417295D-08, 0.5076821746017D+01, 0.1567108171867D+02,
     :  0.3764194617832D-08, 0.8164676232075D+00, 0.3185192151914D+01,
     :  0.4840628226284D-08, 0.1360479453671D+01, 0.1252801878276D+02,
     :  0.4949443923785D-08, 0.2725622229926D+01, 0.1617106187867D+03,
     :  0.4117393089971D-08, 0.6054459628492D+00, 0.5642198095270D+01,
     :  0.3925754020428D-08, 0.8570462135210D+00, 0.2139354194808D+02,
     :  0.3630551757923D-08, 0.3552067338279D+01, 0.6294805223347D+01,
     :  0.3627274802357D-08, 0.3096565085313D+01, 0.6271346477544D+01,
     :  0.3806143885093D-08, 0.6367751709777D+00, 0.1725304118033D+02 /
      DATA ((E0(I,J,1),I=1,3),J=381,390) /
     :  0.4433254641565D-08, 0.4848461503937D+01, 0.7445550607224D+01,
     :  0.3712319846576D-08, 0.1331950643655D+01, 0.4194847048887D+00,
     :  0.3849847534783D-08, 0.4958368297746D+00, 0.9562891316684D+00,
     :  0.3483955430165D-08, 0.2237215515707D+01, 0.1161697602389D+02,
     :  0.3961912730982D-08, 0.3332402188575D+01, 0.2277943724828D+02,
     :  0.3419978244481D-08, 0.5785600576016D+01, 0.1362553364512D+02,
     :  0.3329417758177D-08, 0.9812676559709D-01, 0.1685848245639D+02,
     :  0.4207206893193D-08, 0.9494780468236D+00, 0.2986433403208D+02,
     :  0.3268548976410D-08, 0.1739332095686D+00, 0.5749861718712D+01,
     :  0.3321880082685D-08, 0.1423354800666D+01, 0.6279143387820D+01 /
      DATA ((E0(I,J,1),I=1,3),J=391,400) /
     :  0.4503173010852D-08, 0.2314972675293D+00, 0.1385561574497D+01,
     :  0.4316599090954D-08, 0.1012646782616D+00, 0.4176041334900D+01,
     :  0.3283493323850D-08, 0.5233306881265D+01, 0.6287008313071D+01,
     :  0.3164033542343D-08, 0.4005597257511D+01, 0.2099539292909D+02,
     :  0.4159720956725D-08, 0.5365676242020D+01, 0.5905702259363D+01,
     :  0.3565176892217D-08, 0.4284440620612D+01, 0.3932462625300D-02,
     :  0.3514440950221D-08, 0.4270562636575D+01, 0.7335344340001D+01,
     :  0.3540596871909D-08, 0.5953553201060D+01, 0.1234573916645D+02,
     :  0.2960769905118D-08, 0.1115180417718D+01, 0.2670964694522D+02,
     :  0.2962213739684D-08, 0.3863811918186D+01, 0.6408777551755D+00 /
      DATA ((E0(I,J,1),I=1,3),J=401,410) /
     :  0.3883556700251D-08, 0.1268617928302D+01, 0.6660449441528D+01,
     :  0.2919225516346D-08, 0.4908605223265D+01, 0.1375773836557D+01,
     :  0.3115158863370D-08, 0.3744519976885D+01, 0.3802769619140D-01,
     :  0.4099438144212D-08, 0.4173244670532D+01, 0.4480965020977D+02,
     :  0.2899531858964D-08, 0.5910601428850D+01, 0.2059724391010D+02,
     :  0.3289733429855D-08, 0.2488050078239D+01, 0.1081813534213D+02,
     :  0.3933075612875D-08, 0.1122363652883D+01, 0.3773735910827D+00,
     :  0.3021403764467D-08, 0.4951973724904D+01, 0.2982630633589D+02,
     :  0.2798598949757D-08, 0.5117057845513D+01, 0.1937891852345D+02,
     :  0.3397421302707D-08, 0.6104159180476D+01, 0.6923953605621D+01 /
      DATA ((E0(I,J,1),I=1,3),J=411,420) /
     :  0.3720398002179D-08, 0.1184933429829D+01, 0.3066615496545D+02,
     :  0.3598484186267D-08, 0.3505282086105D+01, 0.6147450479709D+01,
     :  0.3694594027310D-08, 0.2286651088141D+01, 0.2636725487657D+01,
     :  0.2680444152969D-08, 0.1871816775482D+00, 0.6816289982179D+01,
     :  0.3497574865641D-08, 0.3143251755431D+01, 0.6418701221183D+01,
     :  0.3130274129494D-08, 0.2462167316018D+01, 0.1235996607578D+02,
     :  0.3241119069551D-08, 0.4256374004686D+01, 0.1652265972112D+02,
     :  0.2601960842061D-08, 0.4970362941425D+01, 0.1045450126711D+02,
     :  0.2690601527504D-08, 0.2372657824898D+01, 0.3163918923335D+00,
     :  0.2908688152664D-08, 0.4232652627721D+01, 0.2828699048865D+02 /
      DATA ((E0(I,J,1),I=1,3),J=421,430) /
     :  0.3120456131875D-08, 0.3925747001137D+00, 0.2195415756911D+02,
     :  0.3148855423384D-08, 0.3093478330445D+01, 0.1172006883645D+02,
     :  0.3051044261017D-08, 0.5560948248212D+01, 0.6055599646783D+01,
     :  0.2826006876660D-08, 0.5072790310072D+01, 0.5120601093667D+01,
     :  0.3100034191711D-08, 0.4998530231096D+01, 0.1799603123222D+02,
     :  0.2398771640101D-08, 0.2561739802176D+01, 0.6255674361143D+01,
     :  0.2384002842728D-08, 0.4087420284111D+01, 0.6310477339748D+01,
     :  0.2842146517568D-08, 0.2515048217955D+01, 0.5469525544182D+01,
     :  0.2847674371340D-08, 0.5235326497443D+01, 0.1034429499989D+02,
     :  0.2903722140764D-08, 0.1088200795797D+01, 0.6510552054109D+01 /
      DATA ((E0(I,J,1),I=1,3),J=431,440) /
     :  0.3187610710605D-08, 0.4710624424816D+01, 0.1693792562116D+03,
     :  0.3048869992813D-08, 0.2857975896445D+00, 0.8390110365991D+01,
     :  0.2860216950984D-08, 0.2241619020815D+01, 0.2243449970715D+00,
     :  0.2701117683113D-08, 0.6651573305272D-01, 0.6129297044991D+01,
     :  0.2509891590152D-08, 0.1285135324585D+01, 0.1044027435778D+02,
     :  0.2623200252223D-08, 0.2981229834530D+00, 0.6436854655901D+01,
     :  0.2622541669202D-08, 0.6122470726189D+01, 0.9380959548977D+01,
     :  0.2818435667099D-08, 0.4251087148947D+01, 0.5934151399930D+01,
     :  0.2365196797465D-08, 0.3465070460790D+01, 0.2470570524223D+02,
     :  0.2358704646143D-08, 0.5791603815350D+01, 0.8671969964381D+01 /
      DATA ((E0(I,J,1),I=1,3),J=441,450) /
     :  0.2388299481390D-08, 0.4142483772941D+01, 0.7096626156709D+01,
     :  0.1996041217224D-08, 0.2101901889496D+01, 0.1727188400790D+02,
     :  0.2687593060336D-08, 0.1526689456959D+01, 0.7075506709219D+02,
     :  0.2618913670810D-08, 0.2397684236095D+01, 0.6632000300961D+01,
     :  0.2571523050364D-08, 0.5751929456787D+00, 0.6206810014183D+01,
     :  0.2582135006946D-08, 0.5595464352926D+01, 0.4873985990671D+02,
     :  0.2372530190361D-08, 0.5092689490655D+01, 0.1590676413561D+02,
     :  0.2357178484712D-08, 0.4444363527851D+01, 0.3097883698531D+01,
     :  0.2451590394723D-08, 0.3108251687661D+01, 0.6612329252343D+00,
     :  0.2370045949608D-08, 0.2608133861079D+01, 0.3459636466239D+02 /
      DATA ((E0(I,J,1),I=1,3),J=451,460) /
     :  0.2268997267358D-08, 0.3639717753384D+01, 0.2844914056730D-01,
     :  0.1731432137906D-08, 0.1741898445707D+00, 0.2019909489111D+02,
     :  0.1629869741622D-08, 0.3902225646724D+01, 0.3035599730800D+02,
     :  0.2206215801974D-08, 0.4971131250731D+01, 0.6281667977667D+01,
     :  0.2205469554680D-08, 0.1677462357110D+01, 0.6284483723224D+01,
     :  0.2148792362509D-08, 0.4236259604006D+01, 0.1980482729015D+02,
     :  0.1873733657847D-08, 0.5926814998687D+01, 0.2876692439167D+02,
     :  0.2026573758959D-08, 0.4349643351962D+01, 0.2449240616245D+02,
     :  0.1807770325110D-08, 0.5700940482701D+01, 0.2045286941806D+02,
     :  0.1881174408581D-08, 0.6601286363430D+00, 0.2358125818164D+02 /
      DATA ((E0(I,J,1),I=1,3),J=461,470) /
     :  0.1368023671690D-08, 0.2211098592752D+01, 0.2473415438279D+02,
     :  0.1720017916280D-08, 0.4942488551129D+01, 0.1679593901136D+03,
     :  0.1702427665131D-08, 0.1452233856386D+01, 0.3338575901272D+03,
     :  0.1414032510054D-08, 0.5525357721439D+01, 0.1624205518357D+03,
     :  0.1652626045364D-08, 0.4108794283624D+01, 0.8956999012000D+02,
     :  0.1642957769686D-08, 0.7344335209984D+00, 0.5267006960365D+02,
     :  0.1614952403624D-08, 0.3541213951363D+01, 0.3332657872986D+02,
     :  0.1535988291188D-08, 0.4031094072151D+01, 0.3852657435933D+02,
     :  0.1593193738177D-08, 0.4185136203609D+01, 0.2282781046519D+03,
     :  0.1074569126382D-08, 0.1720485636868D+01, 0.8397383534231D+02 /
      DATA ((E0(I,J,1),I=1,3),J=471,480) /
     :  0.1074408214509D-08, 0.2758613420318D+01, 0.8401985929482D+02,
     :  0.9700199670465D-09, 0.4216686842097D+01, 0.7826370942180D+02,
     :  0.1258433517061D-08, 0.2575068876639D+00, 0.3115650189215D+03,
     :  0.1240303229539D-08, 0.4800844956756D+00, 0.1784300471910D+03,
     :  0.9018345948127D-09, 0.3896756361552D+00, 0.5886454391678D+02,
     :  0.1135301432805D-08, 0.3700805023550D+00, 0.7842370451713D+02,
     :  0.9215887951370D-09, 0.4364579276638D+01, 0.1014262087719D+03,
     :  0.1055401054147D-08, 0.2156564222111D+01, 0.5660027930059D+02,
     :  0.1008725979831D-08, 0.5454015785234D+01, 0.4245678405627D+02,
     :  0.7217398104321D-09, 0.1597772562175D+01, 0.2457074661053D+03 /
      DATA ((E0(I,J,1),I=1,3),J=481,490) /
     :  0.6912033134447D-09, 0.5824090621461D+01, 0.1679936946371D+03,
     :  0.6833881523549D-09, 0.3578778482835D+01, 0.6053048899753D+02,
     :  0.4887304205142D-09, 0.3724362812423D+01, 0.9656299901946D+02,
     :  0.5173709754788D-09, 0.5422427507933D+01, 0.2442876000072D+03,
     :  0.4671353097145D-09, 0.2396106924439D+01, 0.1435713242844D+03,
     :  0.5652608439480D-09, 0.2804028838685D+01, 0.8365903305582D+02,
     :  0.5604061331253D-09, 0.1638816006247D+01, 0.8433466158131D+02,
     :  0.4712723365400D-09, 0.8979003224474D+00, 0.3164282286739D+03,
     :  0.4909967465112D-09, 0.3210426725516D+01, 0.4059982187939D+03,
     :  0.4771358267658D-09, 0.5308027211629D+01, 0.1805255418145D+03 /
      DATA ((E0(I,J,1),I=1,3),J=491,500) /
     :  0.3943451445989D-09, 0.2195145341074D+01, 0.2568537517081D+03,
     :  0.3952109120244D-09, 0.5081189491586D+01, 0.2449975330562D+03,
     :  0.3788134594789D-09, 0.4345171264441D+01, 0.1568131045107D+03,
     :  0.3738330190479D-09, 0.2613062847997D+01, 0.3948519331910D+03,
     :  0.3099866678136D-09, 0.2846760817689D+01, 0.1547176098872D+03,
     :  0.2002962716768D-09, 0.4921360989412D+01, 0.2268582385539D+03,
     :  0.2198291338754D-09, 0.1130360117454D+00, 0.1658638954901D+03,
     :  0.1491958330784D-09, 0.4228195232278D+01, 0.2219950288015D+03,
     :  0.1475384076173D-09, 0.3005721811604D+00, 0.3052819430710D+03,
     :  0.1661626624624D-09, 0.7830125621203D+00, 0.2526661704812D+03 /
      DATA ((E0(I,J,1),I=1,3),J=501,NE0X) /
     :  0.9015823460025D-10, 0.3807792942715D+01, 0.4171445043968D+03 /

*  Sun-to-Earth, T^1, X
      DATA ((E1(I,J,1),I=1,3),J=  1, 10) /
     :  0.1234046326004D-05, 0.0000000000000D+00, 0.0000000000000D+00,
     :  0.5150068824701D-06, 0.6002664557501D+01, 0.1256615170089D+02,
     :  0.1290743923245D-07, 0.5959437664199D+01, 0.1884922755134D+02,
     :  0.1068615564952D-07, 0.2015529654209D+01, 0.6283075850446D+01,
     :  0.2079619142538D-08, 0.1732960531432D+01, 0.6279552690824D+01,
     :  0.2078009243969D-08, 0.4915604476996D+01, 0.6286599010068D+01,
     :  0.6206330058856D-09, 0.3616457953824D+00, 0.4705732307012D+01,
     :  0.5989335313746D-09, 0.3802607304474D+01, 0.6256777527156D+01,
     :  0.5958495663840D-09, 0.2845866560031D+01, 0.6309374173736D+01,
     :  0.4866923261539D-09, 0.5213203771824D+01, 0.7755226100720D+00 /
      DATA ((E1(I,J,1),I=1,3),J= 11, 20) /
     :  0.4267785823142D-09, 0.4368189727818D+00, 0.1059381944224D+01,
     :  0.4610675141648D-09, 0.1837249181372D-01, 0.7860419393880D+01,
     :  0.3626989993973D-09, 0.2161590545326D+01, 0.5753384878334D+01,
     :  0.3563071194389D-09, 0.1452631954746D+01, 0.5884926831456D+01,
     :  0.3557015642807D-09, 0.4470593393054D+01, 0.6812766822558D+01,
     :  0.3210412089122D-09, 0.5195926078314D+01, 0.6681224869435D+01,
     :  0.2875473577986D-09, 0.5916256610193D+01, 0.2513230340178D+02,
     :  0.2842913681629D-09, 0.1149902426047D+01, 0.6127655567643D+01,
     :  0.2751248215916D-09, 0.5502088574662D+01, 0.6438496133249D+01,
     :  0.2481432881127D-09, 0.2921989846637D+01, 0.5486777812467D+01 /
      DATA ((E1(I,J,1),I=1,3),J= 21, 30) /
     :  0.2059885976560D-09, 0.3718070376585D+01, 0.7079373888424D+01,
     :  0.2015522342591D-09, 0.5979395259740D+01, 0.6290189305114D+01,
     :  0.1995364084253D-09, 0.6772087985494D+00, 0.6275962395778D+01,
     :  0.1957436436943D-09, 0.2899210654665D+01, 0.5507553240374D+01,
     :  0.1651609818948D-09, 0.6228206482192D+01, 0.1150676975667D+02,
     :  0.1822980550699D-09, 0.1469348746179D+01, 0.1179062909082D+02,
     :  0.1675223159760D-09, 0.3813910555688D+01, 0.7058598460518D+01,
     :  0.1706491764745D-09, 0.3004380506684D+00, 0.7113454667900D-02,
     :  0.1392952362615D-09, 0.1440393973406D+01, 0.7962980379786D+00,
     :  0.1209868266342D-09, 0.4150425791727D+01, 0.4694002934110D+01 /
      DATA ((E1(I,J,1),I=1,3),J= 31, 40) /
     :  0.1009827202611D-09, 0.3290040429843D+01, 0.3738761453707D+01,
     :  0.1047261388602D-09, 0.4229590090227D+01, 0.6282095334605D+01,
     :  0.1047006652004D-09, 0.2418967680575D+01, 0.6284056366286D+01,
     :  0.9609993143095D-10, 0.4627943659201D+01, 0.6069776770667D+01,
     :  0.9590900593873D-10, 0.1894393939924D+01, 0.4136910472696D+01,
     :  0.9146249188071D-10, 0.2010647519562D+01, 0.6496374930224D+01,
     :  0.8545274480290D-10, 0.5529846956226D-01, 0.1194447056968D+01,
     :  0.8224377881194D-10, 0.1254304102174D+01, 0.1589072916335D+01,
     :  0.6183529510410D-10, 0.3360862168815D+01, 0.8827390247185D+01,
     :  0.6259255147141D-10, 0.4755628243179D+01, 0.8429241228195D+01 /
      DATA ((E1(I,J,1),I=1,3),J= 41, 50) /
     :  0.5539291694151D-10, 0.5371746955142D+01, 0.4933208510675D+01,
     :  0.7328259466314D-10, 0.4927699613906D+00, 0.4535059491685D+01,
     :  0.6017835843560D-10, 0.5776682001734D-01, 0.1255903824622D+02,
     :  0.7079827775243D-10, 0.4395059432251D+01, 0.5088628793478D+01,
     :  0.5170358878213D-10, 0.5154062619954D+01, 0.1176985366291D+02,
     :  0.4872301838682D-10, 0.6289611648973D+00, 0.6040347114260D+01,
     :  0.5249869411058D-10, 0.5617272046949D+01, 0.3154687086868D+01,
     :  0.4716172354411D-10, 0.3965901800877D+01, 0.5331357529664D+01,
     :  0.4871214940964D-10, 0.4627507050093D+01, 0.1256967486051D+02,
     :  0.4598076850751D-10, 0.6023631226459D+01, 0.6525804586632D+01 /
      DATA ((E1(I,J,1),I=1,3),J= 51, 60) /
     :  0.4562196089485D-10, 0.4138562084068D+01, 0.3930209696940D+01,
     :  0.4325493872224D-10, 0.1330845906564D+01, 0.7632943190217D+01,
     :  0.5673781176748D-10, 0.2558752615657D+01, 0.5729506548653D+01,
     :  0.3961436642503D-10, 0.2728071734630D+01, 0.7234794171227D+01,
     :  0.5101868209058D-10, 0.4113444965144D+01, 0.6836645152238D+01,
     :  0.5257043167676D-10, 0.6195089830590D+01, 0.8031092209206D+01,
     :  0.5076613989393D-10, 0.2305124132918D+01, 0.7477522907414D+01,
     :  0.3342169352778D-10, 0.5415998155071D+01, 0.1097707878456D+02,
     :  0.3545881983591D-10, 0.3727160564574D+01, 0.4164311961999D+01,
     :  0.3364063738599D-10, 0.2901121049204D+00, 0.1137170464392D+02 /
      DATA ((E1(I,J,1),I=1,3),J= 61, 70) /
     :  0.3357039670776D-10, 0.1652229354331D+01, 0.5223693906222D+01,
     :  0.4307412268687D-10, 0.4938909587445D+01, 0.1592596075957D+01,
     :  0.3405769115435D-10, 0.2408890766511D+01, 0.3128388763578D+01,
     :  0.3001926198480D-10, 0.4862239006386D+01, 0.1748016358760D+01,
     :  0.2778264787325D-10, 0.5241168661353D+01, 0.7342457794669D+01,
     :  0.2676159480666D-10, 0.3423593942199D+01, 0.2146165377750D+01,
     :  0.2954273399939D-10, 0.1881721265406D+01, 0.5368044267797D+00,
     :  0.3309362888795D-10, 0.1931525677349D+01, 0.8018209333619D+00,
     :  0.2810283608438D-10, 0.2414659495050D+01, 0.5225775174439D+00,
     :  0.3378045637764D-10, 0.4238019163430D+01, 0.1554202828031D+00 /
      DATA ((E1(I,J,1),I=1,3),J= 71,NE1X) /
     :  0.2558134979840D-10, 0.1828225235805D+01, 0.5230807360890D+01,
     :  0.2273755578447D-10, 0.5858184283998D+01, 0.7084896783808D+01,
     :  0.2294176037690D-10, 0.4514589779057D+01, 0.1726015463500D+02,
     :  0.2533506099435D-10, 0.2355717851551D+01, 0.5216580451554D+01,
     :  0.2716685375812D-10, 0.2221003625100D+01, 0.8635942003952D+01,
     :  0.2419043435198D-10, 0.5955704951635D+01, 0.4690479774488D+01,
     :  0.2521232544812D-10, 0.1395676848521D+01, 0.5481254917084D+01,
     :  0.2630195021491D-10, 0.5727468918743D+01, 0.2629832328990D-01,
     :  0.2548395840944D-10, 0.2628351859400D-03, 0.1349867339771D+01 /

*  Sun-to-Earth, T^2, X
      DATA ((E2(I,J,1),I=1,3),J=  1,NE2X) /
     : -0.4143818297913D-10, 0.0000000000000D+00, 0.0000000000000D+00,
     :  0.2171497694435D-10, 0.4398225628264D+01, 0.1256615170089D+02,
     :  0.9845398442516D-11, 0.2079720838384D+00, 0.6283075850446D+01,
     :  0.9256833552682D-12, 0.4191264694361D+01, 0.1884922755134D+02,
     :  0.1022049384115D-12, 0.5381133195658D+01, 0.8399684731857D+02 /

*  Sun-to-Earth, T^0, Y
      DATA ((E0(I,J,2),I=1,3),J=  1, 10) /
     :  0.9998921098898D+00, 0.1826583913846D+00, 0.6283075850446D+01,
     : -0.2442700893735D-01, 0.0000000000000D+00, 0.0000000000000D+00,
     :  0.8352929742915D-02, 0.1395277998680D+00, 0.1256615170089D+02,
     :  0.1046697300177D-03, 0.9641423109763D-01, 0.1884922755134D+02,
     :  0.3110841876663D-04, 0.5381140401712D+01, 0.8399684731857D+02,
     :  0.2570269094593D-04, 0.5301016407128D+01, 0.5296909721118D+00,
     :  0.2147389623610D-04, 0.2662510869850D+01, 0.1577343543434D+01,
     :  0.1680344384050D-04, 0.5207904119704D+01, 0.6279552690824D+01,
     :  0.1679117312193D-04, 0.4582187486968D+01, 0.6286599010068D+01,
     :  0.1440512068440D-04, 0.1900688517726D+01, 0.2352866153506D+01 /
      DATA ((E0(I,J,2),I=1,3),J= 11, 20) /
     :  0.1135139664999D-04, 0.5273108538556D+01, 0.5223693906222D+01,
     :  0.9345482571018D-05, 0.4503047687738D+01, 0.1203646072878D+02,
     :  0.9007418719568D-05, 0.1605621059637D+01, 0.1021328554739D+02,
     :  0.5671536712314D-05, 0.5812849070861D+00, 0.1059381944224D+01,
     :  0.7451401861666D-05, 0.2807346794836D+01, 0.3981490189893D+00,
     :  0.6393470057114D-05, 0.6029224133855D+01, 0.5753384878334D+01,
     :  0.6814275881697D-05, 0.6472990145974D+00, 0.4705732307012D+01,
     :  0.6113705628887D-05, 0.3813843419700D+01, 0.6812766822558D+01,
     :  0.4503851367273D-05, 0.4527804370996D+01, 0.5884926831456D+01,
     :  0.4522249141926D-05, 0.5991783029224D+01, 0.6256777527156D+01 /
      DATA ((E0(I,J,2),I=1,3),J= 21, 30) /
     :  0.4501794307018D-05, 0.3798703844397D+01, 0.6309374173736D+01,
     :  0.5514927480180D-05, 0.3961257833388D+01, 0.5507553240374D+01,
     :  0.4062862799995D-05, 0.5256247296369D+01, 0.6681224869435D+01,
     :  0.5414900429712D-05, 0.5499032014097D+01, 0.7755226100720D+00,
     :  0.5463153987424D-05, 0.6173092454097D+01, 0.1414349524433D+02,
     :  0.5071611859329D-05, 0.2870244247651D+01, 0.7860419393880D+01,
     :  0.2195112094455D-05, 0.2952338617201D+01, 0.1150676975667D+02,
     :  0.2279139233919D-05, 0.5951775132933D+01, 0.7058598460518D+01,
     :  0.2278386100876D-05, 0.4845456398785D+01, 0.4694002934110D+01,
     :  0.2559088003308D-05, 0.6945321117311D+00, 0.1216800268190D+02 /
      DATA ((E0(I,J,2),I=1,3),J= 31, 40) /
     :  0.2561079286856D-05, 0.6167224608301D+01, 0.7099330490126D+00,
     :  0.1792755796387D-05, 0.1400122509632D+01, 0.7962980379786D+00,
     :  0.1818715656502D-05, 0.4703347611830D+01, 0.6283142985870D+01,
     :  0.1818744924791D-05, 0.5086748900237D+01, 0.6283008715021D+01,
     :  0.1554518791390D-05, 0.5331008042713D-01, 0.2513230340178D+02,
     :  0.2063265737239D-05, 0.4283680484178D+01, 0.1179062909082D+02,
     :  0.1497613520041D-05, 0.6074207826073D+01, 0.5486777812467D+01,
     :  0.2000617940427D-05, 0.2501426281450D+01, 0.1778984560711D+02,
     :  0.1289731195580D-05, 0.3646340599536D+01, 0.7079373888424D+01,
     :  0.1282657998934D-05, 0.3232864804902D+01, 0.3738761453707D+01 /
      DATA ((E0(I,J,2),I=1,3),J= 41, 50) /
     :  0.1528915968658D-05, 0.5581433416669D+01, 0.2132990797783D+00,
     :  0.1187304098432D-05, 0.5453576453694D+01, 0.9437762937313D+01,
     :  0.7842782928118D-06, 0.2823953922273D+00, 0.8827390247185D+01,
     :  0.7352892280868D-06, 0.1124369580175D+01, 0.1589072916335D+01,
     :  0.6570189360797D-06, 0.2089154042840D+01, 0.1176985366291D+02,
     :  0.6324967590410D-06, 0.6704855581230D+00, 0.6262300422539D+01,
     :  0.6298289872283D-06, 0.2836414855840D+01, 0.6303851278352D+01,
     :  0.6476686465855D-06, 0.4852433866467D+00, 0.7113454667900D-02,
     :  0.8587034651234D-06, 0.1453511005668D+01, 0.1672837615881D+03,
     :  0.8068948788113D-06, 0.9224087798609D+00, 0.6069776770667D+01 /
      DATA ((E0(I,J,2),I=1,3),J= 51, 60) /
     :  0.8353786011661D-06, 0.4631707184895D+01, 0.3340612434717D+01,
     :  0.6009324532132D-06, 0.1829498827726D+01, 0.4136910472696D+01,
     :  0.7558158559566D-06, 0.2588596800317D+01, 0.6496374930224D+01,
     :  0.5809279504503D-06, 0.5516818853476D+00, 0.1097707878456D+02,
     :  0.5374131950254D-06, 0.6275674734960D+01, 0.1194447056968D+01,
     :  0.5711160507326D-06, 0.1091905956872D+01, 0.6282095334605D+01,
     :  0.5710183170746D-06, 0.2415001635090D+01, 0.6284056366286D+01,
     :  0.5144373590610D-06, 0.6020336443438D+01, 0.6290189305114D+01,
     :  0.5103108927267D-06, 0.3775634564605D+01, 0.6275962395778D+01,
     :  0.4960654697891D-06, 0.1073450946756D+01, 0.6127655567643D+01 /
      DATA ((E0(I,J,2),I=1,3),J= 61, 70) /
     :  0.4786385689280D-06, 0.2431178012310D+01, 0.6438496133249D+01,
     :  0.6109911263665D-06, 0.5343356157914D+01, 0.3154687086868D+01,
     :  0.4839898944024D-06, 0.5830833594047D-01, 0.8018209333619D+00,
     :  0.4734822623919D-06, 0.4536080134821D+01, 0.3128388763578D+01,
     :  0.4834741473290D-06, 0.2585090489754D+00, 0.7084896783808D+01,
     :  0.5134858581156D-06, 0.4213317172603D+01, 0.1235285262111D+02,
     :  0.5064004264978D-06, 0.4814418806478D+00, 0.1185621865188D+02,
     :  0.3753476772761D-06, 0.1599953399788D+01, 0.8429241228195D+01,
     :  0.4935264014283D-06, 0.2157417556873D+01, 0.2544314396739D+01,
     :  0.3950929600897D-06, 0.3359394184254D+01, 0.5481254917084D+01 /
      DATA ((E0(I,J,2),I=1,3),J= 71, 80) /
     :  0.4895849789777D-06, 0.5165704376558D+01, 0.9225539266174D+01,
     :  0.4215241688886D-06, 0.2065368800993D+01, 0.1726015463500D+02,
     :  0.3796773731132D-06, 0.1468606346612D+01, 0.4265981595566D+00,
     :  0.3114178142515D-06, 0.3615638079474D+01, 0.2146165377750D+01,
     :  0.3260664220838D-06, 0.4417134922435D+01, 0.4164311961999D+01,
     :  0.3976996123008D-06, 0.4700866883004D+01, 0.5856477690889D+01,
     :  0.2801459672924D-06, 0.4538902060922D+01, 0.1256967486051D+02,
     :  0.3638931868861D-06, 0.1334197991475D+01, 0.1807370494127D+02,
     :  0.2487013269476D-06, 0.3749275558275D+01, 0.2629832328990D-01,
     :  0.3034165481994D-06, 0.4236622030873D+00, 0.4535059491685D+01 /
      DATA ((E0(I,J,2),I=1,3),J= 81, 90) /
     :  0.2676278825586D-06, 0.5970848007811D+01, 0.3930209696940D+01,
     :  0.2764903818918D-06, 0.5194636754501D+01, 0.1256262854127D+02,
     :  0.2485149930507D-06, 0.1002434207846D+01, 0.5088628793478D+01,
     :  0.2199305540941D-06, 0.3066773098403D+01, 0.1255903824622D+02,
     :  0.2571106500435D-06, 0.7588312459063D+00, 0.1336797263425D+02,
     :  0.2049751817158D-06, 0.3444977434856D+01, 0.1137170464392D+02,
     :  0.2599707296297D-06, 0.1873128542205D+01, 0.7143069561767D+02,
     :  0.1785018072217D-06, 0.5015891306615D+01, 0.1748016358760D+01,
     :  0.2324833891115D-06, 0.4618271239730D+01, 0.1831953657923D+02,
     :  0.1709711119545D-06, 0.5300003455669D+01, 0.4933208510675D+01 /
      DATA ((E0(I,J,2),I=1,3),J= 91,100) /
     :  0.2107159351716D-06, 0.2229819815115D+01, 0.7477522907414D+01,
     :  0.1750333080295D-06, 0.6161485880008D+01, 0.1044738781244D+02,
     :  0.2000598210339D-06, 0.2967357299999D+01, 0.8031092209206D+01,
     :  0.1380920248681D-06, 0.3027007923917D+01, 0.8635942003952D+01,
     :  0.1412460470299D-06, 0.6037597163798D+01, 0.2942463415728D+01,
     :  0.1888459803001D-06, 0.8561476243374D+00, 0.1561374759853D+03,
     :  0.1788370542585D-06, 0.4869736290209D+01, 0.1592596075957D+01,
     :  0.1360893296167D-06, 0.3626411886436D+01, 0.1309584267300D+02,
     :  0.1506846530160D-06, 0.1550975377427D+01, 0.1649636139783D+02,
     :  0.1800913376176D-06, 0.2075826033190D+01, 0.1729818233119D+02 /
      DATA ((E0(I,J,2),I=1,3),J=101,110) /
     :  0.1436261390649D-06, 0.6148876420255D+01, 0.2042657109477D+02,
     :  0.1220227114151D-06, 0.4382583879906D+01, 0.7632943190217D+01,
     :  0.1337883603592D-06, 0.2036644327361D+01, 0.1213955354133D+02,
     :  0.1159326650738D-06, 0.3892276994687D+01, 0.5331357529664D+01,
     :  0.1352853128569D-06, 0.1447950649744D+01, 0.1673046366289D+02,
     :  0.1433408296083D-06, 0.4457854692961D+01, 0.7342457794669D+01,
     :  0.1234701666518D-06, 0.1538818147151D+01, 0.6279485555400D+01,
     :  0.1234027192007D-06, 0.1968523220760D+01, 0.6286666145492D+01,
     :  0.1244024091797D-06, 0.5779803499985D+01, 0.1511046609763D+02,
     :  0.1097934945516D-06, 0.6210975221388D+00, 0.1098880815746D+02 /
      DATA ((E0(I,J,2),I=1,3),J=111,120) /
     :  0.1254611329856D-06, 0.2591963807998D+01, 0.1572083878776D+02,
     :  0.1158247286784D-06, 0.2483612812670D+01, 0.5729506548653D+01,
     :  0.9039078252960D-07, 0.3857554579796D+01, 0.9623688285163D+01,
     :  0.9108024978836D-07, 0.5826368512984D+01, 0.7234794171227D+01,
     :  0.8887068108436D-07, 0.3475694573987D+01, 0.6148010737701D+01,
     :  0.8632374035438D-07, 0.3059070488983D-01, 0.6418140963190D+01,
     :  0.7893186992967D-07, 0.1583194837728D+01, 0.2118763888447D+01,
     :  0.8297650201172D-07, 0.8519770534637D+00, 0.1471231707864D+02,
     :  0.1019759578988D-06, 0.1319598738732D+00, 0.1349867339771D+01,
     :  0.1010037696236D-06, 0.9937860115618D+00, 0.6836645152238D+01 /
      DATA ((E0(I,J,2),I=1,3),J=121,130) /
     :  0.1047727548266D-06, 0.1382138405399D+01, 0.5999216516294D+01,
     :  0.7351993881086D-07, 0.3833397851735D+01, 0.6040347114260D+01,
     :  0.9868771092341D-07, 0.2124913814390D+01, 0.6566935184597D+01,
     :  0.7007321959390D-07, 0.5946305343763D+01, 0.6525804586632D+01,
     :  0.6861411679709D-07, 0.4574654977089D+01, 0.7238675589263D+01,
     :  0.7554519809614D-07, 0.5949232686844D+01, 0.1253985337760D+02,
     :  0.9541880448335D-07, 0.3495242990564D+01, 0.2122839202813D+02,
     :  0.7185606722155D-07, 0.4310113471661D+01, 0.6245048154254D+01,
     :  0.7131360871710D-07, 0.5480309323650D+01, 0.6321103546637D+01,
     :  0.6651142021039D-07, 0.5411097713654D+01, 0.5327476111629D+01 /
      DATA ((E0(I,J,2),I=1,3),J=131,140) /
     :  0.8538618213667D-07, 0.1827849973951D+01, 0.1101510648075D+02,
     :  0.8634954288044D-07, 0.5443584943349D+01, 0.5643178611111D+01,
     :  0.7449415051484D-07, 0.2011535459060D+01, 0.5368044267797D+00,
     :  0.7421047599169D-07, 0.3464562529249D+01, 0.2354323048545D+02,
     :  0.6140694354424D-07, 0.5657556228815D+01, 0.1296430071988D+02,
     :  0.6353525143033D-07, 0.3463816593821D+01, 0.1990745094947D+01,
     :  0.6221964013447D-07, 0.1532259498697D+01, 0.9517183207817D+00,
     :  0.5852480257244D-07, 0.1375396598875D+01, 0.9555997388169D+00,
     :  0.6398637498911D-07, 0.2405645801972D+01, 0.2407292145756D+02,
     :  0.7039744069878D-07, 0.5397541799027D+01, 0.5225775174439D+00 /
      DATA ((E0(I,J,2),I=1,3),J=141,150) /
     :  0.6977997694382D-07, 0.4762347105419D+01, 0.1097355562493D+02,
     :  0.7460629558396D-07, 0.2711944692164D+01, 0.2200391463820D+02,
     :  0.5376577536101D-07, 0.2352980430239D+01, 0.1431416805965D+02,
     :  0.7530607893556D-07, 0.1943940180699D+01, 0.1842262939178D+02,
     :  0.6822928971605D-07, 0.4337651846959D+01, 0.1554202828031D+00,
     :  0.6220772380094D-07, 0.6716871369278D+00, 0.1845107853235D+02,
     :  0.6586950799043D-07, 0.2229714460505D+01, 0.5216580451554D+01,
     :  0.5873800565771D-07, 0.7627013920580D+00, 0.6398972393349D+00,
     :  0.6264346929745D-07, 0.6202785478961D+00, 0.6277552955062D+01,
     :  0.6257929115669D-07, 0.2886775596668D+01, 0.6288598745829D+01 /
      DATA ((E0(I,J,2),I=1,3),J=151,160) /
     :  0.5343536033409D-07, 0.1977241012051D+01, 0.4690479774488D+01,
     :  0.5587849781714D-07, 0.1922923484825D+01, 0.1551045220144D+01,
     :  0.6905100845603D-07, 0.3570757164631D+01, 0.1030928125552D+00,
     :  0.6178957066649D-07, 0.5197558947765D+01, 0.5230807360890D+01,
     :  0.6187270224331D-07, 0.8193497368922D+00, 0.5650292065779D+01,
     :  0.5385664291426D-07, 0.5406336665586D+01, 0.7771377146812D+02,
     :  0.6329363917926D-07, 0.2837760654536D+01, 0.2608790314060D+02,
     :  0.4546018761604D-07, 0.2933580297050D+01, 0.5535693017924D+00,
     :  0.6196091049375D-07, 0.4157871494377D+01, 0.8467247584405D+02,
     :  0.6159555108218D-07, 0.3211703561703D+01, 0.2394243902548D+03 /
      DATA ((E0(I,J,2),I=1,3),J=161,170) /
     :  0.4995340539317D-07, 0.1459098102922D+01, 0.4732030630302D+01,
     :  0.5457031243572D-07, 0.1430457676136D+01, 0.6179983037890D+01,
     :  0.4863461418397D-07, 0.2196425916730D+01, 0.9027992316901D+02,
     :  0.5342947626870D-07, 0.2086612890268D+01, 0.6386168663001D+01,
     :  0.5674296648439D-07, 0.2760204966535D+01, 0.6915859635113D+01,
     :  0.4745783120161D-07, 0.4245368971862D+01, 0.6282970628506D+01,
     :  0.4745676961198D-07, 0.5544725787016D+01, 0.6283181072386D+01,
     :  0.4049796869973D-07, 0.2213984363586D+01, 0.6254626709878D+01,
     :  0.4248333596940D-07, 0.8075781952896D+00, 0.7875671926403D+01,
     :  0.4027178070205D-07, 0.1293268540378D+01, 0.6311524991013D+01 /
      DATA ((E0(I,J,2),I=1,3),J=171,180) /
     :  0.4066543943476D-07, 0.3986141175804D+01, 0.3634620989887D+01,
     :  0.4858863787880D-07, 0.1276112738231D+01, 0.5760498333002D+01,
     :  0.5277398263530D-07, 0.4916111741527D+01, 0.2515860172507D+02,
     :  0.4105635656559D-07, 0.1725805864426D+01, 0.6709674010002D+01,
     :  0.4376781925772D-07, 0.2243642442106D+01, 0.6805653367890D+01,
     :  0.3235827894693D-07, 0.3614135118271D+01, 0.1066495398892D+01,
     :  0.3073244740308D-07, 0.2460873393460D+01, 0.5863591145557D+01,
     :  0.3088609271373D-07, 0.5678431771790D+01, 0.9917696840332D+01,
     :  0.3393022279836D-07, 0.3814017477291D+01, 0.1391601904066D+02,
     :  0.3038686508802D-07, 0.4660216229171D+01, 0.1256621883632D+02 /
      DATA ((E0(I,J,2),I=1,3),J=181,190) /
     :  0.4019677752497D-07, 0.5906906243735D+01, 0.1334167431096D+02,
     :  0.3288834998232D-07, 0.9536146445882D+00, 0.1620077269078D+02,
     :  0.3889973794631D-07, 0.3942205097644D+01, 0.7478166569050D-01,
     :  0.3050438987141D-07, 0.1624810271286D+01, 0.1805292951336D+02,
     :  0.3601142564638D-07, 0.4030467142575D+01, 0.6208294184755D+01,
     :  0.3689015557141D-07, 0.3648878818694D+01, 0.5966683958112D+01,
     :  0.3563471893565D-07, 0.5749584017096D+01, 0.6357857516136D+01,
     :  0.2776183170667D-07, 0.2630124187070D+01, 0.3523159621801D-02,
     :  0.2922350530341D-07, 0.1790346403629D+01, 0.1272157198369D+02,
     :  0.3511076917302D-07, 0.6142198301611D+01, 0.6599467742779D+01 /
      DATA ((E0(I,J,2),I=1,3),J=191,200) /
     :  0.3619351007632D-07, 0.1432421386492D+01, 0.6019991944201D+01,
     :  0.2561254711098D-07, 0.2302822475792D+01, 0.1259245002418D+02,
     :  0.2626903942920D-07, 0.8660470994571D+00, 0.6702560555334D+01,
     :  0.2550187397083D-07, 0.6069721995383D+01, 0.1057540660594D+02,
     :  0.2535873526138D-07, 0.1079020331795D-01, 0.3141537925223D+02,
     :  0.3519786153847D-07, 0.3809066902283D+01, 0.2505706758577D+03,
     :  0.3424651492873D-07, 0.2075435114417D+01, 0.6546159756691D+01,
     :  0.2372676630861D-07, 0.2057803120154D+01, 0.2388894113936D+01,
     :  0.2710980779541D-07, 0.1510068488010D+01, 0.1202934727411D+02,
     :  0.3038710889704D-07, 0.5043617528901D+01, 0.1256608456547D+02 /
      DATA ((E0(I,J,2),I=1,3),J=201,210) /
     :  0.2220364130585D-07, 0.3694793218205D+01, 0.1336244973887D+02,
     :  0.3025880825460D-07, 0.5450618999049D-01, 0.2908881142201D+02,
     :  0.2784493486864D-07, 0.3381164084502D+01, 0.1494531617769D+02,
     :  0.2294414142438D-07, 0.4382309025210D+01, 0.6076890225335D+01,
     :  0.2012723294724D-07, 0.9142212256518D+00, 0.6262720680387D+01,
     :  0.2036357831958D-07, 0.5676172293154D+01, 0.4701116388778D+01,
     :  0.2003474823288D-07, 0.2592767977625D+01, 0.6303431020504D+01,
     :  0.2207144900109D-07, 0.5404976271180D+01, 0.6489261475556D+01,
     :  0.2481664905135D-07, 0.4373284587027D+01, 0.1204357418345D+02,
     :  0.2674949182295D-07, 0.5859182188482D+01, 0.4590910121555D+01 /
      DATA ((E0(I,J,2),I=1,3),J=211,220) /
     :  0.2450554720322D-07, 0.4555381557451D+01, 0.1495633313810D+00,
     :  0.2601975986457D-07, 0.3933165584959D+01, 0.1965104848470D+02,
     :  0.2199860022848D-07, 0.5227977189087D+01, 0.1351787002167D+02,
     :  0.2448121172316D-07, 0.4858060353949D+01, 0.1162474756779D+01,
     :  0.1876014864049D-07, 0.5690546553605D+01, 0.6279194432410D+01,
     :  0.1874513219396D-07, 0.4099539297446D+01, 0.6286957268481D+01,
     :  0.2156380842559D-07, 0.4382594769913D+00, 0.1813929450232D+02,
     :  0.1981691240061D-07, 0.1829784152444D+01, 0.4686889479442D+01,
     :  0.2329992648539D-07, 0.2836254278973D+01, 0.1002183730415D+02,
     :  0.1765184135302D-07, 0.2803494925833D+01, 0.4292330755499D+01 /
      DATA ((E0(I,J,2),I=1,3),J=221,230) /
     :  0.2436368366085D-07, 0.2836897959677D+01, 0.9514313292143D+02,
     :  0.2164089203889D-07, 0.6127522446024D+01, 0.6037244212485D+01,
     :  0.1847755034221D-07, 0.3683163635008D+01, 0.2427287361862D+00,
     :  0.1674798769966D-07, 0.3316993867246D+00, 0.1311972100268D+02,
     :  0.2222542124356D-07, 0.8294097805480D+00, 0.1266924451345D+02,
     :  0.2071074505925D-07, 0.3659492220261D+01, 0.6528907488406D+01,
     :  0.1608224471835D-07, 0.4774492067182D+01, 0.1352175143971D+02,
     :  0.1857583439071D-07, 0.2873120597682D+01, 0.8662240327241D+01,
     :  0.1793018836159D-07, 0.5282441177929D+00, 0.6819880277225D+01,
     :  0.1575391221692D-07, 0.1320789654258D+01, 0.1102062672231D+00 /
      DATA ((E0(I,J,2),I=1,3),J=231,240) /
     :  0.1840132009557D-07, 0.1917110916256D+01, 0.6514761976723D+02,
     :  0.1760917288281D-07, 0.2972635937132D+01, 0.5746271423666D+01,
     :  0.1561779518516D-07, 0.4372569261981D+01, 0.6272439236156D+01,
     :  0.1558687885205D-07, 0.5416424926425D+01, 0.6293712464735D+01,
     :  0.1951359382579D-07, 0.3094448898752D+01, 0.2301353951334D+02,
     :  0.1569144275614D-07, 0.2802103689808D+01, 0.1765478049437D+02,
     :  0.1479130389462D-07, 0.2136435020467D+01, 0.2077542790660D-01,
     :  0.1467828510764D-07, 0.7072627435674D+00, 0.1052268489556D+01,
     :  0.1627627337440D-07, 0.3947607143237D+01, 0.6327837846670D+00,
     :  0.1503498479758D-07, 0.4079248909190D+01, 0.7626583626240D-01 /
      DATA ((E0(I,J,2),I=1,3),J=241,250) /
     :  0.1297967708237D-07, 0.6269637122840D+01, 0.1149965630200D+02,
     :  0.1374416896634D-07, 0.4175657970702D+01, 0.6016468784579D+01,
     :  0.1783812325219D-07, 0.1476540547560D+01, 0.3301902111895D+02,
     :  0.1525884228756D-07, 0.4653477715241D+01, 0.9411464614024D+01,
     :  0.1451067396763D-07, 0.2573001128225D+01, 0.1277945078067D+02,
     :  0.1297713111950D-07, 0.5612799618771D+01, 0.6549682916313D+01,
     :  0.1462784012820D-07, 0.4189661623870D+01, 0.1863592847156D+02,
     :  0.1384185980007D-07, 0.2656915472196D+01, 0.2379164476796D+01,
     :  0.1221497599801D-07, 0.5612515760138D+01, 0.1257326515556D+02,
     :  0.1560574525896D-07, 0.4783414317919D+01, 0.1887552587463D+02 /
      DATA ((E0(I,J,2),I=1,3),J=251,260) /
     :  0.1544598372036D-07, 0.2694431138063D+01, 0.1820933031200D+02,
     :  0.1531678928696D-07, 0.4105103489666D+01, 0.2593412433514D+02,
     :  0.1349321503795D-07, 0.3082437194015D+00, 0.5120601093667D+01,
     :  0.1252030290917D-07, 0.6124072334087D+01, 0.6993008899458D+01,
     :  0.1459243816687D-07, 0.3733103981697D+01, 0.3813291813120D-01,
     :  0.1226103625262D-07, 0.1267127706817D+01, 0.2435678079171D+02,
     :  0.1019449641504D-07, 0.4367790112269D+01, 0.1725663147538D+02,
     :  0.1380789433607D-07, 0.3387201768700D+01, 0.2458316379602D+00,
     :  0.1019453421658D-07, 0.9204143073737D+00, 0.6112403035119D+01,
     :  0.1297929434405D-07, 0.5786874896426D+01, 0.1249137003520D+02 /
      DATA ((E0(I,J,2),I=1,3),J=261,270) /
     :  0.9912677786097D-08, 0.3164232870746D+01, 0.6247047890016D+01,
     :  0.9829386098599D-08, 0.2586762413351D+01, 0.6453748665772D+01,
     :  0.1226807746104D-07, 0.6239068436607D+01, 0.5429879531333D+01,
     :  0.1192691755997D-07, 0.1867380051424D+01, 0.6290122169689D+01,
     :  0.9836499227081D-08, 0.3424716293727D+00, 0.6319103810876D+01,
     :  0.9642862564285D-08, 0.5661372990657D+01, 0.8273820945392D+01,
     :  0.1165184404862D-07, 0.5768367239093D+01, 0.1778273215245D+02,
     :  0.1175794418818D-07, 0.1657351222943D+01, 0.6276029531202D+01,
     :  0.1018948635601D-07, 0.6458292350865D+00, 0.1254537627298D+02,
     :  0.9500383606676D-08, 0.1054306140741D+01, 0.1256517118505D+02 /
      DATA ((E0(I,J,2),I=1,3),J=271,280) /
     :  0.1227512202906D-07, 0.2505278379114D+01, 0.2248384854122D+02,
     :  0.9664792009993D-08, 0.4289737277000D+01, 0.6259197520765D+01,
     :  0.9613285666331D-08, 0.5500597673141D+01, 0.6306954180126D+01,
     :  0.1117906736211D-07, 0.2361405953468D+01, 0.1779695906178D+02,
     :  0.9611378640782D-08, 0.2851310576269D+01, 0.2061856251104D+00,
     :  0.8845354852370D-08, 0.6208777705343D+01, 0.1692165728891D+01,
     :  0.1054046966600D-07, 0.5413091423934D+01, 0.2204125344462D+00,
     :  0.1215539124483D-07, 0.5613969479755D+01, 0.8257698122054D+02,
     :  0.9932460955209D-08, 0.1106124877015D+01, 0.1017725758696D+02,
     :  0.8785804715043D-08, 0.2869224476477D+01, 0.9491756770005D+00 /
      DATA ((E0(I,J,2),I=1,3),J=281,290) /
     :  0.8538084097562D-08, 0.6159640899344D+01, 0.6393282117669D+01,
     :  0.8648994369529D-08, 0.1374901198784D+01, 0.4804209201333D+01,
     :  0.1039063219067D-07, 0.5171080641327D+01, 0.1550861511662D+02,
     :  0.8867983926439D-08, 0.8317320304902D+00, 0.3903911373650D+01,
     :  0.8327495955244D-08, 0.3605591969180D+01, 0.6172869583223D+01,
     :  0.9243088356133D-08, 0.6114299196843D+01, 0.6267823317922D+01,
     :  0.9205657357835D-08, 0.3675153683737D+01, 0.6298328382969D+01,
     :  0.1033269714606D-07, 0.3313328813024D+01, 0.5573142801433D+01,
     :  0.8001706275552D-08, 0.2019980960053D+01, 0.2648454860559D+01,
     :  0.9171858254191D-08, 0.8992015524177D+00, 0.1498544001348D+03 /
      DATA ((E0(I,J,2),I=1,3),J=291,300) /
     :  0.1075327150242D-07, 0.2898669963648D+01, 0.3694923081589D+02,
     :  0.9884866689828D-08, 0.4946715904478D+01, 0.1140367694411D+02,
     :  0.9541835576677D-08, 0.2371787888469D+01, 0.1256713221673D+02,
     :  0.7739903376237D-08, 0.2213775190612D+01, 0.7834121070590D+01,
     :  0.7311962684106D-08, 0.3429378787739D+01, 0.1192625446156D+02,
     :  0.9724904869624D-08, 0.6195878564404D+01, 0.2280573557157D+02,
     :  0.9251628983612D-08, 0.6511509527390D+00, 0.2787043132925D+01,
     :  0.7320763787842D-08, 0.6001083639421D+01, 0.6282655592598D+01,
     :  0.7320296650962D-08, 0.3789073265087D+01, 0.6283496108294D+01,
     :  0.7947032271039D-08, 0.1059659582204D+01, 0.1241073141809D+02 /
      DATA ((E0(I,J,2),I=1,3),J=301,310) /
     :  0.9005277053115D-08, 0.1280315624361D+01, 0.6281591679874D+01,
     :  0.8995601652048D-08, 0.2224439106766D+01, 0.6284560021018D+01,
     :  0.8288040568796D-08, 0.5234914433867D+01, 0.1241658836951D+02,
     :  0.6359381347255D-08, 0.4137989441490D+01, 0.1596186371003D+01,
     :  0.8699572228626D-08, 0.1758411009497D+01, 0.6133512519065D+01,
     :  0.6456797542736D-08, 0.5919285089994D+01, 0.1685848245639D+02,
     :  0.7424573475452D-08, 0.5414616938827D+01, 0.4061219149443D+01,
     :  0.7235671196168D-08, 0.1496516557134D+01, 0.1610006857377D+03,
     :  0.8104015182733D-08, 0.1919918242764D+01, 0.8460828644453D+00,
     :  0.8098576535937D-08, 0.3819615855458D+01, 0.3894181736510D+01 /
      DATA ((E0(I,J,2),I=1,3),J=311,320) /
     :  0.6275292346625D-08, 0.6244264115141D+01, 0.8531963191132D+00,
     :  0.6052432989112D-08, 0.5037731872610D+00, 0.1567108171867D+02,
     :  0.5705651535817D-08, 0.2984557271995D+01, 0.1258692712880D+02,
     :  0.5789650115138D-08, 0.6087038140697D+01, 0.1193336791622D+02,
     :  0.5512132153377D-08, 0.5855668994076D+01, 0.1232342296471D+02,
     :  0.7388890819102D-08, 0.2443128574740D+01, 0.4907302013889D+01,
     :  0.5467593991798D-08, 0.3017561234194D+01, 0.1884211409667D+02,
     :  0.6388519802999D-08, 0.5887386712935D+01, 0.5217580628120D+02,
     :  0.6106777149944D-08, 0.3483461059895D+00, 0.1422690933580D-01,
     :  0.7383420275489D-08, 0.5417387056707D+01, 0.2358125818164D+02 /
      DATA ((E0(I,J,2),I=1,3),J=321,330) /
     :  0.5505208141738D-08, 0.2848193644783D+01, 0.1151388321134D+02,
     :  0.6310757462877D-08, 0.2349882520828D+01, 0.1041998632314D+02,
     :  0.6166904929691D-08, 0.5728575944077D+00, 0.6151533897323D+01,
     :  0.5263442042754D-08, 0.4495796125937D+01, 0.1885275071096D+02,
     :  0.5591828082629D-08, 0.1355441967677D+01, 0.4337116142245D+00,
     :  0.5397051680497D-08, 0.1673422864307D+01, 0.6286362197481D+01,
     :  0.5396992745159D-08, 0.1833502206373D+01, 0.6279789503410D+01,
     :  0.6572913000726D-08, 0.3331122065824D+01, 0.1176433076753D+02,
     :  0.5123421866413D-08, 0.2165327142679D+01, 0.1245594543367D+02,
     :  0.5930495725999D-08, 0.2931146089284D+01, 0.6414617803568D+01 /
      DATA ((E0(I,J,2),I=1,3),J=331,340) /
     :  0.6431797403933D-08, 0.4134407994088D+01, 0.1350651127443D+00,
     :  0.5003182207604D-08, 0.3805420303749D+01, 0.1096996532989D+02,
     :  0.5587731032504D-08, 0.1082469260599D+01, 0.6062663316000D+01,
     :  0.5935263407816D-08, 0.8384333678401D+00, 0.5326786718777D+01,
     :  0.4756019827760D-08, 0.3552588749309D+01, 0.3104930017775D+01,
     :  0.6599951172637D-08, 0.4320826409528D+01, 0.4087944051283D+02,
     :  0.5902606868464D-08, 0.4811879454445D+01, 0.5849364236221D+01,
     :  0.5921147809031D-08, 0.9942628922396D-01, 0.1581959461667D+01,
     :  0.5505382581266D-08, 0.2466557607764D+01, 0.6503488384892D+01,
     :  0.5353771071862D-08, 0.4551978748683D+01, 0.1735668374386D+03 /
      DATA ((E0(I,J,2),I=1,3),J=341,350) /
     :  0.5063282210946D-08, 0.5710812312425D+01, 0.1248988586463D+02,
     :  0.5926120403383D-08, 0.1333998428358D+01, 0.2673594526851D+02,
     :  0.5211016176149D-08, 0.4649315360760D+01, 0.2460261242967D+02,
     :  0.5347075084894D-08, 0.5512754081205D+01, 0.4171425416666D+01,
     :  0.4872609773574D-08, 0.1308025299938D+01, 0.5333900173445D+01,
     :  0.4727711321420D-08, 0.2144908368062D+01, 0.7232251527446D+01,
     :  0.6029426018652D-08, 0.5567259412084D+01, 0.3227113045244D+03,
     :  0.4321485284369D-08, 0.5230667156451D+01, 0.9388005868221D+01,
     :  0.4476406760553D-08, 0.6134081115303D+01, 0.5547199253223D+01,
     :  0.5835268277420D-08, 0.4783808492071D+01, 0.7285056171570D+02 /
      DATA ((E0(I,J,2),I=1,3),J=351,360) /
     :  0.5172183602748D-08, 0.5161817911099D+01, 0.1884570439172D+02,
     :  0.5693571465184D-08, 0.1381646203111D+01, 0.9723862754494D+02,
     :  0.4060634965349D-08, 0.3876705259495D+00, 0.4274518229222D+01,
     :  0.3967398770473D-08, 0.5029491776223D+01, 0.3496032717521D+01,
     :  0.3943754005255D-08, 0.1923162955490D+01, 0.6244942932314D+01,
     :  0.4781323427824D-08, 0.4633332586423D+01, 0.2929661536378D+02,
     :  0.3871483781204D-08, 0.1616650009743D+01, 0.6321208768577D+01,
     :  0.5141741733997D-08, 0.9817316704659D-01, 0.1232032006293D+02,
     :  0.4002385978497D-08, 0.3656161212139D+01, 0.7018952447668D+01,
     :  0.4901092604097D-08, 0.4404098713092D+01, 0.1478866649112D+01 /
      DATA ((E0(I,J,2),I=1,3),J=361,370) /
     :  0.3740932630345D-08, 0.5181188732639D+00, 0.6922973089781D+01,
     :  0.4387283718538D-08, 0.3254859566869D+01, 0.2331413144044D+03,
     :  0.5019197802033D-08, 0.3086773224677D+01, 0.1715706182245D+02,
     :  0.3834931695175D-08, 0.2797882673542D+01, 0.1491901785440D+02,
     :  0.3760413942497D-08, 0.2892676280217D+01, 0.1726726808967D+02,
     :  0.3719717204628D-08, 0.5861046025739D+01, 0.6297302759782D+01,
     :  0.4145623530149D-08, 0.2168239627033D+01, 0.1376059875786D+02,
     :  0.3932788425380D-08, 0.6271811124181D+01, 0.7872148766781D+01,
     :  0.3686377476857D-08, 0.3936853151404D+01, 0.6268848941110D+01,
     :  0.3779077950339D-08, 0.1404148734043D+01, 0.4157198507331D+01 /
      DATA ((E0(I,J,2),I=1,3),J=371,380) /
     :  0.4091334550598D-08, 0.2452436180854D+01, 0.9779108567966D+01,
     :  0.3926694536146D-08, 0.6102292739040D+01, 0.1098419223922D+02,
     :  0.4841000253289D-08, 0.6072760457276D+01, 0.1252801878276D+02,
     :  0.4949340130240D-08, 0.1154832815171D+01, 0.1617106187867D+03,
     :  0.3761557737360D-08, 0.5527545321897D+01, 0.3185192151914D+01,
     :  0.3647396268188D-08, 0.1525035688629D+01, 0.6271346477544D+01,
     :  0.3932405074189D-08, 0.5570681040569D+01, 0.2139354194808D+02,
     :  0.3631322501141D-08, 0.1981240601160D+01, 0.6294805223347D+01,
     :  0.4130007425139D-08, 0.2050060880201D+01, 0.2195415756911D+02,
     :  0.4433905965176D-08, 0.3277477970321D+01, 0.7445550607224D+01 /
      DATA ((E0(I,J,2),I=1,3),J=381,390) /
     :  0.3851814176947D-08, 0.5210690074886D+01, 0.9562891316684D+00,
     :  0.3485807052785D-08, 0.6653274904611D+00, 0.1161697602389D+02,
     :  0.3979772816991D-08, 0.1767941436148D+01, 0.2277943724828D+02,
     :  0.3402607460500D-08, 0.3421746306465D+01, 0.1087398597200D+02,
     :  0.4049993000926D-08, 0.1127144787547D+01, 0.3163918923335D+00,
     :  0.3420511182382D-08, 0.4214794779161D+01, 0.1362553364512D+02,
     :  0.3640772365012D-08, 0.5324905497687D+01, 0.1725304118033D+02,
     :  0.3323037987501D-08, 0.6135761838271D+01, 0.6279143387820D+01,
     :  0.4503141663637D-08, 0.1802305450666D+01, 0.1385561574497D+01,
     :  0.4314560055588D-08, 0.4812299731574D+01, 0.4176041334900D+01 /
      DATA ((E0(I,J,2),I=1,3),J=391,400) /
     :  0.3294226949110D-08, 0.3657547059723D+01, 0.6287008313071D+01,
     :  0.3215657197281D-08, 0.4866676894425D+01, 0.5749861718712D+01,
     :  0.4129362656266D-08, 0.3809342558906D+01, 0.5905702259363D+01,
     :  0.3137762976388D-08, 0.2494635174443D+01, 0.2099539292909D+02,
     :  0.3514010952384D-08, 0.2699961831678D+01, 0.7335344340001D+01,
     :  0.3327607571530D-08, 0.3318457714816D+01, 0.5436992986000D+01,
     :  0.3541066946675D-08, 0.4382703582466D+01, 0.1234573916645D+02,
     :  0.3216179847052D-08, 0.5271066317054D+01, 0.3802769619140D-01,
     :  0.2959045059570D-08, 0.5819591585302D+01, 0.2670964694522D+02,
     :  0.3884040326665D-08, 0.5980934960428D+01, 0.6660449441528D+01 /
      DATA ((E0(I,J,2),I=1,3),J=401,410) /
     :  0.2922027539886D-08, 0.3337290282483D+01, 0.1375773836557D+01,
     :  0.4110846382042D-08, 0.5742978187327D+01, 0.4480965020977D+02,
     :  0.2934508411032D-08, 0.2278075804200D+01, 0.6408777551755D+00,
     :  0.3966896193000D-08, 0.5835747858477D+01, 0.3773735910827D+00,
     :  0.3286695827610D-08, 0.5838898193902D+01, 0.3932462625300D-02,
     :  0.3720643094196D-08, 0.1122212337858D+01, 0.1646033343740D+02,
     :  0.3285508906174D-08, 0.9182250996416D+00, 0.1081813534213D+02,
     :  0.3753880575973D-08, 0.5174761973266D+01, 0.5642198095270D+01,
     :  0.3022129385587D-08, 0.3381611020639D+01, 0.2982630633589D+02,
     :  0.2798569205621D-08, 0.3546193723922D+01, 0.1937891852345D+02 /
      DATA ((E0(I,J,2),I=1,3),J=411,420) /
     :  0.3397872070505D-08, 0.4533203197934D+01, 0.6923953605621D+01,
     :  0.3708099772977D-08, 0.2756168198616D+01, 0.3066615496545D+02,
     :  0.3599283541510D-08, 0.1934395469918D+01, 0.6147450479709D+01,
     :  0.3688702753059D-08, 0.7149920971109D+00, 0.2636725487657D+01,
     :  0.2681084724003D-08, 0.4899819493154D+01, 0.6816289982179D+01,
     :  0.3495993460759D-08, 0.1572418915115D+01, 0.6418701221183D+01,
     :  0.3130770324995D-08, 0.8912190180489D+00, 0.1235996607578D+02,
     :  0.2744353821941D-08, 0.3800821940055D+01, 0.2059724391010D+02,
     :  0.2842732906341D-08, 0.2644717440029D+01, 0.2828699048865D+02,
     :  0.3046882682154D-08, 0.3987793020179D+01, 0.6055599646783D+01 /
      DATA ((E0(I,J,2),I=1,3),J=421,430) /
     :  0.2399072455143D-08, 0.9908826440764D+00, 0.6255674361143D+01,
     :  0.2384306274204D-08, 0.2516149752220D+01, 0.6310477339748D+01,
     :  0.2977324500559D-08, 0.5849195642118D+01, 0.1652265972112D+02,
     :  0.3062835258972D-08, 0.1681660100162D+01, 0.1172006883645D+02,
     :  0.3109682589231D-08, 0.5804143987737D+00, 0.2751146787858D+02,
     :  0.2903920355299D-08, 0.5800768280123D+01, 0.6510552054109D+01,
     :  0.2823221989212D-08, 0.9241118370216D+00, 0.5469525544182D+01,
     :  0.3187949696649D-08, 0.3139776445735D+01, 0.1693792562116D+03,
     :  0.2922559771655D-08, 0.3549440782984D+01, 0.2630839062450D+00,
     :  0.2436302066603D-08, 0.4735540696319D+01, 0.3946258593675D+00 /
      DATA ((E0(I,J,2),I=1,3),J=431,440) /
     :  0.3049473043606D-08, 0.4998289124561D+01, 0.8390110365991D+01,
     :  0.2863682575784D-08, 0.6709515671102D+00, 0.2243449970715D+00,
     :  0.2641750517966D-08, 0.5410978257284D+01, 0.2986433403208D+02,
     :  0.2704093466243D-08, 0.4778317207821D+01, 0.6129297044991D+01,
     :  0.2445522177011D-08, 0.6009020662222D+01, 0.1171295538178D+02,
     :  0.2623608810230D-08, 0.5010449777147D+01, 0.6436854655901D+01,
     :  0.2079259704053D-08, 0.5980943768809D+01, 0.2019909489111D+02,
     :  0.2820225596771D-08, 0.2679965110468D+01, 0.5934151399930D+01,
     :  0.2365221950927D-08, 0.1894231148810D+01, 0.2470570524223D+02,
     :  0.2359682077149D-08, 0.4220752950780D+01, 0.8671969964381D+01 /
      DATA ((E0(I,J,2),I=1,3),J=441,450) /
     :  0.2387577137206D-08, 0.2571783940617D+01, 0.7096626156709D+01,
     :  0.1982102089816D-08, 0.5169765997119D+00, 0.1727188400790D+02,
     :  0.2687502389925D-08, 0.6239078264579D+01, 0.7075506709219D+02,
     :  0.2207751669135D-08, 0.2031184412677D+01, 0.4377611041777D+01,
     :  0.2618370214274D-08, 0.8266079985979D+00, 0.6632000300961D+01,
     :  0.2591951887361D-08, 0.8819350522008D+00, 0.4873985990671D+02,
     :  0.2375055656248D-08, 0.3520944177789D+01, 0.1590676413561D+02,
     :  0.2472019978911D-08, 0.1551431908671D+01, 0.6612329252343D+00,
     :  0.2368157127199D-08, 0.4178610147412D+01, 0.3459636466239D+02,
     :  0.1764846605693D-08, 0.1506764000157D+01, 0.1980094587212D+02 /
      DATA ((E0(I,J,2),I=1,3),J=451,460) /
     :  0.2291769608798D-08, 0.2118250611782D+01, 0.2844914056730D-01,
     :  0.2209997316943D-08, 0.3363255261678D+01, 0.2666070658668D+00,
     :  0.2292699097923D-08, 0.4200423956460D+00, 0.1484170571900D-02,
     :  0.1629683015329D-08, 0.2331362582487D+01, 0.3035599730800D+02,
     :  0.2206492862426D-08, 0.3400274026992D+01, 0.6281667977667D+01,
     :  0.2205746568257D-08, 0.1066051230724D+00, 0.6284483723224D+01,
     :  0.2026310767991D-08, 0.2779066487979D+01, 0.2449240616245D+02,
     :  0.1762977622163D-08, 0.9951450691840D+00, 0.2045286941806D+02,
     :  0.1368535049606D-08, 0.6402447365817D+00, 0.2473415438279D+02,
     :  0.1720598775450D-08, 0.2303524214705D+00, 0.1679593901136D+03 /
      DATA ((E0(I,J,2),I=1,3),J=461,470) /
     :  0.1702429015449D-08, 0.6164622655048D+01, 0.3338575901272D+03,
     :  0.1414033197685D-08, 0.3954561185580D+01, 0.1624205518357D+03,
     :  0.1573768958043D-08, 0.2028286308984D+01, 0.3144167757552D+02,
     :  0.1650705184447D-08, 0.2304040666128D+01, 0.5267006960365D+02,
     :  0.1651087618855D-08, 0.2538461057280D+01, 0.8956999012000D+02,
     :  0.1616409518983D-08, 0.5111054348152D+01, 0.3332657872986D+02,
     :  0.1537175173581D-08, 0.5601130666603D+01, 0.3852657435933D+02,
     :  0.1593191980553D-08, 0.2614340453411D+01, 0.2282781046519D+03,
     :  0.1499480170643D-08, 0.3624721577264D+01, 0.2823723341956D+02,
     :  0.1493807843235D-08, 0.4214569879008D+01, 0.2876692439167D+02 /
      DATA ((E0(I,J,2),I=1,3),J=471,480) /
     :  0.1074571199328D-08, 0.1496911744704D+00, 0.8397383534231D+02,
     :  0.1074406983417D-08, 0.1187817671922D+01, 0.8401985929482D+02,
     :  0.9757576855851D-09, 0.2655703035858D+01, 0.7826370942180D+02,
     :  0.1258432887565D-08, 0.4969896184844D+01, 0.3115650189215D+03,
     :  0.1240336343282D-08, 0.5192460776926D+01, 0.1784300471910D+03,
     :  0.9016107005164D-09, 0.1960356923057D+01, 0.5886454391678D+02,
     :  0.1135392360918D-08, 0.5082427809068D+01, 0.7842370451713D+02,
     :  0.9216046089565D-09, 0.2793775037273D+01, 0.1014262087719D+03,
     :  0.1061276615030D-08, 0.3726144311409D+01, 0.5660027930059D+02,
     :  0.1010110596263D-08, 0.7404080708937D+00, 0.4245678405627D+02 /
      DATA ((E0(I,J,2),I=1,3),J=481,490) /
     :  0.7217424756199D-09, 0.2697449980577D-01, 0.2457074661053D+03,
     :  0.6912003846756D-09, 0.4253296276335D+01, 0.1679936946371D+03,
     :  0.6871814664847D-09, 0.5148072412354D+01, 0.6053048899753D+02,
     :  0.4887158016343D-09, 0.2153581148294D+01, 0.9656299901946D+02,
     :  0.5161802866314D-09, 0.3852750634351D+01, 0.2442876000072D+03,
     :  0.5652599559057D-09, 0.1233233356270D+01, 0.8365903305582D+02,
     :  0.4710812608586D-09, 0.5610486976767D+01, 0.3164282286739D+03,
     :  0.4909977500324D-09, 0.1639629524123D+01, 0.4059982187939D+03,
     :  0.4772641839378D-09, 0.3737100368583D+01, 0.1805255418145D+03,
     :  0.4487562567153D-09, 0.1158417054478D+00, 0.8433466158131D+02 /
      DATA ((E0(I,J,2),I=1,3),J=491,500) /
     :  0.3943441230497D-09, 0.6243502862796D+00, 0.2568537517081D+03,
     :  0.3952236913598D-09, 0.3510377382385D+01, 0.2449975330562D+03,
     :  0.3788898363417D-09, 0.5916128302299D+01, 0.1568131045107D+03,
     :  0.3738329328831D-09, 0.1042266763456D+01, 0.3948519331910D+03,
     :  0.2451199165151D-09, 0.1166788435700D+01, 0.1435713242844D+03,
     :  0.2436734402904D-09, 0.3254726114901D+01, 0.2268582385539D+03,
     :  0.2213605274325D-09, 0.1687210598530D+01, 0.1658638954901D+03,
     :  0.1491521204829D-09, 0.2657541786794D+01, 0.2219950288015D+03,
     :  0.1474995329744D-09, 0.5013089805819D+01, 0.3052819430710D+03,
     :  0.1661939475656D-09, 0.5495315428418D+01, 0.2526661704812D+03 /
      DATA ((E0(I,J,2),I=1,3),J=501,NE0Y) /
     :  0.9015946748003D-10, 0.2236989966505D+01, 0.4171445043968D+03 /

*  Sun-to-Earth, T^1, Y
      DATA ((E1(I,J,2),I=1,3),J=  1, 10) /
     :  0.9304690546528D-06, 0.0000000000000D+00, 0.0000000000000D+00,
     :  0.5150715570663D-06, 0.4431807116294D+01, 0.1256615170089D+02,
     :  0.1290825411056D-07, 0.4388610039678D+01, 0.1884922755134D+02,
     :  0.4645466665386D-08, 0.5827263376034D+01, 0.6283075850446D+01,
     :  0.2079625310718D-08, 0.1621698662282D+00, 0.6279552690824D+01,
     :  0.2078189850907D-08, 0.3344713435140D+01, 0.6286599010068D+01,
     :  0.6207190138027D-09, 0.5074049319576D+01, 0.4705732307012D+01,
     :  0.5989826532569D-09, 0.2231842216620D+01, 0.6256777527156D+01,
     :  0.5961360812618D-09, 0.1274975769045D+01, 0.6309374173736D+01,
     :  0.4874165471016D-09, 0.3642277426779D+01, 0.7755226100720D+00 /
      DATA ((E1(I,J,2),I=1,3),J= 11, 20) /
     :  0.4283834034360D-09, 0.5148765510106D+01, 0.1059381944224D+01,
     :  0.4652389287529D-09, 0.4715794792175D+01, 0.7860419393880D+01,
     :  0.3751707476401D-09, 0.6617207370325D+00, 0.5753384878334D+01,
     :  0.3559998806198D-09, 0.6155548875404D+01, 0.5884926831456D+01,
     :  0.3558447558857D-09, 0.2898827297664D+01, 0.6812766822558D+01,
     :  0.3211116927106D-09, 0.3625813502509D+01, 0.6681224869435D+01,
     :  0.2875609914672D-09, 0.4345435813134D+01, 0.2513230340178D+02,
     :  0.2843109704069D-09, 0.5862263940038D+01, 0.6127655567643D+01,
     :  0.2744676468427D-09, 0.3926419475089D+01, 0.6438496133249D+01,
     :  0.2481285237789D-09, 0.1351976572828D+01, 0.5486777812467D+01 /
      DATA ((E1(I,J,2),I=1,3),J= 21, 30) /
     :  0.2060338481033D-09, 0.2147556998591D+01, 0.7079373888424D+01,
     :  0.2015822358331D-09, 0.4408358972216D+01, 0.6290189305114D+01,
     :  0.2001195944195D-09, 0.5385829822531D+01, 0.6275962395778D+01,
     :  0.1953667642377D-09, 0.1304933746120D+01, 0.5507553240374D+01,
     :  0.1839744078713D-09, 0.6173567228835D+01, 0.1179062909082D+02,
     :  0.1643334294845D-09, 0.4635942997523D+01, 0.1150676975667D+02,
     :  0.1768051018652D-09, 0.5086283558874D+01, 0.7113454667900D-02,
     :  0.1674874205489D-09, 0.2243332137241D+01, 0.7058598460518D+01,
     :  0.1421445397609D-09, 0.6186899771515D+01, 0.7962980379786D+00,
     :  0.1255163958267D-09, 0.5730238465658D+01, 0.4694002934110D+01 /
      DATA ((E1(I,J,2),I=1,3),J= 31, 40) /
     :  0.1013945281961D-09, 0.1726055228402D+01, 0.3738761453707D+01,
     :  0.1047294335852D-09, 0.2658801228129D+01, 0.6282095334605D+01,
     :  0.1047103879392D-09, 0.8481047835035D+00, 0.6284056366286D+01,
     :  0.9530343962826D-10, 0.3079267149859D+01, 0.6069776770667D+01,
     :  0.9604637611690D-10, 0.3258679792918D+00, 0.4136910472696D+01,
     :  0.9153518537177D-10, 0.4398599886584D+00, 0.6496374930224D+01,
     :  0.8562458214922D-10, 0.4772686794145D+01, 0.1194447056968D+01,
     :  0.8232525360654D-10, 0.5966220721679D+01, 0.1589072916335D+01,
     :  0.6150223411438D-10, 0.1780985591923D+01, 0.8827390247185D+01,
     :  0.6272087858000D-10, 0.3184305429012D+01, 0.8429241228195D+01 /
      DATA ((E1(I,J,2),I=1,3),J= 41, 50) /
     :  0.5540476311040D-10, 0.3801260595433D+01, 0.4933208510675D+01,
     :  0.7331901699361D-10, 0.5205948591865D+01, 0.4535059491685D+01,
     :  0.6018528702791D-10, 0.4770139083623D+01, 0.1255903824622D+02,
     :  0.5150530724804D-10, 0.3574796899585D+01, 0.1176985366291D+02,
     :  0.6471933741811D-10, 0.2679787266521D+01, 0.5088628793478D+01,
     :  0.5317460644174D-10, 0.9528763345494D+00, 0.3154687086868D+01,
     :  0.4832187748783D-10, 0.5329322498232D+01, 0.6040347114260D+01,
     :  0.4716763555110D-10, 0.2395235316466D+01, 0.5331357529664D+01,
     :  0.4871509139861D-10, 0.3056663648823D+01, 0.1256967486051D+02,
     :  0.4598417696768D-10, 0.4452762609019D+01, 0.6525804586632D+01 /
      DATA ((E1(I,J,2),I=1,3),J= 51, 60) /
     :  0.5674189533175D-10, 0.9879680872193D+00, 0.5729506548653D+01,
     :  0.4073560328195D-10, 0.5939127696986D+01, 0.7632943190217D+01,
     :  0.5040994945359D-10, 0.4549875824510D+01, 0.8031092209206D+01,
     :  0.5078185134679D-10, 0.7346659893982D+00, 0.7477522907414D+01,
     :  0.3769343537061D-10, 0.1071317188367D+01, 0.7234794171227D+01,
     :  0.4980331365299D-10, 0.2500345341784D+01, 0.6836645152238D+01,
     :  0.3458236594757D-10, 0.3825159450711D+01, 0.1097707878456D+02,
     :  0.3578859493602D-10, 0.5299664791549D+01, 0.4164311961999D+01,
     :  0.3370504646419D-10, 0.5002316301593D+01, 0.1137170464392D+02,
     :  0.3299873338428D-10, 0.2526123275282D+01, 0.3930209696940D+01 /
      DATA ((E1(I,J,2),I=1,3),J= 61, 70) /
     :  0.4304917318409D-10, 0.3368078557132D+01, 0.1592596075957D+01,
     :  0.3402418753455D-10, 0.8385495425800D+00, 0.3128388763578D+01,
     :  0.2778460572146D-10, 0.3669905203240D+01, 0.7342457794669D+01,
     :  0.2782710128902D-10, 0.2691664812170D+00, 0.1748016358760D+01,
     :  0.2711725179646D-10, 0.4707487217718D+01, 0.5296909721118D+00,
     :  0.2981760946340D-10, 0.3190260867816D+00, 0.5368044267797D+00,
     :  0.2811672977772D-10, 0.3196532315372D+01, 0.7084896783808D+01,
     :  0.2863454474467D-10, 0.2263240324780D+00, 0.5223693906222D+01,
     :  0.3333464634051D-10, 0.3498451685065D+01, 0.8018209333619D+00,
     :  0.3312991747609D-10, 0.5839154477412D+01, 0.1554202828031D+00 /
      DATA ((E1(I,J,2),I=1,3),J= 71,NE1Y) /
     :  0.2813255564006D-10, 0.8268044346621D+00, 0.5225775174439D+00,
     :  0.2665098083966D-10, 0.3934021725360D+01, 0.5216580451554D+01,
     :  0.2349795705216D-10, 0.5197620913779D+01, 0.2146165377750D+01,
     :  0.2330352293961D-10, 0.2984999231807D+01, 0.1726015463500D+02,
     :  0.2728001683419D-10, 0.6521679638544D+00, 0.8635942003952D+01,
     :  0.2484061007669D-10, 0.3468955561097D+01, 0.5230807360890D+01,
     :  0.2646328768427D-10, 0.1013724533516D+01, 0.2629832328990D-01,
     :  0.2518630264831D-10, 0.6108081057122D+01, 0.5481254917084D+01,
     :  0.2421901455384D-10, 0.1651097776260D+01, 0.1349867339771D+01,
     :  0.6348533267831D-11, 0.3220226560321D+01, 0.8433466158131D+02 /

*  Sun-to-Earth, T^2, Y
      DATA ((E2(I,J,2),I=1,3),J=  1,NE2Y) /
     :  0.5063375872532D-10, 0.0000000000000D+00, 0.0000000000000D+00,
     :  0.2173815785980D-10, 0.2827805833053D+01, 0.1256615170089D+02,
     :  0.1010231999920D-10, 0.4634612377133D+01, 0.6283075850446D+01,
     :  0.9259745317636D-12, 0.2620612076189D+01, 0.1884922755134D+02,
     :  0.1022202095812D-12, 0.3809562326066D+01, 0.8399684731857D+02 /

*  Sun-to-Earth, T^0, Z
      DATA ((E0(I,J,3),I=1,3),J=  1, 10) /
     :  0.2796207639075D-05, 0.3198701560209D+01, 0.8433466158131D+02,
     :  0.1016042198142D-05, 0.5422360395913D+01, 0.5507553240374D+01,
     :  0.8044305033647D-06, 0.3880222866652D+01, 0.5223693906222D+01,
     :  0.4385347909274D-06, 0.3704369937468D+01, 0.2352866153506D+01,
     :  0.3186156414906D-06, 0.3999639363235D+01, 0.1577343543434D+01,
     :  0.2272412285792D-06, 0.3984738315952D+01, 0.1047747311755D+01,
     :  0.1645620103007D-06, 0.3565412516841D+01, 0.5856477690889D+01,
     :  0.1815836921166D-06, 0.4984507059020D+01, 0.6283075850446D+01,
     :  0.1447461676364D-06, 0.3702753570108D+01, 0.9437762937313D+01,
     :  0.1430760876382D-06, 0.3409658712357D+01, 0.1021328554739D+02 /
      DATA ((E0(I,J,3),I=1,3),J= 11, 20) /
     :  0.1120445753226D-06, 0.4829561570246D+01, 0.1414349524433D+02,
     :  0.1090232840797D-06, 0.2080729178066D+01, 0.6812766822558D+01,
     :  0.9715727346551D-07, 0.3476295881948D+01, 0.4694002934110D+01,
     :  0.1036267136217D-06, 0.4056639536648D+01, 0.7109288135493D+02,
     :  0.8752665271340D-07, 0.4448159519911D+01, 0.5753384878334D+01,
     :  0.8331864956004D-07, 0.4991704044208D+01, 0.7084896783808D+01,
     :  0.6901658670245D-07, 0.4325358994219D+01, 0.6275962395778D+01,
     :  0.9144536848998D-07, 0.1141826375363D+01, 0.6620890113188D+01,
     :  0.7205085037435D-07, 0.3624344170143D+01, 0.5296909721118D+00,
     :  0.7697874654176D-07, 0.5554257458998D+01, 0.1676215758509D+03 /
      DATA ((E0(I,J,3),I=1,3),J= 21, 30) /
     :  0.5197545738384D-07, 0.6251760961735D+01, 0.1807370494127D+02,
     :  0.5031345378608D-07, 0.2497341091913D+01, 0.4705732307012D+01,
     :  0.4527110205840D-07, 0.2335079920992D+01, 0.6309374173736D+01,
     :  0.4753355798089D-07, 0.7094148987474D+00, 0.5884926831456D+01,
     :  0.4296951977516D-07, 0.1101916352091D+01, 0.6681224869435D+01,
     :  0.3855341568387D-07, 0.1825495405486D+01, 0.5486777812467D+01,
     :  0.5253930970990D-07, 0.4424740687208D+01, 0.7860419393880D+01,
     :  0.4024630496471D-07, 0.5120498157053D+01, 0.1336797263425D+02,
     :  0.4061069791453D-07, 0.6029771435451D+01, 0.3930209696940D+01,
     :  0.3797883804205D-07, 0.4435193600836D+00, 0.3154687086868D+01 /
      DATA ((E0(I,J,3),I=1,3),J= 31, 40) /
     :  0.2933033225587D-07, 0.5124157356507D+01, 0.1059381944224D+01,
     :  0.3503000930426D-07, 0.5421830162065D+01, 0.6069776770667D+01,
     :  0.3670096214050D-07, 0.4582101667297D+01, 0.1219403291462D+02,
     :  0.2905609437008D-07, 0.1926566420072D+01, 0.1097707878456D+02,
     :  0.2466827821713D-07, 0.6090174539834D+00, 0.6496374930224D+01,
     :  0.2691647295332D-07, 0.1393432595077D+01, 0.2200391463820D+02,
     :  0.2150554667946D-07, 0.4308671715951D+01, 0.5643178611111D+01,
     :  0.2237481922680D-07, 0.8133968269414D+00, 0.8635942003952D+01,
     :  0.1817741038157D-07, 0.3755205127454D+01, 0.3340612434717D+01,
     :  0.2227820762132D-07, 0.2759558596664D+01, 0.1203646072878D+02 /
      DATA ((E0(I,J,3),I=1,3),J= 41, 50) /
     :  0.1944713772307D-07, 0.5699645869121D+01, 0.1179062909082D+02,
     :  0.1527340520662D-07, 0.1986749091746D+01, 0.3981490189893D+00,
     :  0.1577282574914D-07, 0.3205017217983D+01, 0.5088628793478D+01,
     :  0.1424738825424D-07, 0.6256747903666D+01, 0.2544314396739D+01,
     :  0.1616563121701D-07, 0.2601671259394D+00, 0.1729818233119D+02,
     :  0.1401210391692D-07, 0.4686939173506D+01, 0.7058598460518D+01,
     :  0.1488726974214D-07, 0.2815862451372D+01, 0.2593412433514D+02,
     :  0.1692626442388D-07, 0.4956894109797D+01, 0.1564752902480D+03,
     :  0.1123571582910D-07, 0.2381192697696D+01, 0.3738761453707D+01,
     :  0.9903308606317D-08, 0.4294851657684D+01, 0.9225539266174D+01 /
      DATA ((E0(I,J,3),I=1,3),J= 51, 60) /
     :  0.9174533187191D-08, 0.3075171510642D+01, 0.4164311961999D+01,
     :  0.8645985631457D-08, 0.5477534821633D+00, 0.8429241228195D+01,
     : -0.1085876492688D-07, 0.0000000000000D+00, 0.0000000000000D+00,
     :  0.9264309077815D-08, 0.5968571670097D+01, 0.7079373888424D+01,
     :  0.8243116984954D-08, 0.1489098777643D+01, 0.1044738781244D+02,
     :  0.8268102113708D-08, 0.3512977691983D+01, 0.1150676975667D+02,
     :  0.9043613988227D-08, 0.1290704408221D+00, 0.1101510648075D+02,
     :  0.7432912038789D-08, 0.1991086893337D+01, 0.2608790314060D+02,
     :  0.8586233727285D-08, 0.4238357924414D+01, 0.2986433403208D+02,
     :  0.7612230060131D-08, 0.2911090150166D+01, 0.4732030630302D+01 /
      DATA ((E0(I,J,3),I=1,3),J= 61, 70) /
     :  0.7097787751408D-08, 0.1908938392390D+01, 0.8031092209206D+01,
     :  0.7640237040175D-08, 0.6129219000168D+00, 0.7962980379786D+00,
     :  0.7070445688081D-08, 0.1380417036651D+01, 0.2146165377750D+01,
     :  0.7690770957702D-08, 0.1680504249084D+01, 0.2122839202813D+02,
     :  0.8051292542594D-08, 0.5127423484511D+01, 0.2942463415728D+01,
     :  0.5902709104515D-08, 0.2020274190917D+01, 0.7755226100720D+00,
     :  0.5134567496462D-08, 0.2606778676418D+01, 0.1256615170089D+02,
     :  0.5525802046102D-08, 0.1613011769663D+01, 0.8018209333619D+00,
     :  0.5880724784221D-08, 0.4604483417236D+01, 0.4690479774488D+01,
     :  0.5211699081370D-08, 0.5718964114193D+01, 0.8827390247185D+01 /
      DATA ((E0(I,J,3),I=1,3),J= 71, 80) /
     :  0.4891849573562D-08, 0.3689658932196D+01, 0.2132990797783D+00,
     :  0.5150246069997D-08, 0.4099769855122D+01, 0.6480980550449D+02,
     :  0.5102434319633D-08, 0.5660834602509D+01, 0.3379454372902D+02,
     :  0.5083405254252D-08, 0.9842221218974D+00, 0.4136910472696D+01,
     :  0.4206562585682D-08, 0.1341363634163D+00, 0.3128388763578D+01,
     :  0.4663249683579D-08, 0.8130132735866D+00, 0.5216580451554D+01,
     :  0.4099474416530D-08, 0.5791497770644D+01, 0.4265981595566D+00,
     :  0.4628251220767D-08, 0.1249802769331D+01, 0.1572083878776D+02,
     :  0.5024068728142D-08, 0.4795684802743D+01, 0.6290189305114D+01,
     :  0.5120234327758D-08, 0.3810420387208D+01, 0.5230807360890D+01 /
      DATA ((E0(I,J,3),I=1,3),J= 81, 90) /
     :  0.5524029815280D-08, 0.1029264714351D+01, 0.2397622045175D+03,
     :  0.4757415718860D-08, 0.3528044781779D+01, 0.1649636139783D+02,
     :  0.3915786131127D-08, 0.5593889282646D+01, 0.1589072916335D+01,
     :  0.4869053149991D-08, 0.3299636454433D+01, 0.7632943190217D+01,
     :  0.3649365703729D-08, 0.1286049002584D+01, 0.6206810014183D+01,
     :  0.3992493949002D-08, 0.3100307589464D+01, 0.2515860172507D+02,
     :  0.3320247477418D-08, 0.6212683940807D+01, 0.1216800268190D+02,
     :  0.3287123739696D-08, 0.4699118445928D+01, 0.7234794171227D+01,
     :  0.3472776811103D-08, 0.2630507142004D+01, 0.7342457794669D+01,
     :  0.3423253294767D-08, 0.2946432844305D+01, 0.9623688285163D+01 /
      DATA ((E0(I,J,3),I=1,3),J= 91,100) /
     :  0.3896173898244D-08, 0.1224834179264D+01, 0.6438496133249D+01,
     :  0.3388455337924D-08, 0.1543807616351D+01, 0.1494531617769D+02,
     :  0.3062704716523D-08, 0.1191777572310D+01, 0.8662240327241D+01,
     :  0.3270075600400D-08, 0.5483498767737D+01, 0.1194447056968D+01,
     :  0.3101209215259D-08, 0.8000833804348D+00, 0.3772475342596D+02,
     :  0.2780883347311D-08, 0.4077980721888D+00, 0.5863591145557D+01,
     :  0.2903605931824D-08, 0.2617490302147D+01, 0.1965104848470D+02,
     :  0.2682014743119D-08, 0.2634703158290D+01, 0.7238675589263D+01,
     :  0.2534360108492D-08, 0.6102446114873D+01, 0.6836645152238D+01,
     :  0.2392564882509D-08, 0.3681820208691D+01, 0.5849364236221D+01 /
      DATA ((E0(I,J,3),I=1,3),J=101,110) /
     :  0.2656667254856D-08, 0.6216045388886D+01, 0.6133512519065D+01,
     :  0.2331242096773D-08, 0.5864949777744D+01, 0.4535059491685D+01,
     :  0.2287898363668D-08, 0.4566628532802D+01, 0.7477522907414D+01,
     :  0.2336944521306D-08, 0.2442722126930D+01, 0.1137170464392D+02,
     :  0.3156632236269D-08, 0.1626628050682D+01, 0.2509084901204D+03,
     :  0.2982612402766D-08, 0.2803604512609D+01, 0.1748016358760D+01,
     :  0.2774031674807D-08, 0.4654002897158D+01, 0.8223916695780D+02,
     :  0.2295236548638D-08, 0.4326518333253D+01, 0.3378142627421D+00,
     :  0.2190714699873D-08, 0.4519614578328D+01, 0.2908881142201D+02,
     :  0.2191495845045D-08, 0.3012626912549D+01, 0.1673046366289D+02 /
      DATA ((E0(I,J,3),I=1,3),J=111,120) /
     :  0.2492901628386D-08, 0.1290101424052D+00, 0.1543797956245D+03,
     :  0.1993778064319D-08, 0.3864046799414D+01, 0.1778984560711D+02,
     :  0.1898146479022D-08, 0.5053777235891D+01, 0.2042657109477D+02,
     :  0.1918280127634D-08, 0.2222470192548D+01, 0.4165496312290D+02,
     :  0.1916351061607D-08, 0.8719067257774D+00, 0.7737595720538D+02,
     :  0.1834720181466D-08, 0.4031491098040D+01, 0.2358125818164D+02,
     :  0.1249201523806D-08, 0.5938379466835D+01, 0.3301902111895D+02,
     :  0.1477304050539D-08, 0.6544722606797D+00, 0.9548094718417D+02,
     :  0.1264316431249D-08, 0.2059072853236D+01, 0.8399684731857D+02,
     :  0.1203526495039D-08, 0.3644813532605D+01, 0.4558517281984D+02 /
      DATA ((E0(I,J,3),I=1,3),J=121,130) /
     :  0.9221681059831D-09, 0.3241815055602D+01, 0.7805158573086D+02,
     :  0.7849278367646D-09, 0.5043812342457D+01, 0.5217580628120D+02,
     :  0.7983392077387D-09, 0.5000024502753D+01, 0.1501922143975D+03,
     :  0.7925395431654D-09, 0.1398734871821D-01, 0.9061773743175D+02,
     :  0.7640473285886D-09, 0.5067111723130D+01, 0.4951538251678D+02,
     :  0.5398937754482D-09, 0.5597382200075D+01, 0.1613385000004D+03,
     :  0.5626247550193D-09, 0.2601338209422D+01, 0.7318837597844D+02,
     :  0.5525197197855D-09, 0.5814832109256D+01, 0.1432335100216D+03,
     :  0.5407629837898D-09, 0.3384820609076D+01, 0.3230491187871D+03,
     :  0.3856739119801D-09, 0.1072391840473D+01, 0.2334791286671D+03 /
      DATA ((E0(I,J,3),I=1,3),J=131,NE0Z) /
     :  0.3856425239987D-09, 0.2369540393327D+01, 0.1739046517013D+03,
     :  0.4350867755983D-09, 0.5255575751082D+01, 0.1620484330494D+03,
     :  0.3844113924996D-09, 0.5482356246182D+01, 0.9757644180768D+02,
     :  0.2854869155431D-09, 0.9573634763143D+00, 0.1697170704744D+03,
     :  0.1719227671416D-09, 0.1887203025202D+01, 0.2265204242912D+03,
     :  0.1527846879755D-09, 0.3982183931157D+01, 0.3341954043900D+03,
     :  0.1128229264847D-09, 0.2787457156298D+01, 0.3119028331842D+03 /

*  Sun-to-Earth, T^1, Z
      DATA ((E1(I,J,3),I=1,3),J=  1, 10) /
     :  0.2278290449966D-05, 0.3413716033863D+01, 0.6283075850446D+01,
     :  0.5429458209830D-07, 0.0000000000000D+00, 0.0000000000000D+00,
     :  0.1903240492525D-07, 0.3370592358297D+01, 0.1256615170089D+02,
     :  0.2385409276743D-09, 0.3327914718416D+01, 0.1884922755134D+02,
     :  0.8676928342573D-10, 0.1824006811264D+01, 0.5223693906222D+01,
     :  0.7765442593544D-10, 0.3888564279247D+01, 0.5507553240374D+01,
     :  0.7066158332715D-10, 0.5194267231944D+01, 0.2352866153506D+01,
     :  0.7092175288657D-10, 0.2333246960021D+01, 0.8399684731857D+02,
     :  0.5357582213535D-10, 0.2224031176619D+01, 0.5296909721118D+00,
     :  0.3828035865021D-10, 0.2156710933584D+01, 0.6279552690824D+01 /
      DATA ((E1(I,J,3),I=1,3),J= 11,NE1Z) /
     :  0.3824857220427D-10, 0.1529755219915D+01, 0.6286599010068D+01,
     :  0.3286995181628D-10, 0.4879512900483D+01, 0.1021328554739D+02 /

*  Sun-to-Earth, T^2, Z
      DATA ((E2(I,J,3),I=1,3),J=  1,NE2Z) /
     :  0.9722666114891D-10, 0.5152219582658D+01, 0.6283075850446D+01,
     : -0.3494819171909D-11, 0.0000000000000D+00, 0.0000000000000D+00,
     :  0.6713034376076D-12, 0.6440188750495D+00, 0.1256615170089D+02 /

*  SSB-to-Sun, T^0, X
      DATA ((S0(I,J,1),I=1,3),J=  1, 10) /
     :  0.4956757536410D-02, 0.3741073751789D+01, 0.5296909721118D+00,
     :  0.2718490072522D-02, 0.4016011511425D+01, 0.2132990797783D+00,
     :  0.1546493974344D-02, 0.2170528330642D+01, 0.3813291813120D-01,
     :  0.8366855276341D-03, 0.2339614075294D+01, 0.7478166569050D-01,
     :  0.2936777942117D-03, 0.0000000000000D+00, 0.0000000000000D+00,
     :  0.1201317439469D-03, 0.4090736353305D+01, 0.1059381944224D+01,
     :  0.7578550887230D-04, 0.3241518088140D+01, 0.4265981595566D+00,
     :  0.1941787367773D-04, 0.1012202064330D+01, 0.2061856251104D+00,
     :  0.1889227765991D-04, 0.3892520416440D+01, 0.2204125344462D+00,
     :  0.1937896968613D-04, 0.4797779441161D+01, 0.1495633313810D+00 /
      DATA ((S0(I,J,1),I=1,3),J= 11, 20) /
     :  0.1434506110873D-04, 0.3868960697933D+01, 0.5225775174439D+00,
     :  0.1406659911580D-04, 0.4759766557397D+00, 0.5368044267797D+00,
     :  0.1179022300202D-04, 0.7774961520598D+00, 0.7626583626240D-01,
     :  0.8085864460959D-05, 0.3254654471465D+01, 0.3664874755930D-01,
     :  0.7622752967615D-05, 0.4227633103489D+01, 0.3961708870310D-01,
     :  0.6209171139066D-05, 0.2791828325711D+00, 0.7329749511860D-01,
     :  0.4366435633970D-05, 0.4440454875925D+01, 0.1589072916335D+01,
     :  0.3792124889348D-05, 0.5156393842356D+01, 0.7113454667900D-02,
     :  0.3154548963402D-05, 0.6157005730093D+01, 0.4194847048887D+00,
     :  0.3088359882942D-05, 0.2494567553163D+01, 0.6398972393349D+00 /
      DATA ((S0(I,J,1),I=1,3),J= 21, 30) /
     :  0.2788440902136D-05, 0.4934318747989D+01, 0.1102062672231D+00,
     :  0.3039928456376D-05, 0.4895077702640D+01, 0.6283075850446D+01,
     :  0.2272258457679D-05, 0.5278394064764D+01, 0.1030928125552D+00,
     :  0.2162007057957D-05, 0.5802978019099D+01, 0.3163918923335D+00,
     :  0.1767632855737D-05, 0.3415346595193D-01, 0.1021328554739D+02,
     :  0.1349413459362D-05, 0.2001643230755D+01, 0.1484170571900D-02,
     :  0.1170141900476D-05, 0.2424750491620D+01, 0.6327837846670D+00,
     :  0.1054355266820D-05, 0.3123311487576D+01, 0.4337116142245D+00,
     :  0.9800822461610D-06, 0.3026258088130D+01, 0.1052268489556D+01,
     :  0.1091203749931D-05, 0.3157811670347D+01, 0.1162474756779D+01 /
      DATA ((S0(I,J,1),I=1,3),J= 31, 40) /
     :  0.6960236715913D-06, 0.8219570542313D+00, 0.1066495398892D+01,
     :  0.5689257296909D-06, 0.1323052375236D+01, 0.9491756770005D+00,
     :  0.6613172135802D-06, 0.2765348881598D+00, 0.8460828644453D+00,
     :  0.6277702517571D-06, 0.5794064466382D+01, 0.1480791608091D+00,
     :  0.6304884066699D-06, 0.7323555380787D+00, 0.2243449970715D+00,
     :  0.4897850467382D-06, 0.3062464235399D+01, 0.3340612434717D+01,
     :  0.3759148598786D-06, 0.4588290469664D+01, 0.3516457698740D-01,
     :  0.3110520548195D-06, 0.1374299536572D+01, 0.6373574839730D-01,
     :  0.3064708359780D-06, 0.4222267485047D+01, 0.1104591729320D-01,
     :  0.2856347168241D-06, 0.3714202944973D+01, 0.1510475019529D+00 /
      DATA ((S0(I,J,1),I=1,3),J= 41, 50) /
     :  0.2840945514288D-06, 0.2847972875882D+01, 0.4110125927500D-01,
     :  0.2378951599405D-06, 0.3762072563388D+01, 0.2275259891141D+00,
     :  0.2714229481417D-06, 0.1036049980031D+01, 0.2535050500000D-01,
     :  0.2323551717307D-06, 0.4682388599076D+00, 0.8582758298370D-01,
     :  0.1881790512219D-06, 0.4790565425418D+01, 0.2118763888447D+01,
     :  0.2261353968371D-06, 0.1669144912212D+01, 0.7181332454670D-01,
     :  0.2214546389848D-06, 0.3937717281614D+01, 0.2968341143800D-02,
     :  0.2184915594933D-06, 0.1129169845099D+00, 0.7775000683430D-01,
     :  0.2000164937936D-06, 0.4030009638488D+01, 0.2093666171530D+00,
     :  0.1966105136719D-06, 0.8745955786834D+00, 0.2172315424036D+00 /
      DATA ((S0(I,J,1),I=1,3),J= 51, 60) /
     :  0.1904742332624D-06, 0.5919743598964D+01, 0.2022531624851D+00,
     :  0.1657399705031D-06, 0.2549141484884D+01, 0.7358765972222D+00,
     :  0.1574070533987D-06, 0.5277533020230D+01, 0.7429900518901D+00,
     :  0.1832261651039D-06, 0.3064688127777D+01, 0.3235053470014D+00,
     :  0.1733615346569D-06, 0.3011432799094D+01, 0.1385174140878D+00,
     :  0.1549124014496D-06, 0.4005569132359D+01, 0.5154640627760D+00,
     :  0.1637044713838D-06, 0.1831375966632D+01, 0.8531963191132D+00,
     :  0.1123420082383D-06, 0.1180270407578D+01, 0.1990721704425D+00,
     :  0.1083754165740D-06, 0.3414101320863D+00, 0.5439178814476D+00,
     :  0.1156638012655D-06, 0.6130479452594D+00, 0.5257585094865D+00 /
      DATA ((S0(I,J,1),I=1,3),J= 61, 70) /
     :  0.1142548785134D-06, 0.3724761948846D+01, 0.5336234347371D+00,
     :  0.7921463895965D-07, 0.2435425589361D+01, 0.1478866649112D+01,
     :  0.7428600285231D-07, 0.3542144398753D+01, 0.2164800718209D+00,
     :  0.8323211246747D-07, 0.3525058072354D+01, 0.1692165728891D+01,
     :  0.7257595116312D-07, 0.1364299431982D+01, 0.2101180877357D+00,
     :  0.7111185833236D-07, 0.2460478875808D+01, 0.4155522422634D+00,
     :  0.6868090383716D-07, 0.4397327670704D+01, 0.1173197218910D+00,
     :  0.7226419974175D-07, 0.4042647308905D+01, 0.1265567569334D+01,
     :  0.6955642383177D-07, 0.2865047906085D+01, 0.9562891316684D+00,
     :  0.7492139296331D-07, 0.5014278994215D+01, 0.1422690933580D-01 /
      DATA ((S0(I,J,1),I=1,3),J= 71, 80) /
     :  0.6598363128857D-07, 0.2376730020492D+01, 0.6470106940028D+00,
     :  0.7381147293385D-07, 0.3272990384244D+01, 0.1581959461667D+01,
     :  0.6402909624032D-07, 0.5302290955138D+01, 0.9597935788730D-01,
     :  0.6237454263857D-07, 0.5444144425332D+01, 0.7084920306520D-01,
     :  0.5241198544016D-07, 0.4215359579205D+01, 0.5265099800692D+00,
     :  0.5144463853918D-07, 0.1218916689916D+00, 0.5328719641544D+00,
     :  0.5868164772299D-07, 0.2369402002213D+01, 0.7871412831580D-01,
     :  0.6233195669151D-07, 0.1254922242403D+01, 0.2608790314060D+02,
     :  0.6068463791422D-07, 0.5679713760431D+01, 0.1114304132498D+00,
     :  0.4359361135065D-07, 0.6097219641646D+00, 0.1375773836557D+01 /
      DATA ((S0(I,J,1),I=1,3),J= 81, 90) /
     :  0.4686510366826D-07, 0.4786231041431D+01, 0.1143987543936D+00,
     :  0.3758977287225D-07, 0.1167368068139D+01, 0.1596186371003D+01,
     :  0.4282051974778D-07, 0.1519471064319D+01, 0.2770348281756D+00,
     :  0.5153765386113D-07, 0.1860532322984D+01, 0.2228608264996D+00,
     :  0.4575129387188D-07, 0.7632857887158D+00, 0.1465949902372D+00,
     :  0.3326844933286D-07, 0.1298219485285D+01, 0.5070101000000D-01,
     :  0.3748617450984D-07, 0.1046510321062D+01, 0.4903339079539D+00,
     :  0.2816756661499D-07, 0.3434522346190D+01, 0.2991266627620D+00,
     :  0.3412750405039D-07, 0.2523766270318D+01, 0.3518164938661D+00,
     :  0.2655796761776D-07, 0.2904422260194D+01, 0.6256703299991D+00 /
      DATA ((S0(I,J,1),I=1,3),J= 91,100) /
     :  0.2963597929458D-07, 0.5923900431149D+00, 0.1099462426779D+00,
     :  0.2539523734781D-07, 0.4851947722567D+01, 0.1256615170089D+02,
     :  0.2283087914139D-07, 0.3400498595496D+01, 0.6681224869435D+01,
     :  0.2321309799331D-07, 0.5789099148673D+01, 0.3368040641550D-01,
     :  0.2549657649750D-07, 0.3991856479792D-01, 0.1169588211447D+01,
     :  0.2290462303977D-07, 0.2788567577052D+01, 0.1045155034888D+01,
     :  0.1945398522914D-07, 0.3290896998176D+01, 0.1155361302111D+01,
     :  0.1849171512638D-07, 0.2698060129367D+01, 0.4452511715700D-02,
     :  0.1647199834254D-07, 0.3016735644085D+01, 0.4408250688924D+00,
     :  0.1529530765273D-07, 0.5573043116178D+01, 0.6521991896920D-01 /
      DATA ((S0(I,J,1),I=1,3),J=101,110) /
     :  0.1433199339978D-07, 0.1481192356147D+01, 0.9420622223326D+00,
     :  0.1729134193602D-07, 0.1422817538933D+01, 0.2108507877249D+00,
     :  0.1716463931346D-07, 0.3469468901855D+01, 0.2157473718317D+00,
     :  0.1391206061378D-07, 0.6122436220547D+01, 0.4123712502208D+00,
     :  0.1404746661924D-07, 0.1647765641936D+01, 0.4258542984690D-01,
     :  0.1410452399455D-07, 0.5989729161964D+01, 0.2258291676434D+00,
     :  0.1089828772168D-07, 0.2833705509371D+01, 0.4226656969313D+00,
     :  0.1047374564948D-07, 0.5090690007331D+00, 0.3092784376656D+00,
     :  0.1358279126532D-07, 0.5128990262836D+01, 0.7923417740620D-01,
     :  0.1020456476148D-07, 0.9632772880808D+00, 0.1456308687557D+00 /
      DATA ((S0(I,J,1),I=1,3),J=111,120) /
     :  0.1033428735328D-07, 0.3223779318418D+01, 0.1795258541446D+01,
     :  0.1412435841540D-07, 0.2410271572721D+01, 0.1525316725248D+00,
     :  0.9722759371574D-08, 0.2333531395690D+01, 0.8434341241180D-01,
     :  0.9657334084704D-08, 0.6199270974168D+01, 0.1272681024002D+01,
     :  0.1083641148690D-07, 0.2864222292929D+01, 0.7032915397480D-01,
     :  0.1067318403838D-07, 0.5833458866568D+00, 0.2123349582968D+00,
     :  0.1062366201976D-07, 0.4307753989494D+01, 0.2142632012598D+00,
     :  0.1236364149266D-07, 0.2873917870593D+01, 0.1847279083684D+00,
     :  0.1092759489593D-07, 0.2959887266733D+01, 0.1370332435159D+00,
     :  0.8912069362899D-08, 0.5141213702562D+01, 0.2648454860559D+01 /
      DATA ((S0(I,J,1),I=1,3),J=121,130) /
     :  0.9656467707970D-08, 0.4532182462323D+01, 0.4376440768498D+00,
     :  0.8098386150135D-08, 0.2268906338379D+01, 0.2880807454688D+00,
     :  0.7857714675000D-08, 0.4055544260745D+01, 0.2037373330570D+00,
     :  0.7288455940646D-08, 0.5357901655142D+01, 0.1129145838217D+00,
     :  0.9450595950552D-08, 0.4264926963939D+01, 0.5272426800584D+00,
     :  0.9381718247537D-08, 0.7489366976576D-01, 0.5321392641652D+00,
     :  0.7079052646038D-08, 0.1923311052874D+01, 0.6288513220417D+00,
     :  0.9259004415344D-08, 0.2970256853438D+01, 0.1606092486742D+00,
     :  0.8259801499742D-08, 0.3327056314697D+01, 0.8389694097774D+00,
     :  0.6476334355779D-08, 0.2954925505727D+01, 0.2008557621224D+01 /
      DATA ((S0(I,J,1),I=1,3),J=131,140) /
     :  0.5984021492007D-08, 0.9138753105829D+00, 0.2042657109477D+02,
     :  0.5989546863181D-08, 0.3244464082031D+01, 0.2111650433779D+01,
     :  0.6233108606023D-08, 0.4995232638403D+00, 0.4305306221819D+00,
     :  0.6877299149965D-08, 0.2834987233449D+01, 0.9561746721300D-02,
     :  0.8311234227190D-08, 0.2202951835758D+01, 0.3801276407308D+00,
     :  0.6599472832414D-08, 0.4478581462618D+01, 0.1063314406849D+01,
     :  0.6160491096549D-08, 0.5145858696411D+01, 0.1368660381889D+01,
     :  0.6164772043891D-08, 0.3762976697911D+00, 0.4234171675140D+00,
     :  0.6363248684450D-08, 0.3162246718685D+01, 0.1253008786510D-01,
     :  0.6448587520999D-08, 0.3442693302119D+01, 0.5287268506303D+00 /
      DATA ((S0(I,J,1),I=1,3),J=141,150) /
     :  0.6431662283977D-08, 0.8977549136606D+00, 0.5306550935933D+00,
     :  0.6351223158474D-08, 0.4306447410369D+01, 0.5217580628120D+02,
     :  0.5476721393451D-08, 0.3888529177855D+01, 0.2221856701002D+01,
     :  0.5341772572619D-08, 0.2655560662512D+01, 0.7466759693650D-01,
     :  0.5337055758302D-08, 0.5164990735946D+01, 0.7489573444450D-01,
     :  0.5373120816787D-08, 0.6041214553456D+01, 0.1274714967946D+00,
     :  0.5392351705426D-08, 0.9177763485932D+00, 0.1055449481598D+01,
     :  0.6688495850205D-08, 0.3089608126937D+01, 0.2213766559277D+00,
     :  0.5072003660362D-08, 0.4311316541553D+01, 0.2132517061319D+00,
     :  0.5070726650455D-08, 0.5790675464444D+00, 0.2133464534247D+00 /
      DATA ((S0(I,J,1),I=1,3),J=151,160) /
     :  0.5658012950032D-08, 0.2703945510675D+01, 0.7287631425543D+00,
     :  0.4835509924854D-08, 0.2975422976065D+01, 0.7160067364790D-01,
     :  0.6479821978012D-08, 0.1324168733114D+01, 0.2209183458640D-01,
     :  0.6230636494980D-08, 0.2860103632836D+01, 0.3306188016693D+00,
     :  0.4649239516213D-08, 0.4832259763403D+01, 0.7796265773310D-01,
     :  0.6487325792700D-08, 0.2726165825042D+01, 0.3884652414254D+00,
     :  0.4682823682770D-08, 0.6966602455408D+00, 0.1073608853559D+01,
     :  0.5704230804976D-08, 0.5669634104606D+01, 0.8731175355560D-01,
     :  0.6125413585489D-08, 0.1513386538915D+01, 0.7605151500000D-01,
     :  0.6035825038187D-08, 0.1983509168227D+01, 0.9846002785331D+00 /
      DATA ((S0(I,J,1),I=1,3),J=161,170) /
     :  0.4331123462303D-08, 0.2782892992807D+01, 0.4297791515992D+00,
     :  0.4681107685143D-08, 0.5337232886836D+01, 0.2127790306879D+00,
     :  0.4669105829655D-08, 0.5837133792160D+01, 0.2138191288687D+00,
     :  0.5138823602365D-08, 0.3080560200507D+01, 0.7233337363710D-01,
     :  0.4615856664534D-08, 0.1661747897471D+01, 0.8603097737811D+00,
     :  0.4496916702197D-08, 0.2112508027068D+01, 0.7381754420900D-01,
     :  0.4278479042945D-08, 0.5716528462627D+01, 0.7574578717200D-01,
     :  0.3840525503932D-08, 0.6424172726492D+00, 0.3407705765729D+00,
     :  0.4866636509685D-08, 0.4919244697715D+01, 0.7722995774390D-01,
     :  0.3526100639296D-08, 0.2550821052734D+01, 0.6225157782540D-01 /
      DATA ((S0(I,J,1),I=1,3),J=171,180) /
     :  0.3939558488075D-08, 0.3939331491710D+01, 0.5268983110410D-01,
     :  0.4041268772576D-08, 0.2275337571218D+01, 0.3503323232942D+00,
     :  0.3948761842853D-08, 0.1999324200790D+01, 0.1451108196653D+00,
     :  0.3258394550029D-08, 0.9121001378200D+00, 0.5296435984654D+00,
     :  0.3257897048761D-08, 0.3428428660869D+01, 0.5297383457582D+00,
     :  0.3842559031298D-08, 0.6132927720035D+01, 0.9098186128426D+00,
     :  0.3109920095448D-08, 0.7693650193003D+00, 0.3932462625300D-02,
     :  0.3132237775119D-08, 0.3621293854908D+01, 0.2346394437820D+00,
     :  0.3942189421510D-08, 0.4841863659733D+01, 0.3180992042600D-02,
     :  0.3796972285340D-08, 0.1814174994268D+01, 0.1862120789403D+00 /
      DATA ((S0(I,J,1),I=1,3),J=181,190) /
     :  0.3995640233688D-08, 0.1386990406091D+01, 0.4549093064213D+00,
     :  0.2875013727414D-08, 0.9178318587177D+00, 0.1905464808669D+01,
     :  0.3073719932844D-08, 0.2688923811835D+01, 0.3628624111593D+00,
     :  0.2731016580075D-08, 0.1188259127584D+01, 0.2131850110243D+00,
     :  0.2729549896546D-08, 0.3702160634273D+01, 0.2134131485323D+00,
     :  0.3339372892449D-08, 0.7199163960331D+00, 0.2007689919132D+00,
     :  0.2898833764204D-08, 0.1916709364999D+01, 0.5291709230214D+00,
     :  0.2894536549362D-08, 0.2424043195547D+01, 0.5302110212022D+00,
     :  0.3096872473843D-08, 0.4445894977497D+01, 0.2976424921901D+00,
     :  0.2635672326810D-08, 0.3814366984117D+01, 0.1485980103780D+01 /
      DATA ((S0(I,J,1),I=1,3),J=191,200) /
     :  0.3649302697001D-08, 0.2924200596084D+01, 0.6044726378023D+00,
     :  0.3127954585895D-08, 0.1842251648327D+01, 0.1084620721060D+00,
     :  0.2616040173947D-08, 0.4155841921984D+01, 0.1258454114666D+01,
     :  0.2597395859860D-08, 0.1158045978874D+00, 0.2103781122809D+00,
     :  0.2593286172210D-08, 0.4771850408691D+01, 0.2162200472757D+00,
     :  0.2481823585747D-08, 0.4608842558889D+00, 0.1062562936266D+01,
     :  0.2742219550725D-08, 0.1538781127028D+01, 0.5651155736444D+00,
     :  0.3199558469610D-08, 0.3226647822878D+00, 0.7036329877322D+00,
     :  0.2666088542957D-08, 0.1967991731219D+00, 0.1400015846597D+00,
     :  0.2397067430580D-08, 0.3707036669873D+01, 0.2125476091956D+00 /
      DATA ((S0(I,J,1),I=1,3),J=201,210) /
     :  0.2376570772738D-08, 0.1182086628042D+01, 0.2140505503610D+00,
     :  0.2547228007887D-08, 0.4906256820629D+01, 0.1534957940063D+00,
     :  0.2265575594114D-08, 0.3414949866857D+01, 0.2235935264888D+00,
     :  0.2464381430585D-08, 0.4599122275378D+01, 0.2091065926078D+00,
     :  0.2433408527044D-08, 0.2830751145445D+00, 0.2174915669488D+00,
     :  0.2443605509076D-08, 0.4212046432538D+01, 0.1739420156204D+00,
     :  0.2319779262465D-08, 0.9881978408630D+00, 0.7530171478090D-01,
     :  0.2284622835465D-08, 0.5565347331588D+00, 0.7426161660010D-01,
     :  0.2467268750783D-08, 0.5655708150766D+00, 0.2526561439362D+00,
     :  0.2808513492782D-08, 0.1418405053408D+01, 0.5636314030725D+00 /
      DATA ((S0(I,J,1),I=1,3),J=211,NS0X) /
     :  0.2329528932532D-08, 0.4069557545675D+01, 0.1056200952181D+01,
     :  0.9698639532817D-09, 0.1074134313634D+01, 0.7826370942180D+02 /

*  SSB-to-Sun, T^1, X
      DATA ((S1(I,J,1),I=1,3),J=  1, 10) /
     : -0.1296310361520D-07, 0.0000000000000D+00, 0.0000000000000D+00,
     :  0.8975769009438D-08, 0.1128891609250D+01, 0.4265981595566D+00,
     :  0.7771113441307D-08, 0.2706039877077D+01, 0.2061856251104D+00,
     :  0.7538303866642D-08, 0.2191281289498D+01, 0.2204125344462D+00,
     :  0.6061384579336D-08, 0.3248167319958D+01, 0.1059381944224D+01,
     :  0.5726994235594D-08, 0.5569981398610D+01, 0.5225775174439D+00,
     :  0.5616492836424D-08, 0.5057386614909D+01, 0.5368044267797D+00,
     :  0.1010881584769D-08, 0.3473577116095D+01, 0.7113454667900D-02,
     :  0.7259606157626D-09, 0.3651858593665D+00, 0.6398972393349D+00,
     :  0.8755095026935D-09, 0.1662835408338D+01, 0.4194847048887D+00 /
      DATA ((S1(I,J,1),I=1,3),J= 11, 20) /
     :  0.5370491182812D-09, 0.1327673878077D+01, 0.4337116142245D+00,
     :  0.5743773887665D-09, 0.4250200846687D+01, 0.2132990797783D+00,
     :  0.4408103140300D-09, 0.3598752574277D+01, 0.1589072916335D+01,
     :  0.3101892374445D-09, 0.4887822983319D+01, 0.1052268489556D+01,
     :  0.3209453713578D-09, 0.9702272295114D+00, 0.5296909721118D+00,
     :  0.3017228286064D-09, 0.5484462275949D+01, 0.1066495398892D+01,
     :  0.3200700038601D-09, 0.2846613338643D+01, 0.1495633313810D+00,
     :  0.2137637279911D-09, 0.5692163292729D+00, 0.3163918923335D+00,
     :  0.1899686386727D-09, 0.2061077157189D+01, 0.2275259891141D+00,
     :  0.1401994545308D-09, 0.4177771136967D+01, 0.1102062672231D+00 /
      DATA ((S1(I,J,1),I=1,3),J= 21, 30) /
     :  0.1578057810499D-09, 0.5782460597335D+01, 0.7626583626240D-01,
     :  0.1237713253351D-09, 0.5705900866881D+01, 0.5154640627760D+00,
     :  0.1313076837395D-09, 0.5163438179576D+01, 0.3664874755930D-01,
     :  0.1184963304860D-09, 0.3054804427242D+01, 0.6327837846670D+00,
     :  0.1238130878565D-09, 0.2317292575962D+01, 0.3961708870310D-01,
     :  0.1015959527736D-09, 0.2194643645526D+01, 0.7329749511860D-01,
     :  0.9017954423714D-10, 0.2868603545435D+01, 0.1990721704425D+00,
     :  0.8668024955603D-10, 0.4923849675082D+01, 0.5439178814476D+00,
     :  0.7756083930103D-10, 0.3014334135200D+01, 0.9491756770005D+00,
     :  0.7536503401741D-10, 0.2704886279769D+01, 0.1030928125552D+00 /
      DATA ((S1(I,J,1),I=1,3),J= 31, 40) /
     :  0.5483308679332D-10, 0.6010983673799D+01, 0.8531963191132D+00,
     :  0.5184339620428D-10, 0.1952704573291D+01, 0.2093666171530D+00,
     :  0.5108658712030D-10, 0.2958575786649D+01, 0.2172315424036D+00,
     :  0.5019424524650D-10, 0.1736317621318D+01, 0.2164800718209D+00,
     :  0.4909312625978D-10, 0.3167216416257D+01, 0.2101180877357D+00,
     :  0.4456638901107D-10, 0.7697579923471D+00, 0.3235053470014D+00,
     :  0.4227030350925D-10, 0.3490910137928D+01, 0.6373574839730D-01,
     :  0.4095456040093D-10, 0.5178888984491D+00, 0.6470106940028D+00,
     :  0.4990537041422D-10, 0.3323887668974D+01, 0.1422690933580D-01,
     :  0.4321170010845D-10, 0.4288484987118D+01, 0.7358765972222D+00 /
      DATA ((S1(I,J,1),I=1,3),J= 41,NS1X) /
     :  0.3544072091802D-10, 0.6021051579251D+01, 0.5265099800692D+00,
     :  0.3480198638687D-10, 0.4600027054714D+01, 0.5328719641544D+00,
     :  0.3440287244435D-10, 0.4349525970742D+01, 0.8582758298370D-01,
     :  0.3330628322713D-10, 0.2347391505082D+01, 0.1104591729320D-01,
     :  0.2973060707184D-10, 0.4789409286400D+01, 0.5257585094865D+00,
     :  0.2932606766089D-10, 0.5831693799927D+01, 0.5336234347371D+00,
     :  0.2876972310953D-10, 0.2692638514771D+01, 0.1173197218910D+00,
     :  0.2827488278556D-10, 0.2056052487960D+01, 0.2022531624851D+00,
     :  0.2515028239756D-10, 0.7411863262449D+00, 0.9597935788730D-01,
     :  0.2853033744415D-10, 0.3948481024894D+01, 0.2118763888447D+01 /

*  SSB-to-Sun, T^2, X
      DATA ((S2(I,J,1),I=1,3),J=  1,NS2X) /
     :  0.1603551636587D-11, 0.4404109410481D+01, 0.2061856251104D+00,
     :  0.1556935889384D-11, 0.4818040873603D+00, 0.2204125344462D+00,
     :  0.1182594414915D-11, 0.9935762734472D+00, 0.5225775174439D+00,
     :  0.1158794583180D-11, 0.3353180966450D+01, 0.5368044267797D+00,
     :  0.9597358943932D-12, 0.5567045358298D+01, 0.2132990797783D+00,
     :  0.6511516579605D-12, 0.5630872420788D+01, 0.4265981595566D+00,
     :  0.7419792747688D-12, 0.2156188581957D+01, 0.5296909721118D+00,
     :  0.3951972655848D-12, 0.1981022541805D+01, 0.1059381944224D+01,
     :  0.4478223877045D-12, 0.0000000000000D+00, 0.0000000000000D+00 /

*  SSB-to-Sun, T^0, Y
      DATA ((S0(I,J,2),I=1,3),J=  1, 10) /
     :  0.4955392320126D-02, 0.2170467313679D+01, 0.5296909721118D+00,
     :  0.2722325167392D-02, 0.2444433682196D+01, 0.2132990797783D+00,
     :  0.1546579925346D-02, 0.5992779281546D+00, 0.3813291813120D-01,
     :  0.8363140252966D-03, 0.7687356310801D+00, 0.7478166569050D-01,
     :  0.3385792683603D-03, 0.0000000000000D+00, 0.0000000000000D+00,
     :  0.1201192221613D-03, 0.2520035601514D+01, 0.1059381944224D+01,
     :  0.7587125720554D-04, 0.1669954006449D+01, 0.4265981595566D+00,
     :  0.1964155361250D-04, 0.5707743963343D+01, 0.2061856251104D+00,
     :  0.1891900364909D-04, 0.2320960679937D+01, 0.2204125344462D+00,
     :  0.1937373433356D-04, 0.3226940689555D+01, 0.1495633313810D+00 /
      DATA ((S0(I,J,2),I=1,3),J= 11, 20) /
     :  0.1437139941351D-04, 0.2301626908096D+01, 0.5225775174439D+00,
     :  0.1406267683099D-04, 0.5188579265542D+01, 0.5368044267797D+00,
     :  0.1178703080346D-04, 0.5489483248476D+01, 0.7626583626240D-01,
     :  0.8079835186041D-05, 0.1683751835264D+01, 0.3664874755930D-01,
     :  0.7623253594652D-05, 0.2656400462961D+01, 0.3961708870310D-01,
     :  0.6248667483971D-05, 0.4992775362055D+01, 0.7329749511860D-01,
     :  0.4366353695038D-05, 0.2869706279678D+01, 0.1589072916335D+01,
     :  0.3829101568895D-05, 0.3572131359950D+01, 0.7113454667900D-02,
     :  0.3175733773908D-05, 0.4535372530045D+01, 0.4194847048887D+00,
     :  0.3092437902159D-05, 0.9230153317909D+00, 0.6398972393349D+00 /
      DATA ((S0(I,J,2),I=1,3),J= 21, 30) /
     :  0.2874168812154D-05, 0.3363143761101D+01, 0.1102062672231D+00,
     :  0.3040119321826D-05, 0.3324250895675D+01, 0.6283075850446D+01,
     :  0.2699723308006D-05, 0.2917882441928D+00, 0.1030928125552D+00,
     :  0.2134832683534D-05, 0.4220997202487D+01, 0.3163918923335D+00,
     :  0.1770412139433D-05, 0.4747318496462D+01, 0.1021328554739D+02,
     :  0.1377264209373D-05, 0.4305058462401D+00, 0.1484170571900D-02,
     :  0.1127814538960D-05, 0.8538177240740D+00, 0.6327837846670D+00,
     :  0.1055608090130D-05, 0.1551800742580D+01, 0.4337116142245D+00,
     :  0.9802673861420D-06, 0.1459646735377D+01, 0.1052268489556D+01,
     :  0.1090329461951D-05, 0.1587351228711D+01, 0.1162474756779D+01 /
      DATA ((S0(I,J,2),I=1,3),J= 31, 40) /
     :  0.6959590025090D-06, 0.5534442628766D+01, 0.1066495398892D+01,
     :  0.5664914529542D-06, 0.6030673003297D+01, 0.9491756770005D+00,
     :  0.6607787763599D-06, 0.4989507233927D+01, 0.8460828644453D+00,
     :  0.6269725742838D-06, 0.4222951804572D+01, 0.1480791608091D+00,
     :  0.6301889697863D-06, 0.5444316669126D+01, 0.2243449970715D+00,
     :  0.4891042662861D-06, 0.1490552839784D+01, 0.3340612434717D+01,
     :  0.3457083123290D-06, 0.3030475486049D+01, 0.3516457698740D-01,
     :  0.3032559967314D-06, 0.2652038793632D+01, 0.1104591729320D-01,
     :  0.2841133988903D-06, 0.1276744786829D+01, 0.4110125927500D-01,
     :  0.2855564444432D-06, 0.2143368674733D+01, 0.1510475019529D+00 /
      DATA ((S0(I,J,2),I=1,3),J= 41, 50) /
     :  0.2765157135038D-06, 0.5444186109077D+01, 0.6373574839730D-01,
     :  0.2382312465034D-06, 0.2190521137593D+01, 0.2275259891141D+00,
     :  0.2808060365077D-06, 0.5735195064841D+01, 0.2535050500000D-01,
     :  0.2332175234405D-06, 0.9481985524859D-01, 0.7181332454670D-01,
     :  0.2322488199659D-06, 0.5180499361533D+01, 0.8582758298370D-01,
     :  0.1881850258423D-06, 0.3219788273885D+01, 0.2118763888447D+01,
     :  0.2196111392808D-06, 0.2366941159761D+01, 0.2968341143800D-02,
     :  0.2183810335519D-06, 0.4825445110915D+01, 0.7775000683430D-01,
     :  0.2002733093326D-06, 0.2457148995307D+01, 0.2093666171530D+00,
     :  0.1967111767229D-06, 0.5586291545459D+01, 0.2172315424036D+00 /
      DATA ((S0(I,J,2),I=1,3),J= 51, 60) /
     :  0.1568473250543D-06, 0.3708003123320D+01, 0.7429900518901D+00,
     :  0.1852528314300D-06, 0.4310638151560D+01, 0.2022531624851D+00,
     :  0.1832111226447D-06, 0.1494665322656D+01, 0.3235053470014D+00,
     :  0.1746805502310D-06, 0.1451378500784D+01, 0.1385174140878D+00,
     :  0.1555730966650D-06, 0.1068040418198D+01, 0.7358765972222D+00,
     :  0.1554883462559D-06, 0.2442579035461D+01, 0.5154640627760D+00,
     :  0.1638380568746D-06, 0.2597913420625D+00, 0.8531963191132D+00,
     :  0.1159938593640D-06, 0.5834512021280D+01, 0.1990721704425D+00,
     :  0.1083427965695D-06, 0.5054033177950D+01, 0.5439178814476D+00,
     :  0.1156480369431D-06, 0.5325677432457D+01, 0.5257585094865D+00 /
      DATA ((S0(I,J,2),I=1,3),J= 61, 70) /
     :  0.1141308860095D-06, 0.2153403923857D+01, 0.5336234347371D+00,
     :  0.7913146470946D-07, 0.8642846847027D+00, 0.1478866649112D+01,
     :  0.7439752463733D-07, 0.1970628496213D+01, 0.2164800718209D+00,
     :  0.7280277104079D-07, 0.6073307250609D+01, 0.2101180877357D+00,
     :  0.8319567719136D-07, 0.1954371928334D+01, 0.1692165728891D+01,
     :  0.7137705549290D-07, 0.8904989440909D+00, 0.4155522422634D+00,
     :  0.6900825396225D-07, 0.2825717714977D+01, 0.1173197218910D+00,
     :  0.7245757216635D-07, 0.2481677513331D+01, 0.1265567569334D+01,
     :  0.6961165696255D-07, 0.1292955312978D+01, 0.9562891316684D+00,
     :  0.7571804456890D-07, 0.3427517575069D+01, 0.1422690933580D-01 /
      DATA ((S0(I,J,2),I=1,3),J= 71, 80) /
     :  0.6605425721904D-07, 0.8052192701492D+00, 0.6470106940028D+00,
     :  0.7375477357248D-07, 0.1705076390088D+01, 0.1581959461667D+01,
     :  0.7041664951470D-07, 0.4848356967891D+00, 0.9597935788730D-01,
     :  0.6322199535763D-07, 0.3878069473909D+01, 0.7084920306520D-01,
     :  0.5244380279191D-07, 0.2645560544125D+01, 0.5265099800692D+00,
     :  0.5143125704988D-07, 0.4834486101370D+01, 0.5328719641544D+00,
     :  0.5871866319373D-07, 0.7981472548900D+00, 0.7871412831580D-01,
     :  0.6300822573871D-07, 0.5979398788281D+01, 0.2608790314060D+02,
     :  0.6062154271548D-07, 0.4108655402756D+01, 0.1114304132498D+00,
     :  0.4361912339976D-07, 0.5322624319280D+01, 0.1375773836557D+01 /
      DATA ((S0(I,J,2),I=1,3),J= 81, 90) /
     :  0.4417005920067D-07, 0.6240817359284D+01, 0.2770348281756D+00,
     :  0.4686806749936D-07, 0.3214977301156D+01, 0.1143987543936D+00,
     :  0.3758892132305D-07, 0.5879809634765D+01, 0.1596186371003D+01,
     :  0.5151351332319D-07, 0.2893377688007D+00, 0.2228608264996D+00,
     :  0.4554683578572D-07, 0.5475427144122D+01, 0.1465949902372D+00,
     :  0.3442381385338D-07, 0.5992034796640D+01, 0.5070101000000D-01,
     :  0.2831093954933D-07, 0.5367350273914D+01, 0.3092784376656D+00,
     :  0.3756267090084D-07, 0.5758171285420D+01, 0.4903339079539D+00,
     :  0.2816374679892D-07, 0.1863718700923D+01, 0.2991266627620D+00,
     :  0.3419307025569D-07, 0.9524347534130D+00, 0.3518164938661D+00 /
      DATA ((S0(I,J,2),I=1,3),J= 91,100) /
     :  0.2904250494239D-07, 0.5304471615602D+01, 0.1099462426779D+00,
     :  0.2471734511206D-07, 0.1297069793530D+01, 0.6256703299991D+00,
     :  0.2539620831872D-07, 0.3281126083375D+01, 0.1256615170089D+02,
     :  0.2281017868007D-07, 0.1829122133165D+01, 0.6681224869435D+01,
     :  0.2275319473335D-07, 0.5797198160181D+01, 0.3932462625300D-02,
     :  0.2547755368442D-07, 0.4752697708330D+01, 0.1169588211447D+01,
     :  0.2285979669317D-07, 0.1223205292886D+01, 0.1045155034888D+01,
     :  0.1913386560994D-07, 0.1757532993389D+01, 0.1155361302111D+01,
     :  0.1809020525147D-07, 0.4246116108791D+01, 0.3368040641550D-01,
     :  0.1649213300201D-07, 0.1445162890627D+01, 0.4408250688924D+00 /
      DATA ((S0(I,J,2),I=1,3),J=101,110) /
     :  0.1834972793932D-07, 0.1126917567225D+01, 0.4452511715700D-02,
     :  0.1439550648138D-07, 0.6160756834764D+01, 0.9420622223326D+00,
     :  0.1487645457041D-07, 0.4358761931792D+01, 0.4123712502208D+00,
     :  0.1731729516660D-07, 0.6134456753344D+01, 0.2108507877249D+00,
     :  0.1717747163567D-07, 0.1898186084455D+01, 0.2157473718317D+00,
     :  0.1418190430374D-07, 0.4180286741266D+01, 0.6521991896920D-01,
     :  0.1404844134873D-07, 0.7654053565412D-01, 0.4258542984690D-01,
     :  0.1409842846538D-07, 0.4418612420312D+01, 0.2258291676434D+00,
     :  0.1090948346291D-07, 0.1260615686131D+01, 0.4226656969313D+00,
     :  0.1357577323612D-07, 0.3558248818690D+01, 0.7923417740620D-01 /
      DATA ((S0(I,J,2),I=1,3),J=111,120) /
     :  0.1018154061960D-07, 0.5676087241256D+01, 0.1456308687557D+00,
     :  0.1412073972109D-07, 0.8394392632422D+00, 0.1525316725248D+00,
     :  0.1030938326496D-07, 0.1653593274064D+01, 0.1795258541446D+01,
     :  0.1180081567104D-07, 0.1285802592036D+01, 0.7032915397480D-01,
     :  0.9708510575650D-08, 0.7631889488106D+00, 0.8434341241180D-01,
     :  0.9637689663447D-08, 0.4630642649176D+01, 0.1272681024002D+01,
     :  0.1068910429389D-07, 0.5294934032165D+01, 0.2123349582968D+00,
     :  0.1063716179336D-07, 0.2736266800832D+01, 0.2142632012598D+00,
     :  0.1234858713814D-07, 0.1302891146570D+01, 0.1847279083684D+00,
     :  0.8912631189738D-08, 0.3570415993621D+01, 0.2648454860559D+01 /
      DATA ((S0(I,J,2),I=1,3),J=121,130) /
     :  0.1036378285534D-07, 0.4236693440949D+01, 0.1370332435159D+00,
     :  0.9667798501561D-08, 0.2960768892398D+01, 0.4376440768498D+00,
     :  0.8108314201902D-08, 0.6987781646841D+00, 0.2880807454688D+00,
     :  0.7648364324628D-08, 0.2499017863863D+01, 0.2037373330570D+00,
     :  0.7286136828406D-08, 0.3787426951665D+01, 0.1129145838217D+00,
     :  0.9448237743913D-08, 0.2694354332983D+01, 0.5272426800584D+00,
     :  0.9374276106428D-08, 0.4787121277064D+01, 0.5321392641652D+00,
     :  0.7100226287462D-08, 0.3530238792101D+00, 0.6288513220417D+00,
     :  0.9253056659571D-08, 0.1399478925664D+01, 0.1606092486742D+00,
     :  0.6636432145504D-08, 0.3479575438447D+01, 0.1368660381889D+01 /
      DATA ((S0(I,J,2),I=1,3),J=131,140) /
     :  0.6469975312932D-08, 0.1383669964800D+01, 0.2008557621224D+01,
     :  0.7335849729765D-08, 0.1243698166898D+01, 0.9561746721300D-02,
     :  0.8743421205855D-08, 0.3776164289301D+01, 0.3801276407308D+00,
     :  0.5993635744494D-08, 0.5627122113596D+01, 0.2042657109477D+02,
     :  0.5981008479693D-08, 0.1674336636752D+01, 0.2111650433779D+01,
     :  0.6188535145838D-08, 0.5214925208672D+01, 0.4305306221819D+00,
     :  0.6596074017566D-08, 0.2907653268124D+01, 0.1063314406849D+01,
     :  0.6630815126226D-08, 0.2127643669658D+01, 0.8389694097774D+00,
     :  0.6156772830040D-08, 0.5082160803295D+01, 0.4234171675140D+00,
     :  0.6446960563014D-08, 0.1872100916905D+01, 0.5287268506303D+00 /
      DATA ((S0(I,J,2),I=1,3),J=141,150) /
     :  0.6429324424668D-08, 0.5610276103577D+01, 0.5306550935933D+00,
     :  0.6302232396465D-08, 0.1592152049607D+01, 0.1253008786510D-01,
     :  0.6399244436159D-08, 0.2746214421532D+01, 0.5217580628120D+02,
     :  0.5474965172558D-08, 0.2317666374383D+01, 0.2221856701002D+01,
     :  0.5339293190692D-08, 0.1084724961156D+01, 0.7466759693650D-01,
     :  0.5334733683389D-08, 0.3594106067745D+01, 0.7489573444450D-01,
     :  0.5392665782110D-08, 0.5630254365606D+01, 0.1055449481598D+01,
     :  0.6682075673789D-08, 0.1518480041732D+01, 0.2213766559277D+00,
     :  0.5079130495960D-08, 0.2739765115711D+01, 0.2132517061319D+00,
     :  0.5077759793261D-08, 0.5290711290094D+01, 0.2133464534247D+00 /
      DATA ((S0(I,J,2),I=1,3),J=151,160) /
     :  0.4832037368310D-08, 0.1404473217200D+01, 0.7160067364790D-01,
     :  0.6463279674802D-08, 0.6038381695210D+01, 0.2209183458640D-01,
     :  0.6240592771560D-08, 0.1290170653666D+01, 0.3306188016693D+00,
     :  0.4672013521493D-08, 0.3261895939677D+01, 0.7796265773310D-01,
     :  0.6500650750348D-08, 0.1154522312095D+01, 0.3884652414254D+00,
     :  0.6344161389053D-08, 0.6206111545062D+01, 0.7605151500000D-01,
     :  0.4682518370646D-08, 0.5409118796685D+01, 0.1073608853559D+01,
     :  0.5329460015591D-08, 0.1202985784864D+01, 0.7287631425543D+00,
     :  0.5701588675898D-08, 0.4098715257064D+01, 0.8731175355560D-01,
     :  0.6030690867211D-08, 0.4132033218460D+00, 0.9846002785331D+00 /
      DATA ((S0(I,J,2),I=1,3),J=161,170) /
     :  0.4336256312655D-08, 0.1211415991827D+01, 0.4297791515992D+00,
     :  0.4688498808975D-08, 0.3765479072409D+01, 0.2127790306879D+00,
     :  0.4675578609335D-08, 0.4265540037226D+01, 0.2138191288687D+00,
     :  0.4225578112158D-08, 0.5237566010676D+01, 0.3407705765729D+00,
     :  0.5139422230028D-08, 0.1507173079513D+01, 0.7233337363710D-01,
     :  0.4619995093571D-08, 0.9023957449848D-01, 0.8603097737811D+00,
     :  0.4494776255461D-08, 0.5414930552139D+00, 0.7381754420900D-01,
     :  0.4274026276788D-08, 0.4145735303659D+01, 0.7574578717200D-01,
     :  0.5018141789353D-08, 0.3344408829055D+01, 0.3180992042600D-02,
     :  0.4866163952181D-08, 0.3348534657607D+01, 0.7722995774390D-01 /
      DATA ((S0(I,J,2),I=1,3),J=171,180) /
     :  0.4111986020501D-08, 0.4198823597220D+00, 0.1451108196653D+00,
     :  0.3356142784950D-08, 0.5609144747180D+01, 0.1274714967946D+00,
     :  0.4070575554551D-08, 0.7028411059224D+00, 0.3503323232942D+00,
     :  0.3257451857278D-08, 0.5624697983086D+01, 0.5296435984654D+00,
     :  0.3256973703026D-08, 0.1857842076707D+01, 0.5297383457582D+00,
     :  0.3830771508640D-08, 0.4562887279931D+01, 0.9098186128426D+00,
     :  0.3725024005962D-08, 0.2358058692652D+00, 0.1084620721060D+00,
     :  0.3136763921756D-08, 0.2049731526845D+01, 0.2346394437820D+00,
     :  0.3795147256194D-08, 0.2432356296933D+00, 0.1862120789403D+00,
     :  0.2877342229911D-08, 0.5631101279387D+01, 0.1905464808669D+01 /
      DATA ((S0(I,J,2),I=1,3),J=181,190) /
     :  0.3076931798805D-08, 0.1117615737392D+01, 0.3628624111593D+00,
     :  0.2734765945273D-08, 0.5899826516955D+01, 0.2131850110243D+00,
     :  0.2733405296885D-08, 0.2130562964070D+01, 0.2134131485323D+00,
     :  0.2898552353410D-08, 0.3462387048225D+00, 0.5291709230214D+00,
     :  0.2893736103681D-08, 0.8534352781543D+00, 0.5302110212022D+00,
     :  0.3095717734137D-08, 0.2875061429041D+01, 0.2976424921901D+00,
     :  0.2636190425832D-08, 0.2242512846659D+01, 0.1485980103780D+01,
     :  0.3645512095537D-08, 0.1354016903958D+01, 0.6044726378023D+00,
     :  0.2808173547723D-08, 0.6705114365631D-01, 0.6225157782540D-01,
     :  0.2625012866888D-08, 0.4775705748482D+01, 0.5268983110410D-01 /
      DATA ((S0(I,J,2),I=1,3),J=191,200) /
     :  0.2572233995651D-08, 0.2638924216139D+01, 0.1258454114666D+01,
     :  0.2604238824792D-08, 0.4826358927373D+01, 0.2103781122809D+00,
     :  0.2596886385239D-08, 0.3200388483118D+01, 0.2162200472757D+00,
     :  0.3228057304264D-08, 0.5384848409563D+01, 0.2007689919132D+00,
     :  0.2481601798252D-08, 0.5173373487744D+01, 0.1062562936266D+01,
     :  0.2745977498864D-08, 0.6250966149853D+01, 0.5651155736444D+00,
     :  0.2669878833811D-08, 0.4906001352499D+01, 0.1400015846597D+00,
     :  0.3203986611711D-08, 0.5034333010005D+01, 0.7036329877322D+00,
     :  0.3354961227212D-08, 0.6108262423137D+01, 0.4549093064213D+00,
     :  0.2400407324558D-08, 0.2135399294955D+01, 0.2125476091956D+00 /
      DATA ((S0(I,J,2),I=1,3),J=201,210) /
     :  0.2379905859802D-08, 0.5893721933961D+01, 0.2140505503610D+00,
     :  0.2550844302187D-08, 0.3331940762063D+01, 0.1534957940063D+00,
     :  0.2268824211001D-08, 0.1843418461035D+01, 0.2235935264888D+00,
     :  0.2464700891204D-08, 0.3029548547230D+01, 0.2091065926078D+00,
     :  0.2436814726024D-08, 0.4994717970364D+01, 0.2174915669488D+00,
     :  0.2443623894745D-08, 0.2645102591375D+01, 0.1739420156204D+00,
     :  0.2318701783838D-08, 0.5700547397897D+01, 0.7530171478090D-01,
     :  0.2284448700256D-08, 0.5268898905872D+01, 0.7426161660010D-01,
     :  0.2468848123510D-08, 0.5276280575078D+01, 0.2526561439362D+00,
     :  0.2814052350303D-08, 0.6130168623475D+01, 0.5636314030725D+00 /
      DATA ((S0(I,J,2),I=1,3),J=211,NS0Y) /
     :  0.2243662755220D-08, 0.6631692457995D+00, 0.8886590321940D-01,
     :  0.2330795855941D-08, 0.2499435487702D+01, 0.1056200952181D+01,
     :  0.9757679038404D-09, 0.5796846023126D+01, 0.7826370942180D+02 /

*  SSB-to-Sun, T^1, Y
      DATA ((S1(I,J,2),I=1,3),J=  1, 10) /
     :  0.8989047573576D-08, 0.5840593672122D+01, 0.4265981595566D+00,
     :  0.7815938401048D-08, 0.1129664707133D+01, 0.2061856251104D+00,
     :  0.7550926713280D-08, 0.6196589104845D+00, 0.2204125344462D+00,
     :  0.6056556925895D-08, 0.1677494667846D+01, 0.1059381944224D+01,
     :  0.5734142698204D-08, 0.4000920852962D+01, 0.5225775174439D+00,
     :  0.5614341822459D-08, 0.3486722577328D+01, 0.5368044267797D+00,
     :  0.1028678147656D-08, 0.1877141024787D+01, 0.7113454667900D-02,
     :  0.7270792075266D-09, 0.5077167301739D+01, 0.6398972393349D+00,
     :  0.8734141726040D-09, 0.9069550282609D-01, 0.4194847048887D+00,
     :  0.5377371402113D-09, 0.6039381844671D+01, 0.4337116142245D+00 /
      DATA ((S1(I,J,2),I=1,3),J= 11, 20) /
     :  0.4729719431571D-09, 0.2153086311760D+01, 0.2132990797783D+00,
     :  0.4458052820973D-09, 0.5059830025565D+01, 0.5296909721118D+00,
     :  0.4406855467908D-09, 0.2027971692630D+01, 0.1589072916335D+01,
     :  0.3101659310977D-09, 0.3317677981860D+01, 0.1052268489556D+01,
     :  0.3016749232545D-09, 0.3913703482532D+01, 0.1066495398892D+01,
     :  0.3198541352656D-09, 0.1275513098525D+01, 0.1495633313810D+00,
     :  0.2142065389871D-09, 0.5301351614597D+01, 0.3163918923335D+00,
     :  0.1902615247592D-09, 0.4894943352736D+00, 0.2275259891141D+00,
     :  0.1613410990871D-09, 0.2449891130437D+01, 0.1102062672231D+00,
     :  0.1576992165097D-09, 0.4211421447633D+01, 0.7626583626240D-01 /
      DATA ((S1(I,J,2),I=1,3),J= 21, 30) /
     :  0.1241637259894D-09, 0.4140803368133D+01, 0.5154640627760D+00,
     :  0.1313974830355D-09, 0.3591920305503D+01, 0.3664874755930D-01,
     :  0.1181697118258D-09, 0.1506314382788D+01, 0.6327837846670D+00,
     :  0.1238239742779D-09, 0.7461405378404D+00, 0.3961708870310D-01,
     :  0.1010107068241D-09, 0.6271010795475D+00, 0.7329749511860D-01,
     :  0.9226316616509D-10, 0.1259158839583D+01, 0.1990721704425D+00,
     :  0.8664946419555D-10, 0.3353244696934D+01, 0.5439178814476D+00,
     :  0.7757230468978D-10, 0.1447677295196D+01, 0.9491756770005D+00,
     :  0.7693168628139D-10, 0.1120509896721D+01, 0.1030928125552D+00,
     :  0.5487897454612D-10, 0.4439380426795D+01, 0.8531963191132D+00 /
      DATA ((S1(I,J,2),I=1,3),J= 31, 40) /
     :  0.5196118677218D-10, 0.3788856619137D+00, 0.2093666171530D+00,
     :  0.5110853339935D-10, 0.1386879372016D+01, 0.2172315424036D+00,
     :  0.5027804534813D-10, 0.1647881805466D+00, 0.2164800718209D+00,
     :  0.4922485922674D-10, 0.1594315079862D+01, 0.2101180877357D+00,
     :  0.6155599524400D-10, 0.0000000000000D+00, 0.0000000000000D+00,
     :  0.4447147832161D-10, 0.5480720918976D+01, 0.3235053470014D+00,
     :  0.4144691276422D-10, 0.1931371033660D+01, 0.6373574839730D-01,
     :  0.4099950625452D-10, 0.5229611294335D+01, 0.6470106940028D+00,
     :  0.5060541682953D-10, 0.1731112486298D+01, 0.1422690933580D-01,
     :  0.4293615946300D-10, 0.2714571038925D+01, 0.7358765972222D+00 /
      DATA ((S1(I,J,2),I=1,3),J= 41,NS1Y) /
     :  0.3545659845763D-10, 0.4451041444634D+01, 0.5265099800692D+00,
     :  0.3479112041196D-10, 0.3029385448081D+01, 0.5328719641544D+00,
     :  0.3438516493570D-10, 0.2778507143731D+01, 0.8582758298370D-01,
     :  0.3297341285033D-10, 0.7898709807584D+00, 0.1104591729320D-01,
     :  0.2972585818015D-10, 0.3218785316973D+01, 0.5257585094865D+00,
     :  0.2931707295017D-10, 0.4260731012098D+01, 0.5336234347371D+00,
     :  0.2897198149403D-10, 0.1120753978101D+01, 0.1173197218910D+00,
     :  0.2832293240878D-10, 0.4597682717827D+00, 0.2022531624851D+00,
     :  0.2864348326612D-10, 0.2169939928448D+01, 0.9597935788730D-01,
     :  0.2852714675471D-10, 0.2377659870578D+01, 0.2118763888447D+01 /

*  SSB-to-Sun, T^2, Y
      DATA ((S2(I,J,2),I=1,3),J=  1,NS2Y) /
     :  0.1609114495091D-11, 0.2831096993481D+01, 0.2061856251104D+00,
     :  0.1560330784946D-11, 0.5193058213906D+01, 0.2204125344462D+00,
     :  0.1183535479202D-11, 0.5707003443890D+01, 0.5225775174439D+00,
     :  0.1158183066182D-11, 0.1782400404928D+01, 0.5368044267797D+00,
     :  0.1032868027407D-11, 0.4036925452011D+01, 0.2132990797783D+00,
     :  0.6540142847741D-12, 0.4058241056717D+01, 0.4265981595566D+00,
     :  0.7305236491596D-12, 0.6175401942957D+00, 0.5296909721118D+00,
     : -0.5580725052968D-12, 0.0000000000000D+00, 0.0000000000000D+00,
     :  0.3946122651015D-12, 0.4108265279171D+00, 0.1059381944224D+01 /

*  SSB-to-Sun, T^0, Z
      DATA ((S0(I,J,3),I=1,3),J=  1, 10) /
     :  0.1181255122986D-03, 0.4607918989164D+00, 0.2132990797783D+00,
     :  0.1127777651095D-03, 0.4169146331296D+00, 0.5296909721118D+00,
     :  0.4777754401806D-04, 0.4582657007130D+01, 0.3813291813120D-01,
     :  0.1129354285772D-04, 0.5758735142480D+01, 0.7478166569050D-01,
     : -0.1149543637123D-04, 0.0000000000000D+00, 0.0000000000000D+00,
     :  0.3298730512306D-05, 0.5978801994625D+01, 0.4265981595566D+00,
     :  0.2733376706079D-05, 0.7665413691040D+00, 0.1059381944224D+01,
     :  0.9426389657270D-06, 0.3710201265838D+01, 0.2061856251104D+00,
     :  0.8187517749552D-06, 0.3390675605802D+00, 0.2204125344462D+00,
     :  0.4080447871819D-06, 0.4552296640088D+00, 0.5225775174439D+00 /
      DATA ((S0(I,J,3),I=1,3),J= 11, 20) /
     :  0.3169973017028D-06, 0.3445455899321D+01, 0.5368044267797D+00,
     :  0.2438098615549D-06, 0.5664675150648D+01, 0.3664874755930D-01,
     :  0.2601897517235D-06, 0.1931894095697D+01, 0.1495633313810D+00,
     :  0.2314558080079D-06, 0.3666319115574D+00, 0.3961708870310D-01,
     :  0.1962549548002D-06, 0.3167411699020D+01, 0.7626583626240D-01,
     :  0.2180518287925D-06, 0.1544420746580D+01, 0.7113454667900D-02,
     :  0.1451382442868D-06, 0.1583756740070D+01, 0.1102062672231D+00,
     :  0.1358439007389D-06, 0.5239941758280D+01, 0.6398972393349D+00,
     :  0.1050585898028D-06, 0.2266958352859D+01, 0.3163918923335D+00,
     :  0.1050029870186D-06, 0.2711495250354D+01, 0.4194847048887D+00 /
      DATA ((S0(I,J,3),I=1,3),J= 21, 30) /
     :  0.9934920679800D-07, 0.1116208151396D+01, 0.1589072916335D+01,
     :  0.1048395331560D-06, 0.3408619600206D+01, 0.1021328554739D+02,
     :  0.8370147196668D-07, 0.3810459401087D+01, 0.2535050500000D-01,
     :  0.7989856510998D-07, 0.3769910473647D+01, 0.7329749511860D-01,
     :  0.5441221655233D-07, 0.2416994903374D+01, 0.1030928125552D+00,
     :  0.4610812906784D-07, 0.5858503336994D+01, 0.4337116142245D+00,
     :  0.3923022803444D-07, 0.3354170010125D+00, 0.1484170571900D-02,
     :  0.2610725582128D-07, 0.5410600646324D+01, 0.6327837846670D+00,
     :  0.2455279767721D-07, 0.6120216681403D+01, 0.1162474756779D+01,
     :  0.2375530706525D-07, 0.6055443426143D+01, 0.1052268489556D+01 /
      DATA ((S0(I,J,3),I=1,3),J= 31, 40) /
     :  0.1782967577553D-07, 0.3146108708004D+01, 0.8460828644453D+00,
     :  0.1581687095238D-07, 0.6255496089819D+00, 0.3340612434717D+01,
     :  0.1594657672461D-07, 0.3782604300261D+01, 0.1066495398892D+01,
     :  0.1563448615040D-07, 0.1997775733196D+01, 0.2022531624851D+00,
     :  0.1463624258525D-07, 0.1736316792088D+00, 0.3516457698740D-01,
     :  0.1331585056673D-07, 0.4331941830747D+01, 0.9491756770005D+00,
     :  0.1130634557637D-07, 0.6152017751825D+01, 0.2968341143800D-02,
     :  0.1028949607145D-07, 0.2101792614637D+00, 0.2275259891141D+00,
     :  0.1024074971618D-07, 0.4071833211074D+01, 0.5070101000000D-01,
     :  0.8826956060303D-08, 0.4861633688145D+00, 0.2093666171530D+00 /
      DATA ((S0(I,J,3),I=1,3),J= 41, 50) /
     :  0.8572230171541D-08, 0.5268190724302D+01, 0.4110125927500D-01,
     :  0.7649332643544D-08, 0.5134543417106D+01, 0.2608790314060D+02,
     :  0.8581673291033D-08, 0.2920218146681D+01, 0.1480791608091D+00,
     :  0.8430589300938D-08, 0.3604576619108D+01, 0.2172315424036D+00,
     :  0.7776165501012D-08, 0.3772942249792D+01, 0.6373574839730D-01,
     :  0.8311070234408D-08, 0.6200412329888D+01, 0.3235053470014D+00,
     :  0.6927365212582D-08, 0.4543353113437D+01, 0.8531963191132D+00,
     :  0.6791574208598D-08, 0.2882188406238D+01, 0.7181332454670D-01,
     :  0.5593100811839D-08, 0.1776646892780D+01, 0.7429900518901D+00,
     :  0.4553381853021D-08, 0.3949617611240D+01, 0.7775000683430D-01 /
      DATA ((S0(I,J,3),I=1,3),J= 51, 60) /
     :  0.5758000450068D-08, 0.3859251775075D+01, 0.1990721704425D+00,
     :  0.4281283457133D-08, 0.1466294631206D+01, 0.2118763888447D+01,
     :  0.4206935661097D-08, 0.5421776011706D+01, 0.1104591729320D-01,
     :  0.4213751641837D-08, 0.3412048993322D+01, 0.2243449970715D+00,
     :  0.5310506239878D-08, 0.5421641370995D+00, 0.5154640627760D+00,
     :  0.3827450341320D-08, 0.8887314524995D+00, 0.1510475019529D+00,
     :  0.4292435241187D-08, 0.1405043757194D+01, 0.1422690933580D-01,
     :  0.3189780702289D-08, 0.1060049293445D+01, 0.1173197218910D+00,
     :  0.3226611928069D-08, 0.6270858897442D+01, 0.2164800718209D+00,
     :  0.2893897608830D-08, 0.5117563223301D+01, 0.6470106940028D+00 /
      DATA ((S0(I,J,3),I=1,3),J= 61,NS0Z) /
     :  0.3239852024578D-08, 0.4079092237983D+01, 0.2101180877357D+00,
     :  0.2956892222200D-08, 0.1594917021704D+01, 0.3092784376656D+00,
     :  0.2980177912437D-08, 0.5258787667564D+01, 0.4155522422634D+00,
     :  0.3163725690776D-08, 0.3854589225479D+01, 0.8582758298370D-01,
     :  0.2662262399118D-08, 0.3561326430187D+01, 0.5257585094865D+00,
     :  0.2766689135729D-08, 0.3180732086830D+00, 0.1385174140878D+00,
     :  0.2411600278464D-08, 0.3324798335058D+01, 0.5439178814476D+00,
     :  0.2483527695131D-08, 0.4169069291947D+00, 0.5336234347371D+00,
     :  0.7788777276590D-09, 0.1900569908215D+01, 0.5217580628120D+02 /

*  SSB-to-Sun, T^1, Z
      DATA ((S1(I,J,3),I=1,3),J=  1, 10) /
     :  0.5444220475678D-08, 0.1803825509310D+01, 0.2132990797783D+00,
     :  0.3883412695596D-08, 0.4668616389392D+01, 0.5296909721118D+00,
     :  0.1334341434551D-08, 0.0000000000000D+00, 0.0000000000000D+00,
     :  0.3730001266883D-09, 0.5401405918943D+01, 0.2061856251104D+00,
     :  0.2894929197956D-09, 0.4932415609852D+01, 0.2204125344462D+00,
     :  0.2857950357701D-09, 0.3154625362131D+01, 0.7478166569050D-01,
     :  0.2499226432292D-09, 0.3657486128988D+01, 0.4265981595566D+00,
     :  0.1937705443593D-09, 0.5740434679002D+01, 0.1059381944224D+01,
     :  0.1374894396320D-09, 0.1712857366891D+01, 0.5368044267797D+00,
     :  0.1217248678408D-09, 0.2312090870932D+01, 0.5225775174439D+00 /
      DATA ((S1(I,J,3),I=1,3),J= 11,NS1Z) /
     :  0.7961052740870D-10, 0.5283368554163D+01, 0.3813291813120D-01,
     :  0.4979225949689D-10, 0.4298290471860D+01, 0.4194847048887D+00,
     :  0.4388552286597D-10, 0.6145515047406D+01, 0.7113454667900D-02,
     :  0.2586835212560D-10, 0.3019448001809D+01, 0.6398972393349D+00 /

*  SSB-to-Sun, T^2, Z
      DATA ((S2(I,J,3),I=1,3),J=  1,NS2Z) /
     :  0.3749920358054D-12, 0.3230285558668D+01, 0.2132990797783D+00,
     :  0.2735037220939D-12, 0.6154322683046D+01, 0.5296909721118D+00 /

* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

*  Time since reference epoch, years.
      T = ( DATE - DJM0 ) / DJY
      T2 = T*T

*  X then Y then Z.
      DO K=1,3

*     Initialize position and velocity component.
         XYZ = 0D0
         XYZD = 0D0

*     ------------------------------------------------
*     Obtain component of Sun to Earth ecliptic vector
*     ------------------------------------------------

*     Sun to Earth, T^0 terms.
         DO J=1,NE0(K)
            A = E0(1,J,K)
            B = E0(2,J,K)
            C = E0(3,J,K)
            P = B + C*T
            XYZ  = XYZ  + A*COS(P)
            XYZD = XYZD - A*C*SIN(P)
         END DO

*     Sun to Earth, T^1 terms.
         DO J=1,NE1(K)
            A = E1(1,J,K)
            B = E1(2,J,K)
            C = E1(3,J,K)
            CT = C*T
            P = B + CT
            CP = COS(P)
            XYZ  = XYZ  + A*T*CP
            XYZD = XYZD + A*(CP-CT*SIN(P))
         END DO

*     Sun to Earth, T^2 terms.
         DO J=1,NE2(K)
            A = E2(1,J,K)
            B = E2(2,J,K)
            C = E2(3,J,K)
            CT = C*T
            P = B + CT
            CP = COS(P)
            XYZ  = XYZ  + A*T2*CP
            XYZD = XYZD + A*T*(2D0*CP-CT*SIN(P))
         END DO

*     Heliocentric Earth position and velocity component.
         HP(K) = XYZ
         HV(K) = XYZD / DJY

*     ------------------------------------------------
*     Obtain component of SSB to Earth ecliptic vector
*     ------------------------------------------------

*     SSB to Sun, T^0 terms.
         DO J=1,NS0(K)
            A = S0(1,J,K)
            B = S0(2,J,K)
            C = S0(3,J,K)
            P = B + C*T
            XYZ  = XYZ  + A*COS(P)
            XYZD = XYZD - A*C*SIN(P)
         END DO

*     SSB to Sun, T^1 terms.
         DO J=1,NS1(K)
            A = S1(1,J,K)
            B = S1(2,J,K)
            C = S1(3,J,K)
            CT = C*T
            P = B + CT
            CP = COS(P)
            XYZ  = XYZ  + A*T*CP
            XYZD = XYZD + A*(CP-CT*SIN(P))
         END DO

*     SSB to Sun, T^2 terms.
         DO J=1,NS2(K)
            A = S2(1,J,K)
            B = S2(2,J,K)
            C = S2(3,J,K)
            CT = C*T
            P = B + CT
            CP = COS(P)
            XYZ  = XYZ  + A*T2*CP
            XYZD = XYZD + A*T*(2D0*CP-CT*SIN(P))
         END DO

*     Barycentric Earth position and velocity component.
         BP(K) = XYZ
         BV(K) = XYZD / DJY

*     Next Cartesian component.
      END DO

*  Rotate from ecliptic to ICRS coordinates and return the results.
      X = HP(1)
      Y = HP(2)
      Z = HP(3)
      PH(1) =      X + AM12*Y + AM13*Z
      PH(2) = AM21*X + AM22*Y + AM23*Z
      PH(3) =          AM32*Y + AM33*Z
      X = HV(1)
      Y = HV(2)
      Z = HV(3)
      VH(1) =      X + AM12*Y + AM13*Z
      VH(2) = AM21*X + AM22*Y + AM23*Z
      VH(3) =          AM32*Y + AM33*Z
      X = BP(1)
      Y = BP(2)
      Z = BP(3)
      PB(1) =      X + AM12*Y + AM13*Z
      PB(2) = AM21*X + AM22*Y + AM23*Z
      PB(3) =          AM32*Y + AM33*Z
      X = BV(1)
      Y = BV(2)
      Z = BV(3)
      VB(1) =      X + AM12*Y + AM13*Z
      VB(2) = AM21*X + AM22*Y + AM23*Z
      VB(3) =          AM32*Y + AM33*Z

      END
