/*
 * Copyright © 2013 Intel Corporation
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#include "piglit-util-gl.h"
#include "xfb3_common.h"

/**
 * @file set_invalid_varyings.c 
 *
 * Tests that varyings for separate attributes cannot be set using keywords
 * reserved solely for the interleaved use. The spec:
 *
 * "The error INVALID_OPERATION is generated by TransformFeedbackVaryings
 *  if any pointer in <varyings> identifies the special names "gl_NextBuffer",
 *  "gl_SkipComponents1", "gl_SkipComponents2", "gl_SkipComponents3", or
 *  "gl_SkipComponents4" and <bufferMode> is not INTERLEAVED_ATTRIBS_NV, or if
 *  the number of "gl_NextBuffer" pointers in <varyings> is greater than or
 *  equal to the value of MAX_TRANSFORM_FEEDBACK_BUFFERS."
 */

PIGLIT_GL_TEST_CONFIG_BEGIN

	config.supports_gl_compat_version = 32;
	config.supports_gl_core_version = 32;

PIGLIT_GL_TEST_CONFIG_END

static const char gs_simple_text[] =
	"#version 150\n"
	"layout(points) in;\n"
	"layout(points, max_vertices = 1) out;\n"
	"out float x1_out;\n"
	"void main() {\n"
	"  gl_Position = gl_in[0].gl_Position;\n"
	"  x1_out = 1.0;\n"
	"}";

static const struct {
	const char *description;
	const char *varyings[2];
} invalid_for_separated[] = {
	{
		"gl_NextBuffer should not be accepted in separate mode",
		{ "x1_out", "gl_NextBuffer" }
	},
	{
		"gl_SkipComponents1 should not be accepted in separate mode",
		{ "gl_SkipComponents1", "x1_out" }
	},
	{
		"gl_SkipComponents2 should not be accepted in separate mode",
		{ "gl_SkipComponents2", "x1_out" }
	},
	{
		"gl_SkipComponents3 should not be accepted in separate mode",
		{ "gl_SkipComponents3", "x1_out" }
	},
	{
		"gl_SkipComponents4 should not be accepted in separate mode",
		{ "gl_SkipComponents4", "x1_out" }
	}
};

static bool
try_invalid_separate_mode(GLuint prog)
{
	unsigned i;

	for (i = 0; i < ARRAY_SIZE(invalid_for_separated); ++i) {
		glTransformFeedbackVaryings(prog,
			ARRAY_SIZE(invalid_for_separated[i].varyings),
			invalid_for_separated[i].varyings,
			GL_SEPARATE_ATTRIBS);

		if (!piglit_check_gl_error(GL_INVALID_OPERATION)) {
			printf("fail: %s\n",
				invalid_for_separated[i].description);
			return false;
		}
	}

	return true;
}

/**
 * Dynamically reserve space and setup an array of 'n + n + 1' string pointers
 * having the following layout:
 *
 * [0]: "x_0x00000000"
 * [1]: "gl_NextBuffer"
 * [2]: "x_0x00000001"
 * [3]: "gl_NextBuffer"
 * ...
 * [2n]: "x_0x..n"
 *
 * For testing the upper bound of sepators (gl_NextBuffer) the varying names
 * need to be mutually unique as the driver could simply bail out before
 * encountering 'n' separators simply because the driver did not except already
 * declared varying.
 */
static bool
try_max_separators(GLuint prog, unsigned n)
{
	/* 'n' + 1 varying names and 'n' separators */
	const unsigned var_n = 2 * n + 1;
	const char **vars;
	unsigned i;

	vars = (const char **)malloc(var_n * sizeof(const char *));

	for (i = 0; i <= n; ++i) {
		(void)!asprintf((char **)&vars[2 * i + 0], "x_0x%08x", i);

		if (i < n) {
			static const char separator[] = "gl_NextBuffer";
			vars[2 * i + 1] = separator;
		}
	}

	glTransformFeedbackVaryings(prog, var_n, vars, GL_INTERLEAVED_ATTRIBS);

	for (i = 0; i <= n; ++i)
		free((char *)vars[2 * i + 0]);

	free(vars);

	return piglit_check_gl_error(GL_INVALID_OPERATION);
}

void
piglit_init(int argc, char **argv)
{
	bool pass = true;
	GLuint prog;
	GLint max_attrib_n;

	piglit_require_extension("GL_ARB_transform_feedback3");

	glGetIntegerv(GL_MAX_TRANSFORM_FEEDBACK_SEPARATE_ATTRIBS,
		&max_attrib_n);
	if (!max_attrib_n) {
		printf("Maximum number of separete attributes is zero\n");
		piglit_report_result(PIGLIT_FAIL);
	}

	prog = piglit_build_simple_program_multiple_shaders(
			GL_VERTEX_SHADER, vs_pass_thru_text,
			GL_GEOMETRY_SHADER, gs_simple_text, 0);

	/* Try too many attributes */
	pass = try_invalid_separate_mode(prog) && pass;
	pass = try_max_separators(prog, max_attrib_n) && pass;

	piglit_report_result(pass ? PIGLIT_PASS : PIGLIT_FAIL);
}

enum piglit_result
piglit_display(void)
{
	/* Should never be reached */
	return PIGLIT_FAIL;
}
