use std::io;
use std::sync::Mutex;
use git2::{Oid, Repository};
use serde::Serialize;
use openpgp::{
    Cert,
    Fingerprint,
    packet::Signature,
};
use super::*;

/// What output format to prefer, when there's an option?
#[derive(Clone)]
pub enum Format {
    /// Output that is meant to be read by humans, instead of programs.
    ///
    /// This type of output has no version, and is not meant to be
    /// parsed by programs.
    HumanReadable,

    /// Output as JSON.
    Json,
}

impl std::str::FromStr for Format {
    type Err = anyhow::Error;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        match s {
            "human-readable" => Ok(Self::HumanReadable),
            "json" => Ok(Self::Json),
            _ => Err(anyhow!("unknown output format {:?}", s)),
        }
    }
}

/// Emits a human-readable description of the policy to stdout.
pub fn describe_policy(p: &Policy) -> Result<()> {
    println!("# OpenPGP policy file for git, version {}",
             p.version);
    println!();

    println!("## Commit Goodlist");
    println!();

    for commit in &p.commit_goodlist {
        println!("  - {}", commit);
    }
    println!();

    println!("## Authorizations");
    println!();

    for (i, (name, auth)) in p.authorization.iter().enumerate()
    {
        println!("{}. {}", i, name);
        let ident = vec![' '; i.to_string().len() + 2]
            .into_iter().collect::<String>();

        if auth.sign_commit {
            println!("{}- may sign commits", ident);
        }
        if auth.sign_tag {
            println!("{}- may sign tags", ident);
        }
        if auth.sign_archive {
            println!("{}- may sign archives", ident);
        }
        if auth.add_user {
            println!("{}- may add users", ident);
        }
        if auth.retire_user {
            println!("{}- may retire users", ident);
        }
        if auth.audit {
            println!("{}- may goodlist commits", ident);
        }
        for cert in auth.certs()? {
            println!("{}- has OpenPGP cert: {}", ident,
                     cert?.fingerprint());
        }
    }
    Ok(())
}

/// Emits a human-readable description of a difference between two
/// policies to stdout.
pub fn describe_diff(p: &Diff) -> Result<()> {
    for change in &p.changes {
        use Change::*;
        match change {
            VersionChange { from, to } =>
                println!("  - Version changed from {} to {}.", from, to),

            GoodlistCommit(oid) =>
                println!("  - Commit {} was added to the goodlist.", oid),
            UngoodlistCommit(oid) =>
                println!("  - Commit {} was removed from the goodlist.", oid),

            AddUser(name) =>
                println!("  - User {:?} was added.", name),

            RetireUser(name) =>
                println!("  - User {:?} was retired.", name),

            AddRight(name, right) =>
                println!("  - User {:?} was granted the right {}.", name, right),
            RemoveRight(name, right) =>
                println!("  - User {:?}'s {} right was revoked.", name, right),

            _ => todo!("the cert stuff"),
        }
    }

    Ok(())
}

// The version of the commit output.  This follows semantic
// versioning.
static COMMIT_JSON_VERSION: &'static str = "1.0.0";

#[derive(Serialize)]
pub struct Commit<'a> {
    version: &'static str,
    #[serde(serialize_with = "crate::utils::serialize_oid")]
    id: &'a Oid,
    #[serde(serialize_with = "crate::utils::serialize_optional_oid")]
    parent_id: Option<&'a Oid>,
    results: Vec<std::result::Result<String, (String, Option<String>)>>,
}

static MISSING_SIGNATURE_HINT: Mutex<bool> = Mutex::new(false);
static MISSING_KEY_HINT: Mutex<bool> = Mutex::new(false);
static MALFORMED_MESSAGE_HINT: Mutex<bool> = Mutex::new(false);

impl<'a> Commit<'a> {
    pub fn new(_git: &Repository,
               id: &'a Oid,
               parent_id: Option<&'a Oid>,
               shadow_policy: &Option<PathBuf>,
               result: &'a sequoia_git::Result<Vec<sequoia_git::Result<(String, Signature, Cert, Fingerprint)>>>)
               -> Result<Self>
    {
        let hint = |e: &Error| -> Option<String> {
            match (shadow_policy, e) {
                (None, _) => None,
                (Some(p), Error::MissingSignature(commit)) => {
                    let mut shown = MISSING_SIGNATURE_HINT.lock().unwrap();
                    if ! *shown {
                        *shown = true;
                        Some(format!("when using an external policy, do\n\n\
                                      git show {1} \n\
                                      \n  and verify that the commit is good.  \
                                      If satisfied, do\n\n\
                                      sq-git policy goodlist --policy-file {0} {1}",
                                     p.display(), commit))
                    } else {
                        None
                    }
                }
                (Some(p), Error::MissingKey(handle)) => {
                    let mut shown = MISSING_KEY_HINT.lock().unwrap();
                    if ! *shown {
                        *shown = true;
                        Some(format!("when using an external policy, do\n\n\
                                      sq keyserver get {1} \n\
                                      \n  and verify that the cert belongs to the \
                                      committer.  If satisfied, do\n\n\
                                      sq-git policy authorize --policy-file {} \
                                      <ROLE-NAME> {} --sign-commit",
                                     p.display(), handle))
                    } else {
                        None
                    }
                }
                (_, Error::Other(e)) => {
                    if let Some(e) = e.downcast_ref::<openpgp::Error>() {
                        if let openpgp::Error::MalformedMessage(_) = e {
                            let mut shown
                                = MALFORMED_MESSAGE_HINT.lock().unwrap();
                            if ! *shown {
                                *shown = true;
                                Some(format!("\
a signature is malformed.  It was probably created by GitHub, which\n\
is known to created invalid signatures.  See the following discussion for\n\
more information:\n\
\n\
https://github.com/orgs/community/discussions/27607"))
                            } else {
                                None
                            }
                        } else {
                            None
                        }
                    } else {
                        None
                    }
                }
                _ => None,
            }
        };

        let mut r = Vec::new();
        match result {
            Ok(results) => {
                for e in results.iter()
                    .filter_map(|r| r.as_ref().err())
                {
                    r.push(Err((e.to_string(), hint(e))));
                }
                for (name, _s, c, _signer_fpr) in results.iter()
                    .filter_map(|r| r.as_ref().ok())
                {
                    r.push(Ok(format!("{} [{}]", name, c.keyid())));
                }
            },
            Err(e) => {
                r.push(Err((e.to_string(), hint(e))));
            },
        }

        Ok(Commit {
            version: COMMIT_JSON_VERSION,
            id,
            parent_id,
            results: r,
        })
    }

    pub fn describe(&self, sink: &mut dyn io::Write) -> Result<()> {
        for r in &self.results {
            let id = if let Some(parent_id) = self.parent_id {
                format!("{}..{}", parent_id, self.id)
            } else {
                self.id.to_string()
            };

            match r {
                Err((e, hint)) => {
                    writeln!(sink, "{}: {}", id, e)?;
                    if let Some(h) = hint {
                        writeln!(sink, "\n  Hint: {}", h)?;
                    }
                },
                Ok(fp) => {
                    writeln!(sink, "{}: {}", id, fp)?;
                },
            }
        }

        Ok(())
    }
}

// The version of the commit output.  This follows semantic
// versioning.
static ARCHIVE_JSON_VERSION: &'static str = "1.0.0";

#[derive(Serialize)]
pub struct Archive {
    version: &'static str,
    results: Vec<std::result::Result<String, String>>,
}

impl Archive {
    pub fn new(result: sequoia_git::Result<Vec<sequoia_git::Result<(String, Signature, Cert, Fingerprint)>>>)
               -> Result<Self>
    {
        let mut r = Vec::new();
        match result {
            Ok(results) => {
                for e in results.iter()
                    .filter_map(|r| r.as_ref().err())
                {
                    r.push(Err(e.to_string()));
                }
                for (name, _s, c, _signer_fpr) in results.iter()
                    .filter_map(|r| r.as_ref().ok())
                {
                    r.push(Ok(format!("{} [{}]", name, c.keyid())));
                }
            },
            Err(e) => {
                r.push(Err(e.to_string()));
            },
        }

        Ok(Self {
            version: ARCHIVE_JSON_VERSION,
            results: r,
        })
    }

    pub fn describe(&self, sink: &mut dyn io::Write) -> Result<()> {
        for r in &self.results {
            match r {
                Err(e) => {
                    writeln!(sink, "{}", e)?;
                },
                Ok(fp) => {
                    writeln!(sink, "{}", fp)?;
                },
            }
        }

        Ok(())
    }
}
