
///////////////////////////////////////////////////////////
//                                                       //
//                         SAGA                          //
//                                                       //
//      System for Automated Geoscientific Analyses      //
//                                                       //
//                     Tool Library                      //
//                    shapes_polygon                     //
//                                                       //
//-------------------------------------------------------//
//                                                       //
//               polygon_generalization.cpp              //
//                                                       //
//                 Copyright (C) 2019 by                 //
//                      Olaf Conrad                      //
//                                                       //
//-------------------------------------------------------//
//                                                       //
// This file is part of 'SAGA - System for Automated     //
// Geoscientific Analyses'. SAGA is free software; you   //
// can redistribute it and/or modify it under the terms  //
// of the GNU General Public License as published by the //
// Free Software Foundation, either version 2 of the     //
// License, or (at your option) any later version.       //
//                                                       //
// SAGA is distributed in the hope that it will be       //
// useful, but WITHOUT ANY WARRANTY; without even the    //
// implied warranty of MERCHANTABILITY or FITNESS FOR A  //
// PARTICULAR PURPOSE. See the GNU General Public        //
// License for more details.                             //
//                                                       //
// You should have received a copy of the GNU General    //
// Public License along with this program; if not, see   //
// <http://www.gnu.org/licenses/>.                       //
//                                                       //
//-------------------------------------------------------//
//                                                       //
//    e-mail:     oconrad@saga-gis.org                   //
//                                                       //
//    contact:    Olaf Conrad                            //
//                Institute of Geography                 //
//                University of Hamburg                  //
//                Germany                                //
//                                                       //
///////////////////////////////////////////////////////////

//---------------------------------------------------------
#include "polygon_generalization.h"


///////////////////////////////////////////////////////////
//														 //
//														 //
//														 //
///////////////////////////////////////////////////////////

//---------------------------------------------------------
CPolygon_Generalization::CPolygon_Generalization(void)
{
	Set_Name		(_TL("Polygon Generalization"));

	Set_Author		("O.Conrad (c) 2019");

	Set_Description	(_TW(
		"A simple generalization tool for polygons. "
		"The tool joins polygons with an area size smaller than "
		"the specified threshold to a neighbouring polygon. "
		"Either the neighbouring polygon with the largest area or "
		"the one with the longest shared edge wins."
	));

	//-----------------------------------------------------
	Parameters.Add_Shapes("",
		"POLYGONS"   , _TL("Shapes"),
		_TL("The input polygons."),
		PARAMETER_INPUT, SHAPE_TYPE_Polygon
	);

	Parameters.Add_Shapes("",
		"GENERALIZED", _TL("Generalized Shapes"),
		_TL("The generalized output polygons."),
		PARAMETER_OUTPUT_OPTIONAL, SHAPE_TYPE_Polygon
	);

	Parameters.Add_Double("",
		"THRESHOLD"  , _TL("Area Threshold"),
		_TL("The maximum area of a polygon to get joined [map units squared]."),
		100., 0., true
	);

	Parameters.Add_Choice("",
		"JOIN_TO"    , _TL("Join to Neighbour with ..."),
		_TL("Choose the method to determine the winner polygon."),
		CSG_String::Format("%s|%s",
			_TL("largest area"),
			_TL("largest shared edge length")
		), 0
	);

	Parameters.Add_Bool("",
		"VERTICES"   , _TL("Check Vertices"),
		_TL(""),
		false
	);

	Parameters.Add_Double("VERTICES",
		"EPSILON"    , _TL("Tolerance"),
		_TL(""),
		0.00001, 0., true
	);
}


///////////////////////////////////////////////////////////
//														 //
///////////////////////////////////////////////////////////

//---------------------------------------------------------
int CPolygon_Generalization::On_Parameters_Enable(CSG_Parameters *pParameters, CSG_Parameter *pParameter)
{
	if( pParameter->Cmp_Identifier("JOIN_TO") )
	{
		pParameters->Set_Enabled("VERTICES", pParameter->asInt() == 1);
	}

	if( pParameter->Cmp_Identifier("VERTICES") )
	{
		pParameters->Set_Enabled("EPSILON", pParameter->asBool());
	}

	return( CSG_Tool::On_Parameters_Enable(pParameters, pParameter) );
}


///////////////////////////////////////////////////////////
//														 //
///////////////////////////////////////////////////////////

//---------------------------------------------------------
bool CPolygon_Generalization::On_Execute(void)
{
	CSG_Shapes *pPolygons = Parameters("POLYGONS")->asShapes();

	if( !pPolygons->is_Valid() )
	{
		Error_Set(_TL("invalid polygons layer"));

		return( false );
	}

	if( pPolygons->Get_Count() <= 1 )
	{
		Error_Set(_TL("nothing to join"));

		return( false );
	}

	//-----------------------------------------------------
	if( Parameters("GENERALIZED")->asShapes() && Parameters("GENERALIZED")->asShapes() != pPolygons )
	{
		CSG_Shapes *pTarget = Parameters("GENERALIZED")->asShapes();

		pTarget->Create(*pPolygons);

		pTarget->Fmt_Name("%s [%s]", pPolygons->Get_Name(), _TL("generalized"));

		pPolygons = pTarget;
	}

	//-----------------------------------------------------
	bool bResult = Set_Joins(pPolygons);

	if( pPolygons == Parameters("POLYGONS")->asShapes() )
	{	// output is always updated automatically - but if input has been modified, this needs a manual update!
		DataObject_Update(pPolygons);
	}

	return( bResult );
}


///////////////////////////////////////////////////////////
//														 //
///////////////////////////////////////////////////////////

//---------------------------------------------------------
enum { FIELD_AREA = 0, FIELD_JOIN, FIELD_JOINS };

//---------------------------------------------------------
bool CPolygon_Generalization::Get_Joins(CSG_Shapes *pPolygons, CSG_Table &Joins)
{
	double Threshold = Parameters["THRESHOLD"].asDouble();

	if( Threshold <= 0. )
	{
		Error_Set(_TL("nothing to join, threshold should be greater zero"));

		return( false );
	}

	//-----------------------------------------------------
	Joins.Destroy();

	Joins.Add_Field("AREA" , SG_DATATYPE_Double);
	Joins.Add_Field("JOIN" , SG_DATATYPE_Long  );
	Joins.Add_Field("JOINS", SG_DATATYPE_Int   );

	for(sLong i=0; i<pPolygons->Get_Count(); i++)
	{
		CSG_Table_Record &Join = *Joins.Add_Record();

		Join.Set_Value(FIELD_AREA , pPolygons->Get_Shape(i)->asPolygon()->Get_Area());
		Join.Set_Value(FIELD_JOIN , -1);
		Join.Set_Value(FIELD_JOINS, 0.);
	}

	CSG_Index Index;

	if( !Joins.Set_Index(Index, FIELD_AREA) )
	{
		Error_Set(_TL("index creation failed"));

		return( false );
	}

	//-----------------------------------------------------
	int     Method = Parameters["JOIN_TO" ].asInt   ();
	bool bVertices = Parameters["VERTICES"].asBool  ();
	double Epsilon = Parameters["EPSILON" ].asDouble();

	sLong nJoins = 0;

	for(sLong i=0; i<Joins.Get_Count() && Set_Progress(i, Joins.Get_Count()); i++)
	{
		CSG_Table_Record &Join = Joins[Index[i]];

		if( Join.asDouble(FIELD_AREA) < Threshold )
		{
			CSG_Shape_Polygon *pPolygon = pPolygons->Get_Shape(Join.Get_Index())->asPolygon();

			sLong maxPolygon = -1; double maxValue = 0.;

			for(sLong j=0; j<pPolygons->Get_Count(); j++)
			{
				CSG_Shape_Polygon *pNeighbour = pPolygons->Get_Shape(j)->asPolygon();

				if( pPolygon != pNeighbour )
				{
					if( Method == 0 ) // largest area
					{
						if( maxValue < pNeighbour->Get_Area() && pPolygon->is_Neighbour(pNeighbour) )
						{
							maxValue = pNeighbour->Get_Area(); maxPolygon = j;
						}
					}
					else if( pPolygon->is_Neighbour(pNeighbour) ) // largest shared edge length
					{
						double sharedLength = pPolygon->Get_Shared_Length(pNeighbour, bVertices, Epsilon);

						if( maxValue < sharedLength )
						{
							maxValue = sharedLength; maxPolygon = j;
						}
					}
				}
			}

			if( maxPolygon >= 0 )
			{
				Join.Set_Value(FIELD_JOIN, maxPolygon);

				CSG_Table_Record &JoinTo = Joins[maxPolygon];

				JoinTo.Add_Value(FIELD_AREA , Join.asDouble(FIELD_AREA));
				JoinTo.Add_Value(FIELD_JOINS, 1);

				nJoins++;
			}
		}
	}

	return( nJoins > 0 );
}


///////////////////////////////////////////////////////////
//														 //
///////////////////////////////////////////////////////////

//---------------------------------------------------------
bool CPolygon_Generalization::Set_Joins(CSG_Shapes *pPolygons)
{
	CSG_Table Joins;

	if( !Get_Joins(pPolygons, Joins) )
	{
		return( false );
	}

	//-----------------------------------------------------
	for(sLong i=0; i<pPolygons->Get_Count() && Set_Progress(i, pPolygons->Get_Count()); i++)
	{
		if( Joins[i].asInt(FIELD_JOINS) > 0 )
		{
			CSG_Shape_Polygon *pPolygon = pPolygons->Get_Shape(i)->asPolygon();

			for(int iPart=0; iPart<pPolygon->Get_Part_Count(); iPart++)
			{
				if( pPolygon->is_Lake(iPart) == pPolygon->is_Clockwise(iPart) )
				{
					pPolygon->Revert_Points(iPart);
				}
			}

			Add_Joins(pPolygons, Joins, pPolygon, i);

			SG_Shape_Get_Dissolve(pPolygon);
		}
	}

	//-----------------------------------------------------
	for(sLong i=pPolygons->Get_Count()-1; i>=0; i--)
	{
		if( pPolygons->Get_Shape(i)->Get_Part_Count() < 1 )
		{
			pPolygons->Del_Shape(i);
		}
	}

	sLong nJoined = Joins.Get_Count() - pPolygons->Get_Count();

	Message_Fmt("\n%s: %lld", _TL("total number of removed polygons"), nJoined);

	return( nJoined > 0 );
}


///////////////////////////////////////////////////////////
//														 //
///////////////////////////////////////////////////////////

//---------------------------------------------------------
bool CPolygon_Generalization::Add_Joins(CSG_Shapes *pPolygons, CSG_Table &Joins, CSG_Shape *pPolygon, sLong id)
{
	for(sLong i=0; i<Joins.Get_Count(); i++)
	{
		if( pPolygon->Get_Index() != i && Joins[i].asInt(FIELD_JOIN) == id )
		{
			CSG_Shape_Polygon *pJoin = pPolygons->Get_Shape(i)->asPolygon();

			for(int iPart=0; iPart<pJoin->Get_Part_Count(); iPart++)
			{
				pPolygon->Add_Part(pJoin->Get_Part(iPart), pJoin->is_Lake(iPart) == pJoin->is_Clockwise(iPart));
			}

			pJoin->Del_Parts();

			if( Joins[i].asInt(FIELD_JOINS) )
			{
				Joins[i].Set_Value(FIELD_JOINS, 0.);

				Add_Joins(pPolygons, Joins, pPolygon, i); // recursive!
			}
		}
	}

	return( true );
}


///////////////////////////////////////////////////////////
//														 //
//														 //
//														 //
///////////////////////////////////////////////////////////

//---------------------------------------------------------
