// Copyright 2025 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package slhdsa

import (
	"slices"
	"testing"
)

type hashTestVector struct {
	n          uint32
	m          uint32
	r          []byte
	pkSeed     []byte
	pkRoot     []byte
	skSeed     []byte
	skPrf      []byte
	optRand    []byte
	msg        []byte
	wantHMsg   []byte
	wantPrf    []byte
	wantPrfMsg []byte
	wantF      []byte
	wantH      []byte
	wantTl     []byte
}

// Hash test vectors, generated using the reference implementation (https://github.com/sphincs/sphincsplus).
// Note that all of these are based on a zero address.
var (
	hashTestVectorShaken16m30 = hashTestVector{
		n:          16,
		m:          30,
		r:          []byte{0xf8, 0x1e, 0x43, 0xe8, 0xed, 0xee, 0xc2, 0x25, 0xb4, 0xd4, 0xee, 0xf6, 0x30, 0x13, 0xb, 0x99},
		pkSeed:     []byte{0xfe, 0x7b, 0xcf, 0x72, 0xbf, 0x0e, 0x9e, 0x08, 0x85, 0x3e, 0xb3, 0x80, 0x71, 0x99, 0x2c, 0xe5},
		pkRoot:     []byte{0xe6, 0x05, 0x10, 0x27, 0xba, 0xff, 0x1e, 0x80, 0x85, 0x09, 0x31, 0x3c, 0x27, 0x50, 0x65, 0xcb},
		skSeed:     []byte{0xd5, 0x03, 0xb1, 0x29, 0x70, 0xbb, 0xb8, 0x84, 0x56, 0x17, 0xd0, 0x16, 0x97, 0x3c, 0x5d, 0x16},
		skPrf:      []byte{0x07, 0xd5, 0x87, 0x3f, 0x99, 0x8f, 0x65, 0xd8, 0x8f, 0x81, 0xcd, 0xae, 0xd1, 0xc9, 0x8b, 0x30},
		optRand:    []byte{0xd5, 0xe0, 0xfe, 0xd3, 0xf4, 0xd4, 0xbf, 0x8f, 0x31, 0xc4, 0xd7, 0xbf, 0x60, 0xbd, 0xc1, 0x35},
		msg:        []byte{0x0, 0x10, 0x55, 0x41, 0xd, 0xbd, 0x5d, 0x32, 0x46, 0x4, 0xd, 0x46, 0xfe, 0x8b, 0x88, 0x42, 0xf2, 0xa2, 0x7, 0x11, 0x68, 0x7c, 0x42, 0xf, 0xf7, 0xf8, 0x8d, 0x89, 0x32, 0x62, 0xd8, 0x12, 0x8b, 0xdc},
		wantHMsg:   []byte{0x16, 0xaa, 0x69, 0x6b, 0xfe, 0x53, 0xc6, 0x8b, 0x38, 0x6c, 0x27, 0x42, 0xf9, 0xaa, 0x86, 0xee, 0xae, 0x99, 0xcf, 0x0, 0xdc, 0xee, 0x98, 0x7c, 0xdf, 0x96, 0x8c, 0x47, 0x24, 0x26},
		wantPrf:    []byte{0x15, 0xf4, 0x71, 0x7a, 0xdb, 0x75, 0xdb, 0xe7, 0x62, 0xd8, 0xee, 0x7, 0xa4, 0x7c, 0x17, 0xb9},
		wantPrfMsg: []byte{0x44, 0xf9, 0x96, 0x2, 0xba, 0x6c, 0xc7, 0xe0, 0xcd, 0xa3, 0x9a, 0xd8, 0xdf, 0xc6, 0x9, 0x9b},
		wantF:      []byte{0xf, 0x55, 0xe9, 0x48, 0x6, 0xf, 0xf9, 0xf5, 0xbc, 0xd4, 0x36, 0xc7, 0x1e, 0x2b, 0xd6, 0x62},
		wantH:      []byte{0xf, 0x55, 0xe9, 0x48, 0x6, 0xf, 0xf9, 0xf5, 0xbc, 0xd4, 0x36, 0xc7, 0x1e, 0x2b, 0xd6, 0x62},
		wantTl:     []byte{0xf, 0x55, 0xe9, 0x48, 0x6, 0xf, 0xf9, 0xf5, 0xbc, 0xd4, 0x36, 0xc7, 0x1e, 0x2b, 0xd6, 0x62},
	}
	hashTestVectorShaken16m34 = hashTestVector{
		n:          16,
		m:          34,
		r:          []byte{0xf8, 0x1e, 0x43, 0xe8, 0xed, 0xee, 0xc2, 0x25, 0xb4, 0xd4, 0xee, 0xf6, 0x30, 0x13, 0xb, 0x99},
		pkSeed:     []byte{0xfe, 0x7b, 0xcf, 0x72, 0xbf, 0x0e, 0x9e, 0x08, 0x85, 0x3e, 0xb3, 0x80, 0x71, 0x99, 0x2c, 0xe5},
		pkRoot:     []byte{0xe6, 0x05, 0x10, 0x27, 0xba, 0xff, 0x1e, 0x80, 0x85, 0x09, 0x31, 0x3c, 0x27, 0x50, 0x65, 0xcb},
		skSeed:     []byte{0xd5, 0x03, 0xb1, 0x29, 0x70, 0xbb, 0xb8, 0x84, 0x56, 0x17, 0xd0, 0x16, 0x97, 0x3c, 0x5d, 0x16},
		skPrf:      []byte{0x07, 0xd5, 0x87, 0x3f, 0x99, 0x8f, 0x65, 0xd8, 0x8f, 0x81, 0xcd, 0xae, 0xd1, 0xc9, 0x8b, 0x30},
		optRand:    []byte{0xd5, 0xe0, 0xfe, 0xd3, 0xf4, 0xd4, 0xbf, 0x8f, 0x31, 0xc4, 0xd7, 0xbf, 0x60, 0xbd, 0xc1, 0x35},
		msg:        []byte{0x0, 0x10, 0x55, 0x41, 0xd, 0xbd, 0x5d, 0x32, 0x46, 0x4, 0xd, 0x46, 0xfe, 0x8b, 0x88, 0x42, 0xf2, 0xa2, 0x7, 0x11, 0x68, 0x7c, 0x42, 0xf, 0xf7, 0xf8, 0x8d, 0x89, 0x32, 0x62, 0xd8, 0x12, 0x8b, 0xdc},
		wantHMsg:   []byte{0x16, 0xaa, 0x69, 0x6b, 0xfe, 0x53, 0xc6, 0x8b, 0x38, 0x6c, 0x27, 0x42, 0xf9, 0xaa, 0x86, 0xee, 0xae, 0x99, 0xcf, 0x0, 0xdc, 0xee, 0x98, 0x7c, 0xdf, 0x96, 0x8c, 0x47, 0x24, 0x26, 0x3a, 0x60, 0x4f, 0xd7},
		wantPrf:    []byte{0x15, 0xf4, 0x71, 0x7a, 0xdb, 0x75, 0xdb, 0xe7, 0x62, 0xd8, 0xee, 0x7, 0xa4, 0x7c, 0x17, 0xb9},
		wantPrfMsg: []byte{0x44, 0xf9, 0x96, 0x2, 0xba, 0x6c, 0xc7, 0xe0, 0xcd, 0xa3, 0x9a, 0xd8, 0xdf, 0xc6, 0x9, 0x9b},
		wantF:      []byte{0xf, 0x55, 0xe9, 0x48, 0x6, 0xf, 0xf9, 0xf5, 0xbc, 0xd4, 0x36, 0xc7, 0x1e, 0x2b, 0xd6, 0x62},
		wantH:      []byte{0xf, 0x55, 0xe9, 0x48, 0x6, 0xf, 0xf9, 0xf5, 0xbc, 0xd4, 0x36, 0xc7, 0x1e, 0x2b, 0xd6, 0x62},
		wantTl:     []byte{0xf, 0x55, 0xe9, 0x48, 0x6, 0xf, 0xf9, 0xf5, 0xbc, 0xd4, 0x36, 0xc7, 0x1e, 0x2b, 0xd6, 0x62},
	}
	hashTestVectorShaken24m39 = hashTestVector{
		n:          24,
		m:          39,
		r:          []byte{0x5d, 0xd5, 0xc0, 0xb2, 0xf4, 0x5f, 0xb4, 0x78, 0xe3, 0xb7, 0x5b, 0x33, 0x82, 0x8e, 0x7f, 0xda, 0x20, 0xcb, 0x35, 0xbd, 0xa0, 0x4c, 0xfd, 0x13},
		pkSeed:     []byte{0xfe, 0x70, 0xc, 0x33, 0xd6, 0x30, 0x4d, 0x31, 0xaa, 0x7e, 0xe, 0x5b, 0xb8, 0xeb, 0xbe, 0xe9, 0x5, 0xf5, 0x42, 0x9d, 0x11, 0xe, 0x98, 0xf9},
		pkRoot:     []byte{0xbc, 0x3f, 0x28, 0x92, 0x43, 0xb2, 0x11, 0xb5, 0x3f, 0x15, 0xc3, 0xa, 0xc4, 0xf1, 0x34, 0x82, 0x43, 0x9e, 0x5b, 0x4b, 0xaa, 0x62, 0xf5, 0xd0},
		skSeed:     []byte{0x54, 0xfc, 0x58, 0x34, 0xc7, 0x9f, 0x20, 0x6a, 0x9d, 0x45, 0x83, 0xe0, 0x12, 0xdc, 0x2, 0x2c, 0x10, 0x47, 0xf4, 0x58, 0x50, 0xe1, 0xf7, 0xbf},
		skPrf:      []byte{0xf2, 0xc4, 0xf, 0xa7, 0xff, 0x9b, 0xd5, 0x6b, 0xd1, 0x28, 0x78, 0x89, 0xd7, 0xbc, 0x69, 0xf0, 0x8f, 0x31, 0x1, 0x7a, 0xc7, 0xca, 0x22, 0x39},
		optRand:    []byte{0x2d, 0x32, 0x27, 0xbb, 0xa8, 0xc5, 0xad, 0x18, 0xfc, 0x81, 0x8a, 0x3f, 0xab, 0x8c, 0x36, 0x3b, 0x47, 0x53, 0xce, 0xe1, 0xa3, 0x58, 0x8c, 0xe8},
		msg:        []byte{0x0, 0x10, 0xb2, 0x39, 0x59, 0x9c, 0xbc, 0xa2, 0xd, 0x4d, 0x26, 0x46, 0xb1, 0x20, 0x13, 0xa6, 0x53, 0xd3, 0xd4, 0xee, 0x59, 0xc7, 0x7a, 0x7d, 0x59, 0x79, 0x9, 0xd9, 0x52, 0x46, 0x7c, 0x9, 0xf9, 0x2f},
		wantHMsg:   []byte{0x74, 0xc2, 0xd9, 0xc4, 0x65, 0xd5, 0xf6, 0x4f, 0xcf, 0x40, 0x61, 0x4, 0x84, 0x2c, 0x98, 0xd3, 0x35, 0x16, 0xcd, 0xdb, 0x1f, 0xaf, 0x72, 0xd, 0x61, 0x47, 0x39, 0x70, 0x1d, 0xd, 0x32, 0x94, 0xdd, 0x83, 0xe4, 0x79, 0x7d, 0xf3, 0x84},
		wantPrf:    []byte{0x81, 0xa3, 0x62, 0xd6, 0x57, 0xb6, 0x2, 0x6d, 0x62, 0x13, 0x67, 0x64, 0x42, 0x2d, 0x64, 0xc9, 0x18, 0x51, 0x2a, 0xb6, 0x4d, 0xa, 0x72, 0x98},
		wantPrfMsg: []byte{0x92, 0xa, 0x9a, 0x1b, 0x90, 0x1f, 0x1b, 0x87, 0x8, 0xf8, 0x2d, 0x47, 0xf7, 0xa, 0x90, 0xac, 0x1b, 0xe7, 0xd1, 0x63, 0xef, 0x72, 0x5e, 0x17},
		wantF:      []byte{0xef, 0x45, 0xf6, 0xfc, 0x8b, 0xb, 0xb8, 0x72, 0xe5, 0xda, 0xa9, 0x26, 0x78, 0x56, 0xdf, 0x82, 0x4c, 0x70, 0xc, 0x42, 0xac, 0xff, 0x2c, 0x74},
		wantH:      []byte{0xef, 0x45, 0xf6, 0xfc, 0x8b, 0xb, 0xb8, 0x72, 0xe5, 0xda, 0xa9, 0x26, 0x78, 0x56, 0xdf, 0x82, 0x4c, 0x70, 0xc, 0x42, 0xac, 0xff, 0x2c, 0x74},
		wantTl:     []byte{0xef, 0x45, 0xf6, 0xfc, 0x8b, 0xb, 0xb8, 0x72, 0xe5, 0xda, 0xa9, 0x26, 0x78, 0x56, 0xdf, 0x82, 0x4c, 0x70, 0xc, 0x42, 0xac, 0xff, 0x2c, 0x74},
	}
	hashTestVectorShaken24m42 = hashTestVector{
		n:          24,
		m:          42,
		r:          []byte{0x5d, 0xd5, 0xc0, 0xb2, 0xf4, 0x5f, 0xb4, 0x78, 0xe3, 0xb7, 0x5b, 0x33, 0x82, 0x8e, 0x7f, 0xda, 0x20, 0xcb, 0x35, 0xbd, 0xa0, 0x4c, 0xfd, 0x13},
		pkSeed:     []byte{0xfe, 0x70, 0xc, 0x33, 0xd6, 0x30, 0x4d, 0x31, 0xaa, 0x7e, 0xe, 0x5b, 0xb8, 0xeb, 0xbe, 0xe9, 0x5, 0xf5, 0x42, 0x9d, 0x11, 0xe, 0x98, 0xf9},
		pkRoot:     []byte{0xbc, 0x3f, 0x28, 0x92, 0x43, 0xb2, 0x11, 0xb5, 0x3f, 0x15, 0xc3, 0xa, 0xc4, 0xf1, 0x34, 0x82, 0x43, 0x9e, 0x5b, 0x4b, 0xaa, 0x62, 0xf5, 0xd0},
		skSeed:     []byte{0x54, 0xfc, 0x58, 0x34, 0xc7, 0x9f, 0x20, 0x6a, 0x9d, 0x45, 0x83, 0xe0, 0x12, 0xdc, 0x2, 0x2c, 0x10, 0x47, 0xf4, 0x58, 0x50, 0xe1, 0xf7, 0xbf},
		skPrf:      []byte{0xf2, 0xc4, 0xf, 0xa7, 0xff, 0x9b, 0xd5, 0x6b, 0xd1, 0x28, 0x78, 0x89, 0xd7, 0xbc, 0x69, 0xf0, 0x8f, 0x31, 0x1, 0x7a, 0xc7, 0xca, 0x22, 0x39},
		optRand:    []byte{0x2d, 0x32, 0x27, 0xbb, 0xa8, 0xc5, 0xad, 0x18, 0xfc, 0x81, 0x8a, 0x3f, 0xab, 0x8c, 0x36, 0x3b, 0x47, 0x53, 0xce, 0xe1, 0xa3, 0x58, 0x8c, 0xe8},
		msg:        []byte{0x0, 0x10, 0xb2, 0x39, 0x59, 0x9c, 0xbc, 0xa2, 0xd, 0x4d, 0x26, 0x46, 0xb1, 0x20, 0x13, 0xa6, 0x53, 0xd3, 0xd4, 0xee, 0x59, 0xc7, 0x7a, 0x7d, 0x59, 0x79, 0x9, 0xd9, 0x52, 0x46, 0x7c, 0x9, 0xf9, 0x2f},
		wantHMsg:   []byte{0x74, 0xc2, 0xd9, 0xc4, 0x65, 0xd5, 0xf6, 0x4f, 0xcf, 0x40, 0x61, 0x4, 0x84, 0x2c, 0x98, 0xd3, 0x35, 0x16, 0xcd, 0xdb, 0x1f, 0xaf, 0x72, 0xd, 0x61, 0x47, 0x39, 0x70, 0x1d, 0xd, 0x32, 0x94, 0xdd, 0x83, 0xe4, 0x79, 0x7d, 0xf3, 0x84, 0xad, 0xdb, 0x5c},
		wantPrf:    []byte{0x81, 0xa3, 0x62, 0xd6, 0x57, 0xb6, 0x2, 0x6d, 0x62, 0x13, 0x67, 0x64, 0x42, 0x2d, 0x64, 0xc9, 0x18, 0x51, 0x2a, 0xb6, 0x4d, 0xa, 0x72, 0x98},
		wantPrfMsg: []byte{0x92, 0xa, 0x9a, 0x1b, 0x90, 0x1f, 0x1b, 0x87, 0x8, 0xf8, 0x2d, 0x47, 0xf7, 0xa, 0x90, 0xac, 0x1b, 0xe7, 0xd1, 0x63, 0xef, 0x72, 0x5e, 0x17},
		wantF:      []byte{0xef, 0x45, 0xf6, 0xfc, 0x8b, 0xb, 0xb8, 0x72, 0xe5, 0xda, 0xa9, 0x26, 0x78, 0x56, 0xdf, 0x82, 0x4c, 0x70, 0xc, 0x42, 0xac, 0xff, 0x2c, 0x74},
		wantH:      []byte{0xef, 0x45, 0xf6, 0xfc, 0x8b, 0xb, 0xb8, 0x72, 0xe5, 0xda, 0xa9, 0x26, 0x78, 0x56, 0xdf, 0x82, 0x4c, 0x70, 0xc, 0x42, 0xac, 0xff, 0x2c, 0x74},
		wantTl:     []byte{0xef, 0x45, 0xf6, 0xfc, 0x8b, 0xb, 0xb8, 0x72, 0xe5, 0xda, 0xa9, 0x26, 0x78, 0x56, 0xdf, 0x82, 0x4c, 0x70, 0xc, 0x42, 0xac, 0xff, 0x2c, 0x74},
	}
	hashTestVectorShaken32m47 = hashTestVector{
		n:          32,
		m:          47,
		r:          []byte{0x5d, 0xd5, 0xc0, 0xb2, 0xf4, 0x5f, 0xb4, 0x78, 0xe3, 0xb7, 0x5b, 0x33, 0x82, 0x8e, 0x7f, 0xda, 0x20, 0xcb, 0x35, 0xbd, 0xa0, 0x4c, 0xfd, 0x13, 0xf8, 0x1e, 0x43, 0xe8, 0xed, 0xee, 0xc2, 0x25},
		pkSeed:     []byte{0xfe, 0x70, 0xc, 0x33, 0xd6, 0x30, 0x4d, 0x31, 0xaa, 0x7e, 0xe, 0x5b, 0xb8, 0xeb, 0xbe, 0xe9, 0x5, 0xf5, 0x42, 0x9d, 0x11, 0xe, 0x98, 0xf9, 0xf8, 0x1e, 0x43, 0xe8, 0xed, 0xee, 0xc2, 0x25},
		pkRoot:     []byte{0xbc, 0x3f, 0x28, 0x92, 0x43, 0xb2, 0x11, 0xb5, 0x3f, 0x15, 0xc3, 0xa, 0xc4, 0xf1, 0x34, 0x82, 0x43, 0x9e, 0x5b, 0x4b, 0xaa, 0x62, 0xf5, 0xd0, 0xf8, 0x1e, 0x43, 0xe8, 0xed, 0xee, 0xc2, 0x25},
		skSeed:     []byte{0x54, 0xfc, 0x58, 0x34, 0xc7, 0x9f, 0x20, 0x6a, 0x9d, 0x45, 0x83, 0xe0, 0x12, 0xdc, 0x2, 0x2c, 0x10, 0x47, 0xf4, 0x58, 0x50, 0xe1, 0xf7, 0xbf, 0xf8, 0x1e, 0x43, 0xe8, 0xed, 0xee, 0xc2, 0x25},
		skPrf:      []byte{0xf2, 0xc4, 0xf, 0xa7, 0xff, 0x9b, 0xd5, 0x6b, 0xd1, 0x28, 0x78, 0x89, 0xd7, 0xbc, 0x69, 0xf0, 0x8f, 0x31, 0x1, 0x7a, 0xc7, 0xca, 0x22, 0x39, 0xf8, 0x1e, 0x43, 0xe8, 0xed, 0xee, 0xc2, 0x25},
		optRand:    []byte{0x2d, 0x32, 0x27, 0xbb, 0xa8, 0xc5, 0xad, 0x18, 0xfc, 0x81, 0x8a, 0x3f, 0xab, 0x8c, 0x36, 0x3b, 0x47, 0x53, 0xce, 0xe1, 0xa3, 0x58, 0x8c, 0xe8, 0xf8, 0x1e, 0x43, 0xe8, 0xed, 0xee, 0xc2, 0x25},
		msg:        []byte{0x0, 0x10, 0xb2, 0x39, 0x59, 0x9c, 0xbc, 0xa2, 0xd, 0x4d, 0x26, 0x46, 0xb1, 0x20, 0x13, 0xa6, 0x53, 0xd3, 0xd4, 0xee, 0x59, 0xc7, 0x7a, 0x7d, 0x59, 0x79, 0x9, 0xd9, 0x52, 0x46, 0x7c, 0x9, 0xf9, 0x2f},
		wantHMsg:   []byte{0xd8, 0x12, 0xf6, 0x62, 0x13, 0x7d, 0x8d, 0xc6, 0x74, 0xaa, 0xce, 0x36, 0x0, 0x54, 0x85, 0x42, 0x2b, 0x3e, 0xb7, 0x71, 0xf, 0xea, 0x9a, 0x5, 0xf0, 0x1c, 0xfe, 0x71, 0x60, 0x3e, 0xdc, 0xa6, 0xbc, 0xaa, 0x8b, 0xdd, 0xb2, 0x4a, 0xd4, 0xfc, 0x3c, 0x33, 0xf8, 0x8b, 0x96, 0x7d, 0x8c},
		wantPrf:    []byte{0xb9, 0x60, 0xec, 0x82, 0x12, 0x19, 0xa2, 0x2c, 0x35, 0xf9, 0x6e, 0x60, 0x1d, 0x4e, 0x72, 0x81, 0xa6, 0xb8, 0x5f, 0x77, 0x90, 0xeb, 0x20, 0xf9, 0xd, 0xc0, 0x2d, 0x87, 0x80, 0x9b, 0xce, 0x33},
		wantPrfMsg: []byte{0x16, 0xb7, 0xe5, 0x77, 0x88, 0x8c, 0x6d, 0x99, 0x60, 0x34, 0xf, 0x1c, 0xd1, 0x3, 0xc9, 0x51, 0xe2, 0xa0, 0x1c, 0x6, 0x95, 0x34, 0xf7, 0x42, 0x6c, 0xc2, 0x76, 0xc8, 0x2, 0xc, 0xc6, 0xc8},
		wantF:      []byte{0xf8, 0x75, 0x59, 0xac, 0xca, 0xb7, 0x17, 0x5f, 0xc2, 0x72, 0x53, 0xfb, 0x4d, 0x17, 0x12, 0xf, 0xfe, 0xda, 0xc3, 0x30, 0xa2, 0x1, 0xcf, 0x74, 0xe6, 0x63, 0xc3, 0x49, 0xf9, 0x30, 0x8a, 0x10},
		wantH:      []byte{0xf8, 0x75, 0x59, 0xac, 0xca, 0xb7, 0x17, 0x5f, 0xc2, 0x72, 0x53, 0xfb, 0x4d, 0x17, 0x12, 0xf, 0xfe, 0xda, 0xc3, 0x30, 0xa2, 0x1, 0xcf, 0x74, 0xe6, 0x63, 0xc3, 0x49, 0xf9, 0x30, 0x8a, 0x10},
		wantTl:     []byte{0xf8, 0x75, 0x59, 0xac, 0xca, 0xb7, 0x17, 0x5f, 0xc2, 0x72, 0x53, 0xfb, 0x4d, 0x17, 0x12, 0xf, 0xfe, 0xda, 0xc3, 0x30, 0xa2, 0x1, 0xcf, 0x74, 0xe6, 0x63, 0xc3, 0x49, 0xf9, 0x30, 0x8a, 0x10},
	}
	hashTestVectorShaken32m49 = hashTestVector{
		n:          32,
		m:          49,
		r:          []byte{0x5d, 0xd5, 0xc0, 0xb2, 0xf4, 0x5f, 0xb4, 0x78, 0xe3, 0xb7, 0x5b, 0x33, 0x82, 0x8e, 0x7f, 0xda, 0x20, 0xcb, 0x35, 0xbd, 0xa0, 0x4c, 0xfd, 0x13, 0xf8, 0x1e, 0x43, 0xe8, 0xed, 0xee, 0xc2, 0x25},
		pkSeed:     []byte{0xfe, 0x70, 0xc, 0x33, 0xd6, 0x30, 0x4d, 0x31, 0xaa, 0x7e, 0xe, 0x5b, 0xb8, 0xeb, 0xbe, 0xe9, 0x5, 0xf5, 0x42, 0x9d, 0x11, 0xe, 0x98, 0xf9, 0xf8, 0x1e, 0x43, 0xe8, 0xed, 0xee, 0xc2, 0x25},
		pkRoot:     []byte{0xbc, 0x3f, 0x28, 0x92, 0x43, 0xb2, 0x11, 0xb5, 0x3f, 0x15, 0xc3, 0xa, 0xc4, 0xf1, 0x34, 0x82, 0x43, 0x9e, 0x5b, 0x4b, 0xaa, 0x62, 0xf5, 0xd0, 0xf8, 0x1e, 0x43, 0xe8, 0xed, 0xee, 0xc2, 0x25},
		skSeed:     []byte{0x54, 0xfc, 0x58, 0x34, 0xc7, 0x9f, 0x20, 0x6a, 0x9d, 0x45, 0x83, 0xe0, 0x12, 0xdc, 0x2, 0x2c, 0x10, 0x47, 0xf4, 0x58, 0x50, 0xe1, 0xf7, 0xbf, 0xf8, 0x1e, 0x43, 0xe8, 0xed, 0xee, 0xc2, 0x25},
		skPrf:      []byte{0xf2, 0xc4, 0xf, 0xa7, 0xff, 0x9b, 0xd5, 0x6b, 0xd1, 0x28, 0x78, 0x89, 0xd7, 0xbc, 0x69, 0xf0, 0x8f, 0x31, 0x1, 0x7a, 0xc7, 0xca, 0x22, 0x39, 0xf8, 0x1e, 0x43, 0xe8, 0xed, 0xee, 0xc2, 0x25},
		optRand:    []byte{0x2d, 0x32, 0x27, 0xbb, 0xa8, 0xc5, 0xad, 0x18, 0xfc, 0x81, 0x8a, 0x3f, 0xab, 0x8c, 0x36, 0x3b, 0x47, 0x53, 0xce, 0xe1, 0xa3, 0x58, 0x8c, 0xe8, 0xf8, 0x1e, 0x43, 0xe8, 0xed, 0xee, 0xc2, 0x25},
		msg:        []byte{0x0, 0x10, 0xb2, 0x39, 0x59, 0x9c, 0xbc, 0xa2, 0xd, 0x4d, 0x26, 0x46, 0xb1, 0x20, 0x13, 0xa6, 0x53, 0xd3, 0xd4, 0xee, 0x59, 0xc7, 0x7a, 0x7d, 0x59, 0x79, 0x9, 0xd9, 0x52, 0x46, 0x7c, 0x9, 0xf9, 0x2f},
		wantHMsg:   []byte{0xd8, 0x12, 0xf6, 0x62, 0x13, 0x7d, 0x8d, 0xc6, 0x74, 0xaa, 0xce, 0x36, 0x0, 0x54, 0x85, 0x42, 0x2b, 0x3e, 0xb7, 0x71, 0xf, 0xea, 0x9a, 0x5, 0xf0, 0x1c, 0xfe, 0x71, 0x60, 0x3e, 0xdc, 0xa6, 0xbc, 0xaa, 0x8b, 0xdd, 0xb2, 0x4a, 0xd4, 0xfc, 0x3c, 0x33, 0xf8, 0x8b, 0x96, 0x7d, 0x8c, 0x7b, 0x47},
		wantPrf:    []byte{0xb9, 0x60, 0xec, 0x82, 0x12, 0x19, 0xa2, 0x2c, 0x35, 0xf9, 0x6e, 0x60, 0x1d, 0x4e, 0x72, 0x81, 0xa6, 0xb8, 0x5f, 0x77, 0x90, 0xeb, 0x20, 0xf9, 0xd, 0xc0, 0x2d, 0x87, 0x80, 0x9b, 0xce, 0x33},
		wantPrfMsg: []byte{0x16, 0xb7, 0xe5, 0x77, 0x88, 0x8c, 0x6d, 0x99, 0x60, 0x34, 0xf, 0x1c, 0xd1, 0x3, 0xc9, 0x51, 0xe2, 0xa0, 0x1c, 0x6, 0x95, 0x34, 0xf7, 0x42, 0x6c, 0xc2, 0x76, 0xc8, 0x2, 0xc, 0xc6, 0xc8},
		wantF:      []byte{0xf8, 0x75, 0x59, 0xac, 0xca, 0xb7, 0x17, 0x5f, 0xc2, 0x72, 0x53, 0xfb, 0x4d, 0x17, 0x12, 0xf, 0xfe, 0xda, 0xc3, 0x30, 0xa2, 0x1, 0xcf, 0x74, 0xe6, 0x63, 0xc3, 0x49, 0xf9, 0x30, 0x8a, 0x10},
		wantH:      []byte{0xf8, 0x75, 0x59, 0xac, 0xca, 0xb7, 0x17, 0x5f, 0xc2, 0x72, 0x53, 0xfb, 0x4d, 0x17, 0x12, 0xf, 0xfe, 0xda, 0xc3, 0x30, 0xa2, 0x1, 0xcf, 0x74, 0xe6, 0x63, 0xc3, 0x49, 0xf9, 0x30, 0x8a, 0x10},
		wantTl:     []byte{0xf8, 0x75, 0x59, 0xac, 0xca, 0xb7, 0x17, 0x5f, 0xc2, 0x72, 0x53, 0xfb, 0x4d, 0x17, 0x12, 0xf, 0xfe, 0xda, 0xc3, 0x30, 0xa2, 0x1, 0xcf, 0x74, 0xe6, 0x63, 0xc3, 0x49, 0xf9, 0x30, 0x8a, 0x10},
	}
	hashTestVectorSha2C1n16m30 = hashTestVector{
		n:          16,
		m:          30,
		r:          []byte{0xf8, 0x1e, 0x43, 0xe8, 0xed, 0xee, 0xc2, 0x25, 0xb4, 0xd4, 0xee, 0xf6, 0x30, 0x13, 0xb, 0x99},
		pkSeed:     []byte{0xfe, 0x7b, 0xcf, 0x72, 0xbf, 0x0e, 0x9e, 0x08, 0x85, 0x3e, 0xb3, 0x80, 0x71, 0x99, 0x2c, 0xe5},
		pkRoot:     []byte{0xe6, 0x05, 0x10, 0x27, 0xba, 0xff, 0x1e, 0x80, 0x85, 0x09, 0x31, 0x3c, 0x27, 0x50, 0x65, 0xcb},
		skSeed:     []byte{0xd5, 0x03, 0xb1, 0x29, 0x70, 0xbb, 0xb8, 0x84, 0x56, 0x17, 0xd0, 0x16, 0x97, 0x3c, 0x5d, 0x16},
		skPrf:      []byte{0x07, 0xd5, 0x87, 0x3f, 0x99, 0x8f, 0x65, 0xd8, 0x8f, 0x81, 0xcd, 0xae, 0xd1, 0xc9, 0x8b, 0x30},
		optRand:    []byte{0xd5, 0xe0, 0xfe, 0xd3, 0xf4, 0xd4, 0xbf, 0x8f, 0x31, 0xc4, 0xd7, 0xbf, 0x60, 0xbd, 0xc1, 0x35},
		msg:        []byte{0x0, 0x10, 0x55, 0x41, 0xd, 0xbd, 0x5d, 0x32, 0x46, 0x4, 0xd, 0x46, 0xfe, 0x8b, 0x88, 0x42, 0xf2, 0xa2, 0x7, 0x11, 0x68, 0x7c, 0x42, 0xf, 0xf7, 0xf8, 0x8d, 0x89, 0x32, 0x62, 0xd8, 0x12, 0x8b, 0xdc},
		wantHMsg:   []byte{0xb1, 0x4e, 0xeb, 0x51, 0x2, 0x42, 0xdf, 0xc2, 0xbf, 0x38, 0x56, 0xe1, 0x52, 0x24, 0x2e, 0xf, 0xb9, 0x7c, 0xfa, 0x8c, 0xe6, 0x83, 0x96, 0xf3, 0x7f, 0xe4, 0x50, 0x1f, 0x9d, 0xbc},
		wantPrf:    []byte{0x2, 0x5d, 0xc5, 0xba, 0xf0, 0x65, 0xa4, 0xa1, 0xc, 0x87, 0x43, 0x36, 0x39, 0x60, 0x30, 0x1f},
		wantPrfMsg: []byte{0xe7, 0x50, 0xf, 0xdf, 0x6f, 0x9b, 0x1d, 0x7d, 0x55, 0x9e, 0x98, 0xbd, 0xdb, 0x82, 0xb1, 0xea},
		wantF:      []byte{0x28, 0x6d, 0x30, 0x47, 0x1c, 0xa1, 0x4c, 0xf9, 0x26, 0x15, 0x14, 0xec, 0x5e, 0x27, 0x8d, 0xad},
		wantH:      []byte{0x28, 0x6d, 0x30, 0x47, 0x1c, 0xa1, 0x4c, 0xf9, 0x26, 0x15, 0x14, 0xec, 0x5e, 0x27, 0x8d, 0xad},
		wantTl:     []byte{0x28, 0x6d, 0x30, 0x47, 0x1c, 0xa1, 0x4c, 0xf9, 0x26, 0x15, 0x14, 0xec, 0x5e, 0x27, 0x8d, 0xad},
	}
	hashTestVectorSha2C1n16m34 = hashTestVector{
		n:          16,
		m:          34,
		r:          []byte{0xf8, 0x1e, 0x43, 0xe8, 0xed, 0xee, 0xc2, 0x25, 0xb4, 0xd4, 0xee, 0xf6, 0x30, 0x13, 0xb, 0x99},
		pkSeed:     []byte{0xfe, 0x7b, 0xcf, 0x72, 0xbf, 0x0e, 0x9e, 0x08, 0x85, 0x3e, 0xb3, 0x80, 0x71, 0x99, 0x2c, 0xe5},
		pkRoot:     []byte{0xe6, 0x05, 0x10, 0x27, 0xba, 0xff, 0x1e, 0x80, 0x85, 0x09, 0x31, 0x3c, 0x27, 0x50, 0x65, 0xcb},
		skSeed:     []byte{0xd5, 0x03, 0xb1, 0x29, 0x70, 0xbb, 0xb8, 0x84, 0x56, 0x17, 0xd0, 0x16, 0x97, 0x3c, 0x5d, 0x16},
		skPrf:      []byte{0x07, 0xd5, 0x87, 0x3f, 0x99, 0x8f, 0x65, 0xd8, 0x8f, 0x81, 0xcd, 0xae, 0xd1, 0xc9, 0x8b, 0x30},
		optRand:    []byte{0xd5, 0xe0, 0xfe, 0xd3, 0xf4, 0xd4, 0xbf, 0x8f, 0x31, 0xc4, 0xd7, 0xbf, 0x60, 0xbd, 0xc1, 0x35},
		msg:        []byte{0x0, 0x10, 0x55, 0x41, 0xd, 0xbd, 0x5d, 0x32, 0x46, 0x4, 0xd, 0x46, 0xfe, 0x8b, 0x88, 0x42, 0xf2, 0xa2, 0x7, 0x11, 0x68, 0x7c, 0x42, 0xf, 0xf7, 0xf8, 0x8d, 0x89, 0x32, 0x62, 0xd8, 0x12, 0x8b, 0xdc},
		wantHMsg:   []byte{0xb1, 0x4e, 0xeb, 0x51, 0x2, 0x42, 0xdf, 0xc2, 0xbf, 0x38, 0x56, 0xe1, 0x52, 0x24, 0x2e, 0xf, 0xb9, 0x7c, 0xfa, 0x8c, 0xe6, 0x83, 0x96, 0xf3, 0x7f, 0xe4, 0x50, 0x1f, 0x9d, 0xbc, 0x55, 0x43, 0x70, 0x31},
		wantPrf:    []byte{0x2, 0x5d, 0xc5, 0xba, 0xf0, 0x65, 0xa4, 0xa1, 0xc, 0x87, 0x43, 0x36, 0x39, 0x60, 0x30, 0x1f},
		wantPrfMsg: []byte{0xe7, 0x50, 0xf, 0xdf, 0x6f, 0x9b, 0x1d, 0x7d, 0x55, 0x9e, 0x98, 0xbd, 0xdb, 0x82, 0xb1, 0xea},
		wantF:      []byte{0x28, 0x6d, 0x30, 0x47, 0x1c, 0xa1, 0x4c, 0xf9, 0x26, 0x15, 0x14, 0xec, 0x5e, 0x27, 0x8d, 0xad},
		wantH:      []byte{0x28, 0x6d, 0x30, 0x47, 0x1c, 0xa1, 0x4c, 0xf9, 0x26, 0x15, 0x14, 0xec, 0x5e, 0x27, 0x8d, 0xad},
		wantTl:     []byte{0x28, 0x6d, 0x30, 0x47, 0x1c, 0xa1, 0x4c, 0xf9, 0x26, 0x15, 0x14, 0xec, 0x5e, 0x27, 0x8d, 0xad},
	}
	hashTestVectorSha2C35n24m39 = hashTestVector{
		n:          24,
		m:          39,
		r:          []byte{0x5d, 0xd5, 0xc0, 0xb2, 0xf4, 0x5f, 0xb4, 0x78, 0xe3, 0xb7, 0x5b, 0x33, 0x82, 0x8e, 0x7f, 0xda, 0x20, 0xcb, 0x35, 0xbd, 0xa0, 0x4c, 0xfd, 0x13},
		pkSeed:     []byte{0xfe, 0x70, 0xc, 0x33, 0xd6, 0x30, 0x4d, 0x31, 0xaa, 0x7e, 0xe, 0x5b, 0xb8, 0xeb, 0xbe, 0xe9, 0x5, 0xf5, 0x42, 0x9d, 0x11, 0xe, 0x98, 0xf9},
		pkRoot:     []byte{0xbc, 0x3f, 0x28, 0x92, 0x43, 0xb2, 0x11, 0xb5, 0x3f, 0x15, 0xc3, 0xa, 0xc4, 0xf1, 0x34, 0x82, 0x43, 0x9e, 0x5b, 0x4b, 0xaa, 0x62, 0xf5, 0xd0},
		skSeed:     []byte{0x54, 0xfc, 0x58, 0x34, 0xc7, 0x9f, 0x20, 0x6a, 0x9d, 0x45, 0x83, 0xe0, 0x12, 0xdc, 0x2, 0x2c, 0x10, 0x47, 0xf4, 0x58, 0x50, 0xe1, 0xf7, 0xbf},
		skPrf:      []byte{0xf2, 0xc4, 0xf, 0xa7, 0xff, 0x9b, 0xd5, 0x6b, 0xd1, 0x28, 0x78, 0x89, 0xd7, 0xbc, 0x69, 0xf0, 0x8f, 0x31, 0x1, 0x7a, 0xc7, 0xca, 0x22, 0x39},
		optRand:    []byte{0x2d, 0x32, 0x27, 0xbb, 0xa8, 0xc5, 0xad, 0x18, 0xfc, 0x81, 0x8a, 0x3f, 0xab, 0x8c, 0x36, 0x3b, 0x47, 0x53, 0xce, 0xe1, 0xa3, 0x58, 0x8c, 0xe8},
		msg:        []byte{0x0, 0x10, 0xb2, 0x39, 0x59, 0x9c, 0xbc, 0xa2, 0xd, 0x4d, 0x26, 0x46, 0xb1, 0x20, 0x13, 0xa6, 0x53, 0xd3, 0xd4, 0xee, 0x59, 0xc7, 0x7a, 0x7d, 0x59, 0x79, 0x9, 0xd9, 0x52, 0x46, 0x7c, 0x9, 0xf9, 0x2f},
		wantHMsg:   []byte{0xf2, 0x2, 0xec, 0x32, 0xb2, 0x92, 0x81, 0x92, 0x75, 0x93, 0xd, 0x4d, 0xb8, 0xd0, 0x4, 0xec, 0x1a, 0x2e, 0xc0, 0x4e, 0x42, 0x2e, 0xdb, 0xab, 0x8a, 0x44, 0xa8, 0x42, 0x4a, 0x66, 0x2e, 0x4f, 0x4a, 0xda, 0xc7, 0x7c, 0x35, 0xda, 0xd9},
		wantPrf:    []byte{0xe0, 0x54, 0x6f, 0x48, 0xa3, 0x1d, 0x3a, 0xcb, 0x6, 0x1b, 0xc4, 0xac, 0x8a, 0x6c, 0x92, 0x41, 0x10, 0x53, 0xf, 0xe6, 0x43, 0xd6, 0xab, 0x51},
		wantPrfMsg: []byte{0xd8, 0xc2, 0xea, 0xff, 0x63, 0x75, 0xcc, 0x31, 0x52, 0x75, 0x56, 0xdd, 0x54, 0xf7, 0x61, 0xf1, 0x96, 0x47, 0x44, 0xd, 0x98, 0xa2, 0x95, 0xd5},
		wantF:      []byte{0x7f, 0xe5, 0x99, 0x4e, 0x42, 0xbd, 0x2a, 0x2e, 0x94, 0x13, 0xe6, 0xcc, 0xd0, 0x3, 0xa9, 0xb1, 0x1d, 0x43, 0x74, 0x54, 0xd4, 0x35, 0x54, 0x88},
		wantH:      []byte{0x8f, 0xf0, 0x9b, 0x79, 0xa3, 0xc, 0x72, 0x7f, 0xb1, 0x87, 0x86, 0x17, 0x7c, 0xe7, 0xbd, 0xea, 0x1d, 0x54, 0xd3, 0xdd, 0x24, 0x32, 0x49, 0x2b},
		wantTl:     []byte{0x8f, 0xf0, 0x9b, 0x79, 0xa3, 0xc, 0x72, 0x7f, 0xb1, 0x87, 0x86, 0x17, 0x7c, 0xe7, 0xbd, 0xea, 0x1d, 0x54, 0xd3, 0xdd, 0x24, 0x32, 0x49, 0x2b},
	}
	hashTestVectorSha2C35n24m42 = hashTestVector{
		n:          24,
		m:          42,
		r:          []byte{0x5d, 0xd5, 0xc0, 0xb2, 0xf4, 0x5f, 0xb4, 0x78, 0xe3, 0xb7, 0x5b, 0x33, 0x82, 0x8e, 0x7f, 0xda, 0x20, 0xcb, 0x35, 0xbd, 0xa0, 0x4c, 0xfd, 0x13},
		pkSeed:     []byte{0xfe, 0x70, 0xc, 0x33, 0xd6, 0x30, 0x4d, 0x31, 0xaa, 0x7e, 0xe, 0x5b, 0xb8, 0xeb, 0xbe, 0xe9, 0x5, 0xf5, 0x42, 0x9d, 0x11, 0xe, 0x98, 0xf9},
		pkRoot:     []byte{0xbc, 0x3f, 0x28, 0x92, 0x43, 0xb2, 0x11, 0xb5, 0x3f, 0x15, 0xc3, 0xa, 0xc4, 0xf1, 0x34, 0x82, 0x43, 0x9e, 0x5b, 0x4b, 0xaa, 0x62, 0xf5, 0xd0},
		skSeed:     []byte{0x54, 0xfc, 0x58, 0x34, 0xc7, 0x9f, 0x20, 0x6a, 0x9d, 0x45, 0x83, 0xe0, 0x12, 0xdc, 0x2, 0x2c, 0x10, 0x47, 0xf4, 0x58, 0x50, 0xe1, 0xf7, 0xbf},
		skPrf:      []byte{0xf2, 0xc4, 0xf, 0xa7, 0xff, 0x9b, 0xd5, 0x6b, 0xd1, 0x28, 0x78, 0x89, 0xd7, 0xbc, 0x69, 0xf0, 0x8f, 0x31, 0x1, 0x7a, 0xc7, 0xca, 0x22, 0x39},
		optRand:    []byte{0x2d, 0x32, 0x27, 0xbb, 0xa8, 0xc5, 0xad, 0x18, 0xfc, 0x81, 0x8a, 0x3f, 0xab, 0x8c, 0x36, 0x3b, 0x47, 0x53, 0xce, 0xe1, 0xa3, 0x58, 0x8c, 0xe8},
		msg:        []byte{0x0, 0x10, 0xb2, 0x39, 0x59, 0x9c, 0xbc, 0xa2, 0xd, 0x4d, 0x26, 0x46, 0xb1, 0x20, 0x13, 0xa6, 0x53, 0xd3, 0xd4, 0xee, 0x59, 0xc7, 0x7a, 0x7d, 0x59, 0x79, 0x9, 0xd9, 0x52, 0x46, 0x7c, 0x9, 0xf9, 0x2f},
		wantHMsg:   []byte{0xf2, 0x2, 0xec, 0x32, 0xb2, 0x92, 0x81, 0x92, 0x75, 0x93, 0xd, 0x4d, 0xb8, 0xd0, 0x4, 0xec, 0x1a, 0x2e, 0xc0, 0x4e, 0x42, 0x2e, 0xdb, 0xab, 0x8a, 0x44, 0xa8, 0x42, 0x4a, 0x66, 0x2e, 0x4f, 0x4a, 0xda, 0xc7, 0x7c, 0x35, 0xda, 0xd9, 0x64, 0x50, 0x10},
		wantPrf:    []byte{0xe0, 0x54, 0x6f, 0x48, 0xa3, 0x1d, 0x3a, 0xcb, 0x6, 0x1b, 0xc4, 0xac, 0x8a, 0x6c, 0x92, 0x41, 0x10, 0x53, 0xf, 0xe6, 0x43, 0xd6, 0xab, 0x51},
		wantPrfMsg: []byte{0xd8, 0xc2, 0xea, 0xff, 0x63, 0x75, 0xcc, 0x31, 0x52, 0x75, 0x56, 0xdd, 0x54, 0xf7, 0x61, 0xf1, 0x96, 0x47, 0x44, 0xd, 0x98, 0xa2, 0x95, 0xd5},
		wantF:      []byte{0x7f, 0xe5, 0x99, 0x4e, 0x42, 0xbd, 0x2a, 0x2e, 0x94, 0x13, 0xe6, 0xcc, 0xd0, 0x3, 0xa9, 0xb1, 0x1d, 0x43, 0x74, 0x54, 0xd4, 0x35, 0x54, 0x88},
		wantH:      []byte{0x8f, 0xf0, 0x9b, 0x79, 0xa3, 0xc, 0x72, 0x7f, 0xb1, 0x87, 0x86, 0x17, 0x7c, 0xe7, 0xbd, 0xea, 0x1d, 0x54, 0xd3, 0xdd, 0x24, 0x32, 0x49, 0x2b},
		wantTl:     []byte{0x8f, 0xf0, 0x9b, 0x79, 0xa3, 0xc, 0x72, 0x7f, 0xb1, 0x87, 0x86, 0x17, 0x7c, 0xe7, 0xbd, 0xea, 0x1d, 0x54, 0xd3, 0xdd, 0x24, 0x32, 0x49, 0x2b},
	}
	hashTestVectorSha2C35n32m47 = hashTestVector{
		n:          32,
		m:          47,
		r:          []byte{0x5d, 0xd5, 0xc0, 0xb2, 0xf4, 0x5f, 0xb4, 0x78, 0xe3, 0xb7, 0x5b, 0x33, 0x82, 0x8e, 0x7f, 0xda, 0x20, 0xcb, 0x35, 0xbd, 0xa0, 0x4c, 0xfd, 0x13, 0xf8, 0x1e, 0x43, 0xe8, 0xed, 0xee, 0xc2, 0x25},
		pkSeed:     []byte{0xfe, 0x70, 0xc, 0x33, 0xd6, 0x30, 0x4d, 0x31, 0xaa, 0x7e, 0xe, 0x5b, 0xb8, 0xeb, 0xbe, 0xe9, 0x5, 0xf5, 0x42, 0x9d, 0x11, 0xe, 0x98, 0xf9, 0xf8, 0x1e, 0x43, 0xe8, 0xed, 0xee, 0xc2, 0x25},
		pkRoot:     []byte{0xbc, 0x3f, 0x28, 0x92, 0x43, 0xb2, 0x11, 0xb5, 0x3f, 0x15, 0xc3, 0xa, 0xc4, 0xf1, 0x34, 0x82, 0x43, 0x9e, 0x5b, 0x4b, 0xaa, 0x62, 0xf5, 0xd0, 0xf8, 0x1e, 0x43, 0xe8, 0xed, 0xee, 0xc2, 0x25},
		skSeed:     []byte{0x54, 0xfc, 0x58, 0x34, 0xc7, 0x9f, 0x20, 0x6a, 0x9d, 0x45, 0x83, 0xe0, 0x12, 0xdc, 0x2, 0x2c, 0x10, 0x47, 0xf4, 0x58, 0x50, 0xe1, 0xf7, 0xbf, 0xf8, 0x1e, 0x43, 0xe8, 0xed, 0xee, 0xc2, 0x25},
		skPrf:      []byte{0xf2, 0xc4, 0xf, 0xa7, 0xff, 0x9b, 0xd5, 0x6b, 0xd1, 0x28, 0x78, 0x89, 0xd7, 0xbc, 0x69, 0xf0, 0x8f, 0x31, 0x1, 0x7a, 0xc7, 0xca, 0x22, 0x39, 0xf8, 0x1e, 0x43, 0xe8, 0xed, 0xee, 0xc2, 0x25},
		optRand:    []byte{0x2d, 0x32, 0x27, 0xbb, 0xa8, 0xc5, 0xad, 0x18, 0xfc, 0x81, 0x8a, 0x3f, 0xab, 0x8c, 0x36, 0x3b, 0x47, 0x53, 0xce, 0xe1, 0xa3, 0x58, 0x8c, 0xe8, 0xf8, 0x1e, 0x43, 0xe8, 0xed, 0xee, 0xc2, 0x25},
		msg:        []byte{0x0, 0x10, 0xb2, 0x39, 0x59, 0x9c, 0xbc, 0xa2, 0xd, 0x4d, 0x26, 0x46, 0xb1, 0x20, 0x13, 0xa6, 0x53, 0xd3, 0xd4, 0xee, 0x59, 0xc7, 0x7a, 0x7d, 0x59, 0x79, 0x9, 0xd9, 0x52, 0x46, 0x7c, 0x9, 0xf9, 0x2f},
		wantHMsg:   []byte{0xbe, 0xdf, 0xe, 0xc0, 0xd4, 0xd0, 0x8b, 0xf4, 0xc, 0xcd, 0x26, 0x60, 0x5f, 0x51, 0x6a, 0xaa, 0x70, 0xd5, 0xc6, 0x11, 0x8b, 0x97, 0xb6, 0x23, 0x4e, 0x4a, 0xda, 0x91, 0x50, 0xce, 0x3, 0x4b, 0xec, 0xd7, 0x84, 0xdb, 0x29, 0x6f, 0x32, 0xed, 0x68, 0x83, 0x12, 0x7e, 0x9c, 0x44, 0xed},
		wantPrf:    []byte{0x8f, 0xdd, 0x27, 0xf2, 0xc3, 0x9d, 0xd4, 0xce, 0x2f, 0x22, 0x59, 0x23, 0xc1, 0x70, 0x8c, 0xa5, 0x2d, 0x84, 0x6f, 0xe8, 0x90, 0x9a, 0x83, 0x74, 0x4e, 0xe7, 0x5d, 0x28, 0x96, 0x24, 0x5a, 0xfa},
		wantPrfMsg: []byte{0xe4, 0xbe, 0x4, 0xf3, 0x82, 0x6a, 0x5a, 0xd, 0x3d, 0xaa, 0xc7, 0x19, 0x65, 0xa4, 0xc1, 0x65, 0x8f, 0xbb, 0x15, 0x54, 0xb7, 0x53, 0xd4, 0xc0, 0x15, 0x5b, 0x6d, 0x52, 0xe1, 0x84, 0x2a, 0xfe},
		wantF:      []byte{0x44, 0xc1, 0x8f, 0x63, 0x27, 0x78, 0x61, 0xff, 0x25, 0x8e, 0x6a, 0xa7, 0xa, 0x7b, 0x29, 0x4b, 0xe5, 0x87, 0x2a, 0x1, 0xac, 0x93, 0xe4, 0x27, 0xfc, 0xe0, 0x8c, 0x18, 0x9e, 0x88, 0xd8, 0xc8},
		wantH:      []byte{0x56, 0x31, 0x69, 0xc6, 0x88, 0x2d, 0x41, 0xbd, 0x6d, 0x87, 0xee, 0x1, 0xc0, 0xcb, 0x4f, 0xc3, 0x72, 0x7c, 0xcd, 0xe, 0x44, 0x54, 0xab, 0x43, 0x34, 0x90, 0x63, 0x26, 0xc5, 0xe, 0x14, 0x8d},
		wantTl:     []byte{0x56, 0x31, 0x69, 0xc6, 0x88, 0x2d, 0x41, 0xbd, 0x6d, 0x87, 0xee, 0x1, 0xc0, 0xcb, 0x4f, 0xc3, 0x72, 0x7c, 0xcd, 0xe, 0x44, 0x54, 0xab, 0x43, 0x34, 0x90, 0x63, 0x26, 0xc5, 0xe, 0x14, 0x8d},
	}
	hashTestVectorSha2C35n32m49 = hashTestVector{
		n:          32,
		m:          49,
		r:          []byte{0x5d, 0xd5, 0xc0, 0xb2, 0xf4, 0x5f, 0xb4, 0x78, 0xe3, 0xb7, 0x5b, 0x33, 0x82, 0x8e, 0x7f, 0xda, 0x20, 0xcb, 0x35, 0xbd, 0xa0, 0x4c, 0xfd, 0x13, 0xf8, 0x1e, 0x43, 0xe8, 0xed, 0xee, 0xc2, 0x25},
		pkSeed:     []byte{0xfe, 0x70, 0xc, 0x33, 0xd6, 0x30, 0x4d, 0x31, 0xaa, 0x7e, 0xe, 0x5b, 0xb8, 0xeb, 0xbe, 0xe9, 0x5, 0xf5, 0x42, 0x9d, 0x11, 0xe, 0x98, 0xf9, 0xf8, 0x1e, 0x43, 0xe8, 0xed, 0xee, 0xc2, 0x25},
		pkRoot:     []byte{0xbc, 0x3f, 0x28, 0x92, 0x43, 0xb2, 0x11, 0xb5, 0x3f, 0x15, 0xc3, 0xa, 0xc4, 0xf1, 0x34, 0x82, 0x43, 0x9e, 0x5b, 0x4b, 0xaa, 0x62, 0xf5, 0xd0, 0xf8, 0x1e, 0x43, 0xe8, 0xed, 0xee, 0xc2, 0x25},
		skSeed:     []byte{0x54, 0xfc, 0x58, 0x34, 0xc7, 0x9f, 0x20, 0x6a, 0x9d, 0x45, 0x83, 0xe0, 0x12, 0xdc, 0x2, 0x2c, 0x10, 0x47, 0xf4, 0x58, 0x50, 0xe1, 0xf7, 0xbf, 0xf8, 0x1e, 0x43, 0xe8, 0xed, 0xee, 0xc2, 0x25},
		skPrf:      []byte{0xf2, 0xc4, 0xf, 0xa7, 0xff, 0x9b, 0xd5, 0x6b, 0xd1, 0x28, 0x78, 0x89, 0xd7, 0xbc, 0x69, 0xf0, 0x8f, 0x31, 0x1, 0x7a, 0xc7, 0xca, 0x22, 0x39, 0xf8, 0x1e, 0x43, 0xe8, 0xed, 0xee, 0xc2, 0x25},
		optRand:    []byte{0x2d, 0x32, 0x27, 0xbb, 0xa8, 0xc5, 0xad, 0x18, 0xfc, 0x81, 0x8a, 0x3f, 0xab, 0x8c, 0x36, 0x3b, 0x47, 0x53, 0xce, 0xe1, 0xa3, 0x58, 0x8c, 0xe8, 0xf8, 0x1e, 0x43, 0xe8, 0xed, 0xee, 0xc2, 0x25},
		msg:        []byte{0x0, 0x10, 0xb2, 0x39, 0x59, 0x9c, 0xbc, 0xa2, 0xd, 0x4d, 0x26, 0x46, 0xb1, 0x20, 0x13, 0xa6, 0x53, 0xd3, 0xd4, 0xee, 0x59, 0xc7, 0x7a, 0x7d, 0x59, 0x79, 0x9, 0xd9, 0x52, 0x46, 0x7c, 0x9, 0xf9, 0x2f},
		wantHMsg:   []byte{0xbe, 0xdf, 0xe, 0xc0, 0xd4, 0xd0, 0x8b, 0xf4, 0xc, 0xcd, 0x26, 0x60, 0x5f, 0x51, 0x6a, 0xaa, 0x70, 0xd5, 0xc6, 0x11, 0x8b, 0x97, 0xb6, 0x23, 0x4e, 0x4a, 0xda, 0x91, 0x50, 0xce, 0x3, 0x4b, 0xec, 0xd7, 0x84, 0xdb, 0x29, 0x6f, 0x32, 0xed, 0x68, 0x83, 0x12, 0x7e, 0x9c, 0x44, 0xed, 0xaf, 0x9c},
		wantPrf:    []byte{0x8f, 0xdd, 0x27, 0xf2, 0xc3, 0x9d, 0xd4, 0xce, 0x2f, 0x22, 0x59, 0x23, 0xc1, 0x70, 0x8c, 0xa5, 0x2d, 0x84, 0x6f, 0xe8, 0x90, 0x9a, 0x83, 0x74, 0x4e, 0xe7, 0x5d, 0x28, 0x96, 0x24, 0x5a, 0xfa},
		wantPrfMsg: []byte{0xe4, 0xbe, 0x4, 0xf3, 0x82, 0x6a, 0x5a, 0xd, 0x3d, 0xaa, 0xc7, 0x19, 0x65, 0xa4, 0xc1, 0x65, 0x8f, 0xbb, 0x15, 0x54, 0xb7, 0x53, 0xd4, 0xc0, 0x15, 0x5b, 0x6d, 0x52, 0xe1, 0x84, 0x2a, 0xfe},
		wantF:      []byte{0x44, 0xc1, 0x8f, 0x63, 0x27, 0x78, 0x61, 0xff, 0x25, 0x8e, 0x6a, 0xa7, 0xa, 0x7b, 0x29, 0x4b, 0xe5, 0x87, 0x2a, 0x1, 0xac, 0x93, 0xe4, 0x27, 0xfc, 0xe0, 0x8c, 0x18, 0x9e, 0x88, 0xd8, 0xc8},
		wantH:      []byte{0x56, 0x31, 0x69, 0xc6, 0x88, 0x2d, 0x41, 0xbd, 0x6d, 0x87, 0xee, 0x1, 0xc0, 0xcb, 0x4f, 0xc3, 0x72, 0x7c, 0xcd, 0xe, 0x44, 0x54, 0xab, 0x43, 0x34, 0x90, 0x63, 0x26, 0xc5, 0xe, 0x14, 0x8d},
		wantTl:     []byte{0x56, 0x31, 0x69, 0xc6, 0x88, 0x2d, 0x41, 0xbd, 0x6d, 0x87, 0xee, 0x1, 0xc0, 0xcb, 0x4f, 0xc3, 0x72, 0x7c, 0xcd, 0xe, 0x44, 0x54, 0xab, 0x43, 0x34, 0x90, 0x63, 0x26, 0xc5, 0xe, 0x14, 0x8d},
	}
)

func TestHashKat(t *testing.T) {
	for _, tc := range []struct {
		name   string
		vec    hashTestVector
		hMsg   func(r []byte, pkSeed []byte, pkRoot []byte, msg []byte, m uint32) []byte
		prf    func(pkSeed []byte, skSeed []byte, adrs *address, n uint32) []byte
		prfMsg func(skPrf []byte, optRand []byte, msg []byte, n uint32) []byte
		f      func(pkSeed []byte, adrs *address, msg1 []byte, n uint32) []byte
		h      func(pkSeed []byte, adrs *address, msg2 []byte, n uint32) []byte
		tl     func(pkSeed []byte, adrs *address, msgl []byte, n uint32) []byte
	}{
		{
			name:   "Shake-16-30",
			vec:    hashTestVectorShaken16m30,
			hMsg:   shakeHMsg,
			prf:    shakePrf,
			prfMsg: shakePrfMsg,
			f:      shakeF,
			h:      shakeH,
			tl:     shakeTl,
		},
		{
			name:   "Shake-16-34",
			vec:    hashTestVectorShaken16m34,
			hMsg:   shakeHMsg,
			prf:    shakePrf,
			prfMsg: shakePrfMsg,
			f:      shakeF,
			h:      shakeH,
			tl:     shakeTl,
		},
		{
			name:   "Shake-24-39",
			vec:    hashTestVectorShaken24m39,
			hMsg:   shakeHMsg,
			prf:    shakePrf,
			prfMsg: shakePrfMsg,
			f:      shakeF,
			h:      shakeH,
			tl:     shakeTl,
		},
		{
			name:   "Shake-24-42",
			vec:    hashTestVectorShaken24m42,
			hMsg:   shakeHMsg,
			prf:    shakePrf,
			prfMsg: shakePrfMsg,
			f:      shakeF,
			h:      shakeH,
			tl:     shakeTl,
		},
		{
			name:   "Shake-32-47",
			vec:    hashTestVectorShaken32m47,
			hMsg:   shakeHMsg,
			prf:    shakePrf,
			prfMsg: shakePrfMsg,
			f:      shakeF,
			h:      shakeH,
			tl:     shakeTl,
		},
		{
			name:   "Shake-32-49",
			vec:    hashTestVectorShaken32m49,
			hMsg:   shakeHMsg,
			prf:    shakePrf,
			prfMsg: shakePrfMsg,
			f:      shakeF,
			h:      shakeH,
			tl:     shakeTl,
		},
		{
			name:   "Sha2C1-16-30",
			vec:    hashTestVectorSha2C1n16m30,
			hMsg:   sha2C1HMsg,
			prf:    sha2C1Prf,
			prfMsg: sha2C1PrfMsg,
			f:      sha2C1F,
			h:      sha2C1H,
			tl:     sha2C1Tl,
		},
		{
			name:   "Sha2C1-16-34",
			vec:    hashTestVectorSha2C1n16m34,
			hMsg:   sha2C1HMsg,
			prf:    sha2C1Prf,
			prfMsg: sha2C1PrfMsg,
			f:      sha2C1F,
			h:      sha2C1H,
			tl:     sha2C1Tl,
		},
		{
			name:   "Sha2C35-24-39",
			vec:    hashTestVectorSha2C35n24m39,
			hMsg:   sha2C35HMsg,
			prf:    sha2C35Prf,
			prfMsg: sha2C35PrfMsg,
			f:      sha2C35F,
			h:      sha2C35H,
			tl:     sha2C35Tl,
		},
		{
			name:   "Sha2C35-24-42",
			vec:    hashTestVectorSha2C35n24m42,
			hMsg:   sha2C35HMsg,
			prf:    sha2C35Prf,
			prfMsg: sha2C35PrfMsg,
			f:      sha2C35F,
			h:      sha2C35H,
			tl:     sha2C35Tl,
		},
		{
			name:   "Sha2C35-32-47",
			vec:    hashTestVectorSha2C35n32m47,
			hMsg:   sha2C35HMsg,
			prf:    sha2C35Prf,
			prfMsg: sha2C35PrfMsg,
			f:      sha2C35F,
			h:      sha2C35H,
			tl:     sha2C35Tl,
		},
		{
			name:   "Sha2C35-32-49",
			vec:    hashTestVectorSha2C35n32m49,
			hMsg:   sha2C35HMsg,
			prf:    sha2C35Prf,
			prfMsg: sha2C35PrfMsg,
			f:      sha2C35F,
			h:      sha2C35H,
			tl:     sha2C35Tl,
		},
	} {
		t.Run(tc.name, func(t *testing.T) {
			msg := tc.hMsg(tc.vec.r, tc.vec.pkSeed, tc.vec.pkRoot, tc.vec.msg, tc.vec.m)
			if !slices.Equal(msg, tc.vec.wantHMsg) {
				t.Errorf("tc.hMsg() = %v, want %v", msg, tc.vec.wantHMsg)
			}
			prf := tc.prf(tc.vec.pkSeed, tc.vec.skSeed, newAddress(), tc.vec.n)
			if !slices.Equal(prf, tc.vec.wantPrf) {
				t.Errorf("tc.prf() = %v, want %v", msg, tc.vec.wantPrf)
			}
			prfMsg := tc.prfMsg(tc.vec.skPrf, tc.vec.optRand, tc.vec.msg, tc.vec.n)
			if !slices.Equal(prfMsg, tc.vec.wantPrfMsg) {
				t.Errorf("tc.prfMsg() = %v, want %v", msg, tc.vec.wantPrfMsg)
			}
			f := tc.f(tc.vec.pkSeed, newAddress(), tc.vec.msg, tc.vec.n)
			if !slices.Equal(f, tc.vec.wantF) {
				t.Errorf("tc.f() = %v, want %v", msg, tc.vec.wantF)
			}
			h := tc.h(tc.vec.pkSeed, newAddress(), tc.vec.msg, tc.vec.n)
			if !slices.Equal(h, tc.vec.wantH) {
				t.Errorf("tc.h() = %v, want %v", msg, tc.vec.wantH)
			}
			tl := tc.tl(tc.vec.pkSeed, newAddress(), tc.vec.msg, tc.vec.n)
			if !slices.Equal(tl, tc.vec.wantTl) {
				t.Errorf("tc.tl() = %v, want %v", msg, tc.vec.wantTl)
			}
		})
	}
}
