/*  RipOff - Plugin based CD Ripper
 *  Copyright (C) 2006 Bobby Ryan Newberry
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public Licensse as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <libxml/xmlmemory.h>
#include "RipOffConfigFile.h"

static xmlDocPtr config_doc;

void open_config_doc();

gboolean ripoff_config_file_save(gchar *module_name, gchar *key, const gchar *value, gchar *type);
gchar *ripoff_config_file_retrieve_value(gchar *module_name, gchar *key, gchar *type);

void ripoff_config_file_set_config_file_pointer(xmlDocPtr ptr)
{
	config_doc = ptr;
}

xmlDocPtr ripoff_config_file_get_config_file_pointer()
{
	if(config_doc == NULL)
	{
		open_config_doc();
	}

	return config_doc;
}

gboolean ripoff_config_file_save_int(gchar *module_name, gchar *key, const gint value)
{
	xmlChar *string_value = g_malloc(255);

	g_snprintf(string_value, 255, "%i", value);

	return ripoff_config_file_save(module_name, key, string_value, "int");
}

gboolean ripoff_config_file_save_string(gchar *module_name, gchar *key, const gchar *value)
{
	return ripoff_config_file_save(module_name, key, value, (gchar *) "string");
}

gboolean ripoff_config_file_save_boolean(gchar *module_name, gchar *key, const gboolean value)
{
	if(value)
		return ripoff_config_file_save(module_name, key, "1", "boolean");
	else
		return ripoff_config_file_save(module_name, key, "0", "boolean");
}

gboolean ripoff_config_file_save_double(gchar *module_name, gchar *key, const gdouble value)
{
	xmlChar *string_value = g_malloc(255);

	g_snprintf(string_value, 255, "%f", value);

	return ripoff_config_file_save(module_name, key, string_value, "double");
}

gboolean ripoff_config_file_save(gchar *module_name, gchar *key, const gchar *value, gchar *type)
{
	xmlNodePtr cur;
	xmlNodePtr moduleptr;
	gchar *name;
	gchar *node_key;
	gchar *node_type;

	/* Check to see if the doc ptr has already been set if not we go attempt to open it
	   If doc is still */
	if(config_doc == NULL)
	{
		open_config_doc();
		
		if(config_doc == NULL)
			return FALSE;
	}
	
	
	cur = xmlDocGetRootElement(config_doc);
	cur = cur->xmlChildrenNode;
	while (cur != NULL) 
	{
		name = xmlGetProp(cur, "name");
		if (!(xmlStrcmp(name, BAD_CAST module_name)))
		{
			xmlFree(name);
			moduleptr = cur;
			cur = cur->xmlChildrenNode;
			while(cur != NULL)
			{
				node_key = xmlGetProp(cur, "key");
				node_type = xmlGetProp(cur, "type");
				if (!(xmlStrcmp(node_key, BAD_CAST key)) && !(xmlStrcmp(node_type, BAD_CAST type)))
				{
					xmlSetProp(cur, BAD_CAST "value", BAD_CAST value);
					return TRUE;
				}
				xmlFree(node_key);
				xmlFree(node_type);
				cur = cur->next;
			}

			cur = xmlNewChild(moduleptr, NULL, BAD_CAST "pref", NULL);
			xmlSetProp(cur, BAD_CAST "key", BAD_CAST key);
			xmlSetProp(cur, BAD_CAST "value", BAD_CAST value);
			xmlSetProp(cur, BAD_CAST "type", BAD_CAST type);
			return TRUE;
		}

		cur = cur->next;
	}

	/* case for which the specified module has no config file entry. We create
	   it here */

	cur = xmlNewChild(xmlDocGetRootElement(config_doc), NULL, BAD_CAST "module", NULL);
	xmlSetProp(cur, BAD_CAST "name", BAD_CAST module_name);
	cur = xmlNewChild(cur, NULL, BAD_CAST "pref", NULL);
	xmlSetProp(cur, BAD_CAST "key", BAD_CAST key);
	xmlSetProp(cur, BAD_CAST "value", BAD_CAST value);
	xmlSetProp(cur, BAD_CAST "type", BAD_CAST type);

	return TRUE;
}

gint ripoff_config_file_retrieve_int(gchar *module_name, gchar *key, gint default_val)
{
	int integer_value;
	gchar *value = ripoff_config_file_retrieve_value(module_name, key, "int");

	#ifdef DEBUG
	g_printf("String value retrieved %s\n", value);
	#endif

	if(value == NULL)
		return default_val;
	else
	{
		integer_value = atoi(value);
		xmlFree(value);
		return integer_value;
	}
}

gchar *ripoff_config_file_retrieve_string(gchar *module_name, gchar *key, const gchar *default_val)
{
	gchar *value = ripoff_config_file_retrieve_value(module_name, key, "string");

	if(value == NULL)
	{
		if(default_val != NULL)
		{
			value = g_malloc(strlen(default_val)+1);
			strncpy(value, default_val, strlen(default_val)+1);
		}
		else
			return NULL;
	}

	return value;
}


gboolean ripoff_config_file_retrieve_boolean(gchar *module_name, gchar *key, gboolean default_val)
{
	gchar *value = ripoff_config_file_retrieve_value(module_name, key, "boolean");

	if(value == NULL)
		return default_val;
	else
	{
		if(value[0] == '1')
		{
			xmlFree(value);
			return TRUE;
		}
		else
		{
			xmlFree(value);
			return FALSE;
		}
	}
}
		

gdouble ripoff_config_file_retrieve_double(gchar *module_name, gchar *key, gdouble default_val )
{
	gchar *value = ripoff_config_file_retrieve_value(module_name, key, "double");
	double double_value;

	if(value == NULL)
		return default_val;
	else
	{
		double_value = atof(value);
		xmlFree(value);
		return double_value;
	}
}

gchar *ripoff_config_file_retrieve_value(gchar *module_name, gchar *key, gchar *type)
{
	xmlNodePtr cur;
	xmlNodePtr moduleptr;
	gchar *name;
	gchar *node_key;
	gchar *node_type;

	/* Check to see if the doc ptr has already been set if not we go attempt to open it
	   If doc is still */
	if(config_doc == NULL)
	{
		open_config_doc();
		
		if(config_doc == NULL)
			return FALSE;
	}
	
	
	cur = xmlDocGetRootElement(config_doc);
	cur = cur->xmlChildrenNode;
	while (cur != NULL) 
	{
		name = xmlGetProp(cur, "name");
		if (!(xmlStrcmp(name, BAD_CAST module_name)))
		{
			xmlFree(name);
			moduleptr = cur;
			cur = cur->xmlChildrenNode;
			while(cur != NULL)
			{
				node_key = xmlGetProp(cur, "key");
				node_type = xmlGetProp(cur, "type");
				if (!(xmlStrcmp(node_key, BAD_CAST key)) && !(xmlStrcmp(node_type, BAD_CAST type)))
				{
					return xmlGetProp(cur, BAD_CAST "value");
				}

				xmlFree(node_key);
				xmlFree(node_type);
				cur = cur->next;
			}

			return NULL;
		}

		cur = cur->next;
	}

	return NULL;
}

void open_config_doc()
{
	xmlNodePtr root_node;
	gchar *docname;

	if(getenv("HOME") == NULL)
		return;

	docname = g_strconcat(getenv("HOME"), "/.ripoff/prefs.xml", NULL);
	config_doc = xmlParseFile(docname);

	if(config_doc == NULL)
	{
		config_doc = xmlNewDoc(BAD_CAST "1.0");
		root_node = xmlNewNode(NULL, BAD_CAST "RipOffPreferences");
		xmlDocSetRootElement(config_doc, root_node);
		g_free(docname);
		return;
	}

	root_node = xmlDocGetRootElement(config_doc);
	if(root_node == NULL)
	{
		root_node = xmlNewNode(NULL, BAD_CAST "RipOffPreferences");
		xmlDocSetRootElement(config_doc, root_node);
		g_free(docname);
		return;
	}
	else
	{
		if (xmlStrcmp(root_node->name, BAD_CAST "RipOffPreferences"))
		{
			xmlFreeDoc(config_doc);
			return;
		}
	}

	g_free(docname);
}

gboolean ripoff_config_file_write_config_file_to_disc()
{
	gchar *home_dir = getenv("HOME");
	gchar *docname;

	if(home_dir == NULL)
		return FALSE;

	else
	{
		docname = g_strconcat(getenv("HOME"), "/.ripoff/prefs.xml", NULL);

		if(xmlSaveFileEnc(docname, config_doc, "UTF-8") == -1)
		{
			g_free(docname);
			return FALSE;
		}
	
		else
		{
			g_free(docname);
			return TRUE;
		}
	}
}
