#! /usr/bin/env python

'''
Python Interface to SAGA Tools Library
----------
- Category : Imagery
- Name     : Tools
- ID       : imagery_tools

Description
----------
Image processing and analysis tools.
'''

from PySAGA.helper import Tool_Wrapper

def Vegetation_Index_Distance_Based(
    RED=None,
    NIR=None,
    PVI0=None,
    PVI1=None,
    PVI2=None,
    PVI3=None,
    TSAVI=None,
    ATSAVI=None,
    INTERCEPT=None,
    SLOPE=None, Verbose=2):
    '''
    Vegetation Index (Distance Based)
    ----------
    [imagery_tools.0]\n
    Distance based vegetation indices.\n
    Arguments
    ----------
    - RED [`input grid`] : Red Reflectance
    - NIR [`input grid`] : Near Infrared Reflectance
    - PVI0 [`output grid`] : Perpendicular Vegetation Index (Richardson and Wiegand, 1977)
    - PVI1 [`output grid`] : Perpendicular Vegetation Index (Perry and Lautenschlager, 1984)
    - PVI2 [`output grid`] : Perpendicular Vegetation Index (Walther and Shabaani)
    - PVI3 [`output grid`] : Perpendicular Vegetation Index (Qi, et al., 1994)
    - TSAVI [`output grid`] : Transformed Soil Adjusted Vegetation Index (Baret et al. 1989)
    - ATSAVI [`output grid`] : Transformed Soil Adjusted Vegetation Index (Baret and Guyot, 1991)
    - INTERCEPT [`floating point number`] : Intercept of Soil Line. Default: 0.000000
    - SLOPE [`floating point number`] : Slope of Soil Line. Default: 0.500000

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '0', 'Vegetation Index (Distance Based)')
    if Tool.is_Okay():
        Tool.Set_Input ('RED', RED)
        Tool.Set_Input ('NIR', NIR)
        Tool.Set_Output('PVI0', PVI0)
        Tool.Set_Output('PVI1', PVI1)
        Tool.Set_Output('PVI2', PVI2)
        Tool.Set_Output('PVI3', PVI3)
        Tool.Set_Output('TSAVI', TSAVI)
        Tool.Set_Output('ATSAVI', ATSAVI)
        Tool.Set_Option('INTERCEPT', INTERCEPT)
        Tool.Set_Option('SLOPE', SLOPE)
        return Tool.Execute(Verbose)
    return False

def run_tool_imagery_tools_0(
    RED=None,
    NIR=None,
    PVI0=None,
    PVI1=None,
    PVI2=None,
    PVI3=None,
    TSAVI=None,
    ATSAVI=None,
    INTERCEPT=None,
    SLOPE=None, Verbose=2):
    '''
    Vegetation Index (Distance Based)
    ----------
    [imagery_tools.0]\n
    Distance based vegetation indices.\n
    Arguments
    ----------
    - RED [`input grid`] : Red Reflectance
    - NIR [`input grid`] : Near Infrared Reflectance
    - PVI0 [`output grid`] : Perpendicular Vegetation Index (Richardson and Wiegand, 1977)
    - PVI1 [`output grid`] : Perpendicular Vegetation Index (Perry and Lautenschlager, 1984)
    - PVI2 [`output grid`] : Perpendicular Vegetation Index (Walther and Shabaani)
    - PVI3 [`output grid`] : Perpendicular Vegetation Index (Qi, et al., 1994)
    - TSAVI [`output grid`] : Transformed Soil Adjusted Vegetation Index (Baret et al. 1989)
    - ATSAVI [`output grid`] : Transformed Soil Adjusted Vegetation Index (Baret and Guyot, 1991)
    - INTERCEPT [`floating point number`] : Intercept of Soil Line. Default: 0.000000
    - SLOPE [`floating point number`] : Slope of Soil Line. Default: 0.500000

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '0', 'Vegetation Index (Distance Based)')
    if Tool.is_Okay():
        Tool.Set_Input ('RED', RED)
        Tool.Set_Input ('NIR', NIR)
        Tool.Set_Output('PVI0', PVI0)
        Tool.Set_Output('PVI1', PVI1)
        Tool.Set_Output('PVI2', PVI2)
        Tool.Set_Output('PVI3', PVI3)
        Tool.Set_Output('TSAVI', TSAVI)
        Tool.Set_Output('ATSAVI', ATSAVI)
        Tool.Set_Option('INTERCEPT', INTERCEPT)
        Tool.Set_Option('SLOPE', SLOPE)
        return Tool.Execute(Verbose)
    return False

def Vegetation_Index_Slope_Based(
    RED=None,
    NIR=None,
    DVI=None,
    NDVI=None,
    RVI=None,
    NRVI=None,
    TVI=None,
    CTVI=None,
    TTVI=None,
    SAVI=None,
    SOIL=None, Verbose=2):
    '''
    Vegetation Index (Slope Based)
    ----------
    [imagery_tools.1]\n
    Slope based vegetation indices.\n
    (-) Difference Vegetation Index\n
    DVI = NIR - R\n
    (-) Normalized Difference Vegetation Index (Rouse et al. 1974)\n
    NDVI = (NIR - R) / (NIR + R)\n
    (-) Ratio Vegetation Index (Richardson and Wiegand, 1977)\n
    RVI = R / NIR\n
    (-) Normalized Ratio Vegetation Index (Baret and Guyot, 1991)\n
    NRVI = (RVI - 1) / (RVI + 1)\n
    (-) Transformed Vegetation Index (Deering et al., 1975)\n
    TVI = [(NIR - R) / (NIR + R) + 0.5]^0.5\n
    (-) Corrected Transformed Ratio Vegetation Index (Perry and Lautenschlager, 1984)\n
    CTVI = [(NDVI + 0.5) / abs(NDVI + 0.5)] * [abs(NDVI + 0.5)]^0.5\n
    (-) Thiam's Transformed Vegetation Index (Thiam, 1997)\n
    RVI = [abs(NDVI) + 0.5]^0.5\n
    (-) Soil Adjusted Vegetation Index (Huete, 1988)\n
    SAVI = [(NIR - R) / (NIR + R + L)] * (1 + L)\n
    (NIR = near infrared, R = red, S = soil adjustment factor)\n
    Arguments
    ----------
    - RED [`input grid`] : Red Reflectance
    - NIR [`input grid`] : Near Infrared Reflectance
    - DVI [`output grid`] : Difference Vegetation Index
    - NDVI [`output grid`] : Normalized Difference Vegetation Index
    - RVI [`output grid`] : Ratio Vegetation Index
    - NRVI [`output grid`] : Normalized Ratio Vegetation Index
    - TVI [`output grid`] : Transformed Vegetation Index
    - CTVI [`output grid`] : Corrected Transformed Vegetation Index
    - TTVI [`output grid`] : Thiam's Transformed Vegetation Index
    - SAVI [`output grid`] : Soil Adjusted Vegetation Index
    - SOIL [`floating point number`] : Soil Adjustment Factor. Minimum: 0.000000 Maximum: 1.000000 Default: 0.500000 For SAVI, suggested values (after Silleos et al. 2006): 1.0 = very low, 0.5 = intermediate, 0.25 = high density vegetation.

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '1', 'Vegetation Index (Slope Based)')
    if Tool.is_Okay():
        Tool.Set_Input ('RED', RED)
        Tool.Set_Input ('NIR', NIR)
        Tool.Set_Output('DVI', DVI)
        Tool.Set_Output('NDVI', NDVI)
        Tool.Set_Output('RVI', RVI)
        Tool.Set_Output('NRVI', NRVI)
        Tool.Set_Output('TVI', TVI)
        Tool.Set_Output('CTVI', CTVI)
        Tool.Set_Output('TTVI', TTVI)
        Tool.Set_Output('SAVI', SAVI)
        Tool.Set_Option('SOIL', SOIL)
        return Tool.Execute(Verbose)
    return False

def run_tool_imagery_tools_1(
    RED=None,
    NIR=None,
    DVI=None,
    NDVI=None,
    RVI=None,
    NRVI=None,
    TVI=None,
    CTVI=None,
    TTVI=None,
    SAVI=None,
    SOIL=None, Verbose=2):
    '''
    Vegetation Index (Slope Based)
    ----------
    [imagery_tools.1]\n
    Slope based vegetation indices.\n
    (-) Difference Vegetation Index\n
    DVI = NIR - R\n
    (-) Normalized Difference Vegetation Index (Rouse et al. 1974)\n
    NDVI = (NIR - R) / (NIR + R)\n
    (-) Ratio Vegetation Index (Richardson and Wiegand, 1977)\n
    RVI = R / NIR\n
    (-) Normalized Ratio Vegetation Index (Baret and Guyot, 1991)\n
    NRVI = (RVI - 1) / (RVI + 1)\n
    (-) Transformed Vegetation Index (Deering et al., 1975)\n
    TVI = [(NIR - R) / (NIR + R) + 0.5]^0.5\n
    (-) Corrected Transformed Ratio Vegetation Index (Perry and Lautenschlager, 1984)\n
    CTVI = [(NDVI + 0.5) / abs(NDVI + 0.5)] * [abs(NDVI + 0.5)]^0.5\n
    (-) Thiam's Transformed Vegetation Index (Thiam, 1997)\n
    RVI = [abs(NDVI) + 0.5]^0.5\n
    (-) Soil Adjusted Vegetation Index (Huete, 1988)\n
    SAVI = [(NIR - R) / (NIR + R + L)] * (1 + L)\n
    (NIR = near infrared, R = red, S = soil adjustment factor)\n
    Arguments
    ----------
    - RED [`input grid`] : Red Reflectance
    - NIR [`input grid`] : Near Infrared Reflectance
    - DVI [`output grid`] : Difference Vegetation Index
    - NDVI [`output grid`] : Normalized Difference Vegetation Index
    - RVI [`output grid`] : Ratio Vegetation Index
    - NRVI [`output grid`] : Normalized Ratio Vegetation Index
    - TVI [`output grid`] : Transformed Vegetation Index
    - CTVI [`output grid`] : Corrected Transformed Vegetation Index
    - TTVI [`output grid`] : Thiam's Transformed Vegetation Index
    - SAVI [`output grid`] : Soil Adjusted Vegetation Index
    - SOIL [`floating point number`] : Soil Adjustment Factor. Minimum: 0.000000 Maximum: 1.000000 Default: 0.500000 For SAVI, suggested values (after Silleos et al. 2006): 1.0 = very low, 0.5 = intermediate, 0.25 = high density vegetation.

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '1', 'Vegetation Index (Slope Based)')
    if Tool.is_Okay():
        Tool.Set_Input ('RED', RED)
        Tool.Set_Input ('NIR', NIR)
        Tool.Set_Output('DVI', DVI)
        Tool.Set_Output('NDVI', NDVI)
        Tool.Set_Output('RVI', RVI)
        Tool.Set_Output('NRVI', NRVI)
        Tool.Set_Output('TVI', TVI)
        Tool.Set_Output('CTVI', CTVI)
        Tool.Set_Output('TTVI', TTVI)
        Tool.Set_Output('SAVI', SAVI)
        Tool.Set_Option('SOIL', SOIL)
        return Tool.Execute(Verbose)
    return False

def Enhanced_Vegetation_Index(
    RED=None,
    NIR=None,
    BLUE=None,
    EVI=None,
    GAIN=None,
    L=None,
    CBLUE=None,
    CRED=None, Verbose=2):
    '''
    Enhanced Vegetation Index
    ----------
    [imagery_tools.2]\n
    Enhanced Vegetation Index (EVI).\n
    Arguments
    ----------
    - RED [`input grid`] : Red Reflectance
    - NIR [`input grid`] : Near Infrared Reflectance
    - BLUE [`optional input grid`] : Blue Reflectance
    - EVI [`output grid`] : Enhanced Vegetation Index
    - GAIN [`floating point number`] : Gain. Minimum: 0.000000 Default: 2.500000
    - L [`floating point number`] : Canopy Background Adjustment. Minimum: 0.000000 Default: 1.000000
    - CBLUE [`floating point number`] : Aerosol Resistance Coefficient (Blue). Minimum: 0.000000 Default: 7.500000
    - CRED [`floating point number`] : Aerosol Resistance Coefficient (Red). Minimum: 0.000000 Default: 6.000000

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '2', 'Enhanced Vegetation Index')
    if Tool.is_Okay():
        Tool.Set_Input ('RED', RED)
        Tool.Set_Input ('NIR', NIR)
        Tool.Set_Input ('BLUE', BLUE)
        Tool.Set_Output('EVI', EVI)
        Tool.Set_Option('GAIN', GAIN)
        Tool.Set_Option('L', L)
        Tool.Set_Option('CBLUE', CBLUE)
        Tool.Set_Option('CRED', CRED)
        return Tool.Execute(Verbose)
    return False

def run_tool_imagery_tools_2(
    RED=None,
    NIR=None,
    BLUE=None,
    EVI=None,
    GAIN=None,
    L=None,
    CBLUE=None,
    CRED=None, Verbose=2):
    '''
    Enhanced Vegetation Index
    ----------
    [imagery_tools.2]\n
    Enhanced Vegetation Index (EVI).\n
    Arguments
    ----------
    - RED [`input grid`] : Red Reflectance
    - NIR [`input grid`] : Near Infrared Reflectance
    - BLUE [`optional input grid`] : Blue Reflectance
    - EVI [`output grid`] : Enhanced Vegetation Index
    - GAIN [`floating point number`] : Gain. Minimum: 0.000000 Default: 2.500000
    - L [`floating point number`] : Canopy Background Adjustment. Minimum: 0.000000 Default: 1.000000
    - CBLUE [`floating point number`] : Aerosol Resistance Coefficient (Blue). Minimum: 0.000000 Default: 7.500000
    - CRED [`floating point number`] : Aerosol Resistance Coefficient (Red). Minimum: 0.000000 Default: 6.000000

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '2', 'Enhanced Vegetation Index')
    if Tool.is_Okay():
        Tool.Set_Input ('RED', RED)
        Tool.Set_Input ('NIR', NIR)
        Tool.Set_Input ('BLUE', BLUE)
        Tool.Set_Output('EVI', EVI)
        Tool.Set_Option('GAIN', GAIN)
        Tool.Set_Option('L', L)
        Tool.Set_Option('CBLUE', CBLUE)
        Tool.Set_Option('CRED', CRED)
        return Tool.Execute(Verbose)
    return False

def Tasseled_Cap_Transformation(
    BLUE=None,
    GREEN=None,
    RED=None,
    NIR=None,
    MIR1=None,
    MIR2=None,
    BRIGHTNESS=None,
    GREENNESS=None,
    WETNESS=None, Verbose=2):
    '''
    Tasseled Cap Transformation
    ----------
    [imagery_tools.3]\n
    Tasseled Cap Transformation as proposed for Landsat Thematic Mapper.\n
    Arguments
    ----------
    - BLUE [`input grid`] : Blue (TM 1)
    - GREEN [`input grid`] : Green (TM 2)
    - RED [`input grid`] : Red (TM 3)
    - NIR [`input grid`] : Near Infrared (TM 4)
    - MIR1 [`input grid`] : Mid Infrared (TM 5)
    - MIR2 [`input grid`] : Mid Infrared (TM 7)
    - BRIGHTNESS [`output grid`] : Brightness
    - GREENNESS [`output grid`] : Greenness
    - WETNESS [`output grid`] : Wetness

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '3', 'Tasseled Cap Transformation')
    if Tool.is_Okay():
        Tool.Set_Input ('BLUE', BLUE)
        Tool.Set_Input ('GREEN', GREEN)
        Tool.Set_Input ('RED', RED)
        Tool.Set_Input ('NIR', NIR)
        Tool.Set_Input ('MIR1', MIR1)
        Tool.Set_Input ('MIR2', MIR2)
        Tool.Set_Output('BRIGHTNESS', BRIGHTNESS)
        Tool.Set_Output('GREENNESS', GREENNESS)
        Tool.Set_Output('WETNESS', WETNESS)
        return Tool.Execute(Verbose)
    return False

def run_tool_imagery_tools_3(
    BLUE=None,
    GREEN=None,
    RED=None,
    NIR=None,
    MIR1=None,
    MIR2=None,
    BRIGHTNESS=None,
    GREENNESS=None,
    WETNESS=None, Verbose=2):
    '''
    Tasseled Cap Transformation
    ----------
    [imagery_tools.3]\n
    Tasseled Cap Transformation as proposed for Landsat Thematic Mapper.\n
    Arguments
    ----------
    - BLUE [`input grid`] : Blue (TM 1)
    - GREEN [`input grid`] : Green (TM 2)
    - RED [`input grid`] : Red (TM 3)
    - NIR [`input grid`] : Near Infrared (TM 4)
    - MIR1 [`input grid`] : Mid Infrared (TM 5)
    - MIR2 [`input grid`] : Mid Infrared (TM 7)
    - BRIGHTNESS [`output grid`] : Brightness
    - GREENNESS [`output grid`] : Greenness
    - WETNESS [`output grid`] : Wetness

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '3', 'Tasseled Cap Transformation')
    if Tool.is_Okay():
        Tool.Set_Input ('BLUE', BLUE)
        Tool.Set_Input ('GREEN', GREEN)
        Tool.Set_Input ('RED', RED)
        Tool.Set_Input ('NIR', NIR)
        Tool.Set_Input ('MIR1', MIR1)
        Tool.Set_Input ('MIR2', MIR2)
        Tool.Set_Output('BRIGHTNESS', BRIGHTNESS)
        Tool.Set_Output('GREENNESS', GREENNESS)
        Tool.Set_Output('WETNESS', WETNESS)
        return Tool.Execute(Verbose)
    return False

def IHS_Sharpening(
    R=None,
    G=None,
    B=None,
    PAN=None,
    R_SHARP=None,
    G_SHARP=None,
    B_SHARP=None,
    SHARP=None,
    PAN_SYSTEM=None,
    OUTPUT=None,
    RESAMPLING=None,
    PAN_MATCH=None, Verbose=2):
    '''
    IHS Sharpening
    ----------
    [imagery_tools.4]\n
    Intensity, hue, saturation (IHS) sharpening.\n
    Arguments
    ----------
    - R [`input grid`] : Red
    - G [`input grid`] : Green
    - B [`input grid`] : Blue
    - PAN [`input grid`] : Panchromatic Channel
    - R_SHARP [`output grid`] : Red
    - G_SHARP [`output grid`] : Green
    - B_SHARP [`output grid`] : Blue
    - SHARP [`output grid collection`] : Sharpened
    - PAN_SYSTEM [`grid system`] : High Resolution Grid System
    - OUTPUT [`choice`] : Output. Available Choices: [0] single grids [1] grid collection Default: 1
    - RESAMPLING [`choice`] : Resampling. Available Choices: [0] nearest neighbour [1] bilinear [2] cubic convolution Default: 2
    - PAN_MATCH [`choice`] : Panchromatic Channel Matching. Available Choices: [0] normalized [1] standardized Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '4', 'IHS Sharpening')
    if Tool.is_Okay():
        Tool.Set_Input ('R', R)
        Tool.Set_Input ('G', G)
        Tool.Set_Input ('B', B)
        Tool.Set_Input ('PAN', PAN)
        Tool.Set_Output('R_SHARP', R_SHARP)
        Tool.Set_Output('G_SHARP', G_SHARP)
        Tool.Set_Output('B_SHARP', B_SHARP)
        Tool.Set_Output('SHARP', SHARP)
        Tool.Set_Option('PAN_SYSTEM', PAN_SYSTEM)
        Tool.Set_Option('OUTPUT', OUTPUT)
        Tool.Set_Option('RESAMPLING', RESAMPLING)
        Tool.Set_Option('PAN_MATCH', PAN_MATCH)
        return Tool.Execute(Verbose)
    return False

def run_tool_imagery_tools_4(
    R=None,
    G=None,
    B=None,
    PAN=None,
    R_SHARP=None,
    G_SHARP=None,
    B_SHARP=None,
    SHARP=None,
    PAN_SYSTEM=None,
    OUTPUT=None,
    RESAMPLING=None,
    PAN_MATCH=None, Verbose=2):
    '''
    IHS Sharpening
    ----------
    [imagery_tools.4]\n
    Intensity, hue, saturation (IHS) sharpening.\n
    Arguments
    ----------
    - R [`input grid`] : Red
    - G [`input grid`] : Green
    - B [`input grid`] : Blue
    - PAN [`input grid`] : Panchromatic Channel
    - R_SHARP [`output grid`] : Red
    - G_SHARP [`output grid`] : Green
    - B_SHARP [`output grid`] : Blue
    - SHARP [`output grid collection`] : Sharpened
    - PAN_SYSTEM [`grid system`] : High Resolution Grid System
    - OUTPUT [`choice`] : Output. Available Choices: [0] single grids [1] grid collection Default: 1
    - RESAMPLING [`choice`] : Resampling. Available Choices: [0] nearest neighbour [1] bilinear [2] cubic convolution Default: 2
    - PAN_MATCH [`choice`] : Panchromatic Channel Matching. Available Choices: [0] normalized [1] standardized Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '4', 'IHS Sharpening')
    if Tool.is_Okay():
        Tool.Set_Input ('R', R)
        Tool.Set_Input ('G', G)
        Tool.Set_Input ('B', B)
        Tool.Set_Input ('PAN', PAN)
        Tool.Set_Output('R_SHARP', R_SHARP)
        Tool.Set_Output('G_SHARP', G_SHARP)
        Tool.Set_Output('B_SHARP', B_SHARP)
        Tool.Set_Output('SHARP', SHARP)
        Tool.Set_Option('PAN_SYSTEM', PAN_SYSTEM)
        Tool.Set_Option('OUTPUT', OUTPUT)
        Tool.Set_Option('RESAMPLING', RESAMPLING)
        Tool.Set_Option('PAN_MATCH', PAN_MATCH)
        return Tool.Execute(Verbose)
    return False

def Colour_Normalized_Brovey_Sharpening(
    R=None,
    G=None,
    B=None,
    PAN=None,
    R_SHARP=None,
    G_SHARP=None,
    B_SHARP=None,
    SHARP=None,
    PAN_SYSTEM=None,
    OUTPUT=None,
    RESAMPLING=None, Verbose=2):
    '''
    Colour Normalized Brovey Sharpening
    ----------
    [imagery_tools.5]\n
    Colour normalized (Brovey) sharpening.\n
    Arguments
    ----------
    - R [`input grid`] : Red
    - G [`input grid`] : Green
    - B [`input grid`] : Blue
    - PAN [`input grid`] : Panchromatic Channel
    - R_SHARP [`output grid`] : Red
    - G_SHARP [`output grid`] : Green
    - B_SHARP [`output grid`] : Blue
    - SHARP [`output grid collection`] : Sharpened
    - PAN_SYSTEM [`grid system`] : High Resolution Grid System
    - OUTPUT [`choice`] : Output. Available Choices: [0] single grids [1] grid collection Default: 1
    - RESAMPLING [`choice`] : Resampling. Available Choices: [0] nearest neighbour [1] bilinear [2] cubic convolution Default: 2

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '5', 'Colour Normalized Brovey Sharpening')
    if Tool.is_Okay():
        Tool.Set_Input ('R', R)
        Tool.Set_Input ('G', G)
        Tool.Set_Input ('B', B)
        Tool.Set_Input ('PAN', PAN)
        Tool.Set_Output('R_SHARP', R_SHARP)
        Tool.Set_Output('G_SHARP', G_SHARP)
        Tool.Set_Output('B_SHARP', B_SHARP)
        Tool.Set_Output('SHARP', SHARP)
        Tool.Set_Option('PAN_SYSTEM', PAN_SYSTEM)
        Tool.Set_Option('OUTPUT', OUTPUT)
        Tool.Set_Option('RESAMPLING', RESAMPLING)
        return Tool.Execute(Verbose)
    return False

def run_tool_imagery_tools_5(
    R=None,
    G=None,
    B=None,
    PAN=None,
    R_SHARP=None,
    G_SHARP=None,
    B_SHARP=None,
    SHARP=None,
    PAN_SYSTEM=None,
    OUTPUT=None,
    RESAMPLING=None, Verbose=2):
    '''
    Colour Normalized Brovey Sharpening
    ----------
    [imagery_tools.5]\n
    Colour normalized (Brovey) sharpening.\n
    Arguments
    ----------
    - R [`input grid`] : Red
    - G [`input grid`] : Green
    - B [`input grid`] : Blue
    - PAN [`input grid`] : Panchromatic Channel
    - R_SHARP [`output grid`] : Red
    - G_SHARP [`output grid`] : Green
    - B_SHARP [`output grid`] : Blue
    - SHARP [`output grid collection`] : Sharpened
    - PAN_SYSTEM [`grid system`] : High Resolution Grid System
    - OUTPUT [`choice`] : Output. Available Choices: [0] single grids [1] grid collection Default: 1
    - RESAMPLING [`choice`] : Resampling. Available Choices: [0] nearest neighbour [1] bilinear [2] cubic convolution Default: 2

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '5', 'Colour Normalized Brovey Sharpening')
    if Tool.is_Okay():
        Tool.Set_Input ('R', R)
        Tool.Set_Input ('G', G)
        Tool.Set_Input ('B', B)
        Tool.Set_Input ('PAN', PAN)
        Tool.Set_Output('R_SHARP', R_SHARP)
        Tool.Set_Output('G_SHARP', G_SHARP)
        Tool.Set_Output('B_SHARP', B_SHARP)
        Tool.Set_Output('SHARP', SHARP)
        Tool.Set_Option('PAN_SYSTEM', PAN_SYSTEM)
        Tool.Set_Option('OUTPUT', OUTPUT)
        Tool.Set_Option('RESAMPLING', RESAMPLING)
        return Tool.Execute(Verbose)
    return False

def Colour_Normalized_Spectral_Sharpening(
    GRIDS=None,
    PAN=None,
    SINGLES=None,
    COLLECTION=None,
    LO_RES=None,
    OUTPUT=None,
    RESAMPLING=None, Verbose=2):
    '''
    Colour Normalized Spectral Sharpening
    ----------
    [imagery_tools.6]\n
    Colour normalized spectral sharpening.\n
    Arguments
    ----------
    - GRIDS [`input grid list`] : Original Channels
    - PAN [`input grid`] : Panchromatic Channel
    - SINGLES [`output grid list`] : Sharpened Channels
    - COLLECTION [`output grid collection`] : Sharpened Channels
    - LO_RES [`grid system`] : Low Resolution Grid System
    - OUTPUT [`choice`] : Output. Available Choices: [0] single grids [1] grid collection Default: 1
    - RESAMPLING [`choice`] : Resampling. Available Choices: [0] nearest neighbour [1] bilinear [2] cubic convolution Default: 2

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '6', 'Colour Normalized Spectral Sharpening')
    if Tool.is_Okay():
        Tool.Set_Input ('GRIDS', GRIDS)
        Tool.Set_Input ('PAN', PAN)
        Tool.Set_Output('SINGLES', SINGLES)
        Tool.Set_Output('COLLECTION', COLLECTION)
        Tool.Set_Option('LO_RES', LO_RES)
        Tool.Set_Option('OUTPUT', OUTPUT)
        Tool.Set_Option('RESAMPLING', RESAMPLING)
        return Tool.Execute(Verbose)
    return False

def run_tool_imagery_tools_6(
    GRIDS=None,
    PAN=None,
    SINGLES=None,
    COLLECTION=None,
    LO_RES=None,
    OUTPUT=None,
    RESAMPLING=None, Verbose=2):
    '''
    Colour Normalized Spectral Sharpening
    ----------
    [imagery_tools.6]\n
    Colour normalized spectral sharpening.\n
    Arguments
    ----------
    - GRIDS [`input grid list`] : Original Channels
    - PAN [`input grid`] : Panchromatic Channel
    - SINGLES [`output grid list`] : Sharpened Channels
    - COLLECTION [`output grid collection`] : Sharpened Channels
    - LO_RES [`grid system`] : Low Resolution Grid System
    - OUTPUT [`choice`] : Output. Available Choices: [0] single grids [1] grid collection Default: 1
    - RESAMPLING [`choice`] : Resampling. Available Choices: [0] nearest neighbour [1] bilinear [2] cubic convolution Default: 2

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '6', 'Colour Normalized Spectral Sharpening')
    if Tool.is_Okay():
        Tool.Set_Input ('GRIDS', GRIDS)
        Tool.Set_Input ('PAN', PAN)
        Tool.Set_Output('SINGLES', SINGLES)
        Tool.Set_Output('COLLECTION', COLLECTION)
        Tool.Set_Option('LO_RES', LO_RES)
        Tool.Set_Option('OUTPUT', OUTPUT)
        Tool.Set_Option('RESAMPLING', RESAMPLING)
        return Tool.Execute(Verbose)
    return False

def Principal_Component_Based_Image_Sharpening(
    GRIDS=None,
    PAN=None,
    SINGLES=None,
    COLLECTION=None,
    LO_RES=None,
    OUTPUT=None,
    METHOD=None,
    RESAMPLING=None,
    PAN_MATCH=None,
    OVERWRITE=None, Verbose=2):
    '''
    Principal Component Based Image Sharpening
    ----------
    [imagery_tools.7]\n
    Principal component based image sharpening.\n
    Arguments
    ----------
    - GRIDS [`input grid list`] : Original Channels
    - PAN [`input grid`] : Panchromatic Channel
    - SINGLES [`output grid list`] : Sharpened Channels
    - COLLECTION [`output grid collection`] : Sharpened Channels
    - LO_RES [`grid system`] : Low Resolution
    - OUTPUT [`choice`] : Output. Available Choices: [0] single grids [1] grid collection Default: 1
    - METHOD [`choice`] : Method. Available Choices: [0] correlation matrix [1] variance-covariance matrix [2] sums-of-squares-and-cross-products matrix Default: 2
    - RESAMPLING [`choice`] : Resampling. Available Choices: [0] nearest neighbour [1] bilinear [2] cubic convolution Default: 2
    - PAN_MATCH [`choice`] : Panchromatic Channel Matching. Available Choices: [0] normalized [1] standardized Default: 1
    - OVERWRITE [`boolean`] : Overwrite. Default: 1 overwrite previous output if adequate

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '7', 'Principal Component Based Image Sharpening')
    if Tool.is_Okay():
        Tool.Set_Input ('GRIDS', GRIDS)
        Tool.Set_Input ('PAN', PAN)
        Tool.Set_Output('SINGLES', SINGLES)
        Tool.Set_Output('COLLECTION', COLLECTION)
        Tool.Set_Option('LO_RES', LO_RES)
        Tool.Set_Option('OUTPUT', OUTPUT)
        Tool.Set_Option('METHOD', METHOD)
        Tool.Set_Option('RESAMPLING', RESAMPLING)
        Tool.Set_Option('PAN_MATCH', PAN_MATCH)
        Tool.Set_Option('OVERWRITE', OVERWRITE)
        return Tool.Execute(Verbose)
    return False

def run_tool_imagery_tools_7(
    GRIDS=None,
    PAN=None,
    SINGLES=None,
    COLLECTION=None,
    LO_RES=None,
    OUTPUT=None,
    METHOD=None,
    RESAMPLING=None,
    PAN_MATCH=None,
    OVERWRITE=None, Verbose=2):
    '''
    Principal Component Based Image Sharpening
    ----------
    [imagery_tools.7]\n
    Principal component based image sharpening.\n
    Arguments
    ----------
    - GRIDS [`input grid list`] : Original Channels
    - PAN [`input grid`] : Panchromatic Channel
    - SINGLES [`output grid list`] : Sharpened Channels
    - COLLECTION [`output grid collection`] : Sharpened Channels
    - LO_RES [`grid system`] : Low Resolution
    - OUTPUT [`choice`] : Output. Available Choices: [0] single grids [1] grid collection Default: 1
    - METHOD [`choice`] : Method. Available Choices: [0] correlation matrix [1] variance-covariance matrix [2] sums-of-squares-and-cross-products matrix Default: 2
    - RESAMPLING [`choice`] : Resampling. Available Choices: [0] nearest neighbour [1] bilinear [2] cubic convolution Default: 2
    - PAN_MATCH [`choice`] : Panchromatic Channel Matching. Available Choices: [0] normalized [1] standardized Default: 1
    - OVERWRITE [`boolean`] : Overwrite. Default: 1 overwrite previous output if adequate

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '7', 'Principal Component Based Image Sharpening')
    if Tool.is_Okay():
        Tool.Set_Input ('GRIDS', GRIDS)
        Tool.Set_Input ('PAN', PAN)
        Tool.Set_Output('SINGLES', SINGLES)
        Tool.Set_Output('COLLECTION', COLLECTION)
        Tool.Set_Option('LO_RES', LO_RES)
        Tool.Set_Option('OUTPUT', OUTPUT)
        Tool.Set_Option('METHOD', METHOD)
        Tool.Set_Option('RESAMPLING', RESAMPLING)
        Tool.Set_Option('PAN_MATCH', PAN_MATCH)
        Tool.Set_Option('OVERWRITE', OVERWRITE)
        return Tool.Execute(Verbose)
    return False

def Top_of_Atmosphere_Reflectance(
    MSS01=None,
    MSS02=None,
    MSS03=None,
    MSS04=None,
    TM01=None,
    TM02=None,
    TM03=None,
    TM04=None,
    TM05=None,
    TM07=None,
    ETM01=None,
    ETM02=None,
    ETM03=None,
    ETM04=None,
    ETM05=None,
    ETM07=None,
    OLI01=None,
    OLI02=None,
    OLI03=None,
    OLI04=None,
    OLI05=None,
    OLI06=None,
    OLI07=None,
    OLI09=None,
    TM_T06=None,
    ETM_T61=None,
    ETM_T62=None,
    TIRS10=None,
    TIRS11=None,
    PAN08=None,
    SPECTRAL=None,
    THERMAL=None,
    PANBAND=None,
    MSS=None,
    TM=None,
    ETM=None,
    OLI=None,
    TM_T=None,
    ETM_T=None,
    TIRS=None,
    PAN=None,
    METAFILE=None,
    SENSOR=None,
    DATE_ACQU=None,
    DATE_PROD=None,
    SUN_HGT=None,
    GRIDS_OUT=None,
    GRIDS_NAME=None,
    AS_RAD=None,
    AC_METHOD=None,
    AC_DO_CELLS=None,
    AC_RAYLEIGH=None,
    AC_SUN_RAD=None,
    ETM_GAIN_10=None,
    ETM_GAIN_20=None,
    ETM_GAIN_30=None,
    ETM_GAIN_40=None,
    ETM_GAIN_50=None,
    ETM_GAIN_61=None,
    ETM_GAIN_62=None,
    ETM_GAIN_70=None,
    ETM_GAIN_80=None, Verbose=2):
    '''
    Top of Atmosphere Reflectance
    ----------
    [imagery_tools.8]\n
    Calculation of top-of-atmosphere radiance or reflectance and temperature (TOAR) for Landsat MSS/TM/ETM+. This tool incorporates E.J. Tizado's GRASS GIS implementation (i.landsat.toar).\n
    Arguments
    ----------
    - MSS01 [`optional input grid`] : Band 1
    - MSS02 [`optional input grid`] : Band 2
    - MSS03 [`optional input grid`] : Band 3
    - MSS04 [`optional input grid`] : Band 4
    - TM01 [`optional input grid`] : Band 1
    - TM02 [`optional input grid`] : Band 2
    - TM03 [`optional input grid`] : Band 3
    - TM04 [`optional input grid`] : Band 4
    - TM05 [`optional input grid`] : Band 5
    - TM07 [`optional input grid`] : Band 7
    - ETM01 [`optional input grid`] : Band 1
    - ETM02 [`optional input grid`] : Band 2
    - ETM03 [`optional input grid`] : Band 3
    - ETM04 [`optional input grid`] : Band 4
    - ETM05 [`optional input grid`] : Band 5
    - ETM07 [`optional input grid`] : Band 7
    - OLI01 [`optional input grid`] : Band 1
    - OLI02 [`optional input grid`] : Band 2
    - OLI03 [`optional input grid`] : Band 3
    - OLI04 [`optional input grid`] : Band 4
    - OLI05 [`optional input grid`] : Band 5
    - OLI06 [`optional input grid`] : Band 6
    - OLI07 [`optional input grid`] : Band 7
    - OLI09 [`optional input grid`] : Band 9
    - TM_T06 [`optional input grid`] : Band 6
    - ETM_T61 [`optional input grid`] : Band 61
    - ETM_T62 [`optional input grid`] : Band 62
    - TIRS10 [`optional input grid`] : Band 10
    - TIRS11 [`optional input grid`] : Band 11
    - PAN08 [`optional input grid`] : Band 8
    - SPECTRAL [`output grid list`] : Spectral
    - THERMAL [`output grid list`] : Thermal
    - PANBAND [`output grid list`] : Panchromatic
    - MSS [`grid system`] : Spectral
    - TM [`grid system`] : Spectral
    - ETM [`grid system`] : Spectral
    - OLI [`grid system`] : Spectral
    - TM_T [`grid system`] : Thermal
    - ETM_T [`grid system`] : Thermal
    - TIRS [`grid system`] : Thermal
    - PAN [`grid system`] : Panchromatic
    - METAFILE [`file path`] : Metadata File. Name of Landsat metadata file (.met or MTL.txt)
    - SENSOR [`choice`] : Spacecraft Sensor. Available Choices: [0] Landsat-1 MSS [1] Landsat-2 MSS [2] Landsat-3 MSS [3] Landsat-4 MSS [4] Landsat-5 MSS [5] Landsat-4 TM [6] Landsat-5 TM [7] Landsat-7 ETM+ [8] Landsat-8 OLI/TIRS Default: 7
    - DATE_ACQU [`text`] : Image Acquisition Date. Default: 2001-01-01 Image acquisition date (yyyy-mm-dd)
    - DATE_PROD [`text`] : Image Creation Date. Default: 2001-01-01 Image creation date (yyyy-mm-dd)
    - SUN_HGT [`floating point number`] : Suns's Height. Minimum: 0.000000 Maximum: 90.000000 Default: 45.000000 Sun's height above horizon in degree
    - GRIDS_OUT [`boolean`] : Spectral Output as Grid Collection. Default: 1
    - GRIDS_NAME [`text`] : Name. Default: Landsat Calibrated
    - AS_RAD [`boolean`] : At-Sensor Radiance. Default: 0 Output at-sensor radiance for all bands
    - AC_METHOD [`choice`] : Atmospheric Correction. Available Choices: [0] uncorrected [1] corrected [2] dark object subtraction 1 [3] dark object subtraction 2 [4] dark object subtraction 2b [5] dark object subtraction 3 [6] dark object subtraction 4 Default: 0
    - AC_DO_CELLS [`integer number`] : Minimum Number of Dark Object Cells. Minimum: 0 Default: 1000 Minimum pixels to consider digital number as dark object
    - AC_RAYLEIGH [`floating point number`] : Rayleigh Scattering. Minimum: 0.000000 Default: 0.000000 Rayleigh atmosphere (diffuse sky irradiance)
    - AC_SUN_RAD [`floating point number`] : Solar Radiance. Minimum: 0.000000 Maximum: 100.000000 Default: 1.000000 Percent of solar radiance in path radiance
    - ETM_GAIN_10 [`choice`] : Band 1. Available Choices: [0] low [1] high Default: 1
    - ETM_GAIN_20 [`choice`] : Band 2. Available Choices: [0] low [1] high Default: 1
    - ETM_GAIN_30 [`choice`] : Band 3. Available Choices: [0] low [1] high Default: 1
    - ETM_GAIN_40 [`choice`] : Band 4. Available Choices: [0] low [1] high Default: 1
    - ETM_GAIN_50 [`choice`] : Band 5. Available Choices: [0] low [1] high Default: 1
    - ETM_GAIN_61 [`choice`] : Band 61. Available Choices: [0] low [1] high Default: 0
    - ETM_GAIN_62 [`choice`] : Band 62. Available Choices: [0] low [1] high Default: 1
    - ETM_GAIN_70 [`choice`] : Band 7. Available Choices: [0] low [1] high Default: 1
    - ETM_GAIN_80 [`choice`] : Band 8. Available Choices: [0] low [1] high Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '8', 'Top of Atmosphere Reflectance')
    if Tool.is_Okay():
        Tool.Set_Input ('MSS01', MSS01)
        Tool.Set_Input ('MSS02', MSS02)
        Tool.Set_Input ('MSS03', MSS03)
        Tool.Set_Input ('MSS04', MSS04)
        Tool.Set_Input ('TM01', TM01)
        Tool.Set_Input ('TM02', TM02)
        Tool.Set_Input ('TM03', TM03)
        Tool.Set_Input ('TM04', TM04)
        Tool.Set_Input ('TM05', TM05)
        Tool.Set_Input ('TM07', TM07)
        Tool.Set_Input ('ETM01', ETM01)
        Tool.Set_Input ('ETM02', ETM02)
        Tool.Set_Input ('ETM03', ETM03)
        Tool.Set_Input ('ETM04', ETM04)
        Tool.Set_Input ('ETM05', ETM05)
        Tool.Set_Input ('ETM07', ETM07)
        Tool.Set_Input ('OLI01', OLI01)
        Tool.Set_Input ('OLI02', OLI02)
        Tool.Set_Input ('OLI03', OLI03)
        Tool.Set_Input ('OLI04', OLI04)
        Tool.Set_Input ('OLI05', OLI05)
        Tool.Set_Input ('OLI06', OLI06)
        Tool.Set_Input ('OLI07', OLI07)
        Tool.Set_Input ('OLI09', OLI09)
        Tool.Set_Input ('TM_T06', TM_T06)
        Tool.Set_Input ('ETM_T61', ETM_T61)
        Tool.Set_Input ('ETM_T62', ETM_T62)
        Tool.Set_Input ('TIRS10', TIRS10)
        Tool.Set_Input ('TIRS11', TIRS11)
        Tool.Set_Input ('PAN08', PAN08)
        Tool.Set_Output('SPECTRAL', SPECTRAL)
        Tool.Set_Output('THERMAL', THERMAL)
        Tool.Set_Output('PANBAND', PANBAND)
        Tool.Set_Option('MSS', MSS)
        Tool.Set_Option('TM', TM)
        Tool.Set_Option('ETM', ETM)
        Tool.Set_Option('OLI', OLI)
        Tool.Set_Option('TM_T', TM_T)
        Tool.Set_Option('ETM_T', ETM_T)
        Tool.Set_Option('TIRS', TIRS)
        Tool.Set_Option('PAN', PAN)
        Tool.Set_Option('METAFILE', METAFILE)
        Tool.Set_Option('SENSOR', SENSOR)
        Tool.Set_Option('DATE_ACQU', DATE_ACQU)
        Tool.Set_Option('DATE_PROD', DATE_PROD)
        Tool.Set_Option('SUN_HGT', SUN_HGT)
        Tool.Set_Option('GRIDS_OUT', GRIDS_OUT)
        Tool.Set_Option('GRIDS_NAME', GRIDS_NAME)
        Tool.Set_Option('AS_RAD', AS_RAD)
        Tool.Set_Option('AC_METHOD', AC_METHOD)
        Tool.Set_Option('AC_DO_CELLS', AC_DO_CELLS)
        Tool.Set_Option('AC_RAYLEIGH', AC_RAYLEIGH)
        Tool.Set_Option('AC_SUN_RAD', AC_SUN_RAD)
        Tool.Set_Option('ETM_GAIN_10', ETM_GAIN_10)
        Tool.Set_Option('ETM_GAIN_20', ETM_GAIN_20)
        Tool.Set_Option('ETM_GAIN_30', ETM_GAIN_30)
        Tool.Set_Option('ETM_GAIN_40', ETM_GAIN_40)
        Tool.Set_Option('ETM_GAIN_50', ETM_GAIN_50)
        Tool.Set_Option('ETM_GAIN_61', ETM_GAIN_61)
        Tool.Set_Option('ETM_GAIN_62', ETM_GAIN_62)
        Tool.Set_Option('ETM_GAIN_70', ETM_GAIN_70)
        Tool.Set_Option('ETM_GAIN_80', ETM_GAIN_80)
        return Tool.Execute(Verbose)
    return False

def run_tool_imagery_tools_8(
    MSS01=None,
    MSS02=None,
    MSS03=None,
    MSS04=None,
    TM01=None,
    TM02=None,
    TM03=None,
    TM04=None,
    TM05=None,
    TM07=None,
    ETM01=None,
    ETM02=None,
    ETM03=None,
    ETM04=None,
    ETM05=None,
    ETM07=None,
    OLI01=None,
    OLI02=None,
    OLI03=None,
    OLI04=None,
    OLI05=None,
    OLI06=None,
    OLI07=None,
    OLI09=None,
    TM_T06=None,
    ETM_T61=None,
    ETM_T62=None,
    TIRS10=None,
    TIRS11=None,
    PAN08=None,
    SPECTRAL=None,
    THERMAL=None,
    PANBAND=None,
    MSS=None,
    TM=None,
    ETM=None,
    OLI=None,
    TM_T=None,
    ETM_T=None,
    TIRS=None,
    PAN=None,
    METAFILE=None,
    SENSOR=None,
    DATE_ACQU=None,
    DATE_PROD=None,
    SUN_HGT=None,
    GRIDS_OUT=None,
    GRIDS_NAME=None,
    AS_RAD=None,
    AC_METHOD=None,
    AC_DO_CELLS=None,
    AC_RAYLEIGH=None,
    AC_SUN_RAD=None,
    ETM_GAIN_10=None,
    ETM_GAIN_20=None,
    ETM_GAIN_30=None,
    ETM_GAIN_40=None,
    ETM_GAIN_50=None,
    ETM_GAIN_61=None,
    ETM_GAIN_62=None,
    ETM_GAIN_70=None,
    ETM_GAIN_80=None, Verbose=2):
    '''
    Top of Atmosphere Reflectance
    ----------
    [imagery_tools.8]\n
    Calculation of top-of-atmosphere radiance or reflectance and temperature (TOAR) for Landsat MSS/TM/ETM+. This tool incorporates E.J. Tizado's GRASS GIS implementation (i.landsat.toar).\n
    Arguments
    ----------
    - MSS01 [`optional input grid`] : Band 1
    - MSS02 [`optional input grid`] : Band 2
    - MSS03 [`optional input grid`] : Band 3
    - MSS04 [`optional input grid`] : Band 4
    - TM01 [`optional input grid`] : Band 1
    - TM02 [`optional input grid`] : Band 2
    - TM03 [`optional input grid`] : Band 3
    - TM04 [`optional input grid`] : Band 4
    - TM05 [`optional input grid`] : Band 5
    - TM07 [`optional input grid`] : Band 7
    - ETM01 [`optional input grid`] : Band 1
    - ETM02 [`optional input grid`] : Band 2
    - ETM03 [`optional input grid`] : Band 3
    - ETM04 [`optional input grid`] : Band 4
    - ETM05 [`optional input grid`] : Band 5
    - ETM07 [`optional input grid`] : Band 7
    - OLI01 [`optional input grid`] : Band 1
    - OLI02 [`optional input grid`] : Band 2
    - OLI03 [`optional input grid`] : Band 3
    - OLI04 [`optional input grid`] : Band 4
    - OLI05 [`optional input grid`] : Band 5
    - OLI06 [`optional input grid`] : Band 6
    - OLI07 [`optional input grid`] : Band 7
    - OLI09 [`optional input grid`] : Band 9
    - TM_T06 [`optional input grid`] : Band 6
    - ETM_T61 [`optional input grid`] : Band 61
    - ETM_T62 [`optional input grid`] : Band 62
    - TIRS10 [`optional input grid`] : Band 10
    - TIRS11 [`optional input grid`] : Band 11
    - PAN08 [`optional input grid`] : Band 8
    - SPECTRAL [`output grid list`] : Spectral
    - THERMAL [`output grid list`] : Thermal
    - PANBAND [`output grid list`] : Panchromatic
    - MSS [`grid system`] : Spectral
    - TM [`grid system`] : Spectral
    - ETM [`grid system`] : Spectral
    - OLI [`grid system`] : Spectral
    - TM_T [`grid system`] : Thermal
    - ETM_T [`grid system`] : Thermal
    - TIRS [`grid system`] : Thermal
    - PAN [`grid system`] : Panchromatic
    - METAFILE [`file path`] : Metadata File. Name of Landsat metadata file (.met or MTL.txt)
    - SENSOR [`choice`] : Spacecraft Sensor. Available Choices: [0] Landsat-1 MSS [1] Landsat-2 MSS [2] Landsat-3 MSS [3] Landsat-4 MSS [4] Landsat-5 MSS [5] Landsat-4 TM [6] Landsat-5 TM [7] Landsat-7 ETM+ [8] Landsat-8 OLI/TIRS Default: 7
    - DATE_ACQU [`text`] : Image Acquisition Date. Default: 2001-01-01 Image acquisition date (yyyy-mm-dd)
    - DATE_PROD [`text`] : Image Creation Date. Default: 2001-01-01 Image creation date (yyyy-mm-dd)
    - SUN_HGT [`floating point number`] : Suns's Height. Minimum: 0.000000 Maximum: 90.000000 Default: 45.000000 Sun's height above horizon in degree
    - GRIDS_OUT [`boolean`] : Spectral Output as Grid Collection. Default: 1
    - GRIDS_NAME [`text`] : Name. Default: Landsat Calibrated
    - AS_RAD [`boolean`] : At-Sensor Radiance. Default: 0 Output at-sensor radiance for all bands
    - AC_METHOD [`choice`] : Atmospheric Correction. Available Choices: [0] uncorrected [1] corrected [2] dark object subtraction 1 [3] dark object subtraction 2 [4] dark object subtraction 2b [5] dark object subtraction 3 [6] dark object subtraction 4 Default: 0
    - AC_DO_CELLS [`integer number`] : Minimum Number of Dark Object Cells. Minimum: 0 Default: 1000 Minimum pixels to consider digital number as dark object
    - AC_RAYLEIGH [`floating point number`] : Rayleigh Scattering. Minimum: 0.000000 Default: 0.000000 Rayleigh atmosphere (diffuse sky irradiance)
    - AC_SUN_RAD [`floating point number`] : Solar Radiance. Minimum: 0.000000 Maximum: 100.000000 Default: 1.000000 Percent of solar radiance in path radiance
    - ETM_GAIN_10 [`choice`] : Band 1. Available Choices: [0] low [1] high Default: 1
    - ETM_GAIN_20 [`choice`] : Band 2. Available Choices: [0] low [1] high Default: 1
    - ETM_GAIN_30 [`choice`] : Band 3. Available Choices: [0] low [1] high Default: 1
    - ETM_GAIN_40 [`choice`] : Band 4. Available Choices: [0] low [1] high Default: 1
    - ETM_GAIN_50 [`choice`] : Band 5. Available Choices: [0] low [1] high Default: 1
    - ETM_GAIN_61 [`choice`] : Band 61. Available Choices: [0] low [1] high Default: 0
    - ETM_GAIN_62 [`choice`] : Band 62. Available Choices: [0] low [1] high Default: 1
    - ETM_GAIN_70 [`choice`] : Band 7. Available Choices: [0] low [1] high Default: 1
    - ETM_GAIN_80 [`choice`] : Band 8. Available Choices: [0] low [1] high Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '8', 'Top of Atmosphere Reflectance')
    if Tool.is_Okay():
        Tool.Set_Input ('MSS01', MSS01)
        Tool.Set_Input ('MSS02', MSS02)
        Tool.Set_Input ('MSS03', MSS03)
        Tool.Set_Input ('MSS04', MSS04)
        Tool.Set_Input ('TM01', TM01)
        Tool.Set_Input ('TM02', TM02)
        Tool.Set_Input ('TM03', TM03)
        Tool.Set_Input ('TM04', TM04)
        Tool.Set_Input ('TM05', TM05)
        Tool.Set_Input ('TM07', TM07)
        Tool.Set_Input ('ETM01', ETM01)
        Tool.Set_Input ('ETM02', ETM02)
        Tool.Set_Input ('ETM03', ETM03)
        Tool.Set_Input ('ETM04', ETM04)
        Tool.Set_Input ('ETM05', ETM05)
        Tool.Set_Input ('ETM07', ETM07)
        Tool.Set_Input ('OLI01', OLI01)
        Tool.Set_Input ('OLI02', OLI02)
        Tool.Set_Input ('OLI03', OLI03)
        Tool.Set_Input ('OLI04', OLI04)
        Tool.Set_Input ('OLI05', OLI05)
        Tool.Set_Input ('OLI06', OLI06)
        Tool.Set_Input ('OLI07', OLI07)
        Tool.Set_Input ('OLI09', OLI09)
        Tool.Set_Input ('TM_T06', TM_T06)
        Tool.Set_Input ('ETM_T61', ETM_T61)
        Tool.Set_Input ('ETM_T62', ETM_T62)
        Tool.Set_Input ('TIRS10', TIRS10)
        Tool.Set_Input ('TIRS11', TIRS11)
        Tool.Set_Input ('PAN08', PAN08)
        Tool.Set_Output('SPECTRAL', SPECTRAL)
        Tool.Set_Output('THERMAL', THERMAL)
        Tool.Set_Output('PANBAND', PANBAND)
        Tool.Set_Option('MSS', MSS)
        Tool.Set_Option('TM', TM)
        Tool.Set_Option('ETM', ETM)
        Tool.Set_Option('OLI', OLI)
        Tool.Set_Option('TM_T', TM_T)
        Tool.Set_Option('ETM_T', ETM_T)
        Tool.Set_Option('TIRS', TIRS)
        Tool.Set_Option('PAN', PAN)
        Tool.Set_Option('METAFILE', METAFILE)
        Tool.Set_Option('SENSOR', SENSOR)
        Tool.Set_Option('DATE_ACQU', DATE_ACQU)
        Tool.Set_Option('DATE_PROD', DATE_PROD)
        Tool.Set_Option('SUN_HGT', SUN_HGT)
        Tool.Set_Option('GRIDS_OUT', GRIDS_OUT)
        Tool.Set_Option('GRIDS_NAME', GRIDS_NAME)
        Tool.Set_Option('AS_RAD', AS_RAD)
        Tool.Set_Option('AC_METHOD', AC_METHOD)
        Tool.Set_Option('AC_DO_CELLS', AC_DO_CELLS)
        Tool.Set_Option('AC_RAYLEIGH', AC_RAYLEIGH)
        Tool.Set_Option('AC_SUN_RAD', AC_SUN_RAD)
        Tool.Set_Option('ETM_GAIN_10', ETM_GAIN_10)
        Tool.Set_Option('ETM_GAIN_20', ETM_GAIN_20)
        Tool.Set_Option('ETM_GAIN_30', ETM_GAIN_30)
        Tool.Set_Option('ETM_GAIN_40', ETM_GAIN_40)
        Tool.Set_Option('ETM_GAIN_50', ETM_GAIN_50)
        Tool.Set_Option('ETM_GAIN_61', ETM_GAIN_61)
        Tool.Set_Option('ETM_GAIN_62', ETM_GAIN_62)
        Tool.Set_Option('ETM_GAIN_70', ETM_GAIN_70)
        Tool.Set_Option('ETM_GAIN_80', ETM_GAIN_80)
        return Tool.Execute(Verbose)
    return False

def Automated_Cloud_Cover_Assessment(
    BAND2=None,
    BAND3=None,
    BAND4=None,
    BAND5=None,
    BAND6=None,
    CLOUD=None,
    BAND6_GRIDSYSTEM=None,
    B56C=None,
    B45R=None,
    HIST_N=None,
    CSIG=None,
    PASS2=None,
    SHADOW=None,
    FILTER=None, Verbose=2):
    '''
    Automated Cloud Cover Assessment
    ----------
    [imagery_tools.9]\n
    Automated Cloud-Cover Assessment (ACCA) for Landsat TM/ETM+ imagery as proposed by Irish (2000). This tool incorporates E.J. Tizado's GRASS GIS implementation (i.landsat.acca).\n
    Arguments
    ----------
    - BAND2 [`input grid`] : Green. Reflectance
    - BAND3 [`input grid`] : Red. Reflectance
    - BAND4 [`input grid`] : NIR. Reflectance
    - BAND5 [`input grid`] : SWIR. Reflectance
    - BAND6 [`input grid`] : Thermal. Kelvin
    - CLOUD [`output grid`] : Clouds
    - BAND6_GRIDSYSTEM [`grid system`] : Grid system
    - B56C [`floating point number`] : SWIR/Thermal Threshold. Default: 225.000000 Threshold for SWIR/Thermal Composite (step 6).
    - B45R [`floating point number`] : Desert Detection Threshold. Default: 1.000000 Threshold for desert detection (step 10,  NIR/SWIR Ratio).
    - HIST_N [`integer number`] : Temperature Histogram. Minimum: 10 Default: 100 Number of classes in the cloud temperature histogram.
    - CSIG [`boolean`] : Cloud Signature. Default: 1 Always use cloud signature (step 14).
    - PASS2 [`boolean`] : Cloud Differentiation. Default: 0 Differentiate between warm (not ambiguous) and cold clouds.
    - SHADOW [`boolean`] : Shadows. Default: 1 Include a category for cloud shadows.
    - FILTER [`boolean`] : Filter. Default: 1 Apply post-processing filter to remove small holes.

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '9', 'Automated Cloud Cover Assessment')
    if Tool.is_Okay():
        Tool.Set_Input ('BAND2', BAND2)
        Tool.Set_Input ('BAND3', BAND3)
        Tool.Set_Input ('BAND4', BAND4)
        Tool.Set_Input ('BAND5', BAND5)
        Tool.Set_Input ('BAND6', BAND6)
        Tool.Set_Output('CLOUD', CLOUD)
        Tool.Set_Option('BAND6_GRIDSYSTEM', BAND6_GRIDSYSTEM)
        Tool.Set_Option('B56C', B56C)
        Tool.Set_Option('B45R', B45R)
        Tool.Set_Option('HIST_N', HIST_N)
        Tool.Set_Option('CSIG', CSIG)
        Tool.Set_Option('PASS2', PASS2)
        Tool.Set_Option('SHADOW', SHADOW)
        Tool.Set_Option('FILTER', FILTER)
        return Tool.Execute(Verbose)
    return False

def run_tool_imagery_tools_9(
    BAND2=None,
    BAND3=None,
    BAND4=None,
    BAND5=None,
    BAND6=None,
    CLOUD=None,
    BAND6_GRIDSYSTEM=None,
    B56C=None,
    B45R=None,
    HIST_N=None,
    CSIG=None,
    PASS2=None,
    SHADOW=None,
    FILTER=None, Verbose=2):
    '''
    Automated Cloud Cover Assessment
    ----------
    [imagery_tools.9]\n
    Automated Cloud-Cover Assessment (ACCA) for Landsat TM/ETM+ imagery as proposed by Irish (2000). This tool incorporates E.J. Tizado's GRASS GIS implementation (i.landsat.acca).\n
    Arguments
    ----------
    - BAND2 [`input grid`] : Green. Reflectance
    - BAND3 [`input grid`] : Red. Reflectance
    - BAND4 [`input grid`] : NIR. Reflectance
    - BAND5 [`input grid`] : SWIR. Reflectance
    - BAND6 [`input grid`] : Thermal. Kelvin
    - CLOUD [`output grid`] : Clouds
    - BAND6_GRIDSYSTEM [`grid system`] : Grid system
    - B56C [`floating point number`] : SWIR/Thermal Threshold. Default: 225.000000 Threshold for SWIR/Thermal Composite (step 6).
    - B45R [`floating point number`] : Desert Detection Threshold. Default: 1.000000 Threshold for desert detection (step 10,  NIR/SWIR Ratio).
    - HIST_N [`integer number`] : Temperature Histogram. Minimum: 10 Default: 100 Number of classes in the cloud temperature histogram.
    - CSIG [`boolean`] : Cloud Signature. Default: 1 Always use cloud signature (step 14).
    - PASS2 [`boolean`] : Cloud Differentiation. Default: 0 Differentiate between warm (not ambiguous) and cold clouds.
    - SHADOW [`boolean`] : Shadows. Default: 1 Include a category for cloud shadows.
    - FILTER [`boolean`] : Filter. Default: 1 Apply post-processing filter to remove small holes.

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '9', 'Automated Cloud Cover Assessment')
    if Tool.is_Okay():
        Tool.Set_Input ('BAND2', BAND2)
        Tool.Set_Input ('BAND3', BAND3)
        Tool.Set_Input ('BAND4', BAND4)
        Tool.Set_Input ('BAND5', BAND5)
        Tool.Set_Input ('BAND6', BAND6)
        Tool.Set_Output('CLOUD', CLOUD)
        Tool.Set_Option('BAND6_GRIDSYSTEM', BAND6_GRIDSYSTEM)
        Tool.Set_Option('B56C', B56C)
        Tool.Set_Option('B45R', B45R)
        Tool.Set_Option('HIST_N', HIST_N)
        Tool.Set_Option('CSIG', CSIG)
        Tool.Set_Option('PASS2', PASS2)
        Tool.Set_Option('SHADOW', SHADOW)
        Tool.Set_Option('FILTER', FILTER)
        return Tool.Execute(Verbose)
    return False

def Landsat_Import_with_Options(
    BANDS=None,
    FILES=None,
    PROJECTION=None,
    RESAMPLING=None,
    SHOW_RGB=None,
    SHOW_R=None,
    SHOW_G=None,
    SHOW_B=None, Verbose=2):
    '''
    Landsat Import with Options
    ----------
    [imagery_tools.10]\n
    This tool facilitates the import and display of Landsat scenes, which have each band given as a single GeoTIFF file.\n
    The development of this tool has been requested and sponsored by Rohan Fisher, Charles Darwin University, Australia.\n
    Arguments
    ----------
    - BANDS [`output grid list`] : Bands
    - FILES [`file path`] : Files
    - PROJECTION [`choice`] : Coordinate System. Available Choices: [0] UTM North [1] UTM South [2] Geographic Coordinates Default: 0
    - RESAMPLING [`choice`] : Resampling. Available Choices: [0] Nearest Neighbour [1] Bilinear Interpolation [2] Bicubic Spline Interpolation [3] B-Spline Interpolation Default: 3
    - SHOW_RGB [`boolean`] : Show a Composite. Default: 1
    - SHOW_R [`choice`] : Red. Available Choices: [0] no choice available Default: 0
    - SHOW_G [`choice`] : Green. Available Choices: [0] no choice available Default: 0
    - SHOW_B [`choice`] : Blue. Available Choices: [0] no choice available Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '10', 'Landsat Import with Options')
    if Tool.is_Okay():
        Tool.Set_Output('BANDS', BANDS)
        Tool.Set_Option('FILES', FILES)
        Tool.Set_Option('PROJECTION', PROJECTION)
        Tool.Set_Option('RESAMPLING', RESAMPLING)
        Tool.Set_Option('SHOW_RGB', SHOW_RGB)
        Tool.Set_Option('SHOW_R', SHOW_R)
        Tool.Set_Option('SHOW_G', SHOW_G)
        Tool.Set_Option('SHOW_B', SHOW_B)
        return Tool.Execute(Verbose)
    return False

def run_tool_imagery_tools_10(
    BANDS=None,
    FILES=None,
    PROJECTION=None,
    RESAMPLING=None,
    SHOW_RGB=None,
    SHOW_R=None,
    SHOW_G=None,
    SHOW_B=None, Verbose=2):
    '''
    Landsat Import with Options
    ----------
    [imagery_tools.10]\n
    This tool facilitates the import and display of Landsat scenes, which have each band given as a single GeoTIFF file.\n
    The development of this tool has been requested and sponsored by Rohan Fisher, Charles Darwin University, Australia.\n
    Arguments
    ----------
    - BANDS [`output grid list`] : Bands
    - FILES [`file path`] : Files
    - PROJECTION [`choice`] : Coordinate System. Available Choices: [0] UTM North [1] UTM South [2] Geographic Coordinates Default: 0
    - RESAMPLING [`choice`] : Resampling. Available Choices: [0] Nearest Neighbour [1] Bilinear Interpolation [2] Bicubic Spline Interpolation [3] B-Spline Interpolation Default: 3
    - SHOW_RGB [`boolean`] : Show a Composite. Default: 1
    - SHOW_R [`choice`] : Red. Available Choices: [0] no choice available Default: 0
    - SHOW_G [`choice`] : Green. Available Choices: [0] no choice available Default: 0
    - SHOW_B [`choice`] : Blue. Available Choices: [0] no choice available Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '10', 'Landsat Import with Options')
    if Tool.is_Okay():
        Tool.Set_Output('BANDS', BANDS)
        Tool.Set_Option('FILES', FILES)
        Tool.Set_Option('PROJECTION', PROJECTION)
        Tool.Set_Option('RESAMPLING', RESAMPLING)
        Tool.Set_Option('SHOW_RGB', SHOW_RGB)
        Tool.Set_Option('SHOW_R', SHOW_R)
        Tool.Set_Option('SHOW_G', SHOW_G)
        Tool.Set_Option('SHOW_B', SHOW_B)
        return Tool.Execute(Verbose)
    return False

def Textural_Features(
    GRID=None,
    ASM=None,
    CONTRAST=None,
    CORRELATION=None,
    VARIANCE=None,
    IDM=None,
    SUM_AVERAGE=None,
    SUM_ENTROPY=None,
    SUM_VARIANCE=None,
    ENTROPY=None,
    DIF_VARIANCE=None,
    DIF_ENTROPY=None,
    MOC_1=None,
    MOC_2=None,
    DIRECTION=None,
    RADIUS=None,
    DISTANCE=None,
    MAX_CATS=None, Verbose=2):
    '''
    Textural Features
    ----------
    [imagery_tools.11]\n
    Textural features. This tool is based on the GRASS GIS implementation by Carmine Basco (r.texture).\n
    Arguments
    ----------
    - GRID [`input grid`] : Grid
    - ASM [`output grid`] : Angular Second Moment
    - CONTRAST [`output grid`] : Contrast
    - CORRELATION [`output grid`] : Correlation
    - VARIANCE [`output grid`] : Variance
    - IDM [`output grid`] : Inverse Diff Moment
    - SUM_AVERAGE [`output grid`] : Sum Average
    - SUM_ENTROPY [`output grid`] : Sum Entropy
    - SUM_VARIANCE [`output grid`] : Sum Variance
    - ENTROPY [`output grid`] : Entropy
    - DIF_VARIANCE [`output grid`] : Difference Variance
    - DIF_ENTROPY [`output grid`] : Difference Entropy
    - MOC_1 [`output grid`] : Measure of Correlation-1
    - MOC_2 [`output grid`] : Measure of Correlation-2
    - DIRECTION [`choice`] : Direction. Available Choices: [0] all [1] N-S [2] NE-SW [3] E-W [4] SE-NW Default: 0
    - RADIUS [`integer number`] : Radius. Minimum: 1 Default: 1 kernel radius in cells
    - DISTANCE [`integer number`] : Distance. Minimum: 1 Default: 1 The distance between two samples.
    - MAX_CATS [`integer number`] : Maximum Number of Categories. Minimum: 2 Default: 256

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '11', 'Textural Features')
    if Tool.is_Okay():
        Tool.Set_Input ('GRID', GRID)
        Tool.Set_Output('ASM', ASM)
        Tool.Set_Output('CONTRAST', CONTRAST)
        Tool.Set_Output('CORRELATION', CORRELATION)
        Tool.Set_Output('VARIANCE', VARIANCE)
        Tool.Set_Output('IDM', IDM)
        Tool.Set_Output('SUM_AVERAGE', SUM_AVERAGE)
        Tool.Set_Output('SUM_ENTROPY', SUM_ENTROPY)
        Tool.Set_Output('SUM_VARIANCE', SUM_VARIANCE)
        Tool.Set_Output('ENTROPY', ENTROPY)
        Tool.Set_Output('DIF_VARIANCE', DIF_VARIANCE)
        Tool.Set_Output('DIF_ENTROPY', DIF_ENTROPY)
        Tool.Set_Output('MOC_1', MOC_1)
        Tool.Set_Output('MOC_2', MOC_2)
        Tool.Set_Option('DIRECTION', DIRECTION)
        Tool.Set_Option('RADIUS', RADIUS)
        Tool.Set_Option('DISTANCE', DISTANCE)
        Tool.Set_Option('MAX_CATS', MAX_CATS)
        return Tool.Execute(Verbose)
    return False

def run_tool_imagery_tools_11(
    GRID=None,
    ASM=None,
    CONTRAST=None,
    CORRELATION=None,
    VARIANCE=None,
    IDM=None,
    SUM_AVERAGE=None,
    SUM_ENTROPY=None,
    SUM_VARIANCE=None,
    ENTROPY=None,
    DIF_VARIANCE=None,
    DIF_ENTROPY=None,
    MOC_1=None,
    MOC_2=None,
    DIRECTION=None,
    RADIUS=None,
    DISTANCE=None,
    MAX_CATS=None, Verbose=2):
    '''
    Textural Features
    ----------
    [imagery_tools.11]\n
    Textural features. This tool is based on the GRASS GIS implementation by Carmine Basco (r.texture).\n
    Arguments
    ----------
    - GRID [`input grid`] : Grid
    - ASM [`output grid`] : Angular Second Moment
    - CONTRAST [`output grid`] : Contrast
    - CORRELATION [`output grid`] : Correlation
    - VARIANCE [`output grid`] : Variance
    - IDM [`output grid`] : Inverse Diff Moment
    - SUM_AVERAGE [`output grid`] : Sum Average
    - SUM_ENTROPY [`output grid`] : Sum Entropy
    - SUM_VARIANCE [`output grid`] : Sum Variance
    - ENTROPY [`output grid`] : Entropy
    - DIF_VARIANCE [`output grid`] : Difference Variance
    - DIF_ENTROPY [`output grid`] : Difference Entropy
    - MOC_1 [`output grid`] : Measure of Correlation-1
    - MOC_2 [`output grid`] : Measure of Correlation-2
    - DIRECTION [`choice`] : Direction. Available Choices: [0] all [1] N-S [2] NE-SW [3] E-W [4] SE-NW Default: 0
    - RADIUS [`integer number`] : Radius. Minimum: 1 Default: 1 kernel radius in cells
    - DISTANCE [`integer number`] : Distance. Minimum: 1 Default: 1 The distance between two samples.
    - MAX_CATS [`integer number`] : Maximum Number of Categories. Minimum: 2 Default: 256

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '11', 'Textural Features')
    if Tool.is_Okay():
        Tool.Set_Input ('GRID', GRID)
        Tool.Set_Output('ASM', ASM)
        Tool.Set_Output('CONTRAST', CONTRAST)
        Tool.Set_Output('CORRELATION', CORRELATION)
        Tool.Set_Output('VARIANCE', VARIANCE)
        Tool.Set_Output('IDM', IDM)
        Tool.Set_Output('SUM_AVERAGE', SUM_AVERAGE)
        Tool.Set_Output('SUM_ENTROPY', SUM_ENTROPY)
        Tool.Set_Output('SUM_VARIANCE', SUM_VARIANCE)
        Tool.Set_Output('ENTROPY', ENTROPY)
        Tool.Set_Output('DIF_VARIANCE', DIF_VARIANCE)
        Tool.Set_Output('DIF_ENTROPY', DIF_ENTROPY)
        Tool.Set_Output('MOC_1', MOC_1)
        Tool.Set_Output('MOC_2', MOC_2)
        Tool.Set_Option('DIRECTION', DIRECTION)
        Tool.Set_Option('RADIUS', RADIUS)
        Tool.Set_Option('DISTANCE', DISTANCE)
        Tool.Set_Option('MAX_CATS', MAX_CATS)
        return Tool.Execute(Verbose)
    return False

def Local_Statistical_Measures(
    GRID=None,
    CONTRAST=None,
    ENERGY=None,
    ENTROPY=None,
    VARIANCE=None,
    TYPE=None,
    RADIUS=None,
    NORMALIZE=None,
    NORM_MIN=None,
    NORM_MAX=None, Verbose=2):
    '''
    Local Statistical Measures
    ----------
    [imagery_tools.12]\n
    Local Statistical Measures\n
    Arguments
    ----------
    - GRID [`input grid`] : Grid
    - CONTRAST [`output grid`] : Contrast
    - ENERGY [`output grid`] : Energy
    - ENTROPY [`output grid`] : Entropy
    - VARIANCE [`output grid`] : Variance
    - TYPE [`choice`] : Kernel. Available Choices: [0] square [1] circle Default: 1 kernel radius in cells
    - RADIUS [`integer number`] : Radius. Minimum: 1 Default: 1 kernel radius in cells
    - NORMALIZE [`choice`] : Normalization. Available Choices: [0] no [1] scale to range Default: 1
    - NORM_MIN [`floating point number`] : Minimum. Default: 1.000000
    - NORM_MAX [`floating point number`] : Maximum. Default: 255.000000

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '12', 'Local Statistical Measures')
    if Tool.is_Okay():
        Tool.Set_Input ('GRID', GRID)
        Tool.Set_Output('CONTRAST', CONTRAST)
        Tool.Set_Output('ENERGY', ENERGY)
        Tool.Set_Output('ENTROPY', ENTROPY)
        Tool.Set_Output('VARIANCE', VARIANCE)
        Tool.Set_Option('TYPE', TYPE)
        Tool.Set_Option('RADIUS', RADIUS)
        Tool.Set_Option('NORMALIZE', NORMALIZE)
        Tool.Set_Option('NORM_MIN', NORM_MIN)
        Tool.Set_Option('NORM_MAX', NORM_MAX)
        return Tool.Execute(Verbose)
    return False

def run_tool_imagery_tools_12(
    GRID=None,
    CONTRAST=None,
    ENERGY=None,
    ENTROPY=None,
    VARIANCE=None,
    TYPE=None,
    RADIUS=None,
    NORMALIZE=None,
    NORM_MIN=None,
    NORM_MAX=None, Verbose=2):
    '''
    Local Statistical Measures
    ----------
    [imagery_tools.12]\n
    Local Statistical Measures\n
    Arguments
    ----------
    - GRID [`input grid`] : Grid
    - CONTRAST [`output grid`] : Contrast
    - ENERGY [`output grid`] : Energy
    - ENTROPY [`output grid`] : Entropy
    - VARIANCE [`output grid`] : Variance
    - TYPE [`choice`] : Kernel. Available Choices: [0] square [1] circle Default: 1 kernel radius in cells
    - RADIUS [`integer number`] : Radius. Minimum: 1 Default: 1 kernel radius in cells
    - NORMALIZE [`choice`] : Normalization. Available Choices: [0] no [1] scale to range Default: 1
    - NORM_MIN [`floating point number`] : Minimum. Default: 1.000000
    - NORM_MAX [`floating point number`] : Maximum. Default: 255.000000

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '12', 'Local Statistical Measures')
    if Tool.is_Okay():
        Tool.Set_Input ('GRID', GRID)
        Tool.Set_Output('CONTRAST', CONTRAST)
        Tool.Set_Output('ENERGY', ENERGY)
        Tool.Set_Output('ENTROPY', ENTROPY)
        Tool.Set_Output('VARIANCE', VARIANCE)
        Tool.Set_Option('TYPE', TYPE)
        Tool.Set_Option('RADIUS', RADIUS)
        Tool.Set_Option('NORMALIZE', NORMALIZE)
        Tool.Set_Option('NORM_MIN', NORM_MIN)
        Tool.Set_Option('NORM_MAX', NORM_MAX)
        return Tool.Execute(Verbose)
    return False

def Universal_Image_Quality_Index(
    GRID_A=None,
    GRID_B=None,
    QUALITY=None,
    CORRELATION=None,
    LUMINANCE=None,
    CONTRAST=None,
    K1=None,
    K2=None,
    L=None,
    KERNEL_TYPE=None,
    KERNEL_RADIUS=None, Verbose=2):
    '''
    Universal Image Quality Index
    ----------
    [imagery_tools.13]\n
    The Universal Image Quality Index compares two grids (greyscale images) using the three parameters luminance, contrast and structure. This is done for each pixel using a moving window as specified by the kernel radius.\n
    Arguments
    ----------
    - GRID_A [`input grid`] : First Grid
    - GRID_B [`input grid`] : Second Grid
    - QUALITY [`output grid`] : Universal Image Quality Index. The product of correlation, luminance and contrast similarity.
    - CORRELATION [`output grid`] : Correlation. The correlation coefficient.
    - LUMINANCE [`output grid`] : Luminance. The similarity of mean luminance.
    - CONTRAST [`output grid`] : Contrast. The similarity of contrast.
    - K1 [`floating point number`] : k1. Minimum: 0.000000 Maximum: 1.000000 Default: 0.010000
    - K2 [`floating point number`] : k2. Minimum: 0.000000 Maximum: 1.000000 Default: 0.030000
    - L [`integer number`] : L. Minimum: 1 Default: 255 The dynamic range for the image pixel, i.e. the number of different grey values.
    - KERNEL_TYPE [`choice`] : Kernel Type. Available Choices: [0] Square [1] Circle Default: 1 The kernel's shape.
    - KERNEL_RADIUS [`integer number`] : Radius. Minimum: 0 Default: 2 cells

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '13', 'Universal Image Quality Index')
    if Tool.is_Okay():
        Tool.Set_Input ('GRID_A', GRID_A)
        Tool.Set_Input ('GRID_B', GRID_B)
        Tool.Set_Output('QUALITY', QUALITY)
        Tool.Set_Output('CORRELATION', CORRELATION)
        Tool.Set_Output('LUMINANCE', LUMINANCE)
        Tool.Set_Output('CONTRAST', CONTRAST)
        Tool.Set_Option('K1', K1)
        Tool.Set_Option('K2', K2)
        Tool.Set_Option('L', L)
        Tool.Set_Option('KERNEL_TYPE', KERNEL_TYPE)
        Tool.Set_Option('KERNEL_RADIUS', KERNEL_RADIUS)
        return Tool.Execute(Verbose)
    return False

def run_tool_imagery_tools_13(
    GRID_A=None,
    GRID_B=None,
    QUALITY=None,
    CORRELATION=None,
    LUMINANCE=None,
    CONTRAST=None,
    K1=None,
    K2=None,
    L=None,
    KERNEL_TYPE=None,
    KERNEL_RADIUS=None, Verbose=2):
    '''
    Universal Image Quality Index
    ----------
    [imagery_tools.13]\n
    The Universal Image Quality Index compares two grids (greyscale images) using the three parameters luminance, contrast and structure. This is done for each pixel using a moving window as specified by the kernel radius.\n
    Arguments
    ----------
    - GRID_A [`input grid`] : First Grid
    - GRID_B [`input grid`] : Second Grid
    - QUALITY [`output grid`] : Universal Image Quality Index. The product of correlation, luminance and contrast similarity.
    - CORRELATION [`output grid`] : Correlation. The correlation coefficient.
    - LUMINANCE [`output grid`] : Luminance. The similarity of mean luminance.
    - CONTRAST [`output grid`] : Contrast. The similarity of contrast.
    - K1 [`floating point number`] : k1. Minimum: 0.000000 Maximum: 1.000000 Default: 0.010000
    - K2 [`floating point number`] : k2. Minimum: 0.000000 Maximum: 1.000000 Default: 0.030000
    - L [`integer number`] : L. Minimum: 1 Default: 255 The dynamic range for the image pixel, i.e. the number of different grey values.
    - KERNEL_TYPE [`choice`] : Kernel Type. Available Choices: [0] Square [1] Circle Default: 1 The kernel's shape.
    - KERNEL_RADIUS [`integer number`] : Radius. Minimum: 0 Default: 2 cells

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '13', 'Universal Image Quality Index')
    if Tool.is_Okay():
        Tool.Set_Input ('GRID_A', GRID_A)
        Tool.Set_Input ('GRID_B', GRID_B)
        Tool.Set_Output('QUALITY', QUALITY)
        Tool.Set_Output('CORRELATION', CORRELATION)
        Tool.Set_Output('LUMINANCE', LUMINANCE)
        Tool.Set_Output('CONTRAST', CONTRAST)
        Tool.Set_Option('K1', K1)
        Tool.Set_Option('K2', K2)
        Tool.Set_Option('L', L)
        Tool.Set_Option('KERNEL_TYPE', KERNEL_TYPE)
        Tool.Set_Option('KERNEL_RADIUS', KERNEL_RADIUS)
        return Tool.Execute(Verbose)
    return False

def Import_Landsat_Scene(
    EXTENT_SHAPES=None,
    BANDS_SPECTRAL=None,
    BANDS_THERMAL=None,
    BANDS_AUXILIARY=None,
    BAND_INFO=None,
    METAFILE=None,
    MULTI2GRIDS=None,
    SKIP_PAN=None,
    SKIP_AEROSOL=None,
    SKIP_CIRRUS=None,
    CALIBRATION=None,
    DATA_TYPE=None,
    TEMP_UNIT=None,
    PROJECTION=None,
    RESAMPLING=None,
    UTM_ZONE=None,
    UTM_SOUTH=None,
    EXTENT=None,
    EXTENT_XMIN=None,
    EXTENT_XMAX=None,
    EXTENT_YMIN=None,
    EXTENT_YMAX=None,
    EXTENT_GRID=None,
    EXTENT_BUFFER=None, Verbose=2):
    '''
    Import Landsat Scene
    ----------
    [imagery_tools.14]\n
    Import Landsat scenes including metadata from Landsat metadata files. Band data have to be stored in the same folder as the chosen metadata file in uncompressed GeoTIFF format.\n
    Arguments
    ----------
    - EXTENT_SHAPES [`input shapes`] : Shapes Extent
    - BANDS_SPECTRAL [`output grid list`] : Spectral Bands
    - BANDS_THERMAL [`output grid list`] : Thermal Bands
    - BANDS_AUXILIARY [`output grid list`] : Auxiliary Bands
    - BAND_INFO [`output table`] : Band Info
    - METAFILE [`file path`] : Metadata File
    - MULTI2GRIDS [`boolean`] : Spectral Bands as Grid Collection. Default: 1
    - SKIP_PAN [`boolean`] : Skip Panchromatic Band. Default: 1
    - SKIP_AEROSOL [`boolean`] : Skip Aerosol Band. Default: 1
    - SKIP_CIRRUS [`boolean`] : Skip Cirrus Band. Default: 1
    - CALIBRATION [`choice`] : Radiometric Calibration. Available Choices: [0] digital numbers [1] radiance [2] reflectance Default: 0
    - DATA_TYPE [`choice`] : Output Data Type. Available Choices: [0] integers with scaling [1] floating point numbers Default: 0
    - TEMP_UNIT [`choice`] : Temperature Unit. Available Choices: [0] Kelvin [1] Celsius Default: 0
    - PROJECTION [`choice`] : Coordinate System. Available Choices: [0] UTM North [1] UTM South [2] Different UTM Zone [3] Geographic Coordinates Default: 0 If using the extent option in combination with 'Different UTM Zone' or 'Geographic Coordinates' extent is expected to be defined with UTM North coordinates.
    - RESAMPLING [`choice`] : Resampling. Available Choices: [0] Nearest Neighbour [1] Bilinear Interpolation [2] Bicubic Spline Interpolation [3] B-Spline Interpolation Default: 3
    - UTM_ZONE [`integer number`] : Zone. Minimum: 1 Maximum: 60 Default: 32
    - UTM_SOUTH [`boolean`] : South. Default: 0
    - EXTENT [`choice`] : Extent. Available Choices: [0] original [1] user defined [2] grid system [3] shapes extent Default: 0
    - EXTENT_XMIN [`floating point number`] : Left. Default: 0.000000
    - EXTENT_XMAX [`floating point number`] : Right. Default: 0.000000
    - EXTENT_YMIN [`floating point number`] : Bottom. Default: 0.000000
    - EXTENT_YMAX [`floating point number`] : Top. Default: 0.000000
    - EXTENT_GRID [`grid system`] : Grid System
    - EXTENT_BUFFER [`floating point number`] : Buffer. Minimum: 0.000000 Default: 0.000000

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '14', 'Import Landsat Scene')
    if Tool.is_Okay():
        Tool.Set_Input ('EXTENT_SHAPES', EXTENT_SHAPES)
        Tool.Set_Output('BANDS_SPECTRAL', BANDS_SPECTRAL)
        Tool.Set_Output('BANDS_THERMAL', BANDS_THERMAL)
        Tool.Set_Output('BANDS_AUXILIARY', BANDS_AUXILIARY)
        Tool.Set_Output('BAND_INFO', BAND_INFO)
        Tool.Set_Option('METAFILE', METAFILE)
        Tool.Set_Option('MULTI2GRIDS', MULTI2GRIDS)
        Tool.Set_Option('SKIP_PAN', SKIP_PAN)
        Tool.Set_Option('SKIP_AEROSOL', SKIP_AEROSOL)
        Tool.Set_Option('SKIP_CIRRUS', SKIP_CIRRUS)
        Tool.Set_Option('CALIBRATION', CALIBRATION)
        Tool.Set_Option('DATA_TYPE', DATA_TYPE)
        Tool.Set_Option('TEMP_UNIT', TEMP_UNIT)
        Tool.Set_Option('PROJECTION', PROJECTION)
        Tool.Set_Option('RESAMPLING', RESAMPLING)
        Tool.Set_Option('UTM_ZONE', UTM_ZONE)
        Tool.Set_Option('UTM_SOUTH', UTM_SOUTH)
        Tool.Set_Option('EXTENT', EXTENT)
        Tool.Set_Option('EXTENT_XMIN', EXTENT_XMIN)
        Tool.Set_Option('EXTENT_XMAX', EXTENT_XMAX)
        Tool.Set_Option('EXTENT_YMIN', EXTENT_YMIN)
        Tool.Set_Option('EXTENT_YMAX', EXTENT_YMAX)
        Tool.Set_Option('EXTENT_GRID', EXTENT_GRID)
        Tool.Set_Option('EXTENT_BUFFER', EXTENT_BUFFER)
        return Tool.Execute(Verbose)
    return False

def run_tool_imagery_tools_14(
    EXTENT_SHAPES=None,
    BANDS_SPECTRAL=None,
    BANDS_THERMAL=None,
    BANDS_AUXILIARY=None,
    BAND_INFO=None,
    METAFILE=None,
    MULTI2GRIDS=None,
    SKIP_PAN=None,
    SKIP_AEROSOL=None,
    SKIP_CIRRUS=None,
    CALIBRATION=None,
    DATA_TYPE=None,
    TEMP_UNIT=None,
    PROJECTION=None,
    RESAMPLING=None,
    UTM_ZONE=None,
    UTM_SOUTH=None,
    EXTENT=None,
    EXTENT_XMIN=None,
    EXTENT_XMAX=None,
    EXTENT_YMIN=None,
    EXTENT_YMAX=None,
    EXTENT_GRID=None,
    EXTENT_BUFFER=None, Verbose=2):
    '''
    Import Landsat Scene
    ----------
    [imagery_tools.14]\n
    Import Landsat scenes including metadata from Landsat metadata files. Band data have to be stored in the same folder as the chosen metadata file in uncompressed GeoTIFF format.\n
    Arguments
    ----------
    - EXTENT_SHAPES [`input shapes`] : Shapes Extent
    - BANDS_SPECTRAL [`output grid list`] : Spectral Bands
    - BANDS_THERMAL [`output grid list`] : Thermal Bands
    - BANDS_AUXILIARY [`output grid list`] : Auxiliary Bands
    - BAND_INFO [`output table`] : Band Info
    - METAFILE [`file path`] : Metadata File
    - MULTI2GRIDS [`boolean`] : Spectral Bands as Grid Collection. Default: 1
    - SKIP_PAN [`boolean`] : Skip Panchromatic Band. Default: 1
    - SKIP_AEROSOL [`boolean`] : Skip Aerosol Band. Default: 1
    - SKIP_CIRRUS [`boolean`] : Skip Cirrus Band. Default: 1
    - CALIBRATION [`choice`] : Radiometric Calibration. Available Choices: [0] digital numbers [1] radiance [2] reflectance Default: 0
    - DATA_TYPE [`choice`] : Output Data Type. Available Choices: [0] integers with scaling [1] floating point numbers Default: 0
    - TEMP_UNIT [`choice`] : Temperature Unit. Available Choices: [0] Kelvin [1] Celsius Default: 0
    - PROJECTION [`choice`] : Coordinate System. Available Choices: [0] UTM North [1] UTM South [2] Different UTM Zone [3] Geographic Coordinates Default: 0 If using the extent option in combination with 'Different UTM Zone' or 'Geographic Coordinates' extent is expected to be defined with UTM North coordinates.
    - RESAMPLING [`choice`] : Resampling. Available Choices: [0] Nearest Neighbour [1] Bilinear Interpolation [2] Bicubic Spline Interpolation [3] B-Spline Interpolation Default: 3
    - UTM_ZONE [`integer number`] : Zone. Minimum: 1 Maximum: 60 Default: 32
    - UTM_SOUTH [`boolean`] : South. Default: 0
    - EXTENT [`choice`] : Extent. Available Choices: [0] original [1] user defined [2] grid system [3] shapes extent Default: 0
    - EXTENT_XMIN [`floating point number`] : Left. Default: 0.000000
    - EXTENT_XMAX [`floating point number`] : Right. Default: 0.000000
    - EXTENT_YMIN [`floating point number`] : Bottom. Default: 0.000000
    - EXTENT_YMAX [`floating point number`] : Top. Default: 0.000000
    - EXTENT_GRID [`grid system`] : Grid System
    - EXTENT_BUFFER [`floating point number`] : Buffer. Minimum: 0.000000 Default: 0.000000

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '14', 'Import Landsat Scene')
    if Tool.is_Okay():
        Tool.Set_Input ('EXTENT_SHAPES', EXTENT_SHAPES)
        Tool.Set_Output('BANDS_SPECTRAL', BANDS_SPECTRAL)
        Tool.Set_Output('BANDS_THERMAL', BANDS_THERMAL)
        Tool.Set_Output('BANDS_AUXILIARY', BANDS_AUXILIARY)
        Tool.Set_Output('BAND_INFO', BAND_INFO)
        Tool.Set_Option('METAFILE', METAFILE)
        Tool.Set_Option('MULTI2GRIDS', MULTI2GRIDS)
        Tool.Set_Option('SKIP_PAN', SKIP_PAN)
        Tool.Set_Option('SKIP_AEROSOL', SKIP_AEROSOL)
        Tool.Set_Option('SKIP_CIRRUS', SKIP_CIRRUS)
        Tool.Set_Option('CALIBRATION', CALIBRATION)
        Tool.Set_Option('DATA_TYPE', DATA_TYPE)
        Tool.Set_Option('TEMP_UNIT', TEMP_UNIT)
        Tool.Set_Option('PROJECTION', PROJECTION)
        Tool.Set_Option('RESAMPLING', RESAMPLING)
        Tool.Set_Option('UTM_ZONE', UTM_ZONE)
        Tool.Set_Option('UTM_SOUTH', UTM_SOUTH)
        Tool.Set_Option('EXTENT', EXTENT)
        Tool.Set_Option('EXTENT_XMIN', EXTENT_XMIN)
        Tool.Set_Option('EXTENT_XMAX', EXTENT_XMAX)
        Tool.Set_Option('EXTENT_YMIN', EXTENT_YMIN)
        Tool.Set_Option('EXTENT_YMAX', EXTENT_YMAX)
        Tool.Set_Option('EXTENT_GRID', EXTENT_GRID)
        Tool.Set_Option('EXTENT_BUFFER', EXTENT_BUFFER)
        return Tool.Execute(Verbose)
    return False

def Import_Sentinel2_Scene(
    EXTENT_SHAPES=None,
    BANDS=None,
    BANDS_AUX=None,
    METAFILE=None,
    MULTI2GRIDS=None,
    LOAD_60M=None,
    LOAD_AOT=None,
    LOAD_WVP=None,
    LOAD_SCL=None,
    REFLECTANCE=None,
    RESOLUTION=None,
    PROJECTION=None,
    RESAMPLING=None,
    UTM_ZONE=None,
    UTM_SOUTH=None,
    EXTENT=None,
    EXTENT_XMIN=None,
    EXTENT_XMAX=None,
    EXTENT_YMIN=None,
    EXTENT_YMAX=None,
    EXTENT_GRID=None,
    EXTENT_BUFFER=None, Verbose=2):
    '''
    Import Sentinel-2 Scene
    ----------
    [imagery_tools.15]\n
    Import Sentinel-2 scenes from a folder structure as provided by the original ESA download.\n
    Arguments
    ----------
    - EXTENT_SHAPES [`input shapes`] : Shapes Extent
    - BANDS [`output grid list`] : Spectral Bands
    - BANDS_AUX [`output grid list`] : Auxiliary Bands
    - METAFILE [`file path`] : Metadata File
    - MULTI2GRIDS [`boolean`] : Spectral Bands as Grid Collection. Default: 1
    - LOAD_60M [`boolean`] : Aerosol, Vapour, Cirrus. Default: 0
    - LOAD_AOT [`boolean`] : Aerosol Optical Thickness. Default: 0
    - LOAD_WVP [`boolean`] : Water Vapour. Default: 0
    - LOAD_SCL [`boolean`] : Scene Classification. Default: 0
    - REFLECTANCE [`choice`] : Reflectance Values. Available Choices: [0] digital number [1] fraction Default: 1
    - RESOLUTION [`choice`] : Resolution. Available Choices: [0] original [1] 10m [2] 20m Default: 0 Allows to resample all spectral bands to the same target resolution, either 10 or 20m.
    - PROJECTION [`choice`] : Coordinate System. Available Choices: [0] original [1] UTM North [2] UTM South [3] Different UTM Zone [4] Geographic Coordinates Default: 0 If using the extent option in combination with 'Different UTM Zone' or 'Geographic Coordinates' extent is expected to be defined with UTM North coordinates.
    - RESAMPLING [`choice`] : Resampling. Available Choices: [0] Nearest Neighbour [1] Bilinear Interpolation [2] Bicubic Spline Interpolation [3] B-Spline Interpolation Default: 3
    - UTM_ZONE [`integer number`] : Zone. Minimum: 1 Maximum: 60 Default: 32
    - UTM_SOUTH [`boolean`] : South. Default: 0
    - EXTENT [`choice`] : Extent. Available Choices: [0] original [1] user defined [2] grid system [3] shapes extent Default: 0
    - EXTENT_XMIN [`floating point number`] : Left. Default: 0.000000
    - EXTENT_XMAX [`floating point number`] : Right. Default: 0.000000
    - EXTENT_YMIN [`floating point number`] : Bottom. Default: 0.000000
    - EXTENT_YMAX [`floating point number`] : Top. Default: 0.000000
    - EXTENT_GRID [`grid system`] : Grid System
    - EXTENT_BUFFER [`floating point number`] : Buffer. Minimum: 0.000000 Default: 0.000000

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '15', 'Import Sentinel-2 Scene')
    if Tool.is_Okay():
        Tool.Set_Input ('EXTENT_SHAPES', EXTENT_SHAPES)
        Tool.Set_Output('BANDS', BANDS)
        Tool.Set_Output('BANDS_AUX', BANDS_AUX)
        Tool.Set_Option('METAFILE', METAFILE)
        Tool.Set_Option('MULTI2GRIDS', MULTI2GRIDS)
        Tool.Set_Option('LOAD_60M', LOAD_60M)
        Tool.Set_Option('LOAD_AOT', LOAD_AOT)
        Tool.Set_Option('LOAD_WVP', LOAD_WVP)
        Tool.Set_Option('LOAD_SCL', LOAD_SCL)
        Tool.Set_Option('REFLECTANCE', REFLECTANCE)
        Tool.Set_Option('RESOLUTION', RESOLUTION)
        Tool.Set_Option('PROJECTION', PROJECTION)
        Tool.Set_Option('RESAMPLING', RESAMPLING)
        Tool.Set_Option('UTM_ZONE', UTM_ZONE)
        Tool.Set_Option('UTM_SOUTH', UTM_SOUTH)
        Tool.Set_Option('EXTENT', EXTENT)
        Tool.Set_Option('EXTENT_XMIN', EXTENT_XMIN)
        Tool.Set_Option('EXTENT_XMAX', EXTENT_XMAX)
        Tool.Set_Option('EXTENT_YMIN', EXTENT_YMIN)
        Tool.Set_Option('EXTENT_YMAX', EXTENT_YMAX)
        Tool.Set_Option('EXTENT_GRID', EXTENT_GRID)
        Tool.Set_Option('EXTENT_BUFFER', EXTENT_BUFFER)
        return Tool.Execute(Verbose)
    return False

def run_tool_imagery_tools_15(
    EXTENT_SHAPES=None,
    BANDS=None,
    BANDS_AUX=None,
    METAFILE=None,
    MULTI2GRIDS=None,
    LOAD_60M=None,
    LOAD_AOT=None,
    LOAD_WVP=None,
    LOAD_SCL=None,
    REFLECTANCE=None,
    RESOLUTION=None,
    PROJECTION=None,
    RESAMPLING=None,
    UTM_ZONE=None,
    UTM_SOUTH=None,
    EXTENT=None,
    EXTENT_XMIN=None,
    EXTENT_XMAX=None,
    EXTENT_YMIN=None,
    EXTENT_YMAX=None,
    EXTENT_GRID=None,
    EXTENT_BUFFER=None, Verbose=2):
    '''
    Import Sentinel-2 Scene
    ----------
    [imagery_tools.15]\n
    Import Sentinel-2 scenes from a folder structure as provided by the original ESA download.\n
    Arguments
    ----------
    - EXTENT_SHAPES [`input shapes`] : Shapes Extent
    - BANDS [`output grid list`] : Spectral Bands
    - BANDS_AUX [`output grid list`] : Auxiliary Bands
    - METAFILE [`file path`] : Metadata File
    - MULTI2GRIDS [`boolean`] : Spectral Bands as Grid Collection. Default: 1
    - LOAD_60M [`boolean`] : Aerosol, Vapour, Cirrus. Default: 0
    - LOAD_AOT [`boolean`] : Aerosol Optical Thickness. Default: 0
    - LOAD_WVP [`boolean`] : Water Vapour. Default: 0
    - LOAD_SCL [`boolean`] : Scene Classification. Default: 0
    - REFLECTANCE [`choice`] : Reflectance Values. Available Choices: [0] digital number [1] fraction Default: 1
    - RESOLUTION [`choice`] : Resolution. Available Choices: [0] original [1] 10m [2] 20m Default: 0 Allows to resample all spectral bands to the same target resolution, either 10 or 20m.
    - PROJECTION [`choice`] : Coordinate System. Available Choices: [0] original [1] UTM North [2] UTM South [3] Different UTM Zone [4] Geographic Coordinates Default: 0 If using the extent option in combination with 'Different UTM Zone' or 'Geographic Coordinates' extent is expected to be defined with UTM North coordinates.
    - RESAMPLING [`choice`] : Resampling. Available Choices: [0] Nearest Neighbour [1] Bilinear Interpolation [2] Bicubic Spline Interpolation [3] B-Spline Interpolation Default: 3
    - UTM_ZONE [`integer number`] : Zone. Minimum: 1 Maximum: 60 Default: 32
    - UTM_SOUTH [`boolean`] : South. Default: 0
    - EXTENT [`choice`] : Extent. Available Choices: [0] original [1] user defined [2] grid system [3] shapes extent Default: 0
    - EXTENT_XMIN [`floating point number`] : Left. Default: 0.000000
    - EXTENT_XMAX [`floating point number`] : Right. Default: 0.000000
    - EXTENT_YMIN [`floating point number`] : Bottom. Default: 0.000000
    - EXTENT_YMAX [`floating point number`] : Top. Default: 0.000000
    - EXTENT_GRID [`grid system`] : Grid System
    - EXTENT_BUFFER [`floating point number`] : Buffer. Minimum: 0.000000 Default: 0.000000

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '15', 'Import Sentinel-2 Scene')
    if Tool.is_Okay():
        Tool.Set_Input ('EXTENT_SHAPES', EXTENT_SHAPES)
        Tool.Set_Output('BANDS', BANDS)
        Tool.Set_Output('BANDS_AUX', BANDS_AUX)
        Tool.Set_Option('METAFILE', METAFILE)
        Tool.Set_Option('MULTI2GRIDS', MULTI2GRIDS)
        Tool.Set_Option('LOAD_60M', LOAD_60M)
        Tool.Set_Option('LOAD_AOT', LOAD_AOT)
        Tool.Set_Option('LOAD_WVP', LOAD_WVP)
        Tool.Set_Option('LOAD_SCL', LOAD_SCL)
        Tool.Set_Option('REFLECTANCE', REFLECTANCE)
        Tool.Set_Option('RESOLUTION', RESOLUTION)
        Tool.Set_Option('PROJECTION', PROJECTION)
        Tool.Set_Option('RESAMPLING', RESAMPLING)
        Tool.Set_Option('UTM_ZONE', UTM_ZONE)
        Tool.Set_Option('UTM_SOUTH', UTM_SOUTH)
        Tool.Set_Option('EXTENT', EXTENT)
        Tool.Set_Option('EXTENT_XMIN', EXTENT_XMIN)
        Tool.Set_Option('EXTENT_XMAX', EXTENT_XMAX)
        Tool.Set_Option('EXTENT_YMIN', EXTENT_YMIN)
        Tool.Set_Option('EXTENT_YMAX', EXTENT_YMAX)
        Tool.Set_Option('EXTENT_GRID', EXTENT_GRID)
        Tool.Set_Option('EXTENT_BUFFER', EXTENT_BUFFER)
        return Tool.Execute(Verbose)
    return False

def Import_Sentinel3_OLCI_Scene(
    BANDS=None,
    DIRECTORY=None,
    RESOLUTION=None,
    COLLECTION=None,
    CRS_STRING=None, Verbose=2):
    '''
    Import Sentinel-3 OLCI Scene
    ----------
    [imagery_tools.16]\n
    Import Sentinel-3 OLCI (Ocean and Land Colour Instrument) scenes from a folder structure as provided by the original ESA download.\n
    Arguments
    ----------
    - BANDS [`output grid list`] : Bands
    - DIRECTORY [`file path`] : Directory
    - RESOLUTION [`floating point number`] : Target Resolution. Minimum: 0.001000 Default: 0.002778
    - COLLECTION [`boolean`] : Bands as Grid Collection. Default: 1
    - CRS_STRING [`text`] : Coordinate System Definition. Supported formats comprise PROJ and WKT strings, object codes (e.g. "EPSG:4326").

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '16', 'Import Sentinel-3 OLCI Scene')
    if Tool.is_Okay():
        Tool.Set_Output('BANDS', BANDS)
        Tool.Set_Option('DIRECTORY', DIRECTORY)
        Tool.Set_Option('RESOLUTION', RESOLUTION)
        Tool.Set_Option('COLLECTION', COLLECTION)
        Tool.Set_Option('CRS_STRING', CRS_STRING)
        return Tool.Execute(Verbose)
    return False

def run_tool_imagery_tools_16(
    BANDS=None,
    DIRECTORY=None,
    RESOLUTION=None,
    COLLECTION=None,
    CRS_STRING=None, Verbose=2):
    '''
    Import Sentinel-3 OLCI Scene
    ----------
    [imagery_tools.16]\n
    Import Sentinel-3 OLCI (Ocean and Land Colour Instrument) scenes from a folder structure as provided by the original ESA download.\n
    Arguments
    ----------
    - BANDS [`output grid list`] : Bands
    - DIRECTORY [`file path`] : Directory
    - RESOLUTION [`floating point number`] : Target Resolution. Minimum: 0.001000 Default: 0.002778
    - COLLECTION [`boolean`] : Bands as Grid Collection. Default: 1
    - CRS_STRING [`text`] : Coordinate System Definition. Supported formats comprise PROJ and WKT strings, object codes (e.g. "EPSG:4326").

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '16', 'Import Sentinel-3 OLCI Scene')
    if Tool.is_Okay():
        Tool.Set_Output('BANDS', BANDS)
        Tool.Set_Option('DIRECTORY', DIRECTORY)
        Tool.Set_Option('RESOLUTION', RESOLUTION)
        Tool.Set_Option('COLLECTION', COLLECTION)
        Tool.Set_Option('CRS_STRING', CRS_STRING)
        return Tool.Execute(Verbose)
    return False

def Spectral_Profile(
    BANDS=None,
    LOCATION=None,
    PROFILE=None,
    SHOW=None,
    RESAMPLING=None,
    LENGTHS=None, Verbose=2):
    '''
    Spectral Profile
    ----------
    [imagery_tools.17]\n
    Spectral Profile\n
    Arguments
    ----------
    - BANDS [`input grid list`] : Spectral Bands
    - LOCATION [`input shapes`] : Profile Location
    - PROFILE [`output table`] : Spectral Profile
    - SHOW [`boolean`] : Show Diagram. Default: 0
    - RESAMPLING [`choice`] : Resampling. Available Choices: [0] Nearest Neighbour [1] Bilinear Interpolation [2] Bicubic Spline Interpolation [3] B-Spline Interpolation Default: 3
    - LENGTHS [`text`] : Wave Lengths. Default: 0.485 0.56 0.66 0.83 1.65 2.215 11.45 Space separated wave lengths ordered corresponding to the bands in input list. If empty a simple enumeration will be used instead.

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '17', 'Spectral Profile')
    if Tool.is_Okay():
        Tool.Set_Input ('BANDS', BANDS)
        Tool.Set_Input ('LOCATION', LOCATION)
        Tool.Set_Output('PROFILE', PROFILE)
        Tool.Set_Option('SHOW', SHOW)
        Tool.Set_Option('RESAMPLING', RESAMPLING)
        Tool.Set_Option('LENGTHS', LENGTHS)
        return Tool.Execute(Verbose)
    return False

def run_tool_imagery_tools_17(
    BANDS=None,
    LOCATION=None,
    PROFILE=None,
    SHOW=None,
    RESAMPLING=None,
    LENGTHS=None, Verbose=2):
    '''
    Spectral Profile
    ----------
    [imagery_tools.17]\n
    Spectral Profile\n
    Arguments
    ----------
    - BANDS [`input grid list`] : Spectral Bands
    - LOCATION [`input shapes`] : Profile Location
    - PROFILE [`output table`] : Spectral Profile
    - SHOW [`boolean`] : Show Diagram. Default: 0
    - RESAMPLING [`choice`] : Resampling. Available Choices: [0] Nearest Neighbour [1] Bilinear Interpolation [2] Bicubic Spline Interpolation [3] B-Spline Interpolation Default: 3
    - LENGTHS [`text`] : Wave Lengths. Default: 0.485 0.56 0.66 0.83 1.65 2.215 11.45 Space separated wave lengths ordered corresponding to the bands in input list. If empty a simple enumeration will be used instead.

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '17', 'Spectral Profile')
    if Tool.is_Okay():
        Tool.Set_Input ('BANDS', BANDS)
        Tool.Set_Input ('LOCATION', LOCATION)
        Tool.Set_Output('PROFILE', PROFILE)
        Tool.Set_Option('SHOW', SHOW)
        Tool.Set_Option('RESAMPLING', RESAMPLING)
        Tool.Set_Option('LENGTHS', LENGTHS)
        return Tool.Execute(Verbose)
    return False

def Topographic_Correction(
    DEM=None,
    BANDS=None,
    CORRECTED=None,
    DEM_GRIDSYSTEM=None,
    AZIMUTH=None,
    HEIGHT=None,
    METHOD=None,
    MINNAERT=None,
    MAXCELLS=None,
    LIMIT=None,
    LIMIT_MIN=None,
    LIMIT_MAX=None, Verbose=2):
    '''
    Topographic Correction
    ----------
    [imagery_tools.19]\n
    Topographic correction tools for satellite imagery using a digital elevation model (DEM) and the position of the Sun at acquisition time to estimate and remove shading effects.\n
    Arguments
    ----------
    - DEM [`input grid`] : Elevation
    - BANDS [`input grid list`] : Bands
    - CORRECTED [`output grid list`] : Corrected Bands
    - DEM_GRIDSYSTEM [`grid system`] : Grid system
    - AZIMUTH [`floating point number`] : Azimuth. Minimum: 0.000000 Maximum: 360.000000 Default: 180.000000 direction of sun (degree, clockwise from North)
    - HEIGHT [`floating point number`] : Height. Minimum: 0.000000 Maximum: 90.000000 Default: 45.000000 height of sun above horizon (degree)
    - METHOD [`choice`] : Method. Available Choices: [0] Cosine Correction (Teillet et al. 1982) [1] Cosine Correction (Civco 1989) [2] Minnaert Correction [3] Minnaert Correction with Slope (Riano et al. 2003) [4] Minnaert Correction with Slope (Law & Nichol 2004) [5] C Correction [6] Normalization (after Civco, modified by Law & Nichol) Default: 2
    - MINNAERT [`floating point number`] : Minnaert Correction. Minimum: 0.000000 Maximum: 1.000000 Default: 0.500000
    - MAXCELLS [`integer number`] : Maximum Number of Cells. Minimum: 1000 Default: 100000 Maximum number of grid cells used for trend analysis as required by C correction.
    - LIMIT [`boolean`] : Limit Value Range. Default: 0 Avoid over and undershootings specifying value range limits.
    - LIMIT_MIN [`floating point number`] : Minimum. Default: 1.000000
    - LIMIT_MAX [`floating point number`] : Maximum. Default: 255.000000

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '19', 'Topographic Correction')
    if Tool.is_Okay():
        Tool.Set_Input ('DEM', DEM)
        Tool.Set_Input ('BANDS', BANDS)
        Tool.Set_Output('CORRECTED', CORRECTED)
        Tool.Set_Option('DEM_GRIDSYSTEM', DEM_GRIDSYSTEM)
        Tool.Set_Option('AZIMUTH', AZIMUTH)
        Tool.Set_Option('HEIGHT', HEIGHT)
        Tool.Set_Option('METHOD', METHOD)
        Tool.Set_Option('MINNAERT', MINNAERT)
        Tool.Set_Option('MAXCELLS', MAXCELLS)
        Tool.Set_Option('LIMIT', LIMIT)
        Tool.Set_Option('LIMIT_MIN', LIMIT_MIN)
        Tool.Set_Option('LIMIT_MAX', LIMIT_MAX)
        return Tool.Execute(Verbose)
    return False

def run_tool_imagery_tools_19(
    DEM=None,
    BANDS=None,
    CORRECTED=None,
    DEM_GRIDSYSTEM=None,
    AZIMUTH=None,
    HEIGHT=None,
    METHOD=None,
    MINNAERT=None,
    MAXCELLS=None,
    LIMIT=None,
    LIMIT_MIN=None,
    LIMIT_MAX=None, Verbose=2):
    '''
    Topographic Correction
    ----------
    [imagery_tools.19]\n
    Topographic correction tools for satellite imagery using a digital elevation model (DEM) and the position of the Sun at acquisition time to estimate and remove shading effects.\n
    Arguments
    ----------
    - DEM [`input grid`] : Elevation
    - BANDS [`input grid list`] : Bands
    - CORRECTED [`output grid list`] : Corrected Bands
    - DEM_GRIDSYSTEM [`grid system`] : Grid system
    - AZIMUTH [`floating point number`] : Azimuth. Minimum: 0.000000 Maximum: 360.000000 Default: 180.000000 direction of sun (degree, clockwise from North)
    - HEIGHT [`floating point number`] : Height. Minimum: 0.000000 Maximum: 90.000000 Default: 45.000000 height of sun above horizon (degree)
    - METHOD [`choice`] : Method. Available Choices: [0] Cosine Correction (Teillet et al. 1982) [1] Cosine Correction (Civco 1989) [2] Minnaert Correction [3] Minnaert Correction with Slope (Riano et al. 2003) [4] Minnaert Correction with Slope (Law & Nichol 2004) [5] C Correction [6] Normalization (after Civco, modified by Law & Nichol) Default: 2
    - MINNAERT [`floating point number`] : Minnaert Correction. Minimum: 0.000000 Maximum: 1.000000 Default: 0.500000
    - MAXCELLS [`integer number`] : Maximum Number of Cells. Minimum: 1000 Default: 100000 Maximum number of grid cells used for trend analysis as required by C correction.
    - LIMIT [`boolean`] : Limit Value Range. Default: 0 Avoid over and undershootings specifying value range limits.
    - LIMIT_MIN [`floating point number`] : Minimum. Default: 1.000000
    - LIMIT_MAX [`floating point number`] : Maximum. Default: 255.000000

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '19', 'Topographic Correction')
    if Tool.is_Okay():
        Tool.Set_Input ('DEM', DEM)
        Tool.Set_Input ('BANDS', BANDS)
        Tool.Set_Output('CORRECTED', CORRECTED)
        Tool.Set_Option('DEM_GRIDSYSTEM', DEM_GRIDSYSTEM)
        Tool.Set_Option('AZIMUTH', AZIMUTH)
        Tool.Set_Option('HEIGHT', HEIGHT)
        Tool.Set_Option('METHOD', METHOD)
        Tool.Set_Option('MINNAERT', MINNAERT)
        Tool.Set_Option('MAXCELLS', MAXCELLS)
        Tool.Set_Option('LIMIT', LIMIT)
        Tool.Set_Option('LIMIT_MIN', LIMIT_MIN)
        Tool.Set_Option('LIMIT_MAX', LIMIT_MAX)
        return Tool.Execute(Verbose)
    return False

def Cloud_Detection(
    BAND_BLUE=None,
    BAND_GREEN=None,
    BAND_RED=None,
    BAND_NIR=None,
    BAND_SWIR1=None,
    BAND_SWIR2=None,
    BAND_THERMAL=None,
    BAND_CIRRUS=None,
    CLOUDS=None,
    BANDS_SWIR=None,
    BANDS_THERMAL=None,
    BANDS_CIRRUS=None,
    THERMAL_UNIT=None,
    ALGORITHM=None,
    ACCA_B56C=None,
    ACCA_B45R=None,
    ACCA_HIST_N=None,
    ACCA_CSIG=None,
    ACCA_PASS2=None,
    ACCA_SHADOW=None,
    SHADOWS=None,
    SUN_AZIMUTH=None,
    SUN_HEIGHT=None, Verbose=2):
    '''
    Cloud Detection
    ----------
    [imagery_tools.20]\n
    This tool implements pass one of the Function of mask (Fmask) algorithm for cloud and cloud shadow detection in Landsat imagery. Landsat Top of Atmosphere (TOA) reflectance and Brightness Temperature (BT) are used as input.\n
    Alternatively you can choose the scene-average automated cloud-cover assessment (ACCA) algorithm as proposed by Irish (2000) and Irish et al. (2006).\n
    This tool can optionally pass the cloud mask to the "Cloud Shadow Detection" tool as well.\n
    Arguments
    ----------
    - BAND_BLUE [`input grid`] : Blue
    - BAND_GREEN [`input grid`] : Green
    - BAND_RED [`input grid`] : Red
    - BAND_NIR [`input grid`] : Near Infrared
    - BAND_SWIR1 [`input grid`] : Shortwave Infrared 1
    - BAND_SWIR2 [`input grid`] : Shortwave Infrared 2
    - BAND_THERMAL [`optional input grid`] : Thermal
    - BAND_CIRRUS [`optional input grid`] : Cirrus
    - CLOUDS [`output grid`] : Clouds
    - BANDS_SWIR [`grid system`] : Grid System
    - BANDS_THERMAL [`grid system`] : Grid System
    - BANDS_CIRRUS [`grid system`] : Grid System
    - THERMAL_UNIT [`choice`] : Unit. Available Choices: [0] Kelvin [1] Celsius Default: 0
    - ALGORITHM [`choice`] : Algorithm. Available Choices: [0] Fmask [1] ACCA Default: 0
    - ACCA_B56C [`floating point number`] : SWIR/Thermal Threshold. Default: 225.000000 Threshold for SWIR/Thermal Composite (step 6).
    - ACCA_B45R [`floating point number`] : Desert Detection Threshold. Default: 1.000000 Threshold for desert detection (step 10,  NIR/SWIR Ratio).
    - ACCA_HIST_N [`integer number`] : Temperature Histogram. Minimum: 10 Default: 100 Number of classes in the cloud temperature histogram.
    - ACCA_CSIG [`boolean`] : Cloud Signature. Default: 1 Always use cloud signature (step 14).
    - ACCA_PASS2 [`boolean`] : Cloud Differentiation. Default: 0 Differentiate between warm (not ambiguous) and cold clouds.
    - ACCA_SHADOW [`boolean`] : Shadows. Default: 0 Include a category for cloud shadows.
    - SHADOWS [`boolean`] : Shadow Detection. Default: 0 Run cloud shadow detection tool with standard settings.
    - SUN_AZIMUTH [`floating point number`] : Sun's Azimuth. Minimum: 0.000000 Maximum: 360.000000 Default: -180.000000 Direction of sun clockwise from North [degree].
    - SUN_HEIGHT [`floating point number`] : Sun's Height. Minimum: 0.000000 Maximum: 90.000000 Default: 45.000000 Height of sun above horizon [degree].

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '20', 'Cloud Detection')
    if Tool.is_Okay():
        Tool.Set_Input ('BAND_BLUE', BAND_BLUE)
        Tool.Set_Input ('BAND_GREEN', BAND_GREEN)
        Tool.Set_Input ('BAND_RED', BAND_RED)
        Tool.Set_Input ('BAND_NIR', BAND_NIR)
        Tool.Set_Input ('BAND_SWIR1', BAND_SWIR1)
        Tool.Set_Input ('BAND_SWIR2', BAND_SWIR2)
        Tool.Set_Input ('BAND_THERMAL', BAND_THERMAL)
        Tool.Set_Input ('BAND_CIRRUS', BAND_CIRRUS)
        Tool.Set_Output('CLOUDS', CLOUDS)
        Tool.Set_Option('BANDS_SWIR', BANDS_SWIR)
        Tool.Set_Option('BANDS_THERMAL', BANDS_THERMAL)
        Tool.Set_Option('BANDS_CIRRUS', BANDS_CIRRUS)
        Tool.Set_Option('THERMAL_UNIT', THERMAL_UNIT)
        Tool.Set_Option('ALGORITHM', ALGORITHM)
        Tool.Set_Option('ACCA_B56C', ACCA_B56C)
        Tool.Set_Option('ACCA_B45R', ACCA_B45R)
        Tool.Set_Option('ACCA_HIST_N', ACCA_HIST_N)
        Tool.Set_Option('ACCA_CSIG', ACCA_CSIG)
        Tool.Set_Option('ACCA_PASS2', ACCA_PASS2)
        Tool.Set_Option('ACCA_SHADOW', ACCA_SHADOW)
        Tool.Set_Option('SHADOWS', SHADOWS)
        Tool.Set_Option('SUN_AZIMUTH', SUN_AZIMUTH)
        Tool.Set_Option('SUN_HEIGHT', SUN_HEIGHT)
        return Tool.Execute(Verbose)
    return False

def run_tool_imagery_tools_20(
    BAND_BLUE=None,
    BAND_GREEN=None,
    BAND_RED=None,
    BAND_NIR=None,
    BAND_SWIR1=None,
    BAND_SWIR2=None,
    BAND_THERMAL=None,
    BAND_CIRRUS=None,
    CLOUDS=None,
    BANDS_SWIR=None,
    BANDS_THERMAL=None,
    BANDS_CIRRUS=None,
    THERMAL_UNIT=None,
    ALGORITHM=None,
    ACCA_B56C=None,
    ACCA_B45R=None,
    ACCA_HIST_N=None,
    ACCA_CSIG=None,
    ACCA_PASS2=None,
    ACCA_SHADOW=None,
    SHADOWS=None,
    SUN_AZIMUTH=None,
    SUN_HEIGHT=None, Verbose=2):
    '''
    Cloud Detection
    ----------
    [imagery_tools.20]\n
    This tool implements pass one of the Function of mask (Fmask) algorithm for cloud and cloud shadow detection in Landsat imagery. Landsat Top of Atmosphere (TOA) reflectance and Brightness Temperature (BT) are used as input.\n
    Alternatively you can choose the scene-average automated cloud-cover assessment (ACCA) algorithm as proposed by Irish (2000) and Irish et al. (2006).\n
    This tool can optionally pass the cloud mask to the "Cloud Shadow Detection" tool as well.\n
    Arguments
    ----------
    - BAND_BLUE [`input grid`] : Blue
    - BAND_GREEN [`input grid`] : Green
    - BAND_RED [`input grid`] : Red
    - BAND_NIR [`input grid`] : Near Infrared
    - BAND_SWIR1 [`input grid`] : Shortwave Infrared 1
    - BAND_SWIR2 [`input grid`] : Shortwave Infrared 2
    - BAND_THERMAL [`optional input grid`] : Thermal
    - BAND_CIRRUS [`optional input grid`] : Cirrus
    - CLOUDS [`output grid`] : Clouds
    - BANDS_SWIR [`grid system`] : Grid System
    - BANDS_THERMAL [`grid system`] : Grid System
    - BANDS_CIRRUS [`grid system`] : Grid System
    - THERMAL_UNIT [`choice`] : Unit. Available Choices: [0] Kelvin [1] Celsius Default: 0
    - ALGORITHM [`choice`] : Algorithm. Available Choices: [0] Fmask [1] ACCA Default: 0
    - ACCA_B56C [`floating point number`] : SWIR/Thermal Threshold. Default: 225.000000 Threshold for SWIR/Thermal Composite (step 6).
    - ACCA_B45R [`floating point number`] : Desert Detection Threshold. Default: 1.000000 Threshold for desert detection (step 10,  NIR/SWIR Ratio).
    - ACCA_HIST_N [`integer number`] : Temperature Histogram. Minimum: 10 Default: 100 Number of classes in the cloud temperature histogram.
    - ACCA_CSIG [`boolean`] : Cloud Signature. Default: 1 Always use cloud signature (step 14).
    - ACCA_PASS2 [`boolean`] : Cloud Differentiation. Default: 0 Differentiate between warm (not ambiguous) and cold clouds.
    - ACCA_SHADOW [`boolean`] : Shadows. Default: 0 Include a category for cloud shadows.
    - SHADOWS [`boolean`] : Shadow Detection. Default: 0 Run cloud shadow detection tool with standard settings.
    - SUN_AZIMUTH [`floating point number`] : Sun's Azimuth. Minimum: 0.000000 Maximum: 360.000000 Default: -180.000000 Direction of sun clockwise from North [degree].
    - SUN_HEIGHT [`floating point number`] : Sun's Height. Minimum: 0.000000 Maximum: 90.000000 Default: 45.000000 Height of sun above horizon [degree].

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '20', 'Cloud Detection')
    if Tool.is_Okay():
        Tool.Set_Input ('BAND_BLUE', BAND_BLUE)
        Tool.Set_Input ('BAND_GREEN', BAND_GREEN)
        Tool.Set_Input ('BAND_RED', BAND_RED)
        Tool.Set_Input ('BAND_NIR', BAND_NIR)
        Tool.Set_Input ('BAND_SWIR1', BAND_SWIR1)
        Tool.Set_Input ('BAND_SWIR2', BAND_SWIR2)
        Tool.Set_Input ('BAND_THERMAL', BAND_THERMAL)
        Tool.Set_Input ('BAND_CIRRUS', BAND_CIRRUS)
        Tool.Set_Output('CLOUDS', CLOUDS)
        Tool.Set_Option('BANDS_SWIR', BANDS_SWIR)
        Tool.Set_Option('BANDS_THERMAL', BANDS_THERMAL)
        Tool.Set_Option('BANDS_CIRRUS', BANDS_CIRRUS)
        Tool.Set_Option('THERMAL_UNIT', THERMAL_UNIT)
        Tool.Set_Option('ALGORITHM', ALGORITHM)
        Tool.Set_Option('ACCA_B56C', ACCA_B56C)
        Tool.Set_Option('ACCA_B45R', ACCA_B45R)
        Tool.Set_Option('ACCA_HIST_N', ACCA_HIST_N)
        Tool.Set_Option('ACCA_CSIG', ACCA_CSIG)
        Tool.Set_Option('ACCA_PASS2', ACCA_PASS2)
        Tool.Set_Option('ACCA_SHADOW', ACCA_SHADOW)
        Tool.Set_Option('SHADOWS', SHADOWS)
        Tool.Set_Option('SUN_AZIMUTH', SUN_AZIMUTH)
        Tool.Set_Option('SUN_HEIGHT', SUN_HEIGHT)
        return Tool.Execute(Verbose)
    return False

def Cloud_Shadow_Detection(
    CLOUDS=None,
    CAND_GRID_IN=None,
    BAND_GREEN=None,
    BAND_RED=None,
    BAND_NIR=None,
    BAND_SWIR=None,
    BANDS_BRIGHTNESS=None,
    DEM=None,
    BAND_THERMAL=None,
    CLOUD_ID=None,
    CAND_GRID_OUT=None,
    SHADOWS=None,
    CLOUD_INFO=None,
    BANDS_SWIR=None,
    BANDS_THERMAL=None,
    OUTPUT=None,
    CAND_GRID_VALUE=None,
    THERMAL_UNIT=None,
    CANDIDATES=None,
    BRIGHTNESS=None,
    PROCESSING=None,
    SUN_AZIMUTH=None,
    SUN_HEIGHT=None,
    CLOUD_HEIGHT=None, Verbose=2):
    '''
    Cloud Shadow Detection
    ----------
    [imagery_tools.21]\n
    This tool derives cloud shadows from detected clouds based on their spectral characteristics as well as terrain, sun position and expected cloud height. The initial cloud mask can be created by the tool "Cloud Detection".\n
    Arguments
    ----------
    - CLOUDS [`input grid`] : Clouds
    - CAND_GRID_IN [`input grid`] : Candidates
    - BAND_GREEN [`input grid`] : Green
    - BAND_RED [`input grid`] : Red
    - BAND_NIR [`input grid`] : Near Infrared
    - BAND_SWIR [`input grid`] : Shortwave Infrared
    - BANDS_BRIGHTNESS [`input grid list`] : Brightness Bands
    - DEM [`optional input grid`] : Elevation
    - BAND_THERMAL [`optional input grid`] : Thermal
    - CLOUD_ID [`output grid`] : Cloud Number
    - CAND_GRID_OUT [`output grid`] : Candidates
    - SHADOWS [`output grid`] : Shadows
    - CLOUD_INFO [`output table`] : Cloud Info
    - BANDS_SWIR [`grid system`] : Grid System
    - BANDS_THERMAL [`grid system`] : Grid System
    - OUTPUT [`choice`] : Output. Available Choices: [0] visible shadow [1] full shadow [2] shadow and clouds Default: 2
    - CAND_GRID_VALUE [`floating point number`] : Value. Default: 0.000000
    - THERMAL_UNIT [`choice`] : Unit. Available Choices: [0] Kelvin [1] Celsius Default: 0
    - CANDIDATES [`choice`] : Candidates. Available Choices: [0] all cells of candidates grid that are not no-data [1] all cells of candidates grid with a specified value [2] average brightness threshold [3] Irish [4] Irish modified by Tizado Default: 0
    - BRIGHTNESS [`floating point number`] : Brightness Threshold. Minimum: 0.000000 Maximum: 1.000000 Default: 0.050000
    - PROCESSING [`choice`] : Processing. Available Choices: [0] all clouds at once [1] cloud by cloud Default: 0
    - SUN_AZIMUTH [`floating point number`] : Azimuth. Minimum: 0.000000 Maximum: 360.000000 Default: -180.000000 Direction of sun clockwise from North [degree].
    - SUN_HEIGHT [`floating point number`] : Height. Minimum: 0.000000 Maximum: 90.000000 Default: 45.000000 Height of sun above horizon [degree].
    - CLOUD_HEIGHT [`value range`] : Cloud Height. The range of cloud height above ground to be tested.

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '21', 'Cloud Shadow Detection')
    if Tool.is_Okay():
        Tool.Set_Input ('CLOUDS', CLOUDS)
        Tool.Set_Input ('CAND_GRID_IN', CAND_GRID_IN)
        Tool.Set_Input ('BAND_GREEN', BAND_GREEN)
        Tool.Set_Input ('BAND_RED', BAND_RED)
        Tool.Set_Input ('BAND_NIR', BAND_NIR)
        Tool.Set_Input ('BAND_SWIR', BAND_SWIR)
        Tool.Set_Input ('BANDS_BRIGHTNESS', BANDS_BRIGHTNESS)
        Tool.Set_Input ('DEM', DEM)
        Tool.Set_Input ('BAND_THERMAL', BAND_THERMAL)
        Tool.Set_Output('CLOUD_ID', CLOUD_ID)
        Tool.Set_Output('CAND_GRID_OUT', CAND_GRID_OUT)
        Tool.Set_Output('SHADOWS', SHADOWS)
        Tool.Set_Output('CLOUD_INFO', CLOUD_INFO)
        Tool.Set_Option('BANDS_SWIR', BANDS_SWIR)
        Tool.Set_Option('BANDS_THERMAL', BANDS_THERMAL)
        Tool.Set_Option('OUTPUT', OUTPUT)
        Tool.Set_Option('CAND_GRID_VALUE', CAND_GRID_VALUE)
        Tool.Set_Option('THERMAL_UNIT', THERMAL_UNIT)
        Tool.Set_Option('CANDIDATES', CANDIDATES)
        Tool.Set_Option('BRIGHTNESS', BRIGHTNESS)
        Tool.Set_Option('PROCESSING', PROCESSING)
        Tool.Set_Option('SUN_AZIMUTH', SUN_AZIMUTH)
        Tool.Set_Option('SUN_HEIGHT', SUN_HEIGHT)
        Tool.Set_Option('CLOUD_HEIGHT', CLOUD_HEIGHT)
        return Tool.Execute(Verbose)
    return False

def run_tool_imagery_tools_21(
    CLOUDS=None,
    CAND_GRID_IN=None,
    BAND_GREEN=None,
    BAND_RED=None,
    BAND_NIR=None,
    BAND_SWIR=None,
    BANDS_BRIGHTNESS=None,
    DEM=None,
    BAND_THERMAL=None,
    CLOUD_ID=None,
    CAND_GRID_OUT=None,
    SHADOWS=None,
    CLOUD_INFO=None,
    BANDS_SWIR=None,
    BANDS_THERMAL=None,
    OUTPUT=None,
    CAND_GRID_VALUE=None,
    THERMAL_UNIT=None,
    CANDIDATES=None,
    BRIGHTNESS=None,
    PROCESSING=None,
    SUN_AZIMUTH=None,
    SUN_HEIGHT=None,
    CLOUD_HEIGHT=None, Verbose=2):
    '''
    Cloud Shadow Detection
    ----------
    [imagery_tools.21]\n
    This tool derives cloud shadows from detected clouds based on their spectral characteristics as well as terrain, sun position and expected cloud height. The initial cloud mask can be created by the tool "Cloud Detection".\n
    Arguments
    ----------
    - CLOUDS [`input grid`] : Clouds
    - CAND_GRID_IN [`input grid`] : Candidates
    - BAND_GREEN [`input grid`] : Green
    - BAND_RED [`input grid`] : Red
    - BAND_NIR [`input grid`] : Near Infrared
    - BAND_SWIR [`input grid`] : Shortwave Infrared
    - BANDS_BRIGHTNESS [`input grid list`] : Brightness Bands
    - DEM [`optional input grid`] : Elevation
    - BAND_THERMAL [`optional input grid`] : Thermal
    - CLOUD_ID [`output grid`] : Cloud Number
    - CAND_GRID_OUT [`output grid`] : Candidates
    - SHADOWS [`output grid`] : Shadows
    - CLOUD_INFO [`output table`] : Cloud Info
    - BANDS_SWIR [`grid system`] : Grid System
    - BANDS_THERMAL [`grid system`] : Grid System
    - OUTPUT [`choice`] : Output. Available Choices: [0] visible shadow [1] full shadow [2] shadow and clouds Default: 2
    - CAND_GRID_VALUE [`floating point number`] : Value. Default: 0.000000
    - THERMAL_UNIT [`choice`] : Unit. Available Choices: [0] Kelvin [1] Celsius Default: 0
    - CANDIDATES [`choice`] : Candidates. Available Choices: [0] all cells of candidates grid that are not no-data [1] all cells of candidates grid with a specified value [2] average brightness threshold [3] Irish [4] Irish modified by Tizado Default: 0
    - BRIGHTNESS [`floating point number`] : Brightness Threshold. Minimum: 0.000000 Maximum: 1.000000 Default: 0.050000
    - PROCESSING [`choice`] : Processing. Available Choices: [0] all clouds at once [1] cloud by cloud Default: 0
    - SUN_AZIMUTH [`floating point number`] : Azimuth. Minimum: 0.000000 Maximum: 360.000000 Default: -180.000000 Direction of sun clockwise from North [degree].
    - SUN_HEIGHT [`floating point number`] : Height. Minimum: 0.000000 Maximum: 90.000000 Default: 45.000000 Height of sun above horizon [degree].
    - CLOUD_HEIGHT [`value range`] : Cloud Height. The range of cloud height above ground to be tested.

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '21', 'Cloud Shadow Detection')
    if Tool.is_Okay():
        Tool.Set_Input ('CLOUDS', CLOUDS)
        Tool.Set_Input ('CAND_GRID_IN', CAND_GRID_IN)
        Tool.Set_Input ('BAND_GREEN', BAND_GREEN)
        Tool.Set_Input ('BAND_RED', BAND_RED)
        Tool.Set_Input ('BAND_NIR', BAND_NIR)
        Tool.Set_Input ('BAND_SWIR', BAND_SWIR)
        Tool.Set_Input ('BANDS_BRIGHTNESS', BANDS_BRIGHTNESS)
        Tool.Set_Input ('DEM', DEM)
        Tool.Set_Input ('BAND_THERMAL', BAND_THERMAL)
        Tool.Set_Output('CLOUD_ID', CLOUD_ID)
        Tool.Set_Output('CAND_GRID_OUT', CAND_GRID_OUT)
        Tool.Set_Output('SHADOWS', SHADOWS)
        Tool.Set_Output('CLOUD_INFO', CLOUD_INFO)
        Tool.Set_Option('BANDS_SWIR', BANDS_SWIR)
        Tool.Set_Option('BANDS_THERMAL', BANDS_THERMAL)
        Tool.Set_Option('OUTPUT', OUTPUT)
        Tool.Set_Option('CAND_GRID_VALUE', CAND_GRID_VALUE)
        Tool.Set_Option('THERMAL_UNIT', THERMAL_UNIT)
        Tool.Set_Option('CANDIDATES', CANDIDATES)
        Tool.Set_Option('BRIGHTNESS', BRIGHTNESS)
        Tool.Set_Option('PROCESSING', PROCESSING)
        Tool.Set_Option('SUN_AZIMUTH', SUN_AZIMUTH)
        Tool.Set_Option('SUN_HEIGHT', SUN_HEIGHT)
        Tool.Set_Option('CLOUD_HEIGHT', CLOUD_HEIGHT)
        return Tool.Execute(Verbose)
    return False

def Import_SPOT_Scene(
    BANDS=None,
    METAFILE=None,
    PROJECTION=None,
    RESAMPLING=None,
    UTM_ZONE=None,
    UTM_SOUTH=None,
    SHIFT_X=None,
    SHIFT_Y=None, Verbose=2):
    '''
    Import SPOT Scene
    ----------
    [imagery_tools.22]\n
    Imports a SPOT (Satellite Pour l'Observation de la Terre) scene. Currently this is just a simple import support tool for SPOT level 1A data.\n
    Arguments
    ----------
    - BANDS [`output grid list`] : Spectral Bands
    - METAFILE [`file path`] : Metadata File
    - PROJECTION [`choice`] : Coordinate System. Available Choices: [0] Geographic Coordinates [1] Universal Transverse Mercator Default: 1
    - RESAMPLING [`choice`] : Resampling. Available Choices: [0] Nearest Neighbour [1] Bilinear Interpolation [2] Bicubic Spline Interpolation [3] B-Spline Interpolation Default: 3
    - UTM_ZONE [`integer number`] : Zone. Minimum: 1 Maximum: 60 Default: 32
    - UTM_SOUTH [`boolean`] : South. Default: 0
    - SHIFT_X [`floating point number`] : Shift.x. Default: 0.000000
    - SHIFT_Y [`floating point number`] : Shift.y. Default: 0.000000

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '22', 'Import SPOT Scene')
    if Tool.is_Okay():
        Tool.Set_Output('BANDS', BANDS)
        Tool.Set_Option('METAFILE', METAFILE)
        Tool.Set_Option('PROJECTION', PROJECTION)
        Tool.Set_Option('RESAMPLING', RESAMPLING)
        Tool.Set_Option('UTM_ZONE', UTM_ZONE)
        Tool.Set_Option('UTM_SOUTH', UTM_SOUTH)
        Tool.Set_Option('SHIFT_X', SHIFT_X)
        Tool.Set_Option('SHIFT_Y', SHIFT_Y)
        return Tool.Execute(Verbose)
    return False

def run_tool_imagery_tools_22(
    BANDS=None,
    METAFILE=None,
    PROJECTION=None,
    RESAMPLING=None,
    UTM_ZONE=None,
    UTM_SOUTH=None,
    SHIFT_X=None,
    SHIFT_Y=None, Verbose=2):
    '''
    Import SPOT Scene
    ----------
    [imagery_tools.22]\n
    Imports a SPOT (Satellite Pour l'Observation de la Terre) scene. Currently this is just a simple import support tool for SPOT level 1A data.\n
    Arguments
    ----------
    - BANDS [`output grid list`] : Spectral Bands
    - METAFILE [`file path`] : Metadata File
    - PROJECTION [`choice`] : Coordinate System. Available Choices: [0] Geographic Coordinates [1] Universal Transverse Mercator Default: 1
    - RESAMPLING [`choice`] : Resampling. Available Choices: [0] Nearest Neighbour [1] Bilinear Interpolation [2] Bicubic Spline Interpolation [3] B-Spline Interpolation Default: 3
    - UTM_ZONE [`integer number`] : Zone. Minimum: 1 Maximum: 60 Default: 32
    - UTM_SOUTH [`boolean`] : South. Default: 0
    - SHIFT_X [`floating point number`] : Shift.x. Default: 0.000000
    - SHIFT_Y [`floating point number`] : Shift.y. Default: 0.000000

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '22', 'Import SPOT Scene')
    if Tool.is_Okay():
        Tool.Set_Output('BANDS', BANDS)
        Tool.Set_Option('METAFILE', METAFILE)
        Tool.Set_Option('PROJECTION', PROJECTION)
        Tool.Set_Option('RESAMPLING', RESAMPLING)
        Tool.Set_Option('UTM_ZONE', UTM_ZONE)
        Tool.Set_Option('UTM_SOUTH', UTM_SOUTH)
        Tool.Set_Option('SHIFT_X', SHIFT_X)
        Tool.Set_Option('SHIFT_Y', SHIFT_Y)
        return Tool.Execute(Verbose)
    return False

def Decode_Landsat_Quality_Assessment_Bands(
    IN_QA_PIXEL=None,
    IN_QA_RADSAT=None,
    IN_SR_QA_AEROSOL=None,
    OUTPUT=None,
    SENSOR=None,
    IN_QA_PIX_SELECTION=None,
    IN_QA_RADSAT_SELECTION=None,
    IN_SR_QA_AEROSOL_SELECTION=None,
    SELECTION=None,
    GRIDS=None,
    SET_LUT=None, Verbose=2):
    '''
    Decode Landsat Quality Assessment Bands
    ----------
    [imagery_tools.23]\n
    This tool decodes Landsat Multispectral Scanner System (MSS), Thematic Mapper (TM), Enhanced Thematic Mapper Plus (ETM+), and Operational Land Imager/Thermal Infrared Sensor (OLI/TIRS) Quality Assessment (QA) bands. It splits these QA bands into individual bands and optionally aggregates them into a Grid Collection. It is also possible to select individual flags for output.\n
    Currently, the tool supports Pixel, Radiometric Saturation and Surface Reflectance Aerosol (only OLI/TIRS) Quality Assessment bands from Collection 2 (Level 1 and 2). It also provides value interpretation for certain sensors and QA bands, which can be optionally added to the input datasets for classified displaying in the GUI.\n
    Arguments
    ----------
    - IN_QA_PIXEL [`optional input grid`] : Pixel QA Band. "QA_PIXEL"-Suffix
    - IN_QA_RADSAT [`optional input grid`] : Radiometric Saturation QA Band. "QA_RADSAT"-Suffix
    - IN_SR_QA_AEROSOL [`optional input grid`] : SR Aerosol QA Band. "SR_QA_AEROSOL"-Suffix
    - OUTPUT [`output grid list`] : Output
    - SENSOR [`choice`] : Spacecraft (Sensor). Available Choices: [0] Landsat 1-5 (MSS) [1] Landsat 4-7 (TM & ETM+) [2] Landsat 8-9 (OLI/TIRS) Default: 2
    - IN_QA_PIX_SELECTION [`choices`] : Flag Selection. Available Choices:
    - IN_QA_RADSAT_SELECTION [`choices`] : Flag Selection. Available Choices:
    - IN_SR_QA_AEROSOL_SELECTION [`choices`] : Flag Selection. Available Choices:
    - SELECTION [`boolean`] : Select individual Bands. Default: 0
    - GRIDS [`boolean`] : Output as Grid Collection. Default: 0
    - SET_LUT [`boolean`] : Classify Colors of Input. Default: 1

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '23', 'Decode Landsat Quality Assessment Bands')
    if Tool.is_Okay():
        Tool.Set_Input ('IN_QA_PIXEL', IN_QA_PIXEL)
        Tool.Set_Input ('IN_QA_RADSAT', IN_QA_RADSAT)
        Tool.Set_Input ('IN_SR_QA_AEROSOL', IN_SR_QA_AEROSOL)
        Tool.Set_Output('OUTPUT', OUTPUT)
        Tool.Set_Option('SENSOR', SENSOR)
        Tool.Set_Option('IN_QA_PIX_SELECTION', IN_QA_PIX_SELECTION)
        Tool.Set_Option('IN_QA_RADSAT_SELECTION', IN_QA_RADSAT_SELECTION)
        Tool.Set_Option('IN_SR_QA_AEROSOL_SELECTION', IN_SR_QA_AEROSOL_SELECTION)
        Tool.Set_Option('SELECTION', SELECTION)
        Tool.Set_Option('GRIDS', GRIDS)
        Tool.Set_Option('SET_LUT', SET_LUT)
        return Tool.Execute(Verbose)
    return False

def run_tool_imagery_tools_23(
    IN_QA_PIXEL=None,
    IN_QA_RADSAT=None,
    IN_SR_QA_AEROSOL=None,
    OUTPUT=None,
    SENSOR=None,
    IN_QA_PIX_SELECTION=None,
    IN_QA_RADSAT_SELECTION=None,
    IN_SR_QA_AEROSOL_SELECTION=None,
    SELECTION=None,
    GRIDS=None,
    SET_LUT=None, Verbose=2):
    '''
    Decode Landsat Quality Assessment Bands
    ----------
    [imagery_tools.23]\n
    This tool decodes Landsat Multispectral Scanner System (MSS), Thematic Mapper (TM), Enhanced Thematic Mapper Plus (ETM+), and Operational Land Imager/Thermal Infrared Sensor (OLI/TIRS) Quality Assessment (QA) bands. It splits these QA bands into individual bands and optionally aggregates them into a Grid Collection. It is also possible to select individual flags for output.\n
    Currently, the tool supports Pixel, Radiometric Saturation and Surface Reflectance Aerosol (only OLI/TIRS) Quality Assessment bands from Collection 2 (Level 1 and 2). It also provides value interpretation for certain sensors and QA bands, which can be optionally added to the input datasets for classified displaying in the GUI.\n
    Arguments
    ----------
    - IN_QA_PIXEL [`optional input grid`] : Pixel QA Band. "QA_PIXEL"-Suffix
    - IN_QA_RADSAT [`optional input grid`] : Radiometric Saturation QA Band. "QA_RADSAT"-Suffix
    - IN_SR_QA_AEROSOL [`optional input grid`] : SR Aerosol QA Band. "SR_QA_AEROSOL"-Suffix
    - OUTPUT [`output grid list`] : Output
    - SENSOR [`choice`] : Spacecraft (Sensor). Available Choices: [0] Landsat 1-5 (MSS) [1] Landsat 4-7 (TM & ETM+) [2] Landsat 8-9 (OLI/TIRS) Default: 2
    - IN_QA_PIX_SELECTION [`choices`] : Flag Selection. Available Choices:
    - IN_QA_RADSAT_SELECTION [`choices`] : Flag Selection. Available Choices:
    - IN_SR_QA_AEROSOL_SELECTION [`choices`] : Flag Selection. Available Choices:
    - SELECTION [`boolean`] : Select individual Bands. Default: 0
    - GRIDS [`boolean`] : Output as Grid Collection. Default: 0
    - SET_LUT [`boolean`] : Classify Colors of Input. Default: 1

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '23', 'Decode Landsat Quality Assessment Bands')
    if Tool.is_Okay():
        Tool.Set_Input ('IN_QA_PIXEL', IN_QA_PIXEL)
        Tool.Set_Input ('IN_QA_RADSAT', IN_QA_RADSAT)
        Tool.Set_Input ('IN_SR_QA_AEROSOL', IN_SR_QA_AEROSOL)
        Tool.Set_Output('OUTPUT', OUTPUT)
        Tool.Set_Option('SENSOR', SENSOR)
        Tool.Set_Option('IN_QA_PIX_SELECTION', IN_QA_PIX_SELECTION)
        Tool.Set_Option('IN_QA_RADSAT_SELECTION', IN_QA_RADSAT_SELECTION)
        Tool.Set_Option('IN_SR_QA_AEROSOL_SELECTION', IN_SR_QA_AEROSOL_SELECTION)
        Tool.Set_Option('SELECTION', SELECTION)
        Tool.Set_Option('GRIDS', GRIDS)
        Tool.Set_Option('SET_LUT', SET_LUT)
        return Tool.Execute(Verbose)
    return False

def Awesome_Spectral_Indices(
    VV=None,
    HV=None,
    HH=None,
    VH=None,
    A=None,
    B=None,
    G=None,
    G1=None,
    Y=None,
    R=None,
    RE1=None,
    RE2=None,
    N=None,
    RE3=None,
    N2=None,
    WV=None,
    S1=None,
    S2=None,
    T=None,
    T1=None,
    T2=None,
    RESULT=None,
    GRID_SYSTEM_AUX=None,
    GRID_SYSTEM_VNIR=None,
    GRID_SYSTEM_REDEDGE=None,
    GRID_SYSTEM_SWIR=None,
    GRID_SYSTEM_THERMAL=None,
    INDEX=None,
    C1=None,
    C2=None,
    L=None,
    PAR=None,
    ALPHA=None,
    BETA=None,
    c=None,
    CEXP=None,
    EPSILON=None,
    FDELTA=None,
    g=None,
    GAMMA=None,
    k=None,
    LAMBDAG=None,
    LAMBDAN=None,
    LAMBDAR=None,
    LAMBDAS1=None,
    NEXP=None,
    OMEGA=None,
    p=None,
    SIGMA=None,
    SLA=None,
    SLB=None, Verbose=2):
    '''
    'Awesome' Spectral Indices
    ----------
    [imagery_tools.24]\n
    Calculate a spectral index. Select the index of your choice from the index definitions originally provided by the 'Awesome-Spectral-Indices' project. See [Montero et al. (2023)] for details and the following listing of provided spectral index definitions.\n
    ____________\n
    Spectral Indices\n
    ============\n
    Domain	ID	Name\n
    burn	[BAI](https://digital.csic.es/bitstream/10261/6426/1/Martin_Isabel_Serie_Geografica.pdf)	Burned Area Index\n
    burn	[BAIM](https://doi.org/10.1016/j.foreco.2006.08.248)	Burned Area Index adapted to MODIS\n
    burn	[BAIS2](https://doi.org/10.3390/ecrs-2-05177)	Burned Area Index for Sentinel 2\n
    burn	[CSI](https://doi.org/10.1016/j.rse.2005.04.014)	Char Soil Index\n
    burn	[CSIT](https://doi.org/10.1080/01431160600954704)	Char Soil Index Thermal\n
    burn	[MIRBI](https://doi.org/10.1080/01431160110053185)	Mid-Infrared Burn Index\n
    burn	[NSTv1](https://doi.org/10.1016/j.rse.2011.06.010)	NIR-SWIR-Temperature Version 1\n
    burn	[NSTv2](https://doi.org/10.1016/j.rse.2011.06.010)	NIR-SWIR-Temperature Version 2\n
    burn	[NBR](https://doi.org/10.3133/ofr0211)	Normalized Burn Ratio\n
    burn	[NBR2](https://www.usgs.gov/core-science-systems/nli/landsat/landsat-normalized-burn-ratio-2)	Normalized Burn Ratio 2\n
    burn	[NBRplus](https://doi.org/10.3390/rs14071727)	Normalized Burn Ratio Plus\n
    burn	[NBRSWIR](https://doi.org/10.1080/22797254.2020.1738900)	Normalized Burn Ratio SWIR\n
    burn	[NBRT1](https://doi.org/10.1080/01431160500239008)	Normalized Burn Ratio Thermal 1\n
    burn	[NBRT2](https://doi.org/10.1080/01431160500239008)	Normalized Burn Ratio Thermal 2\n
    burn	[NBRT3](https://doi.org/10.1080/01431160500239008)	Normalized Burn Ratio Thermal 3\n
    burn	[NDSWIR](https://doi.org/10.1109/TGRS.2003.819190)	Normalized Difference SWIR\n
    burn	[NDVIT](https://doi.org/10.1080/01431160600954704)	Normalized Difference Vegetation Index Thermal\n
    burn	[SAVIT](https://doi.org/10.1080/01431160600954704)	Soil-Adjusted Vegetation Index Thermal\n
    burn	[VI6T](https://doi.org/10.1080/01431160500239008)	VI6T Index\n
    radar	[DPDD](https://doi.org/10.1016/j.rse.2018.09.003)	Dual-Pol Diagonal Distance\n
    radar	[DpRVIHH](https://www.tandfonline.com/doi/abs/10.5589/m12-043)	Dual-Polarized Radar Vegetation Index HH\n
    radar	[DpRVIVV](https://doi.org/10.3390/app9040655)	Dual-Polarized Radar Vegetation Index VV\n
    radar	[NDPolI](https://www.isprs.org/proceedings/XXXVII/congress/4_pdf/267.pdf)	Normalized Difference Polarization Index\n
    radar	[QpRVI](https://doi.org/10.1109/IGARSS.2001.976856)	Quad-Polarized Radar Vegetation Index\n
    radar	[RFDI](https://doi.org/10.5194/bg-9-179-2012)	Radar Forest Degradation Index\n
    radar	[VHVVD](https://doi.org/10.3390/app9040655)	VH-VV Difference\n
    radar	[VHVVP](https://doi.org/10.1109/IGARSS47720.2021.9554099)	VH-VV Product\n
    radar	[VHVVR](https://doi.org/10.1109/IGARSS47720.2021.9554099)	VH-VV Ratio\n
    radar	[VVVHD](https://doi.org/10.1109/IGARSS47720.2021.9554099)	VV-VH Difference\n
    radar	[VVVHR](https://doi.org/10.3390/app9040655)	VV-VH Ratio\n
    radar	[VVVHS](https://doi.org/10.1109/IGARSS47720.2021.9554099)	VV-VH Sum\n
    radar	[VDDPI](https://doi.org/10.1016/j.rse.2018.09.003)	Vertical Dual De-Polarization Index\n
    snow	[NBSIMS](https://doi.org/10.3390/rs13142777)	Non-Binary Snow Index for Multi-Component Surfaces\n
    snow	[NDGlaI](https://doi.org/10.1080/01431160802385459)	Normalized Difference Glacier Index\n
    snow	[NDSII](https://doi.org/10.1080/01431160802385459)	Normalized Difference Snow Ice Index\n
    snow	[NDSI](https://doi.org/10.1109/IGARSS.1994.399618)	Normalized Difference Snow Index\n
    snow	[NDSInw](https://doi.org/10.3390/w12051339)	Normalized Difference Snow Index with no Water\n
    snow	[NDSaII](https://doi.org/10.1080/01431160119766)	Normalized Difference Snow and Ice Index\n
    snow	[S3](https://doi.org/10.3178/jjshwr.12.28)	S3 Snow Index\n
    snow	[SWI](https://doi.org/10.3390/rs11232774)	Snow Water Index\n
    soil	[BI](http://citeseerx.ist.psu.edu/viewdoc/download?doi=10.1.1.465.8749&rep=rep1&type=pdf)	Bare Soil Index\n
    soil	[BaI](https://doi.org/10.1109/IGARSS.2005.1525743)	Bareness Index\n
    soil	[DBSI](https://doi.org/10.3390/land7030081)	Dry Bareness Index\n
    soil	[EMBI](https://doi.org/10.1016/j.jag.2022.102703)	Enhanced Modified Bare Soil Index\n
    soil	[BITM](https://doi.org/10.1016/S0034-4257(98)00030-3)	Landsat TM-based Brightness Index\n
    soil	[MBI](https://doi.org/10.3390/land10030231)	Modified Bare Soil Index\n
    soil	[NBLI](https://doi.org/10.3390/rs9030249)	Normalized Difference Bare Land Index\n
    soil	[NBLIOLI](https://doi.org/10.3390/rs9030249)	Normalized Difference Bare Land Index for Landsat-OLI\n
    soil	[NDBaI](https://doi.org/10.1109/IGARSS.2005.1526319)	Normalized Difference Bareness Index\n
    soil	[NDSoI](https://doi.org/10.1016/j.jag.2015.02.010)	Normalized Difference Soil Index\n
    soil	[NSDS](https://doi.org/10.3390/land10030231)	Normalized Shortwave Infrared Difference Soil-Moisture\n
    soil	[NSDSI1](https://doi.org/10.1016/j.isprsjprs.2019.06.012)	Normalized Shortwave-Infrared Difference Bare Soil Moisture Index 1\n
    soil	[NSDSI2](https://doi.org/10.1016/j.isprsjprs.2019.06.012)	Normalized Shortwave-Infrared Difference Bare Soil Moisture Index 2\n
    soil	[NSDSI3](https://doi.org/10.1016/j.isprsjprs.2019.06.012)	Normalized Shortwave-Infrared Difference Bare Soil Moisture Index 3\n
    soil	[BIXS](https://doi.org/10.1016/S0034-4257(98)00030-3)	SPOT HRV XS-based Brightness Index\n
    soil	[RI4XS](https://doi.org/10.1016/S0034-4257(98)00030-3)	SPOT HRV XS-based Redness Index 4\n
    soil	[NDSIWV](https://www.semanticscholar.org/paper/Using-WorldView-2-Vis-NIR-MSI-Imagery-to-Support-Wolf/5e5063ccc4ee76b56b721c866e871d47a77f9fb4)	WorldView Normalized Difference Soil Index\n
    urban	[BRBA](https://www.omicsonline.org/scientific-reports/JGRS-SR136.pdf)	Band Ratio for Built-up Area\n
    urban	[BLFEI](https://doi.org/10.1080/10106049.2018.1497094)	Built-Up Land Features Extraction Index\n
    urban	[DBI](https://doi.org/10.3390/land7030081)	Dry Built-Up Index\n
    urban	[EBBI](https://doi.org/10.3390/rs4102957)	Enhanced Built-Up and Bareness Index\n
    urban	[IBI](https://doi.org/10.1080/01431160802039957)	Index-Based Built-Up Index\n
    urban	[NBUI](https://hdl.handle.net/1959.11/29500)	New Built-Up Index\n
    urban	[NHFD](https://www.semanticscholar.org/paper/Using-WorldView-2-Vis-NIR-MSI-Imagery-to-Support-Wolf/5e5063ccc4ee76b56b721c866e871d47a77f9fb4)	Non-Homogeneous Feature Difference\n
    urban	[NBAI](https://www.omicsonline.org/scientific-reports/JGRS-SR136.pdf)	Normalized Built-up Area Index\n
    urban	[NDBI](http://dx.doi.org/10.1080/01431160304987)	Normalized Difference Built-Up Index\n
    urban	[NDISIb](https://doi.org/10.14358/PERS.76.5.557)	Normalized Difference Impervious Surface Index Blue\n
    urban	[NDISIg](https://doi.org/10.14358/PERS.76.5.557)	Normalized Difference Impervious Surface Index Green\n
    urban	[NDISIr](https://doi.org/10.14358/PERS.76.5.557)	Normalized Difference Impervious Surface Index Red\n
    urban	[NDISImndwi](https://doi.org/10.14358/PERS.76.5.557)	Normalized Difference Impervious Surface Index with MNDWI\n
    urban	[NDISIndwi](https://doi.org/10.14358/PERS.76.5.557)	Normalized Difference Impervious Surface Index with NDWI\n
    urban	[PISI](https://doi.org/10.3390/rs10101521)	Perpendicular Impervious Surface Index\n
    urban	[UI](https://www.isprs.org/proceedings/XXXI/congress/part7/321_XXXI-part7.pdf)	Urban Index\n
    urban	[VIBI](http://dx.doi.org/10.1080/01431161.2012.687842)	Vegetation Index Built-up Index\n
    urban	[VgNIRBI](https://doi.org/10.1016/j.ecolind.2015.03.037)	Visible Green-Based Built-Up Index\n
    urban	[VrNIRBI](https://doi.org/10.1016/j.ecolind.2015.03.037)	Visible Red-Based Built-Up Index\n
    vegetation	[ATSAVI](https://doi.org/10.1016/0034-4257(91)90009-U)	Adjusted Transformed Soil-Adjusted Vegetation Index\n
    vegetation	[AVI](http://citeseerx.ist.psu.edu/viewdoc/download?doi=10.1.1.465.8749&rep=rep1&type=pdf)	Advanced Vegetation Index\n
    vegetation	[AFRI1600](https://doi.org/10.1016/S0034-4257(01)00190-0)	Aerosol Free Vegetation Index (1600 nm)\n
    vegetation	[AFRI2100](https://doi.org/10.1016/S0034-4257(01)00190-0)	Aerosol Free Vegetation Index (2100 nm)\n
    vegetation	[ARI](https://doi.org/10.1562/0031-8655(2001)074%3C0038:OPANEO%3E2.0.CO;2)	Anthocyanin Reflectance Index\n
    vegetation	[ARI2](https://doi.org/10.1562/0031-8655(2001)074%3C0038:OPANEO%3E2.0.CO;2)	Anthocyanin Reflectance Index 2\n
    vegetation	[ARVI](https://doi.org/10.1109/36.134076)	Atmospherically Resistant Vegetation Index\n
    vegetation	[BCC](https://doi.org/10.1016/0034-4257(87)90088-5)	Blue Chromatic Coordinate\n
    vegetation	[bNIRv](https://doi.org/10.1029/2024JG008240)	Blue Near-Infrared Reflectance of Vegetation\n
    vegetation	[BNDVI](https://doi.org/10.1016/S1672-6308(07)60027-4)	Blue Normalized Difference Vegetation Index\n
    vegetation	[BWDRVI](https://doi.org/10.2135/cropsci2007.01.0031)	Blue Wide Dynamic Range Vegetation Index\n
    vegetation	[CRI550](https://doi.org/10.1562/0031-8655(2002)0750272ACCIPL2.0.CO2)	Carotenoid Reflectance Index using 550 nm\n
    vegetation	[CRI700](https://doi.org/10.1562/0031-8655(2002)0750272ACCIPL2.0.CO2)	Carotenoid Reflectance Index using 700 nm\n
    vegetation	[CCI](https://doi.org/10.1073/pnas.1606162113)	Chlorophyll Carotenoid Index\n
    vegetation	[CIG](https://doi.org/10.1078/0176-1617-00887)	Chlorophyll Index Green\n
    vegetation	[CIRE](https://doi.org/10.1078/0176-1617-00887)	Chlorophyll Index Red Edge\n
    vegetation	[CVI](https://doi.org/10.1007/s11119-010-9204-3)	Chlorophyll Vegetation Index\n
    vegetation	[DVI](https://doi.org/10.1016/0034-4257(94)00114-3)	Difference Vegetation Index\n
    vegetation	[DVIplus](https://doi.org/10.1016/j.rse.2019.03.028)	Difference Vegetation Index Plus\n
    vegetation	[DSWI1](https://doi.org/10.1080/01431160310001618031)	Disease-Water Stress Index 1\n
    vegetation	[DSWI2](https://doi.org/10.1080/01431160310001618031)	Disease-Water Stress Index 2\n
    vegetation	[DSWI3](https://doi.org/10.1080/01431160310001618031)	Disease-Water Stress Index 3\n
    vegetation	[DSWI4](https://doi.org/10.1080/01431160310001618031)	Disease-Water Stress Index 4\n
    vegetation	[DSWI5](https://doi.org/10.1080/01431160310001618031)	Disease-Water Stress Index 5\n
    vegetation	[DSI](https://www.asprs.org/wp-content/uploads/pers/1999journal/apr/1999_apr_495-501.pdf)	Drought Stress Index\n
    vegetation	[EBI](https://doi.org/10.1016/j.isprsjprs.2019.08.006)	Enhanced Bloom Index\n
    vegetation	[ENDVI](https://doi.org/10.1371/journal.pone.0186193)	Enhanced Normalized Difference Vegetation Index\n
    vegetation	[EVI](https://doi.org/10.1016/S0034-4257(96)00112-5)	Enhanced Vegetation Index\n
    vegetation	[EVIv](https://doi.org/10.1029/2024JG008240)	Enhanced Vegetation Index of Vegetation\n
    vegetation	[ExGR](https://doi.org/10.1016/j.compag.2008.03.009)	ExG - ExR Vegetation Index\n
    vegetation	[ExG](https://doi.org/10.13031/2013.27838)	Excess Green Index\n
    vegetation	[ExR](https://doi.org/10.1117/12.336896)	Excess Red Index\n
    vegetation	[FCVI](https://doi.org/10.1016/j.rse.2020.111676)	Fluorescence Correction Vegetation Index\n
    vegetation	[GDVI](https://doi.org/10.3390/rs6021211)	Generalized Difference Vegetation Index\n
    vegetation	[GM1](https://doi.org/10.1016/S0176-1617(96)80284-7)	Gitelson and Merzlyak Index 1\n
    vegetation	[GM2](https://doi.org/10.1016/S0176-1617(96)80284-7)	Gitelson and Merzlyak Index 2\n
    vegetation	[GEMI](http://dx.doi.org/10.1007/bf00031911)	Global Environment Monitoring Index\n
    vegetation	[GVMI](https://doi.org/10.1016/S0034-4257(02)00037-8)	Global Vegetation Moisture Index\n
    vegetation	[GARI](https://doi.org/10.1016/S0034-4257(96)00072-7)	Green Atmospherically Resistant Vegetation Index\n
    vegetation	[GCC](https://doi.org/10.1016/0034-4257(87)90088-5)	Green Chromatic Coordinate\n
    vegetation	[GLI](http://dx.doi.org/10.1080/10106040108542184)	Green Leaf Index\n
    vegetation	[GNDVI](https://doi.org/10.1016/S0034-4257(96)00072-7)	Green Normalized Difference Vegetation Index\n
    vegetation	[GOSAVI](https://doi.org/10.2134/agronj2004.0314)	Green Optimized Soil Adjusted Vegetation Index\n
    vegetation	[GRVI](https://doi.org/10.2134/agronj2004.0314)	Green Ratio Vegetation Index\n
    vegetation	[GSAVI](https://doi.org/10.2134/agronj2004.0314)	Green Soil Adjusted Vegetation Index\n
    vegetation	[GBNDVI](https://doi.org/10.1016/S1672-6308(07)60027-4)	Green-Blue Normalized Difference Vegetation Index\n
    vegetation	[GRNDVI](https://doi.org/10.1016/S1672-6308(07)60027-4)	Green-Red Normalized Difference Vegetation Index\n
    vegetation	[NIRvH2](https://doi.org/10.1016/j.rse.2021.112723)	Hyperspectral Near-Infrared Reflectance of Vegetation\n
    vegetation	[IPVI](https://doi.org/10.1016/0034-4257(90)90085-Z)	Infrared Percentage Vegetation Index\n
    vegetation	[IRECI](https://doi.org/10.1016/j.isprsjprs.2013.04.007)	Inverted Red-Edge Chlorophyll Index\n
    vegetation	[IKAW](https://doi.org/10.1006/anbo.1997.0544)	Kawashima Index\n
    vegetation	[MCARIOSAVI](https://doi.org/10.1016/S0034-4257(00)00113-9)	MCARI/OSAVI Ratio\n
    vegetation	[MCARIOSAVI705](https://doi.org/10.1016/j.agrformet.2008.03.005)	MCARI/OSAVI Ratio (705 and 750 nm)\n
    vegetation	[MTCI](https://doi.org/10.1080/0143116042000274015)	MERIS Terrestrial Chlorophyll Index\n
    vegetation	[MCARI](http://dx.doi.org/10.1016/S0034-4257(00)00113-9)	Modified Chlorophyll Absorption in Reflectance Index\n
    vegetation	[MCARI705](https://doi.org/10.1016/j.agrformet.2008.03.005)	Modified Chlorophyll Absorption in Reflectance Index (705 and 750 nm)\n
    vegetation	[MCARI1](https://doi.org/10.1016/j.rse.2003.12.013)	Modified Chlorophyll Absorption in Reflectance Index 1\n
    vegetation	[MCARI2](https://doi.org/10.1016/j.rse.2003.12.013)	Modified Chlorophyll Absorption in Reflectance Index 2\n
    vegetation	[MGRVI](https://doi.org/10.1016/j.jag.2015.02.012)	Modified Green Red Vegetation Index\n
    vegetation	[MNLI](https://doi.org/10.1109/TGRS.2003.812910)	Modified Non-Linear Vegetation Index\n
    vegetation	[mND705](https://doi.org/10.1016/S0034-4257(02)00010-X)	Modified Normalized Difference (705, 750 and 445 nm)\n
    vegetation	[MNDVI](https://doi.org/10.1080/014311697216810)	Modified Normalized Difference Vegetation Index\n
    vegetation	[MRBVI](https://doi.org/10.3390/s20185055)	Modified Red Blue Vegetation Index\n
    vegetation	[MSR](https://doi.org/10.1080/07038992.1996.10855178)	Modified Simple Ratio\n
    vegetation	[mSR705](https://doi.org/10.1016/S0034-4257(02)00010-X)	Modified Simple Ratio (705 and 445 nm)\n
    vegetation	[MSR705](https://doi.org/10.1016/j.agrformet.2008.03.005)	Modified Simple Ratio (705 and 750 nm)\n
    vegetation	[MSAVI](https://doi.org/10.1016/0034-4257(94)90134-1)	Modified Soil-Adjusted Vegetation Index\n
    vegetation	[MTVI1](https://doi.org/10.1016/j.rse.2003.12.013)	Modified Triangular Vegetation Index 1\n
    vegetation	[MTVI2](https://doi.org/10.1016/j.rse.2003.12.013)	Modified Triangular Vegetation Index 2\n
    vegetation	[MSI](https://doi.org/10.1016/0034-4257(89)90046-1)	Moisture Stress Index\n
    vegetation	[NIRv](https://doi.org/10.1126/sciadv.1602244)	Near-Infrared Reflectance of Vegetation\n
    vegetation	[NIRvP](https://doi.org/10.1016/j.rse.2021.112763)	Near-Infrared Reflectance of Vegetation and Incoming PAR\n
    vegetation	[IAVI](https://www.jipb.net/EN/abstract/abstract23925.shtml)	New Atmospherically Resistant Vegetation Index\n
    vegetation	[NLI](https://doi.org/10.1080/02757259409532252)	Non-Linear Vegetation Index\n
    vegetation	[ND705](https://doi.org/10.1016/S0034-4257(02)00010-X)	Normalized Difference (705 and 750 nm)\n
    vegetation	[NDDI](https://doi.org/10.1029/2006GL029127)	Normalized Difference Drought Index\n
    vegetation	[NDGI](https://doi.org/10.1016/j.rse.2019.03.028)	Normalized Difference Greenness Index\n
    vegetation	[NDII](https://www.asprs.org/wp-content/uploads/pers/1983journal/jan/1983_jan_77-83.pdf)	Normalized Difference Infrared Index\n
    vegetation	[NDMI](https://doi.org/10.1016/S0034-4257(01)00318-2)	Normalized Difference Moisture Index\n
    vegetation	[NDPI](https://doi.org/10.1016/j.rse.2017.04.031)	Normalized Difference Phenology Index\n
    vegetation	[NDREI](https://doi.org/10.1016/1011-1344(93)06963-4)	Normalized Difference Red Edge Index\n
    vegetation	[NDVI](https://ntrs.nasa.gov/citations/19740022614)	Normalized Difference Vegetation Index\n
    vegetation	[NDVI705](https://doi.org/10.1016/S0176-1617(11)81633-0)	Normalized Difference Vegetation Index (705 and 750 nm)\n
    vegetation	[NDYI](https://doi.org/10.1016/j.rse.2016.06.016)	Normalized Difference Yellowness Index\n
    vegetation	[NormG](https://doi.org/10.2134/agronj2004.0314)	Normalized Green\n
    vegetation	[NGRDI](https://doi.org/10.1016/0034-4257(79)90013-0)	Normalized Green Red Difference Index\n
    vegetation	[NMDI](https://doi.org/10.1029/2007GL031021)	Normalized Multi-band Drought Index\n
    vegetation	[NormNIR](https://doi.org/10.2134/agronj2004.0314)	Normalized NIR\n
    vegetation	[NRFIg](https://doi.org/10.3390/rs13010105)	Normalized Rapeseed Flowering Index Green\n
    vegetation	[NRFIr](https://doi.org/10.3390/rs13010105)	Normalized Rapeseed Flowering Index Red\n
    vegetation	[NormR](https://doi.org/10.2134/agronj2004.0314)	Normalized Red\n
    vegetation	[OCVI](http://dx.doi.org/10.1007/s11119-008-9075-z)	Optimized Chlorophyll Vegetation Index\n
    vegetation	[OSAVI](https://doi.org/10.1016/0034-4257(95)00186-7)	Optimized Soil-Adjusted Vegetation Index\n
    vegetation	[PSRI](https://doi.org/10.1034/j.1399-3054.1999.106119.x)	Plant Senescing Reflectance Index\n
    vegetation	[RVI](https://doi.org/10.2134/agronj1968.00021962006000060016x)	Ratio Vegetation Index\n
    vegetation	[RCC](https://doi.org/10.1016/0034-4257(87)90088-5)	Red Chromatic Coordinate\n
    vegetation	[RENDVI](https://doi.org/10.1016/S0176-1617(11)81633-0)	Red Edge Normalized Difference Vegetation Index\n
    vegetation	[RGBVI](https://doi.org/10.1016/j.jag.2015.02.012)	Red Green Blue Vegetation Index\n
    vegetation	[REDSI](https://doi.org/10.3390/s18030868)	Red-Edge Disease Stress Index\n
    vegetation	[RGRI](https://doi.org/10.1016/j.jag.2014.03.018)	Red-Green Ratio Index\n
    vegetation	[RI](https://www.documentation.ird.fr/hor/fdi:34390)	Redness Index\n
    vegetation	[RDVI](https://doi.org/10.1016/0034-4257(94)00114-3)	Renormalized Difference Vegetation Index\n
    vegetation	[sNIRvSWIR](https://doi.org/10.1029/2024JG008240)	SWIR-enhanced Near-Infrared Reflectance of Vegetation\n
    vegetation	[sNIRvLSWI](https://doi.org/10.1029/2024JG008240)	SWIR-enhanced Near-Infrared Reflectance of Vegetation for LSWI\n
    vegetation	[sNIRvNDPI](https://doi.org/10.1029/2024JG008240)	SWIR-enhanced Near-Infrared Reflectance of Vegetation for NDPI\n
    vegetation	[sNIRvNDVILSWIP](https://doi.org/10.1029/2024JG008240)	SWIR-enhanced Near-Infrared Reflectance of Vegetation for the NDVI-LSWI Product\n
    vegetation	[sNIRvNDVILSWIS](https://doi.org/10.1029/2024JG008240)	SWIR-enhanced Near-Infrared Reflectance of Vegetation for the NDVI-LSWI Sum\n
    vegetation	[SeLI](https://doi.org/10.3390/s19040904)	Sentinel-2 LAI Green Index\n
    vegetation	[S2REP](https://doi.org/10.1016/j.isprsjprs.2013.04.007)	Sentinel-2 Red-Edge Position\n
    vegetation	[SI](http://citeseerx.ist.psu.edu/viewdoc/download?doi=10.1.1.465.8749&rep=rep1&type=pdf)	Shadow Index\n
    vegetation	[SEVI](https://doi.org/10.1080/17538947.2018.1495770)	Shadow-Eliminated Vegetation Index\n
    vegetation	[SR](https://doi.org/10.2307/1936256)	Simple Ratio\n
    vegetation	[SR555](https://doi.org/10.1016/S0176-1617(11)81633-0)	Simple Ratio (555 and 750 nm)\n
    vegetation	[SR705](https://doi.org/10.1016/S0176-1617(11)81633-0)	Simple Ratio (705 and 750 nm)\n
    vegetation	[SR2](https://doi.org/10.1080/01431169308904370)	Simple Ratio (800 and 550 nm)\n
    vegetation	[SR3](https://doi.org/10.1016/S0034-4257(98)00046-7)	Simple Ratio (860, 550 and 708 nm)\n
    vegetation	[SARVI](https://doi.org/10.1109/36.134076)	Soil Adjusted and Atmospherically Resistant Vegetation Index\n
    vegetation	[SAVI](https://doi.org/10.1016/0034-4257(88)90106-X)	Soil-Adjusted Vegetation Index\n
    vegetation	[SAVI2](https://doi.org/10.1080/01431169008955053)	Soil-Adjusted Vegetation Index 2\n
    vegetation	[SLAVI](https://www.asprs.org/wp-content/uploads/pers/2000journal/february/2000_feb_183-191.pdf)	Specific Leaf Area Vegetation Index\n
    vegetation	[SIPI](https://eurekamag.com/research/009/395/009395053.php)	Structure Insensitive Pigment Index\n
    vegetation	[TCARIOSAVI](https://doi.org/10.1016/S0034-4257(02)00018-4)	TCARI/OSAVI Ratio\n
    vegetation	[TCARIOSAVI705](https://doi.org/10.1016/j.agrformet.2008.03.005)	TCARI/OSAVI Ratio (705 and 750 nm)\n
    vegetation	[TCARI](https://doi.org/10.1016/S0034-4257(02)00018-4)	Transformed Chlorophyll Absorption in Reflectance Index\n
    vegetation	[TDVI](https://doi.org/10.1109/IGARSS.2002.1026867)	Transformed Difference Vegetation Index\n
    vegetation	[TRRVI](https://doi.org/10.3390/rs12152359)	Transformed Red Range Vegetation Index\n
    vegetation	[TSAVI](https://doi.org/10.1109/IGARSS.1989.576128)	Transformed Soil-Adjusted Vegetation Index\n
    vegetation	[TTVI](https://doi.org/10.3390/rs12010016)	Transformed Triangular Vegetation Index\n
    vegetation	[TVI](https://ntrs.nasa.gov/citations/19740022614)	Transformed Vegetation Index\n
    vegetation	[TCI](http://dx.doi.org/10.1109/TGRS.2007.904836)	Triangular Chlorophyll Index\n
    vegetation	[TGI](http://dx.doi.org/10.1016/j.jag.2012.07.020)	Triangular Greenness Index\n
    vegetation	[TriVI](http://dx.doi.org/10.1016/S0034-4257(00)00197-8)	Triangular Vegetation Index\n
    vegetation	[EVI2](https://doi.org/10.1016/j.rse.2008.06.006)	Two-Band Enhanced Vegetation Index\n
    vegetation	[VI700](https://doi.org/10.1016/S0034-4257(01)00289-9)	Vegetation Index (700 nm)\n
    vegetation	[VIG](https://doi.org/10.1016/S0034-4257(01)00289-9)	Vegetation Index Green\n
    vegetation	[VARI](https://doi.org/10.1016/S0034-4257(01)00289-9)	Visible Atmospherically Resistant Index\n
    vegetation	[VARI700](https://doi.org/10.1016/S0034-4257(01)00289-9)	Visible Atmospherically Resistant Index (700 nm)\n
    vegetation	[WDVI](https://doi.org/10.1016/0034-4257(89)90076-X)	Weighted Difference Vegetation Index\n
    vegetation	[WDRVI](https://doi.org/10.1078/0176-1617-01176)	Wide Dynamic Range Vegetation Index\n
    water	[ANDWI](https://doi.org/10.1016/j.envsoft.2021.105030)	Augmented Normalized Difference Water Index\n
    water	[AWEInsh](https://doi.org/10.1016/j.rse.2013.08.029)	Automated Water Extraction Index\n
    water	[AWEIsh](https://doi.org/10.1016/j.rse.2013.08.029)	Automated Water Extraction Index with Shadows Elimination\n
    water	[FAI](https://doi.org/10.1016/j.rse.2009.05.012)	Floating Algae Index\n
    water	[LSWI](https://doi.org/10.1016/j.rse.2003.11.008)	Land Surface Water Index\n
    water	[MLSWI26](https://doi.org/10.3390/rs71215805)	Modified Land Surface Water Index (MODIS Bands 2 and 6)\n
    water	[MLSWI27](https://doi.org/10.3390/rs71215805)	Modified Land Surface Water Index (MODIS Bands 2 and 7)\n
    water	[MNDWI](https://doi.org/10.1080/01431160600589179)	Modified Normalized Difference Water Index\n
    water	[MBWI](https://doi.org/10.1016/j.jag.2018.01.018)	Multi-Band Water Index\n
    water	[NDVIMNDWI](https://doi.org/10.1007/978-3-662-45737-5_51)	NDVI-MNDWI Model\n
    water	[NWI](https://doi.org/10.11873/j.issn.1004-0323.2009.2.167)	New Water Index\n
    water	[NDCI](https://doi.org/10.1016/j.rse.2011.10.016)	Normalized Difference Chlorophyll Index\n
    water	[NDPonI](https://doi.org/10.1016/j.rse.2006.07.012)	Normalized Difference Pond Index\n
    water	[NDTI](https://doi.org/10.1016/j.rse.2006.07.012)	Normalized Difference Turbidity Index\n
    water	[NDWI](https://doi.org/10.1080/01431169608948714)	Normalized Difference Water Index\n
    water	[NDWIns](https://doi.org/10.3390/w12051339)	Normalized Difference Water Index with no Snow Cover and Glaciers\n
    water	[OSI](https://doi.org/10.1016/j.mex.2021.101327)	Oil Spill Index\n
    water	[PI](https://doi.org/10.3390/rs12162648)	Plastic Index\n
    water	[RNDVI](https://doi.org/10.3390/rs12162648)	Reversed Normalized Difference Vegetation Index\n
    water	[MuWIR](https://doi.org/10.3390/rs10101643)	Revised Multi-Spectral Water Index\n
    water	[SWM](https://eoscience.esa.int/landtraining2017/files/posters/MILCZAREK.pdf)	Sentinel Water Mask\n
    water	[S2WI](https://doi.org/10.3390/w13121647)	Sentinel-2 Water Index\n
    water	[TWI](https://doi.org/10.3390/rs14215289)	Triangle Water Index\n
    water	[WI1](https://doi.org/10.3390/rs11182186)	Water Index 1\n
    water	[WI2](https://doi.org/10.3390/rs11182186)	Water Index 2\n
    water	[WI2015](https://doi.org/10.1016/j.rse.2015.12.055)	Water Index 2015\n
    water	[WRI](https://doi.org/10.1109/GEOINFORMATICS.2010.5567762)	Water Ratio Index\n
    ============\n
    Arguments
    ----------
    - VV [`input grid`] : VV
    - HV [`input grid`] : HV
    - HH [`input grid`] : HH
    - VH [`input grid`] : VH
    - A [`input grid`] : Aersols. Wavelengths 400 - 455 nm
    - B [`input grid`] : Blue. Wavelengths 450 - 530 nm
    - G [`input grid`] : Green. Wavelengths 510 - 600 nm
    - G1 [`input grid`] : Green 1. Wavelengths 510 - 550 nm
    - Y [`input grid`] : Yellow. Wavelengths 585 - 625 nm
    - R [`input grid`] : Red. Wavelengths 620 - 690 nm
    - RE1 [`input grid`] : Red Edge 1. Wavelengths 695 - 715 nm
    - RE2 [`input grid`] : Red Edge 2. Wavelengths 730 - 750 nm
    - N [`input grid`] : Near-Infrared (NIR). Wavelengths 760 - 900 nm
    - RE3 [`input grid`] : Red Edge 3. Wavelengths 765 - 795 nm
    - N2 [`input grid`] : Near-Infrared (NIR) 2. Wavelengths 850 - 880 nm
    - WV [`input grid`] : Water Vapour. Wavelengths 930 - 960 nm
    - S1 [`input grid`] : Short-wave Infrared (SWIR) 1. Wavelengths 1550 - 1750 nm
    - S2 [`input grid`] : Short-wave Infrared (SWIR) 2. Wavelengths 2080 - 2350 nm
    - T [`input grid`] : Thermal Infrared. Wavelengths 10400 - 12500 nm
    - T1 [`input grid`] : Thermal Infrared 1. Wavelengths 10600 - 11190 nm
    - T2 [`input grid`] : Thermal Infrared 2. Wavelengths 11500 - 12510 nm
    - RESULT [`output grid`] : Spectral Index
    - GRID_SYSTEM_AUX [`grid system`] : Auxiliary Bands
    - GRID_SYSTEM_VNIR [`grid system`] : Visible & Near-Infrared
    - GRID_SYSTEM_REDEDGE [`grid system`] : Red Edge
    - GRID_SYSTEM_SWIR [`grid system`] : Short-wave Infrared
    - GRID_SYSTEM_THERMAL [`grid system`] : Thermal Infrared
    - INDEX [`choice`] : Spectral Index. Available Choices: [0] BAI [1] BAIM [2] BAIS2 [3] CSI [4] CSIT [5] MIRBI [6] NSTv1 [7] NSTv2 [8] NBR [9] NBR2 [10] NBRplus [11] NBRSWIR [12] NBRT1 [13] NBRT2 [14] NBRT3 [15] NDSWIR [16] NDVIT [17] SAVIT [18] VI6T [19] kEVI [20] kIPVI [21] kNDVI [22] kRVI [23] kVARI [24] DPDD [25] DpRVIHH [26] DpRVIVV [27] NDPolI [28] QpRVI [29] RFDI [30] VHVVD [31] VHVVP [32] VHVVR [33] VVVHD [34] VVVHR [35] VVVHS [36] VDDPI [37] NBSIMS [38] NDGlaI [39] NDSII [40] NDSI [41] NDSInw [42] NDSaII [43] S3 [44] SWI [45] BI [46] BaI [47] DBSI [48] EMBI [49] BITM [50] MBI [51] NBLI [52] NBLIOLI [53] NDBaI [54] NDSoI [55] NSDS [56] NSDSI1 [57] NSDSI2 [58] NSDSI3 [59] BIXS [60] RI4XS [61] NDSIWV [62] BRBA [63] BLFEI [64] DBI [65] EBBI [66] IBI [67] NBUI [68] NHFD [69] NBAI [70] NDBI [71] NDISIb [72] NDISIg [73] NDISIr [74] NDISImndwi [75] NDISIndwi [76] PISI [77] UI [78] VIBI [79] VgNIRBI [80] VrNIRBI [81] ATSAVI [82] AVI [83] AFRI1600 [84] AFRI2100 [85] ARI [86] ARI2 [87] ARVI [88] BCC [89] bNIRv [90] BNDVI [91] BWDRVI [92] CRI550 [93] CRI700 [94] CCI [95] CIG [96] CIRE [97] CVI [98] DVI [99] DVIplus [100] DSWI1 [101] DSWI2 [102] DSWI3 [103] DSWI4 [104] DSWI5 [105] DSI [106] EBI [107] ENDVI [108] EVI [109] EVIv [110] ExGR [111] ExG [112] ExR [113] FCVI [114] GDVI [115] GM1 [116] GM2 [117] GEMI [118] GVMI [119] GARI [120] GCC [121] GLI [122] GNDVI [123] GOSAVI [124] GRVI [125] GSAVI [126] GBNDVI [127] GRNDVI [128] NIRvH2 [129] IPVI [130] IRECI [131] IKAW [132] MCARIOSAVI [133] MCARIOSAVI705 [134] MTCI [135] MCARI [136] MCARI705 [137] MCARI1 [138] MCARI2 [139] MGRVI [140] MNLI [141] mND705 [142] MNDVI [143] MRBVI [144] MSR [145] mSR705 [146] MSR705 [147] MSAVI [148] MTVI1 [149] MTVI2 [150] MSI [151] NIRv [152] NIRvP [153] IAVI [154] NLI [155] ND705 [156] NDDI [157] NDGI [158] NDII [159] NDMI [160] NDPI [161] NDREI [162] NDVI [163] NDVI705 [164] NDYI [165] NormG [166] NGRDI [167] NMDI [168] NormNIR [169] NRFIg [170] NRFIr [171] NormR [172] OCVI [173] OSAVI [174] PSRI [175] RVI [176] RCC [177] RENDVI [178] RGBVI [179] REDSI [180] RGRI [181] RI [182] RDVI [183] sNIRvSWIR [184] sNIRvLSWI [185] sNIRvNDPI [186] sNIRvNDVILSWIP [187] sNIRvNDVILSWIS [188] SeLI [189] S2REP [190] SI [191] SEVI [192] SR [193] SR555 [194] SR705 [195] SR2 [196] SR3 [197] SARVI [198] SAVI [199] SAVI2 [200] SLAVI [201] SIPI [202] TCARIOSAVI [203] TCARIOSAVI705 [204] TCARI [205] TDVI [206] TRRVI [207] TSAVI [208] TTVI [209] TVI [210] TCI [211] TGI [212] TriVI [213] EVI2 [214] VI700 [215] VIG [216] VARI [217] VARI700 [218] WDVI [219] WDRVI [220] ANDWI [221] AWEInsh [222] AWEIsh [223] FAI [224] LSWI [225] MLSWI26 [226] MLSWI27 [227] MNDWI [228] MBWI [229] NDVIMNDWI [230] NWI [231] NDCI [232] NDPonI [233] NDTI [234] NDWI [235] NDWIns [236] OSI [237] PI [238] RNDVI [239] MuWIR [240] SWM [241] S2WI [242] TWI [243] WI1 [244] WI2 [245] WI2015 [246] WRI Default: 0
    - C1 [`floating point number`] : C1. Default: 6.000000 Coefficient 1 for the aerosol resistance term
    - C2 [`floating point number`] : C2. Default: 7.500000 Coefficient 2 for the aerosol resistance term
    - L [`floating point number`] : L. Default: 1.000000 Canopy background adjustment
    - PAR [`floating point number`] : PAR. Default: 0.000000 Photosynthetically Active Radiation
    - ALPHA [`floating point number`] : alpha. Default: 0.100000 Weighting coefficient used for WDRVI
    - BETA [`floating point number`] : beta. Default: 0.050000 Calibration parameter used for NDSInw
    - c [`floating point number`] : c. Default: 1.000000 Trade-off parameter in the polynomial kernel
    - CEXP [`floating point number`] : cexp. Default: 1.160000 Exponent used for OCVI
    - EPSILON [`floating point number`] : epsilon. Default: 1.000000 Adjustment constant used for EBI
    - FDELTA [`floating point number`] : fdelta. Default: 0.581000 Adjustment factor used for SEVI
    - g [`floating point number`] : g. Default: 2.500000 Gain factor
    - GAMMA [`floating point number`] : gamma. Default: 1.000000 Weighting coefficient used for ARVI
    - k [`floating point number`] : k. Default: 0.000000 Slope parameter by soil used for NIRvH2
    - LAMBDAG [`floating point number`] : lambdaG. Default: 0.000000 Green central wavelength (nm)
    - LAMBDAN [`floating point number`] : lambdaN. Default: 0.000000 NIR central wavelength (nm)
    - LAMBDAR [`floating point number`] : lambdaR. Default: 0.000000 Red central wavelength (nm)
    - LAMBDAS1 [`floating point number`] : lambdaS1. Default: 0.000000 SWIR1 central wavelength (nm)
    - NEXP [`floating point number`] : nexp. Default: 2.000000 Exponent used for GDVI
    - OMEGA [`floating point number`] : omega. Default: 2.000000 Weighting coefficient used for MBWI
    - p [`floating point number`] : p. Default: 2.000000 Kernel degree in the polynomial kernel
    - SIGMA [`floating point number`] : sigma. Default: 0.500000 Length-scale parameter in the RBF kernel
    - SLA [`floating point number`] : sla. Default: 1.000000 Soil line slope
    - SLB [`floating point number`] : slb. Default: 0.000000 Soil line intercept

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '24', '\'Awesome\' Spectral Indices')
    if Tool.is_Okay():
        Tool.Set_Input ('VV', VV)
        Tool.Set_Input ('HV', HV)
        Tool.Set_Input ('HH', HH)
        Tool.Set_Input ('VH', VH)
        Tool.Set_Input ('A', A)
        Tool.Set_Input ('B', B)
        Tool.Set_Input ('G', G)
        Tool.Set_Input ('G1', G1)
        Tool.Set_Input ('Y', Y)
        Tool.Set_Input ('R', R)
        Tool.Set_Input ('RE1', RE1)
        Tool.Set_Input ('RE2', RE2)
        Tool.Set_Input ('N', N)
        Tool.Set_Input ('RE3', RE3)
        Tool.Set_Input ('N2', N2)
        Tool.Set_Input ('WV', WV)
        Tool.Set_Input ('S1', S1)
        Tool.Set_Input ('S2', S2)
        Tool.Set_Input ('T', T)
        Tool.Set_Input ('T1', T1)
        Tool.Set_Input ('T2', T2)
        Tool.Set_Output('RESULT', RESULT)
        Tool.Set_Option('GRID_SYSTEM_AUX', GRID_SYSTEM_AUX)
        Tool.Set_Option('GRID_SYSTEM_VNIR', GRID_SYSTEM_VNIR)
        Tool.Set_Option('GRID_SYSTEM_REDEDGE', GRID_SYSTEM_REDEDGE)
        Tool.Set_Option('GRID_SYSTEM_SWIR', GRID_SYSTEM_SWIR)
        Tool.Set_Option('GRID_SYSTEM_THERMAL', GRID_SYSTEM_THERMAL)
        Tool.Set_Option('INDEX', INDEX)
        Tool.Set_Option('C1', C1)
        Tool.Set_Option('C2', C2)
        Tool.Set_Option('L', L)
        Tool.Set_Option('PAR', PAR)
        Tool.Set_Option('alpha', ALPHA)
        Tool.Set_Option('beta', BETA)
        Tool.Set_Option('c', c)
        Tool.Set_Option('cexp', CEXP)
        Tool.Set_Option('epsilon', EPSILON)
        Tool.Set_Option('fdelta', FDELTA)
        Tool.Set_Option('g', g)
        Tool.Set_Option('gamma', GAMMA)
        Tool.Set_Option('k', k)
        Tool.Set_Option('lambdaG', LAMBDAG)
        Tool.Set_Option('lambdaN', LAMBDAN)
        Tool.Set_Option('lambdaR', LAMBDAR)
        Tool.Set_Option('lambdaS1', LAMBDAS1)
        Tool.Set_Option('nexp', NEXP)
        Tool.Set_Option('omega', OMEGA)
        Tool.Set_Option('p', p)
        Tool.Set_Option('sigma', SIGMA)
        Tool.Set_Option('sla', SLA)
        Tool.Set_Option('slb', SLB)
        return Tool.Execute(Verbose)
    return False

def run_tool_imagery_tools_24(
    VV=None,
    HV=None,
    HH=None,
    VH=None,
    A=None,
    B=None,
    G=None,
    G1=None,
    Y=None,
    R=None,
    RE1=None,
    RE2=None,
    N=None,
    RE3=None,
    N2=None,
    WV=None,
    S1=None,
    S2=None,
    T=None,
    T1=None,
    T2=None,
    RESULT=None,
    GRID_SYSTEM_AUX=None,
    GRID_SYSTEM_VNIR=None,
    GRID_SYSTEM_REDEDGE=None,
    GRID_SYSTEM_SWIR=None,
    GRID_SYSTEM_THERMAL=None,
    INDEX=None,
    C1=None,
    C2=None,
    L=None,
    PAR=None,
    ALPHA=None,
    BETA=None,
    c=None,
    CEXP=None,
    EPSILON=None,
    FDELTA=None,
    g=None,
    GAMMA=None,
    k=None,
    LAMBDAG=None,
    LAMBDAN=None,
    LAMBDAR=None,
    LAMBDAS1=None,
    NEXP=None,
    OMEGA=None,
    p=None,
    SIGMA=None,
    SLA=None,
    SLB=None, Verbose=2):
    '''
    'Awesome' Spectral Indices
    ----------
    [imagery_tools.24]\n
    Calculate a spectral index. Select the index of your choice from the index definitions originally provided by the 'Awesome-Spectral-Indices' project. See [Montero et al. (2023)] for details and the following listing of provided spectral index definitions.\n
    ____________\n
    Spectral Indices\n
    ============\n
    Domain	ID	Name\n
    burn	[BAI](https://digital.csic.es/bitstream/10261/6426/1/Martin_Isabel_Serie_Geografica.pdf)	Burned Area Index\n
    burn	[BAIM](https://doi.org/10.1016/j.foreco.2006.08.248)	Burned Area Index adapted to MODIS\n
    burn	[BAIS2](https://doi.org/10.3390/ecrs-2-05177)	Burned Area Index for Sentinel 2\n
    burn	[CSI](https://doi.org/10.1016/j.rse.2005.04.014)	Char Soil Index\n
    burn	[CSIT](https://doi.org/10.1080/01431160600954704)	Char Soil Index Thermal\n
    burn	[MIRBI](https://doi.org/10.1080/01431160110053185)	Mid-Infrared Burn Index\n
    burn	[NSTv1](https://doi.org/10.1016/j.rse.2011.06.010)	NIR-SWIR-Temperature Version 1\n
    burn	[NSTv2](https://doi.org/10.1016/j.rse.2011.06.010)	NIR-SWIR-Temperature Version 2\n
    burn	[NBR](https://doi.org/10.3133/ofr0211)	Normalized Burn Ratio\n
    burn	[NBR2](https://www.usgs.gov/core-science-systems/nli/landsat/landsat-normalized-burn-ratio-2)	Normalized Burn Ratio 2\n
    burn	[NBRplus](https://doi.org/10.3390/rs14071727)	Normalized Burn Ratio Plus\n
    burn	[NBRSWIR](https://doi.org/10.1080/22797254.2020.1738900)	Normalized Burn Ratio SWIR\n
    burn	[NBRT1](https://doi.org/10.1080/01431160500239008)	Normalized Burn Ratio Thermal 1\n
    burn	[NBRT2](https://doi.org/10.1080/01431160500239008)	Normalized Burn Ratio Thermal 2\n
    burn	[NBRT3](https://doi.org/10.1080/01431160500239008)	Normalized Burn Ratio Thermal 3\n
    burn	[NDSWIR](https://doi.org/10.1109/TGRS.2003.819190)	Normalized Difference SWIR\n
    burn	[NDVIT](https://doi.org/10.1080/01431160600954704)	Normalized Difference Vegetation Index Thermal\n
    burn	[SAVIT](https://doi.org/10.1080/01431160600954704)	Soil-Adjusted Vegetation Index Thermal\n
    burn	[VI6T](https://doi.org/10.1080/01431160500239008)	VI6T Index\n
    radar	[DPDD](https://doi.org/10.1016/j.rse.2018.09.003)	Dual-Pol Diagonal Distance\n
    radar	[DpRVIHH](https://www.tandfonline.com/doi/abs/10.5589/m12-043)	Dual-Polarized Radar Vegetation Index HH\n
    radar	[DpRVIVV](https://doi.org/10.3390/app9040655)	Dual-Polarized Radar Vegetation Index VV\n
    radar	[NDPolI](https://www.isprs.org/proceedings/XXXVII/congress/4_pdf/267.pdf)	Normalized Difference Polarization Index\n
    radar	[QpRVI](https://doi.org/10.1109/IGARSS.2001.976856)	Quad-Polarized Radar Vegetation Index\n
    radar	[RFDI](https://doi.org/10.5194/bg-9-179-2012)	Radar Forest Degradation Index\n
    radar	[VHVVD](https://doi.org/10.3390/app9040655)	VH-VV Difference\n
    radar	[VHVVP](https://doi.org/10.1109/IGARSS47720.2021.9554099)	VH-VV Product\n
    radar	[VHVVR](https://doi.org/10.1109/IGARSS47720.2021.9554099)	VH-VV Ratio\n
    radar	[VVVHD](https://doi.org/10.1109/IGARSS47720.2021.9554099)	VV-VH Difference\n
    radar	[VVVHR](https://doi.org/10.3390/app9040655)	VV-VH Ratio\n
    radar	[VVVHS](https://doi.org/10.1109/IGARSS47720.2021.9554099)	VV-VH Sum\n
    radar	[VDDPI](https://doi.org/10.1016/j.rse.2018.09.003)	Vertical Dual De-Polarization Index\n
    snow	[NBSIMS](https://doi.org/10.3390/rs13142777)	Non-Binary Snow Index for Multi-Component Surfaces\n
    snow	[NDGlaI](https://doi.org/10.1080/01431160802385459)	Normalized Difference Glacier Index\n
    snow	[NDSII](https://doi.org/10.1080/01431160802385459)	Normalized Difference Snow Ice Index\n
    snow	[NDSI](https://doi.org/10.1109/IGARSS.1994.399618)	Normalized Difference Snow Index\n
    snow	[NDSInw](https://doi.org/10.3390/w12051339)	Normalized Difference Snow Index with no Water\n
    snow	[NDSaII](https://doi.org/10.1080/01431160119766)	Normalized Difference Snow and Ice Index\n
    snow	[S3](https://doi.org/10.3178/jjshwr.12.28)	S3 Snow Index\n
    snow	[SWI](https://doi.org/10.3390/rs11232774)	Snow Water Index\n
    soil	[BI](http://citeseerx.ist.psu.edu/viewdoc/download?doi=10.1.1.465.8749&rep=rep1&type=pdf)	Bare Soil Index\n
    soil	[BaI](https://doi.org/10.1109/IGARSS.2005.1525743)	Bareness Index\n
    soil	[DBSI](https://doi.org/10.3390/land7030081)	Dry Bareness Index\n
    soil	[EMBI](https://doi.org/10.1016/j.jag.2022.102703)	Enhanced Modified Bare Soil Index\n
    soil	[BITM](https://doi.org/10.1016/S0034-4257(98)00030-3)	Landsat TM-based Brightness Index\n
    soil	[MBI](https://doi.org/10.3390/land10030231)	Modified Bare Soil Index\n
    soil	[NBLI](https://doi.org/10.3390/rs9030249)	Normalized Difference Bare Land Index\n
    soil	[NBLIOLI](https://doi.org/10.3390/rs9030249)	Normalized Difference Bare Land Index for Landsat-OLI\n
    soil	[NDBaI](https://doi.org/10.1109/IGARSS.2005.1526319)	Normalized Difference Bareness Index\n
    soil	[NDSoI](https://doi.org/10.1016/j.jag.2015.02.010)	Normalized Difference Soil Index\n
    soil	[NSDS](https://doi.org/10.3390/land10030231)	Normalized Shortwave Infrared Difference Soil-Moisture\n
    soil	[NSDSI1](https://doi.org/10.1016/j.isprsjprs.2019.06.012)	Normalized Shortwave-Infrared Difference Bare Soil Moisture Index 1\n
    soil	[NSDSI2](https://doi.org/10.1016/j.isprsjprs.2019.06.012)	Normalized Shortwave-Infrared Difference Bare Soil Moisture Index 2\n
    soil	[NSDSI3](https://doi.org/10.1016/j.isprsjprs.2019.06.012)	Normalized Shortwave-Infrared Difference Bare Soil Moisture Index 3\n
    soil	[BIXS](https://doi.org/10.1016/S0034-4257(98)00030-3)	SPOT HRV XS-based Brightness Index\n
    soil	[RI4XS](https://doi.org/10.1016/S0034-4257(98)00030-3)	SPOT HRV XS-based Redness Index 4\n
    soil	[NDSIWV](https://www.semanticscholar.org/paper/Using-WorldView-2-Vis-NIR-MSI-Imagery-to-Support-Wolf/5e5063ccc4ee76b56b721c866e871d47a77f9fb4)	WorldView Normalized Difference Soil Index\n
    urban	[BRBA](https://www.omicsonline.org/scientific-reports/JGRS-SR136.pdf)	Band Ratio for Built-up Area\n
    urban	[BLFEI](https://doi.org/10.1080/10106049.2018.1497094)	Built-Up Land Features Extraction Index\n
    urban	[DBI](https://doi.org/10.3390/land7030081)	Dry Built-Up Index\n
    urban	[EBBI](https://doi.org/10.3390/rs4102957)	Enhanced Built-Up and Bareness Index\n
    urban	[IBI](https://doi.org/10.1080/01431160802039957)	Index-Based Built-Up Index\n
    urban	[NBUI](https://hdl.handle.net/1959.11/29500)	New Built-Up Index\n
    urban	[NHFD](https://www.semanticscholar.org/paper/Using-WorldView-2-Vis-NIR-MSI-Imagery-to-Support-Wolf/5e5063ccc4ee76b56b721c866e871d47a77f9fb4)	Non-Homogeneous Feature Difference\n
    urban	[NBAI](https://www.omicsonline.org/scientific-reports/JGRS-SR136.pdf)	Normalized Built-up Area Index\n
    urban	[NDBI](http://dx.doi.org/10.1080/01431160304987)	Normalized Difference Built-Up Index\n
    urban	[NDISIb](https://doi.org/10.14358/PERS.76.5.557)	Normalized Difference Impervious Surface Index Blue\n
    urban	[NDISIg](https://doi.org/10.14358/PERS.76.5.557)	Normalized Difference Impervious Surface Index Green\n
    urban	[NDISIr](https://doi.org/10.14358/PERS.76.5.557)	Normalized Difference Impervious Surface Index Red\n
    urban	[NDISImndwi](https://doi.org/10.14358/PERS.76.5.557)	Normalized Difference Impervious Surface Index with MNDWI\n
    urban	[NDISIndwi](https://doi.org/10.14358/PERS.76.5.557)	Normalized Difference Impervious Surface Index with NDWI\n
    urban	[PISI](https://doi.org/10.3390/rs10101521)	Perpendicular Impervious Surface Index\n
    urban	[UI](https://www.isprs.org/proceedings/XXXI/congress/part7/321_XXXI-part7.pdf)	Urban Index\n
    urban	[VIBI](http://dx.doi.org/10.1080/01431161.2012.687842)	Vegetation Index Built-up Index\n
    urban	[VgNIRBI](https://doi.org/10.1016/j.ecolind.2015.03.037)	Visible Green-Based Built-Up Index\n
    urban	[VrNIRBI](https://doi.org/10.1016/j.ecolind.2015.03.037)	Visible Red-Based Built-Up Index\n
    vegetation	[ATSAVI](https://doi.org/10.1016/0034-4257(91)90009-U)	Adjusted Transformed Soil-Adjusted Vegetation Index\n
    vegetation	[AVI](http://citeseerx.ist.psu.edu/viewdoc/download?doi=10.1.1.465.8749&rep=rep1&type=pdf)	Advanced Vegetation Index\n
    vegetation	[AFRI1600](https://doi.org/10.1016/S0034-4257(01)00190-0)	Aerosol Free Vegetation Index (1600 nm)\n
    vegetation	[AFRI2100](https://doi.org/10.1016/S0034-4257(01)00190-0)	Aerosol Free Vegetation Index (2100 nm)\n
    vegetation	[ARI](https://doi.org/10.1562/0031-8655(2001)074%3C0038:OPANEO%3E2.0.CO;2)	Anthocyanin Reflectance Index\n
    vegetation	[ARI2](https://doi.org/10.1562/0031-8655(2001)074%3C0038:OPANEO%3E2.0.CO;2)	Anthocyanin Reflectance Index 2\n
    vegetation	[ARVI](https://doi.org/10.1109/36.134076)	Atmospherically Resistant Vegetation Index\n
    vegetation	[BCC](https://doi.org/10.1016/0034-4257(87)90088-5)	Blue Chromatic Coordinate\n
    vegetation	[bNIRv](https://doi.org/10.1029/2024JG008240)	Blue Near-Infrared Reflectance of Vegetation\n
    vegetation	[BNDVI](https://doi.org/10.1016/S1672-6308(07)60027-4)	Blue Normalized Difference Vegetation Index\n
    vegetation	[BWDRVI](https://doi.org/10.2135/cropsci2007.01.0031)	Blue Wide Dynamic Range Vegetation Index\n
    vegetation	[CRI550](https://doi.org/10.1562/0031-8655(2002)0750272ACCIPL2.0.CO2)	Carotenoid Reflectance Index using 550 nm\n
    vegetation	[CRI700](https://doi.org/10.1562/0031-8655(2002)0750272ACCIPL2.0.CO2)	Carotenoid Reflectance Index using 700 nm\n
    vegetation	[CCI](https://doi.org/10.1073/pnas.1606162113)	Chlorophyll Carotenoid Index\n
    vegetation	[CIG](https://doi.org/10.1078/0176-1617-00887)	Chlorophyll Index Green\n
    vegetation	[CIRE](https://doi.org/10.1078/0176-1617-00887)	Chlorophyll Index Red Edge\n
    vegetation	[CVI](https://doi.org/10.1007/s11119-010-9204-3)	Chlorophyll Vegetation Index\n
    vegetation	[DVI](https://doi.org/10.1016/0034-4257(94)00114-3)	Difference Vegetation Index\n
    vegetation	[DVIplus](https://doi.org/10.1016/j.rse.2019.03.028)	Difference Vegetation Index Plus\n
    vegetation	[DSWI1](https://doi.org/10.1080/01431160310001618031)	Disease-Water Stress Index 1\n
    vegetation	[DSWI2](https://doi.org/10.1080/01431160310001618031)	Disease-Water Stress Index 2\n
    vegetation	[DSWI3](https://doi.org/10.1080/01431160310001618031)	Disease-Water Stress Index 3\n
    vegetation	[DSWI4](https://doi.org/10.1080/01431160310001618031)	Disease-Water Stress Index 4\n
    vegetation	[DSWI5](https://doi.org/10.1080/01431160310001618031)	Disease-Water Stress Index 5\n
    vegetation	[DSI](https://www.asprs.org/wp-content/uploads/pers/1999journal/apr/1999_apr_495-501.pdf)	Drought Stress Index\n
    vegetation	[EBI](https://doi.org/10.1016/j.isprsjprs.2019.08.006)	Enhanced Bloom Index\n
    vegetation	[ENDVI](https://doi.org/10.1371/journal.pone.0186193)	Enhanced Normalized Difference Vegetation Index\n
    vegetation	[EVI](https://doi.org/10.1016/S0034-4257(96)00112-5)	Enhanced Vegetation Index\n
    vegetation	[EVIv](https://doi.org/10.1029/2024JG008240)	Enhanced Vegetation Index of Vegetation\n
    vegetation	[ExGR](https://doi.org/10.1016/j.compag.2008.03.009)	ExG - ExR Vegetation Index\n
    vegetation	[ExG](https://doi.org/10.13031/2013.27838)	Excess Green Index\n
    vegetation	[ExR](https://doi.org/10.1117/12.336896)	Excess Red Index\n
    vegetation	[FCVI](https://doi.org/10.1016/j.rse.2020.111676)	Fluorescence Correction Vegetation Index\n
    vegetation	[GDVI](https://doi.org/10.3390/rs6021211)	Generalized Difference Vegetation Index\n
    vegetation	[GM1](https://doi.org/10.1016/S0176-1617(96)80284-7)	Gitelson and Merzlyak Index 1\n
    vegetation	[GM2](https://doi.org/10.1016/S0176-1617(96)80284-7)	Gitelson and Merzlyak Index 2\n
    vegetation	[GEMI](http://dx.doi.org/10.1007/bf00031911)	Global Environment Monitoring Index\n
    vegetation	[GVMI](https://doi.org/10.1016/S0034-4257(02)00037-8)	Global Vegetation Moisture Index\n
    vegetation	[GARI](https://doi.org/10.1016/S0034-4257(96)00072-7)	Green Atmospherically Resistant Vegetation Index\n
    vegetation	[GCC](https://doi.org/10.1016/0034-4257(87)90088-5)	Green Chromatic Coordinate\n
    vegetation	[GLI](http://dx.doi.org/10.1080/10106040108542184)	Green Leaf Index\n
    vegetation	[GNDVI](https://doi.org/10.1016/S0034-4257(96)00072-7)	Green Normalized Difference Vegetation Index\n
    vegetation	[GOSAVI](https://doi.org/10.2134/agronj2004.0314)	Green Optimized Soil Adjusted Vegetation Index\n
    vegetation	[GRVI](https://doi.org/10.2134/agronj2004.0314)	Green Ratio Vegetation Index\n
    vegetation	[GSAVI](https://doi.org/10.2134/agronj2004.0314)	Green Soil Adjusted Vegetation Index\n
    vegetation	[GBNDVI](https://doi.org/10.1016/S1672-6308(07)60027-4)	Green-Blue Normalized Difference Vegetation Index\n
    vegetation	[GRNDVI](https://doi.org/10.1016/S1672-6308(07)60027-4)	Green-Red Normalized Difference Vegetation Index\n
    vegetation	[NIRvH2](https://doi.org/10.1016/j.rse.2021.112723)	Hyperspectral Near-Infrared Reflectance of Vegetation\n
    vegetation	[IPVI](https://doi.org/10.1016/0034-4257(90)90085-Z)	Infrared Percentage Vegetation Index\n
    vegetation	[IRECI](https://doi.org/10.1016/j.isprsjprs.2013.04.007)	Inverted Red-Edge Chlorophyll Index\n
    vegetation	[IKAW](https://doi.org/10.1006/anbo.1997.0544)	Kawashima Index\n
    vegetation	[MCARIOSAVI](https://doi.org/10.1016/S0034-4257(00)00113-9)	MCARI/OSAVI Ratio\n
    vegetation	[MCARIOSAVI705](https://doi.org/10.1016/j.agrformet.2008.03.005)	MCARI/OSAVI Ratio (705 and 750 nm)\n
    vegetation	[MTCI](https://doi.org/10.1080/0143116042000274015)	MERIS Terrestrial Chlorophyll Index\n
    vegetation	[MCARI](http://dx.doi.org/10.1016/S0034-4257(00)00113-9)	Modified Chlorophyll Absorption in Reflectance Index\n
    vegetation	[MCARI705](https://doi.org/10.1016/j.agrformet.2008.03.005)	Modified Chlorophyll Absorption in Reflectance Index (705 and 750 nm)\n
    vegetation	[MCARI1](https://doi.org/10.1016/j.rse.2003.12.013)	Modified Chlorophyll Absorption in Reflectance Index 1\n
    vegetation	[MCARI2](https://doi.org/10.1016/j.rse.2003.12.013)	Modified Chlorophyll Absorption in Reflectance Index 2\n
    vegetation	[MGRVI](https://doi.org/10.1016/j.jag.2015.02.012)	Modified Green Red Vegetation Index\n
    vegetation	[MNLI](https://doi.org/10.1109/TGRS.2003.812910)	Modified Non-Linear Vegetation Index\n
    vegetation	[mND705](https://doi.org/10.1016/S0034-4257(02)00010-X)	Modified Normalized Difference (705, 750 and 445 nm)\n
    vegetation	[MNDVI](https://doi.org/10.1080/014311697216810)	Modified Normalized Difference Vegetation Index\n
    vegetation	[MRBVI](https://doi.org/10.3390/s20185055)	Modified Red Blue Vegetation Index\n
    vegetation	[MSR](https://doi.org/10.1080/07038992.1996.10855178)	Modified Simple Ratio\n
    vegetation	[mSR705](https://doi.org/10.1016/S0034-4257(02)00010-X)	Modified Simple Ratio (705 and 445 nm)\n
    vegetation	[MSR705](https://doi.org/10.1016/j.agrformet.2008.03.005)	Modified Simple Ratio (705 and 750 nm)\n
    vegetation	[MSAVI](https://doi.org/10.1016/0034-4257(94)90134-1)	Modified Soil-Adjusted Vegetation Index\n
    vegetation	[MTVI1](https://doi.org/10.1016/j.rse.2003.12.013)	Modified Triangular Vegetation Index 1\n
    vegetation	[MTVI2](https://doi.org/10.1016/j.rse.2003.12.013)	Modified Triangular Vegetation Index 2\n
    vegetation	[MSI](https://doi.org/10.1016/0034-4257(89)90046-1)	Moisture Stress Index\n
    vegetation	[NIRv](https://doi.org/10.1126/sciadv.1602244)	Near-Infrared Reflectance of Vegetation\n
    vegetation	[NIRvP](https://doi.org/10.1016/j.rse.2021.112763)	Near-Infrared Reflectance of Vegetation and Incoming PAR\n
    vegetation	[IAVI](https://www.jipb.net/EN/abstract/abstract23925.shtml)	New Atmospherically Resistant Vegetation Index\n
    vegetation	[NLI](https://doi.org/10.1080/02757259409532252)	Non-Linear Vegetation Index\n
    vegetation	[ND705](https://doi.org/10.1016/S0034-4257(02)00010-X)	Normalized Difference (705 and 750 nm)\n
    vegetation	[NDDI](https://doi.org/10.1029/2006GL029127)	Normalized Difference Drought Index\n
    vegetation	[NDGI](https://doi.org/10.1016/j.rse.2019.03.028)	Normalized Difference Greenness Index\n
    vegetation	[NDII](https://www.asprs.org/wp-content/uploads/pers/1983journal/jan/1983_jan_77-83.pdf)	Normalized Difference Infrared Index\n
    vegetation	[NDMI](https://doi.org/10.1016/S0034-4257(01)00318-2)	Normalized Difference Moisture Index\n
    vegetation	[NDPI](https://doi.org/10.1016/j.rse.2017.04.031)	Normalized Difference Phenology Index\n
    vegetation	[NDREI](https://doi.org/10.1016/1011-1344(93)06963-4)	Normalized Difference Red Edge Index\n
    vegetation	[NDVI](https://ntrs.nasa.gov/citations/19740022614)	Normalized Difference Vegetation Index\n
    vegetation	[NDVI705](https://doi.org/10.1016/S0176-1617(11)81633-0)	Normalized Difference Vegetation Index (705 and 750 nm)\n
    vegetation	[NDYI](https://doi.org/10.1016/j.rse.2016.06.016)	Normalized Difference Yellowness Index\n
    vegetation	[NormG](https://doi.org/10.2134/agronj2004.0314)	Normalized Green\n
    vegetation	[NGRDI](https://doi.org/10.1016/0034-4257(79)90013-0)	Normalized Green Red Difference Index\n
    vegetation	[NMDI](https://doi.org/10.1029/2007GL031021)	Normalized Multi-band Drought Index\n
    vegetation	[NormNIR](https://doi.org/10.2134/agronj2004.0314)	Normalized NIR\n
    vegetation	[NRFIg](https://doi.org/10.3390/rs13010105)	Normalized Rapeseed Flowering Index Green\n
    vegetation	[NRFIr](https://doi.org/10.3390/rs13010105)	Normalized Rapeseed Flowering Index Red\n
    vegetation	[NormR](https://doi.org/10.2134/agronj2004.0314)	Normalized Red\n
    vegetation	[OCVI](http://dx.doi.org/10.1007/s11119-008-9075-z)	Optimized Chlorophyll Vegetation Index\n
    vegetation	[OSAVI](https://doi.org/10.1016/0034-4257(95)00186-7)	Optimized Soil-Adjusted Vegetation Index\n
    vegetation	[PSRI](https://doi.org/10.1034/j.1399-3054.1999.106119.x)	Plant Senescing Reflectance Index\n
    vegetation	[RVI](https://doi.org/10.2134/agronj1968.00021962006000060016x)	Ratio Vegetation Index\n
    vegetation	[RCC](https://doi.org/10.1016/0034-4257(87)90088-5)	Red Chromatic Coordinate\n
    vegetation	[RENDVI](https://doi.org/10.1016/S0176-1617(11)81633-0)	Red Edge Normalized Difference Vegetation Index\n
    vegetation	[RGBVI](https://doi.org/10.1016/j.jag.2015.02.012)	Red Green Blue Vegetation Index\n
    vegetation	[REDSI](https://doi.org/10.3390/s18030868)	Red-Edge Disease Stress Index\n
    vegetation	[RGRI](https://doi.org/10.1016/j.jag.2014.03.018)	Red-Green Ratio Index\n
    vegetation	[RI](https://www.documentation.ird.fr/hor/fdi:34390)	Redness Index\n
    vegetation	[RDVI](https://doi.org/10.1016/0034-4257(94)00114-3)	Renormalized Difference Vegetation Index\n
    vegetation	[sNIRvSWIR](https://doi.org/10.1029/2024JG008240)	SWIR-enhanced Near-Infrared Reflectance of Vegetation\n
    vegetation	[sNIRvLSWI](https://doi.org/10.1029/2024JG008240)	SWIR-enhanced Near-Infrared Reflectance of Vegetation for LSWI\n
    vegetation	[sNIRvNDPI](https://doi.org/10.1029/2024JG008240)	SWIR-enhanced Near-Infrared Reflectance of Vegetation for NDPI\n
    vegetation	[sNIRvNDVILSWIP](https://doi.org/10.1029/2024JG008240)	SWIR-enhanced Near-Infrared Reflectance of Vegetation for the NDVI-LSWI Product\n
    vegetation	[sNIRvNDVILSWIS](https://doi.org/10.1029/2024JG008240)	SWIR-enhanced Near-Infrared Reflectance of Vegetation for the NDVI-LSWI Sum\n
    vegetation	[SeLI](https://doi.org/10.3390/s19040904)	Sentinel-2 LAI Green Index\n
    vegetation	[S2REP](https://doi.org/10.1016/j.isprsjprs.2013.04.007)	Sentinel-2 Red-Edge Position\n
    vegetation	[SI](http://citeseerx.ist.psu.edu/viewdoc/download?doi=10.1.1.465.8749&rep=rep1&type=pdf)	Shadow Index\n
    vegetation	[SEVI](https://doi.org/10.1080/17538947.2018.1495770)	Shadow-Eliminated Vegetation Index\n
    vegetation	[SR](https://doi.org/10.2307/1936256)	Simple Ratio\n
    vegetation	[SR555](https://doi.org/10.1016/S0176-1617(11)81633-0)	Simple Ratio (555 and 750 nm)\n
    vegetation	[SR705](https://doi.org/10.1016/S0176-1617(11)81633-0)	Simple Ratio (705 and 750 nm)\n
    vegetation	[SR2](https://doi.org/10.1080/01431169308904370)	Simple Ratio (800 and 550 nm)\n
    vegetation	[SR3](https://doi.org/10.1016/S0034-4257(98)00046-7)	Simple Ratio (860, 550 and 708 nm)\n
    vegetation	[SARVI](https://doi.org/10.1109/36.134076)	Soil Adjusted and Atmospherically Resistant Vegetation Index\n
    vegetation	[SAVI](https://doi.org/10.1016/0034-4257(88)90106-X)	Soil-Adjusted Vegetation Index\n
    vegetation	[SAVI2](https://doi.org/10.1080/01431169008955053)	Soil-Adjusted Vegetation Index 2\n
    vegetation	[SLAVI](https://www.asprs.org/wp-content/uploads/pers/2000journal/february/2000_feb_183-191.pdf)	Specific Leaf Area Vegetation Index\n
    vegetation	[SIPI](https://eurekamag.com/research/009/395/009395053.php)	Structure Insensitive Pigment Index\n
    vegetation	[TCARIOSAVI](https://doi.org/10.1016/S0034-4257(02)00018-4)	TCARI/OSAVI Ratio\n
    vegetation	[TCARIOSAVI705](https://doi.org/10.1016/j.agrformet.2008.03.005)	TCARI/OSAVI Ratio (705 and 750 nm)\n
    vegetation	[TCARI](https://doi.org/10.1016/S0034-4257(02)00018-4)	Transformed Chlorophyll Absorption in Reflectance Index\n
    vegetation	[TDVI](https://doi.org/10.1109/IGARSS.2002.1026867)	Transformed Difference Vegetation Index\n
    vegetation	[TRRVI](https://doi.org/10.3390/rs12152359)	Transformed Red Range Vegetation Index\n
    vegetation	[TSAVI](https://doi.org/10.1109/IGARSS.1989.576128)	Transformed Soil-Adjusted Vegetation Index\n
    vegetation	[TTVI](https://doi.org/10.3390/rs12010016)	Transformed Triangular Vegetation Index\n
    vegetation	[TVI](https://ntrs.nasa.gov/citations/19740022614)	Transformed Vegetation Index\n
    vegetation	[TCI](http://dx.doi.org/10.1109/TGRS.2007.904836)	Triangular Chlorophyll Index\n
    vegetation	[TGI](http://dx.doi.org/10.1016/j.jag.2012.07.020)	Triangular Greenness Index\n
    vegetation	[TriVI](http://dx.doi.org/10.1016/S0034-4257(00)00197-8)	Triangular Vegetation Index\n
    vegetation	[EVI2](https://doi.org/10.1016/j.rse.2008.06.006)	Two-Band Enhanced Vegetation Index\n
    vegetation	[VI700](https://doi.org/10.1016/S0034-4257(01)00289-9)	Vegetation Index (700 nm)\n
    vegetation	[VIG](https://doi.org/10.1016/S0034-4257(01)00289-9)	Vegetation Index Green\n
    vegetation	[VARI](https://doi.org/10.1016/S0034-4257(01)00289-9)	Visible Atmospherically Resistant Index\n
    vegetation	[VARI700](https://doi.org/10.1016/S0034-4257(01)00289-9)	Visible Atmospherically Resistant Index (700 nm)\n
    vegetation	[WDVI](https://doi.org/10.1016/0034-4257(89)90076-X)	Weighted Difference Vegetation Index\n
    vegetation	[WDRVI](https://doi.org/10.1078/0176-1617-01176)	Wide Dynamic Range Vegetation Index\n
    water	[ANDWI](https://doi.org/10.1016/j.envsoft.2021.105030)	Augmented Normalized Difference Water Index\n
    water	[AWEInsh](https://doi.org/10.1016/j.rse.2013.08.029)	Automated Water Extraction Index\n
    water	[AWEIsh](https://doi.org/10.1016/j.rse.2013.08.029)	Automated Water Extraction Index with Shadows Elimination\n
    water	[FAI](https://doi.org/10.1016/j.rse.2009.05.012)	Floating Algae Index\n
    water	[LSWI](https://doi.org/10.1016/j.rse.2003.11.008)	Land Surface Water Index\n
    water	[MLSWI26](https://doi.org/10.3390/rs71215805)	Modified Land Surface Water Index (MODIS Bands 2 and 6)\n
    water	[MLSWI27](https://doi.org/10.3390/rs71215805)	Modified Land Surface Water Index (MODIS Bands 2 and 7)\n
    water	[MNDWI](https://doi.org/10.1080/01431160600589179)	Modified Normalized Difference Water Index\n
    water	[MBWI](https://doi.org/10.1016/j.jag.2018.01.018)	Multi-Band Water Index\n
    water	[NDVIMNDWI](https://doi.org/10.1007/978-3-662-45737-5_51)	NDVI-MNDWI Model\n
    water	[NWI](https://doi.org/10.11873/j.issn.1004-0323.2009.2.167)	New Water Index\n
    water	[NDCI](https://doi.org/10.1016/j.rse.2011.10.016)	Normalized Difference Chlorophyll Index\n
    water	[NDPonI](https://doi.org/10.1016/j.rse.2006.07.012)	Normalized Difference Pond Index\n
    water	[NDTI](https://doi.org/10.1016/j.rse.2006.07.012)	Normalized Difference Turbidity Index\n
    water	[NDWI](https://doi.org/10.1080/01431169608948714)	Normalized Difference Water Index\n
    water	[NDWIns](https://doi.org/10.3390/w12051339)	Normalized Difference Water Index with no Snow Cover and Glaciers\n
    water	[OSI](https://doi.org/10.1016/j.mex.2021.101327)	Oil Spill Index\n
    water	[PI](https://doi.org/10.3390/rs12162648)	Plastic Index\n
    water	[RNDVI](https://doi.org/10.3390/rs12162648)	Reversed Normalized Difference Vegetation Index\n
    water	[MuWIR](https://doi.org/10.3390/rs10101643)	Revised Multi-Spectral Water Index\n
    water	[SWM](https://eoscience.esa.int/landtraining2017/files/posters/MILCZAREK.pdf)	Sentinel Water Mask\n
    water	[S2WI](https://doi.org/10.3390/w13121647)	Sentinel-2 Water Index\n
    water	[TWI](https://doi.org/10.3390/rs14215289)	Triangle Water Index\n
    water	[WI1](https://doi.org/10.3390/rs11182186)	Water Index 1\n
    water	[WI2](https://doi.org/10.3390/rs11182186)	Water Index 2\n
    water	[WI2015](https://doi.org/10.1016/j.rse.2015.12.055)	Water Index 2015\n
    water	[WRI](https://doi.org/10.1109/GEOINFORMATICS.2010.5567762)	Water Ratio Index\n
    ============\n
    Arguments
    ----------
    - VV [`input grid`] : VV
    - HV [`input grid`] : HV
    - HH [`input grid`] : HH
    - VH [`input grid`] : VH
    - A [`input grid`] : Aersols. Wavelengths 400 - 455 nm
    - B [`input grid`] : Blue. Wavelengths 450 - 530 nm
    - G [`input grid`] : Green. Wavelengths 510 - 600 nm
    - G1 [`input grid`] : Green 1. Wavelengths 510 - 550 nm
    - Y [`input grid`] : Yellow. Wavelengths 585 - 625 nm
    - R [`input grid`] : Red. Wavelengths 620 - 690 nm
    - RE1 [`input grid`] : Red Edge 1. Wavelengths 695 - 715 nm
    - RE2 [`input grid`] : Red Edge 2. Wavelengths 730 - 750 nm
    - N [`input grid`] : Near-Infrared (NIR). Wavelengths 760 - 900 nm
    - RE3 [`input grid`] : Red Edge 3. Wavelengths 765 - 795 nm
    - N2 [`input grid`] : Near-Infrared (NIR) 2. Wavelengths 850 - 880 nm
    - WV [`input grid`] : Water Vapour. Wavelengths 930 - 960 nm
    - S1 [`input grid`] : Short-wave Infrared (SWIR) 1. Wavelengths 1550 - 1750 nm
    - S2 [`input grid`] : Short-wave Infrared (SWIR) 2. Wavelengths 2080 - 2350 nm
    - T [`input grid`] : Thermal Infrared. Wavelengths 10400 - 12500 nm
    - T1 [`input grid`] : Thermal Infrared 1. Wavelengths 10600 - 11190 nm
    - T2 [`input grid`] : Thermal Infrared 2. Wavelengths 11500 - 12510 nm
    - RESULT [`output grid`] : Spectral Index
    - GRID_SYSTEM_AUX [`grid system`] : Auxiliary Bands
    - GRID_SYSTEM_VNIR [`grid system`] : Visible & Near-Infrared
    - GRID_SYSTEM_REDEDGE [`grid system`] : Red Edge
    - GRID_SYSTEM_SWIR [`grid system`] : Short-wave Infrared
    - GRID_SYSTEM_THERMAL [`grid system`] : Thermal Infrared
    - INDEX [`choice`] : Spectral Index. Available Choices: [0] BAI [1] BAIM [2] BAIS2 [3] CSI [4] CSIT [5] MIRBI [6] NSTv1 [7] NSTv2 [8] NBR [9] NBR2 [10] NBRplus [11] NBRSWIR [12] NBRT1 [13] NBRT2 [14] NBRT3 [15] NDSWIR [16] NDVIT [17] SAVIT [18] VI6T [19] kEVI [20] kIPVI [21] kNDVI [22] kRVI [23] kVARI [24] DPDD [25] DpRVIHH [26] DpRVIVV [27] NDPolI [28] QpRVI [29] RFDI [30] VHVVD [31] VHVVP [32] VHVVR [33] VVVHD [34] VVVHR [35] VVVHS [36] VDDPI [37] NBSIMS [38] NDGlaI [39] NDSII [40] NDSI [41] NDSInw [42] NDSaII [43] S3 [44] SWI [45] BI [46] BaI [47] DBSI [48] EMBI [49] BITM [50] MBI [51] NBLI [52] NBLIOLI [53] NDBaI [54] NDSoI [55] NSDS [56] NSDSI1 [57] NSDSI2 [58] NSDSI3 [59] BIXS [60] RI4XS [61] NDSIWV [62] BRBA [63] BLFEI [64] DBI [65] EBBI [66] IBI [67] NBUI [68] NHFD [69] NBAI [70] NDBI [71] NDISIb [72] NDISIg [73] NDISIr [74] NDISImndwi [75] NDISIndwi [76] PISI [77] UI [78] VIBI [79] VgNIRBI [80] VrNIRBI [81] ATSAVI [82] AVI [83] AFRI1600 [84] AFRI2100 [85] ARI [86] ARI2 [87] ARVI [88] BCC [89] bNIRv [90] BNDVI [91] BWDRVI [92] CRI550 [93] CRI700 [94] CCI [95] CIG [96] CIRE [97] CVI [98] DVI [99] DVIplus [100] DSWI1 [101] DSWI2 [102] DSWI3 [103] DSWI4 [104] DSWI5 [105] DSI [106] EBI [107] ENDVI [108] EVI [109] EVIv [110] ExGR [111] ExG [112] ExR [113] FCVI [114] GDVI [115] GM1 [116] GM2 [117] GEMI [118] GVMI [119] GARI [120] GCC [121] GLI [122] GNDVI [123] GOSAVI [124] GRVI [125] GSAVI [126] GBNDVI [127] GRNDVI [128] NIRvH2 [129] IPVI [130] IRECI [131] IKAW [132] MCARIOSAVI [133] MCARIOSAVI705 [134] MTCI [135] MCARI [136] MCARI705 [137] MCARI1 [138] MCARI2 [139] MGRVI [140] MNLI [141] mND705 [142] MNDVI [143] MRBVI [144] MSR [145] mSR705 [146] MSR705 [147] MSAVI [148] MTVI1 [149] MTVI2 [150] MSI [151] NIRv [152] NIRvP [153] IAVI [154] NLI [155] ND705 [156] NDDI [157] NDGI [158] NDII [159] NDMI [160] NDPI [161] NDREI [162] NDVI [163] NDVI705 [164] NDYI [165] NormG [166] NGRDI [167] NMDI [168] NormNIR [169] NRFIg [170] NRFIr [171] NormR [172] OCVI [173] OSAVI [174] PSRI [175] RVI [176] RCC [177] RENDVI [178] RGBVI [179] REDSI [180] RGRI [181] RI [182] RDVI [183] sNIRvSWIR [184] sNIRvLSWI [185] sNIRvNDPI [186] sNIRvNDVILSWIP [187] sNIRvNDVILSWIS [188] SeLI [189] S2REP [190] SI [191] SEVI [192] SR [193] SR555 [194] SR705 [195] SR2 [196] SR3 [197] SARVI [198] SAVI [199] SAVI2 [200] SLAVI [201] SIPI [202] TCARIOSAVI [203] TCARIOSAVI705 [204] TCARI [205] TDVI [206] TRRVI [207] TSAVI [208] TTVI [209] TVI [210] TCI [211] TGI [212] TriVI [213] EVI2 [214] VI700 [215] VIG [216] VARI [217] VARI700 [218] WDVI [219] WDRVI [220] ANDWI [221] AWEInsh [222] AWEIsh [223] FAI [224] LSWI [225] MLSWI26 [226] MLSWI27 [227] MNDWI [228] MBWI [229] NDVIMNDWI [230] NWI [231] NDCI [232] NDPonI [233] NDTI [234] NDWI [235] NDWIns [236] OSI [237] PI [238] RNDVI [239] MuWIR [240] SWM [241] S2WI [242] TWI [243] WI1 [244] WI2 [245] WI2015 [246] WRI Default: 0
    - C1 [`floating point number`] : C1. Default: 6.000000 Coefficient 1 for the aerosol resistance term
    - C2 [`floating point number`] : C2. Default: 7.500000 Coefficient 2 for the aerosol resistance term
    - L [`floating point number`] : L. Default: 1.000000 Canopy background adjustment
    - PAR [`floating point number`] : PAR. Default: 0.000000 Photosynthetically Active Radiation
    - ALPHA [`floating point number`] : alpha. Default: 0.100000 Weighting coefficient used for WDRVI
    - BETA [`floating point number`] : beta. Default: 0.050000 Calibration parameter used for NDSInw
    - c [`floating point number`] : c. Default: 1.000000 Trade-off parameter in the polynomial kernel
    - CEXP [`floating point number`] : cexp. Default: 1.160000 Exponent used for OCVI
    - EPSILON [`floating point number`] : epsilon. Default: 1.000000 Adjustment constant used for EBI
    - FDELTA [`floating point number`] : fdelta. Default: 0.581000 Adjustment factor used for SEVI
    - g [`floating point number`] : g. Default: 2.500000 Gain factor
    - GAMMA [`floating point number`] : gamma. Default: 1.000000 Weighting coefficient used for ARVI
    - k [`floating point number`] : k. Default: 0.000000 Slope parameter by soil used for NIRvH2
    - LAMBDAG [`floating point number`] : lambdaG. Default: 0.000000 Green central wavelength (nm)
    - LAMBDAN [`floating point number`] : lambdaN. Default: 0.000000 NIR central wavelength (nm)
    - LAMBDAR [`floating point number`] : lambdaR. Default: 0.000000 Red central wavelength (nm)
    - LAMBDAS1 [`floating point number`] : lambdaS1. Default: 0.000000 SWIR1 central wavelength (nm)
    - NEXP [`floating point number`] : nexp. Default: 2.000000 Exponent used for GDVI
    - OMEGA [`floating point number`] : omega. Default: 2.000000 Weighting coefficient used for MBWI
    - p [`floating point number`] : p. Default: 2.000000 Kernel degree in the polynomial kernel
    - SIGMA [`floating point number`] : sigma. Default: 0.500000 Length-scale parameter in the RBF kernel
    - SLA [`floating point number`] : sla. Default: 1.000000 Soil line slope
    - SLB [`floating point number`] : slb. Default: 0.000000 Soil line intercept

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('imagery_tools', '24', '\'Awesome\' Spectral Indices')
    if Tool.is_Okay():
        Tool.Set_Input ('VV', VV)
        Tool.Set_Input ('HV', HV)
        Tool.Set_Input ('HH', HH)
        Tool.Set_Input ('VH', VH)
        Tool.Set_Input ('A', A)
        Tool.Set_Input ('B', B)
        Tool.Set_Input ('G', G)
        Tool.Set_Input ('G1', G1)
        Tool.Set_Input ('Y', Y)
        Tool.Set_Input ('R', R)
        Tool.Set_Input ('RE1', RE1)
        Tool.Set_Input ('RE2', RE2)
        Tool.Set_Input ('N', N)
        Tool.Set_Input ('RE3', RE3)
        Tool.Set_Input ('N2', N2)
        Tool.Set_Input ('WV', WV)
        Tool.Set_Input ('S1', S1)
        Tool.Set_Input ('S2', S2)
        Tool.Set_Input ('T', T)
        Tool.Set_Input ('T1', T1)
        Tool.Set_Input ('T2', T2)
        Tool.Set_Output('RESULT', RESULT)
        Tool.Set_Option('GRID_SYSTEM_AUX', GRID_SYSTEM_AUX)
        Tool.Set_Option('GRID_SYSTEM_VNIR', GRID_SYSTEM_VNIR)
        Tool.Set_Option('GRID_SYSTEM_REDEDGE', GRID_SYSTEM_REDEDGE)
        Tool.Set_Option('GRID_SYSTEM_SWIR', GRID_SYSTEM_SWIR)
        Tool.Set_Option('GRID_SYSTEM_THERMAL', GRID_SYSTEM_THERMAL)
        Tool.Set_Option('INDEX', INDEX)
        Tool.Set_Option('C1', C1)
        Tool.Set_Option('C2', C2)
        Tool.Set_Option('L', L)
        Tool.Set_Option('PAR', PAR)
        Tool.Set_Option('alpha', ALPHA)
        Tool.Set_Option('beta', BETA)
        Tool.Set_Option('c', c)
        Tool.Set_Option('cexp', CEXP)
        Tool.Set_Option('epsilon', EPSILON)
        Tool.Set_Option('fdelta', FDELTA)
        Tool.Set_Option('g', g)
        Tool.Set_Option('gamma', GAMMA)
        Tool.Set_Option('k', k)
        Tool.Set_Option('lambdaG', LAMBDAG)
        Tool.Set_Option('lambdaN', LAMBDAN)
        Tool.Set_Option('lambdaR', LAMBDAR)
        Tool.Set_Option('lambdaS1', LAMBDAS1)
        Tool.Set_Option('nexp', NEXP)
        Tool.Set_Option('omega', OMEGA)
        Tool.Set_Option('p', p)
        Tool.Set_Option('sigma', SIGMA)
        Tool.Set_Option('sla', SLA)
        Tool.Set_Option('slb', SLB)
        return Tool.Execute(Verbose)
    return False

