/*
 *                            COPYRIGHT
 *
 *  sch-rnd - modular/flexible schematics editor - ProtelII netlist export
 *  Copyright (C) 2025 Tibor 'Igor2' Palinkas
 *  Copyright (C) 2025 Aron Barath
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 31 Milk Street, # 960789 Boston, MA 02196 USA.
 *
 *  Contact:
 *    Project page: http://repo.hu/projects/sch-rnd
 *    contact lead developer: http://www.repo.hu/projects/sch-rnd/contact.html
 *    mailing list: http://www.repo.hu/projects/sch-rnd/contact.html
 */


#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <librnd/core/compat_misc.h>
#include <librnd/core/safe_fs.h>
#include <librnd/core/plugins.h>
#include <librnd/core/error.h>
#include <libcschem/config.h>
#include <libcschem/plug_io.h>

#include <plugins/lib_netlist_exp/lib_netlist_exp.h>

static csch_plug_io_t eprotelII_net;

static int protelII_export_prio(const char *fn, const char *fmt, csch_plug_io_type_t type)
{
	if (type != CSCH_IOTYP_NETLIST)
		return 0;
	if (rnd_strcasecmp(fmt, "protelII") == 0)
		return 100;
	if (rnd_strcasecmp(fmt, "protel") == 0)
		return 80;
	return 0;
}

static void protelII_print_value(FILE* const f, const char* value)
{
	if(value)
	{
		fputs(value, f);
	}
}

/* Export abstract components; exports refdes, footprint, value and device */
static void protelII_export_comps(FILE *f, csch_abstract_t *abs)
{
	htsp_entry_t *e;
	for(e = htsp_first(&abs->comps); e != NULL; e = htsp_next(&abs->comps, e)) {
		csch_acomp_t *comp = e->value;
		const char *refdes = sch_nle_get_refdes(comp);
		const char *fp, *dev, *val;

		if (refdes == NULL)
			continue;

		if (comp->hdr.omit)
			continue;

		/* Get main symbol attributes the safe way, considering alternate names;
		   these should be always exported and are usually hardwiared in the
		   netlist format. */
		fp = sch_nle_get_alt_attr(&comp->hdr.attr, SCH_NLE_ALTATTR_FOOTPRINT);
		val = sch_nle_get_alt_attr(&comp->hdr.attr, SCH_NLE_ALTATTR_VALUE);
		dev = sch_nle_get_alt_attr(&comp->hdr.attr, SCH_NLE_ALTATTR_DEVICE);

		if(fp==NULL)
		{
			/* just warn the user */
			rnd_message(RND_MSG_ERROR, "protelII: no footprint is set for \"%s\"\n", refdes);
		}

		fprintf(f, "[\r\n");

		fprintf(f, "DESIGNATOR\r\n");
		protelII_print_value(f, refdes);
		fprintf(f, "\r\n");

		fprintf(f, "FOOTPRINT\r\n");
		protelII_print_value(f, fp);
		fprintf(f, "\r\n");

		fprintf(f, "PARTTYPE\r\n");
		if(val)
		{
			protelII_print_value(f, val);
		}
		else
		{
			protelII_print_value(f, dev);
		}
		fprintf(f, "\r\n");

		fprintf(f, "DESCRIPTION\r\n");
		protelII_print_value(f, dev);
		fprintf(f, "\r\n");

		fprintf(f, "PART FIELD 1\r\n\r\n");
		fprintf(f, "LIBRARYFIELD1 1\r\n\r\n");
		fprintf(f, "]\r\n");
	}
}

/* Export abstract nets; exports net's name and a list of refdes-pinnum */
/* pairs connected as well as {device/value}-pinname pairs */
static void protelII_export_nets(FILE *f, csch_abstract_t *abs)
{
	htsp_entry_t *e;
	long n;

	for(e = htsp_first(&abs->nets); e != NULL; e = htsp_next(&abs->nets, e)) {
		csch_anet_t *net = e->value;
		const char *netname = sch_nle_get_netname(net);

		if (net->hdr.omit) continue;

		/* empty netlists are permitted, just print the netname */
		fprintf(f, "(\r\n");
		protelII_print_value(f, netname);
		fprintf(f, "\r\n");

		for(n = 0; n < net->conns.used; n++) {
			csch_aport_t *port = net->conns.array[n];
			const char *refdes = NULL, *pinnums;

			if (port->hdr.type != CSCH_ATYPE_PORT) {
				rnd_message(RND_MSG_ERROR, "protelII: invalid connection (object type)\n");
				continue;
			}

			pinnums = sch_nle_get_pinnum(port);
			if (pinnums == NULL) {
				rnd_message(RND_MSG_ERROR, "protelII: can't determine port's pin number\n");
				continue;
			}

			if (port->parent != NULL) {
				refdes = sch_nle_get_refdes(port->parent);
				if (port->parent->hdr.omit)
					continue; /* omit component */
			}
			if (refdes == NULL) {
				/* This is not an error: no refdes means: do not export (e.g. gnd) */
/*				rnd_message(RND_MSG_ERROR, "protelII: can't determine port's parent component refdes\n");*/
				continue;
			}

			/* split up pinnum at space and create one or more conn lines connecting
			   each pin to the given net */
			SCH_NLE_FOREACH_PINNUM(pinnums, my_num,
				{
					const char* val = NULL;
					const char* dev = NULL;
					const char* pinname = sch_nle_get_alt_attr(&port->hdr.attr, SCH_NLE_ALTATTR_PINNAME);

					if(port->parent)
					{
						val = sch_nle_get_alt_attr(&port->parent->hdr.attr, SCH_NLE_ALTATTR_VALUE);
						dev = sch_nle_get_alt_attr(&port->parent->hdr.attr, SCH_NLE_ALTATTR_DEVICE);
					}

					if(strchr(refdes, '-'))
					{
						rnd_message(RND_MSG_ERROR, "protelII: Broken output! Refdes contains dash (-), and messes things up: %s\n", refdes);
					}

					/* format allows to print symbolic connections here, so */
					/* we preserve symbolic pin-name here (if exists) */

					protelII_print_value(f, refdes);
					fprintf(f, "-%s ", my_num);
					protelII_print_value(f, val ? val : (dev ? dev : refdes));
					fprintf(f, "-%s\n", pinname ? pinname : my_num);
				}
			);
		}

		fprintf(f, ")\r\n");
	}
}


/* Export netlist from the abstract model */
static int protelII_export_project_abst(const char *fn, const char *fmt, csch_abstract_t *abs, rnd_hid_attr_val_t *options)
{
	TODO("get hidlib as an arg")
	rnd_design_t *hidlib = NULL;
	FILE *f = rnd_fopen(hidlib, fn, "w");
	if (f == NULL)
		return -1;

	fprintf(f, "PROTEL NETLIST 2.0\r\n");
	protelII_export_comps(f, abs);
	protelII_export_nets(f, abs);

	fprintf(f,
		"{\r\n"
		"TRACK\r\n"
		"10\r\n"
		"VIA\r\n"
		"50\r\n"
		"NET TOPOLOGY\r\n"
		"SHORTEST\r\n"
		"ROUTING PRIORITY\r\n"
		"MEDIUM\r\n"
		"LAYER\r\n"
		"UNDEFINED\r\n"
		"}\r\n");

	fclose(f);
	return 0;
}

#include "hid_impl.c"

int pplg_check_ver_export_protelII(int ver_needed) { return 0; }

void pplg_uninit_export_protelII(void)
{
	csch_plug_io_unregister(&eprotelII_net);
	rnd_export_remove_opts_by_cookie(protelII_cookie);
	rnd_hid_remove_hid(&protelII_hid);
}

int pplg_init_export_protelII(void)
{
	RND_API_CHK_VER;

	eprotelII_net.name = "export to ProtelII";
	eprotelII_net.export_prio = protelII_export_prio;
	eprotelII_net.export_project_abst = protelII_export_project_abst;
	eprotelII_net.ext_export_project = ".net";
	csch_plug_io_register(&eprotelII_net);


	rnd_hid_nogui_init(&protelII_hid);

	protelII_hid.struct_size = sizeof(rnd_hid_t);
	protelII_hid.name = "protelII";
	protelII_hid.description = "Exports project's ProtelII netlist";
	protelII_hid.exporter = 1;

	protelII_hid.get_export_options = protelII_get_export_options;
	protelII_hid.do_export = protelII_do_export;
	protelII_hid.parse_arguments = protelII_parse_arguments;
	protelII_hid.argument_array = protelII_values;

	protelII_hid.usage = protelII_usage;

	rnd_hid_register_hid(&protelII_hid);
	rnd_hid_load_defaults(&protelII_hid, protelII_options, NUM_OPTIONS);


	return 0;
}

